/* tifffastcrop

 v. 1.3.10

 Copyright (c) 2013-2017 Christophe Deroulers

 Portions are based on libtiff's tiffcp code. tiffcp is:
 Copyright (c) 1988-1997 Sam Leffler
 Copyright (c) 1991-1997 Silicon Graphics, Inc.

 Distributed under the GNU General Public License v3 -- contact the 
 author for commercial use */

/* TODO: fix option "-c jpeg:r" -- presently, it is deactivated */

#include <stdio.h>
#include <stdlib.h> /* exit, strtoul */
#include <string.h>
#include <strings.h> /* strncasecmp */
#include <assert.h>
#include <errno.h>
#include <tiff.h>
#include <tiffio.h>
#include <jpeglib.h>
#include <math.h> /* lroundl */

#include "config.h"

#ifdef HAVE_PNG
 #include <png.h>
# ifndef HAVE_PNG_CONST_BYTEP
 typedef png_byte * png_const_bytep;
# endif
#endif

#define JPEG_MAX_DIMENSION 65500L /* in libjpeg's jmorecfg.h */

#define EXIT_SYNTAX_ERROR        1
#define EXIT_IO_ERROR            2
#define EXIT_UNHANDLED_FILE_TYPE 3
#define EXIT_INSUFFICIENT_MEMORY 4
#define EXIT_UNABLE_TO_ACHIEVE_TILE_DIMENSIONS 5
#define EXIT_GEOMETRY_ERROR	6

#define CopyField(tag, v) \
    if (TIFFGetField(in, tag, &v)) TIFFSetField(TIFFout, tag, v)
#define CopyField2(tag, v1, v2) \
    if (TIFFGetField(in, tag, &v1, &v2)) TIFFSetField(TIFFout, tag, v1, v2)
#define CopyField3(tag, v1, v2, v3) \
    if (TIFFGetField(in, tag, &v1, &v2, &v3)) TIFFSetField(TIFFout, tag, v1, v2, v3)

static uint32 requestedxmin = 0;
static uint32 requestedymin = 0;
static uint32 requestedwidth = 0;
static uint32 requestedlength = 0;
static int verbose = 0;

#define OUTPUT_FORMAT_TIFF 0
#define OUTPUT_FORMAT_JPEG 1
#define OUTPUT_FORMAT_PNG  2
static int output_format = -1;
static const char TIFF_SUFFIX[] = "tif";
static const char JPEG_SUFFIX[] = "jpg";
static const char PNG_SUFFIX[] = "png";
static const char * OUTPUT_SUFFIX[]= {TIFF_SUFFIX, JPEG_SUFFIX, PNG_SUFFIX};

static uint32 defg3opts = (uint32) -1;
static int jpeg_quality = -1, default_jpeg_quality = 75; /* JPEG quality */
static int png_quality = -1, default_png_quality = 6; /* PNG quality */
static int jpegcolormode = JPEGCOLORMODE_RGB;
static uint16 defcompression = (uint16) -1;
static uint16 defpredictor = (uint16) -1;
static int defpreset = -1;
/*static uint16 defphotometric = (uint16) -1;*/


static void my_asprintf(char ** ret, const char * format, ...)
{
	int n;
	char * p;
	va_list ap;

	va_start(ap, format);
	n= vsnprintf(NULL, 0, format, ap);
	va_end(ap);
	p= malloc(n+1);
	if (p == NULL) {
		perror("Insufficient memory for a character string ");
		exit(EXIT_INSUFFICIENT_MEMORY);
	}
	va_start(ap, format);
	vsnprintf(p, n+1, format, ap);
	va_end(ap);
	*ret= p;
}


static char * photometricName(uint16 photometric)
{
	char * s= NULL;
	switch (photometric) {
		case PHOTOMETRIC_MINISWHITE:
			my_asprintf(&s, "MinIsWhite"); break;
		case PHOTOMETRIC_MINISBLACK:
			my_asprintf(&s, "MinIsBlack"); break;
		case PHOTOMETRIC_RGB:
			my_asprintf(&s, "RGB"); break;
		case PHOTOMETRIC_PALETTE:
			my_asprintf(&s, "Palette"); break;
		case PHOTOMETRIC_YCBCR:
			my_asprintf(&s, "YCbCr"); break;
		default:
			my_asprintf(&s, "%u", photometric);
	}
	return s;
}


static int searchNumberOfDigits(uint32 u)
{
        return snprintf(NULL, 0, "%u", u);
}


static char * searchPrefixBeforeLastDot(const char * path)
{
	char * prefix;
	int l= strlen(path)-1;

	while (l >= 0 && path[l] != '.')
		l--;

	if (l < 0)
		l= strlen(path);

	if ((prefix = malloc(l+1)) == NULL) {
		perror("Insufficient memory for a character string ");
		exit(EXIT_INSUFFICIENT_MEMORY);
	}

	strncpy(prefix, path, l);
	prefix[l]= 0;

	return prefix;
}


static const char * searchSuffix(const char * path)
{
	int l= strlen(path)-1;

	while (l >= 0 && path[l] != '.')
		l--;

	if (l < 0)
		l= strlen(path);
	else
		l++;

	return &(path[l]);
}


static void tiffCopyFieldsButDimensions(TIFF* in, TIFF* TIFFout)
{
	uint16 bitspersample, samplesperpixel, compression, shortv, *shortav;
	float floatv;
	char *stringv;
	uint32 longv;

	CopyField(TIFFTAG_SUBFILETYPE, longv);
	CopyField(TIFFTAG_BITSPERSAMPLE, bitspersample);
	CopyField(TIFFTAG_SAMPLESPERPIXEL, samplesperpixel);
	CopyField(TIFFTAG_COMPRESSION, compression);
	CopyField(TIFFTAG_PHOTOMETRIC, shortv);
	CopyField(TIFFTAG_PREDICTOR, shortv);
	CopyField(TIFFTAG_THRESHHOLDING, shortv);
	CopyField(TIFFTAG_FILLORDER, shortv);
	CopyField(TIFFTAG_ORIENTATION, shortv);
	CopyField(TIFFTAG_MINSAMPLEVALUE, shortv);
	CopyField(TIFFTAG_MAXSAMPLEVALUE, shortv);
	CopyField(TIFFTAG_XRESOLUTION, floatv);
	CopyField(TIFFTAG_YRESOLUTION, floatv);
/*	CopyField(TIFFTAG_GROUP3OPTIONS, longv);
	CopyField(TIFFTAG_GROUP4OPTIONS, longv);*/
	CopyField(TIFFTAG_RESOLUTIONUNIT, shortv);
	CopyField(TIFFTAG_PLANARCONFIG, shortv);
	CopyField(TIFFTAG_XPOSITION, floatv);
	CopyField(TIFFTAG_YPOSITION, floatv);
	CopyField(TIFFTAG_IMAGEDEPTH, longv);
	CopyField(TIFFTAG_TILEDEPTH, longv);
	CopyField(TIFFTAG_SAMPLEFORMAT, shortv);
	CopyField2(TIFFTAG_EXTRASAMPLES, shortv, shortav);
	{ uint16 *red, *green, *blue;
	    CopyField3(TIFFTAG_COLORMAP, red, green, blue);
	}
	{ uint16 shortv2;
	    CopyField2(TIFFTAG_PAGENUMBER, shortv, shortv2);
	}
	CopyField(TIFFTAG_ARTIST, stringv);
	CopyField(TIFFTAG_IMAGEDESCRIPTION, stringv);
	CopyField(TIFFTAG_MAKE, stringv);
	CopyField(TIFFTAG_MODEL, stringv);
	CopyField(TIFFTAG_SOFTWARE, stringv);
	CopyField(TIFFTAG_DATETIME, stringv);
	CopyField(TIFFTAG_HOSTCOMPUTER, stringv);
	CopyField(TIFFTAG_PAGENAME, stringv);
	CopyField(TIFFTAG_DOCUMENTNAME, stringv);
	CopyField(TIFFTAG_FAXDCS, stringv);
}


static tsize_t computeWidthInBytes(uint32 width_in_pixels, uint16 bitsperpixel)
{
	if (bitsperpixel % 8 == 0)
		return (tsize_t) width_in_pixels * (bitsperpixel/8);
	else if (8 % bitsperpixel == 0) {
		uint32 pixelsperbyte = 8/bitsperpixel;
		return (tsize_t) ((width_in_pixels + pixelsperbyte - 1)/
			pixelsperbyte);
	} else
		return 0;
}


static void cpBufToBuf(uint8* out_beginningofline, uint32 out_x,
	uint8* in_beginningofline, uint32 in_x,
	uint32 widthtocopyinpixels, uint16 bitsperpixel,
	uint32 rows, int out_linewidthinbytes, int in_linewidthinbytes)
{
	if (bitsperpixel % 8 == 0) { /* Easy case */
		uint8* in  = in_beginningofline  + in_x  * (bitsperpixel/8);
		uint8* out = out_beginningofline + out_x * (bitsperpixel/8);
		while (rows-- > 0) {
			memcpy(out, in, widthtocopyinpixels * (bitsperpixel/8));
			in += in_linewidthinbytes;
			out += out_linewidthinbytes;
		}
		return;
	}

	/* Hard case. Do computations to prepare steps 1, 2, 3: */
	static const uint8 left_masks[] =
	    { 0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff };

	assert(8 % bitsperpixel == 0);
	int pixelsperbyte = 8/bitsperpixel;
	int out_x_start_whole_outbytes = out_x;
	int in_x_start_whole_outbytes  = in_x;
	uint32 in_skew  = in_linewidthinbytes;
	uint32 out_skew = out_linewidthinbytes;

	 /* 1. Copy pixels to complete the first byte (if incomplete) of dest. */
	int out_startpositioninincompletefirstbyte = out_x % pixelsperbyte;
	int in_startpositioninfirstbyte = in_x % pixelsperbyte;
	int out_pixelsinincompletefirstbyte = 0;
	int shift_first_inbyte_to_first_incompl_outbyte =
		out_startpositioninincompletefirstbyte -
		in_startpositioninfirstbyte;
	int shift_second_inbyte_to_first_incompl_outbyte = 0;
	uint8 mask_of_first_inbyte_to_first_incompl_outbyte = 0;
	uint8 mask_of_second_inbyte_to_first_incompl_outbyte = 0;

	if (out_startpositioninincompletefirstbyte) {
		out_pixelsinincompletefirstbyte =
		    8-out_startpositioninincompletefirstbyte;
		if (out_pixelsinincompletefirstbyte > widthtocopyinpixels)
			out_pixelsinincompletefirstbyte = widthtocopyinpixels;
		int pixels_available_in_first_inbyte =
		    8 - in_startpositioninfirstbyte;

		if (pixels_available_in_first_inbyte >= out_pixelsinincompletefirstbyte) {
			mask_of_first_inbyte_to_first_incompl_outbyte =
			    left_masks[out_pixelsinincompletefirstbyte * bitsperpixel] >>
				(in_startpositioninfirstbyte * bitsperpixel);
		} else {
			mask_of_first_inbyte_to_first_incompl_outbyte =
			    left_masks[pixels_available_in_first_inbyte *
					bitsperpixel] >>
				(in_startpositioninfirstbyte * bitsperpixel);
			mask_of_second_inbyte_to_first_incompl_outbyte =
			    left_masks[(out_pixelsinincompletefirstbyte -
				pixels_available_in_first_inbyte) * bitsperpixel];
			shift_second_inbyte_to_first_incompl_outbyte =
			    pixels_available_in_first_inbyte;
			in_skew--;
		}

		in_x_start_whole_outbytes += out_pixelsinincompletefirstbyte;
		out_x_start_whole_outbytes += out_pixelsinincompletefirstbyte;
	}

	/* 2. Write as many whole bytes as possible in dest. */
	/* Examples of bits in in_bytes:
	  6+2|8|1+7 -> (6+)2|6+2|6(+2)  11 bits -> 2 bytes, 1 whole byte
	  6+2|8|7+1 -> (6+)2|6+2|6+2    17 bits -> 3 bytes, 2 whole bytes
	   Strategy: copy whole bytes. Then make an additional, 
	  incomplete byte (which shall have 
	  out_pixelsinincompletelastbyte pixels) with the remaining (not 
	  yet copied) bits of current byte in input and, if these bits 
	  are not enough, the first in_bitstoreadinlastbyte bits of next 
	  byte in input. */
	uint8* out_wholeoutbytes = out_beginningofline +
		out_x_start_whole_outbytes / pixelsperbyte;
	uint8* in = in_beginningofline + in_x / pixelsperbyte;
	uint32 wholebytesperline =
	    ((widthtocopyinpixels-out_pixelsinincompletefirstbyte) *
		bitsperpixel) / 8;
	int in_bitoffset = (in_x_start_whole_outbytes % pixelsperbyte) * bitsperpixel;
	if (in_bitoffset) {
		in_skew -= wholebytesperline + 1;
		out_skew -= wholebytesperline;
	}

	/* 3. Copy pixels to complete the last byte (if incomplete) of dest. */
	int out_pixelsinincompletelastbyte =
		(out_x + widthtocopyinpixels) % pixelsperbyte;
	if (out_pixelsinincompletelastbyte) {
		if (in_bitoffset == 0)
			wholebytesperline++; /* Let memcpy start writing last byte */
		else
			out_skew--;
	}
	int in_bitstoreadinlastbyte =
	    out_pixelsinincompletelastbyte * bitsperpixel - (8-in_bitoffset);
	if (in_bitstoreadinlastbyte < 0)
		in_bitstoreadinlastbyte = 0;

	/* Perform steps 1, 2, 3: */
	while (rows-- > 0) {
		/* 1. */
		if (out_startpositioninincompletefirstbyte) {
			uint8 a = (*in) & mask_of_first_inbyte_to_first_incompl_outbyte;
			if (shift_first_inbyte_to_first_incompl_outbyte >= 0)
				a >>= shift_first_inbyte_to_first_incompl_outbyte;
			else
				a <<= -shift_first_inbyte_to_first_incompl_outbyte;
			if (shift_second_inbyte_to_first_incompl_outbyte) {
				in++;
				a |= ((*in) & mask_of_second_inbyte_to_first_incompl_outbyte)
				    >> shift_second_inbyte_to_first_incompl_outbyte;
			}

			*(out_wholeoutbytes-1) &= left_masks[out_startpositioninincompletefirstbyte];
			*(out_wholeoutbytes-1) |= a;
		}

		/* 2. & 3. */
		if (in_bitoffset == 0) {
			memcpy(out_wholeoutbytes, in, wholebytesperline);
			/* out_pixelsinincompletelastbyte =
			 in_bitstoreadinlastbyte / bitsperpixel in this case */
			if (in_bitstoreadinlastbyte)
				*(out_wholeoutbytes + wholebytesperline) |=
				    (*(in+wholebytesperline))
				    >> (8-in_bitstoreadinlastbyte);
		} else {
			uint32 j = wholebytesperline;
			uint8 acc = (*in++) << in_bitoffset;

			while (j-- > 0) {
				acc |= (*in) >> (8-in_bitoffset);
				*out_wholeoutbytes++ = acc;
				acc = (*in++) << in_bitoffset;
			}
			if (out_pixelsinincompletelastbyte) {
				if (in_bitstoreadinlastbyte)
					acc |= (*in) >> (8-in_bitstoreadinlastbyte);
				*out_wholeoutbytes++ = acc;
			}
		}

		in += in_skew;
		out_wholeoutbytes += out_skew;
	}
}


static int testAndFixInTIFFPhotoAndCompressionParameters(TIFF* in,
	uint16* input_compression)
{
	uint16 bitspersample, input_photometric;

	TIFFGetFieldDefaulted(in, TIFFTAG_BITSPERSAMPLE, &bitspersample);

	TIFFGetFieldDefaulted(in, TIFFTAG_COMPRESSION, input_compression);
	TIFFGetFieldDefaulted(in, TIFFTAG_PHOTOMETRIC, &input_photometric);
	if (verbose) {
		char * pn= photometricName(input_photometric);
		fprintf(stderr, "Input file \"%s\" had compression %u "
			"and photometric interpretation %s.\n",
			TIFFFileName(in), *input_compression, pn);
		free(pn);
	}
	if (*input_compression == COMPRESSION_JPEG) {
		/* like in libtiff's tiffcp.c -- otherwise the reserved
		 size for the tiles is too small and the program
		 segfaults */
		/* Force conversion to RGB */
		TIFFSetField(in, TIFFTAG_JPEGCOLORMODE, JPEGCOLORMODE_RGB);
	} else if (input_photometric == PHOTOMETRIC_YCBCR) {
		/* Otherwise, can't handle subsampled input */
		uint16 subsamplinghor, subsamplingver;

		TIFFGetFieldDefaulted(in, TIFFTAG_YCBCRSUBSAMPLING,
				      &subsamplinghor, &subsamplingver);
		if (subsamplinghor!=1 || subsamplingver!=1) {
			fprintf(stderr, "%s: Can't deal with subsampled image.\n",
				TIFFFileName(in));
			return 0;
		}
	}

	return 1;
}


static int testAndFixOutTIFFPhotoAndCompressionParameters(TIFF* in,
	TIFF* TIFFout)
{
	uint16 input_photometric, input_compression, compression, spp;

	TIFFGetFieldDefaulted(in, TIFFTAG_SAMPLESPERPIXEL, &spp);
	TIFFGetFieldDefaulted(in, TIFFTAG_COMPRESSION, &input_compression);
	TIFFGetFieldDefaulted(in, TIFFTAG_PHOTOMETRIC, &input_photometric);

	if (defcompression != (uint16) -1)
		TIFFSetField(TIFFout, TIFFTAG_COMPRESSION, defcompression);

	TIFFGetField(TIFFout, TIFFTAG_COMPRESSION, &compression);
	if (compression == COMPRESSION_JPEG) {
		if (input_photometric == PHOTOMETRIC_RGB &&
		    jpegcolormode == JPEGCOLORMODE_RGB)
			TIFFSetField(TIFFout, TIFFTAG_PHOTOMETRIC,
				PHOTOMETRIC_YCBCR);
		else
			TIFFSetField(TIFFout, TIFFTAG_PHOTOMETRIC,
				input_photometric);
		TIFFSetField(TIFFout, TIFFTAG_JPEGCOLORMODE,
			jpegcolormode);
		TIFFSetField(TIFFout, TIFFTAG_JPEGQUALITY,
			jpeg_quality);
	} else if (compression == COMPRESSION_SGILOG
			|| compression == COMPRESSION_SGILOG24)
		TIFFSetField(TIFFout, TIFFTAG_PHOTOMETRIC,
		    spp == 1 ? PHOTOMETRIC_LOGL : PHOTOMETRIC_LOGLUV);
	else if (compression == COMPRESSION_LZW ||
		    compression == COMPRESSION_ADOBE_DEFLATE ||
		    compression == COMPRESSION_DEFLATE/* ||
	    	compression == COMPRESSION_LZMA*/) {
		if (defpredictor != (uint16)-1)
			TIFFSetField(TIFFout, TIFFTAG_PREDICTOR,
				defpredictor);
		if (defpreset != -1) {
			if (compression == COMPRESSION_ADOBE_DEFLATE ||
			    compression == COMPRESSION_DEFLATE)
				TIFFSetField(TIFFout,
					TIFFTAG_ZIPQUALITY, defpreset);
			/*else if (compression == COMPRESSION_LZMA)
				TIFFSetField(TIFFout,
					TIFFTAG_LZMAPRESET,
					defpreset);*/
		}
		if (input_compression == COMPRESSION_JPEG &&
		    input_photometric == PHOTOMETRIC_YCBCR)
			TIFFSetField(TIFFout, TIFFTAG_PHOTOMETRIC,
				PHOTOMETRIC_RGB);
	} else if (compression == COMPRESSION_NONE) {
		if (input_compression == COMPRESSION_JPEG &&
		    input_photometric == PHOTOMETRIC_YCBCR)
			TIFFSetField(TIFFout, TIFFTAG_PHOTOMETRIC,
				PHOTOMETRIC_RGB);
	} else if (compression == COMPRESSION_CCITTFAX3 ||
		   compression == COMPRESSION_CCITTFAX4) {
		uint32 longv;
		char *stringv;
		if (compression == COMPRESSION_CCITTFAX3 &&
		    input_compression == COMPRESSION_CCITTFAX3)
			{ CopyField(TIFFTAG_GROUP3OPTIONS, longv); }
		else if (compression == COMPRESSION_CCITTFAX4 &&
		    input_compression == COMPRESSION_CCITTFAX4)
			CopyField(TIFFTAG_GROUP4OPTIONS, longv);
		CopyField(TIFFTAG_BADFAXLINES, longv);
		CopyField(TIFFTAG_CLEANFAXDATA, longv);
		CopyField(TIFFTAG_CONSECUTIVEBADFAXLINES, longv);
		CopyField(TIFFTAG_FAXRECVPARAMS, longv);
		CopyField(TIFFTAG_FAXRECVTIME, longv);
		CopyField(TIFFTAG_FAXSUBADDRESS, stringv);
	}

	/*if (defphotometric != (uint16) -1)
		TIFFSetField(TIFFout, TIFFTAG_PHOTOMETRIC,
			defphotometric);*/

	if (verbose) {
		uint16 output_photometric;
		char * pn;

		TIFFGetField(TIFFout, TIFFTAG_PHOTOMETRIC,
			&output_photometric);
		pn= photometricName(output_photometric);
		fprintf(stderr, "Output file \"%s\" will have "
			"compression %u and photometric "
			"interpretation %u (%s).\n",
			TIFFFileName(TIFFout), compression,
			output_photometric, pn);
		free(pn);
	}

	return 1;
}


static int cpTiles2Strip(TIFF* in, uint32 xmin, uint32 ymin,
	uint32 width, uint32 length, unsigned char * outbuf,
	tsize_t outscanlinesizeinbytes, uint16 bitsperpixel)
{
	tmsize_t inbufsize;
	uint32 intilewidth = (uint32) -1, intilelength = (uint32) -1;
	tsize_t intilewidthinbytes = TIFFTileRowSize(in);
	uint32 y;
	unsigned char * inbuf, * bufp= outbuf;
	int error = 0;

	TIFFGetField(in, TIFFTAG_TILEWIDTH, &intilewidth);
	TIFFGetField(in, TIFFTAG_TILELENGTH, &intilelength);

	inbufsize= TIFFTileSize(in);
	inbuf = (unsigned char *)_TIFFmalloc(inbufsize); /* not malloc
	    because TIFFTileSize returns a tmsize_t */
	if (!inbuf) {
		TIFFError(TIFFFileName(in),
				"Error, can't allocate space for image buffer");
		return (EXIT_INSUFFICIENT_MEMORY);
	}

	for (y = ymin ; y < ymin + length + intilelength ; y += intilelength) {
		uint32 x, out_x = 0;
		uint32 yminoftile = (y/intilelength) * intilelength;
		uint32 ymintocopy = ymin > yminoftile ? ymin : yminoftile;
		uint32 ymaxplusone = yminoftile + intilelength;
		uint32 lengthtocopy;
		unsigned char * inbufrow = inbuf +
		    intilewidthinbytes * (ymintocopy-yminoftile);

		if (ymaxplusone > ymin + length)
			ymaxplusone = ymin + length;
		if (ymaxplusone <= yminoftile)
			break;
		lengthtocopy = ymaxplusone - ymintocopy;

		for (x = xmin ; x < xmin + width + intilewidth ;
		    x += intilewidth) {
			uint32 xminoftile = (x/intilewidth) * intilewidth;
			uint32 xmintocopyintile = xmin > xminoftile ?
			    xmin : xminoftile;
			uint32 xmaxplusone = xminoftile + intilewidth;

			if (xmaxplusone > xmin + width)
				xmaxplusone = xmin + width;
			if (xmaxplusone <= xminoftile)
				break;

			uint32 widthtocopyinpixels =
			    xmaxplusone - xmintocopyintile;

			if (TIFFReadTile(in, inbuf, xminoftile,
			    yminoftile, 0, 0) < 0) {
				TIFFError(TIFFFileName(in),
				    "Error, can't read tile at "
				    UINT32_FORMAT ", " UINT32_FORMAT,
				    xminoftile, yminoftile);
				error = EXIT_IO_ERROR;
				goto done;
			}

			cpBufToBuf(bufp, out_x, inbufrow,
			    xmintocopyintile-xminoftile,
			    widthtocopyinpixels, bitsperpixel,
			    lengthtocopy, outscanlinesizeinbytes,
			    intilewidthinbytes);
			out_x += widthtocopyinpixels;
		}
		bufp += outscanlinesizeinbytes * lengthtocopy;
	}

	done:
	_TIFFfree(inbuf);
	return error;
}

static int cpStrips2Strip(TIFF* in,
	uint32 xmin, uint32 ymin, uint32 width, uint32 length,
	unsigned char * outbuf, tsize_t outscanlinesizeinbytes,
	uint16 bitsperpixel,
	uint32 * y_of_last_read_scanline, uint32 inimagelength)
{
	tsize_t inbufsize;
	uint16 input_compression;
	tsize_t inwidthinbytes = TIFFScanlineSize(in);
	uint32 y;
	unsigned char * inbuf, * bufp= outbuf;
	int error = 0;

	TIFFGetFieldDefaulted(in, TIFFTAG_COMPRESSION, &input_compression);
	inbufsize= TIFFScanlineSize(in);  /* not malloc because
	    TIFFScanlineSize returns a tmsize_t */
	inbuf = (unsigned char *)_TIFFmalloc(inbufsize);
	if (!inbuf) {
		TIFFError(TIFFFileName(in),
				"Error, can't allocate space for image buffer");
		return (EXIT_INSUFFICIENT_MEMORY);
	}

	/* When compression method doesn't support random access: */
	if (input_compression != COMPRESSION_NONE) {
		uint32 y;

		if (*y_of_last_read_scanline > ymin) {
		/* If we need to go back, finish reading to the end, 
		 * then a restart will be automatic */
			for (y = *y_of_last_read_scanline + 1 ;
			     y < inimagelength ; y++)
				if (TIFFReadScanline(in, inbuf, y, 0) < 0) {
					TIFFError(TIFFFileName(in),
					    "Error, can't read scanline at "
					    UINT32_FORMAT " for exhausting",
					    y);
					error = EXIT_IO_ERROR;
					goto done;
				} else
					*y_of_last_read_scanline= y;
		}

		/* Then read up to the point we want to start
		 * copying at */
		for (y = 0 ; y < ymin ; y++)
			if (TIFFReadScanline(in, inbuf, y, 0) < 0) {
				TIFFError(TIFFFileName(in),
				    "Error, can't read scanline at "
				    UINT32_FORMAT " for exhausting", y);
				error = EXIT_IO_ERROR;
				goto done;
			} else
				*y_of_last_read_scanline= y;
	}

	for (y = ymin ; y < ymin + length ; y++) {
		uint32 xmintocopyinscanline = xmin;

		if (TIFFReadScanline(in, inbuf, y, 0) < 0) {
			TIFFError(TIFFFileName(in),
			    "Error, can't read scanline at "
			    UINT32_FORMAT " for copying", y);
			error = EXIT_IO_ERROR;
			goto done;
		} else
			*y_of_last_read_scanline= y;

		cpBufToBuf(bufp, 0, inbuf, xmintocopyinscanline,
		    requestedwidth, bitsperpixel, 1,
		    outscanlinesizeinbytes, inwidthinbytes);
		bufp += outscanlinesizeinbytes;
	}

	done:
	_TIFFfree(inbuf);
	return error;
}


	/* Return 0 if the requested memory size exceeds the machine's
	  addressing size type (size_t) capacity or if biptspersample is
	  unhandled */
static size_t computeMemorySize(uint16 spp, uint16 bitspersample,
	uint32 outwidth, uint32 outlength)
{
	uint16 bitsperpixel = spp * bitspersample;
	uint64 memorysize = outlength;
	if (bitsperpixel % 8 == 0)
		memorysize *= (uint64) outwidth * (bitsperpixel/8);
	else if (8 % bitsperpixel == 0) {
		int pixelsperbyte = 8/bitsperpixel;
		int bytesperrow = (outwidth + pixelsperbyte - 1)/pixelsperbyte;
		memorysize *= bytesperrow;
	} else
		return 0;

	if ((size_t) memorysize != memorysize)
		return 0;
	return memorysize;
}


static int makeExtractFromTIFFFile(const char * infilename,
	const char * outfilename)
{
	TIFF * in;
	uint32 inimagewidth, inimagelength;
	uint32 outwidth = requestedwidth, outlength = requestedlength;
	uint16 planarconfig, spp, bitspersample;
	size_t outmemorysize;
	char * ouroutfilename = NULL;
	unsigned char * outbuf = NULL;
	void * out; /* TIFF* or FILE* */
	uint32 y_of_last_read_scanline = 0;
	int return_code = 0; /* Success */

	if (requestedwidth == 0 || requestedlength == 0) {
		fprintf(stderr, "Requested extract is empty. Can't do it. Aborting.\n");
		return EXIT_GEOMETRY_ERROR;
	}

	in = TIFFOpen(infilename, "r");
	if (in == NULL) {
		if (verbose)
			fprintf(stderr, "Unable to open file \"%s\".\n",
				infilename);
		return EXIT_IO_ERROR;
	}

	if (verbose)
		fprintf(stderr, "File \"%s\" open.\n", infilename);

	TIFFGetField(in, TIFFTAG_IMAGEWIDTH, &inimagewidth);
	TIFFGetField(in, TIFFTAG_IMAGELENGTH, &inimagelength);
	TIFFGetFieldDefaulted(in, TIFFTAG_SAMPLESPERPIXEL, &spp);
	TIFFGetFieldDefaulted(in, TIFFTAG_BITSPERSAMPLE, &bitspersample);

	if (output_format == OUTPUT_FORMAT_JPEG &&
	    (bitspersample != 8 || spp != 3)) {
		TIFFError(TIFFFileName(in),
			"Can't output JPEG file from file with "
			"bits-per-sample %u (not 8) or "
			"samples-per-pixel %u (not 3). Aborting",
			bitspersample, spp);
		TIFFClose(in);
		return EXIT_UNHANDLED_FILE_TYPE;
	} else if (output_format == OUTPUT_FORMAT_PNG) {
		    /* We support only PNG_COLOR_TYPE_GRAY,
		     PNG_COLOR_TYPE_RGB and PNG_COLOR_TYPE_RGB_ALPHA */
		int success = 1;
		if (spp != 1 && spp != 3 && spp != 4) {
			TIFFError(TIFFFileName(in),
				"Can't output PNG file from file with "
				"samples-per-pixel %d (not 1, 3 or 4). Aborting",
				spp);
			success = 0;
		} else switch(spp) {
			case 1:
			if (bitspersample != 1 && bitspersample != 2 &&
			    bitspersample != 4 && bitspersample != 8 &&
			    bitspersample != 16) {
				TIFFError(TIFFFileName(in),
					"Error, can't output PNG file from file with 1 sample per pixel and "
					"bitdepth (bits-per-sample) %u (not 1, 2, 4, 8 or 16). Aborting",
					bitspersample);
				success = 0;
			}
			break;

			case 3:
			case 4:
			if (bitspersample != 8 && bitspersample != 16) {
				TIFFError(TIFFFileName(in),
					"Error, can't output PNG file from file with %u samples per pixel and "
					"bitdepth (bits-per-sample) %u (not 8 or 16). Aborting",
					spp, bitspersample);
				success = 0;
			}
			break;

			default:
			fprintf(stderr, "Internal error.\n");
			return EXIT_UNHANDLED_FILE_TYPE;
		}
		if (!success) {
			TIFFClose(in);
			return EXIT_UNHANDLED_FILE_TYPE;
		}
	} else if (bitspersample % 8 != 0 && 8 % bitspersample != 0) {
		TIFFError(TIFFFileName(in),
			"Error, can't deal with file with "
			"bits-per-sample %d (not a multiple nor a "
			"divisor of 8)",
			bitspersample);
		TIFFClose(in);
		return EXIT_UNHANDLED_FILE_TYPE;
	}

	if (verbose)
		fprintf(stderr, "File \"%s\" has %u bits per sample and %u samples per pixel.\n",
			infilename, (unsigned) bitspersample, (unsigned) spp);

	if (requestedxmin > inimagewidth || requestedymin > inimagelength) {
		fprintf(stderr, "Requested top left corner is outside the image. Aborting.\n");
		TIFFClose(in);
		return EXIT_GEOMETRY_ERROR;
	}
	if (requestedxmin + requestedwidth > inimagewidth ||
	    requestedymin + requestedlength > inimagelength) {
		if (requestedxmin + requestedwidth > inimagewidth)
			requestedwidth = inimagewidth - requestedxmin;
		if (requestedymin + requestedlength > inimagelength)
			requestedlength = inimagelength - requestedymin;
		if (verbose)
			fprintf(stderr, "Requested rectangle extends "
				"outside the image. Adjusting "
				"dimensions to " UINT32_FORMAT "x" 
				UINT32_FORMAT ".\n",
				requestedwidth, requestedlength);
	}

	TIFFGetField(in, TIFFTAG_PLANARCONFIG, &planarconfig);
	if (planarconfig != PLANARCONFIG_CONTIG) {
		TIFFError(TIFFFileName(in),
			"Error, can't deal with file with "
			"planar configuration %d (non contiguous)",
			planarconfig);
		TIFFClose(in);
		return EXIT_UNHANDLED_FILE_TYPE;
	}

	if (output_format == OUTPUT_FORMAT_JPEG &&
	    ( (requestedwidth >= JPEG_MAX_DIMENSION) ||
	      (requestedlength >= JPEG_MAX_DIMENSION) ) ) {
		fprintf(stderr, "At least one requested extract dimension is too large for JPEG files.\n");
		TIFFClose(in);
		return EXIT_UNABLE_TO_ACHIEVE_TILE_DIMENSIONS;
	}

	outmemorysize= computeMemorySize(spp, bitspersample, outwidth,
		outlength);
	outbuf= outmemorysize == 0 ? NULL : malloc(outmemorysize);
	if (outbuf == NULL) {
		fprintf(stderr, "Unable to allocate enough memory to prepare extract ("
		        UINT64_FORMAT " bytes needed).\n",
		        outmemorysize);
		TIFFClose(in);
		return EXIT_INSUFFICIENT_MEMORY;
	}

	{
	char * prefix = searchPrefixBeforeLastDot(infilename);
	if (outfilename == NULL) {
		uint32 ndigitsx = searchNumberOfDigits(inimagewidth),
		    ndigitsy = searchNumberOfDigits(inimagelength);
		my_asprintf(&ouroutfilename, "%s-%0*u-%0*u-%0*ux%0*u.%s",
			    prefix, ndigitsx, requestedxmin, ndigitsy,
			    requestedymin, ndigitsx,
			    requestedwidth, ndigitsy, requestedlength,
			    OUTPUT_SUFFIX[output_format]);
		outfilename = ouroutfilename;
	}
	free(prefix);
	}

	out = output_format != OUTPUT_FORMAT_TIFF ?
	    (void *) fopen(outfilename, "wb") :
	    (void *) TIFFOpen(outfilename, TIFFIsBigEndian(in)?"wb":"wl");
	if (verbose) {
		if (out == NULL)
			fprintf(stderr, "Error: unable to open output file"
				" \"%s\".\n", outfilename);
			else
				fprintf(stderr, "Output file \"%s\" open."
					" Will write extract of size "
					UINT32_FORMAT " x "
					UINT32_FORMAT ".\n",
					outfilename,
					requestedwidth, requestedlength);
	}
	if (ouroutfilename != NULL)
		free(ouroutfilename);
	if (out == NULL) {
		TIFFClose(in);
		return EXIT_IO_ERROR;
	}

	{
		uint16 in_compression;
		testAndFixInTIFFPhotoAndCompressionParameters(in,
		    &in_compression);

		if (jpeg_quality <= 0) {
			if (in_compression == COMPRESSION_JPEG) {
				uint16 in_jpegquality;
				TIFFGetField(in, TIFFTAG_JPEGQUALITY, &in_jpegquality);
				jpeg_quality = in_jpegquality;
			} else
				jpeg_quality = default_jpeg_quality;
		}
	}
	if (png_quality <= 0)
		png_quality = default_png_quality;

	switch(output_format) {
	case OUTPUT_FORMAT_JPEG:
		{
		struct jpeg_compress_struct cinfo;
		struct jpeg_error_mgr jerr;
		uint16 bytesperpixel = (bitspersample/8) * spp;
		tsize_t outscanlinesizeinbytes =
		    requestedwidth * bytesperpixel;
		int error = 0;

		cinfo.err = jpeg_std_error(&jerr);
		jpeg_create_compress(&cinfo);
		jpeg_stdio_dest(&cinfo, out);
		cinfo.image_width = requestedwidth;
		cinfo.image_height = requestedlength;
		cinfo.input_components = spp; /* # of color comp. per pixel */
		cinfo.in_color_space = JCS_RGB; /* colorspace of input image */
		jpeg_set_defaults(&cinfo);
		if (verbose)
			fprintf(stderr, "Quality of produced JPEG will be %d.\n",
				jpeg_quality);
		jpeg_set_quality(&cinfo, jpeg_quality,
		    TRUE /* limit to baseline-JPEG values */);
		jpeg_start_compress(&cinfo, TRUE);

		if (((TIFFIsTiled(in) &&
		      !(error = cpTiles2Strip(in,
			    requestedxmin, requestedymin,
			    requestedwidth, requestedlength,
			    outbuf, outscanlinesizeinbytes, bytesperpixel))) ||
		     (!TIFFIsTiled(in) &&
		      !(error = cpStrips2Strip(in,
			    requestedxmin, requestedymin,
			    requestedwidth, requestedlength,
			    outbuf, outscanlinesizeinbytes, bytesperpixel,
			    &y_of_last_read_scanline,
			    inimagelength))))) {
			if (verbose)
				fprintf(stderr, "Extract prepared.\n");

			uint32 y;
			JSAMPROW row_pointer;
			JSAMPROW* row_pointers =
				malloc(requestedlength * sizeof(JSAMPROW));

			if (row_pointers == NULL) {
				fprintf(stderr, "Error, can't allocate space for row_pointers.\n");
				fclose(out);
				free(outbuf);
				TIFFClose(in);
				return EXIT_INSUFFICIENT_MEMORY;
			}

			for (y = 0, row_pointer = outbuf ; y < requestedlength ;
			    y++, row_pointer += requestedwidth * bytesperpixel)
				row_pointers[y]= row_pointer;

			jpeg_write_scanlines(&cinfo, row_pointers, requestedlength);
			free(row_pointers);
		} else {
			fprintf(stderr, "Error, can't write extract.\n");
			return_code = error;
		}

		jpeg_finish_compress(&cinfo);
		fclose(out);
		jpeg_destroy_compress(&cinfo);
		}
		break;

#ifdef HAVE_PNG
	case OUTPUT_FORMAT_PNG:
		{
		uint16 bitsperpixel = bitspersample * spp;
		tsize_t outscanlinesizeinbytes = computeWidthInBytes(
		    requestedwidth, bitsperpixel);
		int error = 0;

		png_structp png_ptr = png_create_write_struct(
		    PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
		if (!png_ptr)
			{ fclose(out); return EXIT_INSUFFICIENT_MEMORY; }
		png_infop info_ptr = png_create_info_struct(png_ptr);
		if (!info_ptr) {
			fclose(out);
			png_destroy_write_struct(&png_ptr, (png_infopp) NULL);
			return EXIT_INSUFFICIENT_MEMORY;
		}
		if (setjmp(png_jmpbuf(png_ptr))) {
			fclose(out);
			png_destroy_write_struct(&png_ptr, (png_infopp) NULL);
			png_destroy_write_struct(&png_ptr, (png_infopp) &info_ptr);
			fprintf(stderr, "Error, can't write extract.\n");
			return EXIT_INSUFFICIENT_MEMORY;
		}
		png_init_io(png_ptr, out);

		png_set_IHDR(png_ptr, info_ptr, requestedwidth,
		    requestedlength, bitspersample,
		    spp == 4 ? PNG_COLOR_TYPE_RGB_ALPHA :
		    (spp == 3 ? PNG_COLOR_TYPE_RGB : PNG_COLOR_TYPE_GRAY),
		    PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_BASE,
		    PNG_FILTER_TYPE_BASE);
		#ifdef WORDS_BIGENDIAN
		png_set_swap_alpha(png_ptr);
		#else
		png_set_bgr(png_ptr);
		#endif
		png_color_8 sig_bit;
		switch (spp) {
			case 1:
			sig_bit.gray = bitspersample;
			break;

			case 4:
			sig_bit.alpha = bitspersample;
			case 3:
			sig_bit.red = bitspersample;
			sig_bit.green = bitspersample;
			sig_bit.blue = bitspersample;
			break;
		}
		png_set_sBIT(png_ptr, info_ptr, &sig_bit);
		png_set_compression_level(png_ptr, png_quality);
		png_write_info(png_ptr, info_ptr);
		/*png_set_shift(png_ptr, &sig_bit);*/ /* useless 
		 because we support only 1, 2, 4, 8, 16 bit-depths */
		/*png_set_packing(png_ptr);*/ /* Use *only* if bits are 
		 not yet packed */

		if (((TIFFIsTiled(in) &&
		      !(error = cpTiles2Strip(in,
			    requestedxmin, requestedymin,
			    requestedwidth, requestedlength,
			    outbuf, outscanlinesizeinbytes,
			    bitsperpixel))) ||
		     (!TIFFIsTiled(in) &&
		      !(error = cpStrips2Strip(in,
			    requestedxmin, requestedymin,
			    requestedwidth, requestedlength,
			    outbuf, outscanlinesizeinbytes,
			    bitsperpixel,
			    &y_of_last_read_scanline,
			    inimagelength))))) {
			if (verbose)
				fprintf(stderr, "Extract prepared.\n");

			png_const_bytep row_pointer = outbuf;
			uint32 y;
			for (y = 0 ; y < requestedlength ;
			    y++, row_pointer += outscanlinesizeinbytes)
				png_write_row(png_ptr, row_pointer);
		} else
			return_code = error;

		png_write_end(png_ptr, info_ptr);
		fclose(out);
		png_destroy_write_struct(&png_ptr, (png_infopp) & info_ptr);
		png_destroy_info_struct(png_ptr, (png_infopp) & info_ptr);
		}
		break;
#endif

	case OUTPUT_FORMAT_TIFF:
		{
		uint16 bitsperpixel = bitspersample * spp;
		tsize_t outscanlinesizeinbytes;
		int error = 0;

		tiffCopyFieldsButDimensions(in, out);
		TIFFSetField(out, TIFFTAG_IMAGEWIDTH, requestedwidth);
		TIFFSetField(out, TIFFTAG_IMAGELENGTH, requestedlength);
		TIFFSetField(out, TIFFTAG_ROWSPERSTRIP, requestedlength);

		testAndFixOutTIFFPhotoAndCompressionParameters(in, out);
			/* To be done *after* setting compression --
			 * otherwise, ScanlineSize may be wrong */
		outscanlinesizeinbytes = TIFFScanlineSize(out);

		if (((TIFFIsTiled(in) &&
		      !(error = cpTiles2Strip(in, requestedxmin,
			requestedymin, requestedwidth, requestedlength,
			outbuf, outscanlinesizeinbytes, bitsperpixel))) ||
		     (!TIFFIsTiled(in) &&
		      !(error = cpStrips2Strip(in, requestedxmin,
			requestedymin, requestedwidth, requestedlength,
			outbuf, outscanlinesizeinbytes, bitsperpixel,
			&y_of_last_read_scanline,
			inimagelength))))) {
			if (verbose)
				fprintf(stderr, "Extract prepared.\n");

			if (TIFFWriteEncodedStrip(out,
				TIFFComputeStrip(out, 0, 0), outbuf,
				TIFFStripSize(out)) < 0) {
				TIFFError(TIFFFileName(out),
					"Error, can't write strip");
			} else if (verbose)
				fprintf(stderr, "Extract written to "
					"output file \"%s\".\n",
					TIFFFileName(out));
		} else
			return_code = error;

		TIFFClose(out);
		}
		break;

	default:
		fprintf(stderr, "Unsupported output file format.\n");
		free(outbuf);
		TIFFClose(in);
		return EXIT_UNHANDLED_FILE_TYPE;
	}

	if (return_code == 0 && verbose)
		fprintf(stderr, "Extract written.\n");
	free(outbuf);
	TIFFClose(in);
	return return_code;
}


static void usage()
{
	fprintf(stderr, "tifffastcrop v" PACKAGE_VERSION " license GNU GPL v3 (c) 2013-2017 Christophe Deroulers\n\n");
	fprintf(stderr, "Quote \"Deroulers et al., Diagnostic Pathology 2013, 8:92\" in your production\n       http://doi.org/10.1186/1746-1596-8-92\n\n");
	fprintf(stderr, "Usage: tifffastcrop [options] input.tif [output_name]\n\n");
	fprintf(stderr, " Extracts (crops), without loading the full image input.tif into memory, a\nrectangular region from it, and saves it. Output file name is output_name if\ngiven, otherwise a name derived from input.tif. Output file format is guessed\nfrom output_name's extension if possible. Options:\n");
	fprintf(stderr, " -v                verbose monitoring\n");
	fprintf(stderr, " -T                report TIFF errors/warnings on stderr (no dialog boxes)\n");
	fprintf(stderr, " -E x,y,w,l        region to extract/crop (x,y: coordinates of top left corner,\n");
	fprintf(stderr, "   w,l: width and length in pixels)\n");
        fprintf(stderr, " -j[#]             output JPEG file (with quality #, 0-100, default 75)\n");
#ifdef HAVE_PNG
        fprintf(stderr, " -p[#]             output PNG file (with quality #, 0-9, default 6)\n");
#endif
	fprintf(stderr, " -c none[:opts]    output TIFF file with no compression \n");
	fprintf(stderr, " -c x[:opts]       output TIFF compressed with encoding x (jpeg, lzw, zip, ...)\n");
	fprintf(stderr, "When output file format can't be guessed from the output filename extension, it is TIFF with same compression as input.\n\n");
	fprintf(stderr, "JPEG-compressed TIFF options:\n");
	fprintf(stderr, " #   set compression quality level (0-100, default 75)\n");
/*	fprintf(stderr, " r  output color image as RGB rather than YCbCr\n");*/
	fprintf(stderr, "LZW, Deflate (ZIP) and LZMA2 options:\n");
	fprintf(stderr, " #   set predictor value\n");
	fprintf(stderr, " p#  set compression level (preset)\n");
	fprintf(stderr, "For example, -c lzw:2 to get LZW-encoded data with horizontal differencing,\n");
	fprintf(stderr, "-c zip:3:p9 for Deflate encoding with maximum compression level and floating\n");
	fprintf(stderr, "point predictor, -c jpeg:r:50 for JPEG-encoded RGB data at quality 50%%.\n");
}


static int processZIPOptions(char* cp)
{
	if ( (cp = strchr(cp, ':')) ) {
		do {
			cp++;
			if ((*cp) >= '0' && (*cp) <= '9')
				defpredictor = atoi(cp);
			else if (*cp == 'p')
				defpreset = atoi(++cp);
			else
				return 0;
		} while( (cp = strchr(cp, ':')) );
	}
	return 1;
}


static int processG3Options(char* cp)
{
	if( (cp = strchr(cp, ':')) ) {
		if (defg3opts == (uint32) -1)
			defg3opts = 0;
		do {
			cp++;
			if (strncasecmp(cp, "1d", 2) != 0)
				defg3opts &= ~GROUP3OPT_2DENCODING;
			else if (strncasecmp(cp, "2d", 2) != 0)
				defg3opts |= GROUP3OPT_2DENCODING;
			else if (strncasecmp(cp, "fill", 4) != 0)
				defg3opts |= GROUP3OPT_FILLBITS;
			else
				return 0;
		} while( (cp = strchr(cp, ':')) );
	}
	return 1;
}


static int processCompressOptions(char* opt)
{
	if (strncasecmp(opt, "none", 4) == 0) {
		/*char * cp = strchr(opt, ':');*/

		defcompression = COMPRESSION_NONE;
		/*while (cp) {
			if (cp[1] == 'r' )
				defphotometric = PHOTOMETRIC_RGB;
			else if (cp[1] == 'y' )
				defphotometric = PHOTOMETRIC_YCBCR;
			else
				return 0;

			cp = strchr(cp+1,':');
		}*/
	}
	else if (strcasecmp(opt, "packbits") == 0)
		defcompression = COMPRESSION_PACKBITS;
	else if (strncasecmp(opt, "jpeg", 4) == 0) {
		char* cp = strchr(opt, ':');

		defcompression = COMPRESSION_JPEG;
		while( cp ) {
			if (cp[1] >= '0' && cp[1] <= '9') {
				unsigned long u;
				u = strtoul(cp+1, NULL, 10);
				if (u == 0 || u > 100)
					return 0;
				jpeg_quality = (int) u;
			}
			/*else if (cp[1] == 'r' )
				jpegcolormode = JPEGCOLORMODE_RAW;*/
			else
				return 0;

			cp = strchr(cp+1,':');
		}
	} else if (strncasecmp(opt, "g3", 2) == 0) {
		if (!processG3Options(opt))
			return 0;
		defcompression = COMPRESSION_CCITTFAX3;
	} else if (strcasecmp(opt, "g4") == 0) {
		defcompression = COMPRESSION_CCITTFAX4;
	} else if (strncasecmp(opt, "lzw", 3) == 0) {
		char* cp = strchr(opt, ':');
		if (cp)
			defpredictor = atoi(cp+1);
		defcompression = COMPRESSION_LZW;
	} else if (strncasecmp(opt, "zip", 3) == 0) {
		if (!processZIPOptions(opt))
			return 0;
		defcompression = COMPRESSION_ADOBE_DEFLATE;
	}/* else if (strncasecmp(opt, "lzma", 4) == 0) {
		if (!processZIPOptions(opt) )
			return 0;
		defcompression = COMPRESSION_LZMA;
	}*/ else if (strncasecmp(opt, "jbig", 4) == 0) {
		defcompression = COMPRESSION_JBIG;
	} else if (strncasecmp(opt, "sgilog", 6) == 0) {
		defcompression = COMPRESSION_SGILOG;
	} else
		return (0);

	return (1);
}


static int processExtractGeometryOptions(char* cp)
{
	while (*cp == ' ')
		cp++;
	if (sscanf(cp, UINT32_FORMAT "," UINT32_FORMAT ","
	    UINT32_FORMAT "," UINT32_FORMAT,
	    &requestedxmin, &requestedymin, &requestedwidth, 
	    &requestedlength) != 4)
		return 0;
	return 1;
}


static void stderrErrorHandler(const char* module, const char* fmt, va_list ap)
{
	if (module != NULL)
		fprintf(stderr, "%s: ", module);
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, ".\n");
}


static void stderrWarningHandler(const char* module, const char* fmt, va_list ap)
{
	if (!verbose)
		return;
	if (module != NULL)
		fprintf(stderr, "%s: ", module);
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, ".\n");
}


int main(int argc, char * argv[])
{
	int arg = 1;
	int seen_extract_geometry_on_the_command_line = 0;

	while (arg < argc && argv[arg][0] == '-') {

		if (argv[arg][1] == 'v')
			verbose = 1;
		else if (argv[arg][1] == 'T') {
			TIFFSetErrorHandler(stderrErrorHandler);
			TIFFSetWarningHandler(stderrWarningHandler);
			}
		else if (argv[arg][1] == 'E') {
			if (arg+1 >= argc ||
			    !processExtractGeometryOptions(argv[arg+1])) {
				fprintf(stderr, "Syntax error in the "
					"specification of region to "
					"extract (option -E).\n");
				usage();
				return EXIT_SYNTAX_ERROR;
			}
			seen_extract_geometry_on_the_command_line = 1;
			arg++;
		} else if (argv[arg][1] == 'c') {
			output_format = OUTPUT_FORMAT_TIFF;
			if (arg+1 >= argc ||
			    !processCompressOptions(argv[arg+1])) {
				usage();
				return EXIT_SYNTAX_ERROR;
			}
			arg++;
		} else if (argv[arg][1] == 'j') {
			output_format = OUTPUT_FORMAT_JPEG;

			if (argv[arg][2] != 0) {
				unsigned long u;

				u= strtoul(&(argv[arg][2]), NULL, 10);
				if (u == 0 || u > 100) {
					fprintf(stderr, "Expected optional non-null integer percent number after -j, got \"%s\"\n",
					    &(argv[arg][2]));
					usage();
					return EXIT_SYNTAX_ERROR;
				}
				jpeg_quality = (int) u;
			}
#ifdef HAVE_PNG
		} else if (argv[arg][1] == 'p') {
			output_format = OUTPUT_FORMAT_PNG;

			if (argv[arg][2] != 0) {
				unsigned long u;

				u= strtoul(&(argv[arg][2]), NULL, 10);
				if (u > 9) {
					fprintf(stderr, "Expected optional non-null integer percent number after -j, got \"%s\"\n",
					    &(argv[arg][2]));
					usage();
					return EXIT_SYNTAX_ERROR;
				}
				png_quality = (int) u;
			}
#endif
		} else {
			fprintf(stderr, "Unknown option \"%s\"\n", argv[arg]);
			usage();
			return EXIT_SYNTAX_ERROR;
		}

	arg++;
	}

	if (argc > 1 && !seen_extract_geometry_on_the_command_line) {
		fprintf(stderr, "The extract's position and size must be specified on the command line as argument to the '-E' option. Aborting.\n");
		return EXIT_GEOMETRY_ERROR;
	}

	if (output_format < 0) {
		output_format = OUTPUT_FORMAT_TIFF;
		if (argc >= arg+2) { /* Try to guess from output file name */
			const char * suffix = searchSuffix(argv[arg+1]);
			if (strcasecmp(suffix, "png") == 0)
				output_format = OUTPUT_FORMAT_PNG;
			else if (strcasecmp(suffix, "jpeg") == 0 ||
				 strcasecmp(suffix, "jpg") == 0)
				output_format = OUTPUT_FORMAT_JPEG;
		}
	}
	if (verbose)
		fprintf(stderr, "Output file will have format %s.\n",
			OUTPUT_SUFFIX[output_format]);

	if (argc >= arg+2) {
		return makeExtractFromTIFFFile(argv[arg], argv[arg+1]);
	} else if (argc >= arg+1) {
		return makeExtractFromTIFFFile(argv[arg], NULL);
	} else {
		usage();
		return EXIT_SYNTAX_ERROR;
	}

}
