// Copyright (C) 2016 The Qt Company Ltd.
// Copyright (C) 2016 Intel Corporation.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only

#include "tst_bench_qhash.h"

#include <QFile>
#include <QHash>
#include <QString>
#include <QStringList>
#include <QUuid>
#include <QTest>

static constexpr quint64 RandomSeed32 = 1045982819;
static constexpr quint64 RandomSeed64 = qHashMulti(0, RandomSeed32, RandomSeed32);

class tst_QHash : public QObject
{
    Q_OBJECT

private slots:
    void initTestCase();
    void qhash_current_data() { data(); }
    void qhash_current() { qhash_template<QString>(); }
    void qhash_qt50_data() { data(); }
    void qhash_qt50() { qhash_template<Qt50String>(); }
    void qhash_qt4_data() { data(); }
    void qhash_qt4() { qhash_template<Qt4String>(); }
    void qhash_javaString_data() { data(); }
    void qhash_javaString() { qhash_template<JavaString>(); }

    void hashing_current_data() { data(); }
    void hashing_current() { hashing_template<QString>(); }
    void hashing_qbytearray_data() { data(); }
    void hashing_qbytearray() { hashing_template<QByteArray>(); }
    void hashing_qt50_data() { data(); }
    void hashing_qt50()  { hashing_template<Qt50String>(); }
    void hashing_qt4_data() { data(); }
    void hashing_qt4() { hashing_template<Qt4String>(); }
    void hashing_javaString_data() { data(); }
    void hashing_javaString() { hashing_template<JavaString>(); }

    void hashing_nonzero_current_data() { data(); }
    void hashing_nonzero_current() { hashing_nonzero_template<QString>(); }
    void hashing_nonzero_qbytearray_data() { data(); }
    void hashing_nonzero_qbytearray() { hashing_nonzero_template<QByteArray>(); }
    void hashing_nonzero_qlatin1string_data() { data(); }
    void hashing_nonzero_qlatin1string() { hashing_nonzero_template<OwningLatin1String>(); }

private:
    void data();
    template <typename String> void qhash_template();
    template <typename String, size_t Seed = 0> void hashing_template();
    template <typename String> void hashing_nonzero_template()
    { hashing_template<String, size_t(RandomSeed64)>(); }

    QStringList smallFilePaths;
    QStringList uuids;
    QStringList dict;
    QStringList numbers;
    QStringList longstrings;
};

///////////////////// QHash /////////////////////

#include <QDebug>

void tst_QHash::initTestCase()
{
    QHashSeed::setDeterministicGlobalSeed();

    // small list of strings (that happen to look like file paths produced long
    // ago by cd ../.. && find . -print, but that's irrelevant).
    QFile smallPathsData(QFINDTESTDATA("paths_small_data.txt"));
    QVERIFY(smallPathsData.open(QIODevice::ReadOnly));
    smallFilePaths = QString::fromLatin1(smallPathsData.readAll()).split(QLatin1Char('\n'));
    QVERIFY(!smallFilePaths.isEmpty());

    // list of UUIDs
    // guaranteed to be completely random, generated by http://xkcd.com/221/
    QUuid ns = QUuid("{f43d2ef3-2fe9-4563-a6f5-5a0100c2d699}");
    uuids.reserve(smallFilePaths.size());
    longstrings.reserve(smallFilePaths.size());

    for (const QString &path : std::as_const(smallFilePaths))
        uuids.append(QUuid::createUuidV5(ns, path).toString());
    for (qsizetype i = 0; i < uuids.size(); ++i)
        longstrings.append(uuids.at(i).repeated(8));

    // lots of strings with alphabetical characters, vaguely reminiscent of
    // a dictionary.
    //
    // this programmatically generates a series like:
    //  AAAAAA
    //  AAAAAB
    //  AAAAAC
    //  ...
    //  AAAAAZ
    //  AAAABZ
    //  ...
    //  AAAAZZ
    //  AAABZZ
    QByteArray id("AAAAAAA");

    if (dict.isEmpty()) {
        for (int i = id.size() - 1; i > 0;) {
            dict.append(id);
            char c = id.at(i);
            id[i] = ++c;

            if (c == 'Z') {
                // wrap to next digit
                i--;
                id[i] = 'A';
            }
        }
    }

    // string versions of numbers.
    for (int i = 5000000; i < 5005001; ++i)
        numbers.append(QString::number(i));
}

void tst_QHash::data()
{
    QTest::addColumn<QStringList>("items");
    QTest::newRow("paths-small") << smallFilePaths;
    QTest::newRow("uuids-list") << uuids;
    QTest::newRow("longstrings-list") << longstrings;
    QTest::newRow("dictionary") << dict;
    QTest::newRow("numbers") << numbers;
}

template <typename Str> void tst_QHash::qhash_template()
{
    QFETCH(const QStringList, items);
    QHash<Str, int> hash;

    QList<Str> realitems;
    realitems.reserve(items.size());
    for (const QString &s : items)
        realitems.append(s);

    QBENCHMARK {
        for (int i = 0, n = realitems.size(); i != n; ++i) {
            hash[realitems.at(i)] = i;
        }
    }
}

template <typename Str, size_t Seed> void tst_QHash::hashing_template()
{
    // just the hashing function
    QFETCH(const QStringList, items);

    QList<Str> realitems;
    realitems.reserve(items.size());
    for (const QString &s : items) {
        if constexpr (std::is_same_v<QString::value_type, typename Str::value_type>) {
            realitems.append(s);
        } else if constexpr (sizeof(typename Str::value_type) == 1) {
            realitems.append(Str(s.toLatin1()));
        }
    }

    QBENCHMARK {
        for (int i = 0, n = realitems.size(); i != n; ++i) {
            volatile size_t h = qHash(realitems.at(i), Seed);
            (void)h;
#ifdef Q_CC_GNU
            // "use" h
            asm ("" : "+r" (h));
#endif
        }
    }
}

QTEST_MAIN(tst_QHash)

#include "tst_bench_qhash.moc"
