/*******************************************************************************
* Copyright 2004-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : DJACOBI RCI example
*
********************************************************************************
*/
/*
  The program computes the Jacobi matrix of the function on the basis of RCI
  using the central difference.
*/

#include <stdlib.h>
#include <stdio.h>
#include "mkl_service.h"

#include "mkl_rci.h"
#include "mkl_types.h"

/* users objective function */
inline void extended_powell (MKL_INT *, MKL_INT *, double *, double *);

int main (void)
{
  /* n - number of function variables
     m - dimension of function value */
  MKL_INT n = 4, m = 4;
  /* jacobi matrix */
  double *a;
  /* solution vector. contains values x for f(x)
     temporary arrays f1 & f2 that contain f1 = f(x+eps) | f2 = f(x-eps) */
  double *x, *f1, *f2;
  /* precisions for jacobi_matrix calculation */
  double eps = 0.000001;
  /* jacobi-matrix solver handle */
  _JACOBIMATRIX_HANDLE_t handle;
  /* controls of rci cycle */
  MKL_INT successful, rci_request, i;

  if ((a = (double *) mkl_malloc(sizeof (double) * n * m, 64)) == NULL)
    {
      printf ("\n#fail: jacobi matrix allocation failed\n");
    }
  if ((x = (double *) mkl_malloc(sizeof (double) * n, 64)) == NULL)
    {
      printf ("\n#fail: vector X allocation failed\n");
      mkl_free (a);
      return 1;
    }
  if ((f1 = (double *) mkl_malloc(sizeof (double) * n, 64)) == NULL)
    {
      printf ("\n#fail: vector F1 allocation failed\n");
      mkl_free (x);
      mkl_free (a);
      return 1;
    }
  if ((f2 = (double *) mkl_malloc(sizeof (double) * n, 64)) == NULL)
    {
      printf ("\n#fail: vector F2 allocation failed\n");
      mkl_free (f1);
      mkl_free (x);
      mkl_free (a);
      return 1;
    }

  /* set the x values */
  for (i = 0; i < n; i++)
    x[i] = 10.0;
  /* initialize solver (allocate memory, set initial values) */
  if (djacobi_init (&handle, &n, &m, x, a, &eps) != TR_SUCCESS)
    {
      /* if function does not complete successfully then print error message */
      printf ("\n#fail: error in djacobi_init\n");
      fflush (0);
      MKL_Free_Buffers ();
      return 1;
    }
  /* set initial rci cycle variables */
  rci_request = 0;
  successful = 0;
  /* rci cycle */
  while (successful == 0)
    {
      /* call solver */
      if (djacobi_solve (&handle, f1, f2, &rci_request) != TR_SUCCESS)
        {
      /* if function does not complete successfully then print error message */
          printf ("\n#fail: error in djacobi_solve\n");
          fflush (0);
          MKL_Free_Buffers ();
          return 1;
        }
      if (rci_request == 1)
        {
          /* calculate function value f1 = f(x+eps) */
          extended_powell (&m, &n, x, f1);
        }
      else if (rci_request == 2)
        {
          /* calculate function value f1 = f(x-eps) */
          extended_powell (&m, &n, x, f2);
        }
      else if (rci_request == 0)
        /* exit rci cycle */
        successful = 1;
    } /* rci cycle */
  /* free handle memory */
  if (djacobi_delete (&handle) != TR_SUCCESS)
    {
      /* if function does not complete successfully then print error message */
      printf ("\n#fail: error in djacobi_delete\n");
      fflush (0);
      MKL_Free_Buffers ();
      return 1;
    }
  MKL_Free_Buffers ();
  mkl_free (f2);
  mkl_free (f1);
  mkl_free (x);
  mkl_free (a);
  printf ("#pass\n");
  fflush (0);
  return 0;
}

/*
    routine for extended Powell function calculation
    m in : dimension of function value
    n in : number of function variables
    x in : vector for function calculation
    f out: function value f(x)
*/
void extended_powell (MKL_INT * m, MKL_INT * n, double *x, double *f)
{
  MKL_INT i;

  if (*m != * n) {
    printf("|     for this example's function m must be equal to n\n");
    exit(-1);
  }

  for (i = 0; i < (*m) / 4; i++)
  {
      f[4 * i] = x[4 * i] + 10.0 * x[4 * i + 1];
      f[4 * i + 1] = 2.2360679774 * (x[4 * i + 2] - x[4 * i + 3]);
      f[4 * i + 2] = (x[4 * i + 1] - 2.0 * x[4 * i + 2]) *
                     (x[4 * i + 1] - 2.0 * x[4 * i + 2]);
      f[4 * i + 3] = 3.1622776601 * (x[4 * i] - x[4 * i + 3]) *
                                    (x[4 * i] - x[4 * i + 3]);
  }
  return;
}
