/**
 * (C) Copyright IBM Corp. 2007
 *
 * THIS FILE IS PROVIDED UNDER THE TERMS OF THE COMMON PUBLIC LICENSE 
 * ("AGREEMENT"). ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS FILE 
 * CONSTITUTES RECIPIENTS ACCEPTANCE OF THE AGREEMENT.
 *
 * You can obtain a current copy of the Common Public License from
 * http://www.opensource.org/licenses/cpl1.0.php
 *
 * @author: Endre Bak, IBM, ebak@de.ibm.com
 * 
 * Change History
 * Flag       Date        Prog         Description
 *------------------------------------------------------------------------------- 
 * 1804402    2007-09-28  ebak      IPv6 ready SLP
 * 1892103    2008-02-12  ebak      SLP improvements
 */


package org.sblim.slp.internal.msg;

import java.util.Iterator;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import org.sblim.slp.ServiceLocationException;

/**
 * RequestMessage
 *
 */
public abstract class RequestMessage extends SLPMessage {
	
	private SortedSet iPrevResponderSet;
	private List iScopeList;
	
	
	/**
	 * Ctor.
	 * @param pFunctionID
	 * @param pPrevResponderSet - set of address strings
	 * @param pScopeList - list of scope strings
	 */
	public RequestMessage(int pFunctionID, SortedSet pPrevResponderSet, List pScopeList) {
		super(pFunctionID);
		init(pPrevResponderSet, pScopeList);
	}
	
	/**
	 * Ctor.
	 * @param pFunctionID
	 * @param pLangTag
	 * @param pPrevResponderSet - set of address strings
	 * @param pScopeList - list of scope strings
	 */
	public RequestMessage(
		int pFunctionID, String pLangTag, SortedSet pPrevResponderSet, List pScopeList
	) {
		super(pFunctionID, pLangTag);
		init(pPrevResponderSet, pScopeList);
	}
	
	/**
	 * Ctor.
	 * @param pHeader
	 * @param pPrevResponderSet - set of address strings
	 * @param pScopeList - list of scope strings
	 */
	public RequestMessage(
		MsgHeader pHeader, SortedSet pPrevResponderSet, List pScopeList
	) {
		super(pHeader);
		init(pPrevResponderSet, pScopeList);
	}
	
	/**
	 * getPrevResponderSet
	 * @return SortedSet
	 */
	public SortedSet getPrevResponderSet() { return iPrevResponderSet; }

	/**
	 * getPrevRespondersItr
	 * @return Iterator
	 */
	public Iterator getPrevRespondersItr() {
		return iPrevResponderSet==null ? null : iPrevResponderSet.iterator();
	}
	
	/**
	 * updatePrevResponders
	 * @param pResponder
	 * @return boolean
	 */
	public boolean updatePrevResponders(String pResponder) {
		if (iPrevResponderSet == null) iPrevResponderSet = new TreeSet();
		return iPrevResponderSet.add(pResponder);
	}
	
	/**
	 * getScopeList
	 * @return List of scope strings
	 */
	public List getScopeList() { return iScopeList; }
	
	/**
	 * isAllowedResponseType
	 * @param pRspMsg
	 * @return boolean
	 */
	public boolean isAllowedResponseType(SLPMessage pRspMsg) {
		if (pRspMsg == null) return false;
		int id = pRspMsg.getFunctionID();
		int[] rspIDs = getAllowedResponseIDs();
		if (rspIDs == null) return true;
		for(int i = 0; i < rspIDs.length; i++)
			if (id == rspIDs[i]) return true;
		return false;
	}
	
	
	/**
	 * serializeWithoutResponders
	 * @param pSetMulticastFlag
	 * @param pDatagramLimited
	 * @param pKeepXID
	 * @return byte[]
	 * @throws ServiceLocationException
	 */
	public byte[] serializeWithoutResponders(
		boolean pSetMulticastFlag, boolean pDatagramLimited, boolean pKeepXID
	) throws ServiceLocationException {
		
		return serialize(pSetMulticastFlag, pDatagramLimited, pKeepXID, new SkipResponders());
	}
	
	protected boolean serializeBody(SLPOutputStream pOutStr, SerializeOption pSkipResponders) throws ServiceLocationException {
		if (
			!pOutStr.writeStringList(
				pSkipResponders == null ? getPrevRespondersItr() : null
			)
		)
			throw new ServiceLocationException(
				ServiceLocationException.PREVIOUS_RESPONDER_OVERFLOW,
				"Previous responder list has been overflown!"
			);
		return serializeRequestBody(pOutStr);
	}

	protected abstract boolean serializeRequestBody(SLPOutputStream pOutStr)
	throws ServiceLocationException;
	
	protected abstract int[] getAllowedResponseIDs();
	
	private void init(SortedSet pPrevResponderSet, List pScopeList) {
		iPrevResponderSet = pPrevResponderSet; iScopeList = pScopeList;
	}
	
	class SkipResponders extends SerializeOption {
		/*
		 * non-null instance indicates that the PreviousResponderList serialization have to
		 * be skipped
		 */
	}
	
}