#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <locale.h>
#include <gnome.h>
#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include "gimlet.h"
#include <signal.h>
#include <strings.h>
#include <X11/Xatom.h>
#include "iiimcf.h"

static const char *JDSONLYENV = "SESSIONTYPE";

gboolean 
is_on_CDE ()
{
  char *value;

  value = getenv (JDSONLYENV);
  if (value == NULL)
    return TRUE;

  return FALSE;
}

gboolean
is_panel_running ()
{
  Display *xdisplay;
  Atom switcher_atom;

  xdisplay = gdk_x11_get_default_xdisplay ();
  switcher_atom = XInternAtom (xdisplay, SWITCHER, FALSE);
  
  return None != XGetSelectionOwner(xdisplay, switcher_atom);
}

gboolean
is_systray_supported ()
{
  Display *xdisplay;
  Atom selection_atom;
  Window manager_window;
  GdkDisplay *display;
  
  gboolean retval = False;
  int i, screen_count;
  char buffer[256];

  display = gdk_display_get_default ();
  xdisplay = gdk_x11_display_get_xdisplay (display);

  screen_count = gdk_display_get_n_screens (display);

  XGrabServer (xdisplay);

  for (i=0; i<screen_count; i++) {
    g_snprintf (buffer, sizeof (buffer),
                "_NET_SYSTEM_TRAY_S%d", i);
                
    selection_atom = XInternAtom (xdisplay, buffer, FALSE);
    manager_window = XGetSelectionOwner (xdisplay, selection_atom);

    if (manager_window != None) {
      retval = True;
      break;
    }
  }

  XUngrabServer (xdisplay);
  XFlush (xdisplay);
  
  return retval;
}

GdkPixbuf *
init_pixbuf (const gchar *name)
{
  GdkPixbuf *pixbuf;
  GError    *error = NULL;
  char      *path = NULL;

  path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP,
				    name, FALSE, NULL);
  if (!path)
    path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_PIXMAP,
				      name, FALSE, NULL);
  if (!path)
    return NULL;

  pixbuf = gdk_pixbuf_new_from_file (path, &error);

  if (error)
    {
      g_warning ("Cannot load '%s': %s", path, error->message);
      g_error_free (error);
    }
  g_free (path);
  return pixbuf;
}

gchar *
get_language_short_name (gchar *lc_name)
{
  if (g_str_has_prefix (lc_name, "x-kbl-"))
    {
      gchar *lcn = g_strndup (lc_name + 6, 2);
      if (strlen (lcn) > 0)
	{
	  lcn[0] = g_ascii_toupper (lcn[0]);
	}
      return lcn;
    }

  if (strlen (lc_name) == 5 &&
      lc_name[0] == 'z' &&
      lc_name[1] == 'h')
    {
      if (strcmp (lc_name, "zh_CN") == 0)
        return (g_strdup ("SC"));
      else
        return (g_strdup ("TC"));
    }


  gchar *ll = g_strndup (lc_name, 2);
  ll[0] = g_ascii_toupper (ll[0]);
  return ll;
}

static IIIMCF_handle iiim;
static gboolean 
cf_init (Display *display) 
{
  IIIMCF_attr attr = NULL;
  IIIMF_status st;

  st = iiimcf_initialize (IIIMCF_ATTR_NULL);
  if (st != IIIMF_STATUS_SUCCESS) {
    return FALSE;
  }

  st = iiimcf_create_attr (&attr);
  if (st != IIIMF_STATUS_SUCCESS) {
    return FALSE;
  }

  st = iiimcf_attr_put_string_value (attr, IIIMCF_ATTR_CLIENT_TYPE,
				     "GTK IIIMCF Module");
  if (st != IIIMF_STATUS_SUCCESS) {
    return FALSE;
  }

  st = iiimcf_attr_put_string_value (attr, IIIMCF_ATTR_CLIENT_X_DISPLAY_NAME,
				     XDisplayString (display));
  if (st != IIIMF_STATUS_SUCCESS) {
    return FALSE;
  }

  {
    Atom iiimd;
    Atom client_group;
    Window iiimx;
    char path[PATH_MAX];
    char client_group_id[PATH_MAX];

    iiimd = XInternAtom (display, "IIIM_SERVER", True);
    if (iiimd != None) {
      iiimx = XGetSelectionOwner (display, iiimd);
      if (iiimx != None) {
	Atom type;
	int format;
	unsigned long length;
	unsigned long nitem;
	unsigned char *data;
	data = NULL;
	XGetWindowProperty (display, iiimx, iiimd,
			    0, INT_MAX, False, XA_STRING,
			    &type, &format, &nitem, &length, &data);
	if (data != NULL && !strncmp ("uds:", (char *)data, 4)) {
	  strncpy (path, (char *)data + 4, sizeof (path));
	  path[(sizeof (path)) - 1] = '\0';
	  XFree (data);
	  st = iiimcf_attr_put_string_value (attr,
					     IIIMCF_ATTR_SERVER_ADDRESS,
					     path);
	  if (st != IIIMF_STATUS_SUCCESS) {
	    return FALSE;
	  }
	  st = iiimcf_attr_put_string_value (attr,
					     IIIMCF_ATTR_SERVER_SERVICE,
					     "");
	  if (st != IIIMF_STATUS_SUCCESS) {
	    return FALSE;
	  }
	}
      }
    }
    client_group = XInternAtom (display, "IIIM_CLIENT_GROUP", True);
    if (client_group != NULL) {
      iiimx = XGetSelectionOwner (display, client_group);
      if (iiimx != None) {
	Atom type;
	int format;
	unsigned long length;
	unsigned long nitem;
	unsigned char *data;
	data = NULL;
	XGetWindowProperty (display, iiimx, client_group,
			    0, INT_MAX, False, XA_STRING,
			    &type, &format, &nitem, &length, &data);
	if (data != NULL) {
	  strncpy (client_group_id, (char *)data, sizeof (client_group_id));
	  path[(sizeof (client_group_id)) - 1] = '\0';
	  XFree (data);
	  st = iiimcf_attr_put_string_value (attr,
					     IIIMCF_ATTR_CLIENT_GROUP,
					     client_group_id);
	  if (st != IIIMF_STATUS_SUCCESS) {
	    return FALSE;
	  }
	}
      }
    }
  }

  st = iiimcf_create_handle (attr, &iiim);
  if (st != IIIMF_STATUS_SUCCESS) {
    return FALSE;
  }

  st = iiimcf_destroy_attr (attr);
  if (st != IIIMF_STATUS_SUCCESS) {
    return FALSE;
  }

  return TRUE;
}

static void
cf_fini ()
{
    if (iiim)
        iiimcf_destroy_handle (iiim);

    iiimcf_finalize ();
}

gchar *
get_initial_language ()
{
  gchar *remember_last_le = gconf_client_get_string (gconf_client_get_default (),
                                                     KEY_REMEMBER_LAST_LE, NULL);

  if (remember_last_le && 0 == strcmp (remember_last_le, V_TRUE)) {
      gchar *last_le = gconf_client_get_string (gconf_client_get_default (),
                                                KEY_LANG_LAST_SELECTION, NULL);
      if (last_le) {
          gchar **elms = NULL;
          elms = g_strsplit (last_le, ":", -1);
          gchar *lang = get_language_short_name (elms[0]);
          g_free (remember_last_le);
          g_free (last_le);
          return lang;
      }
  }

  /* initialize language by guessing from locale */
  gchar *lang = NULL;
  gchar *locale = setlocale (LC_CTYPE, NULL);

  if (cf_init (gdk_x11_get_default_xdisplay ())) {
      const char * ll;
      IIIMCF_language cflang = iiimcf_get_input_language_for_locale (iiim, locale, False);
      IIIMF_status st = cflang? iiimcf_get_language_id (cflang, &ll): IIIMF_STATUS_FAIL;
      if (IIIMF_STATUS_SUCCESS == st)
          lang = get_language_short_name (ll);
  }
  cf_fini ();

  if (!lang)
    lang = g_strdup ("en");
  
  return lang;
}

IMStatusPlacement
status_placement_num_id (char *str)
{
  if (str && (strcmp (str, V_NONE) == 0))
    return NONE;
  if (str && (strcmp (str, V_PANEL) == 0))
    return ON_DESKTOP_PANEL;
  else if (str && (strcmp (str, V_ATTACH_FRAME) == 0))
    return ATTACH_TO_APP_FRAME;
  else if (str && (strcmp (str, V_DESKTOP_DEFAULT) == 0))
    return is_on_CDE()?ATTACH_TO_APP_FRAME:ON_DESKTOP_PANEL;
}

void
ignore_child_signal()
{
  struct sigaction sa;

  sa.sa_handler = SIG_IGN;
  sa.sa_flags = SA_NOCLDWAIT;
  sigemptyset (&sa.sa_mask);
  sigaction (SIGCHLD, &sa, NULL);
}

void
start_iiimx_settings_init ()
{
  const char *pathname = "/usr/lib/iiim/iiimx-settings-init";

  if (access (pathname, X_OK) == 0) {
    pid_t pid;
    pid = fork();
    if (pid == 0) {
      char *argv[2];
      argv[0] = "iiimx-settings-init";
      argv[1] = NULL;
      execv (pathname, argv);
      _exit(1);
    }
  }
}

void
occupy_all_workspaces_on_CDE (GtkWidget *gtkwin)
{
  Display *xdisplay = GDK_WINDOW_XDISPLAY (gtkwin->window);
  Window xwin = GDK_WINDOW_XID (gtkwin->window);

  guint32 data[3]={0,1,1};
  Atom DT_WORKSPACE_HINTS =  XInternAtom (xdisplay, "_DT_WORKSPACE_HINTS", False);
  XChangeProperty (xdisplay, xwin, DT_WORKSPACE_HINTS, DT_WORKSPACE_HINTS, 32, PropModeReplace, data, 3);
}

void
sm_disable_on_CDE (GtkWidget *gtkwin)
{
  GdkWindow *leader_window = NULL;
  GdkDisplay *display = gtk_widget_get_display (gtkwin);
  leader_window = gdk_display_get_default_group (display);
  
  XSetCommand(GDK_WINDOW_XDISPLAY(gtkwin->window),
              GDK_WINDOW_XID(leader_window),
              NULL, 0);
}

void
sm_disable_on_gnome ()
{
  GnomeClient *client = gnome_master_client ();
  gnome_client_disconnect (client);
}

/* get the motif window manager decoration size in CDE */
void
get_mwm_decoration_size (GtkWidget *widget, int *px, int *py, int *pw, int *ph)
{
  Window parent, root;
  Window *children;
  int nchildren;

  int x, y, w, h;
  x = y = w = h = 0;
 
  Display *disp = GDK_WINDOW_XDISPLAY(widget->window);
  Window   win  = GDK_WINDOW_XID (widget->window);

  XQueryTree(disp, win, &root, &parent, &children, &nchildren);

  if (root != parent) {
    XWindowAttributes winattr;
    XGetWindowAttributes(disp, parent, &winattr);
    x = winattr.x;
    y = winattr.y;
    w = winattr.width;
    h = winattr.height;
  }
  
  if (px) *px = x;
  if (py) *py = y;
  if (pw) *pw = w;
  if (ph) *ph = h;
}
