#include <config.h>

#include <gnome.h>
#include <gconf/gconf-client.h>

#include "gimlet.h"

#define MAX_LANG_ELEM_LEN 256
#define SEPARATOR ";"
#define UI_SEPARATOR " - "

extern GdkPixbuf *kbd_pixbuf;

static void
_g_slist_free_data (gpointer data, gpointer user_data)
{
  g_free (data);
}

static gchar *
get_default_lename_from_save_list (gchar *iiim_lang)
{
  GSList *lang_list = gconf_client_get_list (gconf_client_get_default (),
					     KEY_LANG_TO_SAVE_LIST,
					     GCONF_VALUE_STRING, NULL);
  gchar *lang_part;
  gchar *ret = NULL;

  if (g_str_has_prefix (iiim_lang, "x-kbl-") &&
      g_str_has_suffix (iiim_lang, "custom"))
    {
      return g_strdup ("unitle");
    }

  lang_part = g_strconcat (iiim_lang, SEPARATOR, NULL);

  GSList *lp;
  for (lp = lang_list; lp; lp = g_slist_next (lp))
    {
      if (g_str_has_prefix ((gchar *)lp->data, lang_part))
	{
	  gchar **elm = g_strsplit ((gchar *)lp->data, SEPARATOR, -1);

	  if (elm[1] != NULL)
	      ret = g_strdup (elm[1]);

	  g_strfreev (elm);

          if (ret)
            break;
	}
    }

  if (lang_list != NULL) {
    g_slist_foreach (lang_list, _g_slist_free_data, NULL);
    g_slist_free (lang_list);
  }

  g_free (lang_part);

  return ret;
}

static void 
ensure_default_engine (gchar *lang_id, 
                       gchar *engine_id)
{
  GConfClient *client = gconf_client_get_default ();
  GSList *lang_list = gconf_client_get_list (client,
					     KEY_LANG_TO_SAVE_LIST,
					     GCONF_VALUE_STRING, NULL);
  gchar *lang_part = g_strconcat (lang_id, SEPARATOR, NULL);
  GSList *update_list = NULL;
  gboolean need_update = FALSE;

  GSList *lp;
  for (lp = lang_list; lp; lp = g_slist_next (lp))
    {
      if (g_str_has_prefix ((gchar *)lp->data, lang_part))
	{
	  /* found target language */
	  gchar **elm = g_strsplit ((gchar *)lp->data, SEPARATOR, -1);
	  if (elm[1] != NULL && strcmp (elm[1], engine_id))
	    {
	      /* target engine is not default, so make it default */
	      gchar buf[MAX_LANG_ELEM_LEN];
	      gchar *p = buf;
	      gchar **elmp;
	      p = g_stpcpy (p, lang_part);
	      p = g_stpcpy (p, engine_id);
	      p = g_stpcpy (p, SEPARATOR);
	      for (elmp = elm + 1; *elmp; elmp++)
		{
		  if (!strcmp (*elmp, engine_id) || !strcmp (*elmp, ""))
		    continue;
		  p = g_stpcpy (p, *elmp);
		  p = g_stpcpy (p, SEPARATOR);
		}
	      update_list = g_slist_append (update_list, g_strdup (buf));
	      need_update = TRUE;
	    }
	  else
	    {
	      g_strfreev (elm);
	      break;
	    }
	  g_strfreev (elm);
	}
      else
	{
	  update_list = g_slist_append (update_list, g_strdup(lp->data));
	}
    }

  if (need_update)
    gconf_client_set_list (client, KEY_LANG_TO_SAVE_LIST,
                           GCONF_VALUE_STRING, update_list, NULL);

  if (lang_list != NULL) {
    g_slist_foreach (lang_list, _g_slist_free_data, NULL);
    g_slist_free (lang_list);
  }

  if (update_list != NULL) {
    g_slist_foreach (update_list, _g_slist_free_data, NULL);
    g_slist_free (update_list);      
  }

  g_free (lang_part);
}

static void
lang_menu_item_activate_cb (GtkWidget *item, gpointer user_data)
{
  gchar *lang, *engine, *lang_engine, *delimiter = ":";
  gchar *on_off_policy, *switch_policy, *conversion_mode;

  lang = g_strdup (g_object_get_data (G_OBJECT (item),
				      "iiim-lang-name"));
  engine = g_strdup (g_object_get_data (G_OBJECT (item),
					"iiim-lename"));

  /* if plural engines are available for one language and one engine is selected
   * in use list, then need to specify lename explicitly even if lename is NULL here */
  if (engine == NULL)
    engine = get_default_lename_from_save_list (lang);

  if (engine)
    lang_engine = g_strconcat (lang, delimiter, engine, NULL);
  else
    lang_engine = g_strdup (lang);

  /* update current lang to client */
  gimlet_notify_language_to_client (lang_engine);

  /* remember the last selection */
  gconf_client_set_string (gconf_client_get_default (), 
                           KEY_LANG_LAST_SELECTION, lang_engine, NULL);

  /* update on_off_policy to client */
  on_off_policy = gconf_client_get_string (gconf_client_get_default (),
                                           KEY_ON_OFF_POLICY, NULL);
  if (on_off_policy == NULL || !strcmp (on_off_policy, "Activate") ||
      gimlet->conversion_mode)
    conversion_mode = "on";
  else
    conversion_mode = "off";

  if (engine!= NULL) {
    switch_policy = gconf_client_get_string (gconf_client_get_default (),
                                             KEY_LANG_SWITCH_POLICY, NULL);
    if (switch_policy != NULL && !strcmp (switch_policy, "Desktop"))
      ensure_default_engine (lang, engine);
  }

  /* update conversion mode to client */
  gimlet_notify_conversion_to_client (conversion_mode);

  /* update the panel */
  g_free (gimlet->current_iiim_lang);
  gimlet->current_iiim_lang = g_strdup (lang);
  gimlet_update_lang (gimlet);

  /* cleanup */
  g_free (lang); g_free (engine); g_free (lang_engine);
  g_free (on_off_policy); g_free (switch_policy);
}

static void
extract_lang_props (gchar *langstr, gchar **lang_id, gchar **lang_ui,
		    gchar **lang_engine, gchar **lang_engine_ui)
{
  gchar *p, *np;
  gint len;

  gchar **elems;
  gchar **ui_elems;

  elems = g_strsplit ((const gchar *)langstr, SEPARATOR, 4);
  *lang_id = g_strdup (elems[0]);

  if (strcmp (elems[2], ""))
    {
      *lang_engine = g_strdup (elems[2]);

      ui_elems = g_strsplit ((const gchar *)(elems[1]), UI_SEPARATOR, 2);
      *lang_ui = g_strdup (ui_elems[0]);
      *lang_engine_ui = g_strdup (ui_elems[1]);
      g_strfreev (ui_elems);
    }
  else
    {
      *lang_ui = g_strdup (elems[1]);
      *lang_engine = NULL;
      *lang_engine_ui = NULL;
    }
  
  g_strfreev (elems);
}

static void
item_add_data_and_signal (GtkWidget *menuitem, gchar *lang_id, gchar *ui,
				 gchar *engine_id, GimletWindow *gimlet)
{
  g_object_set_data (G_OBJECT (menuitem), "iiim-lang-name", g_strdup (lang_id));
  g_object_set_data (G_OBJECT (menuitem), "iiim-display-name", g_strdup (ui));
  if (engine_id != NULL)
      g_object_set_data (G_OBJECT (menuitem), "iiim-lename", g_strdup (engine_id));
  
  g_signal_connect (GTK_MENU_ITEM (menuitem), "activate",
		    G_CALLBACK (lang_menu_item_activate_cb), gimlet);
}

static void
item_free_data (GtkWidget *menuitem, gpointer user_data)
{
  GtkWidget *submenu = gtk_menu_item_get_submenu ((GtkMenuItem *)menuitem);

  if (submenu) {
      g_list_foreach (((GtkMenuShell*)submenu)->children, (GFunc)item_free_data, NULL);
  } else {
    g_free (g_object_get_data (G_OBJECT (menuitem), "iiim-lang-name"));
    g_free (g_object_get_data (G_OBJECT (menuitem), "iiim-display-name"));
    g_free (g_object_get_data (G_OBJECT (menuitem), "iiim-lename"));
  }
}

static GtkWidget *
create_lang_menu ()
{
  GSList *lang_list = NULL, *lp;
  GtkWidget *menu;
  GtkWidget *menuitem;
  gchar* prev_lang = NULL;
  GtkWidget *le_submenu;
  GtkWidget *le_submenu_item;

  menu = gtk_menu_new ();

  lang_list = gconf_client_get_list (gconf_client_get_default (),
				     KEY_LANG_TO_USE_LIST, GCONF_VALUE_STRING,
				     NULL);

  for (lp = lang_list; lp; lp = g_slist_next (lp))
    {
      /* need to keep lang_id and UI string */
      gchar *lang_id, *lang_ui, *lang_engine, *lang_engine_ui;

      extract_lang_props ((gchar *)lp->data, &lang_id, &lang_ui, &lang_engine, &lang_engine_ui);

      if (prev_lang == NULL)
	{
	  menuitem = gtk_image_menu_item_new_with_label (lang_ui);
	  if (g_str_has_prefix (lang_id, "x-kbl-") && kbd_pixbuf != NULL)
	    {
	      GtkWidget *kbd_image = gtk_image_new_from_pixbuf (kbd_pixbuf);
	      gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), kbd_image);
	    }

	  gtk_widget_show (menuitem);
	  gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

	  if(lang_engine != NULL)
	    {
	      prev_lang = g_strdup (lang_id);
	      le_submenu = gtk_menu_new ();
	      le_submenu_item = gtk_menu_item_new_with_label (lang_engine_ui);
	      item_add_data_and_signal (le_submenu_item, lang_id, lang_engine_ui, lang_engine, gimlet);
	      gtk_widget_show (le_submenu_item);
	      gtk_widget_show (le_submenu);
	      gtk_menu_shell_append (GTK_MENU_SHELL (le_submenu), le_submenu_item);
	      gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), le_submenu);
	    }
	  else
	    {
	      item_add_data_and_signal (menuitem, lang_id, lang_ui, lang_engine, gimlet);
	    }
	}
      else /* prev_lang is not NULL */
	{
	  if (!strcmp (lang_id, prev_lang))
	    {
	      /* use existing le_submenu */
	      /* engine must be not NULL here */
	      le_submenu_item = gtk_menu_item_new_with_label (lang_engine_ui);
	      item_add_data_and_signal (le_submenu_item, lang_id, lang_engine_ui, lang_engine, gimlet);
	      gtk_widget_show (le_submenu_item);
	      gtk_menu_shell_append (GTK_MENU_SHELL (le_submenu), le_submenu_item);
	    }
	  else /* prev_lang is needed to update */
	    {
	      g_free (prev_lang);

	      /* next language */
	      menuitem = gtk_image_menu_item_new_with_label (lang_ui);
	      if (g_str_has_prefix (lang_id, "x-kbl-") && kbd_pixbuf != NULL)
		{
		  GtkWidget *kbd_image = gtk_image_new_from_pixbuf (kbd_pixbuf);
		  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), kbd_image);
		}
	      gtk_widget_show (menuitem);
	      gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	      
	      if (lang_engine != NULL)
		{
		  prev_lang = g_strdup (lang_id);
		  le_submenu = gtk_menu_new ();
		  le_submenu_item = gtk_menu_item_new_with_label (lang_engine_ui);
		  item_add_data_and_signal (le_submenu_item, lang_id, lang_engine_ui, lang_engine, gimlet);
		  gtk_widget_show (le_submenu_item);
		  gtk_widget_show (le_submenu);
		  gtk_menu_shell_append (GTK_MENU_SHELL (le_submenu), le_submenu_item);
		  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), le_submenu);
		}
	      else
		{
		  prev_lang = NULL;
		  item_add_data_and_signal (menuitem, lang_id, lang_ui, lang_engine, gimlet);
		}
	    }
	}

      /* cleanup */
      g_free (lang_id); g_free (lang_ui); g_free (lang_engine); g_free (lang_engine_ui);
    }

  if (lang_list != NULL) {
    g_slist_foreach (lang_list, _g_slist_free_data, NULL);
    g_slist_free (lang_list);
  }

  return menu;
}

gchar *
get_language_full_name_by_short (gchar *short_name)
{
  gchar *ret = NULL;
  GSList *lang_list = NULL, *lp;

  lang_list = gconf_client_get_list (gconf_client_get_default (),
				     KEY_LANG_TO_USE_LIST, GCONF_VALUE_STRING,
				     NULL);

  for (lp = lang_list; lp; lp = g_slist_next (lp)) {
    gchar *lang_id, *lang_ui, *lang_engine, *lang_engine_ui;
    extract_lang_props ((gchar *)lp->data, &lang_id, &lang_ui, &lang_engine, &lang_engine_ui);
    
    if ( 0 == strcasecmp (lang_id, short_name)) 
      ret = g_strdup (lang_ui);

    g_free (lang_id); g_free (lang_ui); g_free (lang_engine); g_free (lang_engine_ui);

    if (ret)
      break;
  }

  if (lang_list != NULL) {
    g_slist_foreach (lang_list, _g_slist_free_data, NULL);
    g_slist_free (lang_list);
  }
  return ret;
}

void
gimlet_show_lang_menu ()
{
  if (NULL == gimlet->lang_menu) 
    {
      gimlet->lang_menu = create_lang_menu ();
    }
  else if (gimlet->le_list_changed) 
    {
      /* free all the associated data in menuitem */
      g_list_foreach (((GtkMenuShell *)gimlet->lang_menu)->children, (GFunc)item_free_data, NULL);
      gtk_widget_destroy (gimlet->lang_menu);
      
      /* re-create the language menu */
      gimlet->lang_menu = create_lang_menu ();
      gimlet->le_list_changed = FALSE;
    }

  gtk_menu_popup (GTK_MENU (gimlet->lang_menu), NULL, NULL, NULL, NULL, 1, 0);
}

static void
start_properties (GtkWidget *item,
		  gpointer user_data)
{
  char *pathname = "/usr/bin/iiim-properties";

  if (access (pathname, X_OK) == 0)
    {
      pid_t pid;

      pid = fork ();
      if (pid == 0)
	{
	  char *argv[2];

	  argv[0] = "iiim-properties";
	  argv[1] = NULL;
	  execv (pathname, argv);
	  _exit (1);	   
	}
    }
}

static void
start_vkb (GtkWidget *item,
		  gpointer user_data)
{
  char *pathname = "/usr/bin/iiim-keyboard";

  if (access (pathname, X_OK) == 0)
    {
      pid_t pid;

      pid = fork ();
      if (pid == 0)
	{
	  char *argv[2];

	  argv[0] = "iiim-keyboard";
	  argv[1] = NULL;
	  execv (pathname, argv);
	  _exit (1);	   
	}
    }
}

static void
start_help (GtkWidget *item,
            gpointer user_data)
{
  gnome_url_show ("ghelp:iiim-properties#imswitcher", NULL);
}

static void
start_about (GtkWidget *item,
             gpointer user_data)
{
  static GtkWidget *about = NULL;

  gchar *authors [] = {
    "Hidetoshi Tajima",
    "Jens Petersen <petersen@redhat.com>",
    "Federic Zhang",
    "Yong Sun",
    NULL
  };

  if (about != NULL) {
    gtk_window_present (GTK_WINDOW (about));
    return;
  }

  about = gnome_about_new (_("Input Method Switcher"),
                           VERSION,
                           "Copyright (c) 2002, 2010 Oracle and/or its affiliates.\nCopyright (c) 2002,2004 Red Hat, Inc.",
                           _("Switch Input Languages on the panel."),
                           (const gchar**)authors,
                           NULL, NULL, NULL);

  gtk_window_set_destroy_with_parent (GTK_WINDOW(about), TRUE);

  gtk_signal_connect (GTK_OBJECT(about),
                      "destroy",
                      GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                      &about);

  gtk_widget_show (about);
}

static GtkWidget *
create_help_menu ()
{
  GtkWidget *menu;
  GtkWidget *item;
  GtkWidget *image;

  /* to be rewritted later */
  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_from_stock ("gtk-preferences", NULL);
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  g_signal_connect (GTK_MENU_ITEM (item),
		    "activate",
		    G_CALLBACK (start_properties),
		    NULL);

  item = gtk_image_menu_item_new_with_label (_("Virtual Keyboard"));
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  g_signal_connect (GTK_MENU_ITEM (item),
		    "activate",
		    G_CALLBACK (start_vkb),
		    NULL);

  item = gtk_image_menu_item_new_from_stock ("gtk-help", NULL);
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  g_signal_connect (GTK_MENU_ITEM (item),
		    "activate",
		    G_CALLBACK (start_help),
		    NULL);

#if GTK_CHECK_VERSION(2, 6, 0)
  item = gtk_image_menu_item_new_from_stock ("gtk-about", NULL);
#else
  item = gtk_image_menu_item_new_with_mnemonic (_("_About"));
#endif
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  g_signal_connect (GTK_MENU_ITEM (item),
		    "activate",
		    G_CALLBACK (start_about),
		    NULL);

#if 0
  item = gtk_separator_menu_item_new ();
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  gtk_widget_set_sensitive (item, FALSE);

  item = gtk_image_menu_item_new_from_stock ("gtk-quit", NULL);
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  g_signal_connect (GTK_MENU_ITEM (item),
		    "activate",
		    G_CALLBACK (gtk_main_quit),
		    NULL);
#endif

  return menu;
}

void
gimlet_show_help_menu ()
{
  if (NULL == gimlet->help_menu) {
    gimlet->help_menu = create_help_menu ();
  }

  gtk_menu_popup (GTK_MENU (gimlet->help_menu), NULL, NULL, NULL, NULL, 1, 0);
}
