#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#if defined(lint) && defined(HAVE_NOTE_H)
#include <note.h>
#endif /* lint && HAVE_NOTE_H */

#include <iiimp-data.h>

#include "iiimp-dataP.h"


IIIMP_hotkey *
iiimp_hotkey_new(
    IIIMP_data_s *		data_s,
    IIIMP_card16		hotkey_id,
    IIIMP_card8			state,
    IIIMP_card8			action,
    IIIMP_keyevent_list *	keyevent_list,
    IIIMP_string *		label)
{
    IIIMP_hotkey *	data;
    size_t		nbyte;
    int			i;

    data = (IIIMP_hotkey *)malloc(sizeof (IIIMP_hotkey));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = 0;
    nbyte += 2;	/* hotkey_id */
    nbyte += 1;	/* state: enabled/disabled */
    nbyte += 1;	/* action */
    nbyte += 4;	/* byte length of LISTofKEYEVENT */
    if (NULL == keyevent_list) {	/* keyevent */
	nbyte += 0;
    } else {
	nbyte += keyevent_list->nbyte;
    }
    if (NULL == label) {	/* label */
	label += 4;
    } else {
	nbyte += label->nbyte;
    }

    data->nbyte = nbyte;
    data->hotkey_id = hotkey_id;
    data->state = state;
    data->action = action;
    data->keyevent_list = keyevent_list;
    data->label = label;
    data->next = NULL;

    return data;
}


void
iiimp_hotkey_pack(
    IIIMP_data_s *	data_s,
    IIIMP_hotkey  *	m,
    size_t *		nbyte,
    uchar_t **		ptr)
{
    size_t	rest;
    uchar_t *	p;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUTU16(m->hotkey_id, rest, p, data_s->byte_swap);
    PUTU8(m->state, rest, p, data_s->byte_swap);
    PUTU8(m->action, rest, p, data_s->byte_swap);
    if (NULL == m->keyevent_list) {
	PUTU32(0, rest, p, data_s->byte_swap);
    } else {
	PUTU32(m->keyevent_list->nbyte, rest, p, data_s->byte_swap);
	iiimp_keyevent_list_pack(data_s, m->keyevent_list, &rest, &p);
    }
    if (NULL == m->label) {
	PUTU16(0, rest, p, data_s->byte_swap);	/* length */
	PUTU16(0, rest, p, data_s->byte_swap);	/* padding */
    } else {
	iiimp_string_pack(data_s, m->label, &rest, &p);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


void
iiimp_hotkey_list_pack(
    IIIMP_data_s *	data_s,
    IIIMP_hotkey *	m,
    size_t *		nbyte,
    uchar_t **		ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    for (; NULL != m; m = m->next) {
	iiimp_hotkey_pack(data_s, m, &rest, &p);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


void
iiimp_hotkey_delete(IIIMP_data_s * data_s, IIIMP_hotkey * m)
{
    if (NULL == m) return;
    iiimp_keyevent_list_delete(data_s, m->keyevent_list);
    iiimp_string_delete(data_s, m->label);
    free(m);
}


void
iiimp_hotkey_list_delete(IIIMP_data_s * data_s, IIIMP_hotkey * m)
{
    IIIMP_hotkey *	p;
    IIIMP_hotkey *	n;
    for (p = m; NULL != p; p = n) {
	n = p->next;
	iiimp_hotkey_delete(data_s, p);
    }
    return;
}


IIIMP_hotkey *
iiimp_hotkey_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_hotkey *	data;
    int			len;
    size_t		rest;
    const uchar_t *	p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (2 + 1 + 1 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_hotkey *)malloc(sizeof (IIIMP_hotkey));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    data->next = NULL;

    GETU16(data->hotkey_id, rest, p, data_s->byte_swap);
    GETU8(data->state, rest, p, data_s->byte_swap);
    GETU8(data->action, rest, p, data_s->byte_swap);
    GETU32(len, rest, p, data_s->byte_swap);

    data->keyevent_list = iiimp_keyevent_list_unpack(data_s, &rest, &p, len);
    data->label = iiimp_string_unpack(data_s, &rest, &p, rest);

    data->nbyte = (2 + 1 + 1 + 4 +
		   ((NULL == data->keyevent_list) ?
		    4 : data->keyevent_list->nbyte) +
		   ((NULL == data->label) ? (2 + 2) : data->label->nbyte));

    *nbyte -= (nbyte_max - rest);
    *ptr = p;

    return data;
}


IIIMP_hotkey *
iiimp_hotkey_list_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_hotkey *	hotkey;
    IIIMP_hotkey *	hotkey_head;
    IIIMP_hotkey **	hotkey_next;
    size_t		rest;
    const uchar_t *	p;

    rest = nbyte_max;
    p = *ptr;
    hotkey_head = NULL;
    hotkey = NULL;
    hotkey_next = &hotkey_head;

    if (((*nbyte) < nbyte_max) || (0 == rest)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    while (0 < rest) {
	hotkey = iiimp_hotkey_unpack(data_s, &rest, &p, rest);
	if (NULL == hotkey) {
	    iiimp_hotkey_list_delete(data_s, hotkey_head);
	    return NULL;
	} else {
	    *hotkey_next = hotkey;
	    hotkey_next = &(hotkey->next);
	}
    }

    *nbyte -= (nbyte_max - rest);
    *ptr = p;

    return hotkey_head;
}


void
iiimp_hotkey_print(
    IIIMP_data_s *	data_s,
    IIIMP_hotkey *	m)
{
    int			i;
    const char *	p;

    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tHOTKEY = \n");
    (void)fprintf(data_s->print_fp, "\thotkey ID = %d\n", m->hotkey_id);
    switch (m->state) {
    case 0:
	p = "disabled";
	break;
    case 1:
	p = "enabled";
	break;
    default:
	p = "unknown";
	break;
    }
    (void)fprintf(data_s->print_fp, "\thotkey state = %s (%d)\n", p, m->state);
    switch (m->action) {
    case 0:
	p = "event forwarding off";
	break;
    case 1:
	p = "event forwarding on";
	break;
    case 2:
	p = "event forwarding toggle";
	break;
    default:
	p = "unknown";
	break;
    }
    (void)fprintf(data_s->print_fp, "\thotkey action = %s (%d)\n", p, m->action);
    (void)fprintf(data_s->print_fp, "\thotkey keyevent =\n");
    iiimp_keyevent_list_print(data_s, m->keyevent_list);
    (void)fprintf(data_s->print_fp, "\thotkey label = ");
    iiimp_string_print(data_s, m->label);
    (void)fputc('\n', data_s->print_fp);
}


void
iiimp_hotkey_list_print(
    IIIMP_data_s *	data_s,
    IIIMP_hotkey *	m)
{
    for (; NULL != m; m = m->next) {
	iiimp_hotkey_print(data_s, m);
    }
}


/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
