#include <stdio.h>
#include <sys/stat.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "lookup_data.h"

#include "imbean.h"

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

int lookup_data_read_profile_to_memory(char **info_ptr);
int lookup_data_parse_profile(lookup_data_t *lookup_data, char *xml_str, int size);

/*********************************************************************/
/*                       symbol_group utilities                      */
/*********************************************************************/
symbol_group_t *symbol_group_new()
{
    symbol_group_t *symbol_group = NULL;

    symbol_group = (symbol_group_t *) calloc(1, sizeof(symbol_group_t));

    return symbol_group;
}

void symbol_group_destroy(symbol_group_t *symbol_group)
{
    if (symbol_group == NULL)
        return;

    if (symbol_group->id)
        free ((char *)symbol_group->id);

    if (symbol_group->name)
        free ((char *)symbol_group->name);

    if (symbol_group->symbols != NULL) {
        int i;
        for (i = 0; i < symbol_group->num_symbols; i ++) {
            if (symbol_group->symbols[i])
                free ((char *) symbol_group->symbols[i]);
        }
        free ((char *)symbol_group->symbols);
    }

    free ((char *)symbol_group);
}

void symbol_group_print(symbol_group_t *symbol_group)
{
#if DEBUG
    if (symbol_group == NULL)
        return;

    printf("Symbol Group: \n");
    if (symbol_group->id)
        printf("    id: %s\n", symbol_group->id);
    if (symbol_group->name)
        printf("    name: %s\n", symbol_group->name);

    if (symbol_group->symbols != NULL) {
        int i;
        printf("    symbols: ");
        for (i = 0; i < symbol_group->num_symbols; i ++) {
            if (symbol_group->symbols[i])
                printf ("%s ", (char *) symbol_group->symbols[i]);
        }
        printf("\n");
    }
#endif
}

#define SYMBOL_LIST_NUM_ALLOC  10
int symbol_group_pushback_symbol(symbol_group_t *symbol_group,
                                 char *symbol)
{
    int i, num_symbols;

    if (symbol_group == NULL || symbol == NULL)
        return LOOKUP_ERROR;

    if (symbol_group->symbols == NULL) {
        symbol_group->symbols = (char **) calloc (SYMBOL_LIST_NUM_ALLOC,
                                                  sizeof(char *));
        if (symbol_group->symbols == NULL)
            return LOOKUP_ERROR;
    }

    num_symbols = symbol_group->num_symbols;
    if ((num_symbols + 1) % SYMBOL_LIST_NUM_ALLOC == 0) {
        int num = num_symbols + 1 + SYMBOL_LIST_NUM_ALLOC;

        symbol_group->symbols = (char **)realloc(symbol_group->symbols,
                                                 num * sizeof(char *));
        if (symbol_group->symbols == NULL)
            return LOOKUP_ERROR;

        for (i = num_symbols; i < num; i++)
            symbol_group->symbols[i] = NULL;
    }

    symbol_group->symbols[num_symbols] = (char *)strdup(symbol);
    if (symbol_group->symbols[num_symbols] == NULL)
        return LOOKUP_ERROR;

    symbol_group->num_symbols ++;

    return LOOKUP_OK;
}

/*********************************************************************/
/*                       lookup_data_t  utilities                    */
/*********************************************************************/
lookup_data_t *lookup_data_new()
{
    lookup_data_t *lookup_data = NULL;
    char *xml_str = NULL;
    int  len_xml_str;

    lookup_data = (lookup_data_t *) calloc(1, sizeof(lookup_data_t));
    if (lookup_data == NULL)
        return NULL;

    len_xml_str = lookup_data_read_profile_to_memory(&xml_str);
    if (len_xml_str <= 0) {
        return lookup_data;
    }

    lookup_data_parse_profile(lookup_data, xml_str, len_xml_str);

    if (xml_str != NULL)
        free ((char *) xml_str);

    return lookup_data;
}

void lookup_data_destroy(lookup_data_t *lookup_data)
{
    if (lookup_data == NULL)
        return;

    if (lookup_data->symbol_groups) {
        int i;

        for (i = 0; i < lookup_data->num_symbol_groups; i++) {
            if (lookup_data->symbol_groups[i])
                symbol_group_destroy(lookup_data->symbol_groups[i]);
        }
        free ((char *)lookup_data->symbol_groups);
    }

    free ((char *)lookup_data);
}

void lookup_data_print(lookup_data_t *lookup_data)
{
    if (lookup_data == NULL)
        return;

    if (lookup_data->symbol_groups) {
        int i;

        for (i = 0; i < lookup_data->num_symbol_groups; i++) {
            if (lookup_data->symbol_groups[i])
                symbol_group_print(lookup_data->symbol_groups[i]);
        }
    }
}

#define SYMBOL_GROUP_LIST_NUM_ALLOC  10
int lookup_data_pushback_symbol_group(lookup_data_t *lookup_data,
                                      symbol_group_t *symbol_group)
{
    int i, num_symbol_groups;

    if (lookup_data == NULL || symbol_group == NULL)
        return LOOKUP_ERROR;

    if (lookup_data->symbol_groups == NULL) {
        lookup_data->symbol_groups = (symbol_group_t **)
                                          calloc (SYMBOL_GROUP_LIST_NUM_ALLOC,
                                                  sizeof(symbol_group_t *));
        if (lookup_data->symbol_groups == NULL)
            return LOOKUP_ERROR;
    }

    num_symbol_groups = lookup_data->num_symbol_groups;
    if ((num_symbol_groups + 1) % SYMBOL_GROUP_LIST_NUM_ALLOC == 0) {
        int num = num_symbol_groups + 1 + SYMBOL_GROUP_LIST_NUM_ALLOC;

        lookup_data->symbol_groups = (symbol_group_t **)realloc(lookup_data->symbol_groups,
                                                                num * sizeof(symbol_group_t *));
        if (lookup_data->symbol_groups == NULL)
            return LOOKUP_ERROR;

        for (i = num_symbol_groups; i < num; i++)
            lookup_data->symbol_groups[i] = NULL;
    }

    lookup_data->symbol_groups[num_symbol_groups] = symbol_group;
    lookup_data->num_symbol_groups ++;

    return LOOKUP_OK;
}

/*********************************************************************/
/*                       profile utilities                           */
/*********************************************************************/
#define LOOKUO_DATA_FILE_NAME "symbol_groups_tw.xml"
int lookup_data_read_profile_to_memory(char **info_ptr)
{
    char file_name[256];

    struct stat file_stat;
    int ret, file_size;
    FILE *fd;

    char *xml_str = NULL;

    *info_ptr = NULL;

    snprintf(file_name, 256, "%s/%s", LE_AUX_MODULES_DIR, LOOKUO_DATA_FILE_NAME);
    DEBUG_printf("file name :%s\n", file_name);

    ret = stat(file_name, &file_stat);
    if (ret == -1) {
        return 0;
    }

    file_size = file_stat.st_size;
    if (file_size == 0)
        return 0;

    xml_str = (char *) calloc(1, file_size + 1);
    fd = fopen(file_name, "r");
    if (!fd) {
        free ((char *) xml_str);
        return 0;
    }

    fread((char *) xml_str, file_size, 1, fd);
    xml_str[file_size] = 0;

    fclose(fd);

    *info_ptr = xml_str;
    return (file_size + 1);
}

static char utf8_skip_data[256] = {
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
  3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,5,5,5,5,6,6,1,1
};

void lookup_data_parse_symbol_group_item(symbol_group_t *symbol_group,
                                         IbmlProperty *ibml_property)
{
    char *name, *value;
    char symbol[16], *ptr;

    name = (char *) ibml_property->name;
    if (!name || !*name)
        return;

    value = (char *) ibml_property->value;
    if (!value || !*value)
        return;

    if (!strcasecmp(name, "name")) {
        if (symbol_group->name)
            free ((char *)symbol_group->name);
        symbol_group->name = (char *) strdup(value);
    } else if (!strcasecmp(name, "symbols")) {
        ptr = value;
        while (ptr && *ptr) {
            int len;

            if (*ptr == ' ' ||
                *ptr == '\t' ||
                *ptr == '\n') {
                ptr ++;
                continue;
            }

            memset(symbol, 0, 16);

            /* get byte len of next utf8 character */
            len = utf8_skip_data[*(unsigned char *)ptr];
            strncpy(symbol, ptr, len);

            if (*symbol)
                symbol_group_pushback_symbol(symbol_group, symbol);

            ptr = ptr + len;
        }
    }

    return;
}

void lookup_data_parse_symbol_groups_info(lookup_data_t *lookup_data,
                                          IbmlCategory *ibml_category)
{
    symbol_group_t *symbol_group;

    IbmlElement *ibml_element;
    IbmlProperty *ibml_property;

    int ret, i, j;

    char *id, *scope;

    if (ibml_category == NULL)
        return;
    if (ibml_category->num_elements <= 0)
        return;

    for (i = 0; i < ibml_category->num_elements; i++) {
        ibml_element = ibml_category->elements[i];
        if (!ibml_element)
            continue;
        if (ibml_element->num_properties <= 0)
            continue;

        id = (char *) ibml_element->id;
        scope = (char *) ibml_element->scope;
        DEBUG_printf("id:%s, scope:%s\n", id ? id : "NULL",
               scope ? scope : "NULL");
        if (!id || !*id)
            continue;

        symbol_group = (symbol_group_t *) symbol_group_new();
        if (symbol_group == NULL)
            continue;

        symbol_group->id = (char *) strdup(id);

        for (j = 0; j < ibml_element->num_properties; j++) {
            ibml_property = ibml_element->properties[j];
            if (!ibml_property)
                continue;

            lookup_data_parse_symbol_group_item(symbol_group, ibml_property);
        }

        if (symbol_group->id == NULL ||
            symbol_group->name == NULL) {
            symbol_group_destroy (symbol_group);
            continue;
        }

        ret = lookup_data_pushback_symbol_group(lookup_data, symbol_group);
        if (ret == LOOKUP_ERROR) {
            symbol_group_destroy (symbol_group);
            continue;
        }
    }

    return;
}

#define SYMBOL_GROUPS_CATAGORY "symbol_groups"
int lookup_data_parse_profile(lookup_data_t *lookup_data, char *xml_str, int size)
{
    IbmlData *ibml_data;
    IbmlCategory *ibml_category;
    int i;

    if (xml_str == NULL || size <= 0)
        return LOOKUP_ERROR;

    ibml_data = (IbmlData *) imbean_config_new_from_memory(xml_str, size);
    if (ibml_data == NULL)
        return LOOKUP_ERROR;

    for (i = 0; i < ibml_data->num_categories; i++) {
        ibml_category = ibml_data->categories[i];

        if (!ibml_category->scope || !*ibml_category->scope)
            continue;

        if (!strcasecmp(ibml_category->scope, SYMBOL_GROUPS_CATAGORY)) {
            lookup_data_parse_symbol_groups_info(lookup_data, ibml_category);
        }
    }

    ibml_data_free(ibml_data);
    return LOOKUP_OK;
}
