/*

Copyright (c) 2006 Yuuhei TERAMOTO <tera@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "syscfg.h"
#include "phonogram_conv.h"

static int	get_accepted( PGC_UTFCHAR *inbuf, pgc_trans_list lst, pgc_trans *trans );
static int	get_inprocess( PGC_UTFCHAR *inbuf, pgc_trans_list lst, pgc_trans *trans );
static int	get_rejected( PGC_UTFCHAR *inbuf, pgc_trans_list lst, pgc_trans *trans );
static int	choose_NN( PGC_UTFCHAR *inbuf, pgc_trans_list lst, pgc_trans_list *choosed );
static int	choose_minunits( pgc_trans_list lst, pgc_trans_list *choosed );
static int	choose_undetermined( pgc_trans_list lst, pgc_trans_list *choosed );
static int	get_NN_count( PGC_UTFCHAR *inbuf, pgc_trans trans, size_t *size );

int
romaji_kana_convert(
    pgc_descriptor pgcd,
    PGC_UTFCHAR *inbuf, 
    size_t inlen,
    int eot,
    pgc_trans * trans
)
{
    int ret,res;
    pgc_trans_list lst = NULL;
    *trans = NULL;

    ret = pgc_convert( pgcd, inbuf, inlen, eot, &lst );

    if( ret == PGC_ACCEPTED ){
	if( (res = get_accepted( inbuf, lst, trans )) != 0 ){
	    ret = -1;
	}
    }else if( ret == PGC_INPROCESS ){
	if( (res = get_inprocess( inbuf, lst, trans )) != 0 ){
	    ret = -1;
	}
    }else if( ret == PGC_REJECTED ){
	if( (res = get_rejected( inbuf, lst, trans )) != 0 ){
	    ret = -1;
	}
    }
    pgc_trans_list_destroy( lst );
    return ret;
}

static int
get_accepted(
    PGC_UTFCHAR *inbuf,
    pgc_trans_list lst,
    pgc_trans *trans
)
{
    pgc_trans tr;
    pgc_trans_list choosed1=NULL,choosed2=NULL;
    pgc_trans_list lst1,lst2;
    int ret=0;


    if( (ret = choose_NN( inbuf, lst, &choosed1 ) ) != 0 ){
	return ret;
    }
    if( choosed1 == NULL ){
	lst1 = lst;
    }else{
	lst1 = choosed1;
    }
    if( (ret = choose_minunits( lst1, &choosed2 )) != 0 ){
	return ret;
    }
    if( choosed2 == NULL ){
	lst2 = lst1;
    }else{
	lst2 = choosed2;
    }
    tr = pgc_trans_list_next( lst2, NULL );
    *trans = pgc_trans_dup( tr );
    pgc_trans_list_destroy( choosed1 );
    pgc_trans_list_destroy( choosed2 );
    return 0;
}

static int
get_inprocess(
    PGC_UTFCHAR *inbuf,
    pgc_trans_list lst,
    pgc_trans *trans
)
{
    pgc_trans tr;
    pgc_trans_list choosed1=NULL,choosed2=NULL,choosed3=NULL;
    pgc_trans_list lst1, lst2, lst3;
    int ret=0;


    if( (ret = choose_NN( inbuf, lst, &choosed1 ) ) != 0 ){
	return ret;
    }
    if( choosed1 == NULL ){
	lst1 = lst;
    }else{
	lst1 = choosed1;
    }
    if( (ret = choose_undetermined( lst1, &choosed2 )) != 0 ){
	return ret;
    }
    if( choosed2 == NULL ){
	lst2 = lst1;
    }else{
	lst2 = choosed2;
    }
    if( (ret = choose_minunits( lst2, &choosed3 )) != 0 ){
	return ret;
    }
    if( choosed3 == NULL ){
	lst3 = lst2;
    }else{
	lst3 = choosed3;
    }
    tr = pgc_trans_list_next( lst3, NULL );
    *trans = pgc_trans_dup( tr );
    pgc_trans_list_destroy( choosed1 );
    pgc_trans_list_destroy( choosed2 );
    pgc_trans_list_destroy( choosed3 );
    return 0;
}

static int
get_rejected(
    PGC_UTFCHAR *inbuf,
    pgc_trans_list lst,
    pgc_trans *trans
)
{
    pgc_trans tr;
    pgc_trans_list choosed1=NULL,choosed2=NULL;
    pgc_trans_list lst1, lst2;
    int ret=0;


    if( (ret = choose_NN( inbuf, lst, &choosed1 ) ) != 0 ){
	return ret;
    }
    if( choosed1 == NULL ){
	lst1 = lst;
    }else{
	lst1 = choosed1;
    }
    if( (ret = choose_minunits( lst1, &choosed2 )) != 0 ){
	return ret;
    }
    if( choosed2 == NULL ){
	lst2 = lst1;
    }else{
	lst2 = choosed2;
    }
    tr = pgc_trans_list_next( lst2, NULL );
    *trans = pgc_trans_dup( tr );
    pgc_trans_list_destroy( choosed1 );
    pgc_trans_list_destroy( choosed2 );
    return 0;
}

static int
choose_NN(
    PGC_UTFCHAR *inbuf,
    pgc_trans_list lst,
    pgc_trans_list *choosed
)
{
    size_t max=(size_t)-1;
    pgc_trans tr;
    *choosed = NULL;
    for( tr = pgc_trans_list_next( lst, NULL ) ;
	 tr != NULL ; tr = pgc_trans_list_next( lst, tr ) ){
	size_t cc;
	if( get_NN_count( inbuf, tr, &cc ) != 0 ){
	    return -1;
	}
	if( max == (size_t)-1 || cc > max ){
	    max = cc;
	}
    }
    if( max == 0 ){
	return 0;
    }
    for( tr = pgc_trans_list_next( lst, NULL ) ;
	 tr != NULL ; tr = pgc_trans_list_next( lst, tr ) ){
	size_t cc;
	if( get_NN_count( inbuf, tr, &cc ) != 0 ){
	    return -1;
	}
	if( cc == max ){
	    pgc_trans tr2 = pgc_trans_dup(tr);
	    if( tr2 == NULL ){
		return -1;
	    }
	    *choosed = pgc_trans_list_append( *choosed, tr2 );
	}
    }
    return 0;
}

static int
choose_minunits(
    pgc_trans_list lst,
    pgc_trans_list *choosed
)
{
    pgc_trans tr;
    size_t min=(size_t)-1;
    *choosed = NULL;
    for( tr = pgc_trans_list_next( lst, NULL ) ;
	 tr != NULL ; tr = pgc_trans_list_next( lst, tr ) ){
	size_t cc;
	if( pgc_trans_unit_count( tr, &cc ) != 0 ){
	    return -1;
	}
	if( cc > 0 && (min == (size_t)-1 || min > cc) ){
	    min = cc;
	}
    }
    if( min == (size_t)-1 ){
	return 0;
    }
    for( tr = pgc_trans_list_next( lst, NULL ) ;
	 tr != NULL ; tr = pgc_trans_list_next( lst, tr ) ){
	size_t cc;
	if( pgc_trans_unit_count( tr, &cc ) != 0 ){
	    return -1;
	}
	if( cc == min ){
	    pgc_trans tr2 = pgc_trans_dup(tr);
	    if( tr2 == NULL ){
		return -1;
	    }
	    *choosed = pgc_trans_list_append( *choosed, tr2 );
	}
    }
    return 0;
}

static int
choose_undetermined(
    pgc_trans_list lst,
    pgc_trans_list *choosed
)
{
    size_t min=(size_t)-1;
    pgc_trans tr;
    *choosed = NULL;
    size_t sp, ep;

    for( tr = pgc_trans_list_next( lst, NULL ) ;
	 tr != NULL ; tr = pgc_trans_list_next( lst, tr ) ){
	if( pgc_trans_get_source_pos( tr, 0, PGC_TRANS_LAST, &sp, &ep ) != 0 ){
	    return -1;
	}
	if( min == (size_t)-1 || ep < min ){
	    min = ep;
	}
    }
    for( tr = pgc_trans_list_next( lst, NULL ) ;
	 tr != NULL ; tr = pgc_trans_list_next( lst, tr ) ){
	if( pgc_trans_get_source_pos( tr, 0, PGC_TRANS_LAST, &sp, &ep ) != 0 ){
	    return -1;
	}
	if( ep == min ){
	    pgc_trans tr2 = pgc_trans_dup(tr);
	    if( tr2 == NULL ){
		return -1;
	    }
	    *choosed = pgc_trans_list_append( *choosed, tr2 );
	}
    }
    return 0;
}

static int
get_NN_count(
    PGC_UTFCHAR *inbuf,
    pgc_trans trans,
    size_t *count
)
{
    size_t ii,max,sp,ep;
    *count = 0;
    if( pgc_trans_unit_count( trans, &max ) != 0 ){
	return -1;
    }
    for( ii = 0 ; ii < max ; ii++ ){
	if( pgc_trans_get_source_pos( trans, ii, ii, &sp, &ep ) != 0 ){
	    continue;
	}
	if( sp+2 == ep && 
	    *(inbuf+sp) == 'n' && *(inbuf+ep-1) == 'n' ){
	    *count = *count + 1;
	}
    }
    return 0;
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
