/*

Copyright (c) 2006 Yuuhei TERAMOTO <tera@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#ifndef PHONOGRAM_CONV_H_INCLUDED
#define PHONOGRAM_CONV_H_INCLUDED

#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#define SIZEOF_UNSIGNED_CHAR 1
#define SIZEOF_UNSIGNED_INT 4
#define SIZEOF_UNSIGNED_LONG 4
#define SIZEOF_UNSIGNED_SHORT 2
#define HAVE_ALLOCA_H 1
#endif
#include <stdio.h>
#include "syscfg.h"
typedef CARD16BIT PGC_UTFCHAR;
#include "iconv.h"

typedef struct pgc_descriptor_rec *pgc_descriptor; 
typedef struct pgc_trans_unit_rec *pgc_trans_unit;
typedef struct pgc_trans_rec *pgc_trans;
typedef struct pgc_trans_rec *pgc_trans_list;

enum pgc_result {
  PGC_INPROCESS = 0,
  PGC_ACCEPTED,
  PGC_REJECTED
};

#define PGC_TRANS_LAST ((size_t)-1)

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Get descriptor used for conversion.
 *
 * After conversion, you must discard the descriptor with pgc_close().
 * You can use one descriptor in a number of conversion.
 *
 * rule_path:	location of unit table which include conversion rule.
 * outenc:	conversion output encode. ex) "UTF-16", "EUC-JP".
 *
 * return: descriptor. 
 *         null: failure.
 */
pgc_descriptor	pgc_open( char *rule_path, char *outenc );

/*
 * Close conversion descriptor.
 */
void		pgc_close( pgc_descriptor pgcd );

/*
 * Convert.
 *
 * pgcd:	conversion descriptor
 * inbuf:	string to be converted.
 * inlen:       string length. unit is PGC_UTFCHAR.
 * eot:         1:string is whole text to be converted.
 *              0:string is incomplete, some more string may follow.
 *
 * trans_list:  result of conversion is set.
 *              you can manipulate trans_list with pgc_trans_...() functions.
 *              you must discard trans_list with pgc_trans_list_destroy().
 *
 * return:
 *         PGC_ACCEPTED: input is accepted and conversion is completed.
 *                       trans_list hold conversion results.
 *         PGC_INPROCESS:conversion is in process, not completed.
 *                       trans_list hold half converted strings.
 *         PGC_REJECTED: input is rejected, won't be accepted with any more following input.
 *                       trans_list hold half converted strings.
 *         < 0:          failure.
 *         if eot is 1, return value is PGC_ACCEPTED or PGC_REJECTED.
 *         if eot is 0, return value is PGC_INPROCESS or PGC_REJECTED.
 *
 * usage sample:
 *         See CUI converter source. (command/phonogram_conv.c)
 */
int pgc_convert( pgc_descriptor pgcd, PGC_UTFCHAR *inbuf, size_t inlen, int eot, pgc_trans_list *trans_list );

/*
 * Japanese Romaji-Kana Convert. 
 *
 * This function is a wrapper of pgc_convert().
 * Difference is that this function select one from results of pgc_convert(),
 * according to romaji-kana conversion rule.
 *
 * pgcd:	conversion descriptor
 * inbuf:	string to be converted.
 * inlen:       string length. unit is PGC_UTFCHAR.
 * eot:         1:string is whole text to be converted.
 *              0:string is part of text, some more string may follow.
 *
 * trans:  result of conversion is set.
 *              you can manipulate trans with pgc_trans_...() functions.
 *              you must discard trans with pgc_trans_destroy().
 *
 * return:
 *         PGC_ACCEPTED: input is accepted and conversion is completed.
 *                       trans_list hold conversion results.
 *         PGC_INPROCESS:conversion is in process, not completed.
 *                       trans_list hold half converted strings.
 *         PGC_REJECTED: input is rejected, won't be accepted with any more following input.
 *                       trans_list hold null.
 *         < 0:          failure.
 *         if eot is 1, return value is PGC_ACCEPTED or PGC_REJECTED.
 *         if eot is 0, return value is PGC_INPROCESS or PGC_REJECTED.
 */
int		romaji_kana_convert( pgc_descriptor pgcd, PGC_UTFCHAR *inbuf, size_t inlen, int eot, pgc_trans * trans );

/*
 * Destroy translation list (conversion results )
 * list: list to be destroied. Null is allowed but do nothing.
 */
void		pgc_trans_list_destroy( pgc_trans_list list );


/*
 * Get next item from translation list.
 * cur: previous of item to be returned. If null, first item is returned.
 */
pgc_trans	pgc_trans_list_next( pgc_trans_list list, pgc_trans cur );

/*
 * List size.
 */
int		pgc_trans_list_size( pgc_trans_list list, size_t *size );

/*
 * Add item to the end of list.
 */
pgc_trans_list	pgc_trans_list_append( pgc_trans_list list, pgc_trans trans );

/*
 * About Translation.
 *
 * Translation consists of unit.
 *   unit0  unit1 unit2.... unitN
 *
 * Unit
 *   Unit is correspond to record of `Unit Table' describing translation rules.
 *
 *   Unit holds part of whole translation string and position in source.
 *
 * ex)
 *
 *  source:
 *    watashi
 *
 *  translation:    
 *		position in source	part of translation
 *    unit0	0-1			U+308F
 *    unit1	2-3			U+305F
 *    unit2	4-6			U+3057
 */

/*
 * Count of units that constitute a translation.
 */
int		pgc_trans_unit_count(pgc_trans trans, size_t *size);

/*
 * Get part(or while) of translation string.
 * su:	index of start unit that you want to get the string from. (index is 0 origin)
 * eu:  next index of end unit. PGC_TRANS_LAST indicates last unit of translation.
 * sp:	output. start position(bytes) of retrieved string in the whole translation string.
 * ep:  outout. next position(bytes) of end char of retrieved string in whole translation string.
 * buf: outout. set the string. If null, no outout.
 * size: size of buf(bytes).
 */
int		pgc_trans_get_string( pgc_trans trans, size_t su, size_t eu,
				      size_t *sp, size_t *ep, unsigned char *buf, size_t bufsize );
/*
 * Compare translation strings. Case sensitive.
 * return: ret=0 -> t1 equal t2 , ret<0 -> t1<t2, ret>0 -> t1>t2.
 */
int		pgc_trans_strcmp( pgc_trans t1, pgc_trans t2 );

/*
 * Get the position(bytes) in source.
 * su:	index of start unit.
 * eu:  next index of end unit. PGC_TRANS_LAST indicates last unit of translation.
 */
int		pgc_trans_get_source_pos( pgc_trans trans, size_t su, size_t eu, size_t *sp, size_t *ep );

/*
 * Destroy translation.
 */
void		pgc_trans_destroy( pgc_trans trsl );

/*
 * change the encoding of translation string.
 * icd: iconv descripter used in conversion.
 */
int		pgc_trans_reencode( iconv_t icd, pgc_trans trans );

pgc_trans	pgc_trans_create();
pgc_trans	pgc_trans_dup( pgc_trans trsl );
pgc_trans	pgc_trans_add( pgc_trans trsl, pgc_trans_unit unit );

/*
 * Debug function.
 * Dump contents of translation.
 */
void		pgc_trans_print( pgc_trans trans, FILE *fp );


#ifdef __cplusplus
} //extern "C"
#endif

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */

#endif // PHONOGRAM_CONV_H_INCLUDED
