/*
  Copyright 2002-2003 Sun Microsystems, Inc. All Rights Reserved.
  Copyright 2006 Yuuhei TERAMOTO <tera@terre-sys.com>

  Permission is hereby granted, free of charge, to any person obtaining a
  copy of this software and associated documentation files (the
  "Software"), to deal in the Software without restriction, including
  without limitation the rights to use, copy, modify, merge, publish,
  distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to
  the following conditions: The above copyright notice and this
  permission notice shall be included in all copies or substantial
  portions of the Software.


  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
  IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
  THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
  ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


  Except as contained in this notice, the names of The Open Group and/or
  Sun Microsystems, Inc. shall not be used in advertising or otherwise to
  promote the sale, use or other dealings in this Software without prior
  written authorization from The Open Group and/or Sun Microsystems,
  Inc., as applicable.


  X Window System is a trademark of The Open Group

  OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
  logo, LBX, X Window System, and Xinerama are trademarks of the Open
  Group. All other trademarks and registered trademarks mentioned herein
  are the property of their respective owners. No right, title or
  interest in or to any trademark, service mark, logo or trade name of
  Sun Microsystems, Inc. or its licensors is granted.

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <stdarg.h>
#include "syscfg.h"
#include "encode.h"
#include "codetable.h"
#include "transintrn.h"

#define MAX_INPUT_KEY_NUM	256

static void assert_( char * fmt,... );

/*********************** Load Codetable File ***********************/

/*
 * LoadCodeTableHeader -- load the input table header from codetable binary file
 */
int LoadCodeTableHeader (file_name, hztbl)
     char *file_name;
     CodeTableStruct *hztbl;
{
  FILE *ifile;
  char ctFlag[256];
  int  ver;

  /* read table from file to memory buffer  */
  ifile = fopen (file_name, "r");
  if (! ifile) {
    fprintf(stderr, "Unable to open the input table file \"%s\"\n",file_name);
    return(-1);
  }

  /* Read CodeTable File Flag */
  if (fread (ctFlag, strlen(CODETABLE_FLAG), 1, ifile) != 1) {
    fprintf (stderr, "Codetable File read Error:%s\n", file_name);
    fclose(ifile);
    return(-1);
  }

  if (strncmp (ctFlag, CODETABLE_FLAG, strlen(CODETABLE_FLAG)) != 0) {
    fprintf (stderr, "File is not in CodeTable format\n");
    fclose(ifile);
    return(-1);
  }

  /* Read CodeTable Version Flag */
  if (fread ((char *)(&ver), sizeof (int), 1, ifile) != 1)  {
    fprintf (stderr, "Codetable File read Error:%s\n", file_name);
    fclose(ifile);
    return(-1);
  }

  if (ver != CODETABLE_VERSION) {
    fprintf (stderr, "File is not in correct Version Number\n");
    fclose(ifile);
    return(-1);
  }

  if (fread((char *)hztbl, sizeof(CodeTableStruct), 1, ifile) != 1) {
    fprintf(stderr, "Error in loading input table for %s\n", file_name);
    fclose(ifile);
    return(-1);
  }

  fclose (ifile);
  return(0);
}

/*
 * LoadCodeTable -- load the input table from codetable binary file
 */
int LoadCodeTable (file_name, hztbl)
     char *file_name;
     CodeTableStruct *hztbl;
{
  FILE *ifile;
  char ctFlag[256];
  int  ver, i;

  ifile = fopen (file_name, "r");
  if (! ifile) {
    return(-1);
  }

  /* Read CodeTable File Flag */
  if (fread (ctFlag, strlen(CODETABLE_FLAG), 1, ifile) != 1) {
    fprintf (stderr, "Codetable File read Error:%s\n", file_name);
    fclose (ifile);
    return(-1);
  }

  if (strncmp (ctFlag, CODETABLE_FLAG, strlen(CODETABLE_FLAG)) != 0) {
    fprintf (stderr, "File is not in CodeTable format\n");
    fclose (ifile);
    return(-1);
  }

  /* Read CodeTable Version Flag */
  if (fread ((char *)(&ver), sizeof (int), 1, ifile) != 1)  {
    fprintf (stderr, "Codetable File read Error:%s\n", file_name);
    fclose (ifile);
    return(-1);
  }

  if (ver != CODETABLE_VERSION) {
    fprintf (stderr, "File is not in correct Version Number\n");
    fclose (ifile);
    return(-1);
  }

  if (fread((char *)hztbl, sizeof(CodeTableStruct), 1, ifile) == 0) {
    fclose (ifile);
    return(-1);
  }

  /* malloc memory for codetable information */
  hztbl->nodeList = (tableNode *)calloc(hztbl->sizeNodeList,sizeof(tableNode));
  hztbl->hzList = (unsigned char *)calloc(hztbl->sizeHZList, sizeof(unsigned char));
  hztbl->keyprompt = (keyPrompt *)calloc(MAX_USEDCODES_NUM, sizeof(keyPrompt));
  hztbl->functionkey = (functionKey *)calloc(MAX_FUNCTIONKEY_NUM, sizeof(functionKey));
  if ((! hztbl->hzList) || (! hztbl->nodeList) ||
      (! hztbl->keyprompt) || (! hztbl->functionkey)) {
    fclose (ifile);
    return(-1);
  }

  if ((fread ((char *)(hztbl->nodeList), sizeof(tableNode),
	      (int)hztbl->sizeNodeList, ifile) != hztbl->sizeNodeList) ||
      (fread ((char *)hztbl->hzList, sizeof(unsigned char),
	      (int)(hztbl->sizeHZList), ifile) != hztbl->sizeHZList))
    {
      fclose (ifile);
      return(-1);
    }
	
  if (GETBIT(hztbl->bSectionsFlag, KEYPROMPT_SECTION)) {
    if (fread((char *)(&(hztbl->keyprompt[0])), MAX_USEDCODES_NUM, 
	      sizeof(keyPrompt), ifile) != sizeof(keyPrompt))
      {
	fclose (ifile);
	return(-1);
      }
  }

  if (GETBIT(hztbl->bSectionsFlag, FUNCTIONKEY_SECTION)) {
    if (fread((char *)(&(hztbl->functionkey[0])), MAX_FUNCTIONKEY_NUM, 
	      sizeof(functionKey), ifile) != sizeof(functionKey))
      {
	fclose (ifile);
	return(-1);
      }
  }

  for (i=0; i<MAX_USEDCODES_NUM; i++) {
    if (hztbl->keyprompt[i].prompt[0] == 0) {
      hztbl->keyprompt[i].prompt[0] = i;
      hztbl->keyprompt[i].prompt[1] = 0;
    }
  }

  fclose (ifile);
  return(0);
}

/*
 * UnloadCodeTable -- unload the input table, free the memory
 */
void UnloadCodeTable(hztbl)
     CodeTableStruct *hztbl;
{
  if (! hztbl) return;

  if (hztbl->nodeList)  free ((char *)(hztbl->nodeList));
  if (hztbl->hzList)  free ((char *)(hztbl->hzList));
  if (hztbl->keyprompt)  free ((char *)(hztbl->keyprompt));
  if (hztbl->functionkey)  free ((char *)(hztbl->functionkey));
}

/*********************** Traversal in Codetable Node ***********************/
/* Search Context */
#define PGC_INTERMEDIATE_MAX (256)
struct search_state_;
typedef struct search_context_ {
  CodeTableStruct *hztbl;
  int		  *inbuf;
  size_t	  inlen;
  struct search_state_  *intmd[PGC_INTERMEDIATE_MAX];
} search_context;

/* Search State */
typedef struct search_state_ {
  search_context *ctx;
  int		*inptr;
  size_t 	inleft;
  int   	*lastback;
  int		*intrslstart;
  pgc_trans_rec *trsl;
  tableNode 	*curnode;
  int		result;
  struct search_state_ *next;
} search_state;

typedef search_state *search_state_list;

static int root_search( search_state *st, search_state_list *lst );
static int node_search( search_state *st, search_state_list *lst );
static int translate( search_state *st, search_state_list *translated );
static int transit( search_state *st, search_state_list *transited, search_state_list *untransited );
static int has_same_intermediate( search_context *ctx, search_state *st );
static int add_intermediate( search_context *ctx, search_state *st );
static int eval_states( search_state_list lst, pgc_trans_rec ** result);

static int set_translation( search_state *, unsigned char *, int size );
static int get_next_input_code( search_state * );
static int peek_next_input_code( search_state * );
static int input_back( search_state * st );
static tableNode * get_next_node( search_state *, int, int );
static int has_next_node( search_state * );
     search_state *st;
static int is_root_node( search_state * );
static int is_leaf_node( search_state * );
static void initialize_context( search_context *ctx, CodeTableStruct *hztbl, int * inbuf, size_t inlen );
static void destruct_context( search_context *ctx );
static search_state * create_state( search_context * ctx );
static search_state * dup_state( search_state * );
static void destroy_state( search_state * );
static search_state_list append_states( search_state_list, search_state_list );
static void destroy_state_list( search_state_list );
static search_state * next_state( search_state_list, search_state *);
static void print_state( search_state * st, FILE * );

int
codetable_search(hztbl, inbuf, inlen, result )
     CodeTableStruct *hztbl;
     int *inbuf;
     size_t inlen;
     pgc_trans_rec ** result;
{
  int ret = -1;
  search_state_list lst = NULL;
  search_context ctx;
  initialize_context( &ctx, hztbl, inbuf, inlen );
  search_state *first = create_state( &ctx );
  if( first == NULL ){
    return -1;
  }
  if( root_search( first, &lst ) != 0 ){
    goto end;
  }else{
#ifdef DEBUG
    {
      search_state *st;
      size_t cc=0;
      for( st = next_state( lst, NULL ); st != NULL ; st = next_state( lst, st ) ){
	print_state( st, stderr );
	cc++;
      }
      fprintf( stderr, "total %d states\n",cc );
    }
#endif
    ret = eval_states( lst, result );
  }
 end:
  destroy_state( first );
  destroy_state_list( lst );
  destruct_context( &ctx );
  return(ret);
}

static int
root_search( st, lst )
     search_state * st;
     search_state_list *lst;
{
  *lst = NULL;
  st->curnode = &st->ctx->hztbl->nodeList[0];
  int code = peek_next_input_code( st );
  if( code == PGC_ENDOFTEXT || code == 0 ){
    get_next_input_code( st );
    search_state * newst = dup_state(st);
    if( newst == NULL ) return -1;
    if( code == PGC_ENDOFTEXT ){
      newst->result = PGC_ACCEPTED;
    }else{
      newst->result = PGC_INPROCESS;
    }
    *lst = append_states( *lst, newst );
    return 0;
  }else{
    return node_search( st, lst );
  }
}

static int
node_search( st, lst )
     search_state * st;
     search_state_list *lst;
{
  int res=0;
  search_state_list translated=NULL, transited=NULL, untransited=NULL ;
  *lst = NULL;

  res = translate( st, &translated );
  if( res != 0 ){
    return res;
  }
  if( translated != NULL ){
    search_state *tst;
    for( tst = next_state(translated,NULL) ; tst != NULL ; tst = next_state(translated,tst) ){
      search_state_list rst;
      if( has_same_intermediate( tst->ctx, tst ) ){ // decrease redundant transition
	continue;
      }
      if( (res = add_intermediate( tst->ctx, tst )) != 0 ){
	goto err;
      }
      if( (res = root_search( tst, &rst )) != 0 ){
	goto err;
      }
      *lst = append_states( *lst, rst );
    }
    destroy_state_list( translated );
    translated = NULL;
  }

  res = transit( st, &transited, &untransited );
  if( res != 0 ){
    return res;
  }
  *lst = append_states( *lst, untransited );
  if( transited != NULL ){
    search_state * tst;
    for( tst = next_state(transited,NULL) ; tst != NULL ; tst = next_state(transited,tst) ){
      search_state_list rst;
      if( (res = node_search( tst, &rst )) != 0 ){
	goto err;
      }
      *lst = append_states( *lst, rst );
    }
    destroy_state_list( transited );
    transited = NULL;
  }
end:
  return res;
err:
  if( res == 0 ){
    res = -1;
  }
  destroy_state_list( translated );
  destroy_state_list( transited );
  destroy_state_list( untransited );
  destroy_state_list( *lst );
  *lst = NULL;
  goto end;
}

static int
translate( st, translated )
     search_state *st;
     search_state_list *translated;
{
  int count,ii,hzlen;
  tableNode * node = st->curnode;
  CodeTableStruct *hztbl = st->ctx->hztbl;
  *translated = NULL;
  count = node->num_HZchoice;
  if( count == 0 ){
    return 0;
  }
  for( ii = 0 ; ii < count ; ii++ ){
    search_state *newst;
    unsigned char *hzptr = hztbl->hzList + node->pos_HZidx;
    if (*hzptr == HZ_PHRASE_TAG) {
      hzlen = *(hzptr + 1);
      hzptr += 2;
    } else {
      hzlen = get_char_len_by_encodeid(hztbl->Encode, hzptr);
    }
    newst = dup_state( st );
    if( newst == NULL ){
      return -1;
    }
    set_translation( newst, hzptr, hzlen );
    hzptr += hzlen;
    *translated = append_states( *translated, newst );
  }
  return 0;
}

static int
transit( st, transited, untransited )
     search_state *st;
     search_state_list *transited;
     search_state_list *untransited;
{
  int ii;
  *transited = NULL;
  *untransited = NULL;
  int code = get_next_input_code( st );
  if( code == 0 ){
    if( has_next_node( st ) ){
      search_state *s = dup_state(st);
      if( s == NULL )	return -1;
      *untransited = append_states( *untransited, s );
    }
    return 0;
  }
  for( ii = 0 ; ; ii++ ){
    tableNode *node;
    search_state *newst;
    node = get_next_node( st, code, ii );
    if( node == NULL ){
      if( ii == 0 ){
	search_state *errst = dup_state(st);
	if( errst == NULL )	return -1;
	errst->result = PGC_REJECTED;
	*untransited = append_states( *untransited, errst );
      }
      break;
    }
    newst = dup_state( st );
    if( newst == NULL ){
      goto err;
    }
    newst->curnode = node;
    *transited = append_states( *transited, newst );
  }
  return 0;
 err:
  destroy_state_list( *transited );
  *transited = NULL;
  destroy_state_list( *untransited );
  *untransited = NULL;
  return -1;
}

static int
has_same_intermediate( ctx, st )
  search_context* ctx;
  search_state *st;
{
  size_t ii;
  for( ii = 0 ; ii < PGC_INTERMEDIATE_MAX ; ii++ ){
    search_state *cur = ctx->intmd[ii];
    if( cur == NULL ){
      break;
    }
    if( cur->inptr == st->inptr &&
	pgc_trans_strcmp( cur->trsl, st->trsl ) == 0 ){
      return 1;
    }
  }
  return 0;
}

static int
add_intermediate( ctx, st )
  search_context* ctx;
  search_state *st;
{
  size_t ii;
  for( ii = 0 ; ii < PGC_INTERMEDIATE_MAX ; ii++ ){
    if( ctx->intmd[ii] == NULL ){
      search_state *newst = dup_state( st );
      if( newst == NULL ){
	return -1;
      }
      ctx->intmd[ii] = newst;
      break;
    }
  }
  return 0;
}

static int
eval_states( lst, result )
     search_state_list lst;
     pgc_trans_rec ** result;
{
  int ret = PGC_REJECTED;
  pgc_trans_rec *first=NULL,*last=NULL;
  search_state *st;

  for( st = next_state( lst, NULL ) ; st != NULL; st = next_state( lst, st ) ){
    if( st->result == PGC_ACCEPTED ){
      ret = PGC_ACCEPTED;
      break;
    }
  }
  if( ret != PGC_ACCEPTED ){
    for( st = next_state( lst, NULL ) ; st != NULL; st = next_state( lst, st ) ){
      if( st->result == PGC_INPROCESS ){
	ret = PGC_INPROCESS;
	break;
      }
    }
  }
  for( st = next_state( lst, NULL ); st != NULL ; st = next_state( lst, st ) ){
    if( st->result == ret ){
      if( first == NULL ){
	first = st->trsl;
	last = first;
      }else{
	last->next = st->trsl;
	last = last->next;
      }
      st->trsl = NULL;
    }
  }
  if( last != NULL ){
    last->next = NULL;
  }
  *result = first;
  return ret;
}

static tableNode *
get_next_node( st, code, index )
     search_state *st;
     int code;
     int index;
{
  int ii,c=0;
  tableNode *node = &(st->ctx->hztbl->nodeList[st->curnode->pos_NextKey]);
  for( ii = 0; ii < st->curnode->num_NextKeys ; ii++, node++ ){
    if( node->key == code ){
      if( c == index ){
	return node;
      }
      c++;
    }
  }
  return NULL;
}

static int
has_next_node( st )
     search_state *st;
{
  return st->curnode->num_NextKeys > 0 ? 1 : 0;
}

static int
is_root_node( st )
     search_state * st;
{
  return st->curnode == &st->ctx->hztbl->nodeList[0] ? 1 : 0;
}


static int
is_leaf_node( st )
     search_state * st;
{
  return st->curnode->num_NextKeys == 0 ? 1 : 0;
}


static int
get_next_input_code( st )
     search_state * st;
{
  if( st->inleft == 0 ){
    return 0;
  }
  st->inleft--;
  return *st->inptr++;
}

static int
input_back( st )
     search_state * st;
{
  if( st->lastback >= st->inptr - 1 ){
    // protection against reading forever.
    assert_("phonogram_conv>input_back(). input back may cause forever reading.");
    return -1;
  }
  st->inleft++;
  st->inptr--;
  if( st->intrslstart > st->inptr ){
    st->intrslstart = st->inptr;
  }
  st->lastback = st->inptr;
  return 0;
}

static int
peek_next_input_code( st )
     search_state * st;
{
  if( st->inleft == 0 ){
    return 0;
  }
  return *st->inptr;
}

static int set_translation( st, p, size )
     search_state * st;
     unsigned char *p;
     int size;
{
  int ret;
  pgc_trans_unit_rec unit;
  search_context *ctx;
  if( st->trsl == NULL ){
    st->trsl = pgc_trans_create();
    if( st->trsl == NULL ){
      return -1;
    }
  }
  ctx = st->ctx;
  // Japanese `sokuon' 
  // ex) kko
  while( *(p+size-1) == PGC_BACK ){
    input_back( st );
    size--;
  }
  ret = set_translation_unit( &unit, p, size, 
			      st->intrslstart - ctx->inbuf, st->inptr - ctx->inbuf );
  if( ret != 0 ){
    return ret;
  }
  st->trsl = pgc_trans_add( st->trsl, &unit );
  if( st->trsl == NULL ){
    return -1;
  }
  st->intrslstart = st->inptr;
  return 0;
}

static void
initialize_context( ctx, hztbl, inbuf, inlen )
  search_context *ctx;
  CodeTableStruct *hztbl;
  int * inbuf;
  size_t inlen;
{
  ctx->hztbl = hztbl;
  ctx->inbuf = inbuf;
  ctx->inlen = inlen;
  memset( ctx->intmd, 0, sizeof(ctx->intmd) );
}

static void
destruct_context( ctx )
  search_context *ctx;
{
  size_t ii;
  for( ii = 0 ; ii < PGC_INTERMEDIATE_MAX ; ii++ ){
    if( ctx->intmd[ii] != NULL ){
      destroy_state( ctx->intmd[ii] );
      ctx->intmd[ii] = NULL;
    }
  }
}

static search_state *
create_state( ctx )
  search_context *ctx;
{
  search_state * st = (search_state *)malloc( sizeof(*st) );
  if( st == NULL ){
    return NULL;
  }
  memset( st, 0, sizeof(*st) );
  st->ctx = ctx;
  st->inptr = ctx->inbuf;
  st->intrslstart = ctx->inbuf;
  st->inleft = ctx->inlen;
  st->trsl = pgc_trans_create();
  return st;
}

static search_state *
dup_state( st )
     search_state * st;
{
  search_state *newst;
  if( st == NULL ){
    return NULL;

  }
  newst = (search_state*)malloc( sizeof(*newst) );
  if( newst == NULL ){
    return NULL;
  }
  memcpy( newst, st, sizeof(*newst) );
  newst->trsl = pgc_trans_dup( st->trsl );
  newst->next = NULL;
  return newst;
}

static void
destroy_state( st )
  search_state * st;
{
  if( st != NULL ){
    pgc_trans_destroy( st->trsl );
    free( st );
  }
}

static void
destroy_state_list( stlst )
  search_state_list stlst;
{
  search_state * st;
  for( st = stlst ; st != NULL; ){
    search_state * s = st->next;
    destroy_state( st );
    st = s;
  }
}

static search_state_list
append_states( to, lst )
     search_state_list to;
     search_state_list lst;
{
  search_state * last;
  if( to == NULL ){
    return lst;
  }
  for( last = to ; last->next != NULL ; last = last->next ){
    ;
  }
  last->next = lst;
  return to;
}

static search_state *
next_state( lst, st )
     search_state_list lst;
     search_state * st;
{
  if( st == NULL ){
    return lst;
  }else{
    return st->next;
  }
}


static void assert_( 
char * fmt,
... 
)
{
  static char *pre = "ASSERT!";
  char *p,*pa = NULL;
  char buf[256];
  va_list va;
  va_start( va, fmt );

  if( strlen(fmt) + strlen(pre) > sizeof(buf) - 2 ){
    pa = (char*)malloc( strlen(fmt) + strlen(pre) + 2 );
    p = pa;
  }else{
    p = buf;
  }
  strcpy( p, pre );
  strcat( p, fmt );
  strcat( p, "\n" );
  vfprintf(stderr, p, va );
  va_end( va );
  free( pa );  		      
}

static void print_state( st, fp )
     search_state * st;
     FILE * fp;
{
  int ii;
  iconv_t ic;
  char *p = NULL;
  char *nk, *res;
  search_context *ctx;
  fprintf( fp, "#### Phonogram Conv Search State\n");
  ctx = st->ctx;
  // about input string
  if( st->intrslstart < ctx->inbuf ){
    assert_( "phonogram_conv>>print_state() intrslstart < inbuf" );
    return;
  }
  if( st->inptr < ctx->inbuf ){
    assert_("phonogram_conv>>print_state() inptr < instart" );
    return;
  }
  p = (char*)malloc( st->inptr - ctx->inbuf + 2 );
  if( p != NULL ){
    for( ii = 0; ii < st->intrslstart - ctx->inbuf ; ii++ ){
      p[ii] = ctx->inbuf[ii];
    }
    p[ii] = '|';
    for( ; ii < st->inptr - ctx->inbuf ; ii++ ){
      p[ii+1] = ctx->inbuf[ii];
    }
    p[ii+1] = 0;
  }
  fprintf( fp, "\tinput:%s (instart:%x inptr:%x inleft:%x intrslstart:%x)\n",
	  p, (int)ctx->inbuf, (int)st->inptr, st->inleft, (int)st->intrslstart);
  free( p );

  // about status
  if( is_root_node( st ) ){
    nk = "ROOT";
  }else if( is_leaf_node(st) ){
    nk = "LEAF";
  }else{
    nk = "GROUP";
  }
  switch( st->result ){
  case PGC_INPROCESS: res = "IN TRANSIT"; break;
  case PGC_ACCEPTED: res = "ACCEPTED"; break;
  case PGC_REJECTED: res = "REJECTED"; break;
  default: res="Something wrong"; break;
  }
  fprintf( fp, "\tnode:%s state:%s\n", nk, res );

  // about output
  if( st->trsl != NULL ){
    fprintf( fp, "\tTranslations count:%d\n", st->trsl->len );
    for( ii = 0 ; ii < st->trsl->len; ii++ ){
      pgc_trans_unit_rec * unit = st->trsl->data+ii;
      unsigned char in[31];
      unsigned char euc[PGC_TRANS_LENMAX*PGC_TRANS_CHARMAX*3+3];
      unsigned char str[PGC_TRANS_LENMAX*PGC_TRANS_CHARMAX+1];
      int jj;
      if( unit->src_e < unit->src_s ){
	assert_( "phonogram_conv>>print_state() src_e < src_s" );
	return;
      }
      int size = unit->src_e - unit->src_s;
      for( jj = 0; jj < size && jj < 27 ; jj++ ){
	in[jj] = *(ctx->inbuf + unit->src_s + jj);
      }
      if( jj < size ){
	char * cmark = "...";
	strcpy( (char*)in + jj, cmark );
	jj += strlen(cmark);
      }
      in[jj] = 0;

      ic = iconv_open( "EUC-JP", "UTF-8" );
      if( ic == (iconv_t)-1 ){
	euc[0] = 0;
      }else{
	char *inbuf,*outbuf;
	size_t inbytesleft, outbytesleft;
	outbuf = (char*)euc;
	inbuf = (char*)unit->str;
	inbytesleft = unit->bytes;
	outbytesleft = sizeof(euc) - 3 ;
	if( iconv( ic, &inbuf, &inbytesleft, &outbuf, &outbytesleft ) < 0 ){
	  euc[0] = 0;
	}else{
	  *outbuf = 0;
	}
	iconv_close(ic);
      }
      memcpy( str, unit->str, unit->bytes );
      str[unit->bytes] = 0;
      fprintf( fp, "\toutput:\"%s\" \"%s\" ", str, euc );
      for( jj = 0; jj < unit->bytes ; jj++ ){
	fprintf( fp, "%02x", unit->str[jj] );
	if( jj < unit->bytes - 1 ){
	  fprintf( fp, "," );
	}
      }
      fprintf( fp, " (len:%d)", unit->bytes );
      fprintf( fp, "  source:\"%s\" (start:%x end:%x)", in, unit->src_s, unit->src_e );
      fprintf( fp, "\n");
    }
  }
}
