/*

Copyright (c) 2006 Kenji ABE <abek@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#include <stdlib.h>
#include <string.h>

#include "convdef.h"
#include "imcommit.h"
#include "imconv.h"
#include "imlookup.h"
#include "impreedit.h"
#include "imtext.h"
#include "utfcsdef.h"


#define N_CHOICE_PER_WINDOW (10)
#define N_CHOICE_COLUMNS (1)
#define N_CHOICE_ROWS (10)


static void lookup_choice_sub(iml_session_t *session, iml_inst **inst_list);

static void lookup_choice_select_sub(iml_session_t *session, int index, iml_inst **inst_list);

static void make_lookup_choice_draw_inst(iml_session_t *session, iml_inst **inst_list);

static IMText *make_label_text(iml_session_t *session, int index);


static const char LOOKUP_CHOICE_CHARS[] = "1234567890";
static const UTFCHAR LOOKUP_CHOICE_TITLE_PREFIX[] = {0x5019, 0x88DC, 0x9078, 0x629E, '[', 0x0000};
static const UTFCHAR LOOKUP_CHOICE_TITLE_SEPARATOR[] = {'/', 0x0000};
static const UTFCHAR LOOKUP_CHOICE_TITLE_POSTFIX[] = {']', 0x0000};

#define LOOKUP_CHOICE_TITLE_MAX (32) /* > 5 + 10 + 1 + 10 + 1 + 1 */

Bool
handle_lookup_choice_prev(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
RETRY:
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       break;

      case IM_STATE_EDITING:
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
       if (change_im_state(session, IM_STATE_CONVERSION, inst_list, True)) {
	   goto RETRY;
       }
       break;

      case IM_STATE_CONVERSION:
      case IM_STATE_PREDICTION:
       if (psd->lookup_choice_p) {
	   psd->lookup_choice_cur_index--;
	   if (psd->lookup_choice_cur_index < 0) {
	       psd->lookup_choice_cur_index = psd->lookup_choice_count -1;
	   }

	   lookup_choice_sub(session, inst_list);
	   return True;
       }

       if (im_conv_cur_select_prev(psd->im_conv)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }

       start_lookup_choice(session, inst_list);
       return True;

      default:
       ASSERT(0); //cannot happen
       break;
    }

    return False;
}

Bool
handle_lookup_choice_next(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

RETRY:
    switch (psd->im_state) {
      case IM_STATE_OFF:
      case IM_STATE_EDIT_EMPTY:
       break;

      case IM_STATE_EDITING:
      case IM_STATE_LETTER_TYPE_CONVERSION:
      case IM_STATE_CHANGE_SEG:
       if (change_im_state(session, IM_STATE_CONVERSION, inst_list, True)) {
	   goto RETRY;
       }
       break;

      case IM_STATE_CONVERSION:
      case IM_STATE_PREDICTION:
       if (psd->lookup_choice_p) {
	   psd->lookup_choice_cur_index++;
	   if (psd->lookup_choice_count <= psd->lookup_choice_cur_index) {
	       psd->lookup_choice_cur_index = 0;
	   }

	   lookup_choice_sub(session, inst_list);
	   return True;
       }

       if (im_conv_cur_select_next(psd->im_conv)) {
	   make_preedit_draw_inst_from_text(
	       session,
	       im_conv_make_preedit_text(psd->im_conv, session),
	       im_conv_cur_pos(psd->im_conv),
	       inst_list
	   );
       }

       start_lookup_choice(session, inst_list);
       return True;

      default:
       ASSERT(0); //cannot happen
       break;
    }

    return False;
}

Bool
handle_lookup_choice_prev_page(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->lookup_choice_p &&
	psd->lookup_choice_count > N_CHOICE_PER_WINDOW) {
	psd->lookup_choice_list_top_index -= N_CHOICE_PER_WINDOW;
	if (psd->lookup_choice_list_top_index < 0) {
	    psd->lookup_choice_list_top_index = ((psd->lookup_choice_count - 1) / N_CHOICE_PER_WINDOW) * N_CHOICE_PER_WINDOW;
	}

	psd->lookup_choice_cur_index = psd->lookup_choice_list_top_index;

	lookup_choice_sub(session, inst_list);
	return True;
    }

    return False;
}

Bool
handle_lookup_choice_next_page(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->lookup_choice_p &&
	psd->lookup_choice_count > N_CHOICE_PER_WINDOW) {
	psd->lookup_choice_list_top_index += N_CHOICE_PER_WINDOW;
	if (psd->lookup_choice_count <= psd->lookup_choice_list_top_index) {
	    psd->lookup_choice_list_top_index = 0;
	}

	psd->lookup_choice_cur_index = psd->lookup_choice_list_top_index;

	lookup_choice_sub(session, inst_list);
	return True;
    }

    return False;
}

Bool
handle_lookup_choice_select_current(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->lookup_choice_p) {
	lookup_choice_select_sub(session, psd->lookup_choice_cur_index, inst_list);
	return True;
    }
    return False;
}

Bool
handle_lookup_choice_select_by_choice(
    iml_session_t *session,
    IMKeyEventStruct *key_event,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

    if (psd->lookup_choice_p) {
	char *p = strchr(LOOKUP_CHOICE_CHARS, key_event->keyChar);
	if (p) {
	    lookup_choice_select_sub(session, psd->lookup_choice_list_top_index + (p - LOOKUP_CHOICE_CHARS), inst_list);
	    return True;
	}
    }
    return False;
}

static
void
lookup_choice_sub(
    iml_session_t *session,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    ASSERT((psd->im_state == IM_STATE_CONVERSION) || (psd->im_state == IM_STATE_PREDICTION));
    ASSERT(psd->lookup_choice_p);
    if (im_conv_cur_select_cand(psd->im_conv, psd->lookup_choice_cur_index)) {
	make_preedit_draw_inst_from_text(
	    session,
	    im_conv_make_preedit_text(psd->im_conv, session),
	    im_conv_cur_pos(psd->im_conv),
	    inst_list
	);
    }
    make_lookup_choice_draw_inst(session, inst_list);
}

static
void
lookup_choice_select_sub(
    iml_session_t *session,
    int index,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);

    ASSERT((psd->im_state == IM_STATE_CONVERSION) || (psd->im_state == IM_STATE_PREDICTION));
    ASSERT(psd->lookup_choice_p);
    ASSERT(0 <= index);
    ASSERT(index < psd->lookup_choice_count);

    if (im_conv_cur_select_cand(psd->im_conv, index)) {
	make_preedit_draw_inst_from_text(
	    session,
	    im_conv_make_preedit_text(psd->im_conv, session),
	    im_conv_cur_pos(psd->im_conv),
	    inst_list
	);
    }

    end_lookup_choice(session, inst_list);

    if (im_conv_cur_move_next(psd->im_conv)) {
	make_preedit_draw_inst_from_text(
	    session,
	    im_conv_make_preedit_text(psd->im_conv, session),
	    im_conv_cur_pos(psd->im_conv),
	    inst_list
	);
    } else {
	handle_commit(session, NULL, inst_list);
    }
}

void
start_lookup_choice(
    iml_session_t *session,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (!psd->lookup_choice_p) {
	if ((psd->im_state == IM_STATE_CONVERSION) ||
	    (psd->im_state == IM_STATE_PREDICTION)) {
	    iml_methods_t *mtbl = METHODS_TBL(session);
	    IMLookupStartCallbackStruct *luscs;
	    LayoutInfo* li;
	    iml_inst *inst;

	    psd->lookup_choice_p = True;

	    psd->lookup_choice_cur_index = im_conv_cur_cand_index(psd->im_conv);
	    psd->lookup_choice_count = im_conv_cur_cand_count(psd->im_conv);
	    psd->lookup_choice_list_top_index = (psd->lookup_choice_cur_index / N_CHOICE_PER_WINDOW) * N_CHOICE_PER_WINDOW;

	    luscs = mtbl->iml_new(session, sizeof(IMLookupStartCallbackStruct));
	    memset(luscs, 0, sizeof(IMLookupStartCallbackStruct));

	    luscs->whoIsMaster = IMIsMaster;
	    luscs->IMPreference = li = mtbl->iml_new(session, sizeof(LayoutInfo));
	    memset(li, 0, sizeof(LayoutInfo));

	    li->choice_per_window = N_CHOICE_PER_WINDOW;
	    li->ncolumns = N_CHOICE_COLUMNS;
	    li->nrows = N_CHOICE_ROWS;
	    li->drawUpDirection = DrawUpHorizontally;
	    li->whoOwnsLabel = IMOwnsLabel;

	    inst = mtbl->iml_make_lookup_start_inst(session, luscs);
	    mtbl->iml_link_inst_tail(inst_list, inst);
	}
    }

    ASSERT(psd->lookup_choice_p);
    make_lookup_choice_draw_inst(session, inst_list);
}

void
end_lookup_choice(
    iml_session_t *session,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    if (psd->lookup_choice_p) {
	iml_methods_t *mtbl = METHODS_TBL(session);
	iml_inst *inst = mtbl->iml_make_lookup_done_inst(session);
	mtbl->iml_link_inst_tail(inst_list, inst);
	psd->lookup_choice_p = False;
    }
}

static
void
make_lookup_choice_draw_inst(
    iml_session_t *session,
    iml_inst **inst_list
)
{
    PrivateSessionData *psd = SESSION_DATA(session);
    iml_methods_t *mtbl = METHODS_TBL(session);
    iml_inst *inst;
    IMLookupDrawCallbackStruct *ludcs;
    IMChoiceObject *choice;
    int i;
    UTFCHAR buff[LOOKUP_CHOICE_TITLE_MAX], *p;

    ASSERT(psd->lookup_choice_p);
    ASSERT((psd->im_state == IM_STATE_CONVERSION) || (psd->im_state == IM_STATE_PREDICTION));
    ASSERT(psd->im_conv);

    ASSERT(0 <= psd->lookup_choice_cur_index);
    ASSERT(psd->lookup_choice_cur_index < psd->lookup_choice_count);
    psd->lookup_choice_list_top_index = (psd->lookup_choice_cur_index / N_CHOICE_ROWS) * N_CHOICE_ROWS;

    ludcs = mtbl->iml_new(session, sizeof(IMLookupDrawCallbackStruct));
    memset(ludcs, 0, sizeof(IMLookupDrawCallbackStruct));

    ludcs->n_choices = psd->lookup_choice_count - psd->lookup_choice_list_top_index;
    if (ludcs->n_choices > N_CHOICE_PER_WINDOW) {
	ludcs->n_choices = N_CHOICE_PER_WINDOW;
    }

    ludcs->choices = choice = mtbl->iml_new(session, sizeof(IMChoiceObject) * ludcs->n_choices);
    for (i=0; i<ludcs->n_choices; ++i, ++choice) {
	choice->label = make_label_text(session, i);

	choice->value = im_conv_make_cur_cand_text(psd->im_conv, psd->lookup_choice_list_top_index + i, session);
	if (ludcs->max_len < choice->value->char_length) {
	    ludcs->max_len = choice->value->char_length;
	}
    }

    ludcs->index_of_first_candidate = 0;
    ludcs->index_of_last_candidate = ludcs->n_choices -1;
    ludcs->index_of_current_candidate = psd->lookup_choice_cur_index - psd->lookup_choice_list_top_index;

    p = utfcscpy(buff, LOOKUP_CHOICE_TITLE_PREFIX);
    p = utfcsi2s(p, psd->lookup_choice_cur_index +1);
    p = utfcscpy(p, LOOKUP_CHOICE_TITLE_SEPARATOR);
    p = utfcsi2s(p, psd->lookup_choice_count);
    p = utfcscpy(p, LOOKUP_CHOICE_TITLE_POSTFIX);
    ludcs->title = make_im_text(session, buff, p - buff, 0);

    inst = mtbl->iml_make_lookup_draw_inst(session, ludcs);
    mtbl->iml_link_inst_tail(inst_list, inst);
}

static
IMText *
make_label_text(
    iml_session_t *session,
    int index
)
{
    UTFCHAR buff[1];
    if (0 <= index &&
	index < sizeof(LOOKUP_CHOICE_CHARS)/sizeof(LOOKUP_CHOICE_CHARS[0])) {
	buff[0] = LOOKUP_CHOICE_CHARS[index];
	return make_im_text(session, buff, 1, 0);
    }

    return NULL;
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
