/*

Copyright (c) 2006 Kenji ABE <abek@terre-sys.com>

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.

*/

#include <ctype.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>

#include "convdef.h"
#include "dynbuff.h"
#include "imcaret.h"
#include "iminput.h"
#include "impreedit.h"
#include "imtext.h"
#include "phonogram_conv/include/phonogram_conv.h"

#define INIT_BUFF_SIZE (4)

typedef struct {
    enum IM_INPUT_LETTER_TYPE type;
    const char *dict_name;
    const UTFCHAR status_char;
    Bool wide_p;
    Bool normalize_lower_p;
/*     Bool conversion_p; */
    pgc_descriptor pgc_conv;
} im_input_type_pack;

typedef struct im_input_seg_rec {
    struct im_input_seg_rec *prev;
    struct im_input_seg_rec *next;

    dyn_buffer_t raw_buff;
    int raw_len;
    int converted_len;

    dyn_buffer_t cooked_buff;
    int cooked_len;
} im_input_seg;

typedef struct im_input_rec {
    im_input_type_pack* input_type;

    im_input_seg* head;
    im_input_seg* tail;
    im_input_seg* cur;

    int whole_len;

    dyn_buffer_t buff;
} im_input;


static im_input_seg *im_input_create_seg(int raw_size, int cooked_size);
static void im_input_free_seg(im_input_seg *seg);

static int im_input_seg_offset(const im_input_seg *seg);
static int im_input_seg_len(const im_input_seg *seg);

static void im_input_insert_seg(im_input_t input, im_input_seg *new_seg, im_input_seg *prev, im_input_seg *next);
static void im_input_cut_seg(im_input_t input, im_input_seg *seg);

static void im_input_delete_sub(im_input_t input, im_input_seg *seg);

static Bool im_input_cur_move_sub(
    im_input_t input,
    im_input_seg *prev,
    im_input_seg *next
);
static Bool im_input_merge_cur_prev_seg(im_input *input);

static void im_input_convert_cur_seg(im_input *input, im_input_type_pack *type, int eot);

static IMText *im_input_make_text_sub(im_input_t input, iml_session_t *session, int deco);
/* static IMText *im_input_make_seg_text(im_input_t input, im_input_seg *seg, int deco, iml_session_t *session); */

static int im_input_get_string_to_buff(im_input_t input, UTFCHAR *buff, int buff_size);
static int im_input_get_seg_string_to_buff(im_input_seg *seg, UTFCHAR *buff, int buff_size);

static UTFCHAR *im_input_get_converted_string(
    im_input_t input,
    iml_session_t *session,
    enum IM_INPUT_LETTER_TYPE type,
    UTFCHAR *raw_string,
    int raw_len,
    int *len
);


static im_input_type_pack INPUT_TYPES[N_IM_INPUT_LETTER_TYPE] = {
	{
		IM_INPUT_LETTER_TYPE_ALPHA,
		NULL,
		'a',
		False,
		False,
/* 		False, */
		NULL,
	},
	{
		IM_INPUT_LETTER_TYPE_HIRAGANA,
		"hirazen.data",
		0x3042,
		True,
		True,
/* 		True, */
		NULL,
	},
	{
		IM_INPUT_LETTER_TYPE_KATAKANA,
		"katazen.data",
		0x30A2,
		True,
		True,
/* 		True, */
		NULL,
	},
 	{
		IM_INPUT_LETTER_TYPE_HW_KATAKANA,
		"katahan.data",
		0xFF71,
		False,
		True,
/* 		True, */
		NULL,
	},
	{
		IM_INPUT_LETTER_TYPE_FW_ALPHA,
		"alphazen.data",
		0xFF21,
		True,
		False,
/* 		False, */
		NULL,
	},
};

static const char DEFAULT_ENCODING[] = "UTF-16";


Bool
im_input_init(
    const char *dict_dir
)
{
    char buff[PATH_MAX];
    int i;

    for (i=0; i<N_IM_INPUT_LETTER_TYPE; ++i) {
	if (INPUT_TYPES[i].pgc_conv == NULL &&
	    INPUT_TYPES[i].dict_name) {
	    snprintf(buff, sizeof(buff), "%s/%s", dict_dir, INPUT_TYPES[i].dict_name);
	    INPUT_TYPES[i].pgc_conv = pgc_open(buff, (char*)DEFAULT_ENCODING);
	    if (INPUT_TYPES[i].pgc_conv == NULL) {
		return False;
	    }
	}
    }
    return True;
}

void
im_input_term(void)
{
    int i;
    for (i=0; i<N_IM_INPUT_LETTER_TYPE; ++i) {
	if (INPUT_TYPES[i].pgc_conv) {
	    pgc_close(INPUT_TYPES[i].pgc_conv);
	    INPUT_TYPES[i].pgc_conv = NULL;
	}
    }
}

im_input_t
im_input_create(void)
{
    im_input_t input = calloc(1, sizeof(im_input));
    if (input) {
	input->input_type = &INPUT_TYPES[IM_INPUT_LETTER_TYPE_HIRAGANA];
	input->head = input->tail = input->cur = im_input_create_seg(INIT_BUFF_SIZE, 0);
	if (input->cur) {
	    input->buff = dyn_buffer_create();
	    if (input->buff) {
		return input;
	    }
	}
	
	im_input_free(input);
    }
    return NULL;
}

void
im_input_free(
    im_input_t input
)
{
    if (input) {
	im_input_seg *seg = input->head, *next;
	while (seg) {
	    next = seg->next;
	    im_input_free_seg(seg);
	    seg = next;
	}

	if (input->buff) {
	    dyn_buffer_free(input->buff);
	}

	free(input);
    }
}

static
im_input_seg *
im_input_create_seg(
    int raw_size,
    int cooked_size
)
{
    im_input_seg *new_seg = calloc(1, sizeof(im_input_seg));
    if (new_seg) {
	if ((new_seg->raw_buff = dyn_buffer_create()) &&
	    DYN_BUFFER_ENSURE_SIZE(new_seg->raw_buff, UTFCHAR, raw_size) &&
	    (new_seg->cooked_buff = dyn_buffer_create()) &&
	    DYN_BUFFER_ENSURE_SIZE(new_seg->cooked_buff, UTFCHAR, cooked_size)) {
	    return new_seg;
	}

	im_input_free_seg(new_seg);
    }
    return NULL;
}

static
void
im_input_free_seg(
    im_input_seg *seg
)
{
    if (seg) {
	if (seg->raw_buff) {
	    dyn_buffer_free(seg->raw_buff);
	}
	if (seg->cooked_buff) {
	    dyn_buffer_free(seg->cooked_buff);
	}
	free(seg);
    }
}

Bool
im_input_empty_p(
    im_input_t input
)
{
    return input && input->whole_len <= 0;
}

Bool
im_input_wide_p(
    im_input_t input
)
{
    return input && input->input_type && input->input_type->wide_p;
}

UTFCHAR
im_input_get_status_char(
    im_input_t input
)
{
    if (input && input->input_type) {
	return input->input_type->status_char;
    }
    return ' ';
}

enum IM_INPUT_LETTER_TYPE
im_input_get_input_letter_type(
    im_input_t input
)
{
    if (input && input->input_type) {
	return input->input_type->type;
    }
    return IM_INPUT_LETTER_TYPE_ALPHA;
}

void
im_input_set_input_letter_type(
    im_input_t input,
    enum IM_INPUT_LETTER_TYPE new_type
)
{
    if (input && 0 <= new_type && new_type < N_IM_INPUT_LETTER_TYPE) {
	input->input_type = &INPUT_TYPES[new_type];
	ASSERT(input->input_type->type == new_type);
    }
}

void
im_input_convert_letter_type(
    im_input_t input,
    enum IM_INPUT_LETTER_TYPE type
)
{
    if (input && 0 <= type && type < N_IM_INPUT_LETTER_TYPE) {
	int len = 0;
	im_input_seg *seg;
	UTFCHAR *p, *p0;

	for (seg = input->head; seg; seg = seg->next) {
	    len += seg->raw_len;
	}

	if (!DYN_BUFFER_ENSURE_SIZE(input->buff, UTFCHAR, len)) {
	    return;
	}

	p = p0 = DYN_BUFFER_GET_BUFFER(input->buff, UTFCHAR);
	for (seg = input->head; seg; seg = seg->next) {
	    memmove(p, DYN_BUFFER_GET_BUFFER(seg->raw_buff, UTFCHAR), seg->raw_len * sizeof(UTFCHAR));
	    p += seg->raw_len;
	}

	im_input_clear(input);

	if (!DYN_BUFFER_ENSURE_SIZE(input->cur->raw_buff, UTFCHAR, len)) {
	    return;
	}

	p = DYN_BUFFER_GET_BUFFER(input->cur->raw_buff, UTFCHAR);
	memmove(p, p0, len * sizeof(UTFCHAR));
	input->cur->raw_len = len;
	input->whole_len = len;

	im_input_convert_cur_seg(input, &INPUT_TYPES[type], 1);
    }
}

void
im_input_clear(
    im_input_t input
)
{
    if (input && input->cur) {
	im_input_seg *seg = input->cur->prev, *tmp;
	while (seg) {
	    tmp = seg->prev;
	    im_input_free_seg(seg);
	    seg = tmp;
	}

	seg = input->cur->next;
	while (seg) {
	    tmp = seg->next;
	    im_input_free_seg(seg);
	    seg = tmp;
	}

	input->cur->prev =  input->cur->next = NULL;
	input->cur->raw_len =
	    input->cur->converted_len =
	    input->cur->cooked_len = 0;

	input->head = input->tail = input->cur;
	input->whole_len = 0;
    }
}

void
im_input_add_char(
    im_input_t input,
    int ch
)
{
    ASSERT(input);
    if (input && 
	input->cur && 
	DYN_BUFFER_ENSURE_SIZE(input->cur->raw_buff, UTFCHAR, input->cur->raw_len +1)) {
	DYN_BUFFER_GET_BUFFER(input->cur->raw_buff, UTFCHAR)[input->cur->raw_len] = ch;
	input->cur->raw_len++;
	input->whole_len++;

	im_input_convert_cur_seg(input, input->input_type, 0);
    }
}

void
im_input_break(
    im_input_t input
)
{
    ASSERT(input);
    im_input_convert_cur_seg(input, input->input_type, 1);

    ASSERT(input->cur);
    if (input->cur->raw_len > 0) {
	int len = input->cur->raw_len;
	im_input_seg *seg;
	UTFCHAR *src = DYN_BUFFER_GET_BUFFER(input->cur->raw_buff, UTFCHAR);

	if (input->cur->cooked_len > 0) {
	    seg = im_input_create_seg(input->cur->converted_len, input->cur->cooked_len);
	    if (seg == NULL) {
		return;
	    }

	    memmove(DYN_BUFFER_GET_BUFFER(seg->raw_buff, UTFCHAR), src, input->cur->converted_len * sizeof(UTFCHAR));
	    src += (seg->raw_len = seg->converted_len = input->cur->converted_len);

	    memmove(
		DYN_BUFFER_GET_BUFFER(seg->cooked_buff, UTFCHAR),
		DYN_BUFFER_GET_BUFFER(input->cur->cooked_buff, UTFCHAR),
		input->cur->cooked_len * sizeof(UTFCHAR)
	    );
	    seg->cooked_len = input->cur->cooked_len;
	    input->cur->cooked_len = 0;

	    im_input_insert_seg(input, seg, input->cur->prev, input->cur);
	}

	while (len > 0) {
	    seg = im_input_create_seg(1, 0);
	    if (seg == NULL) {
		break;
	    }

	    *DYN_BUFFER_GET_BUFFER(seg->raw_buff, UTFCHAR) = *src++;
	    seg->raw_len = 1;
	    --len;

	    im_input_insert_seg(input, seg, input->cur->prev, input->cur);
	}

	input->cur->raw_len = input->cur->cooked_len = 0;
    }
}

void
im_input_reinit(
    im_input_t input
)
{
    if (input && input->cur) {
	im_input_cut_seg(input, input->cur);
	im_input_insert_seg(input, input->cur, input->tail, NULL);
	im_input_merge_cur_prev_seg(input);
    }
}

void
im_input_delete_prev(
    im_input_t input
)
{
    if (input && input->cur) {
	if (input->cur->raw_len > 0) {
	    int org_len = im_input_seg_len(input->cur), new_len;

	    if (input->cur->converted_len < input->cur->raw_len) {
		input->cur->raw_len--;
	    } else {
		input->cur->raw_len =
		    input->cur->converted_len = 
		    input->cur->cooked_len = 0;
	    }

	    new_len = im_input_seg_len(input->cur);

	    input->whole_len -= org_len;
	    input->whole_len += new_len;

	    im_input_merge_cur_prev_seg(input);
	} else if (input->cur->prev) {
	    im_input_delete_sub(input, input->cur->prev);
	}
    }
}

void
im_input_delete_next(
    im_input_t input
)
{
    if (input && input->cur && input->cur->next) {
	im_input_delete_sub(input, input->cur->next);
    }
}

int
im_input_cur_pos(
    im_input_t input
)
{
    int offset = 0;
    if (input && input->cur) {
	offset = im_input_seg_offset(input->cur) + im_input_seg_len(input->cur);
    }
    return offset;
}

static
int
im_input_seg_offset(
    const im_input_seg *seg
)
{
    int offset = 0;
    if (seg) {
	seg = seg->prev;
	while (seg) {
	    offset += im_input_seg_len(seg);
	    seg = seg->prev;
	}
    }
    return offset;
}

static
int
im_input_seg_len(
    const im_input_seg *seg
)
{
    int len = 0;

    ASSERT(seg);

    if (seg->cooked_len > 0) {
	len += seg->cooked_len;
    }

    if (seg->converted_len < seg->raw_len) {
	len += seg->raw_len - seg->converted_len;
    }

    return len;
}

Bool
im_input_cur_move_to_head(
    im_input_t input
)
{
    if (input &&
	input->cur &&
	(input->cur != input->head ||
	 (input->cur->converted_len > 0))) {
	im_input_break(input);
	return im_input_cur_move_sub(input, NULL, input->head);
    }
    return False;
}

Bool
im_input_cur_move_to_tail(
    im_input_t input
)
{
    if (input && 
	input->cur &&
	input->cur != input->tail) {
	im_input_break(input);
	return im_input_cur_move_sub(input, input->tail, NULL);
    }
    return False;
}

Bool
im_input_cur_move_prev(
    im_input_t input
)
{
    if (input && 
	input->cur &&
	(input->cur->prev ||
	 (input->cur->converted_len > 0))) {
	im_input_break(input);

	ASSERT(input->cur->prev);
	return im_input_cur_move_sub(input, input->cur->prev->prev, input->cur->prev);
    }
    return False;
}

Bool
im_input_cur_move_next(
    im_input_t input
)
{
    if (input && 
	input->cur &&
	input->cur->next) {
	im_input_break(input);

	ASSERT(input->cur->next);
	return im_input_cur_move_sub(input, input->cur->next, input->cur->next->next);
    }
    return False;
}

static
void
im_input_insert_seg(
    im_input_t input,
    im_input_seg *new_seg,
    im_input_seg *prev,
    im_input_seg *next
)
{
    ASSERT(input);
    ASSERT(new_seg);

    if (prev) {
	prev->next = new_seg;
    } else {
	input->head = new_seg;
    }

    new_seg->prev = prev;
    new_seg->next = next;

    if (next) {
	next->prev = new_seg;
    } else {
	input->tail = new_seg;
    }

    input->whole_len += im_input_seg_len(new_seg);
}

static
void
im_input_cut_seg(
    im_input_t input,
    im_input_seg *seg
)
{
    im_input_seg *prev, *next;

    ASSERT(seg);
    prev = seg->prev;
    next = seg->next;

    seg->prev = seg->next = NULL;

    if (prev) {
	prev->next = next;
    } else {
	input->head = next;
    }

    if (next) {
	next->prev = prev;
    } else {
	input->tail = prev;
    }

    input->whole_len -= im_input_seg_len(seg);
}

static
void
im_input_delete_sub(
    im_input_t input,
    im_input_seg *seg
)
{
    ASSERT(input);
    ASSERT(seg);
    ASSERT(input->cur != seg);
    ASSERT(seg->prev || seg->next);

    im_input_cut_seg(input, seg);
    im_input_free_seg(seg);
}

static
Bool
im_input_cur_move_sub(
    im_input_t input,
    im_input_seg *prev,
    im_input_seg *next
)
{
    ASSERT(input);
    ASSERT(input->cur != prev);
    ASSERT(input->cur != next);

    im_input_cut_seg(input, input->cur);
    im_input_insert_seg(input, input->cur, prev, next);

    if (input->cur->raw_len > 0) {
	return True;
    }

    im_input_merge_cur_prev_seg(input);
    return False;
}

static
Bool
im_input_merge_cur_prev_seg(
    im_input *input
)
{
    ASSERT(input);
    ASSERT(input->cur);

    if (input->cur->raw_len <= 0 &&
	input->cur->prev &&
	input->cur->prev->raw_len > input->cur->prev->converted_len) {
	im_input_seg *seg = input->cur, *prev = input->cur->prev;
	im_input_cut_seg(input, seg);
	input->cur = prev;
	im_input_free_seg(seg);
	return True;
    }

    return False;
}

static
void
im_input_convert_cur_seg(
    im_input *input,
    im_input_type_pack *type,
    int eot
)
{
    ASSERT(input);
    ASSERT(input->cur);
    ASSERT(type);
    if (input->cur->raw_len > 0) {
	int len;
	im_input_seg *seg;
	UTFCHAR *src, *dest;

	len = input->cur->raw_len;
	src = DYN_BUFFER_GET_BUFFER(input->cur->raw_buff, UTFCHAR);

	if (type->pgc_conv == NULL) {
	    while (len > 0) {
		seg = im_input_create_seg(1, 1);
		if (seg == NULL) {
		    break;
		}

		memmove(DYN_BUFFER_GET_BUFFER(seg->raw_buff, UTFCHAR), src, sizeof(UTFCHAR));
		memmove(DYN_BUFFER_GET_BUFFER(seg->cooked_buff, UTFCHAR), src, sizeof(UTFCHAR));
		seg->raw_len = seg->converted_len = seg->cooked_len = 1;
		++src;
		--len;
		im_input_insert_seg(input, seg, input->cur->prev, input->cur);
	    }

	    if (len > 0) {
		dest = DYN_BUFFER_GET_BUFFER(input->cur->raw_buff, UTFCHAR);
		src = dest + input->cur->raw_len - len;
		memmove(dest, src, len * sizeof(UTFCHAR));
	    }
	    input->cur->raw_len = len;
	    input->cur->converted_len = input->cur->cooked_len = 0;
	} else {
	    UTFCHAR *src0;
	    int org_cur_len, r;
	    size_t raw_moved, i, count, rsp, rep, rlen, rsum, csp, cep, clen;
	    pgc_trans conv_result;

	    src0 = src;
	    if (type->normalize_lower_p) {
		UTFCHAR *dest0;
		int i;

		if (!DYN_BUFFER_ENSURE_SIZE(input->buff, UTFCHAR, len)) {
		    return;
		}

		dest = dest0 = DYN_BUFFER_GET_BUFFER(input->buff, UTFCHAR);
		for (i=0; i<len; ++i, ++src, ++dest) {
		    if (isascii(*src) && isupper(*src)) {
			*dest = tolower(*src);
		    } else {
			*dest = *src;
		    }
		}

		src = dest0;
	    }

	    org_cur_len = im_input_seg_len(input->cur);
	    raw_moved = 0;

	    while (len > 0) {
		rsum = 0;
		r = romaji_kana_convert(type->pgc_conv, src, len, eot, &conv_result);

		switch (r) {
		  case PGC_ACCEPTED:
		  case PGC_INPROCESS:
		  case PGC_REJECTED:
		   pgc_trans_unit_count(conv_result, &count);

		   if (count > 0) {
		       if (r == PGC_INPROCESS) {
			   --count;
		       }

		       for (i=0; i<count; ++i) {
			   pgc_trans_get_source_pos(conv_result, i, i, &rsp, &rep);
			   pgc_trans_get_string(conv_result, i, i, &csp, &cep, NULL, 0);

			   rlen = rep - rsp;
			   clen = (cep - csp) / sizeof(UTFCHAR);
			   seg = im_input_create_seg(rlen, clen);
			   if (seg == NULL) {
			       len = 0;
			       goto CLEAN_UP;
			   }

			   dest = DYN_BUFFER_GET_BUFFER(seg->raw_buff, UTFCHAR);
			   memcpy(dest, src0 + rsp, rlen * sizeof(UTFCHAR));
			   seg->raw_len = seg->converted_len = rlen;
			   rsum += rlen;

			   pgc_trans_get_string(
			       conv_result,
			       i,
			       i,
			       &csp,
			       &cep,
			       DYN_BUFFER_GET_BUFFER(seg->cooked_buff, unsigned char),
			       DYN_BUFFER_GET_SIZE(seg->cooked_buff, unsigned char)
			   );
			   seg->cooked_len = clen;
			   im_input_insert_seg(input, seg, input->cur->prev, input->cur);
		       }

		       if (r == PGC_INPROCESS) {
			   pgc_trans_get_source_pos(conv_result, count, count, &rsp, &rep);
			   pgc_trans_get_string(conv_result, count, count, &csp, &cep, NULL, 0);

			   rlen = rep - rsp;
			   clen = (cep - csp) / sizeof(UTFCHAR);

			   if (!DYN_BUFFER_ENSURE_SIZE(input->cur->cooked_buff, UTFCHAR, clen)) {
			       goto CLEAN_UP;
			   }
			   
			   input->cur->converted_len = rlen;

			   pgc_trans_get_string(
			       conv_result,
			       count,
			       count,
			       &csp,
			       &cep,
			       DYN_BUFFER_GET_BUFFER(input->cur->cooked_buff, unsigned char),
			       DYN_BUFFER_GET_SIZE(input->cur->cooked_buff, unsigned char)
			   );
			   input->cur->cooked_len = clen;
		       }
		   }

		   switch (r) {
		     case PGC_ACCEPTED:
		      input->cur->converted_len = input->cur->cooked_len = 0;
		      /* not break, but fall through. */
		     case PGC_INPROCESS:
		      raw_moved += rsum;
		      len = 0;
		      break;

		     default: /* case PGC_REJECTED: */
		      if (input->cur->cooked_len > 0) {
			  rlen = input->cur->converted_len;
			  clen = input->cur->cooked_len;
		      } else {
			  rlen = 1;
			  clen = 0;
		      }

		      seg = im_input_create_seg(rlen, clen);
		      if (seg == NULL) {
			  len = 0;
			  goto CLEAN_UP;
		      }

		      ASSERT(rlen > 0);
		      memmove(DYN_BUFFER_GET_BUFFER(seg->raw_buff, UTFCHAR), src0+rsum, rlen * sizeof(UTFCHAR));
		      seg->raw_len = rlen;

		      rsum += rlen;

		      if (clen > 0) {
			  memmove(
			      DYN_BUFFER_GET_BUFFER(seg->cooked_buff, UTFCHAR),
			      DYN_BUFFER_GET_BUFFER(input->cur->cooked_buff, UTFCHAR),
			      clen * sizeof(UTFCHAR)
			  );
			  seg->converted_len = rlen;
			  seg->cooked_len = clen;

			  input->cur->converted_len = input->cur->cooked_len = 0;
		      }

		      im_input_insert_seg(input, seg, input->cur->prev, input->cur);

		      raw_moved += rsum;
		      src += rsum;
		      src0 += rsum;
		      len -= rsum;
		      break;
		   }
		   break;

		  default:
		   len = 0;
		   break;
		}
CLEAN_UP:
		pgc_trans_destroy(conv_result);
	    }

	    if (raw_moved > 0) {
		if (input->cur->raw_len > raw_moved) {
		    dest = DYN_BUFFER_GET_BUFFER(input->cur->raw_buff, UTFCHAR);
		    src = dest + raw_moved;
		    memmove(dest, src, (input->cur->raw_len - raw_moved) * sizeof(UTFCHAR));
		}
		input->cur->raw_len -= raw_moved;
	    }

	    input->whole_len -= org_cur_len;
	    input->whole_len += im_input_seg_len(input->cur);
	}
    }
}

IMText *
im_input_make_preedit_text(
    im_input_t input,
    iml_session_t *session,
    int deco
)
{
    return im_input_make_text_sub(input, session, deco);
}

IMText *
im_input_make_commit_text(
    im_input_t input,
    iml_session_t *session
)
{
    return im_input_make_text_sub(input, session, 0);
}

static
IMText *
im_input_make_text_sub(
    im_input_t input,
    iml_session_t *session,
    int deco
)
{
    IMText *im_txt;

    ASSERT(input);
    ASSERT(session);

    im_txt = make_im_text(session, NULL, input->whole_len, deco);
    if (im_txt && input->whole_len > 0) {
	im_txt->char_length = im_input_get_string_to_buff(input, im_txt->text.utf_chars, input->whole_len);
    }

    return im_txt;
}

/* static */
/* IMText * */
/* im_input_make_seg_text( */
/*     im_input_t input, */
/*     im_input_seg *seg, */
/*     int deco, */
/*     iml_session_t *session */
/* ) */
/* { */
/*     IMText *im_txt; */
/*     int len; */

/*     ASSERT(input); */
/*     ASSERT(seg); */
/*     ASSERT(session); */

/*     im_txt = make_im_text(session, NULL, len = im_input_seg_len(seg), deco); */
/*     if (im_txt && len > 0) { */
/* 	im_txt->char_length = im_input_get_seg_string(seg, im_txt->text.utf_chars, len); */
/*     } */

/*     return im_txt; */
/* } */

UTFCHAR *
im_input_get_string(
    im_input_t input,
    iml_session_t *session,
    int *length
)
{
    UTFCHAR *p;
    int dummy;

    if (length == NULL) {
	length = &dummy;
    }

    p = NULL;
    *length = 0;
    if (input && session) {
	iml_methods_t *mtbl = METHODS_TBL(session);

	p = mtbl->iml_new(session, input->whole_len * sizeof(UTFCHAR));
	if (p) {
	    *length = im_input_get_string_to_buff(input, p, input->whole_len);
	}
    }

    return p;
}

static 
int
im_input_get_string_to_buff(
    im_input_t input,
    UTFCHAR *buff,
    int buff_size
)
{
    im_input_seg *seg;
    int len = 0, s;

    ASSERT(input);
    ASSERT(buff);

    for (seg = input->head; seg && buff_size > 0; seg = seg->next) {
	s = im_input_get_seg_string_to_buff(seg, buff, buff_size);
	buff += s;
	buff_size -= s;
	len += s;
    }

    return len;
}

static
int
im_input_get_seg_string_to_buff(
    im_input_seg *seg,
    UTFCHAR *buff,
    int buff_size
)
{
    UTFCHAR *p;
    int size;

    ASSERT(seg);
    ASSERT(buff);
    ASSERT(buff_size > 0);

    p = buff;

    if (seg->cooked_len > 0) {
	size = seg->cooked_len;
	if (size > buff_size) {
	    size = buff_size;
	}

	memcpy(p, DYN_BUFFER_GET_BUFFER(seg->cooked_buff, UTFCHAR), size * sizeof(UTFCHAR));

	p += size;
	buff_size -= size;
    }

    if (buff_size > 0) {
	size = seg->raw_len - seg->converted_len;
	if (size > 0) {
	    if (size > buff_size) {
		size = buff_size;
	    }

	    memcpy(p, DYN_BUFFER_GET_BUFFER(seg->raw_buff, UTFCHAR) + seg->converted_len, size * sizeof(UTFCHAR));

	    p += size;
	    buff_size -= size;
	}
    }

    return p - buff;
}

UTFCHAR *
im_input_get_converted_range_strig(
    im_input_t input,
    iml_session_t *session,
    int start,
    int end,
    enum IM_INPUT_LETTER_TYPE type,
    int *length
)
{
    UTFCHAR *converted = NULL;
    int dummy;
    if (length == NULL) {
	length = &dummy;
    }

    *length = 0;

    if (input && session && 0 <= type && type < N_IM_INPUT_LETTER_TYPE) {
	UTFCHAR *unconverted = im_input_get_raw_range_string(input, session, start, end, length);
	if (unconverted) {
	    converted = im_input_get_converted_string(input, session, type, unconverted, *length, length);
	}
    }
    return converted;
}

UTFCHAR *
im_input_get_raw_range_string(
    im_input_t input,
    iml_session_t *session,
    int start,
    int end,
    int *length
)
{
    UTFCHAR *str = NULL;
    int dummy;
    if (length == NULL) {
	length = &dummy;
    }

    *length = 0;

    if (input && session) {
	if (start < 0) {
	    start = 0;
	}

	if (start < input->whole_len) {
	    if (input->whole_len < end) {
		end = input->whole_len;
	    }
	    
	    if (start < end) {
		iml_methods_t *mtbl;
		im_input_seg *seg_start, *seg_end;
		int offset = 0, l = 0;
		UTFCHAR *p;

		seg_start = input->head;
		while (seg_start && offset < start) {
		    offset += im_input_seg_len(seg_start);
		    seg_start = seg_start->next;
		}

		seg_end = seg_start;
		while (seg_end && offset < end) {
		    offset += im_input_seg_len(seg_end);
		    l += seg_end->raw_len;
		    seg_end = seg_end->next;
		}

		mtbl = METHODS_TBL(session);
		str = p = mtbl->iml_new(session, (l+1) * sizeof(UTFCHAR));
		if (p) {
		    while (seg_start && seg_start != seg_end) {
			memmove(p, DYN_BUFFER_GET_BUFFER(seg_start->raw_buff, UTFCHAR), seg_start->raw_len * sizeof(UTFCHAR));
			p += seg_start->raw_len;
			seg_start = seg_start->next;
		    }
		    *p = 0x0000;
		    *length = l;
		}
	    }
	}
    }

    return str;
}

static
UTFCHAR *
im_input_get_converted_string(
    im_input_t input,
    iml_session_t *session,
    enum IM_INPUT_LETTER_TYPE type,
    UTFCHAR *raw_string,
    int raw_len,
    int *length
)
{
    iml_methods_t *mtbl;
    UTFCHAR *src, *p, *q;
    im_input_type_pack *input_type;
    int i, len, r;
    pgc_trans conv_result;
    size_t count, csp, cep, clen, rsp, rep, rlen;

    ASSERT(0 <= type && type < N_IM_INPUT_LETTER_TYPE);
    input_type = &INPUT_TYPES[type];

    ASSERT(length);
    if (input_type->pgc_conv == NULL) {
	*length = raw_len;
	return raw_string;
    }

    mtbl = METHODS_TBL(session);

    src = raw_string;
    if (input_type->normalize_lower_p) {
	p = src = mtbl->iml_new(session, raw_len * sizeof(UTFCHAR));
	q = raw_string;
	for (i=0; i<raw_len; ++i, ++p, ++q) {
	    if (isascii(*q) && isupper(*q)) {
		*p = tolower(*q);
	    } else {
		*p = *q;
	    }
	}
    }

    p = NULL;
    len = 0;
    while (raw_len > 0) {
	r = romaji_kana_convert(input_type->pgc_conv, src, raw_len, 1, &conv_result);

	switch (r) {
	  case PGC_ACCEPTED:
	  case PGC_INPROCESS:
	  case PGC_REJECTED:
	   pgc_trans_unit_count(conv_result, &count);

	   if (count > 0) {
	       pgc_trans_get_string(conv_result, 0, PGC_TRANS_LAST, &csp, &cep, NULL, 0);
	       clen = (cep - csp) / sizeof(UTFCHAR);
	       ASSERT(clen > 0);

	       if (!DYN_BUFFER_ENSURE_SIZE(input->buff, UTFCHAR, len + clen)) {
		   raw_len = 0;
		   break;
	       }

	       p = DYN_BUFFER_GET_BUFFER(input->buff, UTFCHAR) + len;
	       pgc_trans_get_string(conv_result, 0, PGC_TRANS_LAST, &csp, &cep, (unsigned char*)p, clen * sizeof(UTFCHAR));
	       len += clen;

	       pgc_trans_get_source_pos(conv_result, 0, count -1, &rsp, &rep);
	       rlen = rep - rsp;
	       src += rlen;
	       raw_string += rlen;
	       raw_len -= rlen;
	   }

	   switch (r) {
	     case PGC_REJECTED:
	      if (raw_len > 0) {
		  if (!DYN_BUFFER_ENSURE_SIZE(input->buff, UTFCHAR, len + 1)) {
		      raw_len = 0;
		      break;
		  }

		  p = DYN_BUFFER_GET_BUFFER(input->buff, UTFCHAR) + len;
		  *p = *raw_string;
		  ++len;
		  ++src;
		  ++raw_string;
		  --raw_len;
	      }
	      /* not break, but fall through. */
	     case PGC_ACCEPTED:
	      goto CLEAN_UP;

	     default: /*case PGC_INPROCESS: */
	      break;
	   }
	   /* not break, but fall through. */
	  default:
	   if (raw_len > 0) {
	       if (!DYN_BUFFER_ENSURE_SIZE(input->buff, UTFCHAR, len + raw_len)) {
		   raw_len = 0;
		   break;
	       }

	       p = DYN_BUFFER_GET_BUFFER(input->buff, UTFCHAR) + len;
	       memmove(p, raw_string, raw_len * sizeof(UTFCHAR));
	       len += raw_len;
	       src += raw_len;
	       raw_string += raw_len;
	       raw_len = 0;
	   }
	   break;
	}
CLEAN_UP:
	pgc_trans_destroy(conv_result);
    }

    if (len > 0) {
	p = mtbl->iml_new(session, (len + 1) * sizeof(UTFCHAR));
	if (p) {
	    memmove(p, DYN_BUFFER_GET_BUFFER(input->buff, UTFCHAR), len * sizeof(UTFCHAR));
	    p[len] = 0x0000;
	}
    }

    *length = len;
    return p;
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
