#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <poll.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include <iiimcf.h>

#include "commonIM.h"
#include "iiimpIM.h"
#include "iiimpAux.h"
#include "iiimpAuxProxy.h"

#include "XimpIm.h"
#include "Xlcint.h"


/* XAUX PROXY service number */
#define XAUX_PROXY_SERVICE_AUX_SETVALUE		(1)
#define XAUX_PROXY_SERVICE_IM_ID		(2)
#define XAUX_PROXY_SERVICE_IC_ID		(3)
#define XAUX_PROXY_SERVICE_DATA_SET		(4)
#define XAUX_PROXY_SERVICE_DATA_GET		(5)
#define XAUX_PROXY_SERVICE_DISPLAY		(6)
#define XAUX_PROXY_SERVICE_WINDOW		(7)
#define XAUX_PROXY_SERVICE_POINT		(8)
#define XAUX_PROXY_SERVICE_POINT_CARET		(9)
#define XAUX_PROXY_SERVICE_UTF16_MB		(10)
#define XAUX_PROXY_SERVICE_MB_UTF16		(11)
#define XAUX_PROXY_SERVICE_COMPOSE		(12)
#define XAUX_PROXY_SERVICE_COMPOSE_SIZE		(13)
#define XAUX_PROXY_SERVICE_DECOMPOSE		(14)
#define XAUX_PROXY_SERVICE_DECOMPOSE_FREE	(15)
#define XAUX_PROXY_SERVICE_REGISTER_X_FILTER	(16)
#define XAUX_PROXY_SERVICE_UNREGISTER_X_FILTER	(17)
#define XAUX_PROXY_SERVICE_SERVER		(18)
#define XAUX_PROXY_SERVICE_CLIENT_WINDOW	(19)
#define XAUX_PROXY_SERVICE_FOCUS_WINDOW		(20)
#define XAUX_PROXY_SERVICE_SCREEN_NUMBER	(21)
#define XAUX_PROXY_SERVICE_POINT_SCREEN		(22)
#define XAUX_PROXY_SERVICE_POINT_CARET_SCREEN	(23)
#define XAUX_PROXY_SERVICE_GET_CONVERSION_MODE	(24)
#define XAUX_PROXY_SERVICE_SET_CONVERSION_MODE	(25)
#define XAUX_PROXY_SERVICE_AUX_GETVALUE		(26)
#define XAUX_PROXY_SERVICE_AUX_GET_FROM_ID	(27)

/* XAUX PROXY message type */
#define	XAUX_PROXY_MESSAGE_TYPE_NONE		(0)
#define	XAUX_PROXY_MESSAGE_TYPE_RECEIPT		(1)
#define	XAUX_PROXY_MESSAGE_TYPE_SERVICE		(2)
#define	XAUX_PROXY_MESSAGE_TYPE_SIMPLE		(3)
#define	XAUX_PROXY_MESSAGE_TYPE_VALUE		(4)

/* XAUX PROXY Window Name */
#define XAUX_PROXY_WINDOW_NAME_SERVER	"IIIM XAUX Proxy Server"
#define XAUX_PROXY_WINDOW_NAME_CLIENT	"IIIM XAUX Proxy Client"


/*
 * data type definition
 */

typedef struct xaux_proxy_message_queue {
    unsigned int			server_id;
    unsigned short			im_id;
    unsigned short			ic_id;
    Display *				display;
    Atom				atom;
    Window				window_target;
    Window				window_self;
    int					message_type;
    unsigned char *			message;
    size_t				message_size;
    time_t				timestamp;
    struct xaux_proxy_message_queue *	next;
} xaux_proxy_message_queue_t;

typedef struct xaux_proxy_xevent_list {
    XEvent				event;
    struct xaux_proxy_xevent_list *	next;
} xaux_proxy_xevent_list_t;

typedef struct xaux_proxy_ic {
    Display *			display;
    Atom			atom;
    Window			window_target;
    Window			window_self;
    int				server_id;
    int				im_id;
    int				ic_id;
    XicCommon			xic;
    aux_t *			aux;
    Bool			aux_self;
    IIIMP_data_s *		data_s;
    struct xaux_proxy_ic *	next;
} xaux_proxy_ic_t;

/* local difinitin of aux_im_data_t */
struct _aux_im_data {
    int			im_id;
    int			ic_id;
    void *		ae; /* dummy */
    void *		data;
    aux_im_data_t *	next;
};


/*
 * local function prototype
 */

static void
xaux_proxy_ic_delete_alone(xaux_proxy_ic_t * ic);

static void
xaux_proxy_ic_delete(
    xaux_proxy_ic_t *	ic,
    Window		window_target,
    Window		window_self,
    XicCommon		xic);

static const xaux_proxy_ic_t *
xaux_proxy_receiver_find(
    Display *		display,
    Window		window_target,
    Window		window_self,
    int			server_id,
    IIIMP_card16	im_id,
    IIIMP_card16	ic_id,
    XicCommon		xic,
    aux_t *		aux);

static Bool
xaux_proxy_window_target_filter(
    Display *	display,
    Window	window,
    XEvent *	event,
    XPointer	client_data);

static Bool
xaux_proxy_window_self_filter(
    Display *	display,
    Window	window,
    XEvent *	event,
    XPointer	client_data);

static void
xaux_proxy_message_process(
    const xaux_proxy_ic_t *	ic,
    IIIMP_message *		message
);

static xaux_proxy_ic_t *
xaux_proxy_ic_new(
    Display *	display,
    Window	window_target,
    Window	window_self,
    int		server_id,
    int		im_id,
    int		ic_id,
    XicCommon	xic,
    aux_t *	aux);

static xaux_proxy_ic_t *
xaux_proxy_ic_get(
    Display *		display,
    Window		window_target,
    Window		window_self,
    int			server_id,
    IIIMP_card16	im_id,
    IIIMP_card16	ic_id,
    XicCommon		xic,
    aux_t *		aux);


/*
 * X auxiliary object service method - client
 */
static void		service_client_aux_setvalue(aux_t *,
						    const unsigned char *, int);
static void		service_client_aux_getvalue(aux_t *,
						    const unsigned char *, int);
static int		service_client_im_id(aux_t *);
static int		service_client_ic_id(aux_t *);
static void		service_client_data_set(aux_t *, int, void *);
static void *		service_client_data_get(aux_t *, int);
static Display *	service_client_display(aux_t *);
static Window		service_client_window(aux_t *);
static XPoint *		service_client_point(aux_t *, XPoint *);
static XPoint *		service_client_point_caret(aux_t *, XPoint *);
static size_t		service_client_utf16_mb(const char **, size_t *,
						char **, size_t *);
static size_t		service_client_mb_utf16(const char **, size_t *,
						char **, size_t *);
static unsigned char *	service_client_compose(const aux_data_t *, int *);
static int		service_client_compose_size(aux_data_type_t,
						    const unsigned char *);
static aux_data_t *	service_client_decompose(aux_data_type_t,
						 const unsigned char *);
static void		service_client_decompose_free(aux_data_t *);
static void		service_client_register_X_filter(Display *, Window,
							 int, int,
							 Bool (* filter)(Display *,
									 Window,
									 XEvent *,
									 XPointer),
							 XPointer);
static void		service_client_unregister_X_filter(Display *, Window,
							   Bool (* filter)(Display *,
									   Window,
									   XEvent *,
									   XPointer),
							   XPointer);
static Bool		service_client_server(aux_t *);
static Window		service_client_client_window(aux_t *);
static Window		service_client_focus_window(aux_t *);
static int		service_client_screen_number(aux_t *);
static int		service_client_point_screen(aux_t *, XPoint *);
static int		service_client_point_caret_screen(aux_t *, XPoint *);
static Bool		service_client_get_conversion_mode(aux_t*);
static void		service_client_set_conversion_mode(aux_t*, int);
static aux_t *		service_client_aux_get_from_id(int, int, CARD16 *, int);

static aux_service_t xaux_proxy_service_client = {
	service_client_aux_setvalue,
	service_client_im_id,
	service_client_ic_id,
	service_client_data_set,
	service_client_data_get,
	service_client_display,
	service_client_window,
	service_client_point,
	service_client_point_caret,
	service_client_utf16_mb,
	service_client_mb_utf16,
	service_client_compose,
	service_client_compose_size,
	service_client_decompose,
	service_client_decompose_free,
	service_client_register_X_filter,
	service_client_unregister_X_filter,
	service_client_server,
	service_client_client_window,
	service_client_focus_window,
	service_client_screen_number,
	service_client_point_screen,
	service_client_point_caret_screen,
	service_client_get_conversion_mode,
	service_client_set_conversion_mode,
	service_client_aux_getvalue,
	service_client_aux_get_from_id
};


/*
 * X auxiliary object service method - proxy
 */
static void		service_proxy_aux_setvalue(aux_t *,
						   const unsigned char *, int);
static void		service_proxy_aux_getvalue(aux_t *,
						   const unsigned char *, int);
static int		service_proxy_im_id(aux_t *);
static int		service_proxy_ic_id(aux_t *);
static void		service_proxy_data_set(aux_t *, int, void *);
static void *		service_proxy_data_get(aux_t *, int);
static Display *	service_proxy_display(aux_t *);
static Window		service_proxy_window(aux_t *);
static XPoint *		service_proxy_point(aux_t *, XPoint *);
static XPoint *		service_proxy_point_caret(aux_t *, XPoint *);
static size_t		service_proxy_utf16_mb(const char **, size_t *,
					       char **, size_t *);
static size_t		service_proxy_mb_utf16(const char **, size_t *,
					       char **, size_t *);
static unsigned char *	service_proxy_compose(const aux_data_t *, int *);
static int		service_proxy_compose_size(aux_data_type_t,
						   const unsigned char *);
static aux_data_t *	service_proxy_decompose(aux_data_type_t,
						const unsigned char *);
static void		service_proxy_decompose_free(aux_data_t *);
static void		service_proxy_register_X_filter(Display *, Window,
							int, int,
							Bool (* filter)(Display *,
									Window,
									XEvent *,
									XPointer),
							XPointer);
static void		service_proxy_unregister_X_filter(Display *, Window,
							  Bool (* filter)(Display *,
									  Window,
									  XEvent *,
									  XPointer),
							  XPointer);
static Bool		service_proxy_server(aux_t *);
static Window		service_proxy_client_window(aux_t *);
static Window		service_proxy_focus_window(aux_t *);
static int		service_proxy_screen_number(aux_t *);
static int		service_proxy_point_screen(aux_t *, XPoint *);
static int		service_proxy_point_caret_screen(aux_t *, XPoint *);
static Bool		service_proxy_get_conversion_mode(aux_t*);
static void		service_proxy_set_conversion_mode(aux_t*, int);
static aux_t *		service_proxy_aux_get_from_id(int, int, CARD16 *, int);


aux_service_t xaux_proxy_service_proxy = {
	service_proxy_aux_setvalue,
	service_proxy_im_id,
	service_proxy_ic_id,
	service_proxy_data_set,
	service_proxy_data_get,
	service_proxy_display,
	service_proxy_window,
	service_proxy_point,
	service_proxy_point_caret,
	service_proxy_utf16_mb,
	service_proxy_mb_utf16,
	service_proxy_compose,
	service_proxy_compose_size,
	service_proxy_decompose,
	service_proxy_decompose_free,
	service_proxy_register_X_filter,
	service_proxy_unregister_X_filter,
	service_proxy_server,
	service_proxy_client_window,
	service_proxy_focus_window,
	service_proxy_screen_number,
	service_proxy_point_screen,
	service_proxy_point_caret_screen,
	service_proxy_get_conversion_mode,
	service_proxy_set_conversion_mode,
	service_proxy_aux_getvalue,
	service_proxy_aux_get_from_id
};


/*
 * static variables
 */

static Window		xaux_proxy_window = None;
static Atom		xaux_proxy_atom = None;
static int		xaux_proxy_role_val = 0;

static xaux_proxy_message_queue_t *	xaux_proxy_message_queue_head;
static xaux_proxy_message_queue_t *	xaux_proxy_message_queue_tail;
static xaux_proxy_message_queue_t *	xaux_proxy_message_queue_last;

static xaux_proxy_ic_t *	xaux_proxy_ic;


/*
 * X error handler
 */

static int
xaux_proxy_xerror_handler(
    Display *		d,
    XErrorEvent *	e
)
{
    /* ignore all error event */
    return 0;
}


/*
 * X call wrappers to avoid BadWindow error
 */

static Status
xaux_proxy_XSendEvent(
    Display *	display,
    Window	w,
    Bool	propagate,
    int		event_mask,
    XEvent *	event_send
)
{
    int		(*oeh)(Display *, XErrorEvent *);
    Status	status;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    status = XSendEvent(display, w, propagate, event_mask, event_send);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return status;
}


static int
xaux_proxy_XChangeProperty(
    Display *		display,
    Window		w,
    Atom		property,
    Atom		type,
    int			format,
    int			mode,
    unsigned char *	data,
    int			nelements
)
{
    int	(*oeh)(Display *, XErrorEvent *);
    int	r;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    r = XChangeProperty(display, w, property, type,
			format, mode, data, nelements);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return r;
}


static int
xaux_proxy_XGetWindowProperty(
    Display *		display,
    Window		w,
    Atom		property,
    long		long_offset,
    long		long_length,
    Bool		delete,
    Atom		req_type,
    Atom *		actual_type_return,
    int *		actual_format_return,
    unsigned long *	nitems_return,
    unsigned long *	bytes_after_return,
    unsigned char **	prop_return
)
{
    int	(*oeh)(Display *, XErrorEvent *);
    int	r;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    r = XGetWindowProperty(display, w, property, long_offset, long_length,
			   delete, req_type,
			   actual_type_return, actual_format_return,
			   nitems_return, bytes_after_return, prop_return);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return r;
}


static int
xaux_proxy_XSelectInput(
    Display *	display,
    Window	w,
    long	event_mask
)
{
    int	(*oeh)(Display *, XErrorEvent *);
    int	r;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    r = XSelectInput(display, w, event_mask);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return r;
}

static int
xaux_proxy_XStoreName(
    Display *	display,
    Window	w,
    char *	window_name
)
{
    int	(*oeh)(Display *, XErrorEvent *);
    int	r;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    r = XStoreName(display, w, window_name);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return r;
}


static Status
xaux_proxy_XFetchName(
    Display *	display,
    Window	w,
    char **	window_name_return
)
{
    int		(*oeh)(Display *, XErrorEvent *);
    Status	s;

    XSync(display, False);
    oeh = XSetErrorHandler(xaux_proxy_xerror_handler);

    s = XFetchName(display, w, window_name_return);

    XSync(display, False);
    XSetErrorHandler(oeh);

    return s;
}


/*
 * miscellaneous utility
 */

static int
xaux_string_length(const IIIMP_card16 * p)
{
    int n;
    n = 0;
    for (; 0 != *p; p++) n++;
    return n;
}


/*
 * role
 */

int
xaux_proxy_role(XicCommon ic)
{
    char *	client_type;

    if (XAUX_PROXY_ROLE_UNKNOWN == xaux_proxy_role_val) {
	if (NULL == ic) {
	    return XAUX_PROXY_ROLE_CLIENT;
	}
	client_type = XIM_IIIMP(ic->core.im, client_type);
	if (0 == strcmp(client_type, "IIIMX XIM Server")) {
	    xaux_proxy_role_val = XAUX_PROXY_ROLE_SERVER;
	} else {
	    xaux_proxy_role_val = XAUX_PROXY_ROLE_CLIENT;
	}
    }

    return xaux_proxy_role_val;
}


/*
 * xua_im_data
 */

static aux_im_data_t *
xaux_proxy_aux_im_data_new(
    int	im_id,
    int	ic_id
)
{
    aux_im_data_t *	im_data;

    im_data = (aux_im_data_t *)malloc(sizeof (aux_im_data_t));
    if (NULL == im_data) return NULL;

    im_data->im_id = im_id;
    im_data->ic_id = ic_id;
    im_data->ae = NULL;
    im_data->data = NULL;
    im_data->next = NULL;

    return im_data;
}


static void
xaux_proxy_aux_im_data_delete(
    aux_im_data_t *	im_data
)
{
    if (NULL == im_data) return;
    free(im_data);
}


/*
 * xaux_proxy_aux_im_data
 */

static aux_t *
xaux_proxy_aux_new(
    xaux_proxy_ic_t *	ic
)
{
    aux_t *	aux;

    aux = (aux_t *)malloc(sizeof (aux_t));
    if (NULL == aux) return NULL;

    aux->ic = (XicCommon)ic;
    if(XAUX_PROXY_ROLE_CLIENT == xaux_proxy_role(NULL)) {
	/*
	 * for client, this function, xaux_proxy_aux_new(), is  not called
	 */
/* 	aux->service = &xaux_proxy_service_client; */
	aux->service = NULL;
    } else {
	aux->service = &xaux_proxy_service_proxy;
    }
    aux->im = xaux_proxy_aux_im_data_new(ic->im_id, ic->ic_id);
    if (NULL == aux->im) {
	free(aux);
	return NULL;
    }

    aux->im_list = aux->im;

    return aux;
}


static void
xaux_proxy_aux_delete(
    aux_t *	aux
)
{
    if (NULL == aux) return;

    xaux_proxy_aux_im_data_delete(aux->im);
    free(aux);
}


/*
 * xaux_proxy_message_queue_t
 */

xaux_proxy_message_queue_t *
xaux_proxy_message_queue_new(
    unsigned int	server_id,
    unsigned short	im_id,
    unsigned short	ic_id,
    Display *		display,
    Atom		atom,
    Window		window_target,
    Window		window_self,
    int			message_type,
    unsigned char *	message,
    size_t		message_size)
{
    xaux_proxy_message_queue_t *	m;

    m = (xaux_proxy_message_queue_t *)
	malloc(sizeof (xaux_proxy_message_queue_t));
    if (NULL == m) return NULL;

    m->server_id = server_id;
    m->im_id = im_id;
    m->ic_id = ic_id;
    m->display = display;
    m->atom = atom;
    m->window_target = window_target;
    m->window_self = window_self;
    m->message_type = message_type;
    m->message = (unsigned char *)malloc(message_size);
    if (NULL == m->message) {
	free(m);
	return NULL;
    }
    memcpy(m->message, message, message_size);
    m->message_size = message_size;
    m->timestamp = 0;
    m->next = NULL;

    return m;
}


void
xaux_proxy_message_queue_delete(
    xaux_proxy_message_queue_t *	m)
{
    if (NULL == m) return;
    free(m->message);
    free(m);
}


void
xaux_proxy_message_queue_delete_all(void)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	*m_prev = m->next;
	xaux_proxy_message_queue_delete(m);
    }

    xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
    xaux_proxy_message_queue_head = NULL;
    xaux_proxy_message_queue_last = NULL;
}


void
xaux_proxy_message_queue_delete_display(
    Display *	display
)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	if (display == m->display) {
	    *m_prev = m->next;
	    xaux_proxy_message_queue_delete(m);
	} else {
	    m_prev = (&(m->next));
	}
    }

    if ((NULL != xaux_proxy_message_queue_last) &&
	(display == xaux_proxy_message_queue_last->display)) {
	xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
	xaux_proxy_message_queue_last = NULL;
    }
}


void
xaux_proxy_message_queue_delete_window_self(
    Window	window
)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	if (window == m->window_self) {
	    *m_prev = m->next;
	    xaux_proxy_message_queue_delete(m);
	} else {
	    m_prev = (&(m->next));
	}
    }

    if ((NULL != xaux_proxy_message_queue_last) &&
	(window == xaux_proxy_message_queue_last->window_self)) {
	xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
	xaux_proxy_message_queue_last = NULL;
    }
}


void
xaux_proxy_message_queue_delete_window_target(
    Window	window
)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	if (window == m->window_target) {
	    *m_prev = m->next;
	    xaux_proxy_message_queue_delete(m);
	} else {
	    m_prev = (&(m->next));
	}
    }

    if ((NULL != xaux_proxy_message_queue_last) &&
	(window == xaux_proxy_message_queue_last->window_target)) {
	xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
	xaux_proxy_message_queue_last = NULL;
    }
}


void
xaux_proxy_message_queue_unsent_purge(void)
{
    xaux_proxy_message_queue_t *	m;
    xaux_proxy_message_queue_t **	m_prev;

    if (NULL == xaux_proxy_message_queue_last) return;

    for (m_prev = (&xaux_proxy_message_queue_head); NULL != (m = *m_prev); ) {
	if (xaux_proxy_message_queue_last->window_target == m->window_target) {
	    *m_prev = m->next;
	    xaux_proxy_message_queue_delete(m);
	} else {
	    m_prev = (&(m->next));
	}
    }

    xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
    xaux_proxy_message_queue_last = NULL;
}


void
xaux_proxy_message_queue_append(
    xaux_proxy_message_queue_t *	m)
{
    if (NULL == xaux_proxy_message_queue_head) {
	xaux_proxy_message_queue_head = m;
    } else {
	xaux_proxy_message_queue_tail->next = m;
    }
    xaux_proxy_message_queue_tail = m;
}


xaux_proxy_message_queue_t *
xaux_proxy_message_queue_pop(void)
{
    xaux_proxy_message_queue_t *	m;

    if (NULL == xaux_proxy_message_queue_head) return NULL;

    m = xaux_proxy_message_queue_head;
    xaux_proxy_message_queue_head = m->next;
    if (NULL == xaux_proxy_message_queue_head) {
	xaux_proxy_message_queue_tail = NULL;
    }

    return m;
}


/*
 * xaux_proxy_message simple data handling
 */

void
xaux_proxy_message_data_to_cm(
    Atom			atom,
    Window			window_receiver,
    Window			window_sender,
    int				server_id,
    unsigned short		im_id,
    unsigned short		ic_id,
    int				message_type,
    unsigned long		data0,
    unsigned long		data1,
    XClientMessageEvent *	cm)
{
    cm->type = ClientMessage;
    cm->message_type = atom;
    cm->format = 32;
    cm->window = window_receiver;
    cm->data.l[0] = window_sender;
    cm->data.l[1] = ((im_id << 16) | ic_id);
    cm->data.l[2] = ((message_type << 16) | server_id);
    cm->data.l[3] = data0;
    cm->data.l[4] = data1;
}


void
xaux_proxy_message_cm_to_data(
    XClientMessageEvent *	cm,
    unsigned long *		data0,
    unsigned long *		data1)
{
    if (NULL != data0) *data0 = cm->data.l[3];
    if (NULL != data1) *data1 = cm->data.l[4];
}


/*
 * short message exchange
 */

Bool
xaux_proxy_message_short_exchange(
    Display *			display,
    Window			window_target,
    Window			window_self,
    Atom			atom,
    int				server_id,
    unsigned short		im_id,
    unsigned short		ic_id,
    int				message_type,
    unsigned long		data0,
    unsigned long		data1,
    unsigned long *		data0_ret,
    unsigned long *		data1_ret)
{
    XEvent			e;
    XClientMessageEvent		cm;
    struct pollfd		fds;
    struct timeval		timeval;
    struct timeval		timeval_current;
    Bool			b;
    long			ms;
    xaux_proxy_xevent_list_t *	el;
    xaux_proxy_xevent_list_t *	el0;

    xaux_proxy_message_data_to_cm(atom, window_target, window_self,
				  server_id, im_id, ic_id, message_type,
				  data0, data1, &cm);

    xaux_proxy_XSendEvent(display, window_target, True, 0, (XEvent *)(&cm));

    if (NULL == data0_ret) return True;

    fds.fd = ConnectionNumber(display);
    fds.events = POLLIN;
    fds.revents = 0;

    gettimeofday(&timeval, NULL);
    timeval_current = timeval;

    b = False;
    el = NULL;

    for (;;) {
	/* timeout == 10 seconds */
	gettimeofday(&timeval_current, NULL);
	ms = (((timeval_current.tv_sec - timeval.tv_sec) * 1000000) +
	      (timeval_current.tv_usec - timeval.tv_usec));
	if (10000000 <= ms) break;

	b = XCheckTypedWindowEvent(display, window_self, ClientMessage, &e);
	if (True == b) {
	    if (XAUX_PROXY_MESSAGE_TYPE_RECEIPT != e.xclient.data.l[2]) {
		el0 = (xaux_proxy_xevent_list_t *)
		    malloc(sizeof (xaux_proxy_xevent_list_t));
		el0->event = e;
		el0->next = el;
		el = el0;
		continue;
	    }

	    xaux_proxy_message_cm_to_data(&cm, data0_ret, data1_ret);

	    break;
	}
	b = XCheckTypedWindowEvent(display, window_target, DestroyNotify, &e);
	if (True == b) {
	    b = False;
	    break;
	}
	poll(&fds, 1, 1000);
    }

    for (; NULL != el; el = el0) {
	XPutBackEvent(display, &(el->event));
	el0 = el->next;
	free(el);
    }

    return b;
}


/*
 * send long message.
 *
 * add a message to the quque.  if pending message does not
 * exist, send out a message at the top of the queue.
 */

void
xaux_proxy_message_send(
    Display *		display,
    Window		window_target,
    Window		window_self,
    Atom		atom,
    unsigned int	server_id,
    unsigned short	im_id,
    unsigned short	ic_id,
    int			message_type,
    unsigned char *	message,
    size_t		message_size)
{
    XClientMessageEvent		cm;
    xaux_proxy_message_queue_t *	m;
    time_t			time_current;

    if (NULL != message) {
	m = xaux_proxy_message_queue_new(server_id, im_id, ic_id, display, atom,
					 window_target, window_self,
					 message_type, message, message_size);
	if (NULL == m) return;

	xaux_proxy_message_queue_append(m);
    }

    time_current = time(NULL);

    if (NULL != xaux_proxy_message_queue_last) {
	if (time_current < (xaux_proxy_message_queue_last->timestamp + 10)) {
	    return;
	}
	xaux_proxy_message_queue_unsent_purge();
    }

    m = xaux_proxy_message_queue_pop();
    if (NULL == m) return;

    xaux_proxy_message_queue_last = m;
    m->timestamp = time_current;

    if (8 < m->message_size) {
	xaux_proxy_XChangeProperty(m->display, m->window_self, m->atom,
				   XA_CARDINAL, 8, PropModeReplace,
				   m->message, m->message_size);
    }

    xaux_proxy_message_short_exchange(m->display, m->window_target, m->window_self,
				      m->atom, m->server_id, m->im_id, m->ic_id,
				      m->message_type, 0, 0, NULL, NULL);

    return;
}


/*
 * send pending message if left on the queue.
 */

void
xaux_proxy_message_ping(Display * display)
{
    xaux_proxy_message_send(display, None, None, None, 0, 0, 0,
			    XAUX_PROXY_MESSAGE_TYPE_NONE, NULL, 0);
}


void
xaux_proxy_message_send_cleanup(void)
{
    xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
    xaux_proxy_message_queue_last = NULL;
}


static void
xaux_proxy_message_process_start(
    const xaux_proxy_ic_t *	ic,
    IIIMP_message *		m
)
{
    IIIMP_card16 *		p;
    IIIMP_string *		name;
    int				len;
    int				size;

    name = m->v.aux_start.input_method_name;
    if (NULL == name->ptr) return;

    len = name->len;

    p = (IIIMP_card16 *)malloc((sizeof (IIIMP_card16)) * (len + 1));
    if (NULL == p) return;

    memcpy(p, name->ptr, (sizeof (IIIMP_card16)) * len);
    *(p + len) = 0;

    xaux_proxy_xaux_so_start(ic->aux, p);

    free(p);
}


static void
xaux_proxy_message_process(
    const xaux_proxy_ic_t *	ic,
    IIIMP_message *		m
)
{
    if ((NULL == ic) || (NULL == m)) return;

    switch (m->opcode) {
    case IM_AUX_START:
	xaux_proxy_message_process_start(ic, m);
	break;
    case IM_AUX_DONE:
    case IM_AUX_DRAW:
    case IM_AUX_SETVALUES:
    case IM_AUX_GETVALUES:
    case IM_AUX_GETVALUES_REPLY:
	break;
    default:
	break;
    }
    return;
}


/*
 * filter to handle message
 */

static Bool
xaux_proxy_window_self_filter(
    Display *	display,
    Window	window,
    XEvent *	event,
    XPointer	client_data)
{
    xaux_proxy_message_queue_t *	m;
    XClientMessageEvent			cm;
    Atom				atom;
    Window				window_sender;
    unsigned int			server_id;
    unsigned short			im_id;
    unsigned short			ic_id;
    int					byte_order;
    int					message_type;
    unsigned char *			message;
    Atom				actual_type_return;
    int					actual_format_return;
    unsigned long			nitem_return;
    unsigned long			bytes_after_return;
    unsigned char *			prop;
    unsigned int			buf[8];
    IIIMP_message *			p;
    size_t				nbyte;
    const uchar_t *			ptr;
    XicCommon				xic;
    const xaux_proxy_ic_t *		ic;
    int					r;

    if ((ClientMessage != event->type) || (event->xclient.window != window)) {
	return False;
    }

    xic = (XicCommon)client_data;

    atom = event->xclient.message_type;
    window_sender = event->xclient.data.l[0];
    im_id = ((0xffff0000 & event->xclient.data.l[1]) >> 16);
    ic_id = (0x0000ffff & event->xclient.data.l[1]);
    byte_order = ((0x80000000 & event->xclient.data.l[2]) >> 31);
    message_type = ((0x7fff0000 & event->xclient.data.l[2]) >> 16);
    server_id = (0x0000ffff & event->xclient.data.l[2]);

    switch (message_type) {
    case XAUX_PROXY_MESSAGE_TYPE_RECEIPT: /* receipt */
	xaux_proxy_message_queue_delete(xaux_proxy_message_queue_last);
	xaux_proxy_message_queue_last = NULL;
	xaux_proxy_message_ping(display);
	break;

    case XAUX_PROXY_MESSAGE_TYPE_SERVICE: /* service routines */
	xaux_proxy_message_short_exchange(display, window_sender, window,
					  atom, server_id, im_id, ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_RECEIPT,
					  0, 0, NULL, NULL);
	break;

    case XAUX_PROXY_MESSAGE_TYPE_SIMPLE: /* IM_AUX_START/DONE */
    case XAUX_PROXY_MESSAGE_TYPE_VALUE: /* IM_AUX_DRAW/SETVALUES/GETVALUES */
	ic = xaux_proxy_receiver_find(display, window_sender, window,
				      server_id, im_id, ic_id, NULL, NULL);
	if (NULL == ic) return False;


	r = xaux_proxy_XGetWindowProperty(display, window_sender, atom,
					  0, 0xffff, True, XA_CARDINAL,
					  &actual_type_return,
					  &actual_format_return,
					  &nitem_return,
					  &bytes_after_return,
					  &prop);

	xaux_proxy_message_short_exchange(display, window_sender, window,
					  atom, server_id, im_id, ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_RECEIPT,
					  0, 0, NULL, NULL);

	if ((Success != r) || (NULL == prop)) return True;

	nbyte = (nitem_return - 4);
	ptr = (prop + 4);
	p = iiimp_message_unpack(ic->data_s, *prop, &nbyte, &ptr);

	XFree(prop);

	if (NULL == p) return True;

	xaux_proxy_message_process(ic, p);

	xaux_proxy_message_ping(display);

	switch (message_type) {
	case XAUX_PROXY_MESSAGE_TYPE_SIMPLE:
	    /* IM_AUX_START/DONE */
	    break;
	case XAUX_PROXY_MESSAGE_TYPE_VALUE:
	    /* IM_AUX_DRAW/SETVALUES/GETVALUES */
	    break;
	}

	iiimp_message_delete(ic->data_s, p);

	break;

    default:
	break;
    }

    return True;
}


/*
 * setup - client
 */

Bool
xaux_proxy_setup_client(
    Display *		d,
    int			server_id,
    int			im_id,
    int			ic_id,
    XicCommon		xic
)
{
    if (NULL == d) return False;

    if (None == xaux_proxy_atom) {
	xaux_proxy_atom = XInternAtom(d, "IIIM_XAUX_PROXY", True);
	if (None == xaux_proxy_atom) return False;
    }

    return True;
}


/*
 * setup - server
 */

Bool
xaux_proxy_setup_server(
    Display *		d
)
{
    char		buf[256];

    if (NULL == d) return False;

    xaux_proxy_role_val = XAUX_PROXY_ROLE_SERVER;

    if (None == xaux_proxy_atom) {
	xaux_proxy_atom = XInternAtom(d, "IIIM_XAUX_PROXY", False);
	if (None == xaux_proxy_atom) return False;
    }

    if (None == xaux_proxy_window) {
	xaux_proxy_window =
	    XCreateSimpleWindow(d, RootWindow(d, 0), 0, 0, 1, 1, 0, 0, 0);
	if (None == xaux_proxy_window) return False;

	XSetSelectionOwner(d, xaux_proxy_atom, xaux_proxy_window, CurrentTime);

	snprintf(buf, sizeof (buf),
		 "%s %d", XAUX_PROXY_WINDOW_NAME_SERVER, getpid());
	XStoreName(d, xaux_proxy_window, buf);

	_XRegisterFilterByType(d, xaux_proxy_window,
			       ClientMessage, ClientMessage,
			       xaux_proxy_window_self_filter, NULL);
    }

    return True;
}


/*
 * finish
 */

void
xaux_proxy_finish(Display * d)
{
    /* this function is not called by anyone as of 2006-01-25 */
    Atom	a;
    Window	w;

    if (NULL == d) return;

    if(XAUX_PROXY_ROLE_CLIENT == xaux_proxy_role(NULL)) {
	a = XInternAtom(d, "IIIM_XAUX_PROXY", True);
	if (None != a) {
	    w = XGetSelectionOwner(d, a);
	    if (None == w) {
		_XUnregisterFilter(d, w, xaux_proxy_window_target_filter, NULL);
	    }
	}
    }

    if (None != xaux_proxy_window) {
	XDestroyWindow(d, xaux_proxy_window);
	xaux_proxy_window = None;
    }
    xaux_proxy_atom = None;
}


/*
 * xaux_proxy_ic_t
 */

static xaux_proxy_ic_t *
xaux_proxy_ic_new(
    Display *	display,
    Window	window_target,
    Window	window_self,
    int		server_id,
    int		im_id,
    int		ic_id,
    XicCommon	xic,
    aux_t *	aux)
{
    xaux_proxy_ic_t *	ic;
    char		buf[256];

    ic = (xaux_proxy_ic_t *)malloc(sizeof (xaux_proxy_ic_t));
    if (NULL == ic) return NULL;

    ic->display = display;
    ic->atom = None;
    ic->window_target = window_target;
    ic->window_self = window_self;
    ic->server_id = server_id;
    ic->im_id = im_id;
    ic->ic_id = ic_id;
    ic->xic = xic;
    ic->aux = aux;
    ic->aux_self = False;
    ic->data_s = NULL;
    ic->next = NULL;

    ic->atom = XInternAtom(display, "IIIM_XAUX_PROXY", True);
    if (None == ic->atom) {
	xaux_proxy_ic_delete_alone(ic);
	return NULL;
    }

    if (None == window_target) {
	ic->window_target = XGetSelectionOwner(display, ic->atom);
	if (None == ic->window_target) {
	    xaux_proxy_ic_delete_alone(ic);
	    return NULL;
	}
    }

    if (None == window_self) {
	ic->window_self = XCreateSimpleWindow(display, RootWindow(display, 0),
					      0, 0, 1, 1, 0, 0, 0);
	if (None == ic->window_self) {
	    xaux_proxy_ic_delete_alone(ic);
	    return NULL;
	}

	snprintf(buf, sizeof (buf),
		 "%s %d", XAUX_PROXY_WINDOW_NAME_CLIENT, getpid());
	XStoreName(display, ic->window_self, buf);

	_XRegisterFilterByType(display, ic->window_self,
			       ClientMessage, ClientMessage,
			       xaux_proxy_window_self_filter, (XPointer)xic);
    }

    ic->data_s = iiimp_data_s_new();
    if (NULL == ic->data_s) {
	xaux_proxy_ic_delete_alone(ic);
	return NULL;
    }

    if (NULL == aux) {
	ic->aux = xaux_proxy_aux_new(ic);
	if (NULL == ic->aux) {
	    xaux_proxy_ic_delete_alone(ic);
	    return NULL;
	}
	ic->aux_self = True;
    }

    return ic;
}


static void
xaux_proxy_ic_delete_alone(xaux_proxy_ic_t * ic)
{
    if (NULL == ic) return;

    if ((xaux_proxy_window != ic->window_self) && (None != ic->window_self)) {
	_XUnregisterFilter(ic->display, ic->window_self,
			   xaux_proxy_window_self_filter, (XPointer)(ic->xic));
	XDestroyWindow(ic->display, ic->window_self);
    }

    iiimp_data_s_delete(ic->data_s);
    if (True == ic->aux_self) xaux_proxy_aux_delete(ic->aux);

    free(ic);
}


static void
xaux_proxy_ic_delete(
    xaux_proxy_ic_t *	ic,
    Window		window_target,
    Window		window_self,
    XicCommon		xic
)
{
    xaux_proxy_ic_t *	p;
    xaux_proxy_ic_t **	p_prev;

    if ((NULL == ic) && (None == window_self)) return;

    for (p_prev = (&xaux_proxy_ic); NULL != (p = *p_prev); ) {
	if ((ic == p) ||
	    ((None != window_target) && (window_target == p->window_target)) ||
	    ((None != window_self) && (window_self == p->window_self)) ||
	    ((NULL != xic) && (xic == p->xic))) {
	    *p_prev = p->next;
	    xaux_proxy_ic_delete_alone(p);
	} else {
	    p_prev = (&(p->next));
	}
    }
}


static void
xaux_proxy_ic_delete_all(
)
{
    xaux_proxy_ic_t *	p;
    xaux_proxy_ic_t **	p_prev;

    for (p_prev = (&xaux_proxy_ic); NULL != (p = *p_prev); ) {
	*p_prev = p->next;
	xaux_proxy_ic_delete_alone(p);
    }
}


static xaux_proxy_ic_t *
xaux_proxy_ic_get(
    Display *		display,
    Window		window_target,
    Window		window_self,
    int			server_id,
    IIIMP_card16	im_id,
    IIIMP_card16	ic_id,
    XicCommon		xic,
    aux_t *		aux
)
{
    xaux_proxy_ic_t *	ic;
    xaux_proxy_ic_t **	ic_prev;
    char *		p;
    char *		n;
    Status		s;
    int			r;
    Window		w;
    char		buf[256];

    ic = NULL;
    ic_prev = (&xaux_proxy_ic);
    w = None;

    r = xaux_proxy_role(NULL);

    if (XAUX_PROXY_ROLE_CLIENT == r) {
	while (NULL != (ic = *ic_prev)) {
	    if ((xic == ic->xic) ||
		((im_id == ic->im_id) &&
		 (window_self == ic->window_self) &&
		 (server_id == ic->server_id) &&
		 (ic_id == ic->ic_id))) {
		*ic_prev = ic->next;
		break;
	    } else {
		ic_prev = (&(ic->next));
	    }
	}
    } else {
	while (NULL != (ic = *ic_prev)) {
	    if ((window_target == ic->window_target) &&
		(server_id == ic->server_id) &&
		(im_id == ic->im_id) &&
		(ic_id == ic->ic_id)) {
		*ic_prev = ic->next;
		break;
	    } else {
		ic_prev = (&(ic->next));
	    }
	}
    }

    if (NULL == ic) {
	if (NULL == display) return NULL;
	ic = xaux_proxy_ic_new(display, window_target, window_self,
			       server_id, im_id, ic_id, xic, aux);
	if (NULL == ic) return NULL;
    }

    if (None == ic->window_target) {
	if (None != window_target) {
	    ic->window_target = window_target;
	} else if (XAUX_PROXY_ROLE_CLIENT == r) {
	    if (None == ic->atom) return NULL;
	    ic->window_target = XGetSelectionOwner(display, ic->atom);
	}
	if (None == ic->window_target) {
	    xaux_proxy_ic_delete(ic, None, None, NULL);
	    return NULL;
	}
    }

    xaux_proxy_XSelectInput(ic->display, ic->window_target,
			    StructureNotifyMask);

    if (XAUX_PROXY_ROLE_CLIENT == r) {
	n = XAUX_PROXY_WINDOW_NAME_SERVER;
    } else {
	n = XAUX_PROXY_WINDOW_NAME_CLIENT;
    }

    s = xaux_proxy_XFetchName(ic->display, ic->window_target, &p);

    if (0 == s) {
	return NULL;
    } else if ((NULL == p) || (0 != strncmp(p, n, strlen(n)))) {
	xaux_proxy_XSelectInput(ic->display, ic->window_self, NoEventMask);
	if (NULL != p) XFree(p);
	return NULL;
    }
    XFree(p);

    _XRegisterFilterByType(ic->display, ic->window_target,
			   DestroyNotify, DestroyNotify,
			   xaux_proxy_window_target_filter, NULL);

    ic->next = xaux_proxy_ic;
    xaux_proxy_ic = ic;

    return ic;
}


static const xaux_proxy_ic_t *
xaux_proxy_receiver_find(
    Display *		display,
    Window		window_target,
    Window		window_self,
    int			server_id,
    IIIMP_card16	im_id,
    IIIMP_card16	ic_id,
    XicCommon		xic,
    aux_t *		aux)
{
    xaux_proxy_ic_t *	ic;

    ic = xaux_proxy_ic_get(display, window_target, window_self,
			   server_id, im_id, ic_id, xic, aux);
    if (NULL == ic) return NULL;

    if (None == ic->window_target) {
	if (XAUX_PROXY_ROLE_CLIENT == xaux_proxy_role(NULL)) {
	    ic->window_target = XGetSelectionOwner(ic->display, ic->atom);
	}
    }

    return ic;
}


static Bool
xaux_proxy_window_target_filter(
    Display *	display,
    Window	window,
    XEvent *	event,
    XPointer	client_data)
{
#if 0
    xaux_proxy_ic_t *	ic;
    xaux_proxy_ic_t **	ic_prev;
#endif /* 0 */

    if (DestroyNotify != event->type) return False;

    _XUnregisterFilter(display, window,
		       xaux_proxy_window_target_filter, client_data);

    xaux_proxy_ic_delete(NULL, window, None, NULL);

#if 0
    for (ic_prev = (&xaux_proxy_ic); NULL != (ic = *ic_prev); ) {
	if (window == ic->window_target) {
	    *ic_prev = ic->next;
	    xaux_proxy_ic_delete(ic);
	} else {
	    ic_prev = (&(ic->next));
	}
    }
#endif /* 0 */

    return True;
}


void
xaux_proxy_ic_destroy(
    XicCommon	xic
)
{
    xaux_proxy_ic_delete_all(NULL, None, None, xic);
}


void
xaux_proxy_destroy(
    Display *	d
)
{
    Atom	a;
    Window	w;

    xaux_proxy_ic_delete_all();

    if (NULL == d) return;

    if(XAUX_PROXY_ROLE_CLIENT == xaux_proxy_role(NULL)) {
	a = XInternAtom(d, "IIIM_XAUX_PROXY", True);
	if (None != a) {
	    w = XGetSelectionOwner(d, a);
	    if (None == w) {
		_XUnregisterFilter(d, w, xaux_proxy_window_target_filter, NULL);
	    }
	}
    }

    if (None != xaux_proxy_window) {
	XDestroyWindow(d, xaux_proxy_window);
	xaux_proxy_window = None;
    }
    xaux_proxy_atom = None;
}


void
xaux_proxy_aux_simple(
    XicCommon		xic,
    int			type,
    int			server_id,
    int			im_id,
    int			ic_id,
    IIIMCF_event	ev)
{
    IIIMP_data_s *		data_s;
    IIIMF_status		st;
    const IIIMP_card16 *	aux_name;
    IIIMP_card32		class_index;
    IIIMP_string *		input_method_name;
    size_t			len;
    uchar_t *			buf;
    size_t			buf_size;
    const xaux_proxy_ic_t *	ic;

    ic = xaux_proxy_receiver_find(xic->core.im->core.display, None, None,
				  server_id, im_id, ic_id,
				  xic, XIC_IIIMP(xic, aux));
    if (NULL == ic) return;

    st = iiimcf_get_aux_event_value(ev, &aux_name, &class_index,
				    NULL, NULL, NULL, NULL);
    if (IIIMF_STATUS_SUCCESS != st) return;

    len = xaux_string_length(aux_name);

    input_method_name = iiimp_string_new(ic->data_s, len, aux_name);
    if (NULL == input_method_name) return;

    buf = iiimp_aux_simple_pack(ic->data_s,
				type,
				im_id,
				ic_id,
				class_index,
				input_method_name,
				&buf_size);
    iiimp_string_delete(ic->data_s, input_method_name);
    if (NULL ==buf) return;

    xaux_proxy_message_send(ic->display, ic->window_target, ic->window_self,
			    ic->atom, server_id, im_id, ic_id,
			    XAUX_PROXY_MESSAGE_TYPE_SIMPLE, buf, buf_size);

    free(buf);

    return;
}


void
xaux_proxy_aux_value(
    XicCommon		xic,
    int			type,
    int			server_id,
    int			im_id,
    int			ic_id,
    IIIMCF_event	ev)
{
    IIIMP_data_s *		data_s;
    IIIMF_status		st;
    const IIIMP_card16 *	aux_name;
    const IIIMP_card16 *	p;
    IIIMP_card32		class_index;
    int				num_intvals;
    const IIIMP_card32 *	pintvals;
    int				num_strvals;
    const IIIMP_card16 **	pstrs;
    IIIMP_string *		input_method_name;
    size_t			len;
    IIIMP_card32_list *		integer_value;
    IIIMP_string *		string_value;
    IIIMP_string *		string_value_last;
    IIIMP_string *		string_value_cur;
    uchar_t *			buf;
    size_t			buf_size;
    int				i;

    st = iiimcf_get_aux_event_value(ev, &aux_name, &class_index,
				    &num_intvals, &pintvals, &num_strvals, &pstrs);
    if (IIIMF_STATUS_SUCCESS != st) return;

#if 0
    if (NULL == xaux_proxy_data_s) {
	xaux_proxy_data_s = iiimp_data_s_new();
	if (NULL == xaux_proxy_data_s) return;
    }
#endif

    len = xaux_string_length(aux_name);
    input_method_name = iiimp_string_new(data_s, len, aux_name);
    if (NULL == input_method_name) return;

    integer_value = NULL;
    iiimp_card32_list_new(data_s, num_intvals, pintvals);

    string_value = NULL;
    for (i = 0; i < num_strvals; i++) {
	len = xaux_string_length(*(pstrs + i));
	string_value_cur =
	    iiimp_string_new(data_s, len, *(pstrs + i));
	if (NULL == string_value_last) {
	    string_value_last = string_value_cur;
	} else {
	    string_value_last->next = string_value_cur;
	}
    }

    buf = iiimp_aux_value_pack(data_s,
			       type,
			       im_id,
			       ic_id,
			       class_index,
			       input_method_name,
			       integer_value,
			       string_value,
			       &buf_size);
    iiimp_string_delete(data_s, input_method_name);
    iiimp_card32_list_delete(data_s, integer_value);
    iiimp_string_list_delete(data_s, string_value);
    if (NULL == buf) return;

    free(buf);
}


/*
 * aux service function - client
 */

static void
service_client_aux_setvalue(
    aux_t *			aux,
    const unsigned char *	p,
    int				len
)
{
#if 0
    AUXComposed *pac = (AUXComposed*) p;

    if (pac->ev) {
	IMAuxSetValues(aux->ic, pac->ev);
    } else if (pac->pad) {
	int i;
	aux_data_t *pad = pac->pad;
	IIIMF_status st;
	IIIMCF_event ev;
	IIIMP_card32 *pintvals;
	const IIIMP_card16 **pstrs;

#ifdef ENABLE_TRACE
	static void show_AC(aux_data_t *pad);
	show_AC(pac->pad);
#endif
	if (pad->integer_count > 0) {
	    pintvals = (IIIMP_card32*) malloc(sizeof(*pintvals) * pad->integer_count);
	    if (!pintvals) return;
	    for (i = 0; i < pad->integer_count; i++) {
		pintvals[i] = pad->integer_list[i];
	    }
	} else {
	    pintvals = NULL;
	}
	if (pad->string_count > 0) {
	    pstrs = (const IIIMP_card16**) malloc(sizeof(*pstrs) * pad->string_count);
	    if (!pstrs) {
		if (pintvals) free(pintvals);
		return;
	    }
	    for (i = 0; i < pad->string_count; i++) {
		pstrs[i] = (const IIIMP_card16*) pad->string_list[i].ptr;
	    }
	} else {
	    pstrs = NULL;
	}

	st = iiimcf_create_aux_setvalues_event((IIIMP_card16*) pad->aux_name,
					       pad->aux_index,
					       pad->integer_count,
					       pintvals,
					       pad->string_count,
					       pstrs,
					       &ev);
	if (st == IIIMF_STATUS_SUCCESS) {
	    IMAuxSetValues(aux->ic, ev);
	}

	if (pintvals) free(pintvals);
	if (pstrs) free(pstrs);
    }
#endif /* 0 */
}


static void
service_client_aux_getvalue(
    aux_t *			aux,
    const unsigned char *	p,
    int				len
)
{
#if 0
    AUXComposed *pac = (AUXComposed*) p;

    if (pac->ev) {
	IMAuxGetValues(aux->ic, pac->ev);
    } else if (pac->pad) {
	int i;
	aux_data_t *pad = pac->pad;
	IIIMF_status st;
	IIIMCF_event ev;
	IIIMP_card32 *pintvals;
	const IIIMP_card16 **pstrs;

#ifdef ENABLE_TRACE
	static void show_AC(aux_data_t *pad);
	show_AC(pac->pad);
#endif
	if (pad->integer_count > 0) {
	    pintvals = (IIIMP_card32*) malloc(sizeof(*pintvals) * pad->integer_count);
	    if (!pintvals) return;
	    for (i = 0; i < pad->integer_count; i++) {
		pintvals[i] = pad->integer_list[i];
	    }
	} else {
	    pintvals = NULL;
	}
	if (pad->string_count > 0) {
	    pstrs = (const IIIMP_card16**) malloc(sizeof(*pstrs) * pad->string_count);
	    if (!pstrs) {
		if (pintvals) free(pintvals);
		return;
	    }
	    for (i = 0; i < pad->string_count; i++) {
		pstrs[i] = (const IIIMP_card16*) pad->string_list[i].ptr;
	    }
	} else {
	    pstrs = NULL;
	}

	st = iiimcf_create_aux_getvalues_event((IIIMP_card16*) pad->aux_name,
					       pad->aux_index,
					       pad->integer_count,
					       pintvals,
					       pad->string_count,
					       pstrs,
					       &ev);
	if (st == IIIMF_STATUS_SUCCESS) {
	    IMAuxGetValues(aux->ic, ev);
	}

	if (pintvals) free(pintvals);
	if (pstrs) free(pstrs);
    }
#endif /* 0 */
}

static int
service_client_im_id(
    aux_t *	aux
)
{
    return aux->im->im_id;
}


static int
service_client_ic_id(
    aux_t *	aux
)
{
    return aux->im->ic_id;
}


static void
service_client_data_set(
    aux_t *	aux,
    int		im_id,
    void *	data
)
{
    aux_im_data_t *	aux_im;

    for (aux_im = aux->im; NULL != aux_im; aux_im = aux_im->next) {
	if (im_id == aux_im->im_id) {
	    aux_im->data = data;
	}
    }

    return;
}


static void *
service_client_data_get(
    aux_t *	aux,
    int		im_id
)
{
    aux_im_data_t *	aux_im;

    for (aux_im = aux->im; NULL != aux_im; aux_im = aux_im->next) {
	if (im_id == aux_im->im_id) {
	    return aux_im->data;
	}
    }

    return NULL;
}


static Display *
service_client_display(
    aux_t *	aux
)
{
    xaux_proxy_ic_t *	ic;
    ic = (xaux_proxy_ic_t*)(aux->ic);
    return ic->display;
}


static Window
service_client_window(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_WINDOW, 0,
					  &data0_ret, NULL);
    if (False == b) {
	return None;
    } else {
	return (Window)(data0_ret);
    }
}


static XPoint *
service_client_point(
    aux_t *	aux,
    XPoint *	point
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_POINT, 0,
					  &data0_ret, NULL);
    if (False == b) {
	point->x = -1;
	point->y = -1;
    } else {
	point->x = ((0xFFFF0000 & data0_ret) >> 16);
	point->y = (0x0000FFFF & data0_ret);
    }

    return point;
}


static XPoint *
service_client_point_caret(
    aux_t *	aux,
    XPoint *	point
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_POINT_CARET, 0,
					  &data0_ret, NULL);
    if (False == b) {
	point->x = -1;
	point->y = -1;
    } else {
	point->x = ((0xFFFF0000 & data0_ret) >> 16);
	point->y = (0x0000FFFF & data0_ret);
    }

    return point;
}


static size_t
service_client_utf16_mb(
    const char **	inbuf,
    size_t *		inbytesleft,
    char **		outbuf,
    size_t *		outbytesleft
)
{
    int	r;

    r = IIimpConvertFromUTF16((char *)(*inbuf), *inbytesleft,
			      outbuf, outbytesleft);
    return (size_t)r;
}


static size_t
service_client_mb_utf16(
    const char **	inbuf,
    size_t *		inbytesleft,
    char **		outbuf,
    size_t *		outbytesleft
)
{
    int	r;

    r = IIimpConvertToUTF16((char *)(*inbuf), *inbytesleft,
			    outbuf, outbytesleft);
    return (size_t)r;
}


static unsigned char *
service_client_compose(
    const aux_data_t *	pad,
    int *		size
)
{
#if 0
    AUXComposed *	pac;

    pac = create_composed_from_aux_data(pad);
    return (unsigned char *)pac;
#else /* !0 */
    return NULL;
#endif /* !0 */
}


static int
service_client_compose_size(
    aux_data_type_t		type,
    const unsigned char *	p
)
{
    /* now this function is dummy... */
    return 0;
}


static aux_data_t *
service_client_decompose(
    aux_data_type_t		type,
    const unsigned char *	p
)
{
#if 0
    AUXComposed *pac = (AUXComposed*) p;

    if (pac->pad) {
	pac = create_composed_from_aux_data(pac->pad);
	if (!pac) return NULL;
	pac->pad->type = type;
#ifdef ENABLE_TRACE
	show_AC(pac->pad);
#endif
	return pac->pad;
    }

    if (pac->ev) {
	pac = create_composed_from_event(pac->aux, pac->ev);
	if (!pac) return NULL;
	pac->pad->type = type;
#ifdef ENABLE_TRACE
	show_AC(pac->pad);
#endif
	return pac->pad;
    }

#endif /* 0 */
    return NULL;
}


static void
service_client_decompose_free(
    aux_data_t *	pad
)
{
    if (NULL == pad) return;
    if (NULL != pad->string_ptr) return;
    free(pad->string_ptr);
}


static void
service_client_register_X_filter(
    Display *	display,
    Window	window,
    int		start_type,
    int		end_type,
    Bool	(* filter)(Display *, Window, XEvent *, XPointer),
    XPointer	client_data
)
{
    _XRegisterFilterByType(display, window, start_type, end_type,
			   filter, client_data);

    return;
}


static void
service_client_unregister_X_filter(
    Display *	display,
    Window	window,
    Bool	(* filter)(Display *, Window, XEvent *, XPointer),
    XPointer	client_data
)
{
    _XUnregisterFilter(display, window, filter, client_data);
}


static Bool
service_client_server(
    aux_t *	aux
)
{
    return True;
}


static Window
service_client_client_window(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_CLIENT_WINDOW, 0,
					  &data0_ret, NULL);
    if (False == b) {
	return None;
    } else {
	return (Window)(data0_ret);
    }
}


static Window
service_client_focus_window(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_FOCUS_WINDOW, 0,
					  &data0_ret, NULL);
    if (False == b) {
	return None;
    } else {
	return (Window)(data0_ret);
    }
}


static int
service_client_screen_number(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_SCREEN_NUMBER, 0,
					  &data0_ret, NULL);
    if (False == b) {
	return 0;
    } else {
	return data0_ret;
    }
}


static int
service_client_point_screen(
    aux_t *	aux,
    XPoint *	point
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;
    unsigned long	data1_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_POINT_SCREEN, 0,
					  &data0_ret, &data1_ret);
    if (False == b) {
	return -1;
    } else {
	point->x = ((0xFFFF0000 & data0_ret) >> 16);
	point->y = (0x0000FFFF & data0_ret);
	return data1_ret;
    }
}


static int
service_client_point_caret_screen(
    aux_t *	aux,
    XPoint *	point
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;
    unsigned long	data1_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_POINT_CARET_SCREEN,
					  0,
					  &data0_ret, &data1_ret);
    if (False == b) {
	return -1;
    } else {
	point->x = ((0xFFFF0000 & data0_ret) >> 16);
	point->y = (0x0000FFFF & data0_ret);
	return data1_ret;
    }
}


static Bool
service_client_get_conversion_mode(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_GET_CONVERSION_MODE,
					  0,
					  &data0_ret, NULL);
    if (False == b) {
	return False;
    } else {
	return data0_ret;
    }
}

static void
service_client_set_conversion_mode(
    aux_t *	aux,
    int		conversion_mode
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_SET_CONVERSION_MODE,
					  conversion_mode,
					  NULL, NULL);
    return;
}


static aux_t *
service_client_aux_get_from_id(
    int		im_id,
    int		ic_id,
    CARD16 *	aux_name,
    int		aux_name_length
)
{
    return NULL;
}


/*
 * aux service function - proxy
 */

static void
service_proxy_aux_setvalue(
    aux_t *			aux,
    const unsigned char *	p,
    int				len
)
{
#if 0
    AUXComposed *pac = (AUXComposed*) p;

    if (pac->ev) {
	IMAuxSetValues(aux->ic, pac->ev);
    } else if (pac->pad) {
	int i;
	aux_data_t *pad = pac->pad;
	IIIMF_status st;
	IIIMCF_event ev;
	IIIMP_card32 *pintvals;
	const IIIMP_card16 **pstrs;

#ifdef ENABLE_TRACE
	static void show_AC(aux_data_t *pad);
	show_AC(pac->pad);
#endif
	if (pad->integer_count > 0) {
	    pintvals = (IIIMP_card32*) malloc(sizeof(*pintvals) * pad->integer_count);
	    if (!pintvals) return;
	    for (i = 0; i < pad->integer_count; i++) {
		pintvals[i] = pad->integer_list[i];
	    }
	} else {
	    pintvals = NULL;
	}
	if (pad->string_count > 0) {
	    pstrs = (const IIIMP_card16**) malloc(sizeof(*pstrs) * pad->string_count);
	    if (!pstrs) {
		if (pintvals) free(pintvals);
		return;
	    }
	    for (i = 0; i < pad->string_count; i++) {
		pstrs[i] = (const IIIMP_card16*) pad->string_list[i].ptr;
	    }
	} else {
	    pstrs = NULL;
	}

	st = iiimcf_create_aux_setvalues_event((IIIMP_card16*) pad->aux_name,
					       pad->aux_index,
					       pad->integer_count,
					       pintvals,
					       pad->string_count,
					       pstrs,
					       &ev);
	if (st == IIIMF_STATUS_SUCCESS) {
	    IMAuxSetValues(aux->ic, ev);
	}

	if (pintvals) free(pintvals);
	if (pstrs) free(pstrs);
    }
#endif /* 0 */
}


static void
service_proxy_aux_getvalue(
    aux_t *			aux,
    const unsigned char *	p,
    int				len
)
{
#if 0
    AUXComposed *pac = (AUXComposed*) p;

    if (pac->ev) {
	IMAuxGetValues(aux->ic, pac->ev);
    } else if (pac->pad) {
	int i;
	aux_data_t *pad = pac->pad;
	IIIMF_status st;
	IIIMCF_event ev;
	IIIMP_card32 *pintvals;
	const IIIMP_card16 **pstrs;

#ifdef ENABLE_TRACE
	static void show_AC(aux_data_t *pad);
	show_AC(pac->pad);
#endif
	if (pad->integer_count > 0) {
	    pintvals = (IIIMP_card32*) malloc(sizeof(*pintvals) * pad->integer_count);
	    if (!pintvals) return;
	    for (i = 0; i < pad->integer_count; i++) {
		pintvals[i] = pad->integer_list[i];
	    }
	} else {
	    pintvals = NULL;
	}
	if (pad->string_count > 0) {
	    pstrs = (const IIIMP_card16**) malloc(sizeof(*pstrs) * pad->string_count);
	    if (!pstrs) {
		if (pintvals) free(pintvals);
		return;
	    }
	    for (i = 0; i < pad->string_count; i++) {
		pstrs[i] = (const IIIMP_card16*) pad->string_list[i].ptr;
	    }
	} else {
	    pstrs = NULL;
	}

	st = iiimcf_create_aux_getvalues_event((IIIMP_card16*) pad->aux_name,
					       pad->aux_index,
					       pad->integer_count,
					       pintvals,
					       pad->string_count,
					       pstrs,
					       &ev);
	if (st == IIIMF_STATUS_SUCCESS) {
	    IMAuxGetValues(aux->ic, ev);
	}

	if (pintvals) free(pintvals);
	if (pstrs) free(pstrs);
    }
#endif /* 0 */
}

static int
service_proxy_im_id(
    aux_t *	aux
)
{
    return aux->im->im_id;
}


static int
service_proxy_ic_id(
    aux_t *	aux
)
{
    return aux->im->ic_id;
}


static void
service_proxy_data_set(
    aux_t *	aux,
    int		im_id,
    void *	data
)
{
    aux_im_data_t *	aux_im;

    for (aux_im = aux->im; NULL != aux_im; aux_im = aux_im->next) {
	if (im_id == aux_im->im_id) {
	    aux_im->data = data;
	}
    }

    return;
}


static void *
service_proxy_data_get(
    aux_t *	aux,
    int		im_id
)
{
    aux_im_data_t *	aux_im;

    for (aux_im = aux->im; NULL != aux_im; aux_im = aux_im->next) {
	if (im_id == aux_im->im_id) {
	    return aux_im->data;
	}
    }

    return NULL;
}


static Display *
service_proxy_display(
    aux_t *	aux
)
{
    xaux_proxy_ic_t *	ic;
    ic = (xaux_proxy_ic_t*)(aux->ic);
    return ic->display;
}


static Window
service_proxy_window(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_WINDOW, 0,
					  &data0_ret, NULL);
    if (False == b) {
	return None;
    } else {
	return (Window)(data0_ret);
    }
}


static XPoint *
service_proxy_point(
    aux_t *	aux,
    XPoint *	point
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_POINT, 0,
					  &data0_ret, NULL);
    if (False == b) {
	point->x = -1;
	point->y = -1;
    } else {
	point->x = ((0xFFFF0000 & data0_ret) >> 16);
	point->y = (0x0000FFFF & data0_ret);
    }

    return point;
}


static XPoint *
service_proxy_point_caret(
    aux_t *	aux,
    XPoint *	point
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_POINT_CARET, 0,
					  &data0_ret, NULL);
    if (False == b) {
	point->x = -1;
	point->y = -1;
    } else {
	point->x = ((0xFFFF0000 & data0_ret) >> 16);
	point->y = (0x0000FFFF & data0_ret);
    }

    return point;
}


static size_t
service_proxy_utf16_mb(
    const char **	inbuf,
    size_t *		inbytesleft,
    char **		outbuf,
    size_t *		outbytesleft
)
{
    int	r;

    r = IIimpConvertFromUTF16((char *)(*inbuf), *inbytesleft,
			      outbuf, outbytesleft);
    return (size_t)r;
}


static size_t
service_proxy_mb_utf16(
    const char **	inbuf,
    size_t *		inbytesleft,
    char **		outbuf,
    size_t *		outbytesleft
)
{
    int	r;

    r = IIimpConvertToUTF16((char *)(*inbuf), *inbytesleft,
			    outbuf, outbytesleft);
    return (size_t)r;
}


static unsigned char *
service_proxy_compose(
    const aux_data_t *	pad,
    int *		size
)
{
#if 0
    AUXComposed *	pac;

    pac = create_composed_from_aux_data(pad);
    return (unsigned char *)pac;
#else /* !0 */
    return NULL;
#endif /* !0 */
}


static int
service_proxy_compose_size(
    aux_data_type_t		type,
    const unsigned char *	p
)
{
    /* now this function is dummy... */
    return 0;
}


static aux_data_t *
service_proxy_decompose(
    aux_data_type_t		type,
    const unsigned char *	p
)
{
#if 0
    AUXComposed *pac = (AUXComposed*) p;

    if (pac->pad) {
	pac = create_composed_from_aux_data(pac->pad);
	if (!pac) return NULL;
	pac->pad->type = type;
#ifdef ENABLE_TRACE
	show_AC(pac->pad);
#endif
	return pac->pad;
    }

    if (pac->ev) {
	pac = create_composed_from_event(pac->aux, pac->ev);
	if (!pac) return NULL;
	pac->pad->type = type;
#ifdef ENABLE_TRACE
	show_AC(pac->pad);
#endif
	return pac->pad;
    }

#endif /* 0 */
    return NULL;
}


static void
service_proxy_decompose_free(
    aux_data_t *	pad
)
{
    if (NULL == pad) return;
    if (NULL != pad->string_ptr) return;
    free(pad->string_ptr);
}


static void
service_proxy_register_X_filter(
    Display *	display,
    Window	window,
    int		start_type,
    int		end_type,
    Bool	(* filter)(Display *, Window, XEvent *, XPointer),
    XPointer	client_data
)
{
    _XRegisterFilterByType(display, window, start_type, end_type,
			   filter, client_data);

    return;
}


static void
service_proxy_unregister_X_filter(
    Display *	display,
    Window	window,
    Bool	(* filter)(Display *, Window, XEvent *, XPointer),
    XPointer	client_data
)
{
    _XUnregisterFilter(display, window, filter, client_data);
}


static Bool
service_proxy_server(
    aux_t *	aux
)
{
    return True;
}


static Window
service_proxy_client_window(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_CLIENT_WINDOW, 0,
					  &data0_ret, NULL);
    if (False == b) {
	return None;
    } else {
	return (Window)(data0_ret);
    }
}


static Window
service_proxy_focus_window(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_FOCUS_WINDOW, 0,
					  &data0_ret, NULL);
    if (False == b) {
	return None;
    } else {
	return (Window)(data0_ret);
    }
}


static int
service_proxy_screen_number(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_SCREEN_NUMBER, 0,
					  &data0_ret, NULL);
    if (False == b) {
	return 0;
    } else {
	return data0_ret;
    }
}


static int
service_proxy_point_screen(
    aux_t *	aux,
    XPoint *	point
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;
    unsigned long	data1_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_POINT_SCREEN, 0,
					  &data0_ret, &data1_ret);
    if (False == b) {
	return -1;
    } else {
	point->x = ((0xFFFF0000 & data0_ret) >> 16);
	point->y = (0x0000FFFF & data0_ret);
	return data1_ret;
    }
}


static int
service_proxy_point_caret_screen(
    aux_t *	aux,
    XPoint *	point
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;
    unsigned long	data1_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_POINT_CARET_SCREEN,
					  0,
					  &data0_ret, &data1_ret);
    if (False == b) {
	return -1;
    } else {
	point->x = ((0xFFFF0000 & data0_ret) >> 16);
	point->y = (0x0000FFFF & data0_ret);
	return data1_ret;
    }
}


static Bool
service_proxy_get_conversion_mode(
    aux_t *	aux
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_GET_CONVERSION_MODE,
					  0,
					  &data0_ret, NULL);
    if (False == b) {
	return False;
    } else {
	return data0_ret;
    }
}

static void
service_proxy_set_conversion_mode(
    aux_t *	aux,
    int		conversion_mode
)
{
    Bool		b;
    xaux_proxy_ic_t *	ic;
    unsigned long	data0_ret;

    ic = (xaux_proxy_ic_t*)(aux->ic);

    b = xaux_proxy_message_short_exchange(ic->display,
					  ic->window_self, ic->window_target,
					  ic->atom, ic->server_id,
					  ic->im_id, ic->ic_id,
					  XAUX_PROXY_MESSAGE_TYPE_SERVICE,
					  XAUX_PROXY_SERVICE_SET_CONVERSION_MODE,
					  conversion_mode,
					  NULL, NULL);
    return;
}


static aux_t *
service_proxy_aux_get_from_id(
    int		im_id,
    int		ic_id,
    CARD16 *	aux_name,
    int		aux_name_length
)
{
    return NULL;
}


/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
