# -*- coding: utf-8 -*-
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for
# license information.
###########################################################################

import os
import sys
import pathlib
import setuptools

from setuptools import setup, find_packages

TOP_DIR = os.path.dirname(__file__) or os.getcwd()
PACKAGE_NAME = "onnxruntime_extensions"

# setup.py cannot be debugged in pip command line, so the command classes are refactored into another file
cmds_dir = pathlib.Path(TOP_DIR) / ".pyproject"
sys.path.append(str(cmds_dir))
# noinspection PyUnresolvedReferences
import cmdclass as _cmds  # noqa: E402

_cmds.prepare_env(TOP_DIR)


# read version from the package file.
def read_version():
    version_str = "0.1.0"
    with open(os.path.join(TOP_DIR, "version.txt"), "r") as f:
        version_str = f.readline().strip()

    # special handling for Onebranch building
    if os.getenv("BUILD_SOURCEBRANCHNAME", "").startswith("rel-"):
        return version_str

    # is it a dev build or release?
    rel_br, cid = _cmds.read_git_refs(TOP_DIR) if os.path.isdir(os.path.join(TOP_DIR, ".git")) else (True, None)

    if rel_br:
        return version_str

    build_id = os.getenv("BUILD_BUILDID", None)
    if build_id is not None:
        version_str += ".{}".format(build_id)
    else:
        version_str += "+" + cid[:7]
    return version_str


def write_py_version(ext_version):
    text = ["# Generated by setup.py, DON'T MANUALLY UPDATE IT!\n", '__version__ = "{}"\n'.format(ext_version)]
    with open(os.path.join(TOP_DIR, "onnxruntime_extensions/_version.py"), "w") as _fver:
        _fver.writelines(text)


ext_modules = [setuptools.extension.Extension(name=str("onnxruntime_extensions._extensions_pydll"), sources=[])]

packages = find_packages()
package_dir = {k: os.path.join(".", k.replace(".", "/")) for k in packages}
package_data = {
    "onnxruntime_extensions": ["*.so", "*.pyd"],
}

long_description = ""
with open(os.path.join(TOP_DIR, "README.md"), "r", encoding="utf-8") as _f:
    long_description += _f.read()
    start_pos = long_description.find("# Introduction")
    start_pos = 0 if start_pos < 0 else start_pos
    end_pos = long_description.find("# Contributing")
    long_description = long_description[start_pos:end_pos]
ortx_version = read_version()
write_py_version(ortx_version)

setup(
    name=PACKAGE_NAME,
    version=ortx_version,
    packages=packages,
    package_dir=package_dir,
    package_data=package_data,
    description="ONNXRuntime Extensions",
    long_description=long_description,
    long_description_content_type="text/markdown",
    license="MIT License",
    author="Microsoft Corporation",
    author_email="onnxruntime@microsoft.com",
    url="https://github.com/microsoft/onnxruntime-extensions",
    ext_modules=ext_modules,
    cmdclass=_cmds.ortx_cmdclass,
    include_package_data=True,
    install_requires=[],
    classifiers=[
        "Development Status :: 4 - Beta",
        "Environment :: Console",
        "Intended Audience :: Developers",
        "Operating System :: MacOS :: MacOS X",
        "Operating System :: Microsoft :: Windows",
        "Operating System :: POSIX :: Linux",
        "Programming Language :: C++",
        "Programming Language :: Python",
        "Programming Language :: Python :: Implementation :: CPython",
        "License :: OSI Approved :: MIT License",
    ],
)
