---
name: Enhance release changelog with AI

on:
  release:
    types: [published]
  workflow_dispatch:
    inputs:
      tag:
        description: 'Release tag to test (e.g., v19.0.0)'
        required: true
        type: string
      dry_run:
        description: 'Dry run mode - generate changelog but do not update release'
        required: false
        type: choice
        options:
          - 'true'
          - 'false'
        default: 'true'

permissions:
  contents: write
  models: read
  actions: read
  discussions: read
  issues: read
  pull-requests: read
  repository-projects: read
  security-events: read

timeout_minutes: 15

tools:
  github:
    toolsets: [all]
  bash: ["git", "cat", "echo", "jq", "curl", "head", "wc", "grep", "sed", "sort"]
  web-fetch:

engine:
  id: copilot

steps:
  - name: Checkout repository
    uses: actions/checkout@v4
    with:
      fetch-depth: 0
      persist-credentials: false

---

# Enhanced Changelog Generator for oh-my-posh

You are a release notes editor for the open-source project "oh-my-posh", a cross-shell prompt theme engine written in Go.

## Your Task

Generate a clear, human-friendly Markdown changelog for this release. Use concise language and organize with headings.

## Context

- Repository: `${{ github.repository }}`
- Release Tag: `${{ github.event.inputs.tag || github.event.release.tag_name }}`
- Release ID: `${{ github.event.release.id }}`
- Dry Run: `${{ github.event.inputs.dry_run }}`

**IMPORTANT Git History**: The repository is checked out with full history (`fetch-depth: 0`), so all tags and commits are available.

**IMPORTANT Release Data**: Release event data is available in the GitHub context:
- For release events: Release information is available via API
- Release ID: `${{ github.event.release.id }}`
- Release tag: `${{ github.event.release.tag_name }}`
- Release name: `${{ github.event.release.name }}`
- Use API to fetch full release details including body

**IMPORTANT**: You are running in a standard GitHub Actions environment. You CAN:
- Use the GitHub MCP tool to call GitHub API (e.g., `get_release`, `get_repository`, etc.)
- Run git commands on the full repository history  
- Use allowed bash commands: git, cat, echo, jq, curl, head, wc, grep, sed, sort
- Use `echo`, bash variables, and heredocs to create temporary files in `/tmp/gh-aw/agent/`
- Write to `$GITHUB_STEP_SUMMARY`

Do NOT attempt to:
- Access files outside `/home/runner/work/` or `/tmp/gh-aw/` without user permission
- Use interactive prompts

## Step-by-Step Process

### 1. Gather Release Context

Use the GitHub API tool to get release information:

```bash
# Get release details by tag or ID
# Use github tool: get_release with owner, repo, and release_id or tag
```

For release events, the release ID is available as: `${{ github.event.release.id }}`
For workflow_dispatch, use the tag: `${{ github.event.inputs.tag }}`

Store the release ID for later use when updating the release body.

### 2. Determine Diff Range

Find the previous tag to establish the comparison range:

```bash
# Get current tag
CURRENT_TAG="${{ github.event.inputs.tag || github.event.release.tag_name }}"

# Find the previous tag - list tags in reverse version order and get the one before current
PREV_TAG=$(git tag --sort=-version:refname | grep -A1 "^$CURRENT_TAG$" | tail -n1 | grep -v "^$CURRENT_TAG$")

# If no previous tag, use first commit (handle multiple root commits)
if [ -z "$PREV_TAG" ] || [ "$PREV_TAG" = "$CURRENT_TAG" ]; then
  PREV_TAG=$(git rev-list --max-parents=0 HEAD | head -n1)
fi

echo "Comparing $PREV_TAG...$CURRENT_TAG"
```

### 3. Collect Commits and Changes

Gather the following information using git commands:

- **Commit subjects**: `git log --no-merges --pretty=format:'%s' "$PREV_TAG...$CURRENT_TAG"`
- **Detailed commits**: `git log --no-merges --pretty=format:'- %s%n%b%n' "$PREV_TAG...$CURRENT_TAG"`
- **Changed files**: `git diff --name-status "$PREV_TAG...$CURRENT_TAG"`
- **Contributors**: Use `git shortlog -sne "$PREV_TAG...$CURRENT_TAG"` to extract contributors
  - Format as GitHub profile links: `[@username](https://github.com/username)`
  - Exclude: Jan De Dobbeleer, dependabot, renovate, github-actions, and other bots
  - Extract GitHub username from email addresses (e.g., noreply@github.com patterns)

### 4. Collect Issue Context

Extract issue numbers from commit messages and fetch their details for context:

```bash
# Extract issue numbers from commits (patterns: fixes #123, closes #456, #789)
git log --no-merges --pretty=format:'%s %b' "$PREV_TAG...$CURRENT_TAG" | \
  grep -oiE '(fix(es|ed)?|close(s|d)?|resolve(s|d)?)?[[:space:]]*#[0-9]+' | \
  grep -oE '[0-9]+' | sort -u
```

For each unique issue number found, use the GitHub API tool to fetch details:
- Use `get_issue` with owner, repo, and issue_number parameters

Limit to the first 20 issues if there are many. Use issue titles and labels to provide context in the changelog.

### 5. Read Version Configuration

Read the `.versionrc.json` file from the repository to understand which commit types should be included.

**CRITICAL**: Respect the .versionrc.json configuration:

- ONLY include these sections with these exact names:
  - "Features" (for feat: commits)
  - "Bug Fixes" (for fix: commits)
  - "Refactor" (for refactor: commits)
  - "Reverts" (for revert: commits)
  - "Themes" (for theme: commits)
- DO NOT include chore, ci, docs, perf, or test commits (marked as hidden in .versionrc.json)
- Use ONLY the section names specified above, not generic names like "Other"
- ONLY show sections that have actual changes - omit empty sections entirely

### 6. Generate Enhanced Changelog

Create a comprehensive changelog that includes:

**Segment Changes** (public-facing):

- When you see changes to `src/segments/*.go` files (excluding `*_test.go`), these are prompt segments that users
  configure
- A segment is a customizable component users add to their shell prompt (e.g., git status, battery level, current
  directory)
- Mention segment changes by their user-facing name (infer from the file name), not file paths
- Focus on what users can now do or configure differently with that segment
- **IMPORTANT**: For segment property additions or new enum values:
  - Generate a minimal, valid JSON example showing the new property or value
  - Validate segment examples using curl to call the MCP validation endpoint:
    ```bash
    # Example: validate a segment JSON (escape inner quotes and newlines)
    curl -X POST https://ohmyposh.dev/api/mcp \
      -H "Content-Type: application/json" \
      -d '{
        "jsonrpc": "2.0",
        "method": "tools/call",
        "params": {
          "name": "validate_segment",
          "arguments": {
            "content": "{\"type\": \"path\", \"style\": \"powerline\", \"foreground\": \"#ffffff\", \"background\": \"#0000ff\", \"template\": \"{{ .Path }}\"}",
            "format": "json"
          }
        },
        "id": 1
      }'
    ```
  - Parse the JSON response to check if `result.content[0].text` contains `"valid": true`
  - Only include validated examples in the changelog
  - Example format:
    ```json
    {
      "type": "segment-name",
      "style": "powerline",
      "foreground": "#ffffff",
      "background": "#0000ff",
      "template": "example",
      "new_property": "value"
    }
    ```

**Configuration Changes** (when schema changes affect full configurations):

- For new configuration-level properties (not segment-specific), provide minimal valid configuration examples
- Validate configuration examples using curl to call the MCP validation endpoint:
  ```bash
  curl -X POST https://ohmyposh.dev/api/mcp \
    -H "Content-Type: application/json" \
    -d '{"jsonrpc":"2.0","method":"tools/call","params":{"name":"validate_config","arguments":{"content":"...","format":"json"}},"id":1}'
  ```
- Parse the JSON response to check if `result.content[0].text` contains `"valid": true`
- Only include validated examples in the changelog
- Example format:
  ```json
  {
    "$schema": "https://raw.githubusercontent.com/JanDeDobbeleer/oh-my-posh/main/themes/schema.json",
    "version": 2,
    "new_property": "value",
    "blocks": []
  }
  ```

**Goals**:

- Summarize highlights up front with context and impact
- Group changes ONLY by the section names from .versionrc.json above (skip empty sections)
- Call out breaking changes and required migrations with explicit before/after examples or commands
- Add practical usage notes or snippets to help users adopt new features or changes
- For segment changes with new properties/values, include validated JSON examples showing usage (use curl to validate)
- For configuration changes, include validated minimal configuration examples (use curl to validate)
- Credit contributors at the end (they are pre-filtered and formatted as GitHub profile links) - ONLY if contributors
  list is not empty
- Include a "Full diff" link footer

**Requirements**:

- Output valid Markdown only, no front matter, no HTML, no title heading
- Do not include a title like "Changelog for vX.Y.Z" - start directly with the content
- Keep to ~300-800 words unless there are many breaking changes
- Prefer code blocks for examples with proper language tags (bash, json, yaml, toml, powershell)
- **CRITICAL**: All JSON segment and configuration examples MUST be validated using curl to call the MCP server:
  - For segments: `curl -X POST https://ohmyposh.dev/api/mcp -H "Content-Type: application/json" -d '{"jsonrpc":"2.0","method":"tools/call","params":{"name":"validate_segment","arguments":{"content":"JSON_HERE","format":"json"}},"id":1}'`
  - For configs: `curl -X POST https://ohmyposh.dev/api/mcp -H "Content-Type: application/json" -d '{"jsonrpc":"2.0","method":"tools/call","params":{"name":"validate_config","arguments":{"content":"JSON_HERE","format":"json"}},"id":1}'`
  - Parse response and check for `"valid": true` in the result
  - If validation fails, fix the example and validate again
  - Do NOT include invalid examples in the changelog
- Do not invent features not present in the commits/diff
- Do not list individual file paths unless they are user-facing config/theme files

### 7. Update Release Body

**Check dry run mode first**: If `${{ github.event.inputs.dry_run }}` equals `'true'` (string), do NOT update the release.

If NOT in dry run mode (value is `'false'` or not provided for release events), use the GitHub API tool to update the release:

- Use the appropriate GitHub API tool method to update the release body
- Pass the release ID and new body content

Where RELEASE_ID comes from the release info fetched in step 1.

If in dry run mode:
- Display the generated changelog clearly
- Add a prominent note that this is a DRY RUN and no changes were made

### 8. Generate Summary

Output the enhanced changelog to the GitHub Action summary and in your response:

```bash
# Write to GitHub Action summary
cat << 'EOF' >> $GITHUB_STEP_SUMMARY
[Your enhanced changelog content here]
EOF
```

Display in your response with clear formatting:

- If dry run: Show a clear "🔍 DRY RUN MODE" header and note that no changes were made
- If not dry run: Show a "✅ RELEASE UPDATED" header confirming the update
- Display the full enhanced changelog
- Include basic statistics (commit count, comparison range)

## Output Format

The enhanced changelog should be structured as:

```markdown
[Opening paragraph summarizing key highlights and impact]

## Features
[New features from feat: commits]

## Bug Fixes
[Bug fixes from fix: commits]

## Refactor
[Refactoring changes from refactor: commits]

## Reverts
[Reverted changes from revert: commits]

## Themes
[Theme changes from theme: commits]

## Contributors
[List of contributor links - only if list is not empty]

---
**Full Changelog**: [compare link]
```

## Important Notes

- Use the GitHub MCP tool for all GitHub API operations (not `gh` CLI)
- Use git commands for repository analysis
- Use bash commands (git, cat, echo, jq, curl, head, wc, grep, sed, sort) as needed
- Handle errors gracefully and provide clear feedback
- Respect the conventional commit format used in oh-my-posh
- Focus on user-facing impact, not internal implementation details
