///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoFactor.cc
// -------------
// Cego procedure factor structure class implementation                                    
//
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2019 Bjoern Lemke 
//
// IMPLEMENTATION MODULE
//
// Class: CegoFactor
// 
// Description: Container class to describe a factor element
//
// Status: CLEAN
// 
///////////////////////////////////////////////////////////////////////////////

// INCLUDES
#include <lfcbase/Exception.h>

#include "CegoFactor.h"
#include "CegoExpr.h"
#include "CegoSelect.h"
#include "CegoCaseCond.h"
#include "CegoXMLdef.h"
#include "CegoTypeConverter.h"

#include "CegoDatabaseFormater.h"

#include <string.h>
#include <stdlib.h>

CegoFactor::CegoFactor(char* buf, CegoDistManager *pGTM, int tabSetId)
{
    _pAttrDesc = 0;
    _pExpr = 0;
    _pFetch = 0;
    _pBlock = 0;
    _pAggr = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
    decode(buf, pGTM, tabSetId);
}

CegoFactor::CegoFactor(Element* pFactorElement, CegoDistManager *pGTM)
{
    _pAttrDesc = 0;
    _pExpr = 0;
    _pFetch = 0;
    _pBlock = 0;
    _pAggr = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
    fromElement(pFactorElement, pGTM);
}

CegoFactor::CegoFactor(const CegoFactor& pf)
{
    *this = pf;
}

CegoFactor::CegoFactor(const Chain& varName)
{
    _type = VAR;
    _varName = varName;
    _pAttrDesc = 0;
    _pExpr = 0;
    _pFetch = 0;
    _pBlock = 0;
    _pAggr = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
}

CegoFactor::CegoFactor(const CegoFieldValue& fv)
{
    _type = CONSTVAL;
    _fv = fv;
    _pAttrDesc = 0;
    _pExpr = 0;
    _pBlock = 0;
    _pFetch = 0;
    _pAggr = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
}

CegoFactor::CegoFactor(CegoExpr *pExpr)
{
    _type = EXPR;
    _pAttrDesc = 0;
    _pExpr = pExpr;
    _pFetch = 0;
    _pBlock = 0;
    _pAggr = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
}

CegoFactor::CegoFactor(CegoProcFetch *pFetch)
{
    _type = FETCH;
    _pAttrDesc = 0;
    _pExpr = 0;
    _pFetch = pFetch;
    _pBlock = 0;
    _pAggr = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
}

CegoFactor::CegoFactor(CegoAttrDesc* pAttrDesc, bool isAttrRef)
{
    _type = ATTR;
    _isAttrRef = isAttrRef;
    _pAttrDesc = pAttrDesc;
    _pFetch = 0;
    _pBlock = 0;
    _pExpr = 0;
    _pAggr = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
}

CegoFactor::CegoFactor(CegoFunction* pFunction)
{
    _type = FUNCTION;
    _pFunction = pFunction;
    _pBlock = 0;
    _pAttrDesc = 0;
    _pExpr = 0;
    _pFetch = 0;
    _pAggr = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
}

CegoFactor::CegoFactor(CegoAggregation* pAggr)
{
    _type = AGGREGATION;
    _pAggr = pAggr;
    _pBlock = 0;
    _pFetch = 0;
    _pAttrDesc = 0;
    _pExpr = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pFLA = 0;
    _pCaseCond = 0;
    _flaCached = false;
}

CegoFactor::CegoFactor(CegoSelect* pSelect)
{
    _type = QUERY;
    _pBlock = 0;
    _pAttrDesc = 0;
    _pExpr = 0;
    _pFetch = 0;
    _pAggr = 0;
    _pFLA = 0;
    _pFunction = 0;
    _pSelect = pSelect;
    _pCaseCond = 0;
    _flaCached = false;
}

CegoFactor::CegoFactor(CegoCaseCond* pCaseCond)
{
    _type = CASECOND;
    _pBlock = 0;
    _pAttrDesc = 0;
    _pExpr = 0;
    _pFetch = 0;
    _pAggr = 0;
    _pFLA = 0;
    _pFunction = 0;
    _pSelect = 0;
    _pCaseCond = pCaseCond;
    _flaCached = false;
}

CegoFactor::~CegoFactor()
{
    if ( _pAttrDesc && _isAttrRef == false)
	delete _pAttrDesc;
    if ( _pExpr )
	delete _pExpr;
    if ( _pFetch )
	delete _pFetch;
    if ( _pFunction )
	delete _pFunction;
    if ( _pAggr )
	delete _pAggr;
    if ( _pSelect )
	delete _pSelect;
    if ( _pCaseCond )
	delete _pCaseCond;
}    

void CegoFactor::cleanUp()
{
    switch ( _type ) 
    {
    case ATTR:
    {
	_pFLA = 0;
	break;
    }
    case EXPR:
    {
	_pExpr->cleanUp();
	break;
    }
    case FUNCTION:
    {
	_pFunction->cleanUp();
	break;
    }
    case AGGREGATION:
    {
	_pAggr->cleanUp();
	break;
    }
    case QUERY:
    {
	_pSelect->cleanUp();
	break;
    }
    case CASECOND:
    {
	_pCaseCond->cleanUp();
	break;
    }
    case CONSTVAL:
    case VAR:
    case FETCH:
	// TODO : check if fetch cursor should be reset
	break;
    }
}

void CegoFactor::setTabSetId(int tabSetId)
{
    switch ( _type ) 
    {
    case EXPR:
    {
	_pExpr->setTabSetId(tabSetId);
	break;
    }
    case FUNCTION:
    {
	_pFunction->setTabSetId(tabSetId);
	break;
    }
    case AGGREGATION:
    {
	// TODO
	// _pAggr->setTabSetId(tabSetId);
	break;
    }
    case QUERY:
    {
	_pSelect->setTabSetId(tabSetId);
	break;
    }
    case CASECOND:
    {
	_pCaseCond->setTabSetId(tabSetId);
	break;
    }
    default:
	// nothing to do
	break;
    }
}


void CegoFactor::setFieldListArray(ListT<CegoField>** pFLA)
{

    /*
    cout << "Setting FieldListArray for " << toChain("") <<  " : "  << endl;
    if ( pFLA )
    {
	int i=0;
	while ( pFLA[i] )
	{
	    
	    CegoField *pXF = pFLA[i]->First();
	    while ( pXF )
	    {
		cout << "FLA : Table = " << pXF->getTableName() << "(" << pXF->getTableAlias() << ")." << pXF->getAttrName() << endl;
		pXF = pFLA[i]->Next();
	    }
	    i++;
	}
    }
    */
    
    // we have to reset fla cache, since we set a new value for it
    // otherwise, we might point to invalid memory

    // for performance reasons, we don't reset _flaCached
    // so we have to make sure, pFLA structure keeps the same
    
    // if ( pFLA != _pFLA )
    //	_flaCached = false;
    
    // cout << "Setting field list of size " << size  << endl;
    switch ( _type ) 
    {
    case ATTR:
    {
	_pFLA = pFLA;
	break;		
    }
    case EXPR:
    {
	_pExpr->setFieldListArray(pFLA);
	break;
    }
    case FUNCTION:
    {
	_pFunction->setFieldListArray(pFLA);
	break;
    }
    case AGGREGATION:
    {
	_pAggr->setFieldListArray(pFLA);
	break;
    }
    case QUERY:
    {
	// cout << "Setting parent join buf for query " << _pSelect->toChain(" ") << endl;
	_pSelect->setParentJoinBuf(pFLA);
	_pSelect->prepare();
	break;
    }
    case CASECOND:
    {
	_pCaseCond->setFieldListArray(pFLA);
	break;
    }
    case CONSTVAL:
    case VAR:
    case FETCH:
	break;
    }
}

void CegoFactor::setBlock(CegoProcBlock *pBlock)
{
    switch ( _type ) 
    {
    case VAR:
    {
	_pBlock = pBlock;
	break;
    }
    case EXPR:
    {
	_pExpr->setBlock(pBlock);
	break;
    }
    case FETCH:
    {
	_pFetch->setBlock(pBlock);
	break;
    }
    case FUNCTION:
    {
	_pFunction->setBlock(pBlock);
	break;
    }
    case AGGREGATION:
    {
	_pAggr->setBlock(pBlock);
	break;
    }
    case QUERY:
    {
	_pSelect->setProcBlock(pBlock);
	_pSelect->prepare();
	break;
    }
    case CASECOND:
    {
	_pCaseCond->setBlock(pBlock);
	break;
    }
    case CONSTVAL:
    case ATTR:
    {
	// nothing to do
	break;
    }
    }
}

void CegoFactor::clearAttrCache()
{
    // cout << "Clearing attr cache .." << endl;
    if ( _pAttrDesc )
	_flaCached = false;   
    if ( _pExpr )
	_pExpr->clearAttrCache();
    if ( _pFunction )
	_pFunction->clearAttrCache();
    if ( _pAggr )
        _pAggr->clearAttrCache();
    if ( _pCaseCond )
	_pCaseCond->clearAttrCache();
}

void CegoFactor::getFunctionList(SetT<CegoObject>& funcList)
{
    if ( _type == FUNCTION )
    {
	funcList.Insert(CegoObject(CegoObject::PROCEDURE, _pFunction->getName(), _pFunction->getTabSetId()));
    }
}


CegoFactor& CegoFactor::operator = ( const CegoFactor& pf)
{
    _fv = pf._fv;
    _varName = pf._varName;
    _pAttrDesc = pf._pAttrDesc;
    _isAttrRef = pf._isAttrRef;
    _pExpr = pf._pExpr;
    _pFetch = pf._pFetch;
    _pFunction = pf._pFunction;
    _pAggr = pf._pAggr;
    _pSelect = pf._pSelect;
    _type = pf._type;
    _pBlock = pf._pBlock;
    _pCaseCond = pf._pCaseCond;
    _flaCached = pf._flaCached;
    _pFLA = pf._pFLA;
    return (*this);
}

bool CegoFactor::checkConst() const
{
    return _type == CONSTVAL;
}

bool CegoFactor::checkVar(Chain& var) const
{
    if ( _type == VAR )
    {
	var = _varName;
	return true;
    }
    return false;
}

CegoAttrDesc* CegoFactor::checkAttr() const
{
    if ( _type == ATTR )
    {
	return _pAttrDesc;
    }
    return 0;
}

bool CegoFactor::checkLob(CegoFieldValue& fv)
{
    switch ( _type )
    {
    case CONSTVAL:
    {
	if ( _fv.getType() == BLOB_TYPE || _fv.getType() == CLOB_TYPE )
	{
	    fv = _fv;
	    return true;
	}
	break;
    }
    case VAR:
    {
	CegoFieldValue fv;
	if ( _pBlock )
	{
	    fv = _pBlock->getValue(_varName);
	    if ( fv.getType() == BLOB_TYPE || fv.getType() == CLOB_TYPE )
	    {
		return true;		
	    }
	}
	break;
    }
    case EXPR:
    case FETCH:
    case ATTR:
    case FUNCTION:
    case QUERY:
    case AGGREGATION:
    case CASECOND:
	return false;
    }
    return false;
}

CegoAttrDesc* CegoFactor::getAttr() const
{
    return _pAttrDesc;
}

CegoFunction* CegoFactor::getFunction() const
{
    return _pFunction;
}

CegoAggregation* CegoFactor::getAggregation() const
{
    return _pAggr;
}

CegoSelect* CegoFactor::getSelect() const
{
    return _pSelect; 
}

CegoCaseCond* CegoFactor::getCaseCond() const
{
    return _pCaseCond;
}

void CegoFactor::getFieldList(ListT<CegoField>& fl) const
{
    switch ( _type )
    {
	
    case CegoFactor::ATTR:
    {
	if ( _pFLA == 0)
	{
	    Chain msg = Chain("Cannot get value for attribute " + _pAttrDesc->toChain());
	    throw Exception(EXLOC, msg);
	}
	
	int i=0;
	CegoField* pF = 0;
	while ( _pFLA[i] && pF == 0 )
	{		    
	    if ( ( pF = _pFLA[i]->Find(CegoField(_pAttrDesc->getTableName(), _pAttrDesc->getAttrName()))) != 0 )
	    {
		fl.Insert(*pF);	       
	    }
	    i++;
	}
	break;
    }
    case CegoFactor::EXPR:
    {
	_pExpr->getFieldList(fl);
	break;
    }
    case CegoFactor::FUNCTION:
    {
	_pFunction->getFieldList(fl);
	break;
    }
    case CegoFactor::AGGREGATION:
    {
	if ( _pAggr->getExpr() )
	    _pAggr->getExpr()->getFieldList(fl);
	break;
    }
    case CegoFactor::CASECOND:
    {
	_pCaseCond->getFieldList(fl);
	break;
    }
    case CegoFactor::QUERY:
    {
	_pSelect->getFieldList(fl);
	break;
    }
    case CONSTVAL:
    case VAR:
    case FETCH:
	break;
    }
}

ListT<CegoAttrDesc*> CegoFactor::getAttrRefList() const
{	
    ListT<CegoAttrDesc*> attrList;

    switch ( _type )
    {
	
    case CegoFactor::ATTR:
    {
	attrList.Insert(_pAttrDesc);
	break;
    }
    case CegoFactor::EXPR:
    {
	attrList = _pExpr->getAttrRefList();
	break;
    }
    case CegoFactor::FUNCTION:
    {
	attrList = _pFunction->getAttrRefList();
	break;
    }
    case CegoFactor::AGGREGATION:
    {
	if ( _pAggr->getExpr() )
	    attrList = _pAggr->getExpr()->getAttrRefList();
	break;
    }
    case CegoFactor::QUERY:
    {
	attrList = _pSelect->getAttrRefList();
	break;
    }
    case CegoFactor::CASECOND:
    {
	attrList = _pCaseCond->getAttrRefList();
	break;
    }
    case CONSTVAL:
    case VAR:
    case FETCH:
	break;

    }
    return attrList;
}

int CegoFactor::evalReferences(CegoContentObject *pCO, const ListT<CegoField>& fl)
{
    int refCount = 0;

    switch ( _type )
    {

    case CegoFactor::ATTR:
    {
	refCount += _pAttrDesc->evalReferences(pCO, fl);
	break;
    }
    case CegoFactor::EXPR:
    {
	refCount += _pExpr->evalReferences(pCO, fl);
	break;
    }
    case CegoFactor::FUNCTION:
    {
	refCount += _pFunction->evalReferences(pCO, fl);
	break;
    }
    case CegoFactor::AGGREGATION:
    {
	if ( _pAggr->getExpr() )
	{
	    refCount += _pAggr->getExpr()->evalReferences(pCO, fl);
	}
	break;
    }
    case CegoFactor::QUERY:
    {
	// reset of parent join buf is done via cleanUp
	// _pSelect->setParentJoinBuf();
	_pSelect->prepare();
	refCount += _pSelect->evalExtTableReferences(pCO, fl);
	break;
    }
    case CegoFactor::CASECOND:
    {
	refCount += _pCaseCond->evalReferences(pCO, fl);
	break;
    }
    case CONSTVAL:
    case VAR:
    case FETCH:
	break;

    }

    return refCount;
}

CegoField CegoFactor::evalField(const ListT<CegoField>& fl) const
{
    CegoField f;

    switch ( _type )
    {
    case CegoFactor::CONSTVAL:
    {
	// cout << "GOT Len " << _fv.getLength() << endl;
	// cout << "GOT dim " << _fv.getDim() << endl;
	
	// f = CegoField("CONST", "CONST", "CONST", _fv.getType(), _fv.getDim() );
	f = CegoField("CONST", "CONST", "CONST", _fv.getType(), _fv.getLength(), _fv.getDim() );
	break;
    }
    case CegoFactor::EXPR:
    {
	return _pExpr->evalField(fl);
    }
    case CegoFactor::FETCH:
    {
	f = CegoField("FETCH", "FETCH", "FETCH", BOOL_TYPE, 1 );
	break;
    }
    case CegoFactor::VAR:
    {
	f = CegoField("VAR", "VAR", "VAR", VARCHAR_TYPE, 20 );
	break;
    }
    case CegoFactor::ATTR:
    {
	if ( fl.Size() == 0 )
	{       	
	    f = CegoField(_pAttrDesc->getTableName(), 
			  _pAttrDesc->getAttrName());
	    
	    // Chain msg = Chain("Fieldlist not set for " + _attrDesc.toChain());
	    // throw Exception(EXLOC, msg);
	}
	else
	{

	    CegoField *pF = fl.Find(CegoField(_pAttrDesc->getTableName(), _pAttrDesc->getAttrName()));
	    if ( pF )
	    {
		f = CegoField(_pAttrDesc->getTableName(), 
			      _pAttrDesc->getTableName(),
			      _pAttrDesc->getAttrName(),
			      pF->getType(),
			      pF->getLength(),
			      pF->getDim());
	    }
	    else
	    {
		Chain msg("Unknown attribute field <" + _pAttrDesc->getTableName() + "." + _pAttrDesc->getAttrName() + ">");
		throw Exception(EXLOC, msg);
	    }
	}
	break;
    }
    case CegoFactor::FUNCTION:
    {
	f = CegoField("FUNC", "FUNC", _pFunction->toChain(), _pFunction->getReturnType(), _pFunction->getReturnTypeLen(fl));
	break;
    }
    case CegoFactor::AGGREGATION:
    {
	if ( _pAggr->getExpr() )
	{
	    CegoField af = _pAggr->getExpr()->evalField(fl);
	    if ( _pAggr->getType() == CegoAggregation::COUNT )
	    {
		f = CegoField("AGGR", "AGGR", _pAggr->toChain(), LONG_TYPE, sizeof(long long));
	    }
	    else
	    {
		f = CegoField("AGGR", "AGGR", _pAggr->toChain(), af.getType(), af.getLength(), af.getDim() );
	    }
	}
	else
	{
	    // must be count(*)
	    f = CegoField("AGGR", "AGGR", _pAggr->toChain(), LONG_TYPE, sizeof(long long));
	}
	break;
    }
    case CegoFactor::QUERY:
    {
	ListT<CegoField> schema;
	_pSelect->getSchema(schema);
	CegoField *pF = schema.First();
	if ( pF )
	    f = *pF;
	
	// f = CegoField("QUERY", "QUERY", Chain("select(..)"), VARCHAR_TYPE, 20 );
	break;
    }
    case CegoFactor::CASECOND:
    {
	return _pCaseCond->evalField(fl);
    }
    default:
	break;
    }
    return f;
}

CegoFieldValue CegoFactor::evalFieldValue()
{
    
    switch ( _type )
    {
    case CONSTVAL:
    {
	return _fv;
    }
    case EXPR:
    {
	return _pExpr->evalFieldValue();
    }
    case FETCH:
    {
	char *pC = new (char);
	*pC = 1;
	CegoFieldValue fv(BOOL_TYPE, pC, 1, true);

	if ( _pFetch->fetch() )
	    *pC = 1;
	else
	    *pC = 0;

	return fv;
    }
    case VAR:
    {
	CegoFieldValue fv;
	if ( _pBlock )
	{
	    fv = _pBlock->getValue(_varName);
	}
	return fv;
    }
    case CegoFactor::ATTR:
    {

	// cout << "Checking " << _pAttrDesc->getTableName() << " / " << _pAttrDesc->getAttrName() << endl;
	
	if ( _pFLA == 0 )
	{
	    Chain msg = Chain("Cannot get value for attribute " + _pAttrDesc->toChain());
	    throw Exception(EXLOC, msg);
	}
	
	if ( _flaCached )
	{
	    // cout << "Using cached position " << _flaCacheArrayPos << "/" << _flaCacheFieldPos << endl;
	    return (*_pFLA[_flaCacheArrayPos])[_flaCacheFieldPos].getValue();
	}
	
	CegoField *pF = 0;
	
	_flaCacheArrayPos=0;
	while ( _pFLA[_flaCacheArrayPos] && pF == 0)
	{
	    CegoField *pF = _pFLA[_flaCacheArrayPos]->First();

	    _flaCacheFieldPos = 0;
	    while ( pF )
	    {
		// cout << "Checking  " << pF->getTableAlias()  << "/" <<  pF->getTableName() << "/" << pF->getAttrName() << endl;
		if ( *pF == CegoField(_pAttrDesc->getTableName(), _pAttrDesc->getAttrName()))
		{
		    // cout << "Found value at position " << _flaCacheArrayPos << "/" << _flaCacheFieldPos << endl;
		    _flaCached = true;
		    return pF->getValue();
		}  
		pF = _pFLA[_flaCacheArrayPos]->Next();
		_flaCacheFieldPos++;
	    }
	    _flaCacheArrayPos++;
	}
	if ( pF == 0 )
	{
	    Chain msg("Unknown attribute field <" + _pAttrDesc->getTableName() + "." + _pAttrDesc->getAttrName() + ">");
	    throw Exception(EXLOC, msg);
	}
    }
    case CegoFactor::FUNCTION:
    {
	return _pFunction->evalFieldValue();	
    }
    case CegoFactor::AGGREGATION:
    {
	return _pAggr->getFieldValue();	
    }
    case CegoFactor::QUERY:
    {
	try 
	{
	    _pSelect->prepare();
	    _pSelect->reset(false);

	    ListT<CegoField> fl;
	   
	    if ( ! _pSelect->nextTuple(fl) )
	    {
		_pSelect->reset(true);
		return CegoFieldValue();
	    }
	    else
	    {
		// if fetch result is just one, we can complete and add to query cache ( if enabled ) 
		ListT<CegoField> nfl;
		_pSelect->nextTuple(nfl);
	    }
	    
	    CegoField *pSF = fl.First();
	    _pSelect->reset(true);
	    CegoFieldValue val = pSF->getValue();
	    return pSF->getValue();
	}
	catch ( Exception e ) 
	{
	    _pSelect->cleanUp();
	    throw e;
	}
    }
    case CegoFactor::CASECOND:
    {
	return _pCaseCond->evalFieldValue();
    }
    default:
	break;
    }
}

ListT<CegoAggregation*> CegoFactor::getAggregationList()
{
    ListT<CegoAggregation*> aggList;

    switch ( _type )
    {
    case CegoFactor::ATTR:
    case CegoFactor::CONSTVAL:
    case CegoFactor::VAR:
    {
	// nothing to add
	break;
    }
    case CegoFactor::EXPR:
    {
	aggList = _pExpr->getAggregationList();
	break;
    }
    case CegoFactor::FUNCTION:
    {
	aggList = _pFunction->getAggregationList();
	break;
    }
    case CegoFactor::QUERY:
    {
	throw Exception(EXLOC, "Query not supported in aggregation");
    }
    case CegoFactor::CASECOND:
    {
	aggList = _pCaseCond->getAggregationList();
	break;
    }
    case CegoFactor::FETCH:
    {
	throw Exception(EXLOC, "Fetch not supported in aggregation");
    }
    case CegoFactor::AGGREGATION:
    {
	aggList.Insert ( _pAggr );
	break;
    }
    }
    return aggList;
}

CegoExpr* CegoFactor::getExpr() const
{
    return _pExpr;
}

const CegoFactor::FacType CegoFactor::getType() const
{
    return _type;
}

const Chain& CegoFactor::getVarName() const
{
    return _varName;
}

const CegoFieldValue& CegoFactor::getFieldValue() const
{
    return _fv;
}

CegoFactor* CegoFactor::clone(bool isAttrRef)
{
    switch ( _type ) 
    {
    case CegoFactor::ATTR:
    {
	if ( isAttrRef )
	{
	    return ( new CegoFactor ( _pAttrDesc, true ));
	}
	else
	    return ( new CegoFactor( _pAttrDesc->clone()));
    }
    case CegoFactor::CONSTVAL:
    {
	return ( new CegoFactor( _fv ) );
    }
    case CegoFactor::VAR:
    {
	return ( new CegoFactor( _varName ) );
    }
    case CegoFactor::EXPR:
    {
	return ( new CegoFactor( _pExpr->clone(isAttrRef) ) );
    }
    case CegoFactor::FUNCTION:
    {
	return ( new CegoFactor( _pFunction->clone(isAttrRef) ) );
    }
    case CegoFactor::AGGREGATION:
    {
	return ( new CegoFactor( _pAggr->clone(isAttrRef) ) );
    }
    case CegoFactor::QUERY:
    {
	return ( new CegoFactor( _pSelect->clone(isAttrRef) ) );
    }
    case CegoFactor::CASECOND:
    {
	return ( new CegoFactor( _pCaseCond->clone(isAttrRef) ) );
    }
    case CegoFactor::FETCH:
	throw Exception(EXLOC, "Clone of fetch not implemented");
    }
    return 0;
}

Chain CegoFactor::getId() const
{
    Chain s;
    switch (_type)
    {
    case CegoFactor::CONSTVAL:
    {
	s = _fv.valAsChain(true);
	break;
    }
    case CegoFactor::ATTR:
    {
	s = _pAttrDesc->getId();
	break;
    }
    case CegoFactor::VAR:
    {
	CegoFieldValue fv;
	if ( _pBlock )
	{
	    fv = _pBlock->getValue(_varName);
	}
	s = fv.toChain();
	break;
    }
    case CegoFactor::EXPR:
    {
	s = Chain("(") + _pExpr->getId() + Chain(")");
	break;
    }
    case CegoFactor::FETCH:
    {
	s = _pFetch->toChain();
	break;
    }
    case CegoFactor::AGGREGATION:
    {
	s = _pAggr->getId();
	break;
    }
    case CegoFactor::FUNCTION:
    {
	s = _pFunction->getId();
	break;
    }
    case CegoFactor::QUERY:
    {
	s = Chain("(") + _pSelect->getQueryId() + Chain(")");
	break;
    }
    case CegoFactor::CASECOND:
    {
	s = _pCaseCond->getId();
	break;
    }
    }
    return s;
}

Chain CegoFactor::toChain(const Chain& indent) const
{
    Chain s;
    switch (_type)
    {
    case CegoFactor::CONSTVAL:
	s = _fv.toChain();
	break;
    case CegoFactor::ATTR:
	s = _pAttrDesc->toChain();
	break;
    case CegoFactor::VAR:
	s = Chain(":") + _varName;
	break;
    case CegoFactor::EXPR:
	s = Chain("(") + _pExpr->toChain(indent) + Chain(")");
	break;
    case CegoFactor::FETCH:
	s = _pFetch->toChain();
	break;
    case CegoFactor::AGGREGATION:
	s = _pAggr->toChain();
	break;
    case CegoFactor::FUNCTION:
	s = _pFunction->toChain();
	break;
    case CegoFactor::QUERY:
	s = Chain("( ") + _pSelect->toChain(indent + Chain(" ")) + Chain(" )");
	break;
    case CegoFactor::CASECOND:
	s = _pCaseCond->toChain();
	break;
    }
    return s;
}

Chain CegoFactor::dbFormat(CegoDatabaseFormater *pForm) const
{
    return pForm->formatFactor(_type, _fv, _pAttrDesc, _varName, _pExpr, _pFetch, _pAggr, _pFunction, _pSelect, _pCaseCond);
}

Element* CegoFactor::toElement() const
{
    Element* pFactorElement = new Element(XML_FACTOR_ELEMENT);

    switch (_type)
    {
    case CegoFactor::CONSTVAL:
    {
	pFactorElement->setAttribute( XML_FACTOR_ATTR, XML_CONST_VALUE );
	pFactorElement->setAttribute( XML_TYPE_ATTR, CegoTypeConverter::getTypeString(_fv.getType()));
	pFactorElement->setAttribute( XML_VALUE_ATTR, _fv.valAsChain() );
	break;
    }
    case CegoFactor::ATTR:
    {
	pFactorElement->setAttribute( XML_FACTOR_ATTR, XML_ATTR_VALUE );
	if ( _pAttrDesc->getTableName().length() > 0 )
	    pFactorElement->setAttribute( XML_TABLENAME_ATTR, _pAttrDesc->getTableName() );
	pFactorElement->setAttribute( XML_ATTRNAME_ATTR, _pAttrDesc->getAttrName() );
	break;
    }
    case CegoFactor::VAR:
    {
	pFactorElement->setAttribute( XML_FACTOR_ATTR, XML_VAR_VALUE );
	pFactorElement->setAttribute( XML_VALUE_ATTR, _varName );
	break;
    }
    case CegoFactor::EXPR:
    {
	pFactorElement->setAttribute( XML_FACTOR_ATTR, XML_EXPR_VALUE );
	pFactorElement->addContent( _pExpr->toElement() );
	break;
    }
    case CegoFactor::FETCH:
    {
	throw Exception(EXLOC, "Cursor fetch not supported in distributed query");
    }
    case CegoFactor::CASECOND:
    {
	throw Exception(EXLOC, "case condition not supported in distributed query");
    }
    case CegoFactor::AGGREGATION:
    {
	throw Exception(EXLOC, "Aggregation not supported in distributed query");
    }
    case CegoFactor::FUNCTION:
    {
	pFactorElement->setAttribute( XML_FACTOR_ATTR, XML_FUNCTION_VALUE );
	pFactorElement->addContent( _pFunction->toElement() );
	break;
    }
    case CegoFactor::QUERY:
    {
	pFactorElement->setAttribute( XML_FACTOR_ATTR, XML_SELECT_VALUE );
	pFactorElement->addContent( _pSelect->toElement() );
	break;
    }
    }
    return pFactorElement;
}

void CegoFactor::fromElement(Element *pFactorElement, CegoDistManager* pGTM)
{
    Chain factorTypeString = pFactorElement->getAttributeValue( XML_FACTOR_ATTR );

    if ( factorTypeString == Chain(XML_CONST_VALUE) )
    {
	_type = CegoFactor::CONSTVAL;
	_fv = CegoFieldValue( CegoTypeConverter::getTypeId(pFactorElement->getAttributeValue( XML_TYPE_ATTR )),
			      pFactorElement->getAttributeValue( XML_VALUE_ATTR ));
	
    }
    else if ( factorTypeString == Chain(XML_ATTR_VALUE) )
    {
	_type = CegoFactor::ATTR;
	Chain tableName = pFactorElement->getAttributeValue( XML_TABLENAME_ATTR );
	if ( tableName.length() > 0 )
	    _pAttrDesc = new CegoAttrDesc( pFactorElement->getAttributeValue( XML_TABLENAME_ATTR ),
				      pFactorElement->getAttributeValue( XML_ATTRNAME_ATTR ));
	else
	    _pAttrDesc = new CegoAttrDesc( pFactorElement->getAttributeValue( XML_ATTRNAME_ATTR ));
    }
    else if ( factorTypeString == Chain(XML_VAR_VALUE) )
    {
	_type = CegoFactor::VAR;
	_varName =  pFactorElement->getAttributeValue( XML_VALUE_ATTR );
    }
    else if ( factorTypeString == Chain(XML_EXPR_VALUE) )
    {
	_type = CegoFactor::EXPR;
	ListT<Element*> el = pFactorElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr = new CegoExpr(*pEE, pGTM);
	}
    }
    else if ( factorTypeString == Chain(XML_FETCH_VALUE) )
    {
	throw Exception(EXLOC, "Cursor fetch not supported in distributed query");
    }
    else if ( factorTypeString == Chain(XML_AGGREGATION_VALUE) )
    {
	throw Exception(EXLOC, "Aggregation not supported in distributed query");
    }
    else if ( factorTypeString == Chain(XML_FUNCTION_VALUE) )
    {
	_type = CegoFactor::FUNCTION;
	ListT<Element*> fl = pFactorElement->getChildren(XML_FUNCTION_ELEMENT);
	Element **pFE = fl.First();
	if ( pFE )
	{
	    _pFunction = new CegoFunction(*pFE, pGTM);
	}
    }
    else if ( factorTypeString == Chain(XML_SELECT_VALUE) )
    {
	_type = CegoFactor::QUERY;
	ListT<Element*> sl = pFactorElement->getChildren(XML_SELECT_ELEMENT);
	Element **pSE = sl.First();
	if ( pSE )
	{
	    _pSelect = new CegoSelect(*pSE, pGTM);
	}
    }
}

void CegoFactor::encode(char *buf)
{    
    char* pE = (char*)buf;

    switch (_type )
    {
    case CegoFactor::CONSTVAL:
    {
	memcpy( pE, &_type, sizeof(CegoFactor::FacType));
	pE = pE + sizeof(CegoFactor::FacType);
	_fv.encode(pE);
	pE = pE + _fv.getEncodingLength();
	break;
    }
    case CegoFactor::ATTR:
    {
	memcpy( pE, &_type, sizeof(CegoFactor::FacType));
	pE = pE + sizeof(CegoFactor::FacType);
	_pAttrDesc->encode(pE);
	pE = pE + _pAttrDesc->getEncodingLength();
	break;
    }
    case CegoFactor::VAR:
    {
	// cast to const 
	CegoFactor::FacType type = CegoFactor::CONSTVAL;
	memcpy( pE, &type, sizeof(CegoFactor::FacType));
	pE = pE + sizeof(CegoFactor::FacType);

	CegoFieldValue fv;
	if ( _pBlock )
	{
	    fv = _pBlock->getValue(_varName);
	}

	fv.encode(pE);
	pE = pE + fv.getEncodingLength();
	break;
    }
    case CegoFactor::EXPR:
    {
	memcpy( pE, &_type, sizeof(CegoFactor::FacType));
	pE = pE + sizeof(CegoFactor::FacType);
	_pExpr->encode(pE);
	pE = pE + _pExpr->getEncodingLength();
	break;
    }
    case CegoFactor::FETCH:
    {
	throw Exception(EXLOC, "No encoding supported for fetch");
	break;
    }
    case CegoFactor::CASECOND:
    {
	memcpy( pE, &_type, sizeof(CegoFactor::FacType));
	pE = pE + sizeof(CegoFactor::FacType);
	_pCaseCond->encode(pE);
	pE = pE + _pCaseCond->getEncodingLength();
	break;
    }
    case CegoFactor::AGGREGATION:
    {
	memcpy( pE, &_type, sizeof(CegoFactor::FacType));
	pE = pE + sizeof(CegoFactor::FacType);
	_pAggr->encode(pE);
	pE = pE + _pAggr->getEncodingLength();
	break;
    }
    case CegoFactor::FUNCTION:
    {
	memcpy( pE, &_type, sizeof(CegoFactor::FacType));
	pE = pE + sizeof(CegoFactor::FacType);
	_pFunction->encode(pE);
	pE = pE + _pFunction->getEncodingLength();
	break;
    }
    case CegoFactor::QUERY:
    {
	memcpy( pE, &_type, sizeof(CegoFactor::FacType));
	pE = pE + sizeof(CegoFactor::FacType);
	_pSelect->encode(pE);
	pE = pE + _pSelect->getEncodingLength();
	break;
    }
    }
}

void CegoFactor::decode(char *buf, CegoDistManager* pGTM, int tabSetId)
{
    char* pE = (char*)buf;

    memcpy( &_type, pE, sizeof(CegoFactor::FacType));
    pE = pE + sizeof(CegoFactor::FacType);
    
    switch (_type )
    {
    case CegoFactor::CONSTVAL:
    {
	_fv.decode(pE);
	pE = pE + _fv.getEncodingLength();
	break;
    }
    case CegoFactor::ATTR:
    {
	_pAttrDesc = new CegoAttrDesc();
	_pAttrDesc->decode(pE);
	pE = pE + _pAttrDesc->getEncodingLength();
	break;
    }
    case CegoFactor::VAR:
    {
	throw Exception(EXLOC, "No decoding supported for var");
	break;
    }
    case CegoFactor::EXPR:
    {
	_pExpr = new CegoExpr(pE, pGTM, tabSetId);
	pE = pE + _pExpr->getEncodingLength();
	break;
    }
    case CegoFactor::FETCH:
    {
	throw Exception(EXLOC, "No decoding supported for fetch");
	break;
    }
    case CegoFactor::CASECOND:
    {
	_pCaseCond = new CegoCaseCond(pE, pGTM, tabSetId);
	pE = pE + _pCaseCond->getEncodingLength();
	break;
    }
    case CegoFactor::AGGREGATION:
    {
	_pAggr = new CegoAggregation(pE, pGTM, tabSetId);
	pE = pE + _pAggr->getEncodingLength();
	break;
    }
    case CegoFactor::FUNCTION:
    {
	_pFunction = new CegoFunction(pE, pGTM, tabSetId);
	pE = pE + _pFunction->getEncodingLength();
	break;
    }
    case CegoFactor::QUERY:
    {
	_pSelect = new CegoSelect(pE, pGTM, tabSetId);
	pE = pE + _pSelect->getEncodingLength();
	break;
    }
    }
}

int CegoFactor::getEncodingLength() const
{   
    int len = 0;

    len += sizeof(CegoFactor::FacType);

    switch (_type )
    {
    case CegoFactor::CONSTVAL:
    {
	len += _fv.getEncodingLength();
	break;
    }
    case CegoFactor::ATTR:
    {
	len += _pAttrDesc->getEncodingLength();
	break;
    }
    case CegoFactor::VAR:
    {
	CegoFieldValue fv;
	if ( _pBlock )
	{
	    fv = _pBlock->getValue(_varName);
	}
	len += fv.getEncodingLength();
	break;
    }
    case CegoFactor::EXPR:
    {
	len += _pExpr->getEncodingLength();
	break;
    }
    case CegoFactor::FETCH:
    {
	// no encoding supported for procedures
	break;
    }
    case CegoFactor::CASECOND:
    {
	len += _pCaseCond->getEncodingLength();
	break;
    }
    case CegoFactor::AGGREGATION:
    {
	len += _pAggr->getEncodingLength();
	break;
    }
    case CegoFactor::FUNCTION:
    {
	len += _pFunction->getEncodingLength();
	break;
    }
    case CegoFactor::QUERY:
    {
	len +=_pSelect->getEncodingLength();
	break;
    }
    }
    
    return len;    
}

void CegoFactor::getPlanList(ListT<Element*>& planList)
{
    if ( _pSelect )
	planList.Insert(_pSelect->getPlan());
}

ostream& operator << (ostream& s, const CegoFactor& pf)
{
    s << pf.toChain(Chain(""));
    return s;
}
