/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2025 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TGUI_CANVAS_SDL_GPU_HPP
#define TGUI_CANVAS_SDL_GPU_HPP

#include <TGUI/Backend/Renderer/SDL_GPU/BackendTextureSDLGPU.hpp>

#if !TGUI_BUILD_AS_CXX_MODULE
    #include <TGUI/Backend/Renderer/BackendRenderTarget.hpp>
    #include <TGUI/Widgets/CanvasBase.hpp>
#endif

#include <TGUI/extlibs/IncludeSDL.hpp>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TGUI_MODULE_EXPORT namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief CanvasSDLGPU provides a way to directly render SDL contents on a widget
    ///
    /// When gui.draw() is called, all widgets are drawn at once. If you wish to have custom SDL rendering inbetween
    /// TGUI widgets (e.g. draw to the background of a child window) then you need to use a CanvasSDLGPU widget.
    ///
    /// The canvas widget is essentially just a wrapper around a texture with the SDL_GPU_TEXTUREUSAGE_COLOR_TARGET flag.
    /// You draw your SDL contents on top of the canvas instead of on the window. The canvas is then added to the gui between
    /// the widgets where you want the rendering to appear.
    ///
    /// Example:
    /// @code
    /// SDL_GPUColorTargetInfo colorTargetInfo = {};
    /// colorTargetInfo.texture = canvas->getTexture();
    /// colorTargetInfo.clear_color = clearColor;
    /// colorTargetInfo.load_op = SDL_GPU_LOADOP_CLEAR;
    /// colorTargetInfo.store_op = SDL_GPU_STOREOP_STORE;
    /// SDL_GPURenderPass* renderPass = SDL_BeginGPURenderPass(cmdBuffer, &colorTargetInfo, 1, NULL);
    /// // Draw to the texture of the canvas
    /// SDL_EndGPURenderPass(renderPass);
    /// @endcode
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API CanvasSDLGPU : public CanvasBase
    {
    public:

        using Ptr = std::shared_ptr<CanvasSDLGPU>; //!< Shared widget pointer
        using ConstPtr = std::shared_ptr<const CanvasSDLGPU>; //!< Shared constant widget pointer

        static constexpr const char StaticWidgetType[] = "CanvasSDLGPU"; //!< Type name of the widget

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// @brief Constructor
        /// @param typeName     Type of the widget
        /// @param initRenderer Should the renderer be initialized? Should be true unless a derived class initializes it.
        /// @see create
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasSDLGPU(const char* typeName = StaticWidgetType, bool initRenderer = true);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Copy constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasSDLGPU(const CanvasSDLGPU& copy);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Default move constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasSDLGPU(CanvasSDLGPU&& copy) = default;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Overload of copy assignment operator
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasSDLGPU& operator= (const CanvasSDLGPU& right);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Default move assignment operator
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        CanvasSDLGPU& operator= (CanvasSDLGPU&& right) = default;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Creates a new canvas widget
        ///
        /// @param size  Size of the canvas
        ///
        /// @return The new canvas
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static CanvasSDLGPU::Ptr create(const Layout2d& size = {"100%", "100%"});

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Makes a copy of another canvas
        ///
        /// @param canvas  The other canvas
        ///
        /// @return The new canvas
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static CanvasSDLGPU::Ptr copy(const CanvasSDLGPU::ConstPtr& canvas);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the size of the widget
        ///
        /// @param size  The new size of the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setSize(const Layout2d& size) override;
        using Widget::setSize;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draw the widget to a render target
        ///
        /// @param target Render target to draw to
        /// @param states Current render states
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void draw(BackendRenderTarget& target, RenderStates states) const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Access the internal texture on which you can render
        ///
        /// @return Pointer to the internal texture
        ///
        /// The texture has a size that is equal to the size of the canvas widget.
        /// The pixel format of the texture is SDL_GPU_TEXTUREFORMAT_R8G8B8A8_UNORM.
        /// The texture was created with SDL_GPU_TEXTUREUSAGE_COLOR_TARGET and SDL_GPU_TEXTUREUSAGE_SAMPLER flags.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD SDL_GPUTexture* getTexture()
        {
            return m_backendTexture->getInternalTexture();
        }

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Makes a copy of the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Widget::Ptr clone() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        Vector2u m_textureSize;
        std::shared_ptr<BackendTextureSDLGPU> m_backendTexture;
    };
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_CANVAS_SDL_GPU_HPP
