







open Ctypes
open Foreign

(* Under Linux, disable GNAT's handling of SIGSEGV, which is incompatible with
   what the OCaml runtime is already doing. *)
let () =
  if Sys.unix then
    ignore
      (Dl.dlopen
        ~filename:"liblangkit_sigsegv_handler.so"
        ~flags:[Dl.RTLD_NOW]
        : Dl.library)

let so_ext = if Sys.unix then "so" else "dll"
let c_lib_name = Format.sprintf "liblktlang.%s" so_ext
let c_lib = Dl.dlopen ~filename:c_lib_name ~flags:[Dl.RTLD_NOW]

let add_gc_link ~from ~to_ =
  let r = ref (Some (Obj.repr to_)) in
  let finaliser _ = r := None in
  Gc.finalise finaliser from

exception FileReadError of string

exception BadTypeError of string

exception OutOfBoundsError of string

exception InvalidInput of string

exception InvalidSymbolError of string

exception InvalidUnitNameError of string

exception NativeException of string

exception PreconditionFailure of string

exception PropertyError of string

exception TemplateArgsError of string

exception TemplateFormatError of string

exception TemplateInstantiationError of string

exception StaleReferenceError of string

exception SyntaxError of string

exception UnknownCharset of string

exception MalformedTreeError of string





let char_ptr_of_string str =
  coerce string (ptr char) str

let string_of_char_ptr str =
  coerce (ptr char) string str

module Exception = struct

  type t = {
    kind : int;
    information : string;
  }

  let c_struct : t structure typ = structure "exception"
  let kind = field c_struct "kind" int
  let information = field c_struct "information" (ptr char)
  let () = seal c_struct

  let wrap c_value_ptr =
    if is_null c_value_ptr then
      None
    else
      let c_value = !@ c_value_ptr in
      let c_information = getf c_value information in
      (* eng/codepeer/infer#220: Avoid a crash when the "information" field is
         null. This is never supposed to happen, but has been observed to
         happen randomly. For now, consider that there is no exception when
         this is the case. *)
      let information =
        if is_null c_information then
          ""
        else
          string_of_char_ptr c_information
      in
      Some {
        kind = getf c_value kind;
        information
      }

  let unwrap value =
    match value with
    | None ->
        from_voidp c_struct null
    | Some value ->
        let c_value = make c_struct in
        let c_information = char_ptr_of_string value.information in
        setf c_value kind value.kind;
        add_gc_link ~from:c_value ~to_:c_information;
        setf c_value information c_information;
        allocate c_struct c_value

  let c_type = view (ptr c_struct) ~read:wrap ~write:unwrap

end

let get_last_exception = foreign ~from:c_lib
  "lkt_get_last_exception"
  (void @-> returning Exception.c_type)

(* When declaring an imported function with foreign, use raisable instead of
 returning, to check get_last_exception before returning *)
let raisable typ =
  let read value =
    match get_last_exception () with
    | None -> value
    | Some exc ->
        (match exc.kind with
         | 0 ->
             raise (FileReadError exc.information)
         | 1 ->
             raise (BadTypeError exc.information)
         | 2 ->
             raise (OutOfBoundsError exc.information)
         | 3 ->
             raise (InvalidInput exc.information)
         | 4 ->
             raise (InvalidSymbolError exc.information)
         | 5 ->
             raise (InvalidUnitNameError exc.information)
         | 6 ->
             raise (NativeException exc.information)
         | 7 ->
             raise (PreconditionFailure exc.information)
         | 8 ->
             raise (PropertyError exc.information)
         | 9 ->
             raise (TemplateArgsError exc.information)
         | 10 ->
             raise (TemplateFormatError exc.information)
         | 11 ->
             raise (TemplateInstantiationError exc.information)
         | 12 ->
             raise (StaleReferenceError exc.information)
         | 13 ->
             raise (SyntaxError exc.information)
         | 14 ->
             raise (UnknownCharset exc.information)
         | 15 ->
             raise (MalformedTreeError exc.information)
         | _ -> assert false)
  in
  let write value = value in
  let new_typ = view typ ~read ~write in
  returning new_typ

(* Convert a char ptr encoding an utf8 string to an ocaml String (cannot use
   char_ptr_of_string here as bytes could contain null characters in the
   middle) *)
let string_of_bytes bytes length =
  String.init (Unsigned.Size_t.to_int (!@ length)) (fun i -> !@ (!@ bytes +@ i))

(* Convert an OCaml String encoded in utf8, to a char* and its length. The
   returned char* is not null terminated *)
let bytes_of_string str =
  let length = String.length str in
  let bytes = allocate_n char ~count:length in
  String.iteri (fun i c ->
    bytes +@ i <-@ c
  ) str ;
  (Unsigned.Size_t.of_int length), bytes

let free = foreign ~from:c_lib "lkt_free"
 (ptr char @-> returning void)

module Text = struct
  type t = string

  let c_struct : t structure typ = structure "text"

  let chars = field c_struct "chars" (ptr uint32_t)

  let length = field c_struct "length" size_t

  let is_allocated = field c_struct "is_allocated" bool

  let () = seal c_struct

  let destroy_text = foreign ~from:c_lib "lkt_destroy_text"
    (ptr c_struct @-> raisable void)

  let text_to_utf8 = foreign ~from:c_lib "lkt_text_to_utf8"
    (ptr c_struct @-> ptr (ptr char) @-> ptr size_t @-> raisable void)

  let text_from_utf8 = foreign ~from:c_lib "lkt_text_from_utf8"
    (ptr char @-> size_t @-> ptr c_struct @-> raisable void)

  let wrap (c_value : t structure) : t =
     let bytes = allocate (ptr char) (from_voidp char null) in
     let length = allocate (size_t) (Unsigned.Size_t.of_int 0) in
     text_to_utf8 (addr c_value) bytes length ;
     let r = string_of_bytes bytes length in
     free (!@ bytes);
     destroy_text (addr c_value);
     r

  let unwrap (value : t) : t structure =
     let length, bytes = bytes_of_string value in
     let c_value = make c_struct in
     text_from_utf8 bytes length (addr c_value);
     c_value

  let c_type = c_struct
end

module Character = struct
  (* Characters are encoded as strings because ocaml char are not unicode
   characters *)
  type t = string

  let char_to_utf8 = foreign ~from:c_lib "lkt_char_to_utf8"
    (uint32_t @-> ptr (ptr char) @-> ptr size_t @-> raisable void)

  let char_from_utf8 = foreign ~from:c_lib "lkt_char_from_utf8"
    (ptr char @-> size_t @-> ptr uint32_t @-> raisable void)

  let wrap (c_value : Unsigned.UInt32.t) : t =
     let bytes = allocate (ptr char) (from_voidp char null) in
     let length = allocate (size_t) (Unsigned.Size_t.of_int 0) in
     char_to_utf8 c_value bytes length ;
     let r = string_of_bytes bytes length in
     free (!@ bytes);
     r

  let chr i =
    wrap (Unsigned.UInt32.of_int i)

  let unwrap (value : string) : Unsigned.UInt32.t =
     let length, bytes = bytes_of_string value in
     let c_value = allocate uint32_t (Unsigned.UInt32.of_int 0) in
     char_from_utf8 bytes length c_value;
     !@ c_value

  let code i =
    Unsigned.UInt32.to_int (unwrap i)

  let c_type = view uint32_t ~read:wrap ~write:unwrap
end

module StringType = struct
  type t = string

  let c_struct : t structure typ = structure "string"
  let length_field = field c_struct "length" int
  let _ = field c_struct "ref_count" int
  (* Langkit strings are encoded in UTF-32 (native endianity). *)
  let content_field = field c_struct "content" uint32_t
  let () = seal c_struct

  let c_type = ptr c_struct

  let string_to_utf8 = foreign ~from:c_lib "lkt_string_to_utf8"
    (c_type @-> ptr (ptr char) @-> ptr size_t @-> raisable void)

  let string_from_utf8 =
    foreign ~from:c_lib "lkt_string_from_utf8"
      (ptr char @-> size_t @-> ptr c_type @-> raisable void)

  let string_dec_ref = foreign ~from:c_lib "lkt_string_dec_ref"
    (c_type @-> raisable void)

  let wrap ?(dec_ref=true) c_value_ptr =
    let bytes = allocate (ptr char) (from_voidp char null) in
    let length = allocate (size_t) (Unsigned.Size_t.of_int 0) in
    string_to_utf8 c_value_ptr bytes length ;
    let r = string_of_bytes bytes length in
    free (!@ bytes);
    if dec_ref then string_dec_ref c_value_ptr;
    r

  let unwrap value =
    let length, bytes = bytes_of_string value in
    let c_value = allocate c_type (from_voidp c_struct null) in
    string_from_utf8 bytes length c_value;
    !@ c_value
end

module BigInteger = struct

  type t = Z.t

  let c_type = ptr void

  let create = foreign ~from:c_lib "lkt_create_big_integer"
    (ptr Text.c_type @-> raisable c_type)

  let text = foreign ~from:c_lib "lkt_big_integer_text"
    (c_type @-> ptr Text.c_type @-> raisable void)

  let decref = foreign ~from:c_lib "lkt_big_integer_decref"
    (c_type @-> raisable void)

  let wrap ?(dec_ref=true) (c_value : unit ptr) : t =
    let c_text_ptr = allocate_n Text.c_type ~count:1 in
    text c_value c_text_ptr;
    if dec_ref then decref c_value;
    Z.of_string (Text.wrap (!@ c_text_ptr))

  let unwrap (value : t) : unit ptr =
    let c_text = Text.unwrap (Z.to_string value) in
    let r = create (addr c_text) in
    Text.destroy_text (addr c_text);
    r
end

module AnalysisUnitKind = struct
  type t =
  | UnitSpecification
  | UnitBody

  let name () = "AnalysisUnitKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> UnitSpecification
    | 1 -> UnitBody
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | UnitSpecification -> 0
    | UnitBody -> 1

   let c_type = view int ~read:wrap ~write:unwrap
end

module CompletionItemKind = struct
  type t =
  | TextKind
  | MethodKind
  | FunctionKind
  | ConstructorKind
  | FieldKind
  | VariableKind
  | ClassKind
  | InterfaceKind
  | ModuleKind
  | PropertyKind
  | UnitKind
  | ValueKind
  | EnumKind
  | KeywordKind
  | SnippetKind
  | ColorKind
  | FileKind
  | ReferenceKind
  | FolderKind
  | EnumMemberKind
  | ConstantKind
  | StructKind
  | EventKind
  | OperatorKind
  | TypeParameterKind

  let name () = "CompletionItemKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> TextKind
    | 1 -> MethodKind
    | 2 -> FunctionKind
    | 3 -> ConstructorKind
    | 4 -> FieldKind
    | 5 -> VariableKind
    | 6 -> ClassKind
    | 7 -> InterfaceKind
    | 8 -> ModuleKind
    | 9 -> PropertyKind
    | 10 -> UnitKind
    | 11 -> ValueKind
    | 12 -> EnumKind
    | 13 -> KeywordKind
    | 14 -> SnippetKind
    | 15 -> ColorKind
    | 16 -> FileKind
    | 17 -> ReferenceKind
    | 18 -> FolderKind
    | 19 -> EnumMemberKind
    | 20 -> ConstantKind
    | 21 -> StructKind
    | 22 -> EventKind
    | 23 -> OperatorKind
    | 24 -> TypeParameterKind
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | TextKind -> 0
    | MethodKind -> 1
    | FunctionKind -> 2
    | ConstructorKind -> 3
    | FieldKind -> 4
    | VariableKind -> 5
    | ClassKind -> 6
    | InterfaceKind -> 7
    | ModuleKind -> 8
    | PropertyKind -> 9
    | UnitKind -> 10
    | ValueKind -> 11
    | EnumKind -> 12
    | KeywordKind -> 13
    | SnippetKind -> 14
    | ColorKind -> 15
    | FileKind -> 16
    | ReferenceKind -> 17
    | FolderKind -> 18
    | EnumMemberKind -> 19
    | ConstantKind -> 20
    | StructKind -> 21
    | EventKind -> 22
    | OperatorKind -> 23
    | TypeParameterKind -> 24

   let c_type = view int ~read:wrap ~write:unwrap
end

module DesignatedEnvKind = struct
  type t =
  | None
  | CurrentEnv
  | NamedEnv
  | DirectEnv

  let name () = "DesignatedEnvKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> None
    | 1 -> CurrentEnv
    | 2 -> NamedEnv
    | 3 -> DirectEnv
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | None -> 0
    | CurrentEnv -> 1
    | NamedEnv -> 2
    | DirectEnv -> 3

   let c_type = view int ~read:wrap ~write:unwrap
end

module GrammarRule = struct
  type t =
  | MainRuleRule
  | IdRule
  | RefIdRule
  | TypeRefIdRule
  | DefIdRule
  | DocRule
  | ImportStmtRule
  | ImportsRule
  | LexerDeclRule
  | GrammarDeclRule
  | GrammarRuleRule
  | LexerRuleRule
  | LexerFamilyDeclRule
  | LexerCaseRuleRule
  | LexerCaseAltRule
  | LexerCaseSendRule
  | GrammarPrimaryRule
  | GrammarExprRule
  | GrammarPickRule
  | GrammarImplicitPickRule
  | GrammarOptRule
  | GrammarOptErrorRule
  | GrammarCutRule
  | GrammarStopcutRule
  | GrammarOrExprRule
  | GrammarDiscardExprRule
  | TokenLiteralRule
  | TokenNoCaseLiteralRule
  | TokenPatternRule
  | TokenPatternLiteralRule
  | ParseNodeExprRule
  | GrammarRuleRefRule
  | GrammarListExprRule
  | GrammarListSepRule
  | GrammarSkipRule
  | GrammarNullRule
  | GrammarTokenRule
  | TypeDeclRule
  | GenericDeclRule
  | GenericParamTypeRule
  | EnumLitDeclRule
  | FunDeclRule
  | LambdaParamDeclRule
  | FunParamDeclRule
  | FunParamListRule
  | LambdaParamListRule
  | FieldDeclRule
  | BareDeclRule
  | DeclRule
  | TypeMemberRefRule
  | TypeExprRule
  | TypeRefRule
  | TypeListRule
  | DeclsRule
  | DeclBlockRule
  | ValDeclRule
  | DynvarDeclRule
  | VarBindRule
  | EnvSpecActionRule
  | EnvSpecDeclRule
  | BlockRule
  | PatternRule
  | FilPatternRule
  | ValuePatternRule
  | RegexPatternRule
  | BoolPatternRule
  | EllipsisPatternRule
  | IntegerPatternRule
  | ListPatternRule
  | TuplePatternRule
  | PatternArgRule
  | SelectorCallRule
  | ExprRule
  | RelRule
  | EqRule
  | Arith1Rule
  | Arith2Rule
  | Arith3Rule
  | IsaOrPrimaryRule
  | LogicPropagateCallRule
  | PrimaryRule
  | MatchExprRule
  | NumLitRule
  | BigNumLitRule
  | StringLitRule
  | BlockStringLitRule
  | CharLitRule
  | IfExprRule
  | RaiseExprRule
  | TryExprRule
  | ArrayLiteralRule
  | CallableRefRule
  | NullCondQualRule
  | BasicExprRule
  | TermRule
  | BasicNameRule
  | LambdaExprRule
  | NullLitRule
  | ArgumentRule
  | ArgsRule
  | DeclAnnotationArgsRule
  | DeclAnnotationRule

  let name () = "GrammarRule"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> MainRuleRule
    | 1 -> IdRule
    | 2 -> RefIdRule
    | 3 -> TypeRefIdRule
    | 4 -> DefIdRule
    | 5 -> DocRule
    | 6 -> ImportStmtRule
    | 7 -> ImportsRule
    | 8 -> LexerDeclRule
    | 9 -> GrammarDeclRule
    | 10 -> GrammarRuleRule
    | 11 -> LexerRuleRule
    | 12 -> LexerFamilyDeclRule
    | 13 -> LexerCaseRuleRule
    | 14 -> LexerCaseAltRule
    | 15 -> LexerCaseSendRule
    | 16 -> GrammarPrimaryRule
    | 17 -> GrammarExprRule
    | 18 -> GrammarPickRule
    | 19 -> GrammarImplicitPickRule
    | 20 -> GrammarOptRule
    | 21 -> GrammarOptErrorRule
    | 22 -> GrammarCutRule
    | 23 -> GrammarStopcutRule
    | 24 -> GrammarOrExprRule
    | 25 -> GrammarDiscardExprRule
    | 26 -> TokenLiteralRule
    | 27 -> TokenNoCaseLiteralRule
    | 28 -> TokenPatternRule
    | 29 -> TokenPatternLiteralRule
    | 30 -> ParseNodeExprRule
    | 31 -> GrammarRuleRefRule
    | 32 -> GrammarListExprRule
    | 33 -> GrammarListSepRule
    | 34 -> GrammarSkipRule
    | 35 -> GrammarNullRule
    | 36 -> GrammarTokenRule
    | 37 -> TypeDeclRule
    | 38 -> GenericDeclRule
    | 39 -> GenericParamTypeRule
    | 40 -> EnumLitDeclRule
    | 41 -> FunDeclRule
    | 42 -> LambdaParamDeclRule
    | 43 -> FunParamDeclRule
    | 44 -> FunParamListRule
    | 45 -> LambdaParamListRule
    | 46 -> FieldDeclRule
    | 47 -> BareDeclRule
    | 48 -> DeclRule
    | 49 -> TypeMemberRefRule
    | 50 -> TypeExprRule
    | 51 -> TypeRefRule
    | 52 -> TypeListRule
    | 53 -> DeclsRule
    | 54 -> DeclBlockRule
    | 55 -> ValDeclRule
    | 56 -> DynvarDeclRule
    | 57 -> VarBindRule
    | 58 -> EnvSpecActionRule
    | 59 -> EnvSpecDeclRule
    | 60 -> BlockRule
    | 61 -> PatternRule
    | 62 -> FilPatternRule
    | 63 -> ValuePatternRule
    | 64 -> RegexPatternRule
    | 65 -> BoolPatternRule
    | 66 -> EllipsisPatternRule
    | 67 -> IntegerPatternRule
    | 68 -> ListPatternRule
    | 69 -> TuplePatternRule
    | 70 -> PatternArgRule
    | 71 -> SelectorCallRule
    | 72 -> ExprRule
    | 73 -> RelRule
    | 74 -> EqRule
    | 75 -> Arith1Rule
    | 76 -> Arith2Rule
    | 77 -> Arith3Rule
    | 78 -> IsaOrPrimaryRule
    | 79 -> LogicPropagateCallRule
    | 80 -> PrimaryRule
    | 81 -> MatchExprRule
    | 82 -> NumLitRule
    | 83 -> BigNumLitRule
    | 84 -> StringLitRule
    | 85 -> BlockStringLitRule
    | 86 -> CharLitRule
    | 87 -> IfExprRule
    | 88 -> RaiseExprRule
    | 89 -> TryExprRule
    | 90 -> ArrayLiteralRule
    | 91 -> CallableRefRule
    | 92 -> NullCondQualRule
    | 93 -> BasicExprRule
    | 94 -> TermRule
    | 95 -> BasicNameRule
    | 96 -> LambdaExprRule
    | 97 -> NullLitRule
    | 98 -> ArgumentRule
    | 99 -> ArgsRule
    | 100 -> DeclAnnotationArgsRule
    | 101 -> DeclAnnotationRule
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | MainRuleRule -> 0
    | IdRule -> 1
    | RefIdRule -> 2
    | TypeRefIdRule -> 3
    | DefIdRule -> 4
    | DocRule -> 5
    | ImportStmtRule -> 6
    | ImportsRule -> 7
    | LexerDeclRule -> 8
    | GrammarDeclRule -> 9
    | GrammarRuleRule -> 10
    | LexerRuleRule -> 11
    | LexerFamilyDeclRule -> 12
    | LexerCaseRuleRule -> 13
    | LexerCaseAltRule -> 14
    | LexerCaseSendRule -> 15
    | GrammarPrimaryRule -> 16
    | GrammarExprRule -> 17
    | GrammarPickRule -> 18
    | GrammarImplicitPickRule -> 19
    | GrammarOptRule -> 20
    | GrammarOptErrorRule -> 21
    | GrammarCutRule -> 22
    | GrammarStopcutRule -> 23
    | GrammarOrExprRule -> 24
    | GrammarDiscardExprRule -> 25
    | TokenLiteralRule -> 26
    | TokenNoCaseLiteralRule -> 27
    | TokenPatternRule -> 28
    | TokenPatternLiteralRule -> 29
    | ParseNodeExprRule -> 30
    | GrammarRuleRefRule -> 31
    | GrammarListExprRule -> 32
    | GrammarListSepRule -> 33
    | GrammarSkipRule -> 34
    | GrammarNullRule -> 35
    | GrammarTokenRule -> 36
    | TypeDeclRule -> 37
    | GenericDeclRule -> 38
    | GenericParamTypeRule -> 39
    | EnumLitDeclRule -> 40
    | FunDeclRule -> 41
    | LambdaParamDeclRule -> 42
    | FunParamDeclRule -> 43
    | FunParamListRule -> 44
    | LambdaParamListRule -> 45
    | FieldDeclRule -> 46
    | BareDeclRule -> 47
    | DeclRule -> 48
    | TypeMemberRefRule -> 49
    | TypeExprRule -> 50
    | TypeRefRule -> 51
    | TypeListRule -> 52
    | DeclsRule -> 53
    | DeclBlockRule -> 54
    | ValDeclRule -> 55
    | DynvarDeclRule -> 56
    | VarBindRule -> 57
    | EnvSpecActionRule -> 58
    | EnvSpecDeclRule -> 59
    | BlockRule -> 60
    | PatternRule -> 61
    | FilPatternRule -> 62
    | ValuePatternRule -> 63
    | RegexPatternRule -> 64
    | BoolPatternRule -> 65
    | EllipsisPatternRule -> 66
    | IntegerPatternRule -> 67
    | ListPatternRule -> 68
    | TuplePatternRule -> 69
    | PatternArgRule -> 70
    | SelectorCallRule -> 71
    | ExprRule -> 72
    | RelRule -> 73
    | EqRule -> 74
    | Arith1Rule -> 75
    | Arith2Rule -> 76
    | Arith3Rule -> 77
    | IsaOrPrimaryRule -> 78
    | LogicPropagateCallRule -> 79
    | PrimaryRule -> 80
    | MatchExprRule -> 81
    | NumLitRule -> 82
    | BigNumLitRule -> 83
    | StringLitRule -> 84
    | BlockStringLitRule -> 85
    | CharLitRule -> 86
    | IfExprRule -> 87
    | RaiseExprRule -> 88
    | TryExprRule -> 89
    | ArrayLiteralRule -> 90
    | CallableRefRule -> 91
    | NullCondQualRule -> 92
    | BasicExprRule -> 93
    | TermRule -> 94
    | BasicNameRule -> 95
    | LambdaExprRule -> 96
    | NullLitRule -> 97
    | ArgumentRule -> 98
    | ArgsRule -> 99
    | DeclAnnotationArgsRule -> 100
    | DeclAnnotationRule -> 101

   let c_type = view int ~read:wrap ~write:unwrap
end

module LookupKind = struct
  type t =
  | Recursive
  | Flat
  | Minimal

  let name () = "LookupKind"

  let wrap (c_value : int) : t =
    match c_value with
    | 0 -> Recursive
    | 1 -> Flat
    | 2 -> Minimal
    | _ -> assert false

  let unwrap (value : t) : int =
    match value with
    | Recursive -> 0
    | Flat -> 1
    | Minimal -> 2

   let c_type = view int ~read:wrap ~write:unwrap
end


let free = foreign ~from:c_lib
  "lkt_free"
  (ptr void @-> returning void)

(** Assuming char_ptr is a valid char*, convert it to a native Ocaml
  * string and free the C pointer.
  *)
let unwrap_str char_ptr =
  let str = Ctypes.coerce (ptr char) string char_ptr in
  free (Ctypes.coerce (ptr char) (ptr void) char_ptr);
  str


let default_grammar_rule = GrammarRule.MainRuleRule

module Sloc = struct
  type t = {
    line : int;
    column : int;
  }

  let c_struct : t structure typ = structure "sloc"
  let line = field c_struct "line" uint32_t
  let column = field c_struct "column" uint16_t
  let () = seal c_struct

  let wrap (c_value : t structure) : t = {
    line = Unsigned.UInt32.to_int (getf c_value line);
    column = Unsigned.UInt16.to_int (getf c_value column);
  }

  let unwrap (value : t) : t structure =
    let c_value = make c_struct in
    setf c_value line (Unsigned.UInt32.of_int (value.line));
    setf c_value column (Unsigned.UInt16.of_int (value.column));
    c_value

  let c_type = view c_struct ~read:wrap ~write:unwrap
end

module SlocRange = struct
  type t = {
    loc_start : Sloc.t;
    loc_end : Sloc.t;
  }

  let c_struct : t structure typ = structure "sloc_range"
  let loc_start = field c_struct "loc_start" Sloc.c_type
  let loc_end = field c_struct "loc_end" Sloc.c_type
  let () = seal c_struct

  let wrap (c_value : t structure) : t = {
    loc_start = getf c_value loc_start;
    loc_end = getf c_value loc_end;
  }

  let unwrap (value : t) : t structure =
    let c_value = make c_struct in
    setf c_value loc_start value.loc_start;
    setf c_value loc_end value.loc_end;
    c_value

  let c_type = view c_struct ~read:wrap ~write:unwrap

  let pp fmt sloc_range =
    Format.fprintf fmt "<SlocRange %d:%d-%d:%d>"
      sloc_range.loc_start.line
      sloc_range.loc_start.column
      sloc_range.loc_end.line
      sloc_range.loc_end.column
end

module Diagnostic = struct
  type t = {
    sloc_range : SlocRange.t;
    message : string
  }

  let c_struct : t structure typ = structure "diagnostic"
  let sloc_range = field c_struct "sloc_range" SlocRange.c_type
  let message = field c_struct "message" Text.c_struct
  let () = seal c_struct

  let wrap (c_value : t structure) : t = {
    sloc_range = getf c_value sloc_range;
    message = Text.wrap (getf c_value message);
  }

  let unwrap (value : t) : t structure =
    let c_value = make c_struct in
    (* sloc_range is not a pointer, thus we have a copy here which is safe. *)
    setf c_value sloc_range value.sloc_range;
    (* message is not a pointer, thus we have a copy here which is safe.
       HOWEVER, there is a link from value.message to another pointer which can
       be freed by the GC if we don't propagate this link. *)
    let c_value_message = Text.unwrap value.message in
    add_gc_link ~from:c_value ~to_:c_value_message;
    setf c_value message c_value_message;
    c_value

  let c_type = view c_struct ~read:wrap ~write:unwrap
end

module TokenData = struct
  type t = unit ptr
end

module Token = struct
  (* We don't have access to AnalysisContextStruct at this point. We don't need
     to do anything with the context value except pass it around, so map it as
     an opaque pointer instead. *)
  type dummy_context = unit ptr

  type t = {
    context : dummy_context;
    token_data : TokenData.t;
    token_index : int;
    trivia_index : int;
    kind : int;
    text : string;
    sloc_range : SlocRange.t;
  }

  let c_type : t structure typ = structure "token"
  let context = field c_type "context" (ptr void)
  let token_data = field c_type "token_data" (ptr void)
  let token_index = field c_type "token_index" int
  let trivia_index = field c_type "trivia_index" int
  let () = seal c_type

  let _token_get_kind = foreign ~from:c_lib
    "lkt_token_get_kind"
    (ptr c_type @-> raisable int)

  let _token_kind_name = foreign ~from:c_lib
    "lkt_token_kind_name"
    (int @-> raisable (ptr char))

  let _token_sloc_range = foreign ~from:c_lib
    "lkt_token_sloc_range"
    (ptr c_type @-> ptr SlocRange.c_type @-> raisable void)

  let token_kind_name kind =
    unwrap_str (_token_kind_name kind)

  let token_range_text = foreign ~from:c_lib
    "lkt_token_range_text"
    (ptr c_type @-> ptr c_type @-> ptr Text.c_type @-> raisable int)

  let wrap (c_value : t structure) : t option =
  let token_data = getf c_value token_data in
  if is_null token_data then
    None
  else
    Some {
      context = getf c_value context;
      token_data;
      token_index = getf c_value token_index;
      trivia_index = getf c_value trivia_index;
      kind = _token_get_kind (addr c_value);
      text =
        (let c_result_ptr = allocate_n Text.c_type ~count:1 in
         let _ = token_range_text (addr c_value) (addr c_value) c_result_ptr in
         Text.wrap (!@ c_result_ptr));
      sloc_range =
        (let c_result_ptr = allocate_n SlocRange.c_type ~count:1 in
         let _ = _token_sloc_range (addr c_value) c_result_ptr in
         !@ c_result_ptr);
    }

  let unwrap (value : t) : t structure =
    let c_value = make c_type in
    setf c_value context value.context;
    setf c_value token_data value.token_data;
    setf c_value token_index value.token_index;
    setf c_value trivia_index value.trivia_index;
    c_value

  let kind_name token = token_kind_name (_token_get_kind (addr (unwrap token)))

  let sloc_range token =
    let c_result_ptr = allocate_n SlocRange.c_type ~count:1 in
    let _ = _token_sloc_range (addr (unwrap token)) c_result_ptr in
    !@ c_result_ptr

  let token_next = foreign ~from:c_lib
    "lkt_token_next"
    (ptr c_type @-> ptr c_type @-> raisable void)

  let token_previous = foreign ~from:c_lib
    "lkt_token_previous"
    (ptr c_type @-> ptr c_type @-> raisable void)

  let is_equivalent = foreign ~from:c_lib
    "lkt_token_is_equivalent"
    (ptr c_type @-> ptr c_type @-> raisable bool)

  let pp fmt token =
    let pp_text fmt = function
      | "" -> Format.pp_print_string fmt ""
      | _ as text -> Format.fprintf fmt " %S" text
    in
    Format.fprintf fmt "<Token %s%a at %a>"
      (kind_name token)
      pp_text token.text
      SlocRange.pp token.sloc_range

  let text_range token_first token_last =
    let c_result_ptr = allocate_n Text.c_type ~count:1 in
    let res =
      token_range_text
        (addr (unwrap token_first))
        (addr (unwrap token_last))
        c_result_ptr
    in
    if res = 0 then
      raise (Invalid_argument
        (Format.asprintf "%a and %a come from different units"
          pp token_first
          pp token_last));
    Text.wrap (!@ c_result_ptr)

  let text token = text_range token token

  let next token =
    let c_next_token_ptr = allocate_n c_type ~count:1 in
    token_next (addr (unwrap token)) c_next_token_ptr ;
    wrap (!@ c_next_token_ptr)

  let previous token =
    let c_next_token_ptr = allocate_n c_type ~count:1 in
    token_previous (addr (unwrap token)) c_next_token_ptr ;
    wrap (!@ c_next_token_ptr)

  let is_trivia token =
    token.trivia_index != 0

  let index token =
    match token.trivia_index with
    | 0 ->
        token.token_index - 1
    | _ ->
        token.trivia_index - 1

  let compare one other =
    let open Stdlib in
    let compare_token_data = compare one.token_data other.token_data in
    if compare_token_data = 0 then
      let compare_token_index = compare one.token_index other.token_index in
      if compare_token_index = 0 then
        compare one.trivia_index other.trivia_index
      else
        compare_token_index
    else
      compare_token_data

  let equal one other =
    compare one other = 0

  let hash token =
    Hashtbl.hash
      (token.token_data
       , token.token_index
       , token.trivia_index)

  let is_equivalent one other =
    is_equivalent (addr (unwrap one)) (addr (unwrap other))

end

module BareNode = struct
  type t = unit ptr
end

module Rebindings = struct
  type t = unit ptr
end

module FileReader = struct
  (* Use a pointer to pointer to register a finaliser when creating a value of
     this type. *)
  type t = unit ptr ptr

  let dec_ref =
    foreign ~from:c_lib "lkt_dec_ref_file_reader"
      (ptr void @-> raisable void)

  let read v =
    let finalise arg =
      dec_ref (!@ arg)
    in
    allocate (ptr void) ~finalise v

  let c_type = view (ptr void) ~read ~write:(!@)
end

module UnitProvider = struct
  (* The real C type of a context is a void*. But we use a pointer to this
     type, to be able to allocate a value of t and attach a finalizer to it. *)
  type t = unit ptr ptr

  let c_type = ptr void

  let null = allocate c_type null

  

end



      
module MetadataStruct = struct
  type t

   
  let c_type : t structure typ = structure "metadata"
   let dummy = field c_type "dummy" char
  let () = seal c_type

end

      
module EntityInfoStruct = struct
  type t

   
  let c_type : t structure typ = structure "entity_info"
  let md =
    field c_type "md" MetadataStruct.c_type
  let rebindings =
    field c_type "rebindings" (ptr void)
  let from_rebound =
    field c_type "from_rebound" bool
  let () = seal c_type

end

         
module EntityStruct = struct
  type t

   
  let c_type : t structure typ = structure "lkt_node"
  let node =
    field c_type "node" (ptr void)
  let info =
    field c_type "info" EntityInfoStruct.c_type
  let () = seal c_type

end

      
module AnalysisUnitStruct : sig
  type t = unit ptr

  val c_type : t typ

  val unit_root : t -> EntityStruct.t structure ptr -> unit

  val unit_diagnostic_count : t -> int

  val unit_diagnostic : t -> int -> Diagnostic.t ptr -> int

  val unit_filename : t -> char ptr

  val unit_reparse_from_file : t -> string -> int

  val unit_reparse_from_buffer :
    t -> string -> string -> Unsigned.size_t -> int

  val unit_first_token : t -> Token.t structure ptr -> unit

  val unit_last_token : t -> Token.t structure ptr -> unit

  val unit_token_count : t -> int

  val unit_trivia_count : t -> int
end = struct
  (* Module defining the c structure of an analysis unit *)

  type t = unit ptr
  let c_type = ptr void

  let unit_root = foreign ~from:c_lib "lkt_unit_root"
    (c_type @-> ptr EntityStruct.c_type @-> raisable void)

  let unit_diagnostic_count = foreign ~from:c_lib
    "lkt_unit_diagnostic_count"
    (c_type @-> raisable int)

  let unit_diagnostic = foreign ~from:c_lib
    "lkt_unit_diagnostic"
    (c_type @-> int @-> ptr Diagnostic.c_type @-> raisable int)

  let unit_filename = foreign ~from:c_lib
    "lkt_unit_filename"
    (c_type @-> raisable (ptr char))

  let unit_reparse_from_file = foreign ~from:c_lib
    "lkt_unit_reparse_from_file"
    (c_type
     @-> string
     @-> raisable int)

  let unit_reparse_from_buffer = foreign ~from:c_lib
    "lkt_unit_reparse_from_buffer"
    (c_type
     @-> string
     @-> string
     @-> size_t
     @-> raisable int)

  let unit_first_token = foreign ~from:c_lib
    "lkt_unit_first_token"
    (c_type
     @-> ptr Token.c_type
     @-> raisable void)

  let unit_last_token = foreign ~from:c_lib
    "lkt_unit_last_token"
    (c_type
     @-> ptr Token.c_type
     @-> raisable void)

  let unit_token_count = foreign ~from:c_lib
    "lkt_unit_token_count"
    (c_type @-> raisable int)

  let unit_trivia_count = foreign ~from:c_lib
    "lkt_unit_trivia_count"
    (c_type @-> raisable int)
end


      
module AnalysisContextStruct : sig
  type t

  val c_type : t typ

  val allocate_analysis_context : ?keep:'a -> unit -> t
  val initialize_analysis_context :
    t -> string -> unit ptr -> unit ptr -> unit ptr -> bool -> int -> unit

  val get_analysis_unit_from_file :
    t -> string -> string -> bool -> GrammarRule.t -> AnalysisUnitStruct.t

  val get_analysis_unit_from_buffer :
    t
    -> string (* Filename *)
    -> string (* Charset *)
    -> string (* Buffer *)
    -> Unsigned.size_t (* Buffer size *)
    -> GrammarRule.t
    -> AnalysisUnitStruct.t
end = struct
  (* Module defining the c structure of an analysis context *)

  (* The real C type of a context is a void*. But we use a pointer to this
     type, to be able to allocate a value of t and attach a finalizer to it.
     See wrap function *)
  type t = unit ptr ptr

  let unwrap (value : t) : unit ptr = !@value

  (* The read part is not required as the only function returning a c_type is
     allocate_analysis_context which is manually written to take an object
     to keep alive as argument *)
  let c_type = view (ptr void) ~read:(fun _ -> assert false) ~write:unwrap

  let context_decref =
    foreign ~from:c_lib "lkt_context_decref"
      (c_type @-> raisable void)

  let c_allocate_analysis_context =
    foreign ~from:c_lib "lkt_allocate_analysis_context"
      ( void @-> raisable (ptr void) )

  let allocate_analysis_context ?keep () =
    (* To deallocate cleanly the context, we need to call context_decref.
       Allocate a value and attach a finalizer to it. Use the keep option
       to keep an object alive while the analysis context is. *)
    let ref_keep = ref keep in
    let finalise arg =
      ref_keep := None;
      context_decref arg
    in
    let c_value = c_allocate_analysis_context () in
    allocate ~finalise (ptr void) c_value

  let initialize_analysis_context =
    foreign ~from:c_lib "lkt_initialize_analysis_context"
      ( c_type @-> string @-> ptr void @-> UnitProvider.c_type @-> ptr void
      @-> bool @-> int @-> raisable void )

  let get_analysis_unit_from_file =
    foreign ~from:c_lib "lkt_get_analysis_unit_from_file"
      ( c_type @-> string @-> string @-> bool @-> GrammarRule.c_type
      @-> raisable AnalysisUnitStruct.c_type )

  let get_analysis_unit_from_buffer =
    foreign ~from:c_lib "lkt_get_analysis_unit_from_buffer"
      ( c_type @-> string (* Filename *) @-> string (* Charset *)
      @-> string (* Buffer *) @-> size_t (* Buffer size *)
      @-> GrammarRule.c_type
      @-> raisable AnalysisUnitStruct.c_type )

end

      
module Symbol : sig
  type t = string

  val c_type : t structure typ

  val wrap : (t structure) -> t

  val unwrap : AnalysisContextStruct.t -> t -> (t structure)

  val symbol_text : t structure ptr -> Text.t structure ptr -> unit

  val context_symbol :
    AnalysisContextStruct.t -> Text.t structure ptr -> t structure ptr -> int
end = struct
  type t = string

  let c_type : t structure typ = structure "symbol"
  let data = field c_type "data" (ptr void)
  let bounds = field c_type "bounds" (ptr void)
  let () = seal c_type

  let symbol_text = foreign ~from:c_lib "lkt_symbol_text"
    (ptr c_type @-> ptr Text.c_type @-> raisable void)

  let wrap (c_value : t structure) : t =
    let c_result_ptr = allocate_n Text.c_type ~count:1 in
    symbol_text (addr c_value) c_result_ptr;
    Text.wrap (!@ c_result_ptr)

  let context_symbol = foreign ~from:c_lib "lkt_context_symbol"
    (AnalysisContextStruct.c_type
     @-> ptr Text.c_type
     @-> ptr c_type
     @-> raisable int)

  let unwrap (ctx : AnalysisContextStruct.t) (value : t) : t structure =
    let result = make c_type in
    let c_text = Text.unwrap value in
    let code =
      context_symbol ctx (addr c_text) (addr result)
    in
    Text.destroy_text (addr c_text);
    if code = 0 then
      raise (InvalidSymbolError value) ;
    result
end

      
module CompleteItemStruct = struct
  type t

   
  let c_type : t structure typ = structure "complete_item"
  let declaration =
    field c_type "declaration" EntityStruct.c_type
  let () = seal c_type

end

      
module DecodedCharValueStruct = struct
  type t

   
  let c_type : t structure typ = structure "decoded_char_value"
  let value =
    field c_type "value" Character.c_type
  let has_error =
    field c_type "has_error" bool
  let error_sloc =
    field c_type "error_sloc" Sloc.c_type
  let error_message =
    field c_type "error_message" StringType.c_type
  let () = seal c_type

end

      
module DecodedStringValueStruct = struct
  type t

   
  let c_type : t structure typ = structure "decoded_string_value"
  let value =
    field c_type "value" StringType.c_type
  let has_error =
    field c_type "has_error" bool
  let error_sloc =
    field c_type "error_sloc" Sloc.c_type
  let error_message =
    field c_type "error_message" StringType.c_type
  let () = seal c_type

end

      
module LogicContextStruct = struct
  type t

   
  let c_type : t structure typ = structure "logic_context"
  let ref_node =
    field c_type "ref_node" EntityStruct.c_type
  let decl_node =
    field c_type "decl_node" EntityStruct.c_type
  let () = seal c_type

end

      
module RefResultStruct = struct
  type t

   
  let c_type : t structure typ = structure "ref_result"
  let ref_ =
    field c_type "ref_" EntityStruct.c_type
  let () = seal c_type

end

         
module LktNodeArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "lkt_node_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    EntityStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "lkt_node_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "lkt_node_array_dec_ref"
    (c_type @-> raisable void)

end

         
module LogicContextArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "logic_context_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    LogicContextStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "lkt_internal_logic_context_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "lkt_internal_logic_context_array_dec_ref"
    (c_type @-> raisable void)

end

      
module SolverDiagnosticStruct = struct
  type t

   
  let c_type : t structure typ = structure "solver_diagnostic"
  let message_template =
    field c_type "message_template" StringType.c_type
  let args =
    field c_type "args" LktNodeArrayStruct.c_type
  let location =
    field c_type "location" (ptr void)
  let contexts =
    field c_type "contexts" LogicContextArrayStruct.c_type
  let round =
    field c_type "round" int
  let () = seal c_type

end

         
module SolverDiagnosticArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "solver_diagnostic_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    SolverDiagnosticStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "lkt_internal_solver_diagnostic_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "lkt_internal_solver_diagnostic_array_dec_ref"
    (c_type @-> raisable void)

end

      
module SolverResultStruct = struct
  type t

   
  let c_type : t structure typ = structure "solver_result"
  let success =
    field c_type "success" bool
  let diagnostics =
    field c_type "diagnostics" SolverDiagnosticArrayStruct.c_type
  let () = seal c_type

end

         
module CompleteItemArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "complete_item_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    CompleteItemStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "lkt_internal_complete_item_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "lkt_internal_complete_item_array_dec_ref"
    (c_type @-> raisable void)

end

         
module RefResultArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "ref_result_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    RefResultStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "lkt_internal_ref_result_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "lkt_internal_ref_result_array_dec_ref"
    (c_type @-> raisable void)

end

         
module AnalysisUnitArrayStruct = struct
  

  type t

  let c_struct : t structure typ = structure "analysis_unit_array"
  let n = field c_struct "n" int
  let _ = field c_struct "ref_count" int
  let items = field c_struct "items"
    AnalysisUnitStruct.c_type
  let () = seal c_struct

  let c_type = ptr c_struct

  let create = foreign ~from:c_lib "lkt_analysis_unit_array_create"
    (int @-> raisable c_type)
  let dec_ref = foreign ~from:c_lib "lkt_analysis_unit_array_dec_ref"
    (c_type @-> raisable void)

end


module CFunctions = struct
  let node_kind = foreign ~from:c_lib "lkt_node_kind"
    (ptr EntityStruct.c_type @-> raisable int)

  let image = foreign ~from:c_lib "lkt_node_image"
    (ptr EntityStruct.c_type
     @-> ptr Text.c_type
     @-> raisable void)

  let node_sloc_range = foreign ~from:c_lib
    "lkt_node_sloc_range"
    (ptr EntityStruct.c_type
     @-> ptr SlocRange.c_type
     @-> raisable void)

  let lookup_in_node = foreign ~from:c_lib
    "lkt_lookup_in_node"
    (ptr EntityStruct.c_type
     @-> ptr Sloc.c_type
     @-> ptr EntityStruct.c_type
     @-> raisable void)

  let node_is_token_node = foreign ~from:c_lib
    "lkt_node_is_token_node"
    (ptr EntityStruct.c_type
     @-> raisable bool)

  let lkt_node_parent = foreign ~from:c_lib
    "lkt_lkt_node_parent"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_parents = foreign ~from:c_lib
    "lkt_lkt_node_parents"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr LktNodeArrayStruct.c_type
    @-> raisable int)

  let lkt_node_children = foreign ~from:c_lib
    "lkt_lkt_node_children"
    (ptr EntityStruct.c_type
    @-> ptr LktNodeArrayStruct.c_type
    @-> raisable int)

  let lkt_node_token_start = foreign ~from:c_lib
    "lkt_lkt_node_token_start"
    (ptr EntityStruct.c_type
    @-> ptr Token.c_type
    @-> raisable int)

  let lkt_node_token_end = foreign ~from:c_lib
    "lkt_lkt_node_token_end"
    (ptr EntityStruct.c_type
    @-> ptr Token.c_type
    @-> raisable int)

  let lkt_node_child_index = foreign ~from:c_lib
    "lkt_lkt_node_child_index"
    (ptr EntityStruct.c_type
    @-> ptr int
    @-> raisable int)

  let lkt_node_previous_sibling = foreign ~from:c_lib
    "lkt_lkt_node_previous_sibling"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_next_sibling = foreign ~from:c_lib
    "lkt_lkt_node_next_sibling"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_unit = foreign ~from:c_lib
    "lkt_lkt_node_unit"
    (ptr EntityStruct.c_type
    @-> ptr AnalysisUnitStruct.c_type
    @-> raisable int)

  let lkt_node_is_ghost = foreign ~from:c_lib
    "lkt_lkt_node_is_ghost"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let lkt_node_full_sloc_image = foreign ~from:c_lib
    "lkt_lkt_node_full_sloc_image"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let lkt_node_completion_item_kind_to_int = foreign ~from:c_lib
    "lkt_lkt_node_completion_item_kind_to_int"
    (ptr EntityStruct.c_type
        
    @-> CompletionItemKind.c_type
    @-> ptr int
    @-> raisable int)

  let lkt_node_p_set_solver_debug_mode = foreign ~from:c_lib
    "lkt_lkt_node_p_set_solver_debug_mode"
    (ptr EntityStruct.c_type
        
    @-> bool
    @-> ptr bool
    @-> raisable int)

  let lkt_node_p_basic_trait_gen = foreign ~from:c_lib
    "lkt_lkt_node_p_basic_trait_gen"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_basic_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_basic_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_node_gen_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_node_gen_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_node_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_node_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_indexable_gen_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_indexable_gen_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_indexable_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_indexable_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_token_node_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_token_node_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_error_node_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_error_node_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_char_type = foreign ~from:c_lib
    "lkt_lkt_node_p_char_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_int_type = foreign ~from:c_lib
    "lkt_lkt_node_p_int_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_bool_type = foreign ~from:c_lib
    "lkt_lkt_node_p_bool_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_bigint_type = foreign ~from:c_lib
    "lkt_lkt_node_p_bigint_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_string_type = foreign ~from:c_lib
    "lkt_lkt_node_p_string_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_symbol_type = foreign ~from:c_lib
    "lkt_lkt_node_p_symbol_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_property_error_type = foreign ~from:c_lib
    "lkt_lkt_node_p_property_error_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_regexp_type = foreign ~from:c_lib
    "lkt_lkt_node_p_regexp_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_entity_gen_type = foreign ~from:c_lib
    "lkt_lkt_node_p_entity_gen_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_entity_type = foreign ~from:c_lib
    "lkt_lkt_node_p_entity_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_logicvar_type = foreign ~from:c_lib
    "lkt_lkt_node_p_logicvar_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_equation_type = foreign ~from:c_lib
    "lkt_lkt_node_p_equation_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_array_gen_type = foreign ~from:c_lib
    "lkt_lkt_node_p_array_gen_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_array_type = foreign ~from:c_lib
    "lkt_lkt_node_p_array_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_astlist_gen_type = foreign ~from:c_lib
    "lkt_lkt_node_p_astlist_gen_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_astlist_type = foreign ~from:c_lib
    "lkt_lkt_node_p_astlist_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_node_builder_gen_type = foreign ~from:c_lib
    "lkt_lkt_node_p_node_builder_gen_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_node_builder_type = foreign ~from:c_lib
    "lkt_lkt_node_p_node_builder_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_iterator_gen_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_iterator_gen_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_iterator_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_iterator_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_analysis_unit_gen_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_analysis_unit_gen_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_analysis_unit_trait = foreign ~from:c_lib
    "lkt_lkt_node_p_analysis_unit_trait"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_topmost_invalid_decl = foreign ~from:c_lib
    "lkt_lkt_node_p_topmost_invalid_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lkt_node_p_nameres_diagnostics = foreign ~from:c_lib
    "lkt_lkt_node_p_nameres_diagnostics"
    (ptr EntityStruct.c_type
    @-> ptr SolverDiagnosticArrayStruct.c_type
    @-> raisable int)

  let lkt_node_p_solve_enclosing_context = foreign ~from:c_lib
    "lkt_lkt_node_p_solve_enclosing_context"
    (ptr EntityStruct.c_type
    @-> ptr SolverResultStruct.c_type
    @-> raisable int)

  let lkt_node_p_xref_entry_point = foreign ~from:c_lib
    "lkt_lkt_node_p_xref_entry_point"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let lkt_node_p_complete = foreign ~from:c_lib
    "lkt_lkt_node_p_complete"
    (ptr EntityStruct.c_type
    @-> ptr CompleteItemArrayStruct.c_type
    @-> raisable int)

  let argument_f_name = foreign ~from:c_lib
    "lkt_argument_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let argument_f_value = foreign ~from:c_lib
    "lkt_argument_f_value"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_lexer_case_rule_alt_f_send = foreign ~from:c_lib
    "lkt_base_lexer_case_rule_alt_f_send"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lexer_case_rule_cond_alt_f_cond_exprs = foreign ~from:c_lib
    "lkt_lexer_case_rule_cond_alt_f_cond_exprs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_match_branch_f_expr = foreign ~from:c_lib
    "lkt_base_match_branch_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_match_branch_p_match_part = foreign ~from:c_lib
    "lkt_base_match_branch_p_match_part"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let match_branch_f_decl = foreign ~from:c_lib
    "lkt_match_branch_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let pattern_match_branch_f_pattern = foreign ~from:c_lib
    "lkt_pattern_match_branch_f_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let block_expr_clause_f_clause = foreign ~from:c_lib
    "lkt_block_expr_clause_f_clause"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let class_qualifier_p_as_bool = foreign ~from:c_lib
    "lkt_class_qualifier_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let decl_f_syn_name = foreign ~from:c_lib
    "lkt_decl_f_syn_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_p_custom_image = foreign ~from:c_lib
    "lkt_decl_p_custom_image"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let decl_p_decl_type_name = foreign ~from:c_lib
    "lkt_decl_p_decl_type_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let decl_p_def_ids = foreign ~from:c_lib
    "lkt_decl_p_def_ids"
    (ptr EntityStruct.c_type
    @-> ptr LktNodeArrayStruct.c_type
    @-> raisable int)

  let decl_p_as_bare_decl = foreign ~from:c_lib
    "lkt_decl_p_as_bare_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_p_get_type = foreign ~from:c_lib
    "lkt_decl_p_get_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_p_get_cast_type = foreign ~from:c_lib
    "lkt_decl_p_get_cast_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_p_get_keep_type = foreign ~from:c_lib
    "lkt_decl_p_get_keep_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_p_get_suffix_type = foreign ~from:c_lib
    "lkt_decl_p_get_suffix_type"
    (ptr EntityStruct.c_type
        
    @-> ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_p_is_generic = foreign ~from:c_lib
    "lkt_decl_p_is_generic"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let decl_p_return_type_is_instantiated = foreign ~from:c_lib
    "lkt_decl_p_return_type_is_instantiated"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let decl_p_is_instantiated = foreign ~from:c_lib
    "lkt_decl_p_is_instantiated"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let decl_p_name = foreign ~from:c_lib
    "lkt_decl_p_name"
    (ptr EntityStruct.c_type
    @-> ptr Symbol.c_type
    @-> raisable int)

  let decl_p_full_name = foreign ~from:c_lib
    "lkt_decl_p_full_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let base_grammar_rule_decl_f_expr = foreign ~from:c_lib
    "lkt_base_grammar_rule_decl_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let explicitly_typed_decl_f_decl_type = foreign ~from:c_lib
    "lkt_explicitly_typed_decl_f_decl_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let component_decl_f_default_val = foreign ~from:c_lib
    "lkt_component_decl_f_default_val"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let field_decl_f_trait_ref = foreign ~from:c_lib
    "lkt_field_decl_f_trait_ref"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let fun_param_decl_f_decl_annotations = foreign ~from:c_lib
    "lkt_fun_param_decl_f_decl_annotations"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let val_decl_f_expr = foreign ~from:c_lib
    "lkt_val_decl_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let fun_decl_f_params = foreign ~from:c_lib
    "lkt_fun_decl_f_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let fun_decl_f_return_type = foreign ~from:c_lib
    "lkt_fun_decl_f_return_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let fun_decl_f_trait_ref = foreign ~from:c_lib
    "lkt_fun_decl_f_trait_ref"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let fun_decl_f_body = foreign ~from:c_lib
    "lkt_fun_decl_f_body"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let fun_decl_p_is_dynamic_combiner = foreign ~from:c_lib
    "lkt_fun_decl_p_is_dynamic_combiner"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let fun_decl_p_find_all_overrides = foreign ~from:c_lib
    "lkt_fun_decl_p_find_all_overrides"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitArrayStruct.c_type
    @-> ptr LktNodeArrayStruct.c_type
    @-> raisable int)

  let env_spec_decl_f_actions = foreign ~from:c_lib
    "lkt_env_spec_decl_f_actions"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_decl_f_generic_param_decls = foreign ~from:c_lib
    "lkt_generic_decl_f_generic_param_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_decl_f_decl = foreign ~from:c_lib
    "lkt_generic_decl_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_decl_f_rules = foreign ~from:c_lib
    "lkt_grammar_decl_f_rules"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lexer_decl_f_rules = foreign ~from:c_lib
    "lkt_lexer_decl_f_rules"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lexer_family_decl_f_rules = foreign ~from:c_lib
    "lkt_lexer_family_decl_f_rules"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_decl_f_traits = foreign ~from:c_lib
    "lkt_type_decl_f_traits"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_decl_f_syn_base_type = foreign ~from:c_lib
    "lkt_type_decl_f_syn_base_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_decl_p_def_id = foreign ~from:c_lib
    "lkt_type_decl_p_def_id"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_decl_p_base_type = foreign ~from:c_lib
    "lkt_type_decl_p_base_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_decl_p_base_type_if_entity = foreign ~from:c_lib
    "lkt_type_decl_p_base_type_if_entity"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_param_type_decl_f_has_class = foreign ~from:c_lib
    "lkt_generic_param_type_decl_f_has_class"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let named_type_decl_f_decls = foreign ~from:c_lib
    "lkt_named_type_decl_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_class_decl_f_branches = foreign ~from:c_lib
    "lkt_enum_class_decl_f_branches"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_type_decl_f_literals = foreign ~from:c_lib
    "lkt_enum_type_decl_f_literals"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_annotation_f_name = foreign ~from:c_lib
    "lkt_decl_annotation_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_annotation_f_args = foreign ~from:c_lib
    "lkt_decl_annotation_f_args"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let decl_annotation_args_f_args = foreign ~from:c_lib
    "lkt_decl_annotation_args_f_args"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let elsif_branch_f_cond_expr = foreign ~from:c_lib
    "lkt_elsif_branch_f_cond_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let elsif_branch_f_then_expr = foreign ~from:c_lib
    "lkt_elsif_branch_f_then_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let enum_class_case_f_decls = foreign ~from:c_lib
    "lkt_enum_class_case_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let excludes_null_p_as_bool = foreign ~from:c_lib
    "lkt_excludes_null_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let expr_p_get_type = foreign ~from:c_lib
    "lkt_expr_p_get_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let expr_p_get_generic_type = foreign ~from:c_lib
    "lkt_expr_p_get_generic_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let expr_p_get_expected_type = foreign ~from:c_lib
    "lkt_expr_p_get_expected_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let expr_p_referenced_decl = foreign ~from:c_lib
    "lkt_expr_p_referenced_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let any_of_f_expr = foreign ~from:c_lib
    "lkt_any_of_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let any_of_f_values = foreign ~from:c_lib
    "lkt_any_of_f_values"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let array_literal_f_exprs = foreign ~from:c_lib
    "lkt_array_literal_f_exprs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let array_literal_f_element_type = foreign ~from:c_lib
    "lkt_array_literal_f_element_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_call_expr_f_name = foreign ~from:c_lib
    "lkt_base_call_expr_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let base_call_expr_f_args = foreign ~from:c_lib
    "lkt_base_call_expr_f_args"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let bin_op_f_left = foreign ~from:c_lib
    "lkt_bin_op_f_left"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let bin_op_f_op = foreign ~from:c_lib
    "lkt_bin_op_f_op"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let bin_op_f_right = foreign ~from:c_lib
    "lkt_bin_op_f_right"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let block_expr_f_clauses = foreign ~from:c_lib
    "lkt_block_expr_f_clauses"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let cast_expr_f_expr = foreign ~from:c_lib
    "lkt_cast_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let cast_expr_f_null_cond = foreign ~from:c_lib
    "lkt_cast_expr_f_null_cond"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let cast_expr_f_excludes_null = foreign ~from:c_lib
    "lkt_cast_expr_f_excludes_null"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let cast_expr_f_dest_type = foreign ~from:c_lib
    "lkt_cast_expr_f_dest_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let dot_expr_f_prefix = foreign ~from:c_lib
    "lkt_dot_expr_f_prefix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let dot_expr_f_null_cond = foreign ~from:c_lib
    "lkt_dot_expr_f_null_cond"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let dot_expr_f_suffix = foreign ~from:c_lib
    "lkt_dot_expr_f_suffix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let error_on_null_f_expr = foreign ~from:c_lib
    "lkt_error_on_null_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_instantiation_f_name = foreign ~from:c_lib
    "lkt_generic_instantiation_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_instantiation_f_args = foreign ~from:c_lib
    "lkt_generic_instantiation_f_args"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_discard_f_expr = foreign ~from:c_lib
    "lkt_grammar_discard_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_dont_skip_f_expr = foreign ~from:c_lib
    "lkt_grammar_dont_skip_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_dont_skip_f_dont_skip = foreign ~from:c_lib
    "lkt_grammar_dont_skip_f_dont_skip"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_list_f_list_type = foreign ~from:c_lib
    "lkt_grammar_list_f_list_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_list_f_kind = foreign ~from:c_lib
    "lkt_grammar_list_f_kind"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_list_f_expr = foreign ~from:c_lib
    "lkt_grammar_list_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_list_f_sep = foreign ~from:c_lib
    "lkt_grammar_list_f_sep"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_null_f_name = foreign ~from:c_lib
    "lkt_grammar_null_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_opt_f_expr = foreign ~from:c_lib
    "lkt_grammar_opt_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_opt_error_f_expr = foreign ~from:c_lib
    "lkt_grammar_opt_error_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_opt_error_group_f_expr = foreign ~from:c_lib
    "lkt_grammar_opt_error_group_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_opt_group_f_expr = foreign ~from:c_lib
    "lkt_grammar_opt_group_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_or_expr_f_sub_exprs = foreign ~from:c_lib
    "lkt_grammar_or_expr_f_sub_exprs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_pick_f_exprs = foreign ~from:c_lib
    "lkt_grammar_pick_f_exprs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_predicate_f_expr = foreign ~from:c_lib
    "lkt_grammar_predicate_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_predicate_f_prop_ref = foreign ~from:c_lib
    "lkt_grammar_predicate_f_prop_ref"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_rule_ref_f_node_name = foreign ~from:c_lib
    "lkt_grammar_rule_ref_f_node_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_skip_f_name = foreign ~from:c_lib
    "lkt_grammar_skip_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_stop_cut_f_expr = foreign ~from:c_lib
    "lkt_grammar_stop_cut_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let parse_node_expr_f_node_name = foreign ~from:c_lib
    "lkt_parse_node_expr_f_node_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let parse_node_expr_f_sub_exprs = foreign ~from:c_lib
    "lkt_parse_node_expr_f_sub_exprs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let token_lit_p_denoted_value = foreign ~from:c_lib
    "lkt_token_lit_p_denoted_value"
    (ptr EntityStruct.c_type
    @-> ptr DecodedStringValueStruct.c_type
    @-> raisable int)

  let token_no_case_lit_f_lit = foreign ~from:c_lib
    "lkt_token_no_case_lit_f_lit"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let token_pattern_concat_f_left = foreign ~from:c_lib
    "lkt_token_pattern_concat_f_left"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let token_pattern_concat_f_right = foreign ~from:c_lib
    "lkt_token_pattern_concat_f_right"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let token_pattern_lit_p_denoted_value = foreign ~from:c_lib
    "lkt_token_pattern_lit_p_denoted_value"
    (ptr EntityStruct.c_type
    @-> ptr DecodedStringValueStruct.c_type
    @-> raisable int)

  let token_ref_f_token_name = foreign ~from:c_lib
    "lkt_token_ref_f_token_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let token_ref_f_expr = foreign ~from:c_lib
    "lkt_token_ref_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let id_p_custom_image = foreign ~from:c_lib
    "lkt_id_p_custom_image"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let def_id_p_name = foreign ~from:c_lib
    "lkt_def_id_p_name"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let def_id_p_get_implementatinons = foreign ~from:c_lib
    "lkt_def_id_p_get_implementatinons"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitArrayStruct.c_type
    @-> ptr LktNodeArrayStruct.c_type
    @-> raisable int)

  let def_id_p_decl_detail = foreign ~from:c_lib
    "lkt_def_id_p_decl_detail"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let def_id_p_completion_item_kind = foreign ~from:c_lib
    "lkt_def_id_p_completion_item_kind"
    (ptr EntityStruct.c_type
    @-> ptr int
    @-> raisable int)

  let def_id_p_doc = foreign ~from:c_lib
    "lkt_def_id_p_doc"
    (ptr EntityStruct.c_type
    @-> ptr StringType.c_type
    @-> raisable int)

  let def_id_p_find_all_references = foreign ~from:c_lib
    "lkt_def_id_p_find_all_references"
    (ptr EntityStruct.c_type
        
    @-> AnalysisUnitArrayStruct.c_type
    @-> ptr RefResultArrayStruct.c_type
    @-> raisable int)

  let ref_id_p_referenced_defining_name = foreign ~from:c_lib
    "lkt_ref_id_p_referenced_defining_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_expr_f_cond_expr = foreign ~from:c_lib
    "lkt_if_expr_f_cond_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_expr_f_then_expr = foreign ~from:c_lib
    "lkt_if_expr_f_then_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_expr_f_alternatives = foreign ~from:c_lib
    "lkt_if_expr_f_alternatives"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let if_expr_f_else_expr = foreign ~from:c_lib
    "lkt_if_expr_f_else_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let isa_f_expr = foreign ~from:c_lib
    "lkt_isa_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let isa_f_pattern = foreign ~from:c_lib
    "lkt_isa_f_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let keep_expr_f_expr = foreign ~from:c_lib
    "lkt_keep_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let keep_expr_f_null_cond = foreign ~from:c_lib
    "lkt_keep_expr_f_null_cond"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let keep_expr_f_keep_type = foreign ~from:c_lib
    "lkt_keep_expr_f_keep_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lambda_expr_f_params = foreign ~from:c_lib
    "lkt_lambda_expr_f_params"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lambda_expr_f_return_type = foreign ~from:c_lib
    "lkt_lambda_expr_f_return_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lambda_expr_f_body = foreign ~from:c_lib
    "lkt_lambda_expr_f_body"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let char_lit_p_denoted_value = foreign ~from:c_lib
    "lkt_char_lit_p_denoted_value"
    (ptr EntityStruct.c_type
    @-> ptr DecodedCharValueStruct.c_type
    @-> raisable int)

  let null_lit_f_dest_type = foreign ~from:c_lib
    "lkt_null_lit_f_dest_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let string_lit_p_denoted_value = foreign ~from:c_lib
    "lkt_string_lit_p_denoted_value"
    (ptr EntityStruct.c_type
    @-> ptr DecodedStringValueStruct.c_type
    @-> raisable int)

  let string_lit_p_is_prefixed_string = foreign ~from:c_lib
    "lkt_string_lit_p_is_prefixed_string"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let string_lit_p_prefix = foreign ~from:c_lib
    "lkt_string_lit_p_prefix"
    (ptr EntityStruct.c_type
    @-> ptr Character.c_type
    @-> raisable int)

  let string_lit_p_is_regexp_literal = foreign ~from:c_lib
    "lkt_string_lit_p_is_regexp_literal"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let block_string_lit_f_lines = foreign ~from:c_lib
    "lkt_block_string_lit_f_lines"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let logic_assign_f_dest_var = foreign ~from:c_lib
    "lkt_logic_assign_f_dest_var"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let logic_assign_f_value = foreign ~from:c_lib
    "lkt_logic_assign_f_value"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let logic_expr_f_expr = foreign ~from:c_lib
    "lkt_logic_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let logic_propagate_f_dest_var = foreign ~from:c_lib
    "lkt_logic_propagate_f_dest_var"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let logic_propagate_f_call = foreign ~from:c_lib
    "lkt_logic_propagate_f_call"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let logic_unify_f_lhs = foreign ~from:c_lib
    "lkt_logic_unify_f_lhs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let logic_unify_f_rhs = foreign ~from:c_lib
    "lkt_logic_unify_f_rhs"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let match_expr_f_match_expr = foreign ~from:c_lib
    "lkt_match_expr_f_match_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let match_expr_f_branches = foreign ~from:c_lib
    "lkt_match_expr_f_branches"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let not_expr_f_expr = foreign ~from:c_lib
    "lkt_not_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let paren_expr_f_expr = foreign ~from:c_lib
    "lkt_paren_expr_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let raise_expr_f_dest_type = foreign ~from:c_lib
    "lkt_raise_expr_f_dest_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let raise_expr_f_except_expr = foreign ~from:c_lib
    "lkt_raise_expr_f_except_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subscript_expr_f_prefix = foreign ~from:c_lib
    "lkt_subscript_expr_f_prefix"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subscript_expr_f_null_cond = foreign ~from:c_lib
    "lkt_subscript_expr_f_null_cond"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let subscript_expr_f_index = foreign ~from:c_lib
    "lkt_subscript_expr_f_index"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let try_expr_f_try_expr = foreign ~from:c_lib
    "lkt_try_expr_f_try_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let try_expr_f_or_expr = foreign ~from:c_lib
    "lkt_try_expr_f_or_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let un_op_f_op = foreign ~from:c_lib
    "lkt_un_op_f_op"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let un_op_f_expr = foreign ~from:c_lib
    "lkt_un_op_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let full_decl_f_doc = foreign ~from:c_lib
    "lkt_full_decl_f_doc"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let full_decl_f_decl_annotations = foreign ~from:c_lib
    "lkt_full_decl_f_decl_annotations"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let full_decl_f_decl = foreign ~from:c_lib
    "lkt_full_decl_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let full_decl_p_has_annotation = foreign ~from:c_lib
    "lkt_full_decl_p_has_annotation"
    (ptr EntityStruct.c_type
        
    @-> ptr Symbol.c_type
    @-> ptr bool
    @-> raisable int)

  let grammar_list_sep_f_token = foreign ~from:c_lib
    "lkt_grammar_list_sep_f_token"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let grammar_list_sep_f_extra = foreign ~from:c_lib
    "lkt_grammar_list_sep_f_extra"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let import_f_name = foreign ~from:c_lib
    "lkt_import_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let import_p_referenced_unit = foreign ~from:c_lib
    "lkt_import_p_referenced_unit"
    (ptr EntityStruct.c_type
    @-> ptr AnalysisUnitStruct.c_type
    @-> raisable int)

  let langkit_root_f_imports = foreign ~from:c_lib
    "lkt_langkit_root_f_imports"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let langkit_root_f_decls = foreign ~from:c_lib
    "lkt_langkit_root_f_decls"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let langkit_root_p_fetch_prelude = foreign ~from:c_lib
    "lkt_langkit_root_p_fetch_prelude"
    (ptr EntityStruct.c_type
    @-> ptr AnalysisUnitStruct.c_type
    @-> raisable int)

  let lexer_case_rule_f_expr = foreign ~from:c_lib
    "lkt_lexer_case_rule_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lexer_case_rule_f_alts = foreign ~from:c_lib
    "lkt_lexer_case_rule_f_alts"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lexer_case_rule_send_f_sent = foreign ~from:c_lib
    "lkt_lexer_case_rule_send_f_sent"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let lexer_case_rule_send_f_match_size = foreign ~from:c_lib
    "lkt_lexer_case_rule_send_f_match_size"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let null_cond_qualifier_p_as_bool = foreign ~from:c_lib
    "lkt_null_cond_qualifier_p_as_bool"
    (ptr EntityStruct.c_type
    @-> ptr bool
    @-> raisable int)

  let binding_pattern_f_decl = foreign ~from:c_lib
    "lkt_binding_pattern_f_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let binding_pattern_f_sub_pattern = foreign ~from:c_lib
    "lkt_binding_pattern_f_sub_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let ellipsis_pattern_f_binding = foreign ~from:c_lib
    "lkt_ellipsis_pattern_f_binding"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let extended_pattern_f_sub_pattern = foreign ~from:c_lib
    "lkt_extended_pattern_f_sub_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let extended_pattern_f_details = foreign ~from:c_lib
    "lkt_extended_pattern_f_details"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let filtered_pattern_f_sub_pattern = foreign ~from:c_lib
    "lkt_filtered_pattern_f_sub_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let filtered_pattern_f_predicate = foreign ~from:c_lib
    "lkt_filtered_pattern_f_predicate"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let list_pattern_f_sub_patterns = foreign ~from:c_lib
    "lkt_list_pattern_f_sub_patterns"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let not_pattern_f_sub_pattern = foreign ~from:c_lib
    "lkt_not_pattern_f_sub_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let or_pattern_f_left_sub_pattern = foreign ~from:c_lib
    "lkt_or_pattern_f_left_sub_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let or_pattern_f_right_sub_pattern = foreign ~from:c_lib
    "lkt_or_pattern_f_right_sub_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let paren_pattern_f_sub_pattern = foreign ~from:c_lib
    "lkt_paren_pattern_f_sub_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let tuple_pattern_f_sub_patterns = foreign ~from:c_lib
    "lkt_tuple_pattern_f_sub_patterns"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_pattern_f_type_name = foreign ~from:c_lib
    "lkt_type_pattern_f_type_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let field_pattern_detail_f_id = foreign ~from:c_lib
    "lkt_field_pattern_detail_f_id"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let field_pattern_detail_f_expected_value = foreign ~from:c_lib
    "lkt_field_pattern_detail_f_expected_value"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let property_pattern_detail_f_call = foreign ~from:c_lib
    "lkt_property_pattern_detail_f_call"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let property_pattern_detail_f_expected_value = foreign ~from:c_lib
    "lkt_property_pattern_detail_f_expected_value"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let selector_pattern_detail_f_call = foreign ~from:c_lib
    "lkt_selector_pattern_detail_f_call"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let selector_pattern_detail_f_sub_pattern = foreign ~from:c_lib
    "lkt_selector_pattern_detail_f_sub_pattern"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let selector_call_f_quantifier = foreign ~from:c_lib
    "lkt_selector_call_f_quantifier"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let selector_call_f_binding = foreign ~from:c_lib
    "lkt_selector_call_f_binding"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let selector_call_f_selector_call = foreign ~from:c_lib
    "lkt_selector_call_f_selector_call"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let type_ref_p_referenced_decl = foreign ~from:c_lib
    "lkt_type_ref_p_referenced_decl"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let function_type_ref_f_param_types = foreign ~from:c_lib
    "lkt_function_type_ref_f_param_types"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let function_type_ref_f_return_type = foreign ~from:c_lib
    "lkt_function_type_ref_f_return_type"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_type_ref_f_type_name = foreign ~from:c_lib
    "lkt_generic_type_ref_f_type_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let generic_type_ref_f_args = foreign ~from:c_lib
    "lkt_generic_type_ref_f_args"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let simple_type_ref_f_type_name = foreign ~from:c_lib
    "lkt_simple_type_ref_f_type_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let var_bind_f_name = foreign ~from:c_lib
    "lkt_var_bind_f_name"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)

  let var_bind_f_expr = foreign ~from:c_lib
    "lkt_var_bind_f_expr"
    (ptr EntityStruct.c_type
    @-> ptr EntityStruct.c_type
    @-> raisable int)


end

type analysis_context = {
  c_value : AnalysisContextStruct.t;
  unit_provider : UnitProvider.t
}

and analysis_unit = {
  c_value : AnalysisUnitStruct.t;
  context : analysis_context
}

and entity = EntityStruct.t structure


  and entity_info = {
      rebindings :
         Rebindings.t;
      from_rebound :
         bool;
  }





  
   
  (**
    * argument
    * base_lexer_case_rule_alt
    * base_match_branch
    * block_expr_clause
    * block_string_line
    * class_qualifier
    * decl
    * decl_annotation
    * decl_annotation_args
    * dyn_env_wrapper
    * elsif_branch
    * enum_class_case
    * excludes_null
    * expr
    * full_decl
    * grammar_list_sep
    * import
    * langkit_root
    * lexer_case_rule
    * lexer_case_rule_send
    * list_kind
    * lkt_node_base_list
    * null_cond_qualifier
    * op
    * pattern
    * pattern_detail
    * selector_call
    * type_ref
    * var_bind
    *)
  and lkt_node =
    [
    | `Argument
        of argument_fields
    | `LexerCaseRuleCondAlt
        of lexer_case_rule_cond_alt_fields
    | `LexerCaseRuleDefaultAlt
        of lexer_case_rule_default_alt_fields
    | `MatchBranch
        of match_branch_fields
    | `PatternMatchBranch
        of pattern_match_branch_fields
    | `BlockExprClause
        of block_expr_clause_fields
    | `BlockStringLine
        of block_string_line_fields
    | `ClassQualifierAbsent
        of class_qualifier_absent_fields
    | `ClassQualifierPresent
        of class_qualifier_present_fields
    | `GrammarRuleDecl
        of grammar_rule_decl_fields
    | `SyntheticLexerDecl
        of synthetic_lexer_decl_fields
    | `NodeDecl
        of node_decl_fields
    | `SelfDecl
        of self_decl_fields
    | `BindingValDecl
        of binding_val_decl_fields
    | `EnumLitDecl
        of enum_lit_decl_fields
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    | `FunDecl
        of fun_decl_fields
    | `EnvSpecDecl
        of env_spec_decl_fields
    | `ErrorDecl
        of error_decl_fields
    | `GenericDecl
        of generic_decl_fields
    | `GrammarDecl
        of grammar_decl_fields
    | `LexerDecl
        of lexer_decl_fields
    | `LexerFamilyDecl
        of lexer_family_decl_fields
    | `SynthFunDecl
        of synth_fun_decl_fields
    | `SynthParamDecl
        of synth_param_decl_fields
    | `AnyTypeDecl
        of any_type_decl_fields
    | `EnumClassAltDecl
        of enum_class_alt_decl_fields
    | `FunctionType
        of function_type_fields
    | `GenericParamTypeDecl
        of generic_param_type_decl_fields
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    | `EnumTypeDecl
        of enum_type_decl_fields
    | `StructDecl
        of struct_decl_fields
    | `TraitDecl
        of trait_decl_fields
    | `DeclAnnotation
        of decl_annotation_fields
    | `DeclAnnotationArgs
        of decl_annotation_args_fields
    | `DynEnvWrapper
        of dyn_env_wrapper_fields
    | `ElsifBranch
        of elsif_branch_fields
    | `EnumClassCase
        of enum_class_case_fields
    | `ExcludesNullAbsent
        of excludes_null_absent_fields
    | `ExcludesNullPresent
        of excludes_null_present_fields
    | `AnyOf
        of any_of_fields
    | `ArrayLiteral
        of array_literal_fields
    | `CallExpr
        of call_expr_fields
    | `LogicPredicate
        of logic_predicate_fields
    | `LogicPropagateCall
        of logic_propagate_call_fields
    | `BinOp
        of bin_op_fields
    | `BlockExpr
        of block_expr_fields
    | `CastExpr
        of cast_expr_fields
    | `DotExpr
        of dot_expr_fields
    | `ErrorOnNull
        of error_on_null_fields
    | `GenericInstantiation
        of generic_instantiation_fields
    | `GrammarCut
        of grammar_cut_fields
    | `GrammarDiscard
        of grammar_discard_fields
    | `GrammarDontSkip
        of grammar_dont_skip_fields
    | `GrammarList
        of grammar_list_fields
    | `GrammarNull
        of grammar_null_fields
    | `GrammarOpt
        of grammar_opt_fields
    | `GrammarOptError
        of grammar_opt_error_fields
    | `GrammarOptErrorGroup
        of grammar_opt_error_group_fields
    | `GrammarOptGroup
        of grammar_opt_group_fields
    | `GrammarOrExpr
        of grammar_or_expr_fields
    | `GrammarPick
        of grammar_pick_fields
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    | `GrammarPredicate
        of grammar_predicate_fields
    | `GrammarRuleRef
        of grammar_rule_ref_fields
    | `GrammarSkip
        of grammar_skip_fields
    | `GrammarStopCut
        of grammar_stop_cut_fields
    | `ParseNodeExpr
        of parse_node_expr_fields
    | `TokenLit
        of token_lit_fields
    | `TokenNoCaseLit
        of token_no_case_lit_fields
    | `TokenPatternConcat
        of token_pattern_concat_fields
    | `TokenPatternLit
        of token_pattern_lit_fields
    | `TokenRef
        of token_ref_fields
    | `Id
        of id_fields
    | `DefId
        of def_id_fields
    | `ModuleRefId
        of module_ref_id_fields
    | `RefId
        of ref_id_fields
    | `IfExpr
        of if_expr_fields
    | `Isa
        of isa_fields
    | `KeepExpr
        of keep_expr_fields
    | `LambdaExpr
        of lambda_expr_fields
    | `BigNumLit
        of big_num_lit_fields
    | `CharLit
        of char_lit_fields
    | `NullLit
        of null_lit_fields
    | `NumLit
        of num_lit_fields
    | `BlockStringLit
        of block_string_lit_fields
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    | `LogicAssign
        of logic_assign_fields
    | `LogicExpr
        of logic_expr_fields
    | `LogicPropagate
        of logic_propagate_fields
    | `LogicUnify
        of logic_unify_fields
    | `MatchExpr
        of match_expr_fields
    | `NotExpr
        of not_expr_fields
    | `ParenExpr
        of paren_expr_fields
    | `RaiseExpr
        of raise_expr_fields
    | `SubscriptExpr
        of subscript_expr_fields
    | `TryExpr
        of try_expr_fields
    | `UnOp
        of un_op_fields
    | `FullDecl
        of full_decl_fields
    | `GrammarListSep
        of grammar_list_sep_fields
    | `Import
        of import_fields
    | `LangkitRoot
        of langkit_root_fields
    | `LexerCaseRule
        of lexer_case_rule_fields
    | `LexerCaseRuleSend
        of lexer_case_rule_send_fields
    | `ListKindOne
        of list_kind_one_fields
    | `ListKindZero
        of list_kind_zero_fields
    | `ArgumentList
        of argument_list_fields
    | `BaseLexerCaseRuleAltList
        of base_lexer_case_rule_alt_list_fields
    | `BaseMatchBranchList
        of base_match_branch_list_fields
    | `BlockStringLineList
        of block_string_line_list_fields
    | `CallExprList
        of call_expr_list_fields
    | `DeclAnnotationList
        of decl_annotation_list_fields
    | `ElsifBranchList
        of elsif_branch_list_fields
    | `EnumClassAltDeclList
        of enum_class_alt_decl_list_fields
    | `EnumClassCaseList
        of enum_class_case_list_fields
    | `EnumLitDeclList
        of enum_lit_decl_list_fields
    | `ExprList
        of expr_list_fields
    | `AnyOfList
        of any_of_list_fields
    | `FullDeclList
        of full_decl_list_fields
    | `DeclBlock
        of decl_block_fields
    | `GenericParamDeclList
        of generic_param_decl_list_fields
    | `FunParamDeclList
        of fun_param_decl_list_fields
    | `GrammarExprList
        of grammar_expr_list_fields
    | `GrammarExprListList
        of grammar_expr_list_list_fields
    | `ImportList
        of import_list_fields
    | `LambdaParamDeclList
        of lambda_param_decl_list_fields
    | `LktNodeList
        of lkt_node_list_fields
    | `PatternDetailList
        of pattern_detail_list_fields
    | `PatternList
        of pattern_list_fields
    | `RefIdList
        of ref_id_list_fields
    | `TypeRefList
        of type_ref_list_fields
    | `SyntheticTypeRefList
        of synthetic_type_ref_list_fields
    | `NullCondQualifierAbsent
        of null_cond_qualifier_absent_fields
    | `NullCondQualifierPresent
        of null_cond_qualifier_present_fields
    | `OpAmp
        of op_amp_fields
    | `OpAnd
        of op_and_fields
    | `OpDiv
        of op_div_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpLogicAnd
        of op_logic_and_fields
    | `OpLogicOr
        of op_logic_or_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMult
        of op_mult_fields
    | `OpNe
        of op_ne_fields
    | `OpOr
        of op_or_fields
    | `OpOrInt
        of op_or_int_fields
    | `OpPlus
        of op_plus_fields
    | `AnyTypePattern
        of any_type_pattern_fields
    | `BindingPattern
        of binding_pattern_fields
    | `BoolPatternFalse
        of bool_pattern_false_fields
    | `BoolPatternTrue
        of bool_pattern_true_fields
    | `EllipsisPattern
        of ellipsis_pattern_fields
    | `ExtendedPattern
        of extended_pattern_fields
    | `FilteredPattern
        of filtered_pattern_fields
    | `IntegerPattern
        of integer_pattern_fields
    | `ListPattern
        of list_pattern_fields
    | `NotPattern
        of not_pattern_fields
    | `NullPattern
        of null_pattern_fields
    | `OrPattern
        of or_pattern_fields
    | `ParenPattern
        of paren_pattern_fields
    | `RegexPattern
        of regex_pattern_fields
    | `TuplePattern
        of tuple_pattern_fields
    | `TypePattern
        of type_pattern_fields
    | `FieldPatternDetail
        of field_pattern_detail_fields
    | `PropertyPatternDetail
        of property_pattern_detail_fields
    | `SelectorPatternDetail
        of selector_pattern_detail_fields
    | `SelectorCall
        of selector_call_fields
    | `DefaultListTypeRef
        of default_list_type_ref_fields
    | `FunctionTypeRef
        of function_type_ref_fields
    | `GenericTypeRef
        of generic_type_ref_fields
    | `SimpleTypeRef
        of simple_type_ref_fields
    | `VarBind
        of var_bind_fields
    ]

  
   
  and argument =
    [
    | `Argument
        of argument_fields
    ]
  and argument_fields = 
  {
         
    f_name: ref_id
    option
    Lazy.t;
         
    f_value: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_lexer_case_rule_alt =
    [
    | `LexerCaseRuleCondAlt
        of lexer_case_rule_cond_alt_fields
    | `LexerCaseRuleDefaultAlt
        of lexer_case_rule_default_alt_fields
    ]

  
   
  and lexer_case_rule_cond_alt =
    [
    | `LexerCaseRuleCondAlt
        of lexer_case_rule_cond_alt_fields
    ]
  and lexer_case_rule_cond_alt_fields = 
  {
         
    f_cond_exprs: ref_id_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_case_rule_default_alt =
    [
    | `LexerCaseRuleDefaultAlt
        of lexer_case_rule_default_alt_fields
    ]
  and lexer_case_rule_default_alt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_match_branch =
    [
    | `MatchBranch
        of match_branch_fields
    | `PatternMatchBranch
        of pattern_match_branch_fields
    ]

  
   
  and match_branch =
    [
    | `MatchBranch
        of match_branch_fields
    ]
  and match_branch_fields = 
  {
         
    f_decl: match_val_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_match_branch =
    [
    | `PatternMatchBranch
        of pattern_match_branch_fields
    ]
  and pattern_match_branch_fields = 
  {
         
    f_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_expr_clause =
    [
    | `BlockExprClause
        of block_expr_clause_fields
    ]
  and block_expr_clause_fields = 
  {
         
    f_clause: [
      | `ValDecl
          of val_decl_fields
      | `VarBind
          of var_bind_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_string_line =
    [
    | `BlockStringLine
        of block_string_line_fields
    ]
  and block_string_line_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and class_qualifier =
    [
    | `ClassQualifierAbsent
        of class_qualifier_absent_fields
    | `ClassQualifierPresent
        of class_qualifier_present_fields
    ]

  
   
  and class_qualifier_absent =
    [
    | `ClassQualifierAbsent
        of class_qualifier_absent_fields
    ]
  and class_qualifier_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and class_qualifier_present =
    [
    | `ClassQualifierPresent
        of class_qualifier_present_fields
    ]
  and class_qualifier_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * base_grammar_rule_decl
    * base_val_decl
    * env_spec_decl
    * error_decl
    * generic_decl
    * grammar_decl
    * lexer_decl
    * lexer_family_decl
    * synth_fun_decl
    * synth_param_decl
    * type_decl
    *)
  and decl =
    [
    | `GrammarRuleDecl
        of grammar_rule_decl_fields
    | `SyntheticLexerDecl
        of synthetic_lexer_decl_fields
    | `NodeDecl
        of node_decl_fields
    | `SelfDecl
        of self_decl_fields
    | `BindingValDecl
        of binding_val_decl_fields
    | `EnumLitDecl
        of enum_lit_decl_fields
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    | `FunDecl
        of fun_decl_fields
    | `EnvSpecDecl
        of env_spec_decl_fields
    | `ErrorDecl
        of error_decl_fields
    | `GenericDecl
        of generic_decl_fields
    | `GrammarDecl
        of grammar_decl_fields
    | `LexerDecl
        of lexer_decl_fields
    | `LexerFamilyDecl
        of lexer_family_decl_fields
    | `SynthFunDecl
        of synth_fun_decl_fields
    | `SynthParamDecl
        of synth_param_decl_fields
    | `AnyTypeDecl
        of any_type_decl_fields
    | `EnumClassAltDecl
        of enum_class_alt_decl_fields
    | `FunctionType
        of function_type_fields
    | `GenericParamTypeDecl
        of generic_param_type_decl_fields
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    | `EnumTypeDecl
        of enum_type_decl_fields
    | `StructDecl
        of struct_decl_fields
    | `TraitDecl
        of trait_decl_fields
    ]

  
   
  and base_grammar_rule_decl =
    [
    | `GrammarRuleDecl
        of grammar_rule_decl_fields
    | `SyntheticLexerDecl
        of synthetic_lexer_decl_fields
    ]

  
   
  and grammar_rule_decl =
    [
    | `GrammarRuleDecl
        of grammar_rule_decl_fields
    ]
  and grammar_rule_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_lexer_decl =
    [
    | `SyntheticLexerDecl
        of synthetic_lexer_decl_fields
    ]
  and synthetic_lexer_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * node_decl
    * self_decl
    * user_val_decl
    *)
  and base_val_decl =
    [
    | `NodeDecl
        of node_decl_fields
    | `SelfDecl
        of self_decl_fields
    | `BindingValDecl
        of binding_val_decl_fields
    | `EnumLitDecl
        of enum_lit_decl_fields
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    | `FunDecl
        of fun_decl_fields
    ]

  
   
  and node_decl =
    [
    | `NodeDecl
        of node_decl_fields
    ]
  and node_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and self_decl =
    [
    | `SelfDecl
        of self_decl_fields
    ]
  and self_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * binding_val_decl
    * enum_lit_decl
    * explicitly_typed_decl
    * fun_decl
    *)
  and user_val_decl =
    [
    | `BindingValDecl
        of binding_val_decl_fields
    | `EnumLitDecl
        of enum_lit_decl_fields
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    | `FunDecl
        of fun_decl_fields
    ]

  
   
  and binding_val_decl =
    [
    | `BindingValDecl
        of binding_val_decl_fields
    ]
  and binding_val_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_lit_decl =
    [
    | `EnumLitDecl
        of enum_lit_decl_fields
    ]
  and enum_lit_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * component_decl
    * dyn_var_decl
    * match_val_decl
    * val_decl
    *)
  and explicitly_typed_decl =
    [
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    | `DynVarDecl
        of dyn_var_decl_fields
    | `MatchValDecl
        of match_val_decl_fields
    | `ValDecl
        of val_decl_fields
    ]

  
   
  and component_decl =
    [
    | `FieldDecl
        of field_decl_fields
    | `FunParamDecl
        of fun_param_decl_fields
    | `LambdaParamDecl
        of lambda_param_decl_fields
    ]

  
   
  and field_decl =
    [
    | `FieldDecl
        of field_decl_fields
    ]
  and field_decl_fields = 
  {
         
    f_trait_ref: dot_expr
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and fun_param_decl =
    [
    | `FunParamDecl
        of fun_param_decl_fields
    ]
  and fun_param_decl_fields = 
  {
         
    f_decl_annotations: decl_annotation_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lambda_param_decl =
    [
    | `LambdaParamDecl
        of lambda_param_decl_fields
    ]
  and lambda_param_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and dyn_var_decl =
    [
    | `DynVarDecl
        of dyn_var_decl_fields
    ]
  and dyn_var_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and match_val_decl =
    [
    | `MatchValDecl
        of match_val_decl_fields
    ]
  and match_val_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and val_decl =
    [
    | `ValDecl
        of val_decl_fields
    ]
  and val_decl_fields = 
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and fun_decl =
    [
    | `FunDecl
        of fun_decl_fields
    ]
  and fun_decl_fields = 
  {
         
    f_params: fun_param_decl_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
         
    f_trait_ref: dot_expr
    option
    Lazy.t;
         
    f_body: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and env_spec_decl =
    [
    | `EnvSpecDecl
        of env_spec_decl_fields
    ]
  and env_spec_decl_fields = 
  {
         
    f_actions: call_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and error_decl =
    [
    | `ErrorDecl
        of error_decl_fields
    ]
  and error_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_decl =
    [
    | `GenericDecl
        of generic_decl_fields
    ]
  and generic_decl_fields = 
  {
         
    f_generic_param_decls: generic_param_decl_list
    Lazy.t;
         
    f_decl: [
      | `ClassDecl
          of class_decl_fields
      | `DynVarDecl
          of dyn_var_decl_fields
      | `EnumClassDecl
          of enum_class_decl_fields
      | `EnumTypeDecl
          of enum_type_decl_fields
      | `EnvSpecDecl
          of env_spec_decl_fields
      | `FieldDecl
          of field_decl_fields
      | `FunDecl
          of fun_decl_fields
      | `GenericDecl
          of generic_decl_fields
      | `GrammarDecl
          of grammar_decl_fields
      | `GrammarRuleDecl
          of grammar_rule_decl_fields
      | `LexerDecl
          of lexer_decl_fields
      | `StructDecl
          of struct_decl_fields
      | `TraitDecl
          of trait_decl_fields
      | `ValDecl
          of val_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_decl =
    [
    | `GrammarDecl
        of grammar_decl_fields
    ]
  and grammar_decl_fields = 
  {
         
    f_rules: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_decl =
    [
    | `LexerDecl
        of lexer_decl_fields
    ]
  and lexer_decl_fields = 
  {
         
    f_rules: lkt_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_family_decl =
    [
    | `LexerFamilyDecl
        of lexer_family_decl_fields
    ]
  and lexer_family_decl_fields = 
  {
         
    f_rules: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synth_fun_decl =
    [
    | `SynthFunDecl
        of synth_fun_decl_fields
    ]
  and synth_fun_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and synth_param_decl =
    [
    | `SynthParamDecl
        of synth_param_decl_fields
    ]
  and synth_param_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * any_type_decl
    * enum_class_alt_decl
    * function_type
    * generic_param_type_decl
    * named_type_decl
    *)
  and type_decl =
    [
    | `AnyTypeDecl
        of any_type_decl_fields
    | `EnumClassAltDecl
        of enum_class_alt_decl_fields
    | `FunctionType
        of function_type_fields
    | `GenericParamTypeDecl
        of generic_param_type_decl_fields
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    | `EnumTypeDecl
        of enum_type_decl_fields
    | `StructDecl
        of struct_decl_fields
    | `TraitDecl
        of trait_decl_fields
    ]

  
   
  and any_type_decl =
    [
    | `AnyTypeDecl
        of any_type_decl_fields
    ]
  and any_type_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_alt_decl =
    [
    | `EnumClassAltDecl
        of enum_class_alt_decl_fields
    ]
  and enum_class_alt_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and function_type =
    [
    | `FunctionType
        of function_type_fields
    ]
  and function_type_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_param_type_decl =
    [
    | `GenericParamTypeDecl
        of generic_param_type_decl_fields
    ]
  and generic_param_type_decl_fields = 
  {
         
    f_has_class: class_qualifier
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * basic_class_decl
    * enum_type_decl
    * struct_decl
    * trait_decl
    *)
  and named_type_decl =
    [
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    | `EnumTypeDecl
        of enum_type_decl_fields
    | `StructDecl
        of struct_decl_fields
    | `TraitDecl
        of trait_decl_fields
    ]

  
   
  and basic_class_decl =
    [
    | `ClassDecl
        of class_decl_fields
    | `EnumClassDecl
        of enum_class_decl_fields
    ]

  
   
  and class_decl =
    [
    | `ClassDecl
        of class_decl_fields
    ]
  and class_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_decl =
    [
    | `EnumClassDecl
        of enum_class_decl_fields
    ]
  and enum_class_decl_fields = 
  {
         
    f_branches: enum_class_case_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_type_decl =
    [
    | `EnumTypeDecl
        of enum_type_decl_fields
    ]
  and enum_type_decl_fields = 
  {
         
    f_literals: enum_lit_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and struct_decl =
    [
    | `StructDecl
        of struct_decl_fields
    ]
  and struct_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and trait_decl =
    [
    | `TraitDecl
        of trait_decl_fields
    ]
  and trait_decl_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_annotation =
    [
    | `DeclAnnotation
        of decl_annotation_fields
    ]
  and decl_annotation_fields = 
  {
         
    f_name: id
    Lazy.t;
         
    f_args: decl_annotation_args
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_annotation_args =
    [
    | `DeclAnnotationArgs
        of decl_annotation_args_fields
    ]
  and decl_annotation_args_fields = 
  {
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and dyn_env_wrapper =
    [
    | `DynEnvWrapper
        of dyn_env_wrapper_fields
    ]
  and dyn_env_wrapper_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and elsif_branch =
    [
    | `ElsifBranch
        of elsif_branch_fields
    ]
  and elsif_branch_fields = 
  {
         
    f_cond_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_case =
    [
    | `EnumClassCase
        of enum_class_case_fields
    ]
  and enum_class_case_fields = 
  {
         
    f_decls: enum_class_alt_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and excludes_null =
    [
    | `ExcludesNullAbsent
        of excludes_null_absent_fields
    | `ExcludesNullPresent
        of excludes_null_present_fields
    ]

  
   
  and excludes_null_absent =
    [
    | `ExcludesNullAbsent
        of excludes_null_absent_fields
    ]
  and excludes_null_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and excludes_null_present =
    [
    | `ExcludesNullPresent
        of excludes_null_present_fields
    ]
  and excludes_null_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * any_of
    * array_literal
    * base_call_expr
    * bin_op
    * block_expr
    * cast_expr
    * dot_expr
    * error_on_null
    * generic_instantiation
    * grammar_expr
    * id
    * if_expr
    * isa
    * keep_expr
    * lambda_expr
    * lit
    * logic_assign
    * logic_expr
    * logic_propagate
    * logic_unify
    * match_expr
    * not_expr
    * paren_expr
    * raise_expr
    * subscript_expr
    * try_expr
    * un_op
    *)
  and expr =
    [
    | `AnyOf
        of any_of_fields
    | `ArrayLiteral
        of array_literal_fields
    | `CallExpr
        of call_expr_fields
    | `LogicPredicate
        of logic_predicate_fields
    | `LogicPropagateCall
        of logic_propagate_call_fields
    | `BinOp
        of bin_op_fields
    | `BlockExpr
        of block_expr_fields
    | `CastExpr
        of cast_expr_fields
    | `DotExpr
        of dot_expr_fields
    | `ErrorOnNull
        of error_on_null_fields
    | `GenericInstantiation
        of generic_instantiation_fields
    | `GrammarCut
        of grammar_cut_fields
    | `GrammarDiscard
        of grammar_discard_fields
    | `GrammarDontSkip
        of grammar_dont_skip_fields
    | `GrammarList
        of grammar_list_fields
    | `GrammarNull
        of grammar_null_fields
    | `GrammarOpt
        of grammar_opt_fields
    | `GrammarOptError
        of grammar_opt_error_fields
    | `GrammarOptErrorGroup
        of grammar_opt_error_group_fields
    | `GrammarOptGroup
        of grammar_opt_group_fields
    | `GrammarOrExpr
        of grammar_or_expr_fields
    | `GrammarPick
        of grammar_pick_fields
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    | `GrammarPredicate
        of grammar_predicate_fields
    | `GrammarRuleRef
        of grammar_rule_ref_fields
    | `GrammarSkip
        of grammar_skip_fields
    | `GrammarStopCut
        of grammar_stop_cut_fields
    | `ParseNodeExpr
        of parse_node_expr_fields
    | `TokenLit
        of token_lit_fields
    | `TokenNoCaseLit
        of token_no_case_lit_fields
    | `TokenPatternConcat
        of token_pattern_concat_fields
    | `TokenPatternLit
        of token_pattern_lit_fields
    | `TokenRef
        of token_ref_fields
    | `Id
        of id_fields
    | `DefId
        of def_id_fields
    | `ModuleRefId
        of module_ref_id_fields
    | `RefId
        of ref_id_fields
    | `IfExpr
        of if_expr_fields
    | `Isa
        of isa_fields
    | `KeepExpr
        of keep_expr_fields
    | `LambdaExpr
        of lambda_expr_fields
    | `BigNumLit
        of big_num_lit_fields
    | `CharLit
        of char_lit_fields
    | `NullLit
        of null_lit_fields
    | `NumLit
        of num_lit_fields
    | `BlockStringLit
        of block_string_lit_fields
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    | `LogicAssign
        of logic_assign_fields
    | `LogicExpr
        of logic_expr_fields
    | `LogicPropagate
        of logic_propagate_fields
    | `LogicUnify
        of logic_unify_fields
    | `MatchExpr
        of match_expr_fields
    | `NotExpr
        of not_expr_fields
    | `ParenExpr
        of paren_expr_fields
    | `RaiseExpr
        of raise_expr_fields
    | `SubscriptExpr
        of subscript_expr_fields
    | `TryExpr
        of try_expr_fields
    | `UnOp
        of un_op_fields
    ]

  
   
  and any_of =
    [
    | `AnyOf
        of any_of_fields
    ]
  and any_of_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_values: any_of_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and array_literal =
    [
    | `ArrayLiteral
        of array_literal_fields
    ]
  and array_literal_fields = 
  {
         
    f_exprs: expr_list
    Lazy.t;
         
    f_element_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * call_expr
    * logic_call_expr
    *)
  and base_call_expr =
    [
    | `CallExpr
        of call_expr_fields
    | `LogicPredicate
        of logic_predicate_fields
    | `LogicPropagateCall
        of logic_propagate_call_fields
    ]

  
   
  and call_expr =
    [
    | `CallExpr
        of call_expr_fields
    ]
  and call_expr_fields = 
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_call_expr =
    [
    | `LogicPredicate
        of logic_predicate_fields
    | `LogicPropagateCall
        of logic_propagate_call_fields
    ]

  
   
  and logic_predicate =
    [
    | `LogicPredicate
        of logic_predicate_fields
    ]
  and logic_predicate_fields = 
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_propagate_call =
    [
    | `LogicPropagateCall
        of logic_propagate_call_fields
    ]
  and logic_propagate_call_fields = 
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and bin_op =
    [
    | `BinOp
        of bin_op_fields
    ]
  and bin_op_fields = 
  {
         
    f_left: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_op: op
    Lazy.t;
         
    f_right: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_expr =
    [
    | `BlockExpr
        of block_expr_fields
    ]
  and block_expr_fields = 
  {
         
    f_clauses: lkt_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and cast_expr =
    [
    | `CastExpr
        of cast_expr_fields
    ]
  and cast_expr_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_excludes_null: excludes_null
    Lazy.t;
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and dot_expr =
    [
    | `DotExpr
        of dot_expr_fields
    ]
  and dot_expr_fields = 
  {
         
    f_prefix: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_suffix: ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and error_on_null =
    [
    | `ErrorOnNull
        of error_on_null_fields
    ]
  and error_on_null_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_instantiation =
    [
    | `GenericInstantiation
        of generic_instantiation_fields
    ]
  and generic_instantiation_fields = 
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: type_ref_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * grammar_cut
    * grammar_discard
    * grammar_dont_skip
    * grammar_list
    * grammar_null
    * grammar_opt
    * grammar_opt_error
    * grammar_opt_error_group
    * grammar_opt_group
    * grammar_or_expr
    * grammar_pick
    * grammar_predicate
    * grammar_rule_ref
    * grammar_skip
    * grammar_stop_cut
    * parse_node_expr
    * token_lit
    * token_no_case_lit
    * token_pattern_concat
    * token_pattern_lit
    * token_ref
    *)
  and grammar_expr =
    [
    | `GrammarCut
        of grammar_cut_fields
    | `GrammarDiscard
        of grammar_discard_fields
    | `GrammarDontSkip
        of grammar_dont_skip_fields
    | `GrammarList
        of grammar_list_fields
    | `GrammarNull
        of grammar_null_fields
    | `GrammarOpt
        of grammar_opt_fields
    | `GrammarOptError
        of grammar_opt_error_fields
    | `GrammarOptErrorGroup
        of grammar_opt_error_group_fields
    | `GrammarOptGroup
        of grammar_opt_group_fields
    | `GrammarOrExpr
        of grammar_or_expr_fields
    | `GrammarPick
        of grammar_pick_fields
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    | `GrammarPredicate
        of grammar_predicate_fields
    | `GrammarRuleRef
        of grammar_rule_ref_fields
    | `GrammarSkip
        of grammar_skip_fields
    | `GrammarStopCut
        of grammar_stop_cut_fields
    | `ParseNodeExpr
        of parse_node_expr_fields
    | `TokenLit
        of token_lit_fields
    | `TokenNoCaseLit
        of token_no_case_lit_fields
    | `TokenPatternConcat
        of token_pattern_concat_fields
    | `TokenPatternLit
        of token_pattern_lit_fields
    | `TokenRef
        of token_ref_fields
    ]

  
   
  and grammar_cut =
    [
    | `GrammarCut
        of grammar_cut_fields
    ]
  and grammar_cut_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_discard =
    [
    | `GrammarDiscard
        of grammar_discard_fields
    ]
  and grammar_discard_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_dont_skip =
    [
    | `GrammarDontSkip
        of grammar_dont_skip_fields
    ]
  and grammar_dont_skip_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_dont_skip: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_list =
    [
    | `GrammarList
        of grammar_list_fields
    ]
  and grammar_list_fields = 
  {
         
    f_list_type: type_ref
    Lazy.t;
         
    f_kind: list_kind
    Lazy.t;
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_sep: grammar_list_sep
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_null =
    [
    | `GrammarNull
        of grammar_null_fields
    ]
  and grammar_null_fields = 
  {
         
    f_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_opt =
    [
    | `GrammarOpt
        of grammar_opt_fields
    ]
  and grammar_opt_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_opt_error =
    [
    | `GrammarOptError
        of grammar_opt_error_fields
    ]
  and grammar_opt_error_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_opt_error_group =
    [
    | `GrammarOptErrorGroup
        of grammar_opt_error_group_fields
    ]
  and grammar_opt_error_group_fields = 
  {
         
    f_expr: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_opt_group =
    [
    | `GrammarOptGroup
        of grammar_opt_group_fields
    ]
  and grammar_opt_group_fields = 
  {
         
    f_expr: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_or_expr =
    [
    | `GrammarOrExpr
        of grammar_or_expr_fields
    ]
  and grammar_or_expr_fields = 
  {
         
    f_sub_exprs: grammar_expr_list_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_pick =
    [
    | `GrammarPick
        of grammar_pick_fields
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    ]
  and grammar_pick_fields = 
  {
         
    f_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_implicit_pick =
    [
    | `GrammarImplicitPick
        of grammar_implicit_pick_fields
    ]
  and grammar_implicit_pick_fields = 
  {
         
    f_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_predicate =
    [
    | `GrammarPredicate
        of grammar_predicate_fields
    ]
  and grammar_predicate_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_prop_ref: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_rule_ref =
    [
    | `GrammarRuleRef
        of grammar_rule_ref_fields
    ]
  and grammar_rule_ref_fields = 
  {
         
    f_node_name: ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_skip =
    [
    | `GrammarSkip
        of grammar_skip_fields
    ]
  and grammar_skip_fields = 
  {
         
    f_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_stop_cut =
    [
    | `GrammarStopCut
        of grammar_stop_cut_fields
    ]
  and grammar_stop_cut_fields = 
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and parse_node_expr =
    [
    | `ParseNodeExpr
        of parse_node_expr_fields
    ]
  and parse_node_expr_fields = 
  {
         
    f_node_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
         
    f_sub_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_lit =
    [
    | `TokenLit
        of token_lit_fields
    ]
  and token_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_no_case_lit =
    [
    | `TokenNoCaseLit
        of token_no_case_lit_fields
    ]
  and token_no_case_lit_fields = 
  {
         
    f_lit: token_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_pattern_concat =
    [
    | `TokenPatternConcat
        of token_pattern_concat_fields
    ]
  and token_pattern_concat_fields = 
  {
         
    f_left: [
      | `TokenPatternConcat
          of token_pattern_concat_fields
      | `TokenPatternLit
          of token_pattern_lit_fields
    ]
    Lazy.t;
         
    f_right: token_pattern_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_pattern_lit =
    [
    | `TokenPatternLit
        of token_pattern_lit_fields
    ]
  and token_pattern_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and token_ref =
    [
    | `TokenRef
        of token_ref_fields
    ]
  and token_ref_fields = 
  {
         
    f_token_name: ref_id
    Lazy.t;
         
    f_expr: token_lit
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and id =
    [
    | `Id
        of id_fields
    | `DefId
        of def_id_fields
    | `ModuleRefId
        of module_ref_id_fields
    | `RefId
        of ref_id_fields
    ]
  and id_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and def_id =
    [
    | `DefId
        of def_id_fields
    ]
  and def_id_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and module_ref_id =
    [
    | `ModuleRefId
        of module_ref_id_fields
    ]
  and module_ref_id_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and ref_id =
    [
    | `RefId
        of ref_id_fields
    ]
  and ref_id_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and if_expr =
    [
    | `IfExpr
        of if_expr_fields
    ]
  and if_expr_fields = 
  {
         
    f_cond_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_alternatives: elsif_branch_list
    Lazy.t;
         
    f_else_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and isa =
    [
    | `Isa
        of isa_fields
    ]
  and isa_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and keep_expr =
    [
    | `KeepExpr
        of keep_expr_fields
    ]
  and keep_expr_fields = 
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_keep_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lambda_expr =
    [
    | `LambdaExpr
        of lambda_expr_fields
    ]
  and lambda_expr_fields = 
  {
         
    f_params: lambda_param_decl_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
         
    f_body: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * big_num_lit
    * char_lit
    * null_lit
    * num_lit
    * string_lit
    *)
  and lit =
    [
    | `BigNumLit
        of big_num_lit_fields
    | `CharLit
        of char_lit_fields
    | `NullLit
        of null_lit_fields
    | `NumLit
        of num_lit_fields
    | `BlockStringLit
        of block_string_lit_fields
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    ]

  
   
  and big_num_lit =
    [
    | `BigNumLit
        of big_num_lit_fields
    ]
  and big_num_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and char_lit =
    [
    | `CharLit
        of char_lit_fields
    ]
  and char_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_lit =
    [
    | `NullLit
        of null_lit_fields
    ]
  and null_lit_fields = 
  {
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and num_lit =
    [
    | `NumLit
        of num_lit_fields
    ]
  and num_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * block_string_lit
    * single_line_string_lit
    *)
  and string_lit =
    [
    | `BlockStringLit
        of block_string_lit_fields
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    ]

  
   
  and block_string_lit =
    [
    | `BlockStringLit
        of block_string_lit_fields
    ]
  and block_string_lit_fields = 
  {
         
    f_lines: block_string_line_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and single_line_string_lit =
    [
    | `SingleLineStringLit
        of single_line_string_lit_fields
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    ]
  and single_line_string_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_single_line_string_lit =
    [
    | `PatternSingleLineStringLit
        of pattern_single_line_string_lit_fields
    ]
  and pattern_single_line_string_lit_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_assign =
    [
    | `LogicAssign
        of logic_assign_fields
    ]
  and logic_assign_fields = 
  {
         
    f_dest_var: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_value: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_expr =
    [
    | `LogicExpr
        of logic_expr_fields
    ]
  and logic_expr_fields = 
  {
         
    f_expr: [
      | `CallExpr
          of call_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_propagate =
    [
    | `LogicPropagate
        of logic_propagate_fields
    ]
  and logic_propagate_fields = 
  {
         
    f_dest_var: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_call: logic_propagate_call
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and logic_unify =
    [
    | `LogicUnify
        of logic_unify_fields
    ]
  and logic_unify_fields = 
  {
         
    f_lhs: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_rhs: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and match_expr =
    [
    | `MatchExpr
        of match_expr_fields
    ]
  and match_expr_fields = 
  {
         
    f_match_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_branches: base_match_branch_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and not_expr =
    [
    | `NotExpr
        of not_expr_fields
    ]
  and not_expr_fields = 
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and paren_expr =
    [
    | `ParenExpr
        of paren_expr_fields
    ]
  and paren_expr_fields = 
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and raise_expr =
    [
    | `RaiseExpr
        of raise_expr_fields
    ]
  and raise_expr_fields = 
  {
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
         
    f_except_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and subscript_expr =
    [
    | `SubscriptExpr
        of subscript_expr_fields
    ]
  and subscript_expr_fields = 
  {
         
    f_prefix: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_index: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and try_expr =
    [
    | `TryExpr
        of try_expr_fields
    ]
  and try_expr_fields = 
  {
         
    f_try_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_or_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and un_op =
    [
    | `UnOp
        of un_op_fields
    ]
  and un_op_fields = 
  {
         
    f_op: [
      | `OpMinus
          of op_minus_fields
      | `OpPlus
          of op_plus_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and full_decl =
    [
    | `FullDecl
        of full_decl_fields
    ]
  and full_decl_fields = 
  {
         
    f_doc: string_lit
    option
    Lazy.t;
         
    f_decl_annotations: decl_annotation_list
    Lazy.t;
         
    f_decl: [
      | `ClassDecl
          of class_decl_fields
      | `DynVarDecl
          of dyn_var_decl_fields
      | `EnumClassDecl
          of enum_class_decl_fields
      | `EnumTypeDecl
          of enum_type_decl_fields
      | `EnvSpecDecl
          of env_spec_decl_fields
      | `FieldDecl
          of field_decl_fields
      | `FunDecl
          of fun_decl_fields
      | `GenericDecl
          of generic_decl_fields
      | `GenericParamTypeDecl
          of generic_param_type_decl_fields
      | `GrammarDecl
          of grammar_decl_fields
      | `GrammarRuleDecl
          of grammar_rule_decl_fields
      | `LexerDecl
          of lexer_decl_fields
      | `LexerFamilyDecl
          of lexer_family_decl_fields
      | `StructDecl
          of struct_decl_fields
      | `TraitDecl
          of trait_decl_fields
      | `ValDecl
          of val_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_list_sep =
    [
    | `GrammarListSep
        of grammar_list_sep_fields
    ]
  and grammar_list_sep_fields = 
  {
         
    f_token: grammar_expr
    Lazy.t;
         
    f_extra: id
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and import =
    [
    | `Import
        of import_fields
    ]
  and import_fields = 
  {
         
    f_name: module_ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and langkit_root =
    [
    | `LangkitRoot
        of langkit_root_fields
    ]
  and langkit_root_fields = 
  {
         
    f_imports: import_list
    Lazy.t;
         
    f_decls: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_case_rule =
    [
    | `LexerCaseRule
        of lexer_case_rule_fields
    ]
  and lexer_case_rule_fields = 
  {
         
    f_expr: [
      | `GrammarCut
          of grammar_cut_fields
      | `GrammarDiscard
          of grammar_discard_fields
      | `GrammarImplicitPick
          of grammar_implicit_pick_fields
      | `GrammarList
          of grammar_list_fields
      | `GrammarNull
          of grammar_null_fields
      | `GrammarOpt
          of grammar_opt_fields
      | `GrammarOptError
          of grammar_opt_error_fields
      | `GrammarOptErrorGroup
          of grammar_opt_error_group_fields
      | `GrammarOptGroup
          of grammar_opt_group_fields
      | `GrammarOrExpr
          of grammar_or_expr_fields
      | `GrammarPick
          of grammar_pick_fields
      | `GrammarRuleRef
          of grammar_rule_ref_fields
      | `GrammarSkip
          of grammar_skip_fields
      | `GrammarStopCut
          of grammar_stop_cut_fields
      | `ParseNodeExpr
          of parse_node_expr_fields
      | `TokenLit
          of token_lit_fields
      | `TokenNoCaseLit
          of token_no_case_lit_fields
      | `TokenPatternConcat
          of token_pattern_concat_fields
      | `TokenPatternLit
          of token_pattern_lit_fields
      | `TokenRef
          of token_ref_fields
    ]
    Lazy.t;
         
    f_alts: base_lexer_case_rule_alt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lexer_case_rule_send =
    [
    | `LexerCaseRuleSend
        of lexer_case_rule_send_fields
    ]
  and lexer_case_rule_send_fields = 
  {
         
    f_sent: ref_id
    Lazy.t;
         
    f_match_size: num_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and list_kind =
    [
    | `ListKindOne
        of list_kind_one_fields
    | `ListKindZero
        of list_kind_zero_fields
    ]

  
   
  and list_kind_one =
    [
    | `ListKindOne
        of list_kind_one_fields
    ]
  and list_kind_one_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and list_kind_zero =
    [
    | `ListKindZero
        of list_kind_zero_fields
    ]
  and list_kind_zero_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * argument_list
    * base_lexer_case_rule_alt_list
    * base_match_branch_list
    * block_string_line_list
    * call_expr_list
    * decl_annotation_list
    * elsif_branch_list
    * enum_class_alt_decl_list
    * enum_class_case_list
    * enum_lit_decl_list
    * expr_list
    * full_decl_list
    * fun_param_decl_list
    * grammar_expr_list
    * grammar_expr_list_list
    * import_list
    * lambda_param_decl_list
    * lkt_node_list
    * pattern_detail_list
    * pattern_list
    * ref_id_list
    * type_ref_list
    *)
  and lkt_node_base_list =
    [
    | `ArgumentList
        of argument_list_fields
    | `BaseLexerCaseRuleAltList
        of base_lexer_case_rule_alt_list_fields
    | `BaseMatchBranchList
        of base_match_branch_list_fields
    | `BlockStringLineList
        of block_string_line_list_fields
    | `CallExprList
        of call_expr_list_fields
    | `DeclAnnotationList
        of decl_annotation_list_fields
    | `ElsifBranchList
        of elsif_branch_list_fields
    | `EnumClassAltDeclList
        of enum_class_alt_decl_list_fields
    | `EnumClassCaseList
        of enum_class_case_list_fields
    | `EnumLitDeclList
        of enum_lit_decl_list_fields
    | `ExprList
        of expr_list_fields
    | `AnyOfList
        of any_of_list_fields
    | `FullDeclList
        of full_decl_list_fields
    | `DeclBlock
        of decl_block_fields
    | `GenericParamDeclList
        of generic_param_decl_list_fields
    | `FunParamDeclList
        of fun_param_decl_list_fields
    | `GrammarExprList
        of grammar_expr_list_fields
    | `GrammarExprListList
        of grammar_expr_list_list_fields
    | `ImportList
        of import_list_fields
    | `LambdaParamDeclList
        of lambda_param_decl_list_fields
    | `LktNodeList
        of lkt_node_list_fields
    | `PatternDetailList
        of pattern_detail_list_fields
    | `PatternList
        of pattern_list_fields
    | `RefIdList
        of ref_id_list_fields
    | `TypeRefList
        of type_ref_list_fields
    | `SyntheticTypeRefList
        of synthetic_type_ref_list_fields
    ]

  
   
  and argument_list =
    [
    | `ArgumentList
        of argument_list_fields
    ]
  and argument_list_fields = 
  {
    list : argument list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_lexer_case_rule_alt_list =
    [
    | `BaseLexerCaseRuleAltList
        of base_lexer_case_rule_alt_list_fields
    ]
  and base_lexer_case_rule_alt_list_fields = 
  {
    list : base_lexer_case_rule_alt list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and base_match_branch_list =
    [
    | `BaseMatchBranchList
        of base_match_branch_list_fields
    ]
  and base_match_branch_list_fields = 
  {
    list : base_match_branch list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and block_string_line_list =
    [
    | `BlockStringLineList
        of block_string_line_list_fields
    ]
  and block_string_line_list_fields = 
  {
    list : block_string_line list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and call_expr_list =
    [
    | `CallExprList
        of call_expr_list_fields
    ]
  and call_expr_list_fields = 
  {
    list : call_expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_annotation_list =
    [
    | `DeclAnnotationList
        of decl_annotation_list_fields
    ]
  and decl_annotation_list_fields = 
  {
    list : decl_annotation list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and elsif_branch_list =
    [
    | `ElsifBranchList
        of elsif_branch_list_fields
    ]
  and elsif_branch_list_fields = 
  {
    list : elsif_branch list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_alt_decl_list =
    [
    | `EnumClassAltDeclList
        of enum_class_alt_decl_list_fields
    ]
  and enum_class_alt_decl_list_fields = 
  {
    list : enum_class_alt_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_class_case_list =
    [
    | `EnumClassCaseList
        of enum_class_case_list_fields
    ]
  and enum_class_case_list_fields = 
  {
    list : enum_class_case list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and enum_lit_decl_list =
    [
    | `EnumLitDeclList
        of enum_lit_decl_list_fields
    ]
  and enum_lit_decl_list_fields = 
  {
    list : enum_lit_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and expr_list =
    [
    | `ExprList
        of expr_list_fields
    | `AnyOfList
        of any_of_list_fields
    ]
  and expr_list_fields = 
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and any_of_list =
    [
    | `AnyOfList
        of any_of_list_fields
    ]
  and any_of_list_fields = 
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and full_decl_list =
    [
    | `FullDeclList
        of full_decl_list_fields
    | `DeclBlock
        of decl_block_fields
    | `GenericParamDeclList
        of generic_param_decl_list_fields
    ]
  and full_decl_list_fields = 
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and decl_block =
    [
    | `DeclBlock
        of decl_block_fields
    ]
  and decl_block_fields = 
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_param_decl_list =
    [
    | `GenericParamDeclList
        of generic_param_decl_list_fields
    ]
  and generic_param_decl_list_fields = 
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and fun_param_decl_list =
    [
    | `FunParamDeclList
        of fun_param_decl_list_fields
    ]
  and fun_param_decl_list_fields = 
  {
    list : fun_param_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_expr_list =
    [
    | `GrammarExprList
        of grammar_expr_list_fields
    ]
  and grammar_expr_list_fields = 
  {
    list : grammar_expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and grammar_expr_list_list =
    [
    | `GrammarExprListList
        of grammar_expr_list_list_fields
    ]
  and grammar_expr_list_list_fields = 
  {
    list : grammar_expr_list list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and import_list =
    [
    | `ImportList
        of import_list_fields
    ]
  and import_list_fields = 
  {
    list : import list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lambda_param_decl_list =
    [
    | `LambdaParamDeclList
        of lambda_param_decl_list_fields
    ]
  and lambda_param_decl_list_fields = 
  {
    list : lambda_param_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and lkt_node_list =
    [
    | `LktNodeList
        of lkt_node_list_fields
    ]
  and lkt_node_list_fields = 
  {
    list : lkt_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_detail_list =
    [
    | `PatternDetailList
        of pattern_detail_list_fields
    ]
  and pattern_detail_list_fields = 
  {
    list : pattern_detail list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_list =
    [
    | `PatternList
        of pattern_list_fields
    ]
  and pattern_list_fields = 
  {
    list : pattern list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and ref_id_list =
    [
    | `RefIdList
        of ref_id_list_fields
    ]
  and ref_id_list_fields = 
  {
    list : ref_id list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and type_ref_list =
    [
    | `TypeRefList
        of type_ref_list_fields
    | `SyntheticTypeRefList
        of synthetic_type_ref_list_fields
    ]
  and type_ref_list_fields = 
  {
    list : type_ref list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and synthetic_type_ref_list =
    [
    | `SyntheticTypeRefList
        of synthetic_type_ref_list_fields
    ]
  and synthetic_type_ref_list_fields = 
  {
    list : type_ref list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_cond_qualifier =
    [
    | `NullCondQualifierAbsent
        of null_cond_qualifier_absent_fields
    | `NullCondQualifierPresent
        of null_cond_qualifier_present_fields
    ]

  
   
  and null_cond_qualifier_absent =
    [
    | `NullCondQualifierAbsent
        of null_cond_qualifier_absent_fields
    ]
  and null_cond_qualifier_absent_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_cond_qualifier_present =
    [
    | `NullCondQualifierPresent
        of null_cond_qualifier_present_fields
    ]
  and null_cond_qualifier_present_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op =
    [
    | `OpAmp
        of op_amp_fields
    | `OpAnd
        of op_and_fields
    | `OpDiv
        of op_div_fields
    | `OpEq
        of op_eq_fields
    | `OpGt
        of op_gt_fields
    | `OpGte
        of op_gte_fields
    | `OpLogicAnd
        of op_logic_and_fields
    | `OpLogicOr
        of op_logic_or_fields
    | `OpLt
        of op_lt_fields
    | `OpLte
        of op_lte_fields
    | `OpMinus
        of op_minus_fields
    | `OpMult
        of op_mult_fields
    | `OpNe
        of op_ne_fields
    | `OpOr
        of op_or_fields
    | `OpOrInt
        of op_or_int_fields
    | `OpPlus
        of op_plus_fields
    ]

  
   
  and op_amp =
    [
    | `OpAmp
        of op_amp_fields
    ]
  and op_amp_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_and =
    [
    | `OpAnd
        of op_and_fields
    ]
  and op_and_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_div =
    [
    | `OpDiv
        of op_div_fields
    ]
  and op_div_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_eq =
    [
    | `OpEq
        of op_eq_fields
    ]
  and op_eq_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_gt =
    [
    | `OpGt
        of op_gt_fields
    ]
  and op_gt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_gte =
    [
    | `OpGte
        of op_gte_fields
    ]
  and op_gte_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_logic_and =
    [
    | `OpLogicAnd
        of op_logic_and_fields
    ]
  and op_logic_and_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_logic_or =
    [
    | `OpLogicOr
        of op_logic_or_fields
    ]
  and op_logic_or_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_lt =
    [
    | `OpLt
        of op_lt_fields
    ]
  and op_lt_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_lte =
    [
    | `OpLte
        of op_lte_fields
    ]
  and op_lte_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_minus =
    [
    | `OpMinus
        of op_minus_fields
    ]
  and op_minus_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_mult =
    [
    | `OpMult
        of op_mult_fields
    ]
  and op_mult_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_ne =
    [
    | `OpNe
        of op_ne_fields
    ]
  and op_ne_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_or =
    [
    | `OpOr
        of op_or_fields
    ]
  and op_or_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_or_int =
    [
    | `OpOrInt
        of op_or_int_fields
    ]
  and op_or_int_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and op_plus =
    [
    | `OpPlus
        of op_plus_fields
    ]
  and op_plus_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  (**
    * any_type_pattern
    * binding_pattern
    * bool_pattern
    * ellipsis_pattern
    * extended_pattern
    * filtered_pattern
    * integer_pattern
    * list_pattern
    * not_pattern
    * null_pattern
    * or_pattern
    * paren_pattern
    * regex_pattern
    * tuple_pattern
    * type_pattern
    *)
  and pattern =
    [
    | `AnyTypePattern
        of any_type_pattern_fields
    | `BindingPattern
        of binding_pattern_fields
    | `BoolPatternFalse
        of bool_pattern_false_fields
    | `BoolPatternTrue
        of bool_pattern_true_fields
    | `EllipsisPattern
        of ellipsis_pattern_fields
    | `ExtendedPattern
        of extended_pattern_fields
    | `FilteredPattern
        of filtered_pattern_fields
    | `IntegerPattern
        of integer_pattern_fields
    | `ListPattern
        of list_pattern_fields
    | `NotPattern
        of not_pattern_fields
    | `NullPattern
        of null_pattern_fields
    | `OrPattern
        of or_pattern_fields
    | `ParenPattern
        of paren_pattern_fields
    | `RegexPattern
        of regex_pattern_fields
    | `TuplePattern
        of tuple_pattern_fields
    | `TypePattern
        of type_pattern_fields
    ]

  
   
  and any_type_pattern =
    [
    | `AnyTypePattern
        of any_type_pattern_fields
    ]
  and any_type_pattern_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and binding_pattern =
    [
    | `BindingPattern
        of binding_pattern_fields
    ]
  and binding_pattern_fields = 
  {
         
    f_decl: binding_val_decl
    Lazy.t;
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and bool_pattern =
    [
    | `BoolPatternFalse
        of bool_pattern_false_fields
    | `BoolPatternTrue
        of bool_pattern_true_fields
    ]

  
   
  and bool_pattern_false =
    [
    | `BoolPatternFalse
        of bool_pattern_false_fields
    ]
  and bool_pattern_false_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and bool_pattern_true =
    [
    | `BoolPatternTrue
        of bool_pattern_true_fields
    ]
  and bool_pattern_true_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and ellipsis_pattern =
    [
    | `EllipsisPattern
        of ellipsis_pattern_fields
    ]
  and ellipsis_pattern_fields = 
  {
         
    f_binding: id
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and extended_pattern =
    [
    | `ExtendedPattern
        of extended_pattern_fields
    ]
  and extended_pattern_fields = 
  {
         
    f_sub_pattern: [
      | `AnyTypePattern
          of any_type_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_details: pattern_detail_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and filtered_pattern =
    [
    | `FilteredPattern
        of filtered_pattern_fields
    ]
  and filtered_pattern_fields = 
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_predicate: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and integer_pattern =
    [
    | `IntegerPattern
        of integer_pattern_fields
    ]
  and integer_pattern_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and list_pattern =
    [
    | `ListPattern
        of list_pattern_fields
    ]
  and list_pattern_fields = 
  {
         
    f_sub_patterns: pattern_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and not_pattern =
    [
    | `NotPattern
        of not_pattern_fields
    ]
  and not_pattern_fields = 
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and null_pattern =
    [
    | `NullPattern
        of null_pattern_fields
    ]
  and null_pattern_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and or_pattern =
    [
    | `OrPattern
        of or_pattern_fields
    ]
  and or_pattern_fields = 
  {
         
    f_left_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_right_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and paren_pattern =
    [
    | `ParenPattern
        of paren_pattern_fields
    ]
  and paren_pattern_fields = 
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and regex_pattern =
    [
    | `RegexPattern
        of regex_pattern_fields
    ]
  and regex_pattern_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and tuple_pattern =
    [
    | `TuplePattern
        of tuple_pattern_fields
    ]
  and tuple_pattern_fields = 
  {
         
    f_sub_patterns: pattern_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and type_pattern =
    [
    | `TypePattern
        of type_pattern_fields
    ]
  and type_pattern_fields = 
  {
         
    f_type_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and pattern_detail =
    [
    | `FieldPatternDetail
        of field_pattern_detail_fields
    | `PropertyPatternDetail
        of property_pattern_detail_fields
    | `SelectorPatternDetail
        of selector_pattern_detail_fields
    ]

  
   
  and field_pattern_detail =
    [
    | `FieldPatternDetail
        of field_pattern_detail_fields
    ]
  and field_pattern_detail_fields = 
  {
         
    f_id: id
    Lazy.t;
         
    f_expected_value: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and property_pattern_detail =
    [
    | `PropertyPatternDetail
        of property_pattern_detail_fields
    ]
  and property_pattern_detail_fields = 
  {
         
    f_call: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_expected_value: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and selector_pattern_detail =
    [
    | `SelectorPatternDetail
        of selector_pattern_detail_fields
    ]
  and selector_pattern_detail_fields = 
  {
         
    f_call: selector_call
    Lazy.t;
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and selector_call =
    [
    | `SelectorCall
        of selector_call_fields
    ]
  and selector_call_fields = 
  {
         
    f_quantifier: id
    Lazy.t;
         
    f_binding: id
    option
    Lazy.t;
         
    f_selector_call: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and type_ref =
    [
    | `DefaultListTypeRef
        of default_list_type_ref_fields
    | `FunctionTypeRef
        of function_type_ref_fields
    | `GenericTypeRef
        of generic_type_ref_fields
    | `SimpleTypeRef
        of simple_type_ref_fields
    ]

  
   
  and default_list_type_ref =
    [
    | `DefaultListTypeRef
        of default_list_type_ref_fields
    ]
  and default_list_type_ref_fields = 
  {
    c_value : entity;
    context : analysis_context
  }


  
   
  and function_type_ref =
    [
    | `FunctionTypeRef
        of function_type_ref_fields
    ]
  and function_type_ref_fields = 
  {
         
    f_param_types: type_ref_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and generic_type_ref =
    [
    | `GenericTypeRef
        of generic_type_ref_fields
    ]
  and generic_type_ref_fields = 
  {
         
    f_type_name: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
         
    f_args: type_ref_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and simple_type_ref =
    [
    | `SimpleTypeRef
        of simple_type_ref_fields
    ]
  and simple_type_ref_fields = 
  {
         
    f_type_name: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  
   
  and var_bind =
    [
    | `VarBind
        of var_bind_fields
    ]
  and var_bind_fields = 
  {
         
    f_name: ref_id
    Lazy.t;
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }






let rec unwrap_lkt_node value =
  (* This is the unique unwrap function that can be called for any node. *)
  match (value :> lkt_node) with
  | `Argument fields -> fields.c_value
  | `LexerCaseRuleCondAlt fields -> fields.c_value
  | `LexerCaseRuleDefaultAlt fields -> fields.c_value
  | `MatchBranch fields -> fields.c_value
  | `PatternMatchBranch fields -> fields.c_value
  | `BlockExprClause fields -> fields.c_value
  | `BlockStringLine fields -> fields.c_value
  | `ClassQualifierAbsent fields -> fields.c_value
  | `ClassQualifierPresent fields -> fields.c_value
  | `GrammarRuleDecl fields -> fields.c_value
  | `SyntheticLexerDecl fields -> fields.c_value
  | `NodeDecl fields -> fields.c_value
  | `SelfDecl fields -> fields.c_value
  | `BindingValDecl fields -> fields.c_value
  | `EnumLitDecl fields -> fields.c_value
  | `FieldDecl fields -> fields.c_value
  | `FunParamDecl fields -> fields.c_value
  | `LambdaParamDecl fields -> fields.c_value
  | `DynVarDecl fields -> fields.c_value
  | `MatchValDecl fields -> fields.c_value
  | `ValDecl fields -> fields.c_value
  | `FunDecl fields -> fields.c_value
  | `EnvSpecDecl fields -> fields.c_value
  | `ErrorDecl fields -> fields.c_value
  | `GenericDecl fields -> fields.c_value
  | `GrammarDecl fields -> fields.c_value
  | `LexerDecl fields -> fields.c_value
  | `LexerFamilyDecl fields -> fields.c_value
  | `SynthFunDecl fields -> fields.c_value
  | `SynthParamDecl fields -> fields.c_value
  | `AnyTypeDecl fields -> fields.c_value
  | `EnumClassAltDecl fields -> fields.c_value
  | `FunctionType fields -> fields.c_value
  | `GenericParamTypeDecl fields -> fields.c_value
  | `ClassDecl fields -> fields.c_value
  | `EnumClassDecl fields -> fields.c_value
  | `EnumTypeDecl fields -> fields.c_value
  | `StructDecl fields -> fields.c_value
  | `TraitDecl fields -> fields.c_value
  | `DeclAnnotation fields -> fields.c_value
  | `DeclAnnotationArgs fields -> fields.c_value
  | `DynEnvWrapper fields -> fields.c_value
  | `ElsifBranch fields -> fields.c_value
  | `EnumClassCase fields -> fields.c_value
  | `ExcludesNullAbsent fields -> fields.c_value
  | `ExcludesNullPresent fields -> fields.c_value
  | `AnyOf fields -> fields.c_value
  | `ArrayLiteral fields -> fields.c_value
  | `CallExpr fields -> fields.c_value
  | `LogicPredicate fields -> fields.c_value
  | `LogicPropagateCall fields -> fields.c_value
  | `BinOp fields -> fields.c_value
  | `BlockExpr fields -> fields.c_value
  | `CastExpr fields -> fields.c_value
  | `DotExpr fields -> fields.c_value
  | `ErrorOnNull fields -> fields.c_value
  | `GenericInstantiation fields -> fields.c_value
  | `GrammarCut fields -> fields.c_value
  | `GrammarDiscard fields -> fields.c_value
  | `GrammarDontSkip fields -> fields.c_value
  | `GrammarList fields -> fields.c_value
  | `GrammarNull fields -> fields.c_value
  | `GrammarOpt fields -> fields.c_value
  | `GrammarOptError fields -> fields.c_value
  | `GrammarOptErrorGroup fields -> fields.c_value
  | `GrammarOptGroup fields -> fields.c_value
  | `GrammarOrExpr fields -> fields.c_value
  | `GrammarPick fields -> fields.c_value
  | `GrammarImplicitPick fields -> fields.c_value
  | `GrammarPredicate fields -> fields.c_value
  | `GrammarRuleRef fields -> fields.c_value
  | `GrammarSkip fields -> fields.c_value
  | `GrammarStopCut fields -> fields.c_value
  | `ParseNodeExpr fields -> fields.c_value
  | `TokenLit fields -> fields.c_value
  | `TokenNoCaseLit fields -> fields.c_value
  | `TokenPatternConcat fields -> fields.c_value
  | `TokenPatternLit fields -> fields.c_value
  | `TokenRef fields -> fields.c_value
  | `Id fields -> fields.c_value
  | `DefId fields -> fields.c_value
  | `ModuleRefId fields -> fields.c_value
  | `RefId fields -> fields.c_value
  | `IfExpr fields -> fields.c_value
  | `Isa fields -> fields.c_value
  | `KeepExpr fields -> fields.c_value
  | `LambdaExpr fields -> fields.c_value
  | `BigNumLit fields -> fields.c_value
  | `CharLit fields -> fields.c_value
  | `NullLit fields -> fields.c_value
  | `NumLit fields -> fields.c_value
  | `BlockStringLit fields -> fields.c_value
  | `SingleLineStringLit fields -> fields.c_value
  | `PatternSingleLineStringLit fields -> fields.c_value
  | `LogicAssign fields -> fields.c_value
  | `LogicExpr fields -> fields.c_value
  | `LogicPropagate fields -> fields.c_value
  | `LogicUnify fields -> fields.c_value
  | `MatchExpr fields -> fields.c_value
  | `NotExpr fields -> fields.c_value
  | `ParenExpr fields -> fields.c_value
  | `RaiseExpr fields -> fields.c_value
  | `SubscriptExpr fields -> fields.c_value
  | `TryExpr fields -> fields.c_value
  | `UnOp fields -> fields.c_value
  | `FullDecl fields -> fields.c_value
  | `GrammarListSep fields -> fields.c_value
  | `Import fields -> fields.c_value
  | `LangkitRoot fields -> fields.c_value
  | `LexerCaseRule fields -> fields.c_value
  | `LexerCaseRuleSend fields -> fields.c_value
  | `ListKindOne fields -> fields.c_value
  | `ListKindZero fields -> fields.c_value
  | `ArgumentList fields -> fields.c_value
  | `BaseLexerCaseRuleAltList fields -> fields.c_value
  | `BaseMatchBranchList fields -> fields.c_value
  | `BlockStringLineList fields -> fields.c_value
  | `CallExprList fields -> fields.c_value
  | `DeclAnnotationList fields -> fields.c_value
  | `ElsifBranchList fields -> fields.c_value
  | `EnumClassAltDeclList fields -> fields.c_value
  | `EnumClassCaseList fields -> fields.c_value
  | `EnumLitDeclList fields -> fields.c_value
  | `ExprList fields -> fields.c_value
  | `AnyOfList fields -> fields.c_value
  | `FullDeclList fields -> fields.c_value
  | `DeclBlock fields -> fields.c_value
  | `GenericParamDeclList fields -> fields.c_value
  | `FunParamDeclList fields -> fields.c_value
  | `GrammarExprList fields -> fields.c_value
  | `GrammarExprListList fields -> fields.c_value
  | `ImportList fields -> fields.c_value
  | `LambdaParamDeclList fields -> fields.c_value
  | `LktNodeList fields -> fields.c_value
  | `PatternDetailList fields -> fields.c_value
  | `PatternList fields -> fields.c_value
  | `RefIdList fields -> fields.c_value
  | `TypeRefList fields -> fields.c_value
  | `SyntheticTypeRefList fields -> fields.c_value
  | `NullCondQualifierAbsent fields -> fields.c_value
  | `NullCondQualifierPresent fields -> fields.c_value
  | `OpAmp fields -> fields.c_value
  | `OpAnd fields -> fields.c_value
  | `OpDiv fields -> fields.c_value
  | `OpEq fields -> fields.c_value
  | `OpGt fields -> fields.c_value
  | `OpGte fields -> fields.c_value
  | `OpLogicAnd fields -> fields.c_value
  | `OpLogicOr fields -> fields.c_value
  | `OpLt fields -> fields.c_value
  | `OpLte fields -> fields.c_value
  | `OpMinus fields -> fields.c_value
  | `OpMult fields -> fields.c_value
  | `OpNe fields -> fields.c_value
  | `OpOr fields -> fields.c_value
  | `OpOrInt fields -> fields.c_value
  | `OpPlus fields -> fields.c_value
  | `AnyTypePattern fields -> fields.c_value
  | `BindingPattern fields -> fields.c_value
  | `BoolPatternFalse fields -> fields.c_value
  | `BoolPatternTrue fields -> fields.c_value
  | `EllipsisPattern fields -> fields.c_value
  | `ExtendedPattern fields -> fields.c_value
  | `FilteredPattern fields -> fields.c_value
  | `IntegerPattern fields -> fields.c_value
  | `ListPattern fields -> fields.c_value
  | `NotPattern fields -> fields.c_value
  | `NullPattern fields -> fields.c_value
  | `OrPattern fields -> fields.c_value
  | `ParenPattern fields -> fields.c_value
  | `RegexPattern fields -> fields.c_value
  | `TuplePattern fields -> fields.c_value
  | `TypePattern fields -> fields.c_value
  | `FieldPatternDetail fields -> fields.c_value
  | `PropertyPatternDetail fields -> fields.c_value
  | `SelectorPatternDetail fields -> fields.c_value
  | `SelectorCall fields -> fields.c_value
  | `DefaultListTypeRef fields -> fields.c_value
  | `FunctionTypeRef fields -> fields.c_value
  | `GenericTypeRef fields -> fields.c_value
  | `SimpleTypeRef fields -> fields.c_value
  | `VarBind fields -> fields.c_value


   

  and unwrap_entity_info value =
    let c_value = make EntityInfoStruct.c_type in
    let field_c_value =
      value.rebindings
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      EntityInfoStruct.rebindings
      field_c_value;
    let field_c_value =
      value.from_rebound
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      EntityInfoStruct.from_rebound
      field_c_value;
    c_value




   




and unwrap_analysis_unit
  (unit : analysis_unit) = unit.c_value

let rec wrap_lkt_node context c_value =
  (* Top level wrap function that dispatch to wrap function of concrete types
     depending on the node kind *)
  if is_null (getf c_value EntityStruct.node) then
    raise (SyntaxError "null node")
  else
    let kind = CFunctions.node_kind (addr c_value) in
    match kind with
    | 1 ->
        (wrap_argument context (c_value)
         :> lkt_node)
    | 2 ->
        (wrap_lexer_case_rule_cond_alt context (c_value)
         :> lkt_node)
    | 3 ->
        (wrap_lexer_case_rule_default_alt context (c_value)
         :> lkt_node)
    | 4 ->
        (wrap_match_branch context (c_value)
         :> lkt_node)
    | 5 ->
        (wrap_pattern_match_branch context (c_value)
         :> lkt_node)
    | 6 ->
        (wrap_block_expr_clause context (c_value)
         :> lkt_node)
    | 7 ->
        (wrap_block_string_line context (c_value)
         :> lkt_node)
    | 8 ->
        (wrap_class_qualifier_absent context (c_value)
         :> lkt_node)
    | 9 ->
        (wrap_class_qualifier_present context (c_value)
         :> lkt_node)
    | 10 ->
        (wrap_grammar_rule_decl context (c_value)
         :> lkt_node)
    | 11 ->
        (wrap_synthetic_lexer_decl context (c_value)
         :> lkt_node)
    | 12 ->
        (wrap_node_decl context (c_value)
         :> lkt_node)
    | 13 ->
        (wrap_self_decl context (c_value)
         :> lkt_node)
    | 14 ->
        (wrap_binding_val_decl context (c_value)
         :> lkt_node)
    | 15 ->
        (wrap_enum_lit_decl context (c_value)
         :> lkt_node)
    | 16 ->
        (wrap_field_decl context (c_value)
         :> lkt_node)
    | 17 ->
        (wrap_fun_param_decl context (c_value)
         :> lkt_node)
    | 18 ->
        (wrap_lambda_param_decl context (c_value)
         :> lkt_node)
    | 19 ->
        (wrap_dyn_var_decl context (c_value)
         :> lkt_node)
    | 20 ->
        (wrap_match_val_decl context (c_value)
         :> lkt_node)
    | 21 ->
        (wrap_val_decl context (c_value)
         :> lkt_node)
    | 22 ->
        (wrap_fun_decl context (c_value)
         :> lkt_node)
    | 23 ->
        (wrap_env_spec_decl context (c_value)
         :> lkt_node)
    | 24 ->
        (wrap_error_decl context (c_value)
         :> lkt_node)
    | 25 ->
        (wrap_generic_decl context (c_value)
         :> lkt_node)
    | 26 ->
        (wrap_grammar_decl context (c_value)
         :> lkt_node)
    | 27 ->
        (wrap_lexer_decl context (c_value)
         :> lkt_node)
    | 28 ->
        (wrap_lexer_family_decl context (c_value)
         :> lkt_node)
    | 29 ->
        (wrap_synth_fun_decl context (c_value)
         :> lkt_node)
    | 30 ->
        (wrap_synth_param_decl context (c_value)
         :> lkt_node)
    | 31 ->
        (wrap_any_type_decl context (c_value)
         :> lkt_node)
    | 32 ->
        (wrap_enum_class_alt_decl context (c_value)
         :> lkt_node)
    | 33 ->
        (wrap_function_type context (c_value)
         :> lkt_node)
    | 34 ->
        (wrap_generic_param_type_decl context (c_value)
         :> lkt_node)
    | 35 ->
        (wrap_class_decl context (c_value)
         :> lkt_node)
    | 36 ->
        (wrap_enum_class_decl context (c_value)
         :> lkt_node)
    | 37 ->
        (wrap_enum_type_decl context (c_value)
         :> lkt_node)
    | 38 ->
        (wrap_struct_decl context (c_value)
         :> lkt_node)
    | 39 ->
        (wrap_trait_decl context (c_value)
         :> lkt_node)
    | 40 ->
        (wrap_decl_annotation context (c_value)
         :> lkt_node)
    | 41 ->
        (wrap_decl_annotation_args context (c_value)
         :> lkt_node)
    | 42 ->
        (wrap_dyn_env_wrapper context (c_value)
         :> lkt_node)
    | 43 ->
        (wrap_elsif_branch context (c_value)
         :> lkt_node)
    | 44 ->
        (wrap_enum_class_case context (c_value)
         :> lkt_node)
    | 45 ->
        (wrap_excludes_null_absent context (c_value)
         :> lkt_node)
    | 46 ->
        (wrap_excludes_null_present context (c_value)
         :> lkt_node)
    | 47 ->
        (wrap_any_of context (c_value)
         :> lkt_node)
    | 48 ->
        (wrap_array_literal context (c_value)
         :> lkt_node)
    | 49 ->
        (wrap_call_expr context (c_value)
         :> lkt_node)
    | 50 ->
        (wrap_logic_predicate context (c_value)
         :> lkt_node)
    | 51 ->
        (wrap_logic_propagate_call context (c_value)
         :> lkt_node)
    | 52 ->
        (wrap_bin_op context (c_value)
         :> lkt_node)
    | 53 ->
        (wrap_block_expr context (c_value)
         :> lkt_node)
    | 54 ->
        (wrap_cast_expr context (c_value)
         :> lkt_node)
    | 55 ->
        (wrap_dot_expr context (c_value)
         :> lkt_node)
    | 56 ->
        (wrap_error_on_null context (c_value)
         :> lkt_node)
    | 57 ->
        (wrap_generic_instantiation context (c_value)
         :> lkt_node)
    | 58 ->
        (wrap_grammar_cut context (c_value)
         :> lkt_node)
    | 59 ->
        (wrap_grammar_discard context (c_value)
         :> lkt_node)
    | 60 ->
        (wrap_grammar_dont_skip context (c_value)
         :> lkt_node)
    | 61 ->
        (wrap_grammar_list context (c_value)
         :> lkt_node)
    | 62 ->
        (wrap_grammar_null context (c_value)
         :> lkt_node)
    | 63 ->
        (wrap_grammar_opt context (c_value)
         :> lkt_node)
    | 64 ->
        (wrap_grammar_opt_error context (c_value)
         :> lkt_node)
    | 65 ->
        (wrap_grammar_opt_error_group context (c_value)
         :> lkt_node)
    | 66 ->
        (wrap_grammar_opt_group context (c_value)
         :> lkt_node)
    | 67 ->
        (wrap_grammar_or_expr context (c_value)
         :> lkt_node)
    | 68 ->
        (wrap_grammar_pick context (c_value)
         :> lkt_node)
    | 69 ->
        (wrap_grammar_implicit_pick context (c_value)
         :> lkt_node)
    | 70 ->
        (wrap_grammar_predicate context (c_value)
         :> lkt_node)
    | 71 ->
        (wrap_grammar_rule_ref context (c_value)
         :> lkt_node)
    | 72 ->
        (wrap_grammar_skip context (c_value)
         :> lkt_node)
    | 73 ->
        (wrap_grammar_stop_cut context (c_value)
         :> lkt_node)
    | 74 ->
        (wrap_parse_node_expr context (c_value)
         :> lkt_node)
    | 75 ->
        (wrap_token_lit context (c_value)
         :> lkt_node)
    | 76 ->
        (wrap_token_no_case_lit context (c_value)
         :> lkt_node)
    | 77 ->
        (wrap_token_pattern_concat context (c_value)
         :> lkt_node)
    | 78 ->
        (wrap_token_pattern_lit context (c_value)
         :> lkt_node)
    | 79 ->
        (wrap_token_ref context (c_value)
         :> lkt_node)
    | 80 ->
        (wrap_id context (c_value)
         :> lkt_node)
    | 81 ->
        (wrap_def_id context (c_value)
         :> lkt_node)
    | 82 ->
        (wrap_module_ref_id context (c_value)
         :> lkt_node)
    | 83 ->
        (wrap_ref_id context (c_value)
         :> lkt_node)
    | 84 ->
        (wrap_if_expr context (c_value)
         :> lkt_node)
    | 85 ->
        (wrap_isa context (c_value)
         :> lkt_node)
    | 86 ->
        (wrap_keep_expr context (c_value)
         :> lkt_node)
    | 87 ->
        (wrap_lambda_expr context (c_value)
         :> lkt_node)
    | 88 ->
        (wrap_big_num_lit context (c_value)
         :> lkt_node)
    | 89 ->
        (wrap_char_lit context (c_value)
         :> lkt_node)
    | 90 ->
        (wrap_null_lit context (c_value)
         :> lkt_node)
    | 91 ->
        (wrap_num_lit context (c_value)
         :> lkt_node)
    | 92 ->
        (wrap_block_string_lit context (c_value)
         :> lkt_node)
    | 93 ->
        (wrap_single_line_string_lit context (c_value)
         :> lkt_node)
    | 94 ->
        (wrap_pattern_single_line_string_lit context (c_value)
         :> lkt_node)
    | 95 ->
        (wrap_logic_assign context (c_value)
         :> lkt_node)
    | 96 ->
        (wrap_logic_expr context (c_value)
         :> lkt_node)
    | 97 ->
        (wrap_logic_propagate context (c_value)
         :> lkt_node)
    | 98 ->
        (wrap_logic_unify context (c_value)
         :> lkt_node)
    | 99 ->
        (wrap_match_expr context (c_value)
         :> lkt_node)
    | 100 ->
        (wrap_not_expr context (c_value)
         :> lkt_node)
    | 101 ->
        (wrap_paren_expr context (c_value)
         :> lkt_node)
    | 102 ->
        (wrap_raise_expr context (c_value)
         :> lkt_node)
    | 103 ->
        (wrap_subscript_expr context (c_value)
         :> lkt_node)
    | 104 ->
        (wrap_try_expr context (c_value)
         :> lkt_node)
    | 105 ->
        (wrap_un_op context (c_value)
         :> lkt_node)
    | 106 ->
        (wrap_full_decl context (c_value)
         :> lkt_node)
    | 107 ->
        (wrap_grammar_list_sep context (c_value)
         :> lkt_node)
    | 108 ->
        (wrap_import context (c_value)
         :> lkt_node)
    | 109 ->
        (wrap_langkit_root context (c_value)
         :> lkt_node)
    | 110 ->
        (wrap_lexer_case_rule context (c_value)
         :> lkt_node)
    | 111 ->
        (wrap_lexer_case_rule_send context (c_value)
         :> lkt_node)
    | 112 ->
        (wrap_list_kind_one context (c_value)
         :> lkt_node)
    | 113 ->
        (wrap_list_kind_zero context (c_value)
         :> lkt_node)
    | 114 ->
        (wrap_argument_list context (c_value)
         :> lkt_node)
    | 115 ->
        (wrap_base_lexer_case_rule_alt_list context (c_value)
         :> lkt_node)
    | 116 ->
        (wrap_base_match_branch_list context (c_value)
         :> lkt_node)
    | 117 ->
        (wrap_block_string_line_list context (c_value)
         :> lkt_node)
    | 118 ->
        (wrap_call_expr_list context (c_value)
         :> lkt_node)
    | 119 ->
        (wrap_decl_annotation_list context (c_value)
         :> lkt_node)
    | 120 ->
        (wrap_elsif_branch_list context (c_value)
         :> lkt_node)
    | 121 ->
        (wrap_enum_class_alt_decl_list context (c_value)
         :> lkt_node)
    | 122 ->
        (wrap_enum_class_case_list context (c_value)
         :> lkt_node)
    | 123 ->
        (wrap_enum_lit_decl_list context (c_value)
         :> lkt_node)
    | 124 ->
        (wrap_expr_list context (c_value)
         :> lkt_node)
    | 125 ->
        (wrap_any_of_list context (c_value)
         :> lkt_node)
    | 126 ->
        (wrap_full_decl_list context (c_value)
         :> lkt_node)
    | 127 ->
        (wrap_decl_block context (c_value)
         :> lkt_node)
    | 128 ->
        (wrap_generic_param_decl_list context (c_value)
         :> lkt_node)
    | 129 ->
        (wrap_fun_param_decl_list context (c_value)
         :> lkt_node)
    | 130 ->
        (wrap_grammar_expr_list context (c_value)
         :> lkt_node)
    | 131 ->
        (wrap_grammar_expr_list_list context (c_value)
         :> lkt_node)
    | 132 ->
        (wrap_import_list context (c_value)
         :> lkt_node)
    | 133 ->
        (wrap_lambda_param_decl_list context (c_value)
         :> lkt_node)
    | 134 ->
        (wrap_lkt_node_list context (c_value)
         :> lkt_node)
    | 135 ->
        (wrap_pattern_detail_list context (c_value)
         :> lkt_node)
    | 136 ->
        (wrap_pattern_list context (c_value)
         :> lkt_node)
    | 137 ->
        (wrap_ref_id_list context (c_value)
         :> lkt_node)
    | 138 ->
        (wrap_type_ref_list context (c_value)
         :> lkt_node)
    | 139 ->
        (wrap_synthetic_type_ref_list context (c_value)
         :> lkt_node)
    | 140 ->
        (wrap_null_cond_qualifier_absent context (c_value)
         :> lkt_node)
    | 141 ->
        (wrap_null_cond_qualifier_present context (c_value)
         :> lkt_node)
    | 142 ->
        (wrap_op_amp context (c_value)
         :> lkt_node)
    | 143 ->
        (wrap_op_and context (c_value)
         :> lkt_node)
    | 144 ->
        (wrap_op_div context (c_value)
         :> lkt_node)
    | 145 ->
        (wrap_op_eq context (c_value)
         :> lkt_node)
    | 146 ->
        (wrap_op_gt context (c_value)
         :> lkt_node)
    | 147 ->
        (wrap_op_gte context (c_value)
         :> lkt_node)
    | 148 ->
        (wrap_op_logic_and context (c_value)
         :> lkt_node)
    | 149 ->
        (wrap_op_logic_or context (c_value)
         :> lkt_node)
    | 150 ->
        (wrap_op_lt context (c_value)
         :> lkt_node)
    | 151 ->
        (wrap_op_lte context (c_value)
         :> lkt_node)
    | 152 ->
        (wrap_op_minus context (c_value)
         :> lkt_node)
    | 153 ->
        (wrap_op_mult context (c_value)
         :> lkt_node)
    | 154 ->
        (wrap_op_ne context (c_value)
         :> lkt_node)
    | 155 ->
        (wrap_op_or context (c_value)
         :> lkt_node)
    | 156 ->
        (wrap_op_or_int context (c_value)
         :> lkt_node)
    | 157 ->
        (wrap_op_plus context (c_value)
         :> lkt_node)
    | 158 ->
        (wrap_any_type_pattern context (c_value)
         :> lkt_node)
    | 159 ->
        (wrap_binding_pattern context (c_value)
         :> lkt_node)
    | 160 ->
        (wrap_bool_pattern_false context (c_value)
         :> lkt_node)
    | 161 ->
        (wrap_bool_pattern_true context (c_value)
         :> lkt_node)
    | 162 ->
        (wrap_ellipsis_pattern context (c_value)
         :> lkt_node)
    | 163 ->
        (wrap_extended_pattern context (c_value)
         :> lkt_node)
    | 164 ->
        (wrap_filtered_pattern context (c_value)
         :> lkt_node)
    | 165 ->
        (wrap_integer_pattern context (c_value)
         :> lkt_node)
    | 166 ->
        (wrap_list_pattern context (c_value)
         :> lkt_node)
    | 167 ->
        (wrap_not_pattern context (c_value)
         :> lkt_node)
    | 168 ->
        (wrap_null_pattern context (c_value)
         :> lkt_node)
    | 169 ->
        (wrap_or_pattern context (c_value)
         :> lkt_node)
    | 170 ->
        (wrap_paren_pattern context (c_value)
         :> lkt_node)
    | 171 ->
        (wrap_regex_pattern context (c_value)
         :> lkt_node)
    | 172 ->
        (wrap_tuple_pattern context (c_value)
         :> lkt_node)
    | 173 ->
        (wrap_type_pattern context (c_value)
         :> lkt_node)
    | 174 ->
        (wrap_field_pattern_detail context (c_value)
         :> lkt_node)
    | 175 ->
        (wrap_property_pattern_detail context (c_value)
         :> lkt_node)
    | 176 ->
        (wrap_selector_pattern_detail context (c_value)
         :> lkt_node)
    | 177 ->
        (wrap_selector_call context (c_value)
         :> lkt_node)
    | 178 ->
        (wrap_default_list_type_ref context (c_value)
         :> lkt_node)
    | 179 ->
        (wrap_function_type_ref context (c_value)
         :> lkt_node)
    | 180 ->
        (wrap_generic_type_ref context (c_value)
         :> lkt_node)
    | 181 ->
        (wrap_simple_type_ref context (c_value)
         :> lkt_node)
    | 182 ->
        (wrap_var_bind context (c_value)
         :> lkt_node)
    | _ -> assert false

      


  and wrap_argument context c_value
   : argument =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.argument_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_ref_id context (field_c_value))
      in
         

      node
    in
    let f_value () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.argument_f_value
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Argument {
        f_name
          = Lazy.from_fun f_name ;
        f_value
          = Lazy.from_fun f_value ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_lexer_case_rule_alt context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `LexerCaseRuleCondAlt _
      | `LexerCaseRuleDefaultAlt _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_lexer_case_rule_cond_alt context c_value
   : lexer_case_rule_cond_alt =
    let f_cond_exprs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lexer_case_rule_cond_alt_f_cond_exprs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ref_id_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LexerCaseRuleCondAlt {
        f_cond_exprs
          = Lazy.from_fun f_cond_exprs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_lexer_case_rule_default_alt context c_value
   : lexer_case_rule_default_alt =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LexerCaseRuleDefaultAlt {
        c_value = c_value;
        context = context
      }

      

  and wrap_base_match_branch context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `MatchBranch _
      | `PatternMatchBranch _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_match_branch context c_value
   : match_branch =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.match_branch_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_match_val_decl context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `MatchBranch {
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_pattern_match_branch context c_value
   : pattern_match_branch =
    let f_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.pattern_match_branch_f_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `OrPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PatternMatchBranch {
        f_pattern
          = Lazy.from_fun f_pattern ;
        c_value = c_value;
        context = context
      }

      


  and wrap_block_expr_clause context c_value
   : block_expr_clause =
    let f_clause () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.block_expr_clause_f_clause
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_lkt_node context (field_c_value)
      in
         

      match node with
            
      | `ValDecl _
      | `VarBind _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BlockExprClause {
        f_clause
          = Lazy.from_fun f_clause ;
        c_value = c_value;
        context = context
      }

      


  and wrap_block_string_line context c_value
   : block_string_line =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BlockStringLine {
        c_value = c_value;
        context = context
      }

      

  and wrap_class_qualifier context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `ClassQualifierAbsent _
      | `ClassQualifierPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_class_qualifier_absent context c_value
   : class_qualifier_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ClassQualifierAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_class_qualifier_present context c_value
   : class_qualifier_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ClassQualifierPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `GrammarRuleDecl _
      | `SyntheticLexerDecl _
      | `NodeDecl _
      | `SelfDecl _
      | `BindingValDecl _
      | `EnumLitDecl _
      | `FieldDecl _
      | `FunParamDecl _
      | `LambdaParamDecl _
      | `DynVarDecl _
      | `MatchValDecl _
      | `ValDecl _
      | `FunDecl _
      | `EnvSpecDecl _
      | `ErrorDecl _
      | `GenericDecl _
      | `GrammarDecl _
      | `LexerDecl _
      | `LexerFamilyDecl _
      | `SynthFunDecl _
      | `SynthParamDecl _
      | `AnyTypeDecl _
      | `EnumClassAltDecl _
      | `FunctionType _
      | `GenericParamTypeDecl _
      | `ClassDecl _
      | `EnumClassDecl _
      | `EnumTypeDecl _
      | `StructDecl _
      | `TraitDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_base_grammar_rule_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `GrammarRuleDecl _
      | `SyntheticLexerDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_grammar_rule_decl context c_value
   : grammar_rule_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarRuleDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_lexer_decl context c_value
   : synthetic_lexer_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticLexerDecl {
        c_value = c_value;
        context = context
      }

      

  and wrap_base_val_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `NodeDecl _
      | `SelfDecl _
      | `BindingValDecl _
      | `EnumLitDecl _
      | `FieldDecl _
      | `FunParamDecl _
      | `LambdaParamDecl _
      | `DynVarDecl _
      | `MatchValDecl _
      | `ValDecl _
      | `FunDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_node_decl context c_value
   : node_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NodeDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_self_decl context c_value
   : self_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SelfDecl {
        c_value = c_value;
        context = context
      }

      

  and wrap_user_val_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `BindingValDecl _
      | `EnumLitDecl _
      | `FieldDecl _
      | `FunParamDecl _
      | `LambdaParamDecl _
      | `DynVarDecl _
      | `MatchValDecl _
      | `ValDecl _
      | `FunDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_binding_val_decl context c_value
   : binding_val_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BindingValDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_lit_decl context c_value
   : enum_lit_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumLitDecl {
        c_value = c_value;
        context = context
      }

      

  and wrap_explicitly_typed_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `FieldDecl _
      | `FunParamDecl _
      | `LambdaParamDecl _
      | `DynVarDecl _
      | `MatchValDecl _
      | `ValDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_component_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `FieldDecl _
      | `FunParamDecl _
      | `LambdaParamDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_field_decl context c_value
   : field_decl =
    let f_trait_ref () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.field_decl_f_trait_ref
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_dot_expr context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FieldDecl {
        f_trait_ref
          = Lazy.from_fun f_trait_ref ;
        c_value = c_value;
        context = context
      }

      


  and wrap_fun_param_decl context c_value
   : fun_param_decl =
    let f_decl_annotations () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.fun_param_decl_f_decl_annotations
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_decl_annotation_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FunParamDecl {
        f_decl_annotations
          = Lazy.from_fun f_decl_annotations ;
        c_value = c_value;
        context = context
      }

      


  and wrap_lambda_param_decl context c_value
   : lambda_param_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LambdaParamDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_dyn_var_decl context c_value
   : dyn_var_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DynVarDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_match_val_decl context c_value
   : match_val_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `MatchValDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_val_decl context c_value
   : val_decl =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.val_decl_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ValDecl {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_fun_decl context c_value
   : fun_decl =
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.fun_decl_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_fun_param_decl_list context (field_c_value)
      in
         

      node
    in
    let f_return_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.fun_decl_f_return_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      match node with
            
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ -> assert false
    in
    let f_trait_ref () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.fun_decl_f_trait_ref
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_dot_expr context (field_c_value))
      in
         

      node
    in
    let f_body () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.fun_decl_f_body
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AnyOf _
      | Some `ArrayLiteral _
      | Some `BigNumLit _
      | Some `BinOp _
      | Some `BlockExpr _
      | Some `BlockStringLit _
      | Some `CallExpr _
      | Some `CastExpr _
      | Some `CharLit _
      | Some `DotExpr _
      | Some `ErrorOnNull _
      | Some `GenericInstantiation _
      | Some `IfExpr _
      | Some `Isa _
      | Some `KeepExpr _
      | Some `LambdaExpr _
      | Some `LogicAssign _
      | Some `LogicExpr _
      | Some `LogicPredicate _
      | Some `LogicPropagate _
      | Some `LogicUnify _
      | Some `MatchExpr _
      | Some `NotExpr _
      | Some `NullLit _
      | Some `NumLit _
      | Some `ParenExpr _
      | Some `PatternSingleLineStringLit _
      | Some `RaiseExpr _
      | Some `RefId _
      | Some `SingleLineStringLit _
      | Some `SubscriptExpr _
      | Some `TryExpr _
      | Some `UnOp _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FunDecl {
        f_params
          = Lazy.from_fun f_params ;
        f_return_type
          = Lazy.from_fun f_return_type ;
        f_trait_ref
          = Lazy.from_fun f_trait_ref ;
        f_body
          = Lazy.from_fun f_body ;
        c_value = c_value;
        context = context
      }

      


  and wrap_env_spec_decl context c_value
   : env_spec_decl =
    let f_actions () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.env_spec_decl_f_actions
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_call_expr_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnvSpecDecl {
        f_actions
          = Lazy.from_fun f_actions ;
        c_value = c_value;
        context = context
      }

      


  and wrap_error_decl context c_value
   : error_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ErrorDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_decl context c_value
   : generic_decl =
    let f_generic_param_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_decl_f_generic_param_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_generic_param_decl_list context (field_c_value)
      in
         

      node
    in
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_decl_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_decl context (field_c_value)
      in
         

      match node with
            
      | `ClassDecl _
      | `DynVarDecl _
      | `EnumClassDecl _
      | `EnumTypeDecl _
      | `EnvSpecDecl _
      | `FieldDecl _
      | `FunDecl _
      | `GenericDecl _
      | `GrammarDecl _
      | `GrammarRuleDecl _
      | `LexerDecl _
      | `StructDecl _
      | `TraitDecl _
      | `ValDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericDecl {
        f_generic_param_decls
          = Lazy.from_fun f_generic_param_decls ;
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_decl context c_value
   : grammar_decl =
    let f_rules () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_decl_f_rules
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_full_decl_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarDecl {
        f_rules
          = Lazy.from_fun f_rules ;
        c_value = c_value;
        context = context
      }

      


  and wrap_lexer_decl context c_value
   : lexer_decl =
    let f_rules () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lexer_decl_f_rules
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_lkt_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LexerDecl {
        f_rules
          = Lazy.from_fun f_rules ;
        c_value = c_value;
        context = context
      }

      


  and wrap_lexer_family_decl context c_value
   : lexer_family_decl =
    let f_rules () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lexer_family_decl_f_rules
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_full_decl_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LexerFamilyDecl {
        f_rules
          = Lazy.from_fun f_rules ;
        c_value = c_value;
        context = context
      }

      


  and wrap_synth_fun_decl context c_value
   : synth_fun_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SynthFunDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_synth_param_decl context c_value
   : synth_param_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SynthParamDecl {
        c_value = c_value;
        context = context
      }

      

  and wrap_type_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `AnyTypeDecl _
      | `EnumClassAltDecl _
      | `FunctionType _
      | `GenericParamTypeDecl _
      | `ClassDecl _
      | `EnumClassDecl _
      | `EnumTypeDecl _
      | `StructDecl _
      | `TraitDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_any_type_decl context c_value
   : any_type_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AnyTypeDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_class_alt_decl context c_value
   : enum_class_alt_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumClassAltDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_function_type context c_value
   : function_type =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FunctionType {
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_param_type_decl context c_value
   : generic_param_type_decl =
    let f_has_class () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_param_type_decl_f_has_class
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_class_qualifier context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericParamTypeDecl {
        f_has_class
          = Lazy.from_fun f_has_class ;
        c_value = c_value;
        context = context
      }

      

  and wrap_named_type_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `ClassDecl _
      | `EnumClassDecl _
      | `EnumTypeDecl _
      | `StructDecl _
      | `TraitDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      

  and wrap_basic_class_decl context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `ClassDecl _
      | `EnumClassDecl _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_class_decl context c_value
   : class_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ClassDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_class_decl context c_value
   : enum_class_decl =
    let f_branches () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.enum_class_decl_f_branches
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_enum_class_case_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumClassDecl {
        f_branches
          = Lazy.from_fun f_branches ;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_type_decl context c_value
   : enum_type_decl =
    let f_literals () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.enum_type_decl_f_literals
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_enum_lit_decl_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumTypeDecl {
        f_literals
          = Lazy.from_fun f_literals ;
        c_value = c_value;
        context = context
      }

      


  and wrap_struct_decl context c_value
   : struct_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `StructDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_trait_decl context c_value
   : trait_decl =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TraitDecl {
        c_value = c_value;
        context = context
      }

      


  and wrap_decl_annotation context c_value
   : decl_annotation =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decl_annotation_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_id context (field_c_value)
      in
         

      node
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decl_annotation_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_decl_annotation_args context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeclAnnotation {
        f_name
          = Lazy.from_fun f_name ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      


  and wrap_decl_annotation_args context c_value
   : decl_annotation_args =
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.decl_annotation_args_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_argument_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeclAnnotationArgs {
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      


  and wrap_dyn_env_wrapper context c_value
   : dyn_env_wrapper =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DynEnvWrapper {
        c_value = c_value;
        context = context
      }

      


  and wrap_elsif_branch context c_value
   : elsif_branch =
    let f_cond_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.elsif_branch_f_cond_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    let f_then_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.elsif_branch_f_then_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ElsifBranch {
        f_cond_expr
          = Lazy.from_fun f_cond_expr ;
        f_then_expr
          = Lazy.from_fun f_then_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_class_case context c_value
   : enum_class_case =
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.enum_class_case_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_enum_class_alt_decl_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumClassCase {
        f_decls
          = Lazy.from_fun f_decls ;
        c_value = c_value;
        context = context
      }

      

  and wrap_excludes_null context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `ExcludesNullAbsent _
      | `ExcludesNullPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_excludes_null_absent context c_value
   : excludes_null_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExcludesNullAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_excludes_null_present context c_value
   : excludes_null_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExcludesNullPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_expr context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `AnyOf _
      | `ArrayLiteral _
      | `CallExpr _
      | `LogicPredicate _
      | `LogicPropagateCall _
      | `BinOp _
      | `BlockExpr _
      | `CastExpr _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `GrammarCut _
      | `GrammarDiscard _
      | `GrammarDontSkip _
      | `GrammarList _
      | `GrammarNull _
      | `GrammarOpt _
      | `GrammarOptError _
      | `GrammarOptErrorGroup _
      | `GrammarOptGroup _
      | `GrammarOrExpr _
      | `GrammarPick _
      | `GrammarImplicitPick _
      | `GrammarPredicate _
      | `GrammarRuleRef _
      | `GrammarSkip _
      | `GrammarStopCut _
      | `ParseNodeExpr _
      | `TokenLit _
      | `TokenNoCaseLit _
      | `TokenPatternConcat _
      | `TokenPatternLit _
      | `TokenRef _
      | `Id _
      | `DefId _
      | `ModuleRefId _
      | `RefId _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `BigNumLit _
      | `CharLit _
      | `NullLit _
      | `NumLit _
      | `BlockStringLit _
      | `SingleLineStringLit _
      | `PatternSingleLineStringLit _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `ParenExpr _
      | `RaiseExpr _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_any_of context c_value
   : any_of =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.any_of_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      as e -> e
      | _ -> assert false
    in
    let f_values () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.any_of_f_values
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_any_of_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AnyOf {
        f_expr
          = Lazy.from_fun f_expr ;
        f_values
          = Lazy.from_fun f_values ;
        c_value = c_value;
        context = context
      }

      


  and wrap_array_literal context c_value
   : array_literal =
    let f_exprs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.array_literal_f_exprs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr_list context (field_c_value)
      in
         

      node
    in
    let f_element_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.array_literal_f_element_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_ref context (field_c_value))
      in
         

      match node with
            
      | Some `FunctionTypeRef _
      | Some `GenericTypeRef _
      | Some `SimpleTypeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ArrayLiteral {
        f_exprs
          = Lazy.from_fun f_exprs ;
        f_element_type
          = Lazy.from_fun f_element_type ;
        c_value = c_value;
        context = context
      }

      

  and wrap_base_call_expr context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `CallExpr _
      | `LogicPredicate _
      | `LogicPropagateCall _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_call_expr context c_value
   : call_expr =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_call_expr_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_call_expr_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_argument_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CallExpr {
        f_name
          = Lazy.from_fun f_name ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      

  and wrap_logic_call_expr context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `LogicPredicate _
      | `LogicPropagateCall _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_logic_predicate context c_value
   : logic_predicate =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_call_expr_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_call_expr_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_argument_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LogicPredicate {
        f_name
          = Lazy.from_fun f_name ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      


  and wrap_logic_propagate_call context c_value
   : logic_propagate_call =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_call_expr_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.base_call_expr_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_argument_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LogicPropagateCall {
        f_name
          = Lazy.from_fun f_name ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      


  and wrap_bin_op context c_value
   : bin_op =
    let f_left () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_left
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    let f_op () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_op
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_op context (field_c_value)
      in
         

      node
    in
    let f_right () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.bin_op_f_right
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BinOp {
        f_left
          = Lazy.from_fun f_left ;
        f_op
          = Lazy.from_fun f_op ;
        f_right
          = Lazy.from_fun f_right ;
        c_value = c_value;
        context = context
      }

      


  and wrap_block_expr context c_value
   : block_expr =
    let f_clauses () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.block_expr_f_clauses
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_lkt_node_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BlockExpr {
        f_clauses
          = Lazy.from_fun f_clauses ;
        c_value = c_value;
        context = context
      }

      


  and wrap_cast_expr context c_value
   : cast_expr =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.cast_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_null_cond () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.cast_expr_f_null_cond
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_null_cond_qualifier context (field_c_value)
      in
         

      node
    in
    let f_excludes_null () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.cast_expr_f_excludes_null
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_excludes_null context (field_c_value)
      in
         

      node
    in
    let f_dest_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.cast_expr_f_dest_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      match node with
            
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CastExpr {
        f_expr
          = Lazy.from_fun f_expr ;
        f_null_cond
          = Lazy.from_fun f_null_cond ;
        f_excludes_null
          = Lazy.from_fun f_excludes_null ;
        f_dest_type
          = Lazy.from_fun f_dest_type ;
        c_value = c_value;
        context = context
      }

      


  and wrap_dot_expr context c_value
   : dot_expr =
    let f_prefix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.dot_expr_f_prefix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_null_cond () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.dot_expr_f_null_cond
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_null_cond_qualifier context (field_c_value)
      in
         

      node
    in
    let f_suffix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.dot_expr_f_suffix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ref_id context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DotExpr {
        f_prefix
          = Lazy.from_fun f_prefix ;
        f_null_cond
          = Lazy.from_fun f_null_cond ;
        f_suffix
          = Lazy.from_fun f_suffix ;
        c_value = c_value;
        context = context
      }

      


  and wrap_error_on_null context c_value
   : error_on_null =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.error_on_null_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ErrorOnNull {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_instantiation context c_value
   : generic_instantiation =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_instantiation_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_instantiation_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericInstantiation {
        f_name
          = Lazy.from_fun f_name ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      

  and wrap_grammar_expr context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `GrammarCut _
      | `GrammarDiscard _
      | `GrammarDontSkip _
      | `GrammarList _
      | `GrammarNull _
      | `GrammarOpt _
      | `GrammarOptError _
      | `GrammarOptErrorGroup _
      | `GrammarOptGroup _
      | `GrammarOrExpr _
      | `GrammarPick _
      | `GrammarImplicitPick _
      | `GrammarPredicate _
      | `GrammarRuleRef _
      | `GrammarSkip _
      | `GrammarStopCut _
      | `ParseNodeExpr _
      | `TokenLit _
      | `TokenNoCaseLit _
      | `TokenPatternConcat _
      | `TokenPatternLit _
      | `TokenRef _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_grammar_cut context c_value
   : grammar_cut =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarCut {
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_discard context c_value
   : grammar_discard =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_discard_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarDiscard {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_dont_skip context c_value
   : grammar_dont_skip =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_dont_skip_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    let f_dont_skip () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_dont_skip_f_dont_skip
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarDontSkip {
        f_expr
          = Lazy.from_fun f_expr ;
        f_dont_skip
          = Lazy.from_fun f_dont_skip ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_list context c_value
   : grammar_list =
    let f_list_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_list_f_list_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      node
    in
    let f_kind () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_list_f_kind
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_list_kind context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_list_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    let f_sep () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_list_f_sep
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_grammar_list_sep context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarList {
        f_list_type
          = Lazy.from_fun f_list_type ;
        f_kind
          = Lazy.from_fun f_kind ;
        f_expr
          = Lazy.from_fun f_expr ;
        f_sep
          = Lazy.from_fun f_sep ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_null context c_value
   : grammar_null =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_null_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      match node with
            
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarNull {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_opt context c_value
   : grammar_opt =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_opt_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarOpt {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_opt_error context c_value
   : grammar_opt_error =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_opt_error_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarOptError {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_opt_error_group context c_value
   : grammar_opt_error_group =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_opt_error_group_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarOptErrorGroup {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_opt_group context c_value
   : grammar_opt_group =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_opt_group_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarOptGroup {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_or_expr context c_value
   : grammar_or_expr =
    let f_sub_exprs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_or_expr_f_sub_exprs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr_list_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarOrExpr {
        f_sub_exprs
          = Lazy.from_fun f_sub_exprs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_pick context c_value
   : grammar_pick =
    let f_exprs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_pick_f_exprs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarPick {
        f_exprs
          = Lazy.from_fun f_exprs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_implicit_pick context c_value
   : grammar_implicit_pick =
    let f_exprs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_pick_f_exprs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarImplicitPick {
        f_exprs
          = Lazy.from_fun f_exprs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_predicate context c_value
   : grammar_predicate =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_predicate_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    let f_prop_ref () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_predicate_f_prop_ref
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `DotExpr _
      | `RefId _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarPredicate {
        f_expr
          = Lazy.from_fun f_expr ;
        f_prop_ref
          = Lazy.from_fun f_prop_ref ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_rule_ref context c_value
   : grammar_rule_ref =
    let f_node_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_rule_ref_f_node_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ref_id context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarRuleRef {
        f_node_name
          = Lazy.from_fun f_node_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_skip context c_value
   : grammar_skip =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_skip_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      match node with
            
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarSkip {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_stop_cut context c_value
   : grammar_stop_cut =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_stop_cut_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarStopCut {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_parse_node_expr context c_value
   : parse_node_expr =
    let f_node_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.parse_node_expr_f_node_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      match node with
            
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ -> assert false
    in
    let f_sub_exprs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.parse_node_expr_f_sub_exprs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParseNodeExpr {
        f_node_name
          = Lazy.from_fun f_node_name ;
        f_sub_exprs
          = Lazy.from_fun f_sub_exprs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_token_lit context c_value
   : token_lit =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TokenLit {
        c_value = c_value;
        context = context
      }

      


  and wrap_token_no_case_lit context c_value
   : token_no_case_lit =
    let f_lit () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.token_no_case_lit_f_lit
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_token_lit context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TokenNoCaseLit {
        f_lit
          = Lazy.from_fun f_lit ;
        c_value = c_value;
        context = context
      }

      


  and wrap_token_pattern_concat context c_value
   : token_pattern_concat =
    let f_left () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.token_pattern_concat_f_left
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      match node with
            
      | `TokenPatternConcat _
      | `TokenPatternLit _
      as e -> e
      | _ -> assert false
    in
    let f_right () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.token_pattern_concat_f_right
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_token_pattern_lit context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TokenPatternConcat {
        f_left
          = Lazy.from_fun f_left ;
        f_right
          = Lazy.from_fun f_right ;
        c_value = c_value;
        context = context
      }

      


  and wrap_token_pattern_lit context c_value
   : token_pattern_lit =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TokenPatternLit {
        c_value = c_value;
        context = context
      }

      


  and wrap_token_ref context c_value
   : token_ref =
    let f_token_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.token_ref_f_token_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ref_id context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.token_ref_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_token_lit context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TokenRef {
        f_token_name
          = Lazy.from_fun f_token_name ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_id context c_value
   : id =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Id {
        c_value = c_value;
        context = context
      }

      


  and wrap_def_id context c_value
   : def_id =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DefId {
        c_value = c_value;
        context = context
      }

      


  and wrap_module_ref_id context c_value
   : module_ref_id =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ModuleRefId {
        c_value = c_value;
        context = context
      }

      


  and wrap_ref_id context c_value
   : ref_id =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RefId {
        c_value = c_value;
        context = context
      }

      


  and wrap_if_expr context c_value
   : if_expr =
    let f_cond_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_expr_f_cond_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    let f_then_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_expr_f_then_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    let f_alternatives () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_expr_f_alternatives
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_elsif_branch_list context (field_c_value)
      in
         

      node
    in
    let f_else_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.if_expr_f_else_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IfExpr {
        f_cond_expr
          = Lazy.from_fun f_cond_expr ;
        f_then_expr
          = Lazy.from_fun f_then_expr ;
        f_alternatives
          = Lazy.from_fun f_alternatives ;
        f_else_expr
          = Lazy.from_fun f_else_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_isa context c_value
   : isa =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.isa_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      as e -> e
      | _ -> assert false
    in
    let f_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.isa_f_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `OrPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Isa {
        f_expr
          = Lazy.from_fun f_expr ;
        f_pattern
          = Lazy.from_fun f_pattern ;
        c_value = c_value;
        context = context
      }

      


  and wrap_keep_expr context c_value
   : keep_expr =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.keep_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_null_cond () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.keep_expr_f_null_cond
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_null_cond_qualifier context (field_c_value)
      in
         

      node
    in
    let f_keep_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.keep_expr_f_keep_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      match node with
            
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `KeepExpr {
        f_expr
          = Lazy.from_fun f_expr ;
        f_null_cond
          = Lazy.from_fun f_null_cond ;
        f_keep_type
          = Lazy.from_fun f_keep_type ;
        c_value = c_value;
        context = context
      }

      


  and wrap_lambda_expr context c_value
   : lambda_expr =
    let f_params () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lambda_expr_f_params
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_lambda_param_decl_list context (field_c_value)
      in
         

      node
    in
    let f_return_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lambda_expr_f_return_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_ref context (field_c_value))
      in
         

      match node with
            
      | Some `FunctionTypeRef _
      | Some `GenericTypeRef _
      | Some `SimpleTypeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_body () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lambda_expr_f_body
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LambdaExpr {
        f_params
          = Lazy.from_fun f_params ;
        f_return_type
          = Lazy.from_fun f_return_type ;
        f_body
          = Lazy.from_fun f_body ;
        c_value = c_value;
        context = context
      }

      

  and wrap_lit context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `BigNumLit _
      | `CharLit _
      | `NullLit _
      | `NumLit _
      | `BlockStringLit _
      | `SingleLineStringLit _
      | `PatternSingleLineStringLit _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_big_num_lit context c_value
   : big_num_lit =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BigNumLit {
        c_value = c_value;
        context = context
      }

      


  and wrap_char_lit context c_value
   : char_lit =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CharLit {
        c_value = c_value;
        context = context
      }

      


  and wrap_null_lit context c_value
   : null_lit =
    let f_dest_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.null_lit_f_dest_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_ref context (field_c_value))
      in
         

      match node with
            
      | Some `FunctionTypeRef _
      | Some `GenericTypeRef _
      | Some `SimpleTypeRef _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullLit {
        f_dest_type
          = Lazy.from_fun f_dest_type ;
        c_value = c_value;
        context = context
      }

      


  and wrap_num_lit context c_value
   : num_lit =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NumLit {
        c_value = c_value;
        context = context
      }

      

  and wrap_string_lit context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `BlockStringLit _
      | `SingleLineStringLit _
      | `PatternSingleLineStringLit _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_block_string_lit context c_value
   : block_string_lit =
    let f_lines () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.block_string_lit_f_lines
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_block_string_line_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BlockStringLit {
        f_lines
          = Lazy.from_fun f_lines ;
        c_value = c_value;
        context = context
      }

      


  and wrap_single_line_string_lit context c_value
   : single_line_string_lit =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SingleLineStringLit {
        c_value = c_value;
        context = context
      }

      


  and wrap_pattern_single_line_string_lit context c_value
   : pattern_single_line_string_lit =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PatternSingleLineStringLit {
        c_value = c_value;
        context = context
      }

      


  and wrap_logic_assign context c_value
   : logic_assign =
    let f_dest_var () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.logic_assign_f_dest_var
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      as e -> e
      | _ -> assert false
    in
    let f_value () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.logic_assign_f_value
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LogicAssign {
        f_dest_var
          = Lazy.from_fun f_dest_var ;
        f_value
          = Lazy.from_fun f_value ;
        c_value = c_value;
        context = context
      }

      


  and wrap_logic_expr context c_value
   : logic_expr =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.logic_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `CallExpr _
      | `RefId _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LogicExpr {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_logic_propagate context c_value
   : logic_propagate =
    let f_dest_var () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.logic_propagate_f_dest_var
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      as e -> e
      | _ -> assert false
    in
    let f_call () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.logic_propagate_f_call
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_logic_propagate_call context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LogicPropagate {
        f_dest_var
          = Lazy.from_fun f_dest_var ;
        f_call
          = Lazy.from_fun f_call ;
        c_value = c_value;
        context = context
      }

      


  and wrap_logic_unify context c_value
   : logic_unify =
    let f_lhs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.logic_unify_f_lhs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      as e -> e
      | _ -> assert false
    in
    let f_rhs () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.logic_unify_f_rhs
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LogicUnify {
        f_lhs
          = Lazy.from_fun f_lhs ;
        f_rhs
          = Lazy.from_fun f_rhs ;
        c_value = c_value;
        context = context
      }

      


  and wrap_match_expr context c_value
   : match_expr =
    let f_match_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.match_expr_f_match_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    let f_branches () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.match_expr_f_branches
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_match_branch_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `MatchExpr {
        f_match_expr
          = Lazy.from_fun f_match_expr ;
        f_branches
          = Lazy.from_fun f_branches ;
        c_value = c_value;
        context = context
      }

      


  and wrap_not_expr context c_value
   : not_expr =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.not_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NotExpr {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_paren_expr context c_value
   : paren_expr =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.paren_expr_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParenExpr {
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_raise_expr context c_value
   : raise_expr =
    let f_dest_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.raise_expr_f_dest_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_type_ref context (field_c_value))
      in
         

      match node with
            
      | Some `FunctionTypeRef _
      | Some `GenericTypeRef _
      | Some `SimpleTypeRef _
      | None as e -> e
      | _ -> assert false
    in
    let f_except_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.raise_expr_f_except_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RaiseExpr {
        f_dest_type
          = Lazy.from_fun f_dest_type ;
        f_except_expr
          = Lazy.from_fun f_except_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_subscript_expr context c_value
   : subscript_expr =
    let f_prefix () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subscript_expr_f_prefix
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_null_cond () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subscript_expr_f_null_cond
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_null_cond_qualifier context (field_c_value)
      in
         

      node
    in
    let f_index () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.subscript_expr_f_index
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SubscriptExpr {
        f_prefix
          = Lazy.from_fun f_prefix ;
        f_null_cond
          = Lazy.from_fun f_null_cond ;
        f_index
          = Lazy.from_fun f_index ;
        c_value = c_value;
        context = context
      }

      


  and wrap_try_expr context c_value
   : try_expr =
    let f_try_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.try_expr_f_try_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    let f_or_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.try_expr_f_or_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_expr context (field_c_value))
      in
         

      match node with
            
      | Some `AnyOf _
      | Some `ArrayLiteral _
      | Some `BigNumLit _
      | Some `BinOp _
      | Some `BlockExpr _
      | Some `BlockStringLit _
      | Some `CallExpr _
      | Some `CastExpr _
      | Some `CharLit _
      | Some `DotExpr _
      | Some `ErrorOnNull _
      | Some `GenericInstantiation _
      | Some `IfExpr _
      | Some `Isa _
      | Some `KeepExpr _
      | Some `LambdaExpr _
      | Some `LogicAssign _
      | Some `LogicExpr _
      | Some `LogicPredicate _
      | Some `LogicPropagate _
      | Some `LogicUnify _
      | Some `MatchExpr _
      | Some `NotExpr _
      | Some `NullLit _
      | Some `NumLit _
      | Some `ParenExpr _
      | Some `PatternSingleLineStringLit _
      | Some `RaiseExpr _
      | Some `RefId _
      | Some `SingleLineStringLit _
      | Some `SubscriptExpr _
      | Some `TryExpr _
      | Some `UnOp _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TryExpr {
        f_try_expr
          = Lazy.from_fun f_try_expr ;
        f_or_expr
          = Lazy.from_fun f_or_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_un_op context c_value
   : un_op =
    let f_op () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.un_op_f_op
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_op context (field_c_value)
      in
         

      match node with
            
      | `OpMinus _
      | `OpPlus _
      as e -> e
      | _ -> assert false
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.un_op_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `UnOp {
        f_op
          = Lazy.from_fun f_op ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }

      


  and wrap_full_decl context c_value
   : full_decl =
    let f_doc () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.full_decl_f_doc
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_string_lit context (field_c_value))
      in
         

      node
    in
    let f_decl_annotations () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.full_decl_f_decl_annotations
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_decl_annotation_list context (field_c_value)
      in
         

      node
    in
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.full_decl_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_decl context (field_c_value)
      in
         

      match node with
            
      | `ClassDecl _
      | `DynVarDecl _
      | `EnumClassDecl _
      | `EnumTypeDecl _
      | `EnvSpecDecl _
      | `FieldDecl _
      | `FunDecl _
      | `GenericDecl _
      | `GenericParamTypeDecl _
      | `GrammarDecl _
      | `GrammarRuleDecl _
      | `LexerDecl _
      | `LexerFamilyDecl _
      | `StructDecl _
      | `TraitDecl _
      | `ValDecl _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FullDecl {
        f_doc
          = Lazy.from_fun f_doc ;
        f_decl_annotations
          = Lazy.from_fun f_decl_annotations ;
        f_decl
          = Lazy.from_fun f_decl ;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_list_sep context c_value
   : grammar_list_sep =
    let f_token () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_list_sep_f_token
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      node
    in
    let f_extra () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.grammar_list_sep_f_extra
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_id context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarListSep {
        f_token
          = Lazy.from_fun f_token ;
        f_extra
          = Lazy.from_fun f_extra ;
        c_value = c_value;
        context = context
      }

      


  and wrap_import context c_value
   : import =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.import_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_module_ref_id context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `Import {
        f_name
          = Lazy.from_fun f_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_langkit_root context c_value
   : langkit_root =
    let f_imports () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.langkit_root_f_imports
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_import_list context (field_c_value)
      in
         

      node
    in
    let f_decls () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.langkit_root_f_decls
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_full_decl_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LangkitRoot {
        f_imports
          = Lazy.from_fun f_imports ;
        f_decls
          = Lazy.from_fun f_decls ;
        c_value = c_value;
        context = context
      }

      


  and wrap_lexer_case_rule context c_value
   : lexer_case_rule =
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lexer_case_rule_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_grammar_expr context (field_c_value)
      in
         

      match node with
            
      | `GrammarCut _
      | `GrammarDiscard _
      | `GrammarImplicitPick _
      | `GrammarList _
      | `GrammarNull _
      | `GrammarOpt _
      | `GrammarOptError _
      | `GrammarOptErrorGroup _
      | `GrammarOptGroup _
      | `GrammarOrExpr _
      | `GrammarPick _
      | `GrammarRuleRef _
      | `GrammarSkip _
      | `GrammarStopCut _
      | `ParseNodeExpr _
      | `TokenLit _
      | `TokenNoCaseLit _
      | `TokenPatternConcat _
      | `TokenPatternLit _
      | `TokenRef _
      as e -> e
      | _ -> assert false
    in
    let f_alts () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lexer_case_rule_f_alts
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_base_lexer_case_rule_alt_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LexerCaseRule {
        f_expr
          = Lazy.from_fun f_expr ;
        f_alts
          = Lazy.from_fun f_alts ;
        c_value = c_value;
        context = context
      }

      


  and wrap_lexer_case_rule_send context c_value
   : lexer_case_rule_send =
    let f_sent () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lexer_case_rule_send_f_sent
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ref_id context (field_c_value)
      in
         

      node
    in
    let f_match_size () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.lexer_case_rule_send_f_match_size
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_num_lit context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LexerCaseRuleSend {
        f_sent
          = Lazy.from_fun f_sent ;
        f_match_size
          = Lazy.from_fun f_match_size ;
        c_value = c_value;
        context = context
      }

      

  and wrap_list_kind context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `ListKindOne _
      | `ListKindZero _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_list_kind_one context c_value
   : list_kind_one =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ListKindOne {
        c_value = c_value;
        context = context
      }

      


  and wrap_list_kind_zero context c_value
   : list_kind_zero =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ListKindZero {
        c_value = c_value;
        context = context
      }

      

  and wrap_lkt_node_base_list context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `ArgumentList _
      | `BaseLexerCaseRuleAltList _
      | `BaseMatchBranchList _
      | `BlockStringLineList _
      | `CallExprList _
      | `DeclAnnotationList _
      | `ElsifBranchList _
      | `EnumClassAltDeclList _
      | `EnumClassCaseList _
      | `EnumLitDeclList _
      | `ExprList _
      | `AnyOfList _
      | `FullDeclList _
      | `DeclBlock _
      | `GenericParamDeclList _
      | `FunParamDeclList _
      | `GrammarExprList _
      | `GrammarExprListList _
      | `ImportList _
      | `LambdaParamDeclList _
      | `LktNodeList _
      | `PatternDetailList _
      | `PatternList _
      | `RefIdList _
      | `TypeRefList _
      | `SyntheticTypeRefList _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_argument_list context c_value
   : argument_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_argument context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ArgumentList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_base_lexer_case_rule_alt_list context c_value
   : base_lexer_case_rule_alt_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_base_lexer_case_rule_alt context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BaseLexerCaseRuleAltList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_base_match_branch_list context c_value
   : base_match_branch_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_base_match_branch context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BaseMatchBranchList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_block_string_line_list context c_value
   : block_string_line_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_block_string_line context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BlockStringLineList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_call_expr_list context c_value
   : call_expr_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_call_expr context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `CallExprList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_decl_annotation_list context c_value
   : decl_annotation_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_decl_annotation context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeclAnnotationList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_elsif_branch_list context c_value
   : elsif_branch_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_elsif_branch context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ElsifBranchList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_class_alt_decl_list context c_value
   : enum_class_alt_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_enum_class_alt_decl context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumClassAltDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_class_case_list context c_value
   : enum_class_case_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_enum_class_case context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumClassCaseList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_enum_lit_decl_list context c_value
   : enum_lit_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_enum_lit_decl context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EnumLitDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_expr_list context c_value
   : expr_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_expr context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExprList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_any_of_list context c_value
   : any_of_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_expr context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AnyOfList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_full_decl_list context c_value
   : full_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_full_decl context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FullDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_decl_block context c_value
   : decl_block =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_full_decl context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DeclBlock {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_param_decl_list context c_value
   : generic_param_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_full_decl context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericParamDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_fun_param_decl_list context c_value
   : fun_param_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_fun_param_decl context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FunParamDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_expr_list context c_value
   : grammar_expr_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_grammar_expr context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarExprList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_grammar_expr_list_list context c_value
   : grammar_expr_list_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_grammar_expr_list context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GrammarExprListList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_import_list context c_value
   : import_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_import context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ImportList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_lambda_param_decl_list context c_value
   : lambda_param_decl_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_lambda_param_decl context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LambdaParamDeclList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_lkt_node_list context c_value
   : lkt_node_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_lkt_node context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `LktNodeList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_pattern_detail_list context c_value
   : pattern_detail_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_pattern_detail context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PatternDetailList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_pattern_list context c_value
   : pattern_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_pattern context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PatternList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_ref_id_list context c_value
   : ref_id_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_ref_id context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RefIdList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_type_ref_list context c_value
   : type_ref_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_type_ref context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TypeRefList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      


  and wrap_synthetic_type_ref_list context c_value
   : synthetic_type_ref_list =
    let list () =
      let c_value_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      let _ : int =
        CFunctions.lkt_node_children
          (addr c_value)
          (c_value_ptr)
      in
      let c_value = !@(!@(c_value_ptr)) in
      let length = getf c_value LktNodeArrayStruct.n in
      let items = c_value @. LktNodeArrayStruct.items in
      let f i =
        let fresh = allocate EntityStruct.c_type !@(items +@ i) in
        (* This can raise a SyntaxError, which is expected *)
        wrap_type_ref context ((!@ fresh))
      in
      let result = List.init length f in
      LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
      result
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SyntheticTypeRefList {
        list = Lazy.from_fun list;
        c_value = c_value;
        context = context
      }

      

  and wrap_null_cond_qualifier context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `NullCondQualifierAbsent _
      | `NullCondQualifierPresent _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_null_cond_qualifier_absent context c_value
   : null_cond_qualifier_absent =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullCondQualifierAbsent {
        c_value = c_value;
        context = context
      }

      


  and wrap_null_cond_qualifier_present context c_value
   : null_cond_qualifier_present =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullCondQualifierPresent {
        c_value = c_value;
        context = context
      }

      

  and wrap_op context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `OpAmp _
      | `OpAnd _
      | `OpDiv _
      | `OpEq _
      | `OpGt _
      | `OpGte _
      | `OpLogicAnd _
      | `OpLogicOr _
      | `OpLt _
      | `OpLte _
      | `OpMinus _
      | `OpMult _
      | `OpNe _
      | `OpOr _
      | `OpOrInt _
      | `OpPlus _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_op_amp context c_value
   : op_amp =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpAmp {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_and context c_value
   : op_and =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpAnd {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_div context c_value
   : op_div =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpDiv {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_eq context c_value
   : op_eq =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpEq {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_gt context c_value
   : op_gt =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpGt {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_gte context c_value
   : op_gte =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpGte {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_logic_and context c_value
   : op_logic_and =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpLogicAnd {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_logic_or context c_value
   : op_logic_or =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpLogicOr {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_lt context c_value
   : op_lt =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpLt {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_lte context c_value
   : op_lte =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpLte {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_minus context c_value
   : op_minus =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpMinus {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_mult context c_value
   : op_mult =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpMult {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_ne context c_value
   : op_ne =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpNe {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_or context c_value
   : op_or =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpOr {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_or_int context c_value
   : op_or_int =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpOrInt {
        c_value = c_value;
        context = context
      }

      


  and wrap_op_plus context c_value
   : op_plus =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OpPlus {
        c_value = c_value;
        context = context
      }

      

  and wrap_pattern context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `AnyTypePattern _
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `EllipsisPattern _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `OrPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_any_type_pattern context c_value
   : any_type_pattern =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `AnyTypePattern {
        c_value = c_value;
        context = context
      }

      


  and wrap_binding_pattern context c_value
   : binding_pattern =
    let f_decl () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.binding_pattern_f_decl
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_binding_val_decl context (field_c_value)
      in
         

      node
    in
    let f_sub_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.binding_pattern_f_sub_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_pattern context (field_c_value))
      in
         

      match node with
            
      | Some `BindingPattern _
      | Some `BoolPatternFalse _
      | Some `BoolPatternTrue _
      | Some `ExtendedPattern _
      | Some `IntegerPattern _
      | Some `ListPattern _
      | Some `NotPattern _
      | Some `NullPattern _
      | Some `ParenPattern _
      | Some `RegexPattern _
      | Some `TuplePattern _
      | Some `TypePattern _
      | None as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BindingPattern {
        f_decl
          = Lazy.from_fun f_decl ;
        f_sub_pattern
          = Lazy.from_fun f_sub_pattern ;
        c_value = c_value;
        context = context
      }

      

  and wrap_bool_pattern context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_bool_pattern_false context c_value
   : bool_pattern_false =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BoolPatternFalse {
        c_value = c_value;
        context = context
      }

      


  and wrap_bool_pattern_true context c_value
   : bool_pattern_true =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `BoolPatternTrue {
        c_value = c_value;
        context = context
      }

      


  and wrap_ellipsis_pattern context c_value
   : ellipsis_pattern =
    let f_binding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.ellipsis_pattern_f_binding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_id context (field_c_value))
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `EllipsisPattern {
        f_binding
          = Lazy.from_fun f_binding ;
        c_value = c_value;
        context = context
      }

      


  and wrap_extended_pattern context c_value
   : extended_pattern =
    let f_sub_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.extended_pattern_f_sub_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `AnyTypePattern _
      | `ParenPattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    let f_details () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.extended_pattern_f_details
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern_detail_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ExtendedPattern {
        f_sub_pattern
          = Lazy.from_fun f_sub_pattern ;
        f_details
          = Lazy.from_fun f_details ;
        c_value = c_value;
        context = context
      }

      


  and wrap_filtered_pattern context c_value
   : filtered_pattern =
    let f_sub_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.filtered_pattern_f_sub_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    let f_predicate () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.filtered_pattern_f_predicate
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FilteredPattern {
        f_sub_pattern
          = Lazy.from_fun f_sub_pattern ;
        f_predicate
          = Lazy.from_fun f_predicate ;
        c_value = c_value;
        context = context
      }

      


  and wrap_integer_pattern context c_value
   : integer_pattern =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `IntegerPattern {
        c_value = c_value;
        context = context
      }

      


  and wrap_list_pattern context c_value
   : list_pattern =
    let f_sub_patterns () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.list_pattern_f_sub_patterns
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ListPattern {
        f_sub_patterns
          = Lazy.from_fun f_sub_patterns ;
        c_value = c_value;
        context = context
      }

      


  and wrap_not_pattern context c_value
   : not_pattern =
    let f_sub_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.not_pattern_f_sub_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NotPattern {
        f_sub_pattern
          = Lazy.from_fun f_sub_pattern ;
        c_value = c_value;
        context = context
      }

      


  and wrap_null_pattern context c_value
   : null_pattern =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `NullPattern {
        c_value = c_value;
        context = context
      }

      


  and wrap_or_pattern context c_value
   : or_pattern =
    let f_left_sub_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.or_pattern_f_left_sub_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    let f_right_sub_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.or_pattern_f_right_sub_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `OrPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `OrPattern {
        f_left_sub_pattern
          = Lazy.from_fun f_left_sub_pattern ;
        f_right_sub_pattern
          = Lazy.from_fun f_right_sub_pattern ;
        c_value = c_value;
        context = context
      }

      


  and wrap_paren_pattern context c_value
   : paren_pattern =
    let f_sub_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.paren_pattern_f_sub_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `OrPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `ParenPattern {
        f_sub_pattern
          = Lazy.from_fun f_sub_pattern ;
        c_value = c_value;
        context = context
      }

      


  and wrap_regex_pattern context c_value
   : regex_pattern =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `RegexPattern {
        c_value = c_value;
        context = context
      }

      


  and wrap_tuple_pattern context c_value
   : tuple_pattern =
    let f_sub_patterns () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.tuple_pattern_f_sub_patterns
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TuplePattern {
        f_sub_patterns
          = Lazy.from_fun f_sub_patterns ;
        c_value = c_value;
        context = context
      }

      


  and wrap_type_pattern context c_value
   : type_pattern =
    let f_type_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.type_pattern_f_type_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      match node with
            
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `TypePattern {
        f_type_name
          = Lazy.from_fun f_type_name ;
        c_value = c_value;
        context = context
      }

      

  and wrap_pattern_detail context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `FieldPatternDetail _
      | `PropertyPatternDetail _
      | `SelectorPatternDetail _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_field_pattern_detail context c_value
   : field_pattern_detail =
    let f_id () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.field_pattern_detail_f_id
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_id context (field_c_value)
      in
         

      node
    in
    let f_expected_value () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.field_pattern_detail_f_expected_value
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `OrPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FieldPatternDetail {
        f_id
          = Lazy.from_fun f_id ;
        f_expected_value
          = Lazy.from_fun f_expected_value ;
        c_value = c_value;
        context = context
      }

      


  and wrap_property_pattern_detail context c_value
   : property_pattern_detail =
    let f_call () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.property_pattern_detail_f_call
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    let f_expected_value () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.property_pattern_detail_f_expected_value
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `OrPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `PropertyPatternDetail {
        f_call
          = Lazy.from_fun f_call ;
        f_expected_value
          = Lazy.from_fun f_expected_value ;
        c_value = c_value;
        context = context
      }

      


  and wrap_selector_pattern_detail context c_value
   : selector_pattern_detail =
    let f_call () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.selector_pattern_detail_f_call
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_selector_call context (field_c_value)
      in
         

      node
    in
    let f_sub_pattern () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.selector_pattern_detail_f_sub_pattern
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_pattern context (field_c_value)
      in
         

      match node with
            
      | `BindingPattern _
      | `BoolPatternFalse _
      | `BoolPatternTrue _
      | `ExtendedPattern _
      | `FilteredPattern _
      | `IntegerPattern _
      | `ListPattern _
      | `NotPattern _
      | `NullPattern _
      | `OrPattern _
      | `ParenPattern _
      | `RegexPattern _
      | `TuplePattern _
      | `TypePattern _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SelectorPatternDetail {
        f_call
          = Lazy.from_fun f_call ;
        f_sub_pattern
          = Lazy.from_fun f_sub_pattern ;
        c_value = c_value;
        context = context
      }

      


  and wrap_selector_call context c_value
   : selector_call =
    let f_quantifier () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.selector_call_f_quantifier
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_id context (field_c_value)
      in
         

      node
    in
    let f_binding () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.selector_call_f_binding
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         if is_null (getf field_c_value EntityStruct.node) then None else Some (wrap_id context (field_c_value))
      in
         

      node
    in
    let f_selector_call () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.selector_call_f_selector_call
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `ArrayLiteral _
      | `BigNumLit _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `KeepExpr _
      | `LogicExpr _
      | `LogicPredicate _
      | `MatchExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SelectorCall {
        f_quantifier
          = Lazy.from_fun f_quantifier ;
        f_binding
          = Lazy.from_fun f_binding ;
        f_selector_call
          = Lazy.from_fun f_selector_call ;
        c_value = c_value;
        context = context
      }

      

  and wrap_type_ref context c_value =
    (* This is an abstract node, call the root wrap function and filter to get
     the desired type *)
    match wrap_lkt_node context (c_value) with
      | `DefaultListTypeRef _
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ ->
          (* This should not happen if the types are correct *)
          assert false

      


  and wrap_default_list_type_ref context c_value
   : default_list_type_ref =
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `DefaultListTypeRef {
        c_value = c_value;
        context = context
      }

      


  and wrap_function_type_ref context c_value
   : function_type_ref =
    let f_param_types () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.function_type_ref_f_param_types
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref_list context (field_c_value)
      in
         

      node
    in
    let f_return_type () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.function_type_ref_f_return_type
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref context (field_c_value)
      in
         

      match node with
            
      | `FunctionTypeRef _
      | `GenericTypeRef _
      | `SimpleTypeRef _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `FunctionTypeRef {
        f_param_types
          = Lazy.from_fun f_param_types ;
        f_return_type
          = Lazy.from_fun f_return_type ;
        c_value = c_value;
        context = context
      }

      


  and wrap_generic_type_ref context c_value
   : generic_type_ref =
    let f_type_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_type_ref_f_type_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `DotExpr _
      | `RefId _
      as e -> e
      | _ -> assert false
    in
    let f_args () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.generic_type_ref_f_args
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_type_ref_list context (field_c_value)
      in
         

      node
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `GenericTypeRef {
        f_type_name
          = Lazy.from_fun f_type_name ;
        f_args
          = Lazy.from_fun f_args ;
        c_value = c_value;
        context = context
      }

      


  and wrap_simple_type_ref context c_value
   : simple_type_ref =
    let f_type_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.simple_type_ref_f_type_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `DotExpr _
      | `RefId _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `SimpleTypeRef {
        f_type_name
          = Lazy.from_fun f_type_name ;
        c_value = c_value;
        context = context
      }

      


  and wrap_var_bind context c_value
   : var_bind =
    let f_name () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.var_bind_f_name
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_ref_id context (field_c_value)
      in
         

      node
    in
    let f_expr () =
      let field_c_value = make EntityStruct.c_type in
      let _ : int = CFunctions.var_bind_f_expr
        (addr c_value)
        (addr field_c_value)
      in
      let node =
         wrap_expr context (field_c_value)
      in
         

      match node with
            
      | `AnyOf _
      | `ArrayLiteral _
      | `BigNumLit _
      | `BinOp _
      | `BlockExpr _
      | `BlockStringLit _
      | `CallExpr _
      | `CastExpr _
      | `CharLit _
      | `DotExpr _
      | `ErrorOnNull _
      | `GenericInstantiation _
      | `IfExpr _
      | `Isa _
      | `KeepExpr _
      | `LambdaExpr _
      | `LogicAssign _
      | `LogicExpr _
      | `LogicPredicate _
      | `LogicPropagate _
      | `LogicUnify _
      | `MatchExpr _
      | `NotExpr _
      | `NullLit _
      | `NumLit _
      | `ParenExpr _
      | `PatternSingleLineStringLit _
      | `RaiseExpr _
      | `RefId _
      | `SingleLineStringLit _
      | `SubscriptExpr _
      | `TryExpr _
      | `UnOp _
      as e -> e
      | _ -> assert false
    in
    if is_null (getf c_value EntityStruct.node) then
      raise (SyntaxError "null node")
    else
      `VarBind {
        f_name
          = Lazy.from_fun f_name ;
        f_expr
          = Lazy.from_fun f_expr ;
        c_value = c_value;
        context = context
      }



   
  and wrap_entity_info ?(dec_ref=true) c_value = {
    rebindings = (getf c_value EntityInfoStruct.rebindings);
    from_rebound = (getf c_value EntityInfoStruct.from_rebound);
  }



   


and wrap_analysis_unit context c_value
   : analysis_unit = {
 c_value=c_value;
 context=context;
}

module Entity = struct
  type t = entity

  let info value =
    wrap_entity_info (getf value EntityStruct.info)

  let compare e1 e2 =
    let open Stdlib in
    let compare_node =
      compare (getf e1 EntityStruct.node) (getf e2 EntityStruct.node)
    in
    if compare_node = 0 then
      compare
        (getf (getf e1 EntityStruct.info) EntityInfoStruct.rebindings)
        (getf (getf e2 EntityStruct.info) EntityInfoStruct.rebindings)
    else
      compare_node

  let equal e1 e2 =
    compare e1 e2 = 0

  let hash e =
    Hashtbl.hash
      ( getf e EntityStruct.node
      , getf (getf e EntityStruct.info) EntityInfoStruct.rebindings )
end

module AnalysisUnit = struct
  type t = analysis_unit

  let root (unit : t) =
    let c_value = make EntityStruct.c_type in
    AnalysisUnitStruct.unit_root
      (unwrap_analysis_unit (unit))
      (addr c_value);
    if is_null (getf c_value EntityStruct.node) then None else Some (wrap_lkt_node unit.context (c_value))

  let diagnostics (unit : t) =
    let c_unit = unwrap_analysis_unit (unit) in
    let length = AnalysisUnitStruct.unit_diagnostic_count c_unit in
    let f i =
      let diag = allocate_n Diagnostic.c_type ~count:1 in
      let _ : int = AnalysisUnitStruct.unit_diagnostic c_unit i diag in
      !@ diag
    in
    List.init length f

  let filename (unit : t) =
    unwrap_str( AnalysisUnitStruct.unit_filename
      (unwrap_analysis_unit (unit)))

  let reparse ?charset:(charset="") ?buffer (unit : t) =
    match buffer with
    | None ->
        ignore
          (AnalysisUnitStruct.unit_reparse_from_file unit.c_value charset)
    | Some buffer ->
        ignore (AnalysisUnitStruct.unit_reparse_from_buffer unit.c_value
          charset buffer (Unsigned.Size_t.of_int (String.length buffer)))

  let first_token (unit : t) =
    let c_unit = unwrap_analysis_unit (unit) in
    let result_ptr = allocate_n Token.c_type ~count:1 in
    AnalysisUnitStruct.unit_first_token c_unit result_ptr ;
    Token.wrap (!@ result_ptr)

  let last_token (unit : t) =
    let c_unit = unwrap_analysis_unit (unit) in
    let result_ptr = allocate_n Token.c_type ~count:1 in
    AnalysisUnitStruct.unit_last_token c_unit result_ptr ;
    Token.wrap (!@ result_ptr)

  let token_count (unit : t) =
    AnalysisUnitStruct.unit_token_count
      (unwrap_analysis_unit (unit))

  let trivia_count (unit : t) =
    AnalysisUnitStruct.unit_trivia_count
      (unwrap_analysis_unit (unit))

  
  let fold_tokens f init node =
    match first_token node, last_token node with
    | Some tok_start, Some tok_end ->
        let rec aux acc tok_curr =
          let new_acc = f acc tok_curr in
          if Token.equal tok_curr tok_end then
            new_acc
          else (
            match Token.next tok_curr with
            | Some tok_next ->
                aux new_acc tok_next
            | None ->
                new_acc )
        in
        aux init tok_start
    | _ ->
        init

  let iter_tokens f node =
    match first_token node, last_token node with
    | Some tok_start, Some tok_end ->
        let rec aux tok_curr =
          f tok_curr;
          if not (Token.equal tok_curr tok_end) then (
            match Token.next tok_curr with
            | Some tok_next ->
                aux tok_next
            | None ->
                () )
        in
        aux tok_start
    | _ ->
        ()

  let map_tokens f node =
    match first_token node, last_token node with
    | Some tok_start, Some tok_end ->
        let rec aux tok_curr =
          let value = f tok_curr in
          if Token.equal tok_curr tok_end then
            [value]
          else (
            match Token.next tok_curr with
            | Some tok_next ->
                value :: aux tok_next
            | None ->
                [value] )
        in
        aux tok_start
    | _ ->
        []

  let tokens node =
    map_tokens (fun x -> x) node

end

module AnalysisContext = struct
  type t = analysis_context

  let create
    ?charset:(charset="")
    ?with_trivia:(with_trivia=true)
    ?tab_stop:(tab_stop=8)
    ?unit_provider:(unit_provider=UnitProvider.null)
    ?file_reader () : t =
    if tab_stop < 1 then
      raise (Invalid_argument "Invalid tab_stop (positive integer expected)") ;
    let c_context = AnalysisContextStruct.allocate_analysis_context () in
    AnalysisContextStruct.initialize_analysis_context
      c_context
      charset
      (match file_reader with Some v -> !@v | None -> null)
      (!@unit_provider)
      Ctypes.null (* TODO: bind the event handlers API to OCaml *)
      with_trivia
      tab_stop ;
    { c_value= c_context
      ; unit_provider= unit_provider }

  let get_from_file
    ?charset:(charset="")
    ?reparse:(reparse=false)
    ?grammar_rule:(grammar_rule=default_grammar_rule)
    (ctx : t)
    filename : AnalysisUnit.t =

    wrap_analysis_unit ctx (AnalysisContextStruct.get_analysis_unit_from_file ctx.c_value filename charset reparse grammar_rule)

  let get_from_buffer
    ?charset:(charset="")
    ?grammar_rule:(grammar_rule=default_grammar_rule)
    (ctx : t)
    filename
    buffer : AnalysisUnit.t =

    wrap_analysis_unit ctx (AnalysisContextStruct.get_analysis_unit_from_buffer ctx.c_value filename charset buffer (Unsigned.Size_t.of_int (String.length buffer)) grammar_rule)
end

   
module CompleteItem : sig
   
  
  type t = {
      declaration :
         decl option;
  }


  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> CompleteItemStruct.t structure
    -> t

  val unwrap : t -> CompleteItemStruct.t structure


end = struct
   
  
  type t = {
      declaration :
         decl option;
  }


  
   
  let wrap
    ?(dec_ref=true)
    context
    c_value = {
    declaration = if is_null (getf (getf c_value CompleteItemStruct.declaration) EntityStruct.node) then None else Some (wrap_decl context ((getf c_value CompleteItemStruct.declaration)));
  }


  
   

  let unwrap value =
    let c_value = make CompleteItemStruct.c_type in
    let field_c_value =
      match value.declaration with Some n -> unwrap_lkt_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      CompleteItemStruct.declaration
      field_c_value;
    c_value



end

   
module DecodedCharValue : sig
   
  
  type t = {
      value :
         string;
      has_error :
         bool;
      error_sloc :
         Sloc.t;
      error_message :
         string;
  }


  val wrap : ?dec_ref:bool -> DecodedCharValueStruct.t structure -> t

  val unwrap : t -> DecodedCharValueStruct.t structure


end = struct
   
  
  type t = {
      value :
         string;
      has_error :
         bool;
      error_sloc :
         Sloc.t;
      error_message :
         string;
  }


  
   
  let wrap ?(dec_ref=true) c_value = {
    value = (getf c_value DecodedCharValueStruct.value);
    has_error = (getf c_value DecodedCharValueStruct.has_error);
    error_sloc = (getf c_value DecodedCharValueStruct.error_sloc);
    error_message = StringType.wrap ~dec_ref:dec_ref ((getf c_value DecodedCharValueStruct.error_message));
  }


  
   

  let unwrap value =
    let c_value = make DecodedCharValueStruct.c_type in
    let field_c_value =
      value.value
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DecodedCharValueStruct.value
      field_c_value;
    let field_c_value =
      value.has_error
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DecodedCharValueStruct.has_error
      field_c_value;
    let field_c_value =
      value.error_sloc
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DecodedCharValueStruct.error_sloc
      field_c_value;
    let field_c_value =
      StringType.unwrap (value.error_message)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DecodedCharValueStruct.error_message
      field_c_value;
    c_value



end

   
module DecodedStringValue : sig
   
  
  type t = {
      value :
         string;
      has_error :
         bool;
      error_sloc :
         Sloc.t;
      error_message :
         string;
  }


  val wrap : ?dec_ref:bool -> DecodedStringValueStruct.t structure -> t

  val unwrap : t -> DecodedStringValueStruct.t structure


end = struct
   
  
  type t = {
      value :
         string;
      has_error :
         bool;
      error_sloc :
         Sloc.t;
      error_message :
         string;
  }


  
   
  let wrap ?(dec_ref=true) c_value = {
    value = StringType.wrap ~dec_ref:dec_ref ((getf c_value DecodedStringValueStruct.value));
    has_error = (getf c_value DecodedStringValueStruct.has_error);
    error_sloc = (getf c_value DecodedStringValueStruct.error_sloc);
    error_message = StringType.wrap ~dec_ref:dec_ref ((getf c_value DecodedStringValueStruct.error_message));
  }


  
   

  let unwrap value =
    let c_value = make DecodedStringValueStruct.c_type in
    let field_c_value =
      StringType.unwrap (value.value)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DecodedStringValueStruct.value
      field_c_value;
    let field_c_value =
      value.has_error
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DecodedStringValueStruct.has_error
      field_c_value;
    let field_c_value =
      value.error_sloc
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DecodedStringValueStruct.error_sloc
      field_c_value;
    let field_c_value =
      StringType.unwrap (value.error_message)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      DecodedStringValueStruct.error_message
      field_c_value;
    c_value



end

   
module LogicContext : sig
   
  
  type t = {
      ref_node :
         lkt_node option;
      decl_node :
         lkt_node option;
  }


  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> LogicContextStruct.t structure
    -> t

  val unwrap : t -> LogicContextStruct.t structure


end = struct
   
  
  type t = {
      ref_node :
         lkt_node option;
      decl_node :
         lkt_node option;
  }


  
   
  let wrap
    ?(dec_ref=true)
    context
    c_value = {
    ref_node = if is_null (getf (getf c_value LogicContextStruct.ref_node) EntityStruct.node) then None else Some (wrap_lkt_node context ((getf c_value LogicContextStruct.ref_node)));
    decl_node = if is_null (getf (getf c_value LogicContextStruct.decl_node) EntityStruct.node) then None else Some (wrap_lkt_node context ((getf c_value LogicContextStruct.decl_node)));
  }


  
   

  let unwrap value =
    let c_value = make LogicContextStruct.c_type in
    let field_c_value =
      match value.ref_node with Some n -> unwrap_lkt_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      LogicContextStruct.ref_node
      field_c_value;
    let field_c_value =
      match value.decl_node with Some n -> unwrap_lkt_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      LogicContextStruct.decl_node
      field_c_value;
    c_value



end

   
module RefResult : sig
   
  
  type t = {
      ref_ :
         ref_id option;
  }


  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> RefResultStruct.t structure
    -> t

  val unwrap : t -> RefResultStruct.t structure


end = struct
   
  
  type t = {
      ref_ :
         ref_id option;
  }


  
   
  let wrap
    ?(dec_ref=true)
    context
    c_value = {
    ref_ = if is_null (getf (getf c_value RefResultStruct.ref_) EntityStruct.node) then None else Some (wrap_ref_id context ((getf c_value RefResultStruct.ref_)));
  }


  
   

  let unwrap value =
    let c_value = make RefResultStruct.c_type in
    let field_c_value =
      match value.ref_ with Some n -> unwrap_lkt_node (n) | None -> make EntityStruct.c_type
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      RefResultStruct.ref_
      field_c_value;
    c_value



end

   
module LktNodeArray : sig
   
  type t = lkt_node list

  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> LktNodeArrayStruct.t structure ptr
    -> t

  val unwrap : t -> LktNodeArrayStruct.t structure ptr

end = struct
   
  type t = lkt_node list

  let wrap ?(dec_ref=true) (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value LktNodeArrayStruct.n in
    let items = c_value @. LktNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      (* Do not dec_ref the item here since this is the responsability of
         the array *)
      wrap_lkt_node context (!@ fresh)
    in
    let result = List.init length f in
    if dec_ref then LktNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = LktNodeArrayStruct.create (List.length value) in
    let items = result |-> LktNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_lkt_node (v)
    in
    List.iteri f value;
    result


end

   
module LogicContextArray : sig
   
  type t = LogicContext.t list

  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> LogicContextArrayStruct.t structure ptr
    -> t

  val unwrap : t -> LogicContextArrayStruct.t structure ptr

end = struct
   
  type t = LogicContext.t list

  let wrap ?(dec_ref=true) (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value LogicContextArrayStruct.n in
    let items = c_value @. LogicContextArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate LogicContextStruct.c_type (!@ (items +@ i))
      in
      (* Do not dec_ref the item here since this is the responsability of
         the array *)
      LogicContext.wrap context (!@ fresh)
    in
    let result = List.init length f in
    if dec_ref then LogicContextArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = LogicContextArrayStruct.create (List.length value) in
    let items = result |-> LogicContextArrayStruct.items in
    let f i v =
      items +@ i <-@
        LogicContext.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module SolverDiagnostic : sig
   
  
  type t = {
      message_template :
         string;
      args :
         lkt_node list;
      location :
         BareNode.t;
      contexts :
         LogicContext.t list;
      round :
         int;
  }


  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> SolverDiagnosticStruct.t structure
    -> t

  val unwrap : t -> SolverDiagnosticStruct.t structure


end = struct
   
  
  type t = {
      message_template :
         string;
      args :
         lkt_node list;
      location :
         BareNode.t;
      contexts :
         LogicContext.t list;
      round :
         int;
  }


  
   
  let wrap
    ?(dec_ref=true)
    context
    c_value = {
    message_template = StringType.wrap ~dec_ref:dec_ref ((getf c_value SolverDiagnosticStruct.message_template));
    args = LktNodeArray.wrap ~dec_ref:dec_ref context ((getf c_value SolverDiagnosticStruct.args));
    location = (getf c_value SolverDiagnosticStruct.location);
    contexts = LogicContextArray.wrap ~dec_ref:dec_ref context ((getf c_value SolverDiagnosticStruct.contexts));
    round = (getf c_value SolverDiagnosticStruct.round);
  }


  
   

  let unwrap value =
    let c_value = make SolverDiagnosticStruct.c_type in
    let field_c_value =
      StringType.unwrap (value.message_template)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SolverDiagnosticStruct.message_template
      field_c_value;
    let field_c_value =
      LktNodeArray.unwrap (value.args)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SolverDiagnosticStruct.args
      field_c_value;
    let field_c_value =
      value.location
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SolverDiagnosticStruct.location
      field_c_value;
    let field_c_value =
      LogicContextArray.unwrap (value.contexts)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SolverDiagnosticStruct.contexts
      field_c_value;
    let field_c_value =
      value.round
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SolverDiagnosticStruct.round
      field_c_value;
    c_value



end

   
module SolverDiagnosticArray : sig
   
  type t = SolverDiagnostic.t list

  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> SolverDiagnosticArrayStruct.t structure ptr
    -> t

  val unwrap : t -> SolverDiagnosticArrayStruct.t structure ptr

end = struct
   
  type t = SolverDiagnostic.t list

  let wrap ?(dec_ref=true) (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value SolverDiagnosticArrayStruct.n in
    let items = c_value @. SolverDiagnosticArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate SolverDiagnosticStruct.c_type (!@ (items +@ i))
      in
      (* Do not dec_ref the item here since this is the responsability of
         the array *)
      SolverDiagnostic.wrap ~dec_ref:false context (!@ fresh)
    in
    let result = List.init length f in
    if dec_ref then SolverDiagnosticArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = SolverDiagnosticArrayStruct.create (List.length value) in
    let items = result |-> SolverDiagnosticArrayStruct.items in
    let f i v =
      items +@ i <-@
        SolverDiagnostic.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module SolverResult : sig
   
  
  type t = {
      success :
         bool;
      diagnostics :
         SolverDiagnostic.t list;
  }


  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> SolverResultStruct.t structure
    -> t

  val unwrap : t -> SolverResultStruct.t structure


end = struct
   
  
  type t = {
      success :
         bool;
      diagnostics :
         SolverDiagnostic.t list;
  }


  
   
  let wrap
    ?(dec_ref=true)
    context
    c_value = {
    success = (getf c_value SolverResultStruct.success);
    diagnostics = SolverDiagnosticArray.wrap ~dec_ref:dec_ref context ((getf c_value SolverResultStruct.diagnostics));
  }


  
   

  let unwrap value =
    let c_value = make SolverResultStruct.c_type in
    let field_c_value =
      value.success
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SolverResultStruct.success
      field_c_value;
    let field_c_value =
      SolverDiagnosticArray.unwrap (value.diagnostics)
    in
    add_gc_link ~from:c_value ~to_:field_c_value;
    setf c_value
      SolverResultStruct.diagnostics
      field_c_value;
    c_value



end

   
module CompleteItemArray : sig
   
  type t = CompleteItem.t list

  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> CompleteItemArrayStruct.t structure ptr
    -> t

  val unwrap : t -> CompleteItemArrayStruct.t structure ptr

end = struct
   
  type t = CompleteItem.t list

  let wrap ?(dec_ref=true) (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value CompleteItemArrayStruct.n in
    let items = c_value @. CompleteItemArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate CompleteItemStruct.c_type (!@ (items +@ i))
      in
      (* Do not dec_ref the item here since this is the responsability of
         the array *)
      CompleteItem.wrap context (!@ fresh)
    in
    let result = List.init length f in
    if dec_ref then CompleteItemArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = CompleteItemArrayStruct.create (List.length value) in
    let items = result |-> CompleteItemArrayStruct.items in
    let f i v =
      items +@ i <-@
        CompleteItem.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module DefIdArray : sig
   
  type t = def_id list

  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> LktNodeArrayStruct.t structure ptr
    -> t

  val unwrap : t -> LktNodeArrayStruct.t structure ptr

end = struct
   
  type t = def_id list

  let wrap ?(dec_ref=true) (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value LktNodeArrayStruct.n in
    let items = c_value @. LktNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      (* Do not dec_ref the item here since this is the responsability of
         the array *)
      wrap_def_id context (!@ fresh)
    in
    let result = List.init length f in
    if dec_ref then LktNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = LktNodeArrayStruct.create (List.length value) in
    let items = result |-> LktNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_lkt_node (v)
    in
    List.iteri f value;
    result


end

   
module FunDeclArray : sig
   
  type t = fun_decl list

  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> LktNodeArrayStruct.t structure ptr
    -> t

  val unwrap : t -> LktNodeArrayStruct.t structure ptr

end = struct
   
  type t = fun_decl list

  let wrap ?(dec_ref=true) (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value LktNodeArrayStruct.n in
    let items = c_value @. LktNodeArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate EntityStruct.c_type (!@ (items +@ i))
      in
      (* Do not dec_ref the item here since this is the responsability of
         the array *)
      wrap_fun_decl context (!@ fresh)
    in
    let result = List.init length f in
    if dec_ref then LktNodeArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = LktNodeArrayStruct.create (List.length value) in
    let items = result |-> LktNodeArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_lkt_node (v)
    in
    List.iteri f value;
    result


end

   
module RefResultArray : sig
   
  type t = RefResult.t list

  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> RefResultArrayStruct.t structure ptr
    -> t

  val unwrap : t -> RefResultArrayStruct.t structure ptr

end = struct
   
  type t = RefResult.t list

  let wrap ?(dec_ref=true) (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value RefResultArrayStruct.n in
    let items = c_value @. RefResultArrayStruct.items in
    let f i =
      (* we want to allocate a fresh value for a record, otherwize, the c value
       * will still point to the memory at array location *)
      let fresh =
        allocate RefResultStruct.c_type (!@ (items +@ i))
      in
      (* Do not dec_ref the item here since this is the responsability of
         the array *)
      RefResult.wrap context (!@ fresh)
    in
    let result = List.init length f in
    if dec_ref then RefResultArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = RefResultArrayStruct.create (List.length value) in
    let items = result |-> RefResultArrayStruct.items in
    let f i v =
      items +@ i <-@
        RefResult.unwrap (v)
    in
    List.iteri f value;
    result


end

   
module AnalysisUnitArray : sig
   
  type t = analysis_unit list

  val wrap :
       ?dec_ref:bool
    -> analysis_context
    -> AnalysisUnitArrayStruct.t structure ptr
    -> t

  val unwrap : t -> AnalysisUnitArrayStruct.t structure ptr

end = struct
   
  type t = analysis_unit list

  let wrap ?(dec_ref=true) (context : analysis_context) c_value_ptr =
    let c_value = !@ c_value_ptr in
    let length = getf c_value AnalysisUnitArrayStruct.n in
    let items = c_value @. AnalysisUnitArrayStruct.items in
    let f i =
      (* Do not dec_ref the item here since this is the responsability of
         the array *)
      wrap_analysis_unit context (!@ (items +@ i))
    in
    let result = List.init length f in
    if dec_ref then AnalysisUnitArrayStruct.dec_ref c_value_ptr;
    result

  let unwrap value =
    let result = AnalysisUnitArrayStruct.create (List.length value) in
    let items = result |-> AnalysisUnitArrayStruct.items in
    let f i v =
      items +@ i <-@
        unwrap_analysis_unit (v)
    in
    List.iteri f value;
    result


end


let context node =
  (* Given any node, extract the context field *)
  match (node :> lkt_node) with
  | `Argument fields -> fields.context
  | `LexerCaseRuleCondAlt fields -> fields.context
  | `LexerCaseRuleDefaultAlt fields -> fields.context
  | `MatchBranch fields -> fields.context
  | `PatternMatchBranch fields -> fields.context
  | `BlockExprClause fields -> fields.context
  | `BlockStringLine fields -> fields.context
  | `ClassQualifierAbsent fields -> fields.context
  | `ClassQualifierPresent fields -> fields.context
  | `GrammarRuleDecl fields -> fields.context
  | `SyntheticLexerDecl fields -> fields.context
  | `NodeDecl fields -> fields.context
  | `SelfDecl fields -> fields.context
  | `BindingValDecl fields -> fields.context
  | `EnumLitDecl fields -> fields.context
  | `FieldDecl fields -> fields.context
  | `FunParamDecl fields -> fields.context
  | `LambdaParamDecl fields -> fields.context
  | `DynVarDecl fields -> fields.context
  | `MatchValDecl fields -> fields.context
  | `ValDecl fields -> fields.context
  | `FunDecl fields -> fields.context
  | `EnvSpecDecl fields -> fields.context
  | `ErrorDecl fields -> fields.context
  | `GenericDecl fields -> fields.context
  | `GrammarDecl fields -> fields.context
  | `LexerDecl fields -> fields.context
  | `LexerFamilyDecl fields -> fields.context
  | `SynthFunDecl fields -> fields.context
  | `SynthParamDecl fields -> fields.context
  | `AnyTypeDecl fields -> fields.context
  | `EnumClassAltDecl fields -> fields.context
  | `FunctionType fields -> fields.context
  | `GenericParamTypeDecl fields -> fields.context
  | `ClassDecl fields -> fields.context
  | `EnumClassDecl fields -> fields.context
  | `EnumTypeDecl fields -> fields.context
  | `StructDecl fields -> fields.context
  | `TraitDecl fields -> fields.context
  | `DeclAnnotation fields -> fields.context
  | `DeclAnnotationArgs fields -> fields.context
  | `DynEnvWrapper fields -> fields.context
  | `ElsifBranch fields -> fields.context
  | `EnumClassCase fields -> fields.context
  | `ExcludesNullAbsent fields -> fields.context
  | `ExcludesNullPresent fields -> fields.context
  | `AnyOf fields -> fields.context
  | `ArrayLiteral fields -> fields.context
  | `CallExpr fields -> fields.context
  | `LogicPredicate fields -> fields.context
  | `LogicPropagateCall fields -> fields.context
  | `BinOp fields -> fields.context
  | `BlockExpr fields -> fields.context
  | `CastExpr fields -> fields.context
  | `DotExpr fields -> fields.context
  | `ErrorOnNull fields -> fields.context
  | `GenericInstantiation fields -> fields.context
  | `GrammarCut fields -> fields.context
  | `GrammarDiscard fields -> fields.context
  | `GrammarDontSkip fields -> fields.context
  | `GrammarList fields -> fields.context
  | `GrammarNull fields -> fields.context
  | `GrammarOpt fields -> fields.context
  | `GrammarOptError fields -> fields.context
  | `GrammarOptErrorGroup fields -> fields.context
  | `GrammarOptGroup fields -> fields.context
  | `GrammarOrExpr fields -> fields.context
  | `GrammarPick fields -> fields.context
  | `GrammarImplicitPick fields -> fields.context
  | `GrammarPredicate fields -> fields.context
  | `GrammarRuleRef fields -> fields.context
  | `GrammarSkip fields -> fields.context
  | `GrammarStopCut fields -> fields.context
  | `ParseNodeExpr fields -> fields.context
  | `TokenLit fields -> fields.context
  | `TokenNoCaseLit fields -> fields.context
  | `TokenPatternConcat fields -> fields.context
  | `TokenPatternLit fields -> fields.context
  | `TokenRef fields -> fields.context
  | `Id fields -> fields.context
  | `DefId fields -> fields.context
  | `ModuleRefId fields -> fields.context
  | `RefId fields -> fields.context
  | `IfExpr fields -> fields.context
  | `Isa fields -> fields.context
  | `KeepExpr fields -> fields.context
  | `LambdaExpr fields -> fields.context
  | `BigNumLit fields -> fields.context
  | `CharLit fields -> fields.context
  | `NullLit fields -> fields.context
  | `NumLit fields -> fields.context
  | `BlockStringLit fields -> fields.context
  | `SingleLineStringLit fields -> fields.context
  | `PatternSingleLineStringLit fields -> fields.context
  | `LogicAssign fields -> fields.context
  | `LogicExpr fields -> fields.context
  | `LogicPropagate fields -> fields.context
  | `LogicUnify fields -> fields.context
  | `MatchExpr fields -> fields.context
  | `NotExpr fields -> fields.context
  | `ParenExpr fields -> fields.context
  | `RaiseExpr fields -> fields.context
  | `SubscriptExpr fields -> fields.context
  | `TryExpr fields -> fields.context
  | `UnOp fields -> fields.context
  | `FullDecl fields -> fields.context
  | `GrammarListSep fields -> fields.context
  | `Import fields -> fields.context
  | `LangkitRoot fields -> fields.context
  | `LexerCaseRule fields -> fields.context
  | `LexerCaseRuleSend fields -> fields.context
  | `ListKindOne fields -> fields.context
  | `ListKindZero fields -> fields.context
  | `ArgumentList fields -> fields.context
  | `BaseLexerCaseRuleAltList fields -> fields.context
  | `BaseMatchBranchList fields -> fields.context
  | `BlockStringLineList fields -> fields.context
  | `CallExprList fields -> fields.context
  | `DeclAnnotationList fields -> fields.context
  | `ElsifBranchList fields -> fields.context
  | `EnumClassAltDeclList fields -> fields.context
  | `EnumClassCaseList fields -> fields.context
  | `EnumLitDeclList fields -> fields.context
  | `ExprList fields -> fields.context
  | `AnyOfList fields -> fields.context
  | `FullDeclList fields -> fields.context
  | `DeclBlock fields -> fields.context
  | `GenericParamDeclList fields -> fields.context
  | `FunParamDeclList fields -> fields.context
  | `GrammarExprList fields -> fields.context
  | `GrammarExprListList fields -> fields.context
  | `ImportList fields -> fields.context
  | `LambdaParamDeclList fields -> fields.context
  | `LktNodeList fields -> fields.context
  | `PatternDetailList fields -> fields.context
  | `PatternList fields -> fields.context
  | `RefIdList fields -> fields.context
  | `TypeRefList fields -> fields.context
  | `SyntheticTypeRefList fields -> fields.context
  | `NullCondQualifierAbsent fields -> fields.context
  | `NullCondQualifierPresent fields -> fields.context
  | `OpAmp fields -> fields.context
  | `OpAnd fields -> fields.context
  | `OpDiv fields -> fields.context
  | `OpEq fields -> fields.context
  | `OpGt fields -> fields.context
  | `OpGte fields -> fields.context
  | `OpLogicAnd fields -> fields.context
  | `OpLogicOr fields -> fields.context
  | `OpLt fields -> fields.context
  | `OpLte fields -> fields.context
  | `OpMinus fields -> fields.context
  | `OpMult fields -> fields.context
  | `OpNe fields -> fields.context
  | `OpOr fields -> fields.context
  | `OpOrInt fields -> fields.context
  | `OpPlus fields -> fields.context
  | `AnyTypePattern fields -> fields.context
  | `BindingPattern fields -> fields.context
  | `BoolPatternFalse fields -> fields.context
  | `BoolPatternTrue fields -> fields.context
  | `EllipsisPattern fields -> fields.context
  | `ExtendedPattern fields -> fields.context
  | `FilteredPattern fields -> fields.context
  | `IntegerPattern fields -> fields.context
  | `ListPattern fields -> fields.context
  | `NotPattern fields -> fields.context
  | `NullPattern fields -> fields.context
  | `OrPattern fields -> fields.context
  | `ParenPattern fields -> fields.context
  | `RegexPattern fields -> fields.context
  | `TuplePattern fields -> fields.context
  | `TypePattern fields -> fields.context
  | `FieldPatternDetail fields -> fields.context
  | `PropertyPatternDetail fields -> fields.context
  | `SelectorPatternDetail fields -> fields.context
  | `SelectorCall fields -> fields.context
  | `DefaultListTypeRef fields -> fields.context
  | `FunctionTypeRef fields -> fields.context
  | `GenericTypeRef fields -> fields.context
  | `SimpleTypeRef fields -> fields.context
  | `VarBind fields -> fields.context

type _ node =
  | LktNode :
      lkt_node node
  | Argument :
      argument node
  | BaseLexerCaseRuleAlt :
      base_lexer_case_rule_alt node
  | LexerCaseRuleCondAlt :
      lexer_case_rule_cond_alt node
  | LexerCaseRuleDefaultAlt :
      lexer_case_rule_default_alt node
  | BaseMatchBranch :
      base_match_branch node
  | MatchBranch :
      match_branch node
  | PatternMatchBranch :
      pattern_match_branch node
  | BlockExprClause :
      block_expr_clause node
  | BlockStringLine :
      block_string_line node
  | ClassQualifier :
      class_qualifier node
  | ClassQualifierAbsent :
      class_qualifier_absent node
  | ClassQualifierPresent :
      class_qualifier_present node
  | Decl :
      decl node
  | BaseGrammarRuleDecl :
      base_grammar_rule_decl node
  | GrammarRuleDecl :
      grammar_rule_decl node
  | SyntheticLexerDecl :
      synthetic_lexer_decl node
  | BaseValDecl :
      base_val_decl node
  | NodeDecl :
      node_decl node
  | SelfDecl :
      self_decl node
  | UserValDecl :
      user_val_decl node
  | BindingValDecl :
      binding_val_decl node
  | EnumLitDecl :
      enum_lit_decl node
  | ExplicitlyTypedDecl :
      explicitly_typed_decl node
  | ComponentDecl :
      component_decl node
  | FieldDecl :
      field_decl node
  | FunParamDecl :
      fun_param_decl node
  | LambdaParamDecl :
      lambda_param_decl node
  | DynVarDecl :
      dyn_var_decl node
  | MatchValDecl :
      match_val_decl node
  | ValDecl :
      val_decl node
  | FunDecl :
      fun_decl node
  | EnvSpecDecl :
      env_spec_decl node
  | ErrorDecl :
      error_decl node
  | GenericDecl :
      generic_decl node
  | GrammarDecl :
      grammar_decl node
  | LexerDecl :
      lexer_decl node
  | LexerFamilyDecl :
      lexer_family_decl node
  | SynthFunDecl :
      synth_fun_decl node
  | SynthParamDecl :
      synth_param_decl node
  | TypeDecl :
      type_decl node
  | AnyTypeDecl :
      any_type_decl node
  | EnumClassAltDecl :
      enum_class_alt_decl node
  | FunctionType :
      function_type node
  | GenericParamTypeDecl :
      generic_param_type_decl node
  | NamedTypeDecl :
      named_type_decl node
  | BasicClassDecl :
      basic_class_decl node
  | ClassDecl :
      class_decl node
  | EnumClassDecl :
      enum_class_decl node
  | EnumTypeDecl :
      enum_type_decl node
  | StructDecl :
      struct_decl node
  | TraitDecl :
      trait_decl node
  | DeclAnnotation :
      decl_annotation node
  | DeclAnnotationArgs :
      decl_annotation_args node
  | DynEnvWrapper :
      dyn_env_wrapper node
  | ElsifBranch :
      elsif_branch node
  | EnumClassCase :
      enum_class_case node
  | ExcludesNull :
      excludes_null node
  | ExcludesNullAbsent :
      excludes_null_absent node
  | ExcludesNullPresent :
      excludes_null_present node
  | Expr :
      expr node
  | AnyOf :
      any_of node
  | ArrayLiteral :
      array_literal node
  | BaseCallExpr :
      base_call_expr node
  | CallExpr :
      call_expr node
  | LogicCallExpr :
      logic_call_expr node
  | LogicPredicate :
      logic_predicate node
  | LogicPropagateCall :
      logic_propagate_call node
  | BinOp :
      bin_op node
  | BlockExpr :
      block_expr node
  | CastExpr :
      cast_expr node
  | DotExpr :
      dot_expr node
  | ErrorOnNull :
      error_on_null node
  | GenericInstantiation :
      generic_instantiation node
  | GrammarExpr :
      grammar_expr node
  | GrammarCut :
      grammar_cut node
  | GrammarDiscard :
      grammar_discard node
  | GrammarDontSkip :
      grammar_dont_skip node
  | GrammarList :
      grammar_list node
  | GrammarNull :
      grammar_null node
  | GrammarOpt :
      grammar_opt node
  | GrammarOptError :
      grammar_opt_error node
  | GrammarOptErrorGroup :
      grammar_opt_error_group node
  | GrammarOptGroup :
      grammar_opt_group node
  | GrammarOrExpr :
      grammar_or_expr node
  | GrammarPick :
      grammar_pick node
  | GrammarImplicitPick :
      grammar_implicit_pick node
  | GrammarPredicate :
      grammar_predicate node
  | GrammarRuleRef :
      grammar_rule_ref node
  | GrammarSkip :
      grammar_skip node
  | GrammarStopCut :
      grammar_stop_cut node
  | ParseNodeExpr :
      parse_node_expr node
  | TokenLit :
      token_lit node
  | TokenNoCaseLit :
      token_no_case_lit node
  | TokenPatternConcat :
      token_pattern_concat node
  | TokenPatternLit :
      token_pattern_lit node
  | TokenRef :
      token_ref node
  | Id :
      id node
  | DefId :
      def_id node
  | ModuleRefId :
      module_ref_id node
  | RefId :
      ref_id node
  | IfExpr :
      if_expr node
  | Isa :
      isa node
  | KeepExpr :
      keep_expr node
  | LambdaExpr :
      lambda_expr node
  | Lit :
      lit node
  | BigNumLit :
      big_num_lit node
  | CharLit :
      char_lit node
  | NullLit :
      null_lit node
  | NumLit :
      num_lit node
  | StringLit :
      string_lit node
  | BlockStringLit :
      block_string_lit node
  | SingleLineStringLit :
      single_line_string_lit node
  | PatternSingleLineStringLit :
      pattern_single_line_string_lit node
  | LogicAssign :
      logic_assign node
  | LogicExpr :
      logic_expr node
  | LogicPropagate :
      logic_propagate node
  | LogicUnify :
      logic_unify node
  | MatchExpr :
      match_expr node
  | NotExpr :
      not_expr node
  | ParenExpr :
      paren_expr node
  | RaiseExpr :
      raise_expr node
  | SubscriptExpr :
      subscript_expr node
  | TryExpr :
      try_expr node
  | UnOp :
      un_op node
  | FullDecl :
      full_decl node
  | GrammarListSep :
      grammar_list_sep node
  | Import :
      import node
  | LangkitRoot :
      langkit_root node
  | LexerCaseRule :
      lexer_case_rule node
  | LexerCaseRuleSend :
      lexer_case_rule_send node
  | ListKind :
      list_kind node
  | ListKindOne :
      list_kind_one node
  | ListKindZero :
      list_kind_zero node
  | LktNodeBaseList :
      lkt_node_base_list node
  | ArgumentList :
      argument_list node
  | BaseLexerCaseRuleAltList :
      base_lexer_case_rule_alt_list node
  | BaseMatchBranchList :
      base_match_branch_list node
  | BlockStringLineList :
      block_string_line_list node
  | CallExprList :
      call_expr_list node
  | DeclAnnotationList :
      decl_annotation_list node
  | ElsifBranchList :
      elsif_branch_list node
  | EnumClassAltDeclList :
      enum_class_alt_decl_list node
  | EnumClassCaseList :
      enum_class_case_list node
  | EnumLitDeclList :
      enum_lit_decl_list node
  | ExprList :
      expr_list node
  | AnyOfList :
      any_of_list node
  | FullDeclList :
      full_decl_list node
  | DeclBlock :
      decl_block node
  | GenericParamDeclList :
      generic_param_decl_list node
  | FunParamDeclList :
      fun_param_decl_list node
  | GrammarExprList :
      grammar_expr_list node
  | GrammarExprListList :
      grammar_expr_list_list node
  | ImportList :
      import_list node
  | LambdaParamDeclList :
      lambda_param_decl_list node
  | LktNodeList :
      lkt_node_list node
  | PatternDetailList :
      pattern_detail_list node
  | PatternList :
      pattern_list node
  | RefIdList :
      ref_id_list node
  | TypeRefList :
      type_ref_list node
  | SyntheticTypeRefList :
      synthetic_type_ref_list node
  | NullCondQualifier :
      null_cond_qualifier node
  | NullCondQualifierAbsent :
      null_cond_qualifier_absent node
  | NullCondQualifierPresent :
      null_cond_qualifier_present node
  | Op :
      op node
  | OpAmp :
      op_amp node
  | OpAnd :
      op_and node
  | OpDiv :
      op_div node
  | OpEq :
      op_eq node
  | OpGt :
      op_gt node
  | OpGte :
      op_gte node
  | OpLogicAnd :
      op_logic_and node
  | OpLogicOr :
      op_logic_or node
  | OpLt :
      op_lt node
  | OpLte :
      op_lte node
  | OpMinus :
      op_minus node
  | OpMult :
      op_mult node
  | OpNe :
      op_ne node
  | OpOr :
      op_or node
  | OpOrInt :
      op_or_int node
  | OpPlus :
      op_plus node
  | Pattern :
      pattern node
  | AnyTypePattern :
      any_type_pattern node
  | BindingPattern :
      binding_pattern node
  | BoolPattern :
      bool_pattern node
  | BoolPatternFalse :
      bool_pattern_false node
  | BoolPatternTrue :
      bool_pattern_true node
  | EllipsisPattern :
      ellipsis_pattern node
  | ExtendedPattern :
      extended_pattern node
  | FilteredPattern :
      filtered_pattern node
  | IntegerPattern :
      integer_pattern node
  | ListPattern :
      list_pattern node
  | NotPattern :
      not_pattern node
  | NullPattern :
      null_pattern node
  | OrPattern :
      or_pattern node
  | ParenPattern :
      paren_pattern node
  | RegexPattern :
      regex_pattern node
  | TuplePattern :
      tuple_pattern node
  | TypePattern :
      type_pattern node
  | PatternDetail :
      pattern_detail node
  | FieldPatternDetail :
      field_pattern_detail node
  | PropertyPatternDetail :
      property_pattern_detail node
  | SelectorPatternDetail :
      selector_pattern_detail node
  | SelectorCall :
      selector_call node
  | TypeRef :
      type_ref node
  | DefaultListTypeRef :
      default_list_type_ref node
  | FunctionTypeRef :
      function_type_ref node
  | GenericTypeRef :
      generic_type_ref node
  | SimpleTypeRef :
      simple_type_ref node
  | VarBind :
      var_bind node

module VarBind = struct
  type t =
    [
      | `VarBind of
          var_bind_fields
    ]

  type fields = var_bind_fields =
    
  {
         
    f_name: ref_id
    Lazy.t;
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> var_bind) with
    | `VarBind fields ->
        Lazy.force fields.f_name
  let f_expr node =
    match (node :> var_bind) with
    | `VarBind fields ->
        Lazy.force fields.f_expr



end

module SimpleTypeRef = struct
  type t =
    [
      | `SimpleTypeRef of
          simple_type_ref_fields
    ]

  type fields = simple_type_ref_fields =
    
  {
         
    f_type_name: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_type_name node =
    match (node :> simple_type_ref) with
    | `SimpleTypeRef fields ->
        Lazy.force fields.f_type_name



end

module GenericTypeRef = struct
  type t =
    [
      | `GenericTypeRef of
          generic_type_ref_fields
    ]

  type fields = generic_type_ref_fields =
    
  {
         
    f_type_name: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
         
    f_args: type_ref_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_type_name node =
    match (node :> generic_type_ref) with
    | `GenericTypeRef fields ->
        Lazy.force fields.f_type_name
  let f_args node =
    match (node :> generic_type_ref) with
    | `GenericTypeRef fields ->
        Lazy.force fields.f_args



end

module FunctionTypeRef = struct
  type t =
    [
      | `FunctionTypeRef of
          function_type_ref_fields
    ]

  type fields = function_type_ref_fields =
    
  {
         
    f_param_types: type_ref_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_param_types node =
    match (node :> function_type_ref) with
    | `FunctionTypeRef fields ->
        Lazy.force fields.f_param_types
  let f_return_type node =
    match (node :> function_type_ref) with
    | `FunctionTypeRef fields ->
        Lazy.force fields.f_return_type



end

module DefaultListTypeRef = struct
  type t =
    [
      | `DefaultListTypeRef of
          default_list_type_ref_fields
    ]

  type fields = default_list_type_ref_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module TypeRef = struct
  type t =
    [
      | DefaultListTypeRef.t
      | FunctionTypeRef.t
      | GenericTypeRef.t
      | SimpleTypeRef.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_referenced_decl
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.type_ref_p_referenced_decl
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))





end

module SelectorCall = struct
  type t =
    [
      | `SelectorCall of
          selector_call_fields
    ]

  type fields = selector_call_fields =
    
  {
         
    f_quantifier: id
    Lazy.t;
         
    f_binding: id
    option
    Lazy.t;
         
    f_selector_call: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_quantifier node =
    match (node :> selector_call) with
    | `SelectorCall fields ->
        Lazy.force fields.f_quantifier
  let f_binding node =
    match (node :> selector_call) with
    | `SelectorCall fields ->
        Lazy.force fields.f_binding
  let f_selector_call node =
    match (node :> selector_call) with
    | `SelectorCall fields ->
        Lazy.force fields.f_selector_call



end

module SelectorPatternDetail = struct
  type t =
    [
      | `SelectorPatternDetail of
          selector_pattern_detail_fields
    ]

  type fields = selector_pattern_detail_fields =
    
  {
         
    f_call: selector_call
    Lazy.t;
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_call node =
    match (node :> selector_pattern_detail) with
    | `SelectorPatternDetail fields ->
        Lazy.force fields.f_call
  let f_sub_pattern node =
    match (node :> selector_pattern_detail) with
    | `SelectorPatternDetail fields ->
        Lazy.force fields.f_sub_pattern



end

module PropertyPatternDetail = struct
  type t =
    [
      | `PropertyPatternDetail of
          property_pattern_detail_fields
    ]

  type fields = property_pattern_detail_fields =
    
  {
         
    f_call: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_expected_value: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_call node =
    match (node :> property_pattern_detail) with
    | `PropertyPatternDetail fields ->
        Lazy.force fields.f_call
  let f_expected_value node =
    match (node :> property_pattern_detail) with
    | `PropertyPatternDetail fields ->
        Lazy.force fields.f_expected_value



end

module FieldPatternDetail = struct
  type t =
    [
      | `FieldPatternDetail of
          field_pattern_detail_fields
    ]

  type fields = field_pattern_detail_fields =
    
  {
         
    f_id: id
    Lazy.t;
         
    f_expected_value: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_id node =
    match (node :> field_pattern_detail) with
    | `FieldPatternDetail fields ->
        Lazy.force fields.f_id
  let f_expected_value node =
    match (node :> field_pattern_detail) with
    | `FieldPatternDetail fields ->
        Lazy.force fields.f_expected_value



end

module PatternDetail = struct
  type t =
    [
      | FieldPatternDetail.t
      | PropertyPatternDetail.t
      | SelectorPatternDetail.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module TypePattern = struct
  type t =
    [
      | `TypePattern of
          type_pattern_fields
    ]

  type fields = type_pattern_fields =
    
  {
         
    f_type_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_type_name node =
    match (node :> type_pattern) with
    | `TypePattern fields ->
        Lazy.force fields.f_type_name



end

module TuplePattern = struct
  type t =
    [
      | `TuplePattern of
          tuple_pattern_fields
    ]

  type fields = tuple_pattern_fields =
    
  {
         
    f_sub_patterns: pattern_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_sub_patterns node =
    match (node :> tuple_pattern) with
    | `TuplePattern fields ->
        Lazy.force fields.f_sub_patterns



end

module RegexPattern = struct
  type t =
    [
      | `RegexPattern of
          regex_pattern_fields
    ]

  type fields = regex_pattern_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ParenPattern = struct
  type t =
    [
      | `ParenPattern of
          paren_pattern_fields
    ]

  type fields = paren_pattern_fields =
    
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_sub_pattern node =
    match (node :> paren_pattern) with
    | `ParenPattern fields ->
        Lazy.force fields.f_sub_pattern



end

module OrPattern = struct
  type t =
    [
      | `OrPattern of
          or_pattern_fields
    ]

  type fields = or_pattern_fields =
    
  {
         
    f_left_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_right_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_left_sub_pattern node =
    match (node :> or_pattern) with
    | `OrPattern fields ->
        Lazy.force fields.f_left_sub_pattern
  let f_right_sub_pattern node =
    match (node :> or_pattern) with
    | `OrPattern fields ->
        Lazy.force fields.f_right_sub_pattern



end

module NullPattern = struct
  type t =
    [
      | `NullPattern of
          null_pattern_fields
    ]

  type fields = null_pattern_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module NotPattern = struct
  type t =
    [
      | `NotPattern of
          not_pattern_fields
    ]

  type fields = not_pattern_fields =
    
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_sub_pattern node =
    match (node :> not_pattern) with
    | `NotPattern fields ->
        Lazy.force fields.f_sub_pattern



end

module ListPattern = struct
  type t =
    [
      | `ListPattern of
          list_pattern_fields
    ]

  type fields = list_pattern_fields =
    
  {
         
    f_sub_patterns: pattern_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_sub_patterns node =
    match (node :> list_pattern) with
    | `ListPattern fields ->
        Lazy.force fields.f_sub_patterns



end

module IntegerPattern = struct
  type t =
    [
      | `IntegerPattern of
          integer_pattern_fields
    ]

  type fields = integer_pattern_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module FilteredPattern = struct
  type t =
    [
      | `FilteredPattern of
          filtered_pattern_fields
    ]

  type fields = filtered_pattern_fields =
    
  {
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_predicate: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_sub_pattern node =
    match (node :> filtered_pattern) with
    | `FilteredPattern fields ->
        Lazy.force fields.f_sub_pattern
  let f_predicate node =
    match (node :> filtered_pattern) with
    | `FilteredPattern fields ->
        Lazy.force fields.f_predicate



end

module ExtendedPattern = struct
  type t =
    [
      | `ExtendedPattern of
          extended_pattern_fields
    ]

  type fields = extended_pattern_fields =
    
  {
         
    f_sub_pattern: [
      | `AnyTypePattern
          of any_type_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
         
    f_details: pattern_detail_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_sub_pattern node =
    match (node :> extended_pattern) with
    | `ExtendedPattern fields ->
        Lazy.force fields.f_sub_pattern
  let f_details node =
    match (node :> extended_pattern) with
    | `ExtendedPattern fields ->
        Lazy.force fields.f_details



end

module EllipsisPattern = struct
  type t =
    [
      | `EllipsisPattern of
          ellipsis_pattern_fields
    ]

  type fields = ellipsis_pattern_fields =
    
  {
         
    f_binding: id
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_binding node =
    match (node :> ellipsis_pattern) with
    | `EllipsisPattern fields ->
        Lazy.force fields.f_binding



end

module BoolPatternTrue = struct
  type t =
    [
      | `BoolPatternTrue of
          bool_pattern_true_fields
    ]

  type fields = bool_pattern_true_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BoolPatternFalse = struct
  type t =
    [
      | `BoolPatternFalse of
          bool_pattern_false_fields
    ]

  type fields = bool_pattern_false_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BoolPattern = struct
  type t =
    [
      | BoolPatternFalse.t
      | BoolPatternTrue.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BindingPattern = struct
  type t =
    [
      | `BindingPattern of
          binding_pattern_fields
    ]

  type fields = binding_pattern_fields =
    
  {
         
    f_decl: binding_val_decl
    Lazy.t;
         
    f_sub_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_decl node =
    match (node :> binding_pattern) with
    | `BindingPattern fields ->
        Lazy.force fields.f_decl
  let f_sub_pattern node =
    match (node :> binding_pattern) with
    | `BindingPattern fields ->
        Lazy.force fields.f_sub_pattern



end

module AnyTypePattern = struct
  type t =
    [
      | `AnyTypePattern of
          any_type_pattern_fields
    ]

  type fields = any_type_pattern_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module Pattern = struct
  type t =
    [
      | AnyTypePattern.t
      | BindingPattern.t
      | BoolPattern.t
      | EllipsisPattern.t
      | ExtendedPattern.t
      | FilteredPattern.t
      | IntegerPattern.t
      | ListPattern.t
      | NotPattern.t
      | NullPattern.t
      | OrPattern.t
      | ParenPattern.t
      | RegexPattern.t
      | TuplePattern.t
      | TypePattern.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpPlus = struct
  type t =
    [
      | `OpPlus of
          op_plus_fields
    ]

  type fields = op_plus_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpOrInt = struct
  type t =
    [
      | `OpOrInt of
          op_or_int_fields
    ]

  type fields = op_or_int_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpOr = struct
  type t =
    [
      | `OpOr of
          op_or_fields
    ]

  type fields = op_or_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpNe = struct
  type t =
    [
      | `OpNe of
          op_ne_fields
    ]

  type fields = op_ne_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpMult = struct
  type t =
    [
      | `OpMult of
          op_mult_fields
    ]

  type fields = op_mult_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpMinus = struct
  type t =
    [
      | `OpMinus of
          op_minus_fields
    ]

  type fields = op_minus_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpLte = struct
  type t =
    [
      | `OpLte of
          op_lte_fields
    ]

  type fields = op_lte_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpLt = struct
  type t =
    [
      | `OpLt of
          op_lt_fields
    ]

  type fields = op_lt_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpLogicOr = struct
  type t =
    [
      | `OpLogicOr of
          op_logic_or_fields
    ]

  type fields = op_logic_or_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpLogicAnd = struct
  type t =
    [
      | `OpLogicAnd of
          op_logic_and_fields
    ]

  type fields = op_logic_and_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpGte = struct
  type t =
    [
      | `OpGte of
          op_gte_fields
    ]

  type fields = op_gte_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpGt = struct
  type t =
    [
      | `OpGt of
          op_gt_fields
    ]

  type fields = op_gt_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpEq = struct
  type t =
    [
      | `OpEq of
          op_eq_fields
    ]

  type fields = op_eq_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpDiv = struct
  type t =
    [
      | `OpDiv of
          op_div_fields
    ]

  type fields = op_div_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpAnd = struct
  type t =
    [
      | `OpAnd of
          op_and_fields
    ]

  type fields = op_and_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module OpAmp = struct
  type t =
    [
      | `OpAmp of
          op_amp_fields
    ]

  type fields = op_amp_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module Op = struct
  type t =
    [
      | OpAmp.t
      | OpAnd.t
      | OpDiv.t
      | OpEq.t
      | OpGt.t
      | OpGte.t
      | OpLogicAnd.t
      | OpLogicOr.t
      | OpLt.t
      | OpLte.t
      | OpMinus.t
      | OpMult.t
      | OpNe.t
      | OpOr.t
      | OpOrInt.t
      | OpPlus.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module NullCondQualifierPresent = struct
  type t =
    [
      | `NullCondQualifierPresent of
          null_cond_qualifier_present_fields
    ]

  type fields = null_cond_qualifier_present_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module NullCondQualifierAbsent = struct
  type t =
    [
      | `NullCondQualifierAbsent of
          null_cond_qualifier_absent_fields
    ]

  type fields = null_cond_qualifier_absent_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module NullCondQualifier = struct
  type t =
    [
      | NullCondQualifierAbsent.t
      | NullCondQualifierPresent.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_as_bool
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.null_cond_qualifier_p_as_bool
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr





end

module SyntheticTypeRefList = struct
  type t =
    [
      | `SyntheticTypeRefList of
          synthetic_type_ref_list_fields
    ]

  type fields = synthetic_type_ref_list_fields =
    
  {
    list : type_ref list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> synthetic_type_ref_list) with
    | `SyntheticTypeRefList fields ->
        Lazy.force fields.list



end

module TypeRefList = struct
  type t =
    [
      | `TypeRefList of
          type_ref_list_fields
      | `SyntheticTypeRefList of
          synthetic_type_ref_list_fields
    ]

  type fields = type_ref_list_fields =
    
  {
    list : type_ref list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> type_ref_list) with
    | `TypeRefList fields ->
        Lazy.force fields.list
    | `SyntheticTypeRefList fields ->
        Lazy.force fields.list



end

module RefIdList = struct
  type t =
    [
      | `RefIdList of
          ref_id_list_fields
    ]

  type fields = ref_id_list_fields =
    
  {
    list : ref_id list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> ref_id_list) with
    | `RefIdList fields ->
        Lazy.force fields.list



end

module PatternList = struct
  type t =
    [
      | `PatternList of
          pattern_list_fields
    ]

  type fields = pattern_list_fields =
    
  {
    list : pattern list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> pattern_list) with
    | `PatternList fields ->
        Lazy.force fields.list



end

module PatternDetailList = struct
  type t =
    [
      | `PatternDetailList of
          pattern_detail_list_fields
    ]

  type fields = pattern_detail_list_fields =
    
  {
    list : pattern_detail list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> pattern_detail_list) with
    | `PatternDetailList fields ->
        Lazy.force fields.list



end

module LktNodeList = struct
  type t =
    [
      | `LktNodeList of
          lkt_node_list_fields
    ]

  type fields = lkt_node_list_fields =
    
  {
    list : lkt_node list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> lkt_node_list) with
    | `LktNodeList fields ->
        Lazy.force fields.list



end

module LambdaParamDeclList = struct
  type t =
    [
      | `LambdaParamDeclList of
          lambda_param_decl_list_fields
    ]

  type fields = lambda_param_decl_list_fields =
    
  {
    list : lambda_param_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> lambda_param_decl_list) with
    | `LambdaParamDeclList fields ->
        Lazy.force fields.list



end

module ImportList = struct
  type t =
    [
      | `ImportList of
          import_list_fields
    ]

  type fields = import_list_fields =
    
  {
    list : import list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> import_list) with
    | `ImportList fields ->
        Lazy.force fields.list



end

module GrammarExprListList = struct
  type t =
    [
      | `GrammarExprListList of
          grammar_expr_list_list_fields
    ]

  type fields = grammar_expr_list_list_fields =
    
  {
    list : grammar_expr_list list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> grammar_expr_list_list) with
    | `GrammarExprListList fields ->
        Lazy.force fields.list



end

module GrammarExprList = struct
  type t =
    [
      | `GrammarExprList of
          grammar_expr_list_fields
    ]

  type fields = grammar_expr_list_fields =
    
  {
    list : grammar_expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> grammar_expr_list) with
    | `GrammarExprList fields ->
        Lazy.force fields.list



end

module FunParamDeclList = struct
  type t =
    [
      | `FunParamDeclList of
          fun_param_decl_list_fields
    ]

  type fields = fun_param_decl_list_fields =
    
  {
    list : fun_param_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> fun_param_decl_list) with
    | `FunParamDeclList fields ->
        Lazy.force fields.list



end

module GenericParamDeclList = struct
  type t =
    [
      | `GenericParamDeclList of
          generic_param_decl_list_fields
    ]

  type fields = generic_param_decl_list_fields =
    
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> generic_param_decl_list) with
    | `GenericParamDeclList fields ->
        Lazy.force fields.list



end

module DeclBlock = struct
  type t =
    [
      | `DeclBlock of
          decl_block_fields
    ]

  type fields = decl_block_fields =
    
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> decl_block) with
    | `DeclBlock fields ->
        Lazy.force fields.list



end

module FullDeclList = struct
  type t =
    [
      | `FullDeclList of
          full_decl_list_fields
      | `DeclBlock of
          decl_block_fields
      | `GenericParamDeclList of
          generic_param_decl_list_fields
    ]

  type fields = full_decl_list_fields =
    
  {
    list : full_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> full_decl_list) with
    | `FullDeclList fields ->
        Lazy.force fields.list
    | `DeclBlock fields ->
        Lazy.force fields.list
    | `GenericParamDeclList fields ->
        Lazy.force fields.list



end

module AnyOfList = struct
  type t =
    [
      | `AnyOfList of
          any_of_list_fields
    ]

  type fields = any_of_list_fields =
    
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> any_of_list) with
    | `AnyOfList fields ->
        Lazy.force fields.list



end

module ExprList = struct
  type t =
    [
      | `ExprList of
          expr_list_fields
      | `AnyOfList of
          any_of_list_fields
    ]

  type fields = expr_list_fields =
    
  {
    list : expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> expr_list) with
    | `ExprList fields ->
        Lazy.force fields.list
    | `AnyOfList fields ->
        Lazy.force fields.list



end

module EnumLitDeclList = struct
  type t =
    [
      | `EnumLitDeclList of
          enum_lit_decl_list_fields
    ]

  type fields = enum_lit_decl_list_fields =
    
  {
    list : enum_lit_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> enum_lit_decl_list) with
    | `EnumLitDeclList fields ->
        Lazy.force fields.list



end

module EnumClassCaseList = struct
  type t =
    [
      | `EnumClassCaseList of
          enum_class_case_list_fields
    ]

  type fields = enum_class_case_list_fields =
    
  {
    list : enum_class_case list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> enum_class_case_list) with
    | `EnumClassCaseList fields ->
        Lazy.force fields.list



end

module EnumClassAltDeclList = struct
  type t =
    [
      | `EnumClassAltDeclList of
          enum_class_alt_decl_list_fields
    ]

  type fields = enum_class_alt_decl_list_fields =
    
  {
    list : enum_class_alt_decl list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> enum_class_alt_decl_list) with
    | `EnumClassAltDeclList fields ->
        Lazy.force fields.list



end

module ElsifBranchList = struct
  type t =
    [
      | `ElsifBranchList of
          elsif_branch_list_fields
    ]

  type fields = elsif_branch_list_fields =
    
  {
    list : elsif_branch list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> elsif_branch_list) with
    | `ElsifBranchList fields ->
        Lazy.force fields.list



end

module DeclAnnotationList = struct
  type t =
    [
      | `DeclAnnotationList of
          decl_annotation_list_fields
    ]

  type fields = decl_annotation_list_fields =
    
  {
    list : decl_annotation list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> decl_annotation_list) with
    | `DeclAnnotationList fields ->
        Lazy.force fields.list



end

module CallExprList = struct
  type t =
    [
      | `CallExprList of
          call_expr_list_fields
    ]

  type fields = call_expr_list_fields =
    
  {
    list : call_expr list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> call_expr_list) with
    | `CallExprList fields ->
        Lazy.force fields.list



end

module BlockStringLineList = struct
  type t =
    [
      | `BlockStringLineList of
          block_string_line_list_fields
    ]

  type fields = block_string_line_list_fields =
    
  {
    list : block_string_line list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> block_string_line_list) with
    | `BlockStringLineList fields ->
        Lazy.force fields.list



end

module BaseMatchBranchList = struct
  type t =
    [
      | `BaseMatchBranchList of
          base_match_branch_list_fields
    ]

  type fields = base_match_branch_list_fields =
    
  {
    list : base_match_branch list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> base_match_branch_list) with
    | `BaseMatchBranchList fields ->
        Lazy.force fields.list



end

module BaseLexerCaseRuleAltList = struct
  type t =
    [
      | `BaseLexerCaseRuleAltList of
          base_lexer_case_rule_alt_list_fields
    ]

  type fields = base_lexer_case_rule_alt_list_fields =
    
  {
    list : base_lexer_case_rule_alt list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> base_lexer_case_rule_alt_list) with
    | `BaseLexerCaseRuleAltList fields ->
        Lazy.force fields.list



end

module ArgumentList = struct
  type t =
    [
      | `ArgumentList of
          argument_list_fields
    ]

  type fields = argument_list_fields =
    
  {
    list : argument list Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))



  let f_list node =
    match (node :> argument_list) with
    | `ArgumentList fields ->
        Lazy.force fields.list



end

module LktNodeBaseList = struct
  type t =
    [
      | ArgumentList.t
      | BaseLexerCaseRuleAltList.t
      | BaseMatchBranchList.t
      | BlockStringLineList.t
      | CallExprList.t
      | DeclAnnotationList.t
      | ElsifBranchList.t
      | EnumClassAltDeclList.t
      | EnumClassCaseList.t
      | EnumLitDeclList.t
      | ExprList.t
      | FullDeclList.t
      | FunParamDeclList.t
      | GrammarExprList.t
      | GrammarExprListList.t
      | ImportList.t
      | LambdaParamDeclList.t
      | LktNodeList.t
      | PatternDetailList.t
      | PatternList.t
      | RefIdList.t
      | TypeRefList.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ListKindZero = struct
  type t =
    [
      | `ListKindZero of
          list_kind_zero_fields
    ]

  type fields = list_kind_zero_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ListKindOne = struct
  type t =
    [
      | `ListKindOne of
          list_kind_one_fields
    ]

  type fields = list_kind_one_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ListKind = struct
  type t =
    [
      | ListKindOne.t
      | ListKindZero.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module LexerCaseRuleSend = struct
  type t =
    [
      | `LexerCaseRuleSend of
          lexer_case_rule_send_fields
    ]

  type fields = lexer_case_rule_send_fields =
    
  {
         
    f_sent: ref_id
    Lazy.t;
         
    f_match_size: num_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_sent node =
    match (node :> lexer_case_rule_send) with
    | `LexerCaseRuleSend fields ->
        Lazy.force fields.f_sent
  let f_match_size node =
    match (node :> lexer_case_rule_send) with
    | `LexerCaseRuleSend fields ->
        Lazy.force fields.f_match_size



end

module LexerCaseRule = struct
  type t =
    [
      | `LexerCaseRule of
          lexer_case_rule_fields
    ]

  type fields = lexer_case_rule_fields =
    
  {
         
    f_expr: [
      | `GrammarCut
          of grammar_cut_fields
      | `GrammarDiscard
          of grammar_discard_fields
      | `GrammarImplicitPick
          of grammar_implicit_pick_fields
      | `GrammarList
          of grammar_list_fields
      | `GrammarNull
          of grammar_null_fields
      | `GrammarOpt
          of grammar_opt_fields
      | `GrammarOptError
          of grammar_opt_error_fields
      | `GrammarOptErrorGroup
          of grammar_opt_error_group_fields
      | `GrammarOptGroup
          of grammar_opt_group_fields
      | `GrammarOrExpr
          of grammar_or_expr_fields
      | `GrammarPick
          of grammar_pick_fields
      | `GrammarRuleRef
          of grammar_rule_ref_fields
      | `GrammarSkip
          of grammar_skip_fields
      | `GrammarStopCut
          of grammar_stop_cut_fields
      | `ParseNodeExpr
          of parse_node_expr_fields
      | `TokenLit
          of token_lit_fields
      | `TokenNoCaseLit
          of token_no_case_lit_fields
      | `TokenPatternConcat
          of token_pattern_concat_fields
      | `TokenPatternLit
          of token_pattern_lit_fields
      | `TokenRef
          of token_ref_fields
    ]
    Lazy.t;
         
    f_alts: base_lexer_case_rule_alt_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> lexer_case_rule) with
    | `LexerCaseRule fields ->
        Lazy.force fields.f_expr
  let f_alts node =
    match (node :> lexer_case_rule) with
    | `LexerCaseRule fields ->
        Lazy.force fields.f_alts



end

module LangkitRoot = struct
  type t =
    [
      | `LangkitRoot of
          langkit_root_fields
    ]

  type fields = langkit_root_fields =
    
  {
         
    f_imports: import_list
    Lazy.t;
         
    f_decls: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_fetch_prelude
    (node)
    =
      let result_ptr =
        allocate_n AnalysisUnitStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.langkit_root_p_fetch_prelude
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      wrap_analysis_unit (context node) (!@ result_ptr)


  let f_imports node =
    match (node :> langkit_root) with
    | `LangkitRoot fields ->
        Lazy.force fields.f_imports
  let f_decls node =
    match (node :> langkit_root) with
    | `LangkitRoot fields ->
        Lazy.force fields.f_decls



end

module Import = struct
  type t =
    [
      | `Import of
          import_fields
    ]

  type fields = import_fields =
    
  {
         
    f_name: module_ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_referenced_unit
    (node)
    =
      let result_ptr =
        allocate_n AnalysisUnitStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.import_p_referenced_unit
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      wrap_analysis_unit (context node) (!@ result_ptr)


  let f_name node =
    match (node :> import) with
    | `Import fields ->
        Lazy.force fields.f_name



end

module GrammarListSep = struct
  type t =
    [
      | `GrammarListSep of
          grammar_list_sep_fields
    ]

  type fields = grammar_list_sep_fields =
    
  {
         
    f_token: grammar_expr
    Lazy.t;
         
    f_extra: id
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_token node =
    match (node :> grammar_list_sep) with
    | `GrammarListSep fields ->
        Lazy.force fields.f_token
  let f_extra node =
    match (node :> grammar_list_sep) with
    | `GrammarListSep fields ->
        Lazy.force fields.f_extra



end

module FullDecl = struct
  type t =
    [
      | `FullDecl of
          full_decl_fields
    ]

  type fields = full_decl_fields =
    
  {
         
    f_doc: string_lit
    option
    Lazy.t;
         
    f_decl_annotations: decl_annotation_list
    Lazy.t;
         
    f_decl: [
      | `ClassDecl
          of class_decl_fields
      | `DynVarDecl
          of dyn_var_decl_fields
      | `EnumClassDecl
          of enum_class_decl_fields
      | `EnumTypeDecl
          of enum_type_decl_fields
      | `EnvSpecDecl
          of env_spec_decl_fields
      | `FieldDecl
          of field_decl_fields
      | `FunDecl
          of fun_decl_fields
      | `GenericDecl
          of generic_decl_fields
      | `GenericParamTypeDecl
          of generic_param_type_decl_fields
      | `GrammarDecl
          of grammar_decl_fields
      | `GrammarRuleDecl
          of grammar_rule_decl_fields
      | `LexerDecl
          of lexer_decl_fields
      | `LexerFamilyDecl
          of lexer_family_decl_fields
      | `StructDecl
          of struct_decl_fields
      | `TraitDecl
          of trait_decl_fields
      | `ValDecl
          of val_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_has_annotation
    (node)
    (name)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_name =
            
        allocate Symbol.c_type (Symbol.unwrap (context node).c_value (name))
      in
      let _ : int =
        CFunctions.full_decl_p_has_annotation
          (addr (unwrap_lkt_node (node)))
          c_name
          (result_ptr)
      in
         
      !@ result_ptr


  let f_doc node =
    match (node :> full_decl) with
    | `FullDecl fields ->
        Lazy.force fields.f_doc
  let f_decl_annotations node =
    match (node :> full_decl) with
    | `FullDecl fields ->
        Lazy.force fields.f_decl_annotations
  let f_decl node =
    match (node :> full_decl) with
    | `FullDecl fields ->
        Lazy.force fields.f_decl



end

module UnOp = struct
  type t =
    [
      | `UnOp of
          un_op_fields
    ]

  type fields = un_op_fields =
    
  {
         
    f_op: [
      | `OpMinus
          of op_minus_fields
      | `OpPlus
          of op_plus_fields
    ]
    Lazy.t;
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_op node =
    match (node :> un_op) with
    | `UnOp fields ->
        Lazy.force fields.f_op
  let f_expr node =
    match (node :> un_op) with
    | `UnOp fields ->
        Lazy.force fields.f_expr



end

module TryExpr = struct
  type t =
    [
      | `TryExpr of
          try_expr_fields
    ]

  type fields = try_expr_fields =
    
  {
         
    f_try_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_or_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_try_expr node =
    match (node :> try_expr) with
    | `TryExpr fields ->
        Lazy.force fields.f_try_expr
  let f_or_expr node =
    match (node :> try_expr) with
    | `TryExpr fields ->
        Lazy.force fields.f_or_expr



end

module SubscriptExpr = struct
  type t =
    [
      | `SubscriptExpr of
          subscript_expr_fields
    ]

  type fields = subscript_expr_fields =
    
  {
         
    f_prefix: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_index: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_prefix node =
    match (node :> subscript_expr) with
    | `SubscriptExpr fields ->
        Lazy.force fields.f_prefix
  let f_null_cond node =
    match (node :> subscript_expr) with
    | `SubscriptExpr fields ->
        Lazy.force fields.f_null_cond
  let f_index node =
    match (node :> subscript_expr) with
    | `SubscriptExpr fields ->
        Lazy.force fields.f_index



end

module RaiseExpr = struct
  type t =
    [
      | `RaiseExpr of
          raise_expr_fields
    ]

  type fields = raise_expr_fields =
    
  {
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
         
    f_except_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_dest_type node =
    match (node :> raise_expr) with
    | `RaiseExpr fields ->
        Lazy.force fields.f_dest_type
  let f_except_expr node =
    match (node :> raise_expr) with
    | `RaiseExpr fields ->
        Lazy.force fields.f_except_expr



end

module ParenExpr = struct
  type t =
    [
      | `ParenExpr of
          paren_expr_fields
    ]

  type fields = paren_expr_fields =
    
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> paren_expr) with
    | `ParenExpr fields ->
        Lazy.force fields.f_expr



end

module NotExpr = struct
  type t =
    [
      | `NotExpr of
          not_expr_fields
    ]

  type fields = not_expr_fields =
    
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> not_expr) with
    | `NotExpr fields ->
        Lazy.force fields.f_expr



end

module MatchExpr = struct
  type t =
    [
      | `MatchExpr of
          match_expr_fields
    ]

  type fields = match_expr_fields =
    
  {
         
    f_match_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_branches: base_match_branch_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_match_expr node =
    match (node :> match_expr) with
    | `MatchExpr fields ->
        Lazy.force fields.f_match_expr
  let f_branches node =
    match (node :> match_expr) with
    | `MatchExpr fields ->
        Lazy.force fields.f_branches



end

module LogicUnify = struct
  type t =
    [
      | `LogicUnify of
          logic_unify_fields
    ]

  type fields = logic_unify_fields =
    
  {
         
    f_lhs: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_rhs: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_lhs node =
    match (node :> logic_unify) with
    | `LogicUnify fields ->
        Lazy.force fields.f_lhs
  let f_rhs node =
    match (node :> logic_unify) with
    | `LogicUnify fields ->
        Lazy.force fields.f_rhs



end

module LogicPropagate = struct
  type t =
    [
      | `LogicPropagate of
          logic_propagate_fields
    ]

  type fields = logic_propagate_fields =
    
  {
         
    f_dest_var: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_call: logic_propagate_call
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_dest_var node =
    match (node :> logic_propagate) with
    | `LogicPropagate fields ->
        Lazy.force fields.f_dest_var
  let f_call node =
    match (node :> logic_propagate) with
    | `LogicPropagate fields ->
        Lazy.force fields.f_call



end

module LogicExpr = struct
  type t =
    [
      | `LogicExpr of
          logic_expr_fields
    ]

  type fields = logic_expr_fields =
    
  {
         
    f_expr: [
      | `CallExpr
          of call_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> logic_expr) with
    | `LogicExpr fields ->
        Lazy.force fields.f_expr



end

module LogicAssign = struct
  type t =
    [
      | `LogicAssign of
          logic_assign_fields
    ]

  type fields = logic_assign_fields =
    
  {
         
    f_dest_var: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_value: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_dest_var node =
    match (node :> logic_assign) with
    | `LogicAssign fields ->
        Lazy.force fields.f_dest_var
  let f_value node =
    match (node :> logic_assign) with
    | `LogicAssign fields ->
        Lazy.force fields.f_value



end

module PatternSingleLineStringLit = struct
  type t =
    [
      | `PatternSingleLineStringLit of
          pattern_single_line_string_lit_fields
    ]

  type fields = pattern_single_line_string_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module SingleLineStringLit = struct
  type t =
    [
      | `SingleLineStringLit of
          single_line_string_lit_fields
      | `PatternSingleLineStringLit of
          pattern_single_line_string_lit_fields
    ]

  type fields = single_line_string_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BlockStringLit = struct
  type t =
    [
      | `BlockStringLit of
          block_string_lit_fields
    ]

  type fields = block_string_lit_fields =
    
  {
         
    f_lines: block_string_line_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_lines node =
    match (node :> block_string_lit) with
    | `BlockStringLit fields ->
        Lazy.force fields.f_lines



end

module StringLit = struct
  type t =
    [
      | BlockStringLit.t
      | SingleLineStringLit.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_denoted_value
    (node)
    =
      let result_ptr =
        allocate_n DecodedStringValueStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.string_lit_p_denoted_value
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      DecodedStringValue.wrap ~dec_ref:true (!@ result_ptr)

let p_is_prefixed_string
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.string_lit_p_is_prefixed_string
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let p_prefix
    (node)
    =
      let result_ptr =
        allocate_n Character.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.string_lit_p_prefix
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let p_is_regexp_literal
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.string_lit_p_is_regexp_literal
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr





end

module NumLit = struct
  type t =
    [
      | `NumLit of
          num_lit_fields
    ]

  type fields = num_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module NullLit = struct
  type t =
    [
      | `NullLit of
          null_lit_fields
    ]

  type fields = null_lit_fields =
    
  {
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_dest_type node =
    match (node :> null_lit) with
    | `NullLit fields ->
        Lazy.force fields.f_dest_type



end

module CharLit = struct
  type t =
    [
      | `CharLit of
          char_lit_fields
    ]

  type fields = char_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_denoted_value
    (node)
    =
      let result_ptr =
        allocate_n DecodedCharValueStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.char_lit_p_denoted_value
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      DecodedCharValue.wrap ~dec_ref:true (!@ result_ptr)





end

module BigNumLit = struct
  type t =
    [
      | `BigNumLit of
          big_num_lit_fields
    ]

  type fields = big_num_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module Lit = struct
  type t =
    [
      | BigNumLit.t
      | CharLit.t
      | NullLit.t
      | NumLit.t
      | StringLit.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module LambdaExpr = struct
  type t =
    [
      | `LambdaExpr of
          lambda_expr_fields
    ]

  type fields = lambda_expr_fields =
    
  {
         
    f_params: lambda_param_decl_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
         
    f_body: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_params node =
    match (node :> lambda_expr) with
    | `LambdaExpr fields ->
        Lazy.force fields.f_params
  let f_return_type node =
    match (node :> lambda_expr) with
    | `LambdaExpr fields ->
        Lazy.force fields.f_return_type
  let f_body node =
    match (node :> lambda_expr) with
    | `LambdaExpr fields ->
        Lazy.force fields.f_body



end

module KeepExpr = struct
  type t =
    [
      | `KeepExpr of
          keep_expr_fields
    ]

  type fields = keep_expr_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_keep_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> keep_expr) with
    | `KeepExpr fields ->
        Lazy.force fields.f_expr
  let f_null_cond node =
    match (node :> keep_expr) with
    | `KeepExpr fields ->
        Lazy.force fields.f_null_cond
  let f_keep_type node =
    match (node :> keep_expr) with
    | `KeepExpr fields ->
        Lazy.force fields.f_keep_type



end

module Isa = struct
  type t =
    [
      | `Isa of
          isa_fields
    ]

  type fields = isa_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> isa) with
    | `Isa fields ->
        Lazy.force fields.f_expr
  let f_pattern node =
    match (node :> isa) with
    | `Isa fields ->
        Lazy.force fields.f_pattern



end

module IfExpr = struct
  type t =
    [
      | `IfExpr of
          if_expr_fields
    ]

  type fields = if_expr_fields =
    
  {
         
    f_cond_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_alternatives: elsif_branch_list
    Lazy.t;
         
    f_else_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_cond_expr node =
    match (node :> if_expr) with
    | `IfExpr fields ->
        Lazy.force fields.f_cond_expr
  let f_then_expr node =
    match (node :> if_expr) with
    | `IfExpr fields ->
        Lazy.force fields.f_then_expr
  let f_alternatives node =
    match (node :> if_expr) with
    | `IfExpr fields ->
        Lazy.force fields.f_alternatives
  let f_else_expr node =
    match (node :> if_expr) with
    | `IfExpr fields ->
        Lazy.force fields.f_else_expr



end

module RefId = struct
  type t =
    [
      | `RefId of
          ref_id_fields
    ]

  type fields = ref_id_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_referenced_defining_name
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.ref_id_p_referenced_defining_name
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_def_id (context node) (!@ result_ptr))





end

module ModuleRefId = struct
  type t =
    [
      | `ModuleRefId of
          module_ref_id_fields
    ]

  type fields = module_ref_id_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module DefId = struct
  type t =
    [
      | `DefId of
          def_id_fields
    ]

  type fields = def_id_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_name
    (node)
    =
      let result_ptr =
        allocate_n StringType.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.def_id_p_name
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      StringType.wrap ~dec_ref:true (!@ result_ptr)

let p_get_implementatinons
    (node)
    (units)
    =
      let result_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_units =
            
        AnalysisUnitArray.unwrap (units)
      in
      let _ : int =
        CFunctions.def_id_p_get_implementatinons
          (addr (unwrap_lkt_node (node)))
          c_units
          (result_ptr)
      in
         
               AnalysisUnitArrayStruct.dec_ref c_units ;
      DefIdArray.wrap ~dec_ref:true (context node) (!@ result_ptr)

let p_decl_detail
    (node)
    =
      let result_ptr =
        allocate_n StringType.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.def_id_p_decl_detail
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      StringType.wrap ~dec_ref:true (!@ result_ptr)

let p_completion_item_kind
    (node)
    =
      let result_ptr =
        allocate_n int ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.def_id_p_completion_item_kind
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let p_doc
    (node)
    =
      let result_ptr =
        allocate_n StringType.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.def_id_p_doc
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      StringType.wrap ~dec_ref:true (!@ result_ptr)

let p_find_all_references
    (node)
    (units)
    =
      let result_ptr =
        allocate_n RefResultArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_units =
            
        AnalysisUnitArray.unwrap (units)
      in
      let _ : int =
        CFunctions.def_id_p_find_all_references
          (addr (unwrap_lkt_node (node)))
          c_units
          (result_ptr)
      in
         
               AnalysisUnitArrayStruct.dec_ref c_units ;
      RefResultArray.wrap ~dec_ref:true (context node) (!@ result_ptr)





end

module Id = struct
  type t =
    [
      | `Id of
          id_fields
      | `DefId of
          def_id_fields
      | `ModuleRefId of
          module_ref_id_fields
      | `RefId of
          ref_id_fields
    ]

  type fields = id_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_custom_image
    (node)
    =
      let result_ptr =
        allocate_n StringType.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.id_p_custom_image
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      StringType.wrap ~dec_ref:true (!@ result_ptr)





end

module TokenRef = struct
  type t =
    [
      | `TokenRef of
          token_ref_fields
    ]

  type fields = token_ref_fields =
    
  {
         
    f_token_name: ref_id
    Lazy.t;
         
    f_expr: token_lit
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_token_name node =
    match (node :> token_ref) with
    | `TokenRef fields ->
        Lazy.force fields.f_token_name
  let f_expr node =
    match (node :> token_ref) with
    | `TokenRef fields ->
        Lazy.force fields.f_expr



end

module TokenPatternLit = struct
  type t =
    [
      | `TokenPatternLit of
          token_pattern_lit_fields
    ]

  type fields = token_pattern_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_denoted_value
    (node)
    =
      let result_ptr =
        allocate_n DecodedStringValueStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.token_pattern_lit_p_denoted_value
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      DecodedStringValue.wrap ~dec_ref:true (!@ result_ptr)





end

module TokenPatternConcat = struct
  type t =
    [
      | `TokenPatternConcat of
          token_pattern_concat_fields
    ]

  type fields = token_pattern_concat_fields =
    
  {
         
    f_left: [
      | `TokenPatternConcat
          of token_pattern_concat_fields
      | `TokenPatternLit
          of token_pattern_lit_fields
    ]
    Lazy.t;
         
    f_right: token_pattern_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_left node =
    match (node :> token_pattern_concat) with
    | `TokenPatternConcat fields ->
        Lazy.force fields.f_left
  let f_right node =
    match (node :> token_pattern_concat) with
    | `TokenPatternConcat fields ->
        Lazy.force fields.f_right



end

module TokenNoCaseLit = struct
  type t =
    [
      | `TokenNoCaseLit of
          token_no_case_lit_fields
    ]

  type fields = token_no_case_lit_fields =
    
  {
         
    f_lit: token_lit
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_lit node =
    match (node :> token_no_case_lit) with
    | `TokenNoCaseLit fields ->
        Lazy.force fields.f_lit



end

module TokenLit = struct
  type t =
    [
      | `TokenLit of
          token_lit_fields
    ]

  type fields = token_lit_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_denoted_value
    (node)
    =
      let result_ptr =
        allocate_n DecodedStringValueStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.token_lit_p_denoted_value
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      DecodedStringValue.wrap ~dec_ref:true (!@ result_ptr)





end

module ParseNodeExpr = struct
  type t =
    [
      | `ParseNodeExpr of
          parse_node_expr_fields
    ]

  type fields = parse_node_expr_fields =
    
  {
         
    f_node_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
         
    f_sub_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_node_name node =
    match (node :> parse_node_expr) with
    | `ParseNodeExpr fields ->
        Lazy.force fields.f_node_name
  let f_sub_exprs node =
    match (node :> parse_node_expr) with
    | `ParseNodeExpr fields ->
        Lazy.force fields.f_sub_exprs



end

module GrammarStopCut = struct
  type t =
    [
      | `GrammarStopCut of
          grammar_stop_cut_fields
    ]

  type fields = grammar_stop_cut_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> grammar_stop_cut) with
    | `GrammarStopCut fields ->
        Lazy.force fields.f_expr



end

module GrammarSkip = struct
  type t =
    [
      | `GrammarSkip of
          grammar_skip_fields
    ]

  type fields = grammar_skip_fields =
    
  {
         
    f_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> grammar_skip) with
    | `GrammarSkip fields ->
        Lazy.force fields.f_name



end

module GrammarRuleRef = struct
  type t =
    [
      | `GrammarRuleRef of
          grammar_rule_ref_fields
    ]

  type fields = grammar_rule_ref_fields =
    
  {
         
    f_node_name: ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_node_name node =
    match (node :> grammar_rule_ref) with
    | `GrammarRuleRef fields ->
        Lazy.force fields.f_node_name



end

module GrammarPredicate = struct
  type t =
    [
      | `GrammarPredicate of
          grammar_predicate_fields
    ]

  type fields = grammar_predicate_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_prop_ref: [
      | `DotExpr
          of dot_expr_fields
      | `RefId
          of ref_id_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> grammar_predicate) with
    | `GrammarPredicate fields ->
        Lazy.force fields.f_expr
  let f_prop_ref node =
    match (node :> grammar_predicate) with
    | `GrammarPredicate fields ->
        Lazy.force fields.f_prop_ref



end

module GrammarImplicitPick = struct
  type t =
    [
      | `GrammarImplicitPick of
          grammar_implicit_pick_fields
    ]

  type fields = grammar_implicit_pick_fields =
    
  {
         
    f_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_exprs node =
    match (node :> grammar_implicit_pick) with
    | `GrammarImplicitPick fields ->
        Lazy.force fields.f_exprs



end

module GrammarPick = struct
  type t =
    [
      | `GrammarPick of
          grammar_pick_fields
      | `GrammarImplicitPick of
          grammar_implicit_pick_fields
    ]

  type fields = grammar_pick_fields =
    
  {
         
    f_exprs: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_exprs node =
    match (node :> grammar_pick) with
    | `GrammarPick fields ->
        Lazy.force fields.f_exprs
    | `GrammarImplicitPick fields ->
        Lazy.force fields.f_exprs



end

module GrammarOrExpr = struct
  type t =
    [
      | `GrammarOrExpr of
          grammar_or_expr_fields
    ]

  type fields = grammar_or_expr_fields =
    
  {
         
    f_sub_exprs: grammar_expr_list_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_sub_exprs node =
    match (node :> grammar_or_expr) with
    | `GrammarOrExpr fields ->
        Lazy.force fields.f_sub_exprs



end

module GrammarOptGroup = struct
  type t =
    [
      | `GrammarOptGroup of
          grammar_opt_group_fields
    ]

  type fields = grammar_opt_group_fields =
    
  {
         
    f_expr: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> grammar_opt_group) with
    | `GrammarOptGroup fields ->
        Lazy.force fields.f_expr



end

module GrammarOptErrorGroup = struct
  type t =
    [
      | `GrammarOptErrorGroup of
          grammar_opt_error_group_fields
    ]

  type fields = grammar_opt_error_group_fields =
    
  {
         
    f_expr: grammar_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> grammar_opt_error_group) with
    | `GrammarOptErrorGroup fields ->
        Lazy.force fields.f_expr



end

module GrammarOptError = struct
  type t =
    [
      | `GrammarOptError of
          grammar_opt_error_fields
    ]

  type fields = grammar_opt_error_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> grammar_opt_error) with
    | `GrammarOptError fields ->
        Lazy.force fields.f_expr



end

module GrammarOpt = struct
  type t =
    [
      | `GrammarOpt of
          grammar_opt_fields
    ]

  type fields = grammar_opt_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> grammar_opt) with
    | `GrammarOpt fields ->
        Lazy.force fields.f_expr



end

module GrammarNull = struct
  type t =
    [
      | `GrammarNull of
          grammar_null_fields
    ]

  type fields = grammar_null_fields =
    
  {
         
    f_name: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> grammar_null) with
    | `GrammarNull fields ->
        Lazy.force fields.f_name



end

module GrammarList = struct
  type t =
    [
      | `GrammarList of
          grammar_list_fields
    ]

  type fields = grammar_list_fields =
    
  {
         
    f_list_type: type_ref
    Lazy.t;
         
    f_kind: list_kind
    Lazy.t;
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_sep: grammar_list_sep
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_list_type node =
    match (node :> grammar_list) with
    | `GrammarList fields ->
        Lazy.force fields.f_list_type
  let f_kind node =
    match (node :> grammar_list) with
    | `GrammarList fields ->
        Lazy.force fields.f_kind
  let f_expr node =
    match (node :> grammar_list) with
    | `GrammarList fields ->
        Lazy.force fields.f_expr
  let f_sep node =
    match (node :> grammar_list) with
    | `GrammarList fields ->
        Lazy.force fields.f_sep



end

module GrammarDontSkip = struct
  type t =
    [
      | `GrammarDontSkip of
          grammar_dont_skip_fields
    ]

  type fields = grammar_dont_skip_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
         
    f_dont_skip: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> grammar_dont_skip) with
    | `GrammarDontSkip fields ->
        Lazy.force fields.f_expr
  let f_dont_skip node =
    match (node :> grammar_dont_skip) with
    | `GrammarDontSkip fields ->
        Lazy.force fields.f_dont_skip



end

module GrammarDiscard = struct
  type t =
    [
      | `GrammarDiscard of
          grammar_discard_fields
    ]

  type fields = grammar_discard_fields =
    
  {
         
    f_expr: grammar_expr
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> grammar_discard) with
    | `GrammarDiscard fields ->
        Lazy.force fields.f_expr



end

module GrammarCut = struct
  type t =
    [
      | `GrammarCut of
          grammar_cut_fields
    ]

  type fields = grammar_cut_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module GrammarExpr = struct
  type t =
    [
      | GrammarCut.t
      | GrammarDiscard.t
      | GrammarDontSkip.t
      | GrammarList.t
      | GrammarNull.t
      | GrammarOpt.t
      | GrammarOptError.t
      | GrammarOptErrorGroup.t
      | GrammarOptGroup.t
      | GrammarOrExpr.t
      | GrammarPick.t
      | GrammarPredicate.t
      | GrammarRuleRef.t
      | GrammarSkip.t
      | GrammarStopCut.t
      | ParseNodeExpr.t
      | TokenLit.t
      | TokenNoCaseLit.t
      | TokenPatternConcat.t
      | TokenPatternLit.t
      | TokenRef.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module GenericInstantiation = struct
  type t =
    [
      | `GenericInstantiation of
          generic_instantiation_fields
    ]

  type fields = generic_instantiation_fields =
    
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: type_ref_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> generic_instantiation) with
    | `GenericInstantiation fields ->
        Lazy.force fields.f_name
  let f_args node =
    match (node :> generic_instantiation) with
    | `GenericInstantiation fields ->
        Lazy.force fields.f_args



end

module ErrorOnNull = struct
  type t =
    [
      | `ErrorOnNull of
          error_on_null_fields
    ]

  type fields = error_on_null_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> error_on_null) with
    | `ErrorOnNull fields ->
        Lazy.force fields.f_expr



end

module DotExpr = struct
  type t =
    [
      | `DotExpr of
          dot_expr_fields
    ]

  type fields = dot_expr_fields =
    
  {
         
    f_prefix: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_suffix: ref_id
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_prefix node =
    match (node :> dot_expr) with
    | `DotExpr fields ->
        Lazy.force fields.f_prefix
  let f_null_cond node =
    match (node :> dot_expr) with
    | `DotExpr fields ->
        Lazy.force fields.f_null_cond
  let f_suffix node =
    match (node :> dot_expr) with
    | `DotExpr fields ->
        Lazy.force fields.f_suffix



end

module CastExpr = struct
  type t =
    [
      | `CastExpr of
          cast_expr_fields
    ]

  type fields = cast_expr_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_null_cond: null_cond_qualifier
    Lazy.t;
         
    f_excludes_null: excludes_null
    Lazy.t;
         
    f_dest_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> cast_expr) with
    | `CastExpr fields ->
        Lazy.force fields.f_expr
  let f_null_cond node =
    match (node :> cast_expr) with
    | `CastExpr fields ->
        Lazy.force fields.f_null_cond
  let f_excludes_null node =
    match (node :> cast_expr) with
    | `CastExpr fields ->
        Lazy.force fields.f_excludes_null
  let f_dest_type node =
    match (node :> cast_expr) with
    | `CastExpr fields ->
        Lazy.force fields.f_dest_type



end

module BlockExpr = struct
  type t =
    [
      | `BlockExpr of
          block_expr_fields
    ]

  type fields = block_expr_fields =
    
  {
         
    f_clauses: lkt_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_clauses node =
    match (node :> block_expr) with
    | `BlockExpr fields ->
        Lazy.force fields.f_clauses



end

module BinOp = struct
  type t =
    [
      | `BinOp of
          bin_op_fields
    ]

  type fields = bin_op_fields =
    
  {
         
    f_left: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_op: op
    Lazy.t;
         
    f_right: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_left node =
    match (node :> bin_op) with
    | `BinOp fields ->
        Lazy.force fields.f_left
  let f_op node =
    match (node :> bin_op) with
    | `BinOp fields ->
        Lazy.force fields.f_op
  let f_right node =
    match (node :> bin_op) with
    | `BinOp fields ->
        Lazy.force fields.f_right



end

module LogicPropagateCall = struct
  type t =
    [
      | `LogicPropagateCall of
          logic_propagate_call_fields
    ]

  type fields = logic_propagate_call_fields =
    
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> logic_propagate_call) with
    | `LogicPropagateCall fields ->
        Lazy.force fields.f_name
  let f_args node =
    match (node :> logic_propagate_call) with
    | `LogicPropagateCall fields ->
        Lazy.force fields.f_args



end

module LogicPredicate = struct
  type t =
    [
      | `LogicPredicate of
          logic_predicate_fields
    ]

  type fields = logic_predicate_fields =
    
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> logic_predicate) with
    | `LogicPredicate fields ->
        Lazy.force fields.f_name
  let f_args node =
    match (node :> logic_predicate) with
    | `LogicPredicate fields ->
        Lazy.force fields.f_args



end

module LogicCallExpr = struct
  type t =
    [
      | LogicPredicate.t
      | LogicPropagateCall.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> logic_call_expr) with
    | `LogicPredicate fields ->
        Lazy.force fields.f_name
    | `LogicPropagateCall fields ->
        Lazy.force fields.f_name
  let f_args node =
    match (node :> logic_call_expr) with
    | `LogicPredicate fields ->
        Lazy.force fields.f_args
    | `LogicPropagateCall fields ->
        Lazy.force fields.f_args



end

module CallExpr = struct
  type t =
    [
      | `CallExpr of
          call_expr_fields
    ]

  type fields = call_expr_fields =
    
  {
         
    f_name: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `KeepExpr
          of keep_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
    ]
    Lazy.t;
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> call_expr) with
    | `CallExpr fields ->
        Lazy.force fields.f_name
  let f_args node =
    match (node :> call_expr) with
    | `CallExpr fields ->
        Lazy.force fields.f_args



end

module BaseCallExpr = struct
  type t =
    [
      | CallExpr.t
      | LogicCallExpr.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> base_call_expr) with
    | `CallExpr fields ->
        Lazy.force fields.f_name
    | `LogicPredicate fields ->
        Lazy.force fields.f_name
    | `LogicPropagateCall fields ->
        Lazy.force fields.f_name
  let f_args node =
    match (node :> base_call_expr) with
    | `CallExpr fields ->
        Lazy.force fields.f_args
    | `LogicPredicate fields ->
        Lazy.force fields.f_args
    | `LogicPropagateCall fields ->
        Lazy.force fields.f_args



end

module ArrayLiteral = struct
  type t =
    [
      | `ArrayLiteral of
          array_literal_fields
    ]

  type fields = array_literal_fields =
    
  {
         
    f_exprs: expr_list
    Lazy.t;
         
    f_element_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_exprs node =
    match (node :> array_literal) with
    | `ArrayLiteral fields ->
        Lazy.force fields.f_exprs
  let f_element_type node =
    match (node :> array_literal) with
    | `ArrayLiteral fields ->
        Lazy.force fields.f_element_type



end

module AnyOf = struct
  type t =
    [
      | `AnyOf of
          any_of_fields
    ]

  type fields = any_of_fields =
    
  {
         
    f_expr: [
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `MatchExpr
          of match_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
    ]
    Lazy.t;
         
    f_values: any_of_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> any_of) with
    | `AnyOf fields ->
        Lazy.force fields.f_expr
  let f_values node =
    match (node :> any_of) with
    | `AnyOf fields ->
        Lazy.force fields.f_values



end

module Expr = struct
  type t =
    [
      | AnyOf.t
      | ArrayLiteral.t
      | BaseCallExpr.t
      | BinOp.t
      | BlockExpr.t
      | CastExpr.t
      | DotExpr.t
      | ErrorOnNull.t
      | GenericInstantiation.t
      | GrammarExpr.t
      | Id.t
      | IfExpr.t
      | Isa.t
      | KeepExpr.t
      | LambdaExpr.t
      | Lit.t
      | LogicAssign.t
      | LogicExpr.t
      | LogicPropagate.t
      | LogicUnify.t
      | MatchExpr.t
      | NotExpr.t
      | ParenExpr.t
      | RaiseExpr.t
      | SubscriptExpr.t
      | TryExpr.t
      | UnOp.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_get_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.expr_p_get_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))

let p_get_generic_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.expr_p_get_generic_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))

let p_get_expected_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.expr_p_get_expected_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))

let p_referenced_decl
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.expr_p_referenced_decl
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_decl (context node) (!@ result_ptr))





end

module ExcludesNullPresent = struct
  type t =
    [
      | `ExcludesNullPresent of
          excludes_null_present_fields
    ]

  type fields = excludes_null_present_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ExcludesNullAbsent = struct
  type t =
    [
      | `ExcludesNullAbsent of
          excludes_null_absent_fields
    ]

  type fields = excludes_null_absent_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ExcludesNull = struct
  type t =
    [
      | ExcludesNullAbsent.t
      | ExcludesNullPresent.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_as_bool
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.excludes_null_p_as_bool
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr





end

module EnumClassCase = struct
  type t =
    [
      | `EnumClassCase of
          enum_class_case_fields
    ]

  type fields = enum_class_case_fields =
    
  {
         
    f_decls: enum_class_alt_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_decls node =
    match (node :> enum_class_case) with
    | `EnumClassCase fields ->
        Lazy.force fields.f_decls



end

module ElsifBranch = struct
  type t =
    [
      | `ElsifBranch of
          elsif_branch_fields
    ]

  type fields = elsif_branch_fields =
    
  {
         
    f_cond_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
         
    f_then_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_cond_expr node =
    match (node :> elsif_branch) with
    | `ElsifBranch fields ->
        Lazy.force fields.f_cond_expr
  let f_then_expr node =
    match (node :> elsif_branch) with
    | `ElsifBranch fields ->
        Lazy.force fields.f_then_expr



end

module DynEnvWrapper = struct
  type t =
    [
      | `DynEnvWrapper of
          dyn_env_wrapper_fields
    ]

  type fields = dyn_env_wrapper_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module DeclAnnotationArgs = struct
  type t =
    [
      | `DeclAnnotationArgs of
          decl_annotation_args_fields
    ]

  type fields = decl_annotation_args_fields =
    
  {
         
    f_args: argument_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_args node =
    match (node :> decl_annotation_args) with
    | `DeclAnnotationArgs fields ->
        Lazy.force fields.f_args



end

module DeclAnnotation = struct
  type t =
    [
      | `DeclAnnotation of
          decl_annotation_fields
    ]

  type fields = decl_annotation_fields =
    
  {
         
    f_name: id
    Lazy.t;
         
    f_args: decl_annotation_args
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> decl_annotation) with
    | `DeclAnnotation fields ->
        Lazy.force fields.f_name
  let f_args node =
    match (node :> decl_annotation) with
    | `DeclAnnotation fields ->
        Lazy.force fields.f_args



end

module TraitDecl = struct
  type t =
    [
      | `TraitDecl of
          trait_decl_fields
    ]

  type fields = trait_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module StructDecl = struct
  type t =
    [
      | `StructDecl of
          struct_decl_fields
    ]

  type fields = struct_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module EnumTypeDecl = struct
  type t =
    [
      | `EnumTypeDecl of
          enum_type_decl_fields
    ]

  type fields = enum_type_decl_fields =
    
  {
         
    f_literals: enum_lit_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_literals node =
    match (node :> enum_type_decl) with
    | `EnumTypeDecl fields ->
        Lazy.force fields.f_literals



end

module EnumClassDecl = struct
  type t =
    [
      | `EnumClassDecl of
          enum_class_decl_fields
    ]

  type fields = enum_class_decl_fields =
    
  {
         
    f_branches: enum_class_case_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_branches node =
    match (node :> enum_class_decl) with
    | `EnumClassDecl fields ->
        Lazy.force fields.f_branches



end

module ClassDecl = struct
  type t =
    [
      | `ClassDecl of
          class_decl_fields
    ]

  type fields = class_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BasicClassDecl = struct
  type t =
    [
      | ClassDecl.t
      | EnumClassDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module NamedTypeDecl = struct
  type t =
    [
      | BasicClassDecl.t
      | EnumTypeDecl.t
      | StructDecl.t
      | TraitDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module GenericParamTypeDecl = struct
  type t =
    [
      | `GenericParamTypeDecl of
          generic_param_type_decl_fields
    ]

  type fields = generic_param_type_decl_fields =
    
  {
         
    f_has_class: class_qualifier
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_has_class node =
    match (node :> generic_param_type_decl) with
    | `GenericParamTypeDecl fields ->
        Lazy.force fields.f_has_class



end

module FunctionType = struct
  type t =
    [
      | `FunctionType of
          function_type_fields
    ]

  type fields = function_type_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module EnumClassAltDecl = struct
  type t =
    [
      | `EnumClassAltDecl of
          enum_class_alt_decl_fields
    ]

  type fields = enum_class_alt_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module AnyTypeDecl = struct
  type t =
    [
      | `AnyTypeDecl of
          any_type_decl_fields
    ]

  type fields = any_type_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module TypeDecl = struct
  type t =
    [
      | AnyTypeDecl.t
      | EnumClassAltDecl.t
      | FunctionType.t
      | GenericParamTypeDecl.t
      | NamedTypeDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_def_id
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.type_decl_p_def_id
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_def_id (context node) (!@ result_ptr))

let p_base_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.type_decl_p_base_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_ref (context node) (!@ result_ptr))

let p_base_type_if_entity
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.type_decl_p_base_type_if_entity
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))





end

module SynthParamDecl = struct
  type t =
    [
      | `SynthParamDecl of
          synth_param_decl_fields
    ]

  type fields = synth_param_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module SynthFunDecl = struct
  type t =
    [
      | `SynthFunDecl of
          synth_fun_decl_fields
    ]

  type fields = synth_fun_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module LexerFamilyDecl = struct
  type t =
    [
      | `LexerFamilyDecl of
          lexer_family_decl_fields
    ]

  type fields = lexer_family_decl_fields =
    
  {
         
    f_rules: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_rules node =
    match (node :> lexer_family_decl) with
    | `LexerFamilyDecl fields ->
        Lazy.force fields.f_rules



end

module LexerDecl = struct
  type t =
    [
      | `LexerDecl of
          lexer_decl_fields
    ]

  type fields = lexer_decl_fields =
    
  {
         
    f_rules: lkt_node_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_rules node =
    match (node :> lexer_decl) with
    | `LexerDecl fields ->
        Lazy.force fields.f_rules



end

module GrammarDecl = struct
  type t =
    [
      | `GrammarDecl of
          grammar_decl_fields
    ]

  type fields = grammar_decl_fields =
    
  {
         
    f_rules: full_decl_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_rules node =
    match (node :> grammar_decl) with
    | `GrammarDecl fields ->
        Lazy.force fields.f_rules



end

module GenericDecl = struct
  type t =
    [
      | `GenericDecl of
          generic_decl_fields
    ]

  type fields = generic_decl_fields =
    
  {
         
    f_generic_param_decls: generic_param_decl_list
    Lazy.t;
         
    f_decl: [
      | `ClassDecl
          of class_decl_fields
      | `DynVarDecl
          of dyn_var_decl_fields
      | `EnumClassDecl
          of enum_class_decl_fields
      | `EnumTypeDecl
          of enum_type_decl_fields
      | `EnvSpecDecl
          of env_spec_decl_fields
      | `FieldDecl
          of field_decl_fields
      | `FunDecl
          of fun_decl_fields
      | `GenericDecl
          of generic_decl_fields
      | `GrammarDecl
          of grammar_decl_fields
      | `GrammarRuleDecl
          of grammar_rule_decl_fields
      | `LexerDecl
          of lexer_decl_fields
      | `StructDecl
          of struct_decl_fields
      | `TraitDecl
          of trait_decl_fields
      | `ValDecl
          of val_decl_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_generic_param_decls node =
    match (node :> generic_decl) with
    | `GenericDecl fields ->
        Lazy.force fields.f_generic_param_decls
  let f_decl node =
    match (node :> generic_decl) with
    | `GenericDecl fields ->
        Lazy.force fields.f_decl



end

module ErrorDecl = struct
  type t =
    [
      | `ErrorDecl of
          error_decl_fields
    ]

  type fields = error_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module EnvSpecDecl = struct
  type t =
    [
      | `EnvSpecDecl of
          env_spec_decl_fields
    ]

  type fields = env_spec_decl_fields =
    
  {
         
    f_actions: call_expr_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_actions node =
    match (node :> env_spec_decl) with
    | `EnvSpecDecl fields ->
        Lazy.force fields.f_actions



end

module FunDecl = struct
  type t =
    [
      | `FunDecl of
          fun_decl_fields
    ]

  type fields = fun_decl_fields =
    
  {
         
    f_params: fun_param_decl_list
    Lazy.t;
         
    f_return_type: [
      | `FunctionTypeRef
          of function_type_ref_fields
      | `GenericTypeRef
          of generic_type_ref_fields
      | `SimpleTypeRef
          of simple_type_ref_fields
    ]
    Lazy.t;
         
    f_trait_ref: dot_expr
    option
    Lazy.t;
         
    f_body: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_is_dynamic_combiner
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.fun_decl_p_is_dynamic_combiner
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let p_find_all_overrides
    (node)
    (units)
    =
      let result_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_units =
            
        AnalysisUnitArray.unwrap (units)
      in
      let _ : int =
        CFunctions.fun_decl_p_find_all_overrides
          (addr (unwrap_lkt_node (node)))
          c_units
          (result_ptr)
      in
         
               AnalysisUnitArrayStruct.dec_ref c_units ;
      FunDeclArray.wrap ~dec_ref:true (context node) (!@ result_ptr)


  let f_params node =
    match (node :> fun_decl) with
    | `FunDecl fields ->
        Lazy.force fields.f_params
  let f_return_type node =
    match (node :> fun_decl) with
    | `FunDecl fields ->
        Lazy.force fields.f_return_type
  let f_trait_ref node =
    match (node :> fun_decl) with
    | `FunDecl fields ->
        Lazy.force fields.f_trait_ref
  let f_body node =
    match (node :> fun_decl) with
    | `FunDecl fields ->
        Lazy.force fields.f_body



end

module ValDecl = struct
  type t =
    [
      | `ValDecl of
          val_decl_fields
    ]

  type fields = val_decl_fields =
    
  {
         
    f_expr: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_expr node =
    match (node :> val_decl) with
    | `ValDecl fields ->
        Lazy.force fields.f_expr



end

module MatchValDecl = struct
  type t =
    [
      | `MatchValDecl of
          match_val_decl_fields
    ]

  type fields = match_val_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module DynVarDecl = struct
  type t =
    [
      | `DynVarDecl of
          dyn_var_decl_fields
    ]

  type fields = dyn_var_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module LambdaParamDecl = struct
  type t =
    [
      | `LambdaParamDecl of
          lambda_param_decl_fields
    ]

  type fields = lambda_param_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module FunParamDecl = struct
  type t =
    [
      | `FunParamDecl of
          fun_param_decl_fields
    ]

  type fields = fun_param_decl_fields =
    
  {
         
    f_decl_annotations: decl_annotation_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_decl_annotations node =
    match (node :> fun_param_decl) with
    | `FunParamDecl fields ->
        Lazy.force fields.f_decl_annotations



end

module FieldDecl = struct
  type t =
    [
      | `FieldDecl of
          field_decl_fields
    ]

  type fields = field_decl_fields =
    
  {
         
    f_trait_ref: dot_expr
    option
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_trait_ref node =
    match (node :> field_decl) with
    | `FieldDecl fields ->
        Lazy.force fields.f_trait_ref



end

module ComponentDecl = struct
  type t =
    [
      | FieldDecl.t
      | FunParamDecl.t
      | LambdaParamDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ExplicitlyTypedDecl = struct
  type t =
    [
      | ComponentDecl.t
      | DynVarDecl.t
      | MatchValDecl.t
      | ValDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module EnumLitDecl = struct
  type t =
    [
      | `EnumLitDecl of
          enum_lit_decl_fields
    ]

  type fields = enum_lit_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BindingValDecl = struct
  type t =
    [
      | `BindingValDecl of
          binding_val_decl_fields
    ]

  type fields = binding_val_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module UserValDecl = struct
  type t =
    [
      | BindingValDecl.t
      | EnumLitDecl.t
      | ExplicitlyTypedDecl.t
      | FunDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module SelfDecl = struct
  type t =
    [
      | `SelfDecl of
          self_decl_fields
    ]

  type fields = self_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module NodeDecl = struct
  type t =
    [
      | `NodeDecl of
          node_decl_fields
    ]

  type fields = node_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BaseValDecl = struct
  type t =
    [
      | NodeDecl.t
      | SelfDecl.t
      | UserValDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module SyntheticLexerDecl = struct
  type t =
    [
      | `SyntheticLexerDecl of
          synthetic_lexer_decl_fields
    ]

  type fields = synthetic_lexer_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module GrammarRuleDecl = struct
  type t =
    [
      | `GrammarRuleDecl of
          grammar_rule_decl_fields
    ]

  type fields = grammar_rule_decl_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BaseGrammarRuleDecl = struct
  type t =
    [
      | GrammarRuleDecl.t
      | SyntheticLexerDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module Decl = struct
  type t =
    [
      | BaseGrammarRuleDecl.t
      | BaseValDecl.t
      | EnvSpecDecl.t
      | ErrorDecl.t
      | GenericDecl.t
      | GrammarDecl.t
      | LexerDecl.t
      | LexerFamilyDecl.t
      | SynthFunDecl.t
      | SynthParamDecl.t
      | TypeDecl.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_custom_image
    (node)
    =
      let result_ptr =
        allocate_n StringType.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_custom_image
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      StringType.wrap ~dec_ref:true (!@ result_ptr)

let p_decl_type_name
    (node)
    =
      let result_ptr =
        allocate_n StringType.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_decl_type_name
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      StringType.wrap ~dec_ref:true (!@ result_ptr)

let p_def_ids
    (node)
    =
      let result_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_def_ids
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      DefIdArray.wrap ~dec_ref:true (context node) (!@ result_ptr)

let p_as_bare_decl
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_as_bare_decl
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_decl (context node) (!@ result_ptr))

let p_get_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_get_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))

let p_get_cast_type
    (node)
    (cast_to)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_cast_to =
            
        addr (unwrap_lkt_node (cast_to))
      in
      let _ : int =
        CFunctions.decl_p_get_cast_type
          (addr (unwrap_lkt_node (node)))
          c_cast_to
          (result_ptr)
      in
         
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))

let p_get_keep_type
    (node)
    (keep_type)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_keep_type =
            
        addr (unwrap_lkt_node (keep_type))
      in
      let _ : int =
        CFunctions.decl_p_get_keep_type
          (addr (unwrap_lkt_node (node)))
          c_keep_type
          (result_ptr)
      in
         
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))

let p_get_suffix_type
    (node)
    (prefix_type)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_prefix_type =
            
        addr (unwrap_lkt_node (prefix_type))
      in
      let _ : int =
        CFunctions.decl_p_get_suffix_type
          (addr (unwrap_lkt_node (node)))
          c_prefix_type
          (result_ptr)
      in
         
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_type_decl (context node) (!@ result_ptr))

let p_is_generic
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_is_generic
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let p_return_type_is_instantiated
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_return_type_is_instantiated
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let p_is_instantiated
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_is_instantiated
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let p_name
    (node)
    =
      let result_ptr =
        allocate_n Symbol.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_name
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      Symbol.wrap (!@ result_ptr)

let p_full_name
    (node)
    =
      let result_ptr =
        allocate_n StringType.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.decl_p_full_name
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      StringType.wrap ~dec_ref:true (!@ result_ptr)





end

module ClassQualifierPresent = struct
  type t =
    [
      | `ClassQualifierPresent of
          class_qualifier_present_fields
    ]

  type fields = class_qualifier_present_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ClassQualifierAbsent = struct
  type t =
    [
      | `ClassQualifierAbsent of
          class_qualifier_absent_fields
    ]

  type fields = class_qualifier_absent_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module ClassQualifier = struct
  type t =
    [
      | ClassQualifierAbsent.t
      | ClassQualifierPresent.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_as_bool
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.class_qualifier_p_as_bool
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr





end

module BlockStringLine = struct
  type t =
    [
      | `BlockStringLine of
          block_string_line_fields
    ]

  type fields = block_string_line_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module BlockExprClause = struct
  type t =
    [
      | `BlockExprClause of
          block_expr_clause_fields
    ]

  type fields = block_expr_clause_fields =
    
  {
         
    f_clause: [
      | `ValDecl
          of val_decl_fields
      | `VarBind
          of var_bind_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_clause node =
    match (node :> block_expr_clause) with
    | `BlockExprClause fields ->
        Lazy.force fields.f_clause



end

module PatternMatchBranch = struct
  type t =
    [
      | `PatternMatchBranch of
          pattern_match_branch_fields
    ]

  type fields = pattern_match_branch_fields =
    
  {
         
    f_pattern: [
      | `BindingPattern
          of binding_pattern_fields
      | `BoolPatternFalse
          of bool_pattern_false_fields
      | `BoolPatternTrue
          of bool_pattern_true_fields
      | `ExtendedPattern
          of extended_pattern_fields
      | `FilteredPattern
          of filtered_pattern_fields
      | `IntegerPattern
          of integer_pattern_fields
      | `ListPattern
          of list_pattern_fields
      | `NotPattern
          of not_pattern_fields
      | `NullPattern
          of null_pattern_fields
      | `OrPattern
          of or_pattern_fields
      | `ParenPattern
          of paren_pattern_fields
      | `RegexPattern
          of regex_pattern_fields
      | `TuplePattern
          of tuple_pattern_fields
      | `TypePattern
          of type_pattern_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_pattern node =
    match (node :> pattern_match_branch) with
    | `PatternMatchBranch fields ->
        Lazy.force fields.f_pattern



end

module MatchBranch = struct
  type t =
    [
      | `MatchBranch of
          match_branch_fields
    ]

  type fields = match_branch_fields =
    
  {
         
    f_decl: match_val_decl
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_decl node =
    match (node :> match_branch) with
    | `MatchBranch fields ->
        Lazy.force fields.f_decl



end

module BaseMatchBranch = struct
  type t =
    [
      | MatchBranch.t
      | PatternMatchBranch.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let p_match_part
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.base_match_branch_p_match_part
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_lkt_node (context node) (!@ result_ptr))





end

module LexerCaseRuleDefaultAlt = struct
  type t =
    [
      | `LexerCaseRuleDefaultAlt of
          lexer_case_rule_default_alt_fields
    ]

  type fields = lexer_case_rule_default_alt_fields =
    
  {
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module LexerCaseRuleCondAlt = struct
  type t =
    [
      | `LexerCaseRuleCondAlt of
          lexer_case_rule_cond_alt_fields
    ]

  type fields = lexer_case_rule_cond_alt_fields =
    
  {
         
    f_cond_exprs: ref_id_list
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_cond_exprs node =
    match (node :> lexer_case_rule_cond_alt) with
    | `LexerCaseRuleCondAlt fields ->
        Lazy.force fields.f_cond_exprs



end

module BaseLexerCaseRuleAlt = struct
  type t =
    [
      | LexerCaseRuleCondAlt.t
      | LexerCaseRuleDefaultAlt.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))





end

module Argument = struct
  type t =
    [
      | `Argument of
          argument_fields
    ]

  type fields = argument_fields =
    
  {
         
    f_name: ref_id
    option
    Lazy.t;
         
    f_value: [
      | `AnyOf
          of any_of_fields
      | `ArrayLiteral
          of array_literal_fields
      | `BigNumLit
          of big_num_lit_fields
      | `BinOp
          of bin_op_fields
      | `BlockExpr
          of block_expr_fields
      | `BlockStringLit
          of block_string_lit_fields
      | `CallExpr
          of call_expr_fields
      | `CastExpr
          of cast_expr_fields
      | `CharLit
          of char_lit_fields
      | `DotExpr
          of dot_expr_fields
      | `ErrorOnNull
          of error_on_null_fields
      | `GenericInstantiation
          of generic_instantiation_fields
      | `IfExpr
          of if_expr_fields
      | `Isa
          of isa_fields
      | `KeepExpr
          of keep_expr_fields
      | `LambdaExpr
          of lambda_expr_fields
      | `LogicAssign
          of logic_assign_fields
      | `LogicExpr
          of logic_expr_fields
      | `LogicPredicate
          of logic_predicate_fields
      | `LogicPropagate
          of logic_propagate_fields
      | `LogicUnify
          of logic_unify_fields
      | `MatchExpr
          of match_expr_fields
      | `NotExpr
          of not_expr_fields
      | `NullLit
          of null_lit_fields
      | `NumLit
          of num_lit_fields
      | `ParenExpr
          of paren_expr_fields
      | `PatternSingleLineStringLit
          of pattern_single_line_string_lit_fields
      | `RaiseExpr
          of raise_expr_fields
      | `RefId
          of ref_id_fields
      | `SingleLineStringLit
          of single_line_string_lit_fields
      | `SubscriptExpr
          of subscript_expr_fields
      | `TryExpr
          of try_expr_fields
      | `UnOp
          of un_op_fields
    ]
    Lazy.t;
    c_value : entity;
    context : analysis_context
  }


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))


  let f_name node =
    match (node :> argument) with
    | `Argument fields ->
        Lazy.force fields.f_name
  let f_value node =
    match (node :> argument) with
    | `Argument fields ->
        Lazy.force fields.f_value



end

module LktNode = struct
  type t =
    [
      | Argument.t
      | BaseLexerCaseRuleAlt.t
      | BaseMatchBranch.t
      | BlockExprClause.t
      | BlockStringLine.t
      | ClassQualifier.t
      | Decl.t
      | DeclAnnotation.t
      | DeclAnnotationArgs.t
      | DynEnvWrapper.t
      | ElsifBranch.t
      | EnumClassCase.t
      | ExcludesNull.t
      | Expr.t
      | FullDecl.t
      | GrammarListSep.t
      | Import.t
      | LangkitRoot.t
      | LexerCaseRule.t
      | LexerCaseRuleSend.t
      | ListKind.t
      | LktNodeBaseList.t
      | NullCondQualifier.t
      | Op.t
      | Pattern.t
      | PatternDetail.t
      | SelectorCall.t
      | TypeRef.t
      | VarBind.t
    ]


  let equal node1 node2 =
    Entity.equal
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let compare node1 node2 =
    Entity.compare
      (unwrap_lkt_node ((node1 :> lkt_node)))
      (unwrap_lkt_node ((node2 :> lkt_node)))

  let hash node =
    Entity.hash
      (unwrap_lkt_node ((node :> lkt_node)))

let parent
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_parent
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_lkt_node (context node) (!@ result_ptr))

let parents
    ?(with_self=true)
    (node)
    =
      let result_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_with_self =
            
        with_self
      in
      let _ : int =
        CFunctions.lkt_node_parents
          (addr (unwrap_lkt_node (node)))
          c_with_self
          (result_ptr)
      in
         
      LktNodeArray.wrap ~dec_ref:true (context node) (!@ result_ptr)

let children
    (node)
    =
      let result_ptr =
        allocate_n LktNodeArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_children
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      LktNodeArray.wrap ~dec_ref:true (context node) (!@ result_ptr)

let token_start
    (node)
    =
      let result_ptr =
        allocate_n Token.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_token_start
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      Token.wrap (!@ result_ptr)

let token_end
    (node)
    =
      let result_ptr =
        allocate_n Token.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_token_end
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      Token.wrap (!@ result_ptr)

let child_index
    (node)
    =
      let result_ptr =
        allocate_n int ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_child_index
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let previous_sibling
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_previous_sibling
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_lkt_node (context node) (!@ result_ptr))

let next_sibling
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_next_sibling
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_lkt_node (context node) (!@ result_ptr))

let unit
    (node)
    =
      let result_ptr =
        allocate_n AnalysisUnitStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_unit
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      wrap_analysis_unit (context node) (!@ result_ptr)

let is_ghost
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_is_ghost
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let full_sloc_image
    (node)
    =
      let result_ptr =
        allocate_n StringType.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_full_sloc_image
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      StringType.wrap ~dec_ref:true (!@ result_ptr)

let completion_item_kind_to_int
    (node)
    (kind)
    =
      let result_ptr =
        allocate_n int ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_kind =
            
        kind
      in
      let _ : int =
        CFunctions.lkt_node_completion_item_kind_to_int
          (addr (unwrap_lkt_node (node)))
          c_kind
          (result_ptr)
      in
         
      !@ result_ptr

let p_set_solver_debug_mode
    (node)
    (enable)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let c_enable =
            
        enable
      in
      let _ : int =
        CFunctions.lkt_node_p_set_solver_debug_mode
          (addr (unwrap_lkt_node (node)))
          c_enable
          (result_ptr)
      in
         
      !@ result_ptr

let p_basic_trait_gen
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_basic_trait_gen
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_basic_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_basic_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_trait_decl (context node) (!@ result_ptr))

let p_node_gen_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_node_gen_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_node_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_node_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_trait_decl (context node) (!@ result_ptr))

let p_indexable_gen_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_indexable_gen_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_indexable_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_indexable_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_trait_decl (context node) (!@ result_ptr))

let p_token_node_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_token_node_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_error_node_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_error_node_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_char_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_char_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_int_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_int_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_bool_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_bool_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_bigint_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_bigint_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_string_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_string_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_symbol_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_symbol_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_property_error_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_property_error_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_regexp_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_regexp_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_entity_gen_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_entity_gen_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_entity_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_entity_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_logicvar_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_logicvar_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_equation_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_equation_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_array_gen_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_array_gen_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_array_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_array_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_astlist_gen_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_astlist_gen_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_astlist_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_astlist_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_node_builder_gen_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_node_builder_gen_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_node_builder_type
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_node_builder_type
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_named_type_decl (context node) (!@ result_ptr))

let p_iterator_gen_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_iterator_gen_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_iterator_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_iterator_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_trait_decl (context node) (!@ result_ptr))

let p_analysis_unit_gen_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_analysis_unit_gen_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_generic_decl (context node) (!@ result_ptr))

let p_analysis_unit_trait
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_analysis_unit_trait
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_trait_decl (context node) (!@ result_ptr))

let p_topmost_invalid_decl
    (node)
    =
      let result_ptr =
        allocate_n EntityStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_topmost_invalid_decl
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_lkt_node (context node) (!@ result_ptr))

let p_nameres_diagnostics
    (node)
    =
      let result_ptr =
        allocate_n SolverDiagnosticArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_nameres_diagnostics
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      SolverDiagnosticArray.wrap ~dec_ref:true (context node) (!@ result_ptr)

let p_solve_enclosing_context
    (node)
    =
      let result_ptr =
        allocate_n SolverResultStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_solve_enclosing_context
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      SolverResult.wrap ~dec_ref:true (context node) (!@ result_ptr)

let p_xref_entry_point
    (node)
    =
      let result_ptr =
        allocate_n bool ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_xref_entry_point
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      !@ result_ptr

let p_complete
    (node)
    =
      let result_ptr =
        allocate_n CompleteItemArrayStruct.c_type ~count:1
      in
      (* The result of this call should already be checked by the raisable
         mechanism *)
      let _ : int =
        CFunctions.lkt_node_p_complete
          (addr (unwrap_lkt_node (node)))
          (result_ptr)
      in
      CompleteItemArray.wrap ~dec_ref:true (context node) (!@ result_ptr)





  let kind_name = function
    | #var_bind ->
        "VarBind"
    | #simple_type_ref ->
        "SimpleTypeRef"
    | #generic_type_ref ->
        "GenericTypeRef"
    | #function_type_ref ->
        "FunctionTypeRef"
    | #default_list_type_ref ->
        "DefaultListTypeRef"
    | #selector_call ->
        "SelectorCall"
    | #selector_pattern_detail ->
        "SelectorPatternDetail"
    | #property_pattern_detail ->
        "PropertyPatternDetail"
    | #field_pattern_detail ->
        "FieldPatternDetail"
    | #type_pattern ->
        "TypePattern"
    | #tuple_pattern ->
        "TuplePattern"
    | #regex_pattern ->
        "RegexPattern"
    | #paren_pattern ->
        "ParenPattern"
    | #or_pattern ->
        "OrPattern"
    | #null_pattern ->
        "NullPattern"
    | #not_pattern ->
        "NotPattern"
    | #list_pattern ->
        "ListPattern"
    | #integer_pattern ->
        "IntegerPattern"
    | #filtered_pattern ->
        "FilteredPattern"
    | #extended_pattern ->
        "ExtendedPattern"
    | #ellipsis_pattern ->
        "EllipsisPattern"
    | #bool_pattern_true ->
        "BoolPatternTrue"
    | #bool_pattern_false ->
        "BoolPatternFalse"
    | #binding_pattern ->
        "BindingPattern"
    | #any_type_pattern ->
        "AnyTypePattern"
    | #op_plus ->
        "OpPlus"
    | #op_or_int ->
        "OpOrInt"
    | #op_or ->
        "OpOr"
    | #op_ne ->
        "OpNe"
    | #op_mult ->
        "OpMult"
    | #op_minus ->
        "OpMinus"
    | #op_lte ->
        "OpLte"
    | #op_lt ->
        "OpLt"
    | #op_logic_or ->
        "OpLogicOr"
    | #op_logic_and ->
        "OpLogicAnd"
    | #op_gte ->
        "OpGte"
    | #op_gt ->
        "OpGt"
    | #op_eq ->
        "OpEq"
    | #op_div ->
        "OpDiv"
    | #op_and ->
        "OpAnd"
    | #op_amp ->
        "OpAmp"
    | #null_cond_qualifier_present ->
        "NullCondQualifierPresent"
    | #null_cond_qualifier_absent ->
        "NullCondQualifierAbsent"
    | #synthetic_type_ref_list ->
        "SyntheticTypeRefList"
    | #type_ref_list ->
        "TypeRefList"
    | #ref_id_list ->
        "RefIdList"
    | #pattern_list ->
        "PatternList"
    | #pattern_detail_list ->
        "PatternDetailList"
    | #lkt_node_list ->
        "LktNodeList"
    | #lambda_param_decl_list ->
        "LambdaParamDeclList"
    | #import_list ->
        "ImportList"
    | #grammar_expr_list_list ->
        "GrammarExprListList"
    | #grammar_expr_list ->
        "GrammarExprList"
    | #fun_param_decl_list ->
        "FunParamDeclList"
    | #generic_param_decl_list ->
        "GenericParamDeclList"
    | #decl_block ->
        "DeclBlock"
    | #full_decl_list ->
        "FullDeclList"
    | #any_of_list ->
        "AnyOfList"
    | #expr_list ->
        "ExprList"
    | #enum_lit_decl_list ->
        "EnumLitDeclList"
    | #enum_class_case_list ->
        "EnumClassCaseList"
    | #enum_class_alt_decl_list ->
        "EnumClassAltDeclList"
    | #elsif_branch_list ->
        "ElsifBranchList"
    | #decl_annotation_list ->
        "DeclAnnotationList"
    | #call_expr_list ->
        "CallExprList"
    | #block_string_line_list ->
        "BlockStringLineList"
    | #base_match_branch_list ->
        "BaseMatchBranchList"
    | #base_lexer_case_rule_alt_list ->
        "BaseLexerCaseRuleAltList"
    | #argument_list ->
        "ArgumentList"
    | #list_kind_zero ->
        "ListKindZero"
    | #list_kind_one ->
        "ListKindOne"
    | #lexer_case_rule_send ->
        "LexerCaseRuleSend"
    | #lexer_case_rule ->
        "LexerCaseRule"
    | #langkit_root ->
        "LangkitRoot"
    | #import ->
        "Import"
    | #grammar_list_sep ->
        "GrammarListSep"
    | #full_decl ->
        "FullDecl"
    | #un_op ->
        "UnOp"
    | #try_expr ->
        "TryExpr"
    | #subscript_expr ->
        "SubscriptExpr"
    | #raise_expr ->
        "RaiseExpr"
    | #paren_expr ->
        "ParenExpr"
    | #not_expr ->
        "NotExpr"
    | #match_expr ->
        "MatchExpr"
    | #logic_unify ->
        "LogicUnify"
    | #logic_propagate ->
        "LogicPropagate"
    | #logic_expr ->
        "LogicExpr"
    | #logic_assign ->
        "LogicAssign"
    | #pattern_single_line_string_lit ->
        "PatternSingleLineStringLit"
    | #single_line_string_lit ->
        "SingleLineStringLit"
    | #block_string_lit ->
        "BlockStringLit"
    | #num_lit ->
        "NumLit"
    | #null_lit ->
        "NullLit"
    | #char_lit ->
        "CharLit"
    | #big_num_lit ->
        "BigNumLit"
    | #lambda_expr ->
        "LambdaExpr"
    | #keep_expr ->
        "KeepExpr"
    | #isa ->
        "Isa"
    | #if_expr ->
        "IfExpr"
    | #ref_id ->
        "RefId"
    | #module_ref_id ->
        "ModuleRefId"
    | #def_id ->
        "DefId"
    | #id ->
        "Id"
    | #token_ref ->
        "TokenRef"
    | #token_pattern_lit ->
        "TokenPatternLit"
    | #token_pattern_concat ->
        "TokenPatternConcat"
    | #token_no_case_lit ->
        "TokenNoCaseLit"
    | #token_lit ->
        "TokenLit"
    | #parse_node_expr ->
        "ParseNodeExpr"
    | #grammar_stop_cut ->
        "GrammarStopCut"
    | #grammar_skip ->
        "GrammarSkip"
    | #grammar_rule_ref ->
        "GrammarRuleRef"
    | #grammar_predicate ->
        "GrammarPredicate"
    | #grammar_implicit_pick ->
        "GrammarImplicitPick"
    | #grammar_pick ->
        "GrammarPick"
    | #grammar_or_expr ->
        "GrammarOrExpr"
    | #grammar_opt_group ->
        "GrammarOptGroup"
    | #grammar_opt_error_group ->
        "GrammarOptErrorGroup"
    | #grammar_opt_error ->
        "GrammarOptError"
    | #grammar_opt ->
        "GrammarOpt"
    | #grammar_null ->
        "GrammarNull"
    | #grammar_list ->
        "GrammarList"
    | #grammar_dont_skip ->
        "GrammarDontSkip"
    | #grammar_discard ->
        "GrammarDiscard"
    | #grammar_cut ->
        "GrammarCut"
    | #generic_instantiation ->
        "GenericInstantiation"
    | #error_on_null ->
        "ErrorOnNull"
    | #dot_expr ->
        "DotExpr"
    | #cast_expr ->
        "CastExpr"
    | #block_expr ->
        "BlockExpr"
    | #bin_op ->
        "BinOp"
    | #logic_propagate_call ->
        "LogicPropagateCall"
    | #logic_predicate ->
        "LogicPredicate"
    | #call_expr ->
        "CallExpr"
    | #array_literal ->
        "ArrayLiteral"
    | #any_of ->
        "AnyOf"
    | #excludes_null_present ->
        "ExcludesNullPresent"
    | #excludes_null_absent ->
        "ExcludesNullAbsent"
    | #enum_class_case ->
        "EnumClassCase"
    | #elsif_branch ->
        "ElsifBranch"
    | #dyn_env_wrapper ->
        "DynEnvWrapper"
    | #decl_annotation_args ->
        "DeclAnnotationArgs"
    | #decl_annotation ->
        "DeclAnnotation"
    | #trait_decl ->
        "TraitDecl"
    | #struct_decl ->
        "StructDecl"
    | #enum_type_decl ->
        "EnumTypeDecl"
    | #enum_class_decl ->
        "EnumClassDecl"
    | #class_decl ->
        "ClassDecl"
    | #generic_param_type_decl ->
        "GenericParamTypeDecl"
    | #function_type ->
        "FunctionType"
    | #enum_class_alt_decl ->
        "EnumClassAltDecl"
    | #any_type_decl ->
        "AnyTypeDecl"
    | #synth_param_decl ->
        "SynthParamDecl"
    | #synth_fun_decl ->
        "SynthFunDecl"
    | #lexer_family_decl ->
        "LexerFamilyDecl"
    | #lexer_decl ->
        "LexerDecl"
    | #grammar_decl ->
        "GrammarDecl"
    | #generic_decl ->
        "GenericDecl"
    | #error_decl ->
        "ErrorDecl"
    | #env_spec_decl ->
        "EnvSpecDecl"
    | #fun_decl ->
        "FunDecl"
    | #val_decl ->
        "ValDecl"
    | #match_val_decl ->
        "MatchValDecl"
    | #dyn_var_decl ->
        "DynVarDecl"
    | #lambda_param_decl ->
        "LambdaParamDecl"
    | #fun_param_decl ->
        "FunParamDecl"
    | #field_decl ->
        "FieldDecl"
    | #enum_lit_decl ->
        "EnumLitDecl"
    | #binding_val_decl ->
        "BindingValDecl"
    | #self_decl ->
        "SelfDecl"
    | #node_decl ->
        "NodeDecl"
    | #synthetic_lexer_decl ->
        "SyntheticLexerDecl"
    | #grammar_rule_decl ->
        "GrammarRuleDecl"
    | #class_qualifier_present ->
        "ClassQualifierPresent"
    | #class_qualifier_absent ->
        "ClassQualifierAbsent"
    | #block_string_line ->
        "BlockStringLine"
    | #block_expr_clause ->
        "BlockExprClause"
    | #pattern_match_branch ->
        "PatternMatchBranch"
    | #match_branch ->
        "MatchBranch"
    | #lexer_case_rule_default_alt ->
        "LexerCaseRuleDefaultAlt"
    | #lexer_case_rule_cond_alt ->
        "LexerCaseRuleCondAlt"
    | #argument ->
        "Argument"

  let text node =
    match token_start node, token_end node with
    | Some tok_start, Some tok_end ->
        Token.text_range tok_start tok_end
    | _ ->
        ""

  let image node =
    let c_result_ptr = allocate_n Text.c_type ~count:1 in
    CFunctions.image
      (addr (unwrap_lkt_node (node)))
      c_result_ptr;
    (Text.wrap (!@ c_result_ptr))

  let is_token_node node =
    let node_c_value = unwrap_lkt_node (node) in
    CFunctions.node_is_token_node (addr node_c_value)

  let sloc_range node =
    let c_result_ptr = allocate_n SlocRange.c_type ~count:1 in
    CFunctions.node_sloc_range
      (addr (unwrap_lkt_node (node)))
      c_result_ptr;
    !@ c_result_ptr

  
  let fold_tokens f init node =
    match token_start node, token_end node with
    | Some tok_start, Some tok_end ->
        let rec aux acc tok_curr =
          let new_acc = f acc tok_curr in
          if Token.equal tok_curr tok_end then
            new_acc
          else (
            match Token.next tok_curr with
            | Some tok_next ->
                aux new_acc tok_next
            | None ->
                new_acc )
        in
        aux init tok_start
    | _ ->
        init

  let iter_tokens f node =
    match token_start node, token_end node with
    | Some tok_start, Some tok_end ->
        let rec aux tok_curr =
          f tok_curr;
          if not (Token.equal tok_curr tok_end) then (
            match Token.next tok_curr with
            | Some tok_next ->
                aux tok_next
            | None ->
                () )
        in
        aux tok_start
    | _ ->
        ()

  let map_tokens f node =
    match token_start node, token_end node with
    | Some tok_start, Some tok_end ->
        let rec aux tok_curr =
          let value = f tok_curr in
          if Token.equal tok_curr tok_end then
            [value]
          else (
            match Token.next tok_curr with
            | Some tok_next ->
                value :: aux tok_next
            | None ->
                [value] )
        in
        aux tok_start
    | _ ->
        []

  let tokens node =
    map_tokens (fun x -> x) node


  let lookup node sloc =
    let node_c_value = unwrap_lkt_node (node) in
    let sloc_ptr = allocate Sloc.c_type sloc in
    let result_ptr = allocate_n EntityStruct.c_type ~count:1 in
    CFunctions.lookup_in_node
      (addr node_c_value) sloc_ptr result_ptr;
    if is_null (getf !@ result_ptr EntityStruct.node) then None else Some (wrap_lkt_node (context node) (!@ result_ptr))

  let children_opt node =
    let node_c_value = unwrap_lkt_node (node) in
    let context = context node in
    let c_value_ptr = allocate_n LktNodeArrayStruct.c_type ~count:1 in
    let _ : int =
      CFunctions.lkt_node_children
        (addr node_c_value)
        (c_value_ptr)
    in
    let c_value = !@(!@(c_value_ptr)) in
    let length = getf c_value LktNodeArrayStruct.n in
    let items = c_value @. LktNodeArrayStruct.items in
    let f i =
      let fresh = allocate EntityStruct.c_type !@(items +@ i) in
      if is_null (getf !@ fresh EntityStruct.node) then None else Some (wrap_lkt_node context (!@ fresh))
    in
    let result = List.init length f in
    LktNodeArrayStruct.dec_ref (!@ c_value_ptr);
    result

  let iter_fields f node =
    children_opt (node :> lkt_node)
    |> List.iter (function None -> () | Some node -> f node)

  let fold_fields f acc node =
    children_opt (node :> lkt_node)
    |> List.fold_left (fun x -> function None -> x | Some node -> f x node) acc

  let exists_fields p node =
    children_opt (node :> lkt_node)
    |> List.exists (function | None -> false | Some node -> p node)

  let for_all_fields p node =
    children_opt (node :> lkt_node)
    |> List.for_all (function | None -> true | Some node -> p node)

  let fold f acc node =
    (* Use an auxiliary function here to have a better type for the function *)
    let rec aux acc node = fold_fields aux (f acc node) node in
    aux acc (node :> lkt_node)

  let iter f node =
    (* Use an auxiliary function here to have a better type for the function *)
    let rec aux node = f node; iter_fields aux node in
    aux (node :> lkt_node)

  let filter p node =
    fold (fun acc node -> if p node then node :: acc else acc) [] node
    |> List.rev

  let exists p node =
    (* Use an auxiliary function here to have a better type for the function *)
    let rec aux node =
      p node || exists_fields aux node in aux (node :> lkt_node)

  let for_all p node =
    (* Use an auxiliary function here to have a better type for the function *)
    let rec aux node = p node && for_all_fields aux node in
    aux (node :> lkt_node)

  let lookup_with_kind :
    type a. a node -> [< lkt_node] -> Sloc.t -> a option =
    fun node_type node sloc ->
      let lookup_res = lookup node sloc in
      let rec aux : a node -> [< lkt_node] -> a option =
        fun node_type node ->
        match node_type, node with
        | LktNode
          , (#lkt_node as node) ->
          Some node
        | Argument
          , (#argument as node) ->
          Some node
        | BaseLexerCaseRuleAlt
          , (#base_lexer_case_rule_alt as node) ->
          Some node
        | LexerCaseRuleCondAlt
          , (#lexer_case_rule_cond_alt as node) ->
          Some node
        | LexerCaseRuleDefaultAlt
          , (#lexer_case_rule_default_alt as node) ->
          Some node
        | BaseMatchBranch
          , (#base_match_branch as node) ->
          Some node
        | MatchBranch
          , (#match_branch as node) ->
          Some node
        | PatternMatchBranch
          , (#pattern_match_branch as node) ->
          Some node
        | BlockExprClause
          , (#block_expr_clause as node) ->
          Some node
        | BlockStringLine
          , (#block_string_line as node) ->
          Some node
        | ClassQualifier
          , (#class_qualifier as node) ->
          Some node
        | ClassQualifierAbsent
          , (#class_qualifier_absent as node) ->
          Some node
        | ClassQualifierPresent
          , (#class_qualifier_present as node) ->
          Some node
        | Decl
          , (#decl as node) ->
          Some node
        | BaseGrammarRuleDecl
          , (#base_grammar_rule_decl as node) ->
          Some node
        | GrammarRuleDecl
          , (#grammar_rule_decl as node) ->
          Some node
        | SyntheticLexerDecl
          , (#synthetic_lexer_decl as node) ->
          Some node
        | BaseValDecl
          , (#base_val_decl as node) ->
          Some node
        | NodeDecl
          , (#node_decl as node) ->
          Some node
        | SelfDecl
          , (#self_decl as node) ->
          Some node
        | UserValDecl
          , (#user_val_decl as node) ->
          Some node
        | BindingValDecl
          , (#binding_val_decl as node) ->
          Some node
        | EnumLitDecl
          , (#enum_lit_decl as node) ->
          Some node
        | ExplicitlyTypedDecl
          , (#explicitly_typed_decl as node) ->
          Some node
        | ComponentDecl
          , (#component_decl as node) ->
          Some node
        | FieldDecl
          , (#field_decl as node) ->
          Some node
        | FunParamDecl
          , (#fun_param_decl as node) ->
          Some node
        | LambdaParamDecl
          , (#lambda_param_decl as node) ->
          Some node
        | DynVarDecl
          , (#dyn_var_decl as node) ->
          Some node
        | MatchValDecl
          , (#match_val_decl as node) ->
          Some node
        | ValDecl
          , (#val_decl as node) ->
          Some node
        | FunDecl
          , (#fun_decl as node) ->
          Some node
        | EnvSpecDecl
          , (#env_spec_decl as node) ->
          Some node
        | ErrorDecl
          , (#error_decl as node) ->
          Some node
        | GenericDecl
          , (#generic_decl as node) ->
          Some node
        | GrammarDecl
          , (#grammar_decl as node) ->
          Some node
        | LexerDecl
          , (#lexer_decl as node) ->
          Some node
        | LexerFamilyDecl
          , (#lexer_family_decl as node) ->
          Some node
        | SynthFunDecl
          , (#synth_fun_decl as node) ->
          Some node
        | SynthParamDecl
          , (#synth_param_decl as node) ->
          Some node
        | TypeDecl
          , (#type_decl as node) ->
          Some node
        | AnyTypeDecl
          , (#any_type_decl as node) ->
          Some node
        | EnumClassAltDecl
          , (#enum_class_alt_decl as node) ->
          Some node
        | FunctionType
          , (#function_type as node) ->
          Some node
        | GenericParamTypeDecl
          , (#generic_param_type_decl as node) ->
          Some node
        | NamedTypeDecl
          , (#named_type_decl as node) ->
          Some node
        | BasicClassDecl
          , (#basic_class_decl as node) ->
          Some node
        | ClassDecl
          , (#class_decl as node) ->
          Some node
        | EnumClassDecl
          , (#enum_class_decl as node) ->
          Some node
        | EnumTypeDecl
          , (#enum_type_decl as node) ->
          Some node
        | StructDecl
          , (#struct_decl as node) ->
          Some node
        | TraitDecl
          , (#trait_decl as node) ->
          Some node
        | DeclAnnotation
          , (#decl_annotation as node) ->
          Some node
        | DeclAnnotationArgs
          , (#decl_annotation_args as node) ->
          Some node
        | DynEnvWrapper
          , (#dyn_env_wrapper as node) ->
          Some node
        | ElsifBranch
          , (#elsif_branch as node) ->
          Some node
        | EnumClassCase
          , (#enum_class_case as node) ->
          Some node
        | ExcludesNull
          , (#excludes_null as node) ->
          Some node
        | ExcludesNullAbsent
          , (#excludes_null_absent as node) ->
          Some node
        | ExcludesNullPresent
          , (#excludes_null_present as node) ->
          Some node
        | Expr
          , (#expr as node) ->
          Some node
        | AnyOf
          , (#any_of as node) ->
          Some node
        | ArrayLiteral
          , (#array_literal as node) ->
          Some node
        | BaseCallExpr
          , (#base_call_expr as node) ->
          Some node
        | CallExpr
          , (#call_expr as node) ->
          Some node
        | LogicCallExpr
          , (#logic_call_expr as node) ->
          Some node
        | LogicPredicate
          , (#logic_predicate as node) ->
          Some node
        | LogicPropagateCall
          , (#logic_propagate_call as node) ->
          Some node
        | BinOp
          , (#bin_op as node) ->
          Some node
        | BlockExpr
          , (#block_expr as node) ->
          Some node
        | CastExpr
          , (#cast_expr as node) ->
          Some node
        | DotExpr
          , (#dot_expr as node) ->
          Some node
        | ErrorOnNull
          , (#error_on_null as node) ->
          Some node
        | GenericInstantiation
          , (#generic_instantiation as node) ->
          Some node
        | GrammarExpr
          , (#grammar_expr as node) ->
          Some node
        | GrammarCut
          , (#grammar_cut as node) ->
          Some node
        | GrammarDiscard
          , (#grammar_discard as node) ->
          Some node
        | GrammarDontSkip
          , (#grammar_dont_skip as node) ->
          Some node
        | GrammarList
          , (#grammar_list as node) ->
          Some node
        | GrammarNull
          , (#grammar_null as node) ->
          Some node
        | GrammarOpt
          , (#grammar_opt as node) ->
          Some node
        | GrammarOptError
          , (#grammar_opt_error as node) ->
          Some node
        | GrammarOptErrorGroup
          , (#grammar_opt_error_group as node) ->
          Some node
        | GrammarOptGroup
          , (#grammar_opt_group as node) ->
          Some node
        | GrammarOrExpr
          , (#grammar_or_expr as node) ->
          Some node
        | GrammarPick
          , (#grammar_pick as node) ->
          Some node
        | GrammarImplicitPick
          , (#grammar_implicit_pick as node) ->
          Some node
        | GrammarPredicate
          , (#grammar_predicate as node) ->
          Some node
        | GrammarRuleRef
          , (#grammar_rule_ref as node) ->
          Some node
        | GrammarSkip
          , (#grammar_skip as node) ->
          Some node
        | GrammarStopCut
          , (#grammar_stop_cut as node) ->
          Some node
        | ParseNodeExpr
          , (#parse_node_expr as node) ->
          Some node
        | TokenLit
          , (#token_lit as node) ->
          Some node
        | TokenNoCaseLit
          , (#token_no_case_lit as node) ->
          Some node
        | TokenPatternConcat
          , (#token_pattern_concat as node) ->
          Some node
        | TokenPatternLit
          , (#token_pattern_lit as node) ->
          Some node
        | TokenRef
          , (#token_ref as node) ->
          Some node
        | Id
          , (#id as node) ->
          Some node
        | DefId
          , (#def_id as node) ->
          Some node
        | ModuleRefId
          , (#module_ref_id as node) ->
          Some node
        | RefId
          , (#ref_id as node) ->
          Some node
        | IfExpr
          , (#if_expr as node) ->
          Some node
        | Isa
          , (#isa as node) ->
          Some node
        | KeepExpr
          , (#keep_expr as node) ->
          Some node
        | LambdaExpr
          , (#lambda_expr as node) ->
          Some node
        | Lit
          , (#lit as node) ->
          Some node
        | BigNumLit
          , (#big_num_lit as node) ->
          Some node
        | CharLit
          , (#char_lit as node) ->
          Some node
        | NullLit
          , (#null_lit as node) ->
          Some node
        | NumLit
          , (#num_lit as node) ->
          Some node
        | StringLit
          , (#string_lit as node) ->
          Some node
        | BlockStringLit
          , (#block_string_lit as node) ->
          Some node
        | SingleLineStringLit
          , (#single_line_string_lit as node) ->
          Some node
        | PatternSingleLineStringLit
          , (#pattern_single_line_string_lit as node) ->
          Some node
        | LogicAssign
          , (#logic_assign as node) ->
          Some node
        | LogicExpr
          , (#logic_expr as node) ->
          Some node
        | LogicPropagate
          , (#logic_propagate as node) ->
          Some node
        | LogicUnify
          , (#logic_unify as node) ->
          Some node
        | MatchExpr
          , (#match_expr as node) ->
          Some node
        | NotExpr
          , (#not_expr as node) ->
          Some node
        | ParenExpr
          , (#paren_expr as node) ->
          Some node
        | RaiseExpr
          , (#raise_expr as node) ->
          Some node
        | SubscriptExpr
          , (#subscript_expr as node) ->
          Some node
        | TryExpr
          , (#try_expr as node) ->
          Some node
        | UnOp
          , (#un_op as node) ->
          Some node
        | FullDecl
          , (#full_decl as node) ->
          Some node
        | GrammarListSep
          , (#grammar_list_sep as node) ->
          Some node
        | Import
          , (#import as node) ->
          Some node
        | LangkitRoot
          , (#langkit_root as node) ->
          Some node
        | LexerCaseRule
          , (#lexer_case_rule as node) ->
          Some node
        | LexerCaseRuleSend
          , (#lexer_case_rule_send as node) ->
          Some node
        | ListKind
          , (#list_kind as node) ->
          Some node
        | ListKindOne
          , (#list_kind_one as node) ->
          Some node
        | ListKindZero
          , (#list_kind_zero as node) ->
          Some node
        | LktNodeBaseList
          , (#lkt_node_base_list as node) ->
          Some node
        | ArgumentList
          , (#argument_list as node) ->
          Some node
        | BaseLexerCaseRuleAltList
          , (#base_lexer_case_rule_alt_list as node) ->
          Some node
        | BaseMatchBranchList
          , (#base_match_branch_list as node) ->
          Some node
        | BlockStringLineList
          , (#block_string_line_list as node) ->
          Some node
        | CallExprList
          , (#call_expr_list as node) ->
          Some node
        | DeclAnnotationList
          , (#decl_annotation_list as node) ->
          Some node
        | ElsifBranchList
          , (#elsif_branch_list as node) ->
          Some node
        | EnumClassAltDeclList
          , (#enum_class_alt_decl_list as node) ->
          Some node
        | EnumClassCaseList
          , (#enum_class_case_list as node) ->
          Some node
        | EnumLitDeclList
          , (#enum_lit_decl_list as node) ->
          Some node
        | ExprList
          , (#expr_list as node) ->
          Some node
        | AnyOfList
          , (#any_of_list as node) ->
          Some node
        | FullDeclList
          , (#full_decl_list as node) ->
          Some node
        | DeclBlock
          , (#decl_block as node) ->
          Some node
        | GenericParamDeclList
          , (#generic_param_decl_list as node) ->
          Some node
        | FunParamDeclList
          , (#fun_param_decl_list as node) ->
          Some node
        | GrammarExprList
          , (#grammar_expr_list as node) ->
          Some node
        | GrammarExprListList
          , (#grammar_expr_list_list as node) ->
          Some node
        | ImportList
          , (#import_list as node) ->
          Some node
        | LambdaParamDeclList
          , (#lambda_param_decl_list as node) ->
          Some node
        | LktNodeList
          , (#lkt_node_list as node) ->
          Some node
        | PatternDetailList
          , (#pattern_detail_list as node) ->
          Some node
        | PatternList
          , (#pattern_list as node) ->
          Some node
        | RefIdList
          , (#ref_id_list as node) ->
          Some node
        | TypeRefList
          , (#type_ref_list as node) ->
          Some node
        | SyntheticTypeRefList
          , (#synthetic_type_ref_list as node) ->
          Some node
        | NullCondQualifier
          , (#null_cond_qualifier as node) ->
          Some node
        | NullCondQualifierAbsent
          , (#null_cond_qualifier_absent as node) ->
          Some node
        | NullCondQualifierPresent
          , (#null_cond_qualifier_present as node) ->
          Some node
        | Op
          , (#op as node) ->
          Some node
        | OpAmp
          , (#op_amp as node) ->
          Some node
        | OpAnd
          , (#op_and as node) ->
          Some node
        | OpDiv
          , (#op_div as node) ->
          Some node
        | OpEq
          , (#op_eq as node) ->
          Some node
        | OpGt
          , (#op_gt as node) ->
          Some node
        | OpGte
          , (#op_gte as node) ->
          Some node
        | OpLogicAnd
          , (#op_logic_and as node) ->
          Some node
        | OpLogicOr
          , (#op_logic_or as node) ->
          Some node
        | OpLt
          , (#op_lt as node) ->
          Some node
        | OpLte
          , (#op_lte as node) ->
          Some node
        | OpMinus
          , (#op_minus as node) ->
          Some node
        | OpMult
          , (#op_mult as node) ->
          Some node
        | OpNe
          , (#op_ne as node) ->
          Some node
        | OpOr
          , (#op_or as node) ->
          Some node
        | OpOrInt
          , (#op_or_int as node) ->
          Some node
        | OpPlus
          , (#op_plus as node) ->
          Some node
        | Pattern
          , (#pattern as node) ->
          Some node
        | AnyTypePattern
          , (#any_type_pattern as node) ->
          Some node
        | BindingPattern
          , (#binding_pattern as node) ->
          Some node
        | BoolPattern
          , (#bool_pattern as node) ->
          Some node
        | BoolPatternFalse
          , (#bool_pattern_false as node) ->
          Some node
        | BoolPatternTrue
          , (#bool_pattern_true as node) ->
          Some node
        | EllipsisPattern
          , (#ellipsis_pattern as node) ->
          Some node
        | ExtendedPattern
          , (#extended_pattern as node) ->
          Some node
        | FilteredPattern
          , (#filtered_pattern as node) ->
          Some node
        | IntegerPattern
          , (#integer_pattern as node) ->
          Some node
        | ListPattern
          , (#list_pattern as node) ->
          Some node
        | NotPattern
          , (#not_pattern as node) ->
          Some node
        | NullPattern
          , (#null_pattern as node) ->
          Some node
        | OrPattern
          , (#or_pattern as node) ->
          Some node
        | ParenPattern
          , (#paren_pattern as node) ->
          Some node
        | RegexPattern
          , (#regex_pattern as node) ->
          Some node
        | TuplePattern
          , (#tuple_pattern as node) ->
          Some node
        | TypePattern
          , (#type_pattern as node) ->
          Some node
        | PatternDetail
          , (#pattern_detail as node) ->
          Some node
        | FieldPatternDetail
          , (#field_pattern_detail as node) ->
          Some node
        | PropertyPatternDetail
          , (#property_pattern_detail as node) ->
          Some node
        | SelectorPatternDetail
          , (#selector_pattern_detail as node) ->
          Some node
        | SelectorCall
          , (#selector_call as node) ->
          Some node
        | TypeRef
          , (#type_ref as node) ->
          Some node
        | DefaultListTypeRef
          , (#default_list_type_ref as node) ->
          Some node
        | FunctionTypeRef
          , (#function_type_ref as node) ->
          Some node
        | GenericTypeRef
          , (#generic_type_ref as node) ->
          Some node
        | SimpleTypeRef
          , (#simple_type_ref as node) ->
          Some node
        | VarBind
          , (#var_bind as node) ->
          Some node
        | _ -> (match parent node with
                | Some parent_node -> aux node_type parent_node
                | _ -> None) in
    match lookup_res with
      | Some node -> aux node_type node
      | _ -> None

  let as_a : type a. a node -> [< lkt_node ] -> a option =
   fun node_type node ->
    match node_type, (node :> lkt_node) with
    | LktNode
      , (#lkt_node as node) ->
        Some node
    | Argument
      , (#argument as node) ->
        Some node
    | BaseLexerCaseRuleAlt
      , (#base_lexer_case_rule_alt as node) ->
        Some node
    | LexerCaseRuleCondAlt
      , (#lexer_case_rule_cond_alt as node) ->
        Some node
    | LexerCaseRuleDefaultAlt
      , (#lexer_case_rule_default_alt as node) ->
        Some node
    | BaseMatchBranch
      , (#base_match_branch as node) ->
        Some node
    | MatchBranch
      , (#match_branch as node) ->
        Some node
    | PatternMatchBranch
      , (#pattern_match_branch as node) ->
        Some node
    | BlockExprClause
      , (#block_expr_clause as node) ->
        Some node
    | BlockStringLine
      , (#block_string_line as node) ->
        Some node
    | ClassQualifier
      , (#class_qualifier as node) ->
        Some node
    | ClassQualifierAbsent
      , (#class_qualifier_absent as node) ->
        Some node
    | ClassQualifierPresent
      , (#class_qualifier_present as node) ->
        Some node
    | Decl
      , (#decl as node) ->
        Some node
    | BaseGrammarRuleDecl
      , (#base_grammar_rule_decl as node) ->
        Some node
    | GrammarRuleDecl
      , (#grammar_rule_decl as node) ->
        Some node
    | SyntheticLexerDecl
      , (#synthetic_lexer_decl as node) ->
        Some node
    | BaseValDecl
      , (#base_val_decl as node) ->
        Some node
    | NodeDecl
      , (#node_decl as node) ->
        Some node
    | SelfDecl
      , (#self_decl as node) ->
        Some node
    | UserValDecl
      , (#user_val_decl as node) ->
        Some node
    | BindingValDecl
      , (#binding_val_decl as node) ->
        Some node
    | EnumLitDecl
      , (#enum_lit_decl as node) ->
        Some node
    | ExplicitlyTypedDecl
      , (#explicitly_typed_decl as node) ->
        Some node
    | ComponentDecl
      , (#component_decl as node) ->
        Some node
    | FieldDecl
      , (#field_decl as node) ->
        Some node
    | FunParamDecl
      , (#fun_param_decl as node) ->
        Some node
    | LambdaParamDecl
      , (#lambda_param_decl as node) ->
        Some node
    | DynVarDecl
      , (#dyn_var_decl as node) ->
        Some node
    | MatchValDecl
      , (#match_val_decl as node) ->
        Some node
    | ValDecl
      , (#val_decl as node) ->
        Some node
    | FunDecl
      , (#fun_decl as node) ->
        Some node
    | EnvSpecDecl
      , (#env_spec_decl as node) ->
        Some node
    | ErrorDecl
      , (#error_decl as node) ->
        Some node
    | GenericDecl
      , (#generic_decl as node) ->
        Some node
    | GrammarDecl
      , (#grammar_decl as node) ->
        Some node
    | LexerDecl
      , (#lexer_decl as node) ->
        Some node
    | LexerFamilyDecl
      , (#lexer_family_decl as node) ->
        Some node
    | SynthFunDecl
      , (#synth_fun_decl as node) ->
        Some node
    | SynthParamDecl
      , (#synth_param_decl as node) ->
        Some node
    | TypeDecl
      , (#type_decl as node) ->
        Some node
    | AnyTypeDecl
      , (#any_type_decl as node) ->
        Some node
    | EnumClassAltDecl
      , (#enum_class_alt_decl as node) ->
        Some node
    | FunctionType
      , (#function_type as node) ->
        Some node
    | GenericParamTypeDecl
      , (#generic_param_type_decl as node) ->
        Some node
    | NamedTypeDecl
      , (#named_type_decl as node) ->
        Some node
    | BasicClassDecl
      , (#basic_class_decl as node) ->
        Some node
    | ClassDecl
      , (#class_decl as node) ->
        Some node
    | EnumClassDecl
      , (#enum_class_decl as node) ->
        Some node
    | EnumTypeDecl
      , (#enum_type_decl as node) ->
        Some node
    | StructDecl
      , (#struct_decl as node) ->
        Some node
    | TraitDecl
      , (#trait_decl as node) ->
        Some node
    | DeclAnnotation
      , (#decl_annotation as node) ->
        Some node
    | DeclAnnotationArgs
      , (#decl_annotation_args as node) ->
        Some node
    | DynEnvWrapper
      , (#dyn_env_wrapper as node) ->
        Some node
    | ElsifBranch
      , (#elsif_branch as node) ->
        Some node
    | EnumClassCase
      , (#enum_class_case as node) ->
        Some node
    | ExcludesNull
      , (#excludes_null as node) ->
        Some node
    | ExcludesNullAbsent
      , (#excludes_null_absent as node) ->
        Some node
    | ExcludesNullPresent
      , (#excludes_null_present as node) ->
        Some node
    | Expr
      , (#expr as node) ->
        Some node
    | AnyOf
      , (#any_of as node) ->
        Some node
    | ArrayLiteral
      , (#array_literal as node) ->
        Some node
    | BaseCallExpr
      , (#base_call_expr as node) ->
        Some node
    | CallExpr
      , (#call_expr as node) ->
        Some node
    | LogicCallExpr
      , (#logic_call_expr as node) ->
        Some node
    | LogicPredicate
      , (#logic_predicate as node) ->
        Some node
    | LogicPropagateCall
      , (#logic_propagate_call as node) ->
        Some node
    | BinOp
      , (#bin_op as node) ->
        Some node
    | BlockExpr
      , (#block_expr as node) ->
        Some node
    | CastExpr
      , (#cast_expr as node) ->
        Some node
    | DotExpr
      , (#dot_expr as node) ->
        Some node
    | ErrorOnNull
      , (#error_on_null as node) ->
        Some node
    | GenericInstantiation
      , (#generic_instantiation as node) ->
        Some node
    | GrammarExpr
      , (#grammar_expr as node) ->
        Some node
    | GrammarCut
      , (#grammar_cut as node) ->
        Some node
    | GrammarDiscard
      , (#grammar_discard as node) ->
        Some node
    | GrammarDontSkip
      , (#grammar_dont_skip as node) ->
        Some node
    | GrammarList
      , (#grammar_list as node) ->
        Some node
    | GrammarNull
      , (#grammar_null as node) ->
        Some node
    | GrammarOpt
      , (#grammar_opt as node) ->
        Some node
    | GrammarOptError
      , (#grammar_opt_error as node) ->
        Some node
    | GrammarOptErrorGroup
      , (#grammar_opt_error_group as node) ->
        Some node
    | GrammarOptGroup
      , (#grammar_opt_group as node) ->
        Some node
    | GrammarOrExpr
      , (#grammar_or_expr as node) ->
        Some node
    | GrammarPick
      , (#grammar_pick as node) ->
        Some node
    | GrammarImplicitPick
      , (#grammar_implicit_pick as node) ->
        Some node
    | GrammarPredicate
      , (#grammar_predicate as node) ->
        Some node
    | GrammarRuleRef
      , (#grammar_rule_ref as node) ->
        Some node
    | GrammarSkip
      , (#grammar_skip as node) ->
        Some node
    | GrammarStopCut
      , (#grammar_stop_cut as node) ->
        Some node
    | ParseNodeExpr
      , (#parse_node_expr as node) ->
        Some node
    | TokenLit
      , (#token_lit as node) ->
        Some node
    | TokenNoCaseLit
      , (#token_no_case_lit as node) ->
        Some node
    | TokenPatternConcat
      , (#token_pattern_concat as node) ->
        Some node
    | TokenPatternLit
      , (#token_pattern_lit as node) ->
        Some node
    | TokenRef
      , (#token_ref as node) ->
        Some node
    | Id
      , (#id as node) ->
        Some node
    | DefId
      , (#def_id as node) ->
        Some node
    | ModuleRefId
      , (#module_ref_id as node) ->
        Some node
    | RefId
      , (#ref_id as node) ->
        Some node
    | IfExpr
      , (#if_expr as node) ->
        Some node
    | Isa
      , (#isa as node) ->
        Some node
    | KeepExpr
      , (#keep_expr as node) ->
        Some node
    | LambdaExpr
      , (#lambda_expr as node) ->
        Some node
    | Lit
      , (#lit as node) ->
        Some node
    | BigNumLit
      , (#big_num_lit as node) ->
        Some node
    | CharLit
      , (#char_lit as node) ->
        Some node
    | NullLit
      , (#null_lit as node) ->
        Some node
    | NumLit
      , (#num_lit as node) ->
        Some node
    | StringLit
      , (#string_lit as node) ->
        Some node
    | BlockStringLit
      , (#block_string_lit as node) ->
        Some node
    | SingleLineStringLit
      , (#single_line_string_lit as node) ->
        Some node
    | PatternSingleLineStringLit
      , (#pattern_single_line_string_lit as node) ->
        Some node
    | LogicAssign
      , (#logic_assign as node) ->
        Some node
    | LogicExpr
      , (#logic_expr as node) ->
        Some node
    | LogicPropagate
      , (#logic_propagate as node) ->
        Some node
    | LogicUnify
      , (#logic_unify as node) ->
        Some node
    | MatchExpr
      , (#match_expr as node) ->
        Some node
    | NotExpr
      , (#not_expr as node) ->
        Some node
    | ParenExpr
      , (#paren_expr as node) ->
        Some node
    | RaiseExpr
      , (#raise_expr as node) ->
        Some node
    | SubscriptExpr
      , (#subscript_expr as node) ->
        Some node
    | TryExpr
      , (#try_expr as node) ->
        Some node
    | UnOp
      , (#un_op as node) ->
        Some node
    | FullDecl
      , (#full_decl as node) ->
        Some node
    | GrammarListSep
      , (#grammar_list_sep as node) ->
        Some node
    | Import
      , (#import as node) ->
        Some node
    | LangkitRoot
      , (#langkit_root as node) ->
        Some node
    | LexerCaseRule
      , (#lexer_case_rule as node) ->
        Some node
    | LexerCaseRuleSend
      , (#lexer_case_rule_send as node) ->
        Some node
    | ListKind
      , (#list_kind as node) ->
        Some node
    | ListKindOne
      , (#list_kind_one as node) ->
        Some node
    | ListKindZero
      , (#list_kind_zero as node) ->
        Some node
    | LktNodeBaseList
      , (#lkt_node_base_list as node) ->
        Some node
    | ArgumentList
      , (#argument_list as node) ->
        Some node
    | BaseLexerCaseRuleAltList
      , (#base_lexer_case_rule_alt_list as node) ->
        Some node
    | BaseMatchBranchList
      , (#base_match_branch_list as node) ->
        Some node
    | BlockStringLineList
      , (#block_string_line_list as node) ->
        Some node
    | CallExprList
      , (#call_expr_list as node) ->
        Some node
    | DeclAnnotationList
      , (#decl_annotation_list as node) ->
        Some node
    | ElsifBranchList
      , (#elsif_branch_list as node) ->
        Some node
    | EnumClassAltDeclList
      , (#enum_class_alt_decl_list as node) ->
        Some node
    | EnumClassCaseList
      , (#enum_class_case_list as node) ->
        Some node
    | EnumLitDeclList
      , (#enum_lit_decl_list as node) ->
        Some node
    | ExprList
      , (#expr_list as node) ->
        Some node
    | AnyOfList
      , (#any_of_list as node) ->
        Some node
    | FullDeclList
      , (#full_decl_list as node) ->
        Some node
    | DeclBlock
      , (#decl_block as node) ->
        Some node
    | GenericParamDeclList
      , (#generic_param_decl_list as node) ->
        Some node
    | FunParamDeclList
      , (#fun_param_decl_list as node) ->
        Some node
    | GrammarExprList
      , (#grammar_expr_list as node) ->
        Some node
    | GrammarExprListList
      , (#grammar_expr_list_list as node) ->
        Some node
    | ImportList
      , (#import_list as node) ->
        Some node
    | LambdaParamDeclList
      , (#lambda_param_decl_list as node) ->
        Some node
    | LktNodeList
      , (#lkt_node_list as node) ->
        Some node
    | PatternDetailList
      , (#pattern_detail_list as node) ->
        Some node
    | PatternList
      , (#pattern_list as node) ->
        Some node
    | RefIdList
      , (#ref_id_list as node) ->
        Some node
    | TypeRefList
      , (#type_ref_list as node) ->
        Some node
    | SyntheticTypeRefList
      , (#synthetic_type_ref_list as node) ->
        Some node
    | NullCondQualifier
      , (#null_cond_qualifier as node) ->
        Some node
    | NullCondQualifierAbsent
      , (#null_cond_qualifier_absent as node) ->
        Some node
    | NullCondQualifierPresent
      , (#null_cond_qualifier_present as node) ->
        Some node
    | Op
      , (#op as node) ->
        Some node
    | OpAmp
      , (#op_amp as node) ->
        Some node
    | OpAnd
      , (#op_and as node) ->
        Some node
    | OpDiv
      , (#op_div as node) ->
        Some node
    | OpEq
      , (#op_eq as node) ->
        Some node
    | OpGt
      , (#op_gt as node) ->
        Some node
    | OpGte
      , (#op_gte as node) ->
        Some node
    | OpLogicAnd
      , (#op_logic_and as node) ->
        Some node
    | OpLogicOr
      , (#op_logic_or as node) ->
        Some node
    | OpLt
      , (#op_lt as node) ->
        Some node
    | OpLte
      , (#op_lte as node) ->
        Some node
    | OpMinus
      , (#op_minus as node) ->
        Some node
    | OpMult
      , (#op_mult as node) ->
        Some node
    | OpNe
      , (#op_ne as node) ->
        Some node
    | OpOr
      , (#op_or as node) ->
        Some node
    | OpOrInt
      , (#op_or_int as node) ->
        Some node
    | OpPlus
      , (#op_plus as node) ->
        Some node
    | Pattern
      , (#pattern as node) ->
        Some node
    | AnyTypePattern
      , (#any_type_pattern as node) ->
        Some node
    | BindingPattern
      , (#binding_pattern as node) ->
        Some node
    | BoolPattern
      , (#bool_pattern as node) ->
        Some node
    | BoolPatternFalse
      , (#bool_pattern_false as node) ->
        Some node
    | BoolPatternTrue
      , (#bool_pattern_true as node) ->
        Some node
    | EllipsisPattern
      , (#ellipsis_pattern as node) ->
        Some node
    | ExtendedPattern
      , (#extended_pattern as node) ->
        Some node
    | FilteredPattern
      , (#filtered_pattern as node) ->
        Some node
    | IntegerPattern
      , (#integer_pattern as node) ->
        Some node
    | ListPattern
      , (#list_pattern as node) ->
        Some node
    | NotPattern
      , (#not_pattern as node) ->
        Some node
    | NullPattern
      , (#null_pattern as node) ->
        Some node
    | OrPattern
      , (#or_pattern as node) ->
        Some node
    | ParenPattern
      , (#paren_pattern as node) ->
        Some node
    | RegexPattern
      , (#regex_pattern as node) ->
        Some node
    | TuplePattern
      , (#tuple_pattern as node) ->
        Some node
    | TypePattern
      , (#type_pattern as node) ->
        Some node
    | PatternDetail
      , (#pattern_detail as node) ->
        Some node
    | FieldPatternDetail
      , (#field_pattern_detail as node) ->
        Some node
    | PropertyPatternDetail
      , (#property_pattern_detail as node) ->
        Some node
    | SelectorPatternDetail
      , (#selector_pattern_detail as node) ->
        Some node
    | SelectorCall
      , (#selector_call as node) ->
        Some node
    | TypeRef
      , (#type_ref as node) ->
        Some node
    | DefaultListTypeRef
      , (#default_list_type_ref as node) ->
        Some node
    | FunctionTypeRef
      , (#function_type_ref as node) ->
        Some node
    | GenericTypeRef
      , (#generic_type_ref as node) ->
        Some node
    | SimpleTypeRef
      , (#simple_type_ref as node) ->
        Some node
    | VarBind
      , (#var_bind as node) ->
        Some node
    | _ ->
        None

  let find : type a. a node ->  [< lkt_node ] -> a =
    fun node_type node ->
      let exception Found of a in
      let aux node =
        match node_type, node with
        | LktNode
          , (#lkt_node as node) ->
            raise (Found node)
        | Argument
          , (#argument as node) ->
            raise (Found node)
        | BaseLexerCaseRuleAlt
          , (#base_lexer_case_rule_alt as node) ->
            raise (Found node)
        | LexerCaseRuleCondAlt
          , (#lexer_case_rule_cond_alt as node) ->
            raise (Found node)
        | LexerCaseRuleDefaultAlt
          , (#lexer_case_rule_default_alt as node) ->
            raise (Found node)
        | BaseMatchBranch
          , (#base_match_branch as node) ->
            raise (Found node)
        | MatchBranch
          , (#match_branch as node) ->
            raise (Found node)
        | PatternMatchBranch
          , (#pattern_match_branch as node) ->
            raise (Found node)
        | BlockExprClause
          , (#block_expr_clause as node) ->
            raise (Found node)
        | BlockStringLine
          , (#block_string_line as node) ->
            raise (Found node)
        | ClassQualifier
          , (#class_qualifier as node) ->
            raise (Found node)
        | ClassQualifierAbsent
          , (#class_qualifier_absent as node) ->
            raise (Found node)
        | ClassQualifierPresent
          , (#class_qualifier_present as node) ->
            raise (Found node)
        | Decl
          , (#decl as node) ->
            raise (Found node)
        | BaseGrammarRuleDecl
          , (#base_grammar_rule_decl as node) ->
            raise (Found node)
        | GrammarRuleDecl
          , (#grammar_rule_decl as node) ->
            raise (Found node)
        | SyntheticLexerDecl
          , (#synthetic_lexer_decl as node) ->
            raise (Found node)
        | BaseValDecl
          , (#base_val_decl as node) ->
            raise (Found node)
        | NodeDecl
          , (#node_decl as node) ->
            raise (Found node)
        | SelfDecl
          , (#self_decl as node) ->
            raise (Found node)
        | UserValDecl
          , (#user_val_decl as node) ->
            raise (Found node)
        | BindingValDecl
          , (#binding_val_decl as node) ->
            raise (Found node)
        | EnumLitDecl
          , (#enum_lit_decl as node) ->
            raise (Found node)
        | ExplicitlyTypedDecl
          , (#explicitly_typed_decl as node) ->
            raise (Found node)
        | ComponentDecl
          , (#component_decl as node) ->
            raise (Found node)
        | FieldDecl
          , (#field_decl as node) ->
            raise (Found node)
        | FunParamDecl
          , (#fun_param_decl as node) ->
            raise (Found node)
        | LambdaParamDecl
          , (#lambda_param_decl as node) ->
            raise (Found node)
        | DynVarDecl
          , (#dyn_var_decl as node) ->
            raise (Found node)
        | MatchValDecl
          , (#match_val_decl as node) ->
            raise (Found node)
        | ValDecl
          , (#val_decl as node) ->
            raise (Found node)
        | FunDecl
          , (#fun_decl as node) ->
            raise (Found node)
        | EnvSpecDecl
          , (#env_spec_decl as node) ->
            raise (Found node)
        | ErrorDecl
          , (#error_decl as node) ->
            raise (Found node)
        | GenericDecl
          , (#generic_decl as node) ->
            raise (Found node)
        | GrammarDecl
          , (#grammar_decl as node) ->
            raise (Found node)
        | LexerDecl
          , (#lexer_decl as node) ->
            raise (Found node)
        | LexerFamilyDecl
          , (#lexer_family_decl as node) ->
            raise (Found node)
        | SynthFunDecl
          , (#synth_fun_decl as node) ->
            raise (Found node)
        | SynthParamDecl
          , (#synth_param_decl as node) ->
            raise (Found node)
        | TypeDecl
          , (#type_decl as node) ->
            raise (Found node)
        | AnyTypeDecl
          , (#any_type_decl as node) ->
            raise (Found node)
        | EnumClassAltDecl
          , (#enum_class_alt_decl as node) ->
            raise (Found node)
        | FunctionType
          , (#function_type as node) ->
            raise (Found node)
        | GenericParamTypeDecl
          , (#generic_param_type_decl as node) ->
            raise (Found node)
        | NamedTypeDecl
          , (#named_type_decl as node) ->
            raise (Found node)
        | BasicClassDecl
          , (#basic_class_decl as node) ->
            raise (Found node)
        | ClassDecl
          , (#class_decl as node) ->
            raise (Found node)
        | EnumClassDecl
          , (#enum_class_decl as node) ->
            raise (Found node)
        | EnumTypeDecl
          , (#enum_type_decl as node) ->
            raise (Found node)
        | StructDecl
          , (#struct_decl as node) ->
            raise (Found node)
        | TraitDecl
          , (#trait_decl as node) ->
            raise (Found node)
        | DeclAnnotation
          , (#decl_annotation as node) ->
            raise (Found node)
        | DeclAnnotationArgs
          , (#decl_annotation_args as node) ->
            raise (Found node)
        | DynEnvWrapper
          , (#dyn_env_wrapper as node) ->
            raise (Found node)
        | ElsifBranch
          , (#elsif_branch as node) ->
            raise (Found node)
        | EnumClassCase
          , (#enum_class_case as node) ->
            raise (Found node)
        | ExcludesNull
          , (#excludes_null as node) ->
            raise (Found node)
        | ExcludesNullAbsent
          , (#excludes_null_absent as node) ->
            raise (Found node)
        | ExcludesNullPresent
          , (#excludes_null_present as node) ->
            raise (Found node)
        | Expr
          , (#expr as node) ->
            raise (Found node)
        | AnyOf
          , (#any_of as node) ->
            raise (Found node)
        | ArrayLiteral
          , (#array_literal as node) ->
            raise (Found node)
        | BaseCallExpr
          , (#base_call_expr as node) ->
            raise (Found node)
        | CallExpr
          , (#call_expr as node) ->
            raise (Found node)
        | LogicCallExpr
          , (#logic_call_expr as node) ->
            raise (Found node)
        | LogicPredicate
          , (#logic_predicate as node) ->
            raise (Found node)
        | LogicPropagateCall
          , (#logic_propagate_call as node) ->
            raise (Found node)
        | BinOp
          , (#bin_op as node) ->
            raise (Found node)
        | BlockExpr
          , (#block_expr as node) ->
            raise (Found node)
        | CastExpr
          , (#cast_expr as node) ->
            raise (Found node)
        | DotExpr
          , (#dot_expr as node) ->
            raise (Found node)
        | ErrorOnNull
          , (#error_on_null as node) ->
            raise (Found node)
        | GenericInstantiation
          , (#generic_instantiation as node) ->
            raise (Found node)
        | GrammarExpr
          , (#grammar_expr as node) ->
            raise (Found node)
        | GrammarCut
          , (#grammar_cut as node) ->
            raise (Found node)
        | GrammarDiscard
          , (#grammar_discard as node) ->
            raise (Found node)
        | GrammarDontSkip
          , (#grammar_dont_skip as node) ->
            raise (Found node)
        | GrammarList
          , (#grammar_list as node) ->
            raise (Found node)
        | GrammarNull
          , (#grammar_null as node) ->
            raise (Found node)
        | GrammarOpt
          , (#grammar_opt as node) ->
            raise (Found node)
        | GrammarOptError
          , (#grammar_opt_error as node) ->
            raise (Found node)
        | GrammarOptErrorGroup
          , (#grammar_opt_error_group as node) ->
            raise (Found node)
        | GrammarOptGroup
          , (#grammar_opt_group as node) ->
            raise (Found node)
        | GrammarOrExpr
          , (#grammar_or_expr as node) ->
            raise (Found node)
        | GrammarPick
          , (#grammar_pick as node) ->
            raise (Found node)
        | GrammarImplicitPick
          , (#grammar_implicit_pick as node) ->
            raise (Found node)
        | GrammarPredicate
          , (#grammar_predicate as node) ->
            raise (Found node)
        | GrammarRuleRef
          , (#grammar_rule_ref as node) ->
            raise (Found node)
        | GrammarSkip
          , (#grammar_skip as node) ->
            raise (Found node)
        | GrammarStopCut
          , (#grammar_stop_cut as node) ->
            raise (Found node)
        | ParseNodeExpr
          , (#parse_node_expr as node) ->
            raise (Found node)
        | TokenLit
          , (#token_lit as node) ->
            raise (Found node)
        | TokenNoCaseLit
          , (#token_no_case_lit as node) ->
            raise (Found node)
        | TokenPatternConcat
          , (#token_pattern_concat as node) ->
            raise (Found node)
        | TokenPatternLit
          , (#token_pattern_lit as node) ->
            raise (Found node)
        | TokenRef
          , (#token_ref as node) ->
            raise (Found node)
        | Id
          , (#id as node) ->
            raise (Found node)
        | DefId
          , (#def_id as node) ->
            raise (Found node)
        | ModuleRefId
          , (#module_ref_id as node) ->
            raise (Found node)
        | RefId
          , (#ref_id as node) ->
            raise (Found node)
        | IfExpr
          , (#if_expr as node) ->
            raise (Found node)
        | Isa
          , (#isa as node) ->
            raise (Found node)
        | KeepExpr
          , (#keep_expr as node) ->
            raise (Found node)
        | LambdaExpr
          , (#lambda_expr as node) ->
            raise (Found node)
        | Lit
          , (#lit as node) ->
            raise (Found node)
        | BigNumLit
          , (#big_num_lit as node) ->
            raise (Found node)
        | CharLit
          , (#char_lit as node) ->
            raise (Found node)
        | NullLit
          , (#null_lit as node) ->
            raise (Found node)
        | NumLit
          , (#num_lit as node) ->
            raise (Found node)
        | StringLit
          , (#string_lit as node) ->
            raise (Found node)
        | BlockStringLit
          , (#block_string_lit as node) ->
            raise (Found node)
        | SingleLineStringLit
          , (#single_line_string_lit as node) ->
            raise (Found node)
        | PatternSingleLineStringLit
          , (#pattern_single_line_string_lit as node) ->
            raise (Found node)
        | LogicAssign
          , (#logic_assign as node) ->
            raise (Found node)
        | LogicExpr
          , (#logic_expr as node) ->
            raise (Found node)
        | LogicPropagate
          , (#logic_propagate as node) ->
            raise (Found node)
        | LogicUnify
          , (#logic_unify as node) ->
            raise (Found node)
        | MatchExpr
          , (#match_expr as node) ->
            raise (Found node)
        | NotExpr
          , (#not_expr as node) ->
            raise (Found node)
        | ParenExpr
          , (#paren_expr as node) ->
            raise (Found node)
        | RaiseExpr
          , (#raise_expr as node) ->
            raise (Found node)
        | SubscriptExpr
          , (#subscript_expr as node) ->
            raise (Found node)
        | TryExpr
          , (#try_expr as node) ->
            raise (Found node)
        | UnOp
          , (#un_op as node) ->
            raise (Found node)
        | FullDecl
          , (#full_decl as node) ->
            raise (Found node)
        | GrammarListSep
          , (#grammar_list_sep as node) ->
            raise (Found node)
        | Import
          , (#import as node) ->
            raise (Found node)
        | LangkitRoot
          , (#langkit_root as node) ->
            raise (Found node)
        | LexerCaseRule
          , (#lexer_case_rule as node) ->
            raise (Found node)
        | LexerCaseRuleSend
          , (#lexer_case_rule_send as node) ->
            raise (Found node)
        | ListKind
          , (#list_kind as node) ->
            raise (Found node)
        | ListKindOne
          , (#list_kind_one as node) ->
            raise (Found node)
        | ListKindZero
          , (#list_kind_zero as node) ->
            raise (Found node)
        | LktNodeBaseList
          , (#lkt_node_base_list as node) ->
            raise (Found node)
        | ArgumentList
          , (#argument_list as node) ->
            raise (Found node)
        | BaseLexerCaseRuleAltList
          , (#base_lexer_case_rule_alt_list as node) ->
            raise (Found node)
        | BaseMatchBranchList
          , (#base_match_branch_list as node) ->
            raise (Found node)
        | BlockStringLineList
          , (#block_string_line_list as node) ->
            raise (Found node)
        | CallExprList
          , (#call_expr_list as node) ->
            raise (Found node)
        | DeclAnnotationList
          , (#decl_annotation_list as node) ->
            raise (Found node)
        | ElsifBranchList
          , (#elsif_branch_list as node) ->
            raise (Found node)
        | EnumClassAltDeclList
          , (#enum_class_alt_decl_list as node) ->
            raise (Found node)
        | EnumClassCaseList
          , (#enum_class_case_list as node) ->
            raise (Found node)
        | EnumLitDeclList
          , (#enum_lit_decl_list as node) ->
            raise (Found node)
        | ExprList
          , (#expr_list as node) ->
            raise (Found node)
        | AnyOfList
          , (#any_of_list as node) ->
            raise (Found node)
        | FullDeclList
          , (#full_decl_list as node) ->
            raise (Found node)
        | DeclBlock
          , (#decl_block as node) ->
            raise (Found node)
        | GenericParamDeclList
          , (#generic_param_decl_list as node) ->
            raise (Found node)
        | FunParamDeclList
          , (#fun_param_decl_list as node) ->
            raise (Found node)
        | GrammarExprList
          , (#grammar_expr_list as node) ->
            raise (Found node)
        | GrammarExprListList
          , (#grammar_expr_list_list as node) ->
            raise (Found node)
        | ImportList
          , (#import_list as node) ->
            raise (Found node)
        | LambdaParamDeclList
          , (#lambda_param_decl_list as node) ->
            raise (Found node)
        | LktNodeList
          , (#lkt_node_list as node) ->
            raise (Found node)
        | PatternDetailList
          , (#pattern_detail_list as node) ->
            raise (Found node)
        | PatternList
          , (#pattern_list as node) ->
            raise (Found node)
        | RefIdList
          , (#ref_id_list as node) ->
            raise (Found node)
        | TypeRefList
          , (#type_ref_list as node) ->
            raise (Found node)
        | SyntheticTypeRefList
          , (#synthetic_type_ref_list as node) ->
            raise (Found node)
        | NullCondQualifier
          , (#null_cond_qualifier as node) ->
            raise (Found node)
        | NullCondQualifierAbsent
          , (#null_cond_qualifier_absent as node) ->
            raise (Found node)
        | NullCondQualifierPresent
          , (#null_cond_qualifier_present as node) ->
            raise (Found node)
        | Op
          , (#op as node) ->
            raise (Found node)
        | OpAmp
          , (#op_amp as node) ->
            raise (Found node)
        | OpAnd
          , (#op_and as node) ->
            raise (Found node)
        | OpDiv
          , (#op_div as node) ->
            raise (Found node)
        | OpEq
          , (#op_eq as node) ->
            raise (Found node)
        | OpGt
          , (#op_gt as node) ->
            raise (Found node)
        | OpGte
          , (#op_gte as node) ->
            raise (Found node)
        | OpLogicAnd
          , (#op_logic_and as node) ->
            raise (Found node)
        | OpLogicOr
          , (#op_logic_or as node) ->
            raise (Found node)
        | OpLt
          , (#op_lt as node) ->
            raise (Found node)
        | OpLte
          , (#op_lte as node) ->
            raise (Found node)
        | OpMinus
          , (#op_minus as node) ->
            raise (Found node)
        | OpMult
          , (#op_mult as node) ->
            raise (Found node)
        | OpNe
          , (#op_ne as node) ->
            raise (Found node)
        | OpOr
          , (#op_or as node) ->
            raise (Found node)
        | OpOrInt
          , (#op_or_int as node) ->
            raise (Found node)
        | OpPlus
          , (#op_plus as node) ->
            raise (Found node)
        | Pattern
          , (#pattern as node) ->
            raise (Found node)
        | AnyTypePattern
          , (#any_type_pattern as node) ->
            raise (Found node)
        | BindingPattern
          , (#binding_pattern as node) ->
            raise (Found node)
        | BoolPattern
          , (#bool_pattern as node) ->
            raise (Found node)
        | BoolPatternFalse
          , (#bool_pattern_false as node) ->
            raise (Found node)
        | BoolPatternTrue
          , (#bool_pattern_true as node) ->
            raise (Found node)
        | EllipsisPattern
          , (#ellipsis_pattern as node) ->
            raise (Found node)
        | ExtendedPattern
          , (#extended_pattern as node) ->
            raise (Found node)
        | FilteredPattern
          , (#filtered_pattern as node) ->
            raise (Found node)
        | IntegerPattern
          , (#integer_pattern as node) ->
            raise (Found node)
        | ListPattern
          , (#list_pattern as node) ->
            raise (Found node)
        | NotPattern
          , (#not_pattern as node) ->
            raise (Found node)
        | NullPattern
          , (#null_pattern as node) ->
            raise (Found node)
        | OrPattern
          , (#or_pattern as node) ->
            raise (Found node)
        | ParenPattern
          , (#paren_pattern as node) ->
            raise (Found node)
        | RegexPattern
          , (#regex_pattern as node) ->
            raise (Found node)
        | TuplePattern
          , (#tuple_pattern as node) ->
            raise (Found node)
        | TypePattern
          , (#type_pattern as node) ->
            raise (Found node)
        | PatternDetail
          , (#pattern_detail as node) ->
            raise (Found node)
        | FieldPatternDetail
          , (#field_pattern_detail as node) ->
            raise (Found node)
        | PropertyPatternDetail
          , (#property_pattern_detail as node) ->
            raise (Found node)
        | SelectorPatternDetail
          , (#selector_pattern_detail as node) ->
            raise (Found node)
        | SelectorCall
          , (#selector_call as node) ->
            raise (Found node)
        | TypeRef
          , (#type_ref as node) ->
            raise (Found node)
        | DefaultListTypeRef
          , (#default_list_type_ref as node) ->
            raise (Found node)
        | FunctionTypeRef
          , (#function_type_ref as node) ->
            raise (Found node)
        | GenericTypeRef
          , (#generic_type_ref as node) ->
            raise (Found node)
        | SimpleTypeRef
          , (#simple_type_ref as node) ->
            raise (Found node)
        | VarBind
          , (#var_bind as node) ->
            raise (Found node)
        | _ ->
          ()
      in
      try
        iter aux node;
        raise Not_found
      with (Found node) -> node



  let findall : type a. a node ->  [< lkt_node ] -> a list =
    fun node_type node ->
      let aux : a list -> [< lkt_node ] -> a list =
       fun acc node ->
        match node_type, node with
        | LktNode
          , (#lkt_node as node) ->
            node :: acc
        | Argument
          , (#argument as node) ->
            node :: acc
        | BaseLexerCaseRuleAlt
          , (#base_lexer_case_rule_alt as node) ->
            node :: acc
        | LexerCaseRuleCondAlt
          , (#lexer_case_rule_cond_alt as node) ->
            node :: acc
        | LexerCaseRuleDefaultAlt
          , (#lexer_case_rule_default_alt as node) ->
            node :: acc
        | BaseMatchBranch
          , (#base_match_branch as node) ->
            node :: acc
        | MatchBranch
          , (#match_branch as node) ->
            node :: acc
        | PatternMatchBranch
          , (#pattern_match_branch as node) ->
            node :: acc
        | BlockExprClause
          , (#block_expr_clause as node) ->
            node :: acc
        | BlockStringLine
          , (#block_string_line as node) ->
            node :: acc
        | ClassQualifier
          , (#class_qualifier as node) ->
            node :: acc
        | ClassQualifierAbsent
          , (#class_qualifier_absent as node) ->
            node :: acc
        | ClassQualifierPresent
          , (#class_qualifier_present as node) ->
            node :: acc
        | Decl
          , (#decl as node) ->
            node :: acc
        | BaseGrammarRuleDecl
          , (#base_grammar_rule_decl as node) ->
            node :: acc
        | GrammarRuleDecl
          , (#grammar_rule_decl as node) ->
            node :: acc
        | SyntheticLexerDecl
          , (#synthetic_lexer_decl as node) ->
            node :: acc
        | BaseValDecl
          , (#base_val_decl as node) ->
            node :: acc
        | NodeDecl
          , (#node_decl as node) ->
            node :: acc
        | SelfDecl
          , (#self_decl as node) ->
            node :: acc
        | UserValDecl
          , (#user_val_decl as node) ->
            node :: acc
        | BindingValDecl
          , (#binding_val_decl as node) ->
            node :: acc
        | EnumLitDecl
          , (#enum_lit_decl as node) ->
            node :: acc
        | ExplicitlyTypedDecl
          , (#explicitly_typed_decl as node) ->
            node :: acc
        | ComponentDecl
          , (#component_decl as node) ->
            node :: acc
        | FieldDecl
          , (#field_decl as node) ->
            node :: acc
        | FunParamDecl
          , (#fun_param_decl as node) ->
            node :: acc
        | LambdaParamDecl
          , (#lambda_param_decl as node) ->
            node :: acc
        | DynVarDecl
          , (#dyn_var_decl as node) ->
            node :: acc
        | MatchValDecl
          , (#match_val_decl as node) ->
            node :: acc
        | ValDecl
          , (#val_decl as node) ->
            node :: acc
        | FunDecl
          , (#fun_decl as node) ->
            node :: acc
        | EnvSpecDecl
          , (#env_spec_decl as node) ->
            node :: acc
        | ErrorDecl
          , (#error_decl as node) ->
            node :: acc
        | GenericDecl
          , (#generic_decl as node) ->
            node :: acc
        | GrammarDecl
          , (#grammar_decl as node) ->
            node :: acc
        | LexerDecl
          , (#lexer_decl as node) ->
            node :: acc
        | LexerFamilyDecl
          , (#lexer_family_decl as node) ->
            node :: acc
        | SynthFunDecl
          , (#synth_fun_decl as node) ->
            node :: acc
        | SynthParamDecl
          , (#synth_param_decl as node) ->
            node :: acc
        | TypeDecl
          , (#type_decl as node) ->
            node :: acc
        | AnyTypeDecl
          , (#any_type_decl as node) ->
            node :: acc
        | EnumClassAltDecl
          , (#enum_class_alt_decl as node) ->
            node :: acc
        | FunctionType
          , (#function_type as node) ->
            node :: acc
        | GenericParamTypeDecl
          , (#generic_param_type_decl as node) ->
            node :: acc
        | NamedTypeDecl
          , (#named_type_decl as node) ->
            node :: acc
        | BasicClassDecl
          , (#basic_class_decl as node) ->
            node :: acc
        | ClassDecl
          , (#class_decl as node) ->
            node :: acc
        | EnumClassDecl
          , (#enum_class_decl as node) ->
            node :: acc
        | EnumTypeDecl
          , (#enum_type_decl as node) ->
            node :: acc
        | StructDecl
          , (#struct_decl as node) ->
            node :: acc
        | TraitDecl
          , (#trait_decl as node) ->
            node :: acc
        | DeclAnnotation
          , (#decl_annotation as node) ->
            node :: acc
        | DeclAnnotationArgs
          , (#decl_annotation_args as node) ->
            node :: acc
        | DynEnvWrapper
          , (#dyn_env_wrapper as node) ->
            node :: acc
        | ElsifBranch
          , (#elsif_branch as node) ->
            node :: acc
        | EnumClassCase
          , (#enum_class_case as node) ->
            node :: acc
        | ExcludesNull
          , (#excludes_null as node) ->
            node :: acc
        | ExcludesNullAbsent
          , (#excludes_null_absent as node) ->
            node :: acc
        | ExcludesNullPresent
          , (#excludes_null_present as node) ->
            node :: acc
        | Expr
          , (#expr as node) ->
            node :: acc
        | AnyOf
          , (#any_of as node) ->
            node :: acc
        | ArrayLiteral
          , (#array_literal as node) ->
            node :: acc
        | BaseCallExpr
          , (#base_call_expr as node) ->
            node :: acc
        | CallExpr
          , (#call_expr as node) ->
            node :: acc
        | LogicCallExpr
          , (#logic_call_expr as node) ->
            node :: acc
        | LogicPredicate
          , (#logic_predicate as node) ->
            node :: acc
        | LogicPropagateCall
          , (#logic_propagate_call as node) ->
            node :: acc
        | BinOp
          , (#bin_op as node) ->
            node :: acc
        | BlockExpr
          , (#block_expr as node) ->
            node :: acc
        | CastExpr
          , (#cast_expr as node) ->
            node :: acc
        | DotExpr
          , (#dot_expr as node) ->
            node :: acc
        | ErrorOnNull
          , (#error_on_null as node) ->
            node :: acc
        | GenericInstantiation
          , (#generic_instantiation as node) ->
            node :: acc
        | GrammarExpr
          , (#grammar_expr as node) ->
            node :: acc
        | GrammarCut
          , (#grammar_cut as node) ->
            node :: acc
        | GrammarDiscard
          , (#grammar_discard as node) ->
            node :: acc
        | GrammarDontSkip
          , (#grammar_dont_skip as node) ->
            node :: acc
        | GrammarList
          , (#grammar_list as node) ->
            node :: acc
        | GrammarNull
          , (#grammar_null as node) ->
            node :: acc
        | GrammarOpt
          , (#grammar_opt as node) ->
            node :: acc
        | GrammarOptError
          , (#grammar_opt_error as node) ->
            node :: acc
        | GrammarOptErrorGroup
          , (#grammar_opt_error_group as node) ->
            node :: acc
        | GrammarOptGroup
          , (#grammar_opt_group as node) ->
            node :: acc
        | GrammarOrExpr
          , (#grammar_or_expr as node) ->
            node :: acc
        | GrammarPick
          , (#grammar_pick as node) ->
            node :: acc
        | GrammarImplicitPick
          , (#grammar_implicit_pick as node) ->
            node :: acc
        | GrammarPredicate
          , (#grammar_predicate as node) ->
            node :: acc
        | GrammarRuleRef
          , (#grammar_rule_ref as node) ->
            node :: acc
        | GrammarSkip
          , (#grammar_skip as node) ->
            node :: acc
        | GrammarStopCut
          , (#grammar_stop_cut as node) ->
            node :: acc
        | ParseNodeExpr
          , (#parse_node_expr as node) ->
            node :: acc
        | TokenLit
          , (#token_lit as node) ->
            node :: acc
        | TokenNoCaseLit
          , (#token_no_case_lit as node) ->
            node :: acc
        | TokenPatternConcat
          , (#token_pattern_concat as node) ->
            node :: acc
        | TokenPatternLit
          , (#token_pattern_lit as node) ->
            node :: acc
        | TokenRef
          , (#token_ref as node) ->
            node :: acc
        | Id
          , (#id as node) ->
            node :: acc
        | DefId
          , (#def_id as node) ->
            node :: acc
        | ModuleRefId
          , (#module_ref_id as node) ->
            node :: acc
        | RefId
          , (#ref_id as node) ->
            node :: acc
        | IfExpr
          , (#if_expr as node) ->
            node :: acc
        | Isa
          , (#isa as node) ->
            node :: acc
        | KeepExpr
          , (#keep_expr as node) ->
            node :: acc
        | LambdaExpr
          , (#lambda_expr as node) ->
            node :: acc
        | Lit
          , (#lit as node) ->
            node :: acc
        | BigNumLit
          , (#big_num_lit as node) ->
            node :: acc
        | CharLit
          , (#char_lit as node) ->
            node :: acc
        | NullLit
          , (#null_lit as node) ->
            node :: acc
        | NumLit
          , (#num_lit as node) ->
            node :: acc
        | StringLit
          , (#string_lit as node) ->
            node :: acc
        | BlockStringLit
          , (#block_string_lit as node) ->
            node :: acc
        | SingleLineStringLit
          , (#single_line_string_lit as node) ->
            node :: acc
        | PatternSingleLineStringLit
          , (#pattern_single_line_string_lit as node) ->
            node :: acc
        | LogicAssign
          , (#logic_assign as node) ->
            node :: acc
        | LogicExpr
          , (#logic_expr as node) ->
            node :: acc
        | LogicPropagate
          , (#logic_propagate as node) ->
            node :: acc
        | LogicUnify
          , (#logic_unify as node) ->
            node :: acc
        | MatchExpr
          , (#match_expr as node) ->
            node :: acc
        | NotExpr
          , (#not_expr as node) ->
            node :: acc
        | ParenExpr
          , (#paren_expr as node) ->
            node :: acc
        | RaiseExpr
          , (#raise_expr as node) ->
            node :: acc
        | SubscriptExpr
          , (#subscript_expr as node) ->
            node :: acc
        | TryExpr
          , (#try_expr as node) ->
            node :: acc
        | UnOp
          , (#un_op as node) ->
            node :: acc
        | FullDecl
          , (#full_decl as node) ->
            node :: acc
        | GrammarListSep
          , (#grammar_list_sep as node) ->
            node :: acc
        | Import
          , (#import as node) ->
            node :: acc
        | LangkitRoot
          , (#langkit_root as node) ->
            node :: acc
        | LexerCaseRule
          , (#lexer_case_rule as node) ->
            node :: acc
        | LexerCaseRuleSend
          , (#lexer_case_rule_send as node) ->
            node :: acc
        | ListKind
          , (#list_kind as node) ->
            node :: acc
        | ListKindOne
          , (#list_kind_one as node) ->
            node :: acc
        | ListKindZero
          , (#list_kind_zero as node) ->
            node :: acc
        | LktNodeBaseList
          , (#lkt_node_base_list as node) ->
            node :: acc
        | ArgumentList
          , (#argument_list as node) ->
            node :: acc
        | BaseLexerCaseRuleAltList
          , (#base_lexer_case_rule_alt_list as node) ->
            node :: acc
        | BaseMatchBranchList
          , (#base_match_branch_list as node) ->
            node :: acc
        | BlockStringLineList
          , (#block_string_line_list as node) ->
            node :: acc
        | CallExprList
          , (#call_expr_list as node) ->
            node :: acc
        | DeclAnnotationList
          , (#decl_annotation_list as node) ->
            node :: acc
        | ElsifBranchList
          , (#elsif_branch_list as node) ->
            node :: acc
        | EnumClassAltDeclList
          , (#enum_class_alt_decl_list as node) ->
            node :: acc
        | EnumClassCaseList
          , (#enum_class_case_list as node) ->
            node :: acc
        | EnumLitDeclList
          , (#enum_lit_decl_list as node) ->
            node :: acc
        | ExprList
          , (#expr_list as node) ->
            node :: acc
        | AnyOfList
          , (#any_of_list as node) ->
            node :: acc
        | FullDeclList
          , (#full_decl_list as node) ->
            node :: acc
        | DeclBlock
          , (#decl_block as node) ->
            node :: acc
        | GenericParamDeclList
          , (#generic_param_decl_list as node) ->
            node :: acc
        | FunParamDeclList
          , (#fun_param_decl_list as node) ->
            node :: acc
        | GrammarExprList
          , (#grammar_expr_list as node) ->
            node :: acc
        | GrammarExprListList
          , (#grammar_expr_list_list as node) ->
            node :: acc
        | ImportList
          , (#import_list as node) ->
            node :: acc
        | LambdaParamDeclList
          , (#lambda_param_decl_list as node) ->
            node :: acc
        | LktNodeList
          , (#lkt_node_list as node) ->
            node :: acc
        | PatternDetailList
          , (#pattern_detail_list as node) ->
            node :: acc
        | PatternList
          , (#pattern_list as node) ->
            node :: acc
        | RefIdList
          , (#ref_id_list as node) ->
            node :: acc
        | TypeRefList
          , (#type_ref_list as node) ->
            node :: acc
        | SyntheticTypeRefList
          , (#synthetic_type_ref_list as node) ->
            node :: acc
        | NullCondQualifier
          , (#null_cond_qualifier as node) ->
            node :: acc
        | NullCondQualifierAbsent
          , (#null_cond_qualifier_absent as node) ->
            node :: acc
        | NullCondQualifierPresent
          , (#null_cond_qualifier_present as node) ->
            node :: acc
        | Op
          , (#op as node) ->
            node :: acc
        | OpAmp
          , (#op_amp as node) ->
            node :: acc
        | OpAnd
          , (#op_and as node) ->
            node :: acc
        | OpDiv
          , (#op_div as node) ->
            node :: acc
        | OpEq
          , (#op_eq as node) ->
            node :: acc
        | OpGt
          , (#op_gt as node) ->
            node :: acc
        | OpGte
          , (#op_gte as node) ->
            node :: acc
        | OpLogicAnd
          , (#op_logic_and as node) ->
            node :: acc
        | OpLogicOr
          , (#op_logic_or as node) ->
            node :: acc
        | OpLt
          , (#op_lt as node) ->
            node :: acc
        | OpLte
          , (#op_lte as node) ->
            node :: acc
        | OpMinus
          , (#op_minus as node) ->
            node :: acc
        | OpMult
          , (#op_mult as node) ->
            node :: acc
        | OpNe
          , (#op_ne as node) ->
            node :: acc
        | OpOr
          , (#op_or as node) ->
            node :: acc
        | OpOrInt
          , (#op_or_int as node) ->
            node :: acc
        | OpPlus
          , (#op_plus as node) ->
            node :: acc
        | Pattern
          , (#pattern as node) ->
            node :: acc
        | AnyTypePattern
          , (#any_type_pattern as node) ->
            node :: acc
        | BindingPattern
          , (#binding_pattern as node) ->
            node :: acc
        | BoolPattern
          , (#bool_pattern as node) ->
            node :: acc
        | BoolPatternFalse
          , (#bool_pattern_false as node) ->
            node :: acc
        | BoolPatternTrue
          , (#bool_pattern_true as node) ->
            node :: acc
        | EllipsisPattern
          , (#ellipsis_pattern as node) ->
            node :: acc
        | ExtendedPattern
          , (#extended_pattern as node) ->
            node :: acc
        | FilteredPattern
          , (#filtered_pattern as node) ->
            node :: acc
        | IntegerPattern
          , (#integer_pattern as node) ->
            node :: acc
        | ListPattern
          , (#list_pattern as node) ->
            node :: acc
        | NotPattern
          , (#not_pattern as node) ->
            node :: acc
        | NullPattern
          , (#null_pattern as node) ->
            node :: acc
        | OrPattern
          , (#or_pattern as node) ->
            node :: acc
        | ParenPattern
          , (#paren_pattern as node) ->
            node :: acc
        | RegexPattern
          , (#regex_pattern as node) ->
            node :: acc
        | TuplePattern
          , (#tuple_pattern as node) ->
            node :: acc
        | TypePattern
          , (#type_pattern as node) ->
            node :: acc
        | PatternDetail
          , (#pattern_detail as node) ->
            node :: acc
        | FieldPatternDetail
          , (#field_pattern_detail as node) ->
            node :: acc
        | PropertyPatternDetail
          , (#property_pattern_detail as node) ->
            node :: acc
        | SelectorPatternDetail
          , (#selector_pattern_detail as node) ->
            node :: acc
        | SelectorCall
          , (#selector_call as node) ->
            node :: acc
        | TypeRef
          , (#type_ref as node) ->
            node :: acc
        | DefaultListTypeRef
          , (#default_list_type_ref as node) ->
            node :: acc
        | FunctionTypeRef
          , (#function_type_ref as node) ->
            node :: acc
        | GenericTypeRef
          , (#generic_type_ref as node) ->
            node :: acc
        | SimpleTypeRef
          , (#simple_type_ref as node) ->
            node :: acc
        | VarBind
          , (#var_bind as node) ->
            node :: acc
        | _ ->
          acc
      in
      List.rev (fold aux [] node)

  let fields_with_names node =
    let aux i x =
      (Format.sprintf "item_%d" i), x
    in
    match (node :> lkt_node) with
    | `VarBind value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `SimpleTypeRef value ->
        [
        (try
           ("type_name"
           , Some (Lazy.force value.f_type_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("type_name", None) );
        ]
    | `GenericTypeRef value ->
        [
        (try
           ("type_name"
           , Some (Lazy.force value.f_type_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("type_name", None) );
        (try
           ("args"
           , Some (Lazy.force value.f_args
                    :> lkt_node))
        with SyntaxError _ ->
          ("args", None) );
        ]
    | `FunctionTypeRef value ->
        [
        (try
           ("param_types"
           , Some (Lazy.force value.f_param_types
                    :> lkt_node))
        with SyntaxError _ ->
          ("param_types", None) );
        (try
           ("return_type"
           , Some (Lazy.force value.f_return_type
                    :> lkt_node))
        with SyntaxError _ ->
          ("return_type", None) );
        ]
    | `DefaultListTypeRef value ->
        [
        ]
    | `SelectorCall value ->
        [
        (try
           ("quantifier"
           , Some (Lazy.force value.f_quantifier
                    :> lkt_node))
        with SyntaxError _ ->
          ("quantifier", None) );
        ("binding"
        , (Lazy.force value.f_binding
           :> lkt_node option));
        (try
           ("selector_call"
           , Some (Lazy.force value.f_selector_call
                    :> lkt_node))
        with SyntaxError _ ->
          ("selector_call", None) );
        ]
    | `SelectorPatternDetail value ->
        [
        (try
           ("call"
           , Some (Lazy.force value.f_call
                    :> lkt_node))
        with SyntaxError _ ->
          ("call", None) );
        (try
           ("sub_pattern"
           , Some (Lazy.force value.f_sub_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_pattern", None) );
        ]
    | `PropertyPatternDetail value ->
        [
        (try
           ("call"
           , Some (Lazy.force value.f_call
                    :> lkt_node))
        with SyntaxError _ ->
          ("call", None) );
        (try
           ("expected_value"
           , Some (Lazy.force value.f_expected_value
                    :> lkt_node))
        with SyntaxError _ ->
          ("expected_value", None) );
        ]
    | `FieldPatternDetail value ->
        [
        (try
           ("id"
           , Some (Lazy.force value.f_id
                    :> lkt_node))
        with SyntaxError _ ->
          ("id", None) );
        (try
           ("expected_value"
           , Some (Lazy.force value.f_expected_value
                    :> lkt_node))
        with SyntaxError _ ->
          ("expected_value", None) );
        ]
    | `TypePattern value ->
        [
        (try
           ("type_name"
           , Some (Lazy.force value.f_type_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("type_name", None) );
        ]
    | `TuplePattern value ->
        [
        (try
           ("sub_patterns"
           , Some (Lazy.force value.f_sub_patterns
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_patterns", None) );
        ]
    | `RegexPattern value ->
        [
        ]
    | `ParenPattern value ->
        [
        (try
           ("sub_pattern"
           , Some (Lazy.force value.f_sub_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_pattern", None) );
        ]
    | `OrPattern value ->
        [
        (try
           ("left_sub_pattern"
           , Some (Lazy.force value.f_left_sub_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("left_sub_pattern", None) );
        (try
           ("right_sub_pattern"
           , Some (Lazy.force value.f_right_sub_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("right_sub_pattern", None) );
        ]
    | `NullPattern value ->
        [
        ]
    | `NotPattern value ->
        [
        (try
           ("sub_pattern"
           , Some (Lazy.force value.f_sub_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_pattern", None) );
        ]
    | `ListPattern value ->
        [
        (try
           ("sub_patterns"
           , Some (Lazy.force value.f_sub_patterns
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_patterns", None) );
        ]
    | `IntegerPattern value ->
        [
        ]
    | `FilteredPattern value ->
        [
        (try
           ("sub_pattern"
           , Some (Lazy.force value.f_sub_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_pattern", None) );
        (try
           ("predicate"
           , Some (Lazy.force value.f_predicate
                    :> lkt_node))
        with SyntaxError _ ->
          ("predicate", None) );
        ]
    | `ExtendedPattern value ->
        [
        (try
           ("sub_pattern"
           , Some (Lazy.force value.f_sub_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_pattern", None) );
        (try
           ("details"
           , Some (Lazy.force value.f_details
                    :> lkt_node))
        with SyntaxError _ ->
          ("details", None) );
        ]
    | `EllipsisPattern value ->
        [
        ("binding"
        , (Lazy.force value.f_binding
           :> lkt_node option));
        ]
    | `BoolPatternTrue value ->
        [
        ]
    | `BoolPatternFalse value ->
        [
        ]
    | `BindingPattern value ->
        [
        (try
           ("decl"
           , Some (Lazy.force value.f_decl
                    :> lkt_node))
        with SyntaxError _ ->
          ("decl", None) );
        ("sub_pattern"
        , (Lazy.force value.f_sub_pattern
           :> lkt_node option));
        ]
    | `AnyTypePattern value ->
        [
        ]
    | `OpPlus value ->
        [
        ]
    | `OpOrInt value ->
        [
        ]
    | `OpOr value ->
        [
        ]
    | `OpNe value ->
        [
        ]
    | `OpMult value ->
        [
        ]
    | `OpMinus value ->
        [
        ]
    | `OpLte value ->
        [
        ]
    | `OpLt value ->
        [
        ]
    | `OpLogicOr value ->
        [
        ]
    | `OpLogicAnd value ->
        [
        ]
    | `OpGte value ->
        [
        ]
    | `OpGt value ->
        [
        ]
    | `OpEq value ->
        [
        ]
    | `OpDiv value ->
        [
        ]
    | `OpAnd value ->
        [
        ]
    | `OpAmp value ->
        [
        ]
    | `NullCondQualifierPresent value ->
        [
        ]
    | `NullCondQualifierAbsent value ->
        [
        ]
    | `SyntheticTypeRefList value ->
        List.mapi aux (children_opt node)
    | `TypeRefList value ->
        List.mapi aux (children_opt node)
    | `RefIdList value ->
        List.mapi aux (children_opt node)
    | `PatternList value ->
        List.mapi aux (children_opt node)
    | `PatternDetailList value ->
        List.mapi aux (children_opt node)
    | `LktNodeList value ->
        List.mapi aux (children_opt node)
    | `LambdaParamDeclList value ->
        List.mapi aux (children_opt node)
    | `ImportList value ->
        List.mapi aux (children_opt node)
    | `GrammarExprListList value ->
        List.mapi aux (children_opt node)
    | `GrammarExprList value ->
        List.mapi aux (children_opt node)
    | `FunParamDeclList value ->
        List.mapi aux (children_opt node)
    | `GenericParamDeclList value ->
        List.mapi aux (children_opt node)
    | `DeclBlock value ->
        List.mapi aux (children_opt node)
    | `FullDeclList value ->
        List.mapi aux (children_opt node)
    | `AnyOfList value ->
        List.mapi aux (children_opt node)
    | `ExprList value ->
        List.mapi aux (children_opt node)
    | `EnumLitDeclList value ->
        List.mapi aux (children_opt node)
    | `EnumClassCaseList value ->
        List.mapi aux (children_opt node)
    | `EnumClassAltDeclList value ->
        List.mapi aux (children_opt node)
    | `ElsifBranchList value ->
        List.mapi aux (children_opt node)
    | `DeclAnnotationList value ->
        List.mapi aux (children_opt node)
    | `CallExprList value ->
        List.mapi aux (children_opt node)
    | `BlockStringLineList value ->
        List.mapi aux (children_opt node)
    | `BaseMatchBranchList value ->
        List.mapi aux (children_opt node)
    | `BaseLexerCaseRuleAltList value ->
        List.mapi aux (children_opt node)
    | `ArgumentList value ->
        List.mapi aux (children_opt node)
    | `ListKindZero value ->
        [
        ]
    | `ListKindOne value ->
        [
        ]
    | `LexerCaseRuleSend value ->
        [
        (try
           ("sent"
           , Some (Lazy.force value.f_sent
                    :> lkt_node))
        with SyntaxError _ ->
          ("sent", None) );
        (try
           ("match_size"
           , Some (Lazy.force value.f_match_size
                    :> lkt_node))
        with SyntaxError _ ->
          ("match_size", None) );
        ]
    | `LexerCaseRule value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        (try
           ("alts"
           , Some (Lazy.force value.f_alts
                    :> lkt_node))
        with SyntaxError _ ->
          ("alts", None) );
        ]
    | `LangkitRoot value ->
        [
        (try
           ("imports"
           , Some (Lazy.force value.f_imports
                    :> lkt_node))
        with SyntaxError _ ->
          ("imports", None) );
        (try
           ("decls"
           , Some (Lazy.force value.f_decls
                    :> lkt_node))
        with SyntaxError _ ->
          ("decls", None) );
        ]
    | `Import value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        ]
    | `GrammarListSep value ->
        [
        (try
           ("token"
           , Some (Lazy.force value.f_token
                    :> lkt_node))
        with SyntaxError _ ->
          ("token", None) );
        ("extra"
        , (Lazy.force value.f_extra
           :> lkt_node option));
        ]
    | `FullDecl value ->
        [
        ("doc"
        , (Lazy.force value.f_doc
           :> lkt_node option));
        (try
           ("decl_annotations"
           , Some (Lazy.force value.f_decl_annotations
                    :> lkt_node))
        with SyntaxError _ ->
          ("decl_annotations", None) );
        (try
           ("decl"
           , Some (Lazy.force value.f_decl
                    :> lkt_node))
        with SyntaxError _ ->
          ("decl", None) );
        ]
    | `UnOp value ->
        [
        (try
           ("op"
           , Some (Lazy.force value.f_op
                    :> lkt_node))
        with SyntaxError _ ->
          ("op", None) );
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `TryExpr value ->
        [
        (try
           ("try_expr"
           , Some (Lazy.force value.f_try_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("try_expr", None) );
        ("or_expr"
        , (Lazy.force value.f_or_expr
           :> lkt_node option));
        ]
    | `SubscriptExpr value ->
        [
        (try
           ("prefix"
           , Some (Lazy.force value.f_prefix
                    :> lkt_node))
        with SyntaxError _ ->
          ("prefix", None) );
        (try
           ("null_cond"
           , Some (Lazy.force value.f_null_cond
                    :> lkt_node))
        with SyntaxError _ ->
          ("null_cond", None) );
        (try
           ("index"
           , Some (Lazy.force value.f_index
                    :> lkt_node))
        with SyntaxError _ ->
          ("index", None) );
        ]
    | `RaiseExpr value ->
        [
        ("dest_type"
        , (Lazy.force value.f_dest_type
           :> lkt_node option));
        (try
           ("except_expr"
           , Some (Lazy.force value.f_except_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("except_expr", None) );
        ]
    | `ParenExpr value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `NotExpr value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `MatchExpr value ->
        [
        (try
           ("match_expr"
           , Some (Lazy.force value.f_match_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("match_expr", None) );
        (try
           ("branches"
           , Some (Lazy.force value.f_branches
                    :> lkt_node))
        with SyntaxError _ ->
          ("branches", None) );
        ]
    | `LogicUnify value ->
        [
        (try
           ("lhs"
           , Some (Lazy.force value.f_lhs
                    :> lkt_node))
        with SyntaxError _ ->
          ("lhs", None) );
        (try
           ("rhs"
           , Some (Lazy.force value.f_rhs
                    :> lkt_node))
        with SyntaxError _ ->
          ("rhs", None) );
        ]
    | `LogicPropagate value ->
        [
        (try
           ("dest_var"
           , Some (Lazy.force value.f_dest_var
                    :> lkt_node))
        with SyntaxError _ ->
          ("dest_var", None) );
        (try
           ("call"
           , Some (Lazy.force value.f_call
                    :> lkt_node))
        with SyntaxError _ ->
          ("call", None) );
        ]
    | `LogicExpr value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `LogicAssign value ->
        [
        (try
           ("dest_var"
           , Some (Lazy.force value.f_dest_var
                    :> lkt_node))
        with SyntaxError _ ->
          ("dest_var", None) );
        (try
           ("value"
           , Some (Lazy.force value.f_value
                    :> lkt_node))
        with SyntaxError _ ->
          ("value", None) );
        ]
    | `PatternSingleLineStringLit value ->
        [
        ]
    | `SingleLineStringLit value ->
        [
        ]
    | `BlockStringLit value ->
        [
        (try
           ("lines"
           , Some (Lazy.force value.f_lines
                    :> lkt_node))
        with SyntaxError _ ->
          ("lines", None) );
        ]
    | `NumLit value ->
        [
        ]
    | `NullLit value ->
        [
        ("dest_type"
        , (Lazy.force value.f_dest_type
           :> lkt_node option));
        ]
    | `CharLit value ->
        [
        ]
    | `BigNumLit value ->
        [
        ]
    | `LambdaExpr value ->
        [
        (try
           ("params"
           , Some (Lazy.force value.f_params
                    :> lkt_node))
        with SyntaxError _ ->
          ("params", None) );
        ("return_type"
        , (Lazy.force value.f_return_type
           :> lkt_node option));
        (try
           ("body"
           , Some (Lazy.force value.f_body
                    :> lkt_node))
        with SyntaxError _ ->
          ("body", None) );
        ]
    | `KeepExpr value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        (try
           ("null_cond"
           , Some (Lazy.force value.f_null_cond
                    :> lkt_node))
        with SyntaxError _ ->
          ("null_cond", None) );
        (try
           ("keep_type"
           , Some (Lazy.force value.f_keep_type
                    :> lkt_node))
        with SyntaxError _ ->
          ("keep_type", None) );
        ]
    | `Isa value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        (try
           ("pattern"
           , Some (Lazy.force value.f_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("pattern", None) );
        ]
    | `IfExpr value ->
        [
        (try
           ("cond_expr"
           , Some (Lazy.force value.f_cond_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("cond_expr", None) );
        (try
           ("then_expr"
           , Some (Lazy.force value.f_then_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("then_expr", None) );
        (try
           ("alternatives"
           , Some (Lazy.force value.f_alternatives
                    :> lkt_node))
        with SyntaxError _ ->
          ("alternatives", None) );
        (try
           ("else_expr"
           , Some (Lazy.force value.f_else_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("else_expr", None) );
        ]
    | `RefId value ->
        [
        ]
    | `ModuleRefId value ->
        [
        ]
    | `DefId value ->
        [
        ]
    | `Id value ->
        [
        ]
    | `TokenRef value ->
        [
        (try
           ("token_name"
           , Some (Lazy.force value.f_token_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("token_name", None) );
        ("expr"
        , (Lazy.force value.f_expr
           :> lkt_node option));
        ]
    | `TokenPatternLit value ->
        [
        ]
    | `TokenPatternConcat value ->
        [
        (try
           ("left"
           , Some (Lazy.force value.f_left
                    :> lkt_node))
        with SyntaxError _ ->
          ("left", None) );
        (try
           ("right"
           , Some (Lazy.force value.f_right
                    :> lkt_node))
        with SyntaxError _ ->
          ("right", None) );
        ]
    | `TokenNoCaseLit value ->
        [
        (try
           ("lit"
           , Some (Lazy.force value.f_lit
                    :> lkt_node))
        with SyntaxError _ ->
          ("lit", None) );
        ]
    | `TokenLit value ->
        [
        ]
    | `ParseNodeExpr value ->
        [
        (try
           ("node_name"
           , Some (Lazy.force value.f_node_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("node_name", None) );
        (try
           ("sub_exprs"
           , Some (Lazy.force value.f_sub_exprs
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_exprs", None) );
        ]
    | `GrammarStopCut value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `GrammarSkip value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        ]
    | `GrammarRuleRef value ->
        [
        (try
           ("node_name"
           , Some (Lazy.force value.f_node_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("node_name", None) );
        ]
    | `GrammarPredicate value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        (try
           ("prop_ref"
           , Some (Lazy.force value.f_prop_ref
                    :> lkt_node))
        with SyntaxError _ ->
          ("prop_ref", None) );
        ]
    | `GrammarImplicitPick value ->
        [
        (try
           ("exprs"
           , Some (Lazy.force value.f_exprs
                    :> lkt_node))
        with SyntaxError _ ->
          ("exprs", None) );
        ]
    | `GrammarPick value ->
        [
        (try
           ("exprs"
           , Some (Lazy.force value.f_exprs
                    :> lkt_node))
        with SyntaxError _ ->
          ("exprs", None) );
        ]
    | `GrammarOrExpr value ->
        [
        (try
           ("sub_exprs"
           , Some (Lazy.force value.f_sub_exprs
                    :> lkt_node))
        with SyntaxError _ ->
          ("sub_exprs", None) );
        ]
    | `GrammarOptGroup value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `GrammarOptErrorGroup value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `GrammarOptError value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `GrammarOpt value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `GrammarNull value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        ]
    | `GrammarList value ->
        [
        (try
           ("list_type"
           , Some (Lazy.force value.f_list_type
                    :> lkt_node))
        with SyntaxError _ ->
          ("list_type", None) );
        (try
           ("kind"
           , Some (Lazy.force value.f_kind
                    :> lkt_node))
        with SyntaxError _ ->
          ("kind", None) );
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ("sep"
        , (Lazy.force value.f_sep
           :> lkt_node option));
        ]
    | `GrammarDontSkip value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        (try
           ("dont_skip"
           , Some (Lazy.force value.f_dont_skip
                    :> lkt_node))
        with SyntaxError _ ->
          ("dont_skip", None) );
        ]
    | `GrammarDiscard value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `GrammarCut value ->
        [
        ]
    | `GenericInstantiation value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        (try
           ("args"
           , Some (Lazy.force value.f_args
                    :> lkt_node))
        with SyntaxError _ ->
          ("args", None) );
        ]
    | `ErrorOnNull value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `DotExpr value ->
        [
        (try
           ("prefix"
           , Some (Lazy.force value.f_prefix
                    :> lkt_node))
        with SyntaxError _ ->
          ("prefix", None) );
        (try
           ("null_cond"
           , Some (Lazy.force value.f_null_cond
                    :> lkt_node))
        with SyntaxError _ ->
          ("null_cond", None) );
        (try
           ("suffix"
           , Some (Lazy.force value.f_suffix
                    :> lkt_node))
        with SyntaxError _ ->
          ("suffix", None) );
        ]
    | `CastExpr value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        (try
           ("null_cond"
           , Some (Lazy.force value.f_null_cond
                    :> lkt_node))
        with SyntaxError _ ->
          ("null_cond", None) );
        (try
           ("excludes_null"
           , Some (Lazy.force value.f_excludes_null
                    :> lkt_node))
        with SyntaxError _ ->
          ("excludes_null", None) );
        (try
           ("dest_type"
           , Some (Lazy.force value.f_dest_type
                    :> lkt_node))
        with SyntaxError _ ->
          ("dest_type", None) );
        ]
    | `BlockExpr value ->
        [
        (try
           ("clauses"
           , Some (Lazy.force value.f_clauses
                    :> lkt_node))
        with SyntaxError _ ->
          ("clauses", None) );
        ]
    | `BinOp value ->
        [
        (try
           ("left"
           , Some (Lazy.force value.f_left
                    :> lkt_node))
        with SyntaxError _ ->
          ("left", None) );
        (try
           ("op"
           , Some (Lazy.force value.f_op
                    :> lkt_node))
        with SyntaxError _ ->
          ("op", None) );
        (try
           ("right"
           , Some (Lazy.force value.f_right
                    :> lkt_node))
        with SyntaxError _ ->
          ("right", None) );
        ]
    | `LogicPropagateCall value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        (try
           ("args"
           , Some (Lazy.force value.f_args
                    :> lkt_node))
        with SyntaxError _ ->
          ("args", None) );
        ]
    | `LogicPredicate value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        (try
           ("args"
           , Some (Lazy.force value.f_args
                    :> lkt_node))
        with SyntaxError _ ->
          ("args", None) );
        ]
    | `CallExpr value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        (try
           ("args"
           , Some (Lazy.force value.f_args
                    :> lkt_node))
        with SyntaxError _ ->
          ("args", None) );
        ]
    | `ArrayLiteral value ->
        [
        (try
           ("exprs"
           , Some (Lazy.force value.f_exprs
                    :> lkt_node))
        with SyntaxError _ ->
          ("exprs", None) );
        ("element_type"
        , (Lazy.force value.f_element_type
           :> lkt_node option));
        ]
    | `AnyOf value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        (try
           ("values"
           , Some (Lazy.force value.f_values
                    :> lkt_node))
        with SyntaxError _ ->
          ("values", None) );
        ]
    | `ExcludesNullPresent value ->
        [
        ]
    | `ExcludesNullAbsent value ->
        [
        ]
    | `EnumClassCase value ->
        [
        (try
           ("decls"
           , Some (Lazy.force value.f_decls
                    :> lkt_node))
        with SyntaxError _ ->
          ("decls", None) );
        ]
    | `ElsifBranch value ->
        [
        (try
           ("cond_expr"
           , Some (Lazy.force value.f_cond_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("cond_expr", None) );
        (try
           ("then_expr"
           , Some (Lazy.force value.f_then_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("then_expr", None) );
        ]
    | `DynEnvWrapper value ->
        [
        ]
    | `DeclAnnotationArgs value ->
        [
        (try
           ("args"
           , Some (Lazy.force value.f_args
                    :> lkt_node))
        with SyntaxError _ ->
          ("args", None) );
        ]
    | `DeclAnnotation value ->
        [
        (try
           ("name"
           , Some (Lazy.force value.f_name
                    :> lkt_node))
        with SyntaxError _ ->
          ("name", None) );
        ("args"
        , (Lazy.force value.f_args
           :> lkt_node option));
        ]
    | `TraitDecl value ->
        [
        ]
    | `StructDecl value ->
        [
        ]
    | `EnumTypeDecl value ->
        [
        (try
           ("literals"
           , Some (Lazy.force value.f_literals
                    :> lkt_node))
        with SyntaxError _ ->
          ("literals", None) );
        ]
    | `EnumClassDecl value ->
        [
        (try
           ("branches"
           , Some (Lazy.force value.f_branches
                    :> lkt_node))
        with SyntaxError _ ->
          ("branches", None) );
        ]
    | `ClassDecl value ->
        [
        ]
    | `GenericParamTypeDecl value ->
        [
        (try
           ("has_class"
           , Some (Lazy.force value.f_has_class
                    :> lkt_node))
        with SyntaxError _ ->
          ("has_class", None) );
        ]
    | `FunctionType value ->
        [
        ]
    | `EnumClassAltDecl value ->
        [
        ]
    | `AnyTypeDecl value ->
        [
        ]
    | `SynthParamDecl value ->
        [
        ]
    | `SynthFunDecl value ->
        [
        ]
    | `LexerFamilyDecl value ->
        [
        (try
           ("rules"
           , Some (Lazy.force value.f_rules
                    :> lkt_node))
        with SyntaxError _ ->
          ("rules", None) );
        ]
    | `LexerDecl value ->
        [
        (try
           ("rules"
           , Some (Lazy.force value.f_rules
                    :> lkt_node))
        with SyntaxError _ ->
          ("rules", None) );
        ]
    | `GrammarDecl value ->
        [
        (try
           ("rules"
           , Some (Lazy.force value.f_rules
                    :> lkt_node))
        with SyntaxError _ ->
          ("rules", None) );
        ]
    | `GenericDecl value ->
        [
        (try
           ("generic_param_decls"
           , Some (Lazy.force value.f_generic_param_decls
                    :> lkt_node))
        with SyntaxError _ ->
          ("generic_param_decls", None) );
        (try
           ("decl"
           , Some (Lazy.force value.f_decl
                    :> lkt_node))
        with SyntaxError _ ->
          ("decl", None) );
        ]
    | `ErrorDecl value ->
        [
        ]
    | `EnvSpecDecl value ->
        [
        (try
           ("actions"
           , Some (Lazy.force value.f_actions
                    :> lkt_node))
        with SyntaxError _ ->
          ("actions", None) );
        ]
    | `FunDecl value ->
        [
        (try
           ("params"
           , Some (Lazy.force value.f_params
                    :> lkt_node))
        with SyntaxError _ ->
          ("params", None) );
        (try
           ("return_type"
           , Some (Lazy.force value.f_return_type
                    :> lkt_node))
        with SyntaxError _ ->
          ("return_type", None) );
        ("trait_ref"
        , (Lazy.force value.f_trait_ref
           :> lkt_node option));
        ("body"
        , (Lazy.force value.f_body
           :> lkt_node option));
        ]
    | `ValDecl value ->
        [
        (try
           ("expr"
           , Some (Lazy.force value.f_expr
                    :> lkt_node))
        with SyntaxError _ ->
          ("expr", None) );
        ]
    | `MatchValDecl value ->
        [
        ]
    | `DynVarDecl value ->
        [
        ]
    | `LambdaParamDecl value ->
        [
        ]
    | `FunParamDecl value ->
        [
        (try
           ("decl_annotations"
           , Some (Lazy.force value.f_decl_annotations
                    :> lkt_node))
        with SyntaxError _ ->
          ("decl_annotations", None) );
        ]
    | `FieldDecl value ->
        [
        ("trait_ref"
        , (Lazy.force value.f_trait_ref
           :> lkt_node option));
        ]
    | `EnumLitDecl value ->
        [
        ]
    | `BindingValDecl value ->
        [
        ]
    | `SelfDecl value ->
        [
        ]
    | `NodeDecl value ->
        [
        ]
    | `SyntheticLexerDecl value ->
        [
        ]
    | `GrammarRuleDecl value ->
        [
        ]
    | `ClassQualifierPresent value ->
        [
        ]
    | `ClassQualifierAbsent value ->
        [
        ]
    | `BlockStringLine value ->
        [
        ]
    | `BlockExprClause value ->
        [
        (try
           ("clause"
           , Some (Lazy.force value.f_clause
                    :> lkt_node))
        with SyntaxError _ ->
          ("clause", None) );
        ]
    | `PatternMatchBranch value ->
        [
        (try
           ("pattern"
           , Some (Lazy.force value.f_pattern
                    :> lkt_node))
        with SyntaxError _ ->
          ("pattern", None) );
        ]
    | `MatchBranch value ->
        [
        (try
           ("decl"
           , Some (Lazy.force value.f_decl
                    :> lkt_node))
        with SyntaxError _ ->
          ("decl", None) );
        ]
    | `LexerCaseRuleDefaultAlt value ->
        [
        ]
    | `LexerCaseRuleCondAlt value ->
        [
        (try
           ("cond_exprs"
           , Some (Lazy.force value.f_cond_exprs
                    :> lkt_node))
        with SyntaxError _ ->
          ("cond_exprs", None) );
        ]
    | `Argument value ->
        [
        ("name"
        , (Lazy.force value.f_name
           :> lkt_node option));
        (try
           ("value"
           , Some (Lazy.force value.f_value
                    :> lkt_node))
        with SyntaxError _ ->
          ("value", None) );
        ]

  let rec pp_tree fmt node =
    let rec pp_node_field fmt (name, node) =
      match node with
      | Some node ->
          Format.fprintf fmt "@[<v 2>%s:@ %a@]" name pp_node node
      | None ->
          Format.fprintf fmt "@[<v 2>%s: None@]" name
    and pp_node_fields fmt node =
      let name_field_list = fields_with_names node in
      match name_field_list with
      | [] ->
          ()
      | l ->
          Format.fprintf fmt "@ @[<v>%a@]"
            (Format.pp_print_list pp_node_field) l
    and pp_node fmt node =
      let repr = image node in
      let len = String.length repr in
      let erepr = String.sub repr 1 (len - 2) in
      Format.fprintf fmt "@[<v 2>%s%s%a@]"
        erepr
        (if is_token_node node then (": " ^ (text node)) else "")
        pp_node_fields node
    in
    let default = Format.pp_get_formatter_out_functions fmt () in
    let out_indent n =
      let the_end = n in
      let rec make n =
        if n = the_end then ""
        else (if n mod 4 = 2 then "|" else " ") ^ make (n + 1)
      in
      default.out_string (make 0) 0 n
    in
    Format.pp_set_formatter_out_functions fmt {default with out_indent} ;
    Format.fprintf fmt "%a%!" pp_node (node :> lkt_node);
    Format.pp_set_formatter_out_functions fmt default


end

