--
--  Copyright (C) 2014-2022, AdaCore
--  SPDX-License-Identifier: Apache-2.0
--

with Libadalang.Generic_API; use Libadalang.Generic_API;
with Libadalang.Generic_Introspection;
use Libadalang.Generic_Introspection;

package body Libadalang.Rewriting is

   function "+" (Handle : Rewriting_Handle) return G.Rewriting_Handle
   is (G.Rewriting_Handle (Handle));

   function "+" (Handle : Unit_Rewriting_Handle) return G.Unit_Rewriting_Handle
   is (G.Unit_Rewriting_Handle (Handle));

   function "+" (Handle : Node_Rewriting_Handle) return G.Node_Rewriting_Handle
   is (G.Node_Rewriting_Handle (Handle));

   function "+" (Handle : G.Rewriting_Handle) return Rewriting_Handle
   is (Rewriting_Handle (Handle));

   function "+" (Handle : G.Unit_Rewriting_Handle) return Unit_Rewriting_Handle
   is (Unit_Rewriting_Handle (Handle));

   function "+" (Handle : G.Node_Rewriting_Handle) return Node_Rewriting_Handle
   is (Node_Rewriting_Handle (Handle));

   ------------
   -- Handle --
   ------------

   function Handle (Context : Analysis_Context) return Rewriting_Handle is
   begin
      return +G.Handle (To_Generic_Context (Context));
   end Handle;

   -------------
   -- Context --
   -------------

   function Context (Handle : Rewriting_Handle) return Analysis_Context is
   begin
      return From_Generic_Context (G.Context (+Handle));
   end Context;

   ---------------------
   -- Start_Rewriting --
   ---------------------

   function Start_Rewriting
     (Context : Analysis_Context) return Rewriting_Handle is
   begin
      return +G.Start_Rewriting (To_Generic_Context (Context));
   end Start_Rewriting;

   ---------------------
   -- Abort_Rewriting --
   ---------------------

   procedure Abort_Rewriting (Handle : in out Rewriting_Handle) is
      H : G.Rewriting_Handle := +Handle;
   begin
      G.Abort_Rewriting (H);
      Handle := +H;
   end Abort_Rewriting;

   -----------
   -- Apply --
   -----------

   function Apply (Handle : in out Rewriting_Handle) return Apply_Result is
      H : G.Rewriting_Handle := +Handle;
      R : constant G.Apply_Result := G.Apply (H);
   begin
      Handle := +H;
      if R.Success then
         return (Success => True);
      else
         return (Success     => False,
                 Unit        => From_Generic_Unit (R.Unit),
                 Diagnostics => R.Diagnostics);
      end if;
   end Apply;

   ------------------
   -- Unit_Handles --
   ------------------

   function Unit_Handles
     (Handle : Rewriting_Handle) return Unit_Rewriting_Handle_Array
   is
      R : constant G.Unit_Rewriting_Handle_Array := G.Unit_Handles (+Handle);
   begin
      return Result : Unit_Rewriting_Handle_Array (R'Range) do
         for I in R'Range loop
            Result (I) := +R (I);
         end loop;
      end return;
   end Unit_Handles;

   ------------
   -- Handle --
   ------------

   function Handle (Unit : Analysis_Unit) return Unit_Rewriting_Handle is
   begin
      return +G.Handle (To_Generic_Unit (Unit));
   end Handle;

   ----------
   -- Unit --
   ----------

   function Unit (Handle : Unit_Rewriting_Handle) return Analysis_Unit is
   begin
      return From_Generic_Unit (G.Unit (+Handle));
   end Unit;

   ----------
   -- Root --
   ----------

   function Root (Handle : Unit_Rewriting_Handle) return Node_Rewriting_Handle
   is
   begin
      return +G.Root (+Handle);
   end Root;

   --------------
   -- Set_Root --
   --------------

   procedure Set_Root
     (Handle : Unit_Rewriting_Handle; Root : Node_Rewriting_Handle) is
   begin
      G.Set_Root (+Handle, +Root);
   end Set_Root;

   -------------
   -- Unparse --
   -------------

   function Unparse
     (Handle : Unit_Rewriting_Handle) return Unbounded_Text_Type is
   begin
      return G.Unparse (+Handle);
   end Unparse;

   ------------
   -- Handle --
   ------------

   function Handle
     (Node : Ada_Node'Class) return Node_Rewriting_Handle is
   begin
      return +G.Handle (To_Generic_Node (Node));
   end Handle;

   ----------
   -- Node --
   ----------

   function Node
     (Handle : Node_Rewriting_Handle) return Ada_Node is
   begin
      return From_Generic_Node (G.Node (+Handle));
   end Node;

   -------------
   -- Context --
   -------------

   function Context (Handle : Node_Rewriting_Handle) return Rewriting_Handle is
   begin
      return +G.Context (+Handle);
   end Context;

   -------------
   -- Unparse --
   -------------

   function Unparse (Handle : Node_Rewriting_Handle) return Text_Type is
   begin
      return G.Unparse (+Handle);
   end Unparse;

   ----------
   -- Kind --
   ----------

   function Kind (Handle : Node_Rewriting_Handle) return Ada_Node_Kind_Type is
   begin
      return From_Generic_Node_Type (G.Type_Of (+Handle));
   end Kind;

   -----------
   -- Image --
   -----------

   function Image (Handle : Node_Rewriting_Handle) return String is
   begin
      return G.Image (+Handle);
   end Image;

   ----------
   -- Tied --
   ----------

   function Tied (Handle : Node_Rewriting_Handle) return Boolean is
   begin
      return G.Tied (+Handle);
   end Tied;

   ------------
   -- Parent --
   ------------

   function Parent
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      return +G.Parent (+Handle);
   end Parent;

   --------------------
   -- Children_Count --
   --------------------

   function Children_Count (Handle : Node_Rewriting_Handle) return Natural is
   begin
      return G.Children_Count (+Handle);
   end Children_Count;

   -----------
   -- Child --
   -----------

   function Child
     (Handle : Node_Rewriting_Handle;
      Field  : Struct_Member_Ref) return Node_Rewriting_Handle is
   begin
      return +G.Child (+Handle, Field);
   end Child;

   -----------
   -- Child --
   -----------

   function Child
     (Handle : Node_Rewriting_Handle;
      Fields : Struct_Member_Ref_Array) return Node_Rewriting_Handle is
   begin
      return +G.Child (+Handle, Fields);
   end Child;

   --------------
   -- Children --
   --------------

   function Children
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle_Array
   is
      R : constant G.Node_Rewriting_Handle_Array := G.Children (+Handle);
   begin
      return Result : Node_Rewriting_Handle_Array (R'Range) do
         for I in R'Range loop
            Result (I) := +R (I);
         end loop;
      end return;
   end Children;

   ---------------
   -- Set_Child --
   ---------------

   procedure Set_Child
     (Handle : Node_Rewriting_Handle;
      Field  : Struct_Member_Ref;
      Child  : Node_Rewriting_Handle)
   is
   begin
      G.Set_Child (+Handle, Field, +Child);
   end Set_Child;

   ----------
   -- Text --
   ----------

   function Text (Handle : Node_Rewriting_Handle) return Text_Type is
   begin
      return G.Text (+Handle);
   end Text;

   --------------
   -- Set_Text --
   --------------

   procedure Set_Text (Handle : Node_Rewriting_Handle; Text : Text_Type) is
   begin
      G.Set_Text (+Handle, Text);
   end Set_Text;

   -------------
   -- Replace --
   -------------

   procedure Replace (Handle, New_Node : Node_Rewriting_Handle) is
   begin
      G.Replace (+Handle, +New_Node);
   end Replace;

   ------------
   -- Rotate --
   ------------

   procedure Rotate (Handles : Node_Rewriting_Handle_Array) is
      H : G.Node_Rewriting_Handle_Array (Handles'Range);
   begin
      for I in Handles'Range loop
         H (I) := +Handles (I);
      end loop;
      G.Rotate (H);
   end Rotate;

   ------------------
   -- Is_List_Node --
   ------------------

   function Is_List_Node (Handle : Node_Rewriting_Handle) return Boolean is
   begin
      return G.Is_List_Node (+Handle);
   end Is_List_Node;

   -----------------
   -- First_Child --
   -----------------

   function First_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      return +G.First_Child (+Handle);
   end First_Child;

   ----------------
   -- Last_Child --
   ----------------

   function Last_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      return +G.Last_Child (+Handle);
   end Last_Child;

   ----------------
   -- Next_Child --
   ----------------

   function Next_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      return +G.Next_Child (+Handle);
   end Next_Child;

   --------------------
   -- Previous_Child --
   --------------------

   function Previous_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      return +G.Previous_Child (+Handle);
   end Previous_Child;

   -------------------
   -- Insert_Before --
   -------------------

   procedure Insert_Before
     (Handle, New_Sibling : Node_Rewriting_Handle) is
   begin
      G.Insert_Before (+Handle, +New_Sibling);
   end Insert_Before;

   ------------------
   -- Insert_After --
   ------------------

   procedure Insert_After
     (Handle, New_Sibling : Node_Rewriting_Handle) is
   begin
      G.Insert_After (+Handle, +New_Sibling);
   end Insert_After;

   ------------------
   -- Insert_First --
   ------------------

   procedure Insert_First (Handle, New_Child : Node_Rewriting_Handle) is
   begin
      G.Insert_First (+Handle, +New_Child);
   end Insert_First;

   -----------------
   -- Insert_Last --
   -----------------

   procedure Insert_Last (Handle, New_Child : Node_Rewriting_Handle) is
   begin
      G.Insert_Last (+Handle, +New_Child);
   end Insert_Last;

   ------------------
   -- Remove_Child --
   ------------------

   procedure Remove_Child (Handle : Node_Rewriting_Handle) is
   begin
      G.Remove_Child (+Handle);
   end Remove_Child;

   -----------
   -- Clone --
   -----------

   function Clone
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle is
   begin
      return +G.Clone (+Handle);
   end Clone;

   -----------------
   -- Create_Node --
   -----------------

   function Create_Node
     (Handle : Rewriting_Handle;
      Kind   : Ada_Node_Kind_Type) return Node_Rewriting_Handle is
   begin
      return +G.Create_Node (+Handle, From_Index (Self_Id, Node_Kinds (Kind)));
   end Create_Node;

   -----------------------
   -- Create_Token_Node --
   -----------------------

   function Create_Token_Node
     (Handle : Rewriting_Handle;
      Kind   : Ada_Node_Kind_Type;
      Text   : Text_Type) return Node_Rewriting_Handle is
   begin
      return +G.Create_Token_Node
        (+Handle, From_Index (Self_Id, Node_Kinds (Kind)), Text);
   end Create_Token_Node;

   -------------------------
   -- Create_Regular_Node --
   -------------------------

   function Create_Regular_Node
     (Handle   : Rewriting_Handle;
      Kind     : Ada_Node_Kind_Type;
      Children : Node_Rewriting_Handle_Array) return Node_Rewriting_Handle
   is
      C : G.Node_Rewriting_Handle_Array (Children'Range);
   begin
      for I in Children'Range loop
         C (I) := +Children (I);
      end loop;
      return +G.Create_Regular_Node
        (+Handle, From_Index (Self_Id, Node_Kinds (Kind)), C);
   end Create_Regular_Node;

   --------------------------
   -- Create_From_Template --
   --------------------------

   function Create_From_Template
     (Handle    : Rewriting_Handle;
      Template  : Text_Type;
      Arguments : Node_Rewriting_Handle_Array;
      Rule      : Grammar_Rule) return Node_Rewriting_Handle
   is
      A : G.Node_Rewriting_Handle_Array (Arguments'Range);
   begin
      for I in Arguments'Range loop
         A (I) := +Arguments (I);
      end loop;
      return +G.Create_From_Template
        (+Handle, Template, A, To_Generic_Grammar_Rule (Rule));
   end Create_From_Template;


         function Create_Constrained_Array_Indices
           (Handle : Rewriting_Handle
               ; F_List : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Constrained_Array_Indices,
                 (1 => F_List));
         end;


         function Create_Unconstrained_Array_Indices
           (Handle : Rewriting_Handle
               ; F_Types : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Unconstrained_Array_Indices,
                 (1 => F_Types));
         end;


         function Create_Aspect_Assoc
           (Handle : Rewriting_Handle
               ; F_Id : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Aspect_Assoc,
                 (1 => F_Id,
                  2 => F_Expr));
         end;


         function Create_At_Clause
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_At_Clause,
                 (1 => F_Name,
                  2 => F_Expr));
         end;


         function Create_Attribute_Def_Clause
           (Handle : Rewriting_Handle
               ; F_Attribute_Expr : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Attribute_Def_Clause,
                 (1 => F_Attribute_Expr,
                  2 => F_Expr));
         end;


         function Create_Enum_Rep_Clause
           (Handle : Rewriting_Handle
               ; F_Type_Name : Node_Rewriting_Handle
               ; F_Aggregate : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Enum_Rep_Clause,
                 (1 => F_Type_Name,
                  2 => F_Aggregate));
         end;


         function Create_Record_Rep_Clause
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_At_Expr : Node_Rewriting_Handle
               ; F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Record_Rep_Clause,
                 (1 => F_Name,
                  2 => F_At_Expr,
                  3 => F_Components));
         end;


         function Create_Aspect_Spec
           (Handle : Rewriting_Handle
               ; F_Aspect_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Aspect_Spec,
                 (1 => F_Aspect_Assocs));
         end;


         function Create_Contract_Case_Assoc
           (Handle : Rewriting_Handle
               ; F_Guard : Node_Rewriting_Handle
               ; F_Consequence : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Contract_Case_Assoc,
                 (1 => F_Guard,
                  2 => F_Consequence));
         end;


         function Create_Pragma_Argument_Assoc
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Pragma_Argument_Assoc,
                 (1 => F_Name,
                  2 => F_Expr));
         end;


         function Create_Entry_Spec
           (Handle : Rewriting_Handle
               ; F_Entry_Name : Node_Rewriting_Handle
               ; F_Family_Type : Node_Rewriting_Handle
               ; F_Entry_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Entry_Spec,
                 (1 => F_Entry_Name,
                  2 => F_Family_Type,
                  3 => F_Entry_Params));
         end;


         function Create_Subp_Spec
           (Handle : Rewriting_Handle
               ; F_Subp_Kind : Node_Rewriting_Handle
               ; F_Subp_Name : Node_Rewriting_Handle
               ; F_Subp_Params : Node_Rewriting_Handle
               ; F_Subp_Returns : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Subp_Spec,
                 (1 => F_Subp_Kind,
                  2 => F_Subp_Name,
                  3 => F_Subp_Params,
                  4 => F_Subp_Returns));
         end;


         function Create_Synthetic_Binary_Spec
           (Handle : Rewriting_Handle
               ; F_Left_Param : Node_Rewriting_Handle
               ; F_Right_Param : Node_Rewriting_Handle
               ; F_Return_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synthetic_Binary_Spec,
                 (1 => F_Left_Param,
                  2 => F_Right_Param,
                  3 => F_Return_Type_Expr));
         end;


         function Create_Synthetic_Unary_Spec
           (Handle : Rewriting_Handle
               ; F_Right_Param : Node_Rewriting_Handle
               ; F_Return_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synthetic_Unary_Spec,
                 (1 => F_Right_Param,
                  2 => F_Return_Type_Expr));
         end;


         function Create_Component_List
           (Handle : Rewriting_Handle
               ; F_Components : Node_Rewriting_Handle
               ; F_Variant_Part : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Component_List,
                 (1 => F_Components,
                  2 => F_Variant_Part));
         end;


         function Create_Known_Discriminant_Part
           (Handle : Rewriting_Handle
               ; F_Discr_Specs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Known_Discriminant_Part,
                 (1 => F_Discr_Specs));
         end;


         function Create_Entry_Completion_Formal_Params
           (Handle : Rewriting_Handle
               ; F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Entry_Completion_Formal_Params,
                 (1 => F_Params));
         end;


         function Create_Generic_Formal_Part
           (Handle : Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Formal_Part,
                 (1 => F_Decls));
         end;


         function Create_Null_Record_Def
           (Handle : Rewriting_Handle
               ; F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Null_Record_Def,
                 (1 => F_Components));
         end;


         function Create_Record_Def
           (Handle : Rewriting_Handle
               ; F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Record_Def,
                 (1 => F_Components));
         end;


         function Create_Aggregate_Assoc
           (Handle : Rewriting_Handle
               ; F_Designators : Node_Rewriting_Handle
               ; F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Aggregate_Assoc,
                 (1 => F_Designators,
                  2 => F_R_Expr));
         end;


         function Create_Multi_Dim_Array_Assoc
           (Handle : Rewriting_Handle
               ; F_Designators : Node_Rewriting_Handle
               ; F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Multi_Dim_Array_Assoc,
                 (1 => F_Designators,
                  2 => F_R_Expr));
         end;


         function Create_Composite_Constraint_Assoc
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Constraint_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Composite_Constraint_Assoc,
                 (1 => F_Ids,
                  2 => F_Constraint_Expr));
         end;


         function Create_Iterated_Assoc
           (Handle : Rewriting_Handle
               ; F_Spec : Node_Rewriting_Handle
               ; F_Key_Expr : Node_Rewriting_Handle
               ; F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Iterated_Assoc,
                 (1 => F_Spec,
                  2 => F_Key_Expr,
                  3 => F_R_Expr));
         end;


         function Create_Param_Assoc
           (Handle : Rewriting_Handle
               ; F_Designator : Node_Rewriting_Handle
               ; F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Param_Assoc,
                 (1 => F_Designator,
                  2 => F_R_Expr));
         end;


         function Create_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Abstract_State_Decl,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Anonymous_Expr_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Anonymous_Expr_Decl,
                 (1 => F_Name,
                  2 => F_Expr,
                  3 => F_Aspects));
         end;


         function Create_Component_Decl
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Component_Def : Node_Rewriting_Handle
               ; F_Default_Expr : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Component_Decl,
                 (1 => F_Ids,
                  2 => F_Component_Def,
                  3 => F_Default_Expr,
                  4 => F_Aspects));
         end;


         function Create_Discriminant_Spec
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Type_Expr : Node_Rewriting_Handle
               ; F_Default_Expr : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Discriminant_Spec,
                 (1 => F_Ids,
                  2 => F_Type_Expr,
                  3 => F_Default_Expr,
                  4 => F_Aspects));
         end;


         function Create_Generic_Formal_Obj_Decl
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Formal_Obj_Decl,
                 (1 => F_Decl,
                  2 => F_Aspects));
         end;


         function Create_Generic_Formal_Package
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Formal_Package,
                 (1 => F_Decl,
                  2 => F_Aspects));
         end;


         function Create_Generic_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Formal_Subp_Decl,
                 (1 => F_Decl,
                  2 => F_Aspects));
         end;


         function Create_Generic_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Formal_Type_Decl,
                 (1 => F_Decl,
                  2 => F_Aspects));
         end;


         function Create_Param_Spec
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Has_Aliased : Node_Rewriting_Handle
               ; F_Mode : Node_Rewriting_Handle
               ; F_Type_Expr : Node_Rewriting_Handle
               ; F_Default_Expr : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Param_Spec,
                 (1 => F_Ids,
                  2 => F_Has_Aliased,
                  3 => F_Mode,
                  4 => F_Type_Expr,
                  5 => F_Default_Expr,
                  6 => F_Aspects));
         end;


         function Create_Synthetic_Formal_Param_Decl
           (Handle : Rewriting_Handle
               ; F_Param_Type : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synthetic_Formal_Param_Decl,
                 (1 => F_Param_Type,
                  2 => F_Aspects));
         end;


         function Create_Generic_Package_Internal
           (Handle : Rewriting_Handle
               ; F_Package_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Public_Part : Node_Rewriting_Handle
               ; F_Private_Part : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Package_Internal,
                 (1 => F_Package_Name,
                  2 => F_Aspects,
                  3 => F_Public_Part,
                  4 => F_Private_Part,
                  5 => F_End_Name));
         end;


         function Create_Package_Decl
           (Handle : Rewriting_Handle
               ; F_Package_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Public_Part : Node_Rewriting_Handle
               ; F_Private_Part : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Package_Decl,
                 (1 => F_Package_Name,
                  2 => F_Aspects,
                  3 => F_Public_Part,
                  4 => F_Private_Part,
                  5 => F_End_Name));
         end;


         function Create_Discrete_Base_Subtype_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Discrete_Base_Subtype_Decl,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Subtype_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Subtype : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Subtype_Decl,
                 (1 => F_Name,
                  2 => F_Subtype,
                  3 => F_Aspects));
         end;


         function Create_Classwide_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Classwide_Type_Decl,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Incomplete_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Incomplete_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Aspects));
         end;


         function Create_Incomplete_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Is_Tagged : Node_Rewriting_Handle
               ; F_Default_Type : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Incomplete_Formal_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Aspects,
                  4 => F_Is_Tagged,
                  5 => F_Default_Type));
         end;


         function Create_Incomplete_Tagged_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Has_Abstract : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Incomplete_Tagged_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Aspects,
                  4 => F_Has_Abstract));
         end;


         function Create_Protected_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Interfaces : Node_Rewriting_Handle
               ; F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Protected_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Aspects,
                  4 => F_Interfaces,
                  5 => F_Definition));
         end;


         function Create_Task_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Task_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Aspects,
                  4 => F_Definition));
         end;


         function Create_Single_Task_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Single_Task_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Aspects,
                  4 => F_Definition));
         end;


         function Create_Anonymous_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Type_Def : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Anonymous_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Type_Def,
                  4 => F_Aspects));
         end;


         function Create_Synth_Anonymous_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Type_Def : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synth_Anonymous_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Type_Def,
                  4 => F_Aspects));
         end;


         function Create_Concrete_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Type_Def : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Concrete_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Type_Def,
                  4 => F_Aspects));
         end;


         function Create_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Discriminants : Node_Rewriting_Handle
               ; F_Type_Def : Node_Rewriting_Handle
               ; F_Default_Type : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Formal_Type_Decl,
                 (1 => F_Name,
                  2 => F_Discriminants,
                  3 => F_Type_Def,
                  4 => F_Default_Type,
                  5 => F_Aspects));
         end;


         function Create_Abstract_Subp_Decl
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Abstract_Subp_Decl,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Aspects));
         end;


         function Create_Abstract_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Default_Expr : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Abstract_Formal_Subp_Decl,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Default_Expr,
                  4 => F_Aspects));
         end;


         function Create_Concrete_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Default_Expr : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Concrete_Formal_Subp_Decl,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Default_Expr,
                  4 => F_Aspects));
         end;


         function Create_Subp_Decl
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Subp_Decl,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Aspects));
         end;


         function Create_Entry_Decl
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Spec : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Entry_Decl,
                 (1 => F_Overriding,
                  2 => F_Spec,
                  3 => F_Aspects));
         end;


         function Create_Enum_Literal_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Enum_Literal_Decl,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Synthetic_Char_Enum_Lit
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synthetic_Char_Enum_Lit,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Generic_Subp_Internal
           (Handle : Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Subp_Internal,
                 (1 => F_Subp_Spec,
                  2 => F_Aspects));
         end;


         function Create_Synthetic_Subp_Decl
           (Handle : Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Spec : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synthetic_Subp_Decl,
                 (1 => F_Aspects,
                  2 => F_Spec));
         end;


         function Create_Accept_Stmt_Body
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Accept_Stmt_Body,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Expr_Function
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Expr_Function,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Expr,
                  4 => F_Aspects));
         end;


         function Create_Null_Subp_Decl
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Null_Subp_Decl,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Aspects));
         end;


         function Create_Subp_Body
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Subp_Body,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Aspects,
                  4 => F_Decls,
                  5 => F_Stmts,
                  6 => F_End_Name));
         end;


         function Create_Subp_Renaming_Decl
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Renames : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Subp_Renaming_Decl,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Renames,
                  4 => F_Aspects));
         end;


         function Create_Package_Body_Stub
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Package_Body_Stub,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Protected_Body_Stub
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Protected_Body_Stub,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Subp_Body_Stub
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Subp_Body_Stub,
                 (1 => F_Overriding,
                  2 => F_Subp_Spec,
                  3 => F_Aspects));
         end;


         function Create_Task_Body_Stub
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Task_Body_Stub,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Entry_Body
           (Handle : Rewriting_Handle
               ; F_Entry_Name : Node_Rewriting_Handle
               ; F_Index_Spec : Node_Rewriting_Handle
               ; F_Params : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Barrier : Node_Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Entry_Body,
                 (1 => F_Entry_Name,
                  2 => F_Index_Spec,
                  3 => F_Params,
                  4 => F_Aspects,
                  5 => F_Barrier,
                  6 => F_Decls,
                  7 => F_Stmts,
                  8 => F_End_Name));
         end;


         function Create_Package_Body
           (Handle : Rewriting_Handle
               ; F_Package_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Package_Body,
                 (1 => F_Package_Name,
                  2 => F_Aspects,
                  3 => F_Decls,
                  4 => F_Stmts,
                  5 => F_End_Name));
         end;


         function Create_Protected_Body
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Protected_Body,
                 (1 => F_Name,
                  2 => F_Aspects,
                  3 => F_Decls,
                  4 => F_End_Name));
         end;


         function Create_Task_Body
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Task_Body,
                 (1 => F_Name,
                  2 => F_Aspects,
                  3 => F_Decls,
                  4 => F_Stmts,
                  5 => F_End_Name));
         end;


         function Create_Entry_Index_Spec
           (Handle : Rewriting_Handle
               ; F_Id : Node_Rewriting_Handle
               ; F_Subtype : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Entry_Index_Spec,
                 (1 => F_Id,
                  2 => F_Subtype,
                  3 => F_Aspects));
         end;


         function Create_Error_Decl
           (Handle : Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Error_Decl,
                 (1 => F_Aspects));
         end;


         function Create_Exception_Decl
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Renames : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Exception_Decl,
                 (1 => F_Ids,
                  2 => F_Renames,
                  3 => F_Aspects));
         end;


         function Create_Exception_Handler
           (Handle : Rewriting_Handle
               ; F_Exception_Name : Node_Rewriting_Handle
               ; F_Handled_Exceptions : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Exception_Handler,
                 (1 => F_Exception_Name,
                  2 => F_Handled_Exceptions,
                  3 => F_Stmts,
                  4 => F_Aspects));
         end;


         function Create_For_Loop_Var_Decl
           (Handle : Rewriting_Handle
               ; F_Id : Node_Rewriting_Handle
               ; F_Id_Type : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_For_Loop_Var_Decl,
                 (1 => F_Id,
                  2 => F_Id_Type,
                  3 => F_Aspects));
         end;


         function Create_Generic_Package_Decl
           (Handle : Rewriting_Handle
               ; F_Formal_Part : Node_Rewriting_Handle
               ; F_Package_Decl : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Package_Decl,
                 (1 => F_Formal_Part,
                  2 => F_Package_Decl,
                  3 => F_Aspects));
         end;


         function Create_Generic_Subp_Decl
           (Handle : Rewriting_Handle
               ; F_Formal_Part : Node_Rewriting_Handle
               ; F_Subp_Decl : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Subp_Decl,
                 (1 => F_Formal_Part,
                  2 => F_Subp_Decl,
                  3 => F_Aspects));
         end;


         function Create_Generic_Package_Instantiation
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Generic_Pkg_Name : Node_Rewriting_Handle
               ; F_Params : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Package_Instantiation,
                 (1 => F_Name,
                  2 => F_Generic_Pkg_Name,
                  3 => F_Params,
                  4 => F_Aspects));
         end;


         function Create_Generic_Subp_Instantiation
           (Handle : Rewriting_Handle
               ; F_Overriding : Node_Rewriting_Handle
               ; F_Kind : Node_Rewriting_Handle
               ; F_Subp_Name : Node_Rewriting_Handle
               ; F_Generic_Subp_Name : Node_Rewriting_Handle
               ; F_Params : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Subp_Instantiation,
                 (1 => F_Overriding,
                  2 => F_Kind,
                  3 => F_Subp_Name,
                  4 => F_Generic_Subp_Name,
                  5 => F_Params,
                  6 => F_Aspects));
         end;


         function Create_Generic_Package_Renaming_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Renames : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Package_Renaming_Decl,
                 (1 => F_Name,
                  2 => F_Renames,
                  3 => F_Aspects));
         end;


         function Create_Generic_Subp_Renaming_Decl
           (Handle : Rewriting_Handle
               ; F_Kind : Node_Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Renames : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Generic_Subp_Renaming_Decl,
                 (1 => F_Kind,
                  2 => F_Name,
                  3 => F_Renames,
                  4 => F_Aspects));
         end;


         function Create_Label_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Label_Decl,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Named_Stmt_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Named_Stmt_Decl,
                 (1 => F_Name,
                  2 => F_Aspects));
         end;


         function Create_Number_Decl
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Number_Decl,
                 (1 => F_Ids,
                  2 => F_Expr,
                  3 => F_Aspects));
         end;


         function Create_Object_Decl
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Has_Aliased : Node_Rewriting_Handle
               ; F_Has_Constant : Node_Rewriting_Handle
               ; F_Mode : Node_Rewriting_Handle
               ; F_Type_Expr : Node_Rewriting_Handle
               ; F_Default_Expr : Node_Rewriting_Handle
               ; F_Renaming_Clause : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Object_Decl,
                 (1 => F_Ids,
                  2 => F_Has_Aliased,
                  3 => F_Has_Constant,
                  4 => F_Mode,
                  5 => F_Type_Expr,
                  6 => F_Default_Expr,
                  7 => F_Renaming_Clause,
                  8 => F_Aspects));
         end;


         function Create_Extended_Return_Stmt_Object_Decl
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Has_Aliased : Node_Rewriting_Handle
               ; F_Has_Constant : Node_Rewriting_Handle
               ; F_Mode : Node_Rewriting_Handle
               ; F_Type_Expr : Node_Rewriting_Handle
               ; F_Default_Expr : Node_Rewriting_Handle
               ; F_Renaming_Clause : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Extended_Return_Stmt_Object_Decl,
                 (1 => F_Ids,
                  2 => F_Has_Aliased,
                  3 => F_Has_Constant,
                  4 => F_Mode,
                  5 => F_Type_Expr,
                  6 => F_Default_Expr,
                  7 => F_Renaming_Clause,
                  8 => F_Aspects));
         end;


         function Create_No_Type_Object_Renaming_Decl
           (Handle : Rewriting_Handle
               ; F_Ids : Node_Rewriting_Handle
               ; F_Has_Aliased : Node_Rewriting_Handle
               ; F_Has_Constant : Node_Rewriting_Handle
               ; F_Mode : Node_Rewriting_Handle
               ; F_Type_Expr : Node_Rewriting_Handle
               ; F_Default_Expr : Node_Rewriting_Handle
               ; F_Renaming_Clause : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_No_Type_Object_Renaming_Decl,
                 (1 => F_Ids,
                  2 => F_Has_Aliased,
                  3 => F_Has_Constant,
                  4 => F_Mode,
                  5 => F_Type_Expr,
                  6 => F_Default_Expr,
                  7 => F_Renaming_Clause,
                  8 => F_Aspects));
         end;


         function Create_Package_Renaming_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Renames : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Package_Renaming_Decl,
                 (1 => F_Name,
                  2 => F_Renames,
                  3 => F_Aspects));
         end;


         function Create_Single_Protected_Decl
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
               ; F_Interfaces : Node_Rewriting_Handle
               ; F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Single_Protected_Decl,
                 (1 => F_Name,
                  2 => F_Aspects,
                  3 => F_Interfaces,
                  4 => F_Definition));
         end;


         function Create_Single_Task_Decl
           (Handle : Rewriting_Handle
               ; F_Task_Type : Node_Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Single_Task_Decl,
                 (1 => F_Task_Type,
                  2 => F_Aspects));
         end;


         function Create_Synthetic_Object_Decl
           (Handle : Rewriting_Handle
               ; F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synthetic_Object_Decl,
                 (1 => F_Aspects));
         end;


         function Create_Case_Stmt_Alternative
           (Handle : Rewriting_Handle
               ; F_Choices : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Case_Stmt_Alternative,
                 (1 => F_Choices,
                  2 => F_Stmts));
         end;


         function Create_Compilation_Unit
           (Handle : Rewriting_Handle
               ; F_Prelude : Node_Rewriting_Handle
               ; F_Body : Node_Rewriting_Handle
               ; F_Pragmas : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Compilation_Unit,
                 (1 => F_Prelude,
                  2 => F_Body,
                  3 => F_Pragmas));
         end;


         function Create_Component_Clause
           (Handle : Rewriting_Handle
               ; F_Id : Node_Rewriting_Handle
               ; F_Position : Node_Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Component_Clause,
                 (1 => F_Id,
                  2 => F_Position,
                  3 => F_Range));
         end;


         function Create_Component_Def
           (Handle : Rewriting_Handle
               ; F_Has_Aliased : Node_Rewriting_Handle
               ; F_Has_Constant : Node_Rewriting_Handle
               ; F_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Component_Def,
                 (1 => F_Has_Aliased,
                  2 => F_Has_Constant,
                  3 => F_Type_Expr));
         end;


         function Create_Composite_Constraint
           (Handle : Rewriting_Handle
               ; F_Constraints : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Composite_Constraint,
                 (1 => F_Constraints));
         end;


         function Create_Delta_Constraint
           (Handle : Rewriting_Handle
               ; F_Delta : Node_Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Delta_Constraint,
                 (1 => F_Delta,
                  2 => F_Range));
         end;


         function Create_Digits_Constraint
           (Handle : Rewriting_Handle
               ; F_Digits : Node_Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Digits_Constraint,
                 (1 => F_Digits,
                  2 => F_Range));
         end;


         function Create_Range_Constraint
           (Handle : Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Range_Constraint,
                 (1 => F_Range));
         end;


         function Create_Declarative_Part
           (Handle : Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Declarative_Part,
                 (1 => F_Decls));
         end;


         function Create_Private_Part
           (Handle : Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Private_Part,
                 (1 => F_Decls));
         end;


         function Create_Public_Part
           (Handle : Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Public_Part,
                 (1 => F_Decls));
         end;


         function Create_Else_Part
           (Handle : Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Else_Part,
                 (1 => F_Stmts));
         end;


         function Create_Elsif_Expr_Part
           (Handle : Rewriting_Handle
               ; F_Cond_Expr : Node_Rewriting_Handle
               ; F_Then_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Elsif_Expr_Part,
                 (1 => F_Cond_Expr,
                  2 => F_Then_Expr));
         end;


         function Create_Elsif_Stmt_Part
           (Handle : Rewriting_Handle
               ; F_Cond_Expr : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Elsif_Stmt_Part,
                 (1 => F_Cond_Expr,
                  2 => F_Stmts));
         end;


         function Create_Abstract_State_Decl_Expr
           (Handle : Rewriting_Handle
               ; F_State_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Abstract_State_Decl_Expr,
                 (1 => F_State_Decl));
         end;


         function Create_Allocator
           (Handle : Rewriting_Handle
               ; F_Subpool : Node_Rewriting_Handle
               ; F_Type_Or_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Allocator,
                 (1 => F_Subpool,
                  2 => F_Type_Or_Expr));
         end;


         function Create_Aggregate
           (Handle : Rewriting_Handle
               ; F_Ancestor_Expr : Node_Rewriting_Handle
               ; F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Aggregate,
                 (1 => F_Ancestor_Expr,
                  2 => F_Assocs));
         end;


         function Create_Bracket_Aggregate
           (Handle : Rewriting_Handle
               ; F_Ancestor_Expr : Node_Rewriting_Handle
               ; F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Bracket_Aggregate,
                 (1 => F_Ancestor_Expr,
                  2 => F_Assocs));
         end;


         function Create_Delta_Aggregate
           (Handle : Rewriting_Handle
               ; F_Ancestor_Expr : Node_Rewriting_Handle
               ; F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Delta_Aggregate,
                 (1 => F_Ancestor_Expr,
                  2 => F_Assocs));
         end;


         function Create_Bracket_Delta_Aggregate
           (Handle : Rewriting_Handle
               ; F_Ancestor_Expr : Node_Rewriting_Handle
               ; F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Bracket_Delta_Aggregate,
                 (1 => F_Ancestor_Expr,
                  2 => F_Assocs));
         end;


         function Create_Null_Record_Aggregate
           (Handle : Rewriting_Handle
               ; F_Ancestor_Expr : Node_Rewriting_Handle
               ; F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Null_Record_Aggregate,
                 (1 => F_Ancestor_Expr,
                  2 => F_Assocs));
         end;


         function Create_Bin_Op
           (Handle : Rewriting_Handle
               ; F_Left : Node_Rewriting_Handle
               ; F_Op : Node_Rewriting_Handle
               ; F_Right : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Bin_Op,
                 (1 => F_Left,
                  2 => F_Op,
                  3 => F_Right));
         end;


         function Create_Relation_Op
           (Handle : Rewriting_Handle
               ; F_Left : Node_Rewriting_Handle
               ; F_Op : Node_Rewriting_Handle
               ; F_Right : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Relation_Op,
                 (1 => F_Left,
                  2 => F_Op,
                  3 => F_Right));
         end;


         function Create_Case_Expr_Alternative
           (Handle : Rewriting_Handle
               ; F_Choices : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Case_Expr_Alternative,
                 (1 => F_Choices,
                  2 => F_Expr));
         end;


         function Create_Concat_Op
           (Handle : Rewriting_Handle
               ; F_First_Operand : Node_Rewriting_Handle
               ; F_Other_Operands : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Concat_Op,
                 (1 => F_First_Operand,
                  2 => F_Other_Operands));
         end;


         function Create_Concat_Operand
           (Handle : Rewriting_Handle
               ; F_Operator : Node_Rewriting_Handle
               ; F_Operand : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Concat_Operand,
                 (1 => F_Operator,
                  2 => F_Operand));
         end;


         function Create_Case_Expr
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_Cases : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Case_Expr,
                 (1 => F_Expr,
                  2 => F_Cases));
         end;


         function Create_If_Expr
           (Handle : Rewriting_Handle
               ; F_Cond_Expr : Node_Rewriting_Handle
               ; F_Then_Expr : Node_Rewriting_Handle
               ; F_Alternatives : Node_Rewriting_Handle
               ; F_Else_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_If_Expr,
                 (1 => F_Cond_Expr,
                  2 => F_Then_Expr,
                  3 => F_Alternatives,
                  4 => F_Else_Expr));
         end;


         function Create_Contract_Cases
           (Handle : Rewriting_Handle
               ; F_Contract_Cases : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Contract_Cases,
                 (1 => F_Contract_Cases));
         end;


         function Create_Decl_Expr
           (Handle : Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Decl_Expr,
                 (1 => F_Decls,
                  2 => F_Expr));
         end;


         function Create_Format_String_Literal
           (Handle : Rewriting_Handle
               ; F_Opening_Chunk : Node_Rewriting_Handle
               ; F_Mid_Exprs : Node_Rewriting_Handle
               ; F_Trailing_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Format_String_Literal,
                 (1 => F_Opening_Chunk,
                  2 => F_Mid_Exprs,
                  3 => F_Trailing_Expr));
         end;


         function Create_Membership_Expr
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_Op : Node_Rewriting_Handle
               ; F_Membership_Exprs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Membership_Expr,
                 (1 => F_Expr,
                  2 => F_Op,
                  3 => F_Membership_Exprs));
         end;


         function Create_Array_Subcomponent_Choice_Name
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Array_Subcomponent_Choice_Name,
                 (1 => F_Name,
                  2 => F_Suffix));
         end;


         function Create_Attribute_Ref
           (Handle : Rewriting_Handle
               ; F_Prefix : Node_Rewriting_Handle
               ; F_Attribute : Node_Rewriting_Handle
               ; F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Attribute_Ref,
                 (1 => F_Prefix,
                  2 => F_Attribute,
                  3 => F_Args));
         end;


         function Create_Call_Expr
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Call_Expr,
                 (1 => F_Name,
                  2 => F_Suffix));
         end;


         function Create_Defining_Name
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Defining_Name,
                 (1 => F_Name));
         end;


         function Create_Synthetic_Defining_Name
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synthetic_Defining_Name,
                 (1 => F_Name));
         end;


         function Create_Discrete_Subtype_Name
           (Handle : Rewriting_Handle
               ; F_Subtype : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Discrete_Subtype_Name,
                 (1 => F_Subtype));
         end;


         function Create_Dotted_Name
           (Handle : Rewriting_Handle
               ; F_Prefix : Node_Rewriting_Handle
               ; F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Dotted_Name,
                 (1 => F_Prefix,
                  2 => F_Suffix));
         end;


         function Create_End_Name
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_End_Name,
                 (1 => F_Name));
         end;


         function Create_Explicit_Deref
           (Handle : Rewriting_Handle
               ; F_Prefix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Explicit_Deref,
                 (1 => F_Prefix));
         end;


         function Create_Qual_Expr
           (Handle : Rewriting_Handle
               ; F_Prefix : Node_Rewriting_Handle
               ; F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Qual_Expr,
                 (1 => F_Prefix,
                  2 => F_Suffix));
         end;


         function Create_Reduce_Attribute_Ref
           (Handle : Rewriting_Handle
               ; F_Prefix : Node_Rewriting_Handle
               ; F_Attribute : Node_Rewriting_Handle
               ; F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Reduce_Attribute_Ref,
                 (1 => F_Prefix,
                  2 => F_Attribute,
                  3 => F_Args));
         end;


         function Create_Update_Attribute_Ref
           (Handle : Rewriting_Handle
               ; F_Prefix : Node_Rewriting_Handle
               ; F_Attribute : Node_Rewriting_Handle
               ; F_Values : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Update_Attribute_Ref,
                 (1 => F_Prefix,
                  2 => F_Attribute,
                  3 => F_Values));
         end;


         function Create_Paren_Expr
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Paren_Expr,
                 (1 => F_Expr));
         end;


         function Create_Quantified_Expr
           (Handle : Rewriting_Handle
               ; F_Quantifier : Node_Rewriting_Handle
               ; F_Loop_Spec : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Quantified_Expr,
                 (1 => F_Quantifier,
                  2 => F_Loop_Spec,
                  3 => F_Expr));
         end;


         function Create_Raise_Expr
           (Handle : Rewriting_Handle
               ; F_Exception_Name : Node_Rewriting_Handle
               ; F_Error_Message : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Raise_Expr,
                 (1 => F_Exception_Name,
                  2 => F_Error_Message));
         end;


         function Create_Un_Op
           (Handle : Rewriting_Handle
               ; F_Op : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Un_Op,
                 (1 => F_Op,
                  2 => F_Expr));
         end;


         function Create_For_Loop_Iter_Filter
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_For_Loop_Iter_Filter,
                 (1 => F_Expr));
         end;


         function Create_Format_String_Chunk
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_String_Tok : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Format_String_Chunk,
                 (1 => F_Expr,
                  2 => F_String_Tok));
         end;


         function Create_Handled_Stmts
           (Handle : Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_Exceptions : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Handled_Stmts,
                 (1 => F_Stmts,
                  2 => F_Exceptions));
         end;


         function Create_Library_Item
           (Handle : Rewriting_Handle
               ; F_Has_Private : Node_Rewriting_Handle
               ; F_Item : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Library_Item,
                 (1 => F_Has_Private,
                  2 => F_Item));
         end;


         function Create_For_Loop_Spec
           (Handle : Rewriting_Handle
               ; F_Var_Decl : Node_Rewriting_Handle
               ; F_Loop_Type : Node_Rewriting_Handle
               ; F_Has_Reverse : Node_Rewriting_Handle
               ; F_Iter_Expr : Node_Rewriting_Handle
               ; F_Iter_Filter : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_For_Loop_Spec,
                 (1 => F_Var_Decl,
                  2 => F_Loop_Type,
                  3 => F_Has_Reverse,
                  4 => F_Iter_Expr,
                  5 => F_Iter_Filter));
         end;


         function Create_While_Loop_Spec
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_While_Loop_Spec,
                 (1 => F_Expr));
         end;


         function Create_Multi_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Multi_Abstract_State_Decl,
                 (1 => F_Decls));
         end;


         function Create_Params
           (Handle : Rewriting_Handle
               ; F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Params,
                 (1 => F_Params));
         end;


         function Create_Paren_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Paren_Abstract_State_Decl,
                 (1 => F_Decl));
         end;


         function Create_Pp_Elsif_Directive
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_Then_Kw : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Pp_Elsif_Directive,
                 (1 => F_Expr,
                  2 => F_Then_Kw));
         end;


         function Create_Pp_If_Directive
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_Then_Kw : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Pp_If_Directive,
                 (1 => F_Expr,
                  2 => F_Then_Kw));
         end;


         function Create_Pragma_Node
           (Handle : Rewriting_Handle
               ; F_Id : Node_Rewriting_Handle
               ; F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Pragma_Node,
                 (1 => F_Id,
                  2 => F_Args));
         end;


         function Create_Protected_Def
           (Handle : Rewriting_Handle
               ; F_Public_Part : Node_Rewriting_Handle
               ; F_Private_Part : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Protected_Def,
                 (1 => F_Public_Part,
                  2 => F_Private_Part,
                  3 => F_End_Name));
         end;


         function Create_Range_Spec
           (Handle : Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Range_Spec,
                 (1 => F_Range));
         end;


         function Create_Renaming_Clause
           (Handle : Rewriting_Handle
               ; F_Renamed_Object : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Renaming_Clause,
                 (1 => F_Renamed_Object));
         end;


         function Create_Synthetic_Renaming_Clause
           (Handle : Rewriting_Handle
               ; F_Renamed_Object : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Synthetic_Renaming_Clause,
                 (1 => F_Renamed_Object));
         end;


         function Create_Select_When_Part
           (Handle : Rewriting_Handle
               ; F_Cond_Expr : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Select_When_Part,
                 (1 => F_Cond_Expr,
                  2 => F_Stmts));
         end;


         function Create_Accept_Stmt
           (Handle : Rewriting_Handle
               ; F_Body_Decl : Node_Rewriting_Handle
               ; F_Entry_Index_Expr : Node_Rewriting_Handle
               ; F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Accept_Stmt,
                 (1 => F_Body_Decl,
                  2 => F_Entry_Index_Expr,
                  3 => F_Params));
         end;


         function Create_Accept_Stmt_With_Stmts
           (Handle : Rewriting_Handle
               ; F_Body_Decl : Node_Rewriting_Handle
               ; F_Entry_Index_Expr : Node_Rewriting_Handle
               ; F_Params : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Accept_Stmt_With_Stmts,
                 (1 => F_Body_Decl,
                  2 => F_Entry_Index_Expr,
                  3 => F_Params,
                  4 => F_Stmts,
                  5 => F_End_Name));
         end;


         function Create_For_Loop_Stmt
           (Handle : Rewriting_Handle
               ; F_Spec : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_For_Loop_Stmt,
                 (1 => F_Spec,
                  2 => F_Stmts,
                  3 => F_End_Name));
         end;


         function Create_Loop_Stmt
           (Handle : Rewriting_Handle
               ; F_Spec : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Loop_Stmt,
                 (1 => F_Spec,
                  2 => F_Stmts,
                  3 => F_End_Name));
         end;


         function Create_While_Loop_Stmt
           (Handle : Rewriting_Handle
               ; F_Spec : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_While_Loop_Stmt,
                 (1 => F_Spec,
                  2 => F_Stmts,
                  3 => F_End_Name));
         end;


         function Create_Begin_Block
           (Handle : Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Begin_Block,
                 (1 => F_Stmts,
                  2 => F_End_Name));
         end;


         function Create_Decl_Block
           (Handle : Rewriting_Handle
               ; F_Decls : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Decl_Block,
                 (1 => F_Decls,
                  2 => F_Stmts,
                  3 => F_End_Name));
         end;


         function Create_Case_Stmt
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
               ; F_Pragmas : Node_Rewriting_Handle
               ; F_Alternatives : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Case_Stmt,
                 (1 => F_Expr,
                  2 => F_Pragmas,
                  3 => F_Alternatives));
         end;


         function Create_Extended_Return_Stmt
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Extended_Return_Stmt,
                 (1 => F_Decl,
                  2 => F_Stmts));
         end;


         function Create_If_Stmt
           (Handle : Rewriting_Handle
               ; F_Cond_Expr : Node_Rewriting_Handle
               ; F_Then_Stmts : Node_Rewriting_Handle
               ; F_Alternatives : Node_Rewriting_Handle
               ; F_Else_Part : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_If_Stmt,
                 (1 => F_Cond_Expr,
                  2 => F_Then_Stmts,
                  3 => F_Alternatives,
                  4 => F_Else_Part));
         end;


         function Create_Named_Stmt
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
               ; F_Stmt : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Named_Stmt,
                 (1 => F_Decl,
                  2 => F_Stmt));
         end;


         function Create_Select_Stmt
           (Handle : Rewriting_Handle
               ; F_Guards : Node_Rewriting_Handle
               ; F_Else_Part : Node_Rewriting_Handle
               ; F_Then_Abort_Part : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Select_Stmt,
                 (1 => F_Guards,
                  2 => F_Else_Part,
                  3 => F_Then_Abort_Part));
         end;


         function Create_Abort_Stmt
           (Handle : Rewriting_Handle
               ; F_Names : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Abort_Stmt,
                 (1 => F_Names));
         end;


         function Create_Assign_Stmt
           (Handle : Rewriting_Handle
               ; F_Dest : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Assign_Stmt,
                 (1 => F_Dest,
                  2 => F_Expr));
         end;


         function Create_Call_Stmt
           (Handle : Rewriting_Handle
               ; F_Call : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Call_Stmt,
                 (1 => F_Call));
         end;


         function Create_Delay_Stmt
           (Handle : Rewriting_Handle
               ; F_Has_Until : Node_Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Delay_Stmt,
                 (1 => F_Has_Until,
                  2 => F_Expr));
         end;


         function Create_Exit_Stmt
           (Handle : Rewriting_Handle
               ; F_Loop_Name : Node_Rewriting_Handle
               ; F_Cond_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Exit_Stmt,
                 (1 => F_Loop_Name,
                  2 => F_Cond_Expr));
         end;


         function Create_Goto_Stmt
           (Handle : Rewriting_Handle
               ; F_Label_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Goto_Stmt,
                 (1 => F_Label_Name));
         end;


         function Create_Label
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Label,
                 (1 => F_Decl));
         end;


         function Create_Raise_Stmt
           (Handle : Rewriting_Handle
               ; F_Exception_Name : Node_Rewriting_Handle
               ; F_Error_Message : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Raise_Stmt,
                 (1 => F_Exception_Name,
                  2 => F_Error_Message));
         end;


         function Create_Requeue_Stmt
           (Handle : Rewriting_Handle
               ; F_Call_Name : Node_Rewriting_Handle
               ; F_Has_Abort : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Requeue_Stmt,
                 (1 => F_Call_Name,
                  2 => F_Has_Abort));
         end;


         function Create_Return_Stmt
           (Handle : Rewriting_Handle
               ; F_Return_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Return_Stmt,
                 (1 => F_Return_Expr));
         end;


         function Create_Simple_Decl_Stmt
           (Handle : Rewriting_Handle
               ; F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Simple_Decl_Stmt,
                 (1 => F_Decl));
         end;


         function Create_Subunit
           (Handle : Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Body : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Subunit,
                 (1 => F_Name,
                  2 => F_Body));
         end;


         function Create_Task_Def
           (Handle : Rewriting_Handle
               ; F_Interfaces : Node_Rewriting_Handle
               ; F_Public_Part : Node_Rewriting_Handle
               ; F_Private_Part : Node_Rewriting_Handle
               ; F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Task_Def,
                 (1 => F_Interfaces,
                  2 => F_Public_Part,
                  3 => F_Private_Part,
                  4 => F_End_Name));
         end;


         function Create_Then_Abort_Part
           (Handle : Rewriting_Handle
               ; F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Then_Abort_Part,
                 (1 => F_Stmts));
         end;


         function Create_Access_To_Subp_Def
           (Handle : Rewriting_Handle
               ; F_Has_Not_Null : Node_Rewriting_Handle
               ; F_Has_Protected : Node_Rewriting_Handle
               ; F_Subp_Spec : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Access_To_Subp_Def,
                 (1 => F_Has_Not_Null,
                  2 => F_Has_Protected,
                  3 => F_Subp_Spec));
         end;


         function Create_Anonymous_Type_Access_Def
           (Handle : Rewriting_Handle
               ; F_Has_Not_Null : Node_Rewriting_Handle
               ; F_Type_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Anonymous_Type_Access_Def,
                 (1 => F_Has_Not_Null,
                  2 => F_Type_Decl));
         end;


         function Create_Type_Access_Def
           (Handle : Rewriting_Handle
               ; F_Has_Not_Null : Node_Rewriting_Handle
               ; F_Has_All : Node_Rewriting_Handle
               ; F_Has_Constant : Node_Rewriting_Handle
               ; F_Subtype_Indication : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Type_Access_Def,
                 (1 => F_Has_Not_Null,
                  2 => F_Has_All,
                  3 => F_Has_Constant,
                  4 => F_Subtype_Indication));
         end;


         function Create_Array_Type_Def
           (Handle : Rewriting_Handle
               ; F_Indices : Node_Rewriting_Handle
               ; F_Component_Type : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Array_Type_Def,
                 (1 => F_Indices,
                  2 => F_Component_Type));
         end;


         function Create_Derived_Type_Def
           (Handle : Rewriting_Handle
               ; F_Has_Abstract : Node_Rewriting_Handle
               ; F_Has_Limited : Node_Rewriting_Handle
               ; F_Has_Synchronized : Node_Rewriting_Handle
               ; F_Subtype_Indication : Node_Rewriting_Handle
               ; F_Interfaces : Node_Rewriting_Handle
               ; F_Record_Extension : Node_Rewriting_Handle
               ; F_Has_With_Private : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Derived_Type_Def,
                 (1 => F_Has_Abstract,
                  2 => F_Has_Limited,
                  3 => F_Has_Synchronized,
                  4 => F_Subtype_Indication,
                  5 => F_Interfaces,
                  6 => F_Record_Extension,
                  7 => F_Has_With_Private));
         end;


         function Create_Enum_Type_Def
           (Handle : Rewriting_Handle
               ; F_Enum_Literals : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Enum_Type_Def,
                 (1 => F_Enum_Literals));
         end;


         function Create_Interface_Type_Def
           (Handle : Rewriting_Handle
               ; F_Interface_Kind : Node_Rewriting_Handle
               ; F_Interfaces : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Interface_Type_Def,
                 (1 => F_Interface_Kind,
                  2 => F_Interfaces));
         end;


         function Create_Mod_Int_Type_Def
           (Handle : Rewriting_Handle
               ; F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Mod_Int_Type_Def,
                 (1 => F_Expr));
         end;


         function Create_Private_Type_Def
           (Handle : Rewriting_Handle
               ; F_Has_Abstract : Node_Rewriting_Handle
               ; F_Has_Tagged : Node_Rewriting_Handle
               ; F_Has_Limited : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Private_Type_Def,
                 (1 => F_Has_Abstract,
                  2 => F_Has_Tagged,
                  3 => F_Has_Limited));
         end;


         function Create_Decimal_Fixed_Point_Def
           (Handle : Rewriting_Handle
               ; F_Delta : Node_Rewriting_Handle
               ; F_Digits : Node_Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Decimal_Fixed_Point_Def,
                 (1 => F_Delta,
                  2 => F_Digits,
                  3 => F_Range));
         end;


         function Create_Floating_Point_Def
           (Handle : Rewriting_Handle
               ; F_Num_Digits : Node_Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Floating_Point_Def,
                 (1 => F_Num_Digits,
                  2 => F_Range));
         end;


         function Create_Ordinary_Fixed_Point_Def
           (Handle : Rewriting_Handle
               ; F_Delta : Node_Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Ordinary_Fixed_Point_Def,
                 (1 => F_Delta,
                  2 => F_Range));
         end;


         function Create_Record_Type_Def
           (Handle : Rewriting_Handle
               ; F_Has_Abstract : Node_Rewriting_Handle
               ; F_Has_Tagged : Node_Rewriting_Handle
               ; F_Has_Limited : Node_Rewriting_Handle
               ; F_Record_Def : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Record_Type_Def,
                 (1 => F_Has_Abstract,
                  2 => F_Has_Tagged,
                  3 => F_Has_Limited,
                  4 => F_Record_Def));
         end;


         function Create_Signed_Int_Type_Def
           (Handle : Rewriting_Handle
               ; F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Signed_Int_Type_Def,
                 (1 => F_Range));
         end;


         function Create_Anonymous_Type
           (Handle : Rewriting_Handle
               ; F_Type_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Anonymous_Type,
                 (1 => F_Type_Decl));
         end;


         function Create_Subtype_Indication
           (Handle : Rewriting_Handle
               ; F_Has_Not_Null : Node_Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Subtype_Indication,
                 (1 => F_Has_Not_Null,
                  2 => F_Name,
                  3 => F_Constraint));
         end;


         function Create_Constrained_Subtype_Indication
           (Handle : Rewriting_Handle
               ; F_Has_Not_Null : Node_Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Constrained_Subtype_Indication,
                 (1 => F_Has_Not_Null,
                  2 => F_Name,
                  3 => F_Constraint));
         end;


         function Create_Discrete_Subtype_Indication
           (Handle : Rewriting_Handle
               ; F_Has_Not_Null : Node_Rewriting_Handle
               ; F_Name : Node_Rewriting_Handle
               ; F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Discrete_Subtype_Indication,
                 (1 => F_Has_Not_Null,
                  2 => F_Name,
                  3 => F_Constraint));
         end;


         function Create_Unconstrained_Array_Index
           (Handle : Rewriting_Handle
               ; F_Subtype_Name : Node_Rewriting_Handle
               ; F_Lower_Bound : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Unconstrained_Array_Index,
                 (1 => F_Subtype_Name,
                  2 => F_Lower_Bound));
         end;


         function Create_Use_Package_Clause
           (Handle : Rewriting_Handle
               ; F_Packages : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Use_Package_Clause,
                 (1 => F_Packages));
         end;


         function Create_Use_Type_Clause
           (Handle : Rewriting_Handle
               ; F_Has_All : Node_Rewriting_Handle
               ; F_Types : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Use_Type_Clause,
                 (1 => F_Has_All,
                  2 => F_Types));
         end;


         function Create_Value_Sequence
           (Handle : Rewriting_Handle
               ; F_Iter_Assoc : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Value_Sequence,
                 (1 => F_Iter_Assoc));
         end;


         function Create_Variant
           (Handle : Rewriting_Handle
               ; F_Choices : Node_Rewriting_Handle
               ; F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Variant,
                 (1 => F_Choices,
                  2 => F_Components));
         end;


         function Create_Variant_Part
           (Handle : Rewriting_Handle
               ; F_Discr_Name : Node_Rewriting_Handle
               ; F_Variant : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_Variant_Part,
                 (1 => F_Discr_Name,
                  2 => F_Variant));
         end;


         function Create_With_Clause
           (Handle : Rewriting_Handle
               ; F_Has_Limited : Node_Rewriting_Handle
               ; F_Has_Private : Node_Rewriting_Handle
               ; F_Packages : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle is
         begin
            return Create_Regular_Node
              (Handle,
               Ada_With_Clause,
                 (1 => F_Has_Limited,
                  2 => F_Has_Private,
                  3 => F_Packages));
         end;


end Libadalang.Rewriting;
