%define OT_Evaluation_doc
"Numerical math evaluation implementation.

Available constructors:
    EvaluationImplementation()

See also
--------
Function, AggregatedEvaluation,
DatabaseEvaluation,
DualLinearCombinationEvaluation, LinearFunction
"
%enddef
%feature("docstring") OT::EvaluationImplementation
OT_Evaluation_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getCallsNumber_doc
"Accessor to the number of times the function has been called.

Returns
-------
calls_number : int
    Integer that counts the number of times the function has been called
    since its creation."
%enddef
%feature("docstring") OT::EvaluationImplementation::getCallsNumber
OT_Evaluation_getCallsNumber_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getDescription_doc
"Accessor to the description of the inputs and outputs.

Returns
-------
description : :class:`~openturns.Description`
    Description of the inputs and the outputs.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getDescription())
[x1,x2,y0]"
%enddef
%feature("docstring") OT::EvaluationImplementation::getDescription
OT_Evaluation_getDescription_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getInputDescription_doc
"Accessor to the description of the inputs.

Returns
-------
description : :class:`~openturns.Description`
    Description of the inputs.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getInputDescription())
[x1,x2]"
%enddef
%feature("docstring") OT::EvaluationImplementation::getInputDescription
OT_Evaluation_getInputDescription_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getInputDimension_doc
"Accessor to the number of the inputs.

Returns
-------
number_inputs : int
    Number of inputs.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getInputDimension())
2"
%enddef
%feature("docstring") OT::EvaluationImplementation::getInputDimension
OT_Evaluation_getInputDimension_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getMarginal_doc
"Accessor to marginal.

Parameters
----------
indices : int or sequence of ints
    Set of indices for which the marginal is extracted.

Returns
-------
marginal : :class:`~openturns.Function`
    Function corresponding to either :math:`f_i` or
    :math:`(f_i)_{i \\in indices}`, with :math:`f:\\Rset^n \\rightarrow \\Rset^p`
    and :math:`f=(f_0 , \\dots, f_{p-1})`."
%enddef
%feature("docstring") OT::EvaluationImplementation::getMarginal
OT_Evaluation_getMarginal_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getOutputDescription_doc
"Accessor to the description of the outputs.

Returns
-------
description : :class:`~openturns.Description`
    Description of the outputs.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getOutputDescription())
[y0]"
%enddef
%feature("docstring") OT::EvaluationImplementation::getOutputDescription
OT_Evaluation_getOutputDescription_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getOutputDimension_doc
"Accessor to the number of the outputs.

Returns
-------
number_outputs : int
    Number of outputs.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getOutputDimension())
1"
%enddef
%feature("docstring") OT::EvaluationImplementation::getOutputDimension
OT_Evaluation_getOutputDimension_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getParameterDimension_doc
"Accessor to the dimension of the parameter.

Returns
-------
parameter_dimension : int
    Dimension of the parameter."
%enddef
%feature("docstring") OT::EvaluationImplementation::getParameterDimension
OT_Evaluation_getParameterDimension_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_setDescription_doc
"Accessor to the description of the inputs and outputs.

Parameters
----------
description : sequence of str
    Description of the inputs and the outputs.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getDescription())
[x1,x2,y0]
>>> f.setDescription(['a','b','y'])
>>> print(f.getDescription())
[a,b,y]"
%enddef
%feature("docstring") OT::EvaluationImplementation::setDescription
OT_Evaluation_setDescription_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_setInputDescription_doc
"Accessor to the description of the inputs.

Returns
-------
description : :class:`~openturns.Description`
    Description of the inputs."
%enddef
%feature("docstring") OT::EvaluationImplementation::setInputDescription
OT_Evaluation_setInputDescription_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_setOutputDescription_doc
"Accessor to the description of the outputs.

Returns
-------
description : :class:`~openturns.Description`
    Description of the outputs."
%enddef
%feature("docstring") OT::EvaluationImplementation::setOutputDescription
OT_Evaluation_setOutputDescription_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_parameterGradient_doc
"Gradient against the parameters.

Parameters
----------
x : sequence of float
    Input point

Returns
-------
parameter_gradient : :class:`~openturns.Matrix`
    The parameters gradient computed at x."
%enddef
%feature("docstring") OT::EvaluationImplementation::parameterGradient
OT_Evaluation_parameterGradient_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getParameter_doc
"Accessor to the parameter values.

Returns
-------
parameter : :class:`~openturns.Point`
    The parameter values."
%enddef
%feature("docstring") OT::EvaluationImplementation::getParameter
OT_Evaluation_getParameter_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_setParameter_doc
"Accessor to the parameter values.

Parameters
----------
parameter : sequence of float
    The parameter values."
%enddef
%feature("docstring") OT::EvaluationImplementation::setParameter
OT_Evaluation_setParameter_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getParameterDescription_doc
"Accessor to the parameter description.

Returns
-------
parameter : :class:`~openturns.Description`
    The parameter description."
%enddef
%feature("docstring") OT::EvaluationImplementation::getParameterDescription
OT_Evaluation_getParameterDescription_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_setParameterDescription_doc
"Accessor to the parameter description.

Parameters
----------
parameter : :class:`~openturns.Description`
    The parameter description."
%enddef
%feature("docstring") OT::EvaluationImplementation::setParameterDescription
OT_Evaluation_setParameterDescription_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_draw_doc
"Draw the output of function as a :class:`~openturns.Graph`.

Available usages:
    draw(*inputMarg, outputMarg, CP, xiMin, xiMax, ptNb, scale*)

    draw(*firstInputMarg, secondInputMarg, outputMarg, CP, xiMin_xjMin, xiMax_xjMax, ptNbs, scale, isFilled*)

    draw(*xiMin, xiMax, ptNb, scale*)

    draw(*xiMin_xjMin, xiMax_xjMax, ptNbs, scale*)

Parameters
----------
outputMarg, inputMarg : int, :math:`outputMarg, inputMarg \\geq 0`
    *outputMarg* is the index of the marginal to draw as a function of the marginal
    with index *inputMarg*.
firstInputMarg, secondInputMarg : int, :math:`firstInputMarg, secondInputMarg \\geq 0`
    In the 2D case, the marginal *outputMarg* is drawn as a function of the
    two marginals with indexes *firstInputMarg* and *secondInputMarg*.
CP : sequence of float
    Central point.
xiMin, xiMax : float
    Define the interval where the curve is plotted.
xiMin_xjMin, xiMax_xjMax : sequence of float of dimension 2.
    In the 2D case, define the intervals where the curves are plotted.
ptNb : int :math:`ptNb > 0`
    The number of points to draw the curves.
ptNbs : sequence of int of dimension 2 :math:`ptNbs_k > 0, k=1,2`
    The number of points to draw the contour in the 2D case.
scale : bool
    scale indicates whether the logarithmic scale is used either for one or both axes:

    - ot.GraphImplementation.NONE or 0: no log scale is used,

    - ot.GraphImplementation.LOGX or 1: log scale is used only for horizontal data,

    - ot.GraphImplementation.LOGY or 2: log scale is used only for vertical data,

    - ot.GraphImplementation.LOGXY or 3: log scale is used for both data.
isFilled : bool
    isFilled indicates whether the contour graph is filled or not

Notes
-----
We note :math:`f: \\Rset^n \\rightarrow \\Rset^p`
where :math:`\\vect{x} = (x_1, \\dots, x_n)` and
:math:`f(\\vect{x}) = (f_1(\\vect{x}), \\dots,f_p(\\vect{x}))`,
with :math:`n\\geq 1` and :math:`p\\geq 1`.

- In the first usage:

Draws graph of the given 1D *outputMarg* marginal
:math:`f_k: \\Rset^n \\rightarrow \\Rset` as a function of the given 1D *inputMarg*
marginal with respect to the variation of :math:`x_i` in the interval
:math:`[x_i^{min}, x_i^{max}]`, when all the other components of
:math:`\\vect{x}` are fixed to the corresponding ones of the central point *CP*.
Then it draws the graph:
:math:`t\\in [x_i^{min}, x_i^{max}] \\mapsto f_k(CP_1, \\dots, CP_{i-1}, t,  CP_{i+1} \\dots, CP_n)`.

- In the second usage:

Draws the iso-curves of the given *outputMarg* marginal :math:`f_k` as a
function of the given 2D *firstInputMarg* and *secondInputMarg* marginals
with respect to the variation of :math:`(x_i, x_j)` in the interval
:math:`[x_i^{min}, x_i^{max}] \\times [x_j^{min}, x_j^{max}]`, when all the
other components of :math:`\\vect{x}` are fixed to the corresponding ones of the
central point *CP*. Then it draws the graph:
:math:`(t,u) \\in [x_i^{min}, x_i^{max}] \\times [x_j^{min}, x_j^{max}] \\mapsto f_k(CP_1, \\dots, CP_{i-1}, t, CP_{i+1}, \\dots, CP_{j-1}, u,  CP_{j+1} \\dots, CP_n)`.

- In the third usage:

The same as the first usage but only for function :math:`f: \\Rset \\rightarrow \\Rset`.

- In the fourth usage:

The same as the second usage but only for function :math:`f: \\Rset^2 \\rightarrow \\Rset`.


Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> f = ot.SymbolicFunction(['x'], ['sin(2*pi_*x)*exp(-x^2/2)'])
>>> graph = f.draw(-1.2, 1.2, 100)
>>> View(graph).show()"
%enddef
%feature("docstring") OT::EvaluationImplementation::draw
OT_Evaluation_draw_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_drawCrossCuts_doc
"Draw the 2D and 1D cross cuts of a 1D output function as a :class:`~openturns.GridLayout`.

Parameters
----------
centralPoint : sequence of float
    Central point with dimension equal to the input dimension of the function.
xMin, xMax : sequence of float
    Define the interval where the curve is plotted.
pointNumber : sequence of int
    The number of points to draw the contours and the curves.
withMonoDimensionalCuts : bool, optional
    withMonoDimensionalCuts indicates whether the mono dimension cuts are drawn or not
    Default value is specified in the `CrossCuts-DefaultWithMonoDimensionalCuts` ResourceMap key.
isFilled : bool, optional
    isFilled indicates whether the contour graphs are filled or not
    Default value is specified in the `Contour-DefaultIsFilled` ResourceMap key
vMin, vMax : float, optional
    Define the interval used to build the color map for the contours
    If not specified, these values are computed to best fit the graphs.
    Either specify both values or do not specify any.

Notes
-----
We note :math:`f: \\Rset^n \\rightarrow \\Rset^p`
where :math:`\\vect{x} = (x_1, \\dots, x_n)` and
:math:`f(\\vect{x}) = (f_1(\\vect{x}), \\dots, f_p(\\vect{x}))`,
with :math:`n\\geq 1` and :math:`p\\geq 1`.

In all usages, draw the 1D and 2D cross cuts of
:math:`f_k: \\Rset^n \\rightarrow \\Rset` as a function of all input coordinates for 1D cuts
and all couples of coordinates for 2D cuts. Variable coordinates :math:`x_i` are sampled
regularly using :math:`ptNb[i]` points in the interval :math:`[x_i^{min}, x_i^{max}]`,
when all the other components of :math:`\\vect{x}` are fixed to the corresponding components
of the *centralPoint* :math:`\\vect{c}`. In the first usage, vMin and vMax are evaluated as
the min and max of all samples of the function value calculated in all cross cuts performed.

- For 1D cross cuts:

OpenTURNS draws the graph:

.. math::
    y = f_k^{(i)}(s)

for any :math:`s \\in [x_i^{min}, x_i^{max}]` where :math:`f_k^{(i)}(s)` is defined by the equation:

.. math::
    f_k^{(i)}(s) = f_k(c_1, \\dots, c_{i-1}, s,  c_{i+1} \\dots, c_n).

- For 2D cross cuts:

.. math::
    y = f_k^{(i,j)}(s, t)

for any :math:`(s, t) \\in [x_i^{min}, x_i^{max}] \\times [x_j^{min}, x_j^{max}]` where :math:`f_k^{(i,j)}` 
is defined by the equation:

.. math::
    f_k^{(i,j)}(s,t) = f_k(c_1, \\dots, c_{i-1}, s, c_{i+1}, \\dots, c_{j-1}, t,  c_{j+1} \\dots, c_n).

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> f = ot.SymbolicFunction(['x0', 'x1', 'x2'], ['sin(1*pi_*x0) + x1 - x2 ^ 2'])
>>> grid = f.drawCrossCuts([0., 0., 0.], [-3., -3, -3], [3, 3, 3], [100, 20, 20], True, True)
>>> View(grid).show()"
%enddef
%feature("docstring") OT::EvaluationImplementation::drawCrossCuts
OT_Evaluation_drawCrossCuts_doc

// ---------------------------------------------------------------------

%feature("docstring") OT::EvaluationImplementation::isActualImplementation
"Accessor to the validity flag.

Returns
-------
is_impl : bool
    Whether the implementation is valid."

// ---------------------------------------------------------------------

%define OT_Evaluation_isLinear_doc
"Accessor to the linearity of the evaluation.

Returns
-------
linear : bool
    *True* if the evaluation is linear, *False* otherwise."
%enddef
%feature("docstring") OT::EvaluationImplementation::isLinear
OT_Evaluation_isLinear_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_isLinearlyDependent_doc
"Accessor to the linearity of the evaluation with regard to a specific variable.

Parameters
----------
index : int
    The index of the variable with regard to which linearity is evaluated.

Returns
-------
linear : bool
    *True* if the evaluation is linearly dependent on the specified variable, *False* otherwise."
%enddef
%feature("docstring") OT::EvaluationImplementation::isLinearlyDependent
OT_Evaluation_isLinearlyDependent_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_setCheckOutput_doc
"Accessor to the output verification flag.

Parameters
----------
check_output : bool
    Whether to check return values for nan or inf."
%enddef
%feature("docstring") OT::EvaluationImplementation::setCheckOutput
OT_Evaluation_setCheckOutput_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_getCheckOutput_doc
"Accessor to the output verification flag.

Returns
-------
check_output : bool
    Whether to check return values for nan or inf."
%enddef
%feature("docstring") OT::EvaluationImplementation::getCheckOutput
OT_Evaluation_getCheckOutput_doc

// ---------------------------------------------------------------------

%define OT_Evaluation_setStopCallback_doc
"Set up a stop callback.

Can be used to programmatically stop an evaluation.

Parameters
----------
callback : callable
    Returns a bool deciding whether to stop or continue.
"
%enddef
%feature("docstring") OT::EvaluationImplementation::setStopCallback
OT_Evaluation_setStopCallback_doc
