/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { useState, useCallback, useEffect } from 'react';
import { AuthType } from '@google/gemini-cli-core';
import { clearCachedCredentialFile, getErrorMessage, } from '@google/gemini-cli-core';
import { runExitCleanup } from '../../utils/cleanup.js';
export const useAuthCommand = (settings, setAuthError, config) => {
    const [isAuthDialogOpen, setIsAuthDialogOpen] = useState(settings.merged.security?.auth?.selectedType === undefined);
    const openAuthDialog = useCallback(() => {
        setIsAuthDialogOpen(true);
    }, []);
    const [isAuthenticating, setIsAuthenticating] = useState(false);
    useEffect(() => {
        const authFlow = async () => {
            const authType = settings.merged.security?.auth?.selectedType;
            if (isAuthDialogOpen || !authType) {
                return;
            }
            try {
                setIsAuthenticating(true);
                await config.refreshAuth(authType);
                console.log(`Authenticated via "${authType}".`);
            }
            catch (e) {
                setAuthError(`Failed to login. Message: ${getErrorMessage(e)}`);
                openAuthDialog();
            }
            finally {
                setIsAuthenticating(false);
            }
        };
        void authFlow();
    }, [isAuthDialogOpen, settings, config, setAuthError, openAuthDialog]);
    const handleAuthSelect = useCallback(async (authType, scope) => {
        if (authType) {
            await clearCachedCredentialFile();
            settings.setValue(scope, 'security.auth.selectedType', authType);
            if (authType === AuthType.LOGIN_WITH_GOOGLE &&
                config.isBrowserLaunchSuppressed()) {
                runExitCleanup();
                console.log(`
----------------------------------------------------------------
Logging in with Google... Please restart Gemini CLI to continue.
----------------------------------------------------------------
            `);
                process.exit(0);
            }
        }
        setIsAuthDialogOpen(false);
        setAuthError(null);
    }, [settings, setAuthError, config]);
    const cancelAuthentication = useCallback(() => {
        setIsAuthenticating(false);
    }, []);
    return {
        isAuthDialogOpen,
        openAuthDialog,
        handleAuthSelect,
        isAuthenticating,
        cancelAuthentication,
    };
};
//# sourceMappingURL=useAuthCommand.js.map