import json
import os
import shutil
import tarfile
from io import BytesIO

from conan.api.model import PackagesList
from conan.api.output import ConanOutput
from conan.internal.conan_app import ConanApp
from conan.internal.integrity_check import IntegrityChecker
from conans.client.cache.cache import ClientCache
from conans.errors import ConanException
from conans.model.package_ref import PkgReference
from conans.model.recipe_ref import RecipeReference
from conans.util.dates import revision_timestamp_now
from conans.util.files import rmdir, gzopen_without_timestamps, mkdir


class CacheAPI:

    def __init__(self, conan_api):
        self.conan_api = conan_api

    def export_path(self, ref: RecipeReference):
        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        ref.revision = None if ref.revision == "latest" else ref.revision
        ref_layout = app.cache.recipe_layout(ref)
        return _check_folder_existence(ref, "export", ref_layout.export())

    def recipe_metadata_path(self, ref: RecipeReference):
        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        ref = _resolve_latest_ref(app, ref)
        ref_layout = app.cache.recipe_layout(ref)
        return _check_folder_existence(ref, "metadata", ref_layout.metadata())

    def export_source_path(self, ref: RecipeReference):
        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        ref.revision = None if ref.revision == "latest" else ref.revision
        ref_layout = app.cache.recipe_layout(ref)
        return _check_folder_existence(ref, "export_sources", ref_layout.export_sources())

    def source_path(self, ref: RecipeReference):
        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        ref.revision = None if ref.revision == "latest" else ref.revision
        ref_layout = app.cache.recipe_layout(ref)
        return _check_folder_existence(ref, "source", ref_layout.source())

    def build_path(self, pref: PkgReference):
        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        pref = _resolve_latest_pref(app, pref)
        ref_layout = app.cache.pkg_layout(pref)
        return _check_folder_existence(pref, "build", ref_layout.build())

    def package_metadata_path(self, pref: PkgReference):
        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        pref = _resolve_latest_pref(app, pref)
        ref_layout = app.cache.pkg_layout(pref)
        return _check_folder_existence(pref, "metadata", ref_layout.metadata())

    def package_path(self, pref: PkgReference):
        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        pref = _resolve_latest_pref(app, pref)
        ref_layout = app.cache.pkg_layout(pref)
        return _check_folder_existence(pref, "package", ref_layout.package())

    def check_integrity(self, package_list):
        """Check if the recipes and packages are corrupted (it will raise a ConanExcepcion)"""
        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        checker = IntegrityChecker(app)
        checker.check(package_list)

    def clean(self, package_list, source=True, build=True, download=True, temp=True):
        """
        Remove non critical folders from the cache, like source, build and download (.tgz store)
        folders.
        :param package_list: the package lists that should be cleaned
        :param source: boolean, remove the "source" folder if True
        :param build: boolean, remove the "build" folder if True
        :param download: boolen, remove the "download (.tgz)" folder if True
        :param temp: boolean, remove the temporary folders
        :return:
        """

        app = ConanApp(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        if temp:
            rmdir(app.cache.temp_folder)
            # Clean those build folders that didn't succeed to create a package and wont be in DB
            builds_folder = app.cache.builds_folder
            if os.path.isdir(builds_folder):
                for subdir in os.listdir(builds_folder):
                    folder = os.path.join(builds_folder, subdir)
                    manifest = os.path.join(folder, "p", "conanmanifest.txt")
                    info = os.path.join(folder, "p", "conaninfo.txt")
                    if not os.path.exists(manifest) or not os.path.exists(info):
                        rmdir(folder)

        for ref, ref_bundle in package_list.refs().items():
            ref_layout = app.cache.recipe_layout(ref)
            if source:
                rmdir(ref_layout.source())
            if download:
                rmdir(ref_layout.download_export())
            for pref, _ in package_list.prefs(ref, ref_bundle).items():
                pref_layout = app.cache.pkg_layout(pref)
                if build:
                    rmdir(pref_layout.build())
                    # It is important to remove the "build_id" identifier if build-folder is removed
                    app.cache.remove_build_id(pref)
                if download:
                    rmdir(pref_layout.download_package())

    def save(self, package_list, tgz_path):
        cache_folder = self.conan_api.cache_folder
        app = ConanApp(cache_folder, self.conan_api.config.global_conf)
        out = ConanOutput()
        mkdir(os.path.dirname(tgz_path))
        name = os.path.basename(tgz_path)
        with open(tgz_path, "wb") as tgz_handle:
            tgz = gzopen_without_timestamps(name, mode="w", fileobj=tgz_handle)
            for ref, ref_bundle in package_list.refs().items():
                ref_layout = app.cache.recipe_layout(ref)
                recipe_folder = os.path.relpath(ref_layout.base_folder, cache_folder)
                recipe_folder = recipe_folder.replace("\\", "/")  # make win paths portable
                ref_bundle["recipe_folder"] = recipe_folder
                out.info(f"Saving {ref}: {recipe_folder}")
                tgz.add(os.path.join(cache_folder, recipe_folder), recipe_folder, recursive=True)
                for pref, pref_bundle in package_list.prefs(ref, ref_bundle).items():
                    pref_layout = app.cache.pkg_layout(pref)
                    pkg_folder = pref_layout.package()
                    folder = os.path.relpath(pkg_folder, cache_folder)
                    folder = folder.replace("\\", "/")  # make win paths portable
                    pref_bundle["package_folder"] = folder
                    out.info(f"Saving {pref}: {folder}")
                    tgz.add(os.path.join(cache_folder, folder), folder, recursive=True)
                    if os.path.exists(pref_layout.metadata()):
                        metadata_folder = os.path.relpath(pref_layout.metadata(), cache_folder)
                        metadata_folder = metadata_folder.replace("\\", "/")  # make paths portable
                        pref_bundle["metadata_folder"] = metadata_folder
                        out.info(f"Saving {pref} metadata: {metadata_folder}")
                        tgz.add(os.path.join(cache_folder, metadata_folder), metadata_folder,
                                recursive=True)
            serialized = json.dumps(package_list.serialize(), indent=2)
            info = tarfile.TarInfo(name="pkglist.json")
            data = serialized.encode('utf-8')
            info.size = len(data)
            tgz.addfile(tarinfo=info, fileobj=BytesIO(data))
            tgz.close()

    def restore(self, path):
        with open(path, mode='rb') as file_handler:
            the_tar = tarfile.open(fileobj=file_handler)
            fileobj = the_tar.extractfile("pkglist.json")
            pkglist = fileobj.read()
            the_tar.extractall(path=self.conan_api.cache_folder)
            the_tar.close()

        out = ConanOutput()
        package_list = PackagesList.deserialize(json.loads(pkglist))
        cache = ClientCache(self.conan_api.cache_folder, self.conan_api.config.global_conf)
        for ref, ref_bundle in package_list.refs().items():
            ref.timestamp = revision_timestamp_now()
            ref_bundle["timestamp"] = ref.timestamp
            recipe_layout = cache.get_or_create_ref_layout(ref)
            recipe_folder = ref_bundle["recipe_folder"]
            rel_path = os.path.relpath(recipe_layout.base_folder, cache.cache_folder)
            rel_path = rel_path.replace("\\", "/")
            assert rel_path == recipe_folder, f"{rel_path}!={recipe_folder}"
            out.info(f"Restore: {ref} in {recipe_folder}")
            for pref, pref_bundle in package_list.prefs(ref, ref_bundle).items():
                pref.timestamp = revision_timestamp_now()
                pref_bundle["timestamp"] = pref.timestamp
                pkg_layout = cache.get_or_create_pkg_layout(pref)
                pkg_folder = pref_bundle["package_folder"]
                out.info(f"Restore: {pref} in {pkg_folder}")
                # We need to put the package in the final location in the cache
                shutil.move(os.path.join(cache.cache_folder, pkg_folder), pkg_layout.package())
                metadata_folder = pref_bundle.get("metadata_folder")
                if metadata_folder:
                    out.info(f"Restore: {pref} metadata in {metadata_folder}")
                    # We need to put the package in the final location in the cache
                    shutil.move(os.path.join(cache.cache_folder, metadata_folder),
                                pkg_layout.metadata())

        return package_list


def _resolve_latest_ref(app, ref):
    if ref.revision is None or ref.revision == "latest":
        ref.revision = None
        result = app.cache.get_latest_recipe_reference(ref)
        if result is None:
            raise ConanException(f"'{ref}' not found in cache")
        ref = result
    return ref


def _resolve_latest_pref(app, pref):
    pref.ref = _resolve_latest_ref(app, pref.ref)
    if pref.revision is None or pref.revision == "latest":
        pref.revision = None
        result = app.cache.get_latest_package_reference(pref)
        if result is None:
            raise ConanException(f"'{pref.repr_notime()}' not found in cache")
        pref = result
    return pref


def _check_folder_existence(ref, folder_name, folder_path):
    if not os.path.exists(folder_path):
        raise ConanException(f"'{folder_name}' folder does not exist for the reference {ref}")
    return folder_path
