static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2011 Michael Muré <batolettre@gmail.com>                        \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"enum_start (gegl_warp_behavior)                                               \n"
"  enum_value (GEGL_WARP_BEHAVIOR_MOVE,      \"move\",      N_(\"Move pixels\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_GROW,      \"grow\",      N_(\"Grow area\")) \n"
"  enum_value (GEGL_WARP_BEHAVIOR_SHRINK,    \"shrink\",    N_(\"Shrink area\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_SWIRL_CW,  \"swirl-cw\",  N_(\"Swirl clockwise\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_SWIRL_CCW, \"swirl-ccw\", N_(\"Swirl counter-clockwise\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_ERASE,     \"erase\",     N_(\"Erase warping\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_SMOOTH,    \"smooth\",    N_(\"Smooth warping\"))\n"
"enum_end (GeglWarpBehavior)                                                   \n"
"                                                                              \n"
"property_double (strength, _(\"Strength\"), 50)                               \n"
"  value_range (0, 100)                                                        \n"
"                                                                              \n"
"property_double (size, _(\"Size\"), 40.0)                                     \n"
"  value_range (1.0, 10000.0)                                                  \n"
"                                                                              \n"
"property_double (hardness, _(\"Hardness\"), 0.5)                              \n"
"  value_range (0.0, 1.0)                                                      \n"
"                                                                              \n"
"property_path   (stroke, _(\"Stroke\"), NULL)                                 \n"
"                                                                              \n"
"property_enum   (behavior, _(\"Behavior\"),                                   \n"
"                   GeglWarpBehavior, gegl_warp_behavior,                      \n"
"                   GEGL_WARP_BEHAVIOR_MOVE)                                   \n"
"  description   (_(\"Behavior of the op\"))                                   \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_C_SOURCE warp.c                                               \n"
"#define GEGL_OP_NAME     warp                                                 \n"
"                                                                              \n"
"#include \"gegl-plugin.h\"                                                    \n"
"#include \"gegl-path.h\"                                                      \n"
"                                                                              \n"
"static void path_changed (GeglPath            *path,                          \n"
"                          const GeglRectangle *roi,                           \n"
"                          gpointer             userdata);                     \n"
"#include \"gegl-op.h\"                                                        \n"
"                                                                              \n"
"typedef struct {                                                              \n"
"  gdouble     *lookup;                                                        \n"
"  GeglBuffer  *buffer;                                                        \n"
"  gdouble      last_x;                                                        \n"
"  gdouble      last_y;                                                        \n"
"  gboolean     last_point_set;                                                \n"
"} WarpPrivate;                                                                \n"
"                                                                              \n"
"static void                                                                   \n"
"path_changed (GeglPath            *path,                                      \n"
"              const GeglRectangle *roi,                                       \n"
"              gpointer             userdata)                                  \n"
"{                                                                             \n"
"  GeglRectangle   rect = *roi;                                                \n"
"  GeglProperties *o    = GEGL_PROPERTIES (userdata);                          \n"
"  /* invalidate the incoming rectangle */                                     \n"
"                                                                              \n"
"  rect.x -= o->size/2;                                                        \n"
"  rect.y -= o->size/2;                                                        \n"
"  rect.width += o->size;                                                      \n"
"  rect.height += o->size;                                                     \n"
"                                                                              \n"
"  gegl_operation_invalidate (userdata, &rect, FALSE);                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  GeglProperties *o     = GEGL_PROPERTIES (operation);                        \n"
"  WarpPrivate    *priv;                                                       \n"
"                                                                              \n"
"  const Babl *format = babl_format_n (babl_type (\"float\"), 2);              \n"
"  gegl_operation_set_format (operation, \"input\", format);                   \n"
"  gegl_operation_set_format (operation, \"output\", format);                  \n"
"                                                                              \n"
"  if (!o->user_data)                                                          \n"
"    {                                                                         \n"
"      o->user_data = g_slice_new (WarpPrivate);                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  priv = (WarpPrivate*) o->user_data;                                         \n"
"  priv->last_point_set = FALSE;                                               \n"
"  priv->lookup = NULL;                                                        \n"
"  priv->buffer = NULL;                                                        \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"finalize (GObject *object)                                                    \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (object);                               \n"
"                                                                              \n"
"  if (o->user_data)                                                           \n"
"    {                                                                         \n"
"      g_slice_free (WarpPrivate, o->user_data);                               \n"
"      o->user_data = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  G_OBJECT_CLASS (gegl_op_parent_class)->finalize (object);                   \n"
"}                                                                             \n"
"                                                                              \n"
"static gdouble                                                                \n"
"gauss (gdouble f)                                                             \n"
"{                                                                             \n"
"  /* This is not a real gauss function. */                                    \n"
"  /* Approximation is valid if -1 < f < 1 */                                  \n"
"  if (f < -1.0)                                                               \n"
"    return 0.0;                                                               \n"
"                                                                              \n"
"  if (f < -0.5)                                                               \n"
"    {                                                                         \n"
"      f = -1.0 - f;                                                           \n"
"      return (2.0 * f*f);                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (f < 0.5)                                                                \n"
"    return (1.0 - 2.0 * f*f);                                                 \n"
"                                                                              \n"
"  if (f < 1.0)                                                                \n"
"    {                                                                         \n"
"      f = 1.0 - f;                                                            \n"
"      return (2.0 * f*f);                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return 0.0;                                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"/* set up lookup table */                                                     \n"
"static void                                                                   \n"
"calc_lut (GeglProperties  *o)                                                 \n"
"{                                                                             \n"
"  WarpPrivate  *priv = (WarpPrivate*) o->user_data;                           \n"
"  gint          length;                                                       \n"
"  gint          x;                                                            \n"
"  gdouble       exponent;                                                     \n"
"                                                                              \n"
"  length = (gint)(0.5 * o->size + 1.0) + 2;                                   \n"
"                                                                              \n"
"  priv->lookup = g_malloc (length * sizeof (gdouble));                        \n"
"                                                                              \n"
"  if ((1.0 - o->hardness) < 0.0000004)                                        \n"
"    exponent = 1000000.0;                                                     \n"
"  else                                                                        \n"
"    exponent = 0.4 / (1.0 - o->hardness);                                     \n"
"                                                                              \n"
"  for (x = 0; x < length; x++)                                                \n"
"    {                                                                         \n"
"      priv->lookup[x] = gauss (pow (2.0 * x / o->size, exponent));            \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static gdouble                                                                \n"
"get_stamp_force (GeglProperties *o,                                           \n"
"                 gdouble         x,                                           \n"
"                 gdouble         y)                                           \n"
"{                                                                             \n"
"  WarpPrivate  *priv = (WarpPrivate*) o->user_data;                           \n"
"  gfloat        radius;                                                       \n"
"                                                                              \n"
"  if (!priv->lookup)                                                          \n"
"    {                                                                         \n"
"      calc_lut (o);                                                           \n"
"    }                                                                         \n"
"                                                                              \n"
"  radius = hypot (x, y);                                                      \n"
"                                                                              \n"
"  if (radius < 0.5 * o->size + 1)                                             \n"
"    {                                                                         \n"
"      /* linear interpolation */                                              \n"
"      gint a;                                                                 \n"
"      gdouble ratio;                                                          \n"
"      gdouble before, after;                                                  \n"
"                                                                              \n"
"      a = (gint)(radius);                                                     \n"
"      ratio = (radius - a);                                                   \n"
"                                                                              \n"
"      before = priv->lookup[a];                                               \n"
"      after = priv->lookup[a + 1];                                            \n"
"                                                                              \n"
"      return (1.0 - ratio) * before + ratio * after;                          \n"
"    }                                                                         \n"
"                                                                              \n"
"  return 0.0;                                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"stamp (GeglProperties          *o,                                            \n"
"       const GeglRectangle     *result,                                       \n"
"       gdouble                  x,                                            \n"
"       gdouble                  y)                                            \n"
"{                                                                             \n"
"  WarpPrivate         *priv = (WarpPrivate*) o->user_data;                    \n"
"  GeglBufferIterator  *it;                                                    \n"
"  const Babl          *format;                                                \n"
"  gdouble              stamp_force, influence;                                \n"
"  gdouble              x_mean = 0.0;                                          \n"
"  gdouble              y_mean = 0.0;                                          \n"
"  gint                 x_iter, y_iter;                                        \n"
"  GeglRectangle        area;                                                  \n"
"  const GeglRectangle *src_extent;                                            \n"
"  gfloat              *srcbuf, *stampbuf;                                     \n"
"  gint                 buf_rowstride = 0;                                     \n"
"  gfloat               s = 0, c = 0;                                          \n"
"                                                                              \n"
"  area.x = floor (x - o->size / 2.0);                                         \n"
"  area.y = floor (y - o->size / 2.0);                                         \n"
"  area.width   = ceil (x + o->size / 2.0);                                    \n"
"  area.height  = ceil (y + o->size / 2.0);                                    \n"
"  area.width  -= area.x;                                                      \n"
"  area.height -= area.y;                                                      \n"
"                                                                              \n"
"  /* first point of the stroke */                                             \n"
"  if (!priv->last_point_set)                                                  \n"
"    {                                                                         \n"
"      priv->last_x = x;                                                       \n"
"      priv->last_y = y;                                                       \n"
"      priv->last_point_set = TRUE;                                            \n"
"      return;                                                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* don't stamp if outside the roi treated */                                \n"
"  if (!gegl_rectangle_intersect (NULL, result, &area))                        \n"
"    return;                                                                   \n"
"                                                                              \n"
"  format = babl_format_n (babl_type (\"float\"), 2);                          \n"
"                                                                              \n"
"  /* If needed, compute the mean deformation */                               \n"
"  if (o->behavior == GEGL_WARP_BEHAVIOR_SMOOTH)                               \n"
"    {                                                                         \n"
"      gint pixel_count = 0;                                                   \n"
"                                                                              \n"
"      it = gegl_buffer_iterator_new (priv->buffer, &area, 0, format,          \n"
"                                     GEGL_ACCESS_READ, GEGL_ABYSS_NONE);      \n"
"                                                                              \n"
"      while (gegl_buffer_iterator_next (it))                                  \n"
"        {                                                                     \n"
"          gint    n_pixels    = it->length;                                   \n"
"          gfloat *coords      = it->data[0];                                  \n"
"                                                                              \n"
"          while (n_pixels--)                                                  \n"
"            {                                                                 \n"
"              x_mean += coords[0];                                            \n"
"              y_mean += coords[1];                                            \n"
"              coords += 2;                                                    \n"
"            }                                                                 \n"
"          pixel_count += it->roi->width * it->roi->height;                    \n"
"        }                                                                     \n"
"      x_mean /= pixel_count;                                                  \n"
"      y_mean /= pixel_count;                                                  \n"
"    }                                                                         \n"
"  else if (o->behavior == GEGL_WARP_BEHAVIOR_SWIRL_CW ||                      \n"
"           o->behavior == GEGL_WARP_BEHAVIOR_SWIRL_CCW)                       \n"
"    {                                                                         \n"
"      /* swirl by 5 degrees per stamp (for strength 100).                     \n"
"       * not exactly sin/cos factors,                                         \n"
"       * since we calculate an off-center offset-vector */                    \n"
"                                                                              \n"
"      /* note that this is fudged for stamp_force < 1.0 and                   \n"
"       * results in a slight upscaling there. It is a compromise              \n"
"       * between exactness and calculation speed. */                          \n"
"      s = sin (0.01 * o->strength * 5.0 / 180 * G_PI);                        \n"
"      c = cos (0.01 * o->strength * 5.0 / 180 * G_PI) - 1.0;                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  srcbuf = gegl_buffer_linear_open (priv->buffer, NULL, &buf_rowstride, NULL);\n"
"  buf_rowstride /= sizeof (gfloat);                                           \n"
"  src_extent = gegl_buffer_get_extent (priv->buffer);                         \n"
"                                                                              \n"
"  stampbuf = g_new0 (gfloat, 2 * area.height * area.width);                   \n"
"                                                                              \n"
"  for (y_iter = 0; y_iter < area.height; y_iter++)                            \n"
"    {                                                                         \n"
"      for (x_iter = 0; x_iter < area.width; x_iter++)                         \n"
"        {                                                                     \n"
"          gfloat nvx, nvy;                                                    \n"
"          gfloat xi, yi;                                                      \n"
"          gfloat *vals;                                                       \n"
"          gint dx, dy;                                                        \n"
"          gfloat weight_x, weight_y;                                          \n"
"          gfloat *srcptr;                                                     \n"
"                                                                              \n"
"          xi = area.x + x_iter;                                               \n"
"          xi += -x + 0.5;                                                     \n"
"          yi = area.y + y_iter;                                               \n"
"          yi += -y + 0.5;                                                     \n"
"                                                                              \n"
"          stamp_force = get_stamp_force (o, xi, yi);                          \n"
"          influence = 0.01 * o->strength * stamp_force;                       \n"
"                                                                              \n"
"          switch (o->behavior)                                                \n"
"            {                                                                 \n"
"              case GEGL_WARP_BEHAVIOR_MOVE:                                   \n"
"                nvx = influence * (priv->last_x - x);                         \n"
"                nvy = influence * (priv->last_y - y);                         \n"
"                break;                                                        \n"
"              case GEGL_WARP_BEHAVIOR_GROW:                                   \n"
"                nvx = influence * -0.1 * xi;                                  \n"
"                nvy = influence * -0.1 * yi;                                  \n"
"                break;                                                        \n"
"              case GEGL_WARP_BEHAVIOR_SHRINK:                                 \n"
"                nvx = influence * 0.1 * xi;                                   \n"
"                nvy = influence * 0.1 * yi;                                   \n"
"                break;                                                        \n"
"              case GEGL_WARP_BEHAVIOR_SWIRL_CW:                               \n"
"                nvx = stamp_force * ( c * xi + s * yi);                       \n"
"                nvy = stamp_force * (-s * xi + c * yi);                       \n"
"                break;                                                        \n"
"              case GEGL_WARP_BEHAVIOR_SWIRL_CCW:                              \n"
"                nvx = stamp_force * ( c * xi - s * yi);                       \n"
"                nvy = stamp_force * ( s * xi + c * yi);                       \n"
"                break;                                                        \n"
"              case GEGL_WARP_BEHAVIOR_ERASE:                                  \n"
"              case GEGL_WARP_BEHAVIOR_SMOOTH:                                 \n"
"              default:                                                        \n"
"                nvx = 0.0;                                                    \n"
"                nvy = 0.0;                                                    \n"
"                break;                                                        \n"
"            }                                                                 \n"
"                                                                              \n"
"          vals = stampbuf + (y_iter * area.width + x_iter) * 2;               \n"
"                                                                              \n"
"          dx = floorf (nvx);                                                  \n"
"          dy = floorf (nvy);                                                  \n"
"                                                                              \n"
"          if (area.x + x_iter + dx     <  src_extent->x                     ||\n"
"              area.x + x_iter + dx + 1 >= src_extent->x + src_extent->width ||\n"
"              area.y + y_iter + dy     <  src_extent->y                     ||\n"
"              area.y + y_iter + dy + 1 >= src_extent->y + src_extent->height) \n"
"            {                                                                 \n"
"              continue;                                                       \n"
"            }                                                                 \n"
"                                                                              \n"
"          srcptr = srcbuf + (area.y - src_extent->y + y_iter + dy) * buf_rowstride +\n"
"                            (area.x - src_extent->x + x_iter + dx) * 2;       \n"
"                                                                              \n"
"          if (o->behavior == GEGL_WARP_BEHAVIOR_ERASE)                        \n"
"            {                                                                 \n"
"              vals[0] = srcptr[0] * (1.0 - MIN (influence, 1.0));             \n"
"              vals[1] = srcptr[1] * (1.0 - MIN (influence, 1.0));             \n"
"            }                                                                 \n"
"          else if (o->behavior == GEGL_WARP_BEHAVIOR_SMOOTH)                  \n"
"            {                                                                 \n"
"              vals[0] = (1.0 - influence) * srcptr[0] + influence * x_mean;   \n"
"              vals[1] = (1.0 - influence) * srcptr[1] + influence * y_mean;   \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              weight_x = nvx - dx;                                            \n"
"              weight_y = nvy - dy;                                            \n"
"                                                                              \n"
"              /* bilinear interpolation of the vectors */                     \n"
"                                                                              \n"
"              vals[0]  = srcptr[0] * (1.0 - weight_x) * (1.0 - weight_y);     \n"
"              vals[1]  = srcptr[1] * (1.0 - weight_x) * (1.0 - weight_y);     \n"
"                                                                              \n"
"              vals[0] += srcptr[2] * weight_x * (1.0 - weight_y);             \n"
"              vals[1] += srcptr[3] * weight_x * (1.0 - weight_y);             \n"
"                                                                              \n"
"              vals[0] += srcptr[buf_rowstride + 0] * (1.0 - weight_x) * weight_y;\n"
"              vals[1] += srcptr[buf_rowstride + 1] * (1.0 - weight_x) * weight_y;\n"
"                                                                              \n"
"              vals[0] += srcptr[buf_rowstride + 2] * weight_x * weight_y;     \n"
"              vals[1] += srcptr[buf_rowstride + 3] * weight_x * weight_y;     \n"
"                                                                              \n"
"              vals[0] += nvx;                                                 \n"
"              vals[1] += nvy;                                                 \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_buffer_linear_close (priv->buffer, srcbuf);                            \n"
"  gegl_buffer_set (priv->buffer, &area, 0, format,                            \n"
"                   stampbuf, GEGL_AUTO_ROWSTRIDE);                            \n"
"  g_free (stampbuf);                                                          \n"
"                                                                              \n"
"  /* Memorize the stamp location for movement dependant behavior like move */ \n"
"  priv->last_x = x;                                                           \n"
"  priv->last_y = y;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *result,                                         \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglProperties      *o    = GEGL_PROPERTIES (operation);                    \n"
"  WarpPrivate         *priv = (WarpPrivate*) o->user_data;                    \n"
"  gdouble              dist;                                                  \n"
"  gdouble              stamps;                                                \n"
"  gdouble              spacing = MAX (o->size * 0.01, 0.5); /*1% spacing for starters*/\n"
"                                                                              \n"
"  GeglPathPoint        prev, next, lerp;                                      \n"
"  gulong               i;                                                     \n"
"  GeglPathList        *event;                                                 \n"
"                                                                              \n"
"  priv->buffer = gegl_buffer_dup (input);                                     \n"
"                                                                              \n"
"  event = gegl_path_get_path (o->stroke);                                     \n"
"                                                                              \n"
"  prev = *(event->d.point);                                                   \n"
"                                                                              \n"
"  while (event->next)                                                         \n"
"    {                                                                         \n"
"      event = event->next;                                                    \n"
"      next = *(event->d.point);                                               \n"
"      dist = gegl_path_point_dist (&next, &prev);                             \n"
"      stamps = dist / spacing;                                                \n"
"                                                                              \n"
"      if (stamps < 1)                                                         \n"
"        {                                                                     \n"
"          stamp (o, result, next.x, next.y);                                  \n"
"          prev = next;                                                        \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          for (i = 0; i < stamps; i++)                                        \n"
"            {                                                                 \n"
"              gegl_path_point_lerp (&lerp, &prev, &next, (i * spacing) / dist);\n"
"              stamp (o, result, lerp.x, lerp.y);                              \n"
"            }                                                                 \n"
"          prev = lerp;                                                        \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Affect the output buffer */                                              \n"
"  gegl_buffer_copy (priv->buffer, result, GEGL_ABYSS_NONE,                    \n"
"                    output, result);                                          \n"
"  gegl_buffer_set_extent (output, gegl_buffer_get_extent (input));            \n"
"  g_object_unref (priv->buffer);                                              \n"
"                                                                              \n"
"  /* prepare for the recomputing of the op */                                 \n"
"  priv->last_point_set = FALSE;                                               \n"
"                                                                              \n"
"  /* free the LUT */                                                          \n"
"  if (priv->lookup)                                                           \n"
"    {                                                                         \n"
"      g_free (priv->lookup);                                                  \n"
"      priv->lookup = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GObjectClass               *object_class    = G_OBJECT_CLASS (klass);       \n"
"  GeglOperationClass         *operation_class = GEGL_OPERATION_CLASS (klass); \n"
"  GeglOperationFilterClass   *filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);\n"
"                                                                              \n"
"  object_class->finalize   = finalize;                                        \n"
"  operation_class->prepare = prepare;                                         \n"
"  filter_class->process    = process;                                         \n"
"  operation_class->threaded = FALSE;                                          \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",               \"gegl:warp\",                                    \n"
"    \"categories\",         \"transform\",                                    \n"
"    \"title\",              _(\"Warp\"),                                      \n"
"    \"position-dependent\", \"true\",                                         \n"
"    \"description\", _(\"Compute a relative displacement mapping from a stroke\"),\n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"#endif                                                                        \n"
;
