//! User interface

mod actions;
mod app_window;
mod backup_status;
#[allow(dead_code)]
mod builder;
mod dbus;
mod dialog_about;
mod dialog_archive_prefix;
mod dialog_check;
mod dialog_check_result;
mod dialog_delete_archive;
mod dialog_device_missing;
mod dialog_encryption_password;
mod dialog_exclude;
mod dialog_exclude_pattern;
mod dialog_info;
mod dialog_preferences;
mod dialog_prune;
mod dialog_prune_review;
mod dialog_setup;
mod dialog_storage;
mod error;
mod export;
mod globals;
mod headerbar;
mod operation;
mod page_archives;
mod page_backup;
mod page_detail;
mod page_overview;
mod page_schedule;
mod prelude;
mod shell;
mod status;
mod toast_size_estimate;
mod utils;
mod widget;

pub(crate) use globals::{BACKUP_CONFIG, BACKUP_HISTORY, SCHEDULE_STATUS};

use gtk::prelude::*;
use gvdb_macros::include_gresource_from_dir;

use crate::borg;
use crate::config;
use crate::ui;
use crate::ui::prelude::*;
use config::TrackChanges;

static GRESOURCE_BYTES: &[u8] =
    if const_str::equal!("/org/gnome/World/PikaBackup", crate::DBUS_API_PATH) {
        include_gresource_from_dir!("/org/gnome/World/PikaBackup", "data/resources")
    } else if const_str::equal!("/org/gnome/World/PikaBackup/Devel", crate::DBUS_API_PATH) {
        include_gresource_from_dir!("/org/gnome/World/PikaBackup/Devel", "data/resources")
    } else {
        panic!("Invalid DBUS_API_PATH")
    };

// Run application
pub fn main() {
    if std::env::var_os("ZBUS_TRACING").map_or(false, |x| !x.is_empty()) {
        tracing_subscriber::FmtSubscriber::builder()
            .with_max_level(tracing_subscriber::filter::LevelFilter::TRACE)
            .init();
    }

    crate::utils::init_gettext();

    adw_app().connect_startup(on_startup);
    adw_app().connect_activate(on_activate);
    adw_app().connect_shutdown(on_shutdown);

    // Ctrl-C handling
    glib::unix_signal_add(nix::sys::signal::Signal::SIGINT as i32, on_ctrlc);

    gio::resources_register(
        &gio::Resource::from_data(&glib::Bytes::from_static(GRESOURCE_BYTES)).unwrap(),
    );

    adw_app().run();
}

fn on_ctrlc() -> glib::ControlFlow {
    debug!("Quit: SIGINT (Ctrl+C)");

    BORG_OPERATION.with(|operations| {
        for op in operations.load().values() {
            op.set_instruction(borg::Instruction::Abort(borg::Abort::User));
        }
    });

    adw_app().quit();
    glib::ControlFlow::Continue
}

fn on_shutdown(_app: &adw::Application) {
    IS_SHUTDOWN.swap(std::sync::Arc::new(true));

    let result = BACKUP_HISTORY.try_update(|histories| {
        config::Histories::handle_shutdown(histories);
        Ok(())
    });

    if let Err(err) = result {
        error!("Failed to write config during shutdown: {}", err);
    }

    while !ACTIVE_MOUNTS.load().is_empty() {
        async_std::task::block_on(async {
            for repo_id in ACTIVE_MOUNTS.load().iter() {
                if borg::functions::umount(repo_id).await.is_ok() {
                    ACTIVE_MOUNTS.update(|mounts| {
                        mounts.remove(repo_id);
                    });
                }
            }
        })
    }

    debug!("Good bye!");
}

fn on_startup(_app: &adw::Application) {
    debug!("Signal 'startup'");
    ui::utils::config_io::load_config();
    config::ScheduleStatus::update_on_change(&SCHEDULE_STATUS, |err| {
        Err::<(), std::io::Error>(err).handle("Failed to load Schedule Status")
    })
    .handle("Failed to Load Schedule Status");

    // Force adwaita icon theme
    if let Some(settings) = gtk::Settings::default() {
        settings.set_property("gtk-icon-theme-name", "Adwaita");
    }

    ui::actions::init();
    glib::MainContext::default().spawn_local(async {
        ui::dbus::init().await;
    });

    ui::app_window::init();
    ui::headerbar::init();

    ui::page_overview::init();

    ui::page_detail::init();
    ui::page_backup::init::init();
    ui::page_archives::init();
    ui::page_schedule::init::init();

    // init status tracking
    status_tracking();

    adw_app().set_accels_for_action("app.help", &["F1"]);
    adw_app().set_accels_for_action("app.quit", &["<Ctrl>Q"]);
    adw_app().set_accels_for_action("app.setup", &["<Ctrl>N"]);
    adw_app().set_accels_for_action("app.backup-preferences", &["<Ctrl>comma"]);
    adw_app().set_accels_for_action("win.show-help-overlay", &["<Ctrl>question"]);

    if BACKUP_CONFIG.load().iter().count() == 1 {
        if let Some(config) = BACKUP_CONFIG.load().iter().next() {
            ui::page_backup::view_backup_conf(&config.id);
        }
    }
}

fn on_activate(_app: &adw::Application) {
    debug!("Signal 'activate'");
    app_window::show();
}

async fn quit() -> Result<()> {
    debug!("Running quit routine");
    if utils::borg::is_borg_operation_running() {
        if main_ui().window().is_visible() {
            let permission = utils::background_permission().await;

            match permission {
                Ok(()) => {
                    debug!("Hiding main window as backup is currently running");
                    main_ui().window().set_visible(false);
                }
                Err(err) => {
                    err.show().await;

                    ui::utils::confirmation_dialog(
                        &gettext("Abort running backup creation?"),
                        &gettext("The backup will remain incomplete if aborted now."),
                        &gettext("Continue"),
                        &gettext("Abort"),
                    )
                    .await?;
                    quit_real().await;
                }
            }
        } else {
            // Someone wants to quit the app from the shell (eg via backgrounds app list)
            // Or we do something wrong and called this erroneously
            debug!("Received quit request while a backup operation is running. Ignoring");
            let notification = gio::Notification::new(&gettext("A Backup Operation is Running"));
            notification.set_body(Some(&gettext(
                "Pika Backup cannot be quit during a backup operation.",
            )));

            adw_app().send_notification(None, &notification);
        }
    } else {
        quit_real().await;
    }

    Ok(())
}

async fn quit_real() {
    shell::set_status_message(&gettext("Quit")).await;

    adw_app().quit();
}

async fn init_check_borg() -> Result<()> {
    let version_result = utils::spawn_thread("borg::version", borg::version)
        .await?
        .await;

    match version_result {
        Err(err) => {
            let _ = globals::BORG_VERSION.set(format!("Error: {}", err));

            return Err(Message::new(
                gettext("Failed to run “borg”. Is BorgBackup installed correctly?"),
                err,
            )
            .into());
        }
        Ok(version_output) => {
            let _ = globals::BORG_VERSION.set(version_output.clone());

            if let Some(version) = version_output
                .lines()
                .next()
                .and_then(|x| x.split(' ').nth(1))
                .and_then(|v| {
                    // Parse three numbers to a vec
                    v.splitn(3, '.')
                        .map(str::parse::<u32>)
                        .collect::<std::result::Result<Vec<_>, _>>()
                        .ok()
                })
            {
                let version_format = |version: &[u32]| {
                    version
                        .iter()
                        .map(ToString::to_string)
                        .reduce(|acc, s| format!("{acc}.{s}"))
                        .unwrap_or_default()
                };

                if version[..] < borg::MIN_VERSION[..] {
                    return Err(Message::new(
                            gettext("BorgBackup Version Too Old"),
                            gettextf(
                                "The installed version {} of BorgBackup is older than the required version {}. This is unsupported.",
                                &[
                                    &version_format(&version),
                                    &version_format(&borg::MIN_VERSION),
                                ],
                            )).into());
                } else if version[..2] > borg::MAX_VERSION[..] {
                    // Ignore patch version for maximum version determination
                    return Err(Message::new(
                            gettext("BorgBackup Version Too New"),
                            gettextf(
                                "The installed version {} of BorgBackup is new and untested. Version {} is recommended.",
                                &[
                                    &version_format(&version),
                                    &version_format(&borg::MAX_VERSION),
                                ],
                            )).into());
                }
            } else {
                return Err(Message::new(
                    gettext("Failed to Check BorgBackup Version"),
                    gettextf(
                        "The installed version {} might not work.",
                        &[&version_output],
                    ),
                )
                .into());
            }
        }
    }

    Ok(())
}
