/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.s3;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.store.ChecksumIndexInput;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.common.util.StrUtils;
import org.apache.solr.core.DirectoryFactory;
import org.apache.solr.core.backup.repository.BackupRepository;
import org.apache.solr.s3.S3BackupRepositoryConfig;
import org.apache.solr.s3.S3IndexInput;
import org.apache.solr.s3.S3StorageClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3BackupRepository
implements BackupRepository {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final int CHUNK_SIZE = 0x1000000;
    static final String S3_SCHEME = "s3";
    private NamedList<?> config;
    private S3StorageClient client;

    public void init(NamedList<?> args) {
        this.config = args;
        S3BackupRepositoryConfig backupConfig = new S3BackupRepositoryConfig(this.config);
        if (this.client != null) {
            this.client.close();
        }
        this.client = backupConfig.buildClient();
    }

    public <T> T getConfigProperty(String name) {
        return (T)this.config.get(name);
    }

    public URI createURI(String location) {
        if (StrUtils.isNullOrEmpty((String)location)) {
            throw new IllegalArgumentException("cannot create URI with an empty location");
        }
        try {
            URI result = location.startsWith("s3:") ? new URI(location) : (location.startsWith("/") ? new URI(S3_SCHEME, "", location, null) : new URI(S3_SCHEME, "", "/" + location, null));
            return result;
        }
        catch (URISyntaxException ex) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, (Throwable)ex);
        }
    }

    public URI createDirectoryURI(String location) {
        if (StrUtils.isNullOrEmpty((String)location)) {
            throw new IllegalArgumentException("cannot create URI with an empty location");
        }
        if (!((String)location).endsWith("/")) {
            location = (String)location + "/";
        }
        return this.createURI((String)location);
    }

    public URI resolve(URI baseUri, String ... pathComponents) {
        if (!S3_SCHEME.equalsIgnoreCase(baseUri.getScheme())) {
            throw new IllegalArgumentException("URI must begin with 's3:' scheme");
        }
        String path = baseUri + "/" + String.join((CharSequence)"/", pathComponents);
        return URI.create(path).normalize();
    }

    public URI resolveDirectory(URI baseUri, String ... pathComponents) {
        if (pathComponents.length > 0) {
            if (!pathComponents[pathComponents.length - 1].endsWith("/")) {
                pathComponents[pathComponents.length - 1] = pathComponents[pathComponents.length - 1] + "/";
            }
        } else if (!baseUri.toString().endsWith("/")) {
            baseUri = URI.create(baseUri + "/");
        }
        return this.resolve(baseUri, pathComponents);
    }

    public void createDirectory(URI path) throws IOException {
        Objects.requireNonNull(path, "cannot create directory to a null URI");
        String s3Path = S3BackupRepository.getS3Path(path);
        if (log.isDebugEnabled()) {
            log.debug("Create directory '{}'", (Object)s3Path);
        }
        this.client.createDirectory(s3Path);
    }

    public void deleteDirectory(URI path) throws IOException {
        Objects.requireNonNull(path, "cannot delete directory with a null URI");
        String s3Path = S3BackupRepository.getS3Path(path);
        if (log.isDebugEnabled()) {
            log.debug("Delete directory '{}'", (Object)s3Path);
        }
        this.client.deleteDirectory(s3Path);
    }

    public void delete(URI path, Collection<String> files) throws IOException {
        Objects.requireNonNull(path, "cannot delete files without a valid URI path");
        Objects.requireNonNull(files, "collection of files to delete cannot be null");
        if (log.isDebugEnabled()) {
            log.debug("Delete files {} from {}", files, (Object)S3BackupRepository.getS3Path(path));
        }
        Set<String> filesToDelete = files.stream().map(file -> this.resolve(path, (String)file)).map(S3BackupRepository::getS3Path).collect(Collectors.toSet());
        this.client.delete(filesToDelete);
    }

    public boolean exists(URI path) throws IOException {
        Objects.requireNonNull(path, "cannot test for existence of a null URI path");
        String s3Path = S3BackupRepository.getS3Path(path);
        if (log.isDebugEnabled()) {
            log.debug("Path exists '{}'", (Object)s3Path);
        }
        return this.client.pathExists(s3Path);
    }

    public IndexInput openInput(URI path, String fileName, IOContext ctx) throws IOException {
        Objects.requireNonNull(path, "cannot open a input stream without a valid URI path");
        if (StrUtils.isNullOrEmpty((String)fileName)) {
            throw new IllegalArgumentException("need a valid file name to read from S3");
        }
        URI filePath = this.resolve(path, fileName);
        String s3Path = S3BackupRepository.getS3Path(filePath);
        if (log.isDebugEnabled()) {
            log.debug("Read from S3 '{}'", (Object)s3Path);
        }
        return new S3IndexInput(this.client.pullStream(s3Path), s3Path, this.client.length(s3Path));
    }

    public OutputStream createOutput(URI path) throws IOException {
        Objects.requireNonNull(path, "cannot write to S3 without a valid URI path");
        String s3Path = S3BackupRepository.getS3Path(path);
        if (log.isDebugEnabled()) {
            log.debug("Write to S3 '{}'", (Object)s3Path);
        }
        return this.client.pushStream(s3Path);
    }

    public String[] listAll(URI path) throws IOException {
        String s3Path = S3BackupRepository.getS3Path(path);
        if (log.isDebugEnabled()) {
            log.debug("listAll for '{}'", (Object)s3Path);
        }
        return this.client.listDir(s3Path);
    }

    public BackupRepository.PathType getPathType(URI path) throws IOException {
        String s3Path = S3BackupRepository.getS3Path(path);
        if (log.isDebugEnabled()) {
            log.debug("getPathType for '{}'", (Object)s3Path);
        }
        return this.client.isDirectory(s3Path) ? BackupRepository.PathType.DIRECTORY : BackupRepository.PathType.FILE;
    }

    public void copyIndexFileFrom(Directory sourceDir, String sourceFileName, URI dest, String destFileName) throws IOException {
        if (StrUtils.isNullOrEmpty((String)sourceFileName)) {
            throw new IllegalArgumentException("must have a valid source file name to copy");
        }
        if (StrUtils.isNullOrEmpty((String)destFileName)) {
            throw new IllegalArgumentException("must have a valid destination file name to copy");
        }
        URI filePath = this.resolve(dest, destFileName);
        String s3Path = S3BackupRepository.getS3Path(filePath);
        Instant start = Instant.now();
        if (log.isDebugEnabled()) {
            log.debug("Upload started to S3 '{}'", (Object)s3Path);
        }
        try (ChecksumIndexInput indexInput = sourceDir.openChecksumInput(sourceFileName, DirectoryFactory.IOCONTEXT_NO_CACHE);){
            if (indexInput.length() <= (long)CodecUtil.footerLength()) {
                throw new CorruptIndexException("file is too small:" + indexInput.length(), (DataInput)indexInput);
            }
            this.client.createDirectory(S3BackupRepository.getS3Path(dest));
            try (OutputStream outputStream = this.client.pushStream(s3Path);){
                int bufferLen;
                byte[] buffer = new byte[0x1000000];
                for (long remaining = indexInput.length() - (long)CodecUtil.footerLength(); remaining > 0L; remaining -= (long)bufferLen) {
                    bufferLen = remaining >= 0x1000000L ? 0x1000000 : (int)remaining;
                    indexInput.readBytes(buffer, 0, bufferLen);
                    outputStream.write(buffer, 0, bufferLen);
                }
                long checksum = CodecUtil.checkFooter((ChecksumIndexInput)indexInput);
                this.writeFooter(checksum, outputStream);
            }
        }
        long timeElapsed = Duration.between(start, Instant.now()).toMillis();
        if (log.isInfoEnabled()) {
            log.info("Upload to S3: '{}' finished in {}ms", (Object)s3Path, (Object)timeElapsed);
        }
    }

    public void copyIndexFileTo(URI sourceDir, String sourceFileName, Directory dest, String destFileName) throws IOException {
        if (StrUtils.isNullOrEmpty((String)sourceFileName)) {
            throw new IllegalArgumentException("must have a valid source file name to copy");
        }
        if (StrUtils.isNullOrEmpty((String)destFileName)) {
            throw new IllegalArgumentException("must have a valid destination file name to copy");
        }
        URI filePath = this.resolve(sourceDir, sourceFileName);
        String s3Path = S3BackupRepository.getS3Path(filePath);
        Instant start = Instant.now();
        if (log.isDebugEnabled()) {
            log.debug("Download started from S3 '{}'", (Object)s3Path);
        }
        try (InputStream inputStream = this.client.pullStream(s3Path);
             IndexOutput indexOutput = dest.createOutput(destFileName, IOContext.DEFAULT);){
            int len;
            byte[] buffer = new byte[0x1000000];
            while ((len = inputStream.read(buffer)) != -1) {
                indexOutput.writeBytes(buffer, 0, len);
            }
        }
        long timeElapsed = Duration.between(start, Instant.now()).toMillis();
        if (log.isInfoEnabled()) {
            log.info("Download from S3 '{}' finished in {}ms", (Object)s3Path, (Object)timeElapsed);
        }
    }

    public void close() {
        this.client.close();
    }

    private static String getS3Path(URI uri) {
        String host = uri.getHost();
        return host == null ? uri.getPath() : host + uri.getPath();
    }

    private void writeFooter(final long checksum, final OutputStream outputStream) throws IOException {
        IndexOutput out = new IndexOutput("", ""){

            public void writeByte(byte b) throws IOException {
                outputStream.write(b);
            }

            public void writeBytes(byte[] b, int offset, int length) throws IOException {
                outputStream.write(b, offset, length);
            }

            public void close() {
            }

            public long getFilePointer() {
                return 0L;
            }

            public long getChecksum() {
                return checksum;
            }
        };
        CodecUtil.writeFooter((IndexOutput)out);
    }
}

