// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use inherent::inherent;

use super::{CommonMetricData, MetricId};

use crate::ipc::need_ipc;

#[cfg(feature = "with_gecko")]
use super::profiler_utils::{truncate_string_for_marker, TelemetryProfilerCategory};

#[cfg(feature = "with_gecko")]
#[derive(serde::Serialize, serde::Deserialize, Debug)]
struct UrlMetricMarker {
    id: MetricId,
    val: String,
}

#[cfg(feature = "with_gecko")]
impl gecko_profiler::ProfilerMarker for UrlMetricMarker {
    fn marker_type_name() -> &'static str {
        "UrlMetric"
    }

    fn marker_type_display() -> gecko_profiler::MarkerSchema {
        use gecko_profiler::schema::*;
        let mut schema = MarkerSchema::new(&[Location::MarkerChart, Location::MarkerTable]);
        schema.set_tooltip_label("{marker.data.id} {marker.data.val}");
        schema.set_table_label("{marker.name} - {marker.data.id}: {marker.data.val}");
        schema.add_key_label_format_searchable(
            "id",
            "Metric",
            Format::UniqueString,
            Searchable::Searchable,
        );
        schema.add_key_label_format_searchable("val", "Value", Format::Url, Searchable::Searchable);
        schema
    }

    fn stream_json_marker_data(&self, json_writer: &mut gecko_profiler::JSONWriter) {
        let name = self.id.get_name();
        json_writer.unique_string_property("id", &name);
        json_writer.string_property("val", self.val.as_str());
    }
}

/// Developer-facing API for recording URL metrics.
///
/// Instances of this class type are automatically generated by the parsers
/// at build time, allowing developers to record values that were previously
/// registered in the metrics.yaml file.
#[derive(Clone)]
pub enum UrlMetric {
    Parent {
        /// The metric's ID. Used for testing and profiler markers. URL
        /// metrics canot be labeled, so we only store a MetricId. If this
        /// changes, this should be changed to a MetricGetter to distinguish
        /// between metrics and sub-metrics.
        id: MetricId,
        inner: glean::private::UrlMetric,
    },
    Child(UrlMetricIpc),
}
#[derive(Clone, Debug)]
pub struct UrlMetricIpc;

impl UrlMetric {
    /// Create a new Url metric.
    pub fn new(id: MetricId, meta: CommonMetricData) -> Self {
        if need_ipc() {
            UrlMetric::Child(UrlMetricIpc)
        } else {
            UrlMetric::Parent {
                id,
                inner: glean::private::UrlMetric::new(meta),
            }
        }
    }

    #[cfg(test)]
    pub(crate) fn child_metric(&self) -> Self {
        match self {
            UrlMetric::Parent { .. } => UrlMetric::Child(UrlMetricIpc),
            UrlMetric::Child(_) => panic!("Can't get a child metric from a child metric"),
        }
    }
}

#[inherent]
impl glean::traits::Url for UrlMetric {
    pub fn set<S: Into<std::string::String>>(&self, value: S) {
        match self {
            #[allow(unused)]
            UrlMetric::Parent { id, inner } => {
                let value: String = value.into();
                #[cfg(feature = "with_gecko")]
                gecko_profiler::lazy_add_marker!(
                    "Url::set",
                    TelemetryProfilerCategory,
                    UrlMetricMarker {
                        id: *id,
                        val: truncate_string_for_marker(value.clone()),
                    }
                );
                inner.set(value)
            }
            UrlMetric::Child(_) => {
                log::error!(
                    "Unable to set Url metric in non-main process. This operation will be ignored."
                );
                // If we're in automation we can panic so the instrumentor knows they've gone wrong.
                // This is a deliberate violation of Glean's "metric APIs must not throw" design.
                assert!(!crate::ipc::is_in_automation(), "Attempted to set URL metric in non-main process, which is forbidden. This panics in automation.");
                // TODO: Record an error.
            }
        };
    }

    pub fn test_get_value<'a, S: Into<Option<&'a str>>>(
        &self,
        ping_name: S,
    ) -> Option<std::string::String> {
        let ping_name = ping_name.into().map(|s| s.to_string());
        match self {
            UrlMetric::Parent { inner, .. } => inner.test_get_value(ping_name),
            UrlMetric::Child(_) => {
                panic!("Cannot get test value for Url metric in non-main process!")
            }
        }
    }

    pub fn test_get_num_recorded_errors(&self, error: glean::ErrorType) -> i32 {
        match self {
            UrlMetric::Parent { inner, .. } => inner.test_get_num_recorded_errors(error),
            UrlMetric::Child(_) => panic!(
                "Cannot get the number of recorded errors for Url metric in non-main process!"
            ),
        }
    }
}

#[cfg(test)]
mod test {
    use crate::{common_test::*, ipc, metrics};

    #[test]
    fn sets_url_value() {
        let _lock = lock_test();

        let metric = &metrics::test_only_ipc::a_url;

        metric.set("https://example.com");

        assert_eq!(
            "https://example.com",
            metric.test_get_value("test-ping").unwrap()
        );
    }

    #[test]
    fn url_ipc() {
        // UrlMetric doesn't support IPC.
        let _lock = lock_test();

        let parent_metric = &metrics::test_only_ipc::a_url;

        parent_metric.set("https://example.com/parent");

        {
            let child_metric = parent_metric.child_metric();

            let _raii = ipc::test_set_need_ipc(true);

            // Instrumentation calls do not panic.
            child_metric.set("https://example.com/child");

            // (They also shouldn't do anything,
            // but that's not something we can inspect in this test)
        }

        assert!(ipc::replay_from_buf(&ipc::take_buf().unwrap()).is_ok());

        assert!(
            "https://example.com/parent" == parent_metric.test_get_value("test-ping").unwrap(),
            "Url metrics should only work in the parent process"
        );
    }
}
