/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

// This file is a Mako template: http://www.makotemplates.org/

// Please note that valid Rust syntax may be mangled by the Mako parser.
// For example, Vec&Foo> will be mangled as Vec&Foo>. To work around these issues, the code
// can be escaped. In the above example, Vec<&Foo> or Vec< &Foo> achieves the desired result of Vec&Foo>.



use app_units::Au;
use servo_arc::{Arc, UniqueArc};
use std::{ops, ptr};
use std::{fmt, mem};

#[cfg(feature = "servo")] use euclid::SideOffsets2D;
#[cfg(feature = "gecko")] use crate::gecko_bindings::structs::{self, nsCSSPropertyID};
#[cfg(feature = "servo")] use crate::logical_geometry::LogicalMargin;
#[cfg(feature = "servo")] use crate::computed_values;
use crate::logical_geometry::WritingMode;
use malloc_size_of::{MallocSizeOf, MallocSizeOfOps};
use crate::computed_value_flags::*;
use cssparser::Parser;
use crate::media_queries::Device;
use crate::parser::ParserContext;
use crate::selector_parser::PseudoElement;
use crate::stylist::Stylist;
use style_traits::{CssWriter, KeywordsCollectFn, ParseError, SpecifiedValueInfo, StyleParseErrorKind, ToCss};
use crate::stylesheets::{CssRuleType, CssRuleTypes, Origin};
use crate::logical_geometry::{LogicalAxis, LogicalCorner, LogicalSide};
use crate::use_counters::UseCounters;
use crate::rule_tree::StrongRuleNode;
use crate::str::CssStringWriter;
use crate::values::{
    computed,
    resolved,
    specified::{font::SystemFont, length::LineHeightBase, color::ColorSchemeFlags},
};
use std::cell::Cell;
use super::{
    PropertyDeclarationId, PropertyId, NonCustomPropertyId,
    NonCustomPropertyIdSet, PropertyFlags, SourcePropertyDeclaration,
    LonghandIdSet, VariableDeclaration, CustomDeclaration,
    WideKeywordDeclaration, NonCustomPropertyIterator,
};



/// Conversion with fewer impls than From/Into
pub trait MaybeBoxed<Out> {
    /// Convert
    fn maybe_boxed(self) -> Out;
}

impl<T> MaybeBoxed<T> for T {
    #[inline]
    fn maybe_boxed(self) -> T { self }
}

impl<T> MaybeBoxed<Box<T>> for T {
    #[inline]
    fn maybe_boxed(self) -> Box<T> { Box::new(self) }
}

macro_rules! expanded {
    ( $( $name: ident: $value: expr ),+ ) => {
        expanded!( $( $name: $value, )+ )
    };
    ( $( $name: ident: $value: expr, )+ ) => {
        Longhands {
            $(
                $name: MaybeBoxed::maybe_boxed($value),
            )+
        }
    }
}

/// A module with all the code for longhand properties.
#[allow(missing_docs)]
pub mod longhands {
    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */




    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#background-color
    pub mod background_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Color::TRANSPARENT_BLACK }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { SpecifiedValue::transparent() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::Color::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundColor,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_color();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_background_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundColor)
        }
    }




    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds/#the-background-image
    pub mod background_image {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Image as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Image as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Image::None }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Image::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Image as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for background-image.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;



            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-image.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundImage,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_image();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_image::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_image(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundImage)
        }
    }




    
    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds-4/#propdef-background-position-x
    pub mod background_position_x {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::position::HorizontalPosition as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::position::HorizontalPosition as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::LengthPercentage::zero_percent() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { SpecifiedValue::initial_specified_value() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::position::HorizontalPosition as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for background-position-x.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;


            use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
            use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

            // FIXME(emilio): For some reason rust thinks that this alias is
            // unused, even though it's clearly used below?
            #[allow(unused)]
            type AnimatedList = OwnedList<<single_value::T as ToAnimatedValue>::AnimatedValue>;

            impl ToAnimatedZero for AnimatedList {
                fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
            }

            impl Animate for AnimatedList {
                fn animate(
                    &self,
                    other: &Self,
                    procedure: Procedure,
                ) -> Result<Self, ()> {
                    Ok(OwnedList(
                        lists::repeatable_list::animate(&self.0, &other.0, procedure)?
                    ))
                }
            }
            impl ComputeSquaredDistance for AnimatedList {
                fn compute_squared_distance(
                    &self,
                    other: &Self,
                ) -> Result<SquaredDistance, ()> {
                    lists::repeatable_list::squared_distance(&self.0, &other.0)
                }
            }

            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-position-x.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundPositionX,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_position_x();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_position_x::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_position_x(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundPositionX)
        }
    }



    
    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds-4/#propdef-background-position-y
    pub mod background_position_y {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::position::VerticalPosition as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::position::VerticalPosition as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::LengthPercentage::zero_percent() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { SpecifiedValue::initial_specified_value() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::position::VerticalPosition as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for background-position-y.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;


            use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
            use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

            // FIXME(emilio): For some reason rust thinks that this alias is
            // unused, even though it's clearly used below?
            #[allow(unused)]
            type AnimatedList = OwnedList<<single_value::T as ToAnimatedValue>::AnimatedValue>;

            impl ToAnimatedZero for AnimatedList {
                fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
            }

            impl Animate for AnimatedList {
                fn animate(
                    &self,
                    other: &Self,
                    procedure: Procedure,
                ) -> Result<Self, ()> {
                    Ok(OwnedList(
                        lists::repeatable_list::animate(&self.0, &other.0, procedure)?
                    ))
                }
            }
            impl ComputeSquaredDistance for AnimatedList {
                fn compute_squared_distance(
                    &self,
                    other: &Self,
                ) -> Result<SquaredDistance, ()> {
                    lists::repeatable_list::squared_distance(&self.0, &other.0)
                }
            }

            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-position-y.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundPositionY,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_position_y();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_position_y::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_position_y(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundPositionY)
        }
    }





    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds/#the-background-repeat
    pub mod background_repeat {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BackgroundRepeat as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BackgroundRepeat as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BackgroundRepeat::repeat() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BackgroundRepeat::repeat() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BackgroundRepeat as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for background-repeat.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;



            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-repeat.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundRepeat,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_repeat();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_repeat::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_repeat(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundRepeat)
        }
    }





    

    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds/#the-background-attachment
    pub mod background_attachment {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Scroll,
            
            Fixed,
            
            Local,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Scroll
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Scroll
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        }

        /// The definition of the computed value for background-attachment.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;



            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-attachment.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundAttachment,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_attachment();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_attachment::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_attachment(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundAttachment)
        }
    }





    

    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds/#the-background-clip
    pub mod background_clip {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            BorderBox,
            
            PaddingBox,
            
            ContentBox,
            
            Text,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::BorderBox
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::BorderBox
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        }

        /// The definition of the computed value for background-clip.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;



            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-clip.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundClip,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_clip();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_clip::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_clip(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundClip)
        }
    }





    

    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds/#the-background-origin
    pub mod background_origin {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            PaddingBox,
            
            BorderBox,
            
            ContentBox,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::PaddingBox
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::PaddingBox
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        }

        /// The definition of the computed value for background-origin.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;



            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-origin.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundOrigin,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_origin();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_origin::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_origin(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundOrigin)
        }
    }





    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds/#the-background-size
    pub mod background_size {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BackgroundSize as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BackgroundSize as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BackgroundSize::auto() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BackgroundSize::auto() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BackgroundSize as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for background-size.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;


            use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
            use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

            // FIXME(emilio): For some reason rust thinks that this alias is
            // unused, even though it's clearly used below?
            #[allow(unused)]
            type AnimatedList = OwnedList<<single_value::T as ToAnimatedValue>::AnimatedValue>;

            impl ToAnimatedZero for AnimatedList {
                fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
            }

            impl Animate for AnimatedList {
                fn animate(
                    &self,
                    other: &Self,
                    procedure: Procedure,
                ) -> Result<Self, ()> {
                    Ok(OwnedList(
                        lists::repeatable_list::animate(&self.0, &other.0, procedure)?
                    ))
                }
            }
            impl ComputeSquaredDistance for AnimatedList {
                fn compute_squared_distance(
                    &self,
                    other: &Self,
                ) -> Result<SquaredDistance, ()> {
                    lists::repeatable_list::squared_distance(&self.0, &other.0)
                }
            }

            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-size.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundSize,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_size();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_size::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_size(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundSize)
        }
    }




// https://drafts.fxtf.org/compositing/#background-blend-mode

    

    
        
    
    
    /// https://drafts.fxtf.org/compositing/#background-blend-mode
    pub mod background_blend_mode {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Normal,
            
            Multiply,
            
            Screen,
            
            Overlay,
            
            Darken,
            
            Lighten,
            
            ColorDodge,
            
            ColorBurn,
            
            HardLight,
            
            SoftLight,
            
            Difference,
            
            Exclusion,
            
            Hue,
            
            Saturation,
            
            Color,
            
            Luminosity,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Normal
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Normal
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        }

        /// The definition of the computed value for background-blend-mode.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use smallvec::SmallVec;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    SmallVec<[T; 1]>;

            pub type UnderlyingOwnedList<T> =
                    SmallVec<[T; 1]>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;



            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of background-blend-mode.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                let mut v = SmallVec::new();
                v.push(single_value::get_initial_value());
                computed_value::List(v)
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;


            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;

        impl SpecifiedValue {
            fn compute_iter<'a, 'cx, 'cx_a>(
                &'a self,
                context: &'cx Context<'cx_a>,
            ) -> computed_value::Iter<'a, 'cx, 'cx_a> {
                computed_value::Iter::new(context, &self.0)
            }
        }

        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackgroundBlendMode,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_background_blend_mode();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::background_blend_mode::SpecifiedValue>()
                },
            };


                // In the case of a vector property we want to pass down an
                // iterator so that this can be computed without allocation.
                //
                // However, computing requires a context, but the style struct
                // being mutated is on the context. We temporarily remove it,
                // mutate it, and then put it back. Vector longhands cannot
                // touch their own style struct whilst computing, else this will
                // panic.
                let mut s =
                    context.builder.take_background();
                {
                    let iter = specified_value.compute_iter(context);
                    s.set_background_blend_mode(iter);
                }
                context.builder.put_background(s);
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackgroundBlendMode)
        }
    }




    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */





    
    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-top-color
    pub mod border_top_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::Color::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderTopColor.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_top_color();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_top_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderTopColor)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-top-style
    pub mod border_top_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderTopStyle.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_top_style();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_top_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderTopStyle)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-top-width
    pub mod border_top_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::BorderSideWidth::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderTopWidth.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_top_width();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderSideWidth>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_top_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderTopWidth)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-right-color
    pub mod border_right_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::Color::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderRightColor.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_right_color();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_right_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderRightColor)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-right-style
    pub mod border_right_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderRightStyle.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_right_style();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_right_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderRightStyle)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-right-width
    pub mod border_right_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::BorderSideWidth::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderRightWidth.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_right_width();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderSideWidth>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_right_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderRightWidth)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-bottom-color
    pub mod border_bottom_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::Color::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderBottomColor.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_bottom_color();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_bottom_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBottomColor)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-bottom-style
    pub mod border_bottom_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderBottomStyle.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_bottom_style();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_bottom_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBottomStyle)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-bottom-width
    pub mod border_bottom_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::BorderSideWidth::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderBottomWidth.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_bottom_width();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderSideWidth>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_bottom_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBottomWidth)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-left-color
    pub mod border_left_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::Color::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderLeftColor.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_left_color();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_left_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderLeftColor)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-left-style
    pub mod border_left_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderLeftStyle.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_left_style();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_left_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderLeftStyle)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-left-width
    pub mod border_left_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::BorderSideWidth::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderLeftWidth.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_left_width();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderSideWidth>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_left_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderLeftWidth)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-block-start-color
    pub mod border_block_start_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Color as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBlockStartColor)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-block-start-style
    pub mod border_block_start_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBlockStartStyle)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-block-start-width
    pub mod border_block_start_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderSideWidth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBlockStartWidth)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-block-end-color
    pub mod border_block_end_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Color as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBlockEndColor)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-block-end-style
    pub mod border_block_end_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBlockEndStyle)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-block-end-width
    pub mod border_block_end_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderSideWidth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderBlockEndWidth)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-inline-start-color
    pub mod border_inline_start_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Color as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderInlineStartColor)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-inline-start-style
    pub mod border_inline_start_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderInlineStartStyle)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-inline-start-width
    pub mod border_inline_start_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderSideWidth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderInlineStartWidth)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-inline-end-color
    pub mod border_inline_end_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Color as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderInlineEndColor)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-inline-end-style
    pub mod border_inline_end_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderInlineEndStyle)
        }
    }



    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-inline-end-width
    pub mod border_inline_end_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderSideWidth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderInlineEndWidth)
        }
    }



    
    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-top-left-radius
    pub mod border_top_left_radius {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderCornerRadius as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderCornerRadius as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderCornerRadius::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderCornerRadius as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderTopLeftRadius.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_top_left_radius();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::BorderCornerRadius>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_border_top_left_radius(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderTopLeftRadius)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-top-right-radius
    pub mod border_top_right_radius {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderCornerRadius as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderCornerRadius as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderCornerRadius::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderCornerRadius as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderTopRightRadius.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_top_right_radius();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::BorderCornerRadius>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_border_top_right_radius(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderTopRightRadius)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-bottom-right-radius
    pub mod border_bottom_right_radius {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderCornerRadius as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderCornerRadius as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderCornerRadius::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderCornerRadius as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderBottomRightRadius.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_bottom_right_radius();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::BorderCornerRadius>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_border_bottom_right_radius(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderBottomRightRadius)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-bottom-left-radius
    pub mod border_bottom_left_radius {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderCornerRadius as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderCornerRadius as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderCornerRadius::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderCornerRadius as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::BorderBottomLeftRadius.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_bottom_left_radius();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::BorderCornerRadius>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_border_bottom_left_radius(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderBottomLeftRadius)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-start-start-radius
    pub mod border_start_start_radius {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderCornerRadius as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderCornerRadius as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderCornerRadius::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderCornerRadius as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderStartStartRadius)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-start-end-radius
    pub mod border_start_end_radius {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderCornerRadius as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderCornerRadius as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderCornerRadius::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderCornerRadius as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderStartEndRadius)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-end-start-radius
    pub mod border_end_start_radius {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderCornerRadius as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderCornerRadius as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderCornerRadius::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderCornerRadius as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderEndStartRadius)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-logical-props/#propdef-border-end-end-radius
    pub mod border_end_end_radius {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderCornerRadius as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderCornerRadius as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderCornerRadius::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderCornerRadius as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderEndEndRadius)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-break/#propdef-box-decoration-break
    pub mod box_decoration_break {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Slice,
            
            Clone,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Slice
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Slice
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BoxDecorationBreak,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_box_decoration_break();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::box_decoration_break::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_box_decoration_break(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BoxDecorationBreak)
        }
    }




    

    
        
    
    /// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/-moz-float-edge)
    pub mod _moz_float_edge {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            ContentBox,
            
            MarginBox,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::ContentBox
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::ContentBox
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozFloatEdge,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit__moz_float_edge();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::_moz_float_edge::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_float_edge(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozFloatEdge)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#the-background-image
    pub mod border_image_source {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Image as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Image as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Image::None }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Image::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Image as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BorderImageSource,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_image_source();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Image>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_image_source(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderImageSource)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-image-outset
    pub mod border_image_outset {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::NonNegativeLengthOrNumberRect as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::NonNegativeLengthOrNumberRect as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { generics::rect::Rect::all(computed::NonNegativeLengthOrNumber::zero()) }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { generics::rect::Rect::all(specified::NonNegativeLengthOrNumber::zero()) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::NonNegativeLengthOrNumberRect as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BorderImageOutset,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_image_outset();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::NonNegativeLengthOrNumberRect>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_border_image_outset(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderImageOutset)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#the-border-image-repeat
    pub mod border_image_repeat {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderImageRepeat as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderImageRepeat as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderImageRepeat::stretch() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BorderImageRepeat::stretch() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderImageRepeat as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BorderImageRepeat,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_image_repeat();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderImageRepeat>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_image_repeat(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderImageRepeat)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-image-width
    pub mod border_image_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderImageWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderImageWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderImageWidth::all(computed::BorderImageSideWidth::one()) }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BorderImageWidth::all(specified::BorderImageSideWidth::one()) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderImageWidth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BorderImageWidth,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_image_width();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::BorderImageWidth>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_border_image_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderImageWidth)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-backgrounds/#border-image-slice
    pub mod border_image_slice {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderImageSlice as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderImageSlice as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderImageSlice::hundred_percent() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BorderImageSlice::hundred_percent() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderImageSlice as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BorderImageSlice,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_border_image_slice();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::BorderImageSlice>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_border_image_slice(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderImageSlice)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */





    
        
    
    /// https://drafts.csswg.org/css-display/#propdef-display
    pub mod display {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Display as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Display as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Display::inline() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Display::inline() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Display as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Display,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_display();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Display>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_display(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Display)
        }
    }




    

    
        
    
    /// Internal (not web-exposed)
    pub mod _moz_top_layer {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            None,
            
            Auto,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::None
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::None
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozTopLayer,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit__moz_top_layer();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::_moz_top_layer::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_top_layer(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozTopLayer)
        }
    }



// An internal-only property for elements in a top layer
// https://fullscreen.spec.whatwg.org/#top-layer

    

    
        
    



    
        
    
    /// https://drafts.csswg.org/css-position/#position-property
    pub mod position {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::PositionProperty as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::PositionProperty as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::PositionProperty::Static }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::PositionProperty::Static }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::PositionProperty as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Position,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_position();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::PositionProperty>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_position(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Position)
        }
    }



// Changes do not invalidate our element. We handle notify/invalidating
// elements that reference anchor-name elsewhere.

    
        
    
    /// https://drafts.csswg.org/css-anchor-position-1/#propdef-anchor-name
    pub mod anchor_name {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::AnchorName as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::AnchorName as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnchorName::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::AnchorName as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::AnchorName,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_anchor_name();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::AnchorName>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_anchor_name(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::AnchorName)
        }
    }



// Changes do not invalidate our element. We handle notify/invalidating
// any affected descendants elsewhere.

    
        
    
    /// https://drafts.csswg.org/css-anchor-position-1/#propdef-scope
    pub mod anchor_scope {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::AnchorScope as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::AnchorScope as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::AnchorScope::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::AnchorScope as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::AnchorScope,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_anchor_scope();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::AnchorScope>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_anchor_scope(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::AnchorScope)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-box/#propdef-float
    pub mod float {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Float as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Float as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Float::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Float as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Float,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_float();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Float>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_float(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Float)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css2/#propdef-clear
    pub mod clear {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Clear as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Clear as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Clear::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Clear as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Clear,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_clear();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Clear>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_clear(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Clear)
        }
    }




    
        
    
    /// https://www.w3.org/TR/CSS2/visudet.html#propdef-vertical-align
    pub mod vertical_align {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::VerticalAlign as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::VerticalAlign as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::VerticalAlign::baseline() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::VerticalAlign as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::VerticalAlign,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_vertical_align();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::VerticalAlign>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_vertical_align(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::VerticalAlign)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-inline-3/#baseline-source
    pub mod baseline_source {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BaselineSource as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BaselineSource as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BaselineSource::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BaselineSource as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BaselineSource,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_baseline_source();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BaselineSource>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_baseline_source(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BaselineSource)
        }
    }



// CSS 2.1, Section 11 - Visual effects


    

    
        
    


    
    
        
    
    /// Internal, may be standardized in the future:               https://developer.mozilla.org/en-US/docs/Web/CSS/overflow-clip-box
    pub mod overflow_clip_box_inline {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OverflowClipBox as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OverflowClipBox as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverflowClipBox::PaddingBox }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OverflowClipBox as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OverflowClipBoxInline,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_overflow_clip_box_inline();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::OverflowClipBox>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_overflow_clip_box_inline(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverflowClipBoxInline)
        }
    }


    
    
        
    
    /// Internal, may be standardized in the future:               https://developer.mozilla.org/en-US/docs/Web/CSS/overflow-clip-box
    pub mod overflow_clip_box_block {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OverflowClipBox as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OverflowClipBox as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverflowClipBox::PaddingBox }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OverflowClipBox as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OverflowClipBoxBlock,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_overflow_clip_box_block();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::OverflowClipBox>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_overflow_clip_box_block(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverflowClipBoxBlock)
        }
    }



    
    
    
        
    
    /// https://drafts.csswg.org/css-overflow-3/#propdef-overflow-y
    pub mod overflow_y {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Overflow as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Overflow as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Overflow::Visible }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Overflow as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::OverflowY.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_overflow_y();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Overflow>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_overflow_y(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverflowY)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-overflow-3/#propdef-overflow-x
    pub mod overflow_x {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Overflow as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Overflow as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Overflow::Visible }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Overflow as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::OverflowX.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_overflow_x();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Overflow>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_overflow_x(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverflowX)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-overflow-3/#propdef-overflow-block
    pub mod overflow_block {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Overflow as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Overflow as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Overflow::Visible }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Overflow as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverflowBlock)
        }
    }


    
    
    
        
    
    /// https://drafts.csswg.org/css-overflow-3/#propdef-overflow-inline
    pub mod overflow_inline {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Overflow as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Overflow as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Overflow::Visible }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Overflow as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverflowInline)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-scroll-anchoring/#exclusion-api
    pub mod overflow_anchor {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OverflowAnchor as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OverflowAnchor as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverflowAnchor::Auto }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::OverflowAnchor::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OverflowAnchor as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OverflowAnchor,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_overflow_anchor();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::OverflowAnchor>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_overflow_anchor(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverflowAnchor)
        }
    }






    
        
    
    /// https://drafts.csswg.org/css-transforms/#propdef-transform
    pub mod transform {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Transform as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Transform as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { generics::transform::Transform::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Transform as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Transform,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_transform();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Transform>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_transform(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Transform)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-transforms-2/#individual-transforms
    pub mod rotate {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Rotate as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Rotate as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { generics::transform::Rotate::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Rotate as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Rotate,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_rotate();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::Rotate>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_rotate(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::Rotate)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-transforms-2/#individual-transforms
    pub mod scale {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Scale as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Scale as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { generics::transform::Scale::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Scale as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Scale,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_scale();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::Scale>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_scale(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::Scale)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-transforms-2/#individual-transforms
    pub mod translate {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Translate as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Translate as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { generics::transform::Translate::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Translate as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Translate,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_translate();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::Translate>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_translate(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::Translate)
        }
    }



// Motion Path Module Level 1

    
        
    
    /// https://drafts.fxtf.org/motion-1/#offset-path-property
    pub mod offset_path {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OffsetPath as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OffsetPath as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OffsetPath::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OffsetPath as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OffsetPath,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_offset_path();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::OffsetPath>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_offset_path(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OffsetPath)
        }
    }



// Motion Path Module Level 1

    
        
    
    /// https://drafts.fxtf.org/motion-1/#offset-distance-property
    pub mod offset_distance {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::LengthPercentage as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::LengthPercentage as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::LengthPercentage::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::LengthPercentage as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OffsetDistance,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_offset_distance();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::LengthPercentage>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_offset_distance(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OffsetDistance)
        }
    }



// Motion Path Module Level 1

    
        
    
    /// https://drafts.fxtf.org/motion-1/#offset-rotate-property
    pub mod offset_rotate {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OffsetRotate as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OffsetRotate as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OffsetRotate::auto() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OffsetRotate as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OffsetRotate,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_offset_rotate();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::OffsetRotate>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_offset_rotate(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OffsetRotate)
        }
    }



// Motion Path Module Level 1

    
        
    
    /// https://drafts.fxtf.org/motion-1/#offset-anchor-property
    pub mod offset_anchor {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::PositionOrAuto as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::PositionOrAuto as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::PositionOrAuto::auto() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::PositionOrAuto as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OffsetAnchor,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_offset_anchor();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::PositionOrAuto>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_offset_anchor(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::OffsetAnchor)
        }
    }



// Motion Path Module Level 1

    
        
    
    /// https://drafts.fxtf.org/motion-1/#offset-position-property
    pub mod offset_position {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OffsetPosition as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OffsetPosition as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OffsetPosition::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OffsetPosition as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OffsetPosition,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_offset_position();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::OffsetPosition>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_offset_position(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::OffsetPosition)
        }
    }



// CSSOM View Module
// https://www.w3.org/TR/cssom-view-1/

    

    
        
    
    /// https://drafts.csswg.org/cssom-view/#propdef-scroll-behavior
    pub mod scroll_behavior {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Auto,
            
            Smooth,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Auto
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Auto
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ScrollBehavior,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_scroll_behavior();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::scroll_behavior::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_scroll_behavior(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ScrollBehavior)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-scroll-snap-1/#scroll-snap-align
    pub mod scroll_snap_align {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ScrollSnapAlign as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ScrollSnapAlign as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ScrollSnapAlign::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ScrollSnapAlign as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ScrollSnapAlign,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_scroll_snap_align();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ScrollSnapAlign>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_scroll_snap_align(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ScrollSnapAlign)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-scroll-snap-1/#scroll-snap-type
    pub mod scroll_snap_type {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ScrollSnapType as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ScrollSnapType as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ScrollSnapType::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ScrollSnapType as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ScrollSnapType,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_scroll_snap_type();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ScrollSnapType>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_scroll_snap_type(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ScrollSnapType)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-scroll-snap-1/#scroll-snap-stop
    pub mod scroll_snap_stop {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ScrollSnapStop as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ScrollSnapStop as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ScrollSnapStop::Normal }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ScrollSnapStop as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ScrollSnapStop,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_scroll_snap_stop();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ScrollSnapStop>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_scroll_snap_stop(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ScrollSnapStop)
        }
    }



    
    
        
    
    /// https://wicg.github.io/overscroll-behavior/#overscroll-behavior-properties
    pub mod overscroll_behavior_y {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OverscrollBehavior as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OverscrollBehavior as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverscrollBehavior::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OverscrollBehavior as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::OverscrollBehaviorY.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_overscroll_behavior_y();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::OverscrollBehavior>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_overscroll_behavior_y(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverscrollBehaviorY)
        }
    }


    
    
        
    
    /// https://wicg.github.io/overscroll-behavior/#overscroll-behavior-properties
    pub mod overscroll_behavior_x {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OverscrollBehavior as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OverscrollBehavior as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverscrollBehavior::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OverscrollBehavior as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap().logical_group(),
                LonghandId::OverscrollBehaviorX.logical_group(),
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_overscroll_behavior_x();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::OverscrollBehavior>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_overscroll_behavior_x(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverscrollBehaviorX)
        }
    }


    
    
        
    
    /// https://wicg.github.io/overscroll-behavior/#overscroll-behavior-properties
    pub mod overscroll_behavior_block {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OverscrollBehavior as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OverscrollBehavior as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverscrollBehavior::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OverscrollBehavior as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverscrollBehaviorBlock)
        }
    }


    
    
        
    
    /// https://wicg.github.io/overscroll-behavior/#overscroll-behavior-properties
    pub mod overscroll_behavior_inline {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OverscrollBehavior as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OverscrollBehavior as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverscrollBehavior::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OverscrollBehavior as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            declaration.debug_crash("Should physicalize before entering here");
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverscrollBehaviorInline)
        }
    }



// Compositing and Blending Level 1
// http://www.w3.org/TR/compositing-1/

    

    
        
    
    /// https://drafts.fxtf.org/compositing/#isolation
    pub mod isolation {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Auto,
            
            Isolate,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Auto
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Auto
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Isolation,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_isolation();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::isolation::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_isolation(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Isolation)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-break/#propdef-break-after
    pub mod break_after {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BreakBetween as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BreakBetween as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BreakBetween::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BreakBetween as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BreakAfter,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_break_after();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BreakBetween>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_break_after(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BreakAfter)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-break/#propdef-break-before
    pub mod break_before {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BreakBetween as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BreakBetween as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BreakBetween::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BreakBetween as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BreakBefore,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_break_before();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BreakBetween>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_break_before(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BreakBefore)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-break/#propdef-break-inside
    pub mod break_inside {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BreakWithin as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BreakWithin as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BreakWithin::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BreakWithin as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BreakInside,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_break_inside();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BreakWithin>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_break_inside(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BreakInside)
        }
    }



// CSS Basic User Interface Module Level 3
// http://dev.w3.org/csswg/css-ui

    
        
    
    /// https://drafts.csswg.org/css-ui/#propdef-resize
    pub mod resize {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Resize as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Resize as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Resize::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Resize as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Resize,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_resize();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Resize>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_resize(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Resize)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-transforms/#perspective
    pub mod perspective {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Perspective as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Perspective as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Perspective::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Perspective as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Perspective,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_perspective();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Perspective>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_perspective(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Perspective)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-transforms-2/#perspective-origin-property
    pub mod perspective_origin {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Position as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Position as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::position::Position::center() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Position as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::PerspectiveOrigin,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_perspective_origin();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::Position>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_perspective_origin(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::PerspectiveOrigin)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-transforms/#backface-visibility-property
    pub mod backface_visibility {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Visible,
            
            Hidden,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Visible
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Visible
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackfaceVisibility,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_backface_visibility();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::backface_visibility::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_backface_visibility(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackfaceVisibility)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-transforms/#transform-box
    pub mod transform_box {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TransformBox as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TransformBox as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TransformBox::ViewBox }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TransformBox as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TransformBox,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_transform_box();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TransformBox>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_transform_box(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TransformBox)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-transforms-2/#transform-style-property
    pub mod transform_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TransformStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TransformStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TransformStyle::Flat }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TransformStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TransformStyle,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_transform_style();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TransformStyle>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_transform_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TransformStyle)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-transforms/#transform-origin-property
    pub mod transform_origin {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TransformOrigin as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TransformOrigin as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TransformOrigin::initial_value() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TransformOrigin as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TransformOrigin,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_transform_origin();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::TransformOrigin>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_transform_origin(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::TransformOrigin)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-contain/#contain-property
    pub mod contain {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Contain as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Contain as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { specified::Contain::empty() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Contain as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Contain,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_contain();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Contain>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_contain(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Contain)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-contain/#content-visibility
    pub mod content_visibility {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ContentVisibility as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ContentVisibility as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ContentVisibility::Visible }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ContentVisibility as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ContentVisibility,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_content_visibility();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ContentVisibility>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_content_visibility(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ContentVisibility)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-contain-3/#container-type
    pub mod container_type {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ContainerType as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ContainerType as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ContainerType::Normal }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ContainerType as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ContainerType,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_container_type();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ContainerType>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_container_type(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ContainerType)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-contain-3/#container-name
    pub mod container_name {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ContainerName as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ContainerName as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ContainerName::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ContainerName as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ContainerName,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_container_name();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ContainerName>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_container_name(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ContainerName)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-ui-4/#propdef-appearance
    pub mod appearance {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Appearance as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Appearance as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Appearance::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Appearance as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Appearance,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_appearance();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Appearance>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_appearance(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Appearance)
        }
    }



// The inherent widget type of an element, selected by specifying
// `appearance: auto`.

    
        
    
    /// Internal (not web-exposed)
    pub mod _moz_default_appearance {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Appearance as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Appearance as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Appearance::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Appearance as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozDefaultAppearance,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit__moz_default_appearance();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Appearance>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_default_appearance(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozDefaultAppearance)
        }
    }




    

    
        
    
    /// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/-moz-orient)
    pub mod _moz_orient {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Inline,
            
            Block,
            
            Horizontal,
            
            Vertical,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Inline
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Inline
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozOrient,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit__moz_orient();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::_moz_orient::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_orient(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozOrient)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-will-change/#will-change
    pub mod will_change {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::WillChange as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::WillChange as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::WillChange::auto() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::WillChange as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WillChange,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_will_change();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::WillChange>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_will_change(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WillChange)
        }
    }



// The spec issue for the parse_method: https://github.com/w3c/csswg-drafts/issues/4102.

    
        
    
    /// https://drafts.csswg.org/css-shapes/#shape-image-threshold-property
    pub mod shape_image_threshold {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Opacity as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Opacity as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { 0.0 }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Opacity as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ShapeImageThreshold,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_shape_image_threshold();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Opacity>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_shape_image_threshold(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ShapeImageThreshold)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-shapes/#shape-margin-property
    pub mod shape_margin {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::NonNegativeLengthPercentage as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::NonNegativeLengthPercentage as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::NonNegativeLengthPercentage::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::NonNegativeLengthPercentage as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ShapeMargin,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_shape_margin();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::NonNegativeLengthPercentage>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_shape_margin(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ShapeMargin)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-shapes/#shape-outside-property
    pub mod shape_outside {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::basic_shape::ShapeOutside as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::basic_shape::ShapeOutside as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { generics::basic_shape::ShapeOutside::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::basic_shape::ShapeOutside as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ShapeOutside,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_shape_outside();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::basic_shape::ShapeOutside>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_shape_outside(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ShapeOutside)
        }
    }




    
        
    
    /// https://compat.spec.whatwg.org/#touch-action
    pub mod touch_action {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TouchAction as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TouchAction as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TouchAction::auto() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TouchAction as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TouchAction,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_touch_action();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TouchAction>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_touch_action(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TouchAction)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-overflow-3/#line-clamp
    pub mod _webkit_line_clamp {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::LineClamp as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::LineClamp as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::LineClamp::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::LineClamp as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WebkitLineClamp,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit__webkit_line_clamp();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::LineClamp>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__webkit_line_clamp(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WebkitLineClamp)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-overflow-3/#scrollbar-gutter-property
    pub mod scrollbar_gutter {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ScrollbarGutter as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ScrollbarGutter as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ScrollbarGutter::AUTO }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ScrollbarGutter as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ScrollbarGutter,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_scrollbar_gutter();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ScrollbarGutter>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_scrollbar_gutter(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ScrollbarGutter)
        }
    }




    
        
    
    /// Non-standard (https://github.com/atanassov/css-zoom/ is the closest)
    pub mod zoom {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Zoom as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Zoom as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::box_::Zoom::ONE }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Zoom as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Zoom,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_zoom();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Zoom>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_zoom(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Zoom)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */




    
        
    
    /// https://drafts.csswg.org/css-multicol/#propdef-column-width
    pub mod column_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::length::NonNegativeLengthOrAuto as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::length::NonNegativeLengthOrAuto as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::length::NonNegativeLengthOrAuto::auto() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::length::NonNegativeLengthOrAuto::auto() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::length::NonNegativeLengthOrAuto as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColumnWidth,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_column_width();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::length::NonNegativeLengthOrAuto>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_column_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColumnWidth)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-multicol/#propdef-column-count
    pub mod column_count {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ColumnCount as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ColumnCount as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ColumnCount::Auto }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::ColumnCount::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ColumnCount as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColumnCount,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_column_count();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ColumnCount>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_column_count(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColumnCount)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-multicol/#propdef-column-fill
    pub mod column_fill {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Balance,
            
            Auto,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Balance
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Balance
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColumnFill,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_column_fill();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::column_fill::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_column_fill(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColumnFill)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-multicol/#propdef-column-rule-width
    pub mod column_rule_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSideWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSideWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au::from_px(3) }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BorderSideWidth::medium() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderSideWidth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColumnRuleWidth,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_column_rule_width();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderSideWidth>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_column_rule_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColumnRuleWidth)
        }
    }



// https://drafts.csswg.org/css-multicol-1/#crc

    
        
    
    /// https://drafts.csswg.org/css-multicol/#propdef-column-rule-color
    pub mod column_rule_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Color::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Color as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColumnRuleColor,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_column_rule_color();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_column_rule_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColumnRuleColor)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-multicol/#propdef-column-span
    pub mod column_span {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            None,
            
            All,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::None
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::None
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColumnSpan,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_column_span();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::column_span::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_column_span(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColumnSpan)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-multicol/#propdef-column-rule-style
    pub mod column_rule_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderStyle::None }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::BorderStyle::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColumnRuleStyle,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_column_rule_style();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::BorderStyle>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_column_rule_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColumnRuleStyle)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */




    
        
    
    /// https://drafts.csswg.org/css-content/#propdef-content
    pub mod content {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Content as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Content as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Content::normal() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Content::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Content as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Content,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_content();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Content>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_content(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Content)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-lists/#propdef-counter-increment
    pub mod counter_increment {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::CounterIncrement as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::CounterIncrement as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::CounterIncrement as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::CounterIncrement,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_counter_increment();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::CounterIncrement>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_counter_increment(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::CounterIncrement)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-lists-3/#propdef-counter-reset
    pub mod counter_reset {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::CounterReset as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::CounterReset as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::CounterReset as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::CounterReset,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_counter_reset();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::CounterReset>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_counter_reset(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::CounterReset)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-lists-3/#propdef-counter-set
    pub mod counter_set {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::CounterSet as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::CounterSet as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::CounterSet as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::CounterSet,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_counter_set();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::CounterSet>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_counter_set(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::CounterSet)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */




    
        
    
    /// https://drafts.csswg.org/css-color/#transparency
    pub mod opacity {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Opacity as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Opacity as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { 1.0 }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Opacity as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Opacity,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_opacity();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Opacity>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_opacity(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Opacity)
        }
    }




    
        
    
    
    /// https://drafts.csswg.org/css-backgrounds/#box-shadow
    pub mod box_shadow {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BoxShadow as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BoxShadow as T;
        }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BoxShadow as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for box-shadow.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    crate::OwnedSlice<T>;

            pub type UnderlyingOwnedList<T> =
                    crate::OwnedSlice<T>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(if_empty = "none", iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;

            impl From<ComputedList> for UnderlyingList<single_value::T> {
                #[inline]
                fn from(l: ComputedList) -> Self {
                    l.0
                }
            }
            impl From<UnderlyingList<single_value::T>> for ComputedList {
                #[inline]
                fn from(l: UnderlyingList<single_value::T>) -> Self {
                    List(l)
                }
            }

            use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
            use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

            // FIXME(emilio): For some reason rust thinks that this alias is
            // unused, even though it's clearly used below?
            #[allow(unused)]
            type AnimatedList = OwnedList<<single_value::T as ToAnimatedValue>::AnimatedValue>;

            impl ToAnimatedZero for AnimatedList {
                fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
            }

            impl Animate for AnimatedList {
                fn animate(
                    &self,
                    other: &Self,
                    procedure: Procedure,
                ) -> Result<Self, ()> {
                    Ok(OwnedList(
                        lists::with_zero::animate(&self.0, &other.0, procedure)?
                    ))
                }
            }
            impl ComputeSquaredDistance for AnimatedList {
                fn compute_squared_distance(
                    &self,
                    other: &Self,
                ) -> Result<SquaredDistance, ()> {
                    lists::with_zero::squared_distance(&self.0, &other.0)
                }
            }

            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of box-shadow.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(if_empty = "none", iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                computed_value::List(Default::default())
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;

            if input.try_parse(|input| input.expect_ident_matching("none")).is_ok() {
                return Ok(SpecifiedValue(Default::default()))
            }

            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;


        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BoxShadow,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_box_shadow();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::box_shadow::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_box_shadow(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BoxShadow)
        }
    }





    
        
    
    /// https://drafts.fxtf.org/css-masking/#clip-property
    pub mod clip {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ClipRectOrAuto as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ClipRectOrAuto as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ClipRectOrAuto::auto() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::ClipRectOrAuto::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Clip,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_clip();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::ClipRectOrAuto>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_clip(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::Clip)
        }
    }




    
        
    
    
    /// https://drafts.fxtf.org/filters/#propdef-filter
    pub mod filter {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Filter as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Filter as T;
        }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Filter as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for filter.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    crate::OwnedSlice<T>;

            pub type UnderlyingOwnedList<T> =
                    crate::OwnedSlice<T>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            pub struct OwnedList<T>(
                #[css(if_empty = "none", iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;

            impl From<ComputedList> for UnderlyingList<single_value::T> {
                #[inline]
                fn from(l: ComputedList) -> Self {
                    l.0
                }
            }
            impl From<UnderlyingList<single_value::T>> for ComputedList {
                #[inline]
                fn from(l: UnderlyingList<single_value::T>) -> Self {
                    List(l)
                }
            }

            use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
            use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

            // FIXME(emilio): For some reason rust thinks that this alias is
            // unused, even though it's clearly used below?
            #[allow(unused)]
            type AnimatedList = OwnedList<<single_value::T as ToAnimatedValue>::AnimatedValue>;

            impl ToAnimatedZero for AnimatedList {
                fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
            }

            impl Animate for AnimatedList {
                fn animate(
                    &self,
                    other: &Self,
                    procedure: Procedure,
                ) -> Result<Self, ()> {
                    Ok(OwnedList(
                        lists::with_zero::animate(&self.0, &other.0, procedure)?
                    ))
                }
            }
            impl ComputeSquaredDistance for AnimatedList {
                fn compute_squared_distance(
                    &self,
                    other: &Self,
                ) -> Result<SquaredDistance, ()> {
                    lists::with_zero::squared_distance(&self.0, &other.0)
                }
            }

            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of filter.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        pub struct SpecifiedValue(
            #[css(if_empty = "none", iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                computed_value::List(Default::default())
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;

            if input.try_parse(|input| input.expect_ident_matching("none")).is_ok() {
                return Ok(SpecifiedValue(Default::default()))
            }

            let v = style_traits::Space::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;


        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Filter,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_filter();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::filter::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_filter(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Filter)
        }
    }





    
        
    
    
    /// https://drafts.fxtf.org/filter-effects-2/#propdef-backdrop-filter
    pub mod backdrop_filter {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Filter as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Filter as T;
        }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Filter as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for backdrop-filter.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    crate::OwnedSlice<T>;

            pub type UnderlyingOwnedList<T> =
                    crate::OwnedSlice<T>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            pub struct OwnedList<T>(
                #[css(if_empty = "none", iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub type ComputedList = OwnedList<single_value::T>;
            pub use self::OwnedList as List;

            impl From<ComputedList> for UnderlyingList<single_value::T> {
                #[inline]
                fn from(l: ComputedList) -> Self {
                    l.0
                }
            }
            impl From<UnderlyingList<single_value::T>> for ComputedList {
                #[inline]
                fn from(l: UnderlyingList<single_value::T>) -> Self {
                    List(l)
                }
            }

            use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
            use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

            // FIXME(emilio): For some reason rust thinks that this alias is
            // unused, even though it's clearly used below?
            #[allow(unused)]
            type AnimatedList = OwnedList<<single_value::T as ToAnimatedValue>::AnimatedValue>;

            impl ToAnimatedZero for AnimatedList {
                fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
            }

            impl Animate for AnimatedList {
                fn animate(
                    &self,
                    other: &Self,
                    procedure: Procedure,
                ) -> Result<Self, ()> {
                    Ok(OwnedList(
                        lists::with_zero::animate(&self.0, &other.0, procedure)?
                    ))
                }
            }
            impl ComputeSquaredDistance for AnimatedList {
                fn compute_squared_distance(
                    &self,
                    other: &Self,
                ) -> Result<SquaredDistance, ()> {
                    lists::with_zero::squared_distance(&self.0, &other.0)
                }
            }

            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of backdrop-filter.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        pub struct SpecifiedValue(
            #[css(if_empty = "none", iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                computed_value::List(Default::default())
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;

            if input.try_parse(|input| input.expect_ident_matching("none")).is_ok() {
                return Ok(SpecifiedValue(Default::default()))
            }

            let v = style_traits::Space::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;


        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                use std::iter::FromIterator;
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BackdropFilter,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_backdrop_filter();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::backdrop_filter::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_backdrop_filter(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BackdropFilter)
        }
    }





    

    
        
    
    /// https://drafts.fxtf.org/compositing/#propdef-mix-blend-mode
    pub mod mix_blend_mode {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Normal,
            
            Multiply,
            
            Screen,
            
            Overlay,
            
            Darken,
            
            Lighten,
            
            ColorDodge,
            
            ColorBurn,
            
            HardLight,
            
            SoftLight,
            
            Difference,
            
            Exclusion,
            
            Hue,
            
            Saturation,
            
            Color,
            
            Luminosity,
            
            PlusLighter,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Normal
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Normal
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = true;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MixBlendMode,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Initial => {
                                declaration.debug_crash("Unexpected initial or unset for non-inherited property");
                        },
                        CSSWideKeyword::Inherit => {
                                context.rule_cache_conditions.borrow_mut().set_uncacheable();
                                context.builder.inherit_mix_blend_mode();
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::mix_blend_mode::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_mix_blend_mode(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MixBlendMode)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */





    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-family
    pub mod font_family {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontFamily as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontFamily as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontFamily::serif() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontFamily as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontFamily,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_family();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontFamily>()
                },
            };

            if let Some(sf) = specified_value.get_system() {
                longhands::system_font::resolve_system_font(sf, context);
            }

                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_family(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontFamily)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-style
    pub mod font_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontStyle::normal() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontStyle::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontStyle,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_style();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontStyle>()
                },
            };

            if let Some(sf) = specified_value.get_system() {
                longhands::system_font::resolve_system_font(sf, context);
            }

                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontStyle)
        }
    }






    

    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-variant-caps
    pub mod font_variant_caps {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Normal,
            
            SmallCaps,
            
            AllSmallCaps,
            
            PetiteCaps,
            
            AllPetiteCaps,
            
            Unicase,
            
            TitlingCaps,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Normal
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Normal
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontVariantCaps,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_variant_caps();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::font_variant_caps::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_variant_caps(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontVariantCaps)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-weight
    pub mod font_weight {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontWeight as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontWeight as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontWeight::normal() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontWeight::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontWeight as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontWeight,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_weight();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontWeight>()
                },
            };

            if let Some(sf) = specified_value.get_system() {
                longhands::system_font::resolve_system_font(sf, context);
            }

                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_weight(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontWeight)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-size
    pub mod font_size {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontSize as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontSize as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSize::medium() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSize::medium() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            specified::FontSize::parse_quirky(context, input, AllowQuirks::Yes)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontSize,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_size();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontSize>()
                },
            };

            if let Some(sf) = specified_value.get_system() {
                longhands::system_font::resolve_system_font(sf, context);
            }

                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_size(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontSize)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-size-adjust
    pub mod font_size_adjust {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontSizeAdjust as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontSizeAdjust as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSizeAdjust::None }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSizeAdjust::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontSizeAdjust as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontSizeAdjust,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_size_adjust();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontSizeAdjust>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_size_adjust(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontSizeAdjust)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts-4/#font-synthesis-weight
    pub mod font_synthesis_weight {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontSynthesis as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontSynthesis as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSynthesis::Auto }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSynthesis::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontSynthesis as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontSynthesisWeight,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_synthesis_weight();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontSynthesis>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_synthesis_weight(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontSynthesisWeight)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts-4/#font-synthesis-style
    pub mod font_synthesis_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontSynthesisStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontSynthesisStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSynthesisStyle::Auto }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSynthesisStyle::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontSynthesisStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontSynthesisStyle,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_synthesis_style();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontSynthesisStyle>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_synthesis_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontSynthesisStyle)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts-4/#font-synthesis-small-caps
    pub mod font_synthesis_small_caps {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontSynthesis as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontSynthesis as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSynthesis::Auto }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSynthesis::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontSynthesis as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontSynthesisSmallCaps,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_synthesis_small_caps();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontSynthesis>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_synthesis_small_caps(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontSynthesisSmallCaps)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts-4/#font-synthesis-position
    pub mod font_synthesis_position {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontSynthesis as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontSynthesis as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontSynthesis::Auto }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontSynthesis::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontSynthesis as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontSynthesisPosition,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_synthesis_position();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontSynthesis>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_synthesis_position(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontSynthesisPosition)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-stretch
    pub mod font_stretch {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontStretch as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontStretch as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontStretch::hundred() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontStretch::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontStretch as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontStretch,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_stretch();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontStretch>()
                },
            };

            if let Some(sf) = specified_value.get_system() {
                longhands::system_font::resolve_system_font(sf, context);
            }

                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_stretch(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontStretch)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-kerning
    pub mod font_kerning {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Auto,
            
            None,
            
            Normal,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Auto
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Auto
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontKerning,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_kerning();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::font_kerning::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_kerning(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontKerning)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-variant-alternates
    pub mod font_variant_alternates {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontVariantAlternates as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontVariantAlternates as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontVariantAlternates::default() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontVariantAlternates::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontVariantAlternates as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontVariantAlternates,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_variant_alternates();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontVariantAlternates>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_variant_alternates(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontVariantAlternates)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-variant-east-asian
    pub mod font_variant_east_asian {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontVariantEastAsian as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontVariantEastAsian as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontVariantEastAsian::empty() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontVariantEastAsian::empty() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontVariantEastAsian as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontVariantEastAsian,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_variant_east_asian();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontVariantEastAsian>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_variant_east_asian(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontVariantEastAsian)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-variant-emoji
    pub mod font_variant_emoji {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Normal,
            
            Text,
            
            Emoji,
            
            Unicode,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Normal
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Normal
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontVariantEmoji,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_variant_emoji();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::font_variant_emoji::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_variant_emoji(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontVariantEmoji)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-variant-ligatures
    pub mod font_variant_ligatures {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontVariantLigatures as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontVariantLigatures as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontVariantLigatures::empty() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontVariantLigatures::empty() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontVariantLigatures as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontVariantLigatures,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_variant_ligatures();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontVariantLigatures>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_variant_ligatures(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontVariantLigatures)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-variant-numeric
    pub mod font_variant_numeric {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontVariantNumeric as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontVariantNumeric as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontVariantNumeric::empty() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontVariantNumeric::empty() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontVariantNumeric as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontVariantNumeric,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_variant_numeric();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontVariantNumeric>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_variant_numeric(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontVariantNumeric)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-variant-position
    pub mod font_variant_position {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Normal,
            
            Sub,
            
            Super,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Normal
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Normal
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontVariantPosition,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_variant_position();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::font_variant_position::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_variant_position(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontVariantPosition)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#propdef-font-feature-settings
    pub mod font_feature_settings {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontFeatureSettings as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontFeatureSettings as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontFeatureSettings::normal() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontFeatureSettings::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontFeatureSettings as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontFeatureSettings,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_feature_settings();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontFeatureSettings>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_feature_settings(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontFeatureSettings)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts-4/#propdef-font-variation-settings
    pub mod font_variation_settings {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontVariationSettings as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontVariationSettings as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontVariationSettings::normal() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontVariationSettings::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontVariationSettings as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontVariationSettings,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_variation_settings();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontVariationSettings>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_variation_settings(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontVariationSettings)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts-3/#propdef-font-language-override
    pub mod font_language_override {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontLanguageOverride as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontLanguageOverride as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontLanguageOverride::normal() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontLanguageOverride::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontLanguageOverride as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontLanguageOverride,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_language_override();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontLanguageOverride>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_language_override(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontLanguageOverride)
        }
    }




    

    
        
    
    /// https://www.w3.org/TR/css-fonts-4/#font-optical-sizing-def
    pub mod font_optical_sizing {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Auto,
            
            None,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Auto
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Auto
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontOpticalSizing,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_optical_sizing();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::font_optical_sizing::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_optical_sizing(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontOpticalSizing)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-fonts/#font-palette-prop
    pub mod font_palette {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FontPalette as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FontPalette as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::FontPalette::normal() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::FontPalette::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FontPalette as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FontPalette,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_font_palette();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FontPalette>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_font_palette(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FontPalette)
        }
    }




    
        
    
    /// Internal (not web-exposed)
    pub mod _x_lang {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::XLang as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::XLang as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::XLang::get_initial_value() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::XLang as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::XLang,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__x_lang();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::XLang>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__x_lang(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::XLang)
        }
    }




    
        
    
    /// https://mathml-refresh.github.io/mathml-core/#the-math-script-level-property
    pub mod math_depth {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::MathDepth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::MathDepth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { 0 }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::MathDepth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MathDepth,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_math_depth();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::MathDepth>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_math_depth(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MathDepth)
        }
    }




    

    
        
    
    /// https://mathml-refresh.github.io/mathml-core/#the-math-style-property
    pub mod math_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Normal,
            
            Compact,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Normal
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Normal
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

            
            
    
    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
                // We can't match on enum values if we're matching on a u32
                const NORMAL: u32
                    = structs::StyleMathStyle::Normal as u32;
                // We can't match on enum values if we're matching on a u32
                const COMPACT: u32
                    = structs::StyleMathStyle::Compact as u32;
            match kw  {
                    NORMAL => SpecifiedValue::Normal,
                    COMPACT => SpecifiedValue::Compact,
                _ => panic!("Found unexpected value in style struct for math-style property"),
            }
        }
    }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MathStyle,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_math_style();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::math_style::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_math_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MathStyle)
        }
    }




    

    
        
    
    /// Internal (not web-exposed)
    pub mod _moz_math_variant {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            None,
            
            Normal,
            
            Bold,
            
            Italic,
            
            BoldItalic,
            
            Script,
            
            BoldScript,
            
            Fraktur,
            
            DoubleStruck,
            
            BoldFraktur,
            
            SansSerif,
            
            BoldSansSerif,
            
            SansSerifItalic,
            
            SansSerifBoldItalic,
            
            Monospace,
            
            Initial,
            
            Tailed,
            
            Looped,
            
            Stretched,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::None
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::None
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

            
            
    
    #[cfg(feature = "gecko")]
    impl SpecifiedValue {
        /// Obtain a specified value from a Gecko keyword value
        ///
        /// Intended for use with presentation attributes, not style structs
        pub fn from_gecko_keyword(kw: u32) -> Self {
            use crate::gecko_bindings::structs;
                // We can't match on enum values if we're matching on a u32
                const NONE: u32
                    = structs::StyleMathVariant::None as u32;
                // We can't match on enum values if we're matching on a u32
                const NORMAL: u32
                    = structs::StyleMathVariant::Normal as u32;
                // We can't match on enum values if we're matching on a u32
                const BOLD: u32
                    = structs::StyleMathVariant::Bold as u32;
                // We can't match on enum values if we're matching on a u32
                const ITALIC: u32
                    = structs::StyleMathVariant::Italic as u32;
                // We can't match on enum values if we're matching on a u32
                const BOLD_ITALIC: u32
                    = structs::StyleMathVariant::BoldItalic as u32;
                // We can't match on enum values if we're matching on a u32
                const SCRIPT: u32
                    = structs::StyleMathVariant::Script as u32;
                // We can't match on enum values if we're matching on a u32
                const BOLD_SCRIPT: u32
                    = structs::StyleMathVariant::BoldScript as u32;
                // We can't match on enum values if we're matching on a u32
                const FRAKTUR: u32
                    = structs::StyleMathVariant::Fraktur as u32;
                // We can't match on enum values if we're matching on a u32
                const DOUBLE_STRUCK: u32
                    = structs::StyleMathVariant::DoubleStruck as u32;
                // We can't match on enum values if we're matching on a u32
                const BOLD_FRAKTUR: u32
                    = structs::StyleMathVariant::BoldFraktur as u32;
                // We can't match on enum values if we're matching on a u32
                const SANS_SERIF: u32
                    = structs::StyleMathVariant::SansSerif as u32;
                // We can't match on enum values if we're matching on a u32
                const BOLD_SANS_SERIF: u32
                    = structs::StyleMathVariant::BoldSansSerif as u32;
                // We can't match on enum values if we're matching on a u32
                const SANS_SERIF_ITALIC: u32
                    = structs::StyleMathVariant::SansSerifItalic as u32;
                // We can't match on enum values if we're matching on a u32
                const SANS_SERIF_BOLD_ITALIC: u32
                    = structs::StyleMathVariant::SansSerifBoldItalic as u32;
                // We can't match on enum values if we're matching on a u32
                const MONOSPACE: u32
                    = structs::StyleMathVariant::Monospace as u32;
                // We can't match on enum values if we're matching on a u32
                const INITIAL: u32
                    = structs::StyleMathVariant::Initial as u32;
                // We can't match on enum values if we're matching on a u32
                const TAILED: u32
                    = structs::StyleMathVariant::Tailed as u32;
                // We can't match on enum values if we're matching on a u32
                const LOOPED: u32
                    = structs::StyleMathVariant::Looped as u32;
                // We can't match on enum values if we're matching on a u32
                const STRETCHED: u32
                    = structs::StyleMathVariant::Stretched as u32;
            match kw  {
                    NONE => SpecifiedValue::None,
                    NORMAL => SpecifiedValue::Normal,
                    BOLD => SpecifiedValue::Bold,
                    ITALIC => SpecifiedValue::Italic,
                    BOLD_ITALIC => SpecifiedValue::BoldItalic,
                    SCRIPT => SpecifiedValue::Script,
                    BOLD_SCRIPT => SpecifiedValue::BoldScript,
                    FRAKTUR => SpecifiedValue::Fraktur,
                    DOUBLE_STRUCK => SpecifiedValue::DoubleStruck,
                    BOLD_FRAKTUR => SpecifiedValue::BoldFraktur,
                    SANS_SERIF => SpecifiedValue::SansSerif,
                    BOLD_SANS_SERIF => SpecifiedValue::BoldSansSerif,
                    SANS_SERIF_ITALIC => SpecifiedValue::SansSerifItalic,
                    SANS_SERIF_BOLD_ITALIC => SpecifiedValue::SansSerifBoldItalic,
                    MONOSPACE => SpecifiedValue::Monospace,
                    INITIAL => SpecifiedValue::Initial,
                    TAILED => SpecifiedValue::Tailed,
                    LOOPED => SpecifiedValue::Looped,
                    STRETCHED => SpecifiedValue::Stretched,
                _ => panic!("Found unexpected value in style struct for -moz-math-variant property"),
            }
        }
    }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozMathVariant,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__moz_math_variant();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::_moz_math_variant::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_math_variant(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozMathVariant)
        }
    }




    
        
    
    /// Internal (not web-exposed)
    pub mod _x_text_scale {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::XTextScale as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::XTextScale as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::XTextScale::All }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::XTextScale as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::XTextScale,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__x_text_scale();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::XTextScale>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__x_text_scale(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::XTextScale)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css2/visudet.html#propdef-line-height
    pub mod line_height {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::LineHeight as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::LineHeight as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::LineHeight::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::LineHeight as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::LineHeight,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_line_height();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::LineHeight>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_line_height(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::LineHeight)
        }
    }



pub mod system_font {
    //! We deal with system fonts here
    //!
    //! System fonts can only be set as a group via the font shorthand.
    //! They resolve at compute time (not parse time -- this lets the
    //! browser respond to changes to the OS font settings).
    //!
    //! While Gecko handles these as a separate property and keyword
    //! values on each property indicating that the font should be picked
    //! from the -x-system-font property, we avoid this. Instead,
    //! each font longhand has a special SystemFont variant which contains
    //! the specified system font. When the cascade function (in helpers)
    //! detects that a value has a system font, it will resolve it, and
    //! cache it on the ComputedValues. After this, it can be just fetched
    //! whenever a font longhand on the same element needs the system font.
    //!
    //! When a longhand property is holding a SystemFont, it's serialized
    //! to an empty string as if its value comes from a shorthand with
    //! variable reference. We may want to improve this behavior at some
    //! point. See also https://github.com/w3c/csswg-drafts/issues/1586.

    use crate::properties::longhands;
    use std::hash::{Hash, Hasher};
    use crate::values::computed::{ToComputedValue, Context};
    use crate::values::specified::font::SystemFont;
    // ComputedValues are compared at times
    // so we need these impls. We don't want to
    // add Eq to Number (which contains a float)
    // so instead we have an eq impl which skips the
    // cached values
    impl PartialEq for ComputedSystemFont {
        fn eq(&self, other: &Self) -> bool {
            self.system_font == other.system_font
        }
    }
    impl Eq for ComputedSystemFont {}

    impl Hash for ComputedSystemFont {
        fn hash<H: Hasher>(&self, hasher: &mut H) {
            self.system_font.hash(hasher)
        }
    }

    impl ToComputedValue for SystemFont {
        type ComputedValue = ComputedSystemFont;

        fn to_computed_value(&self, cx: &Context) -> Self::ComputedValue {
            use crate::gecko_bindings::bindings;
            use crate::gecko_bindings::structs::nsFont;
            use crate::values::computed::font::FontSize;
            use crate::values::specified::font::KeywordInfo;
            use crate::values::generics::NonNegative;
            use std::mem;

            let mut system = mem::MaybeUninit::<nsFont>::uninit();
            let system = unsafe {
                bindings::Gecko_nsFont_InitSystem(
                    system.as_mut_ptr(),
                    *self,
                    &**cx.style().get_font(),
                    cx.device().document()
                );
                &mut *system.as_mut_ptr()
            };
            let size = NonNegative(cx.maybe_zoom_text(system.size.0));
            let ret = ComputedSystemFont {
                font_family: system.family.clone(),
                font_size: FontSize {
                    computed_size: size,
                    used_size: size,
                    keyword_info: KeywordInfo::none()
                },
                font_weight: system.weight,
                font_stretch: system.stretch,
                font_style: system.style,
                system_font: *self,
            };
            unsafe { bindings::Gecko_nsFont_Destroy(system); }
            ret
        }

        fn from_computed_value(_: &ComputedSystemFont) -> Self {
            unreachable!()
        }
    }

    #[inline]
    /// Compute and cache a system font
    ///
    /// Must be called before attempting to compute a system font
    /// specified value
    pub fn resolve_system_font(system: SystemFont, context: &mut Context) {
        // Checking if context.cached_system_font.is_none() isn't enough,
        // if animating from one system font to another the cached system font
        // may change
        if Some(system) != context.cached_system_font.as_ref().map(|x| x.system_font) {
            let computed = system.to_computed_value(context);
            context.cached_system_font = Some(computed);
        }
    }

    #[derive(Clone, Debug)]
    pub struct ComputedSystemFont {
            pub font_family: longhands::font_family::computed_value::T,
            pub font_size: longhands::font_size::computed_value::T,
            pub font_style: longhands::font_style::computed_value::T,
            pub font_stretch: longhands::font_stretch::computed_value::T,
            pub font_weight: longhands::font_weight::computed_value::T,
        pub system_font: SystemFont,
    }

}


    

    
        
    
    /// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/font-smooth)
    pub mod _moz_osx_font_smoothing {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Auto,
            
            #[parse(aliases = "antialiased")]
            Grayscale,
            
            SubpixelAntialiased,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Auto
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Auto
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozOsxFontSmoothing,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__moz_osx_font_smoothing();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::_moz_osx_font_smoothing::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_osx_font_smoothing(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozOsxFontSmoothing)
        }
    }




    
        
    
    /// Nonstandard (Internal-only)
    pub mod _moz_min_font_size_ratio {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Percentage as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Percentage as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::Percentage::hundred() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Percentage as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozMinFontSizeRatio,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__moz_min_font_size_ratio();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Percentage>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_min_font_size_ratio(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozMinFontSizeRatio)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */



// TODO: collapse. Well, do tables first.

    

    
        
    
    /// https://drafts.csswg.org/css-box/#propdef-visibility
    pub mod visibility {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Visible,
            
            Hidden,
            
            Collapse,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Visible
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Visible
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Visibility,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_visibility();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::visibility::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_visibility(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Visibility)
        }
    }



// CSS Writing Modes Level 3
// https://drafts.csswg.org/css-writing-modes-3

    

    
        
    
    /// https://drafts.csswg.org/css-writing-modes/#propdef-writing-mode
    pub mod writing_mode {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            #[parse(aliases = "lr,lr-tb,rl,rl-tb")]
            HorizontalTb,
            
            #[parse(aliases = "tb,tb-rl")]
            VerticalRl,
            
            VerticalLr,
            
            SidewaysRl,
            
            SidewaysLr,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::HorizontalTb
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::HorizontalTb
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WritingMode,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_writing_mode();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::writing_mode::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_writing_mode(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WritingMode)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-writing-modes/#propdef-direction
    pub mod direction {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Ltr,
            
            Rtl,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Ltr
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Ltr
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Direction,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_direction();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::direction::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_direction(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Direction)
        }
    }




    

    
        
    
    /// None (internal)
    pub mod _moz_box_collapse {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Flex,
            
            Legacy,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Flex
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Flex
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozBoxCollapse,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__moz_box_collapse();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::_moz_box_collapse::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_box_collapse(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozBoxCollapse)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-writing-modes/#propdef-text-orientation
    pub mod text_orientation {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Mixed,
            
            Upright,
            
            #[parse(aliases = "sideways-right")]
            Sideways,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Mixed
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Mixed
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextOrientation,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_orientation();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::text_orientation::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_orientation(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextOrientation)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-color-adjust/#print-color-adjust
    pub mod print_color_adjust {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::PrintColorAdjust as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::PrintColorAdjust as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::PrintColorAdjust::Economy }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::PrintColorAdjust as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::PrintColorAdjust,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_print_color_adjust();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::PrintColorAdjust>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_print_color_adjust(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::PrintColorAdjust)
        }
    }



// According to to CSS-IMAGES-3, `optimizespeed` and `optimizequality` are synonyms for `auto`
// And, firefox doesn't support `pixelated` yet (https://bugzilla.mozilla.org/show_bug.cgi?id=856337)

    
        
    
    /// https://drafts.csswg.org/css-images/#propdef-image-rendering
    pub mod image_rendering {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ImageRendering as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ImageRendering as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::ImageRendering::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ImageRendering as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ImageRendering,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_image_rendering();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ImageRendering>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_image_rendering(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ImageRendering)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-images/#propdef-image-orientation
    pub mod image_orientation {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            FromImage,
            
            None,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::FromImage
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::FromImage
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ImageOrientation,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_image_orientation();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::image_orientation::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_image_orientation(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ImageOrientation)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */



// Section 10 - Text


    

    
        
    
    /// https://www.w3.org/TR/css-inline-3/#propdef-dominant-baseline
    pub mod dominant_baseline {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Auto,
            
            Ideographic,
            
            Alphabetic,
            
            Hanging,
            
            Mathematical,
            
            Central,
            
            Middle,
            
            TextAfterEdge,
            
            TextBeforeEdge,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Auto
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Auto
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::DominantBaseline,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_dominant_baseline();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::dominant_baseline::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_dominant_baseline(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::DominantBaseline)
        }
    }




    

    
        
    
    /// https://svgwg.org/svg2-draft/text.html#TextAnchorProperty
    pub mod text_anchor {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Start,
            
            Middle,
            
            End,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Start
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Start
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextAnchor,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_anchor();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::text_anchor::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_anchor(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextAnchor)
        }
    }



// Section 11 - Painting: Filling, Stroking and Marker Symbols

    

    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#ColorInterpolationProperty
    pub mod color_interpolation {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Srgb,
            
            Auto,
            
            Linearrgb,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Srgb
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Srgb
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColorInterpolation,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_color_interpolation();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::color_interpolation::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_color_interpolation(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColorInterpolation)
        }
    }




    

    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#ColorInterpolationFiltersProperty
    pub mod color_interpolation_filters {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Linearrgb,
            
            Auto,
            
            Srgb,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Linearrgb
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Linearrgb
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ColorInterpolationFilters,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_color_interpolation_filters();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::color_interpolation_filters::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_color_interpolation_filters(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ColorInterpolationFilters)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#SpecifyingFillPaint
    pub mod fill {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SVGPaint as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SVGPaint as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { crate::values::computed::SVGPaint::BLACK }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SVGPaint as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Fill,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_fill();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::SVGPaint>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_fill(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::Fill)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#FillOpacity
    pub mod fill_opacity {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SVGOpacity as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SVGOpacity as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SVGOpacity as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FillOpacity,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_fill_opacity();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::SVGOpacity>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_fill_opacity(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FillOpacity)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#FillRuleProperty
    pub mod fill_rule {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FillRule as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FillRule as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FillRule as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::FillRule,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_fill_rule();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FillRule>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_fill_rule(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::FillRule)
        }
    }




    

    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#ShapeRenderingProperty
    pub mod shape_rendering {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Auto,
            
            Optimizespeed,
            
            Crispedges,
            
            Geometricprecision,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Auto
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Auto
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ShapeRendering,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_shape_rendering();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::shape_rendering::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_shape_rendering(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ShapeRendering)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#SpecifyingStrokePaint
    pub mod stroke {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SVGPaint as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SVGPaint as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SVGPaint as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Stroke,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_stroke();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::SVGPaint>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_stroke(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::Stroke)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#StrokeWidth
    pub mod stroke_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SVGWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SVGWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::SVGWidth::one() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SVGWidth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::StrokeWidth,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_stroke_width();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::SVGWidth>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_stroke_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::StrokeWidth)
        }
    }




    

    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#StrokeLinecapProperty
    pub mod stroke_linecap {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Butt,
            
            Round,
            
            Square,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Butt
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Butt
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::StrokeLinecap,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_stroke_linecap();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::stroke_linecap::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_stroke_linecap(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::StrokeLinecap)
        }
    }




    

    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#StrokeLinejoinProperty
    pub mod stroke_linejoin {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Miter,
            
            Round,
            
            Bevel,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Miter
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Miter
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::StrokeLinejoin,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_stroke_linejoin();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::stroke_linejoin::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_stroke_linejoin(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::StrokeLinejoin)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#StrokeMiterlimitProperty
    pub mod stroke_miterlimit {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::NonNegativeNumber as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::NonNegativeNumber as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { From::from(4.0) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::NonNegativeNumber as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::StrokeMiterlimit,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_stroke_miterlimit();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::NonNegativeNumber>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_stroke_miterlimit(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::StrokeMiterlimit)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#StrokeOpacity
    pub mod stroke_opacity {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SVGOpacity as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SVGOpacity as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SVGOpacity as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::StrokeOpacity,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_stroke_opacity();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::SVGOpacity>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_stroke_opacity(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::StrokeOpacity)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#StrokeDashing
    pub mod stroke_dasharray {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SVGStrokeDashArray as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SVGStrokeDashArray as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SVGStrokeDashArray as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::StrokeDasharray,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_stroke_dasharray();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::SVGStrokeDashArray>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_stroke_dasharray(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::StrokeDasharray)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#StrokeDashing
    pub mod stroke_dashoffset {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SVGLength as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SVGLength as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::SVGLength::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SVGLength as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::StrokeDashoffset,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_stroke_dashoffset();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::SVGLength>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_stroke_dashoffset(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::StrokeDashoffset)
        }
    }



// Section 14 - Clipping, Masking and Compositing

    
        
    
    /// https://svgwg.org/svg2-draft/masking.html#ClipRuleProperty
    pub mod clip_rule {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::FillRule as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::FillRule as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { Default::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::FillRule as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::ClipRule,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_clip_rule();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::FillRule>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_clip_rule(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::ClipRule)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#VertexMarkerProperties
    pub mod marker_start {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::url::UrlOrNone as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::url::UrlOrNone as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::url::UrlOrNone::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::url::UrlOrNone as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MarkerStart,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_marker_start();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::url::UrlOrNone>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_marker_start(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MarkerStart)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#VertexMarkerProperties
    pub mod marker_mid {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::url::UrlOrNone as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::url::UrlOrNone as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::url::UrlOrNone::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::url::UrlOrNone as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MarkerMid,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_marker_mid();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::url::UrlOrNone>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_marker_mid(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MarkerMid)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#VertexMarkerProperties
    pub mod marker_end {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::url::UrlOrNone as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::url::UrlOrNone as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::url::UrlOrNone::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::url::UrlOrNone as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MarkerEnd,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_marker_end();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::url::UrlOrNone>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_marker_end(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MarkerEnd)
        }
    }




    
        
    
    /// https://svgwg.org/svg2-draft/painting.html#PaintOrder
    pub mod paint_order {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SVGPaintOrder as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SVGPaintOrder as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::SVGPaintOrder::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SVGPaintOrder as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::PaintOrder,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_paint_order();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::SVGPaintOrder>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_paint_order(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::PaintOrder)
        }
    }




    
        
    
    /// Nonstandard (https://developer.mozilla.org/en-US/docs/Web/CSS/-moz-context-properties)
    pub mod _moz_context_properties {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::MozContextProperties as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::MozContextProperties as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::MozContextProperties::default() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::MozContextProperties as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozContextProperties,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__moz_context_properties();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::MozContextProperties>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_context_properties(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozContextProperties)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */




    

    
        
    
    /// https://drafts.csswg.org/css-tables/#propdef-border-collapse
    pub mod border_collapse {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Separate,
            
            Collapse,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Separate
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Separate
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BorderCollapse,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_border_collapse();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::border_collapse::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_border_collapse(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::BorderCollapse)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-tables/#propdef-empty-cells
    pub mod empty_cells {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Show,
            
            Hide,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Show
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Show
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::EmptyCells,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_empty_cells();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::empty_cells::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_empty_cells(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::EmptyCells)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-tables/#propdef-caption-side
    pub mod caption_side {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::table::CaptionSide as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::table::CaptionSide as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::table::CaptionSide::Top }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::table::CaptionSide as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::CaptionSide,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_caption_side();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::table::CaptionSide>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_caption_side(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::CaptionSide)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-tables/#propdef-border-spacing
    pub mod border_spacing {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::BorderSpacing as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::BorderSpacing as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::BorderSpacing::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::BorderSpacing as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::BorderSpacing,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_border_spacing();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<Box<crate::values::specified::BorderSpacing>>()
                },
            };


                let computed = (**specified_value).to_computed_value(context);
                context.builder.set_border_spacing(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(Box::new)
                .map(PropertyDeclaration::BorderSpacing)
        }
    }



    
    /* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */




    
        
    
    /// https://drafts.csswg.org/css-color/#color
    pub mod color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::ColorPropertyValue as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::ColorPropertyValue as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { crate::color::AbsoluteColor::BLACK }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::ColorPropertyValue as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Color,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_color();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::ColorPropertyValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Color)
        }
    }



// CSS Text Module Level 3


    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-text-transform
    pub mod text_transform {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TextTransform as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TextTransform as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextTransform::none() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TextTransform as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextTransform,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_transform();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TextTransform>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_transform(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextTransform)
        }
    }




    

    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-hyphens
    pub mod hyphens {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Manual,
            
            None,
            
            Auto,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Manual
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Manual
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::Hyphens,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_hyphens();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::hyphens::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_hyphens(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::Hyphens)
        }
    }



// TODO: Support <percentage>

    

    
        
    
    /// https://drafts.csswg.org/css-size-adjust/#adjustment-control
    pub mod _moz_text_size_adjust {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Auto,
            
            None,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Auto
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Auto
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::MozTextSizeAdjust,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__moz_text_size_adjust();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::_moz_text_size_adjust::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__moz_text_size_adjust(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::MozTextSizeAdjust)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-text-indent
    pub mod text_indent {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TextIndent as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TextIndent as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextIndent::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TextIndent as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextIndent,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_indent();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TextIndent>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_indent(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextIndent)
        }
    }



// Also known as "word-wrap" (which is more popular because of IE), but this is
// the preferred name per CSS-TEXT 6.2.

    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-overflow-wrap
    pub mod overflow_wrap {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::OverflowWrap as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::OverflowWrap as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::OverflowWrap::Normal }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::OverflowWrap as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::OverflowWrap,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_overflow_wrap();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::OverflowWrap>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_overflow_wrap(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::OverflowWrap)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-word-break
    pub mod word_break {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::WordBreak as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::WordBreak as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::WordBreak::Normal }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::WordBreak as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WordBreak,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_word_break();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::WordBreak>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_word_break(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WordBreak)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-text-justify
    pub mod text_justify {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TextJustify as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TextJustify as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextJustify::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TextJustify as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextJustify,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_justify();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TextJustify>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_justify(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextJustify)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-text-align-last
    pub mod text_align_last {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TextAlignLast as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TextAlignLast as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::text::TextAlignLast::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TextAlignLast as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextAlignLast,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_align_last();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TextAlignLast>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_align_last(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextAlignLast)
        }
    }



// TODO make this a shorthand and implement text-align-last/text-align-all

    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-text-align
    pub mod text_align {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TextAlign as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TextAlign as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextAlign::Start }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TextAlign as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextAlign,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_align();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TextAlign>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_align(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextAlign)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-letter-spacing
    pub mod letter_spacing {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::LetterSpacing as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::LetterSpacing as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::LetterSpacing::normal() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::LetterSpacing as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::LetterSpacing,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_letter_spacing();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::LetterSpacing>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_letter_spacing(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::LetterSpacing)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text/#propdef-word-spacing
    pub mod word_spacing {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::WordSpacing as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::WordSpacing as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::WordSpacing::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::WordSpacing as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WordSpacing,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_word_spacing();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::WordSpacing>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_word_spacing(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WordSpacing)
        }
    }



// TODO: `white-space-collapse: discard` not yet supported

    

    
        
    
    /// https://drafts.csswg.org/css-text-4/#propdef-white-space-collapse
    pub mod white_space_collapse {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        pub use self::computed_value::T as SpecifiedValue;
        pub mod computed_value {
            #[cfg_attr(feature = "servo", derive(Deserialize, Hash, Serialize))]
            #[derive(Clone, Copy, Debug, Eq, FromPrimitive, MallocSizeOf, Parse, PartialEq, SpecifiedValueInfo, ToAnimatedValue, ToComputedValue, ToCss, ToResolvedValue, ToShmem)]
            pub enum T {
            
            Collapse,
            
            Preserve,
            
            PreserveBreaks,
            
            BreakSpaces,
            
            #[parse(aliases = "-moz-pre-space")]
            PreserveSpaces,
            }
        }
        #[inline]
        pub fn get_initial_value() -> computed_value::T {
            computed_value::T::Collapse
        }
        #[inline]
        pub fn get_initial_specified_value() -> SpecifiedValue {
            SpecifiedValue::Collapse
        }
        #[inline]
        pub fn parse<'i, 't>(_context: &ParserContext, input: &mut Parser<'i, 't>)
                             -> Result<SpecifiedValue, ParseError<'i>> {
            SpecifiedValue::parse(input)
        }

    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WhiteSpaceCollapse,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_white_space_collapse();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::white_space_collapse::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_white_space_collapse(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WhiteSpaceCollapse)
        }
    }




    
        
    
    
    /// https://drafts.csswg.org/css-text-decor-3/#text-shadow-property
    pub mod text_shadow {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
        #[allow(unused_imports)]
        use smallvec::SmallVec;

        pub mod single_value {
            #[allow(unused_imports)]
            use cssparser::{Parser, BasicParseError};
            #[allow(unused_imports)]
            use crate::parser::{Parse, ParserContext};
            #[allow(unused_imports)]
            use crate::properties::ShorthandId;
            #[allow(unused_imports)]
            use selectors::parser::SelectorParseErrorKind;
            #[allow(unused_imports)]
            use style_traits::{ParseError, StyleParseErrorKind};
            #[allow(unused_imports)]
            use crate::values::computed::{Context, ToComputedValue};
            #[allow(unused_imports)]
            use crate::values::{computed, specified};
            
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::SimpleShadow as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::SimpleShadow as T;
        }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::SimpleShadow as crate::parser::Parse>::parse(context, input)
        }
    
        
        }

        /// The definition of the computed value for text-shadow.
        pub mod computed_value {
            #[allow(unused_imports)]
            use crate::values::animated::ToAnimatedValue;
            #[allow(unused_imports)]
            use crate::values::resolved::ToResolvedValue;
            pub use super::single_value::computed_value as single_value;
            pub use self::single_value::T as SingleComputedValue;
            use crate::values::computed::ComputedVecIter;

            

            // FIXME(emilio): Add an OwnedNonEmptySlice type, and figure out
            // something for transition-name, which is the only remaining user
            // of NotInitial.
            pub type UnderlyingList<T> =
                    crate::ArcSlice<T>;

            pub type UnderlyingOwnedList<T> =
                    crate::OwnedSlice<T>;


            /// The generic type defining the animated and resolved values for
            /// this property.
            ///
            /// Making this type generic allows the compiler to figure out the
            /// animated value for us, instead of having to implement it
            /// manually for every type we care about.
            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToAnimatedValue, ToResolvedValue, ToCss)]
            #[css(comma)]
            pub struct OwnedList<T>(
                #[css(if_empty = "none", iterable)]
                pub UnderlyingOwnedList<T>,
            );

            /// The computed value for this property.
            pub use self::ComputedList as List;

            #[derive(Clone, Debug, MallocSizeOf, PartialEq, ToCss)]
            #[css(comma)]
            pub struct ComputedList(
                #[css(if_empty = "none", iterable)]
                #[ignore_malloc_size_of = "Arc"]
                pub UnderlyingList<single_value::T>,
            );

            type ResolvedList = OwnedList<<single_value::T as ToResolvedValue>::ResolvedValue>;
            impl ToResolvedValue for ComputedList {
                type ResolvedValue = ResolvedList;

                fn to_resolved_value(self, context: &crate::values::resolved::Context) -> Self::ResolvedValue {
                    OwnedList(
                        self.0
                            .iter()
                            .cloned()
                            .map(|v| v.to_resolved_value(context))
                            .collect()
                    )
                }

                fn from_resolved_value(resolved: Self::ResolvedValue) -> Self {
                    let iter =
                        resolved.0.into_iter().map(ToResolvedValue::from_resolved_value);
                    ComputedList(UnderlyingList::from_iter(iter))
                }
            }

            impl From<ComputedList> for UnderlyingList<single_value::T> {
                #[inline]
                fn from(l: ComputedList) -> Self {
                    l.0
                }
            }
            impl From<UnderlyingList<single_value::T>> for ComputedList {
                #[inline]
                fn from(l: UnderlyingList<single_value::T>) -> Self {
                    List(l)
                }
            }

            use crate::values::animated::{Animate, ToAnimatedZero, Procedure, lists};
            use crate::values::distance::{SquaredDistance, ComputeSquaredDistance};

            // FIXME(emilio): For some reason rust thinks that this alias is
            // unused, even though it's clearly used below?
            #[allow(unused)]
            type AnimatedList = OwnedList<<single_value::T as ToAnimatedValue>::AnimatedValue>;
            impl ToAnimatedValue for ComputedList {
                type AnimatedValue = AnimatedList;

                fn to_animated_value(self, context: &crate::values::animated::Context) -> Self::AnimatedValue {
                    OwnedList(
                        self.0.iter().map(|v| v.clone().to_animated_value(context)).collect()
                    )
                }

                fn from_animated_value(animated: Self::AnimatedValue) -> Self {
                    let iter =
                        animated.0.into_iter().map(ToAnimatedValue::from_animated_value);
                    ComputedList(UnderlyingList::from_iter(iter))
                }
            }

            impl ToAnimatedZero for AnimatedList {
                fn to_animated_zero(&self) -> Result<Self, ()> { Err(()) }
            }

            impl Animate for AnimatedList {
                fn animate(
                    &self,
                    other: &Self,
                    procedure: Procedure,
                ) -> Result<Self, ()> {
                    Ok(OwnedList(
                        lists::with_zero::animate(&self.0, &other.0, procedure)?
                    ))
                }
            }
            impl ComputeSquaredDistance for AnimatedList {
                fn compute_squared_distance(
                    &self,
                    other: &Self,
                ) -> Result<SquaredDistance, ()> {
                    lists::with_zero::squared_distance(&self.0, &other.0)
                }
            }

            /// The computed value, effectively a list of single values.
            pub use self::ComputedList as T;

            pub type Iter<'a, 'cx, 'cx_a> = ComputedVecIter<'a, 'cx, 'cx_a, super::single_value::SpecifiedValue>;
        }

        /// The specified value of text-shadow.
        #[derive(Clone, Debug, MallocSizeOf, PartialEq, SpecifiedValueInfo, ToCss, ToShmem)]
        #[css(comma)]
        pub struct SpecifiedValue(
            #[css(if_empty = "none", iterable)]
            pub crate::OwnedSlice<single_value::SpecifiedValue>,
        );

        pub fn get_initial_value() -> computed_value::T {
                computed_value::List(Default::default())
        }

        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            use style_traits::Separator;

            if input.try_parse(|input| input.expect_ident_matching("none")).is_ok() {
                return Ok(SpecifiedValue(Default::default()))
            }

            let v = style_traits::Comma::parse(input, |parser| {
                single_value::parse(context, parser)
            })?;
            Ok(SpecifiedValue(v.into()))
        }

        pub use self::single_value::SpecifiedValue as SingleSpecifiedValue;


        impl ToComputedValue for SpecifiedValue {
            type ComputedValue = computed_value::T;

            #[inline]
            fn to_computed_value(&self, context: &Context) -> computed_value::T {
                computed_value::List(computed_value::UnderlyingList::from_iter(
                    self.0.iter().map(|i| i.to_computed_value(context))
                ))
            }

            #[inline]
            fn from_computed_value(computed: &computed_value::T) -> Self {
                let iter = computed.0.iter().map(ToComputedValue::from_computed_value);
                SpecifiedValue(iter.collect())
            }
        }
    
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextShadow,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_shadow();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<longhands::text_shadow::SpecifiedValue>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_shadow(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextShadow)
        }
    }





    
        
    
    /// https://drafts.csswg.org/css-text-decor/#propdef-text-emphasis-style
    pub mod text_emphasis_style {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TextEmphasisStyle as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TextEmphasisStyle as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextEmphasisStyle::None }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { SpecifiedValue::None }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TextEmphasisStyle as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextEmphasisStyle,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_emphasis_style();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TextEmphasisStyle>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_emphasis_style(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextEmphasisStyle)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text-decor/#propdef-text-emphasis-position
    pub mod text_emphasis_position {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::TextEmphasisPosition as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::TextEmphasisPosition as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::TextEmphasisPosition::AUTO }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::TextEmphasisPosition::AUTO }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::TextEmphasisPosition as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextEmphasisPosition,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_emphasis_position();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::TextEmphasisPosition>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_emphasis_position(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextEmphasisPosition)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text-decor/#propdef-text-emphasis-color
    pub mod text_emphasis_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Color::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Color as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TextEmphasisColor,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_text_emphasis_color();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_text_emphasis_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TextEmphasisColor)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text-3/#tab-size-property
    pub mod tab_size {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::NonNegativeLengthOrNumber as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::NonNegativeLengthOrNumber as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { generics::length::LengthOrNumber::Number(From::from(8.0)) }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::NonNegativeLengthOrNumber as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::TabSize,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_tab_size();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::NonNegativeLengthOrNumber>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_tab_size(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::TabSize)
        }
    }




    
        
    
    /// https://drafts.csswg.org/css-text-3/#line-break-property
    pub mod line_break {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::LineBreak as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::LineBreak as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed::LineBreak::Auto }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::LineBreak as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::LineBreak,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset_line_break();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::LineBreak>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set_line_break(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::LineBreak)
        }
    }



// CSS Compatibility
// https://compat.spec.whatwg.org

    
        
    
    /// https://compat.spec.whatwg.org/#the-webkit-text-fill-color
    pub mod _webkit_text_fill_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Color as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WebkitTextFillColor,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__webkit_text_fill_color();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__webkit_text_fill_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WebkitTextFillColor)
        }
    }




    
        
    
    /// https://compat.spec.whatwg.org/#the-webkit-text-stroke-color
    pub mod _webkit_text_stroke_color {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::Color as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::Color as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { computed_value::T::currentcolor() }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::Color::currentcolor() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::Color as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WebkitTextStrokeColor,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__webkit_text_stroke_color();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::Color>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__webkit_text_stroke_color(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WebkitTextStrokeColor)
        }
    }




    
        
    
    /// https://compat.spec.whatwg.org/#the-webkit-text-stroke-width
    pub mod _webkit_text_stroke_width {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeyword, ComputedValues, PropertyDeclaration};
        #[allow(unused_imports)]
        use crate::properties::style_structs;
        #[allow(unused_imports)]
        use selectors::parser::SelectorParseErrorKind;
        #[allow(unused_imports)]
        use servo_arc::Arc;
        #[allow(unused_imports)]
        use style_traits::{ParseError, StyleParseErrorKind};
        #[allow(unused_imports)]
        use crate::values::computed::{Context, ToComputedValue};
        #[allow(unused_imports)]
        use crate::values::{computed, generics, specified};
        #[allow(unused_imports)]
        use crate::Atom;
        
            
        #[allow(unused_imports)]
        use app_units::Au;
        #[allow(unused_imports)]
        use crate::values::specified::AllowQuirks;
        #[allow(unused_imports)]
        use crate::Zero;
        #[allow(unused_imports)]
        use smallvec::SmallVec;
        pub use crate::values::specified::LineWidth as SpecifiedValue;
        pub mod computed_value {
            pub use crate::values::computed::LineWidth as T;
        }
        #[inline] pub fn get_initial_value() -> computed_value::T { app_units::Au(0) }
        #[inline] pub fn get_initial_specified_value() -> SpecifiedValue { specified::LineWidth::zero() }
        #[allow(unused_variables)]
        #[inline]
        pub fn parse<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<SpecifiedValue, ParseError<'i>> {
            <specified::LineWidth as crate::parser::Parse>::parse(context, input)
        }
    
        
        #[allow(unused_variables)]
        pub unsafe fn cascade_property(
            declaration: &PropertyDeclaration,
            context: &mut computed::Context,
        ) {
            context.for_non_inherited_property = false;
            debug_assert_eq!(
                declaration.id().as_longhand().unwrap(),
                LonghandId::WebkitTextStrokeWidth,
            );
            let specified_value = match *declaration {
                PropertyDeclaration::CSSWideKeyword(ref wk) => {
                    match wk.keyword {
                        CSSWideKeyword::Initial => {
                                context.builder.reset__webkit_text_stroke_width();
                        },
                        CSSWideKeyword::Unset |
                        CSSWideKeyword::Inherit => {
                                declaration.debug_crash("Unexpected inherit or unset for inherited property");
                        }
                        CSSWideKeyword::RevertLayer |
                        CSSWideKeyword::Revert => {
                            declaration.debug_crash("Found revert/revert-layer not deal with");
                        },
                    }
                    return;
                },
                #[cfg(debug_assertions)]
                PropertyDeclaration::WithVariables(..) => {
                    declaration.debug_crash("Found variables not substituted");
                    return;
                },
                _ => unsafe {
                    declaration.unchecked_value_as::<crate::values::specified::LineWidth>()
                },
            };


                let computed = specified_value.to_computed_value(context);
                context.builder.set__webkit_text_stroke_width(computed)
        }

        pub fn parse_declared<'i, 't>(
            context: &ParserContext,
            input: &mut Parser<'i, 't>,
        ) -> Result<PropertyDeclaration, ParseError<'i>> {
                parse(context, input)
                .map(PropertyDeclaration::WebkitTextStrokeWidth)
        }
    }



// CSS Ruby Layout Module Level 1
// https://drafts.csswg.org/css-ruby/

    

    
        
    
    /// https://drafts.csswg.org/css-ruby/#ruby-align-property
    pub mod ruby_align {
        #[allow(unused_imports)]
        use cssparser::{Parser, BasicParseError, Token};
        #[allow(unused_imports)]
        use crate::parser::{Parse, ParserContext};
        #[allow(unused_imports)]
        use crate::properties::{UnparsedValue, ShorthandId};
        #[allow(unused_imports)]
        use crate::error_reporting::ParseErrorReporter;
        #[allow(unused_imports)]
        use crate::properties::longhands;
        #[allow(unused_imports)]
        use crate::properties::{LonghandId, LonghandIdSet};
        #[allow(unused_imports)]
        use crate::properties::{CSSWideKeywor