/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

/// Gecko's pseudo-element definition.
///
/// We intentionally double-box legacy ::-moz-tree pseudo-elements to keep the
/// size of PseudoElement (and thus selector components) small.
#[derive(Clone, Debug, Eq, Hash, MallocSizeOf, PartialEq, ToShmem)]
pub enum PseudoElement {
        /// :after
        After,
        /// :before
        Before,
        /// :marker
        Marker,
        /// :backdrop
        Backdrop,
        /// :cue
        Cue,
        /// :first-letter
        FirstLetter,
        /// :first-line
        FirstLine,
        /// :highlight
        Highlight(AtomIdent),
        /// :selection
        Selection,
        /// :target-text
        TargetText,
        /// :view-transition
        ViewTransition,
        /// :view-transition-group
        /// <pt-name-selector> = '*' | <custom-ident>
        ViewTransitionGroup(AtomIdent),
        /// :view-transition-image-pair
        /// <pt-name-selector> = '*' | <custom-ident>
        ViewTransitionImagePair(AtomIdent),
        /// :view-transition-old
        /// <pt-name-selector> = '*' | <custom-ident>
        ViewTransitionOld(AtomIdent),
        /// :view-transition-new
        /// <pt-name-selector> = '*' | <custom-ident>
        ViewTransitionNew(AtomIdent),
        /// :-moz-focus-inner
        MozFocusInner,
        /// :-moz-number-spin-box
        MozNumberSpinBox,
        /// :-moz-number-spin-up
        MozNumberSpinUp,
        /// :-moz-number-spin-down
        MozNumberSpinDown,
        /// :-moz-search-clear-button
        MozSearchClearButton,
        /// :-moz-progress-bar
        MozProgressBar,
        /// :-moz-range-track
        MozRangeTrack,
        /// :-moz-range-progress
        MozRangeProgress,
        /// :-moz-range-thumb
        MozRangeThumb,
        /// :-moz-meter-bar
        MozMeterBar,
        /// :placeholder
        Placeholder,
        /// :-moz-color-swatch
        MozColorSwatch,
        /// :-moz-text-control-editing-root
        MozTextControlEditingRoot,
        /// :-moz-text-control-preview
        MozTextControlPreview,
        /// :-moz-reveal
        MozReveal,
        /// :file-selector-button
        FileSelectorButton,
        /// :slider-track
        SliderTrack,
        /// :slider-thumb
        SliderThumb,
        /// :slider-fill
        SliderFill,
        /// :details-content
        DetailsContent,
        /// :-moz-oof-placeholder
        OofPlaceholder,
        /// :-moz-hframeset-border
        HorizontalFramesetBorder,
        /// :-moz-vframeset-border
        VerticalFramesetBorder,
        /// :-moz-frameset-blank
        FramesetBlank,
        /// :-moz-table-column-group
        TableColGroup,
        /// :-moz-table-column
        TableCol,
        /// :-moz-page
        Page,
        /// :-moz-page-break
        PageBreak,
        /// :-moz-page-content
        PageContent,
        /// :-moz-printed-sheet
        PrintedSheet,
        /// :-moz-column-span-wrapper
        ColumnSpanWrapper,
        /// :-moz-text
        MozText,
        /// :-moz-first-letter-continuation
        FirstLetterContinuation,
        /// :-moz-block-inside-inline-wrapper
        MozBlockInsideInlineWrapper,
        /// :-moz-mathml-anonymous-block
        MozMathMLAnonymousBlock,
        /// :-moz-line-frame
        MozLineFrame,
        /// :-moz-button-content
        ButtonContent,
        /// :-moz-cell-content
        CellContent,
        /// :-moz-dropdown-list
        DropDownList,
        /// :-moz-fieldset-content
        FieldsetContent,
        /// :-moz-html-canvas-content
        HtmlCanvasContent,
        /// :-moz-inline-table
        InlineTable,
        /// :-moz-table
        Table,
        /// :-moz-table-cell
        TableCell,
        /// :-moz-table-wrapper
        TableWrapper,
        /// :-moz-table-row-group
        TableRowGroup,
        /// :-moz-table-row
        TableRow,
        /// :-moz-canvas
        Canvas,
        /// :-moz-page-sequence
        PageSequence,
        /// :-moz-scrolled-content
        ScrolledContent,
        /// :-moz-scrolled-canvas
        ScrolledCanvas,
        /// :-moz-column-set
        ColumnSet,
        /// :-moz-column-content
        ColumnContent,
        /// :-moz-viewport
        Viewport,
        /// :-moz-viewport-scroll
        ViewportScroll,
        /// :-moz-anonymous-item
        AnonymousItem,
        /// :-moz-block-ruby-content
        BlockRubyContent,
        /// :-moz-ruby
        Ruby,
        /// :-moz-ruby-base
        RubyBase,
        /// :-moz-ruby-base-container
        RubyBaseContainer,
        /// :-moz-ruby-text
        RubyText,
        /// :-moz-ruby-text-container
        RubyTextContainer,
        /// :-moz-tree-column
        MozTreeColumn(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-row
        MozTreeRow(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-separator
        MozTreeSeparator(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-cell
        MozTreeCell(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-indentation
        MozTreeIndentation(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-line
        MozTreeLine(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-twisty
        MozTreeTwisty(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-image
        MozTreeImage(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-cell-text
        MozTreeCellText(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-checkbox
        MozTreeCheckbox(thin_vec::ThinVec<Atom>),
        /// :-moz-tree-drop-feedback
        MozTreeDropFeedback(thin_vec::ThinVec<Atom>),
        /// :-moz-svg-marker-anon-child
        MozSVGMarkerAnonChild,
        /// :-moz-svg-outer-svg-anon-child
        MozSVGOuterSVGAnonChild,
        /// :-moz-svg-foreign-content
        MozSVGForeignContent,
        /// :-moz-svg-text
        MozSVGText,
    /// ::-webkit-* that we don't recognize
    /// https://github.com/whatwg/compat/issues/103
    UnknownWebkit(Atom),
}

/// Important: If you change this, you should also update Gecko's
/// nsCSSPseudoElements::IsEagerlyCascadedInServo.




/// The number of eager pseudo-elements.
pub const EAGER_PSEUDO_COUNT: usize = 4;

/// The number of non-functional pseudo-elements.
pub const SIMPLE_PSEUDO_COUNT: usize = 76;

/// The number of tree pseudo-elements.
pub const TREE_PSEUDO_COUNT: usize = 11;

/// The number of all pseudo-elements.
pub const PSEUDO_COUNT: usize = 92;

/// The list of eager pseudos.
pub const EAGER_PSEUDOS: [PseudoElement; EAGER_PSEUDO_COUNT] = [
    PseudoElement::Before,
    PseudoElement::After,
    PseudoElement::FirstLine,
    PseudoElement::FirstLetter,
];



impl PseudoElement {
    /// Returns an index of the pseudo-element.
    #[inline]
    pub fn index(&self) -> usize {
        match *self {
            PseudoElement::After => 0,
            PseudoElement::Before => 1,
            PseudoElement::Marker => 2,
            PseudoElement::Backdrop => 3,
            PseudoElement::Cue => 4,
            PseudoElement::FirstLetter => 5,
            PseudoElement::FirstLine => 6,
            PseudoElement::Highlight(..) => 7,
            PseudoElement::Selection => 8,
            PseudoElement::TargetText => 9,
            PseudoElement::ViewTransition => 10,
            PseudoElement::ViewTransitionGroup(..) => 11,
            PseudoElement::ViewTransitionImagePair(..) => 12,
            PseudoElement::ViewTransitionOld(..) => 13,
            PseudoElement::ViewTransitionNew(..) => 14,
            PseudoElement::MozFocusInner => 15,
            PseudoElement::MozNumberSpinBox => 16,
            PseudoElement::MozNumberSpinUp => 17,
            PseudoElement::MozNumberSpinDown => 18,
            PseudoElement::MozSearchClearButton => 19,
            PseudoElement::MozProgressBar => 20,
            PseudoElement::MozRangeTrack => 21,
            PseudoElement::MozRangeProgress => 22,
            PseudoElement::MozRangeThumb => 23,
            PseudoElement::MozMeterBar => 24,
            PseudoElement::Placeholder => 25,
            PseudoElement::MozColorSwatch => 26,
            PseudoElement::MozTextControlEditingRoot => 27,
            PseudoElement::MozTextControlPreview => 28,
            PseudoElement::MozReveal => 29,
            PseudoElement::FileSelectorButton => 30,
            PseudoElement::SliderTrack => 31,
            PseudoElement::SliderThumb => 32,
            PseudoElement::SliderFill => 33,
            PseudoElement::DetailsContent => 34,
            PseudoElement::OofPlaceholder => 35,
            PseudoElement::HorizontalFramesetBorder => 36,
            PseudoElement::VerticalFramesetBorder => 37,
            PseudoElement::FramesetBlank => 38,
            PseudoElement::TableColGroup => 39,
            PseudoElement::TableCol => 40,
            PseudoElement::Page => 41,
            PseudoElement::PageBreak => 42,
            PseudoElement::PageContent => 43,
            PseudoElement::PrintedSheet => 44,
            PseudoElement::ColumnSpanWrapper => 45,
            PseudoElement::MozText => 46,
            PseudoElement::FirstLetterContinuation => 47,
            PseudoElement::MozBlockInsideInlineWrapper => 48,
            PseudoElement::MozMathMLAnonymousBlock => 49,
            PseudoElement::MozLineFrame => 50,
            PseudoElement::ButtonContent => 51,
            PseudoElement::CellContent => 52,
            PseudoElement::DropDownList => 53,
            PseudoElement::FieldsetContent => 54,
            PseudoElement::HtmlCanvasContent => 55,
            PseudoElement::InlineTable => 56,
            PseudoElement::Table => 57,
            PseudoElement::TableCell => 58,
            PseudoElement::TableWrapper => 59,
            PseudoElement::TableRowGroup => 60,
            PseudoElement::TableRow => 61,
            PseudoElement::Canvas => 62,
            PseudoElement::PageSequence => 63,
            PseudoElement::ScrolledContent => 64,
            PseudoElement::ScrolledCanvas => 65,
            PseudoElement::ColumnSet => 66,
            PseudoElement::ColumnContent => 67,
            PseudoElement::Viewport => 68,
            PseudoElement::ViewportScroll => 69,
            PseudoElement::AnonymousItem => 70,
            PseudoElement::BlockRubyContent => 71,
            PseudoElement::Ruby => 72,
            PseudoElement::RubyBase => 73,
            PseudoElement::RubyBaseContainer => 74,
            PseudoElement::RubyText => 75,
            PseudoElement::RubyTextContainer => 76,
            PseudoElement::MozTreeColumn(..) => 77,
            PseudoElement::MozTreeRow(..) => 78,
            PseudoElement::MozTreeSeparator(..) => 79,
            PseudoElement::MozTreeCell(..) => 80,
            PseudoElement::MozTreeIndentation(..) => 81,
            PseudoElement::MozTreeLine(..) => 82,
            PseudoElement::MozTreeTwisty(..) => 83,
            PseudoElement::MozTreeImage(..) => 84,
            PseudoElement::MozTreeCellText(..) => 85,
            PseudoElement::MozTreeCheckbox(..) => 86,
            PseudoElement::MozTreeDropFeedback(..) => 87,
            PseudoElement::MozSVGMarkerAnonChild => 88,
            PseudoElement::MozSVGOuterSVGAnonChild => 89,
            PseudoElement::MozSVGForeignContent => 90,
            PseudoElement::MozSVGText => 91,
            PseudoElement::UnknownWebkit(..) => unreachable!(),
        }
    }

    /// Returns an array of `None` values.
    ///
    /// FIXME(emilio): Integer generics can't come soon enough.
    pub fn pseudo_none_array<T>() -> [Option<T>; PSEUDO_COUNT] {
        [
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        ]
    }

    /// Whether this pseudo-element is an anonymous box.
    #[inline]
    pub fn is_anon_box(&self) -> bool {
        match *self {
                    PseudoElement::OofPlaceholder => true,
                    PseudoElement::HorizontalFramesetBorder => true,
                    PseudoElement::VerticalFramesetBorder => true,
                    PseudoElement::FramesetBlank => true,
                    PseudoElement::TableColGroup => true,
                    PseudoElement::TableCol => true,
                    PseudoElement::Page => true,
                    PseudoElement::PageBreak => true,
                    PseudoElement::PageContent => true,
                    PseudoElement::PrintedSheet => true,
                    PseudoElement::ColumnSpanWrapper => true,
                    PseudoElement::MozText => true,
                    PseudoElement::FirstLetterContinuation => true,
                    PseudoElement::MozBlockInsideInlineWrapper => true,
                    PseudoElement::MozMathMLAnonymousBlock => true,
                    PseudoElement::MozLineFrame => true,
                    PseudoElement::ButtonContent => true,
                    PseudoElement::CellContent => true,
                    PseudoElement::DropDownList => true,
                    PseudoElement::FieldsetContent => true,
                    PseudoElement::HtmlCanvasContent => true,
                    PseudoElement::InlineTable => true,
                    PseudoElement::Table => true,
                    PseudoElement::TableCell => true,
                    PseudoElement::TableWrapper => true,
                    PseudoElement::TableRowGroup => true,
                    PseudoElement::TableRow => true,
                    PseudoElement::Canvas => true,
                    PseudoElement::PageSequence => true,
                    PseudoElement::ScrolledContent => true,
                    PseudoElement::ScrolledCanvas => true,
                    PseudoElement::ColumnSet => true,
                    PseudoElement::ColumnContent => true,
                    PseudoElement::Viewport => true,
                    PseudoElement::ViewportScroll => true,
                    PseudoElement::AnonymousItem => true,
                    PseudoElement::BlockRubyContent => true,
                    PseudoElement::Ruby => true,
                    PseudoElement::RubyBase => true,
                    PseudoElement::RubyBaseContainer => true,
                    PseudoElement::RubyText => true,
                    PseudoElement::RubyTextContainer => true,
                    PseudoElement::MozSVGMarkerAnonChild => true,
                    PseudoElement::MozSVGOuterSVGAnonChild => true,
                    PseudoElement::MozSVGForeignContent => true,
                    PseudoElement::MozSVGText => true,
            _ => false,
        }
    }

    /// Whether this pseudo-element is eagerly-cascaded.
    #[inline]
    pub fn is_eager(&self) -> bool {
        matches!(*self,
                 PseudoElement::Before | PseudoElement::After | PseudoElement::FirstLine | PseudoElement::FirstLetter)
    }

    /// Whether this pseudo-element is tree pseudo-element.
    #[inline]
    pub fn is_tree_pseudo_element(&self) -> bool {
        match *self {
            PseudoElement::MozTreeColumn(..) => true,
            PseudoElement::MozTreeRow(..) => true,
            PseudoElement::MozTreeSeparator(..) => true,
            PseudoElement::MozTreeCell(..) => true,
            PseudoElement::MozTreeIndentation(..) => true,
            PseudoElement::MozTreeLine(..) => true,
            PseudoElement::MozTreeTwisty(..) => true,
            PseudoElement::MozTreeImage(..) => true,
            PseudoElement::MozTreeCellText(..) => true,
            PseudoElement::MozTreeCheckbox(..) => true,
            PseudoElement::MozTreeDropFeedback(..) => true,
            _ => false,
        }
    }

    /// Whether this pseudo-element is an unknown Webkit-prefixed pseudo-element.
    #[inline]
    pub fn is_unknown_webkit_pseudo_element(&self) -> bool {
        matches!(*self, PseudoElement::UnknownWebkit(..))
    }

    /// Gets the flags associated to this pseudo-element, or 0 if it's an
    /// anonymous box.
    pub fn flags(&self) -> u32 {
        match *self {
                PseudoElement::After =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_after,
                PseudoElement::Before =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_before,
                PseudoElement::Marker =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_marker,
                PseudoElement::Backdrop =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_backdrop,
                PseudoElement::Cue =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_cue,
                PseudoElement::FirstLetter =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_firstLetter,
                PseudoElement::FirstLine =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_firstLine,
                PseudoElement::Highlight(..) =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_highlight,
                PseudoElement::Selection =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_selection,
                PseudoElement::TargetText =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_targetText,
                PseudoElement::ViewTransition =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_viewTransition,
                PseudoElement::ViewTransitionGroup(..) =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_viewTransitionGroup,
                PseudoElement::ViewTransitionImagePair(..) =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_viewTransitionImagePair,
                PseudoElement::ViewTransitionOld(..) =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_viewTransitionOld,
                PseudoElement::ViewTransitionNew(..) =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_viewTransitionNew,
                PseudoElement::MozFocusInner =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozFocusInner,
                PseudoElement::MozNumberSpinBox =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozNumberSpinBox,
                PseudoElement::MozNumberSpinUp =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozNumberSpinUp,
                PseudoElement::MozNumberSpinDown =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozNumberSpinDown,
                PseudoElement::MozSearchClearButton =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozSearchClearButton,
                PseudoElement::MozProgressBar =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozProgressBar,
                PseudoElement::MozRangeTrack =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozRangeTrack,
                PseudoElement::MozRangeProgress =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozRangeProgress,
                PseudoElement::MozRangeThumb =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozRangeThumb,
                PseudoElement::MozMeterBar =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozMeterBar,
                PseudoElement::Placeholder =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_placeholder,
                PseudoElement::MozColorSwatch =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozColorSwatch,
                PseudoElement::MozTextControlEditingRoot =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozTextControlEditingRoot,
                PseudoElement::MozTextControlPreview =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozTextControlPreview,
                PseudoElement::MozReveal =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_mozReveal,
                PseudoElement::FileSelectorButton =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_fileSelectorButton,
                PseudoElement::SliderTrack =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_sliderTrack,
                PseudoElement::SliderThumb =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_sliderThumb,
                PseudoElement::SliderFill =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_sliderFill,
                PseudoElement::DetailsContent =>
                    structs::SERVO_CSS_PSEUDO_ELEMENT_FLAGS_detailsContent,
                PseudoElement::OofPlaceholder =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::HorizontalFramesetBorder =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::VerticalFramesetBorder =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::FramesetBlank =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::TableColGroup =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::TableCol =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::Page =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::PageBreak =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::PageContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::PrintedSheet =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::ColumnSpanWrapper =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::MozText =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::FirstLetterContinuation =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::MozBlockInsideInlineWrapper =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::MozMathMLAnonymousBlock =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::MozLineFrame =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::ButtonContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::CellContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::DropDownList =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::FieldsetContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::HtmlCanvasContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::InlineTable =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::Table =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::TableCell =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::TableWrapper =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::TableRowGroup =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::TableRow =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::Canvas =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::PageSequence =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::ScrolledContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::ScrolledCanvas =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::ColumnSet =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::ColumnContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::Viewport =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::ViewportScroll =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::AnonymousItem =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::BlockRubyContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::Ruby =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::RubyBase =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::RubyBaseContainer =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::RubyText =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::RubyTextContainer =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::MozTreeColumn(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeRow(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeSeparator(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeCell(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeIndentation(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeLine(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeTwisty(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeImage(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeCellText(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeCheckbox(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozTreeDropFeedback(..) =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS_AND_CHROME,
                PseudoElement::MozSVGMarkerAnonChild =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::MozSVGOuterSVGAnonChild =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::MozSVGForeignContent =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
                PseudoElement::MozSVGText =>
                    structs::CSS_PSEUDO_ELEMENT_ENABLED_IN_UA_SHEETS,
            PseudoElement::UnknownWebkit(..) => 0,
        }
    }

    /// Construct a pseudo-element from a `PseudoStyleType`.
    #[inline]
    pub fn from_pseudo_type(type_: PseudoStyleType, functional_pseudo_parameter: Option<AtomIdent>) -> Option<Self> {
        match type_ {
                PseudoStyleType::after => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::After)
                },
                PseudoStyleType::before => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Before)
                },
                PseudoStyleType::marker => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Marker)
                },
                PseudoStyleType::backdrop => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Backdrop)
                },
                PseudoStyleType::cue => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Cue)
                },
                PseudoStyleType::firstLetter => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::FirstLetter)
                },
                PseudoStyleType::firstLine => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::FirstLine)
                },
                PseudoStyleType::selection => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Selection)
                },
                PseudoStyleType::targetText => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::TargetText)
                },
                PseudoStyleType::viewTransition => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::ViewTransition)
                },
                PseudoStyleType::viewTransitionGroup => {
                    functional_pseudo_parameter.map(PseudoElement::ViewTransitionGroup)
                },
                PseudoStyleType::viewTransitionImagePair => {
                    functional_pseudo_parameter.map(PseudoElement::ViewTransitionImagePair)
                },
                PseudoStyleType::viewTransitionOld => {
                    functional_pseudo_parameter.map(PseudoElement::ViewTransitionOld)
                },
                PseudoStyleType::viewTransitionNew => {
                    functional_pseudo_parameter.map(PseudoElement::ViewTransitionNew)
                },
                PseudoStyleType::mozFocusInner => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozFocusInner)
                },
                PseudoStyleType::mozNumberSpinBox => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozNumberSpinBox)
                },
                PseudoStyleType::mozNumberSpinUp => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozNumberSpinUp)
                },
                PseudoStyleType::mozNumberSpinDown => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozNumberSpinDown)
                },
                PseudoStyleType::mozSearchClearButton => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozSearchClearButton)
                },
                PseudoStyleType::mozProgressBar => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozProgressBar)
                },
                PseudoStyleType::mozRangeTrack => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozRangeTrack)
                },
                PseudoStyleType::mozRangeProgress => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozRangeProgress)
                },
                PseudoStyleType::mozRangeThumb => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozRangeThumb)
                },
                PseudoStyleType::mozMeterBar => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozMeterBar)
                },
                PseudoStyleType::placeholder => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Placeholder)
                },
                PseudoStyleType::mozColorSwatch => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozColorSwatch)
                },
                PseudoStyleType::mozTextControlEditingRoot => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozTextControlEditingRoot)
                },
                PseudoStyleType::mozTextControlPreview => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozTextControlPreview)
                },
                PseudoStyleType::mozReveal => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozReveal)
                },
                PseudoStyleType::fileSelectorButton => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::FileSelectorButton)
                },
                PseudoStyleType::sliderTrack => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::SliderTrack)
                },
                PseudoStyleType::sliderThumb => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::SliderThumb)
                },
                PseudoStyleType::sliderFill => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::SliderFill)
                },
                PseudoStyleType::detailsContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::DetailsContent)
                },
                PseudoStyleType::oofPlaceholder => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::OofPlaceholder)
                },
                PseudoStyleType::horizontalFramesetBorder => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::HorizontalFramesetBorder)
                },
                PseudoStyleType::verticalFramesetBorder => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::VerticalFramesetBorder)
                },
                PseudoStyleType::framesetBlank => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::FramesetBlank)
                },
                PseudoStyleType::tableColGroup => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::TableColGroup)
                },
                PseudoStyleType::tableCol => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::TableCol)
                },
                PseudoStyleType::page => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Page)
                },
                PseudoStyleType::pageBreak => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::PageBreak)
                },
                PseudoStyleType::pageContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::PageContent)
                },
                PseudoStyleType::printedSheet => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::PrintedSheet)
                },
                PseudoStyleType::columnSpanWrapper => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::ColumnSpanWrapper)
                },
                PseudoStyleType::mozText => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozText)
                },
                PseudoStyleType::firstLetterContinuation => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::FirstLetterContinuation)
                },
                PseudoStyleType::mozBlockInsideInlineWrapper => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozBlockInsideInlineWrapper)
                },
                PseudoStyleType::mozMathMLAnonymousBlock => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozMathMLAnonymousBlock)
                },
                PseudoStyleType::mozLineFrame => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozLineFrame)
                },
                PseudoStyleType::buttonContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::ButtonContent)
                },
                PseudoStyleType::cellContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::CellContent)
                },
                PseudoStyleType::dropDownList => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::DropDownList)
                },
                PseudoStyleType::fieldsetContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::FieldsetContent)
                },
                PseudoStyleType::htmlCanvasContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::HtmlCanvasContent)
                },
                PseudoStyleType::inlineTable => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::InlineTable)
                },
                PseudoStyleType::table => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Table)
                },
                PseudoStyleType::tableCell => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::TableCell)
                },
                PseudoStyleType::tableWrapper => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::TableWrapper)
                },
                PseudoStyleType::tableRowGroup => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::TableRowGroup)
                },
                PseudoStyleType::tableRow => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::TableRow)
                },
                PseudoStyleType::canvas => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Canvas)
                },
                PseudoStyleType::pageSequence => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::PageSequence)
                },
                PseudoStyleType::scrolledContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::ScrolledContent)
                },
                PseudoStyleType::scrolledCanvas => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::ScrolledCanvas)
                },
                PseudoStyleType::columnSet => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::ColumnSet)
                },
                PseudoStyleType::columnContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::ColumnContent)
                },
                PseudoStyleType::viewport => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Viewport)
                },
                PseudoStyleType::viewportScroll => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::ViewportScroll)
                },
                PseudoStyleType::anonymousItem => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::AnonymousItem)
                },
                PseudoStyleType::blockRubyContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::BlockRubyContent)
                },
                PseudoStyleType::ruby => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::Ruby)
                },
                PseudoStyleType::rubyBase => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::RubyBase)
                },
                PseudoStyleType::rubyBaseContainer => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::RubyBaseContainer)
                },
                PseudoStyleType::rubyText => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::RubyText)
                },
                PseudoStyleType::rubyTextContainer => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::RubyTextContainer)
                },
                PseudoStyleType::mozSVGMarkerAnonChild => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozSVGMarkerAnonChild)
                },
                PseudoStyleType::mozSVGOuterSVGAnonChild => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozSVGOuterSVGAnonChild)
                },
                PseudoStyleType::mozSVGForeignContent => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozSVGForeignContent)
                },
                PseudoStyleType::mozSVGText => {
                    debug_assert!(functional_pseudo_parameter.is_none());
                    Some(PseudoElement::MozSVGText)
                },
            PseudoStyleType::highlight => {
                match functional_pseudo_parameter {
                    Some(p) => Some(PseudoElement::Highlight(p)),
                    None => None
                }
            }
            _ => None,
        }
    }

    /// Construct a `PseudoStyleType` from a pseudo-element
    #[inline]
    pub fn pseudo_type(&self) -> PseudoStyleType {
        match *self {
                PseudoElement::After => PseudoStyleType::after,
                PseudoElement::Before => PseudoStyleType::before,
                PseudoElement::Marker => PseudoStyleType::marker,
                PseudoElement::Backdrop => PseudoStyleType::backdrop,
                PseudoElement::Cue => PseudoStyleType::cue,
                PseudoElement::FirstLetter => PseudoStyleType::firstLetter,
                PseudoElement::FirstLine => PseudoStyleType::firstLine,
                PseudoElement::Highlight(..) => PseudoStyleType::highlight,
                PseudoElement::Selection => PseudoStyleType::selection,
                PseudoElement::TargetText => PseudoStyleType::targetText,
                PseudoElement::ViewTransition => PseudoStyleType::viewTransition,
                PseudoElement::ViewTransitionGroup(..) => PseudoStyleType::viewTransitionGroup,
                PseudoElement::ViewTransitionImagePair(..) => PseudoStyleType::viewTransitionImagePair,
                PseudoElement::ViewTransitionOld(..) => PseudoStyleType::viewTransitionOld,
                PseudoElement::ViewTransitionNew(..) => PseudoStyleType::viewTransitionNew,
                PseudoElement::MozFocusInner => PseudoStyleType::mozFocusInner,
                PseudoElement::MozNumberSpinBox => PseudoStyleType::mozNumberSpinBox,
                PseudoElement::MozNumberSpinUp => PseudoStyleType::mozNumberSpinUp,
                PseudoElement::MozNumberSpinDown => PseudoStyleType::mozNumberSpinDown,
                PseudoElement::MozSearchClearButton => PseudoStyleType::mozSearchClearButton,
                PseudoElement::MozProgressBar => PseudoStyleType::mozProgressBar,
                PseudoElement::MozRangeTrack => PseudoStyleType::mozRangeTrack,
                PseudoElement::MozRangeProgress => PseudoStyleType::mozRangeProgress,
                PseudoElement::MozRangeThumb => PseudoStyleType::mozRangeThumb,
                PseudoElement::MozMeterBar => PseudoStyleType::mozMeterBar,
                PseudoElement::Placeholder => PseudoStyleType::placeholder,
                PseudoElement::MozColorSwatch => PseudoStyleType::mozColorSwatch,
                PseudoElement::MozTextControlEditingRoot => PseudoStyleType::mozTextControlEditingRoot,
                PseudoElement::MozTextControlPreview => PseudoStyleType::mozTextControlPreview,
                PseudoElement::MozReveal => PseudoStyleType::mozReveal,
                PseudoElement::FileSelectorButton => PseudoStyleType::fileSelectorButton,
                PseudoElement::SliderTrack => PseudoStyleType::sliderTrack,
                PseudoElement::SliderThumb => PseudoStyleType::sliderThumb,
                PseudoElement::SliderFill => PseudoStyleType::sliderFill,
                PseudoElement::DetailsContent => PseudoStyleType::detailsContent,
                PseudoElement::OofPlaceholder => PseudoStyleType::oofPlaceholder,
                PseudoElement::HorizontalFramesetBorder => PseudoStyleType::horizontalFramesetBorder,
                PseudoElement::VerticalFramesetBorder => PseudoStyleType::verticalFramesetBorder,
                PseudoElement::FramesetBlank => PseudoStyleType::framesetBlank,
                PseudoElement::TableColGroup => PseudoStyleType::tableColGroup,
                PseudoElement::TableCol => PseudoStyleType::tableCol,
                PseudoElement::Page => PseudoStyleType::page,
                PseudoElement::PageBreak => PseudoStyleType::pageBreak,
                PseudoElement::PageContent => PseudoStyleType::pageContent,
                PseudoElement::PrintedSheet => PseudoStyleType::printedSheet,
                PseudoElement::ColumnSpanWrapper => PseudoStyleType::columnSpanWrapper,
                PseudoElement::MozText => PseudoStyleType::mozText,
                PseudoElement::FirstLetterContinuation => PseudoStyleType::firstLetterContinuation,
                PseudoElement::MozBlockInsideInlineWrapper => PseudoStyleType::mozBlockInsideInlineWrapper,
                PseudoElement::MozMathMLAnonymousBlock => PseudoStyleType::mozMathMLAnonymousBlock,
                PseudoElement::MozLineFrame => PseudoStyleType::mozLineFrame,
                PseudoElement::ButtonContent => PseudoStyleType::buttonContent,
                PseudoElement::CellContent => PseudoStyleType::cellContent,
                PseudoElement::DropDownList => PseudoStyleType::dropDownList,
                PseudoElement::FieldsetContent => PseudoStyleType::fieldsetContent,
                PseudoElement::HtmlCanvasContent => PseudoStyleType::htmlCanvasContent,
                PseudoElement::InlineTable => PseudoStyleType::inlineTable,
                PseudoElement::Table => PseudoStyleType::table,
                PseudoElement::TableCell => PseudoStyleType::tableCell,
                PseudoElement::TableWrapper => PseudoStyleType::tableWrapper,
                PseudoElement::TableRowGroup => PseudoStyleType::tableRowGroup,
                PseudoElement::TableRow => PseudoStyleType::tableRow,
                PseudoElement::Canvas => PseudoStyleType::canvas,
                PseudoElement::PageSequence => PseudoStyleType::pageSequence,
                PseudoElement::ScrolledContent => PseudoStyleType::scrolledContent,
                PseudoElement::ScrolledCanvas => PseudoStyleType::scrolledCanvas,
                PseudoElement::ColumnSet => PseudoStyleType::columnSet,
                PseudoElement::ColumnContent => PseudoStyleType::columnContent,
                PseudoElement::Viewport => PseudoStyleType::viewport,
                PseudoElement::ViewportScroll => PseudoStyleType::viewportScroll,
                PseudoElement::AnonymousItem => PseudoStyleType::anonymousItem,
                PseudoElement::BlockRubyContent => PseudoStyleType::blockRubyContent,
                PseudoElement::Ruby => PseudoStyleType::ruby,
                PseudoElement::RubyBase => PseudoStyleType::rubyBase,
                PseudoElement::RubyBaseContainer => PseudoStyleType::rubyBaseContainer,
                PseudoElement::RubyText => PseudoStyleType::rubyText,
                PseudoElement::RubyTextContainer => PseudoStyleType::rubyTextContainer,
                PseudoElement::MozTreeColumn(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeRow(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeSeparator(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeCell(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeIndentation(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeLine(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeTwisty(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeImage(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeCellText(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeCheckbox(..) => PseudoStyleType::XULTree,
                PseudoElement::MozTreeDropFeedback(..) => PseudoStyleType::XULTree,
                PseudoElement::MozSVGMarkerAnonChild => PseudoStyleType::mozSVGMarkerAnonChild,
                PseudoElement::MozSVGOuterSVGAnonChild => PseudoStyleType::mozSVGOuterSVGAnonChild,
                PseudoElement::MozSVGForeignContent => PseudoStyleType::mozSVGForeignContent,
                PseudoElement::MozSVGText => PseudoStyleType::mozSVGText,
            PseudoElement::UnknownWebkit(..) => unreachable!(),
        }
    }

    /// Get the argument list of a tree pseudo-element.
    #[inline]
    pub fn tree_pseudo_args(&self) -> Option<&[Atom]> {
        match *self {
            PseudoElement::MozTreeColumn(ref args) => Some(args),
            PseudoElement::MozTreeRow(ref args) => Some(args),
            PseudoElement::MozTreeSeparator(ref args) => Some(args),
            PseudoElement::MozTreeCell(ref args) => Some(args),
            PseudoElement::MozTreeIndentation(ref args) => Some(args),
            PseudoElement::MozTreeLine(ref args) => Some(args),
            PseudoElement::MozTreeTwisty(ref args) => Some(args),
            PseudoElement::MozTreeImage(ref args) => Some(args),
            PseudoElement::MozTreeCellText(ref args) => Some(args),
            PseudoElement::MozTreeCheckbox(ref args) => Some(args),
            PseudoElement::MozTreeDropFeedback(ref args) => Some(args),
            _ => None,
        }
    }

    /// Construct a tree pseudo-element from atom and args.
    #[inline]
    pub fn from_tree_pseudo_atom(atom: &Atom, args: Box<[Atom]>) -> Option<Self> {
            if atom == &atom!(":-moz-tree-column") {
                return Some(PseudoElement::MozTreeColumn(args.into()));
            }
            if atom == &atom!(":-moz-tree-row") {
                return Some(PseudoElement::MozTreeRow(args.into()));
            }
            if atom == &atom!(":-moz-tree-separator") {
                return Some(PseudoElement::MozTreeSeparator(args.into()));
            }
            if atom == &atom!(":-moz-tree-cell") {
                return Some(PseudoElement::MozTreeCell(args.into()));
            }
            if atom == &atom!(":-moz-tree-indentation") {
                return Some(PseudoElement::MozTreeIndentation(args.into()));
            }
            if atom == &atom!(":-moz-tree-line") {
                return Some(PseudoElement::MozTreeLine(args.into()));
            }
            if atom == &atom!(":-moz-tree-twisty") {
                return Some(PseudoElement::MozTreeTwisty(args.into()));
            }
            if atom == &atom!(":-moz-tree-image") {
                return Some(PseudoElement::MozTreeImage(args.into()));
            }
            if atom == &atom!(":-moz-tree-cell-text") {
                return Some(PseudoElement::MozTreeCellText(args.into()));
            }
            if atom == &atom!(":-moz-tree-checkbox") {
                return Some(PseudoElement::MozTreeCheckbox(args.into()));
            }
            if atom == &atom!(":-moz-tree-drop-feedback") {
                return Some(PseudoElement::MozTreeDropFeedback(args.into()));
            }
        None
    }

    /// Constructs a pseudo-element from a string of text.
    ///
    /// Returns `None` if the pseudo-element is not recognised.
    #[inline]
    pub fn from_slice(name: &str, allow_unkown_webkit: bool) -> Option<Self> {
        // We don't need to support tree pseudos because functional
        // pseudo-elements needs arguments, and thus should be created
        // via other methods.
        ascii_case_insensitive_phf_map! {
            pseudo -> PseudoElement = {
                "after" => PseudoElement::After,
                "before" => PseudoElement::Before,
                "marker" => PseudoElement::Marker,
                "backdrop" => PseudoElement::Backdrop,
                "cue" => PseudoElement::Cue,
                "first-letter" => PseudoElement::FirstLetter,
                "first-line" => PseudoElement::FirstLine,
                "selection" => PseudoElement::Selection,
                "target-text" => PseudoElement::TargetText,
                "view-transition" => PseudoElement::ViewTransition,
                "-moz-focus-inner" => PseudoElement::MozFocusInner,
                "-moz-number-spin-box" => PseudoElement::MozNumberSpinBox,
                "-moz-number-spin-up" => PseudoElement::MozNumberSpinUp,
                "-moz-number-spin-down" => PseudoElement::MozNumberSpinDown,
                "-moz-search-clear-button" => PseudoElement::MozSearchClearButton,
                "-moz-progress-bar" => PseudoElement::MozProgressBar,
                "-moz-range-track" => PseudoElement::MozRangeTrack,
                "-moz-range-progress" => PseudoElement::MozRangeProgress,
                "-moz-range-thumb" => PseudoElement::MozRangeThumb,
                "-moz-meter-bar" => PseudoElement::MozMeterBar,
                "placeholder" => PseudoElement::Placeholder,
                "-moz-color-swatch" => PseudoElement::MozColorSwatch,
                "-moz-text-control-editing-root" => PseudoElement::MozTextControlEditingRoot,
                "-moz-text-control-preview" => PseudoElement::MozTextControlPreview,
                "-moz-reveal" => PseudoElement::MozReveal,
                "file-selector-button" => PseudoElement::FileSelectorButton,
                "slider-track" => PseudoElement::SliderTrack,
                "slider-thumb" => PseudoElement::SliderThumb,
                "slider-fill" => PseudoElement::SliderFill,
                "details-content" => PseudoElement::DetailsContent,
                "-moz-oof-placeholder" => PseudoElement::OofPlaceholder,
                "-moz-hframeset-border" => PseudoElement::HorizontalFramesetBorder,
                "-moz-vframeset-border" => PseudoElement::VerticalFramesetBorder,
                "-moz-frameset-blank" => PseudoElement::FramesetBlank,
                "-moz-table-column-group" => PseudoElement::TableColGroup,
                "-moz-table-column" => PseudoElement::TableCol,
                "-moz-page" => PseudoElement::Page,
                "-moz-page-break" => PseudoElement::PageBreak,
                "-moz-page-content" => PseudoElement::PageContent,
                "-moz-printed-sheet" => PseudoElement::PrintedSheet,
                "-moz-column-span-wrapper" => PseudoElement::ColumnSpanWrapper,
                "-moz-text" => PseudoElement::MozText,
                "-moz-first-letter-continuation" => PseudoElement::FirstLetterContinuation,
                "-moz-block-inside-inline-wrapper" => PseudoElement::MozBlockInsideInlineWrapper,
                "-moz-mathml-anonymous-block" => PseudoElement::MozMathMLAnonymousBlock,
                "-moz-line-frame" => PseudoElement::MozLineFrame,
                "-moz-button-content" => PseudoElement::ButtonContent,
                "-moz-cell-content" => PseudoElement::CellContent,
                "-moz-dropdown-list" => PseudoElement::DropDownList,
                "-moz-fieldset-content" => PseudoElement::FieldsetContent,
                "-moz-html-canvas-content" => PseudoElement::HtmlCanvasContent,
                "-moz-inline-table" => PseudoElement::InlineTable,
                "-moz-table" => PseudoElement::Table,
                "-moz-table-cell" => PseudoElement::TableCell,
                "-moz-table-wrapper" => PseudoElement::TableWrapper,
                "-moz-table-row-group" => PseudoElement::TableRowGroup,
                "-moz-table-row" => PseudoElement::TableRow,
                "-moz-canvas" => PseudoElement::Canvas,
                "-moz-page-sequence" => PseudoElement::PageSequence,
                "-moz-scrolled-content" => PseudoElement::ScrolledContent,
                "-moz-scrolled-canvas" => PseudoElement::ScrolledCanvas,
                "-moz-column-set" => PseudoElement::ColumnSet,
                "-moz-column-content" => PseudoElement::ColumnContent,
                "-moz-viewport" => PseudoElement::Viewport,
                "-moz-viewport-scroll" => PseudoElement::ViewportScroll,
                "-moz-anonymous-item" => PseudoElement::AnonymousItem,
                "-moz-block-ruby-content" => PseudoElement::BlockRubyContent,
                "-moz-ruby" => PseudoElement::Ruby,
                "-moz-ruby-base" => PseudoElement::RubyBase,
                "-moz-ruby-base-container" => PseudoElement::RubyBaseContainer,
                "-moz-ruby-text" => PseudoElement::RubyText,
                "-moz-ruby-text-container" => PseudoElement::RubyTextContainer,
                "-moz-svg-marker-anon-child" => PseudoElement::MozSVGMarkerAnonChild,
                "-moz-svg-outer-svg-anon-child" => PseudoElement::MozSVGOuterSVGAnonChild,
                "-moz-svg-foreign-content" => PseudoElement::MozSVGForeignContent,
                "-moz-svg-text" => PseudoElement::MozSVGText,
                // Alias some legacy prefixed pseudos to their standardized name at parse time:
                "-moz-selection" => PseudoElement::Selection,
                "-moz-placeholder" => PseudoElement::Placeholder,
                "-moz-list-bullet" => PseudoElement::Marker,
                "-moz-list-number" => PseudoElement::Marker,
            }
        }
        if let Some(p) = pseudo::get(name) {
            return Some(p.clone());
        }
        if starts_with_ignore_ascii_case(name, "-moz-tree-") {
            return PseudoElement::tree_pseudo_element(name, Default::default())
        }
        const WEBKIT_PREFIX: &str = "-webkit-";
        if allow_unkown_webkit && starts_with_ignore_ascii_case(name, WEBKIT_PREFIX) {
            let part = string_as_ascii_lowercase(&name[WEBKIT_PREFIX.len()..]);
            return Some(PseudoElement::UnknownWebkit(part.into()));
        }
        None
    }

    /// Constructs a tree pseudo-element from the given name and arguments.
    /// "name" must start with "-moz-tree-".
    ///
    /// Returns `None` if the pseudo-element is not recognized.
    #[inline]
    pub fn tree_pseudo_element(name: &str, args: thin_vec::ThinVec<Atom>) -> Option<Self> {
        debug_assert!(starts_with_ignore_ascii_case(name, "-moz-tree-"));
        let tree_part = &name[10..];
            if tree_part.eq_ignore_ascii_case("column") {
                return Some(PseudoElement::MozTreeColumn(args));
            }
            if tree_part.eq_ignore_ascii_case("row") {
                return Some(PseudoElement::MozTreeRow(args));
            }
            if tree_part.eq_ignore_ascii_case("separator") {
                return Some(PseudoElement::MozTreeSeparator(args));
            }
            if tree_part.eq_ignore_ascii_case("cell") {
                return Some(PseudoElement::MozTreeCell(args));
            }
            if tree_part.eq_ignore_ascii_case("indentation") {
                return Some(PseudoElement::MozTreeIndentation(args));
            }
            if tree_part.eq_ignore_ascii_case("line") {
                return Some(PseudoElement::MozTreeLine(args));
            }
            if tree_part.eq_ignore_ascii_case("twisty") {
                return Some(PseudoElement::MozTreeTwisty(args));
            }
            if tree_part.eq_ignore_ascii_case("image") {
                return Some(PseudoElement::MozTreeImage(args));
            }
            if tree_part.eq_ignore_ascii_case("cell-text") {
                return Some(PseudoElement::MozTreeCellText(args));
            }
            if tree_part.eq_ignore_ascii_case("checkbox") {
                return Some(PseudoElement::MozTreeCheckbox(args));
            }
            if tree_part.eq_ignore_ascii_case("drop-feedback") {
                return Some(PseudoElement::MozTreeDropFeedback(args));
            }
        None
    }

    /// Returns true if this pseudo-element matches the given selector.
    pub fn matches(&self, pseudo_selector: &PseudoElement) -> bool {
        if *self == *pseudo_selector {
            return true;
        }

        if std::mem::discriminant(self) != std::mem::discriminant(pseudo_selector) {
            return false;
        }

        match (self, pseudo_selector) {
            (
                &Self::ViewTransitionGroup(ref _name),
                &Self::ViewTransitionGroup(ref selector_name),
            )
            | (
                &Self::ViewTransitionImagePair(ref _name),
                &Self::ViewTransitionImagePair(ref selector_name),
            )
            | (
                &Self::ViewTransitionOld(ref _name),
                &Self::ViewTransitionOld(ref selector_name),
            )
            | (
                &Self::ViewTransitionNew(ref _name),
                &Self::ViewTransitionNew(ref selector_name),
            ) => {
                // Named view transition pseudos accept the universal selector as the name, so we
                // check it first.
                // https://drafts.csswg.org/css-view-transitions-1/#named-view-transition-pseudo
                if selector_name.0 == atom!("*") {
                    return true;
                }
                // We don't need to check if `*_name == *selector_name` here because we already
                // check if the enums are equal above.
                false
            },
            _ => false,
        }
    }
}

impl ToCss for PseudoElement {
    fn to_css<W>(&self, dest: &mut W) -> fmt::Result where W: fmt::Write {
        dest.write_char(':')?;
        match *self {
                PseudoElement::After => dest.write_str(":after")?,
                PseudoElement::Before => dest.write_str(":before")?,
                PseudoElement::Marker => dest.write_str(":marker")?,
                PseudoElement::Backdrop => dest.write_str(":backdrop")?,
                PseudoElement::Cue => dest.write_str(":cue")?,
                PseudoElement::FirstLetter => dest.write_str(":first-letter")?,
                PseudoElement::FirstLine => dest.write_str(":first-line")?,
                PseudoElement::Selection => dest.write_str(":selection")?,
                PseudoElement::TargetText => dest.write_str(":target-text")?,
                PseudoElement::ViewTransition => dest.write_str(":view-transition")?,
                PseudoElement::ViewTransitionGroup(ref name) => {
                    dest.write_str(":view-transition-group(")?;
                    if name.0 == atom!("*") {
                        // serialize_atom_identifier() may serialize "*" as "\*", so we handle it
                        // separately.
                        dest.write_char('*')?;
                    } else {
                        serialize_atom_identifier(name, dest)?;
                    }
                    dest.write_char(')')?;
                }
                PseudoElement::ViewTransitionImagePair(ref name) => {
                    dest.write_str(":view-transition-image-pair(")?;
                    if name.0 == atom!("*") {
                        // serialize_atom_identifier() may serialize "*" as "\*", so we handle it
                        // separately.
                        dest.write_char('*')?;
                    } else {
                        serialize_atom_identifier(name, dest)?;
                    }
                    dest.write_char(')')?;
                }
                PseudoElement::ViewTransitionOld(ref name) => {
                    dest.write_str(":view-transition-old(")?;
                    if name.0 == atom!("*") {
                        // serialize_atom_identifier() may serialize "*" as "\*", so we handle it
                        // separately.
                        dest.write_char('*')?;
                    } else {
                        serialize_atom_identifier(name, dest)?;
                    }
                    dest.write_char(')')?;
                }
                PseudoElement::ViewTransitionNew(ref name) => {
                    dest.write_str(":view-transition-new(")?;
                    if name.0 == atom!("*") {
                        // serialize_atom_identifier() may serialize "*" as "\*", so we handle it
                        // separately.
                        dest.write_char('*')?;
                    } else {
                        serialize_atom_identifier(name, dest)?;
                    }
                    dest.write_char(')')?;
                }
                PseudoElement::MozFocusInner => dest.write_str(":-moz-focus-inner")?,
                PseudoElement::MozNumberSpinBox => dest.write_str(":-moz-number-spin-box")?,
                PseudoElement::MozNumberSpinUp => dest.write_str(":-moz-number-spin-up")?,
                PseudoElement::MozNumberSpinDown => dest.write_str(":-moz-number-spin-down")?,
                PseudoElement::MozSearchClearButton => dest.write_str(":-moz-search-clear-button")?,
                PseudoElement::MozProgressBar => dest.write_str(":-moz-progress-bar")?,
                PseudoElement::MozRangeTrack => dest.write_str(":-moz-range-track")?,
                PseudoElement::MozRangeProgress => dest.write_str(":-moz-range-progress")?,
                PseudoElement::MozRangeThumb => dest.write_str(":-moz-range-thumb")?,
                PseudoElement::MozMeterBar => dest.write_str(":-moz-meter-bar")?,
                PseudoElement::Placeholder => dest.write_str(":placeholder")?,
                PseudoElement::MozColorSwatch => dest.write_str(":-moz-color-swatch")?,
                PseudoElement::MozTextControlEditingRoot => dest.write_str(":-moz-text-control-editing-root")?,
                PseudoElement::MozTextControlPreview => dest.write_str(":-moz-text-control-preview")?,
                PseudoElement::MozReveal => dest.write_str(":-moz-reveal")?,
                PseudoElement::FileSelectorButton => dest.write_str(":file-selector-button")?,
                PseudoElement::SliderTrack => dest.write_str(":slider-track")?,
                PseudoElement::SliderThumb => dest.write_str(":slider-thumb")?,
                PseudoElement::SliderFill => dest.write_str(":slider-fill")?,
                PseudoElement::DetailsContent => dest.write_str(":details-content")?,
                PseudoElement::OofPlaceholder => dest.write_str(":-moz-oof-placeholder")?,
                PseudoElement::HorizontalFramesetBorder => dest.write_str(":-moz-hframeset-border")?,
                PseudoElement::VerticalFramesetBorder => dest.write_str(":-moz-vframeset-border")?,
                PseudoElement::FramesetBlank => dest.write_str(":-moz-frameset-blank")?,
                PseudoElement::TableColGroup => dest.write_str(":-moz-table-column-group")?,
                PseudoElement::TableCol => dest.write_str(":-moz-table-column")?,
                PseudoElement::Page => dest.write_str(":-moz-page")?,
                PseudoElement::PageBreak => dest.write_str(":-moz-page-break")?,
                PseudoElement::PageContent => dest.write_str(":-moz-page-content")?,
                PseudoElement::PrintedSheet => dest.write_str(":-moz-printed-sheet")?,
                PseudoElement::ColumnSpanWrapper => dest.write_str(":-moz-column-span-wrapper")?,
                PseudoElement::MozText => dest.write_str(":-moz-text")?,
                PseudoElement::FirstLetterContinuation => dest.write_str(":-moz-first-letter-continuation")?,
                PseudoElement::MozBlockInsideInlineWrapper => dest.write_str(":-moz-block-inside-inline-wrapper")?,
                PseudoElement::MozMathMLAnonymousBlock => dest.write_str(":-moz-mathml-anonymous-block")?,
                PseudoElement::MozLineFrame => dest.write_str(":-moz-line-frame")?,
                PseudoElement::ButtonContent => dest.write_str(":-moz-button-content")?,
                PseudoElement::CellContent => dest.write_str(":-moz-cell-content")?,
                PseudoElement::DropDownList => dest.write_str(":-moz-dropdown-list")?,
                PseudoElement::FieldsetContent => dest.write_str(":-moz-fieldset-content")?,
                PseudoElement::HtmlCanvasContent => dest.write_str(":-moz-html-canvas-content")?,
                PseudoElement::InlineTable => dest.write_str(":-moz-inline-table")?,
                PseudoElement::Table => dest.write_str(":-moz-table")?,
                PseudoElement::TableCell => dest.write_str(":-moz-table-cell")?,
                PseudoElement::TableWrapper => dest.write_str(":-moz-table-wrapper")?,
                PseudoElement::TableRowGroup => dest.write_str(":-moz-table-row-group")?,
                PseudoElement::TableRow => dest.write_str(":-moz-table-row")?,
                PseudoElement::Canvas => dest.write_str(":-moz-canvas")?,
                PseudoElement::PageSequence => dest.write_str(":-moz-page-sequence")?,
                PseudoElement::ScrolledContent => dest.write_str(":-moz-scrolled-content")?,
                PseudoElement::ScrolledCanvas => dest.write_str(":-moz-scrolled-canvas")?,
                PseudoElement::ColumnSet => dest.write_str(":-moz-column-set")?,
                PseudoElement::ColumnContent => dest.write_str(":-moz-column-content")?,
                PseudoElement::Viewport => dest.write_str(":-moz-viewport")?,
                PseudoElement::ViewportScroll => dest.write_str(":-moz-viewport-scroll")?,
                PseudoElement::AnonymousItem => dest.write_str(":-moz-anonymous-item")?,
                PseudoElement::BlockRubyContent => dest.write_str(":-moz-block-ruby-content")?,
                PseudoElement::Ruby => dest.write_str(":-moz-ruby")?,
                PseudoElement::RubyBase => dest.write_str(":-moz-ruby-base")?,
                PseudoElement::RubyBaseContainer => dest.write_str(":-moz-ruby-base-container")?,
                PseudoElement::RubyText => dest.write_str(":-moz-ruby-text")?,
                PseudoElement::RubyTextContainer => dest.write_str(":-moz-ruby-text-container")?,
                PseudoElement::MozTreeColumn(..) => dest.write_str(":-moz-tree-column")?,
                PseudoElement::MozTreeRow(..) => dest.write_str(":-moz-tree-row")?,
                PseudoElement::MozTreeSeparator(..) => dest.write_str(":-moz-tree-separator")?,
                PseudoElement::MozTreeCell(..) => dest.write_str(":-moz-tree-cell")?,
                PseudoElement::MozTreeIndentation(..) => dest.write_str(":-moz-tree-indentation")?,
                PseudoElement::MozTreeLine(..) => dest.write_str(":-moz-tree-line")?,
                PseudoElement::MozTreeTwisty(..) => dest.write_str(":-moz-tree-twisty")?,
                PseudoElement::MozTreeImage(..) => dest.write_str(":-moz-tree-image")?,
                PseudoElement::MozTreeCellText(..) => dest.write_str(":-moz-tree-cell-text")?,
                PseudoElement::MozTreeCheckbox(..) => dest.write_str(":-moz-tree-checkbox")?,
                PseudoElement::MozTreeDropFeedback(..) => dest.write_str(":-moz-tree-drop-feedback")?,
                PseudoElement::MozSVGMarkerAnonChild => dest.write_str(":-moz-svg-marker-anon-child")?,
                PseudoElement::MozSVGOuterSVGAnonChild => dest.write_str(":-moz-svg-outer-svg-anon-child")?,
                PseudoElement::MozSVGForeignContent => dest.write_str(":-moz-svg-foreign-content")?,
                PseudoElement::MozSVGText => dest.write_str(":-moz-svg-text")?,
            PseudoElement::Highlight(ref name) => {
                dest.write_str(":highlight(")?;
                serialize_atom_identifier(name, dest)?;
                dest.write_char(')')?;
            }
            PseudoElement::UnknownWebkit(ref atom) => {
                dest.write_str(":-webkit-")?;
                serialize_atom_identifier(atom, dest)?;
            }
        }
        if let Some(args) = self.tree_pseudo_args() {
            if !args.is_empty() {
                dest.write_char('(')?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                }
                dest.write_char(')')?;
            }
        }
        Ok(())
    }
}
