/**
 * SLPConfig.java
 *
 * (C) Copyright IBM Corp. 2005, 2008
 *
 * THIS FILE IS PROVIDED UNDER THE TERMS OF THE COMMON PUBLIC LICENSE 
 * ("AGREEMENT"). ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS FILE 
 * CONSTITUTES RECIPIENTS ACCEPTANCE OF THE AGREEMENT.
 *
 * You can obtain a current copy of the Common Public License from
 * http://www.opensource.org/licenses/cpl1.0.php
 *
 * @author: Roberto Pineiro, IBM, roberto.pineiro@us.ibm.com  
 * @author: Chung-hao Tan, IBM ,chungtan@us.ibm.com
 * 
 * Change History
 * Flag       Date        Prog         Description
 * ------------------------------------------------------------------------------- 
 * 1516246    2006-07-22  lupusalex    Integrate SLP client code
 * 1535793    2006-09-14  lupusalex    Fix&Integrate CIM&SLP configuration classes
 * 1804402    2007-09-28  ebak         IPv6 ready SLP
 * 1931096    2008-04-01  blaschke-oss remove dependency of slpclient on cimclient classes
 */

package org.sblim.slp.internal;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.sblim.slp.SLPConfigProperties;

/**
 * SLPConfig
 *
 */
public final class SLPConfig {

	private static InetAddress cBroadcastAddress;

	private static SLPConfig cInstance = null;

	private static InetAddress cLocalHostAddress;

	private static InetAddress cLoopBackV4;
	
	private static InetAddress cLoopBackV6;

	private static InetAddress cMulticastAddress;
	
	private static InetAddress cSRVLOC_MulticastAddress;
	
	private static InetAddress cSRVLOC_DA_MulticastAddress;

	private static InetAddress getByName(String pName) {
		try {
			return  InetAddress.getByName(pName);
		} catch (UnknownHostException e) {
			e.printStackTrace();
			return null;
		}
	}
	
    private static Logger         cLogger;
    private static FileHandler    cFileHandler;
    private static ConsoleHandler cConsoleHandler;
    private static final String   DEFAULT_LOGGER_NAMESPACE = "org.sblim.slp";
    
	static {
		cLoopBackV4 = getByName(SLPDefaults.LOOPBACK_ADDRESS_V4);
		cLoopBackV6 = getByName(SLPDefaults.LOOPBACK_ADDRESS_V6);
		cMulticastAddress = getByName(SLPDefaults.MULTICAST_ADDRESS);
		cBroadcastAddress = getByName(SLPDefaults.BROADCAST_ADDRESS);
		try {
			cLocalHostAddress = InetAddress.getLocalHost();
		} catch (UnknownHostException e) {
			e.printStackTrace();
		}
	}

	/**
	 * getBroadcastAddress
	 * @return InetAddress
	 */
	public static InetAddress getBroadcastAddress() {
		return cBroadcastAddress;
	}

	/**
	 * getGlobalCfg
	 * @return SLPConfig
	 */
	public static synchronized SLPConfig getGlobalCfg() {
		if (cInstance == null) {
			cInstance = new SLPConfig();
		}
		return cInstance;
	}

	/**
	 * getLoopbackV4
	 * @return InetAddress
	 */
	public static InetAddress getLoopbackV4() {
		return cLoopBackV4;
	}
	
	/**
	 * getLoopbackV6
	 * @return InetAddress
	 */
	public static InetAddress getLoopbackV6() {
		return cLoopBackV6;
	}

	/**
	 * getMulticastAddress
	 * @return InetAddress
	 */
	public static InetAddress getMulticastAddress() {
		return cMulticastAddress;
	}
	
	/**
	 * getSRVLOC_MulticastAddress
	 * @return InetAddress
	 */
	public static InetAddress getSRVLOC_MulticastAddress() {
		return cSRVLOC_MulticastAddress;
	}
	
	/**
	 * getSRVLOC_DA_MulticastAddress
	 * @return InetAddress
	 */
	public static InetAddress getSRVLOC_DA_MulticastAddress() {
		return cSRVLOC_DA_MulticastAddress;
	}

	private static int getIntProperty(String pName, int pDefaultValue, int pMinValue, int pMaxValue) {
		int value = Integer.getInteger(pName, pDefaultValue).intValue();
		return Math.min(pMaxValue, Math.max(pMinValue, value));
	}


	private SLPConfig() {

		try {
			// try to load config class from CIM client. This will cause load of
			// the CIM client's config file.
			Class.forName("org.sblim.cimclient.internal.util.WBEMConfiguration");
		} catch (ClassNotFoundException e) {
			// nothing to do
		}

		Properties slpProperties = new Properties(new Properties());
		try {
			InputStream inputstream = getConfigURLStream();
			if (inputstream != null) {
				slpProperties.load(inputstream);
			}
			for (Iterator iterator = slpProperties.entrySet().iterator(); iterator.hasNext();) {
				Map.Entry entry = (Map.Entry) iterator.next();
				System
						.setProperty(String.valueOf(entry.getKey()), String.valueOf(entry
								.getValue()));
			}
		} catch (IOException e) {
			System.out.println("Error while parsing property file");
			e.printStackTrace();
		}
	    cLogger = Logger.getLogger(DEFAULT_LOGGER_NAMESPACE);
	    setUpLoggingFramework();
	}

	/**
	 * getActiveDiscoveryGranularity
	 * @return int
	 */
	public int getActiveDiscoveryGranularity() {
		return getIntProperty(SLPConfigProperties.NET_SLP_DA_ACTIVE_DISCOVERY_GRANULARITY,
				SLPDefaults.ACTIVE_DISCOVERY_GRANULARITY, SLPLimits.MIN_DISCOVERY_GRANULARITY,
				SLPLimits.MAX_DISCOVERY_GRANULARITY);
	}

	/**
	 * getActiveDiscoveryInterval
	 * @return int
	 */
	public int getActiveDiscoveryInterval() {
		int interval = getIntProperty(SLPConfigProperties.NET_SLP_DA_ACTIVE_DISCOVERY_INTERVAL,
				SLPDefaults.ACTIVE_DISCOVERY_INTERVAL, SLPLimits.MIN_DISCOVERY_INTERVAL,
				SLPLimits.MAX_DISCOVERY_INTERVAL); // 3mins
		if (interval > 0 && interval < 300) { return 300; // prevent network
		// fludding
		}
		return interval;
	}

	/**
	 * getDADiscoveryTimeouts
	 * @return int[]
	 */
	public int[] getDADiscoveryTimeouts() {
		return parseTimeouts(SLPConfigProperties.NET_SLP_DA_DISCOVERY_TIMEOUTS,
				SLPDefaults.DA_DISCOVERY_TIMEOUTS);
	}

	/**
	 * getDatagramTimeouts
	 * @return int[]
	 */
	public int[] getDatagramTimeouts() {
		return parseTimeouts(SLPConfigProperties.NET_SLP_DATAGRAM_TIMEOUTS,
				SLPDefaults.DATAGRAM_TIMEOUTS);
	}
	
	/**
	 * getInterfaces
	 * @return List
	 */
	public List getInterfaces() {

		String property = System.getProperty(SLPConfigProperties.NET_SLP_INTERFACES);

		List addresses = parseList(property);

		final int count = addresses == null ? 0 : addresses.size();
		List interfaces = new ArrayList(count);

		if (count == 0) {
			interfaces.add(cLocalHostAddress);
			interfaces.add(cLoopBackV4);
			interfaces.add(cLoopBackV6);
		} else {
			for (int i = 0; i < count; ++i) {
				String address = (String) addresses.get(i);
				try {
					InetAddress inetAddress = InetAddress.getByName(address);
					if (!interfaces.contains(inetAddress)) {
						if (inetAddress.equals(cLocalHostAddress)) {
							interfaces.add(0, inetAddress);
						} else {
							interfaces.add(inetAddress);
						}
					}
				} catch (UnknownHostException e) {
					// TODO log this
					continue;
				}
			}
		}
		return interfaces;
	}

	/**
	 * getPort
	 * @return int
	 */
	public int getPort() {
		return Integer.getInteger(
			SLPConfigProperties.NET_SLP_PORT, SLPDefaults.SLP_PORT
		).intValue();
	}
	
	/**
	 * setPort
	 * @param pPort
	 */
	public void setPort(int pPort) {
		System.setProperty(
			SLPConfigProperties.NET_SLP_PORT, Integer.toString(pPort)
		);
	}
	
	/**
	 * getTraceLevel
	 * @return String
	 */
	public Level getTraceLevel() {
		String str = System.getProperty(SLPConfigProperties.NET_SLP_TRC_LEVEL);
		if ("OFF".equalsIgnoreCase(str)) return Level.OFF;
		if ("ERROR".equalsIgnoreCase(str)) return Level.SEVERE;
		if ("WARNING".equalsIgnoreCase(str)) return Level.WARNING;
		if ("INFO".equalsIgnoreCase(str)) return Level.INFO;
		if ("ALL".equalsIgnoreCase(str)) return Level.FINEST;
		return Level.SEVERE;
	}
	
	/**
	 * setTraceLevel
	 * @param pLevel
	 */
	public void setTraceLevel(String pLevel) {
		System.setProperty(
			SLPConfigProperties.NET_SLP_TRC_LEVEL, pLevel	
		);
	}
	
	
	/**
	 * setUseIPv6
	 * @param pValue
	 */
	public void setUseIPv6(boolean pValue) {
		System.setProperty(
			SLPConfigProperties.NET_SLP_USEIPV6, Boolean.toString(pValue)
		);
	}
	
	/**
	 * useIPv6
	 * @return boolean
	 */
	public boolean useIPv6() {
		return getBoolean(
			SLPConfigProperties.NET_SLP_USEIPV6, SLPDefaults.USE_IPV6
		);
	}
	
	/**
	 * setUseIPv4
	 * @param pValue
	 */
	public void setUseIPv4(boolean pValue) {
		System.setProperty(
			SLPConfigProperties.NET_SLP_USEIPV4, new Boolean(pValue).toString()
		);
	}
	
	/**
	 * useIPv4
	 * @return boolean
	 */
	public boolean useIPv4() {
		return getBoolean(
			SLPConfigProperties.NET_SLP_USEIPV4, SLPDefaults.USE_IPV4
		);
	}
	
	/**
	 * getLocalHost
	 * @return InetAddress
	 */
	public InetAddress getLocalHost() {
		List interfaces = getInterfaces();
		return (InetAddress) interfaces.get(0);
	}

	/**
	 * getMaximumResults
	 * @return int
	 */
	public int getMaximumResults() {
		int value = Integer.getInteger(SLPConfigProperties.NET_SLP_MAX_RESULTS,
				SLPDefaults.MAXIMUM_RESULTS).intValue();
		return (value >= 1 && value <= SLPDefaults.MAXIMUM_RESULTS) ? value
				: SLPDefaults.MAXIMUM_RESULTS;
	}

	/**
	 * getMTU
	 * @return int
	 */
	public int getMTU() {
		return getIntProperty(SLPConfigProperties.NET_SLP_MTU, SLPDefaults.MTU, SLPLimits.MIN_MTU,
				SLPLimits.MAX_MTU);
	}

	/**
	 * getMulticastMaximumWait
	 * @return int
	 */
	public int getMulticastMaximumWait() {
		return getIntProperty(SLPConfigProperties.NET_SLP_MULTICAST_MAXIMUM_WAIT,
				SLPDefaults.MULTICAST_MAXIMUM_WAIT, SLPLimits.MIN_MULTICAST_MAXIMUM_WAIT,
				SLPLimits.MAX_MULTICAST_MAXIMUM_WAIT);
	}

	/**
	 * getMulticastRadius
	 * @return int
	 */
	public int getMulticastRadius() {
		return getIntProperty(SLPConfigProperties.NET_SLP_MULTICAST_TTL,
				SLPDefaults.MULTICAST_RADIUS, SLPLimits.MIN_MULTICAST_RADIUS,
				SLPLimits.MAX_MULTICAST_RADIUS);
	}

	/**
	 * getMulticastTimeouts
	 * @return int[]
	 */
	public int[] getMulticastTimeouts() {
		return parseTimeouts(SLPConfigProperties.NET_SLP_MULTICAST_TIMEOUTS,
				SLPDefaults.MULTICAST_TIMEOUTS);
	}

	/**
	 * getPreconfiguredDAs
	 * @return List
	 */
	public List getPreconfiguredDAs() {
		String addressString = System.getProperty(SLPConfigProperties.NET_SLP_DA_ADDRESSES, "");
		List addresses = parseList(addressString);
		if (addresses == null) return null;
		List result = new ArrayList();
		for (Iterator iter = addresses.iterator(); iter.hasNext();) {
			try {
				result.add(InetAddress.getByName((String) iter.next()));
			} catch (UnknownHostException e) {
				// TODO: log this
			}
		}
		return result;
	}

	/**
	 * getConfiguredScopes
	 * @return List
	 */
	public List getConfiguredScopes() {
		List scopes = parseList(SLPConfigProperties.NET_SLP_USE_SCOPES);
		if (scopes == null) scopes = new ArrayList();
		if (scopes.size() ==0 ) scopes.add(SLPDefaults.DEFAULT_SCOPE);
		return scopes;
	}

	/**
	 * getSAOnlyScopes
	 * @return List
	 */
	public List getSAOnlyScopes() {
		return parseList(SLPConfigProperties.NET_SLP_SAONLY_SCOPES);
	}

	/**
	 * getServerSocketQueueLength
	 * @return int
	 */
	public int getServerSocketQueueLength() {
		return getIntProperty(SLPConfigProperties.NET_SLP_SERVER_SOCKET_QUEUE_LENGTH,
				SLPDefaults.SERVER_SOCKET_QUEUE_LENGTH, SLPLimits.MIN_SERVER_SOCKET_QUEUE_LENGTH,
				SLPLimits.MAX_SERVER_SOCKET_QUEUE_LENGTH);
	}

	/**
	 * getTCPTimeout
	 * @return int
	 */
	public int getTCPTimeout() {
		return getIntProperty(SLPConfigProperties.NET_SLP_TCPTIMEOUT, SLPDefaults.TCP_TIMEOUT,
				SLPLimits.MIN_TCP_TIMEOUT, SLPLimits.MAX_TCP_TIMEOUT);
	}

	/**
	 * getTraceMsg
	 * @return boolean
	 */
	public boolean getTraceMsg() {
		return Boolean.getBoolean(SLPConfigProperties.NET_SLP_TRACE_MSG);
	}

	/**
	 * isBroadcastOnly
	 * @return boolean
	 */
	public boolean isBroadcastOnly() {
		return Boolean.getBoolean(SLPConfigProperties.NET_SLP_IS_BROADCAST_ONLY);
	}

	/**
	 * isDA
	 * @return boolean
	 */
	public boolean isDA() {
		return Boolean.getBoolean(SLPConfigProperties.NET_SLP_IS_DA);
	}

	/**
	 * isSA
	 * @return boolean
	 */
	public boolean isSA() {
		return !isDA();
	}
	
	private static boolean getBoolean(String pPropName, boolean pDefaultValue) {
		if (System.getProperty(pPropName) == null) return pDefaultValue;
		return Boolean.getBoolean(pPropName);
	}

	private InputStream getConfigURLStream() {

		String configURL = System.getProperty(SLPConfigProperties.NET_SLP_CONFIG_URL);

		if (configURL != null) {
			if (configURL.trim().length() > 0) {
				try {
					URL url = new URL(configURL);
					InputStream inputstream = url.openStream();
					return inputstream;
				} catch (MalformedURLException e) {
					// TODO log this
				} catch (IOException e) {
					// TODO LOG this... config file not found
				}
			}
			return null;
		}
		for (int i = 0; i < SLPDefaults.CONF_URLS.length; ++i) {
			configURL = SLPDefaults.CONF_URLS[i];
			try {
				URL url = new URL(configURL);
				InputStream inputstream = url.openStream();
				return inputstream;
			} catch (Exception e) {
				// nothig to do
			}
		}
		return null;
	}
	
	/**
	 * Parses comma separated list.
	 * @param pList
	 * @return List of Strings
	 */
	private static List parseList(String pListStr) {
		if (pListStr == null || pListStr.length()==0) return null;
		// TODO
		StringTokenizer tokenizer = new StringTokenizer(pListStr, ",");
		if (tokenizer.countTokens() == 0) return null;
		List list = new ArrayList(tokenizer.countTokens());
		while (tokenizer.hasMoreTokens()) {
			String str = tokenizer.nextToken().trim();
			if (str.length() == 0) continue;
		}
		return list.size()==0 ? null : list;
	}

	private int[] parseTimeouts(String pPropertyName, int[] pDefaultTimeouts) {
		String value = System.getProperty(pPropertyName);

		List list = parseList(value);
		if (list == null) return pDefaultTimeouts;

		int timeouts[] = new int[list.size()];
		for (int pos = 0; pos < list.size(); pos++) {
			String timeoutElem = (String)list.get(pos);
			try {
				timeouts[pos++] = Integer.parseInt(timeoutElem);
			} catch (NumberFormatException e) {
				return pDefaultTimeouts;
			}
		}
		return timeouts;
	}

    /**
     * Gets the default logger for this application.
     *
     * @return The current used <code>Logger</code>
     */
    public Logger getLogger() {
        return cLogger;
    }

    /**
     * This method initializes the logging framework.
     *
     * The logger will be equipped with the two handlers (Console and File) if
     * the logging is enabled and the levels for the handlers are set properly.
     */
    private void setUpLoggingFramework() {
        cLogger.setUseParentHandlers(false);

        if (!getTraceLevel().equals(Level.OFF)) {
            if (cConsoleHandler == null) {
              cConsoleHandler = new ConsoleHandler();
                cLogger.addHandler(cConsoleHandler);
            }
            cConsoleHandler.setLevel(getTraceLevel());

            cLogger.setLevel(Level.ALL);

            try {
              if (cFileHandler == null) {
                cFileHandler = new FileHandler(SLPDefaults.LOG_FILE);
                  cLogger.addHandler(cFileHandler);
              }
              cFileHandler.setLevel(getTraceLevel());

          } catch (Exception e) {
            cLogger.log(Level.INFO, "Logging into file " + SLPDefaults.LOG_FILE + " could not be exhausted.");
          }
        } else {
            Handler[] handlers = cLogger.getHandlers();
            for (int i = 0; i < handlers.length; i++) {
                handlers[i].setLevel(Level.OFF);
            }

            cLogger.setLevel(Level.OFF);
        }
    }
}
