/////////////////////////////////////////////////////////////////////////////
// Name:        colordlg.h
// Purpose:     interface of wxColourDialog
// Author:      wxWidgets team
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

/**
    @class wxColourDialog

    This class represents the colour chooser dialog.

    Starting from wxWidgets 3.1.3 and currently in the MSW port only, this
    dialog generates wxEVT_COLOUR_CHANGED events while it is being shown, i.e.
    from inside its ShowModal() method, that notify the program about the
    change of the currently selected colour and allow it to e.g. preview the
    effect of selecting this colour. Note that if you react to this event, you
    should also correctly revert to the previously selected colour if the
    dialog is cancelled by the user.

    Example of using this class with dynamic feedback for the selected colour:
    @code
        // Some function for redrawing using the given colour. Ideally, it
        // shouldn't do anything if the colour is the same as the one used
        // before.
        void Redraw(const wxColour& colour);

        wxColourData data;
        data.SetColour(initialColourToUse);
        wxColourDialog dlg(this, &data);
        dlg.Bind(wxEVT_COLOUR_CHANGED, [](wxColourDialogEvent& event) {
                    Redraw(event.GetColour());
                 });
        if ( dlg.ShowModal() == wxID_OK ) {
            // Colour did change.
        } else {
            // Colour didn't change.
        }

        // This call is unnecessary under platforms generating
        // wxEVT_COLOUR_CHANGED if the dialog was accepted and unnecessary
        // under the platforms not generating this event if it was cancelled,
        // so we could check for the different cases explicitly to avoid it,
        // but it's simpler to just always call it.
        Redraw(data.GetColour());
    @endcode

    @library{wxcore}
    @category{cmndlg}

    @see @ref overview_cmndlg_colour, wxColour, wxColourData,
          wxColourDialogEvent, wxGetColourFromUser()
*/
class wxColourDialog : public wxDialog
{
public:
    /**
        Constructor. Pass a parent window, and optionally a pointer to a block
        of colour data, which will be copied to the colour dialog's colour
        data.

        Custom colours from colour data object will be used in the dialog's
        colour palette. Invalid entries in custom colours list will be ignored
        on some platforms(GTK) or replaced with white colour on platforms where
        custom colours palette has fixed size (MSW).

        @see wxColourData
    */
    wxColourDialog(wxWindow* parent, const wxColourData* data = nullptr);

    /**
        Destructor.
    */
    virtual ~wxColourDialog();

    /**
        Same as wxColourDialog().
    */
    bool Create(wxWindow* parent, const wxColourData* data = nullptr);

    /**
        Returns the colour data associated with the colour dialog.
    */
    wxColourData& GetColourData();

    /**
        Shows the dialog, returning wxID_OK if the user pressed OK, and
        wxID_CANCEL otherwise.
    */
    virtual int ShowModal();
};

/**
    This event class is used for the events generated by wxColourDialog.

    @beginEventTable{wxColourPickerEvent}
    @event{EVT_COLOUR_CHANGED(id, func)}
           Generated whenever the currently selected colour in the dialog
           changes. This event is currently only implemented in wxMSW.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxColourDialog

    @since 3.1.3
 */
class wxColourDialogEvent : public wxCommandEvent
{
public:
    wxColourDialogEvent();

    /**
        The constructor is not normally used by the user code.
    */
    wxColourDialogEvent(wxEventType evtType,
                        wxColourDialog* dialog,
                        const wxColour& colour);

    /**
        Retrieve the colour the user has just selected.
    */
    wxColour GetColour() const;

    /**
       Set the colour to be sent with the event.
    */
    void SetColour(const wxColour& colour);
};


wxEventType wxEVT_COLOUR_CHANGED;


// ============================================================================
// Global functions/macros
// ============================================================================

/** @addtogroup group_funcmacro_dialog */
///@{

/**
    Shows the colour selection dialog and returns the colour selected by user
    or invalid colour (use wxColour::IsOk() to test whether a colour is valid)
    if the dialog was cancelled.

    @param parent
        The parent window for the colour selection dialog.
    @param colInit
        If given, this will be the colour initially selected in the dialog.
    @param caption
        If given, this will be used for the dialog caption.
    @param data
        Optional object storing additional colour dialog settings, such as
        custom colours. If none is provided the same settings as the last time
        are used.

    @header{wx/colordlg.h}
*/
wxColour wxGetColourFromUser(wxWindow* parent = nullptr,
                             const wxColour& colInit = wxNullColour,
                             const wxString& caption = wxEmptyString,
                             wxColourData* data = nullptr);

///@}

