/*
 * Rufus: The Reliable USB Formatting Utility
 * DB of the known SHA256 hash values for Rufus downloadable content (GRUB, Syslinux, etc.)
 * as well PE256 hash values for UEFI revoked content (DBX, SkuSiPolicy.p7b)
 * Copyright © 2016-2025 Pete Batard <pete@akeo.ie>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdint.h>
#pragma once

/*
 * NB: Table data was generated from the https://github.com/pbatard/rufus-web/tree/gh-pages/files/
 * directory using the https://github.com/pbatard/rufus-web/blob/gh-pages/files/gendb.sh script.
 */
static uint8_t sha256db[] = {
	0x01, 0x21, 0x98, 0x20, 0xd9, 0x1c, 0x28, 0x7b, 0x35, 0x28, 0x06, 0xf1, 0xb6, 0xa8, 0x0d, 0x4a, 0x8a, 0xf4, 0x5c, 0xf5, 0x04, 0x83, 0xed, 0x6b, 0xe0, 0xc6, 0x0e, 0x7a, 0x66, 0x9c, 0x67, 0x87, // syslinux-6.04/pre2/ldlinux.bss
	0x0a, 0xd1, 0x3c, 0x2a, 0xb9, 0xd1, 0xcc, 0x66, 0x73, 0x70, 0xc4, 0x34, 0xf3, 0xe2, 0xa0, 0x54, 0x1c, 0xb8, 0x37, 0xdb, 0x72, 0x3e, 0x19, 0xae, 0x81, 0xc8, 0xee, 0x0b, 0xbe, 0x10, 0xdc, 0xe0, // grub-2.06@e67a551a-nonstandard/core.img
	0x11, 0x0c, 0x50, 0x1b, 0xfa, 0x9e, 0x72, 0xa8, 0x8c, 0xdb, 0xb8, 0xba, 0x11, 0xe1, 0xf0, 0x76, 0x1a, 0xec, 0x28, 0xbf, 0x04, 0x44, 0x67, 0xff, 0x38, 0x2c, 0x06, 0x95, 0xd5, 0x1f, 0x8a, 0x83, // grub-2.04-nonstandard/core.img
	0x12, 0xbd, 0x22, 0xd2, 0xb3, 0x69, 0x56, 0x0f, 0x89, 0xb8, 0x50, 0x7e, 0x7e, 0x74, 0xeb, 0xc5, 0xea, 0x44, 0x91, 0x48, 0x75, 0xf0, 0xa4, 0xcb, 0x1e, 0xa6, 0xfb, 0x4e, 0xc9, 0x89, 0x58, 0x17, // syslinux-6.03/pre6/ldlinux.sys
	0x15, 0x5f, 0x36, 0x7b, 0xb1, 0x30, 0xfe, 0x05, 0x5c, 0x79, 0x9f, 0x88, 0xb3, 0xc0, 0xc1, 0xa0, 0x0a, 0x18, 0x05, 0x78, 0x22, 0x69, 0xcf, 0x7e, 0x54, 0xaa, 0x61, 0xbd, 0xe3, 0x8e, 0x05, 0x92, // syslinux-6.03/pre3/ldlinux.bss
	0x1c, 0xb7, 0x8b, 0x98, 0xbc, 0xd6, 0x76, 0x7b, 0x01, 0x44, 0xf5, 0x00, 0xaf, 0x81, 0xef, 0x4f, 0x3c, 0x54, 0xea, 0xaf, 0xe3, 0xc9, 0x4e, 0x1f, 0xd6, 0x24, 0x68, 0x41, 0x4e, 0x98, 0x92, 0x25, // syslinux-6.03/pre20/ldlinux.bss
	0x1c, 0xc6, 0x32, 0x21, 0xfd, 0xf4, 0x46, 0xfc, 0xda, 0xc6, 0xc0, 0x56, 0x35, 0x79, 0x54, 0xc1, 0x5b, 0x61, 0x75, 0xca, 0x1b, 0xc2, 0xa4, 0x9f, 0x85, 0x52, 0xec, 0xca, 0x28, 0xac, 0x3e, 0x51, // syslinux-6.02/ldlinux.bss
	0x22, 0x96, 0x82, 0xac, 0x61, 0xb8, 0x8b, 0x11, 0x25, 0xfc, 0xd7, 0xe6, 0x9f, 0x4e, 0x7f, 0x46, 0x7f, 0x68, 0xc5, 0x14, 0x9e, 0xb9, 0x37, 0x1a, 0x98, 0xd8, 0xf2, 0x78, 0x41, 0x40, 0xad, 0x88, // syslinux-5.00/ldlinux.sys
	0x25, 0xd1, 0x38, 0xf3, 0x8b, 0x17, 0x35, 0x79, 0x3e, 0xee, 0x89, 0x19, 0xa3, 0xa0, 0xe5, 0xc9, 0x97, 0x9d, 0x2f, 0xac, 0xc0, 0xb5, 0x70, 0x74, 0x24, 0xe8, 0x04, 0x98, 0x1d, 0x25, 0xf9, 0xc9, // syslinux-6.03/pre10/ldlinux.sys
	0x28, 0xf8, 0x1d, 0xd8, 0x72, 0x0e, 0x6e, 0xfd, 0xc5, 0x30, 0x44, 0xf6, 0xf9, 0xa6, 0xc2, 0x8b, 0x6b, 0x89, 0x54, 0x23, 0xfb, 0x4c, 0xbc, 0x2d, 0xa2, 0xfb, 0xd6, 0x29, 0x70, 0xd6, 0xd1, 0xd6, // grub-2.06-nonstandard/core.img
	0x29, 0xcd, 0xfc, 0x24, 0x0d, 0x08, 0xe4, 0xd6, 0x42, 0x0d, 0x1f, 0x05, 0x7a, 0x0e, 0xb0, 0xb1, 0x07, 0x88, 0x91, 0x65, 0xa8, 0xeb, 0x16, 0x3c, 0x31, 0x72, 0xaa, 0xfc, 0xee, 0x02, 0xac, 0xdd, // grub-2.06~rc1/core.img
	0x2b, 0x07, 0x20, 0x67, 0xf9, 0xcd, 0x6e, 0x8c, 0x62, 0xe7, 0x82, 0xdc, 0x4a, 0xe2, 0x3b, 0x19, 0x2d, 0xe6, 0x28, 0xe7, 0x67, 0x0c, 0x84, 0x68, 0xed, 0x38, 0x0d, 0x36, 0x4a, 0xa7, 0x0b, 0xf1, // grub-2.06/core.img
	0x2b, 0x31, 0x7b, 0x12, 0xab, 0xff, 0x49, 0x66, 0x48, 0x78, 0xdf, 0xe0, 0xb5, 0x00, 0xb8, 0x50, 0x0f, 0x93, 0xf1, 0xd5, 0xe4, 0xd5, 0x28, 0x45, 0x95, 0xcc, 0x2c, 0x15, 0x6c, 0x74, 0x4f, 0x79, // grub-2.02~beta3/core.img
	0x2d, 0x99, 0x55, 0x09, 0xd0, 0xd6, 0xcd, 0xf7, 0x6e, 0xd5, 0x9f, 0x46, 0x53, 0xf1, 0xcb, 0xe1, 0xc9, 0xe6, 0x24, 0xe1, 0xaf, 0xa8, 0xf7, 0x8e, 0xc2, 0x45, 0x5b, 0x24, 0x53, 0x4b, 0x87, 0x76, // syslinux-6.03/pre13/ldlinux.bss
	0x2e, 0x3e, 0xf6, 0x65, 0x1e, 0x59, 0x85, 0x20, 0x26, 0xc5, 0x74, 0xde, 0x52, 0x65, 0xf6, 0x8e, 0x24, 0xf8, 0xe6, 0x27, 0xc5, 0x53, 0xce, 0x8d, 0x32, 0x7c, 0xa7, 0xe0, 0xb4, 0x96, 0x70, 0xa1, // grub4dos-0.4.5c/grldr
	0x31, 0x17, 0x97, 0x0e, 0x89, 0xe2, 0xab, 0xd6, 0x7e, 0x29, 0x26, 0x63, 0xf3, 0xe2, 0xcf, 0xa7, 0xc3, 0xfe, 0xe8, 0x7a, 0x87, 0x0a, 0xb9, 0x20, 0xc8, 0xa1, 0x74, 0xff, 0x5d, 0xfb, 0xfe, 0x8e, // syslinux-6.03/pre20/ldlinux.sys
	0x33, 0x02, 0x03, 0xb6, 0x29, 0x5e, 0x7c, 0x60, 0x72, 0x5f, 0x07, 0xc0, 0x92, 0x15, 0xd1, 0x75, 0xaa, 0x09, 0x23, 0xee, 0xd5, 0x5a, 0xec, 0xb5, 0x72, 0x77, 0x76, 0x4d, 0x6b, 0xdc, 0xf1, 0x31, // syslinux-6.03/pre19/ldlinux.sys
	0x34, 0x08, 0x54, 0xee, 0x4f, 0x7f, 0x10, 0xaa, 0x63, 0x38, 0xba, 0xeb, 0xa4, 0xdd, 0xd1, 0xca, 0x9c, 0xbd, 0x15, 0x6f, 0x04, 0x86, 0x69, 0x66, 0x6c, 0x92, 0xe4, 0x85, 0x47, 0x6e, 0xe9, 0x41, // syslinux-6.03/pre4/ldlinux.sys
	0x34, 0x1b, 0x76, 0x83, 0x3c, 0xc6, 0x1c, 0x1c, 0xd0, 0xde, 0x84, 0x2c, 0xcb, 0xfe, 0x38, 0x4f, 0x0b, 0x21, 0xb1, 0x10, 0x0e, 0xaa, 0x74, 0xf9, 0x5f, 0x07, 0x21, 0x72, 0x4f, 0x00, 0x6f, 0x74, // syslinux-6.03/pre4/ldlinux.bss
	0x37, 0x9f, 0x06, 0xfe, 0xee, 0x22, 0xf0, 0xd5, 0x8f, 0xf9, 0x9b, 0xb3, 0x47, 0xf9, 0x9e, 0x0a, 0x1a, 0x23, 0xaf, 0x03, 0x1e, 0x35, 0x13, 0x02, 0xd2, 0xf7, 0x63, 0x55, 0xc9, 0xc7, 0x90, 0xb2, // syslinux-5.11/ldlinux.c32
	0x38, 0x48, 0x38, 0xcf, 0x85, 0x08, 0x18, 0x04, 0x0a, 0xf2, 0xb3, 0xd8, 0x71, 0x38, 0xb4, 0x49, 0x12, 0xb0, 0x0d, 0x4f, 0x17, 0xb1, 0xba, 0x41, 0xbd, 0x06, 0x1a, 0xf4, 0x6d, 0x56, 0x61, 0x62, // syslinux-6.01/ldlinux.c32
	0x39, 0xde, 0x1f, 0x87, 0x08, 0xef, 0xd8, 0x88, 0x56, 0x91, 0x68, 0x5e, 0xfc, 0x59, 0x45, 0x48, 0xdf, 0x29, 0xfb, 0x20, 0x0b, 0xc1, 0x56, 0x7f, 0x1b, 0xd7, 0xcf, 0xce, 0x6f, 0x95, 0x55, 0xe0, // syslinux-6.03/pre7/ldlinux.sys
	0x3c, 0x39, 0x1b, 0x51, 0x04, 0xec, 0xdb, 0x65, 0x33, 0x01, 0xd2, 0xe2, 0x20, 0x0a, 0x6d, 0xab, 0xc0, 0xdb, 0x59, 0xd8, 0x80, 0x7b, 0xe9, 0x47, 0x99, 0x51, 0xbe, 0x18, 0x61, 0x53, 0x20, 0xad, // syslinux-6.03/pre11/ldlinux.sys
	0x3d, 0x3c, 0xba, 0x34, 0x12, 0xe1, 0xba, 0xf6, 0x01, 0xaf, 0xbb, 0x6d, 0xf0, 0x09, 0x75, 0xa0, 0x8c, 0xc0, 0xad, 0x7c, 0x62, 0xfd, 0x50, 0x15, 0x87, 0x08, 0xeb, 0x4c, 0x8c, 0x3d, 0x95, 0x55, // syslinux-5.02/ldlinux.bss
	0x3d, 0x6b, 0x9d, 0xb7, 0x62, 0xa9, 0x19, 0x01, 0x00, 0xea, 0xbf, 0x3e, 0x13, 0x43, 0xb7, 0xa4, 0x09, 0xc6, 0x77, 0xea, 0x21, 0x04, 0x6f, 0xe0, 0xff, 0xbe, 0x7f, 0xe7, 0xb0, 0xdc, 0xb8, 0x95, // syslinux-6.04/pre2/ldlinux.sys
	0x3f, 0x12, 0x06, 0xe0, 0xcc, 0x45, 0xdb, 0xe1, 0x80, 0xe7, 0x3a, 0xda, 0xeb, 0x22, 0x1b, 0xfc, 0x7d, 0x5a, 0x80, 0x00, 0x95, 0x73, 0x85, 0x49, 0x39, 0x03, 0x79, 0xd7, 0xd0, 0x28, 0x2a, 0xc3, // syslinux-6.03/ldlinux.sys
	0x40, 0xe7, 0x8e, 0x0b, 0x92, 0xb9, 0x80, 0x60, 0xbf, 0x28, 0x9b, 0xa6, 0xab, 0x68, 0x22, 0x5b, 0x61, 0xe2, 0x5e, 0x08, 0x9e, 0xd5, 0xa6, 0x16, 0x94, 0xc5, 0x19, 0x17, 0x92, 0x5f, 0x3e, 0x8d, // syslinux-6.03/pre7/ldlinux.bss
	0x40, 0xe7, 0x99, 0x43, 0xac, 0x71, 0xde, 0xf1, 0xc0, 0x35, 0x5a, 0xec, 0x71, 0xb4, 0xf7, 0x8d, 0xcf, 0x19, 0x20, 0x4a, 0x97, 0xe9, 0x7e, 0x0f, 0x34, 0x17, 0x50, 0x65, 0x32, 0xc9, 0x02, 0x76, // syslinux-6.03/pre16/ldlinux.bss
	0x42, 0x23, 0x03, 0x86, 0x6d, 0xa1, 0xf2, 0xaf, 0x28, 0x25, 0x92, 0x29, 0x17, 0x21, 0xc7, 0x61, 0x23, 0xcf, 0xe9, 0x84, 0x05, 0xb7, 0x60, 0x2a, 0xc7, 0xe3, 0xe9, 0x5b, 0xf3, 0xcb, 0xd4, 0xba, // syslinux-6.00/ldlinux.c32
	0x46, 0x48, 0xe8, 0x13, 0xd7, 0x54, 0x20, 0x39, 0x4e, 0xe6, 0x35, 0xd2, 0xa4, 0x78, 0x44, 0xeb, 0xdc, 0xc9, 0x98, 0xd3, 0x45, 0x8f, 0x95, 0xf9, 0x50, 0x80, 0x50, 0x9b, 0x88, 0x80, 0x44, 0x4a, // syslinux-6.03/pre17/ldlinux.sys
	0x47, 0x0c, 0x04, 0x6d, 0xe8, 0x53, 0x5d, 0x46, 0x0e, 0x05, 0x63, 0x99, 0xa8, 0x89, 0x28, 0x0b, 0x25, 0x55, 0xf2, 0xb7, 0xbf, 0x89, 0x84, 0x9a, 0x7f, 0x27, 0xa4, 0x32, 0x72, 0xb5, 0x3f, 0x12, // syslinux-6.01/ldlinux.sys
	0x4d, 0xee, 0xc3, 0x47, 0xb0, 0xa3, 0x18, 0x46, 0x09, 0x42, 0xb2, 0xeb, 0x3e, 0xd1, 0x75, 0xcf, 0x3f, 0x8c, 0x3e, 0x8c, 0xe4, 0x3b, 0x14, 0xfc, 0x8e, 0x8f, 0x8c, 0x51, 0xcb, 0xa5, 0x5a, 0xdb, // syslinux-5.02/ldlinux.c32
	0x4e, 0xfb, 0x8f, 0xde, 0xb9, 0xe1, 0xe2, 0xa2, 0xc3, 0xf0, 0x11, 0xca, 0x77, 0x3b, 0x95, 0xf4, 0xb9, 0x7f, 0xbc, 0x7f, 0x3d, 0x40, 0x4d, 0x7c, 0x60, 0xf3, 0x4a, 0x61, 0x43, 0x44, 0x7e, 0x7f, // syslinux-5.01/ldlinux.c32
	0x50, 0xca, 0xe9, 0x94, 0x79, 0x68, 0xdf, 0xa6, 0xd7, 0x11, 0xcd, 0xb0, 0x7d, 0x84, 0xf2, 0x56, 0x78, 0xd6, 0x69, 0xbf, 0xb0, 0x29, 0xaa, 0x56, 0xa1, 0x04, 0x55, 0xb0, 0xcd, 0x06, 0xca, 0xd2, // syslinux-6.03/pre3/ldlinux.sys
	0x52, 0x0b, 0x46, 0x87, 0x46, 0x22, 0xb5, 0xa6, 0x57, 0x83, 0xdb, 0x7c, 0x12, 0x42, 0x53, 0x2d, 0x13, 0x7f, 0x95, 0xfe, 0xbf, 0x4d, 0xf9, 0xc2, 0x7e, 0x67, 0x94, 0x3a, 0x21, 0x27, 0xf6, 0x08, // syslinux-5.01/ldlinux.sys
	0x52, 0x80, 0x86, 0xed, 0x16, 0xf4, 0x93, 0xb9, 0x04, 0xf0, 0x75, 0xd1, 0xe9, 0xd8, 0xcb, 0x8a, 0x2e, 0xdc, 0xb4, 0x37, 0x40, 0x30, 0x51, 0xd8, 0x79, 0x40, 0x08, 0x9a, 0x16, 0xed, 0x4d, 0x33, // grub-2.06@e67a551a/core.img
	0x54, 0x4b, 0x3e, 0x86, 0xfc, 0xa2, 0x7a, 0x12, 0xf4, 0x81, 0xfd, 0x32, 0x53, 0x57, 0x1c, 0xc5, 0xe6, 0x1a, 0x19, 0xed, 0x9a, 0x43, 0xbc, 0x12, 0xa8, 0x07, 0x6d, 0x90, 0x8f, 0x98, 0x39, 0x6f, // syslinux-6.03/pre2/ldlinux.sys
	0x55, 0xb9, 0x27, 0xd2, 0x17, 0xe8, 0x83, 0xd9, 0xe1, 0x58, 0xc9, 0x51, 0x80, 0xb1, 0xba, 0x79, 0x97, 0x3a, 0xd3, 0xd9, 0x44, 0xc2, 0xc6, 0xa1, 0xed, 0x49, 0x5c, 0xd3, 0xef, 0xc3, 0x54, 0x1c, // syslinux-5.11/ldlinux.bss
	0x56, 0x7a, 0x32, 0x6a, 0x20, 0x9a, 0xe8, 0x69, 0x9a, 0x7d, 0xb5, 0xc0, 0xa4, 0x34, 0x52, 0x71, 0x05, 0x1e, 0x0a, 0x84, 0x03, 0xae, 0x21, 0x13, 0xd9, 0xd0, 0x12, 0xc4, 0x8d, 0x99, 0xa4, 0x13, // syslinux-6.03/pre12/ldlinux.sys
	0x57, 0x45, 0x0a, 0xf8, 0xa4, 0xe0, 0x06, 0x71, 0x11, 0x46, 0xaa, 0x53, 0x1f, 0xe5, 0x59, 0x8f, 0x9e, 0x92, 0xe7, 0x3d, 0x6c, 0xea, 0x93, 0x06, 0x02, 0x13, 0xa3, 0x08, 0xce, 0x76, 0x11, 0x29, // syslinux-6.03/pre19/ldlinux.bss
	0x5b, 0xf5, 0xb8, 0x10, 0xa4, 0x4a, 0x27, 0xd8, 0x72, 0xae, 0x50, 0x68, 0xa4, 0xc0, 0x06, 0x93, 0xcb, 0xa2, 0x41, 0xd7, 0x96, 0x51, 0x07, 0xfb, 0x20, 0xd0, 0x18, 0x17, 0x92, 0x36, 0x16, 0x81, // syslinux-6.03/pre1/ldlinux.sys
	0x5c, 0x42, 0x3b, 0xc0, 0xc6, 0xb9, 0xc4, 0x12, 0x0f, 0x33, 0x7c, 0x5a, 0xf9, 0x3d, 0xcf, 0xc2, 0xfa, 0x2b, 0xc0, 0xe3, 0x8f, 0xd9, 0x4e, 0x8a, 0x6b, 0x0a, 0x1f, 0x16, 0x1a, 0x88, 0x29, 0x92, // grub-2.03/core.img
	0x5c, 0xef, 0x9a, 0xd0, 0xd0, 0xca, 0x04, 0x09, 0x72, 0x62, 0x24, 0x16, 0x86, 0xc6, 0xc3, 0xa7, 0x30, 0x6a, 0xb9, 0xb9, 0xcd, 0xf2, 0x4b, 0x9d, 0x4e, 0xe3, 0xb1, 0x6a, 0xf0, 0x1a, 0x5a, 0xf2, // syslinux-6.03/ldlinux.c32
	0x5e, 0xd2, 0x81, 0xa6, 0x95, 0xbe, 0x85, 0x95, 0x64, 0x02, 0xb2, 0x30, 0x51, 0x2c, 0x49, 0x96, 0x4a, 0x46, 0x24, 0xc9, 0xd7, 0x55, 0xcd, 0x1c, 0xda, 0xa7, 0x37, 0x6c, 0x2f, 0x7f, 0x41, 0xb9, // syslinux-6.03/pre16/ldlinux.sys
	0x67, 0x64, 0x0d, 0x11, 0xfe, 0x80, 0x78, 0x17, 0x57, 0x0e, 0xf1, 0x6d, 0xeb, 0xe8, 0x21, 0xb5, 0xd3, 0x93, 0x51, 0x78, 0x3a, 0x5f, 0xa7, 0x61, 0x7b, 0xa2, 0x55, 0x30, 0xec, 0x4d, 0xd8, 0x77, // syslinux-6.03/pre6/ldlinux.bss
	0x68, 0x40, 0xb3, 0xda, 0xf8, 0x6d, 0xba, 0x9f, 0x8e, 0x64, 0x22, 0xdb, 0x59, 0x54, 0x67, 0x64, 0xe3, 0xb1, 0x44, 0x61, 0x9a, 0x70, 0xb5, 0xac, 0x27, 0x88, 0x50, 0x5e, 0xdb, 0xc6, 0x17, 0x6c, // grub-2.02~beta2/core.img
	0x6a, 0xbc, 0xdc, 0x80, 0x3a, 0x30, 0x85, 0xea, 0x5f, 0x9d, 0xa1, 0xb4, 0x3e, 0xdb, 0x2e, 0xad, 0xa2, 0x75, 0x36, 0x0d, 0xb8, 0x11, 0xc9, 0xac, 0xf5, 0x9a, 0x55, 0x5f, 0x67, 0x7b, 0x2d, 0x8b, // syslinux-6.03/pre12/ldlinux.bss
	0x73, 0xb6, 0x27, 0x67, 0xa1, 0x62, 0x00, 0xb9, 0xaf, 0x19, 0x3a, 0x7d, 0x5c, 0x94, 0xe9, 0xc2, 0x94, 0xc6, 0xdb, 0xb6, 0xd5, 0xb1, 0x7c, 0x15, 0x03, 0x8c, 0x9f, 0x31, 0x73, 0xc9, 0xa7, 0xbc, // syslinux-6.04/ldlinux.sys
	0x75, 0x6f, 0x89, 0x25, 0x23, 0xc6, 0x8d, 0x27, 0x32, 0x28, 0x8b, 0x5a, 0xd4, 0x2d, 0x7d, 0xc7, 0x4e, 0xa7, 0xa7, 0x08, 0x9b, 0x04, 0x2b, 0x12, 0x5f, 0x5d, 0x74, 0x7f, 0xf3, 0x20, 0xa0, 0x77, // syslinux-4.07/vesamenu.c32
	0x76, 0xd1, 0x2b, 0x17, 0xfd, 0x05, 0xd3, 0x10, 0xb0, 0x5d, 0x00, 0xa8, 0x38, 0x77, 0x06, 0xc5, 0xce, 0x35, 0xd4, 0xb0, 0x50, 0xf2, 0x23, 0x9d, 0x15, 0xd3, 0xc9, 0xac, 0x92, 0xc5, 0xdd, 0xbc, // grub-2.06-nonstandard-gdie/core.img
	0x77, 0x9a, 0x5e, 0xbd, 0x69, 0xd3, 0x28, 0x5e, 0xb9, 0xed, 0x4a, 0xc7, 0xc0, 0x4d, 0x2d, 0x15, 0xcb, 0xa1, 0x8a, 0x1f, 0x97, 0xc7, 0xc4, 0xbe, 0x62, 0x48, 0x93, 0xa9, 0xe1, 0xb0, 0x89, 0x2e, // syslinux-6.03/pre9/ldlinux.sys
	0x78, 0x64, 0x8e, 0xf0, 0xc5, 0x00, 0x41, 0x75, 0xb9, 0xa8, 0xea, 0x33, 0x30, 0x14, 0xea, 0x02, 0xc9, 0x17, 0xf8, 0x23, 0xe7, 0x7a, 0x3e, 0xc9, 0xac, 0xd9, 0xd2, 0x2b, 0x46, 0x02, 0xf3, 0x6d, // syslinux-6.03/pre13/ldlinux.sys
	0x7d, 0xa9, 0xc5, 0x21, 0x76, 0xb8, 0xaf, 0x01, 0x64, 0xea, 0x39, 0x21, 0x22, 0x44, 0xb1, 0x0a, 0xa0, 0xc7, 0x97, 0xe7, 0x65, 0xbb, 0x6b, 0x92, 0x69, 0xb5, 0x8b, 0xc9, 0xe5, 0x0a, 0x9f, 0x18, // syslinux-5.01/ldlinux.bss
	0x82, 0x11, 0xfa, 0xe8, 0xaf, 0xf0, 0x23, 0x3f, 0x05, 0xa8, 0xb7, 0x8c, 0x58, 0x15, 0x25, 0xe2, 0x81, 0xac, 0x98, 0x23, 0x54, 0xa8, 0xc4, 0x3b, 0xb4, 0x96, 0x5e, 0x61, 0xdc, 0x98, 0xb4, 0x62, // syslinux-6.03/pre8/ldlinux.bss
	0x83, 0x57, 0xaa, 0xd3, 0x6a, 0xec, 0x68, 0x21, 0xcb, 0xf2, 0x17, 0x4d, 0xb5, 0xd2, 0x09, 0xef, 0x2c, 0xd2, 0x62, 0x88, 0x12, 0x39, 0xeb, 0xc3, 0xf4, 0xc1, 0xcf, 0x55, 0xab, 0x10, 0xee, 0x55, // grub-2.12~rc1/core.img
	0x83, 0x9b, 0xd0, 0x8a, 0xcb, 0x68, 0x47, 0xd6, 0x55, 0x07, 0xf1, 0x4e, 0x7a, 0x55, 0x6e, 0x91, 0xe6, 0x12, 0x9c, 0x47, 0x86, 0x3f, 0x7d, 0x61, 0xe2, 0xce, 0x6d, 0xb7, 0x8d, 0xf3, 0xd2, 0x3f, // syslinux-6.03/pre9/ldlinux.bss
	0x87, 0xaa, 0x91, 0xf8, 0x7f, 0xba, 0x5f, 0x31, 0x79, 0x43, 0x08, 0xda, 0xa4, 0xa4, 0x8d, 0xad, 0x6c, 0xf6, 0xfa, 0x34, 0x26, 0x4d, 0x66, 0xb8, 0x84, 0xb8, 0xb9, 0xdc, 0x96, 0x42, 0xed, 0x86, // syslinux-5.02/ldlinux.sys
	0x88, 0x14, 0xe5, 0x76, 0xab, 0xc1, 0xaa, 0x44, 0xdd, 0xe9, 0x43, 0xb0, 0xca, 0xae, 0xe8, 0x33, 0xa5, 0x81, 0x01, 0x42, 0x61, 0x4a, 0xde, 0xeb, 0x4c, 0xc7, 0x25, 0xe7, 0x8a, 0x50, 0x45, 0xb7, // syslinux-6.03/ldlinux.bss
	0x8b, 0x93, 0x7e, 0x5e, 0x8b, 0xae, 0x5a, 0xf8, 0xc8, 0x95, 0x63, 0xc0, 0x0e, 0x9c, 0xaf, 0xc6, 0xcd, 0x7c, 0x2c, 0x80, 0x8a, 0xda, 0x7b, 0xf4, 0xad, 0x51, 0x08, 0xda, 0x3e, 0x51, 0xcd, 0x70, // grub-2.00-22/core.img
	0x8e, 0xc8, 0x42, 0x06, 0x94, 0x4c, 0xd4, 0x3d, 0xf6, 0xba, 0x83, 0x63, 0xc0, 0x81, 0xe4, 0xa0, 0x82, 0x9e, 0x71, 0x9a, 0xbf, 0x5a, 0x46, 0x6d, 0x7c, 0x81, 0x0c, 0x2f, 0x5b, 0x6d, 0x13, 0x75, // syslinux-6.03/pre5/ldlinux.sys
	0x95, 0x8d, 0x10, 0xbb, 0x87, 0x28, 0xcc, 0x1f, 0xf1, 0x6a, 0x12, 0xee, 0x6a, 0x60, 0x62, 0x40, 0xa6, 0xb7, 0x4d, 0xab, 0xa0, 0x2b, 0x8c, 0xb8, 0xed, 0x2a, 0xe8, 0x1c, 0x2f, 0xb2, 0x5b, 0x97, // syslinux-6.00/ldlinux.bss
	0x9a, 0x0b, 0xc4, 0x1b, 0xd7, 0x95, 0xed, 0xb0, 0x83, 0x0f, 0x1c, 0xc4, 0x82, 0x4b, 0xfa, 0x9d, 0xe0, 0x9d, 0x68, 0x63, 0x92, 0x09, 0x4f, 0x5a, 0xe7, 0xfb, 0xac, 0xfb, 0xb0, 0x17, 0x9d, 0xa6, // syslinux-6.03/pre1/ldlinux.bss
	0x9b, 0xcc, 0x65, 0x92, 0xa7, 0xba, 0x7e, 0x73, 0x38, 0xf4, 0xbb, 0xba, 0x27, 0xc6, 0x30, 0x16, 0xb9, 0x5e, 0xcb, 0x1e, 0xc6, 0x8c, 0x0b, 0xe9, 0xb6, 0x99, 0xb2, 0xea, 0x69, 0xcb, 0xab, 0xb2, // syslinux-5.00/ldlinux.c32
	0xa6, 0x82, 0x43, 0xa0, 0xf2, 0xe5, 0x90, 0xb8, 0x14, 0x02, 0xd6, 0xfa, 0x62, 0xd4, 0xfd, 0x30, 0x94, 0x8c, 0x00, 0x3d, 0xa1, 0x2b, 0xfe, 0xeb, 0x69, 0xba, 0x20, 0x34, 0x17, 0x27, 0x09, 0x4c, // syslinux-6.03/pre14/ldlinux.sys
	0xa9, 0x4a, 0x99, 0xe6, 0xde, 0x68, 0x81, 0x44, 0x49, 0x2b, 0x38, 0xdb, 0xee, 0x09, 0xde, 0x07, 0x30, 0xe3, 0x2e, 0x1c, 0xfd, 0x0a, 0xb2, 0x54, 0x99, 0x22, 0xff, 0xa8, 0x04, 0x01, 0xad, 0x49, // syslinux-6.03/pre2/ldlinux.bss
	0xa9, 0x95, 0x68, 0x57, 0x9c, 0xd2, 0x51, 0xaf, 0xf1, 0x34, 0xfc, 0xaa, 0xa8, 0x09, 0x91, 0x60, 0x5e, 0x8f, 0xb1, 0x19, 0x74, 0x51, 0xf7, 0x51, 0xaa, 0x4d, 0x6c, 0x84, 0xbf, 0x65, 0xf4, 0xe3, // syslinux-6.03/pre15/ldlinux.sys
	0xaa, 0x18, 0x1a, 0x4e, 0xb2, 0xab, 0xb0, 0xfb, 0xbc, 0x17, 0xc4, 0xc8, 0x5a, 0x17, 0xa0, 0x0b, 0x15, 0x52, 0x4d, 0x7f, 0x2b, 0x0b, 0xea, 0xd4, 0xf8, 0x74, 0x2e, 0x70, 0xa3, 0x3a, 0xfa, 0xf8, // grub-2.04/core.img
	0xad, 0x18, 0x82, 0x0e, 0xcd, 0xe8, 0x74, 0xc7, 0x19, 0x10, 0xbd, 0x24, 0xf3, 0x03, 0x55, 0x17, 0x61, 0x03, 0xd0, 0x21, 0x78, 0x9c, 0xc0, 0x0b, 0xd4, 0x13, 0xd8, 0x0f, 0x5b, 0x02, 0x59, 0xf4, // grub-2.06@e67a551a-nonstandard-gdie/core.img
	0xb1, 0x97, 0x91, 0x33, 0xa4, 0x5e, 0x29, 0x4f, 0x54, 0xee, 0xd0, 0xbd, 0x92, 0xb7, 0xf2, 0xb9, 0xa3, 0xd2, 0x26, 0x45, 0xae, 0x28, 0x20, 0x0e, 0x4a, 0xd8, 0xfd, 0x98, 0xaf, 0x24, 0x1a, 0x07, // grub-2.05/core.img
	0xb2, 0xf0, 0xde, 0x2e, 0x01, 0xa7, 0xe3, 0x4d, 0x96, 0xa9, 0x76, 0x36, 0x43, 0x05, 0xd3, 0x9f, 0xf8, 0x49, 0xd5, 0x0a, 0xd1, 0x3a, 0xd1, 0xf8, 0xa8, 0xb4, 0x41, 0x1d, 0x1d, 0x53, 0x4e, 0x88, // syslinux-5.10/ldlinux.bss
	0xb3, 0xa8, 0xf2, 0x12, 0x89, 0x1c, 0xd2, 0x6e, 0x1a, 0xb4, 0x4a, 0xa8, 0x59, 0x8f, 0x6e, 0x64, 0xc6, 0x83, 0x19, 0x8d, 0x43, 0x00, 0x27, 0x54, 0xec, 0xbd, 0x8a, 0x78, 0xf2, 0xd0, 0xc0, 0x0b, // syslinux-6.03/pre8/ldlinux.sys
	0xb3, 0xdc, 0x31, 0x79, 0xf6, 0x2b, 0x20, 0x51, 0xc9, 0x43, 0xe5, 0x2e, 0xeb, 0xf2, 0x29, 0x8a, 0xa4, 0x7e, 0x7c, 0x0a, 0x97, 0x78, 0xe8, 0x62, 0x77, 0xa7, 0x48, 0x2a, 0x27, 0x0a, 0x7a, 0x8e, // syslinux-6.01/ldlinux.bss
	0xb4, 0x6a, 0xf2, 0x09, 0x19, 0xe9, 0xf2, 0x1f, 0xa1, 0x52, 0x37, 0x5d, 0xda, 0xc4, 0x58, 0x87, 0x08, 0xc1, 0x22, 0xb3, 0x65, 0x7f, 0x09, 0x01, 0x31, 0x4e, 0x83, 0x45, 0x49, 0xa9, 0x6c, 0xe7, // syslinux-4.07/menu.c32
	0xbc, 0x3f, 0xb7, 0x6d, 0xbc, 0xd6, 0x32, 0x37, 0xc3, 0x68, 0x8d, 0x3b, 0x55, 0x3c, 0x99, 0x41, 0x0e, 0x8f, 0xb3, 0x9e, 0x52, 0x00, 0x9d, 0xdc, 0xeb, 0xb8, 0x8e, 0xe7, 0xd2, 0x5a, 0xec, 0xa8, // grub-2.03.5/core.img
	0xbf, 0xf9, 0xc5, 0x89, 0x59, 0x70, 0x4d, 0x1e, 0xbe, 0x25, 0xb4, 0x43, 0xc8, 0x92, 0x3a, 0x42, 0xed, 0x89, 0xaf, 0x8b, 0xd2, 0x92, 0xe7, 0xd7, 0xcb, 0xeb, 0xca, 0xcc, 0xc0, 0x7c, 0xc7, 0x19, // syslinux-5.11/ldlinux.sys
	0xc3, 0x3b, 0x31, 0x5f, 0xec, 0xe4, 0xad, 0xc4, 0xc3, 0xb2, 0x75, 0x13, 0x22, 0x84, 0x66, 0xe4, 0x44, 0x99, 0xcd, 0xa7, 0xfd, 0x63, 0x97, 0xc5, 0xab, 0xe0, 0xf8, 0xce, 0x4f, 0xe3, 0x45, 0x39, // syslinux-5.10/ldlinux.c32
	0xcc, 0x40, 0xba, 0x03, 0x49, 0x78, 0x2c, 0xb4, 0xc9, 0x02, 0x1e, 0x54, 0xdc, 0xc0, 0xa4, 0x54, 0x0c, 0x3a, 0x8b, 0x96, 0x08, 0x8b, 0x3a, 0x56, 0x48, 0x67, 0x19, 0x26, 0xef, 0x44, 0xd2, 0xf0, // syslinux-6.04/ldlinux.bss
	0xcd, 0x08, 0xf4, 0x81, 0xfb, 0x0c, 0xda, 0x19, 0xe1, 0x78, 0x6d, 0x43, 0x8d, 0x5c, 0x13, 0x93, 0xf7, 0xbf, 0xf3, 0x81, 0x31, 0x44, 0x4a, 0x0a, 0x5e, 0x54, 0xc1, 0x04, 0xed, 0x1d, 0x4d, 0xad, // syslinux-6.03/pre10/ldlinux.bss
	0xd0, 0x15, 0x48, 0xaf, 0x46, 0xf6, 0x4f, 0xed, 0x16, 0x69, 0xa5, 0x6e, 0x69, 0x31, 0x56, 0xb5, 0xde, 0x39, 0xd6, 0xbf, 0xa8, 0x6e, 0x9e, 0x07, 0x13, 0x1d, 0x70, 0x06, 0x3c, 0x56, 0x3d, 0xd0, // grub4dos-0.4.5c/grldr.mbr
	0xd2, 0xc6, 0x93, 0x8d, 0xae, 0x5a, 0xd7, 0x16, 0x0e, 0x9e, 0x6c, 0x61, 0xef, 0x46, 0xb7, 0xfd, 0x14, 0x6e, 0x30, 0xc0, 0x3f, 0xdc, 0x8f, 0x5c, 0x6d, 0xbd, 0xeb, 0x86, 0x22, 0xc8, 0xa7, 0xbd, // syslinux-5.00/ldlinux.bss
	0xd3, 0x47, 0x2c, 0x02, 0x26, 0x3a, 0xcf, 0x9c, 0xd1, 0xda, 0x5d, 0xb5, 0x1e, 0x26, 0x3c, 0x54, 0x84, 0xba, 0xd1, 0x3e, 0xa6, 0x86, 0x18, 0xc4, 0x03, 0xd9, 0xcb, 0x01, 0xca, 0x07, 0x0a, 0xee, // syslinux-6.04/ldlinux.c32
	0xd5, 0x50, 0x39, 0xef, 0xb6, 0x8d, 0x6e, 0xec, 0xde, 0x68, 0x61, 0xc9, 0x0b, 0xa9, 0xb7, 0x99, 0x44, 0xd1, 0xaa, 0x8b, 0xc3, 0xd6, 0x01, 0xfb, 0x80, 0xfd, 0x08, 0x7b, 0xc6, 0x13, 0x63, 0xf8, // syslinux-6.02/ldlinux.sys
	0xd6, 0xa7, 0x0c, 0xe1, 0x90, 0x2b, 0x22, 0x1a, 0xcb, 0xc1, 0xd5, 0xf5, 0x94, 0x24, 0xa8, 0x22, 0x7b, 0x62, 0xfe, 0x6d, 0x85, 0x34, 0x86, 0x0e, 0x2b, 0x6e, 0x75, 0x7e, 0x01, 0x02, 0xc5, 0x6f, // grub4dos-0.4.6a/grldr
	0xd9, 0x1c, 0xbd, 0xf9, 0x01, 0x79, 0x5e, 0x98, 0x19, 0x84, 0x9c, 0x75, 0x3e, 0xa5, 0x3a, 0xd8, 0x59, 0xc6, 0xf2, 0x5d, 0x59, 0x17, 0xa6, 0x75, 0x92, 0x58, 0x3f, 0xd6, 0x5e, 0x6d, 0x00, 0xf0, // syslinux-6.03/pre18/ldlinux.bss
	0xda, 0xa3, 0x06, 0xa9, 0x97, 0x0e, 0x37, 0x7d, 0xb4, 0x4c, 0xe7, 0xfa, 0xd8, 0xdd, 0x01, 0xe9, 0xe8, 0x53, 0xc6, 0x8b, 0x1e, 0x3d, 0x84, 0xf3, 0x2a, 0xaf, 0x42, 0x24, 0x6f, 0x77, 0xae, 0x3d, // grub-2.02/core.img
	0xe7, 0x87, 0x08, 0xc7, 0x2c, 0x49, 0x8e, 0xcc, 0x14, 0x9e, 0x30, 0xf0, 0xa9, 0x85, 0xd3, 0x73, 0xa3, 0x00, 0xad, 0x5c, 0xf7, 0xd2, 0x88, 0xc7, 0x7d, 0xe8, 0x05, 0x5e, 0x25, 0x66, 0x28, 0x1f, // syslinux-6.03/pre18/ldlinux.sys
	0xe9, 0x00, 0x3a, 0x56, 0x28, 0x43, 0xa3, 0x7b, 0xec, 0x4b, 0xef, 0xd4, 0xbb, 0x71, 0xfe, 0x3c, 0x5e, 0x6a, 0x50, 0x4d, 0x0b, 0x99, 0x36, 0xaf, 0xb6, 0x9e, 0xdc, 0x37, 0x2f, 0x7c, 0x41, 0xef, // syslinux-6.03/pre14/ldlinux.bss
	0xec, 0x31, 0xc2, 0xb7, 0xba, 0x09, 0xee, 0x69, 0xee, 0xf8, 0x55, 0x31, 0xe2, 0xc2, 0x4f, 0x22, 0x4a, 0xf7, 0xd9, 0x4a, 0x72, 0x25, 0xb2, 0x6f, 0x4f, 0x57, 0xf7, 0x70, 0x5e, 0x7d, 0xda, 0x1a, // syslinux-5.10/ldlinux.sys
	0xec, 0x94, 0x7e, 0x1c, 0x1e, 0x3f, 0xbe, 0xca, 0x4f, 0x55, 0x6d, 0x7a, 0x1a, 0x41, 0x15, 0x60, 0x89, 0x41, 0xf5, 0x48, 0x35, 0xe0, 0x7e, 0x8e, 0x89, 0xf7, 0x02, 0x85, 0x3d, 0xb7, 0xae, 0x2a, // syslinux-6.03/pre15/ldlinux.bss
	0xec, 0xfd, 0xbc, 0x3f, 0x4f, 0x2e, 0x4d, 0x99, 0x16, 0x9c, 0xdd, 0xfb, 0x15, 0x2d, 0x92, 0x4d, 0x7d, 0xe6, 0x8a, 0xb6, 0x7f, 0x4d, 0x12, 0x54, 0x40, 0xfa, 0xbc, 0x9f, 0x00, 0x46, 0xd5, 0xbc, // syslinux-6.03/pre5/ldlinux.bss
	0xee, 0xed, 0xc0, 0x4c, 0x13, 0x73, 0xb5, 0xc4, 0x04, 0x4d, 0x1f, 0xde, 0x0d, 0x2f, 0xb8, 0xe2, 0x8c, 0x74, 0xb1, 0x02, 0x9c, 0x99, 0xed, 0x67, 0x0a, 0x15, 0x98, 0x3f, 0x18, 0xa0, 0x4d, 0x36, // syslinux-6.02/ldlinux.c32
	0xf5, 0x40, 0x26, 0x1c, 0x09, 0x7d, 0xbd, 0x8a, 0x8a, 0x12, 0x9b, 0x68, 0x99, 0x5f, 0x33, 0xab, 0xe7, 0x1c, 0x29, 0x40, 0xf8, 0x87, 0xc6, 0x68, 0x9b, 0xf8, 0xdc, 0x3f, 0x1a, 0xcf, 0x0b, 0x44, // syslinux-6.03/pre17/ldlinux.bss
	0xf5, 0xc6, 0xe8, 0xe2, 0xc1, 0xeb, 0x73, 0x80, 0x28, 0x5f, 0xa9, 0xcb, 0x1c, 0x91, 0x68, 0xe9, 0x2d, 0x5b, 0x3b, 0x55, 0xcd, 0xe0, 0x52, 0xc0, 0x43, 0xba, 0x81, 0xed, 0x17, 0xb9, 0xac, 0xef, // grub4dos-0.4.6a/grldr.mbr
	0xfb, 0x0a, 0x23, 0xca, 0x4d, 0x22, 0xfd, 0xd2, 0xad, 0x4e, 0xfa, 0x1b, 0x21, 0x08, 0xb6, 0x60, 0xd2, 0xff, 0xa3, 0xf2, 0xfb, 0xdd, 0x25, 0x32, 0xdc, 0xf1, 0x3f, 0x49, 0x33, 0x28, 0x46, 0x7b, // syslinux-6.03/pre11/ldlinux.bss
	0xfb, 0x49, 0xfd, 0x45, 0x8c, 0xaf, 0x47, 0x5a, 0x16, 0x05, 0x5e, 0x4a, 0x75, 0x5a, 0xc3, 0xe3, 0x95, 0x52, 0xf4, 0xe9, 0x6c, 0xa2, 0x0d, 0xea, 0x53, 0xf5, 0xc4, 0x09, 0x2b, 0x68, 0xfd, 0x4f, // syslinux-6.00/ldlinux.sys
};

/*
 * Contains the SHA-1 thumbprints of the issuer certificate of the official
 * Secure Boot signing authority (i.e. Microsoft).
 */
static const char db_sb_active_txt[] =
	// 'Microsoft Windows Production PCA 2011'
	"580a6f4cc4e4b669b9ebdc1b2b3e087b80d0678d\n"
	// 'Microsoft Corporation UEFI CA 2011'
	"46def63b5ce61cf8ba0de2e6639c1019d0ed14f3\n"
	// 'Windows UEFI CA 2023'
	"45a0fa32604773c82433c3b7d59e7466b3ac0c67\n"
	// 'Microsoft UEFI CA 2023'
	"b5eeb4a6706048073f0ed296e7f580a790b59eaa";

/*
 * Contains the SHA-1 thumbprints of certificates that are being revoked by DBX.
 * This only includes the 'Microsoft Windows Production PCA 2011' for now.
 */
static const char db_sb_revoked_txt[] =
	"580a6f4cc4e4b669b9ebdc1b2b3e087b80d0678d";

/*
 * Extended SBATLevel.txt that merges Linux SBAT with Microsoft's SVN
 * See https://github.com/pbatard/rufus/issues/2244#issuecomment-2243661539
 * Use as fallback when https://rufus.ie/sbat_level.txt cannot be accessed.
 */
static const char db_sbat_level_txt[] =
	"sbat,1,2025051000\n"
	"shim,4\n"
	"grub,5\n"
	"grub.proxmox,2\n"
	"BOOTMGRSECURITYVERSIONNUMBER,0x70000";
