"""Test ModuleFinder."""

from __future__ import annotations

import os
import py_compile
import sys
from importlib.machinery import BYTECODE_SUFFIXES, SOURCE_SUFFIXES

import pytest

from cx_Freeze import ConstantsModule, ModuleFinder

from .datatest import (
    ABSOLUTE_IMPORT_TEST,
    BYTECODE_TEST,
    CODING_DEFAULT_UTF8_TEST,
    CODING_EXPLICIT_CP1252_TEST,
    CODING_EXPLICIT_UTF8_TEST,
    EDITABLE_PACKAGE_TEST,
    EXTENDED_OPARGS_TEST,
    MAYBE_TEST,
    MAYBE_TEST_NEW,
    NAMESPACE_TEST,
    NAMESPACE_TEST_1,
    NAMESPACE_TEST_2,
    PACKAGE_TEST,
    RELATIVE_IMPORT_TEST,
    RELATIVE_IMPORT_TEST_2,
    RELATIVE_IMPORT_TEST_3,
    RELATIVE_IMPORT_TEST_4,
    SAME_NAME_AS_BAD_TEST,
    SCAN_CODE_IMPORT_CALL_TEST,
    SCAN_CODE_IMPORT_MODULE_TEST,
    SCAN_CODE_TEST,
    SUB_PACKAGE_TEST,
)

# Each test description is a list of 5 items:
#
# 1. a module name that will be imported by ModuleFinder
# 2. a list of module names that ModuleFinder is required to find
# 3. a list of module names that ModuleFinder should complain
#    about because they are not found
# 4. a list of module names that ModuleFinder should complain
#    about because they MAY be not found
# 5. a string specifying packages to create; the format is obvious imo.
#
# Each package will be created in test_dir, and test_dir will be
# removed after the tests again.
# ModuleFinder searches in a path that contains test_dir, plus
# the standard path search directory.


def _do_test(
    test_dir,
    import_this,
    modules,
    missing,  # noqa: ARG001
    maybe_missing,  # noqa: ARG001
    source,
    report=False,
    debug=0,  # noqa: ARG001
    modulefinder_class=ModuleFinder,
    **kwargs,
) -> None:
    test_dir.create(source)
    path = kwargs.pop("path", sys.path)
    finder = modulefinder_class(
        ConstantsModule(), path=[test_dir.path, *path], **kwargs
    )
    finder.include_module(import_this)
    if report:
        finder.report_missing_modules()
    modules = sorted(set(modules))
    found = sorted([module.name for module in finder.modules])
    # check if we found what we expected, not more, not less
    assert found == modules


@pytest.mark.parametrize(
    ("import_this", "modules", "missing", "maybe_missing", "source"),
    [
        ABSOLUTE_IMPORT_TEST,
        CODING_DEFAULT_UTF8_TEST,
        CODING_EXPLICIT_CP1252_TEST,
        CODING_EXPLICIT_UTF8_TEST,
        EXTENDED_OPARGS_TEST,
        MAYBE_TEST,
        MAYBE_TEST_NEW,
        NAMESPACE_TEST,
        NAMESPACE_TEST_1,
        NAMESPACE_TEST_2,
        PACKAGE_TEST,
        RELATIVE_IMPORT_TEST,
        RELATIVE_IMPORT_TEST_2,
        RELATIVE_IMPORT_TEST_3,
        RELATIVE_IMPORT_TEST_4,
        SAME_NAME_AS_BAD_TEST,
        SUB_PACKAGE_TEST,
    ],
    ids=[
        "absolute_import_test",
        "coding_default_utf8_test",
        "coding_explicit_cp1252_test",
        "coding_explicit_utf8_test",
        "extended_opargs_test",
        "maybe_test",
        "maybe_test_new",
        "namespace_test",
        "namespace_test_1",
        "namespace_test_2",
        "package_test",
        "relative_import_test",
        "relative_import_test_2",
        "relative_import_test_3",
        "relative_import_test_4",
        "same_name_as_bad_test",
        "sub_package_test",
    ],
)
def test_finder(
    tmp_package,
    import_this,
    modules,
    missing,
    maybe_missing,
    source,
) -> None:
    """Provides test cases for ModuleFinder class."""
    _do_test(tmp_package, import_this, modules, missing, maybe_missing, source)


def test_bytecode(tmp_package) -> None:
    """Provides bytecode test case for ModuleFinder class."""
    base_path = tmp_package.path / "a"
    source_path = base_path.with_suffix(SOURCE_SUFFIXES[0])
    bytecode_path = base_path.with_suffix(BYTECODE_SUFFIXES[0])
    with source_path.open("wb") as file:
        file.write(b"testing_modulefinder = True\n")
    py_compile.compile(os.fspath(source_path), cfile=os.fspath(bytecode_path))
    os.remove(source_path)
    _do_test(tmp_package, *BYTECODE_TEST)


@pytest.mark.parametrize(
    ("import_this", "modules", "missing", "maybe_missing", "source"),
    [
        SCAN_CODE_TEST,
        SCAN_CODE_IMPORT_CALL_TEST,
        SCAN_CODE_IMPORT_MODULE_TEST,
    ],
    ids=[
        "scan_code_test",
        "scan_code_import_call_test",
        "scan_code_import_module_test",
    ],
)
def test_scancode(
    tmp_package,
    import_this,
    modules,
    missing,
    maybe_missing,
    source,
) -> None:
    """Provides test cases for ModuleFinder class."""
    _do_test(
        tmp_package,
        import_this,
        modules,
        missing,
        maybe_missing,
        source,
        path=[],
    )


def test_zip_include_packages(tmp_package) -> None:
    """Provides test cases for ModuleFinder class."""
    _do_test(
        tmp_package,
        *SUB_PACKAGE_TEST,
        zip_exclude_packages=["*"],
        zip_include_packages=["p"],
        zip_include_all_packages=False,
    )


def test_zip_exclude_packages(tmp_package) -> None:
    """Provides test cases for ModuleFinder class."""
    _do_test(
        tmp_package,
        *SUB_PACKAGE_TEST,
        zip_exclude_packages=["p"],
        zip_include_packages=["*"],
        zip_include_all_packages=True,
    )


def test_editable_packages(tmp_package) -> None:
    """Provides test cases for ModuleFinder class."""
    tmp_package.create(EDITABLE_PACKAGE_TEST[4])
    tmp_package.install(["-e", f"{tmp_package.path}/foo-bar"], backend="pip")
    _do_test(
        tmp_package,
        *EDITABLE_PACKAGE_TEST,
    )
