// META: title=WebCryptoAPI: digest() cSHAKE algorithms
// META: timeout=long

var subtle = crypto.subtle; // Change to test prefixed implementations

var sourceData = {
  empty: new Uint8Array(0),
  short: new Uint8Array([
    21, 110, 234, 124, 193, 76, 86, 203, 148, 219, 3, 10, 74, 157, 149, 255,
  ]),
  medium: new Uint8Array([
    182, 200, 249, 223, 100, 140, 208, 136, 183, 15, 56, 231, 65, 151, 177, 140,
    184, 30, 30, 67, 80, 213, 11, 204, 184, 251, 90, 115, 121, 200, 123, 178,
    227, 214, 237, 84, 97, 237, 30, 159, 54, 243, 64, 163, 150, 42, 68, 107,
    129, 91, 121, 75, 75, 212, 58, 68, 3, 80, 32, 119, 178, 37, 108, 200, 7,
    131, 127, 58, 172, 209, 24, 235, 75, 156, 43, 174, 184, 151, 6, 134, 37,
    171, 172, 161, 147,
  ]),
};

// Test different output lengths for cSHAKE
var digestLengths = [0, 256, 384, 512];

var digestedData = {
  cSHAKE128: {
    0: {
      empty: new Uint8Array([]),
      short: new Uint8Array([]),
      medium: new Uint8Array([]),
    },
    256: {
      empty: new Uint8Array([
        127, 156, 43, 164, 232, 143, 130, 125, 97, 96, 69, 80, 118, 5, 133, 62,
        215, 59, 128, 147, 246, 239, 188, 136, 235, 26, 110, 172, 250, 102, 239,
        38,
      ]),
      short: new Uint8Array([
        222, 166, 45, 115, 230, 181, 156, 247, 37, 208, 50, 13, 102, 0, 137,
        164, 71, 92, 187, 211, 184, 83, 158, 54, 105, 31, 21, 13, 71, 85, 103,
        148,
      ]),
      medium: new Uint8Array([
        177, 172, 213, 58, 3, 231, 106, 34, 30, 82, 234, 87, 142, 4, 47, 104,
        106, 104, 195, 209, 201, 131, 42, 177, 130, 133, 207, 79, 48, 76, 163,
        45,
      ]),
    },
    384: {
      empty: new Uint8Array([
        127, 156, 43, 164, 232, 143, 130, 125, 97, 96, 69, 80, 118, 5, 133, 62,
        215, 59, 128, 147, 246, 239, 188, 136, 235, 26, 110, 172, 250, 102, 239,
        38, 60, 177, 238, 169, 136, 0, 75, 147, 16, 60, 251, 10, 238, 253, 42,
        104,
      ]),
      short: new Uint8Array([
        222, 166, 45, 115, 230, 181, 156, 247, 37, 208, 50, 13, 102, 0, 137,
        164, 71, 92, 187, 211, 184, 83, 158, 54, 105, 31, 21, 13, 71, 85, 103,
        148, 240, 55, 64, 1, 183, 136, 138, 188, 54, 152, 212, 11, 137, 174, 49,
        52,
      ]),
      medium: new Uint8Array([
        177, 172, 213, 58, 3, 231, 106, 34, 30, 82, 234, 87, 142, 4, 47, 104,
        106, 104, 195, 209, 201, 131, 42, 177, 130, 133, 207, 79, 48, 76, 163,
        45, 63, 170, 9, 252, 130, 170, 225, 66, 211, 223, 205, 121, 5, 138, 93,
        92,
      ]),
    },
    512: {
      empty: new Uint8Array([
        127, 156, 43, 164, 232, 143, 130, 125, 97, 96, 69, 80, 118, 5, 133, 62,
        215, 59, 128, 147, 246, 239, 188, 136, 235, 26, 110, 172, 250, 102, 239,
        38, 60, 177, 238, 169, 136, 0, 75, 147, 16, 60, 251, 10, 238, 253, 42,
        104, 110, 1, 250, 74, 88, 232, 163, 99, 156, 168, 161, 227, 249, 174,
        87, 226,
      ]),
      short: new Uint8Array([
        222, 166, 45, 115, 230, 181, 156, 247, 37, 208, 50, 13, 102, 0, 137,
        164, 71, 92, 187, 211, 184, 83, 158, 54, 105, 31, 21, 13, 71, 85, 103,
        148, 240, 55, 64, 1, 183, 136, 138, 188, 54, 152, 212, 11, 137, 174, 49,
        52, 233, 51, 245, 26, 132, 202, 127, 218, 136, 12, 59, 253, 217, 220,
        58, 94,
      ]),
      medium: new Uint8Array([
        177, 172, 213, 58, 3, 231, 106, 34, 30, 82, 234, 87, 142, 4, 47, 104,
        106, 104, 195, 209, 201, 131, 42, 177, 130, 133, 207, 79, 48, 76, 163,
        45, 63, 170, 9, 252, 130, 170, 225, 66, 211, 223, 205, 121, 5, 138, 93,
        92, 60, 17, 189, 45, 17, 195, 248, 169, 51, 31, 98, 172, 221, 186, 225,
        93,
      ]),
    },
  },
  cSHAKE256: {
    0: {
      empty: new Uint8Array([]),
      short: new Uint8Array([]),
      medium: new Uint8Array([]),
    },
    256: {
      empty: new Uint8Array([
        70, 185, 221, 43, 11, 168, 141, 19, 35, 59, 63, 235, 116, 62, 235, 36,
        63, 205, 82, 234, 98, 184, 27, 130, 181, 12, 39, 100, 110, 213, 118, 47,
      ]),
      short: new Uint8Array([
        23, 56, 17, 63, 90, 187, 62, 229, 50, 14, 225, 138, 162, 102, 195, 97,
        122, 116, 117, 219, 216, 237, 154, 152, 89, 148, 253, 221, 97, 18, 173,
        153,
      ]),
      medium: new Uint8Array([
        65, 70, 193, 61, 134, 217, 188, 24, 107, 11, 48, 154, 182, 161, 36, 238,
        12, 116, 186, 38, 184, 198, 13, 204, 123, 62, 213, 5, 150, 154, 168,
        209,
      ]),
    },
    384: {
      empty: new Uint8Array([
        70, 185, 221, 43, 11, 168, 141, 19, 35, 59, 63, 235, 116, 62, 235, 36,
        63, 205, 82, 234, 98, 184, 27, 130, 181, 12, 39, 100, 110, 213, 118, 47,
        215, 93, 196, 221, 216, 192, 242, 0, 203, 5, 1, 157, 103, 181, 146, 246,
      ]),
      short: new Uint8Array([
        23, 56, 17, 63, 90, 187, 62, 229, 50, 14, 225, 138, 162, 102, 195, 97,
        122, 116, 117, 219, 216, 237, 154, 152, 89, 148, 253, 221, 97, 18, 173,
        153, 158, 200, 226, 235, 223, 234, 251, 150, 231, 111, 107, 179, 163,
        173, 186, 67,
      ]),
      medium: new Uint8Array([
        65, 70, 193, 61, 134, 217, 188, 24, 107, 11, 48, 154, 182, 161, 36, 238,
        12, 116, 186, 38, 184, 198, 13, 204, 123, 62, 213, 5, 150, 154, 168,
        209, 144, 40, 198, 49, 121, 153, 160, 133, 177, 230, 182, 167, 133, 206,
        79, 246,
      ]),
    },
    512: {
      empty: new Uint8Array([
        70, 185, 221, 43, 11, 168, 141, 19, 35, 59, 63, 235, 116, 62, 235, 36,
        63, 205, 82, 234, 98, 184, 27, 130, 181, 12, 39, 100, 110, 213, 118, 47,
        215, 93, 196, 221, 216, 192, 242, 0, 203, 5, 1, 157, 103, 181, 146, 246,
        252, 130, 28, 73, 71, 154, 180, 134, 64, 41, 46, 172, 179, 183, 196,
        190,
      ]),
      short: new Uint8Array([
        23, 56, 17, 63, 90, 187, 62, 229, 50, 14, 225, 138, 162, 102, 195, 97,
        122, 116, 117, 219, 216, 237, 154, 152, 89, 148, 253, 221, 97, 18, 173,
        153, 158, 200, 226, 235, 223, 234, 251, 150, 231, 111, 107, 179, 163,
        173, 186, 67, 218, 96, 240, 12, 209, 36, 150, 223, 90, 243, 226, 138,
        230, 211, 222, 66,
      ]),
      medium: new Uint8Array([
        65, 70, 193, 61, 134, 217, 188, 24, 107, 11, 48, 154, 182, 161, 36, 238,
        12, 116, 186, 38, 184, 198, 13, 204, 123, 62, 213, 5, 150, 154, 168,
        209, 144, 40, 198, 49, 121, 153, 160, 133, 177, 230, 182, 167, 133, 206,
        79, 246, 50, 174, 178, 116, 147, 34, 126, 68, 35, 47, 183, 179, 149, 33,
        65, 123,
      ]),
    },
  },
};

// Test cSHAKE digest algorithms with variable output lengths
Object.keys(digestedData).forEach(function (alg) {
  digestLengths.forEach(function (length) {
    Object.keys(sourceData).forEach(function (size) {
      promise_test(function (test) {
        return crypto.subtle
          .digest({ name: alg, length: length }, sourceData[size])
          .then(function (result) {
            assert_true(
              equalBuffers(result, digestedData[alg][length][size]),
              'digest matches expected'
            );
          });
      }, alg + ' with ' + length + ' bit output and ' + size + ' source data');

      promise_test(function (test) {
        var buffer = new Uint8Array(sourceData[size]);
        return crypto.subtle
          .digest({ name: alg, length: length }, buffer)
          .then(function (result) {
            // Alter the buffer after calling digest
            if (buffer.length > 0) {
              buffer[0] = ~buffer[0];
            }
            assert_true(
              equalBuffers(result, digestedData[alg][length][size]),
              'digest matches expected'
            );
          });
      }, alg +
        ' with ' +
        length +
        ' bit output and ' +
        size +
        ' source data and altered buffer after call');
    });
  });
});

function equalBuffers(a, b) {
  if (a.byteLength !== b.byteLength) {
    return false;
  }
  var aBytes = new Uint8Array(a);
  var bBytes = new Uint8Array(b);
  for (var i = 0; i < a.byteLength; i++) {
    if (aBytes[i] !== bBytes[i]) {
      return false;
    }
  }
  return true;
}
