!!****f* ABINIT/contistr03
!! NAME
!! contistr03
!!
!! FUNCTION
!! Carries out specialized metric tensor operations needed for contraction
!! of the 2nd strain derivative of the l=0,1,2,3 nonlocal Kleinman-Bylander
!! pseudopotential operation.  Derivatives are wrt a pair of cartesian
!! strain components.
!! Full advantage is taken of the full permutational symmetry of these
!! tensors.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2025 ABINIT group (DRH)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! INPUTS
!!  istr=1,...6 specifies cartesian strain component 11,22,33,32,31,21
!!  rank=angular momentum
!!  gm(3,3)=metric tensor (array is symmetric but stored as 3x3)
!!  gprimd(3,3)=reciprocal space dimensional primitive translations
!!  aa(2,*)=unique elements of complex right-hand tensor
!!  bb(2,*)=unique elements of complex left-hand tensor
!!
!! OUTPUT
!!  eisnl(3)=contraction for nonlocal internal strain derivative energy
!!
!! NOTES
!! All tensors are stored in a compressed storage mode defined below;
!! input and output conform to this scheme.
!! When tensor elements occur repeatedly due to symmetry, the
!! WEIGHT IS INCLUDED in the output tensor element to simplify later
!! contractions with other tensors of the same rank and form, i.e. the
!! next contraction is then simply a dot product over the unique elements.
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"


module m_contistr03
contains
!!**



subroutine contistr03(istr,rank,gm,gprimd,eisnl,aa,bb)

 use defs_basis
 use m_abicore
 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: istr,rank
!arrays
 real(dp),intent(in) :: aa(2,((rank+1)*(rank+2))/2),bb(2,((rank+4)*(rank+5))/2)
 real(dp),intent(in) :: gm(3,3),gprimd(3,3)
 real(dp),intent(out) :: eisnl(3)

!Local variables-------------------------------
!scalars
 integer :: ii,jj,ka,kb
!arrays
 integer,parameter :: idx(12)=(/1,1,2,2,3,3,3,2,3,1,2,1/)
 real(dp) :: dgm(3,3),tmp(2,3)
 real(dp),allocatable :: cm(:,:,:)

! *************************************************************************

 ABI_MALLOC(cm,(3,((rank+1)*(rank+2))/2,((rank+4)*(rank+5))/2))

 ka=idx(2*istr-1);kb=idx(2*istr)

 do ii = 1,3
   dgm(:,ii)=-(gprimd(ka,:)*gprimd(kb,ii)+gprimd(kb,:)*gprimd(ka,ii))
 end do

 cm(:,:,:)=0.d0
!
!The code below was written by a Mathematica program and formatted by
!a combination of editing scripts.  It is not intended to be read
!by human beings, and certainly not to be modified by one.  Conceivably
!it could be shortened somewhat by identifying common subexpressions.
!

 if(rank==0)then
   cm(1,1,1)=dgm(1,1)
   cm(1,1,2)=dgm(2,2)
   cm(1,1,3)=dgm(3,3)
   cm(1,1,4)=2*dgm(2,3)
   cm(1,1,5)=2*dgm(1,3)
   cm(1,1,6)=2*dgm(1,2)
   cm(2,1,2)=2*dgm(1,2)
   cm(2,1,4)=2*dgm(1,3)
   cm(2,1,6)=dgm(1,1)
   cm(2,1,7)=dgm(2,2)
   cm(2,1,8)=dgm(3,3)
   cm(2,1,9)=2*dgm(2,3)
   cm(3,1,3)=2*dgm(1,3)
   cm(3,1,4)=2*dgm(1,2)
   cm(3,1,5)=dgm(1,1)
   cm(3,1,8)=2*dgm(2,3)
   cm(3,1,9)=dgm(2,2)
   cm(3,1,10)=dgm(3,3)
 elseif(rank==1)then
   cm(1,1,1)=gm(1,1)*dgm(1,1)
   cm(1,2,1)=gm(1,2)*dgm(1,1)
   cm(1,3,1)=gm(1,3)*dgm(1,1)
   cm(1,1,2)=2*gm(1,2)*dgm(1,2)+gm(1,1)*dgm(2,2)
   cm(1,2,2)=2*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2)
   cm(1,3,2)=2*gm(2,3)*dgm(1,2)+gm(1,3)*dgm(2,2)
   cm(1,1,3)=2*gm(1,3)*dgm(1,3)+gm(1,1)*dgm(3,3)
   cm(1,2,3)=2*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3)
   cm(1,3,3)=2*gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3)
   cm(1,1,4)=2*(gm(1,3)*dgm(1,2)+gm(1,2)*dgm(1,3)+gm(1,1)*dgm(2,3))
   cm(1,2,4)=2*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3)+gm(1,2)*dgm(2,3))
   cm(1,3,4)=2*(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3)+gm(1,3)*dgm(2,3))
   cm(1,1,5)=gm(1,3)*dgm(1,1)+2*gm(1,1)*dgm(1,3)
   cm(1,2,5)=gm(2,3)*dgm(1,1)+2*gm(1,2)*dgm(1,3)
   cm(1,3,5)=gm(3,3)*dgm(1,1)+2*gm(1,3)*dgm(1,3)
   cm(1,1,6)=gm(1,2)*dgm(1,1)+2*gm(1,1)*dgm(1,2)
   cm(1,2,6)=gm(2,2)*dgm(1,1)+2*gm(1,2)*dgm(1,2)
   cm(1,3,6)=gm(2,3)*dgm(1,1)+2*gm(1,3)*dgm(1,2)
   cm(1,1,7)=gm(1,2)*dgm(2,2)
   cm(1,2,7)=gm(2,2)*dgm(2,2)
   cm(1,3,7)=gm(2,3)*dgm(2,2)
   cm(1,1,8)=2*gm(1,3)*dgm(2,3)+gm(1,2)*dgm(3,3)
   cm(1,2,8)=2*gm(2,3)*dgm(2,3)+gm(2,2)*dgm(3,3)
   cm(1,3,8)=2*gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3)
   cm(1,1,9)=gm(1,3)*dgm(2,2)+2*gm(1,2)*dgm(2,3)
   cm(1,2,9)=gm(2,3)*dgm(2,2)+2*gm(2,2)*dgm(2,3)
   cm(1,3,9)=gm(3,3)*dgm(2,2)+2*gm(2,3)*dgm(2,3)
   cm(1,1,10)=gm(1,3)*dgm(3,3)
   cm(1,2,10)=gm(2,3)*dgm(3,3)
   cm(1,3,10)=gm(3,3)*dgm(3,3)
   cm(2,1,2)=gm(1,2)*dgm(1,1)+2*gm(1,1)*dgm(1,2)
   cm(2,2,2)=gm(2,2)*dgm(1,1)+2*gm(1,2)*dgm(1,2)
   cm(2,3,2)=gm(2,3)*dgm(1,1)+2*gm(1,3)*dgm(1,2)
   cm(2,1,4)=gm(1,3)*dgm(1,1)+2*gm(1,1)*dgm(1,3)
   cm(2,2,4)=gm(2,3)*dgm(1,1)+2*gm(1,2)*dgm(1,3)
   cm(2,3,4)=gm(3,3)*dgm(1,1)+2*gm(1,3)*dgm(1,3)
   cm(2,1,6)=gm(1,1)*dgm(1,1)
   cm(2,2,6)=gm(1,2)*dgm(1,1)
   cm(2,3,6)=gm(1,3)*dgm(1,1)
   cm(2,1,7)=2*gm(1,2)*dgm(1,2)+gm(1,1)*dgm(2,2)
   cm(2,2,7)=2*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2)
   cm(2,3,7)=2*gm(2,3)*dgm(1,2)+gm(1,3)*dgm(2,2)
   cm(2,1,8)=2*gm(1,3)*dgm(1,3)+gm(1,1)*dgm(3,3)
   cm(2,2,8)=2*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3)
   cm(2,3,8)=2*gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3)
   cm(2,1,9)=2*(gm(1,3)*dgm(1,2)+gm(1,2)*dgm(1,3)+gm(1,1)*dgm(2,3))
   cm(2,2,9)=2*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3)+gm(1,2)*dgm(2,3))
   cm(2,3,9)=2*(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3)+gm(1,3)*dgm(2,3))
   cm(2,1,11)=gm(1,2)*dgm(2,2)
   cm(2,2,11)=gm(2,2)*dgm(2,2)
   cm(2,3,11)=gm(2,3)*dgm(2,2)
   cm(2,1,12)=2*gm(1,3)*dgm(2,3)+gm(1,2)*dgm(3,3)
   cm(2,2,12)=2*gm(2,3)*dgm(2,3)+gm(2,2)*dgm(3,3)
   cm(2,3,12)=2*gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3)
   cm(2,1,13)=gm(1,3)*dgm(2,2)+2*gm(1,2)*dgm(2,3)
   cm(2,2,13)=gm(2,3)*dgm(2,2)+2*gm(2,2)*dgm(2,3)
   cm(2,3,13)=gm(3,3)*dgm(2,2)+2*gm(2,3)*dgm(2,3)
   cm(2,1,14)=gm(1,3)*dgm(3,3)
   cm(2,2,14)=gm(2,3)*dgm(3,3)
   cm(2,3,14)=gm(3,3)*dgm(3,3)
   cm(3,1,3)=gm(1,3)*dgm(1,1)+2*gm(1,1)*dgm(1,3)
   cm(3,2,3)=gm(2,3)*dgm(1,1)+2*gm(1,2)*dgm(1,3)
   cm(3,3,3)=gm(3,3)*dgm(1,1)+2*gm(1,3)*dgm(1,3)
   cm(3,1,4)=gm(1,2)*dgm(1,1)+2*gm(1,1)*dgm(1,2)
   cm(3,2,4)=gm(2,2)*dgm(1,1)+2*gm(1,2)*dgm(1,2)
   cm(3,3,4)=gm(2,3)*dgm(1,1)+2*gm(1,3)*dgm(1,2)
   cm(3,1,5)=gm(1,1)*dgm(1,1)
   cm(3,2,5)=gm(1,2)*dgm(1,1)
   cm(3,3,5)=gm(1,3)*dgm(1,1)
   cm(3,1,8)=2*(gm(1,3)*dgm(1,2)+gm(1,2)*dgm(1,3)+gm(1,1)*dgm(2,3))
   cm(3,2,8)=2*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3)+gm(1,2)*dgm(2,3))
   cm(3,3,8)=2*(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3)+gm(1,3)*dgm(2,3))
   cm(3,1,9)=2*gm(1,2)*dgm(1,2)+gm(1,1)*dgm(2,2)
   cm(3,2,9)=2*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2)
   cm(3,3,9)=2*gm(2,3)*dgm(1,2)+gm(1,3)*dgm(2,2)
   cm(3,1,10)=2*gm(1,3)*dgm(1,3)+gm(1,1)*dgm(3,3)
   cm(3,2,10)=2*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3)
   cm(3,3,10)=2*gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3)
   cm(3,1,12)=gm(1,3)*dgm(2,2)+2*gm(1,2)*dgm(2,3)
   cm(3,2,12)=gm(2,3)*dgm(2,2)+2*gm(2,2)*dgm(2,3)
   cm(3,3,12)=gm(3,3)*dgm(2,2)+2*gm(2,3)*dgm(2,3)
   cm(3,1,13)=gm(1,2)*dgm(2,2)
   cm(3,2,13)=gm(2,2)*dgm(2,2)
   cm(3,3,13)=gm(2,3)*dgm(2,2)
   cm(3,1,14)=2*gm(1,3)*dgm(2,3)+gm(1,2)*dgm(3,3)
   cm(3,2,14)=2*gm(2,3)*dgm(2,3)+gm(2,2)*dgm(3,3)
   cm(3,3,14)=2*gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3)
   cm(3,1,15)=gm(1,3)*dgm(3,3)
   cm(3,2,15)=gm(2,3)*dgm(3,3)
   cm(3,3,15)=gm(3,3)*dgm(3,3)
 elseif(rank==2)then
   cm(1,1,1)=gm(1,1)**2*dgm(1,1)
   cm(1,2,1)=((6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm(1,1))/4.d0
   cm(1,3,1)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm(1,1))/4.d0
   cm(1,4,1)=((6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm(1,1))/2.d0
   cm(1,5,1)=2*gm(1,1)*gm(1,3)*dgm(1,1)
   cm(1,6,1)=2*gm(1,1)*gm(1,2)*dgm(1,1)
   cm(1,1,2)=1.5d0*gm(1,2)**2*dgm(1,1)+4*gm(1,1)*gm(1,2)*dgm(1,2)&
&   +gm(1,1)*(-0.5d0*gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))
   cm(1,2,2)=gm(2,2)**2*dgm(1,1)+1.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)&
&   *(4*gm(1,2)*dgm(1,2)-0.5d0*gm(1,1)*dgm(2,2))
   cm(1,3,2)=1.5d0*gm(2,3)**2*dgm(1,1)-0.5d0*gm(2,2)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*gm(2,3)*dgm(1,2)-2*gm(1,2)*gm(3,3)*dgm(1,2)+1.5d0*gm(1,3)&
&   **2*dgm(2,2)-0.5d0*gm(1,1)*gm(3,3)*dgm(2,2)
   cm(1,4,2)=gm(2,2)*(2*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2))-gm(1,1)&
&   *gm(2,3)*dgm(2,2)+gm(1,2)*(2*gm(2,3)*dgm(1,2)+3*gm(1,3)*dgm(2,2))
   cm(1,5,2)=gm(2,3)*(3*gm(1,2)*dgm(1,1)+6*gm(1,1)*dgm(1,2))+gm(1,3)&
&   *(-gm(2,2)*dgm(1,1)+2*(gm(1,2)*dgm(1,2)+gm(1,1)*dgm(2,2)))
   cm(1,6,2)=2*gm(1,2)**2*dgm(1,2)+6*gm(1,1)*gm(2,2)*dgm(1,2)+2*gm(1,2)&
&   *(gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))
   cm(1,1,3)=1.5d0*gm(1,3)**2*dgm(1,1)+4*gm(1,1)*gm(1,3)*dgm(1,3)&
&   +gm(1,1)*(-0.5d0*gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3))
   cm(1,2,3)=1.5d0*gm(2,3)**2*dgm(1,1)+6*gm(1,2)*gm(2,3)*dgm(1,3)&
&   +1.5d0*gm(1,2)**2*dgm(3,3)+gm(2,2)*(-0.5d0*gm(3,3)*dgm(1,1)-2*gm(1,3)&
&   *dgm(1,3)-0.5d0*gm(1,1)*dgm(3,3))
   cm(1,3,3)=gm(3,3)**2*dgm(1,1)+1.5d0*gm(1,3)**2*dgm(3,3)+gm(3,3)&
&   *(4*gm(1,3)*dgm(1,3)-0.5d0*gm(1,1)*dgm(3,3))
   cm(1,4,3)=gm(2,3)*(2*gm(3,3)*dgm(1,1)+2*gm(1,3)*dgm(1,3)-gm(1,1)&
&   *dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(1,5,3)=2*gm(1,3)**2*dgm(1,3)+6*gm(1,1)*gm(3,3)*dgm(1,3)+2*gm(1,3)&
&   *(gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3))
   cm(1,6,3)=6*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,3)*(3*gm(2,3)*dgm(1,1)&
&   +2*gm(1,2)*dgm(1,3))+gm(1,2)*(-gm(3,3)*dgm(1,1)+2*gm(1,1)*dgm(3,3))
   cm(1,1,4)=gm(1,2)*(3*gm(1,3)*dgm(1,1)+4*gm(1,1)*dgm(1,3))+gm(1,1)&
&   *(-gm(2,3)*dgm(1,1)+4*gm(1,3)*dgm(1,2)+2*gm(1,1)*dgm(2,3))
   cm(1,2,4)=gm(2,2)*(2*gm(2,3)*dgm(1,1)-2*gm(1,3)*dgm(1,2)+4*gm(1,2)&
&   *dgm(1,3)-gm(1,1)*dgm(2,3))+gm(1,2)*(6*gm(2,3)*dgm(1,2)+3*gm(1,2)&
&   *dgm(2,3))
   cm(1,3,4)=4*gm(1,3)*gm(3,3)*dgm(1,2)+gm(2,3)*(2*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*dgm(1,3))+3*gm(1,3)**2*dgm(2,3)+gm(3,3)*(-2*gm(1,2)&
&   *dgm(1,3)-gm(1,1)*dgm(2,3))
   cm(1,4,4)=gm(2,3)**2*dgm(1,1)+gm(2,2)*(3*gm(3,3)*dgm(1,1)+6*gm(1,3)&
&   *dgm(1,3))+gm(2,3)*(2*gm(1,3)*dgm(1,2)+2*gm(1,2)*dgm(1,3)-2*gm(1,1)&
&   *dgm(2,3))+6*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3))
   cm(1,5,4)=3*gm(1,2)*gm(3,3)*dgm(1,1)+2*gm(1,3)**2*dgm(1,2)+6*gm(1,1)&
&   *(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)*(1*gm(2,3)*dgm(1,1)&
&   +2*gm(1,2)*dgm(1,3)+4*gm(1,1)*dgm(2,3))
   cm(1,6,4)=gm(1,3)*(3*gm(2,2)*dgm(1,1)+2*gm(1,2)*dgm(1,2))+2*gm(1,2)&
&   **2*dgm(1,3)+6*gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)&
&   *(1*gm(2,3)*dgm(1,1)+4*gm(1,1)*dgm(2,3))
   cm(1,1,5)=2*gm(1,1)*(gm(1,3)*dgm(1,1)+gm(1,1)*dgm(1,3))
   cm(1,2,5)=-gm(1,3)*gm(2,2)*dgm(1,1)+3*gm(1,2)*gm(2,3)*dgm(1,1)&
&   +3*gm(1,2)**2*dgm(1,3)-gm(1,1)*gm(2,2)*dgm(1,3)
   cm(1,3,5)=2*gm(1,3)*gm(3,3)*dgm(1,1)+3*gm(1,3)**2*dgm(1,3)-gm(1,1)&
&   *gm(3,3)*dgm(1,3)
   cm(1,4,5)=3*gm(1,2)*gm(3,3)*dgm(1,1)-2*gm(1,1)*gm(2,3)*dgm(1,3)&
&   +gm(1,3)*(1*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3))
   cm(1,5,5)=gm(1,3)**2*dgm(1,1)+3*gm(1,1)*gm(3,3)*dgm(1,1)+4*gm(1,1)&
&   *gm(1,3)*dgm(1,3)
   cm(1,6,5)=3*gm(1,1)*gm(2,3)*dgm(1,1)+gm(1,2)*(1*gm(1,3)*dgm(1,1)&
&   +4*gm(1,1)*dgm(1,3))
   cm(1,1,6)=2*gm(1,1)*(gm(1,2)*dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(1,2,6)=2*gm(1,2)*gm(2,2)*dgm(1,1)+3*gm(1,2)**2*dgm(1,2)-gm(1,1)&
&   *gm(2,2)*dgm(1,2)
   cm(1,3,6)=3*gm(1,3)*gm(2,3)*dgm(1,1)+3*gm(1,3)**2*dgm(1,2)-gm(3,3)&
&   *(gm(1,2)*dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(1,4,6)=gm(2,3)*(1*gm(1,2)*dgm(1,1)-2*gm(1,1)*dgm(1,2))+gm(1,3)&
&   *(3*gm(2,2)*dgm(1,1)+6*gm(1,2)*dgm(1,2))
   cm(1,5,6)=gm(1,2)*gm(1,3)*dgm(1,1)+gm(1,1)*(3*gm(2,3)*dgm(1,1)&
&   +4*gm(1,3)*dgm(1,2))
   cm(1,6,6)=gm(1,2)**2*dgm(1,1)+3*gm(1,1)*gm(2,2)*dgm(1,1)+4*gm(1,1)&
&   *gm(1,2)*dgm(1,2)
   cm(1,1,7)=3*gm(1,2)**2*dgm(1,2)-gm(1,1)*gm(2,2)*dgm(1,2)+2*gm(1,1)&
&   *gm(1,2)*dgm(2,2)
   cm(1,2,7)=2*gm(2,2)*(gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(1,3,7)=3*gm(2,3)**2*dgm(1,2)+3*gm(1,3)*gm(2,3)*dgm(2,2)-gm(3,3)&
&   *(gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(1,4,7)=gm(1,2)*gm(2,3)*dgm(2,2)+gm(2,2)*(4*gm(2,3)*dgm(1,2)&
&   +3*gm(1,3)*dgm(2,2))
   cm(1,5,7)=gm(2,3)*(6*gm(1,2)*dgm(1,2)+3*gm(1,1)*dgm(2,2))+gm(1,3)&
&   *(-2*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(1,6,7)=4*gm(1,2)*gm(2,2)*dgm(1,2)+gm(1,2)**2*dgm(2,2)+3*gm(1,1)&
&   *gm(2,2)*dgm(2,2)
   cm(1,1,8)=3*gm(1,3)**2*dgm(1,2)+gm(1,3)*(6*gm(1,2)*dgm(1,3)+4*gm(1,1)&
&   *dgm(2,3))+gm(1,1)*(-gm(3,3)*dgm(1,2)-2*gm(2,3)*dgm(1,3)+2*gm(1,2)&
&   *dgm(3,3))
   cm(1,2,8)=3*gm(2,3)**2*dgm(1,2)+gm(2,3)*(4*gm(2,2)*dgm(1,3)+6*gm(1,2)&
&   *dgm(2,3))+gm(2,2)*(-gm(3,3)*dgm(1,2)-2*gm(1,3)*dgm(2,3)+2*gm(1,2)&
&   *dgm(3,3))
   cm(1,3,8)=2*gm(3,3)**2*dgm(1,2)+3*gm(1,3)*gm(2,3)*dgm(3,3)+gm(3,3)&
&   *(4*gm(2,3)*dgm(1,3)+4*gm(1,3)*dgm(2,3)-gm(1,2)*dgm(3,3))
   cm(1,4,8)=2*gm(2,3)**2*dgm(1,3)+6*gm(1,2)*gm(3,3)*dgm(2,3)+gm(2,3)&
&   *(4*gm(3,3)*dgm(1,2)+2*gm(1,3)*dgm(2,3)+gm(1,2)*dgm(3,3))+gm(2,2)&
&   *(6*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(1,5,8)=6*gm(1,2)*gm(3,3)*dgm(1,3)+2*gm(1,3)**2*dgm(2,3)+gm(1,3)&
&   *(4*gm(3,3)*dgm(1,2)+2*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3))+gm(1,1)&
&   *(6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(1,6,8)=gm(1,2)*(-2*gm(3,3)*dgm(1,2)+2*gm(2,3)*dgm(1,3))+gm(1,3)&
&   *(6*gm(2,3)*dgm(1,2)+6*gm(2,2)*dgm(1,3)+2*gm(1,2)*dgm(2,3))+gm(1,2)&
&   **2*dgm(3,3)+gm(1,1)*(6*gm(2,3)*dgm(2,3)+3*gm(2,2)*dgm(3,3))
   cm(1,1,9)=3*gm(1,2)**2*dgm(1,3)+gm(1,1)*(-2*gm(2,3)*dgm(1,2)-gm(2,2)&
&   *dgm(1,3)+2*gm(1,3)*dgm(2,2))+gm(1,2)*(6*gm(1,3)*dgm(1,2)+4*gm(1,1)&
&   *dgm(2,3))
   cm(1,2,9)=2*gm(2,2)**2*dgm(1,3)+3*gm(1,2)*gm(2,3)*dgm(2,2)+gm(2,2)&
&   *(4*gm(2,3)*dgm(1,2)-gm(1,3)*dgm(2,2)+4*gm(1,2)*dgm(2,3))
   cm(1,3,9)=3*gm(2,3)**2*dgm(1,3)+gm(3,3)*(-gm(2,2)*dgm(1,3)+2*gm(1,3)&
&   *dgm(2,2)-2*gm(1,2)*dgm(2,3))+gm(2,3)*(4*gm(3,3)*dgm(1,2)+6*gm(1,3)&
&   *dgm(2,3))
   cm(1,4,9)=2*gm(2,3)**2*dgm(1,2)+3*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)&
&   *(4*gm(2,2)*dgm(1,3)+gm(1,3)*dgm(2,2)+2*gm(1,2)*dgm(2,3))+6*gm(2,2)&
&   *(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3))
   cm(1,5,9)=6*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)&
&   **2*dgm(2,2)+gm(1,3)*(2*gm(2,3)*dgm(1,2)-2*gm(2,2)*dgm(1,3)+2*gm(1,2)&
&   *dgm(2,3))+gm(1,1)*(3*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))
   cm(1,6,9)=gm(1,2)*(2*gm(2,3)*dgm(1,2)+4*gm(2,2)*dgm(1,3))+gm(1,3)&
&   *(6*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))+2*gm(1,2)**2*dgm(2,3)&
&   +gm(1,1)*(3*gm(2,3)*dgm(2,2)+6*gm(2,2)*dgm(2,3))
   cm(1,1,10)=3*gm(1,3)**2*dgm(1,3)-gm(1,1)*gm(3,3)*dgm(1,3)+2*gm(1,1)&
&   *gm(1,3)*dgm(3,3)
   cm(1,2,10)=3*gm(2,3)**2*dgm(1,3)+3*gm(1,2)*gm(2,3)*dgm(3,3)-gm(2,2)&
&   *(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(1,3,10)=2*gm(3,3)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(1,4,10)=3*gm(1,2)*gm(3,3)*dgm(3,3)+gm(2,3)*(4*gm(3,3)*dgm(1,3)&
&   +gm(1,3)*dgm(3,3))
   cm(1,5,10)=4*gm(1,3)*gm(3,3)*dgm(1,3)+gm(1,3)**2*dgm(3,3)+3*gm(1,1)&
&   *gm(3,3)*dgm(3,3)
   cm(1,6,10)=-2*gm(1,2)*gm(3,3)*dgm(1,3)+3*gm(1,1)*gm(2,3)*dgm(3,3)&
&   +gm(1,3)*(6*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3))
   cm(1,1,11)=((6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm(2,2))/4.d0
   cm(1,2,11)=gm(2,2)**2*dgm(2,2)
   cm(1,3,11)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm(2,2))/4.d0
   cm(1,4,11)=2*gm(2,2)*gm(2,3)*dgm(2,2)
   cm(1,5,11)=((-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm(2,2))/2.d0
   cm(1,6,11)=2*gm(1,2)*gm(2,2)*dgm(2,2)
   cm(1,1,12)=1.5d0*gm(1,3)**2*dgm(2,2)+6*gm(1,2)*gm(1,3)*dgm(2,3)&
&   +1.5d0*gm(1,2)**2*dgm(3,3)+gm(1,1)*(-0.5d0*gm(3,3)*dgm(2,2)-2*gm(2,3)&
&   *dgm(2,3)-0.5d0*gm(2,2)*dgm(3,3))
   cm(1,2,12)=1.5d0*gm(2,3)**2*dgm(2,2)+4*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(-0.5d0*gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3))
   cm(1,3,12)=gm(3,3)**2*dgm(2,2)+1.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)&
&   *(4*gm(2,3)*dgm(2,3)-0.5d0*gm(2,2)*dgm(3,3))
   cm(1,4,12)=2*gm(2,3)**2*dgm(2,3)+6*gm(2,2)*gm(3,3)*dgm(2,3)+2*gm(2,3)&
&   *(gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3))
   cm(1,5,12)=gm(1,3)*(2*gm(3,3)*dgm(2,2)+2*gm(2,3)*dgm(2,3)-gm(2,2)&
&   *dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(1,6,12)=gm(1,3)*(3*gm(2,3)*dgm(2,2)+6*gm(2,2)*dgm(2,3))+gm(1,2)&
&   *(-gm(3,3)*dgm(2,2)+2*(gm(2,3)*dgm(2,3)+gm(2,2)*dgm(3,3)))
   cm(1,1,13)=3*gm(1,2)*gm(1,3)*dgm(2,2)+3*gm(1,2)**2*dgm(2,3)-gm(1,1)&
&   *(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(1,2,13)=2*gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(1,3,13)=2*gm(2,3)*gm(3,3)*dgm(2,2)+3*gm(2,3)**2*dgm(2,3)-gm(2,2)&
&   *gm(3,3)*dgm(2,3)
   cm(1,4,13)=gm(2,3)**2*dgm(2,2)+3*gm(2,2)*gm(3,3)*dgm(2,2)+4*gm(2,2)&
&   *gm(2,3)*dgm(2,3)
   cm(1,5,13)=gm(1,3)*(1*gm(2,3)*dgm(2,2)-2*gm(2,2)*dgm(2,3))+gm(1,2)&
&   *(3*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))
   cm(1,6,13)=3*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,2)*(1*gm(2,3)*dgm(2,2)&
&   +4*gm(2,2)*dgm(2,3))
   cm(1,1,14)=3*gm(1,3)**2*dgm(2,3)+3*gm(1,2)*gm(1,3)*dgm(3,3)-gm(1,1)&
&   *(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(1,2,14)=3*gm(2,3)**2*dgm(2,3)-gm(2,2)*gm(3,3)*dgm(2,3)+2*gm(2,2)&
&   *gm(2,3)*dgm(3,3)
   cm(1,3,14)=2*gm(3,3)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(1,4,14)=4*gm(2,3)*gm(3,3)*dgm(2,3)+gm(2,3)**2*dgm(3,3)+3*gm(2,2)&
&   *gm(3,3)*dgm(3,3)
   cm(1,5,14)=3*gm(1,2)*gm(3,3)*dgm(3,3)+gm(1,3)*(4*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*dgm(3,3))
   cm(1,6,14)=gm(1,3)*(6*gm(2,3)*dgm(2,3)+3*gm(2,2)*dgm(3,3))+gm(1,2)&
&   *(-2*gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(1,1,15)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm(3,3))/4.d0
   cm(1,2,15)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm(3,3))/4.d0
   cm(1,3,15)=gm(3,3)**2*dgm(3,3)
   cm(1,4,15)=2*gm(2,3)*gm(3,3)*dgm(3,3)
   cm(1,5,15)=2*gm(1,3)*gm(3,3)*dgm(3,3)
   cm(1,6,15)=((6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm(3,3))/2.d0
   cm(2,1,2)=2*gm(1,1)*(gm(1,2)*dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(2,2,2)=2*gm(1,2)*gm(2,2)*dgm(1,1)+3*gm(1,2)**2*dgm(1,2)-gm(1,1)&
&   *gm(2,2)*dgm(1,2)
   cm(2,3,2)=3*gm(1,3)*gm(2,3)*dgm(1,1)+3*gm(1,3)**2*dgm(1,2)-gm(3,3)&
&   *(gm(1,2)*dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(2,4,2)=gm(2,3)*(1*gm(1,2)*dgm(1,1)-2*gm(1,1)*dgm(1,2))+gm(1,3)&
&   *(3*gm(2,2)*dgm(1,1)+6*gm(1,2)*dgm(1,2))
   cm(2,5,2)=gm(1,2)*gm(1,3)*dgm(1,1)+gm(1,1)*(3*gm(2,3)*dgm(1,1)&
&   +4*gm(1,3)*dgm(1,2))
   cm(2,6,2)=gm(1,2)**2*dgm(1,1)+3*gm(1,1)*gm(2,2)*dgm(1,1)+4*gm(1,1)&
&   *gm(1,2)*dgm(1,2)
   cm(2,1,4)=2*gm(1,1)*(gm(1,3)*dgm(1,1)+gm(1,1)*dgm(1,3))
   cm(2,2,4)=-gm(1,3)*gm(2,2)*dgm(1,1)+3*gm(1,2)*gm(2,3)*dgm(1,1)&
&   +3*gm(1,2)**2*dgm(1,3)-gm(1,1)*gm(2,2)*dgm(1,3)
   cm(2,3,4)=2*gm(1,3)*gm(3,3)*dgm(1,1)+3*gm(1,3)**2*dgm(1,3)-gm(1,1)&
&   *gm(3,3)*dgm(1,3)
   cm(2,4,4)=3*gm(1,2)*gm(3,3)*dgm(1,1)-2*gm(1,1)*gm(2,3)*dgm(1,3)&
&   +gm(1,3)*(1*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3))
   cm(2,5,4)=gm(1,3)**2*dgm(1,1)+3*gm(1,1)*gm(3,3)*dgm(1,1)+4*gm(1,1)&
&   *gm(1,3)*dgm(1,3)
   cm(2,6,4)=3*gm(1,1)*gm(2,3)*dgm(1,1)+gm(1,2)*(1*gm(1,3)*dgm(1,1)&
&   +4*gm(1,1)*dgm(1,3))
   cm(2,1,6)=gm(1,1)**2*dgm(1,1)
   cm(2,2,6)=((6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm(1,1))/4.d0
   cm(2,3,6)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm(1,1))/4.d0
   cm(2,4,6)=((6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm(1,1))/2.d0
   cm(2,5,6)=2*gm(1,1)*gm(1,3)*dgm(1,1)
   cm(2,6,6)=2*gm(1,1)*gm(1,2)*dgm(1,1)
   cm(2,1,7)=1.5d0*gm(1,2)**2*dgm(1,1)+4*gm(1,1)*gm(1,2)*dgm(1,2)&
&   +gm(1,1)*(-0.5d0*gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))
   cm(2,2,7)=gm(2,2)**2*dgm(1,1)+1.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)&
&   *(4*gm(1,2)*dgm(1,2)-0.5d0*gm(1,1)*dgm(2,2))
   cm(2,3,7)=1.5d0*gm(2,3)**2*dgm(1,1)-0.5d0*gm(2,2)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*gm(2,3)*dgm(1,2)-2*gm(1,2)*gm(3,3)*dgm(1,2)+1.5d0*gm(1,3)&
&   **2*dgm(2,2)-0.5d0*gm(1,1)*gm(3,3)*dgm(2,2)
   cm(2,4,7)=gm(2,2)*(2*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2))-gm(1,1)&
&   *gm(2,3)*dgm(2,2)+gm(1,2)*(2*gm(2,3)*dgm(1,2)+3*gm(1,3)*dgm(2,2))
   cm(2,5,7)=gm(2,3)*(3*gm(1,2)*dgm(1,1)+6*gm(1,1)*dgm(1,2))+gm(1,3)&
&   *(-gm(2,2)*dgm(1,1)+2*(gm(1,2)*dgm(1,2)+gm(1,1)*dgm(2,2)))
   cm(2,6,7)=2*gm(1,2)**2*dgm(1,2)+6*gm(1,1)*gm(2,2)*dgm(1,2)+2*gm(1,2)&
&   *(gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))
   cm(2,1,8)=1.5d0*gm(1,3)**2*dgm(1,1)+4*gm(1,1)*gm(1,3)*dgm(1,3)&
&   +gm(1,1)*(-0.5d0*gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3))
   cm(2,2,8)=1.5d0*gm(2,3)**2*dgm(1,1)+6*gm(1,2)*gm(2,3)*dgm(1,3)&
&   +1.5d0*gm(1,2)**2*dgm(3,3)+gm(2,2)*(-0.5d0*gm(3,3)*dgm(1,1)-2*gm(1,3)&
&   *dgm(1,3)-0.5d0*gm(1,1)*dgm(3,3))
   cm(2,3,8)=gm(3,3)**2*dgm(1,1)+1.5d0*gm(1,3)**2*dgm(3,3)+gm(3,3)&
&   *(4*gm(1,3)*dgm(1,3)-0.5d0*gm(1,1)*dgm(3,3))
   cm(2,4,8)=gm(2,3)*(2*gm(3,3)*dgm(1,1)+2*gm(1,3)*dgm(1,3)-gm(1,1)&
&   *dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(2,5,8)=2*gm(1,3)**2*dgm(1,3)+6*gm(1,1)*gm(3,3)*dgm(1,3)+2*gm(1,3)&
&   *(gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3))
   cm(2,6,8)=6*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,3)*(3*gm(2,3)*dgm(1,1)&
&   +2*gm(1,2)*dgm(1,3))+gm(1,2)*(-gm(3,3)*dgm(1,1)+2*gm(1,1)*dgm(3,3))
   cm(2,1,9)=gm(1,2)*(3*gm(1,3)*dgm(1,1)+4*gm(1,1)*dgm(1,3))+gm(1,1)&
&   *(-gm(2,3)*dgm(1,1)+4*gm(1,3)*dgm(1,2)+2*gm(1,1)*dgm(2,3))
   cm(2,2,9)=gm(2,2)*(2*gm(2,3)*dgm(1,1)-2*gm(1,3)*dgm(1,2)+4*gm(1,2)&
&   *dgm(1,3)-gm(1,1)*dgm(2,3))+gm(1,2)*(6*gm(2,3)*dgm(1,2)+3*gm(1,2)&
&   *dgm(2,3))
   cm(2,3,9)=4*gm(1,3)*gm(3,3)*dgm(1,2)+gm(2,3)*(2*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*dgm(1,3))+3*gm(1,3)**2*dgm(2,3)+gm(3,3)*(-2*gm(1,2)&
&   *dgm(1,3)-gm(1,1)*dgm(2,3))
   cm(2,4,9)=gm(2,3)**2*dgm(1,1)+gm(2,2)*(3*gm(3,3)*dgm(1,1)+6*gm(1,3)&
&   *dgm(1,3))+gm(2,3)*(2*gm(1,3)*dgm(1,2)+2*gm(1,2)*dgm(1,3)-2*gm(1,1)&
&   *dgm(2,3))+6*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3))
   cm(2,5,9)=3*gm(1,2)*gm(3,3)*dgm(1,1)+2*gm(1,3)**2*dgm(1,2)+6*gm(1,1)&
&   *(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)*(1*gm(2,3)*dgm(1,1)&
&   +2*gm(1,2)*dgm(1,3)+4*gm(1,1)*dgm(2,3))
   cm(2,6,9)=gm(1,3)*(3*gm(2,2)*dgm(1,1)+2*gm(1,2)*dgm(1,2))+2*gm(1,2)&
&   **2*dgm(1,3)+6*gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)&
&   *(1*gm(2,3)*dgm(1,1)+4*gm(1,1)*dgm(2,3))
   cm(2,1,11)=3*gm(1,2)**2*dgm(1,2)-gm(1,1)*gm(2,2)*dgm(1,2)+2*gm(1,1)&
&   *gm(1,2)*dgm(2,2)
   cm(2,2,11)=2*gm(2,2)*(gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(2,3,11)=3*gm(2,3)**2*dgm(1,2)+3*gm(1,3)*gm(2,3)*dgm(2,2)-gm(3,3)&
&   *(gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(2,4,11)=gm(1,2)*gm(2,3)*dgm(2,2)+gm(2,2)*(4*gm(2,3)*dgm(1,2)&
&   +3*gm(1,3)*dgm(2,2))
   cm(2,5,11)=gm(2,3)*(6*gm(1,2)*dgm(1,2)+3*gm(1,1)*dgm(2,2))+gm(1,3)&
&   *(-2*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(2,6,11)=4*gm(1,2)*gm(2,2)*dgm(1,2)+gm(1,2)**2*dgm(2,2)+3*gm(1,1)&
&   *gm(2,2)*dgm(2,2)
   cm(2,1,12)=3*gm(1,3)**2*dgm(1,2)+gm(1,3)*(6*gm(1,2)*dgm(1,3)+4*gm(1,1)&
&   *dgm(2,3))+gm(1,1)*(-gm(3,3)*dgm(1,2)-2*gm(2,3)*dgm(1,3)+2*gm(1,2)&
&   *dgm(3,3))
   cm(2,2,12)=3*gm(2,3)**2*dgm(1,2)+gm(2,3)*(4*gm(2,2)*dgm(1,3)+6*gm(1,2)&
&   *dgm(2,3))+gm(2,2)*(-gm(3,3)*dgm(1,2)-2*gm(1,3)*dgm(2,3)+2*gm(1,2)&
&   *dgm(3,3))
   cm(2,3,12)=2*gm(3,3)**2*dgm(1,2)+3*gm(1,3)*gm(2,3)*dgm(3,3)+gm(3,3)&
&   *(4*gm(2,3)*dgm(1,3)+4*gm(1,3)*dgm(2,3)-gm(1,2)*dgm(3,3))
   cm(2,4,12)=2*gm(2,3)**2*dgm(1,3)+6*gm(1,2)*gm(3,3)*dgm(2,3)+gm(2,3)&
&   *(4*gm(3,3)*dgm(1,2)+2*gm(1,3)*dgm(2,3)+gm(1,2)*dgm(3,3))+gm(2,2)&
&   *(6*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(2,5,12)=6*gm(1,2)*gm(3,3)*dgm(1,3)+2*gm(1,3)**2*dgm(2,3)+gm(1,3)&
&   *(4*gm(3,3)*dgm(1,2)+2*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3))+gm(1,1)&
&   *(6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(2,6,12)=gm(1,2)*(-2*gm(3,3)*dgm(1,2)+2*gm(2,3)*dgm(1,3))+gm(1,3)&
&   *(6*gm(2,3)*dgm(1,2)+6*gm(2,2)*dgm(1,3)+2*gm(1,2)*dgm(2,3))+gm(1,2)&
&   **2*dgm(3,3)+gm(1,1)*(6*gm(2,3)*dgm(2,3)+3*gm(2,2)*dgm(3,3))
   cm(2,1,13)=3*gm(1,2)**2*dgm(1,3)+gm(1,1)*(-2*gm(2,3)*dgm(1,2)&
&   -gm(2,2)*dgm(1,3)+2*gm(1,3)*dgm(2,2))+gm(1,2)*(6*gm(1,3)*dgm(1,2)&
&   +4*gm(1,1)*dgm(2,3))
   cm(2,2,13)=2*gm(2,2)**2*dgm(1,3)+3*gm(1,2)*gm(2,3)*dgm(2,2)+gm(2,2)&
&   *(4*gm(2,3)*dgm(1,2)-gm(1,3)*dgm(2,2)+4*gm(1,2)*dgm(2,3))
   cm(2,3,13)=3*gm(2,3)**2*dgm(1,3)+gm(3,3)*(-gm(2,2)*dgm(1,3)+2*gm(1,3)&
&   *dgm(2,2)-2*gm(1,2)*dgm(2,3))+gm(2,3)*(4*gm(3,3)*dgm(1,2)+6*gm(1,3)&
&   *dgm(2,3))
   cm(2,4,13)=2*gm(2,3)**2*dgm(1,2)+3*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)&
&   *(4*gm(2,2)*dgm(1,3)+gm(1,3)*dgm(2,2)+2*gm(1,2)*dgm(2,3))+6*gm(2,2)&
&   *(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3))
   cm(2,5,13)=6*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)&
&   **2*dgm(2,2)+gm(1,3)*(2*gm(2,3)*dgm(1,2)-2*gm(2,2)*dgm(1,3)+2*gm(1,2)&
&   *dgm(2,3))+gm(1,1)*(3*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))
   cm(2,6,13)=gm(1,2)*(2*gm(2,3)*dgm(1,2)+4*gm(2,2)*dgm(1,3))+gm(1,3)&
&   *(6*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))+2*gm(1,2)**2*dgm(2,3)&
&   +gm(1,1)*(3*gm(2,3)*dgm(2,2)+6*gm(2,2)*dgm(2,3))
   cm(2,1,14)=3*gm(1,3)**2*dgm(1,3)-gm(1,1)*gm(3,3)*dgm(1,3)+2*gm(1,1)&
&   *gm(1,3)*dgm(3,3)
   cm(2,2,14)=3*gm(2,3)**2*dgm(1,3)+3*gm(1,2)*gm(2,3)*dgm(3,3)-gm(2,2)&
&   *(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(2,3,14)=2*gm(3,3)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(2,4,14)=3*gm(1,2)*gm(3,3)*dgm(3,3)+gm(2,3)*(4*gm(3,3)*dgm(1,3)&
&   +gm(1,3)*dgm(3,3))
   cm(2,5,14)=4*gm(1,3)*gm(3,3)*dgm(1,3)+gm(1,3)**2*dgm(3,3)+3*gm(1,1)&
&   *gm(3,3)*dgm(3,3)
   cm(2,6,14)=-2*gm(1,2)*gm(3,3)*dgm(1,3)+3*gm(1,1)*gm(2,3)*dgm(3,3)&
&   +gm(1,3)*(6*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3))
   cm(2,1,16)=((6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm(2,2))/4.d0
   cm(2,2,16)=gm(2,2)**2*dgm(2,2)
   cm(2,3,16)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm(2,2))/4.d0
   cm(2,4,16)=2*gm(2,2)*gm(2,3)*dgm(2,2)
   cm(2,5,16)=((-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm(2,2))/2.d0
   cm(2,6,16)=2*gm(1,2)*gm(2,2)*dgm(2,2)
   cm(2,1,17)=1.5d0*gm(1,3)**2*dgm(2,2)+6*gm(1,2)*gm(1,3)*dgm(2,3)&
&   +1.5d0*gm(1,2)**2*dgm(3,3)+gm(1,1)*(-0.5d0*gm(3,3)*dgm(2,2)-2*gm(2,3)&
&   *dgm(2,3)-0.5d0*gm(2,2)*dgm(3,3))
   cm(2,2,17)=1.5d0*gm(2,3)**2*dgm(2,2)+4*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(-0.5d0*gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3))
   cm(2,3,17)=gm(3,3)**2*dgm(2,2)+1.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)&
&   *(4*gm(2,3)*dgm(2,3)-0.5d0*gm(2,2)*dgm(3,3))
   cm(2,4,17)=2*gm(2,3)**2*dgm(2,3)+6*gm(2,2)*gm(3,3)*dgm(2,3)+2*gm(2,3)&
&   *(gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3))
   cm(2,5,17)=gm(1,3)*(2*gm(3,3)*dgm(2,2)+2*gm(2,3)*dgm(2,3)-gm(2,2)&
&   *dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(2,6,17)=gm(1,3)*(3*gm(2,3)*dgm(2,2)+6*gm(2,2)*dgm(2,3))+gm(1,2)&
&   *(-gm(3,3)*dgm(2,2)+2*(gm(2,3)*dgm(2,3)+gm(2,2)*dgm(3,3)))
   cm(2,1,18)=3*gm(1,2)*gm(1,3)*dgm(2,2)+3*gm(1,2)**2*dgm(2,3)-gm(1,1)&
&   *(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(2,2,18)=2*gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(2,3,18)=2*gm(2,3)*gm(3,3)*dgm(2,2)+3*gm(2,3)**2*dgm(2,3)-gm(2,2)&
&   *gm(3,3)*dgm(2,3)
   cm(2,4,18)=gm(2,3)**2*dgm(2,2)+3*gm(2,2)*gm(3,3)*dgm(2,2)+4*gm(2,2)&
&   *gm(2,3)*dgm(2,3)
   cm(2,5,18)=gm(1,3)*(1*gm(2,3)*dgm(2,2)-2*gm(2,2)*dgm(2,3))+gm(1,2)&
&   *(3*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))
   cm(2,6,18)=3*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,2)*(1*gm(2,3)*dgm(2,2)&
&   +4*gm(2,2)*dgm(2,3))
   cm(2,1,19)=3*gm(1,3)**2*dgm(2,3)+3*gm(1,2)*gm(1,3)*dgm(3,3)-gm(1,1)&
&   *(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(2,2,19)=3*gm(2,3)**2*dgm(2,3)-gm(2,2)*gm(3,3)*dgm(2,3)+2*gm(2,2)&
&   *gm(2,3)*dgm(3,3)
   cm(2,3,19)=2*gm(3,3)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(2,4,19)=4*gm(2,3)*gm(3,3)*dgm(2,3)+gm(2,3)**2*dgm(3,3)+3*gm(2,2)&
&   *gm(3,3)*dgm(3,3)
   cm(2,5,19)=3*gm(1,2)*gm(3,3)*dgm(3,3)+gm(1,3)*(4*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*dgm(3,3))
   cm(2,6,19)=gm(1,3)*(6*gm(2,3)*dgm(2,3)+3*gm(2,2)*dgm(3,3))+gm(1,2)&
&   *(-2*gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(2,1,20)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm(3,3))/4.d0
   cm(2,2,20)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm(3,3))/4.d0
   cm(2,3,20)=gm(3,3)**2*dgm(3,3)
   cm(2,4,20)=2*gm(2,3)*gm(3,3)*dgm(3,3)
   cm(2,5,20)=2*gm(1,3)*gm(3,3)*dgm(3,3)
   cm(2,6,20)=((6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm(3,3))/2.d0
   cm(3,1,3)=2*gm(1,1)*(gm(1,3)*dgm(1,1)+gm(1,1)*dgm(1,3))
   cm(3,2,3)=-gm(1,3)*gm(2,2)*dgm(1,1)+3*gm(1,2)*gm(2,3)*dgm(1,1)&
&   +3*gm(1,2)**2*dgm(1,3)-gm(1,1)*gm(2,2)*dgm(1,3)
   cm(3,3,3)=2*gm(1,3)*gm(3,3)*dgm(1,1)+3*gm(1,3)**2*dgm(1,3)-gm(1,1)&
&   *gm(3,3)*dgm(1,3)
   cm(3,4,3)=3*gm(1,2)*gm(3,3)*dgm(1,1)-2*gm(1,1)*gm(2,3)*dgm(1,3)&
&   +gm(1,3)*(1*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3))
   cm(3,5,3)=gm(1,3)**2*dgm(1,1)+3*gm(1,1)*gm(3,3)*dgm(1,1)+4*gm(1,1)&
&   *gm(1,3)*dgm(1,3)
   cm(3,6,3)=3*gm(1,1)*gm(2,3)*dgm(1,1)+gm(1,2)*(1*gm(1,3)*dgm(1,1)&
&   +4*gm(1,1)*dgm(1,3))
   cm(3,1,4)=2*gm(1,1)*(gm(1,2)*dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(3,2,4)=2*gm(1,2)*gm(2,2)*dgm(1,1)+3*gm(1,2)**2*dgm(1,2)-gm(1,1)&
&   *gm(2,2)*dgm(1,2)
   cm(3,3,4)=3*gm(1,3)*gm(2,3)*dgm(1,1)+3*gm(1,3)**2*dgm(1,2)-gm(3,3)&
&   *(gm(1,2)*dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(3,4,4)=gm(2,3)*(1*gm(1,2)*dgm(1,1)-2*gm(1,1)*dgm(1,2))+gm(1,3)&
&   *(3*gm(2,2)*dgm(1,1)+6*gm(1,2)*dgm(1,2))
   cm(3,5,4)=gm(1,2)*gm(1,3)*dgm(1,1)+gm(1,1)*(3*gm(2,3)*dgm(1,1)&
&   +4*gm(1,3)*dgm(1,2))
   cm(3,6,4)=gm(1,2)**2*dgm(1,1)+3*gm(1,1)*gm(2,2)*dgm(1,1)+4*gm(1,1)&
&   *gm(1,2)*dgm(1,2)
   cm(3,1,5)=gm(1,1)**2*dgm(1,1)
   cm(3,2,5)=((6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm(1,1))/4.d0
   cm(3,3,5)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm(1,1))/4.d0
   cm(3,4,5)=((6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*dgm(1,1))/2.d0
   cm(3,5,5)=2*gm(1,1)*gm(1,3)*dgm(1,1)
   cm(3,6,5)=2*gm(1,1)*gm(1,2)*dgm(1,1)
   cm(3,1,8)=gm(1,2)*(3*gm(1,3)*dgm(1,1)+4*gm(1,1)*dgm(1,3))+gm(1,1)&
&   *(-gm(2,3)*dgm(1,1)+4*gm(1,3)*dgm(1,2)+2*gm(1,1)*dgm(2,3))
   cm(3,2,8)=gm(2,2)*(2*gm(2,3)*dgm(1,1)-2*gm(1,3)*dgm(1,2)+4*gm(1,2)&
&   *dgm(1,3)-gm(1,1)*dgm(2,3))+gm(1,2)*(6*gm(2,3)*dgm(1,2)+3*gm(1,2)&
&   *dgm(2,3))
   cm(3,3,8)=4*gm(1,3)*gm(3,3)*dgm(1,2)+gm(2,3)*(2*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*dgm(1,3))+3*gm(1,3)**2*dgm(2,3)+gm(3,3)*(-2*gm(1,2)&
&   *dgm(1,3)-gm(1,1)*dgm(2,3))
   cm(3,4,8)=gm(2,3)**2*dgm(1,1)+gm(2,2)*(3*gm(3,3)*dgm(1,1)+6*gm(1,3)&
&   *dgm(1,3))+gm(2,3)*(2*gm(1,3)*dgm(1,2)+2*gm(1,2)*dgm(1,3)-2*gm(1,1)&
&   *dgm(2,3))+6*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3))
   cm(3,5,8)=3*gm(1,2)*gm(3,3)*dgm(1,1)+2*gm(1,3)**2*dgm(1,2)+6*gm(1,1)&
&   *(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)*(1*gm(2,3)*dgm(1,1)&
&   +2*gm(1,2)*dgm(1,3)+4*gm(1,1)*dgm(2,3))
   cm(3,6,8)=gm(1,3)*(3*gm(2,2)*dgm(1,1)+2*gm(1,2)*dgm(1,2))+2*gm(1,2)&
&   **2*dgm(1,3)+6*gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)&
&   *(1*gm(2,3)*dgm(1,1)+4*gm(1,1)*dgm(2,3))
   cm(3,1,9)=1.5d0*gm(1,2)**2*dgm(1,1)+4*gm(1,1)*gm(1,2)*dgm(1,2)&
&   +gm(1,1)*(-0.5d0*gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))
   cm(3,2,9)=gm(2,2)**2*dgm(1,1)+1.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)&
&   *(4*gm(1,2)*dgm(1,2)-0.5d0*gm(1,1)*dgm(2,2))
   cm(3,3,9)=1.5d0*gm(2,3)**2*dgm(1,1)-0.5d0*gm(2,2)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*gm(2,3)*dgm(1,2)-2*gm(1,2)*gm(3,3)*dgm(1,2)+1.5d0*gm(1,3)&
&   **2*dgm(2,2)-0.5d0*gm(1,1)*gm(3,3)*dgm(2,2)
   cm(3,4,9)=gm(2,2)*(2*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2))-gm(1,1)&
&   *gm(2,3)*dgm(2,2)+gm(1,2)*(2*gm(2,3)*dgm(1,2)+3*gm(1,3)*dgm(2,2))
   cm(3,5,9)=gm(2,3)*(3*gm(1,2)*dgm(1,1)+6*gm(1,1)*dgm(1,2))+gm(1,3)&
&   *(-gm(2,2)*dgm(1,1)+2*(gm(1,2)*dgm(1,2)+gm(1,1)*dgm(2,2)))
   cm(3,6,9)=2*gm(1,2)**2*dgm(1,2)+6*gm(1,1)*gm(2,2)*dgm(1,2)+2*gm(1,2)&
&   *(gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))
   cm(3,1,10)=1.5d0*gm(1,3)**2*dgm(1,1)+4*gm(1,1)*gm(1,3)*dgm(1,3)&
&   +gm(1,1)*(-0.5d0*gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3))
   cm(3,2,10)=1.5d0*gm(2,3)**2*dgm(1,1)+6*gm(1,2)*gm(2,3)*dgm(1,3)&
&   +1.5d0*gm(1,2)**2*dgm(3,3)+gm(2,2)*(-0.5d0*gm(3,3)*dgm(1,1)-2*gm(1,3)&
&   *dgm(1,3)-0.5d0*gm(1,1)*dgm(3,3))
   cm(3,3,10)=gm(3,3)**2*dgm(1,1)+1.5d0*gm(1,3)**2*dgm(3,3)+gm(3,3)&
&   *(4*gm(1,3)*dgm(1,3)-0.5d0*gm(1,1)*dgm(3,3))
   cm(3,4,10)=gm(2,3)*(2*gm(3,3)*dgm(1,1)+2*gm(1,3)*dgm(1,3)-gm(1,1)&
&   *dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(3,5,10)=2*gm(1,3)**2*dgm(1,3)+6*gm(1,1)*gm(3,3)*dgm(1,3)+2*gm(1,3)&
&   *(gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3))
   cm(3,6,10)=6*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,3)*(3*gm(2,3)*dgm(1,1)&
&   +2*gm(1,2)*dgm(1,3))+gm(1,2)*(-gm(3,3)*dgm(1,1)+2*gm(1,1)*dgm(3,3))
   cm(3,1,12)=3*gm(1,2)**2*dgm(1,3)+gm(1,1)*(-2*gm(2,3)*dgm(1,2)&
&   -gm(2,2)*dgm(1,3)+2*gm(1,3)*dgm(2,2))+gm(1,2)*(6*gm(1,3)*dgm(1,2)&
&   +4*gm(1,1)*dgm(2,3))
   cm(3,2,12)=2*gm(2,2)**2*dgm(1,3)+3*gm(1,2)*gm(2,3)*dgm(2,2)+gm(2,2)&
&   *(4*gm(2,3)*dgm(1,2)-gm(1,3)*dgm(2,2)+4*gm(1,2)*dgm(2,3))
   cm(3,3,12)=3*gm(2,3)**2*dgm(1,3)+gm(3,3)*(-gm(2,2)*dgm(1,3)+2*gm(1,3)&
&   *dgm(2,2)-2*gm(1,2)*dgm(2,3))+gm(2,3)*(4*gm(3,3)*dgm(1,2)+6*gm(1,3)&
&   *dgm(2,3))
   cm(3,4,12)=2*gm(2,3)**2*dgm(1,2)+3*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)&
&   *(4*gm(2,2)*dgm(1,3)+gm(1,3)*dgm(2,2)+2*gm(1,2)*dgm(2,3))+6*gm(2,2)&
&   *(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3))
   cm(3,5,12)=6*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)&
&   **2*dgm(2,2)+gm(1,3)*(2*gm(2,3)*dgm(1,2)-2*gm(2,2)*dgm(1,3)+2*gm(1,2)&
&   *dgm(2,3))+gm(1,1)*(3*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))
   cm(3,6,12)=gm(1,2)*(2*gm(2,3)*dgm(1,2)+4*gm(2,2)*dgm(1,3))+gm(1,3)&
&   *(6*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))+2*gm(1,2)**2*dgm(2,3)&
&   +gm(1,1)*(3*gm(2,3)*dgm(2,2)+6*gm(2,2)*dgm(2,3))
   cm(3,1,13)=3*gm(1,2)**2*dgm(1,2)-gm(1,1)*gm(2,2)*dgm(1,2)+2*gm(1,1)&
&   *gm(1,2)*dgm(2,2)
   cm(3,2,13)=2*gm(2,2)*(gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(3,3,13)=3*gm(2,3)**2*dgm(1,2)+3*gm(1,3)*gm(2,3)*dgm(2,2)-gm(3,3)&
&   *(gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(3,4,13)=gm(1,2)*gm(2,3)*dgm(2,2)+gm(2,2)*(4*gm(2,3)*dgm(1,2)&
&   +3*gm(1,3)*dgm(2,2))
   cm(3,5,13)=gm(2,3)*(6*gm(1,2)*dgm(1,2)+3*gm(1,1)*dgm(2,2))+gm(1,3)&
&   *(-2*gm(2,2)*dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(3,6,13)=4*gm(1,2)*gm(2,2)*dgm(1,2)+gm(1,2)**2*dgm(2,2)+3*gm(1,1)&
&   *gm(2,2)*dgm(2,2)
   cm(3,1,14)=3*gm(1,3)**2*dgm(1,2)+gm(1,3)*(6*gm(1,2)*dgm(1,3)+4*gm(1,1)&
&   *dgm(2,3))+gm(1,1)*(-gm(3,3)*dgm(1,2)-2*gm(2,3)*dgm(1,3)+2*gm(1,2)&
&   *dgm(3,3))
   cm(3,2,14)=3*gm(2,3)**2*dgm(1,2)+gm(2,3)*(4*gm(2,2)*dgm(1,3)+6*gm(1,2)&
&   *dgm(2,3))+gm(2,2)*(-gm(3,3)*dgm(1,2)-2*gm(1,3)*dgm(2,3)+2*gm(1,2)&
&   *dgm(3,3))
   cm(3,3,14)=2*gm(3,3)**2*dgm(1,2)+3*gm(1,3)*gm(2,3)*dgm(3,3)+gm(3,3)&
&   *(4*gm(2,3)*dgm(1,3)+4*gm(1,3)*dgm(2,3)-gm(1,2)*dgm(3,3))
   cm(3,4,14)=2*gm(2,3)**2*dgm(1,3)+6*gm(1,2)*gm(3,3)*dgm(2,3)+gm(2,3)&
&   *(4*gm(3,3)*dgm(1,2)+2*gm(1,3)*dgm(2,3)+gm(1,2)*dgm(3,3))+gm(2,2)&
&   *(6*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(3,5,14)=6*gm(1,2)*gm(3,3)*dgm(1,3)+2*gm(1,3)**2*dgm(2,3)+gm(1,3)&
&   *(4*gm(3,3)*dgm(1,2)+2*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3))+gm(1,1)&
&   *(6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(3,6,14)=gm(1,2)*(-2*gm(3,3)*dgm(1,2)+2*gm(2,3)*dgm(1,3))+gm(1,3)&
&   *(6*gm(2,3)*dgm(1,2)+6*gm(2,2)*dgm(1,3)+2*gm(1,2)*dgm(2,3))+gm(1,2)&
&   **2*dgm(3,3)+gm(1,1)*(6*gm(2,3)*dgm(2,3)+3*gm(2,2)*dgm(3,3))
   cm(3,1,15)=3*gm(1,3)**2*dgm(1,3)-gm(1,1)*gm(3,3)*dgm(1,3)+2*gm(1,1)&
&   *gm(1,3)*dgm(3,3)
   cm(3,2,15)=3*gm(2,3)**2*dgm(1,3)+3*gm(1,2)*gm(2,3)*dgm(3,3)-gm(2,2)&
&   *(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(3,3,15)=2*gm(3,3)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(3,4,15)=3*gm(1,2)*gm(3,3)*dgm(3,3)+gm(2,3)*(4*gm(3,3)*dgm(1,3)&
&   +gm(1,3)*dgm(3,3))
   cm(3,5,15)=4*gm(1,3)*gm(3,3)*dgm(1,3)+gm(1,3)**2*dgm(3,3)+3*gm(1,1)&
&   *gm(3,3)*dgm(3,3)
   cm(3,6,15)=-2*gm(1,2)*gm(3,3)*dgm(1,3)+3*gm(1,1)*gm(2,3)*dgm(3,3)&
&   +gm(1,3)*(6*gm(2,3)*dgm(1,3)+gm(1,2)*dgm(3,3))
   cm(3,1,17)=3*gm(1,2)*gm(1,3)*dgm(2,2)+3*gm(1,2)**2*dgm(2,3)-gm(1,1)&
&   *(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(3,2,17)=2*gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(3,3,17)=2*gm(2,3)*gm(3,3)*dgm(2,2)+3*gm(2,3)**2*dgm(2,3)-gm(2,2)&
&   *gm(3,3)*dgm(2,3)
   cm(3,4,17)=gm(2,3)**2*dgm(2,2)+3*gm(2,2)*gm(3,3)*dgm(2,2)+4*gm(2,2)&
&   *gm(2,3)*dgm(2,3)
   cm(3,5,17)=gm(1,3)*(1*gm(2,3)*dgm(2,2)-2*gm(2,2)*dgm(2,3))+gm(1,2)&
&   *(3*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))
   cm(3,6,17)=3*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,2)*(1*gm(2,3)*dgm(2,2)&
&   +4*gm(2,2)*dgm(2,3))
   cm(3,1,18)=((6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*dgm(2,2))/4.d0
   cm(3,2,18)=gm(2,2)**2*dgm(2,2)
   cm(3,3,18)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm(2,2))/4.d0
   cm(3,4,18)=2*gm(2,2)*gm(2,3)*dgm(2,2)
   cm(3,5,18)=((-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*dgm(2,2))/2.d0
   cm(3,6,18)=2*gm(1,2)*gm(2,2)*dgm(2,2)
   cm(3,1,19)=1.5d0*gm(1,3)**2*dgm(2,2)+6*gm(1,2)*gm(1,3)*dgm(2,3)&
&   +1.5d0*gm(1,2)**2*dgm(3,3)+gm(1,1)*(-0.5d0*gm(3,3)*dgm(2,2)-2*gm(2,3)&
&   *dgm(2,3)-0.5d0*gm(2,2)*dgm(3,3))
   cm(3,2,19)=1.5d0*gm(2,3)**2*dgm(2,2)+4*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(-0.5d0*gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3))
   cm(3,3,19)=gm(3,3)**2*dgm(2,2)+1.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)&
&   *(4*gm(2,3)*dgm(2,3)-0.5d0*gm(2,2)*dgm(3,3))
   cm(3,4,19)=2*gm(2,3)**2*dgm(2,3)+6*gm(2,2)*gm(3,3)*dgm(2,3)+2*gm(2,3)&
&   *(gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3))
   cm(3,5,19)=gm(1,3)*(2*gm(3,3)*dgm(2,2)+2*gm(2,3)*dgm(2,3)-gm(2,2)&
&   *dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(3,6,19)=gm(1,3)*(3*gm(2,3)*dgm(2,2)+6*gm(2,2)*dgm(2,3))+gm(1,2)&
&   *(-gm(3,3)*dgm(2,2)+2*(gm(2,3)*dgm(2,3)+gm(2,2)*dgm(3,3)))
   cm(3,1,20)=3*gm(1,3)**2*dgm(2,3)+3*gm(1,2)*gm(1,3)*dgm(3,3)-gm(1,1)&
&   *(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(3,2,20)=3*gm(2,3)**2*dgm(2,3)-gm(2,2)*gm(3,3)*dgm(2,3)+2*gm(2,2)&
&   *gm(2,3)*dgm(3,3)
   cm(3,3,20)=2*gm(3,3)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(3,4,20)=4*gm(2,3)*gm(3,3)*dgm(2,3)+gm(2,3)**2*dgm(3,3)+3*gm(2,2)&
&   *gm(3,3)*dgm(3,3)
   cm(3,5,20)=3*gm(1,2)*gm(3,3)*dgm(3,3)+gm(1,3)*(4*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*dgm(3,3))
   cm(3,6,20)=gm(1,3)*(6*gm(2,3)*dgm(2,3)+3*gm(2,2)*dgm(3,3))+gm(1,2)&
&   *(-2*gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(3,1,21)=((6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*dgm(3,3))/4.d0
   cm(3,2,21)=((6*gm(2,3)**2-2*gm(2,2)*gm(3,3))*dgm(3,3))/4.d0
   cm(3,3,21)=gm(3,3)**2*dgm(3,3)
   cm(3,4,21)=2*gm(2,3)*gm(3,3)*dgm(3,3)
   cm(3,5,21)=2*gm(1,3)*gm(3,3)*dgm(3,3)
   cm(3,6,21)=((6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*dgm(3,3))/2.d0
 elseif(rank==3)then
   cm(1,1,1)=gm(1,1)**3*dgm(1,1)
   cm(1,2,1)=gm(1,1)*(4.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(1,1)
   cm(1,3,1)=gm(1,1)*(4.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(1,1)
   cm(1,4,1)=(gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm(1,1))&
&   /6.d0
   cm(1,5,1)=3*gm(1,1)**2*gm(1,3)*dgm(1,1)
   cm(1,6,1)=3*gm(1,1)**2*gm(1,2)*dgm(1,1)
   cm(1,7,1)=gm(1,2)*(2.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(1,1)
   cm(1,8,1)=((-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm(1,1))/12.d0
   cm(1,9,1)=((90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm(1,1))/12.d0
   cm(1,10,1)=gm(1,3)*(2.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(1,1)
   cm(1,1,2)=gm(1,1)*(4.5d0*gm(1,2)**2*dgm(1,1)+6*gm(1,1)*gm(1,2)&
&   *dgm(1,2)+gm(1,1)*(-1.5d0*gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2)))
   cm(1,2,2)=3*gm(1,2)**3*dgm(1,2)+15*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +gm(1,1)*gm(2,2)*(6*gm(2,2)*dgm(1,1)-1.5d0*gm(1,1)*dgm(2,2))&
&   +gm(1,2)**2*(3*gm(2,2)*dgm(1,1)+4.5d0*gm(1,1)*dgm(2,2))
   cm(1,3,2)=-3*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)*gm(2,3)*(9*gm(1,2)&
&   *dgm(1,1)+24*gm(1,1)*dgm(1,2))+gm(1,1)*(7.5d0*gm(2,3)**2*dgm(1,1)&
&   -1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)-9*gm(1,2)*gm(3,3)*dgm(1,2))-1.5d0*gm(1,1)&
&   **2*gm(3,3)*dgm(2,2)+gm(1,3)**2*(-3*gm(2,2)*dgm(1,1)+3*gm(1,2)&
&   *dgm(1,2)+4.5d0*gm(1,1)*dgm(2,2))
   cm(1,4,2)=gm(1,2)**2*(3*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2))+gm(1,1)&
&   *(gm(2,2)*(12*gm(2,3)*dgm(1,1)+24*gm(1,3)*dgm(1,2))-3*gm(1,1)&
&   *gm(2,3)*dgm(2,2))+gm(1,2)*(6*gm(1,1)*gm(2,3)*dgm(1,2)+gm(1,3)&
&   *(3*gm(2,2)*dgm(1,1)+9*gm(1,1)*dgm(2,2)))
   cm(1,5,2)=1.5d0*gm(1,2)**2*gm(1,3)*dgm(1,1)+gm(1,1)*gm(1,2)*(12*gm(2,3)&
&   *dgm(1,1)+6*gm(1,3)*dgm(1,2))+gm(1,1)*(12*gm(1,1)*gm(2,3)*dgm(1,2)&
&   +gm(1,3)*(-4.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))
   cm(1,6,2)=1.5d0*gm(1,2)**3*dgm(1,1)+6*gm(1,1)*gm(1,2)**2*dgm(1,2)&
&   +12*gm(1,1)**2*gm(2,2)*dgm(1,2)+gm(1,1)*gm(1,2)*(7.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,1)*dgm(2,2))
   cm(1,7,2)=9*gm(1,2)**2*gm(2,2)*dgm(1,2)-3*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +2.5d0*gm(1,2)**3*dgm(2,2)+gm(1,2)*gm(2,2)*(3*gm(2,2)*dgm(1,1)&
&   -1.5d0*gm(1,1)*dgm(2,2))
   cm(1,8,2)=(3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,1)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(2,2))/12.d0
   cm(1,9,2)=gm(2,3)*(3*gm(1,2)**2*dgm(1,2)-9*gm(1,1)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)*(3*gm(2,2)*dgm(1,1)-3*gm(1,1)*dgm(2,2)))+gm(1,3)*(6*gm(2,2)&
&   **2*dgm(1,1)+7.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)*(24*gm(1,2)*dgm(1,2)&
&   -1.5d0*gm(1,1)*dgm(2,2)))
   cm(1,10,2)=(3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,1)+6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,2)+(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm(2,2))/36.d0
   cm(1,1,3)=gm(1,1)*(4.5d0*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(1,3)&
&   *dgm(1,3)+gm(1,1)*(-1.5d0*gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3)))
   cm(1,2,3)=-3*gm(1,3)**2*gm(2,2)*dgm(1,1)-3*gm(1,2)**2*gm(3,3)&
&   *dgm(1,1)+gm(1,3)*(9*gm(1,2)*gm(2,3)*dgm(1,1)+3*gm(1,2)**2*dgm(1,3)&
&   -9*gm(1,1)*gm(2,2)*dgm(1,3))-1.5d0*gm(1,1)**2*gm(2,2)*dgm(3,3)&
&   +gm(1,1)*(7.5d0*gm(2,3)**2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)&
&   +24*gm(1,2)*gm(2,3)*dgm(1,3)+4.5d0*gm(1,2)**2*dgm(3,3))
   cm(1,3,3)=3*gm(1,3)**3*dgm(1,3)+15*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +gm(1,1)*gm(3,3)*(6*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*dgm(3,3))&
&   +gm(1,3)**2*(3*gm(3,3)*dgm(1,1)+4.5d0*gm(1,1)*dgm(3,3))
   cm(1,4,3)=gm(1,3)**2*(3*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3))+gm(1,1)&
&   *(24*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(12*gm(3,3)*dgm(1,1)-3*gm(1,1)&
&   *dgm(3,3)))+gm(1,3)*(6*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)*(3*gm(3,3)&
&   *dgm(1,1)+9*gm(1,1)*dgm(3,3)))
   cm(1,5,3)=1.5d0*gm(1,3)**3*dgm(1,1)+6*gm(1,1)*gm(1,3)**2*dgm(1,3)&
&   +12*gm(1,1)**2*gm(3,3)*dgm(1,3)+gm(1,1)*gm(1,3)*(7.5d0*gm(3,3)&
&   *dgm(1,1)+3*gm(1,1)*dgm(3,3))
   cm(1,6,3)=12*gm(1,1)*gm(2,3)*(gm(1,3)*dgm(1,1)+gm(1,1)*dgm(1,3))&
&   +gm(1,2)*(1.5d0*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(1,3)*dgm(1,3)&
&   +gm(1,1)*(-4.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3)))
   cm(1,7,3)=(3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,1)+6*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm(3,3))/36.d0
   cm(1,8,3)=gm(1,3)**2*(3*gm(2,3)*dgm(1,3)+7.5d0*gm(1,2)*dgm(3,3))&
&   +gm(1,3)*(24*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(3*gm(3,3)*dgm(1,1)&
&   -3*gm(1,1)*dgm(3,3)))+gm(3,3)*(-9*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)&
&   *(6*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*dgm(3,3)))
   cm(1,9,3)=(3*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,1)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(3,3))/12.d0
   cm(1,10,3)=9*gm(1,3)**2*gm(3,3)*dgm(1,3)-3*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +2.5d0*gm(1,3)**3*dgm(3,3)+gm(1,3)*gm(3,3)*(3*gm(3,3)*dgm(1,1)&
&   -1.5d0*gm(1,1)*dgm(3,3))
   cm(1,1,4)=gm(1,1)*(gm(1,2)*(9*gm(1,3)*dgm(1,1)+6*gm(1,1)*dgm(1,3))&
&   +gm(1,1)*(-3*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2)+2*gm(1,1)*dgm(2,3)))
   cm(1,2,4)=3*gm(1,2)**3*dgm(1,3)+gm(1,2)*(3*gm(1,3)*gm(2,2)*dgm(1,1)&
&   +gm(1,1)*(24*gm(2,3)*dgm(1,2)+15*gm(2,2)*dgm(1,3)))+gm(1,1)*gm(2,2)&
&   *(12*gm(2,3)*dgm(1,1)-9*gm(1,3)*dgm(1,2)-3*gm(1,1)*dgm(2,3))&
&   +gm(1,2)**2*(3*gm(2,3)*dgm(1,1)+3*gm(1,3)*dgm(1,2)+9*gm(1,1)&
&   *dgm(2,3))
   cm(1,3,4)=3*gm(1,3)**3*dgm(1,2)+gm(1,3)*(3*gm(1,2)*gm(3,3)*dgm(1,1)&
&   +gm(1,1)*(15*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3)))+gm(1,1)*gm(3,3)&
&   *(12*gm(2,3)*dgm(1,1)-9*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))&
&   +gm(1,3)**2*(3*gm(2,3)*dgm(1,1)+3*gm(1,2)*dgm(1,3)+9*gm(1,1)&
&   *dgm(2,3))
   cm(1,4,4)=9*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)**2*(9*gm(2,2)&
&   *dgm(1,1)+6*gm(1,2)*dgm(1,2))+gm(1,1)*(9*gm(2,3)**2*dgm(1,1)&
&   +15*gm(2,2)*gm(3,3)*dgm(1,1)+24*gm(1,2)*gm(3,3)*dgm(1,2)+6*gm(1,2)&
&   *gm(2,3)*dgm(1,3))-6*gm(1,1)**2*gm(2,3)*dgm(2,3)+gm(1,3)*(6*gm(1,2)&
&   **2*dgm(1,3)+gm(1,1)*(6*gm(2,3)*dgm(1,2)+24*gm(2,2)*dgm(1,3))&
&   +gm(1,2)*(-6*gm(2,3)*dgm(1,1)+18*gm(1,1)*dgm(2,3)))
   cm(1,5,4)=gm(1,2)*(3*gm(1,3)**2*dgm(1,1)+12*gm(1,1)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,3)*dgm(1,3))+gm(1,1)*(6*gm(1,3)**2*dgm(1,2)+12*gm(1,1)&
&   *(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)*(3*gm(2,3)*dgm(1,1)&
&   +6*gm(1,1)*dgm(2,3)))
   cm(1,6,4)=gm(1,2)**2*(3*gm(1,3)*dgm(1,1)+6*gm(1,1)*dgm(1,3))+12*gm(1,1)&
&   *(gm(1,3)*gm(2,2)*dgm(1,1)+gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)&
&   *dgm(1,3)))+gm(1,1)*gm(1,2)*(3*gm(2,3)*dgm(1,1)+6*(gm(1,3)*dgm(1,2)&
&   +gm(1,1)*dgm(2,3)))
   cm(1,7,4)=gm(1,3)*gm(2,2)*(-3*gm(2,2)*dgm(1,1)-6*gm(1,2)*dgm(1,2))&
&   -3*gm(1,1)*gm(2,2)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)&
&   **2*(15*gm(2,3)*dgm(1,2)+9*gm(2,2)*dgm(1,3))+5*gm(1,2)**3*dgm(2,3)&
&   +gm(1,2)*gm(2,2)*(9*gm(2,3)*dgm(1,1)-3*gm(1,1)*dgm(2,3))
   cm(1,8,4)=-6*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(-9*gm(2,3)*gm(3,3)&
&   *dgm(1,2)-6*gm(2,3)**2*dgm(1,3)-3*gm(2,2)*gm(3,3)*dgm(1,3))+gm(1,2)&
&   *gm(3,3)*(3*gm(2,3)*dgm(1,1)-3*gm(1,1)*dgm(2,3))+gm(1,3)*(3*gm(2,3)&
&   **2*dgm(1,1)+gm(3,3)*(12*gm(2,2)*dgm(1,1)+24*gm(1,2)*dgm(1,2))&
&   +gm(2,3)*(18*gm(1,2)*dgm(1,3)-6*gm(1,1)*dgm(2,3)))+gm(1,3)**2*(3*gm(2,3)&
&   *dgm(1,2)+15*(gm(2,2)*dgm(1,3)+gm(1,2)*dgm(2,3)))
   cm(1,9,4)=-6*gm(1,3)**2*gm(2,2)*dgm(1,2)+gm(1,2)**2*(15*gm(3,3)&
&   *dgm(1,2)+3*gm(2,3)*dgm(1,3))+gm(1,1)*(-6*gm(2,3)**2*dgm(1,2)&
&   -3*gm(2,2)*gm(3,3)*dgm(1,2)-9*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,2)&
&   *(3*gm(2,3)**2*dgm(1,1)+12*gm(2,2)*gm(3,3)*dgm(1,1)-6*gm(1,1)&
&   *gm(2,3)*dgm(2,3))+gm(1,3)*(gm(2,2)*(3*gm(2,3)*dgm(1,1)+24*gm(1,2)&
&   *dgm(1,3)-3*gm(1,1)*dgm(2,3))+gm(1,2)*(18*gm(2,3)*dgm(1,2)+15*gm(1,2)&
&   *dgm(2,3)))
   cm(1,10,4)=gm(1,3)**2*(9*gm(3,3)*dgm(1,2)+15*gm(2,3)*dgm(1,3))&
&   -3*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(1,1)+gm(1,1)*(gm(3,3)*dgm(1,2)&
&   +gm(2,3)*dgm(1,3)))+5*gm(1,3)**3*dgm(2,3)+gm(1,3)*gm(3,3)*(9*gm(2,3)&
&   *dgm(1,1)-6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))
   cm(1,1,5)=gm(1,1)**2*(3*gm(1,3)*dgm(1,1)+2*gm(1,1)*dgm(1,3))
   cm(1,2,5)=12*gm(1,1)*gm(1,2)*gm(2,3)*dgm(1,1)+gm(1,1)*gm(2,2)&
&   *(-4.5d0*gm(1,3)*dgm(1,1)-3*gm(1,1)*dgm(1,3))+gm(1,2)**2*(1.5d0*gm(1,3)&
&   *dgm(1,1)+9*gm(1,1)*dgm(1,3))
   cm(1,3,5)=1.5d0*gm(1,3)**3*dgm(1,1)+7.5d0*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm(1,1)+9*gm(1,1)*gm(1,3)**2*dgm(1,3)-3*gm(1,1)**2*gm(3,3)&
&   *dgm(1,3)
   cm(1,4,5)=gm(1,1)*gm(2,3)*(3*gm(1,3)*dgm(1,1)-6*gm(1,1)*dgm(1,3))&
&   +gm(1,2)*(3*gm(1,3)**2*dgm(1,1)+12*gm(1,1)*gm(3,3)*dgm(1,1)+18*gm(1,1)&
&   *gm(1,3)*dgm(1,3))
   cm(1,5,5)=gm(1,1)*(3*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,3)*dgm(1,3))
   cm(1,6,5)=gm(1,1)*(6*gm(1,1)*gm(2,3)*dgm(1,1)+gm(1,2)*(3*gm(1,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(1,3)))
   cm(1,7,5)=7.5d0*gm(1,2)**2*gm(2,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,2)&
&   *gm(2,3)*dgm(1,1)+5*gm(1,2)**3*dgm(1,3)-3*gm(1,2)*gm(2,2)*(gm(1,3)&
&   *dgm(1,1)+gm(1,1)*dgm(1,3))
   cm(1,8,5)=gm(1,1)*gm(3,3)*(-4.5d0*gm(2,3)*dgm(1,1)-3*gm(1,2)*dgm(1,3))&
&   +gm(1,3)**2*(1.5d0*gm(2,3)*dgm(1,1)+15*gm(1,2)*dgm(1,3))+gm(1,3)&
&   *(12*gm(1,2)*gm(3,3)*dgm(1,1)-6*gm(1,1)*gm(2,3)*dgm(1,3))
   cm(1,9,5)=(60*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,1)+40*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(1,3))/240.d0
   cm(1,10,5)=4.5d0*gm(1,3)**2*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(3,3)&
&   **2*dgm(1,1)+5*gm(1,3)**3*dgm(1,3)-3*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm(1,3)
   cm(1,1,6)=gm(1,1)**2*(3*gm(1,2)*dgm(1,1)+2*gm(1,1)*dgm(1,2))
   cm(1,2,6)=1.5d0*gm(1,2)**3*dgm(1,1)+7.5d0*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm(1,1)+9*gm(1,1)*gm(1,2)**2*dgm(1,2)-3*gm(1,1)**2*gm(2,2)&
&   *dgm(1,2)
   cm(1,3,6)=gm(1,2)*(1.5d0*gm(1,3)**2-4.5d0*gm(1,1)*gm(3,3))*dgm(1,1)&
&   +gm(1,1)*(12*gm(1,3)*gm(2,3)*dgm(1,1)+9*gm(1,3)**2*dgm(1,2)-3*gm(1,1)&
&   *gm(3,3)*dgm(1,2))
   cm(1,4,6)=3*gm(1,2)**2*gm(1,3)*dgm(1,1)+gm(1,1)*gm(1,2)*(3*gm(2,3)&
&   *dgm(1,1)+18*gm(1,3)*dgm(1,2))+gm(1,1)*(12*gm(1,3)*gm(2,2)*dgm(1,1)&
&   -6*gm(1,1)*gm(2,3)*dgm(1,2))
   cm(1,5,6)=gm(1,1)*(3*gm(1,2)*gm(1,3)*dgm(1,1)+6*gm(1,1)*(gm(2,3)&
&   *dgm(1,1)+gm(1,3)*dgm(1,2)))
   cm(1,6,6)=gm(1,1)*(3*gm(1,2)**2*dgm(1,1)+6*gm(1,1)*gm(2,2)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,2)*dgm(1,2))
   cm(1,7,6)=4.5d0*gm(1,2)**2*gm(2,2)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,2)&
&   **2*dgm(1,1)+5*gm(1,2)**3*dgm(1,2)-3*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm(1,2)
   cm(1,8,6)=(60*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,1)+40*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(1,2))/240.d0
   cm(1,9,6)=gm(1,1)*gm(2,2)*(-4.5d0*gm(2,3)*dgm(1,1)-3*gm(1,3)*dgm(1,2))&
&   +gm(1,2)**2*(1.5d0*gm(2,3)*dgm(1,1)+15*gm(1,3)*dgm(1,2))+gm(1,2)&
&   *(12*gm(1,3)*gm(2,2)*dgm(1,1)-6*gm(1,1)*gm(2,3)*dgm(1,2))
   cm(1,10,6)=7.5d0*gm(1,3)**2*gm(2,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,3)&
&   *gm(3,3)*dgm(1,1)+5*gm(1,3)**3*dgm(1,2)-3*gm(1,3)*gm(3,3)*(gm(1,2)&
&   *dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(1,1,7)=2.5d0*gm(1,2)**3*dgm(1,1)+9*gm(1,1)*gm(1,2)**2*dgm(1,2)&
&   -3*gm(1,1)**2*gm(2,2)*dgm(1,2)+gm(1,1)*gm(1,2)*(-1.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,1)*dgm(2,2))
   cm(1,2,7)=6*gm(1,2)**2*gm(2,2)*dgm(1,2)+12*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +1.5d0*gm(1,2)**3*dgm(2,2)+gm(1,2)*gm(2,2)*(3*gm(2,2)*dgm(1,1)&
&   +7.5d0*gm(1,1)*dgm(2,2))
   cm(1,3,7)=-6*gm(1,2)**2*gm(3,3)*dgm(1,2)+gm(1,1)*(15*gm(2,3)**2-3*gm(2,2)&
&   *gm(3,3))*dgm(1,2)+gm(1,3)*gm(2,3)*(-3*gm(2,2)*dgm(1,1)+18*gm(1,2)&
&   *dgm(1,2)+12*gm(1,1)*dgm(2,2))+gm(1,3)**2*(-6*gm(2,2)*dgm(1,2)&
&   +1.5d0*gm(1,2)*dgm(2,2))+gm(1,2)*(7.5d0*gm(2,3)**2*dgm(1,1)-1.5d0*gm(2,2)&
&   *gm(3,3)*dgm(1,1)-4.5d0*gm(1,1)*gm(3,3)*dgm(2,2))
   cm(1,4,7)=gm(2,3)*(6*gm(1,2)**2*dgm(1,2)+24*gm(1,1)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)*(9*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))+gm(1,3)*(-3*gm(2,2)&
&   **2*dgm(1,1)+3*gm(1,2)**2*dgm(2,2)+gm(2,2)*(6*gm(1,2)*dgm(1,2)&
&   +12*gm(1,1)*dgm(2,2)))
   cm(1,5,7)=gm(1,2)**2*(7.5d0*gm(2,3)*dgm(1,1)+3*gm(1,3)*dgm(1,2))&
&   +gm(1,1)*(gm(2,2)*(-1.5d0*gm(2,3)*dgm(1,1)-9*gm(1,3)*dgm(1,2))&
&   +6*gm(1,1)*gm(2,3)*dgm(2,2))+gm(1,2)*(24*gm(1,1)*gm(2,3)*dgm(1,2)&
&   +gm(1,3)*(-3*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))
   cm(1,6,7)=3*gm(1,2)**3*dgm(1,2)+15*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)**2*(4.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2))+gm(1,1)&
&   *gm(2,2)*(-1.5d0*gm(2,2)*dgm(1,1)+6*gm(1,1)*dgm(2,2))
   cm(1,7,7)=gm(2,2)*(1*gm(2,2)**2*dgm(1,1)+4.5d0*gm(1,2)**2*dgm(2,2)&
&   +gm(2,2)*(6*gm(1,2)*dgm(1,2)-1.5d0*gm(1,1)*dgm(2,2)))
   cm(1,8,7)=(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,1)&
&   +6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+3*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,2))/12.d0
   cm(1,9,7)=gm(2,2)**2*(3*gm(2,3)*dgm(1,1)+12*gm(1,3)*dgm(1,2))&
&   +1.5d0*gm(1,2)**2*gm(2,3)*dgm(2,2)+gm(2,2)*(-4.5d0*gm(1,1)*gm(2,3)&
&   *dgm(2,2)+gm(1,2)*(6*gm(2,3)*dgm(1,2)+12*gm(1,3)*dgm(2,2)))
   cm(1,10,7)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm(1,1)&
&   +6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&   *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2))/36.d0
   cm(1,1,8)=gm(1,1)*(9*gm(1,3)**2*dgm(1,2)+gm(1,1)*(-3*gm(3,3)*dgm(1,2)&
&   -6*gm(2,3)*dgm(1,3))+gm(1,3)*(-3*gm(2,3)*dgm(1,1)+6*gm(1,1)*dgm(2,3)))&
&   +gm(1,2)*(7.5d0*gm(1,3)**2*dgm(1,1)+18*gm(1,1)*gm(1,3)*dgm(1,3)&
&   +gm(1,1)*(-1.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3)))
   cm(1,2,8)=-6*gm(1,3)**2*gm(2,2)*dgm(1,2)+gm(1,2)**2*(-6*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3))+gm(1,1)*(15*gm(2,3)**2*dgm(1,2)&
&   -3*gm(2,2)*gm(3,3)*dgm(1,2)+24*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,3)&
&   *(gm(2,2)*(12*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3)-9*gm(1,1)*dgm(2,3))&
&   +gm(1,2)*(18*gm(2,3)*dgm(1,2)+3*gm(1,2)*dgm(2,3)))+1.5d0*gm(1,2)&
&   **3*dgm(3,3)+gm(1,2)*(1.5d0*gm(2,3)**2*dgm(1,1)-4.5d0*gm(2,2)&
&   *gm(3,3)*dgm(1,1)+24*gm(1,1)*gm(2,3)*dgm(2,3)+7.5d0*gm(1,1)*gm(2,2)&
&   *dgm(3,3))
   cm(1,3,8)=3*gm(1,3)**3*dgm(2,3)+gm(1,3)**2*(6*gm(3,3)*dgm(1,2)&
&   +6*gm(2,3)*dgm(1,3)+1.5d0*gm(1,2)*dgm(3,3))+gm(3,3)*(gm(1,1)&
&   *(12*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3))+gm(1,2)*(6*gm(3,3)&
&   *dgm(1,1)-4.5d0*gm(1,1)*dgm(3,3)))+gm(1,3)*(gm(3,3)*(6*gm(1,2)&
&   *dgm(1,3)+15*gm(1,1)*dgm(2,3))+gm(2,3)*(3*gm(3,3)*dgm(1,1)+12*gm(1,1)&
&   *dgm(3,3)))
   cm(1,4,8)=18*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(24*gm(2,3)*gm(3,3)&
&   *dgm(1,2)+18*gm(2,3)**2*dgm(1,3)+30*gm(2,2)*gm(3,3)*dgm(1,3))&
&   +gm(1,3)**2*(6*gm(2,3)*dgm(1,2)+18*gm(2,2)*dgm(1,3)+6*gm(1,2)&
&   *dgm(2,3))+gm(1,2)*(24*gm(1,1)*gm(3,3)*dgm(2,3)+3*gm(2,3)*(gm(3,3)&
&   *dgm(1,1)+gm(1,1)*dgm(3,3)))+gm(1,3)*(3*gm(2,3)**2*dgm(1,1)+gm(2,3)&
&   *(-12*gm(1,2)*dgm(1,3)+6*gm(1,1)*dgm(2,3))+12*gm(2,2)*(gm(3,3)&
&   *dgm(1,1)+gm(1,1)*dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(1,2)+3*gm(1,2)&
&   *dgm(3,3)))
   cm(1,5,8)=3*gm(1,3)**3*dgm(1,2)+gm(1,3)**2*(1.5d0*gm(2,3)*dgm(1,1)&
&   +6*(gm(1,2)*dgm(1,3)+gm(1,1)*dgm(2,3)))+gm(1,3)*(gm(1,1)*(15*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3))+gm(1,2)*(12*gm(3,3)*dgm(1,1)+3*gm(1,1)&
&   *dgm(3,3)))+gm(1,1)*(gm(3,3)*(24*gm(1,2)*dgm(1,3)+12*gm(1,1)&
&   *dgm(2,3))+gm(2,3)*(-4.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3)))
   cm(1,6,8)=-3*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)**2*(7.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,2)*dgm(1,2))+gm(1,3)*(6*gm(1,2)**2*dgm(1,3)&
&   +24*gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)*(9*gm(2,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(2,3)))+gm(1,1)*(-3*gm(2,3)**2*dgm(1,1)&
&   -1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)-9*gm(1,2)*gm(3,3)*dgm(1,2)+6*gm(1,2)&
&   *gm(2,3)*dgm(1,3)+3*gm(1,2)**2*dgm(3,3))+gm(1,1)**2*(12*gm(2,3)&
&   *dgm(2,3)+6*gm(2,2)*dgm(3,3))
   cm(1,7,8)=15*gm(1,2)*gm(2,3)*(gm(2,3)*dgm(1,2)+gm(1,2)*dgm(2,3))&
&   +gm(2,2)**2*(-1.5d0*gm(3,3)*dgm(1,1)-6*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)&
&   *dgm(3,3))+gm(2,2)*(4.5d0*gm(2,3)**2*dgm(1,1)+gm(2,3)*(-6*gm(1,3)&
&   *dgm(1,2)+18*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))+gm(1,2)*(-3*gm(3,3)&
&   *dgm(1,2)-6*gm(1,3)*dgm(2,3)+4.5d0*gm(1,2)*dgm(3,3)))
   cm(1,8,8)=gm(2,3)**2*(3*gm(3,3)*dgm(1,1)+6*gm(1,3)*dgm(1,3)-3*gm(1,1)&
&   *dgm(3,3))+gm(1,2)*gm(3,3)*(12*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)&
&   -3*gm(1,2)*dgm(3,3))+gm(2,2)*(6*gm(3,3)**2*dgm(1,1)+24*gm(1,3)&
&   *gm(3,3)*dgm(1,3)+7.5d0*gm(1,3)**2*dgm(3,3)-1.5d0*gm(1,1)*gm(3,3)&
&   *dgm(3,3))+gm(2,3)*(3*gm(1,3)**2*dgm(2,3)+gm(3,3)*(6*gm(1,2)&
&   *dgm(1,3)-9*gm(1,1)*dgm(2,3))+gm(1,3)*(6*gm(3,3)*dgm(1,2)+9*gm(1,2)&
&   *dgm(3,3)))
   cm(1,9,8)=1.5d0*gm(2,3)**3*dgm(1,1)-6*gm(1,3)**2*gm(2,2)*dgm(2,3)&
&   +gm(2,3)**2*(3*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)-6*gm(1,1)&
&   *dgm(2,3))+gm(3,3)*(24*gm(1,2)*gm(2,2)*dgm(1,3)+15*gm(1,2)**2*dgm(2,3)&
&   -3*gm(1,1)*gm(2,2)*dgm(2,3))+gm(1,3)*gm(2,2)*(-9*gm(3,3)*dgm(1,2)&
&   +12*gm(1,2)*dgm(3,3))+gm(2,3)*(gm(2,2)*(7.5d0*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*dgm(1,3)-4.5d0*gm(1,1)*dgm(3,3))+gm(1,2)*(24*gm(3,3)&
&   *dgm(1,2)+18*gm(1,3)*dgm(2,3)+1.5d0*gm(1,2)*dgm(3,3)))
   cm(1,10,8)=gm(2,3)*(3*gm(3,3)**2*dgm(1,1)+7.5d0*gm(1,3)**2*dgm(3,3)&
&   +gm(3,3)*(18*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)*dgm(3,3)))+gm(3,3)&
&   *(9*gm(1,3)**2*dgm(2,3)+gm(3,3)*(-6*gm(1,2)*dgm(1,3)-3*gm(1,1)&
&   *dgm(2,3))+gm(1,3)*(6*gm(3,3)*dgm(1,2)-3*gm(1,2)*dgm(3,3)))
   cm(1,1,9)=gm(1,2)**2*(7.5d0*gm(1,3)*dgm(1,1)+9*gm(1,1)*dgm(1,3))&
&   +gm(1,1)*(gm(1,1)*(-6*gm(2,3)*dgm(1,2)-3*gm(2,2)*dgm(1,3))+gm(1,3)&
&   *(-1.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))+gm(1,1)*gm(1,2)&
&   *(-3*gm(2,3)*dgm(1,1)+18*gm(1,3)*dgm(1,2)+6*gm(1,1)*dgm(2,3))
   cm(1,2,9)=6*gm(1,2)**2*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,1)&
&   *gm(2,2)*(24*gm(2,3)*dgm(1,2)+12*gm(2,2)*dgm(1,3))+gm(1,3)*(6*gm(2,2)&
&   **2*dgm(1,1)+1.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)*(6*gm(1,2)*dgm(1,2)&
&   -4.5d0*gm(1,1)*dgm(2,2)))+3*gm(1,2)**3*dgm(2,3)+gm(1,2)*(3*gm(2,2)&
&   *gm(2,3)*dgm(1,1)+12*gm(1,1)*gm(2,3)*dgm(2,2)+15*gm(1,1)*gm(2,2)&
&   *dgm(2,3))
   cm(1,3,9)=-6*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(24*gm(2,3)*gm(3,3)&
&   *dgm(1,2)+15*gm(2,3)**2*dgm(1,3)-3*gm(2,2)*gm(3,3)*dgm(1,3))&
&   +1.5d0*gm(1,3)**3*dgm(2,2)+gm(1,2)*gm(3,3)*(12*gm(2,3)*dgm(1,1)&
&   -9*gm(1,1)*dgm(2,3))+gm(1,3)**2*(6*gm(2,3)*dgm(1,2)-6*gm(2,2)&
&   *dgm(1,3)+3*gm(1,2)*dgm(2,3))+gm(1,3)*(1.5d0*gm(2,3)**2*dgm(1,1)&
&   +gm(3,3)*(-4.5d0*gm(2,2)*dgm(1,1)+6*gm(1,2)*dgm(1,2)+7.5d0*gm(1,1)&
&   *dgm(2,2))+gm(2,3)*(18*gm(1,2)*dgm(1,3)+24*gm(1,1)*dgm(2,3)))
   cm(1,4,9)=gm(1,2)**2*(18*gm(3,3)*dgm(1,2)+6*gm(2,3)*dgm(1,3))&
&   +gm(1,1)*(18*gm(2,3)**2*dgm(1,2)+30*gm(2,2)*gm(3,3)*dgm(1,2)&
&   +24*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,3)**2*(18*gm(2,2)*dgm(1,2)&
&   +3*gm(1,2)*dgm(2,2))+gm(1,2)*(3*gm(2,3)**2*dgm(1,1)+12*gm(3,3)&
&   *(gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))+6*gm(1,1)*gm(2,3)*dgm(2,3))&
&   +gm(1,3)*(-12*gm(1,2)*gm(2,3)*dgm(1,2)+3*gm(1,1)*gm(2,3)*dgm(2,2)&
&   +6*gm(1,2)**2*dgm(2,3)+gm(2,2)*(3*gm(2,3)*dgm(1,1)+6*gm(1,2)&
&   *dgm(1,3)+24*gm(1,1)*dgm(2,3)))
   cm(1,5,9)=7.5d0*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,1)*(-3*gm(2,3)&
&   **2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)+24*gm(1,2)*gm(3,3)&
&   *dgm(1,2)+24*gm(1,2)*gm(2,3)*dgm(1,3))+gm(1,3)**2*(-3*gm(2,2)&
&   *dgm(1,1)+6*gm(1,2)*dgm(1,2)+3*gm(1,1)*dgm(2,2))+gm(1,1)**2*(6*gm(3,3)&
&   *dgm(2,2)+12*gm(2,3)*dgm(2,3))+gm(1,3)*(3*gm(1,2)**2*dgm(1,3)&
&   +gm(1,1)*(6*gm(2,3)*dgm(1,2)-9*gm(2,2)*dgm(1,3))+gm(1,2)*(9*gm(2,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(2,3)))
   cm(1,6,9)=3*gm(1,2)**3*dgm(1,3)+gm(1,2)*(gm(1,1)*(6*gm(2,3)*dgm(1,2)&
&   +15*gm(2,2)*dgm(1,3))+gm(1,3)*(12*gm(2,2)*dgm(1,1)+3*gm(1,1)&
&   *dgm(2,2)))+gm(1,2)**2*(1.5d0*gm(2,3)*dgm(1,1)+6*(gm(1,3)*dgm(1,2)&
&   +gm(1,1)*dgm(2,3)))+gm(1,1)*(6*gm(1,1)*gm(2,3)*dgm(2,2)+gm(2,2)&
&   *(-4.5d0*gm(2,3)*dgm(1,1)+24*gm(1,3)*dgm(1,2)+12*gm(1,1)*dgm(2,3)))
   cm(1,7,9)=7.5d0*gm(1,2)**2*gm(2,3)*dgm(2,2)+gm(2,2)**2*(3*gm(2,3)&
&   *dgm(1,1)-6*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))&
&   +gm(2,2)*(-1.5d0*gm(1,1)*gm(2,3)*dgm(2,2)+gm(1,2)*(18*gm(2,3)&
&   *dgm(1,2)-3*gm(1,3)*dgm(2,2))+9*gm(1,2)**2*dgm(2,3))
   cm(1,8,9)=1.5d0*gm(2,3)**3*dgm(1,1)+gm(1,3)*gm(3,3)*(24*gm(2,2)&
&   *dgm(1,2)+12*gm(1,2)*dgm(2,2))+15*gm(1,3)**2*gm(2,2)*dgm(2,3)&
&   +gm(2,3)**2*(6*gm(1,3)*dgm(1,2)+3*gm(1,2)*dgm(1,3)-6*gm(1,1)&
&   *dgm(2,3))+gm(3,3)*(-9*gm(1,2)*gm(2,2)*dgm(1,3)-6*gm(1,2)**2*dgm(2,3)&
&   -3*gm(1,1)*gm(2,2)*dgm(2,3))+gm(2,3)*(gm(2,2)*(7.5d0*gm(3,3)&
&   *dgm(1,1)+24*gm(1,3)*dgm(1,3))+(1.5d0*gm(1,3)**2-4.5d0*gm(1,1)&
&   *gm(3,3))*dgm(2,2)+gm(1,2)*(6*gm(3,3)*dgm(1,2)+18*gm(1,3)*dgm(2,3)))
   cm(1,9,9)=gm(2,2)**2*(6*gm(3,3)*dgm(1,1)+12*gm(1,3)*dgm(1,3))&
&   -3*gm(1,1)*gm(2,3)**2*dgm(2,2)+gm(1,2)*gm(2,3)*(6*gm(2,3)*dgm(1,2)&
&   +9*gm(1,3)*dgm(2,2))+gm(1,2)**2*(7.5d0*gm(3,3)*dgm(2,2)+3*gm(2,3)&
&   *dgm(2,3))+gm(2,2)*(3*gm(2,3)**2*dgm(1,1)+(-3*gm(1,3)**2-1.5d0*gm(1,1)&
&   *gm(3,3))*dgm(2,2)+gm(2,3)*(6*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)&
&   -9*gm(1,1)*dgm(2,3))+24*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3)))
   cm(1,10,9)=gm(2,3)**2*(4.5d0*gm(3,3)*dgm(1,1)+15*gm(1,3)*dgm(1,3))&
&   +gm(2,3)*(18*gm(1,3)*gm(3,3)*dgm(1,2)+15*gm(1,3)**2*dgm(2,3)&
&   +gm(3,3)*(-6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3)))+gm(3,3)*(gm(2,2)&
&   *(-1.5d0*gm(3,3)*dgm(1,1)-3*gm(1,3)*dgm(1,3))+(4.5d0*gm(1,3)&
&   **2-1.5d0*gm(1,1)*gm(3,3))*dgm(2,2)-6*gm(1,2)*(gm(3,3)*dgm(1,2)&
&   +gm(1,3)*dgm(2,3)))
   cm(1,1,10)=2.5d0*gm(1,3)**3*dgm(1,1)+9*gm(1,1)*gm(1,3)**2*dgm(1,3)&
&   -3*gm(1,1)**2*gm(3,3)*dgm(1,3)+gm(1,1)*gm(1,3)*(-1.5d0*gm(3,3)&
&   *dgm(1,1)+3*gm(1,1)*dgm(3,3))
   cm(1,2,10)=-6*gm(1,3)**2*gm(2,2)*dgm(1,3)-6*gm(1,2)**2*gm(3,3)&
&   *dgm(1,3)+gm(1,1)*(15*gm(2,3)**2-3*gm(2,2)*gm(3,3))*dgm(1,3)&
&   +gm(1,2)*gm(2,3)*(-3*gm(3,3)*dgm(1,1)+12*gm(1,1)*dgm(3,3))+gm(1,3)&
&   *(7.5d0*gm(2,3)**2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)+18*gm(1,2)&
&   *gm(2,3)*dgm(1,3)+1.5d0*gm(1,2)**2*dgm(3,3)-4.5d0*gm(1,1)*gm(2,2)&
&   *dgm(3,3))
   cm(1,3,10)=6*gm(1,3)**2*gm(3,3)*dgm(1,3)+12*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +1.5d0*gm(1,3)**3*dgm(3,3)+gm(1,3)*gm(3,3)*(3*gm(3,3)*dgm(1,1)&
&   +7.5d0*gm(1,1)*dgm(3,3))
   cm(1,4,10)=gm(1,3)**2*(6*gm(2,3)*dgm(1,3)+3*gm(1,2)*dgm(3,3))&
&   +gm(1,3)*(6*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(9*gm(3,3)*dgm(1,1)&
&   +3*gm(1,1)*dgm(3,3)))+gm(3,3)*(24*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)&
&   *(-3*gm(3,3)*dgm(1,1)+12*gm(1,1)*dgm(3,3)))
   cm(1,5,10)=3*gm(1,3)**3*dgm(1,3)+15*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +gm(1,3)**2*(4.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3))+gm(1,1)&
&   *gm(3,3)*(-1.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3))
   cm(1,6,10)=gm(1,3)**2*(7.5d0*gm(2,3)*dgm(1,1)+3*gm(1,2)*dgm(1,3))&
&   +gm(1,3)*(24*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)*(-3*gm(3,3)*dgm(1,1)&
&   +3*gm(1,1)*dgm(3,3)))+gm(1,1)*(-9*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)&
&   *(-1.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3)))
   cm(1,7,10)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm(1,1)&
&   +6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))/36.d0
   cm(1,8,10)=12*gm(1,2)*gm(3,3)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))&
&   +gm(2,3)*(3*gm(3,3)**2*dgm(1,1)+1.5d0*gm(1,3)**2*dgm(3,3)+gm(3,3)&
&   *(6*gm(1,3)*dgm(1,3)-4.5d0*gm(1,1)*dgm(3,3)))
   cm(1,9,10)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,1)&
&   +6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/12.d0
   cm(1,10,10)=gm(3,3)*(1*gm(3,3)**2*dgm(1,1)+4.5d0*gm(1,3)**2*dgm(3,3)&
&   +gm(3,3)*(6*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)*dgm(3,3)))
   cm(1,1,11)=5*gm(1,2)**3*dgm(1,2)-3*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +4.5d0*gm(1,1)*gm(1,2)**2*dgm(2,2)-1.5d0*gm(1,1)**2*gm(2,2)*dgm(2,2)
   cm(1,2,11)=gm(2,2)*(6*gm(1,2)*gm(2,2)*dgm(1,2)+3*gm(1,2)**2*dgm(2,2)&
&   +6*gm(1,1)*gm(2,2)*dgm(2,2))
   cm(1,3,11)=(16*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,2)+24*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(2,2))/96.d0
   cm(1,4,11)=gm(1,3)*gm(2,2)*(-6*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))&
&   +gm(2,3)*(18*gm(1,2)*gm(2,2)*dgm(1,2)+3*gm(1,2)**2*dgm(2,2)+12*gm(1,1)&
&   *gm(2,2)*dgm(2,2))
   cm(1,5,11)=(16*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,2)+24*(6*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm(2,2))&
&   /96.d0
   cm(1,6,11)=9*gm(1,2)**2*gm(2,2)*dgm(1,2)-3*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +1.5d0*gm(1,2)**3*dgm(2,2)+7.5d0*gm(1,1)*gm(1,2)*gm(2,2)*dgm(2,2)
   cm(1,7,11)=gm(2,2)**2*(2*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))
   cm(1,8,11)=-3*gm(2,2)**2*gm(3,3)*dgm(1,2)+1.5d0*gm(1,2)*gm(2,3)&
&   **2*dgm(2,2)+gm(2,2)*(9*gm(2,3)**2*dgm(1,2)+12*gm(1,3)*gm(2,3)&
&   *dgm(2,2)-4.5d0*gm(1,2)*gm(3,3)*dgm(2,2))
   cm(1,9,11)=gm(2,2)*(3*gm(1,2)*gm(2,3)*dgm(2,2)+6*gm(2,2)*(gm(2,3)&
&   *dgm(1,2)+gm(1,3)*dgm(2,2)))
   cm(1,10,11)=5*gm(2,3)**3*dgm(1,2)+7.5d0*gm(1,3)*gm(2,3)**2*dgm(2,2)&
&   -1.5d0*gm(1,3)*gm(2,2)*gm(3,3)*dgm(2,2)-3*gm(2,3)*gm(3,3)*(gm(2,2)&
&   *dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(1,1,12)=(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(1,2)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(1,3)+gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(2,2)+4*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm(2,3)+gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm(3,3))/12.d0
   cm(1,2,12)=(2*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+2*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm(1,3)+(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,3)+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm(3,3))/4.d0
   cm(1,3,12)=gm(1,2)*gm(3,3)*(12*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3))&
&   -3*gm(1,2)**2*gm(3,3)*dgm(3,3)+gm(1,3)**2*(3*gm(3,3)*dgm(2,2)&
&   +6*gm(2,3)*dgm(2,3)-3*gm(2,2)*dgm(3,3))+gm(1,1)*(6*gm(3,3)**2*dgm(2,2)&
&   +24*gm(2,3)*gm(3,3)*dgm(2,3)+7.5d0*gm(2,3)**2*dgm(3,3)-1.5d0*gm(2,2)&
&   *gm(3,3)*dgm(3,3))+gm(1,3)*(3*gm(2,3)**2*dgm(1,3)+gm(3,3)*(-9*gm(2,2)&
&   *dgm(1,3)+6*gm(1,2)*dgm(2,3))+gm(2,3)*(6*gm(3,3)*dgm(1,2)+9*gm(1,2)&
&   *dgm(3,3)))
   cm(1,4,12)=(2*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+2*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm(1,3)+(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+4*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm(2,3)+(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))&
&   /2.d0
   cm(1,5,12)=(2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,2)+2*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(1,3)+(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm(2,2)+4*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm(2,3)+(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm(3,3))/4.d0
   cm(1,6,12)=(2*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,2)+2*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(2,2)+4*(6*gm(1,2)&
&   **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm(3,3))&
&   /4.d0
   cm(1,7,12)=(360*gm(1,2)*gm(2,3)**2*dgm(2,2)+gm(2,2)*(432*gm(2,3)&
&   **2*dgm(1,2)-72*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)*(-144*gm(1,3)&
&   *dgm(2,2)+864*gm(1,2)*dgm(2,3)))+gm(2,2)**2*(-144*gm(3,3)*dgm(1,2)&
&   +288*gm(2,3)*dgm(1,3)-288*gm(1,3)*dgm(2,3)+144*gm(1,2)*dgm(3,3)))&
&   /48.d0
   cm(1,8,12)=3*gm(2,3)**3*dgm(1,3)+gm(2,3)**2*(6*gm(3,3)*dgm(1,2)&
&   +6*gm(1,3)*dgm(2,3)+1.5d0*gm(1,2)*dgm(3,3))+gm(3,3)*(6*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(12*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)&
&   -4.5d0*gm(1,2)*dgm(3,3)))+gm(2,3)*(gm(3,3)*(3*gm(1,3)*dgm(2,2)&
&   +6*gm(1,2)*dgm(2,3))+gm(2,2)*(15*gm(3,3)*dgm(1,3)+12*gm(1,3)&
&   *dgm(3,3)))
   cm(1,9,12)=3*gm(2,3)**3*dgm(1,2)+gm(2,3)**2*(6*gm(2,2)*dgm(1,3)&
&   +1.5d0*gm(1,3)*dgm(2,2)+6*gm(1,2)*dgm(2,3))+gm(2,3)*(12*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(15*gm(3,3)*dgm(1,2)+6*gm(1,3)*dgm(2,3)&
&   +3*gm(1,2)*dgm(3,3)))+gm(2,2)*(gm(3,3)*(-4.5d0*gm(1,3)*dgm(2,2)&
&   +24*gm(1,2)*dgm(2,3))+gm(2,2)*(12*gm(3,3)*dgm(1,3)+6*gm(1,3)&
&   *dgm(3,3)))
   cm(1,10,12)=gm(2,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+18*gm(1,3)*dgm(2,3)&
&   -3*gm(1,2)*dgm(3,3))+gm(2,3)**2*(9*gm(3,3)*dgm(1,3)+7.5d0*gm(1,3)&
&   *dgm(3,3))+gm(3,3)*(gm(3,3)*(3*gm(1,3)*dgm(2,2)-6*gm(1,2)*dgm(2,3))&
&   +gm(2,2)*(-3*gm(3,3)*dgm(1,3)-1.5d0*gm(1,3)*dgm(3,3)))
   cm(1,1,13)=5*gm(1,2)**3*dgm(1,3)+gm(1,1)*gm(1,2)*(-6*gm(2,3)*dgm(1,2)&
&   -3*gm(2,2)*dgm(1,3)+9*gm(1,3)*dgm(2,2))+gm(1,2)**2*(15*gm(1,3)&
&   *dgm(1,2)+9*gm(1,1)*dgm(2,3))-3*gm(1,1)*(gm(1,3)*gm(2,2)*dgm(1,2)&
&   +gm(1,1)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3)))
   cm(1,2,13)=6*gm(1,2)*gm(2,2)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))&
&   +gm(1,3)*gm(2,2)*(12*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))+12*gm(1,1)&
&   *gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))+gm(1,2)**2*(3*gm(2,3)&
&   *dgm(2,2)+6*gm(2,2)*dgm(2,3))
   cm(1,3,13)=(3*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+3*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm(2,3))/6.d0
   cm(1,4,13)=gm(1,2)*(6*gm(2,3)**2*dgm(1,2)+24*gm(2,2)*gm(3,3)*dgm(1,2)&
&   +18*gm(2,2)*gm(2,3)*dgm(1,3))+9*gm(1,3)**2*gm(2,2)*dgm(2,2)+gm(1,2)&
&   **2*(9*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))+gm(1,1)*(9*gm(2,3)&
&   **2*dgm(2,2)+15*gm(2,2)*gm(3,3)*dgm(2,2)+24*gm(2,2)*gm(2,3)*dgm(2,3))&
&   +gm(1,3)*(-6*gm(2,2)**2*dgm(1,3)-6*gm(1,2)*gm(2,3)*dgm(2,2)+6*gm(2,2)&
&   *(gm(2,3)*dgm(1,2)+gm(1,2)*dgm(2,3)))
   cm(1,5,13)=(3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,2)+(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+3*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm(2,2)+3*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(2,3))/6.d0
   cm(1,6,13)=gm(1,2)**2*(3*gm(2,3)*dgm(1,2)+9*gm(2,2)*dgm(1,3)+3*gm(1,3)&
&   *dgm(2,2))+gm(1,1)*gm(2,2)*(-9*gm(2,3)*dgm(1,2)-3*gm(2,2)*dgm(1,3)&
&   +12*gm(1,3)*dgm(2,2))+3*gm(1,2)**3*dgm(2,3)+gm(1,2)*(24*gm(1,3)&
&   *gm(2,2)*dgm(1,2)+gm(1,1)*(3*gm(2,3)*dgm(2,2)+15*gm(2,2)*dgm(2,3)))
   cm(1,7,13)=gm(2,2)*(2*gm(2,2)**2*dgm(1,3)+9*gm(1,2)*gm(2,3)*dgm(2,2)&
&   +gm(2,2)*(6*gm(2,3)*dgm(1,2)-3*gm(1,3)*dgm(2,2)+6*gm(1,2)*dgm(2,3)))
   cm(1,8,13)=3*gm(2,3)**3*dgm(1,2)+gm(2,2)*gm(3,3)*(-3*gm(2,2)*dgm(1,3)&
&   +12*gm(1,3)*dgm(2,2)-9*gm(1,2)*dgm(2,3))+gm(2,3)**2*(9*gm(2,2)&
&   *dgm(1,3)+3*(gm(1,3)*dgm(2,2)+gm(1,2)*dgm(2,3)))+gm(2,3)*(3*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(15*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)))
   cm(1,9,13)=3*gm(1,2)*gm(2,3)**2*dgm(2,2)+gm(2,2)**2*(12*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3)+12*gm(1,3)*dgm(2,3))+gm(2,2)*(6*gm(2,3)&
&   **2*dgm(1,2)+12*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)*(3*gm(1,3)*dgm(2,2)&
&   +6*gm(1,2)*dgm(2,3)))
   cm(1,10,13)=5*gm(2,3)**3*dgm(1,3)+gm(2,3)*gm(3,3)*(-3*gm(2,2)&
&   *dgm(1,3)+9*gm(1,3)*dgm(2,2)-6*gm(1,2)*dgm(2,3))+gm(2,3)**2*(9*gm(3,3)&
&   *dgm(1,2)+15*gm(1,3)*dgm(2,3))-3*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(2,2)&
&   +gm(2,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3)))
   cm(1,1,14)=5*gm(1,3)**3*dgm(1,2)+gm(1,3)**2*(15*gm(1,2)*dgm(1,3)&
&   +9*gm(1,1)*dgm(2,3))+gm(1,1)*gm(1,3)*(-3*gm(3,3)*dgm(1,2)-6*gm(2,3)&
&   *dgm(1,3)+9*gm(1,2)*dgm(3,3))-3*gm(1,1)*(gm(1,2)*gm(3,3)*dgm(1,3)&
&   +gm(1,1)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3)))
   cm(1,2,14)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,3)+3*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))/6.d0
   cm(1,3,14)=gm(1,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+6*gm(2,3)*dgm(1,3)&
&   +3*gm(1,2)*dgm(3,3))+gm(1,3)**2*(6*gm(3,3)*dgm(2,3)+3*gm(2,3)&
&   *dgm(3,3))+12*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(1,3)+gm(1,1)*(gm(3,3)&
&   *dgm(2,3)+gm(2,3)*dgm(3,3)))
   cm(1,4,14)=(gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+3*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,3)+3*(9*gm(1,3)**2*gm(2,2)&
&   -6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)&
&   **2+15*gm(2,2)*gm(3,3)))*dgm(3,3))/3.d0
   cm(1,5,14)=(gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm(1,3)+3*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm(2,3)+3*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm(3,3))/6.d0
   cm(1,6,14)=((90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,2)+3*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(2,3)+3*(6*gm(1,2)&
&   **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(3,3))/6.d0
   cm(1,7,14)=5*gm(2,3)**3*dgm(1,2)+gm(2,3)**2*(9*gm(2,2)*dgm(1,3)&
&   +15*gm(1,2)*dgm(2,3))+gm(2,2)*gm(2,3)*(-3*gm(3,3)*dgm(1,2)-6*gm(1,3)&
&   *dgm(2,3)+9*gm(1,2)*dgm(3,3))-3*gm(2,2)*(gm(1,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,2)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3)))
   cm(1,8,14)=gm(2,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+6*gm(1,3)*dgm(2,3)&
&   +3*gm(1,2)*dgm(3,3))+gm(2,3)**2*(6*gm(3,3)*dgm(1,3)+3*gm(1,3)&
&   *dgm(3,3))+12*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(2,3)+gm(2,2)*(gm(3,3)&
&   *dgm(1,3)+gm(1,3)*dgm(3,3)))
   cm(1,9,14)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm(1,3)+3*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm(2,3)&
&   +3*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/6.d0
   cm(1,10,14)=gm(3,3)*(2*gm(3,3)**2*dgm(1,2)+9*gm(1,3)*gm(2,3)*dgm(3,3)&
&   +gm(3,3)*(6*gm(2,3)*dgm(1,3)+6*gm(1,3)*dgm(2,3)-3*gm(1,2)*dgm(3,3)))
   cm(1,1,15)=5*gm(1,3)**3*dgm(1,3)-3*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +4.5d0*gm(1,1)*gm(1,3)**2*dgm(3,3)-1.5d0*gm(1,1)**2*gm(3,3)*dgm(3,3)
   cm(1,2,15)=(16*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+24*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(3,3))/96.d0
   cm(1,3,15)=gm(3,3)*(6*gm(1,3)*gm(3,3)*dgm(1,3)+3*gm(1,3)**2*dgm(3,3)&
&   +6*gm(1,1)*gm(3,3)*dgm(3,3))
   cm(1,4,15)=3*gm(1,3)**2*gm(2,3)*dgm(3,3)+gm(1,3)*gm(3,3)*(18*gm(2,3)&
&   *dgm(1,3)+3*gm(1,2)*dgm(3,3))+gm(3,3)*(-6*gm(1,2)*gm(3,3)*dgm(1,3)&
&   +12*gm(1,1)*gm(2,3)*dgm(3,3))
   cm(1,5,15)=9*gm(1,3)**2*gm(3,3)*dgm(1,3)-3*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +1.5d0*gm(1,3)**3*dgm(3,3)+7.5d0*gm(1,1)*gm(1,3)*gm(3,3)*dgm(3,3)
   cm(1,6,15)=(16*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,3)+24*(48*gm(1,1)*gm(1,3)&
&   *gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(3,3))&
&   /96.d0
   cm(1,7,15)=5*gm(2,3)**3*dgm(1,3)+7.5d0*gm(1,2)*gm(2,3)**2*dgm(3,3)&
&   -1.5d0*gm(1,2)*gm(2,2)*gm(3,3)*dgm(3,3)-3*gm(2,2)*gm(2,3)*(gm(3,3)&
&   *dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(1,8,15)=gm(3,3)*(6*gm(1,2)*gm(3,3)*dgm(3,3)+gm(2,3)*(6*gm(3,3)&
&   *dgm(1,3)+3*gm(1,3)*dgm(3,3)))
   cm(1,9,15)=12*gm(1,2)*gm(2,3)*gm(3,3)*dgm(3,3)+gm(2,2)*gm(3,3)&
&   *(-3*gm(3,3)*dgm(1,3)-4.5d0*gm(1,3)*dgm(3,3))+gm(2,3)**2*(9*gm(3,3)&
&   *dgm(1,3)+1.5d0*gm(1,3)*dgm(3,3))
   cm(1,10,15)=gm(3,3)**2*(2*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(1,1,16)=gm(1,2)*(2.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(2,2)
   cm(1,2,16)=3*gm(1,2)*gm(2,2)**2*dgm(2,2)
   cm(1,3,16)=((-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(2,2))/12.d0
   cm(1,4,16)=(gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm(2,2))&
&   /6.d0
   cm(1,5,16)=((-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,2))/12.d0
   cm(1,6,16)=gm(2,2)*(4.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(2,2)
   cm(1,7,16)=gm(2,2)**3*dgm(2,2)
   cm(1,8,16)=gm(2,2)*(4.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(2,2)
   cm(1,9,16)=3*gm(2,2)**2*gm(2,3)*dgm(2,2)
   cm(1,10,16)=gm(2,3)*(2.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(2,2)
   cm(1,1,17)=(1080*gm(1,2)**2*gm(1,3)*dgm(2,3)-216*gm(1,1)*gm(1,3)&
&   *(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))+180*gm(1,2)**3*dgm(3,3)&
&   +gm(1,2)*(540*gm(1,3)**2*dgm(2,2)+gm(1,1)*(-108*gm(3,3)*dgm(2,2)&
&   -432*gm(2,3)*dgm(2,3)-108*gm(2,2)*dgm(3,3))))/72.d0
   cm(1,2,17)=(288*gm(1,3)*gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))&
&   +gm(1,2)*(36*gm(2,3)**2*dgm(2,2)+144*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(-108*gm(3,3)*dgm(2,2)+72*gm(2,2)*dgm(3,3))))/24.d0
   cm(1,3,17)=gm(1,3)*(3*gm(2,3)**2*dgm(2,3)-9*gm(2,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*(3*gm(3,3)*dgm(2,2)-3*gm(2,2)*dgm(3,3)))+gm(1,2)*(6*gm(3,3)&
&   **2*dgm(2,2)+7.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)*(24*gm(2,3)*dgm(2,3)&
&   -1.5d0*gm(2,2)*dgm(3,3)))
   cm(1,4,17)=gm(1,3)*(3*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(12*gm(3,3)*dgm(2,2)-3*gm(2,2)*dgm(3,3)))+gm(1,2)*(6*gm(2,3)&
&   **2*dgm(2,3)+24*gm(2,2)*gm(3,3)*dgm(2,3)+gm(2,3)*(3*gm(3,3)*dgm(2,2)&
&   +9*gm(2,2)*dgm(3,3)))
   cm(1,5,17)=(6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(2,3)+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))&
&   /24.d0
   cm(1,6,17)=7.5d0*gm(1,3)**2*gm(2,2)*dgm(2,2)+gm(1,2)*gm(1,3)*(9*gm(2,3)&
&   *dgm(2,2)+24*gm(2,2)*dgm(2,3))+gm(1,2)**2*(-3*gm(3,3)*dgm(2,2)&
&   +3*gm(2,3)*dgm(2,3)+4.5d0*gm(2,2)*dgm(3,3))+gm(1,1)*(-3*gm(2,3)&
&   **2*dgm(2,2)-9*gm(2,2)*gm(2,3)*dgm(2,3)-1.5d0*gm(2,2)*(gm(3,3)&
&   *dgm(2,2)+gm(2,2)*dgm(3,3)))
   cm(1,7,17)=gm(2,2)*(4.5d0*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(2,3)&
&   *dgm(2,3)+gm(2,2)*(-1.5d0*gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3)))
   cm(1,8,17)=3*gm(2,3)**3*dgm(2,3)+15*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +gm(2,2)*gm(3,3)*(6*gm(3,3)*dgm(2,2)-1.5d0*gm(2,2)*dgm(3,3))&
&   +gm(2,3)**2*(3*gm(3,3)*dgm(2,2)+4.5d0*gm(2,2)*dgm(3,3))
   cm(1,9,17)=1.5d0*gm(2,3)**3*dgm(2,2)+6*gm(2,2)*gm(2,3)**2*dgm(2,3)&
&   +12*gm(2,2)**2*gm(3,3)*dgm(2,3)+gm(2,2)*gm(2,3)*(7.5d0*gm(3,3)&
&   *dgm(2,2)+3*gm(2,2)*dgm(3,3))
   cm(1,10,17)=9*gm(2,3)**2*gm(3,3)*dgm(2,3)-3*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +2.5d0*gm(2,3)**3*dgm(3,3)+gm(2,3)*gm(3,3)*(3*gm(3,3)*dgm(2,2)&
&   -1.5d0*gm(2,2)*dgm(3,3))
   cm(1,1,18)=7.5d0*gm(1,2)**2*gm(1,3)*dgm(2,2)-1.5d0*gm(1,1)*gm(1,3)&
&   *gm(2,2)*dgm(2,2)+5*gm(1,2)**3*dgm(2,3)-3*gm(1,1)*gm(1,2)*(gm(2,3)&
&   *dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(1,2,18)=gm(2,2)*(6*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,2)*(3*gm(2,3)&
&   *dgm(2,2)+6*gm(2,2)*dgm(2,3)))
   cm(1,3,18)=gm(1,3)*(1.5d0*gm(2,3)**2*dgm(2,2)-4.5d0*gm(2,2)*gm(3,3)&
&   *dgm(2,2)-6*gm(2,2)*gm(2,3)*dgm(2,3))+gm(1,2)*(12*gm(2,3)*gm(3,3)&
&   *dgm(2,2)+15*gm(2,3)**2*dgm(2,3)-3*gm(2,2)*gm(3,3)*dgm(2,3))
   cm(1,4,18)=gm(1,3)*gm(2,2)*(3*gm(2,3)*dgm(2,2)-6*gm(2,2)*dgm(2,3))&
&   +gm(1,2)*(3*gm(2,3)**2*dgm(2,2)+12*gm(2,2)*gm(3,3)*dgm(2,2)+18*gm(2,2)&
&   *gm(2,3)*dgm(2,3))
   cm(1,5,18)=(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(2,2)+8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,3))/48.d0
   cm(1,6,18)=12*gm(1,2)*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,1)*gm(2,2)&
&   *(-4.5d0*gm(2,3)*dgm(2,2)-3*gm(2,2)*dgm(2,3))+gm(1,2)**2*(1.5d0*gm(2,3)&
&   *dgm(2,2)+9*gm(2,2)*dgm(2,3))
   cm(1,7,18)=gm(2,2)**2*(3*gm(2,3)*dgm(2,2)+2*gm(2,2)*dgm(2,3))
   cm(1,8,18)=1.5d0*gm(2,3)**3*dgm(2,2)+7.5d0*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm(2,2)+9*gm(2,2)*gm(2,3)**2*dgm(2,3)-3*gm(2,2)**2*gm(3,3)&
&   *dgm(2,3)
   cm(1,9,18)=gm(2,2)*(3*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(3,3)*dgm(2,2)&
&   +6*gm(2,2)*gm(2,3)*dgm(2,3))
   cm(1,10,18)=4.5d0*gm(2,3)**2*gm(3,3)*dgm(2,2)-1.5d0*gm(2,2)*gm(3,3)&
&   **2*dgm(2,2)+5*gm(2,3)**3*dgm(2,3)-3*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm(2,3)
   cm(1,1,19)=(180*gm(1,3)**3*dgm(2,2)+1080*gm(1,2)*gm(1,3)**2*dgm(2,3)&
&   -216*gm(1,1)*gm(1,2)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))+gm(1,3)&
&   *(540*gm(1,2)**2*dgm(3,3)+gm(1,1)*(-108*gm(3,3)*dgm(2,2)-432*gm(2,3)&
&   *dgm(2,3)-108*gm(2,2)*dgm(3,3))))/72.d0
   cm(1,2,19)=(2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(2,2)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm(2,3)+6*gm(2,2)&
&   *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm(3,3))/24.d0
   cm(1,3,19)=12*gm(1,2)*gm(3,3)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))&
&   +gm(1,3)*(3*gm(3,3)**2*dgm(2,2)+1.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)&
&   *(6*gm(2,3)*dgm(2,3)-4.5d0*gm(2,2)*dgm(3,3)))
   cm(1,4,19)=gm(1,3)*(6*gm(2,3)**2*dgm(2,3)+24*gm(2,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*(9*gm(3,3)*dgm(2,2)+3*gm(2,2)*dgm(3,3)))+gm(1,2)*(-3*gm(3,3)&
&   **2*dgm(2,2)+3*gm(2,3)**2*dgm(3,3)+gm(3,3)*(6*gm(2,3)*dgm(2,3)&
&   +12*gm(2,2)*dgm(3,3)))
   cm(1,5,19)=(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(2,2)&
&   +12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm(2,3)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/24.d0
   cm(1,6,19)=gm(1,3)**2*(7.5d0*gm(2,3)*dgm(2,2)+15*gm(2,2)*dgm(2,3))&
&   +gm(1,2)*gm(1,3)*(-3*gm(3,3)*dgm(2,2)+18*gm(2,3)*dgm(2,3)+12*gm(2,2)&
&   *dgm(3,3))+gm(1,2)**2*(-6*gm(3,3)*dgm(2,3)+1.5d0*gm(2,3)*dgm(3,3))&
&   +gm(1,1)*(-1.5d0*gm(2,3)*gm(3,3)*dgm(2,2)-6*gm(2,3)**2*dgm(2,3)&
&   -3*gm(2,2)*gm(3,3)*dgm(2,3)-4.5d0*gm(2,2)*gm(2,3)*dgm(3,3))
   cm(1,7,19)=2.5d0*gm(2,3)**3*dgm(2,2)+9*gm(2,2)*gm(2,3)**2*dgm(2,3)&
&   -3*gm(2,2)**2*gm(3,3)*dgm(2,3)+gm(2,2)*gm(2,3)*(-1.5d0*gm(3,3)&
&   *dgm(2,2)+3*gm(2,2)*dgm(3,3))
   cm(1,8,19)=6*gm(2,3)**2*gm(3,3)*dgm(2,3)+12*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +1.5d0*gm(2,3)**3*dgm(3,3)+gm(2,3)*gm(3,3)*(3*gm(3,3)*dgm(2,2)&
&   +7.5d0*gm(2,2)*dgm(3,3))
   cm(1,9,19)=3*gm(2,3)**3*dgm(2,3)+15*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)**2*(4.5d0*gm(3,3)*dgm(2,2)+3*gm(2,2)*dgm(3,3))+gm(2,2)&
&   *gm(3,3)*(-1.5d0*gm(3,3)*dgm(2,2)+6*gm(2,2)*dgm(3,3))
   cm(1,10,19)=gm(3,3)*(1*gm(3,3)**2*dgm(2,2)+4.5d0*gm(2,3)**2*dgm(3,3)&
&   +gm(3,3)*(6*gm(2,3)*dgm(2,3)-1.5d0*gm(2,2)*dgm(3,3)))
   cm(1,1,20)=5*gm(1,3)**3*dgm(2,3)+7.5d0*gm(1,2)*gm(1,3)**2*dgm(3,3)&
&   -1.5d0*gm(1,1)*gm(1,2)*gm(3,3)*dgm(3,3)-3*gm(1,1)*gm(1,3)*(gm(3,3)&
&   *dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(1,2,20)=(8*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(2,3)+12*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm(3,3))/48.d0
   cm(1,3,20)=gm(3,3)*(6*gm(1,2)*gm(3,3)*dgm(3,3)+gm(1,3)*(6*gm(3,3)&
&   *dgm(2,3)+3*gm(2,3)*dgm(3,3)))
   cm(1,4,20)=gm(1,2)*gm(3,3)*(-6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))&
&   +gm(1,3)*(18*gm(2,3)*gm(3,3)*dgm(2,3)+3*gm(2,3)**2*dgm(3,3)+12*gm(2,2)&
&   *gm(3,3)*dgm(3,3))
   cm(1,5,20)=12*gm(1,2)*gm(1,3)*gm(3,3)*dgm(3,3)+gm(1,1)*gm(3,3)&
&   *(-3*gm(3,3)*dgm(2,3)-4.5d0*gm(2,3)*dgm(3,3))+gm(1,3)**2*(9*gm(3,3)&
&   *dgm(2,3)+1.5d0*gm(2,3)*dgm(3,3))
   cm(1,6,20)=(8*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,3)+12*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(3,3))/48.d0
   cm(1,7,20)=5*gm(2,3)**3*dgm(2,3)-3*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +4.5d0*gm(2,2)*gm(2,3)**2*dgm(3,3)-1.5d0*gm(2,2)**2*gm(3,3)*dgm(3,3)
   cm(1,8,20)=gm(3,3)*(6*gm(2,3)*gm(3,3)*dgm(2,3)+3*gm(2,3)**2*dgm(3,3)&
&   +6*gm(2,2)*gm(3,3)*dgm(3,3))
   cm(1,9,20)=9*gm(2,3)**2*gm(3,3)*dgm(2,3)-3*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +1.5d0*gm(2,3)**3*dgm(3,3)+7.5d0*gm(2,2)*gm(2,3)*gm(3,3)*dgm(3,3)
   cm(1,10,20)=gm(3,3)**2*(2*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(1,1,21)=gm(1,3)*(2.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(3,3)
   cm(1,2,21)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/12.d0
   cm(1,3,21)=3*gm(1,3)*gm(3,3)**2*dgm(3,3)
   cm(1,4,21)=(gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm(3,3))&
&   /6.d0
   cm(1,5,21)=gm(3,3)*(4.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(3,3)
   cm(1,6,21)=((90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(3,3))/12.d0
   cm(1,7,21)=gm(2,3)*(2.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(3,3)
   cm(1,8,21)=3*gm(2,3)*gm(3,3)**2*dgm(3,3)
   cm(1,9,21)=gm(3,3)*(4.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(3,3)
   cm(1,10,21)=gm(3,3)**3*dgm(3,3)
   cm(2,1,2)=gm(1,1)**2*(3*gm(1,2)*dgm(1,1)+2*gm(1,1)*dgm(1,2))
   cm(2,2,2)=1.5d0*gm(1,2)**3*dgm(1,1)+7.5d0*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm(1,1)+9*gm(1,1)*gm(1,2)**2*dgm(1,2)-3*gm(1,1)**2*gm(2,2)&
&   *dgm(1,2)
   cm(2,3,2)=gm(1,2)*(1.5d0*gm(1,3)**2-4.5d0*gm(1,1)*gm(3,3))*dgm(1,1)&
&   +gm(1,1)*(12*gm(1,3)*gm(2,3)*dgm(1,1)+9*gm(1,3)**2*dgm(1,2)-3*gm(1,1)&
&   *gm(3,3)*dgm(1,2))
   cm(2,4,2)=3*gm(1,2)**2*gm(1,3)*dgm(1,1)+gm(1,1)*gm(1,2)*(3*gm(2,3)&
&   *dgm(1,1)+18*gm(1,3)*dgm(1,2))+gm(1,1)*(12*gm(1,3)*gm(2,2)*dgm(1,1)&
&   -6*gm(1,1)*gm(2,3)*dgm(1,2))
   cm(2,5,2)=gm(1,1)*(3*gm(1,2)*gm(1,3)*dgm(1,1)+6*gm(1,1)*(gm(2,3)&
&   *dgm(1,1)+gm(1,3)*dgm(1,2)))
   cm(2,6,2)=gm(1,1)*(3*gm(1,2)**2*dgm(1,1)+6*gm(1,1)*gm(2,2)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,2)*dgm(1,2))
   cm(2,7,2)=4.5d0*gm(1,2)**2*gm(2,2)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,2)&
&   **2*dgm(1,1)+5*gm(1,2)**3*dgm(1,2)-3*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm(1,2)
   cm(2,8,2)=(24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,1)+16*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(1,2))/96.d0
   cm(2,9,2)=gm(1,1)*gm(2,2)*(-4.5d0*gm(2,3)*dgm(1,1)-3*gm(1,3)*dgm(1,2))&
&   +gm(1,2)**2*(1.5d0*gm(2,3)*dgm(1,1)+15*gm(1,3)*dgm(1,2))+gm(1,2)&
&   *(12*gm(1,3)*gm(2,2)*dgm(1,1)-6*gm(1,1)*gm(2,3)*dgm(1,2))
   cm(2,10,2)=7.5d0*gm(1,3)**2*gm(2,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,3)&
&   *gm(3,3)*dgm(1,1)+5*gm(1,3)**3*dgm(1,2)-3*gm(1,3)*gm(3,3)*(gm(1,2)&
&   *dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(2,1,4)=gm(1,1)**2*(3*gm(1,3)*dgm(1,1)+2*gm(1,1)*dgm(1,3))
   cm(2,2,4)=12*gm(1,1)*gm(1,2)*gm(2,3)*dgm(1,1)+gm(1,1)*gm(2,2)&
&   *(-4.5d0*gm(1,3)*dgm(1,1)-3*gm(1,1)*dgm(1,3))+gm(1,2)**2*(1.5d0*gm(1,3)&
&   *dgm(1,1)+9*gm(1,1)*dgm(1,3))
   cm(2,3,4)=1.5d0*gm(1,3)**3*dgm(1,1)+7.5d0*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm(1,1)+9*gm(1,1)*gm(1,3)**2*dgm(1,3)-3*gm(1,1)**2*gm(3,3)&
&   *dgm(1,3)
   cm(2,4,4)=gm(1,1)*gm(2,3)*(3*gm(1,3)*dgm(1,1)-6*gm(1,1)*dgm(1,3))&
&   +gm(1,2)*(3*gm(1,3)**2*dgm(1,1)+12*gm(1,1)*gm(3,3)*dgm(1,1)+18*gm(1,1)&
&   *gm(1,3)*dgm(1,3))
   cm(2,5,4)=gm(1,1)*(3*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,3)*dgm(1,3))
   cm(2,6,4)=gm(1,1)*(6*gm(1,1)*gm(2,3)*dgm(1,1)+gm(1,2)*(3*gm(1,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(1,3)))
   cm(2,7,4)=7.5d0*gm(1,2)**2*gm(2,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,2)&
&   *gm(2,3)*dgm(1,1)+5*gm(1,2)**3*dgm(1,3)-3*gm(1,2)*gm(2,2)*(gm(1,3)&
&   *dgm(1,1)+gm(1,1)*dgm(1,3))
   cm(2,8,4)=gm(1,1)*gm(3,3)*(-4.5d0*gm(2,3)*dgm(1,1)-3*gm(1,2)*dgm(1,3))&
&   +gm(1,3)**2*(1.5d0*gm(2,3)*dgm(1,1)+15*gm(1,2)*dgm(1,3))+gm(1,3)&
&   *(12*gm(1,2)*gm(3,3)*dgm(1,1)-6*gm(1,1)*gm(2,3)*dgm(1,3))
   cm(2,9,4)=(12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,1)+8*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(1,3))/48.d0
   cm(2,10,4)=4.5d0*gm(1,3)**2*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(3,3)&
&   **2*dgm(1,1)+5*gm(1,3)**3*dgm(1,3)-3*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm(1,3)
   cm(2,1,6)=gm(1,1)**3*dgm(1,1)
   cm(2,2,6)=gm(1,1)*(4.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(1,1)
   cm(2,3,6)=gm(1,1)*(4.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(1,1)
   cm(2,4,6)=(gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm(1,1))&
&   /6.d0
   cm(2,5,6)=3*gm(1,1)**2*gm(1,3)*dgm(1,1)
   cm(2,6,6)=3*gm(1,1)**2*gm(1,2)*dgm(1,1)
   cm(2,7,6)=gm(1,2)*(2.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(1,1)
   cm(2,8,6)=((-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm(1,1))/12.d0
   cm(2,9,6)=((90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm(1,1))/12.d0
   cm(2,10,6)=gm(1,3)*(2.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(1,1)
   cm(2,1,7)=gm(1,1)*(4.5d0*gm(1,2)**2*dgm(1,1)+6*gm(1,1)*gm(1,2)&
&   *dgm(1,2)+gm(1,1)*(-1.5d0*gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2)))
   cm(2,2,7)=3*gm(1,2)**3*dgm(1,2)+15*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +gm(1,1)*gm(2,2)*(6*gm(2,2)*dgm(1,1)-1.5d0*gm(1,1)*dgm(2,2))&
&   +gm(1,2)**2*(3*gm(2,2)*dgm(1,1)+4.5d0*gm(1,1)*dgm(2,2))
   cm(2,3,7)=-3*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)*gm(2,3)*(9*gm(1,2)&
&   *dgm(1,1)+24*gm(1,1)*dgm(1,2))+gm(1,1)*(7.5d0*gm(2,3)**2*dgm(1,1)&
&   -1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)-9*gm(1,2)*gm(3,3)*dgm(1,2))-1.5d0*gm(1,1)&
&   **2*gm(3,3)*dgm(2,2)+gm(1,3)**2*(-3*gm(2,2)*dgm(1,1)+3*gm(1,2)&
&   *dgm(1,2)+4.5d0*gm(1,1)*dgm(2,2))
   cm(2,4,7)=gm(1,2)**2*(3*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2))+gm(1,1)&
&   *(gm(2,2)*(12*gm(2,3)*dgm(1,1)+24*gm(1,3)*dgm(1,2))-3*gm(1,1)&
&   *gm(2,3)*dgm(2,2))+gm(1,2)*(6*gm(1,1)*gm(2,3)*dgm(1,2)+gm(1,3)&
&   *(3*gm(2,2)*dgm(1,1)+9*gm(1,1)*dgm(2,2)))
   cm(2,5,7)=1.5d0*gm(1,2)**2*gm(1,3)*dgm(1,1)+gm(1,1)*gm(1,2)*(12*gm(2,3)&
&   *dgm(1,1)+6*gm(1,3)*dgm(1,2))+gm(1,1)*(12*gm(1,1)*gm(2,3)*dgm(1,2)&
&   +gm(1,3)*(-4.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))
   cm(2,6,7)=1.5d0*gm(1,2)**3*dgm(1,1)+6*gm(1,1)*gm(1,2)**2*dgm(1,2)&
&   +12*gm(1,1)**2*gm(2,2)*dgm(1,2)+gm(1,1)*gm(1,2)*(7.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,1)*dgm(2,2))
   cm(2,7,7)=9*gm(1,2)**2*gm(2,2)*dgm(1,2)-3*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +2.5d0*gm(1,2)**3*dgm(2,2)+gm(1,2)*gm(2,2)*(3*gm(2,2)*dgm(1,1)&
&   -1.5d0*gm(1,1)*dgm(2,2))
   cm(2,8,7)=(3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,1)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(2,2))/12.d0
   cm(2,9,7)=gm(2,3)*(3*gm(1,2)**2*dgm(1,2)-9*gm(1,1)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)*(3*gm(2,2)*dgm(1,1)-3*gm(1,1)*dgm(2,2)))+gm(1,3)*(6*gm(2,2)&
&   **2*dgm(1,1)+7.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)*(24*gm(1,2)*dgm(1,2)&
&   -1.5d0*gm(1,1)*dgm(2,2)))
   cm(2,10,7)=(3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,1)+6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)&
&   *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,2)+(90*gm(1,3)&
&   **3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm(2,2))/36.d0
   cm(2,1,8)=gm(1,1)*(4.5d0*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(1,3)&
&   *dgm(1,3)+gm(1,1)*(-1.5d0*gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3)))
   cm(2,2,8)=-3*gm(1,3)**2*gm(2,2)*dgm(1,1)-3*gm(1,2)**2*gm(3,3)&
&   *dgm(1,1)+gm(1,3)*(9*gm(1,2)*gm(2,3)*dgm(1,1)+3*gm(1,2)**2*dgm(1,3)&
&   -9*gm(1,1)*gm(2,2)*dgm(1,3))-1.5d0*gm(1,1)**2*gm(2,2)*dgm(3,3)&
&   +gm(1,1)*(7.5d0*gm(2,3)**2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)&
&   +24*gm(1,2)*gm(2,3)*dgm(1,3)+4.5d0*gm(1,2)**2*dgm(3,3))
   cm(2,3,8)=3*gm(1,3)**3*dgm(1,3)+15*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +gm(1,1)*gm(3,3)*(6*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*dgm(3,3))&
&   +gm(1,3)**2*(3*gm(3,3)*dgm(1,1)+4.5d0*gm(1,1)*dgm(3,3))
   cm(2,4,8)=gm(1,3)**2*(3*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3))+gm(1,1)&
&   *(24*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(12*gm(3,3)*dgm(1,1)-3*gm(1,1)&
&   *dgm(3,3)))+gm(1,3)*(6*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)*(3*gm(3,3)&
&   *dgm(1,1)+9*gm(1,1)*dgm(3,3)))
   cm(2,5,8)=1.5d0*gm(1,3)**3*dgm(1,1)+6*gm(1,1)*gm(1,3)**2*dgm(1,3)&
&   +12*gm(1,1)**2*gm(3,3)*dgm(1,3)+gm(1,1)*gm(1,3)*(7.5d0*gm(3,3)&
&   *dgm(1,1)+3*gm(1,1)*dgm(3,3))
   cm(2,6,8)=12*gm(1,1)*gm(2,3)*(gm(1,3)*dgm(1,1)+gm(1,1)*dgm(1,3))&
&   +gm(1,2)*(1.5d0*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(1,3)*dgm(1,3)&
&   +gm(1,1)*(-4.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3)))
   cm(2,7,8)=(6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,1)+12*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+2*(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm(3,3))/72.d0
   cm(2,8,8)=gm(1,3)**2*(3*gm(2,3)*dgm(1,3)+7.5d0*gm(1,2)*dgm(3,3))&
&   +gm(1,3)*(24*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(3*gm(3,3)*dgm(1,1)&
&   -3*gm(1,1)*dgm(3,3)))+gm(3,3)*(-9*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)&
&   *(6*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*dgm(3,3)))
   cm(2,9,8)=(6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,1)+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(3,3))/24.d0
   cm(2,10,8)=9*gm(1,3)**2*gm(3,3)*dgm(1,3)-3*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +2.5d0*gm(1,3)**3*dgm(3,3)+gm(1,3)*gm(3,3)*(3*gm(3,3)*dgm(1,1)&
&   -1.5d0*gm(1,1)*dgm(3,3))
   cm(2,1,9)=gm(1,1)*(gm(1,2)*(9*gm(1,3)*dgm(1,1)+6*gm(1,1)*dgm(1,3))&
&   +gm(1,1)*(-3*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2)+2*gm(1,1)*dgm(2,3)))
   cm(2,2,9)=3*gm(1,2)**3*dgm(1,3)+gm(1,2)*(3*gm(1,3)*gm(2,2)*dgm(1,1)&
&   +gm(1,1)*(24*gm(2,3)*dgm(1,2)+15*gm(2,2)*dgm(1,3)))+gm(1,1)*gm(2,2)&
&   *(12*gm(2,3)*dgm(1,1)-9*gm(1,3)*dgm(1,2)-3*gm(1,1)*dgm(2,3))&
&   +gm(1,2)**2*(3*gm(2,3)*dgm(1,1)+3*gm(1,3)*dgm(1,2)+9*gm(1,1)&
&   *dgm(2,3))
   cm(2,3,9)=3*gm(1,3)**3*dgm(1,2)+gm(1,3)*(3*gm(1,2)*gm(3,3)*dgm(1,1)&
&   +gm(1,1)*(15*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3)))+gm(1,1)*gm(3,3)&
&   *(12*gm(2,3)*dgm(1,1)-9*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))&
&   +gm(1,3)**2*(3*gm(2,3)*dgm(1,1)+3*gm(1,2)*dgm(1,3)+9*gm(1,1)&
&   *dgm(2,3))
   cm(2,4,9)=9*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)**2*(9*gm(2,2)&
&   *dgm(1,1)+6*gm(1,2)*dgm(1,2))+gm(1,1)*(9*gm(2,3)**2*dgm(1,1)&
&   +15*gm(2,2)*gm(3,3)*dgm(1,1)+24*gm(1,2)*gm(3,3)*dgm(1,2)+6*gm(1,2)&
&   *gm(2,3)*dgm(1,3))-6*gm(1,1)**2*gm(2,3)*dgm(2,3)+gm(1,3)*(6*gm(1,2)&
&   **2*dgm(1,3)+gm(1,1)*(6*gm(2,3)*dgm(1,2)+24*gm(2,2)*dgm(1,3))&
&   +gm(1,2)*(-6*gm(2,3)*dgm(1,1)+18*gm(1,1)*dgm(2,3)))
   cm(2,5,9)=gm(1,2)*(3*gm(1,3)**2*dgm(1,1)+12*gm(1,1)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,3)*dgm(1,3))+gm(1,1)*(6*gm(1,3)**2*dgm(1,2)+12*gm(1,1)&
&   *(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)*(3*gm(2,3)*dgm(1,1)&
&   +6*gm(1,1)*dgm(2,3)))
   cm(2,6,9)=gm(1,2)**2*(3*gm(1,3)*dgm(1,1)+6*gm(1,1)*dgm(1,3))+12*gm(1,1)&
&   *(gm(1,3)*gm(2,2)*dgm(1,1)+gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)&
&   *dgm(1,3)))+gm(1,1)*gm(1,2)*(3*gm(2,3)*dgm(1,1)+6*(gm(1,3)*dgm(1,2)&
&   +gm(1,1)*dgm(2,3)))
   cm(2,7,9)=gm(1,3)*gm(2,2)*(-3*gm(2,2)*dgm(1,1)-6*gm(1,2)*dgm(1,2))&
&   -3*gm(1,1)*gm(2,2)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)&
&   **2*(15*gm(2,3)*dgm(1,2)+9*gm(2,2)*dgm(1,3))+5*gm(1,2)**3*dgm(2,3)&
&   +gm(1,2)*gm(2,2)*(9*gm(2,3)*dgm(1,1)-3*gm(1,1)*dgm(2,3))
   cm(2,8,9)=-6*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(-9*gm(2,3)*gm(3,3)&
&   *dgm(1,2)-6*gm(2,3)**2*dgm(1,3)-3*gm(2,2)*gm(3,3)*dgm(1,3))+gm(1,2)&
&   *gm(3,3)*(3*gm(2,3)*dgm(1,1)-3*gm(1,1)*dgm(2,3))+gm(1,3)*(3*gm(2,3)&
&   **2*dgm(1,1)+gm(3,3)*(12*gm(2,2)*dgm(1,1)+24*gm(1,2)*dgm(1,2))&
&   +gm(2,3)*(18*gm(1,2)*dgm(1,3)-6*gm(1,1)*dgm(2,3)))+gm(1,3)**2*(3*gm(2,3)&
&   *dgm(1,2)+15*(gm(2,2)*dgm(1,3)+gm(1,2)*dgm(2,3)))
   cm(2,9,9)=-6*gm(1,3)**2*gm(2,2)*dgm(1,2)+gm(1,2)**2*(15*gm(3,3)&
&   *dgm(1,2)+3*gm(2,3)*dgm(1,3))+gm(1,1)*(-6*gm(2,3)**2*dgm(1,2)&
&   -3*gm(2,2)*gm(3,3)*dgm(1,2)-9*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,2)&
&   *(3*gm(2,3)**2*dgm(1,1)+12*gm(2,2)*gm(3,3)*dgm(1,1)-6*gm(1,1)&
&   *gm(2,3)*dgm(2,3))+gm(1,3)*(gm(2,2)*(3*gm(2,3)*dgm(1,1)+24*gm(1,2)&
&   *dgm(1,3)-3*gm(1,1)*dgm(2,3))+gm(1,2)*(18*gm(2,3)*dgm(1,2)+15*gm(1,2)&
&   *dgm(2,3)))
   cm(2,10,9)=gm(1,3)**2*(9*gm(3,3)*dgm(1,2)+15*gm(2,3)*dgm(1,3))&
&   -3*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(1,1)+gm(1,1)*(gm(3,3)*dgm(1,2)&
&   +gm(2,3)*dgm(1,3)))+5*gm(1,3)**3*dgm(2,3)+gm(1,3)*gm(3,3)*(9*gm(2,3)&
&   *dgm(1,1)-6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))
   cm(2,1,11)=2.5d0*gm(1,2)**3*dgm(1,1)+9*gm(1,1)*gm(1,2)**2*dgm(1,2)&
&   -3*gm(1,1)**2*gm(2,2)*dgm(1,2)+gm(1,1)*gm(1,2)*(-1.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,1)*dgm(2,2))
   cm(2,2,11)=6*gm(1,2)**2*gm(2,2)*dgm(1,2)+12*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +1.5d0*gm(1,2)**3*dgm(2,2)+gm(1,2)*gm(2,2)*(3*gm(2,2)*dgm(1,1)&
&   +7.5d0*gm(1,1)*dgm(2,2))
   cm(2,3,11)=-6*gm(1,2)**2*gm(3,3)*dgm(1,2)+gm(1,1)*(15*gm(2,3)&
&   **2-3*gm(2,2)*gm(3,3))*dgm(1,2)+gm(1,3)*gm(2,3)*(-3*gm(2,2)*dgm(1,1)&
&   +18*gm(1,2)*dgm(1,2)+12*gm(1,1)*dgm(2,2))+gm(1,3)**2*(-6*gm(2,2)&
&   *dgm(1,2)+1.5d0*gm(1,2)*dgm(2,2))+gm(1,2)*(7.5d0*gm(2,3)**2*dgm(1,1)&
&   -1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)-4.5d0*gm(1,1)*gm(3,3)*dgm(2,2))
   cm(2,4,11)=gm(2,3)*(6*gm(1,2)**2*dgm(1,2)+24*gm(1,1)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)*(9*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))+gm(1,3)*(-3*gm(2,2)&
&   **2*dgm(1,1)+3*gm(1,2)**2*dgm(2,2)+gm(2,2)*(6*gm(1,2)*dgm(1,2)&
&   +12*gm(1,1)*dgm(2,2)))
   cm(2,5,11)=gm(1,2)**2*(7.5d0*gm(2,3)*dgm(1,1)+3*gm(1,3)*dgm(1,2))&
&   +gm(1,1)*(gm(2,2)*(-1.5d0*gm(2,3)*dgm(1,1)-9*gm(1,3)*dgm(1,2))&
&   +6*gm(1,1)*gm(2,3)*dgm(2,2))+gm(1,2)*(24*gm(1,1)*gm(2,3)*dgm(1,2)&
&   +gm(1,3)*(-3*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))
   cm(2,6,11)=3*gm(1,2)**3*dgm(1,2)+15*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)**2*(4.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2))+gm(1,1)&
&   *gm(2,2)*(-1.5d0*gm(2,2)*dgm(1,1)+6*gm(1,1)*dgm(2,2))
   cm(2,7,11)=gm(2,2)*(1*gm(2,2)**2*dgm(1,1)+4.5d0*gm(1,2)**2*dgm(2,2)&
&   +gm(2,2)*(6*gm(1,2)*dgm(1,2)-1.5d0*gm(1,1)*dgm(2,2)))
   cm(2,8,11)=(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,1)&
&   +6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+3*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,2))/12.d0
   cm(2,9,11)=gm(2,2)**2*(3*gm(2,3)*dgm(1,1)+12*gm(1,3)*dgm(1,2))&
&   +1.5d0*gm(1,2)**2*gm(2,3)*dgm(2,2)+gm(2,2)*(-4.5d0*gm(1,1)*gm(2,3)&
&   *dgm(2,2)+gm(1,2)*(6*gm(2,3)*dgm(1,2)+12*gm(1,3)*dgm(2,2)))
   cm(2,10,11)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm(1,1)&
&   +6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&   *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2))/36.d0
   cm(2,1,12)=gm(1,1)*(9*gm(1,3)**2*dgm(1,2)+gm(1,1)*(-3*gm(3,3)&
&   *dgm(1,2)-6*gm(2,3)*dgm(1,3))+gm(1,3)*(-3*gm(2,3)*dgm(1,1)+6*gm(1,1)&
&   *dgm(2,3)))+gm(1,2)*(7.5d0*gm(1,3)**2*dgm(1,1)+18*gm(1,1)*gm(1,3)&
&   *dgm(1,3)+gm(1,1)*(-1.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3)))
   cm(2,2,12)=-6*gm(1,3)**2*gm(2,2)*dgm(1,2)+gm(1,2)**2*(-6*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3))+gm(1,1)*(15*gm(2,3)**2*dgm(1,2)&
&   -3*gm(2,2)*gm(3,3)*dgm(1,2)+24*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,3)&
&   *(gm(2,2)*(12*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3)-9*gm(1,1)*dgm(2,3))&
&   +gm(1,2)*(18*gm(2,3)*dgm(1,2)+3*gm(1,2)*dgm(2,3)))+1.5d0*gm(1,2)&
&   **3*dgm(3,3)+gm(1,2)*(1.5d0*gm(2,3)**2*dgm(1,1)-4.5d0*gm(2,2)&
&   *gm(3,3)*dgm(1,1)+24*gm(1,1)*gm(2,3)*dgm(2,3)+7.5d0*gm(1,1)*gm(2,2)&
&   *dgm(3,3))
   cm(2,3,12)=3*gm(1,3)**3*dgm(2,3)+gm(1,3)**2*(6*gm(3,3)*dgm(1,2)&
&   +6*gm(2,3)*dgm(1,3)+1.5d0*gm(1,2)*dgm(3,3))+gm(3,3)*(gm(1,1)&
&   *(12*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3))+gm(1,2)*(6*gm(3,3)&
&   *dgm(1,1)-4.5d0*gm(1,1)*dgm(3,3)))+gm(1,3)*(gm(3,3)*(6*gm(1,2)&
&   *dgm(1,3)+15*gm(1,1)*dgm(2,3))+gm(2,3)*(3*gm(3,3)*dgm(1,1)+12*gm(1,1)&
&   *dgm(3,3)))
   cm(2,4,12)=18*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(24*gm(2,3)&
&   *gm(3,3)*dgm(1,2)+18*gm(2,3)**2*dgm(1,3)+30*gm(2,2)*gm(3,3)*dgm(1,3))&
&   +gm(1,3)**2*(6*gm(2,3)*dgm(1,2)+18*gm(2,2)*dgm(1,3)+6*gm(1,2)&
&   *dgm(2,3))+gm(1,2)*(24*gm(1,1)*gm(3,3)*dgm(2,3)+3*gm(2,3)*(gm(3,3)&
&   *dgm(1,1)+gm(1,1)*dgm(3,3)))+gm(1,3)*(3*gm(2,3)**2*dgm(1,1)+gm(2,3)&
&   *(-12*gm(1,2)*dgm(1,3)+6*gm(1,1)*dgm(2,3))+12*gm(2,2)*(gm(3,3)&
&   *dgm(1,1)+gm(1,1)*dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(1,2)+3*gm(1,2)&
&   *dgm(3,3)))
   cm(2,5,12)=3*gm(1,3)**3*dgm(1,2)+gm(1,3)**2*(1.5d0*gm(2,3)*dgm(1,1)&
&   +6*(gm(1,2)*dgm(1,3)+gm(1,1)*dgm(2,3)))+gm(1,3)*(gm(1,1)*(15*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3))+gm(1,2)*(12*gm(3,3)*dgm(1,1)+3*gm(1,1)&
&   *dgm(3,3)))+gm(1,1)*(gm(3,3)*(24*gm(1,2)*dgm(1,3)+12*gm(1,1)&
&   *dgm(2,3))+gm(2,3)*(-4.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3)))
   cm(2,6,12)=-3*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)**2*(7.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,2)*dgm(1,2))+gm(1,3)*(6*gm(1,2)**2*dgm(1,3)&
&   +24*gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)*(9*gm(2,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(2,3)))+gm(1,1)*(-3*gm(2,3)**2*dgm(1,1)&
&   -1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)-9*gm(1,2)*gm(3,3)*dgm(1,2)+6*gm(1,2)&
&   *gm(2,3)*dgm(1,3)+3*gm(1,2)**2*dgm(3,3))+gm(1,1)**2*(12*gm(2,3)&
&   *dgm(2,3)+6*gm(2,2)*dgm(3,3))
   cm(2,7,12)=15*gm(1,2)*gm(2,3)*(gm(2,3)*dgm(1,2)+gm(1,2)*dgm(2,3))&
&   +gm(2,2)**2*(-1.5d0*gm(3,3)*dgm(1,1)-6*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)&
&   *dgm(3,3))+gm(2,2)*(4.5d0*gm(2,3)**2*dgm(1,1)+gm(2,3)*(-6*gm(1,3)&
&   *dgm(1,2)+18*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))+gm(1,2)*(-3*gm(3,3)&
&   *dgm(1,2)-6*gm(1,3)*dgm(2,3)+4.5d0*gm(1,2)*dgm(3,3)))
   cm(2,8,12)=gm(2,3)**2*(3*gm(3,3)*dgm(1,1)+6*gm(1,3)*dgm(1,3)-3*gm(1,1)&
&   *dgm(3,3))+gm(1,2)*gm(3,3)*(12*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)&
&   -3*gm(1,2)*dgm(3,3))+gm(2,2)*(6*gm(3,3)**2*dgm(1,1)+24*gm(1,3)&
&   *gm(3,3)*dgm(1,3)+7.5d0*gm(1,3)**2*dgm(3,3)-1.5d0*gm(1,1)*gm(3,3)&
&   *dgm(3,3))+gm(2,3)*(3*gm(1,3)**2*dgm(2,3)+gm(3,3)*(6*gm(1,2)&
&   *dgm(1,3)-9*gm(1,1)*dgm(2,3))+gm(1,3)*(6*gm(3,3)*dgm(1,2)+9*gm(1,2)&
&   *dgm(3,3)))
   cm(2,9,12)=1.5d0*gm(2,3)**3*dgm(1,1)-6*gm(1,3)**2*gm(2,2)*dgm(2,3)&
&   +gm(2,3)**2*(3*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)-6*gm(1,1)&
&   *dgm(2,3))+gm(3,3)*(24*gm(1,2)*gm(2,2)*dgm(1,3)+15*gm(1,2)**2*dgm(2,3)&
&   -3*gm(1,1)*gm(2,2)*dgm(2,3))+gm(1,3)*gm(2,2)*(-9*gm(3,3)*dgm(1,2)&
&   +12*gm(1,2)*dgm(3,3))+gm(2,3)*(gm(2,2)*(7.5d0*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*dgm(1,3)-4.5d0*gm(1,1)*dgm(3,3))+gm(1,2)*(24*gm(3,3)&
&   *dgm(1,2)+18*gm(1,3)*dgm(2,3)+1.5d0*gm(1,2)*dgm(3,3)))
   cm(2,10,12)=gm(2,3)*(3*gm(3,3)**2*dgm(1,1)+7.5d0*gm(1,3)**2*dgm(3,3)&
&   +gm(3,3)*(18*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)*dgm(3,3)))+gm(3,3)&
&   *(9*gm(1,3)**2*dgm(2,3)+gm(3,3)*(-6*gm(1,2)*dgm(1,3)-3*gm(1,1)&
&   *dgm(2,3))+gm(1,3)*(6*gm(3,3)*dgm(1,2)-3*gm(1,2)*dgm(3,3)))
   cm(2,1,13)=gm(1,2)**2*(7.5d0*gm(1,3)*dgm(1,1)+9*gm(1,1)*dgm(1,3))&
&   +gm(1,1)*(gm(1,1)*(-6*gm(2,3)*dgm(1,2)-3*gm(2,2)*dgm(1,3))+gm(1,3)&
&   *(-1.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))+gm(1,1)*gm(1,2)&
&   *(-3*gm(2,3)*dgm(1,1)+18*gm(1,3)*dgm(1,2)+6*gm(1,1)*dgm(2,3))
   cm(2,2,13)=6*gm(1,2)**2*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,1)&
&   *gm(2,2)*(24*gm(2,3)*dgm(1,2)+12*gm(2,2)*dgm(1,3))+gm(1,3)*(6*gm(2,2)&
&   **2*dgm(1,1)+1.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)*(6*gm(1,2)*dgm(1,2)&
&   -4.5d0*gm(1,1)*dgm(2,2)))+3*gm(1,2)**3*dgm(2,3)+gm(1,2)*(3*gm(2,2)&
&   *gm(2,3)*dgm(1,1)+12*gm(1,1)*gm(2,3)*dgm(2,2)+15*gm(1,1)*gm(2,2)&
&   *dgm(2,3))
   cm(2,3,13)=-6*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(24*gm(2,3)&
&   *gm(3,3)*dgm(1,2)+15*gm(2,3)**2*dgm(1,3)-3*gm(2,2)*gm(3,3)*dgm(1,3))&
&   +1.5d0*gm(1,3)**3*dgm(2,2)+gm(1,2)*gm(3,3)*(12*gm(2,3)*dgm(1,1)&
&   -9*gm(1,1)*dgm(2,3))+gm(1,3)**2*(6*gm(2,3)*dgm(1,2)-6*gm(2,2)&
&   *dgm(1,3)+3*gm(1,2)*dgm(2,3))+gm(1,3)*(1.5d0*gm(2,3)**2*dgm(1,1)&
&   +gm(3,3)*(-4.5d0*gm(2,2)*dgm(1,1)+6*gm(1,2)*dgm(1,2)+7.5d0*gm(1,1)&
&   *dgm(2,2))+gm(2,3)*(18*gm(1,2)*dgm(1,3)+24*gm(1,1)*dgm(2,3)))
   cm(2,4,13)=gm(1,2)**2*(18*gm(3,3)*dgm(1,2)+6*gm(2,3)*dgm(1,3))&
&   +gm(1,1)*(18*gm(2,3)**2*dgm(1,2)+30*gm(2,2)*gm(3,3)*dgm(1,2)&
&   +24*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,3)**2*(18*gm(2,2)*dgm(1,2)&
&   +3*gm(1,2)*dgm(2,2))+gm(1,2)*(3*gm(2,3)**2*dgm(1,1)+12*gm(3,3)&
&   *(gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))+6*gm(1,1)*gm(2,3)*dgm(2,3))&
&   +gm(1,3)*(-12*gm(1,2)*gm(2,3)*dgm(1,2)+3*gm(1,1)*gm(2,3)*dgm(2,2)&
&   +6*gm(1,2)**2*dgm(2,3)+gm(2,2)*(3*gm(2,3)*dgm(1,1)+6*gm(1,2)&
&   *dgm(1,3)+24*gm(1,1)*dgm(2,3)))
   cm(2,5,13)=7.5d0*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,1)*(-3*gm(2,3)&
&   **2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)+24*gm(1,2)*gm(3,3)&
&   *dgm(1,2)+24*gm(1,2)*gm(2,3)*dgm(1,3))+gm(1,3)**2*(-3*gm(2,2)&
&   *dgm(1,1)+6*gm(1,2)*dgm(1,2)+3*gm(1,1)*dgm(2,2))+gm(1,1)**2*(6*gm(3,3)&
&   *dgm(2,2)+12*gm(2,3)*dgm(2,3))+gm(1,3)*(3*gm(1,2)**2*dgm(1,3)&
&   +gm(1,1)*(6*gm(2,3)*dgm(1,2)-9*gm(2,2)*dgm(1,3))+gm(1,2)*(9*gm(2,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(2,3)))
   cm(2,6,13)=3*gm(1,2)**3*dgm(1,3)+gm(1,2)*(gm(1,1)*(6*gm(2,3)*dgm(1,2)&
&   +15*gm(2,2)*dgm(1,3))+gm(1,3)*(12*gm(2,2)*dgm(1,1)+3*gm(1,1)&
&   *dgm(2,2)))+gm(1,2)**2*(1.5d0*gm(2,3)*dgm(1,1)+6*(gm(1,3)*dgm(1,2)&
&   +gm(1,1)*dgm(2,3)))+gm(1,1)*(6*gm(1,1)*gm(2,3)*dgm(2,2)+gm(2,2)&
&   *(-4.5d0*gm(2,3)*dgm(1,1)+24*gm(1,3)*dgm(1,2)+12*gm(1,1)*dgm(2,3)))
   cm(2,7,13)=7.5d0*gm(1,2)**2*gm(2,3)*dgm(2,2)+gm(2,2)**2*(3*gm(2,3)&
&   *dgm(1,1)-6*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))&
&   +gm(2,2)*(-1.5d0*gm(1,1)*gm(2,3)*dgm(2,2)+gm(1,2)*(18*gm(2,3)&
&   *dgm(1,2)-3*gm(1,3)*dgm(2,2))+9*gm(1,2)**2*dgm(2,3))
   cm(2,8,13)=1.5d0*gm(2,3)**3*dgm(1,1)+gm(1,3)*gm(3,3)*(24*gm(2,2)&
&   *dgm(1,2)+12*gm(1,2)*dgm(2,2))+15*gm(1,3)**2*gm(2,2)*dgm(2,3)&
&   +gm(2,3)**2*(6*gm(1,3)*dgm(1,2)+3*gm(1,2)*dgm(1,3)-6*gm(1,1)&
&   *dgm(2,3))+gm(3,3)*(-9*gm(1,2)*gm(2,2)*dgm(1,3)-6*gm(1,2)**2*dgm(2,3)&
&   -3*gm(1,1)*gm(2,2)*dgm(2,3))+gm(2,3)*(gm(2,2)*(7.5d0*gm(3,3)&
&   *dgm(1,1)+24*gm(1,3)*dgm(1,3))+(1.5d0*gm(1,3)**2-4.5d0*gm(1,1)&
&   *gm(3,3))*dgm(2,2)+gm(1,2)*(6*gm(3,3)*dgm(1,2)+18*gm(1,3)*dgm(2,3)))
   cm(2,9,13)=gm(2,2)**2*(6*gm(3,3)*dgm(1,1)+12*gm(1,3)*dgm(1,3))&
&   -3*gm(1,1)*gm(2,3)**2*dgm(2,2)+gm(1,2)*gm(2,3)*(6*gm(2,3)*dgm(1,2)&
&   +9*gm(1,3)*dgm(2,2))+gm(1,2)**2*(7.5d0*gm(3,3)*dgm(2,2)+3*gm(2,3)&
&   *dgm(2,3))+gm(2,2)*(3*gm(2,3)**2*dgm(1,1)+(-3*gm(1,3)**2-1.5d0*gm(1,1)&
&   *gm(3,3))*dgm(2,2)+gm(2,3)*(6*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)&
&   -9*gm(1,1)*dgm(2,3))+24*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3)))
   cm(2,10,13)=gm(2,3)**2*(4.5d0*gm(3,3)*dgm(1,1)+15*gm(1,3)*dgm(1,3))&
&   +gm(2,3)*(18*gm(1,3)*gm(3,3)*dgm(1,2)+15*gm(1,3)**2*dgm(2,3)&
&   +gm(3,3)*(-6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3)))+gm(3,3)*(gm(2,2)&
&   *(-1.5d0*gm(3,3)*dgm(1,1)-3*gm(1,3)*dgm(1,3))+(4.5d0*gm(1,3)&
&   **2-1.5d0*gm(1,1)*gm(3,3))*dgm(2,2)-6*gm(1,2)*(gm(3,3)*dgm(1,2)&
&   +gm(1,3)*dgm(2,3)))
   cm(2,1,14)=2.5d0*gm(1,3)**3*dgm(1,1)+9*gm(1,1)*gm(1,3)**2*dgm(1,3)&
&   -3*gm(1,1)**2*gm(3,3)*dgm(1,3)+gm(1,1)*gm(1,3)*(-1.5d0*gm(3,3)&
&   *dgm(1,1)+3*gm(1,1)*dgm(3,3))
   cm(2,2,14)=-6*gm(1,3)**2*gm(2,2)*dgm(1,3)-6*gm(1,2)**2*gm(3,3)&
&   *dgm(1,3)+gm(1,1)*(15*gm(2,3)**2-3*gm(2,2)*gm(3,3))*dgm(1,3)&
&   +gm(1,2)*gm(2,3)*(-3*gm(3,3)*dgm(1,1)+12*gm(1,1)*dgm(3,3))+gm(1,3)&
&   *(7.5d0*gm(2,3)**2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)+18*gm(1,2)&
&   *gm(2,3)*dgm(1,3)+1.5d0*gm(1,2)**2*dgm(3,3)-4.5d0*gm(1,1)*gm(2,2)&
&   *dgm(3,3))
   cm(2,3,14)=6*gm(1,3)**2*gm(3,3)*dgm(1,3)+12*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +1.5d0*gm(1,3)**3*dgm(3,3)+gm(1,3)*gm(3,3)*(3*gm(3,3)*dgm(1,1)&
&   +7.5d0*gm(1,1)*dgm(3,3))
   cm(2,4,14)=gm(1,3)**2*(6*gm(2,3)*dgm(1,3)+3*gm(1,2)*dgm(3,3))&
&   +gm(1,3)*(6*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(9*gm(3,3)*dgm(1,1)&
&   +3*gm(1,1)*dgm(3,3)))+gm(3,3)*(24*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)&
&   *(-3*gm(3,3)*dgm(1,1)+12*gm(1,1)*dgm(3,3)))
   cm(2,5,14)=3*gm(1,3)**3*dgm(1,3)+15*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +gm(1,3)**2*(4.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3))+gm(1,1)&
&   *gm(3,3)*(-1.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3))
   cm(2,6,14)=gm(1,3)**2*(7.5d0*gm(2,3)*dgm(1,1)+3*gm(1,2)*dgm(1,3))&
&   +gm(1,3)*(24*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)*(-3*gm(3,3)*dgm(1,1)&
&   +3*gm(1,1)*dgm(3,3)))+gm(1,1)*(-9*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)&
&   *(-1.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3)))
   cm(2,7,14)=(180*gm(2,3)**3*dgm(1,1)+1080*gm(1,2)*gm(2,3)**2*dgm(1,3)&
&   -216*gm(1,2)*gm(2,2)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))+gm(2,3)&
&   *(540*gm(1,2)**2*dgm(3,3)+gm(2,2)*(-108*gm(3,3)*dgm(1,1)-432*gm(1,3)&
&   *dgm(1,3)-108*gm(1,1)*dgm(3,3))))/72.d0
   cm(2,8,14)=12*gm(1,2)*gm(3,3)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))&
&   +gm(2,3)*(3*gm(3,3)**2*dgm(1,1)+1.5d0*gm(1,3)**2*dgm(3,3)+gm(3,3)&
&   *(6*gm(1,3)*dgm(1,3)-4.5d0*gm(1,1)*dgm(3,3)))
   cm(2,9,14)=(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,1)&
&   +12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/24.d0
   cm(2,10,14)=gm(3,3)*(1*gm(3,3)**2*dgm(1,1)+4.5d0*gm(1,3)**2*dgm(3,3)&
&   +gm(3,3)*(6*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)*dgm(3,3)))
   cm(2,1,16)=5*gm(1,2)**3*dgm(1,2)-3*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +4.5d0*gm(1,1)*gm(1,2)**2*dgm(2,2)-1.5d0*gm(1,1)**2*gm(2,2)*dgm(2,2)
   cm(2,2,16)=gm(2,2)*(6*gm(1,2)*gm(2,2)*dgm(1,2)+3*gm(1,2)**2*dgm(2,2)&
&   +6*gm(1,1)*gm(2,2)*dgm(2,2))
   cm(2,3,16)=(40*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,2)+60*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(2,2))/240.d0
   cm(2,4,16)=gm(1,3)*gm(2,2)*(-6*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))&
&   +gm(2,3)*(18*gm(1,2)*gm(2,2)*dgm(1,2)+3*gm(1,2)**2*dgm(2,2)+12*gm(1,1)&
&   *gm(2,2)*dgm(2,2))
   cm(2,5,16)=(40*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,2)+60*(6*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm(2,2))&
&   /240.d0
   cm(2,6,16)=9*gm(1,2)**2*gm(2,2)*dgm(1,2)-3*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +1.5d0*gm(1,2)**3*dgm(2,2)+7.5d0*gm(1,1)*gm(1,2)*gm(2,2)*dgm(2,2)
   cm(2,7,16)=gm(2,2)**2*(2*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))
   cm(2,8,16)=-3*gm(2,2)**2*gm(3,3)*dgm(1,2)+1.5d0*gm(1,2)*gm(2,3)&
&   **2*dgm(2,2)+gm(2,2)*(9*gm(2,3)**2*dgm(1,2)+12*gm(1,3)*gm(2,3)&
&   *dgm(2,2)-4.5d0*gm(1,2)*gm(3,3)*dgm(2,2))
   cm(2,9,16)=gm(2,2)*(3*gm(1,2)*gm(2,3)*dgm(2,2)+6*gm(2,2)*(gm(2,3)&
&   *dgm(1,2)+gm(1,3)*dgm(2,2)))
   cm(2,10,16)=5*gm(2,3)**3*dgm(1,2)+7.5d0*gm(1,3)*gm(2,3)**2*dgm(2,2)&
&   -1.5d0*gm(1,3)*gm(2,2)*gm(3,3)*dgm(2,2)-3*gm(2,3)*gm(3,3)*(gm(2,2)&
&   *dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(2,1,17)=(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(1,2)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(1,3)+gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(2,2)+4*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm(2,3)+gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm(3,3))/12.d0
   cm(2,2,17)=(2*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+2*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm(1,3)+(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,3)+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm(3,3))/4.d0
   cm(2,3,17)=gm(1,2)*gm(3,3)*(12*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3))&
&   -3*gm(1,2)**2*gm(3,3)*dgm(3,3)+gm(1,3)**2*(3*gm(3,3)*dgm(2,2)&
&   +6*gm(2,3)*dgm(2,3)-3*gm(2,2)*dgm(3,3))+gm(1,1)*(6*gm(3,3)**2*dgm(2,2)&
&   +24*gm(2,3)*gm(3,3)*dgm(2,3)+7.5d0*gm(2,3)**2*dgm(3,3)-1.5d0*gm(2,2)&
&   *gm(3,3)*dgm(3,3))+gm(1,3)*(3*gm(2,3)**2*dgm(1,3)+gm(3,3)*(-9*gm(2,2)&
&   *dgm(1,3)+6*gm(1,2)*dgm(2,3))+gm(2,3)*(6*gm(3,3)*dgm(1,2)+9*gm(1,2)&
&   *dgm(3,3)))
   cm(2,4,17)=(2*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+2*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm(1,3)+(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+4*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm(2,3)+(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))&
&   /2.d0
   cm(2,5,17)=(2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,2)+2*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(1,3)+(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm(2,2)+4*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm(2,3)+(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm(3,3))/4.d0
   cm(2,6,17)=(2*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,2)+2*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(2,2)+4*(6*gm(1,2)&
&   **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm(3,3))&
&   /4.d0
   cm(2,7,17)=7.5d0*gm(1,2)*gm(2,3)**2*dgm(2,2)+gm(2,2)*(9*gm(2,3)&
&   **2*dgm(1,2)-1.5d0*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)*(-3*gm(1,3)&
&   *dgm(2,2)+18*gm(1,2)*dgm(2,3)))+gm(2,2)**2*(-3*gm(3,3)*dgm(1,2)&
&   +6*gm(2,3)*dgm(1,3)-6*gm(1,3)*dgm(2,3)+3*gm(1,2)*dgm(3,3))
   cm(2,8,17)=3*gm(2,3)**3*dgm(1,3)+gm(2,3)**2*(6*gm(3,3)*dgm(1,2)&
&   +6*gm(1,3)*dgm(2,3)+1.5d0*gm(1,2)*dgm(3,3))+gm(3,3)*(6*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(12*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)&
&   -4.5d0*gm(1,2)*dgm(3,3)))+gm(2,3)*(gm(3,3)*(3*gm(1,3)*dgm(2,2)&
&   +6*gm(1,2)*dgm(2,3))+gm(2,2)*(15*gm(3,3)*dgm(1,3)+12*gm(1,3)&
&   *dgm(3,3)))
   cm(2,9,17)=3*gm(2,3)**3*dgm(1,2)+gm(2,3)**2*(6*gm(2,2)*dgm(1,3)&
&   +1.5d0*gm(1,3)*dgm(2,2)+6*gm(1,2)*dgm(2,3))+gm(2,3)*(12*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(15*gm(3,3)*dgm(1,2)+6*gm(1,3)*dgm(2,3)&
&   +3*gm(1,2)*dgm(3,3)))+gm(2,2)*(gm(3,3)*(-4.5d0*gm(1,3)*dgm(2,2)&
&   +24*gm(1,2)*dgm(2,3))+gm(2,2)*(12*gm(3,3)*dgm(1,3)+6*gm(1,3)&
&   *dgm(3,3)))
   cm(2,10,17)=gm(2,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+18*gm(1,3)*dgm(2,3)&
&   -3*gm(1,2)*dgm(3,3))+gm(2,3)**2*(9*gm(3,3)*dgm(1,3)+7.5d0*gm(1,3)&
&   *dgm(3,3))+gm(3,3)*(gm(3,3)*(3*gm(1,3)*dgm(2,2)-6*gm(1,2)*dgm(2,3))&
&   +gm(2,2)*(-3*gm(3,3)*dgm(1,3)-1.5d0*gm(1,3)*dgm(3,3)))
   cm(2,1,18)=5*gm(1,2)**3*dgm(1,3)+gm(1,1)*gm(1,2)*(-6*gm(2,3)*dgm(1,2)&
&   -3*gm(2,2)*dgm(1,3)+9*gm(1,3)*dgm(2,2))+gm(1,2)**2*(15*gm(1,3)&
&   *dgm(1,2)+9*gm(1,1)*dgm(2,3))-3*gm(1,1)*(gm(1,3)*gm(2,2)*dgm(1,2)&
&   +gm(1,1)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3)))
   cm(2,2,18)=6*gm(1,2)*gm(2,2)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))&
&   +gm(1,3)*gm(2,2)*(12*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))+12*gm(1,1)&
&   *gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))+gm(1,2)**2*(3*gm(2,3)&
&   *dgm(2,2)+6*gm(2,2)*dgm(2,3))
   cm(2,3,18)=(3*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+3*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm(2,3))/6.d0
   cm(2,4,18)=gm(1,2)*(6*gm(2,3)**2*dgm(1,2)+24*gm(2,2)*gm(3,3)*dgm(1,2)&
&   +18*gm(2,2)*gm(2,3)*dgm(1,3))+9*gm(1,3)**2*gm(2,2)*dgm(2,2)+gm(1,2)&
&   **2*(9*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))+gm(1,1)*(9*gm(2,3)&
&   **2*dgm(2,2)+15*gm(2,2)*gm(3,3)*dgm(2,2)+24*gm(2,2)*gm(2,3)*dgm(2,3))&
&   +gm(1,3)*(-6*gm(2,2)**2*dgm(1,3)-6*gm(1,2)*gm(2,3)*dgm(2,2)+6*gm(2,2)&
&   *(gm(2,3)*dgm(1,2)+gm(1,2)*dgm(2,3)))
   cm(2,5,18)=(3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,2)+(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+3*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm(2,2)+3*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(2,3))/6.d0
   cm(2,6,18)=gm(1,2)**2*(3*gm(2,3)*dgm(1,2)+9*gm(2,2)*dgm(1,3)+3*gm(1,3)&
&   *dgm(2,2))+gm(1,1)*gm(2,2)*(-9*gm(2,3)*dgm(1,2)-3*gm(2,2)*dgm(1,3)&
&   +12*gm(1,3)*dgm(2,2))+3*gm(1,2)**3*dgm(2,3)+gm(1,2)*(24*gm(1,3)&
&   *gm(2,2)*dgm(1,2)+gm(1,1)*(3*gm(2,3)*dgm(2,2)+15*gm(2,2)*dgm(2,3)))
   cm(2,7,18)=gm(2,2)*(2*gm(2,2)**2*dgm(1,3)+9*gm(1,2)*gm(2,3)*dgm(2,2)&
&   +gm(2,2)*(6*gm(2,3)*dgm(1,2)-3*gm(1,3)*dgm(2,2)+6*gm(1,2)*dgm(2,3)))
   cm(2,8,18)=3*gm(2,3)**3*dgm(1,2)+gm(2,2)*gm(3,3)*(-3*gm(2,2)*dgm(1,3)&
&   +12*gm(1,3)*dgm(2,2)-9*gm(1,2)*dgm(2,3))+gm(2,3)**2*(9*gm(2,2)&
&   *dgm(1,3)+3*(gm(1,3)*dgm(2,2)+gm(1,2)*dgm(2,3)))+gm(2,3)*(3*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(15*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)))
   cm(2,9,18)=3*gm(1,2)*gm(2,3)**2*dgm(2,2)+gm(2,2)**2*(12*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3)+12*gm(1,3)*dgm(2,3))+gm(2,2)*(6*gm(2,3)&
&   **2*dgm(1,2)+12*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)*(3*gm(1,3)*dgm(2,2)&
&   +6*gm(1,2)*dgm(2,3)))
   cm(2,10,18)=5*gm(2,3)**3*dgm(1,3)+gm(2,3)*gm(3,3)*(-3*gm(2,2)&
&   *dgm(1,3)+9*gm(1,3)*dgm(2,2)-6*gm(1,2)*dgm(2,3))+gm(2,3)**2*(9*gm(3,3)&
&   *dgm(1,2)+15*gm(1,3)*dgm(2,3))-3*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(2,2)&
&   +gm(2,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3)))
   cm(2,1,19)=5*gm(1,3)**3*dgm(1,2)+gm(1,3)**2*(15*gm(1,2)*dgm(1,3)&
&   +9*gm(1,1)*dgm(2,3))+gm(1,1)*gm(1,3)*(-3*gm(3,3)*dgm(1,2)-6*gm(2,3)&
&   *dgm(1,3)+9*gm(1,2)*dgm(3,3))-3*gm(1,1)*(gm(1,2)*gm(3,3)*dgm(1,3)&
&   +gm(1,1)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3)))
   cm(2,2,19)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,3)+3*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))/6.d0
   cm(2,3,19)=gm(1,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+6*gm(2,3)*dgm(1,3)&
&   +3*gm(1,2)*dgm(3,3))+gm(1,3)**2*(6*gm(3,3)*dgm(2,3)+3*gm(2,3)&
&   *dgm(3,3))+12*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(1,3)+gm(1,1)*(gm(3,3)&
&   *dgm(2,3)+gm(2,3)*dgm(3,3)))
   cm(2,4,19)=(gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+3*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,3)+3*(9*gm(1,3)**2*gm(2,2)&
&   -6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)&
&   **2+15*gm(2,2)*gm(3,3)))*dgm(3,3))/3.d0
   cm(2,5,19)=(gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm(1,3)+3*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm(2,3)+3*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm(3,3))/6.d0
   cm(2,6,19)=((90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,2)+3*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(2,3)+3*(6*gm(1,2)&
&   **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(3,3))/6.d0
   cm(2,7,19)=5*gm(2,3)**3*dgm(1,2)+gm(2,3)**2*(9*gm(2,2)*dgm(1,3)&
&   +15*gm(1,2)*dgm(2,3))+gm(2,2)*gm(2,3)*(-3*gm(3,3)*dgm(1,2)-6*gm(1,3)&
&   *dgm(2,3)+9*gm(1,2)*dgm(3,3))-3*gm(2,2)*(gm(1,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,2)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3)))
   cm(2,8,19)=gm(2,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+6*gm(1,3)*dgm(2,3)&
&   +3*gm(1,2)*dgm(3,3))+gm(2,3)**2*(6*gm(3,3)*dgm(1,3)+3*gm(1,3)&
&   *dgm(3,3))+12*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(2,3)+gm(2,2)*(gm(3,3)&
&   *dgm(1,3)+gm(1,3)*dgm(3,3)))
   cm(2,9,19)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm(1,3)+3*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm(2,3)&
&   +3*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/6.d0
   cm(2,10,19)=gm(3,3)*(2*gm(3,3)**2*dgm(1,2)+9*gm(1,3)*gm(2,3)*dgm(3,3)&
&   +gm(3,3)*(6*gm(2,3)*dgm(1,3)+6*gm(1,3)*dgm(2,3)-3*gm(1,2)*dgm(3,3)))
   cm(2,1,20)=5*gm(1,3)**3*dgm(1,3)-3*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +4.5d0*gm(1,1)*gm(1,3)**2*dgm(3,3)-1.5d0*gm(1,1)**2*gm(3,3)*dgm(3,3)
   cm(2,2,20)=(8*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(3,3))/48.d0
   cm(2,3,20)=gm(3,3)*(6*gm(1,3)*gm(3,3)*dgm(1,3)+3*gm(1,3)**2*dgm(3,3)&
&   +6*gm(1,1)*gm(3,3)*dgm(3,3))
   cm(2,4,20)=3*gm(1,3)**2*gm(2,3)*dgm(3,3)+gm(1,3)*gm(3,3)*(18*gm(2,3)&
&   *dgm(1,3)+3*gm(1,2)*dgm(3,3))+gm(3,3)*(-6*gm(1,2)*gm(3,3)*dgm(1,3)&
&   +12*gm(1,1)*gm(2,3)*dgm(3,3))
   cm(2,5,20)=9*gm(1,3)**2*gm(3,3)*dgm(1,3)-3*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +1.5d0*gm(1,3)**3*dgm(3,3)+7.5d0*gm(1,1)*gm(1,3)*gm(3,3)*dgm(3,3)
   cm(2,6,20)=(8*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,3)+12*(48*gm(1,1)*gm(1,3)&
&   *gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(3,3))&
&   /48.d0
   cm(2,7,20)=5*gm(2,3)**3*dgm(1,3)+7.5d0*gm(1,2)*gm(2,3)**2*dgm(3,3)&
&   -1.5d0*gm(1,2)*gm(2,2)*gm(3,3)*dgm(3,3)-3*gm(2,2)*gm(2,3)*(gm(3,3)&
&   *dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(2,8,20)=gm(3,3)*(6*gm(1,2)*gm(3,3)*dgm(3,3)+gm(2,3)*(6*gm(3,3)&
&   *dgm(1,3)+3*gm(1,3)*dgm(3,3)))
   cm(2,9,20)=12*gm(1,2)*gm(2,3)*gm(3,3)*dgm(3,3)+gm(2,2)*gm(3,3)&
&   *(-3*gm(3,3)*dgm(1,3)-4.5d0*gm(1,3)*dgm(3,3))+gm(2,3)**2*(9*gm(3,3)&
&   *dgm(1,3)+1.5d0*gm(1,3)*dgm(3,3))
   cm(2,10,20)=gm(3,3)**2*(2*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(2,1,22)=gm(1,2)*(2.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(2,2)
   cm(2,2,22)=3*gm(1,2)*gm(2,2)**2*dgm(2,2)
   cm(2,3,22)=((-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(2,2))/12.d0
   cm(2,4,22)=(gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm(2,2))&
&   /6.d0
   cm(2,5,22)=((-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,2))/12.d0
   cm(2,6,22)=gm(2,2)*(4.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(2,2)
   cm(2,7,22)=gm(2,2)**3*dgm(2,2)
   cm(2,8,22)=gm(2,2)*(4.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(2,2)
   cm(2,9,22)=3*gm(2,2)**2*gm(2,3)*dgm(2,2)
   cm(2,10,22)=gm(2,3)*(2.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(2,2)
   cm(2,1,23)=(3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(2,2)+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(2,3)+(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm(3,3))/36.d0
   cm(2,2,23)=12*gm(1,3)*gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))&
&   +gm(1,2)*(1.5d0*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(-4.5d0*gm(3,3)*dgm(2,2)+3*gm(2,2)*dgm(3,3)))
   cm(2,3,23)=gm(1,3)*(3*gm(2,3)**2*dgm(2,3)-9*gm(2,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*(3*gm(3,3)*dgm(2,2)-3*gm(2,2)*dgm(3,3)))+gm(1,2)*(6*gm(3,3)&
&   **2*dgm(2,2)+7.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)*(24*gm(2,3)*dgm(2,3)&
&   -1.5d0*gm(2,2)*dgm(3,3)))
   cm(2,4,23)=gm(1,3)*(3*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(12*gm(3,3)*dgm(2,2)-3*gm(2,2)*dgm(3,3)))+gm(1,2)*(6*gm(2,3)&
&   **2*dgm(2,3)+24*gm(2,2)*gm(3,3)*dgm(2,3)+gm(2,3)*(3*gm(3,3)*dgm(2,2)&
&   +9*gm(2,2)*dgm(3,3)))
   cm(2,5,23)=(3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(2,3)+(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))&
&   /12.d0
   cm(2,6,23)=7.5d0*gm(1,3)**2*gm(2,2)*dgm(2,2)+gm(1,2)*gm(1,3)*(9*gm(2,3)&
&   *dgm(2,2)+24*gm(2,2)*dgm(2,3))+gm(1,2)**2*(-3*gm(3,3)*dgm(2,2)&
&   +3*gm(2,3)*dgm(2,3)+4.5d0*gm(2,2)*dgm(3,3))+gm(1,1)*(-3*gm(2,3)&
&   **2*dgm(2,2)-9*gm(2,2)*gm(2,3)*dgm(2,3)-1.5d0*gm(2,2)*(gm(3,3)&
&   *dgm(2,2)+gm(2,2)*dgm(3,3)))
   cm(2,7,23)=gm(2,2)*(4.5d0*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(2,3)&
&   *dgm(2,3)+gm(2,2)*(-1.5d0*gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3)))
   cm(2,8,23)=3*gm(2,3)**3*dgm(2,3)+15*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +gm(2,2)*gm(3,3)*(6*gm(3,3)*dgm(2,2)-1.5d0*gm(2,2)*dgm(3,3))&
&   +gm(2,3)**2*(3*gm(3,3)*dgm(2,2)+4.5d0*gm(2,2)*dgm(3,3))
   cm(2,9,23)=1.5d0*gm(2,3)**3*dgm(2,2)+6*gm(2,2)*gm(2,3)**2*dgm(2,3)&
&   +12*gm(2,2)**2*gm(3,3)*dgm(2,3)+gm(2,2)*gm(2,3)*(7.5d0*gm(3,3)&
&   *dgm(2,2)+3*gm(2,2)*dgm(3,3))
   cm(2,10,23)=9*gm(2,3)**2*gm(3,3)*dgm(2,3)-3*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +2.5d0*gm(2,3)**3*dgm(3,3)+gm(2,3)*gm(3,3)*(3*gm(3,3)*dgm(2,2)&
&   -1.5d0*gm(2,2)*dgm(3,3))
   cm(2,1,24)=7.5d0*gm(1,2)**2*gm(1,3)*dgm(2,2)-1.5d0*gm(1,1)*gm(1,3)&
&   *gm(2,2)*dgm(2,2)+5*gm(1,2)**3*dgm(2,3)-3*gm(1,1)*gm(1,2)*(gm(2,3)&
&   *dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(2,2,24)=gm(2,2)*(6*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,2)*(3*gm(2,3)&
&   *dgm(2,2)+6*gm(2,2)*dgm(2,3)))
   cm(2,3,24)=gm(1,3)*(1.5d0*gm(2,3)**2*dgm(2,2)-4.5d0*gm(2,2)*gm(3,3)&
&   *dgm(2,2)-6*gm(2,2)*gm(2,3)*dgm(2,3))+gm(1,2)*(12*gm(2,3)*gm(3,3)&
&   *dgm(2,2)+15*gm(2,3)**2*dgm(2,3)-3*gm(2,2)*gm(3,3)*dgm(2,3))
   cm(2,4,24)=gm(1,3)*gm(2,2)*(3*gm(2,3)*dgm(2,2)-6*gm(2,2)*dgm(2,3))&
&   +gm(1,2)*(3*gm(2,3)**2*dgm(2,2)+12*gm(2,2)*gm(3,3)*dgm(2,2)+18*gm(2,2)&
&   *gm(2,3)*dgm(2,3))
   cm(2,5,24)=(60*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(2,2)+40*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,3))/240.d0
   cm(2,6,24)=12*gm(1,2)*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,1)*gm(2,2)&
&   *(-4.5d0*gm(2,3)*dgm(2,2)-3*gm(2,2)*dgm(2,3))+gm(1,2)**2*(1.5d0*gm(2,3)&
&   *dgm(2,2)+9*gm(2,2)*dgm(2,3))
   cm(2,7,24)=gm(2,2)**2*(3*gm(2,3)*dgm(2,2)+2*gm(2,2)*dgm(2,3))
   cm(2,8,24)=1.5d0*gm(2,3)**3*dgm(2,2)+7.5d0*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm(2,2)+9*gm(2,2)*gm(2,3)**2*dgm(2,3)-3*gm(2,2)**2*gm(3,3)&
&   *dgm(2,3)
   cm(2,9,24)=gm(2,2)*(3*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(3,3)*dgm(2,2)&
&   +6*gm(2,2)*gm(2,3)*dgm(2,3))
   cm(2,10,24)=4.5d0*gm(2,3)**2*gm(3,3)*dgm(2,2)-1.5d0*gm(2,2)*gm(3,3)&
&   **2*dgm(2,2)+5*gm(2,3)**3*dgm(2,3)-3*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm(2,3)
   cm(2,1,25)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm(2,2)&
&   +6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm(2,3)+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(3,3))/36.d0
   cm(2,2,25)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(2,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(2,3)+3*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm(3,3))/12.d0
   cm(2,3,25)=12*gm(1,2)*gm(3,3)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))&
&   +gm(1,3)*(3*gm(3,3)**2*dgm(2,2)+1.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)&
&   *(6*gm(2,3)*dgm(2,3)-4.5d0*gm(2,2)*dgm(3,3)))
   cm(2,4,25)=gm(1,3)*(6*gm(2,3)**2*dgm(2,3)+24*gm(2,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*(9*gm(3,3)*dgm(2,2)+3*gm(2,2)*dgm(3,3)))+gm(1,2)*(-3*gm(3,3)&
&   **2*dgm(2,2)+3*gm(2,3)**2*dgm(3,3)+gm(3,3)*(6*gm(2,3)*dgm(2,3)&
&   +12*gm(2,2)*dgm(3,3)))
   cm(2,5,25)=(gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(2,2)&
&   +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm(2,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/12.d0
   cm(2,6,25)=gm(1,3)**2*(7.5d0*gm(2,3)*dgm(2,2)+15*gm(2,2)*dgm(2,3))&
&   +gm(1,2)*gm(1,3)*(-3*gm(3,3)*dgm(2,2)+18*gm(2,3)*dgm(2,3)+12*gm(2,2)&
&   *dgm(3,3))+gm(1,2)**2*(-6*gm(3,3)*dgm(2,3)+1.5d0*gm(2,3)*dgm(3,3))&
&   +gm(1,1)*(-1.5d0*gm(2,3)*gm(3,3)*dgm(2,2)-6*gm(2,3)**2*dgm(2,3)&
&   -3*gm(2,2)*gm(3,3)*dgm(2,3)-4.5d0*gm(2,2)*gm(2,3)*dgm(3,3))
   cm(2,7,25)=2.5d0*gm(2,3)**3*dgm(2,2)+9*gm(2,2)*gm(2,3)**2*dgm(2,3)&
&   -3*gm(2,2)**2*gm(3,3)*dgm(2,3)+gm(2,2)*gm(2,3)*(-1.5d0*gm(3,3)&
&   *dgm(2,2)+3*gm(2,2)*dgm(3,3))
   cm(2,8,25)=6*gm(2,3)**2*gm(3,3)*dgm(2,3)+12*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +1.5d0*gm(2,3)**3*dgm(3,3)+gm(2,3)*gm(3,3)*(3*gm(3,3)*dgm(2,2)&
&   +7.5d0*gm(2,2)*dgm(3,3))
   cm(2,9,25)=3*gm(2,3)**3*dgm(2,3)+15*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)**2*(4.5d0*gm(3,3)*dgm(2,2)+3*gm(2,2)*dgm(3,3))+gm(2,2)&
&   *gm(3,3)*(-1.5d0*gm(3,3)*dgm(2,2)+6*gm(2,2)*dgm(3,3))
   cm(2,10,25)=gm(3,3)*(1*gm(3,3)**2*dgm(2,2)+4.5d0*gm(2,3)**2*dgm(3,3)&
&   +gm(3,3)*(6*gm(2,3)*dgm(2,3)-1.5d0*gm(2,2)*dgm(3,3)))
   cm(2,1,26)=5*gm(1,3)**3*dgm(2,3)+7.5d0*gm(1,2)*gm(1,3)**2*dgm(3,3)&
&   -1.5d0*gm(1,1)*gm(1,2)*gm(3,3)*dgm(3,3)-3*gm(1,1)*gm(1,3)*(gm(3,3)&
&   *dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(2,2,26)=(16*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(2,3)+24*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm(3,3))/96.d0
   cm(2,3,26)=gm(3,3)*(6*gm(1,2)*gm(3,3)*dgm(3,3)+gm(1,3)*(6*gm(3,3)&
&   *dgm(2,3)+3*gm(2,3)*dgm(3,3)))
   cm(2,4,26)=gm(1,2)*gm(3,3)*(-6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))&
&   +gm(1,3)*(18*gm(2,3)*gm(3,3)*dgm(2,3)+3*gm(2,3)**2*dgm(3,3)+12*gm(2,2)&
&   *gm(3,3)*dgm(3,3))
   cm(2,5,26)=12*gm(1,2)*gm(1,3)*gm(3,3)*dgm(3,3)+gm(1,1)*gm(3,3)&
&   *(-3*gm(3,3)*dgm(2,3)-4.5d0*gm(2,3)*dgm(3,3))+gm(1,3)**2*(9*gm(3,3)&
&   *dgm(2,3)+1.5d0*gm(2,3)*dgm(3,3))
   cm(2,6,26)=(16*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,3)+24*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(3,3))/96.d0
   cm(2,7,26)=5*gm(2,3)**3*dgm(2,3)-3*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +4.5d0*gm(2,2)*gm(2,3)**2*dgm(3,3)-1.5d0*gm(2,2)**2*gm(3,3)*dgm(3,3)
   cm(2,8,26)=gm(3,3)*(6*gm(2,3)*gm(3,3)*dgm(2,3)+3*gm(2,3)**2*dgm(3,3)&
&   +6*gm(2,2)*gm(3,3)*dgm(3,3))
   cm(2,9,26)=9*gm(2,3)**2*gm(3,3)*dgm(2,3)-3*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +1.5d0*gm(2,3)**3*dgm(3,3)+7.5d0*gm(2,2)*gm(2,3)*gm(3,3)*dgm(3,3)
   cm(2,10,26)=gm(3,3)**2*(2*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(2,1,27)=gm(1,3)*(2.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(3,3)
   cm(2,2,27)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/12.d0
   cm(2,3,27)=3*gm(1,3)*gm(3,3)**2*dgm(3,3)
   cm(2,4,27)=(gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm(3,3))&
&   /6.d0
   cm(2,5,27)=gm(3,3)*(4.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(3,3)
   cm(2,6,27)=((90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(3,3))/12.d0
   cm(2,7,27)=gm(2,3)*(2.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(3,3)
   cm(2,8,27)=3*gm(2,3)*gm(3,3)**2*dgm(3,3)
   cm(2,9,27)=gm(3,3)*(4.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(3,3)
   cm(2,10,27)=gm(3,3)**3*dgm(3,3)
   cm(3,1,3)=gm(1,1)**2*(3*gm(1,3)*dgm(1,1)+2*gm(1,1)*dgm(1,3))
   cm(3,2,3)=12*gm(1,1)*gm(1,2)*gm(2,3)*dgm(1,1)+gm(1,1)*gm(2,2)&
&   *(-4.5d0*gm(1,3)*dgm(1,1)-3*gm(1,1)*dgm(1,3))+gm(1,2)**2*(1.5d0*gm(1,3)&
&   *dgm(1,1)+9*gm(1,1)*dgm(1,3))
   cm(3,3,3)=1.5d0*gm(1,3)**3*dgm(1,1)+7.5d0*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm(1,1)+9*gm(1,1)*gm(1,3)**2*dgm(1,3)-3*gm(1,1)**2*gm(3,3)&
&   *dgm(1,3)
   cm(3,4,3)=gm(1,1)*gm(2,3)*(3*gm(1,3)*dgm(1,1)-6*gm(1,1)*dgm(1,3))&
&   +gm(1,2)*(3*gm(1,3)**2*dgm(1,1)+12*gm(1,1)*gm(3,3)*dgm(1,1)+18*gm(1,1)&
&   *gm(1,3)*dgm(1,3))
   cm(3,5,3)=gm(1,1)*(3*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,3)*dgm(1,3))
   cm(3,6,3)=gm(1,1)*(6*gm(1,1)*gm(2,3)*dgm(1,1)+gm(1,2)*(3*gm(1,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(1,3)))
   cm(3,7,3)=7.5d0*gm(1,2)**2*gm(2,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,2)&
&   *gm(2,3)*dgm(1,1)+5*gm(1,2)**3*dgm(1,3)-3*gm(1,2)*gm(2,2)*(gm(1,3)&
&   *dgm(1,1)+gm(1,1)*dgm(1,3))
   cm(3,8,3)=gm(1,1)*gm(3,3)*(-4.5d0*gm(2,3)*dgm(1,1)-3*gm(1,2)*dgm(1,3))&
&   +gm(1,3)**2*(1.5d0*gm(2,3)*dgm(1,1)+15*gm(1,2)*dgm(1,3))+gm(1,3)&
&   *(12*gm(1,2)*gm(3,3)*dgm(1,1)-6*gm(1,1)*gm(2,3)*dgm(1,3))
   cm(3,9,3)=(24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,1)+16*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&   -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(1,3))/96.d0
   cm(3,10,3)=4.5d0*gm(1,3)**2*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(3,3)&
&   **2*dgm(1,1)+5*gm(1,3)**3*dgm(1,3)-3*gm(1,1)*gm(1,3)*gm(3,3)&
&   *dgm(1,3)
   cm(3,1,4)=gm(1,1)**2*(3*gm(1,2)*dgm(1,1)+2*gm(1,1)*dgm(1,2))
   cm(3,2,4)=1.5d0*gm(1,2)**3*dgm(1,1)+7.5d0*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm(1,1)+9*gm(1,1)*gm(1,2)**2*dgm(1,2)-3*gm(1,1)**2*gm(2,2)&
&   *dgm(1,2)
   cm(3,3,4)=gm(1,2)*(1.5d0*gm(1,3)**2-4.5d0*gm(1,1)*gm(3,3))*dgm(1,1)&
&   +gm(1,1)*(12*gm(1,3)*gm(2,3)*dgm(1,1)+9*gm(1,3)**2*dgm(1,2)-3*gm(1,1)&
&   *gm(3,3)*dgm(1,2))
   cm(3,4,4)=3*gm(1,2)**2*gm(1,3)*dgm(1,1)+gm(1,1)*gm(1,2)*(3*gm(2,3)&
&   *dgm(1,1)+18*gm(1,3)*dgm(1,2))+gm(1,1)*(12*gm(1,3)*gm(2,2)*dgm(1,1)&
&   -6*gm(1,1)*gm(2,3)*dgm(1,2))
   cm(3,5,4)=gm(1,1)*(3*gm(1,2)*gm(1,3)*dgm(1,1)+6*gm(1,1)*(gm(2,3)&
&   *dgm(1,1)+gm(1,3)*dgm(1,2)))
   cm(3,6,4)=gm(1,1)*(3*gm(1,2)**2*dgm(1,1)+6*gm(1,1)*gm(2,2)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,2)*dgm(1,2))
   cm(3,7,4)=4.5d0*gm(1,2)**2*gm(2,2)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,2)&
&   **2*dgm(1,1)+5*gm(1,2)**3*dgm(1,2)-3*gm(1,1)*gm(1,2)*gm(2,2)&
&   *dgm(1,2)
   cm(3,8,4)=(12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,1)+8*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(1,2))/48.d0
   cm(3,9,4)=gm(1,1)*gm(2,2)*(-4.5d0*gm(2,3)*dgm(1,1)-3*gm(1,3)*dgm(1,2))&
&   +gm(1,2)**2*(1.5d0*gm(2,3)*dgm(1,1)+15*gm(1,3)*dgm(1,2))+gm(1,2)&
&   *(12*gm(1,3)*gm(2,2)*dgm(1,1)-6*gm(1,1)*gm(2,3)*dgm(1,2))
   cm(3,10,4)=7.5d0*gm(1,3)**2*gm(2,3)*dgm(1,1)-1.5d0*gm(1,1)*gm(2,3)&
&   *gm(3,3)*dgm(1,1)+5*gm(1,3)**3*dgm(1,2)-3*gm(1,3)*gm(3,3)*(gm(1,2)&
&   *dgm(1,1)+gm(1,1)*dgm(1,2))
   cm(3,1,5)=gm(1,1)**3*dgm(1,1)
   cm(3,2,5)=gm(1,1)*(4.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(1,1)
   cm(3,3,5)=gm(1,1)*(4.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(1,1)
   cm(3,4,5)=(gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm(1,1))&
&   /6.d0
   cm(3,5,5)=3*gm(1,1)**2*gm(1,3)*dgm(1,1)
   cm(3,6,5)=3*gm(1,1)**2*gm(1,2)*dgm(1,1)
   cm(3,7,5)=gm(1,2)*(2.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(1,1)
   cm(3,8,5)=((-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm(1,1))/12.d0
   cm(3,9,5)=((90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*gm(2,3))*dgm(1,1))/12.d0
   cm(3,10,5)=gm(1,3)*(2.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(1,1)
   cm(3,1,8)=gm(1,1)*(gm(1,2)*(9*gm(1,3)*dgm(1,1)+6*gm(1,1)*dgm(1,3))&
&   +gm(1,1)*(-3*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2)+2*gm(1,1)*dgm(2,3)))
   cm(3,2,8)=3*gm(1,2)**3*dgm(1,3)+gm(1,2)*(3*gm(1,3)*gm(2,2)*dgm(1,1)&
&   +gm(1,1)*(24*gm(2,3)*dgm(1,2)+15*gm(2,2)*dgm(1,3)))+gm(1,1)*gm(2,2)&
&   *(12*gm(2,3)*dgm(1,1)-9*gm(1,3)*dgm(1,2)-3*gm(1,1)*dgm(2,3))&
&   +gm(1,2)**2*(3*gm(2,3)*dgm(1,1)+3*gm(1,3)*dgm(1,2)+9*gm(1,1)&
&   *dgm(2,3))
   cm(3,3,8)=3*gm(1,3)**3*dgm(1,2)+gm(1,3)*(3*gm(1,2)*gm(3,3)*dgm(1,1)&
&   +gm(1,1)*(15*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3)))+gm(1,1)*gm(3,3)&
&   *(12*gm(2,3)*dgm(1,1)-9*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))&
&   +gm(1,3)**2*(3*gm(2,3)*dgm(1,1)+3*gm(1,2)*dgm(1,3)+9*gm(1,1)&
&   *dgm(2,3))
   cm(3,4,8)=9*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)**2*(9*gm(2,2)&
&   *dgm(1,1)+6*gm(1,2)*dgm(1,2))+gm(1,1)*(9*gm(2,3)**2*dgm(1,1)&
&   +15*gm(2,2)*gm(3,3)*dgm(1,1)+24*gm(1,2)*gm(3,3)*dgm(1,2)+6*gm(1,2)&
&   *gm(2,3)*dgm(1,3))-6*gm(1,1)**2*gm(2,3)*dgm(2,3)+gm(1,3)*(6*gm(1,2)&
&   **2*dgm(1,3)+gm(1,1)*(6*gm(2,3)*dgm(1,2)+24*gm(2,2)*dgm(1,3))&
&   +gm(1,2)*(-6*gm(2,3)*dgm(1,1)+18*gm(1,1)*dgm(2,3)))
   cm(3,5,8)=gm(1,2)*(3*gm(1,3)**2*dgm(1,1)+12*gm(1,1)*gm(3,3)*dgm(1,1)&
&   +6*gm(1,1)*gm(1,3)*dgm(1,3))+gm(1,1)*(6*gm(1,3)**2*dgm(1,2)+12*gm(1,1)&
&   *(gm(3,3)*dgm(1,2)+gm(2,3)*dgm(1,3))+gm(1,3)*(3*gm(2,3)*dgm(1,1)&
&   +6*gm(1,1)*dgm(2,3)))
   cm(3,6,8)=gm(1,2)**2*(3*gm(1,3)*dgm(1,1)+6*gm(1,1)*dgm(1,3))+12*gm(1,1)&
&   *(gm(1,3)*gm(2,2)*dgm(1,1)+gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)&
&   *dgm(1,3)))+gm(1,1)*gm(1,2)*(3*gm(2,3)*dgm(1,1)+6*(gm(1,3)*dgm(1,2)&
&   +gm(1,1)*dgm(2,3)))
   cm(3,7,8)=gm(1,3)*gm(2,2)*(-3*gm(2,2)*dgm(1,1)-6*gm(1,2)*dgm(1,2))&
&   -3*gm(1,1)*gm(2,2)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)&
&   **2*(15*gm(2,3)*dgm(1,2)+9*gm(2,2)*dgm(1,3))+5*gm(1,2)**3*dgm(2,3)&
&   +gm(1,2)*gm(2,2)*(9*gm(2,3)*dgm(1,1)-3*gm(1,1)*dgm(2,3))
   cm(3,8,8)=-6*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(-9*gm(2,3)*gm(3,3)&
&   *dgm(1,2)-6*gm(2,3)**2*dgm(1,3)-3*gm(2,2)*gm(3,3)*dgm(1,3))+gm(1,2)&
&   *gm(3,3)*(3*gm(2,3)*dgm(1,1)-3*gm(1,1)*dgm(2,3))+gm(1,3)*(3*gm(2,3)&
&   **2*dgm(1,1)+gm(3,3)*(12*gm(2,2)*dgm(1,1)+24*gm(1,2)*dgm(1,2))&
&   +gm(2,3)*(18*gm(1,2)*dgm(1,3)-6*gm(1,1)*dgm(2,3)))+gm(1,3)**2*(3*gm(2,3)&
&   *dgm(1,2)+15*(gm(2,2)*dgm(1,3)+gm(1,2)*dgm(2,3)))
   cm(3,9,8)=-6*gm(1,3)**2*gm(2,2)*dgm(1,2)+gm(1,2)**2*(15*gm(3,3)&
&   *dgm(1,2)+3*gm(2,3)*dgm(1,3))+gm(1,1)*(-6*gm(2,3)**2*dgm(1,2)&
&   -3*gm(2,2)*gm(3,3)*dgm(1,2)-9*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,2)&
&   *(3*gm(2,3)**2*dgm(1,1)+12*gm(2,2)*gm(3,3)*dgm(1,1)-6*gm(1,1)&
&   *gm(2,3)*dgm(2,3))+gm(1,3)*(gm(2,2)*(3*gm(2,3)*dgm(1,1)+24*gm(1,2)&
&   *dgm(1,3)-3*gm(1,1)*dgm(2,3))+gm(1,2)*(18*gm(2,3)*dgm(1,2)+15*gm(1,2)&
&   *dgm(2,3)))
   cm(3,10,8)=gm(1,3)**2*(9*gm(3,3)*dgm(1,2)+15*gm(2,3)*dgm(1,3))&
&   -3*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(1,1)+gm(1,1)*(gm(3,3)*dgm(1,2)&
&   +gm(2,3)*dgm(1,3)))+5*gm(1,3)**3*dgm(2,3)+gm(1,3)*gm(3,3)*(9*gm(2,3)&
&   *dgm(1,1)-6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))
   cm(3,1,9)=gm(1,1)*(4.5d0*gm(1,2)**2*dgm(1,1)+6*gm(1,1)*gm(1,2)&
&   *dgm(1,2)+gm(1,1)*(-1.5d0*gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2)))
   cm(3,2,9)=3*gm(1,2)**3*dgm(1,2)+15*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +gm(1,1)*gm(2,2)*(6*gm(2,2)*dgm(1,1)-1.5d0*gm(1,1)*dgm(2,2))&
&   +gm(1,2)**2*(3*gm(2,2)*dgm(1,1)+4.5d0*gm(1,1)*dgm(2,2))
   cm(3,3,9)=-3*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)*gm(2,3)*(9*gm(1,2)&
&   *dgm(1,1)+24*gm(1,1)*dgm(1,2))+gm(1,1)*(7.5d0*gm(2,3)**2*dgm(1,1)&
&   -1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)-9*gm(1,2)*gm(3,3)*dgm(1,2))-1.5d0*gm(1,1)&
&   **2*gm(3,3)*dgm(2,2)+gm(1,3)**2*(-3*gm(2,2)*dgm(1,1)+3*gm(1,2)&
&   *dgm(1,2)+4.5d0*gm(1,1)*dgm(2,2))
   cm(3,4,9)=gm(1,2)**2*(3*gm(2,3)*dgm(1,1)+6*gm(1,3)*dgm(1,2))+gm(1,1)&
&   *(gm(2,2)*(12*gm(2,3)*dgm(1,1)+24*gm(1,3)*dgm(1,2))-3*gm(1,1)&
&   *gm(2,3)*dgm(2,2))+gm(1,2)*(6*gm(1,1)*gm(2,3)*dgm(1,2)+gm(1,3)&
&   *(3*gm(2,2)*dgm(1,1)+9*gm(1,1)*dgm(2,2)))
   cm(3,5,9)=1.5d0*gm(1,2)**2*gm(1,3)*dgm(1,1)+gm(1,1)*gm(1,2)*(12*gm(2,3)&
&   *dgm(1,1)+6*gm(1,3)*dgm(1,2))+gm(1,1)*(12*gm(1,1)*gm(2,3)*dgm(1,2)&
&   +gm(1,3)*(-4.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))
   cm(3,6,9)=1.5d0*gm(1,2)**3*dgm(1,1)+6*gm(1,1)*gm(1,2)**2*dgm(1,2)&
&   +12*gm(1,1)**2*gm(2,2)*dgm(1,2)+gm(1,1)*gm(1,2)*(7.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,1)*dgm(2,2))
   cm(3,7,9)=9*gm(1,2)**2*gm(2,2)*dgm(1,2)-3*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +2.5d0*gm(1,2)**3*dgm(2,2)+gm(1,2)*gm(2,2)*(3*gm(2,2)*dgm(1,1)&
&   -1.5d0*gm(1,1)*dgm(2,2))
   cm(3,8,9)=(6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,1)+12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(2,2))/24.d0
   cm(3,9,9)=gm(2,3)*(3*gm(1,2)**2*dgm(1,2)-9*gm(1,1)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)*(3*gm(2,2)*dgm(1,1)-3*gm(1,1)*dgm(2,2)))+gm(1,3)*(6*gm(2,2)&
&   **2*dgm(1,1)+7.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)*(24*gm(1,2)*dgm(1,2)&
&   -1.5d0*gm(1,1)*dgm(2,2)))
   cm(3,10,9)=(1080*gm(1,3)**2*gm(2,3)*dgm(1,2)-216*gm(2,3)*gm(3,3)&
&   *(gm(1,2)*dgm(1,1)+gm(1,1)*dgm(1,2))+180*gm(1,3)**3*dgm(2,2)&
&   +gm(1,3)*(540*gm(2,3)**2*dgm(1,1)+gm(3,3)*(-108*gm(2,2)*dgm(1,1)&
&   -432*gm(1,2)*dgm(1,2)-108*gm(1,1)*dgm(2,2))))/72.d0
   cm(3,1,10)=gm(1,1)*(4.5d0*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(1,3)&
&   *dgm(1,3)+gm(1,1)*(-1.5d0*gm(3,3)*dgm(1,1)+gm(1,1)*dgm(3,3)))
   cm(3,2,10)=-3*gm(1,3)**2*gm(2,2)*dgm(1,1)-3*gm(1,2)**2*gm(3,3)&
&   *dgm(1,1)+gm(1,3)*(9*gm(1,2)*gm(2,3)*dgm(1,1)+3*gm(1,2)**2*dgm(1,3)&
&   -9*gm(1,1)*gm(2,2)*dgm(1,3))-1.5d0*gm(1,1)**2*gm(2,2)*dgm(3,3)&
&   +gm(1,1)*(7.5d0*gm(2,3)**2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)&
&   +24*gm(1,2)*gm(2,3)*dgm(1,3)+4.5d0*gm(1,2)**2*dgm(3,3))
   cm(3,3,10)=3*gm(1,3)**3*dgm(1,3)+15*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +gm(1,1)*gm(3,3)*(6*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*dgm(3,3))&
&   +gm(1,3)**2*(3*gm(3,3)*dgm(1,1)+4.5d0*gm(1,1)*dgm(3,3))
   cm(3,4,10)=gm(1,3)**2*(3*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3))&
&   +gm(1,1)*(24*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(12*gm(3,3)*dgm(1,1)&
&   -3*gm(1,1)*dgm(3,3)))+gm(1,3)*(6*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)&
&   *(3*gm(3,3)*dgm(1,1)+9*gm(1,1)*dgm(3,3)))
   cm(3,5,10)=1.5d0*gm(1,3)**3*dgm(1,1)+6*gm(1,1)*gm(1,3)**2*dgm(1,3)&
&   +12*gm(1,1)**2*gm(3,3)*dgm(1,3)+gm(1,1)*gm(1,3)*(7.5d0*gm(3,3)&
&   *dgm(1,1)+3*gm(1,1)*dgm(3,3))
   cm(3,6,10)=12*gm(1,1)*gm(2,3)*(gm(1,3)*dgm(1,1)+gm(1,1)*dgm(1,3))&
&   +gm(1,2)*(1.5d0*gm(1,3)**2*dgm(1,1)+6*gm(1,1)*gm(1,3)*dgm(1,3)&
&   +gm(1,1)*(-4.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3)))
   cm(3,7,10)=(3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,1)+6*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm(3,3))/36.d0
   cm(3,8,10)=gm(1,3)**2*(3*gm(2,3)*dgm(1,3)+7.5d0*gm(1,2)*dgm(3,3))&
&   +gm(1,3)*(24*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(3*gm(3,3)*dgm(1,1)&
&   -3*gm(1,1)*dgm(3,3)))+gm(3,3)*(-9*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)&
&   *(6*gm(3,3)*dgm(1,1)-1.5d0*gm(1,1)*dgm(3,3)))
   cm(3,9,10)=(3*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,1)+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(3,3))/12.d0
   cm(3,10,10)=9*gm(1,3)**2*gm(3,3)*dgm(1,3)-3*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +2.5d0*gm(1,3)**3*dgm(3,3)+gm(1,3)*gm(3,3)*(3*gm(3,3)*dgm(1,1)&
&   -1.5d0*gm(1,1)*dgm(3,3))
   cm(3,1,12)=gm(1,2)**2*(7.5d0*gm(1,3)*dgm(1,1)+9*gm(1,1)*dgm(1,3))&
&   +gm(1,1)*(gm(1,1)*(-6*gm(2,3)*dgm(1,2)-3*gm(2,2)*dgm(1,3))+gm(1,3)&
&   *(-1.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))+gm(1,1)*gm(1,2)&
&   *(-3*gm(2,3)*dgm(1,1)+18*gm(1,3)*dgm(1,2)+6*gm(1,1)*dgm(2,3))
   cm(3,2,12)=6*gm(1,2)**2*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,1)&
&   *gm(2,2)*(24*gm(2,3)*dgm(1,2)+12*gm(2,2)*dgm(1,3))+gm(1,3)*(6*gm(2,2)&
&   **2*dgm(1,1)+1.5d0*gm(1,2)**2*dgm(2,2)+gm(2,2)*(6*gm(1,2)*dgm(1,2)&
&   -4.5d0*gm(1,1)*dgm(2,2)))+3*gm(1,2)**3*dgm(2,3)+gm(1,2)*(3*gm(2,2)&
&   *gm(2,3)*dgm(1,1)+12*gm(1,1)*gm(2,3)*dgm(2,2)+15*gm(1,1)*gm(2,2)&
&   *dgm(2,3))
   cm(3,3,12)=-6*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(24*gm(2,3)&
&   *gm(3,3)*dgm(1,2)+15*gm(2,3)**2*dgm(1,3)-3*gm(2,2)*gm(3,3)*dgm(1,3))&
&   +1.5d0*gm(1,3)**3*dgm(2,2)+gm(1,2)*gm(3,3)*(12*gm(2,3)*dgm(1,1)&
&   -9*gm(1,1)*dgm(2,3))+gm(1,3)**2*(6*gm(2,3)*dgm(1,2)-6*gm(2,2)&
&   *dgm(1,3)+3*gm(1,2)*dgm(2,3))+gm(1,3)*(1.5d0*gm(2,3)**2*dgm(1,1)&
&   +gm(3,3)*(-4.5d0*gm(2,2)*dgm(1,1)+6*gm(1,2)*dgm(1,2)+7.5d0*gm(1,1)&
&   *dgm(2,2))+gm(2,3)*(18*gm(1,2)*dgm(1,3)+24*gm(1,1)*dgm(2,3)))
   cm(3,4,12)=gm(1,2)**2*(18*gm(3,3)*dgm(1,2)+6*gm(2,3)*dgm(1,3))&
&   +gm(1,1)*(18*gm(2,3)**2*dgm(1,2)+30*gm(2,2)*gm(3,3)*dgm(1,2)&
&   +24*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,3)**2*(18*gm(2,2)*dgm(1,2)&
&   +3*gm(1,2)*dgm(2,2))+gm(1,2)*(3*gm(2,3)**2*dgm(1,1)+12*gm(3,3)&
&   *(gm(2,2)*dgm(1,1)+gm(1,1)*dgm(2,2))+6*gm(1,1)*gm(2,3)*dgm(2,3))&
&   +gm(1,3)*(-12*gm(1,2)*gm(2,3)*dgm(1,2)+3*gm(1,1)*gm(2,3)*dgm(2,2)&
&   +6*gm(1,2)**2*dgm(2,3)+gm(2,2)*(3*gm(2,3)*dgm(1,1)+6*gm(1,2)&
&   *dgm(1,3)+24*gm(1,1)*dgm(2,3)))
   cm(3,5,12)=7.5d0*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,1)*(-3*gm(2,3)&
&   **2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)+24*gm(1,2)*gm(3,3)&
&   *dgm(1,2)+24*gm(1,2)*gm(2,3)*dgm(1,3))+gm(1,3)**2*(-3*gm(2,2)&
&   *dgm(1,1)+6*gm(1,2)*dgm(1,2)+3*gm(1,1)*dgm(2,2))+gm(1,1)**2*(6*gm(3,3)&
&   *dgm(2,2)+12*gm(2,3)*dgm(2,3))+gm(1,3)*(3*gm(1,2)**2*dgm(1,3)&
&   +gm(1,1)*(6*gm(2,3)*dgm(1,2)-9*gm(2,2)*dgm(1,3))+gm(1,2)*(9*gm(2,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(2,3)))
   cm(3,6,12)=3*gm(1,2)**3*dgm(1,3)+gm(1,2)*(gm(1,1)*(6*gm(2,3)*dgm(1,2)&
&   +15*gm(2,2)*dgm(1,3))+gm(1,3)*(12*gm(2,2)*dgm(1,1)+3*gm(1,1)&
&   *dgm(2,2)))+gm(1,2)**2*(1.5d0*gm(2,3)*dgm(1,1)+6*(gm(1,3)*dgm(1,2)&
&   +gm(1,1)*dgm(2,3)))+gm(1,1)*(6*gm(1,1)*gm(2,3)*dgm(2,2)+gm(2,2)&
&   *(-4.5d0*gm(2,3)*dgm(1,1)+24*gm(1,3)*dgm(1,2)+12*gm(1,1)*dgm(2,3)))
   cm(3,7,12)=7.5d0*gm(1,2)**2*gm(2,3)*dgm(2,2)+gm(2,2)**2*(3*gm(2,3)&
&   *dgm(1,1)-6*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))&
&   +gm(2,2)*(-1.5d0*gm(1,1)*gm(2,3)*dgm(2,2)+gm(1,2)*(18*gm(2,3)&
&   *dgm(1,2)-3*gm(1,3)*dgm(2,2))+9*gm(1,2)**2*dgm(2,3))
   cm(3,8,12)=1.5d0*gm(2,3)**3*dgm(1,1)+gm(1,3)*gm(3,3)*(24*gm(2,2)&
&   *dgm(1,2)+12*gm(1,2)*dgm(2,2))+15*gm(1,3)**2*gm(2,2)*dgm(2,3)&
&   +gm(2,3)**2*(6*gm(1,3)*dgm(1,2)+3*gm(1,2)*dgm(1,3)-6*gm(1,1)&
&   *dgm(2,3))+gm(3,3)*(-9*gm(1,2)*gm(2,2)*dgm(1,3)-6*gm(1,2)**2*dgm(2,3)&
&   -3*gm(1,1)*gm(2,2)*dgm(2,3))+gm(2,3)*(gm(2,2)*(7.5d0*gm(3,3)&
&   *dgm(1,1)+24*gm(1,3)*dgm(1,3))+(1.5d0*gm(1,3)**2-4.5d0*gm(1,1)&
&   *gm(3,3))*dgm(2,2)+gm(1,2)*(6*gm(3,3)*dgm(1,2)+18*gm(1,3)*dgm(2,3)))
   cm(3,9,12)=gm(2,2)**2*(6*gm(3,3)*dgm(1,1)+12*gm(1,3)*dgm(1,3))&
&   -3*gm(1,1)*gm(2,3)**2*dgm(2,2)+gm(1,2)*gm(2,3)*(6*gm(2,3)*dgm(1,2)&
&   +9*gm(1,3)*dgm(2,2))+gm(1,2)**2*(7.5d0*gm(3,3)*dgm(2,2)+3*gm(2,3)&
&   *dgm(2,3))+gm(2,2)*(3*gm(2,3)**2*dgm(1,1)+(-3*gm(1,3)**2-1.5d0*gm(1,1)&
&   *gm(3,3))*dgm(2,2)+gm(2,3)*(6*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)&
&   -9*gm(1,1)*dgm(2,3))+24*gm(1,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3)))
   cm(3,10,12)=gm(2,3)**2*(4.5d0*gm(3,3)*dgm(1,1)+15*gm(1,3)*dgm(1,3))&
&   +gm(2,3)*(18*gm(1,3)*gm(3,3)*dgm(1,2)+15*gm(1,3)**2*dgm(2,3)&
&   +gm(3,3)*(-6*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3)))+gm(3,3)*(gm(2,2)&
&   *(-1.5d0*gm(3,3)*dgm(1,1)-3*gm(1,3)*dgm(1,3))+(4.5d0*gm(1,3)&
&   **2-1.5d0*gm(1,1)*gm(3,3))*dgm(2,2)-6*gm(1,2)*(gm(3,3)*dgm(1,2)&
&   +gm(1,3)*dgm(2,3)))
   cm(3,1,13)=2.5d0*gm(1,2)**3*dgm(1,1)+9*gm(1,1)*gm(1,2)**2*dgm(1,2)&
&   -3*gm(1,1)**2*gm(2,2)*dgm(1,2)+gm(1,1)*gm(1,2)*(-1.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,1)*dgm(2,2))
   cm(3,2,13)=6*gm(1,2)**2*gm(2,2)*dgm(1,2)+12*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +1.5d0*gm(1,2)**3*dgm(2,2)+gm(1,2)*gm(2,2)*(3*gm(2,2)*dgm(1,1)&
&   +7.5d0*gm(1,1)*dgm(2,2))
   cm(3,3,13)=-6*gm(1,2)**2*gm(3,3)*dgm(1,2)+gm(1,1)*(15*gm(2,3)&
&   **2-3*gm(2,2)*gm(3,3))*dgm(1,2)+gm(1,3)*gm(2,3)*(-3*gm(2,2)*dgm(1,1)&
&   +18*gm(1,2)*dgm(1,2)+12*gm(1,1)*dgm(2,2))+gm(1,3)**2*(-6*gm(2,2)&
&   *dgm(1,2)+1.5d0*gm(1,2)*dgm(2,2))+gm(1,2)*(7.5d0*gm(2,3)**2*dgm(1,1)&
&   -1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)-4.5d0*gm(1,1)*gm(3,3)*dgm(2,2))
   cm(3,4,13)=gm(2,3)*(6*gm(1,2)**2*dgm(1,2)+24*gm(1,1)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)*(9*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))+gm(1,3)*(-3*gm(2,2)&
&   **2*dgm(1,1)+3*gm(1,2)**2*dgm(2,2)+gm(2,2)*(6*gm(1,2)*dgm(1,2)&
&   +12*gm(1,1)*dgm(2,2)))
   cm(3,5,13)=gm(1,2)**2*(7.5d0*gm(2,3)*dgm(1,1)+3*gm(1,3)*dgm(1,2))&
&   +gm(1,1)*(gm(2,2)*(-1.5d0*gm(2,3)*dgm(1,1)-9*gm(1,3)*dgm(1,2))&
&   +6*gm(1,1)*gm(2,3)*dgm(2,2))+gm(1,2)*(24*gm(1,1)*gm(2,3)*dgm(1,2)&
&   +gm(1,3)*(-3*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2)))
   cm(3,6,13)=3*gm(1,2)**3*dgm(1,2)+15*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +gm(1,2)**2*(4.5d0*gm(2,2)*dgm(1,1)+3*gm(1,1)*dgm(2,2))+gm(1,1)&
&   *gm(2,2)*(-1.5d0*gm(2,2)*dgm(1,1)+6*gm(1,1)*dgm(2,2))
   cm(3,7,13)=gm(2,2)*(1*gm(2,2)**2*dgm(1,1)+4.5d0*gm(1,2)**2*dgm(2,2)&
&   +gm(2,2)*(6*gm(1,2)*dgm(1,2)-1.5d0*gm(1,1)*dgm(2,2)))
   cm(3,8,13)=(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,1)&
&   +12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,2))/24.d0
   cm(3,9,13)=gm(2,2)**2*(3*gm(2,3)*dgm(1,1)+12*gm(1,3)*dgm(1,2))&
&   +1.5d0*gm(1,2)**2*gm(2,3)*dgm(2,2)+gm(2,2)*(-4.5d0*gm(1,1)*gm(2,3)&
&   *dgm(2,2)+gm(1,2)*(6*gm(2,3)*dgm(1,2)+12*gm(1,3)*dgm(2,2)))
   cm(3,10,13)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm(1,1)&
&   +12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&   *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2))/72.d0
   cm(3,1,14)=gm(1,1)*(9*gm(1,3)**2*dgm(1,2)+gm(1,1)*(-3*gm(3,3)&
&   *dgm(1,2)-6*gm(2,3)*dgm(1,3))+gm(1,3)*(-3*gm(2,3)*dgm(1,1)+6*gm(1,1)&
&   *dgm(2,3)))+gm(1,2)*(7.5d0*gm(1,3)**2*dgm(1,1)+18*gm(1,1)*gm(1,3)&
&   *dgm(1,3)+gm(1,1)*(-1.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3)))
   cm(3,2,14)=-6*gm(1,3)**2*gm(2,2)*dgm(1,2)+gm(1,2)**2*(-6*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3))+gm(1,1)*(15*gm(2,3)**2*dgm(1,2)&
&   -3*gm(2,2)*gm(3,3)*dgm(1,2)+24*gm(2,2)*gm(2,3)*dgm(1,3))+gm(1,3)&
&   *(gm(2,2)*(12*gm(2,3)*dgm(1,1)+6*gm(1,2)*dgm(1,3)-9*gm(1,1)*dgm(2,3))&
&   +gm(1,2)*(18*gm(2,3)*dgm(1,2)+3*gm(1,2)*dgm(2,3)))+1.5d0*gm(1,2)&
&   **3*dgm(3,3)+gm(1,2)*(1.5d0*gm(2,3)**2*dgm(1,1)-4.5d0*gm(2,2)&
&   *gm(3,3)*dgm(1,1)+24*gm(1,1)*gm(2,3)*dgm(2,3)+7.5d0*gm(1,1)*gm(2,2)&
&   *dgm(3,3))
   cm(3,3,14)=3*gm(1,3)**3*dgm(2,3)+gm(1,3)**2*(6*gm(3,3)*dgm(1,2)&
&   +6*gm(2,3)*dgm(1,3)+1.5d0*gm(1,2)*dgm(3,3))+gm(3,3)*(gm(1,1)&
&   *(12*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3))+gm(1,2)*(6*gm(3,3)&
&   *dgm(1,1)-4.5d0*gm(1,1)*dgm(3,3)))+gm(1,3)*(gm(3,3)*(6*gm(1,2)&
&   *dgm(1,3)+15*gm(1,1)*dgm(2,3))+gm(2,3)*(3*gm(3,3)*dgm(1,1)+12*gm(1,1)&
&   *dgm(3,3)))
   cm(3,4,14)=18*gm(1,2)**2*gm(3,3)*dgm(1,3)+gm(1,1)*(24*gm(2,3)&
&   *gm(3,3)*dgm(1,2)+18*gm(2,3)**2*dgm(1,3)+30*gm(2,2)*gm(3,3)*dgm(1,3))&
&   +gm(1,3)**2*(6*gm(2,3)*dgm(1,2)+18*gm(2,2)*dgm(1,3)+6*gm(1,2)&
&   *dgm(2,3))+gm(1,2)*(24*gm(1,1)*gm(3,3)*dgm(2,3)+3*gm(2,3)*(gm(3,3)&
&   *dgm(1,1)+gm(1,1)*dgm(3,3)))+gm(1,3)*(3*gm(2,3)**2*dgm(1,1)+gm(2,3)&
&   *(-12*gm(1,2)*dgm(1,3)+6*gm(1,1)*dgm(2,3))+12*gm(2,2)*(gm(3,3)&
&   *dgm(1,1)+gm(1,1)*dgm(3,3))+gm(1,2)*(6*gm(3,3)*dgm(1,2)+3*gm(1,2)&
&   *dgm(3,3)))
   cm(3,5,14)=3*gm(1,3)**3*dgm(1,2)+gm(1,3)**2*(1.5d0*gm(2,3)*dgm(1,1)&
&   +6*(gm(1,2)*dgm(1,3)+gm(1,1)*dgm(2,3)))+gm(1,3)*(gm(1,1)*(15*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3))+gm(1,2)*(12*gm(3,3)*dgm(1,1)+3*gm(1,1)&
&   *dgm(3,3)))+gm(1,1)*(gm(3,3)*(24*gm(1,2)*dgm(1,3)+12*gm(1,1)&
&   *dgm(2,3))+gm(2,3)*(-4.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3)))
   cm(3,6,14)=-3*gm(1,2)**2*gm(3,3)*dgm(1,1)+gm(1,3)**2*(7.5d0*gm(2,2)&
&   *dgm(1,1)+3*gm(1,2)*dgm(1,2))+gm(1,3)*(6*gm(1,2)**2*dgm(1,3)&
&   +24*gm(1,1)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))+gm(1,2)*(9*gm(2,3)&
&   *dgm(1,1)+6*gm(1,1)*dgm(2,3)))+gm(1,1)*(-3*gm(2,3)**2*dgm(1,1)&
&   -1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)-9*gm(1,2)*gm(3,3)*dgm(1,2)+6*gm(1,2)&
&   *gm(2,3)*dgm(1,3)+3*gm(1,2)**2*dgm(3,3))+gm(1,1)**2*(12*gm(2,3)&
&   *dgm(2,3)+6*gm(2,2)*dgm(3,3))
   cm(3,7,14)=15*gm(1,2)*gm(2,3)*(gm(2,3)*dgm(1,2)+gm(1,2)*dgm(2,3))&
&   +gm(2,2)**2*(-1.5d0*gm(3,3)*dgm(1,1)-6*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)&
&   *dgm(3,3))+gm(2,2)*(4.5d0*gm(2,3)**2*dgm(1,1)+gm(2,3)*(-6*gm(1,3)&
&   *dgm(1,2)+18*gm(1,2)*dgm(1,3)-3*gm(1,1)*dgm(2,3))+gm(1,2)*(-3*gm(3,3)&
&   *dgm(1,2)-6*gm(1,3)*dgm(2,3)+4.5d0*gm(1,2)*dgm(3,3)))
   cm(3,8,14)=gm(2,3)**2*(3*gm(3,3)*dgm(1,1)+6*gm(1,3)*dgm(1,3)-3*gm(1,1)&
&   *dgm(3,3))+gm(1,2)*gm(3,3)*(12*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)&
&   -3*gm(1,2)*dgm(3,3))+gm(2,2)*(6*gm(3,3)**2*dgm(1,1)+24*gm(1,3)&
&   *gm(3,3)*dgm(1,3)+7.5d0*gm(1,3)**2*dgm(3,3)-1.5d0*gm(1,1)*gm(3,3)&
&   *dgm(3,3))+gm(2,3)*(3*gm(1,3)**2*dgm(2,3)+gm(3,3)*(6*gm(1,2)&
&   *dgm(1,3)-9*gm(1,1)*dgm(2,3))+gm(1,3)*(6*gm(3,3)*dgm(1,2)+9*gm(1,2)&
&   *dgm(3,3)))
   cm(3,9,14)=1.5d0*gm(2,3)**3*dgm(1,1)-6*gm(1,3)**2*gm(2,2)*dgm(2,3)&
&   +gm(2,3)**2*(3*gm(1,3)*dgm(1,2)+6*gm(1,2)*dgm(1,3)-6*gm(1,1)&
&   *dgm(2,3))+gm(3,3)*(24*gm(1,2)*gm(2,2)*dgm(1,3)+15*gm(1,2)**2*dgm(2,3)&
&   -3*gm(1,1)*gm(2,2)*dgm(2,3))+gm(1,3)*gm(2,2)*(-9*gm(3,3)*dgm(1,2)&
&   +12*gm(1,2)*dgm(3,3))+gm(2,3)*(gm(2,2)*(7.5d0*gm(3,3)*dgm(1,1)&
&   +6*gm(1,3)*dgm(1,3)-4.5d0*gm(1,1)*dgm(3,3))+gm(1,2)*(24*gm(3,3)&
&   *dgm(1,2)+18*gm(1,3)*dgm(2,3)+1.5d0*gm(1,2)*dgm(3,3)))
   cm(3,10,14)=gm(2,3)*(3*gm(3,3)**2*dgm(1,1)+7.5d0*gm(1,3)**2*dgm(3,3)&
&   +gm(3,3)*(18*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)*dgm(3,3)))+gm(3,3)&
&   *(9*gm(1,3)**2*dgm(2,3)+gm(3,3)*(-6*gm(1,2)*dgm(1,3)-3*gm(1,1)&
&   *dgm(2,3))+gm(1,3)*(6*gm(3,3)*dgm(1,2)-3*gm(1,2)*dgm(3,3)))
   cm(3,1,15)=2.5d0*gm(1,3)**3*dgm(1,1)+9*gm(1,1)*gm(1,3)**2*dgm(1,3)&
&   -3*gm(1,1)**2*gm(3,3)*dgm(1,3)+gm(1,1)*gm(1,3)*(-1.5d0*gm(3,3)&
&   *dgm(1,1)+3*gm(1,1)*dgm(3,3))
   cm(3,2,15)=-6*gm(1,3)**2*gm(2,2)*dgm(1,3)-6*gm(1,2)**2*gm(3,3)&
&   *dgm(1,3)+gm(1,1)*(15*gm(2,3)**2-3*gm(2,2)*gm(3,3))*dgm(1,3)&
&   +gm(1,2)*gm(2,3)*(-3*gm(3,3)*dgm(1,1)+12*gm(1,1)*dgm(3,3))+gm(1,3)&
&   *(7.5d0*gm(2,3)**2*dgm(1,1)-1.5d0*gm(2,2)*gm(3,3)*dgm(1,1)+18*gm(1,2)&
&   *gm(2,3)*dgm(1,3)+1.5d0*gm(1,2)**2*dgm(3,3)-4.5d0*gm(1,1)*gm(2,2)&
&   *dgm(3,3))
   cm(3,3,15)=6*gm(1,3)**2*gm(3,3)*dgm(1,3)+12*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +1.5d0*gm(1,3)**3*dgm(3,3)+gm(1,3)*gm(3,3)*(3*gm(3,3)*dgm(1,1)&
&   +7.5d0*gm(1,1)*dgm(3,3))
   cm(3,4,15)=gm(1,3)**2*(6*gm(2,3)*dgm(1,3)+3*gm(1,2)*dgm(3,3))&
&   +gm(1,3)*(6*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)*(9*gm(3,3)*dgm(1,1)&
&   +3*gm(1,1)*dgm(3,3)))+gm(3,3)*(24*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)&
&   *(-3*gm(3,3)*dgm(1,1)+12*gm(1,1)*dgm(3,3)))
   cm(3,5,15)=3*gm(1,3)**3*dgm(1,3)+15*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +gm(1,3)**2*(4.5d0*gm(3,3)*dgm(1,1)+3*gm(1,1)*dgm(3,3))+gm(1,1)&
&   *gm(3,3)*(-1.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3))
   cm(3,6,15)=gm(1,3)**2*(7.5d0*gm(2,3)*dgm(1,1)+3*gm(1,2)*dgm(1,3))&
&   +gm(1,3)*(24*gm(1,1)*gm(2,3)*dgm(1,3)+gm(1,2)*(-3*gm(3,3)*dgm(1,1)&
&   +3*gm(1,1)*dgm(3,3)))+gm(1,1)*(-9*gm(1,2)*gm(3,3)*dgm(1,3)+gm(2,3)&
&   *(-1.5d0*gm(3,3)*dgm(1,1)+6*gm(1,1)*dgm(3,3)))
   cm(3,7,15)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm(1,1)&
&   +6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&   **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))/36.d0
   cm(3,8,15)=12*gm(1,2)*gm(3,3)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3))&
&   +gm(2,3)*(3*gm(3,3)**2*dgm(1,1)+1.5d0*gm(1,3)**2*dgm(3,3)+gm(3,3)&
&   *(6*gm(1,3)*dgm(1,3)-4.5d0*gm(1,1)*dgm(3,3)))
   cm(3,9,15)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,1)&
&   +6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/12.d0
   cm(3,10,15)=gm(3,3)*(1*gm(3,3)**2*dgm(1,1)+4.5d0*gm(1,3)**2*dgm(3,3)&
&   +gm(3,3)*(6*gm(1,3)*dgm(1,3)-1.5d0*gm(1,1)*dgm(3,3)))
   cm(3,1,17)=5*gm(1,2)**3*dgm(1,3)+gm(1,1)*gm(1,2)*(-6*gm(2,3)*dgm(1,2)&
&   -3*gm(2,2)*dgm(1,3)+9*gm(1,3)*dgm(2,2))+gm(1,2)**2*(15*gm(1,3)&
&   *dgm(1,2)+9*gm(1,1)*dgm(2,3))-3*gm(1,1)*(gm(1,3)*gm(2,2)*dgm(1,2)&
&   +gm(1,1)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3)))
   cm(3,2,17)=6*gm(1,2)*gm(2,2)*(gm(2,3)*dgm(1,2)+gm(2,2)*dgm(1,3))&
&   +gm(1,3)*gm(2,2)*(12*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))+12*gm(1,1)&
&   *gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))+gm(1,2)**2*(3*gm(2,3)&
&   *dgm(2,2)+6*gm(2,2)*dgm(2,3))
   cm(3,3,17)=(3*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+3*(-12*gm(1,3)&
&   **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&   +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm(2,3))/6.d0
   cm(3,4,17)=gm(1,2)*(6*gm(2,3)**2*dgm(1,2)+24*gm(2,2)*gm(3,3)*dgm(1,2)&
&   +18*gm(2,2)*gm(2,3)*dgm(1,3))+9*gm(1,3)**2*gm(2,2)*dgm(2,2)+gm(1,2)&
&   **2*(9*gm(3,3)*dgm(2,2)+6*gm(2,3)*dgm(2,3))+gm(1,1)*(9*gm(2,3)&
&   **2*dgm(2,2)+15*gm(2,2)*gm(3,3)*dgm(2,2)+24*gm(2,2)*gm(2,3)*dgm(2,3))&
&   +gm(1,3)*(-6*gm(2,2)**2*dgm(1,3)-6*gm(1,2)*gm(2,3)*dgm(2,2)+6*gm(2,2)&
&   *(gm(2,3)*dgm(1,2)+gm(1,2)*dgm(2,3)))
   cm(3,5,17)=(3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,2)+(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+3*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm(2,2)+3*(6*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(2,3))/6.d0
   cm(3,6,17)=gm(1,2)**2*(3*gm(2,3)*dgm(1,2)+9*gm(2,2)*dgm(1,3)+3*gm(1,3)&
&   *dgm(2,2))+gm(1,1)*gm(2,2)*(-9*gm(2,3)*dgm(1,2)-3*gm(2,2)*dgm(1,3)&
&   +12*gm(1,3)*dgm(2,2))+3*gm(1,2)**3*dgm(2,3)+gm(1,2)*(24*gm(1,3)&
&   *gm(2,2)*dgm(1,2)+gm(1,1)*(3*gm(2,3)*dgm(2,2)+15*gm(2,2)*dgm(2,3)))
   cm(3,7,17)=gm(2,2)*(2*gm(2,2)**2*dgm(1,3)+9*gm(1,2)*gm(2,3)*dgm(2,2)&
&   +gm(2,2)*(6*gm(2,3)*dgm(1,2)-3*gm(1,3)*dgm(2,2)+6*gm(1,2)*dgm(2,3)))
   cm(3,8,17)=3*gm(2,3)**3*dgm(1,2)+gm(2,2)*gm(3,3)*(-3*gm(2,2)*dgm(1,3)&
&   +12*gm(1,3)*dgm(2,2)-9*gm(1,2)*dgm(2,3))+gm(2,3)**2*(9*gm(2,2)&
&   *dgm(1,3)+3*(gm(1,3)*dgm(2,2)+gm(1,2)*dgm(2,3)))+gm(2,3)*(3*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(15*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)))
   cm(3,9,17)=3*gm(1,2)*gm(2,3)**2*dgm(2,2)+gm(2,2)**2*(12*gm(3,3)&
&   *dgm(1,2)+6*gm(2,3)*dgm(1,3)+12*gm(1,3)*dgm(2,3))+gm(2,2)*(6*gm(2,3)&
&   **2*dgm(1,2)+12*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)*(3*gm(1,3)*dgm(2,2)&
&   +6*gm(1,2)*dgm(2,3)))
   cm(3,10,17)=5*gm(2,3)**3*dgm(1,3)+gm(2,3)*gm(3,3)*(-3*gm(2,2)&
&   *dgm(1,3)+9*gm(1,3)*dgm(2,2)-6*gm(1,2)*dgm(2,3))+gm(2,3)**2*(9*gm(3,3)&
&   *dgm(1,2)+15*gm(1,3)*dgm(2,3))-3*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(2,2)&
&   +gm(2,2)*(gm(3,3)*dgm(1,2)+gm(1,3)*dgm(2,3)))
   cm(3,1,18)=5*gm(1,2)**3*dgm(1,2)-3*gm(1,1)*gm(1,2)*gm(2,2)*dgm(1,2)&
&   +4.5d0*gm(1,1)*gm(1,2)**2*dgm(2,2)-1.5d0*gm(1,1)**2*gm(2,2)*dgm(2,2)
   cm(3,2,18)=gm(2,2)*(6*gm(1,2)*gm(2,2)*dgm(1,2)+3*gm(1,2)**2*dgm(2,2)&
&   +6*gm(1,1)*gm(2,2)*dgm(2,2))
   cm(3,3,18)=(8*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,2)+12*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(2,2))/48.d0
   cm(3,4,18)=gm(1,3)*gm(2,2)*(-6*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))&
&   +gm(2,3)*(18*gm(1,2)*gm(2,2)*dgm(1,2)+3*gm(1,2)**2*dgm(2,2)+12*gm(1,1)&
&   *gm(2,2)*dgm(2,2))
   cm(3,5,18)=(8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,2)+12*(6*gm(1,2)**2*gm(1,3)&
&   -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm(2,2))&
&   /48.d0
   cm(3,6,18)=9*gm(1,2)**2*gm(2,2)*dgm(1,2)-3*gm(1,1)*gm(2,2)**2*dgm(1,2)&
&   +1.5d0*gm(1,2)**3*dgm(2,2)+7.5d0*gm(1,1)*gm(1,2)*gm(2,2)*dgm(2,2)
   cm(3,7,18)=gm(2,2)**2*(2*gm(2,2)*dgm(1,2)+3*gm(1,2)*dgm(2,2))
   cm(3,8,18)=-3*gm(2,2)**2*gm(3,3)*dgm(1,2)+1.5d0*gm(1,2)*gm(2,3)&
&   **2*dgm(2,2)+gm(2,2)*(9*gm(2,3)**2*dgm(1,2)+12*gm(1,3)*gm(2,3)&
&   *dgm(2,2)-4.5d0*gm(1,2)*gm(3,3)*dgm(2,2))
   cm(3,9,18)=gm(2,2)*(3*gm(1,2)*gm(2,3)*dgm(2,2)+6*gm(2,2)*(gm(2,3)&
&   *dgm(1,2)+gm(1,3)*dgm(2,2)))
   cm(3,10,18)=5*gm(2,3)**3*dgm(1,2)+7.5d0*gm(1,3)*gm(2,3)**2*dgm(2,2)&
&   -1.5d0*gm(1,3)*gm(2,2)*gm(3,3)*dgm(2,2)-3*gm(2,3)*gm(3,3)*(gm(2,2)&
&   *dgm(1,2)+gm(1,2)*dgm(2,2))
   cm(3,1,19)=(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(1,2)+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(1,3)+gm(1,1)&
&   *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(2,2)+4*gm(1,1)*(54*gm(1,2)&
&   *gm(1,3)-18*gm(1,1)*gm(2,3))*dgm(2,3)+gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&   *gm(2,2))*dgm(3,3))/12.d0
   cm(3,2,19)=(2*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+2*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&   *gm(2,3))*dgm(1,3)+(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&   *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,2)+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,3)+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&   *gm(2,2))*dgm(3,3))/4.d0
   cm(3,3,19)=gm(1,2)*gm(3,3)*(12*gm(3,3)*dgm(1,2)+24*gm(2,3)*dgm(1,3))&
&   -3*gm(1,2)**2*gm(3,3)*dgm(3,3)+gm(1,3)**2*(3*gm(3,3)*dgm(2,2)&
&   +6*gm(2,3)*dgm(2,3)-3*gm(2,2)*dgm(3,3))+gm(1,1)*(6*gm(3,3)**2*dgm(2,2)&
&   +24*gm(2,3)*gm(3,3)*dgm(2,3)+7.5d0*gm(2,3)**2*dgm(3,3)-1.5d0*gm(2,2)&
&   *gm(3,3)*dgm(3,3))+gm(1,3)*(3*gm(2,3)**2*dgm(1,3)+gm(3,3)*(-9*gm(2,2)&
&   *dgm(1,3)+6*gm(1,2)*dgm(2,3))+gm(2,3)*(6*gm(3,3)*dgm(1,2)+9*gm(1,2)&
&   *dgm(3,3)))
   cm(3,4,19)=(2*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+2*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2+24*gm(2,2)*gm(3,3)))*dgm(1,3)+(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&   *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+4*(9*gm(1,3)&
&   **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&   *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm(2,3)+(6*gm(1,2)*gm(1,3)&
&   *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))&
&   /2.d0
   cm(3,5,19)=(2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,2)+2*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(1,3)+(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm(2,2)+4*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm(2,3)+(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm(3,3))/4.d0
   cm(3,6,19)=(2*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(1,2)+2*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(1,3)+(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(2,2)+4*(6*gm(1,2)&
&   **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(2,3)+(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm(3,3))&
&   /4.d0
   cm(3,7,19)=7.5d0*gm(1,2)*gm(2,3)**2*dgm(2,2)+gm(2,2)*(9*gm(2,3)&
&   **2*dgm(1,2)-1.5d0*gm(1,2)*gm(3,3)*dgm(2,2)+gm(2,3)*(-3*gm(1,3)&
&   *dgm(2,2)+18*gm(1,2)*dgm(2,3)))+gm(2,2)**2*(-3*gm(3,3)*dgm(1,2)&
&   +6*gm(2,3)*dgm(1,3)-6*gm(1,3)*dgm(2,3)+3*gm(1,2)*dgm(3,3))
   cm(3,8,19)=3*gm(2,3)**3*dgm(1,3)+gm(2,3)**2*(6*gm(3,3)*dgm(1,2)&
&   +6*gm(1,3)*dgm(2,3)+1.5d0*gm(1,2)*dgm(3,3))+gm(3,3)*(6*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(12*gm(3,3)*dgm(1,2)+24*gm(1,3)*dgm(2,3)&
&   -4.5d0*gm(1,2)*dgm(3,3)))+gm(2,3)*(gm(3,3)*(3*gm(1,3)*dgm(2,2)&
&   +6*gm(1,2)*dgm(2,3))+gm(2,2)*(15*gm(3,3)*dgm(1,3)+12*gm(1,3)&
&   *dgm(3,3)))
   cm(3,9,19)=3*gm(2,3)**3*dgm(1,2)+gm(2,3)**2*(6*gm(2,2)*dgm(1,3)&
&   +1.5d0*gm(1,3)*dgm(2,2)+6*gm(1,2)*dgm(2,3))+gm(2,3)*(12*gm(1,2)&
&   *gm(3,3)*dgm(2,2)+gm(2,2)*(15*gm(3,3)*dgm(1,2)+6*gm(1,3)*dgm(2,3)&
&   +3*gm(1,2)*dgm(3,3)))+gm(2,2)*(gm(3,3)*(-4.5d0*gm(1,3)*dgm(2,2)&
&   +24*gm(1,2)*dgm(2,3))+gm(2,2)*(12*gm(3,3)*dgm(1,3)+6*gm(1,3)&
&   *dgm(3,3)))
   cm(3,10,19)=gm(2,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+18*gm(1,3)*dgm(2,3)&
&   -3*gm(1,2)*dgm(3,3))+gm(2,3)**2*(9*gm(3,3)*dgm(1,3)+7.5d0*gm(1,3)&
&   *dgm(3,3))+gm(3,3)*(gm(3,3)*(3*gm(1,3)*dgm(2,2)-6*gm(1,2)*dgm(2,3))&
&   +gm(2,2)*(-3*gm(3,3)*dgm(1,3)-1.5d0*gm(1,3)*dgm(3,3)))
   cm(3,1,20)=5*gm(1,3)**3*dgm(1,2)+gm(1,3)**2*(15*gm(1,2)*dgm(1,3)&
&   +9*gm(1,1)*dgm(2,3))+gm(1,1)*gm(1,3)*(-3*gm(3,3)*dgm(1,2)-6*gm(2,3)&
&   *dgm(1,3)+9*gm(1,2)*dgm(3,3))-3*gm(1,1)*(gm(1,2)*gm(3,3)*dgm(1,3)&
&   +gm(1,1)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3)))
   cm(3,2,20)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(1,2)+3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(2,3)+3*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&   +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))/6.d0
   cm(3,3,20)=gm(1,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+6*gm(2,3)*dgm(1,3)&
&   +3*gm(1,2)*dgm(3,3))+gm(1,3)**2*(6*gm(3,3)*dgm(2,3)+3*gm(2,3)&
&   *dgm(3,3))+12*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(1,3)+gm(1,1)*(gm(3,3)&
&   *dgm(2,3)+gm(2,3)*dgm(3,3)))
   cm(3,4,20)=(gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(1,3)+3*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,3)+3*(9*gm(1,3)**2*gm(2,2)&
&   -6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)&
&   **2+15*gm(2,2)*gm(3,3)))*dgm(3,3))/3.d0
   cm(3,5,20)=(gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm(1,3)+3*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)&
&   *gm(3,3))*dgm(2,3)+3*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&   **2+24*gm(1,1)*gm(3,3)))*dgm(3,3))/6.d0
   cm(3,6,20)=((90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,2)+3*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(1,3)+3*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&   +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(2,3)+3*(6*gm(1,2)&
&   **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&   *dgm(3,3))/6.d0
   cm(3,7,20)=5*gm(2,3)**3*dgm(1,2)+gm(2,3)**2*(9*gm(2,2)*dgm(1,3)&
&   +15*gm(1,2)*dgm(2,3))+gm(2,2)*gm(2,3)*(-3*gm(3,3)*dgm(1,2)-6*gm(1,3)&
&   *dgm(2,3)+9*gm(1,2)*dgm(3,3))-3*gm(2,2)*(gm(1,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,2)*(gm(3,3)*dgm(1,3)+gm(1,3)*dgm(3,3)))
   cm(3,8,20)=gm(2,3)*gm(3,3)*(6*gm(3,3)*dgm(1,2)+6*gm(1,3)*dgm(2,3)&
&   +3*gm(1,2)*dgm(3,3))+gm(2,3)**2*(6*gm(3,3)*dgm(1,3)+3*gm(1,3)&
&   *dgm(3,3))+12*gm(3,3)*(gm(1,2)*gm(3,3)*dgm(2,3)+gm(2,2)*(gm(3,3)&
&   *dgm(1,3)+gm(1,3)*dgm(3,3)))
   cm(3,9,20)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm(1,2)&
&   +3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm(1,3)+3*(48*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm(2,3)&
&   +3*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/6.d0
   cm(3,10,20)=gm(3,3)*(2*gm(3,3)**2*dgm(1,2)+9*gm(1,3)*gm(2,3)*dgm(3,3)&
&   +gm(3,3)*(6*gm(2,3)*dgm(1,3)+6*gm(1,3)*dgm(2,3)-3*gm(1,2)*dgm(3,3)))
   cm(3,1,21)=5*gm(1,3)**3*dgm(1,3)-3*gm(1,1)*gm(1,3)*gm(3,3)*dgm(1,3)&
&   +4.5d0*gm(1,1)*gm(1,3)**2*dgm(3,3)-1.5d0*gm(1,1)**2*gm(3,3)*dgm(3,3)
   cm(3,2,21)=(40*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(1,3)+60*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(3,3))/240.d0
   cm(3,3,21)=gm(3,3)*(6*gm(1,3)*gm(3,3)*dgm(1,3)+3*gm(1,3)**2*dgm(3,3)&
&   +6*gm(1,1)*gm(3,3)*dgm(3,3))
   cm(3,4,21)=3*gm(1,3)**2*gm(2,3)*dgm(3,3)+gm(1,3)*gm(3,3)*(18*gm(2,3)&
&   *dgm(1,3)+3*gm(1,2)*dgm(3,3))+gm(3,3)*(-6*gm(1,2)*gm(3,3)*dgm(1,3)&
&   +12*gm(1,1)*gm(2,3)*dgm(3,3))
   cm(3,5,21)=9*gm(1,3)**2*gm(3,3)*dgm(1,3)-3*gm(1,1)*gm(3,3)**2*dgm(1,3)&
&   +1.5d0*gm(1,3)**3*dgm(3,3)+7.5d0*gm(1,1)*gm(1,3)*gm(3,3)*dgm(3,3)
   cm(3,6,21)=(40*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(1,3)+60*(48*gm(1,1)*gm(1,3)&
&   *gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm(3,3))&
&   /240.d0
   cm(3,7,21)=5*gm(2,3)**3*dgm(1,3)+7.5d0*gm(1,2)*gm(2,3)**2*dgm(3,3)&
&   -1.5d0*gm(1,2)*gm(2,2)*gm(3,3)*dgm(3,3)-3*gm(2,2)*gm(2,3)*(gm(3,3)&
&   *dgm(1,3)+gm(1,3)*dgm(3,3))
   cm(3,8,21)=gm(3,3)*(6*gm(1,2)*gm(3,3)*dgm(3,3)+gm(2,3)*(6*gm(3,3)&
&   *dgm(1,3)+3*gm(1,3)*dgm(3,3)))
   cm(3,9,21)=12*gm(1,2)*gm(2,3)*gm(3,3)*dgm(3,3)+gm(2,2)*gm(3,3)&
&   *(-3*gm(3,3)*dgm(1,3)-4.5d0*gm(1,3)*dgm(3,3))+gm(2,3)**2*(9*gm(3,3)&
&   *dgm(1,3)+1.5d0*gm(1,3)*dgm(3,3))
   cm(3,10,21)=gm(3,3)**2*(2*gm(3,3)*dgm(1,3)+3*gm(1,3)*dgm(3,3))
   cm(3,1,23)=7.5d0*gm(1,2)**2*gm(1,3)*dgm(2,2)-1.5d0*gm(1,1)*gm(1,3)&
&   *gm(2,2)*dgm(2,2)+5*gm(1,2)**3*dgm(2,3)-3*gm(1,1)*gm(1,2)*(gm(2,3)&
&   *dgm(2,2)+gm(2,2)*dgm(2,3))
   cm(3,2,23)=gm(2,2)*(6*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,2)*(3*gm(2,3)&
&   *dgm(2,2)+6*gm(2,2)*dgm(2,3)))
   cm(3,3,23)=gm(1,3)*(1.5d0*gm(2,3)**2*dgm(2,2)-4.5d0*gm(2,2)*gm(3,3)&
&   *dgm(2,2)-6*gm(2,2)*gm(2,3)*dgm(2,3))+gm(1,2)*(12*gm(2,3)*gm(3,3)&
&   *dgm(2,2)+15*gm(2,3)**2*dgm(2,3)-3*gm(2,2)*gm(3,3)*dgm(2,3))
   cm(3,4,23)=gm(1,3)*gm(2,2)*(3*gm(2,3)*dgm(2,2)-6*gm(2,2)*dgm(2,3))&
&   +gm(1,2)*(3*gm(2,3)**2*dgm(2,2)+12*gm(2,2)*gm(3,3)*dgm(2,2)+18*gm(2,2)&
&   *gm(2,3)*dgm(2,3))
   cm(3,5,23)=(24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *dgm(2,2)+16*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,3))/96.d0
   cm(3,6,23)=12*gm(1,2)*gm(1,3)*gm(2,2)*dgm(2,2)+gm(1,1)*gm(2,2)&
&   *(-4.5d0*gm(2,3)*dgm(2,2)-3*gm(2,2)*dgm(2,3))+gm(1,2)**2*(1.5d0*gm(2,3)&
&   *dgm(2,2)+9*gm(2,2)*dgm(2,3))
   cm(3,7,23)=gm(2,2)**2*(3*gm(2,3)*dgm(2,2)+2*gm(2,2)*dgm(2,3))
   cm(3,8,23)=1.5d0*gm(2,3)**3*dgm(2,2)+7.5d0*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm(2,2)+9*gm(2,2)*gm(2,3)**2*dgm(2,3)-3*gm(2,2)**2*gm(3,3)&
&   *dgm(2,3)
   cm(3,9,23)=gm(2,2)*(3*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(3,3)*dgm(2,2)&
&   +6*gm(2,2)*gm(2,3)*dgm(2,3))
   cm(3,10,23)=4.5d0*gm(2,3)**2*gm(3,3)*dgm(2,2)-1.5d0*gm(2,2)*gm(3,3)&
&   **2*dgm(2,2)+5*gm(2,3)**3*dgm(2,3)-3*gm(2,2)*gm(2,3)*gm(3,3)&
&   *dgm(2,3)
   cm(3,1,24)=gm(1,2)*(2.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(2,2)
   cm(3,2,24)=3*gm(1,2)*gm(2,2)**2*dgm(2,2)
   cm(3,3,24)=((-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(2,2))/12.d0
   cm(3,4,24)=(gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm(2,2))&
&   /6.d0
   cm(3,5,24)=((-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&   -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(2,2))/12.d0
   cm(3,6,24)=gm(2,2)*(4.5d0*gm(1,2)**2-1.5d0*gm(1,1)*gm(2,2))*dgm(2,2)
   cm(3,7,24)=gm(2,2)**3*dgm(2,2)
   cm(3,8,24)=gm(2,2)*(4.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(2,2)
   cm(3,9,24)=3*gm(2,2)**2*gm(2,3)*dgm(2,2)
   cm(3,10,24)=gm(2,3)*(2.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(2,2)
   cm(3,1,25)=(3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*dgm(2,2)+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(2,3)+(90*gm(1,2)&
&   **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm(3,3))/36.d0
   cm(3,2,25)=12*gm(1,3)*gm(2,2)*(gm(2,3)*dgm(2,2)+gm(2,2)*dgm(2,3))&
&   +gm(1,2)*(1.5d0*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(-4.5d0*gm(3,3)*dgm(2,2)+3*gm(2,2)*dgm(3,3)))
   cm(3,3,25)=gm(1,3)*(3*gm(2,3)**2*dgm(2,3)-9*gm(2,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*(3*gm(3,3)*dgm(2,2)-3*gm(2,2)*dgm(3,3)))+gm(1,2)*(6*gm(3,3)&
&   **2*dgm(2,2)+7.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)*(24*gm(2,3)*dgm(2,3)&
&   -1.5d0*gm(2,2)*dgm(3,3)))
   cm(3,4,25)=gm(1,3)*(3*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(2,3)*dgm(2,3)&
&   +gm(2,2)*(12*gm(3,3)*dgm(2,2)-3*gm(2,2)*dgm(3,3)))+gm(1,2)*(6*gm(2,3)&
&   **2*dgm(2,3)+24*gm(2,2)*gm(3,3)*dgm(2,3)+gm(2,3)*(3*gm(3,3)*dgm(2,2)&
&   +9*gm(2,2)*dgm(3,3)))
   cm(3,5,25)=(3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,2)+6*(-12*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(2,3)+(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&   +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm(3,3))&
&   /12.d0
   cm(3,6,25)=7.5d0*gm(1,3)**2*gm(2,2)*dgm(2,2)+gm(1,2)*gm(1,3)*(9*gm(2,3)&
&   *dgm(2,2)+24*gm(2,2)*dgm(2,3))+gm(1,2)**2*(-3*gm(3,3)*dgm(2,2)&
&   +3*gm(2,3)*dgm(2,3)+4.5d0*gm(2,2)*dgm(3,3))+gm(1,1)*(-3*gm(2,3)&
&   **2*dgm(2,2)-9*gm(2,2)*gm(2,3)*dgm(2,3)-1.5d0*gm(2,2)*(gm(3,3)&
&   *dgm(2,2)+gm(2,2)*dgm(3,3)))
   cm(3,7,25)=gm(2,2)*(4.5d0*gm(2,3)**2*dgm(2,2)+6*gm(2,2)*gm(2,3)&
&   *dgm(2,3)+gm(2,2)*(-1.5d0*gm(3,3)*dgm(2,2)+gm(2,2)*dgm(3,3)))
   cm(3,8,25)=3*gm(2,3)**3*dgm(2,3)+15*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +gm(2,2)*gm(3,3)*(6*gm(3,3)*dgm(2,2)-1.5d0*gm(2,2)*dgm(3,3))&
&   +gm(2,3)**2*(3*gm(3,3)*dgm(2,2)+4.5d0*gm(2,2)*dgm(3,3))
   cm(3,9,25)=1.5d0*gm(2,3)**3*dgm(2,2)+6*gm(2,2)*gm(2,3)**2*dgm(2,3)&
&   +12*gm(2,2)**2*gm(3,3)*dgm(2,3)+gm(2,2)*gm(2,3)*(7.5d0*gm(3,3)&
&   *dgm(2,2)+3*gm(2,2)*dgm(3,3))
   cm(3,10,25)=9*gm(2,3)**2*gm(3,3)*dgm(2,3)-3*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +2.5d0*gm(2,3)**3*dgm(3,3)+gm(2,3)*gm(3,3)*(3*gm(3,3)*dgm(2,2)&
&   -1.5d0*gm(2,2)*dgm(3,3))
   cm(3,1,26)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm(2,2)&
&   +6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&   *gm(3,3)))*dgm(2,3)+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&   *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm(3,3))/36.d0
   cm(3,2,26)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(2,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(2,3)+3*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&   +12*gm(1,2)*gm(2,3))*dgm(3,3))/12.d0
   cm(3,3,26)=12*gm(1,2)*gm(3,3)*(gm(3,3)*dgm(2,3)+gm(2,3)*dgm(3,3))&
&   +gm(1,3)*(3*gm(3,3)**2*dgm(2,2)+1.5d0*gm(2,3)**2*dgm(3,3)+gm(3,3)&
&   *(6*gm(2,3)*dgm(2,3)-4.5d0*gm(2,2)*dgm(3,3)))
   cm(3,4,26)=gm(1,3)*(6*gm(2,3)**2*dgm(2,3)+24*gm(2,2)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)*(9*gm(3,3)*dgm(2,2)+3*gm(2,2)*dgm(3,3)))+gm(1,2)*(-3*gm(3,3)&
&   **2*dgm(2,2)+3*gm(2,3)**2*dgm(3,3)+gm(3,3)*(6*gm(2,3)*dgm(2,3)&
&   +12*gm(2,2)*dgm(3,3)))
   cm(3,5,26)=(gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm(2,2)&
&   +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&   *gm(2,3)*gm(3,3))*dgm(2,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&   *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/12.d0
   cm(3,6,26)=gm(1,3)**2*(7.5d0*gm(2,3)*dgm(2,2)+15*gm(2,2)*dgm(2,3))&
&   +gm(1,2)*gm(1,3)*(-3*gm(3,3)*dgm(2,2)+18*gm(2,3)*dgm(2,3)+12*gm(2,2)&
&   *dgm(3,3))+gm(1,2)**2*(-6*gm(3,3)*dgm(2,3)+1.5d0*gm(2,3)*dgm(3,3))&
&   +gm(1,1)*(-1.5d0*gm(2,3)*gm(3,3)*dgm(2,2)-6*gm(2,3)**2*dgm(2,3)&
&   -3*gm(2,2)*gm(3,3)*dgm(2,3)-4.5d0*gm(2,2)*gm(2,3)*dgm(3,3))
   cm(3,7,26)=2.5d0*gm(2,3)**3*dgm(2,2)+9*gm(2,2)*gm(2,3)**2*dgm(2,3)&
&   -3*gm(2,2)**2*gm(3,3)*dgm(2,3)+gm(2,2)*gm(2,3)*(-1.5d0*gm(3,3)&
&   *dgm(2,2)+3*gm(2,2)*dgm(3,3))
   cm(3,8,26)=6*gm(2,3)**2*gm(3,3)*dgm(2,3)+12*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +1.5d0*gm(2,3)**3*dgm(3,3)+gm(2,3)*gm(3,3)*(3*gm(3,3)*dgm(2,2)&
&   +7.5d0*gm(2,2)*dgm(3,3))
   cm(3,9,26)=3*gm(2,3)**3*dgm(2,3)+15*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +gm(2,3)**2*(4.5d0*gm(3,3)*dgm(2,2)+3*gm(2,2)*dgm(3,3))+gm(2,2)&
&   *gm(3,3)*(-1.5d0*gm(3,3)*dgm(2,2)+6*gm(2,2)*dgm(3,3))
   cm(3,10,26)=gm(3,3)*(1*gm(3,3)**2*dgm(2,2)+4.5d0*gm(2,3)**2*dgm(3,3)&
&   +gm(3,3)*(6*gm(2,3)*dgm(2,3)-1.5d0*gm(2,2)*dgm(3,3)))
   cm(3,1,27)=5*gm(1,3)**3*dgm(2,3)+7.5d0*gm(1,2)*gm(1,3)**2*dgm(3,3)&
&   -1.5d0*gm(1,1)*gm(1,2)*gm(3,3)*dgm(3,3)-3*gm(1,1)*gm(1,3)*(gm(3,3)&
&   *dgm(2,3)+gm(2,3)*dgm(3,3))
   cm(3,2,27)=(40*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*dgm(2,3)+60*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&   +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm(3,3))/240.d0
   cm(3,3,27)=gm(3,3)*(6*gm(1,2)*gm(3,3)*dgm(3,3)+gm(1,3)*(6*gm(3,3)&
&   *dgm(2,3)+3*gm(2,3)*dgm(3,3)))
   cm(3,4,27)=gm(1,2)*gm(3,3)*(-6*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))&
&   +gm(1,3)*(18*gm(2,3)*gm(3,3)*dgm(2,3)+3*gm(2,3)**2*dgm(3,3)+12*gm(2,2)&
&   *gm(3,3)*dgm(3,3))
   cm(3,5,27)=12*gm(1,2)*gm(1,3)*gm(3,3)*dgm(3,3)+gm(1,1)*gm(3,3)&
&   *(-3*gm(3,3)*dgm(2,3)-4.5d0*gm(2,3)*dgm(3,3))+gm(1,3)**2*(9*gm(3,3)&
&   *dgm(2,3)+1.5d0*gm(2,3)*dgm(3,3))
   cm(3,6,27)=(40*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(2,3)+60*(30*gm(1,3)**2*gm(2,2)&
&   +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&   **2-6*gm(2,2)*gm(3,3)))*dgm(3,3))/240.d0
   cm(3,7,27)=5*gm(2,3)**3*dgm(2,3)-3*gm(2,2)*gm(2,3)*gm(3,3)*dgm(2,3)&
&   +4.5d0*gm(2,2)*gm(2,3)**2*dgm(3,3)-1.5d0*gm(2,2)**2*gm(3,3)*dgm(3,3)
   cm(3,8,27)=gm(3,3)*(6*gm(2,3)*gm(3,3)*dgm(2,3)+3*gm(2,3)**2*dgm(3,3)&
&   +6*gm(2,2)*gm(3,3)*dgm(3,3))
   cm(3,9,27)=9*gm(2,3)**2*gm(3,3)*dgm(2,3)-3*gm(2,2)*gm(3,3)**2*dgm(2,3)&
&   +1.5d0*gm(2,3)**3*dgm(3,3)+7.5d0*gm(2,2)*gm(2,3)*gm(3,3)*dgm(3,3)
   cm(3,10,27)=gm(3,3)**2*(2*gm(3,3)*dgm(2,3)+3*gm(2,3)*dgm(3,3))
   cm(3,1,28)=gm(1,3)*(2.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(3,3)
   cm(3,2,28)=((-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&   *gm(3,3)))*dgm(3,3))/12.d0
   cm(3,3,28)=3*gm(1,3)*gm(3,3)**2*dgm(3,3)
   cm(3,4,28)=(gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*dgm(3,3))&
&   /6.d0
   cm(3,5,28)=gm(3,3)*(4.5d0*gm(1,3)**2-1.5d0*gm(1,1)*gm(3,3))*dgm(3,3)
   cm(3,6,28)=((90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&   -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm(3,3))/12.d0
   cm(3,7,28)=gm(2,3)*(2.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(3,3)
   cm(3,8,28)=3*gm(2,3)*gm(3,3)**2*dgm(3,3)
   cm(3,9,28)=gm(3,3)*(4.5d0*gm(2,3)**2-1.5d0*gm(2,2)*gm(3,3))*dgm(3,3)
   cm(3,10,28)=gm(3,3)**3*dgm(3,3)
 end if
!
!contraction to output 3-vector
!
 eisnl(:)=0.d0
 do jj=1,((rank+4)*(rank+5))/2
   tmp(:,:)=0.d0
   do ii=1,((rank+1)*(rank+2))/2
     tmp(:,1)=tmp(:,1)+aa(:,ii)*cm(1,ii,jj)
     tmp(:,2)=tmp(:,2)+aa(:,ii)*cm(2,ii,jj)
     tmp(:,3)=tmp(:,3)+aa(:,ii)*cm(3,ii,jj)
   end do
   eisnl(:)=eisnl(:)+tmp(1,:)*bb(1,jj)+tmp(2,:)*bb(2,jj)
 end do
!factor of 2 multiplied in to drop call to conjugate contraction
!eisnl(:)=0.5d0*eisnl(:)

 ABI_FREE(cm)

end subroutine contistr03
!!***

end module m_contistr03
!!***
