/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import * as child_process from 'node:child_process';
import * as process from 'node:process';
import * as path from 'node:path';
import * as fs from 'node:fs';
import * as os from 'node:os';
import { DetectedIde, getIdeInfo } from './detect-ide.js';
import { GEMINI_CLI_COMPANION_EXTENSION_NAME } from './constants.js';
function getVsCodeCommand(platform = process.platform) {
    return platform === 'win32' ? 'code.cmd' : 'code';
}
async function findVsCodeCommand(platform = process.platform) {
    // 1. Check PATH first.
    const vscodeCommand = getVsCodeCommand(platform);
    try {
        if (platform === 'win32') {
            const result = child_process
                .execSync(`where.exe ${vscodeCommand}`)
                .toString()
                .trim();
            // `where.exe` can return multiple paths. Return the first one.
            const firstPath = result.split(/\r?\n/)[0];
            if (firstPath) {
                return firstPath;
            }
        }
        else {
            child_process.execSync(`command -v ${vscodeCommand}`, {
                stdio: 'ignore',
            });
            return vscodeCommand;
        }
    }
    catch {
        // Not in PATH, continue to check common locations.
    }
    // 2. Check common installation locations.
    const locations = [];
    const homeDir = os.homedir();
    if (platform === 'darwin') {
        // macOS
        locations.push('/Applications/Visual Studio Code.app/Contents/Resources/app/bin/code', path.join(homeDir, 'Library/Application Support/Code/bin/code'));
    }
    else if (platform === 'linux') {
        // Linux
        locations.push('/usr/share/code/bin/code', '/snap/bin/code', path.join(homeDir, '.local/share/code/bin/code'));
    }
    else if (platform === 'win32') {
        // Windows
        locations.push(path.join(process.env['ProgramFiles'] || 'C:\\Program Files', 'Microsoft VS Code', 'bin', 'code.cmd'), path.join(homeDir, 'AppData', 'Local', 'Programs', 'Microsoft VS Code', 'bin', 'code.cmd'));
    }
    for (const location of locations) {
        if (fs.existsSync(location)) {
            return location;
        }
    }
    return null;
}
class VsCodeInstaller {
    ide;
    platform;
    vsCodeCommand;
    ideInfo;
    constructor(ide, platform = process.platform) {
        this.ide = ide;
        this.platform = platform;
        this.vsCodeCommand = findVsCodeCommand(platform);
        this.ideInfo = getIdeInfo(ide);
    }
    async install() {
        const commandPath = await this.vsCodeCommand;
        if (!commandPath) {
            return {
                success: false,
                message: `${this.ideInfo.displayName} CLI not found. Please ensure 'code' is in your system's PATH. For help, see https://code.visualstudio.com/docs/configure/command-line#_code-is-not-recognized-as-an-internal-or-external-command. You can also install the '${GEMINI_CLI_COMPANION_EXTENSION_NAME}' extension manually from the VS Code marketplace.`,
            };
        }
        const command = `"${commandPath}" --install-extension google.gemini-cli-vscode-ide-companion --force`;
        try {
            child_process.execSync(command, { stdio: 'pipe' });
            return {
                success: true,
                message: `${this.ideInfo.displayName} companion extension was installed successfully.`,
            };
        }
        catch (_error) {
            return {
                success: false,
                message: `Failed to install ${this.ideInfo.displayName} companion extension. Please try installing '${GEMINI_CLI_COMPANION_EXTENSION_NAME}' manually from the ${this.ideInfo.displayName} extension marketplace.`,
            };
        }
    }
}
export function getIdeInstaller(ide, platform = process.platform) {
    switch (ide) {
        case DetectedIde.VSCode:
        case DetectedIde.FirebaseStudio:
            return new VsCodeInstaller(ide, platform);
        default:
            return null;
    }
}
//# sourceMappingURL=ide-installer.js.map