pragma Ada_2012;

pragma Style_Checks (Off);
pragma Warnings (Off, "-gnatwu");

with Interfaces.C; use Interfaces.C;
with Interfaces.C.Strings;
with libfswatch_types_h;
with corecrt_h;
with System;

package cevent_h is

  -- * Copyright (c) 2014-2015 Enrico M. Crisostomo
  -- *
  -- * This program is free software; you can redistribute it and/or modify it under
  -- * the terms of the GNU General Public License as published by the Free Software
  -- * Foundation; either version 3, or (at your option) any later version.
  -- *
  -- * This program is distributed in the hope that it will be useful, but WITHOUT
  -- * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  -- * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  -- * details.
  -- *
  -- * You should have received a copy of the GNU General Public License along with
  -- * this program.  If not, see <http://www.gnu.org/licenses/>.
  --  

  --*
  -- * @file
  -- * @brief Event type manipulation.
  -- *
  -- * This header file defines the event types of the `libfswatch` API.
  -- *
  -- * @copyright Copyright (c) 2014-2015 Enrico M. Crisostomo
  -- * @license GNU General Public License v. 3.0
  -- * @author Enrico M. Crisostomo
  -- * @version 1.8.0
  --  

  --*
  -- * @brief Backend-agnostic change flags.
  -- *
  -- * Each element of this enum represents a backend-agnostic change flag.  No
  -- * direct mapping to backend-specific change types is guaranteed to exist: a
  -- * change type may be mapped to multiple `fsw_event_flag` instances included
  -- * the `PlatformSpecific` flag.
  -- *
  -- * The values of event flags are all powers of 2, that is numbers @f$f=2^n@f$
  -- * where @f$n@f$ is an integer.  This representation makes it easy to combine
  -- * flags into a bit mask and encode multiple events flags into a single integer.
  -- *
  -- * A monitor implementation is required to map implementation-specific flags
  -- * into API flags.  Sometimes, though, a perfect match is not possible and the
  -- * following situation may arise:
  -- *
  -- *   - One platform-specific flag must be mapped into multiple API flags.
  -- *
  -- *   - Multiple platform-specific flags must be mapped into a single API flag.
  -- *
  -- *   - A mapping is not possible for some flags, in which case they should be
  -- *     mapped to fsw_event_flag::PlatformSpecific.  The API currently offers no
  -- *     way to retain a platform-specific event flag value in this case.
  --  

   subtype fsw_event_flag is unsigned;
   fsw_event_flag_NoOp : constant fsw_event_flag := 0;
   fsw_event_flag_PlatformSpecific : constant fsw_event_flag := 1;
   fsw_event_flag_Created : constant fsw_event_flag := 2;
   fsw_event_flag_Updated : constant fsw_event_flag := 4;
   fsw_event_flag_Removed : constant fsw_event_flag := 8;
   fsw_event_flag_Renamed : constant fsw_event_flag := 16;
   fsw_event_flag_OwnerModified : constant fsw_event_flag := 32;
   fsw_event_flag_AttributeModified : constant fsw_event_flag := 64;
   fsw_event_flag_MovedFrom : constant fsw_event_flag := 128;
   fsw_event_flag_MovedTo : constant fsw_event_flag := 256;
   fsw_event_flag_IsFile : constant fsw_event_flag := 512;
   fsw_event_flag_IsDir : constant fsw_event_flag := 1024;
   fsw_event_flag_IsSymLink : constant fsw_event_flag := 2048;
   fsw_event_flag_Link : constant fsw_event_flag := 4096;
   fsw_event_flag_Overflow : constant fsw_event_flag := 8192;  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:63

  --*< No event has occurred.  
  --*< Platform-specific placeholder for event type that cannot currently be mapped.  
  --*< An object was created.  
  --*< An object was updated.  
  --*< An object was removed.  
  --*< An object was renamed.  
  --*< The owner of an object was modified.  
  --*< The attributes of an object were modified.  
  --*< An object was moved from this location.  
  --*< An object was moved to this location.  
  --*< The object is a file.  
  --*< The object is a directory.  
  --*< The object is a symbolic link.  
  --*< The link count of an object has changed.  
  --*< The event queue has overflowed.  
   FSW_ALL_EVENT_FLAGS : aliased array (0 .. 14) of aliased fsw_event_flag  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:82
   with Import => True, 
        Convention => C, 
        External_Name => "FSW_ALL_EVENT_FLAGS";

  --*
  --   * @brief Get event flag by name.
  --   *
  --   * This function looks for an event flag called @p name and, if it exists, it
  --   * writes its value onto @p flag and @c FSW_OK, otherwise @p flag is not
  --   * modified and @c FSW_ERR_UNKNOWN_VALUE is returned.
  --   *
  --   * @param[in] name The name of the event flag to look for.
  --   * @param[out] flag The output variable where the event flag is returned.
  --   * @return #FSW_OK if the functions succeeds, #FSW_ERR_UNKNOWN_VALUE
  --   * otherwise.
  --    

   function fsw_get_event_flag_by_name (name : Interfaces.C.Strings.chars_ptr; flag : access fsw_event_flag) return libfswatch_types_h.FSW_STATUS  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:96
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_get_event_flag_by_name";

  --*
  --   * @brief Get the name of an event flag.
  --   *
  --   * This function looks for the name of the specified event @p flag.  If it
  --   * exists, it returns its name, otherwise @c nullptr is returned.
  --   *
  --   * @param[in] flag The event flag to look for.
  --   * @return The name of @p flag, or @c nullptr if it does not exist.
  --    

   function fsw_get_event_flag_name (flag : fsw_event_flag) return Interfaces.C.Strings.chars_ptr  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:107
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_get_event_flag_name";

  --*
  --   * A file change event is represented as an instance of this struct where:
  --   *   - path is the path where the event was triggered.
  --   *   - evt_time the time when the event was triggered.
  --   *   - flags is an array of fsw_event_flag of size flags_num.
  --   *   - flags_num is the size of the flags array.
  --    

   type fsw_cevent is record
      path : Interfaces.C.Strings.chars_ptr;  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:118
      evt_time : aliased corecrt_h.time_t;  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:119
      flags : access fsw_event_flag;  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:120
      flags_num : aliased unsigned;  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:121
   end record
   with Convention => C_Pass_By_Copy;  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:116

  --*
  --   * A function pointer of type FSW_CEVENT_CALLBACK is used by the API as a
  --   * callback to provide information about received events.  The callback is
  --   * passed the following arguments:
  --   *   - events, a const pointer to an array of events of type const fsw_cevent.
  --   *   - event_num, the size of the *events array.
  --   *   - data, optional persisted data for a callback.
  --   *
  --   * The memory used by the fsw_cevent objects will be freed at the end of the
  --   * callback invocation.  A callback should copy such data instead of storing
  --   * a pointer to it.
  --    

   type FSW_CEVENT_CALLBACK is access procedure
        (arg1 : access constant fsw_cevent;
         arg2 : unsigned;
         arg3 : System.Address)
   with Convention => C;  -- D:/a/msys64/mingw64/include/libfswatch/c/cevent.h:136

end cevent_h;

pragma Style_Checks (On);
pragma Warnings (On, "-gnatwu");
