"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.rawAesEncryptedPartsFactory = exports.rawAesEncryptedDataKeyFactory = void 0;
/* For raw AES keyrings the required wrapping information is stored in an EncryptedDataKey.
 * KeyNamespace (identifies the keyring "class"),
 * KeyName (identifies this specific keyring, like a KMS CMK ARN)
 *
 * {
 *   providerId: KeyNamespace
 *   providerInfo: utf8Encode(KeyName + TagLengthBits uInt32BE + IVLength uInt32BE + iv)
 *   encryptedDataKey: wrapped key + authTag
 * }
 *
 * The AAD (encryption context) is the same as the message.
 */
const serialize_1 = require("@aws-crypto/serialize");
const material_management_1 = require("@aws-crypto/material-management");
function rawAesEncryptedDataKeyFactory(toUtf8, fromUtf8) {
    return { rawAesEncryptedDataKey };
    function rawAesEncryptedDataKey(keyNamespace, keyName, iv, ciphertext, authTag) {
        const ivLength = iv.byteLength;
        const authTagBitLength = authTag.byteLength * 8;
        const encryptedDataKey = (0, serialize_1.concatBuffers)(ciphertext, authTag);
        const providerId = keyNamespace;
        const rawInfo = (0, serialize_1.concatBuffers)(fromUtf8(keyName), (0, serialize_1.uInt32BE)(authTagBitLength), (0, serialize_1.uInt32BE)(ivLength), iv);
        const providerInfo = toUtf8(rawInfo);
        return new material_management_1.EncryptedDataKey({
            encryptedDataKey,
            providerId,
            providerInfo,
            rawInfo,
        });
    }
}
exports.rawAesEncryptedDataKeyFactory = rawAesEncryptedDataKeyFactory;
function rawAesEncryptedPartsFactory(fromUtf8) {
    return { rawAesEncryptedParts };
    function rawAesEncryptedParts(suite, keyName, { encryptedDataKey, rawInfo }) {
        /* Precondition: rawInfo must be a Uint8Array. */
        if (!(rawInfo instanceof Uint8Array))
            throw new Error('Malformed Encrypted Data Key.');
        // see above for format, slice off the "string part"
        rawInfo = rawInfo.slice(fromUtf8(keyName).byteLength);
        /* Uint8Array is a view on top of the underlying ArrayBuffer.
         * This means that raw underlying memory stored in the ArrayBuffer
         * may be larger than the Uint8Array.  This is especially true of
         * the Node.js Buffer object.  The offset and length *must* be
         * passed to the DataView otherwise I will get unexpected results.
         */
        const dataView = new DataView(rawInfo.buffer, rawInfo.byteOffset, rawInfo.byteLength);
        /* See above:
         * uInt32BE(authTagBitLength),uInt32BE(ivLength), iv
         */
        const tagLengthBits = dataView.getUint32(0, false); // big endian
        const ivLength = dataView.getUint32(4, false); // big endian
        /* Precondition: The ivLength must match the algorith suite specification. */
        (0, material_management_1.needs)(ivLength === suite.ivLength, 'Malformed providerInfo');
        /* Precondition: The tagLength must match the algorith suite specification. */
        (0, material_management_1.needs)(tagLengthBits === suite.tagLength, 'Malformed providerInfo');
        /* Precondition: The byteLength of rawInfo should match the encoded length. */
        (0, material_management_1.needs)(rawInfo.byteLength === 4 + 4 + ivLength, 'Malformed providerInfo');
        const tagLength = tagLengthBits / 8;
        /* Precondition: The encryptedDataKey byteLength must match the algorith suite specification and encoded length. */
        (0, material_management_1.needs)(encryptedDataKey.byteLength === tagLength + suite.keyLengthBytes, 'Malformed providerInfo');
        const iv = rawInfo.slice(-ivLength);
        const authTag = encryptedDataKey.slice(-tagLength);
        const ciphertext = encryptedDataKey.slice(0, -tagLength);
        return { authTag, ciphertext, iv };
    }
}
exports.rawAesEncryptedPartsFactory = rawAesEncryptedPartsFactory;
//# sourceMappingURL=data:application/json;base64,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