"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.create = create;
const language_core_1 = require("@vue/language-core");
const volar_service_css_1 = require("volar-service-css");
const utils_1 = require("../utils");
function create() {
    const base = (0, volar_service_css_1.create)({ scssDocumentSelector: ['scss', 'postcss'] });
    return {
        ...base,
        create(context) {
            const baseInstance = base.create(context);
            const { 'css/languageService': getCssLs, 'css/stylesheet': getStylesheet, } = baseInstance.provide;
            return {
                ...baseInstance,
                async provideDiagnostics(document, token) {
                    let diagnostics = await baseInstance.provideDiagnostics?.(document, token) ?? [];
                    if (document.languageId === 'postcss') {
                        diagnostics = diagnostics.filter(diag => diag.code !== 'css-semicolonexpected'
                            && diag.code !== 'css-ruleorselectorexpected'
                            && diag.code !== 'unknownAtRules');
                    }
                    return diagnostics;
                },
                /**
                 * If the position is within the virtual code and navigation is enabled,
                 * skip the CSS navigation feature.
                 */
                provideReferences(document, position) {
                    if (isWithinNavigationVirtualCode(document, position)) {
                        return;
                    }
                    return worker(document, (stylesheet, cssLs) => {
                        return cssLs.findReferences(document, position, stylesheet);
                    });
                },
                provideRenameRange(document, position) {
                    if (isWithinNavigationVirtualCode(document, position)) {
                        return;
                    }
                    return worker(document, (stylesheet, cssLs) => {
                        return cssLs.prepareRename(document, position, stylesheet);
                    });
                },
            };
            function isWithinNavigationVirtualCode(document, position) {
                const info = (0, utils_1.resolveEmbeddedCode)(context, document.uri);
                if (!info?.code.id.startsWith('style_')) {
                    return false;
                }
                const block = info.root.sfc.styles.find(style => style.name === info.code.id);
                if (!block) {
                    return false;
                }
                let script;
                for (const [key, value] of info.script.generated.embeddedCodes) {
                    if (key.startsWith('script_')) {
                        script = value;
                        break;
                    }
                }
                if (!script) {
                    return false;
                }
                const offset = document.offsetAt(position) + block.startTagEnd;
                for (const { sourceOffsets, lengths, data } of script.mappings) {
                    if (!sourceOffsets.length || !(0, language_core_1.isRenameEnabled)(data)) {
                        continue;
                    }
                    const start = sourceOffsets[0];
                    const end = sourceOffsets.at(-1) + lengths.at(-1);
                    if (offset >= start && offset <= end) {
                        return true;
                    }
                }
                return false;
            }
            function worker(document, callback) {
                const cssLs = getCssLs(document);
                if (!cssLs) {
                    return;
                }
                return callback(getStylesheet(document, cssLs), cssLs);
            }
        },
    };
}
//# sourceMappingURL=css.js.map