/**
 * grouptoken.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "grouptoken.h"

#include "configurationview.h"
#include "treeitems.h"
#include "entries.h"

#include <kiconloader.h>
#include <kurl.h>
#include <kdebug.h>

using namespace KConfigEditor;

#if 0
static QString nextPartOfURL( const KURL &base, const KURL &full )
{
    QString fullPath = full.path();
    QString basePath = base.path();


    QString extraPath = fullPath.remove( basePath );

    int start = 0;
    if ( extraPath.startsWith( "/" ) )
        start = 1;

    int sep = extraPath.find( "/", start );

    extraPath = extraPath.mid( start, sep );

    return extraPath;
}
#endif

GroupToken::GroupToken( const QString &protocol, const QString &name )
    : Token( KURL() ), m_parent( 0 ), m_propagate( true )
{
    m_url.setProtocol( protocol );
    m_url.addPath( name );

    m_modified = false;
    m_modifiedEntries.setAutoDelete( false ); //m_childDict holds them
    m_sessionChanges.setAutoDelete( false );

    m_childDict.setAutoDelete( true );
    m_entryDict.setAutoDelete( true );
    m_removedEntries.setAutoDelete( true );
}

GroupToken::GroupToken( GroupToken* parent,  const QString &name, bool readOnly )
    : Token( parent->url(), readOnly ), m_parent( parent )
{
    if ( name.startsWith( "/" ) )
        m_url.setPath( name );//we got a full path
    else
        m_url.addPath( name );//we got a name
}

GroupToken::~GroupToken()
{
}

EntryToken*
GroupToken::entry( const QString &name ) const
{
    return m_entryDict[name];
}

EntryToken*
GroupToken::entryContaining( const QString &str ) const
{
    QDictIterator<EntryToken> itr( m_entryDict );

    while( itr.current() ) {
        if ( itr.current()->contains( str ) )
            return itr.current();
        ++itr;
    }
    return 0;
}

bool
GroupToken::removeEntry( const QString& name )
{
    EntryToken *entry = m_entryDict.take( name );
    entryRemoved( this, entry );
    return entry;
}

EntryToken*
GroupToken::createEntry( const QString &name, const QString &typeName,
                         bool hidden )
{
    EntryToken* token = 0;
    EntryToken::Type type = Utils::stringToType( typeName );

    setReadOnly( false );

    switch( type ) {
    case EntryToken::String:
        token = new StringEntry( this, name, hidden );
        break;
    case EntryToken::Password:
        token = new StringEntry( this, name, hidden );
        break;
    case EntryToken::StringList:
        token = new StringListEntry( this, name, hidden );
        break;
    case EntryToken::Font:
        token = new FontEntry( this, name, hidden );
        break;
    case EntryToken::Rect:
        token = new RectEntry( this, name, hidden );
        break;
    case EntryToken::Size:
        token = new SizeEntry( this, name, hidden );
        break;
    case EntryToken::Color:
        token = new ColorEntry( this, name, hidden );
        break;
    case EntryToken::Point:
        token = new PointEntry( this, name, hidden );
        break;
    case EntryToken::Int:
        token = new IntEntry( this, name, hidden );
        break;
    case EntryToken::UInt:
        token = new UIntEntry( this, name, hidden );
        break;
    case EntryToken::Bool:
        token = new BoolEntry( this, name, hidden );
        break;
    case EntryToken::Double:
        token = new DoubleEntry( this, name, hidden );
        break;
    case EntryToken::DateTime:
        token = new DateTimeEntry( this, name, hidden );
        break;
    case EntryToken::Int64:
        token = new Int64Entry( this, name, hidden );
        break;
    case EntryToken::UInt64:
        token = new UInt64Entry( this, name, hidden );
        break;
    case EntryToken::IntList:
        token = new IntListEntry( this, name, hidden );
        break;
    case EntryToken::Enum:
        token = new EnumEntry( this, name, hidden );
        break;
    case EntryToken::Path:
        token = new PathEntry( this, name, hidden );
        break;
    case EntryToken::Invalid:
    default:
        kdDebug()<<"Wrong type with "<< typeName <<endl;
        break;
    }

    if ( token ) {
        m_entryDict.insert( token->url().path(), token );
        entryModified( this, token );
    }

    return token;
}

GroupToken*
GroupToken::createGroup( const QString &name )
{
    GroupToken *group = new GroupToken( this, name, readOnly() );
    m_childDict.insert( group->url().path(), group );
    return group;
}

GroupToken*
GroupToken::group( const QString &group ) const
{
    KURL url( this->url() );
    url.addPath( group );
    return m_childDict[ url.path() ];
}


void
GroupToken::entryModified( GroupToken *group, EntryToken *entry )
{
    if ( !m_propagate )
        return;

    m_modified = true;

    QString url = entry->url().path();
    if ( isTopLevel() ) {
        if ( !m_modifiedEntries[ url ] )
            m_modifiedEntries.insert( url, entry );
        if ( !m_sessionChanges[ url ] )
            m_sessionChanges.insert( url, entry );
    }
    else
        m_view->setPixmap( 0, SmallIcon( "save_all" ) );

    if ( group && group != this ) {
        if ( !m_modifiedGroups[ group->url().path() ] )
            m_modifiedGroups.insert( group->url().path(), group );
    }

    if ( m_parent )
        m_parent->entryModified( this, entry );
}

void
GroupToken::entryRemoved( GroupToken *group, EntryToken *entry )
{
    if ( !m_propagate )
        return;

    m_modified = true;

    QString url = entry->url().path();
    if ( isTopLevel() ) {
        if ( !m_removedEntries[ url ] )
            m_removedEntries.insert( url, entry );
    } else
        m_view->setPixmap( 0, SmallIcon( "save_all" ) );

    if ( group && group != this ) {
        if ( !m_modifiedGroups[ group->url().path() ] )
            m_modifiedGroups.insert( group->url().path(), group );
    }

    if ( m_parent )
        m_parent->entryRemoved( this, entry );
}

/*
 * Pruning is a top-bottom operatation. we start at the toplevel
 * tokens and move down.
 * It's the exact oposite of propagating changes which happens
 * in a bottom-up mode (changes in children are propagated up).
 */
void
GroupToken::prunePendingChanges()
{
    m_modified = false;
    m_view->setPixmap( 0, QPixmap() );

    if ( isTopLevel() ) {
        QDictIterator<EntryToken> modItr( m_modifiedEntries );
        while ( modItr.current() ) {
            modItr.current()->prunePendingChanges();
            ++modItr;
        }
        m_modifiedEntries.clear();
    }

    QDictIterator<GroupToken> grpItr( m_modifiedGroups );
    while ( grpItr.current() ) {
        grpItr.current()->prunePendingChanges();
        ++grpItr;
    }

    m_modifiedGroups.clear();
    m_removedEntries.clear();
}

void
GroupToken::propagateChanges( bool doPropagate )
{
    m_propagate = doPropagate;
}

void
GroupToken::attach( ConfigurationView *confView )
{
    ConfItem *parentItem = 0;

    if ( m_parent )
        parentItem = m_parent->view();
    else
        parentItem = confView->protocolItem( m_url.protocol() );

    m_view = new GroupItem( parentItem, this );
    if ( isTopLevel() )
        static_cast<GroupItem*>( m_view )->updateIcon();
}

void GroupToken::restoreToken( const KURL &entry, QStringList &paths )
{
    KURL url( this->url() );
    url.addPath( paths.front() );

    EntryToken *token = m_entryDict[ url.path() ];
    if ( token ) {
        token->restoreToken( url, paths );
    } else {
        paths.pop_front();
        GroupToken *group = m_childDict[ url.path() ];
        if ( group ) {
            group->restoreToken( entry, paths );
        } else {
            kdWarning()<<"Error at \""<<url.path()<<"\""<<endl;
        }
    }
}

void GroupToken::entryRestored( GroupToken *group, EntryToken *entry )
{
    if ( !m_propagate )
        return;

    QString url = entry->url().path();
    if ( isTopLevel() ) {
        m_modifiedEntries.remove( url );
        m_sessionChanges.remove( url );
    }

    if ( group && group != this )
        m_modifiedGroups.remove( group->url().path() );

    if ( m_modifiedEntries.isEmpty() && m_modifiedGroups.isEmpty() ) {
        m_view->setPixmap( 0, QPixmap() );
        m_modified = true;
    }

    if ( m_parent )
        m_parent->entryRestored( this, entry );
}
