/***************************************************************************
                          kpldoc.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPLDOC_H
#define KPLDOC_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qdatetime.h>
#include <qobject.h>
#include <qptrlist.h>
#include <qstringlist.h>
#include <kio/job.h>
#include <kurl.h>
#include "kplitem.h"
#include "kplnamespace.h"

class QComboBox;
class QHBox;
class QLineEdit;
template<class> class QValueList;
class KFileDialog;
class Kpl;
class ArrayItem;
class FrameItem;
class FunItem;

/**
  * Kpl model class.
  * Provides methods for reading, writing and scaling of plot data.
  * @author Werner Stille
  */
class KplDoc : public QObject
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent QObject.
   */
  KplDoc(QObject* parent);
  /** Destructor. */
  ~KplDoc();
  /** Types of files */
  enum FileType{Unknown, Data, Plot};
  /** Initializes the document generally. */
  void newDocument();
  /**
   * Saves the document under URL.
   * @param url URL of file.
   * @param items list of plot items.
   * @param abs save absolute URL when true.
   * @return true for success.
   */
  bool saveDocument(const KURL& url, QPtrList<KplItem>* items, bool abs = true);
  /**
   * Sets the modified flag for the document after a modifying action on the
   * view connected to the document.
   * @param m true for document modified after last save action.
   */
  inline void setModified(bool m = true) { modified = m; };
  /**
   * Returns if the document is modified or not. Use this to determine if your
   * document needs saving by the user on closing.
   * @return true for document modified after last save action.
   */
  inline bool isModified() const { return modified; };
  /**
   * Sets the current directory path.
   * @param url current path.
   */
  void setCurrentDir(const KURL& url);
  /**
   * Returns the current path.
   * @return current directory path.
   */
  inline const QString& currentDir() const { return m_dir; };
  /**
   * Sets the URL of the document.
   * @param url URL.
   */
  void setURL(const KURL& url);
  /**
   * Returns the URL of the document.
   * @return URL.
   */
  inline const KURL& URL() const { return docURL; };
  /**
   * Returns the modification time of the URL of the document.
   * @return modification date and time.
   */
  inline const QDateTime& URLTime() const { return m_time; };
  /**
   * Returns the type of current document file.
   * @return file type (Plot or Data).
   */
  inline FileType fileType() const { return fType; };
  /**
   * Set the type of current document file.
   * @param iType file type (Plot or Data).
   */
  inline void setFileType(FileType iType) { fType = iType; };
  /**
   * Appends or inserts new item to list and initializes structure.
   * @param ityp type of item. One of the following:
   * Frame, Array, Function, ParFunction, Legend, Arrow.
   * @param index index of item to insert before:
   */
  void newItem(KplItem::ItemTypes ityp, int index = -1);
  /**
   * Reads file and initializes plot items.
   * @param url URL of the file.
   * @param idec 0 for decimal point, 1 for comma.
   * @param type one of the following: Unknown, Data, Plot.
   * @return true for success.
   */
  bool openDocument(const KURL& url, int idec, FileType type);
  /**
   * Restores items for a undo or redo action.
   * @param undo true for undo.
   * @return true for success.
   */
  bool restoreItems(bool undo = true);
  /**
    * Checks if undo action is possible.
    * @return true if possible.
    */
  bool undoAllowed() const;
  /**
    * Checks if redo action is possible.
    * @return true if possible.
    */
  bool redoAllowed() const;
  /**
   * Backup items for undo or redo actions if possible.
   * @param list true when item list should be uddated.
   */
  void backupItems(bool list = true);
  /**
   * Deletes plot item.
   * @param i item index.
   */
  void deleteItem(int i);
  /**
   * Moves plot item.
   * @param is source item index.
   * @param id destination item index.
   */
  void moveItem(int is, int id);
  /**
   * Reorders item list.
   * @param list new index list.
   */
  void orderItems(QValueList<int>& list);
  /**
   * Sets option structure.
   * @param opt pointer to option structure.
   */
  inline void setOptions(KplNamespace::AutoStruct* opt) { aut = *opt; };
  /**
   * Returns option structure.
   * @return pointer to option structure.
   */
  inline KplNamespace::AutoStruct* options() { return &aut; };
  /**
   * Returns separator for output.
   * @return separator.
   */
  inline const QString& separator() const {return separators[aut.iSep]; };
  /**
   * Returns plot item list.
   * @return pointer to item list.
   */
  inline QPtrList<KplItem>* items() { return &itd; };
  /**
   * Returns temporary filename.
   * @return filename.
   */
  QString tmpFile() const;
  /**
   * Copies temporary file to destination.
   * @param src source.
   * @param dest destination.
   */
  void copyTmp(const QString& src, const KURL& dest);
  /**
   * Returns formatted double value string.
   * @param val value.
   */
  inline const QString number(double val) const
    { return QString::number(val, aut.format, aut.prec); };
  /**
   * Retrieves URL for saving.
   * @param parent parent widget.
   * @param url KURL receiving the chosen file URL.
   * @param filter file name filter for file dialog.
   * @return true when file can be saved.
   */
  bool getWriteURL(QWidget* parent, KURL& url, const QString& filter);
  /**
   * Retrieves URL for reading.
   * @param idec 0 for decimal point, 1 for comma.
   * @param filename filename for dialog initialization.
   * @param filter filter for files to be displayed.
   * @return KURL of chosen file.
   */
  KURL getReadURL(int& idec, const QString& filename = QString::null,
                  const QString& filter = "*.dat *.DAT\n*");
  /**
   * Retrieves URLs for reading.
   * @param idec 0 for decimal point, 1 for comma.
   * @return KURL list of chosen files.
   */
  KURL::List getReadURLs(int& idec);
  /**
   * Opens a file selection dialog.
   * @param e QLineEdit widget to display the filename.
   */
  void getFile(QLineEdit* e);
  /**
   * Loads a function parameter file.
   * @param url file URL.
   * @param parent parent widget.
   * @param p double array for parameters.
   * @param idec 0 for decimal point, 1 for comma.
   */
  bool loadPar(const KURL& url, QWidget* parent, double* p, int idec);
  /**
   * Opens a dialog to select a function parameter file.
   * @param parent parent widget.
   * @param p double array for parameters.
   */
  void getPar(QWidget* parent, double* p);
  /**
   * Saves parameters in a file.
   * @param parent parent widget.
   * @param url file URL.
   * @param p double array with parameters.
   * @param err double array with parameters (0 for no error output).
   * @return true for success.
   */
  bool saveParameter(QWidget* parent, const KURL& url, double* p, double* err);
  /**
   * Opens a dialog to save function parameters in a file.
   * @param parent parent widget.
   * @param p double array with parameters.
   */
  void saveFunPar(QWidget* parent, double* p);

signals:
  /**
   * Emitted when changes may require updating the menus and repainting
   * the page view.
   * @param plot true when plot should be uddated.
   * @param list true when item list should be uddated.
   */
  void modelChanged(bool plot, bool list);

protected:
  /** Deallocates storage of all plot items. */
  void freeAllItems();
  /**
   * Sets array and frame item elements by autoscaling.
   * @param fd pointer to frame item.
   * @param ad pointer to array item.
   * @return true for error
   */
  void autoFit(ArrayItem* ad, FunItem* fd);
  /**
   * Inserts a widget for decimal symbol choice into a file dialog.
   * @param idec 0 for decimal point, 1 for comma.
   * @param dlg file dialog.
   * @param hbox horizontal box.
   * @param mode mode of file dialog.
   * @return combobox.
   */
  QComboBox* decimalWidget(int idec, KFileDialog* dlg, QHBox* hbox,
                           unsigned mode);
  /** The modified flag of the current document. */
  bool modified;
  QWidget* app;
  FileType fType;
  KURL docURL;
  QString m_dir;
  QDateTime m_time;
  QPtrList<KplItem> itd;
  QPtrList<QPtrList<KplItem> > itb;
  KplNamespace::AutoStruct aut;
  QStringList separators;
  double pLast[KPL_NPMAX], pErr[KPL_NPMAX];
  double chisq;

  friend class Kpl;
  friend class FitDlg;
  friend class SplFitDlg;

protected slots:
  /** Deletes KFM object and temporary file for upload. */
  void slotIOfinished(KIO::Job*);
};

#endif // KPLDOC_H
