/***************************************************************************
                          funitem.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Aug 29 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FUNITEM_H
#define FUNITEM_H

#include <qmemarray.h>
#include <kurl.h>
#include "kplitem.h"

class QLibrary;
template<class> class QPtrList;
class ArrayItem;
class LMFit;

/**
  * Function item class. Class for function items showing ordinary functions
  * by lines or markers.
  * @author Werner Stille
  */
class FunItem : public ScaledItem
{
public:
  /** Constructor */
  FunItem();
  /** Copy constructor */
  FunItem(const FunItem& f);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  FunItem(KplNamespace::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    * @param url URL of plot file.
    */
  FunItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut, const KURL& url);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param fillStyle > 0 for filled polygon representation.
    * @param symb 0 for circles, < 0 for different markers,
    * >0 for different line types.
    * @param color color for function plot.
    * @param fx normalization factor for x values.
    * @param fy normalization factor for y values.
    * @param xmin minimum argument value.
    * @param xmax maximum argument value.
    * @param dx argument stepsize.
    * @param name function name.
    * @param url URL of the function library.
    * @param relSize relative size of lines or symbols.
    */
  FunItem(bool active, int fillStyle, int symb, const QString& color,
          double fx, double fy, double xmin, double xmax, double dx,
          const QString& name, const KURL& url, double relSize = 1.0);
  /** Destructor */
  virtual ~FunItem();
  /** Assignment operator */
  FunItem& operator=(const FunItem& f);
  virtual ItemTypes iType() const;
  virtual void draw(KplGraph* g);
#ifndef KPL_CLASSES_ONLY
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  virtual void setText(QListViewItem* it, bool* arrays, bool* funcs) const;
  virtual int editItem(QWidget* parent, KplDoc* m, int i);
  virtual void exportTable(QTextStream& ts, KplDoc* m) const;
#endif
  virtual KplItem* copy() const;
  virtual void expoItem(int* iex, int* iey, double* fx, double* fy) const;
  virtual void minMax(double* xmi, double* xma, double* ymi, double* yma) const;
  virtual void setPar(int i, double value, bool yFun);
  /**
   * Performs fit of parameters to a data array.
   * @param ad array item.
   * @param bFit array containing true values for parameters to be fitted.
   * @param err array of structures containing the data error settings.
   * @param nonLin true for nonlinear fit.
   * @param tol tolerance value for stopping nonlinear fit iterations.
   * @param itMax maximum number of nonlinear fit iterations.
   * @param chisq chi-square.
   * @param corr array for parameter correlation matrix.
   * @param pErr array for parameter erros.
   * @param avgErr average error.
   * @return error code.
   */
  int fit(ArrayItem* ad, bool* bFit, KplNamespace::DataErrorStruct* err = 0,
          bool nonLin = true, double tol = 1.0e-14, int itMax = 100,
          double* chisq = 0, double* corr = 0, double* pErr = 0,
          double* avgErr = 0);
  /**
   * Performs fit of parameters of a set of functions to a set of arrays.
   * @param ad array item list.
   * @param fd function item list.
   * @param bFit array containing true values for parameters to be fitted.
   * @param p parameter array.
   * @param fvec array for weighted residuals.
   * @param sig array for data errors.
   * @param err array of structures containing the data error settings.
   * @param nonLin true for nonlinear fit.
   * @param tol tolerance value for stopping nonlinear fit iterations.
   * @param itMax maximum number of nonlinear fit iterations.
   * @param lm object for Levenberg-Marquardt fits.
   * @param chisq chi-square.
   * @param corr array for parameter correlation matrix.
   * @param pErr array for parameter erros.
   * @param avgErr average error.
   * @param parent parant object for lm.
   * @return error code.
   */
  static int fit(QPtrList<ArrayItem>* ad, QPtrList<FunItem>* fd, bool* bFit,
                 double* p, QMemArray<double>* fvec = 0,
                 QMemArray<double>* sig = 0,
                 KplNamespace::DataErrorStruct* err = 0, bool nonLin = true,
                 double tol = 1.0e-14, int itMax = 100, LMFit* lm = 0,
                 double* chisq = 0, double* corr = 0, double* pErr = 0,
                 double* avgErr = 0, QObject* parent = 0);
  /**
   * Gets module handle and address of a shared library function.
   * @param path path of the library.
   * @param name function name.
   * @param lib pointer to library.
   * @param fkt pointer to function address.
   * @return true for success.
   */
  static bool getFuncAddr(const QString& path, const QString& name,
                          QLibrary** lib,
                          double (**fkt)(double, const double*));

protected:
  /**
   * Initializes several elements.
   */
  void init();
  /**
   * Calculates function value table.
   * @param logx true for logarithmic steps
   * @return number of rows.
   */
  int calcTable(bool logx) const;
  mutable bool logxo;
  double tmin, tmax, dt;
  mutable double tmino, tmaxo, dto;
  double (*fkty)(double, const double*);
  mutable double (*fktyo)(double, const double*);
  double py[KPL_NPMAX];
  mutable double pyo[KPL_NPMAX];
  mutable QMemArray<double> tv, yv;
  QString namey;
  KURL pathy;
  QLibrary* liby;

  friend class FitDlg;
  friend class FuncDlg;
  friend class LMFit;
  friend class Kpl;
  friend class KplDoc;
  friend class KplView;
};

/**
  * Parametric function item class.
  * @author Werner Stille
  */
class ParFunItem : public FunItem
{
public:
  /** Constructor */
  ParFunItem();
  /** Copy constructor */
  ParFunItem(const ParFunItem& f);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  ParFunItem(KplNamespace::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    * @param url URL of plot file.
    */
  ParFunItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut,
             const KURL& url);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param fillStyle > 0 for filled polygon representation.
    * @param symb 0 for circles, < 0 for different markers,
    * >0 for different line types.
    * @param color color for function plot.
    * @param fx normalization factor for x values.
    * @param fy normalization factor for y values.
    * @param tmin minimum argument value.
    * @param tmax maximum argument value.
    * @param dt argument stepsize.
    * @param namex x function name.
    * @param urlX URL of the x function library.
    * @param namey y function name.
    * @param urlY URL of the y function library.
    * @param relSize relative size of lines or symbols.
    */
  ParFunItem(bool active, int fillStyle, int symb, const QString& color,
             double fx, double fy, double tmin, double tmax, double dt,
             const QString& namex, const KURL& urlX, const QString& namey,
             const KURL& urlY, double relSize = 1.0);
  /** Destructor */
  virtual ~ParFunItem();
  /** Assignment operator */
  ParFunItem& operator=(const ParFunItem& f);
  virtual ItemTypes iType() const;
  virtual void draw(KplGraph* g);
#ifndef KPL_CLASSES_ONLY
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  virtual void setText(QListViewItem* it, bool* arrays, bool* funcs) const;
  virtual int editItem(QWidget* parent, KplDoc* m, int i);
  virtual void exportTable(QTextStream& ts, KplDoc* m) const;
#endif
  virtual KplItem* copy() const;
  virtual void expoItem(int* iex, int* iey, double* fx, double* fy) const;
  virtual void minMax(double* xmi, double* xma, double* ymi, double* yma) const;
  virtual void setPar(int i, double value, bool yFun);

protected:
  /**
   * Initializes several elements.
   */
  void init();
  /**
   * Calculates function value table.
   * @return number of rows.
   */
  int calcTable() const;
  double (*fktx)(double, const double*);
  mutable double (*fktxo)(double, const double*);
  double px[KPL_NPMAX];
  mutable double pxo[KPL_NPMAX];
  mutable QMemArray<double> xv;
  QString namex;
  KURL pathx;
  QLibrary* libx;
  friend class ParFuncDlg;
  friend class Kpl;
};

#endif
