/***************************************************************************
                          kgraph.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2003 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KGRAPH_H
#define KGRAPH_H

#include <qpainter.h>
#include "kplgraph.h"

/**
  * Graphics class. Provides high level plotting functionality.
  * @author Werner Stille
  */
class KGraph : public KplGraph, public QPainter
{
public:
  /** Constructor */
  KGraph();
  /** Destructor */
  virtual ~KGraph();
  /**
   * Converts from integer units to centimeters.
   * @param p integer coordinates.
   * @param xcm x value in centimeters (output).
   * @param ycm y value in centimeters (output).
   * @return true fr point outside of data window.
   */
  bool i2cm(const QPoint& p, double* xcm, double* ycm) const;
  /**
   * Converts from integer units to user units.
   * @param p integer coordinates.
   * @param fxn normalization factor for x values.
   * @param fyn normalization factor for y values.
   * @param xr x value in user units (output).
   * @param yr y value in user units (output).
   */
  void i2r(const QPoint& p, double fxn, double fyn, double* xr,
           double* yr) const;
  /**
   * Converts from centimeters to integer units.
   * @param xcm x value in centimeters.
   * @param ycm y value in centimeters.
   * @return integer coordinates.
   */
  QPoint fromcm(double xcm, double ycm) const;
  /**
   * Converts from centimeters to user units.
   * @param xcm x value in centimeters.
   * @param ycm y value in centimeters.
   * @param fxn normalization factor for x values.
   * @param fyn normalization factor for y values.
   * @param xr x value in user units (output).
   * @param yr y value in user units (output).
   */
  virtual void cm2r(double xcm, double ycm, double fxn, double fyn,
                    double* xr, double* yr) const;
  /**
   * Sets current (logical) position.
   * @param x x value in user units.
   * @param y y value in user units.
   */
  void move(double x, double y);
  /**
   * Draws line from current to given position.
   * @param x x value in user units.
   * @param y y value in user units.
   */
  void draw(double x, double y);
  /**
   * Draws line between two points.
   * @param x1 x value of first point in user units.
   * @param y1 y value of first point in user units.
   * @param x2 x value of second point in user units.
   * @param y2 y value of second point in user units.
   */
  void line(double x1, double y1, double x2, double y2);
  /**
   * Initializes plot.
   * @param pd QPaintdevice.
   * @param offLeft scroll offset from left margin.
   * @param offTop scroll offset from top margin.
   */
  void plInit(const QPaintDevice* pd, int offLeft = 0, int offTop = 0);
  virtual void format(double x, double y);
  virtual void setColFrame(unsigned icol1);
  virtual void setColGrid(unsigned icol2);
  virtual void setColData(unsigned icol3);
  virtual void setCol(unsigned icol1, unsigned icol2, unsigned icol3);
  virtual double setRelSize(double r);
  virtual void Window(double x0, double xl, double y0, double yl);
  virtual void scale(double xl, double xr, double yu, double yo,
                     bool logx = false, bool logy = false);
  virtual void setDig(int nx, int ny);
  virtual void frame();
  virtual void raster(double xtic, double ytic, int intx, int inty,
                      int mode, double xOff = 0.0, double yOff = 0.0,
                      bool xBottom = true, bool xTop = false,
                      bool yLeft = true, bool yRight = false);
  virtual void setDir(double ang);
  virtual void textcm(const QString& str, double x, double y, int iorg,
                      QRect* bRect = 0);
  virtual void arrow(double x, double y, double dir, double len,
                     QRect* bRect = 0);
  virtual void arc(double x, double y, double w, double h, double a,
                   double alen, double dir = 0.0, QRect* bRect = 0);
  virtual void rectangle(double x, double y, double w, double h,
                         bool filled, double dir = 0.0, QRect* bRect = 0);
  virtual void ellipse(double x, double y, double w, double h,
                       bool filled, double dir = 0.0, QRect* bRect = 0);
  virtual void letH(const QString& str, bool xTop);
  virtual void letX(const QString& str, int iex, bool xTop);
  virtual void letY(const QString& str, int iex, bool yRight);
  virtual void setSymbol(int is);
  /**
   * Draws a circle.
   * @param x0 x value of center in centimeters.
   * @param y0 y value of center in centimeters.
   * @param r radius in centimeters.
   */
  void circle(double x0, double y0, double r);
  /**
   * Draws an arc.
   * @param x x value in centimeters.
   * @param y y value centimeters.
   * @param r radius in centimeters. r < 0: (x, y) = center,
   * r > 0: (x, y) = start point.
   * @param angst start angle in degrees.
   * @param angend end angle in degrees.
   */
  void arcrel(double x, double y, double r, double angst, double angend);
  /**
   * Increment logarithm for next major axis tick.
   * @param x logarithm of tick value.
   * @param f factor to get second tick in a decade.
   * @return mantissa of x.
   */
  static double incLog(double* x, double f);
  virtual void plArray(const double* x, const double* y,
                       double fx, double fy, int n, bool clip = true);
  virtual void plError(const double* x, const double* y, const double* e,
                       double fx, double fy, int n);
  virtual void resetDash();

protected:
  /**
   * Selects line type.
   * @param is line type (1 for full line).
   */
  void linetype(int is);
  /**
   * Plots axis text.
   * @param str text.
   * @param iex logarithm of normalization factor.
   * @param iax 1 for x axus, 2 for y axis.
   * @param oppsite true for nonstandard position of axis text
   */
  void letxy(const QString& str, int iex, int iax, bool opposite = false);
  /**
   * Converts from user units to integer units.
   * @param x x value in user units.
   * @param y y value in user units.
   * @param ix x value in integer units (output).
   * @param iy y value in integer units (output).
   */
  void r2i(double x, double y, int* ix, int* iy);
  /**
   * Plots text.
   * @param str text.
   * @param ix x value in integer units.
   * @param iy y value in integer units.
   * @param iorg 1: left aligned, 2: centered, 3: right aligned.
   * @param bRect bounding rectangle, calculated when non-null.
   */
  void text(const QString& str, int ix, int iy, int iorg, QRect* bRect = 0);
  int minx, maxx, miny, maxy, kxcur, kycur, ndigx, ndigy, minx0,
      miny0, pFactor;
  double textdir, scalx, scaly, xmin, xmax, ymin, ymax, xcur, ycur, dlen,
         dhite, xscal, yscal, relang, relsin, relcos, dcur, factor, fDpi;
  double dash[4];
  QPen pen1, pen2, pen3;
  static const char* face[];

  friend class KplApp;
  friend class FrameItem;
  friend class ThreeDItem;
  friend class LegendItem;
};

#endif
