/***************************************************************************
                          kplview.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qpaintdevicemetrics.h>
#include <qdragobject.h>
#include <qtimer.h>
#include "kplview.h"
#include "kpldoc.h"
#include "kpl.h"

KplView::KplView(KplApp* _parent, double f) :
 KplWidget(_parent), fZoom(f), hPos(0.0), vPos(1.0)
{
  doc = _parent->getDocument();
  fxn = fyn = 1.0;
  setFrameStyle(NoFrame);
  QScrollView::viewport()->setAcceptDrops(true);
  QScrollView::viewport()->setMouseTracking(true);
  eraseWidget();
}

KplView::~KplView()
{
}

#if (KDE_VERSION_MAJOR > 2) || (KDE_VERSION_MINOR > 1)
void KplView::print(KPrinter* printer) const
#else
void KplView::print(QPrinter* printer) const
#endif
{
  KplWidget::print(printer, doc->items());
}

void KplView::layout()
{
  QPaintDeviceMetrics pdm(this);
  hPos = horizontalScrollBar()->isVisible() ?
         double(horizontalScrollBar()->value()) /
         horizontalScrollBar()->maxValue() : 0.0;
  vPos = verticalScrollBar()->isVisible() ?
         double(verticalScrollBar()->value()) /
         verticalScrollBar()->maxValue() : 1.0;
  resizeContents(qRound(fZoom * doc->options()->xf * pdm.logicalDpiX() / 2.54),
                 qRound(fZoom * doc->options()->yf * pdm.logicalDpiY() / 2.54));
  QTimer::singleShot(0, this, SLOT(setScrollBars()));
}

void KplView::eraseWidget()
{
  doPaint(false);
}

void KplView::setZoom(double f)
{
  fZoom = f;
}

double KplView::zoom() const
{
  return fZoom;
}

void KplView::doPaint(bool paint)
{
  layout();
  QScrollView::viewport()->setCursor(waitCursor);
  pm.resize(contentsWidth(), contentsHeight());
  plInit(&pm);
  factor = fZoom;
  format(doc->options()->xf, doc->options()->yf);
  if (paint)
    drawItems(this, doc->items());
  end();
  repaintContents(0, 0, pm.width(), pm.height(), false);
  QScrollView::viewport()->setCursor(crossCursor);
}

void KplView::slotScrollLeft()
{
  horizontalScrollBar()->subtractLine();
}

void KplView::slotScrollRight()
{
  horizontalScrollBar()->addLine();
}

void KplView::slotScrollUp()
{
  verticalScrollBar()->subtractLine();
}

void KplView::slotScrollDown()
{
  verticalScrollBar()->addLine();
}

void KplView::slotScrollPageUp()
{
  verticalScrollBar()->subtractPage();
}

void KplView::slotScrollPageDown()
{
  verticalScrollBar()->addPage();
}

void KplView::slotScrollPageLeft()
{
  horizontalScrollBar()->subtractPage();
}

void KplView::slotScrollPageRight()
{
  horizontalScrollBar()->addPage();
}

void KplView::slotScrollHomeLeft()
{
  horizontalScrollBar()->setValue(0);
}

void KplView::slotScrollHomeDown()
{
  verticalScrollBar()->setValue(verticalScrollBar()->maxValue());
}

void KplView::slotScrollEndRight()
{
  horizontalScrollBar()->setValue(horizontalScrollBar()->maxValue());
}

void KplView::slotScrollEndUp()
{
  verticalScrollBar()->setValue(0);
}

void KplView::paintWidget()
{
  doPaint(true);
}

void KplView::setScrollBars()
{
  horizontalScrollBar()->setValue(hPos * horizontalScrollBar()->maxValue());
  verticalScrollBar()->setValue(vPos * verticalScrollBar()->maxValue());
}

void KplView::contentsMousePressEvent(QMouseEvent* e)
{
  if (e->button() == RightButton)
    emit rightButtonPressed();
  else
    if (e->button() == LeftButton) {
      for (KplItem* item = doc->items()->last(); item;
           item = doc->items()->prev()) {
        if (item->boundingRect().contains(e->pos())) {
          if (sel.contains(item))
            sel.remove(item);
          else
            sel.append(item);
          startPos = dragPos = e->pos();
          drawBoundingRect(item->boundingRect());
          emit mouseMoved(fromcm(item->xPos(), item->yPos()), false);
          break;
        }
      }
    }
}

void KplView::contentsMouseReleaseEvent(QMouseEvent* e)
{
  if (e->button() == LeftButton) {
    if (sel.count() && (!(e->state() & ControlButton))) {
      QPoint d = e->pos() - startPos;
      double snap = 0.05;
      double dx = snap * qRound((d.x() / (scalx * factor)) / snap);
      double dy = snap * qRound((d.y() / (scaly * factor)) / snap);
      if (dx || dy) {
        for (KplItem* it1 = sel.first(); it1; it1 = sel.next()) {
          it1->setPos(it1->xPos() + dx, it1->yPos() + dy);
          KplItem::ItemTypes typ = it1->iType();
          if ((typ == KplItem::Frame) || (typ == KplItem::Array3D) ||
              (typ == KplItem::Function3D))
            if (it1 != doc->items()->getLast())
              for (KplItem* itm = doc->items()->at(doc->items()->find(it1) + 1);
                   itm; itm = doc->items()->next()) {
                typ = itm->iType();
                if ((typ == KplItem::Frame) ||
                    (typ == KplItem::Array3D) ||
                    (typ == KplItem::Function3D))
                  break;
                if (itm->iType() >= KplItem::Legend)
                  itm->setPos(itm->xPos() + dx, itm->yPos() + dy);
              }
        }
        doc->setModified();
        doc->backupItems();
      } else {
        d = dragPos - startPos;
        for (KplItem* it1 = sel.first(); it1; it1 = sel.next()) {
          QRect r = it1->boundingRect();
          r.moveBy(d.x(), d.y());
          drawBoundingRect(r);
        }
      }
      sel.clear();
    }
  }
}

void KplView::contentsMouseMoveEvent(QMouseEvent* e)
{
  if (sel.count()) {
    if (!(e->state() & ControlButton)) {
      KplItem* it1 = sel.first();
      for (KplItem* itm = it1; itm; itm = sel.next()) {
        QRect r = itm->boundingRect();
        QPoint d = dragPos - startPos;
        r.moveBy(d.x(), d.y());
        drawBoundingRect(r);
        d = e->pos() - dragPos;
        r.moveBy(d.x(), d.y());
        drawBoundingRect(r);
      }
      dragPos = e->pos();
      QPoint d = e->pos() - startPos;
      emit mouseMoved(fromcm(d.x() / (scalx * factor) + it1->xPos(),
                             d.y() / (scaly * factor) + it1->yPos()), false);
    }
  } else
    if (QRect(contentsX(), contentsY(), visibleWidth(),
              visibleHeight()).contains(e->pos())) {
      bool showValues = false;
      for (KplItem* item = doc->items()->last(); item;
           item = doc->items()->prev()) {
        KplItem::ItemTypes typ = item->iType();
        if ((typ == KplItem::Frame) || (typ == KplItem::Array3D) ||
            (typ == KplItem::Function3D)) {
          showValues = (item->iType() == KplItem::Frame);
          break;
        }
      }
      emit mouseMoved(e->pos(), showValues);
    } else
      emit mouseLeaved();
}

void KplView::contentsDragEnterEvent(QDragEnterEvent* e)
{
  if (QUriDrag::canDecode(e))
    e->accept(QScrollView::viewport()->rect());
  else
    e->ignore(QScrollView::viewport()->rect());
}

void KplView::contentsDropEvent(QDropEvent* e)
{
  QStrList l;
  if (QUriDrag::decode(e, l)) {
    KURL::List list(QStringList::fromStrList(l));
    emit urlsDropped(&list);
    e->accept();
  } else
    e->ignore();
}

void KplView::drawBoundingRect(const QRect& r)
{
  begin(&pm);
  setRasterOp(NotROP);
  setPen(DashLine);
  drawRect(r);
  end();
  repaintContents(0, 0, pm.width(), pm.height(), false);
}
