/***************************************************************************
                          listedit.cpp  -  description
                             -------------------
    begin                : Fri Sep 24 1999
    copyright            : (C) 1999 by Friedrich W. H. Kossebau
    email                : Friedrich.W.H@Kossebau.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <iostream.h>
#include <qapplication.h>

#include "listedit.h"

ListEdit::ListEdit(QWidget *parent, const char *name ) 
  : QListView(parent,name),
    EditedColumn( 0 ), 
    EditMode( true ), 
    FontMetricsHeight( 0 )
{
  FontList.setAutoDelete( true );
	
  checkFontMetricHeight( font() );
	
	// initialize needed listview properties
	setSorting( -1 );   // noAutoSorting as user would loose control if list shuffles often
	setItemMargin( 2 ); // QLineEdit has a left margin of two pixels, so adjust the listmargin
	// initialize edit properties
	setEditMode( false ); // no editing in the beginning because of no items
	
	Header = header();
//	connect( this, SIGNAL( selectionChanged() ), this, SLOT( updateEditField() ) );	
	connect( Header, SIGNAL( sectionClicked(int) ), this, SLOT( slotSortingChanged( int ) ) );
}


/** creates the EditField */
void ListEdit::initEditField() 
{
	// initialize the Field which is used for editing
	EditField = new QLineEdit( viewport() );  // lays Editfield direct on moving viewport
	addChild( EditField );                    // needed by QListView (see doc) but doesn't work
	EditField->installEventFilter( this );		// to catch the key events which might change field
	EditField->setFrame( false );             // doesn't look nice to me with a border
  connect( EditField, SIGNAL( textChanged( const char * ) ), this, SLOT( setText( const char * ) ) );
	connect( Header, SIGNAL( sizeChange( int, int, int ) ), this, SLOT( updateEditField() ) );
	connect( this, SIGNAL( currentChanged ( QListViewItem * ) ), SLOT( updateEditField() ) );
}

/** well, it comes the time to die */
ListEdit::~ListEdit()
{
  delete EditField;
  
  // delete fonts
  VFontEntry *Entry;
  for( Entry = FontList.first(); Entry; Entry = FontList.next() )
  { 
    if( Entry->Font )
      delete Entry->Font;
  }
}


/** moves the EditField to the according place */
void ListEdit::updateEditField() 
{
  // check if updating makes sense
  if( !EditField ) return;
	
//	bool hadFocus = EditField->hasFocus(); // why that? so commented it out
	
	if (( childCount() == 0 )       // without a child no editing is possible
	    || ( currentItem() == 0 ))	// nothing selected to be edited
		return;

	// to avoid flickering 
	bool updatesWereEnabled = EditField->isUpdatesEnabled();
  EditField->setUpdatesEnabled( false );

  // now start to set up the needed properties...
  
	// set the new editing characteristics
	ItemIsChanging = true;
	EditField->setText( currentItem()->text( header()->mapToLogical( editedColumn() ) ) );
  ItemIsChanging = false;                                           
	EditField->setCursorPosition( cursorPos() );
	
	// set size of EditField to fit for actual edited item
	EditField->resize( columnWidth( editedColumn() )-1, currentItem()->height() );
	// set the font
	EditField->setFont( columnFont( editedColumn() ) );
	// and move it to take a seat in its place
	moveChild( EditField, header()->cellPos(editedColumn()), itemPos(currentItem()) );
//TODO: move sizeinformation to ListEditItem (margin, offset, ...)	
		
	// now makeup is complete, so go back to the party
  if( updatesWereEnabled ) EditField->setUpdatesEnabled( true );
  EditField->repaint();

//	if( hadFocus ) EditField->setFocus(); // why that? s.a.
}


/** passes text to list */
void ListEdit::setText( const char *Text )
{
	if ( currentItem() )
		currentItem()->setText( header()->mapToLogical( editedColumn() ), Text );
  if ( !ItemIsChanging )
	  emit justModified();
}


/**  */
bool ListEdit::eventFilter ( QObject * o, QEvent * e )
{
  if ( o == EditField ) {
		if ( e->type() == Event_KeyPress ) {    // key press
			QKeyEvent *k = (QKeyEvent*)e;
			switch (k->key()) {
			case Key_Left:
					if ( EditField->cursorPosition() == 0 ) {
						changeColumn( ToLeft, -1 );
						return true;
					}
					else break;
			case Key_Right:
					if ( EditField->cursorPosition() == int( QString( EditField->text() ).length() ) ) {
						changeColumn( ToRight, 0 );
						return true;
					}
					else break;
			case Key_Return:
			case Key_Enter:
			case Key_Tab:
					return handleEnterPress( k );
			case Key_Up:
			case Key_Down:
					setCursorPos( EditField->cursorPosition() );
					return false; // let QListView handle the display of current item
					/* EditField->fontMetrics().width( EditField()->text().left( cursor ) )
						 oder falls schon offset (nicht ermittelbar ;-( da, allein Cursorposition */
			default: 
					break;
			}
			return FALSE;
		}
		return FALSE;                           // standard event processing
	}
  
	return QListView::eventFilter( o, e );
}

/** finds the next editable column 
    Direction is +1 or -1
    NewCursorPos is -1 for end else as is
    returns FALSE if nothing found */
bool ListEdit::changeColumn ( const int Direction, const int NewCursorPos = -1 ){

	if( Direction == +1 ) {
		if( editedColumn()  < header()->count()-1 ) 
			EditedColumn++;   // TODO: no direct account to EditedColumn
		else
			return false;
		}	
	else
		if( editedColumn() > 0 )
			EditedColumn-- ;	// TODO: no direct account to EditedColumn
		else
			return false;

	setCursorPos(	NewCursorPos );
		
	updateEditField();	
	return true;
}

/**  */
void ListEdit::resize ( int w, int h ) 
{
  QListView::resize ( w, h );
  updateEditField(); 
}


/** switches Editing on and off
    who would have guessed ;-) */
void ListEdit::setEditMode ( const bool Editing = true ) {

	// is there anything to be edited
	if( !firstChild() ) 
	  EditMode = false;  
	else
	{
  	if ( EditMode == Editing ) return;	// nothing to do	
	
  	EditMode = Editing;
  }
	
	if( EditMode ) 
	{
		if( !EditField ) initEditField(); 
		setEditedColumn( editedColumn() );        
// TODO: check rightmax Column...		
		if( !currentItem() ) setCurrentItem( firstChild() );
		
		viewport()->setCursor( ibeamCursor );	
		updateEditField();
		EditField->show();
		EditField->setFocus();
	}
	else 
	{
		viewport()->setCursor( arrowCursor );	
		if( EditField ) EditField->hide();
	}
}

/**  */
void ListEdit::mousePressEvent ( QMouseEvent *e ){

	if ( isEditMode() ) 
	{	
		QMouseEvent *MouseEvent = e;		
		if ( MouseEvent->button() == LeftButton ) 
		{
			int NewColumn = header()->cellAt( MouseEvent->x() );
			
			if ( NewColumn != -1 ) 
			{
				setEditedColumn( NewColumn );
  			
  			QListViewItem *ClickedItem = itemAt( MouseEvent->pos() );
  			if ( ClickedItem ) 
  			{
				  // move EditField to new object of interest
					setCurrentItem( ClickedItem );
  			}
  			else
  			{
					setCurrentItem( lastItem() );
  			}

 				EditField->setUpdatesEnabled( false );
				updateEditField();
 				
 				// create Mouseevent with according coordinates to hand it to EditField					
 				QPoint Point = MouseEvent->pos();							
 				Point.setX( Point.x() - header()->cellPos( NewColumn ) );
  				QMouseEvent* NewMouseEvent = new QMouseEvent ( Event_MouseButtonPress, Point, LeftButton, LeftButton );
 				QApplication::sendEvent( EditField, NewMouseEvent );  //TODO: fix that mouseclick is not treated like a normal one e.g. doubleclick
					
 				EditField->setUpdatesEnabled( true );
        EditField->repaint();
 				ensureItemVisible( ClickedItem );
 				return;
			}	
		}
/*		else if ( MouseEvent->button() == RightButton ) 
		{
			QPoint p = mapToGlobal( _mouse->pos() );
	
			select( 0L, false );
	QStrList list;
	list.append( manager->getURL() );
	slotPopupMenu( list, p, true );
	return true;
  
		}*/
	}
	QListView::mousePressEvent( e );
}


/** sets EditField to new position */
void ListEdit::slotSortingChanged( int column ){
// TODO: enable Sorting as SIGNAL from HeaderClicking is emitted before sorting...
//       could be done by moving setSorting(-1) to appendVokabel
  setSorting(-1);			// see ListEdit::ListEdit; avoids new (empty) lines to be set at listbegin
	if(EditField) setCursorPos( EditField->cursorPosition() );
	updateEditField();
}


/** handles the pressing of Return,  Enter and Tab */
bool ListEdit::handleEnterPress( const QKeyEvent *k )
{
	if ( !changeColumn( ToRight ) ) 
		return goNextLine();
	return true;
}


/** sets the EditField to the next line in the column Column */
bool ListEdit::goNextLine ( const int Direction = 1, const int NewCursorPos= -1, const int Column = 0 )
{
	QListViewItem *NextItem;
	
	if( Direction == +1 )	NextItem = currentItem()->itemBelow();
	else if (Direction == -1 ) NextItem = currentItem()->itemAbove();
	else return false;

	if ( NextItem ) {
		setEditedColumn( Column );
		setCurrentItem( NextItem );
		setCursorPos( NewCursorPos );
		
		ensureItemVisible( NextItem );
		updateEditField();
		return true;
	}
	else return false;
}


/** sets the edited Column */
void ListEdit::setEditedColumn( const int NewColumn )
{
	if ( NewColumn >= header()->count() ) // Columnindex starts with 0
		EditedColumn = header()->count() - 1;
	else	if ( NewColumn < 0 ) 
		EditedColumn = 0;
	else
		EditedColumn = NewColumn;
}


/** sets the cursor position */
void ListEdit::setCursorPos( const int NewCursorPos = -1 )
{
	if( NewCursorPos != -1 )
		CursorPos = NewCursorPos;
	else
		if( currentItem() )
			CursorPos = QString(currentItem()->text( editedColumn() )).length();
	else CursorPos = 0;
}



/** clears list and resets all edit data */
void ListEdit::clear()
{
	QListView::clear();
	
	setEditMode( false);			// no editing possible
}


/** */
const QFont &ListEdit::columnFont( int Column )
{
  if( FontList.at( Column ) && FontList.at( Column )->Font )
  {
//    cout << "Font der Spalte " << Column << ": " << FontList.at( Column )->Font->family() << "\n";
    return *(FontList.at( Column )->Font);
  }
  else 
    return font();
}   


/** sets the font of column Column */
void ListEdit::setColumnFont( int Column, const QFont & Font )
{
  if( !FontList.at( Column ) )
    return;
  
  if( FontList.at( Column )->Font )
    (*FontList.at( Column )->Font) = Font;
  else
  {
    // no font set until now, so create one
    QFont *NewFont = new QFont( Font );
    
    FontList.at( Column )->Font = NewFont;
  }
  
  checkFontMetricHeight( Font );
}


/** sets the standard font for all columns  */
void ListEdit::setFont ( const QFont & f )
{
  QListView::setFont( f );
  
//cout << "setFont: " << f.rawName() << "\n"; 
  
  checkFontMetricHeight( f );
}

/** */
int ListEdit::addColumn( const char * label, int width = -1 )
{
  int index = QListView::addColumn( label, width );
  
//cout << "addColumn: " << label << "\n";  

  VFontEntry NewEntry = { 0 };
  VFontEntry *Entry = new VFontEntry; 
  (*Entry) = NewEntry;
  FontList.append( Entry );
  
  return index;
}

/** check if new font is higher than the one used */
void ListEdit::checkFontMetricHeight( const QFont &Font )
{
  QFont CheckFont = Font;
  QFontMetrics FontMetrics( CheckFont );
  uint Height = FontMetrics.height();
  
  if( Height >= FontMetricsHeight )
    // new height is higher so have it as list height
    FontMetricsHeight = Height;
  else
  {
    // find highest of used fonts, set standard height as reference
    FontMetrics = QFontMetrics( font() );
    FontMetricsHeight = FontMetrics.height();
  
    VFontEntry *Entry;
    for( Entry = FontList.first(); Entry; Entry = FontList.next() )
    { 
      if( Entry->Font )
      {
        FontMetrics = QFontMetrics( (*Entry->Font) );
        Height = FontMetrics.height();
        if( Height >= FontMetricsHeight )
          FontMetricsHeight = Height;
      }
    }
  }
}



/** returns last item or 0*/
QListViewItem *ListEdit::lastItem()
{
  QListViewItem *Item;                           
	
	if( currentItem() ) Item = currentItem();   // get last used item as that is closer to end -> faster search
	else Item = firstChild(); 
	
	if( Item )
	{	
    while ( Item->nextSibling() )                    // find last item
    	Item = Item->nextSibling();
  	return Item;
  }
  else
    return 0;
}



/** */
ListEditItem::ListEditItem( QListView * parent, const char *Term1 = 0, const char *Term2 = 0 )
		: QListViewItem ( parent,  Term1, Term2 )  
{
}

/** */
ListEditItem::ListEditItem( QListView * parent, QListViewItem * after, const char *Term1 = 0, const char *Term2 = 0 )
		: QListViewItem ( parent, after, Term1, Term2 ) 
{
}

/** */
ListEditItem::~ListEditItem()
{
}


/** draws the text with individuell fonts */ 
void ListEditItem::paintCell ( QPainter * p, const QColorGroup & cg, int column, int width, int align )
{
  ListEdit *List = (ListEdit*)listView();
  p->setFont( List->columnFont( column ) );

//  cerr << "Lnge: " << List->childCount() << ", Zeile: " << column << ", Font: " << p->font().family() << "\n";
  
  QListViewItem::paintCell ( p, cg, column, width, align );
}


/** Performs setup. */
void ListEditItem::setup()
{
  widthChanged();
  
  ListEdit *List = (ListEdit*)listView();
  int Height = List->FontMetricsHeight + 2*List->itemMargin();
  // equal number of pixels looks better (see ListViewItem::setup)
  if ( Height % 2 > 0 )
  	Height++;
  setHeight( Height );
} 






