/* -------------------------------------------------------------------------- */
/*                                                                            */
/* [zmodem.cpp]              ZMODEM File Transport                            */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* Copyright (c) 1997 by Lars Doelle                                          */
/*                                                                            */
/* This file is part of Kom - a serial line terminal for KDE                  */
/*                                                                            */
/* The whole program is available under the GNU Public Software Licence.      */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.        */
/*                                                                            */
/* -------------------------------------------------------------------------- */

/* CURRENT STATE:
   - 70 % of zmodem receiver
   - 70 % of zmodem sender
   - both to mixed into one state machine
*/

/* FIXME:
   - proper change between protocols (kterm)
   - assign valid codes to Zany, Zbadpck, ...
   - check/cleanup "emit event" (proper place)
   - move ::error-event to normal rcv_block.
   - adjust status messages
*/
    
/* TODO: 
   - proper cancel handling (ZMODEM::cancel, aka ZmodemReceiver::terminate)
   - make msgbox for cmd (to test: try to send /p/bin/xgo with sz (nonexist))
   - error recovery
   - time out
   - complete handling of frames (ZCOMMAND, et al?)
   - complete options (ZSINIT ZINIT)
   - support for streams (?)
*/

#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <assert.h>

#include <qstring.h>

#include "zmodem.h"
#include "zmodcon.h"
#include "crc.h"

#include "zmodem.moc"

//typedef unsigned char UINT8;

#define HERE printf("%s(%d): here\n",__FILE__,__LINE__)

enum { Header0, Header1, Header2, HexDta0, HexDta1, DleDta0, DleDta1 };

// "scanModes"
#define scanHeader        7
#define scanPacket     1025
#define scanPacketCRC     2

static void hexdump(char* header, char* buffer, int len)
{ int i;
  printf("%s\r\n  ",header);
  for (i = 0; i < len; i++)
  {
    if (i > 0 && i %  4 == 0) printf(" ");
    if (i > 0 && i %  8 == 0) printf(" ");
    if (i > 0 && i % 16 == 0) printf("\r\n  ");
    printf("%02x ",(UINT8)buffer[i]);
  }
  printf("\r\n");
  fflush(stdout);
}

static QString tagNames [256];
static QString pckNames [256];

static initNames()
{ int i;
  for (i = 0; i < 256; i++)
  { char buf[10]; sprintf(buf,"%5d",i);
    tagNames[i] = buf;
    pckNames[i] = buf;
  }
  tagNames[  0] = "ZRQINIT";     tagNames[  1] = "ZRINIT";
  tagNames[  2] = "ZSINIT";      tagNames[  3] = "ZACK";
  tagNames[  4] = "ZFILE";       tagNames[  5] = "ZSKIP";
  tagNames[  6] = "ZNAK";        tagNames[  7] = "ZABORT";
  tagNames[  8] = "ZFIN";        tagNames[  9] = "ZRPOS";
  tagNames[ 10] = "ZDATA";       tagNames[ 11] = "ZEOF";
  tagNames[ 12] = "ZFERR";       tagNames[ 13] = "ZCRC";
  tagNames[ 14] = "ZCHALLENGE";  tagNames[ 15] = "ZCOMPL";
  tagNames[ 16] = "ZCAN";        tagNames[ 17] = "ZFREECNT";
  tagNames[ 18] = "ZCOMMAND";    tagNames[ 19] = "ZSTDERR";

  pckNames['h'] = "ZCRCE";       pckNames['i'] = "ZCRCG";
  pckNames['j'] = "ZCRCQ";       pckNames['k'] = "ZCRCW";
  pckNames['x'] = "ZHEAD";
}

/* --| Encoding |------------------------------------------------------------ */

static UINT8 nibble_encode(int value)
{
  return value + (value<10?'0':('a'-10));
}

static void hex_encode(UINT8* to, UINT8* from, int len)
{ int i;
  for (i = 0; i < len; i++)
  { 
    to[2*i+0] = nibble_encode(from[i]/16);
    to[2*i+1] = nibble_encode(from[i]%16);
  }
}

static void chksum_encode(UINT8* to, unsigned long value, int len)
{ int i;
  for(i = 0; i<len; i++, value /= 256)
    to[len-1-i] = value%256;
}

static void number_encode(UINT8* to, unsigned long value, int len)
{ int i;
  for(i = 0; i<len; i++, value /= 256)
    to[i] = value%256;
}

//FIXME: make proper options
#define quote_controls 0
#define quote_deletes  0

static int zdle_encode(UINT8 *to, UINT8 *from, int len)
// result is encoded len
{ int i,res = 0;
  for (i = 0; i < len; i++)
  {
    if( quote_deletes && (from[i] & 0x7f) == 0x7f )
    { to[res++] = ZDLE; to[res++] = (from[i]==0x7f?ZRUB0:ZRUB1); }
    else if( quote_controls && (from[i] & 0x1f) == 0)
    { to[res++] = ZDLE; to[res++] = from[i]^0x40; } // X001 00X1 (xon/xoff)
    else if( (from[i] & 0x7d) == 0x11 )
    { to[res++] = ZDLE; to[res++] = from[i]^0x40; }
    else if( from[i] == ZDLE )
    { to[res++] = ZDLE; to[res++] = ZDLEE; }
    else to[res++] = from[i];
  }
  return res;
}

// --| low level sender |------------------------------------------------------

// hex header ///////////////////////////////////////////////////////////

void Zmodem::sendHexHeader(UINT8 typ, unsigned long info)
{ UINT8 hdr[7]; UINT8 buf[4+2*7+3];
printf("snd [%-10s ZHEAD    4] (0x%08lx)\n",tagNames[typ].data(),info);
fflush(stdout);
  hdr[0] = typ; number_encode(hdr+1,info,4);
  chksum_encode(hdr+5,crc16((char*)hdr,5),2);
  buf[0] = ZPAD; buf[1] = ZPAD; buf[2] = ZDLE; buf[3] = ZHEX;
  hex_encode(buf+4,hdr,7);
  buf[18] = '\r'; buf[19] = '\n'; buf[20] = 021; // XON
  mo->send_bytes((char*)buf,4+2*7+3);
  lastFrameType = typ;
  lastFrameInfo = info;
}

// binary header ////////////////////////////////////////////////////////

void Zmodem::sendBinHeader(UINT8 typ, unsigned long info)
{ UINT8 buf[3+2*7]; UINT8 hdr[7];
printf("snd [%-10s ZHEAD    4] (0x%08lx)\n",tagNames[typ].data(),info);
fflush(stdout);
  hdr[0] = typ; number_encode(hdr+1,info,4);
  chksum_encode(hdr+5,crc16((char*)hdr,5),2);
  buf[0] = ZPAD; buf[1] = ZDLE; buf[2] = ZBIN;
  int out = 3+zdle_encode(buf+3,hdr,7);
  mo->send_bytes((char*)buf,out);
  lastFrameType = typ;
  lastFrameInfo = info;
}

// binary packets //////////////////////////////////////////////////////

void Zmodem::sendBinPacket(UINT8* txt, int len, UINT8 type)
{ UINT8 buf[2*1024+2+2*2]; UINT8 tmp[2]; int out;
printf("snd [%-10s %5s %4d]\n",
tagNames[lastFrameType].data(),pckNames[type].data(),len);
fflush(stdout);
  out = zdle_encode(buf,txt,len);
  crc = crc16((char*)txt,len);
  buf[out++] = ZDLE; buf[out++] = type;
  chksum_encode(tmp,upd_crc16(crc,buf+out-1,1),2);  //FIXME: remove upd_crc16
  out += zdle_encode(buf+out,tmp,2);
  mo->send_bytes((char*)buf,out);
//hexdump("Packet-pre",txt,len);
//hexdump("Packet-post",buf,out);
}

// binary frame ////////////////////////////////////////////////////////

void Zmodem::sendBinFrame(UINT8 typ, unsigned long info, UINT8* txt, int len)
{
  sendBinHeader(typ,info); sendBinPacket(txt,len,ZCRCW);
}

// pure text /////////////////////////////////////////////////

void Zmodem::sendString(char* txt)
// this is for "rz\r" and "OO"
{
printf("snd '%s'\n",txt); fflush(stdout);
  mo->send_bytes(txt,strlen(txt));
}

/* --| Block Scanning |------------------------------------------------------ */

#define inRange(L,X,H) (((L) <= (X)) && (X) <= (H))
#define isNibble(X)    (inRange('0',(X),'9')||inRange('a',(X),'f'))
#define isHeader(C)    (inRange('A',(C),'D'))
#define isTrailer(C)   (inRange('h',(C),'k'))

static int nibble_decode(UINT8 nibble)
{
  return nibble - (nibble <= '9' ? '0' : ('a'-10));
}

static unsigned long number_decode(UINT8* UINT8s, int len)
{ int i; unsigned long res = 0;
  for (i=len-1; i >= 0; i--) res = 256*res+UINT8s[i];
  return res;
}

/* --| low level scanner |--------------------------------------------------- */

void Zmodem::rcv_byte(int cin)
{ 
//printf("%d %d %d: %02x (%c)\n",scanState,scanCount,scanLimit,(UINT8)cin,32<=cin&&cin<127?cin:'?');
  
  if ((cin & 0x7d) == 0x11) return; // zmodem silently ignores XON/XOFF

  // 5*ZDLE == cancel
  zdlecnt = (cin == ZDLE) ? zdlecnt+1 : 1;
  if (zdlecnt>=5) { rcv_cancel(); return; } // detect 4*ZDLE

  switch (scanState)
  {
    case Header0 : // cin
         scanState = cin == ZPAD ? Header1 : Header0;
         break;
    case Header1 : // ZPAD cin
         scanState = cin == ZPAD ? Header1 : (cin == ZDLE) ? Header2 : Header0;
         break;
    case Header2 : // ZPAD ZDLE cin
         scanState = cin == ZHEX ? HexDta0 : (cin == ZBIN) ? DleDta0 : Header0;
         break;

    case DleDta0 : // scan fixed len dle encoded data
         if (cin == ZDLE) { scanState = DleDta1; break; }
         goto Push;
    case DleDta1 : // ZDLE cin
         scanState = DleDta0;
         if ((cin & 0x60) == 0x40 ) { cin ^= 0x40; goto Push; }
         if ( cin         == ZRUB0) { cin  = 0x7f; goto Push; }
         if ( cin         == ZRUB1) { cin  = 0xff; goto Push; }
         if ( cin         == ZDLEE) { cin  = ZDLE; goto Push; }
         if ('h' <= cin && cin <= 'k' && scanLimit == scanPacket) // got Framend
         { buffer[buftop++] = cin; setExpect(scanPacketCRC); break; }
         goto Error;

    case HexDta0 : // scan fixed len hex encoded data
         if (!isNibble(cin)) goto Error;
         scanState = HexDta1;
         hex_keep = nibble_decode(cin);
         break;
    case HexDta1 :
         if (!isNibble(cin)) goto Error;
         scanState = HexDta0;
         buffer[buftop++] = 16*hex_keep+nibble_decode(cin);
         if (++scanCount >= scanLimit) goto DoneHeader;
         break;
  }
  return;

Push:
  buffer[buftop++] = cin;
  if (++scanCount >= scanLimit)
  { switch(scanLimit)
    { case scanHeader    : goto DoneHeader;
      case scanPacketCRC : goto DonePacket;
      case scanPacket    : goto Error;
  } }
  return;

DoneHeader:
//hexdump("Header",buffer,buftop);
  if (crc16((char*)buffer,buftop) != 0) goto Error;
  last_hdr_typ = buffer[0];
  rcv_block(buffer[0], ZHEAD, buffer+1, buftop-1);
  switch (buffer[0])
  {
    case ZDATA : case ZFILE : setExpect(scanPacket); break;
    default    : setExpect(scanHeader); break;
  }
  buftop = 0;
  return;

DonePacket:
//hexdump("Packet",buffer,buftop);
  if (crc16((char*)buffer,buftop) != 0) goto Error;
  rcv_block(last_hdr_typ, buffer[buftop-3], buffer, buftop-3);
  switch (buffer[buftop-3])
  {
    case ZCRCE : case ZCRCW : setExpect(scanHeader); break;
    default : setExpect(scanPacket); break;
  }
  buftop = 0;
  return;

Error     :
  { ErrorType tmp = scanLimit==scanHeader?errorHeader:errorPacket;
printf("rcv_error\n");
    buftop = 0;
    setExpect(scanHeader);
    rcv_error(tmp);
    return;
  }
};

void Zmodem::setExpect(int mode)
{
  scanLimit = mode;
  scanCount = 0;
  switch(mode)
  {
    case scanHeader    : scanState = Header0; break;
    case scanPacketCRC : scanState = DleDta0; break;
    case scanPacket    : scanState = DleDta0; break;
  }
}

/* --| constructor |--------------------------------------------------------- */

Zmodem::Zmodem(Modem* mo) : QObject()
{
  initNames();
  this->mo = mo;
  buftop = 0;
  zdlecnt = 0;
  last_hdr_typ=-1;
  setExpect(scanHeader);
  curr_file = NULL;
}

void Zmodem::rcv_cancel()
/* received cancel (CAN*5) */
{
  delete this;
}

void Zmodem::stopProtocol()
// force termination
{
// FIXME: assert proper handling by sender/receiver
  sendString("\030\030\030\030\030\030\030\030");
  disconnect( this, SIGNAL(doneProtocol()), 0, 0 );
  rcv_cancel();
}

Zmodem::~Zmodem()
{
printf("~Zmodem()\n");
//  emit doneProtocol(); // not me // Fixme: remove?
//  emit event(eventDoneProtocol,NULL,0);
}

/* --| Role receiver |------------------------------------------------------- */

#define combine(A,B) ((((UINT16)(A))<<8)|(B))

void Zmodem::evalFileHeader(UINT8* info, int len)
{ long size=0; long tim=0; int mode=0; int filesleft=0; long totalleft=0;

  // decode file packet ////////////////////////////////////////////////////////
  //FIXME: improve

  sscanf((char*)info+strlen((char*)info)+1,"%lu %lo %o 0 %d %ld",
         &size, &tim, &mode, &filesleft, &totalleft);

  // print file info ///////////////////////////////////////////////////////////

  printf("Text: >%s<\r\n",(char*)info+strlen((char*)info)+1);
  printf("-------------- File -------------------- ---- Left ----\r\n");
  printf("Name---------- ----Size -----Time --Mode Files ---Bytes\r\n");
  printf("%14s %8ld %9ld %6o %5d %8ld\r\n",info,size,tim,mode,filesleft,totalleft);

  // open file to receive //////////////////////////////////////////////////////

  curr_file_offset = 0;
//curr_file_size   = size;
  { char buf[100]; 
printf("file: %s/%s\n",downloaddict.data(),info);
    sprintf(buf,"%s/%s",downloaddict.data(),info);
    curr_file = fopen(buf,"w");
printf("file: '%s'.\n",buf);
    assert(curr_file > 0);
  }

  emit event(eventRcvFile,(char*)info,size);
}

#define Zany     'x'
#define Zoo      'o'
#define Znone    'x'
#define Zbadpck  'x'
#define Zbadhdr  'x'

struct Zmodem::ZmodemState Zmodem::rcv_table[] =
{
//  state   Event    Event  condition   state/scan  state   reply
//  ------  -------- ------ ----------- ----------- ------- ------
  { ZRINIT, ZNAK,    ZHEAD, NULL,       scanHeader, ZRINIT, ZRINIT },
  { ZRINIT, ZRQINIT, ZHEAD, NULL,       scanHeader, ZRINIT, ZRINIT },
  { ZRINIT, ZFILE,   ZHEAD, NULL,       scanPacket, ZRINIT, Znone  },
  { ZRINIT, ZFILE,   ZCRCW, makeFile,   scanHeader, ZRPOS,  ZRPOS  },
  { ZRINIT, ZFILE,   ZCRCW, NULL,       scanHeader, ZRINIT, ZSKIP  },
  { ZRINIT, ZFIN,    ZHEAD, quit,       scanHeader, ZFIN,   ZFIN   },
//{ ZRINIT, ZFIN,    ZHEAD, NULL,       scanOO    , ZFIN,   ZFIN   },
  { ZRINIT, Zbadpck, 0,     NULL,       scanHeader, ZRINIT, ZNAK   },
  { ZRINIT, Zbadhdr, 0,     NULL,       scanHeader, ZRINIT, ZNAK   },
  { ZRINIT, Zany,    0,     NULL,       scanHeader, ZRINIT, ZRINIT },
  { ZRPOS,  ZNAK,    ZHEAD, NULL,       scanHeader, ZRPOS,  ZRPOS  },
  { ZRPOS,  ZDATA,   ZHEAD, pkPosOk,    scanPacket, ZRPOS,  Znone  },
  { ZRPOS,  ZDATA,   ZHEAD, NULL,       scanHeader, ZRPOS,  ZRPOS  },
  { ZRPOS,  ZDATA,   ZCRCE, writeBlock, scanHeader, ZRPOS,  Znone  },
  { ZRPOS,  ZDATA,   ZCRCG, writeBlock, scanPacket, ZRPOS,  Znone  },
  { ZRPOS,  ZDATA,   ZCRCW, writeBlock, scanHeader, ZRPOS,  ZACK   },
  { ZRPOS,  ZDATA,   ZCRCQ, writeBlock, scanPacket, ZRPOS,  ZACK   },
//{ ZRPOS,  ZDATA,   ....., NULL,       scanHeader, ZRINIT, ZFERR  },
  { ZRPOS,  ZEOF,    ZHEAD, closeFile,  scanHeader, ZRINIT, ZRINIT },
  { ZRPOS,  ZEOF,    ZHEAD, NULL,       scanHeader, ZRPOS,  ZRPOS  },
  { ZRPOS,  Zbadpck, 0,     NULL,       scanHeader, ZRPOS,  ZNAK   },
  { ZRPOS,  Zbadhdr, 0,     NULL,       scanHeader, ZRPOS,  ZNAK   },
  { ZRPOS,  Zany,    0,     NULL,       scanHeader, ZRPOS,  ZRPOS  },
//{ ZFIN,   Zoo,     0,     quit,                -,     -,  Znone  },
  { ZFIN,   Zany,    0,     NULL,       scanHeader, ZFIN,   Znone  }
};

// predicates //////////////////////////////////////////////////////////////////

bool Zmodem::pkPosOk(UINT8 *info, int len)
{
  return number_decode(info,4) == curr_file_offset;
}

bool Zmodem::closeFile(UINT8 *info, int len)
{
  emit event(eventRcvDone,NULL,0);
  fclose(curr_file); //Fixme: check close success
  curr_file = NULL;
printf("number: %d, offset: %d\n", number_decode(info,4), curr_file_offset);
printf("result: %d\n", number_decode(info,4) == curr_file_offset);
printf("size: %d\n", curr_file_size );
  return number_decode(info,4) == curr_file_offset ;
//    && curr_file_size        == curr_file_offset;
}

bool Zmodem::makeFile(UINT8 *info, int len)
{
  evalFileHeader(info,len);
  return TRUE;
//Fixme: open here;
}

bool Zmodem::writeBlock(UINT8 *info, int len)
{
  emit event(eventRcvData,NULL,len);
  fwrite(info,len,1,curr_file); /*FIXME: check fwrite successfull*/
  curr_file_offset += len;
  return TRUE;
}

bool Zmodem::quit(UINT8 *info, int len)
{
  // there is no secure way to catch "OO", since the modem events
  // are received by the terminal too (before or after this event).
  // this we choose to wait a bit and HOPE that only the "OO"s 
  // are incoming meanwhile.
  emit doneProtocol();
  emit event(eventDoneProtocol,NULL,0);
  QTimer::singleShot( 100, this, SLOT(terminate()) ); // 0.1 sec
  return TRUE;
}

void Zmodem::startReceiver(char* load_path, char* initial)
// start receiving from modem after 'initial'
{
  state = ZRINIT;
  table = rcv_table;
  downloaddict = load_path;
  sendHexHeader(ZRINIT,0); /* here we are! */
  while(*initial) rcv_byte(*initial++);
  connect( mo, SIGNAL(data_in(int)), this, SLOT(rcv_byte(int)) );
}

/* --| Role Sender |------------------------------------------------------- */

bool Zmodem::currFile()
{ struct stat buf; int i; char* basename; char* s; char* path;

  for (; curr_file == NULL && file_list.current() != NULL; file_list.next())
  { char* url = file_list.current();
    // some checks ///////////////////////////////////////////////////////
    if (strncmp(url,"file:/",6)!=0)
    { fprintf(stderr,"%s(%d): '%s' non-file url.\n",__FILE__,__LINE__,url);
      continue; }
    path = url+5;
    if (stat(path,&buf))
    { fprintf(stderr,"%s(%d): cannot stat '%s'.\n",__FILE__,__LINE__,url);
      continue; }
    if (!S_ISREG(buf.st_mode))
    { fprintf(stderr,"%s(%d): '%s' not a file.\n",__FILE__,__LINE__,url);
      continue; }
    // normalizing ///////////////////////////////////////////////////////
    for (s = path+strlen(path); s >= path && *s != '/'; s--);
    basename = s + (*s=='/');
    curr_file = fopen(path,"r");
    if (curr_file == NULL)
    { fprintf(stderr,"%s(%d): cannot open '%s'.\n",__FILE__,__LINE__,url);
      continue; }
    // got it ////////////////////////////////////////////////////////////
    int size = buf.st_size;
    int mode = buf.st_mode & 0777;
    int time = buf.st_mtime;
printf("file: >%s :: %d<\n",file_list.current(),size);
    strcpy((char*)frame_txt,basename);
    int len1 = strlen((char*)frame_txt);
    sprintf((char*)frame_txt+len1+1,"%lu %lo %o", size, time, mode);
    frame_len = len1+1+strlen((char*)frame_txt+len1+1)+1;
    curr_file_offset = 0;
    emit event(eventSndFile,(char*)frame_txt,size);
    break;
  }
  bytes_send = -1; //Fixme: assert proper states
  return curr_file != NULL;
}

bool Zmodem::nextFile()
{
  if (curr_file != NULL) { fclose(curr_file); curr_file = NULL; }
  if (file_list.current() != NULL) file_list.next();
  return currFile();
}

bool Zmodem::zrpos(UINT8 *info, int len)
//FIXME: rename
{ unsigned long code = number_decode(info,4);
  if (curr_file_offset != code)
  {
    //Fixme: if failed or code > curr_file_size?
    fseek(curr_file,code,SEEK_SET);
    curr_file_offset = code;
  }
  return TRUE; //FIXME: curr_file_offset < curr_file_size;
}

void Zmodem::clearToSend()
// signal: modem ready to send more data
// this method prevents locking up the application
//Fixme: we should deactivate the keyboard to have a secure semantic.
{ UINT8 block[1024];
  if (bytes_send < 0) return;
  int tosend = read(fileno(curr_file),block,1024);
  curr_file_offset += tosend;
  if (tosend < 1024)
  {
    sendBinPacket(block,tosend,ZCRCE);
    sendHexHeader(ZEOF,curr_file_offset);
    bytes_send = -1; // stop sending
    state = ZEOF;
  }
  else
    sendBinPacket(block,tosend,ZCRCG);
  emit event(eventSndData,NULL,tosend);
}

//FIXME: add catch-alls
struct Zmodem::ZmodemState Zmodem::snd_table[] =
{
//  state    Event     Event  condition   state/scan  state    reply
//  ------   --------  ------ ----------- ----------- -------- ------
  { ZRQINIT, ZNAK,     ZHEAD, NULL,       scanHeader, ZRQINIT, ZRQINIT},
  { ZRQINIT, ZRINIT,   ZHEAD, currFile,   scanHeader, ZFILE,   ZFILE  },
  { ZRQINIT, ZRINIT,   ZHEAD, NULL,       scanHeader, ZFIN,    ZFIN   },
  { ZRQINIT, ZFIN,     ZHEAD, NULL,       scanHeader, ZFIN,    ZFIN   },
  { ZRQINIT, Zany,     0,     NULL,       scanHeader, ZRQINIT, ZRQINIT},

  { ZFILE,   ZRPOS,    ZHEAD, zrpos,      scanHeader, ZDATA,   ZDATA  },
  { ZFILE,   ZRPOS,    ZHEAD, NULL,       scanHeader, ZEOF,    ZEOF   },
  { ZFILE,   ZSKIP,    ZHEAD, nextFile,   scanHeader, ZFILE,   ZFILE  },
  { ZFILE,   ZSKIP,    ZHEAD, NULL,       scanHeader, ZFIN,    ZFIN   },
  { ZFILE,   ZFIN,     ZHEAD, NULL,       scanHeader, ZFIN,    ZFIN   },
  { ZFILE,   ZRINIT,   ZHEAD, NULL,       scanHeader, ZFILE,   Znone  },
  { ZFILE,   Zany,     0,     NULL,       scanHeader, ZFILE,   Znone  },

  { ZDATA,   ZNAK,     ZHEAD, NULL,       scanHeader, ZDATA,   ZDATA  },
//{ ZDATA,   <clear>,  ZHEAD, moreDta,    scanHeader, ZDATA,   Znone  },
//{ ZDATA,   <clear>,  ZHEAD, NULL,       scanHeader, ZEOF,    ZEOF   },
  { ZDATA,   ZRPOS,    ZHEAD, zrpos,      scanHeader, ZDATA,   ZDATA  },
  { ZDATA,   ZRPOS,    ZHEAD, NULL,       scanHeader, ZEOF,    ZEOF   },
  { ZDATA,   ZACK,     ZHEAD, NULL,       scanHeader, ZDATA,   Znone  },
  { ZDATA,   ZSKIP,    ZHEAD, nextFile,   scanHeader, ZFILE,   ZFILE  },
  { ZDATA,   ZSKIP,    ZHEAD, NULL,       scanHeader, ZFIN,    ZFIN   },
  { ZDATA,   Zany,     0,     NULL,       scanHeader, ZDATA,   Znone  },

  { ZEOF,    ZNAK,     ZHEAD, NULL,       scanHeader, ZEOF,    ZEOF   },
  { ZEOF,    ZRINIT,   ZHEAD, nextFile,   scanHeader, ZFILE,   ZFILE  },
  { ZEOF,    ZRINIT,   ZHEAD, NULL,       scanHeader, ZFIN,    ZFIN   },
  { ZEOF,    ZRPOS,    ZHEAD, zrpos,      scanHeader, ZDATA,   ZDATA  },
  { ZEOF,    ZRPOS,    ZHEAD, NULL,       scanHeader, ZEOF,    ZEOF   },
  { ZEOF,    Zany,     0,     NULL,       scanHeader, ZEOF,    Znone  },

  { ZFIN,    ZNAK,     ZHEAD, NULL,       scanHeader, ZFIN,    ZFIN   },
  { ZFIN,    ZFIN,     ZHEAD, NULL,       scanHeader, ZFIN,    Zoo    }, // OO
  { ZFIN,    Zany,     0,     quit,       scanHeader, ZFIN,    Znone  } 
};

// receiver handling ///////////////////////////////////////////////////////////

void Zmodem::rcv_block(UINT8 type,UINT8 trailer, UINT8* info,int len)
// FIXME: mix with ZmodemReceiver::rcv_block to Zmodem::rcv_block
{ int i;

  // diagnosis /////////////////////////////////////////////////////////////////

  printf("rcv [%-10s %-10s %5s %4d]", 
         tagNames[state].data(), tagNames[type].data(), 
         pckNames[trailer].data(), len);
  if (trailer == ZHEAD) printf(" (0x%08x)",number_decode(info,4));
  printf("\n");
  //hexdump("",info,len);
  fflush(stdout);

  // decide on activity and reponse ////////////////////////////////////////////

  for (i = 0; TRUE; i++)
  {
    if ( ( (  table[i].state == state && 
              ( table[i].type1 == type && table[i].type2 == trailer ||
                table[i].type1 == Zany ) )
         || ( table[i].state == Zany && table[i].type1 == type) )
       && (table[i].pred == NULL || (this->*(table[i].pred))(info,len) ) )
    {
      state = table[i].next;
      setExpect(table[i].follow);
      bytes_send = (table[i].reply == ZDATA) ? 0 : -1;
      switch(table[i].reply)
      {
        // Receiver
        case ZRINIT  : sendHexHeader(table[i].reply,0               ); break;
        case ZNAK    : sendHexHeader(table[i].reply,0               ); break;
        case ZACK    : sendHexHeader(table[i].reply,curr_file_offset); break;
        case ZRPOS   : sendHexHeader(table[i].reply,curr_file_offset); break;
        // Sender
        case ZRQINIT : sendHexHeader(table[i].reply,0               ); break;
        case ZFILE   : sendBinFrame(ZFILE,0,(UINT8*)frame_txt,frame_len);
                       break;
        case ZDATA   : sendBinHeader(table[i].reply,curr_file_offset);
                       bytes_send = 0; //control: triggers 'clearToSend'
                       break;
        case ZEOF    : sendHexHeader(table[i].reply,curr_file_offset); break;
        case Zoo     : sendString("OO"); quit(0,0);                    break;
        // Both
        case ZFIN    : sendHexHeader(table[i].reply,0               ); break;
        case Znone   :                                                 break;
      }
      break;
    }
  }
}

void Zmodem::rcv_error(ErrorType error)
{
/*FIXME: recover*/
/*FIXME: timeout: 1200 byte (assuming 2400 baud) gives 4 sec */
/*must keep some kind of state*/
  emit event(eventRcvError,NULL,0);
}

void Zmodem::terminate()
// FIXME: aka. cancel
{
  delete this;
}

void Zmodem::startSender(QStrList* lst)
// start sending to modem.
{
  state = ZRQINIT;
  file_list = *lst;
  bytes_send = -1;
  table = snd_table;
  sendString("rz\r");     /* start zmodem receiver */
  sendHexHeader(ZRQINIT,0); /* here we are! */
  connect( mo, SIGNAL(data_in(int)), this, SLOT(rcv_byte(int)) );
  connect( mo, SIGNAL(written()), this, SLOT(clearToSend()) );
}
