/* ------------------------------------------------------------------------- */
/*                                                                           */
/* [vt102emu.cpp]              VT102 Terminal Emulation                      */
/*                                                                           */
/* ------------------------------------------------------------------------- */
/*                                                                           */
/* Copyright (c) 1997 by Lars Doelle                                         */
/*                                                                           */
/* This file is part of Kom - a serial line terminal for KDE                 */
/*                                                                           */
/* The whole program is available under the GNU General Public Licence.      */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.       */
/*                                                                           */
/* ------------------------------------------------------------------------- */

//NOTE: search for 'NotImplemented' to find unimplemented features.

//FIXME:
/*
   - For strange reasons, the extend of the rendition attributes ranges over
     all screens and not over the actual screen. We have to find out the
     precise extend.
*/

//TODO: we have no means yet, to change the emulation on the fly.

/* \class

   This class is responsible to scan the escapes sequences of the terminal
   emulation and to map it to their corresponding semantic complements.
   Thus this module knows mainly about decoding escapes sequences and
   is a stateless device w.r.t. the semantics.

   It is also responsible to refresh the TEWidget by certain rules.

   \sa TEWidget \sa TEScreen
*/

/* FIXME
   - more carefull parsing in `interprete_?'.
   - merge interprete_3/5.
*/
/* TODO
   - isolate the refreshing and resizing material to make up a base class.
     Eventually this class could also emulate a dump terminal.
   - add 'underline' mode (used by man)
*/

#include "vt102emu.h"
#include "TEWidget.h"
#include "TEScreen.h"

#include <stdio.h>
#include <unistd.h>
#include <qkeycode.h>

#include <assert.h>

#include "vt102emu.moc"

#define ESC 27
#define MIN(a,b) ((a)<(b)?(a):(b))
#define MAX(a,b) ((a)>(b)?(a):(b))

#define HERE printf("%s(%d): here\n",__FILE__,__LINE__)

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                Emulation                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */

#define CNTL(c) ((c)-'@')

/*!
*/

VT102Emulation::VT102Emulation(TEWidget* gui) : Emulation(gui)
{

  screen[0] = scr;
  screen[1] = new TEScreen(gui->Lines(),gui->Columns());

  reset();

  QObject::connect(gui,SIGNAL(mouseSignal(int,int,int)), 
                   this,SLOT(onMouse(int,int,int)));

  active = TRUE;

  resetMode(MODE_Mouse1000); saveMode(MODE_Mouse1000);
  resetMode(MODE_AppCuKeys); saveMode(MODE_AppCuKeys);
  resetMode(MODE_AppScreen); saveMode(MODE_AppScreen);

  tableInit();
}

/*!
*/

VT102Emulation::~VT102Emulation()
{
  scr = screen[0];
  delete screen[1];
}


void VT102Emulation::onImageSizeChange(int lines, int columns)
{
  if (scr != screen[0]) screen[0]->resizeImage(lines,columns);
  if (scr != screen[1]) screen[1]->resizeImage(lines,columns);
  Emulation::onImageSizeChange(lines,columns);
}

// Interpreting Codes ---------------------------------------------------------
/*
   This section deals with decoding the incoming character stream.
   Decoding means here, that the stream is first seperated into `tokens'
   which are then mapped to a `meaning' provided as operations by the
   `Screen' class.

   The tokens are defined below. They are:

   - CHR        - Printable Characters     (32..255)
   - CTL        - Control Characters       (0..31 but ESC (= 27))
   - ESC        - Escape Codes of the form <ESC><CHR but `[()+*#'>
   - ESC_DE     - Escape Codes of the form <ESC><any of `()+*#'>
   - CSI_PN     - Escape Codes of the form <ESC>'['     {Pn} ';' {Pn} C
   - CSI_PS     - Escape Codes of the form <ESC>'['     {Pn} ';' ...  C
   - CSI_PR     - Escape Codes of the form <ESC>'[' '?' {Pn} ';' ...  C

   The last two forms allow list of arguments. Since the elements of
   the lists are treated individually the same way, they are passed
   as individual tokens to the interpretation. Further, because the
   meaning of the parameters are names (althought represented as numbers),
   they are includes within the token ('N').

*/

#define TY_CONSTR(T,A,N) ( ((((int)N) & 0xffff) << 16) | ((((int)A) & 0xff) << 8) | (((int)T) & 0xff) )

#define TY_CHR___(   )  TY_CONSTR(0,0,0)
#define TY_CTL___(A  )  TY_CONSTR(1,A,0)
#define TY_ESC___(A  )  TY_CONSTR(2,A,0)
#define TY_ESC_CS(   )  TY_CONSTR(3,0,0)
#define TY_ESC_DE(A  )  TY_CONSTR(4,A,0)
#define TY_CSI_PS(A,N)  TY_CONSTR(5,A,N)
#define TY_CSI_PN(A  )  TY_CONSTR(6,A,0)
#define TY_CSI_PR(A,N)  TY_CONSTR(7,A,N)

void VT102Emulation::tau( int code, int p, int q )
{
//scan_buffer_report();
//if (code == TY_CHR___()) printf("%c",p); else
//printf("tau(%d,%d,%d, %d,%d)\n",(code>>0)&0xff,(code>>8)&0xff,(code>>16)&0xffff,p,q);
  switch (code)
  {
    case TY_CHR___(         ) : scr->ShowCharacter        (p         ); break;

    case TY_CTL___(0x24     ) :                                         break;
    case TY_CTL___('\b'     ) : scr->BackSpace            (          ); break;
    case TY_CTL___('\t'     ) : scr->Tabulate             (          ); break;
    case TY_CTL___('\n'     ) : scr->NewLine              (          );
                                     bulkNewline();                     break;
    case TY_CTL___('\r'     ) : scr->Return               (          ); break;
    case TY_CTL___(0x07     ) : gui->Bell                 (          ); break;
    case TY_CTL___(CNTL('N')) : scr->useCharset           (         1); break;
    case TY_CTL___(CNTL('O')) : scr->useCharset           (         0); break;

    case TY_ESC___('D'      ) : scr->index                (          ); break;
    case TY_ESC___('E'      ) : scr->NextLine             (          ); break;
    case TY_ESC___('M'      ) : scr->reverseIndex         (          ); break;
    case TY_ESC___('Z'      ) :      reportTerminalType   (          ); break;
    case TY_ESC___('n'      ) : scr->useCharset           (         2); break;
    case TY_ESC___('o'      ) : scr->useCharset           (         3); break;
    case TY_ESC___('7'      ) : scr->saveCursor           (          ); break;
    case TY_ESC___('8'      ) : scr->restoreCursor        (          ); break;
//  case TY_ESC___('='      ) : /*FIXME: Keypad Application Mode     */ break;
//  case TY_ESC___('>'      ) : /*FIXME: Keypad Numberic    Mode     */ break;

    case TY_ESC_CS(         ) : scr->setCharset           (p-'(',   q); break;

    case TY_ESC_DE('8'      ) : scr->helpAlign            (          ); break;

    case TY_CSI_PS('K',    0) : scr->clearToEndOfLine     (          ); break;
    case TY_CSI_PS('K',    1) : scr->clearToBeginOfLine   (          ); break;
    case TY_CSI_PS('K',    2) : scr->clearEntireLine      (          ); break;
    case TY_CSI_PS('J',    0) : scr->clearToEndOfScreen   (          ); break;
    case TY_CSI_PS('J',    1) : scr->clearToBeginOfScreen (          ); break;
    case TY_CSI_PS('J',    2) : scr->clearEntireScreen    (          ); break;
    case TY_CSI_PS('n',    5) :      reportStatus         (          ); break;
    case TY_CSI_PS('n',    6) :      reportCursorPosition (          ); break;
    case TY_CSI_PS('m',    0) : scr->setDefaultRendition  (          ); break;
    case TY_CSI_PS('m',    1) : scr->setBold              (          ); break;
    case TY_CSI_PS('m',    7) : scr->setInvert            (          ); break;
    case TY_CSI_PS('m',   30) : scr->setForeColor         (         0); break;
    case TY_CSI_PS('m',   31) : scr->setForeColor         (         1); break;
    case TY_CSI_PS('m',   32) : scr->setForeColor         (         2); break;
    case TY_CSI_PS('m',   33) : scr->setForeColor         (         3); break;
    case TY_CSI_PS('m',   34) : scr->setForeColor         (         4); break;
    case TY_CSI_PS('m',   35) : scr->setForeColor         (         5); break;
    case TY_CSI_PS('m',   36) : scr->setForeColor         (         6); break;
    case TY_CSI_PS('m',   37) : scr->setForeColor         (         7); break;
    case TY_CSI_PS('m',   39) : scr->setForeColorToDefault(          ); break;
    case TY_CSI_PS('m',   40) : scr->setBackColor         (         0); break;
    case TY_CSI_PS('m',   41) : scr->setBackColor         (         1); break;
    case TY_CSI_PS('m',   42) : scr->setBackColor         (         2); break;
    case TY_CSI_PS('m',   43) : scr->setBackColor         (         3); break;
    case TY_CSI_PS('m',   44) : scr->setBackColor         (         4); break;
    case TY_CSI_PS('m',   45) : scr->setBackColor         (         5); break;
    case TY_CSI_PS('m',   46) : scr->setBackColor         (         6); break;
    case TY_CSI_PS('m',   47) : scr->setBackColor         (         7); break;
    case TY_CSI_PS('m',   49) : scr->setBackColorToDefault(          ); break;
    case TY_CSI_PS('h',    4) : scr->    setMode      (MODE_Insert   ); break;
    case TY_CSI_PS('l',    4) : scr->  resetMode      (MODE_Insert   ); break;

    case TY_CSI_PN('@'      ) : scr->insertChars          (p         ); break;
    case TY_CSI_PN('A'      ) : scr->cursorUp             (p         ); break;
    case TY_CSI_PN('B'      ) : scr->cursorDown           (p         ); break;
    case TY_CSI_PN('C'      ) : scr->cursorRight          (p         ); break;
    case TY_CSI_PN('D'      ) : scr->cursorLeft           (p         ); break;
    case TY_CSI_PN('H'      ) : scr->setCursorYX          (p,       q); break;
    case TY_CSI_PN('L'      ) : scr->insertLines          (p         ); break;
    case TY_CSI_PN('M'      ) : scr->deleteLines          (p         ); break;
    case TY_CSI_PN('P'      ) : scr->deleteChars          (p         ); break;
    case TY_CSI_PN('f'      ) : scr->setCursorYX          (p,       q); break;
    case TY_CSI_PN('r'      ) : scr->setMargins           (p,       q); break;

    case TY_CSI_PR('h',    1) :          setMode      (MODE_AppCuKeys); break;
    case TY_CSI_PR('l',    1) :        resetMode      (MODE_AppCuKeys); break;
    case TY_CSI_PR('s',    1) :         saveMode      (MODE_AppCuKeys); break;
    case TY_CSI_PR('r',    1) :      restoreMode      (MODE_AppCuKeys); break;
    case TY_CSI_PR('h',    6) : scr->    setMode      (MODE_Origin   ); break;
    case TY_CSI_PR('l',    6) : scr->  resetMode      (MODE_Origin   ); break;
    case TY_CSI_PR('s',    6) : scr->   saveMode      (MODE_Origin   ); break;
    case TY_CSI_PR('r',    6) : scr->restoreMode      (MODE_Origin   ); break;
    case TY_CSI_PR('h',    7) : scr->    setMode      (MODE_Wrap     ); break;
    case TY_CSI_PR('l',    7) : scr->  resetMode      (MODE_Wrap     ); break;
    case TY_CSI_PR('s',    7) : scr->   saveMode      (MODE_Wrap     ); break;
    case TY_CSI_PR('r',    7) : scr->restoreMode      (MODE_Wrap     ); break;
//  case TY_CSI_PR('h',   25) :                                         break;
//  case TY_CSI_PR('l',   25) : /*FIXME: Cursor on/off               */ break;
//  case TY_CSI_PR('s',   25) : /*       happily ignored             */ break;
//  case TY_CSI_PR('r',   25) :                                         break;
    case TY_CSI_PR('h',   47) :          setMode      (MODE_AppScreen); break;
    case TY_CSI_PR('l',   47) :        resetMode      (MODE_AppScreen); break;
    case TY_CSI_PR('s',   47) :         saveMode      (MODE_AppScreen); break;
    case TY_CSI_PR('r',   47) :      restoreMode      (MODE_AppScreen); break;
    case TY_CSI_PR('h', 1000) :          setMode      (MODE_Mouse1000); break;
    case TY_CSI_PR('l', 1000) :        resetMode      (MODE_Mouse1000); break;
    case TY_CSI_PR('s', 1000) :         saveMode      (MODE_Mouse1000); break;
    case TY_CSI_PR('r', 1000) :      restoreMode      (MODE_Mouse1000); break;
    case TY_CSI_PR('h', 1001) : /* IGNORED: hilite mouse tracking    */ break;
    case TY_CSI_PR('l', 1001) : /* IGNORED  hilite mouse tracking    */ break;
    case TY_CSI_PR('s', 1001) : /* IGNORED: hilite mouse tracking    */ break;
    case TY_CSI_PR('r', 1001) : /* IGNORED  hilite mouse tracking    */ break;

    default : ReportErrorToken();    break;
  };
}

// -----------------------------------------------------------------------------
//
// Scanner / Transducer
//
// -----------------------------------------------------------------------------

// Scanning -------------------------------------------------------------------

void VT102Emulation::reset()
{
  ppos = 0; argc = 0; argv[0] = 0; argv[1] = 0;
}

#define CTL  1
#define CHR  2
#define CPN  4
#define DIG  8
#define SCS 16
#define GRP 32

void VT102Emulation::tableInit()
{ int i; UINT8* s;
  for(i =  0;                    i < 256; i++) tbl[ i]  = 0;
  for(i =  0;                    i <  32; i++) tbl[ i] |= (i != ESC) ? CTL : 0;
  for(i = 32;                    i < 256; i++) tbl[ i] |= CHR;
  for(s = (UINT8*)"@ABCDHLMPfr"; *s;      s++) tbl[*s] |= CPN;
  for(s = (UINT8*)"0123456789" ; *s;      s++) tbl[*s] |= DIG;
  for(s = (UINT8*)"()+*"       ; *s;      s++) tbl[*s] |= SCS;
  for(s = (UINT8*)"()+*#["     ; *s;      s++) tbl[*s] |= GRP;
}

/* Ok, here comes the nasty part of the decoder.

   The following defines do not abstract nor explain anything,
   they are only introduced to shorten the lines in the following
   routine, which is their only application.

   - P is the length of the token scanned so far.
   - L (often P-1) is the position on which contents we base a decision.
   - C is a character or a group of characters (taken from 'tbl').
*/

#define lec(P,L,C) (p == (P) &&                     s[(L)]         == (C))
#define les(P,L,C) (p == (P) &&                (tbl[s[(L)]] & (C)) == (C))
#define eec(C)     (p >=  3  &&        t                           == (C))
#define ees(C)     (p >=  3  &&                (tbl[  t   ] & (C)) == (C))
#define eps(C)     (p >=  3  && s[2] != '?' && (tbl[  t   ] & (C)) == (C))
#define epp( )     (p >=  3  && s[2] == '?'                              )
#define ces(C)     (                           (tbl[  cc  ] & (C)) == (C))
#define Dig        a[n] = 10*a[n] + cc - '0';
#define Arg        argc = MIN(argc+1,MAXARGS); argv[argc] = 0;

void VT102Emulation::onRcvByte(int c)
{ unsigned char cc = c; int i;
  if (ces(    CTL)) { tau( TY_CTL___(cc    ),    0,   0); return; }
  pbuf[ppos++] = c; 
  unsigned char* s = pbuf;
  int            p = ppos;
  unsigned char  t = s[p-1];
  int *          a = argv;
  int            n = argc;
  if (lec(1,0,ESC))                                                return;
  if (les(2,1,GRP))                                                return;
  if (lec(3,2,'?'))                                                return;
  if (les(1,0,CHR)) { tau( TY_CHR___(      ), s[0],   0); reset(); return; }
  if (lec(2,0,ESC)) { tau( TY_ESC___(s[1]  ),    0,   0); reset(); return; }
  if (les(3,1,SCS)) { tau( TY_ESC_CS(      ), s[1],s[2]); reset(); return; }
  if (lec(3,1,'#')) { tau( TY_ESC_DE(s[2]  ),    0,   0); reset(); return; }
  if (eps(    CPN)) { tau( TY_CSI_PN(t     ), a[0],a[1]); reset(); return; }
  if (ees(    DIG)) { Dig                                          return; }
  if (eec(    ';')) { Arg                                          return; }
  for (i=0;i<=n;i++)
  if (epp(       ))   tau( TY_CSI_PR(t,a[i]),    0,   0);          else
                      tau( TY_CSI_PS(t,a[i]),    0,   0);
  reset();
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                               Reporting                                   */
/*                                                                           */
/* ------------------------------------------------------------------------- */

/*! shows the contents of the scan buffer.

    This functions is used for diagnostics. It is called by \e ReportErrorToken
    to inform about strings that cannot be decoded or handled by the emulation.

    \sa ReportErrorToken
*/

/*!
*/

static void hexdump(char* s, int len)
{ int i;
  for (i = 0; i < len; i++)
  {
    if (s[i] == '\\') 
      printf("\\\\");
    else
    if ((s[i]&0xff) > 32) 
      printf("%c",s[i]);
    else
      printf("\\%02x",s[i]&0xff);
  }
}

void VT102Emulation::scan_buffer_report()
{
  if (ppos == 0 || ppos == 1 && (pbuf[0] & 0xff) >= 32) return;
  printf("token: "); hexdump((char*)pbuf,ppos); printf("\n");
}

/*!
*/

void VT102Emulation::ReportErrorToken()
{
  printf("undecodable "); scan_buffer_report();
}

void VT102Emulation::NotImplemented()
{
  printf("not implemented "); scan_buffer_report();
}

/*!
*/

void VT102Emulation::sendString(char* s)
{
  emit sndBlock(s,strlen(s));
}

/*!
*/

void VT102Emulation::reportTerminalType()
{
  sendString("\033[?c"); // I'm a BBS VT102 Terminal
}

/*!
*/

void VT102Emulation::reportStatus()
{
  sendString("\033[0n"); //FIXME: what's that?
}

/*!
*/

void VT102Emulation::reportCursorPosition()
{ char tmp[20];
  sprintf(tmp,"\033[%d;%dR",scr->getCursorY()+1,scr->getCursorX()+1);
  sendString(tmp);
}

// Mode Operations -----------------------------------------------------------

// NOTE: experimental section.
//       This is due to the fact that modes have to be handled both on
//       emulation and screen level. Dought that we are able to clean
//       this up.

void VT102Emulation::setMode(int m)
{
  currParm.mode[m] = TRUE;
  switch (m)
  {
    case MODE_AppScreen : setScreen(1); break;
  }
}

void VT102Emulation::resetMode(int m)
{
  currParm.mode[m] = FALSE;
  switch (m)
  {
    case MODE_AppScreen : setScreen(0); break;
  }
}

void VT102Emulation::saveMode(int m)
{
  saveParm.mode[m] = currParm.mode[m];
}

void VT102Emulation::restoreMode(int m)
{
  if(saveParm.mode[m]) setMode(m); else resetMode(m);
}

//NOTE: this is a helper function
BOOL VT102Emulation::getMode(int m)
{
  return currParm.mode[m];
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                Miscelaneous                               */
/*                                                                           */
/* ------------------------------------------------------------------------- */

/*! change between primary and alternate screen
*/

void VT102Emulation::setScreen(int n)
{
  scr = screen[n&1];
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                Mouse Handling                             */
/*                                                                           */
/* ------------------------------------------------------------------------- */

void VT102Emulation::onMouse( int cb, int cx, int cy )
{
  if (getMode(MODE_Mouse1000)) // ?1000 or ?9 on
  { char tmp[20];
    printf("mouse: cb=%d cx=%d cy=%d\n",cb,cx,cy);
    sprintf(tmp,"\033[M%c%c%c",cb+040,cx+040,cy+040);
    sendString(tmp);
  }
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                             Keyboard Handling                             */
/*                                                                           */
/* ------------------------------------------------------------------------- */

/* Structure to hold escape sequences. */
struct keycode { int code; char *sequ; };

#include "other/xterm.keycodes"

static struct keycode* emu_key = xterm_key;

/*!
*/

void VT102Emulation::onKeyPress( QKeyEvent* ev )
{ int key;
  if (!active) return;
  // Note: there 3 ways in rxvt to handle the Meta (Alt) key
  //       1) ignore it
  //       2) preceed the keycode by ESC (what we do here)
  //       3) set the 8th bit of each char in string 
  //          (which may fail for 8bit (european) characters.
  if (ev->state() & AltButton) sendString("\033"); // ESC

  // FIXME: What follows here is a hack around the hack of
  //        generations. The only real choice is to reconfigure
  //        or fix all offending programs.
  //        I start a list of things that i've noticed:
  //        - EMACS: BS(0x08) == C-h == HELP
  //               : 

  // I'm really in the mood to write a long cynic comment
  // on the BS/DEL hack in Linux (designed for VT100).
  //
  // - using 0x08 for BS brakes EMACS (C-h) and LESS (line-edit)
  //   so we finally make the brain dead twist: BS ==> DEL

  key = ev->key();
  switch (key)
  {

    // if you think BS is BS and DEL is DEL,
    // you have never met programs developed on VT100.

    case Key_Backspace : sendString("\x7f"); return;

    // here comes another VT100 "feature"
    // it emits different codes codes in different modes
    // some programs (less) even make use of that

    // FIXME: the keycodes could also be taken from a table
    case Key_Up   : sendString(getMode(MODE_AppCuKeys)?"\x1bOA":"\x1b[A"); return;
    case Key_Down : sendString(getMode(MODE_AppCuKeys)?"\x1bOB":"\x1b[B"); return;
    case Key_Right: sendString(getMode(MODE_AppCuKeys)?"\x1bOC":"\x1b[C"); return;
    case Key_Left : sendString(getMode(MODE_AppCuKeys)?"\x1bOD":"\x1b[D"); return;

  }
  if( (ev->state() & ControlButton) && ev->key() == Key_Space) // Ctrl-Space - emacs
  {
//printf("key : "); hexdump("\01",1); printf("\n");
    sndBlock("\x00",1);
  }
//printf("State/Key: 0x%08x 0x%08x\n",ev->state(),key);
  for(int i = 0; emu_key[i].code != 0; i++) if (emu_key[i].code == key)
  {
//printf("table key: "); hexdump(emu_key[i].sequ,strlen(emu_key[i].sequ)); printf("\n");
    sendString(emu_key[i].sequ);
    return;
  }
  if (ev->ascii()>0)
  { unsigned char c[1]; 
    c[0] = ev->ascii(); 
//printf("ansi key: "); hexdump(c,1); printf("\n");
    emit sndBlock((char*)c,1);
    return;
  }
}
