
/*   tubo_static.c  */

/*  A program independent forking object module for gtk based programs.
 *  
 *  Copyright 2000-2010(C)  Edscott Wilson Garcia under GNU GPL
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <sys/time.h>

#ifndef HAVE_SYS_TYPES_H
# error "This program requieres  the <sys/types.h> header file."
#endif
#ifndef HAVE_SYS_WAIT_H
# error "This program requieres  the <sys/wait.h> header file."
#endif
#ifndef HAVE_SIGNAL_H
# error "This program requieres  the <signal.h> header file."
#endif

/* signal handling */
#ifndef HAVE_SIGNAL
# ifndef HAVE_SIGACTION
#  error "This program needs signals to work!"
# endif
#endif

#define H(x)
#define XH(x)

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <semaphore.h>

#include <unistd.h>
#include <stdarg.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/select.h>


/* public stuff */
#include "tubo.h"
# define NO_RFM_GLOBAL_P 1
//#include "debug.h"
#ifdef TRACE
# undef TRACE
# define TRACE(...) fprintf(stderr, __VA_ARGS__)
#else
# define TRACE(...)
#endif
#ifdef DBG
# undef DBG
# define DBG(...) fprintf(stderr, __VA_ARGS__);fflush(stdout)
#else
# define DBG(...)
#endif

/* private stuff */

/* undefine this variable to leave zombies behind
 * (this will allow you follow program path in debugging): */
//#define FORK_OFF

typedef struct fork_struct {
    pid_t PID;
    int tubo[3][2];
    void (*fork_function) (void *);
    void *fork_function_data;
    void (*stdout_f) (void *stdout_data, void *stream, int childFD);
    void (*stderr_f) (void *stderr_data, void *stream, int childFD);
    void (*tubo_done_f) (void *);
    void *user_function_data;
    unsigned tubo_id;
    int shm_size;
#ifdef BROKEN_SHARED_SEMAPHORES
    int *remote_semaphore;
#else
    sem_t *remote_semaphore;
#endif
    sem_t *local_semaphore;
    char shm_name[256];
    pthread_mutex_t mutex_p;
    int reap_child;
    int check_valid_ansi_sequence;
#if 0
    time_t now;
    off_t count;
#endif
} fork_struct;

// 16 K, size of a memory page in old silicon graphics- cray boxes.
#define TUBO_BLK_SIZE 128*128

static unsigned tubo_id = 0x01;
// this is correct to be global, since it is only used by the independent child
// process which spawns the command: grandchildPID, thus is thread safe.
pid_t grandchildPID;
char *term_shm_name = NULL;

static char *invalid_sequence_p[] = {
    "(",
    ")",
    "M",
    "E",
    "7",
    "8",
    "H",
    "[A",
    "[g",
    "[0g",
    "[3g",
    "#3",
    "#4",
    "#5",
    "#6",
    "5n",
    "6n",
    "[c",
    NULL
};


static 
int
valid_ansi_sequence(char *line){
    char *esc=line;

    while (esc != NULL && *esc != 0) {
        esc=strchr(esc,0x1b);
        if (esc==NULL || *esc==0) return 1;
        esc++;
        char **p;
        for(p = invalid_sequence_p; p && *p; p++) {
            if (strncmp(esc,*p,strlen(*p))==0){
		fprintf (stderr, "sequence <ESC>%s is not in valid_ansi_sequence list\n", *p);
		return 0;
            }
        }
    }
    return 1;
}



static void
signalit (int sig) {
    TRACE ("Tubo: got signal %d (SIGUSR1=%d,SIGUSR2=%d)\n", sig, SIGUSR1, SIGUSR2);

    if(sig == SIGUSR1)
        kill (grandchildPID, SIGTERM);
    if(sig == SIGUSR2)
        kill (grandchildPID, SIGKILL);
    // we want to shm_unlink no longer used memory when we terminate:
    if(sig == SIGTERM) {
        if(term_shm_name) {
            shm_unlink (term_shm_name);
        }
        exit (123);
    };
    return;

}

static void
signal_connections (void
    ) {
#ifdef HAVE_SIGACTION
    struct sigaction act;
    act.sa_handler = signalit;
    sigemptyset (&act.sa_mask);
# ifdef SA_RESTART
    act.sa_flags = SA_RESTART;
# else
    act.sa_flags = 0;
# endif

//      sigaction(SIGHUP,&act,NULL);
    sigaction (SIGTERM, &act, NULL);
//      sigaction(SIGINT,&act,NULL);
//      sigaction(SIGQUIT,&act,NULL);
//      sigaction(SIGABRT,&act,NULL);
    sigaction (SIGUSR1, &act, NULL);
    sigaction (SIGUSR2, &act, NULL);

#else
//      signal(SIGHUP, signalit);
    signal (SIGTERM, signalit);
//      signal(SIGINT, signalit);
//      signal(SIGQUIT, signalit);
//      signal(SIGABRT, signalit);
    signal (SIGUSR1, signalit);
    signal (SIGUSR2, signalit);
#endif
}

/* memcpy is necesary  */
#ifdef __GNUC__
/* memcpy is a GNU extension.*/
# define MEMCPY memcpy
#else
static
void *
MEMCPY (void *dest, const void *src, size_t n) {
    char *destC,
     *srcC;
    size_t i;

    destC = (char *)dest;
    srcC = (char *)src;
    for(i = 0; i < n; i++)
        destC[i] = srcC[i];
    return dest;
}
#endif

static
fork_struct *
TuboClosePipes (fork_struct * forkO) {
    int i;
    TRACE ("Tubo: ClosePipes()\n");
    if(!forkO) {
        return NULL;
    }
    for(i = 0; i < 3; i++) {
        if(forkO->tubo[i][0] > 0) {
            close (forkO->tubo[i][0]);
            forkO->tubo[i][0] = 0;
        }
        if(forkO->tubo[i][1] > 0) {
            close (forkO->tubo[i][1]);
            forkO->tubo[i][1] = 0;
        }
    }
    return NULL;
}

static pthread_mutex_t  
stdmutex = PTHREAD_MUTEX_INITIALIZER;

static
    int
read_fd (int which, void * data) {
    fork_struct *fork_p = (fork_struct *) data;
    char line[TUBO_BLK_SIZE];
    memset (line, 0, TUBO_BLK_SIZE);
    if(which < 1 || which > 2){
        fprintf (stderr, "Tubo: read_fd(): argument out of range\n");
        _exit (1);
    }
    int i;

    for(i = 0; i < TUBO_BLK_SIZE - 1; i++) {
        int result = read (fork_p->tubo[which][0], line + i, 1);
        if(result < 0) {
            fprintf (stderr, "Tubo: read_fd(%d->%d) %s\n", which, fork_p->tubo[which][0], strerror (errno));
            return 0;
        }
        if(result == 0) {
            return 0;
        }
        if(*(line + i) == '\n') {
            *(line + i + 1) = (char)0;
            break;
        }
    }

// check for valid output (if so configured)
    if (fork_p->check_valid_ansi_sequence) {
	if (!valid_ansi_sequence(line)){
	    fprintf(stderr, "Sending SIGTERM to child process (check_valid_ansi_sequence==TRUE)\n");
	    // send child TERM signal: SIGUSR1 gets 
	    // translated to SIGTERM for grandchild
	    kill (fork_p->PID, SIGUSR1); 	    
	    // ignore data
	    return 1;
	}
    }

    pthread_mutex_lock(&stdmutex);

    if(which == 2 && fork_p->stderr_f) {
        (*(fork_p->stderr_f)) (fork_p->user_function_data, (void *)line, fork_p->tubo[0][1]);
    } else if(which == 1 && fork_p->stdout_f) {
        (*(fork_p->stdout_f)) (fork_p->user_function_data, (void *)line, fork_p->tubo[0][1]);
    }
    pthread_mutex_unlock(&stdmutex);
    return 1;

}

static
    void
stdio_f (int which, void * data) {
    fork_struct *fork_p = (fork_struct *) data;

    TRACE ("Tubo: stdio(%d) thread setup!\n", which);
    //map_remote_semaphores(fork_p);
    // greenlight to exec:
    pthread_mutex_lock (&(fork_p->mutex_p));
#ifdef BROKEN_SHARED_SEMAPHORES
    fork_p->remote_semaphore[which - 1] = 1;
#else
    sem_post (fork_p->remote_semaphore + (which - 1));
#endif
    TRACE ("Tubo: stdio_f sem_post to remote_semaphore %d\n", which - 1);
    if(msync (fork_p->remote_semaphore, fork_p->shm_size, MS_SYNC) < 0){
        fprintf (stderr, "msync: %s\n", strerror (errno));
    }
    pthread_mutex_unlock (&(fork_p->mutex_p));

    while(read_fd (which, (void *)fork_p)) ;
    TRACE ("Tubo: closing fd(%d)-> %d\n", which, fork_p->tubo[which][0]);
    close (fork_p->tubo[which][0]);
    // greenlight to fork_finished function:
    TRACE ("Tubo: stdio_f sem_post to local_semaphore %d\n", which - 1);
    sem_post (fork_p->local_semaphore + (which - 1));
}

static
  void *
stdout_thread_f (void * data) {
    stdio_f (1, data);
    return NULL;
}

static
  void *
stderr_thread_f (void * data) {
    stdio_f (2, data);
    return NULL;
}

static void *
threaded_wait_f (void * data) {
    fork_struct *fork_p = (fork_struct *) data;
    int status;
    //map_remote_semaphores(fork_p);
    TRACE ("Tubo: thread=0x%x, wait for 0x%x\n", (unsigned)getpid (), (unsigned)(fork_p->PID));

    if(fork_p->reap_child) {
        waitpid (fork_p->PID, &status, 0);
    } else {
        // leave child in waitable state...
        // no such thing in BSD
#ifdef HAVE_WAITID
        siginfo_t infop;
        waitid (P_PID, fork_p->PID, &infop, WNOWAIT);
#endif
    }
    TRACE ("Tubo: threaded_wait_f complete for 0x%x\n", (unsigned)(fork_p->PID));

    TRACE ("Tubo: threaded_wait_f waiting for semaphore 1...\n");
    sem_wait (fork_p->local_semaphore + 1);
    TRACE ("Tubo: threaded_wait_f waiting for semaphore 0...\n");
    sem_wait (fork_p->local_semaphore);
    TRACE ("Tubo: semaphores OK!\n");

    if(fork_p->tubo_done_f) {
        TRACE ("Tubo: running tubo_done_f() now...\n");
        (*(fork_p->tubo_done_f)) (fork_p->user_function_data);
    }
    // finally, close stdin. 
    if(fork_p->tubo[0][1] > 0)
        close (fork_p->tubo[0][1]);
    pthread_mutex_destroy (&fork_p->mutex_p);
    // this map is no longer needed by the threads:
    munmap (fork_p->remote_semaphore, fork_p->shm_size);
    /* shm_unlink is done by the child... */
    /*TRACE("Tubo: shm_unlink(%s)\n",fork_p->shm_name); */
    /*shm_unlink(fork_p->shm_name); */
    if (fork_p->local_semaphore) free (fork_p->local_semaphore);
    if (data) free (data);
    return NULL;
}

void
tubo_threadwait (void) {
    struct timespec thread_wait = {
        0, 100000000
    };
    nanosleep (&thread_wait, NULL);
}
