//
//
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#define READ_ERROR -1

#if 0
typedef struct thread_stat_t{
    const gchar *path;
    struct stat st;
    gboolean done;
    gboolean cleanup;
} thread_stat_t;

static void *
thread_stat_f(gpointer data){
    thread_stat_t *thread_stat_p = data;

    if (!thread_stat_p) g_error("thread_stat_p cannot be NULL\n");
    if (stat(thread_stat_p->path, &thread_stat_p->st) == 0) {
	thread_stat_p->done = TRUE;
    }
    while (thread_stat_p->cleanup != TRUE) rfm_threadwait();
    g_free(thread_stat_p);
    return NULL;
}

static
gboolean
thread_stat(const gchar *path, struct stat *st_p) {
    thread_stat_t *thread_stat_p = (thread_stat_t *)malloc(sizeof(thread_stat_t));
    if (!thread_stat_p){
	g_error("cannot malloc heartbeat_p: %s\n",strerror(errno));
    }
    memset(thread_stat_p, 0, sizeof(thread_stat_t));
    thread_stat_p->path = path;
    THREAD_CREATE(thread_stat_f, thread_stat_p, "thread_stat_f");
    gint heartbeat=0;
    do {
	rfm_threadwait();
	heartbeat++;
    } while (!thread_stat_p->done && heartbeat < 5);

    if (thread_stat_p->done){
	memcpy(st_p, &thread_stat_p->st, sizeof(struct stat));
	thread_stat_p->cleanup = TRUE;
	return TRUE;
    } 
    thread_stat_p->cleanup = TRUE;
    return FALSE;
}
#endif

/**
 * xfdir_set_entry_tag:
 * @src_en: a valid record_entry_t pointer
 * @tama: Size in bytes.
 * 
 * Sets the entry's tag (which will show up on the status line).
 *
 **/
static void
xfdir_set_entry_tag (widgets_t *widgets_p, record_entry_t * en, off_t tama) {
    int hcount = xfdir_count_hidden_files (en->path);
    int count = xfdir_count_files (en->path);
    gchar *b = g_path_get_basename (en->path);
    gchar *q = rfm_utf_string (b);
    g_free (b);
    gchar *s = NULL;
    g_free (en->tag);

    if (en->module) {
	s=rfm_rational(PLUGIN_DIR, en->module, widgets_p, en, "sizetag");
    } else {
	s = rfm_sizetag (tama, count);
    }
    

    if (hcount) {
        gchar *plural_string = g_strdup_printf(ngettext ("%'u item","%'u items",hcount), hcount);
        en->tag = g_strdup_printf ("%s (%s %s: %s)", q, (s)?s:"", _("Hidden"), plural_string);
        g_free(plural_string);
    } else if (s) {
        en->tag = g_strdup_printf ("%s (%s)", q, s);
    } else {
        en->tag = g_strdup_printf ("%s", q);
    }
    
    g_free (s);
}



// check if monitor should wait:
/**
 * xfdir_monitor_skip:
 * @xfdir_p: pointer to an xfdir_t structure
 * Returns: TRUE if monitor should skip tests on current loop
 * or FALSE if monitor should go ahead with the update tests.
 * 
 * Checks whether the xfdir monitor should skip monitor tests on the current
 * loop or go ahead with monitor tests. It is an exported symbol so that modules
 * may use the same monitor algorithm as the xfdir monitor thread.
 **/
static
  gboolean
xfdir_monitor_skip (xfdir_t * xfdir_p) {
    gboolean skip = FALSE;
    if(xfdir_p == NULL) {
        g_error ("monitor_skip: xfdir_p ==NULL!");
    }
    view_t *view_p = xfdir_p->view_p;


    g_mutex_lock (view_p->mutexes.monitor_loop);
    if(view_p->flags.refresh) {
        skip = TRUE;
        NOOP(stdout, "thread monitor is skipping: xfdir_p->refresh\n");
    }
    g_mutex_unlock (view_p->mutexes.monitor_loop);
	    
    if (view_p->flags.thumbnailer_active) return TRUE;

    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if (status == STATUS_EXIT) return TRUE;
    
    /* popup active? */
    GtkWidget *w = view_p->widgets.popup;
    if(w && gtk_widget_get_visible (w)) {
        skip = TRUE;
        NOOP (stderr, "thread monitor: popup visible\n");
    }
    // private (module) popup active?
    w =	g_object_get_data(G_OBJECT(view_p->widgets.paper), 
		"private_popup_widget");
    if(w && gtk_widget_get_visible (w)) {
        skip = TRUE;
        NOOP (stderr, "thread monitor: private_popup_widget visible\n");
    }

    /* entry window is mapped? */
    else if(view_p->widgets.rename) {
        skip = TRUE;
        NOOP (stderr, "thread monitor: entry window is mapped\n");
    }
    /* tip window is mapped? This could be disabled, but better be safe. */

    else if(view_p->tip_event.tooltip_active) {
        // But is it a valid active state?
	// State should be valid here since
	// condition is taken care of in rodent_mouse.c
	// with the on_leave signal
        NOOP (stderr, "thread monitor: widgets.tooltip_active\n");
        skip = TRUE;
    }
    /* button down? */
    else if(view_p->mouse_event.boxX != -1 && view_p->mouse_event.boxY != -1) {
        skip = TRUE;
        NOOP(stdout, "thread monitor: mouse_event.boxX != -1\n");
    }
    /* dragging? */
    else if(view_p->mouse_event.doing_drag_p){
        skip = TRUE;
    } 
#ifdef DEBUG_TRACE
    static gboolean said = FALSE;
    /* saving cache? */
    if(!skip && said) {
        said = FALSE;
        NOOP ("thread monitor: active\n");
    }
    if(!skip && said) {
        said = FALSE;
        NOOP ("thread monitor: active\n");
    }
    if(skip) {
        if(!said) {
            said = TRUE;
            NOOP ("thread monitor: sleeping\n");
        }
    }
#endif
    return skip;
}

static
  gboolean
xfdir_monitor_disabled (xfdir_t * xfdir_p) {
    gboolean disabled = FALSE;
    if(xfdir_p == NULL) {
        g_error ("monitor_skip: xfdir_p ==NULL!");
    }
    view_t *view_p = xfdir_p->view_p;

    // notebook page not visible?
    if (view_p->flags.type != DESKVIEW_TYPE){
	GtkNotebook *notebook = GTK_NOTEBOOK(*(view_p->widgets.notebook));
	gint current_page = gtk_notebook_get_current_page (notebook);
	GtkWidget *child =gtk_notebook_get_nth_page(notebook, current_page);
	view_t *current_view_p = g_object_get_data (G_OBJECT (child), "view_p");
	if (view_p != current_view_p) {
	    TRACE("xfdir_monitor_disabled: disabled on non visible page\n");
	    disabled = TRUE;
	}
    }
    return disabled;
}


/**
 * xfdir_free_data:
 * @xfdir_p:  pointer to an xfdir_t structure
 *
 *  Frees allocated resources of a loaded xfdir structure
 *
 **/
gint
xfdir_free_data (xfdir_t * xfdir_p) {

    if(!xfdir_p) {
        g_warning ("xfdir_p==NULL in xfdir_free_data");
        return -1;
    }
    if(xfdir_p->gl != NULL && xfdir_p->pathc) {
        int i;
        for(i = 0; i < xfdir_p->pathc; i++) {
            NOOP ("thread monitor: will free xfdir_p->gl[i].pathv=0x%lx\n", (long unsigned)xfdir_p->gl[i].pathv);
            g_free (xfdir_p->gl[i].pathv);
            NOOP ("thread monitor: will destroy xfdir_p->gl[i].en=0x%lx\n", (long unsigned)xfdir_p->gl[i].en);
            rfm_destroy_entry (xfdir_p->gl[i].en);
        }
        NOOP ("thread monitor: will free xfdir_p->gl=0x%lx\n", (long unsigned)xfdir_p->gl);
        g_free (xfdir_p->gl);
        xfdir_p->gl = NULL;
        xfdir_p->pathc = 0;
    }
    NOOP ("thread monitor: will free xfdir_p->data=0x%lx\n", (long unsigned)xfdir_p->data);
    rfm_destroy_entry (xfdir_p->en);
    g_free (xfdir_p->data);
    return 0;
}

static
  gboolean
exit_monitor (xfdir_t * xfdir_p) {
    gboolean result = FALSE;
    view_t *view_p = xfdir_p->view_p;
    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if(status == STATUS_EXIT) return TRUE;
    g_mutex_lock (view_p->mutexes.monitor_id);
    if(view_p->flags.monitor_id != xfdir_p->monitor_id)
        result = TRUE;
    g_mutex_unlock (view_p->mutexes.monitor_id);
    return result;
}

/**
 * xfdir_monitor_continue:
 * @xfdir_p:  pointer to an xfdir_t structure
 * Returns: TRUE if monitor should continue alive or FALSE if monitor should 
 * finish.
 *
 * This function is a generic check whether the local file monitor thread
 * should continue alive or exit. It is an exported symbol so that modules
 * may use the same monitor algorithm as the xfdir monitor thread.
 **/
static
  gboolean
xfdir_monitor_continue (xfdir_t * xfdir_p) {
    if(!xfdir_p) {
        NOOP ("thread monitor: monitor_continue !xfdir_p\n");
        return FALSE;
    }
    view_t *view_p = xfdir_p->view_p;
    /* signaled exit: */
    if(exit_monitor (xfdir_p)) {
        NOOP ("thread monitor: exit_monitor(xfdir_p) = 1\n");
        return FALSE;
    }
    /* invalid view: */
    if(!view_p)
        return FALSE;
    /* root window */
    if(!view_p->en) {
        NOOP ("thread monitor: exit on !view_p->en\n");
        return FALSE;
    }
    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if(status == STATUS_EXIT) return FALSE;
    
    /* directory has vanished */
    if(xfdir_p->en->module==NULL) {
	if(xfdir_p->en->path == NULL || !rfm_g_file_test (xfdir_p->en->path, G_FILE_TEST_EXISTS)) {
	    // this is the test for local files.
	    NOOP ("monitor: !rfm_g_file_test(xfdir_p->en->path,G_FILE_TEST_EXISTS)\n");
	    return FALSE;
	}
    } else {// a module defined test...
    } 
    return TRUE;
}

static off_t
st_sum (struct stat *st) {
    off_t sum;
    if(!st)
        sum = 0;
    else {
        sum = st->st_mtime + st->st_size + st->st_mode + st->st_nlink + st->st_uid + st->st_gid;
    }
    return sum;
}

static
int update_xfdir (xfdir_t * xfdir_p, gboolean fullstat, gint *heartbeat);

static gint
count_files_local (xfdir_t * xfdir) {
    DIR *directory;
    gint count = 0;
    struct dirent *d;

    directory = opendir (xfdir->en->path);
    if(!directory){
	NOOP("xfdir monitor: read_files_local(): unable to open directory! False count==0\n");
        return READ_ERROR;
    }

    NOOP(stderr, "read_files_local: SHOWS_HIDDEN (xfdir->en->type)=%d\n", 
	    SHOWS_HIDDEN (xfdir->en->type));

    while((d = readdir (directory)) != NULL) {
        if(strcmp (d->d_name, ".") == 0) {
	    continue;
	}
        if(!SHOWS_HIDDEN (xfdir->en->type) && 
		d->d_name[0] == '.' && strcmp (d->d_name, "..") != 0) {
	    continue;
	}
        count++;
    }
    closedir (directory);
    // At least the ../ record should have been read. If this
    // is not so, then a read error occurred.
    // (not uncommon in bluetoothed obexfs)
    if (!count) {
	NOOP("xfdir monitor: Count failed! ../ not counted!\n");
	return READ_ERROR;
    }
    return (count);
}


typedef struct xd_t{
    gchar *d_name;
#ifdef HAVE_STRUCT_DIRENT_D_TYPE
    unsigned char d_type;
#endif
}xd_t;

static GSList *
read_files_local (xfdir_t * xfdir_p, gint *heartbeat) {
    if(xfdir_p == NULL) return  NULL;
    GSList *directory_list = NULL;
    NOOP( "read_files_local: SHOWS_HIDDEN (xfdir->en->type)=%d\n", 
	    SHOWS_HIDDEN (xfdir_p->en->type));

    DIR *directory = opendir(xfdir_p->en->path);
    if (!directory) {
	DBG("read_files_local(): Cannot open %s\n", xfdir_p->en->path);
	return NULL;
    }

// http://womble.decadent.org.uk/readdir_r-advisory.html

#if defined(HAVE_FPATHCONF) && defined(HAVE_DIRFD)
    size_t size = offsetof(struct dirent, d_name) + 
	fpathconf(dirfd(directory), _PC_NAME_MAX) + 1;
#else
    size_t size = offsetof(struct dirent, d_name) +
	pathconf(xfdir_p->en->path, _PC_NAME_MAX) + 1;
#endif

    struct dirent *buffer = (struct dirent *)malloc(size);
    if (!buffer) g_error("malloc: %s\n", strerror(errno));

    gint error;
    struct dirent *d;
    while ((error = readdir_r(directory, buffer, &d)) == 0 && d != NULL){
        if(strcmp (d->d_name, ".") == 0) continue;
        if(!SHOWS_HIDDEN (xfdir_p->en->type) && 
		d->d_name[0] == '.' && strcmp (d->d_name, "..") != 0) {
	    continue;
	}
	xd_t *xd_p = (xd_t *)malloc(sizeof(xd_t));
	xd_p->d_name = g_strdup(d->d_name);
#ifdef HAVE_STRUCT_DIRENT_D_TYPE
	xd_p->d_type = d->d_type;
#endif
	directory_list = g_slist_prepend(directory_list, xd_p);
	if (heartbeat) {
	    (*heartbeat)++;
	    NOOP("incrementing heartbeat records to %d\n", *heartbeat);
	}
    }
    if (error) {
        DBG("readdir_r: %s\n", strerror(errno));
    }

    closedir (directory);

    g_free(buffer);

    // At least the ../ record should have been read. If this
    // is not so, then a read error occurred.
    // (not uncommon in bluetoothed obexfs)
    if (!directory_list) {
	DBG("read_files_local(): Count failed! Directory not read!\n");
    }
    return (directory_list);
}


static off_t
st_sum_dir (struct stat *st) {
    off_t sum;
    if(!st) return 0;

    sum = st->st_ino + st->st_mtime + st->st_ctime + st->st_size + st->st_mode + st->st_nlink + st->st_uid + st->st_gid;
    
    return sum;
}

// this function checks if a directory (or module) currently in 
// xfdir structure needs to be updated in the view (in other words,
// is a new xfdir structure necessary?)
static
  gboolean
reload_condition (xfdir_t * xfdir_p, gint * current_count, struct stat *current_st) {
    gboolean instant_return = FALSE;
    view_t *view_p = xfdir_p->view_p;
    record_entry_t *en = xfdir_p->en;
    const gchar *module = en->module;

    // Test 1. Short circuit on instruction from another thread.
    //
    // put a lock on xfdir structure and use the reload variable
    // to shortcircuit test on condition set by other threads
    g_mutex_lock (view_p->mutexes.monitor_loop);
    instant_return = xfdir_p->reload;
    xfdir_p->reload = FALSE;
    g_mutex_unlock (view_p->mutexes.monitor_loop);
    if(instant_return){
	NOOP(stderr, "xfdir monitor: reload_condition instant_return\n");
        return TRUE;
    }

    // Test 2. inode count condition.
    // This will count the number of icons in the view.
    gint instant_count=0;

    if(module) {
        // module defined count test.
 	instant_count = GPOINTER_TO_INT(rfm_natural(PLUGIN_DIR, module,
		    &(view_p->widgets), "module_count"));
	NOOP (stderr, "reload_condition: current count= %d, instant=%d\n",
		*current_count, instant_count);
    } else {
        // local directory test 
        instant_count = count_files_local (xfdir_p);
    }
    if (instant_count == 0) return FALSE;
    if (instant_count == NO_RELOAD_COUNT){
	// This is a direct request to skip the test.
	return FALSE;
    }
    if (instant_count == READ_ERROR){
	// This is a read error. So we skip the test.
	NOOP(stderr, "xfdir monitor: READ_ERROR: reload condition false...\n");
	instant_count = *current_count;
	return FALSE;
    }

    NOOP ("MONITOR current count= %d, instant=%d\n", *current_count, instant_count);
    if(*current_count != instant_count) {
	NOOP(stderr, "reload_condition:  COUNT  %d != %d\n", 
	    *current_count, instant_count);
	// We update the current count, otherwise the reload condition
	// will not cease, and it must cease to trigger the actual reload.
	*current_count=instant_count;
	return TRUE;
    }

    // Stat test is broken for obexfx, because stat information changes each 
    // second... Monitor, nonetheless, is disabled for non-local connections,
    // such as obex.
    //
    //
    //
    // Test 3. Check stat information of current view.
    struct stat instant_st;
    if(module) {
	memset(&instant_st, 0, sizeof(struct stat));
        // module defined test.
	if (rfm_rational(PLUGIN_DIR, module, NULL, NULL, "module_stat")
		&&
	    !rfm_rational(PLUGIN_DIR, module, NULL, &instant_st, "module_stat"))
	{
	    DBG ("reload_condition(): module_stat not correctly defined for %s\n", 
		    module);
	}
	NOOP(stderr, "xfdir monitor: MODULE_STAT current= %d, instant=%d\n",
	    (gint)st_sum_dir (current_st), (gint)st_sum_dir (&instant_st));
    } else {
        // local file test
        if (stat(en->path, &instant_st) < 0) return FALSE;
        //if (!thread_stat(en->path, &instant_st)) return FALSE;
    }
    NOOP("xfdir monitor: current= %d, instant=%d\n",
	    (gint)st_sum_dir (current_st), (gint)st_sum_dir (&instant_st));

    if(current_st && st_sum_dir (current_st) != st_sum_dir (&instant_st)) {
 	// We update the current stat, otherwise the reload condition
	// will not cease, and it must cease to trigger the actual reload.
        NOOP (stderr, "reload_condition(): true on STAT condition\n");
        memcpy (current_st, &instant_st, sizeof (struct stat));
        return TRUE;
    }

    // Test 4. Module specific test: comparison of character data 
    // 	       in xfdir structure.
    if(module && xfdir_p->data) {
        NOOP ("3. xfdir_p->en->module\n"); 
        void *data = rfm_natural(PLUGIN_DIR, module, xfdir_p, "module_data");
        gboolean different = data && 
	    strcmp ((gchar *)(xfdir_p->data), (gchar *)data) != 0;
        if(different) {
            g_free (xfdir_p->data);
            xfdir_p->data = data;
            NOOP(stderr, "reload_condition(): true on module DATA condition\n");
            return TRUE;
        }
        g_free (data);
    }

    return FALSE;
}


// If a full reload is not necessary, this function just updates
// stat information and thumbnails.
static gboolean
update_stat_info (xfdir_t * xfdir_p) {
    int i;
    view_t *view_p = xfdir_p->view_p;
    gboolean something_changed=FALSE;

    NOOP(stderr, "update_stat_info[%s]\n", xfdir_p->en->path );
   

    // We skip first element (which is the dummy "up" icon).
    for(i = 0; i < xfdir_p->pathc; i++) {
	if (xfdir_p->gl[i].en == NULL) continue;
	if (xfdir_p->gl[i].en->st == NULL) continue;
	// This has to be done on local and remote items to see if icon
	// need to be updated. Remote items my be present if we
	// are monitoring a local directory with a remote directory
	// mount point. This needs to be monitored to see if the
	// mount emblem need to be set or unset.
	//
	// Is the work still necessary or has the thread met an exit condition?
        if(exit_monitor (xfdir_p)){
            return FALSE;
	}
	// This is a safeguard, it really should not happen though.
        if(xfdir_p->gl[i].en == NULL){
	    DBG("update_stat_info(): this should not occur\n");
            continue;
	}


	// type might change, say a directory becomes a file, or viceversa...
	record_entry_t *en = 
	    rfm_stat_entry (xfdir_p->gl[i].en->path, xfdir_p->gl[i].en->type);

	// Has item changed from local to remote?
	// This will happen for mount points of remote locations.
	// Also, monitor any other changes on filetype...
	// such as the original a symlink and the new one no...
	// such as mount status changes...
	gint old_type = xfdir_p->gl[i].en->type;
	gint new_type = en->type;
	if (old_type != new_type){
	    NOOP(stderr, "[%s] change of type for %s\n",
		xfdir_p->en->path, xfdir_p->gl[i].en->path);
	    memset(xfdir_p->gl[i].en->st, 0, sizeof(struct stat));
	}   	
        NOOP (stderr, "monitor [%s]:  stat test for %s (%ld ?= %ld)\n",
		xfdir_p->en->path, 
                 xfdir_p->gl[i].en->path, 
		 (long)st_sum (xfdir_p->gl[i].en->st), 
		 (long)st_sum (en->st));
	// This is the stat comparison test.
        if(xfdir_p->gl[i].en->st && st_sum (xfdir_p->gl[i].en->st) != st_sum (en->st)) {
	    // Stat information update is necessary.
            NOOP (stderr, "monitor [%s]:  stat differs for %s\n",
		xfdir_p->en->path, xfdir_p->gl[i].en->path);
	    // Step 1. Update stat information in xfdir structure.
            memcpy (xfdir_p->gl[i].en->st, en->st, sizeof (struct stat));
	    // Step 2. Update entry type.
	    gint keep_type = (xfdir_p->gl[i].en->type) & KEEP_TYPE;
	    xfdir_p->gl[i].en->type = (en->type|keep_type);
	    // No more use for comparison entry.
	    rfm_destroy_entry(en); en = NULL;

	    // Step 3. Update stat information in corresponding view population item.
            // A simple read lock is OK for this operation since the 
	    // view_p->population_pp is not affected.
            if (!rfm_population_read_lock (view_p)) {
		return FALSE;
	    }

            if(exit_monitor (xfdir_p)) {
		// Now, once we got the read lock, is the thread doing required
		// work or not? Conditions may have changed while obtaining
		// the read lock.
		DBG("update_stat(): exit_monitor condition\n");
                rfm_population_read_unlock (view_p);
                return FALSE;
            }

            // Do the update on current view population.
            population_t **tmp;
            for(tmp = view_p->population_pp; *tmp; tmp++) {
                population_t *population_p = *tmp;
		gboolean found =  population_p->en
			       && population_p->en->path
			       && strcmp (population_p->en->path, xfdir_p->gl[i].en->path) == 0;
		if (!found) continue;

		something_changed=TRUE; 
		NOOP (stderr, "updating stat record for %s\n", 
			xfdir_p->gl[i].en->path);
		// Update current population item stat information.
		memcpy (population_p->en->st, xfdir_p->gl[i].en->st, sizeof (struct stat));
		population_p->en->type = xfdir_p->gl[i].en->type;
		// remove any previous popup preview image, including the thumbnail file.
		// (see rodent_population.i:265)
		if (population_p->preview_pixbuf) {
		    NOOP (stderr, "removing preview pixbuf at update_stat\n"); 
		    population_p->preview_pixbuf = NULL;
		    gchar *thumbnail = 
			rfm_get_thumbnail_path (population_p->en->path, PREVIEW_IMAGE_SIZE);
		    if (unlink(thumbnail));
		    g_free(thumbnail);
		    // We might get a new preview here.
		}
		// Are we in the list view?
		// If so we will need to update the displayed stat information.
		if (view_p->view_layout.icon_size < TINY_ICON_SIZE/2) {
		    // Recreate population layout
		    xfdir_make_layout2(view_p, population_p);
		    // Expose layout (threaded)
		    GdkRectangle rect;
		    if (rfm_get_population_label_rect(view_p, population_p, &rect))
		    {
			rfm_thread_expose_rect (view_p, &rect);
		    }
		}
		// Update the icon
		// If stat information was updated, update icon as well.
		// This may imply doing magic on the file...
		// This does the actual greenball update.
		NOOP(stderr, "xfdir monitor: updating icon now: %d\n",
			(population_p->en->st)?population_p->en->st->st_uid:0);

		// if condition hack (do not change the icon for content icons):
		if (population_p->en && IS_SDIR(population_p->en->type) &&
			!IS_MOUNTED_TYPE(population_p->en->type) &&
			population_p->content_icon) ;
	
		else {
		    g_free(population_p->icon_id);
	
		    population_p->icon_id = 
		        rfm_get_entry_icon_id (population_p->en, TRUE); 

		    GdkPixbuf *pixbuf = 
			rfm_get_pixbuf (population_p->icon_id, REDUCED_ICONSIZE(view_p));
		    if (population_p->en){
			NOOP(stderr, "updating icon now: %s -> %s mounted=%d\n", 
			    population_p->en->path, population_p->icon_id,
			    (IS_MOUNTED_TYPE(population_p->en->type))?1:0);
		    }
		    if (pixbuf) {
			// If you wonder why we do not unreference the old
			// pixbuf, the reason why is that the reference 
			// belongs to the pixbuf hashtable, not the population_p.
			population_p->pixbuf = pixbuf;
			GdkRectangle rect;
			if (rfm_get_population_rect(view_p, population_p, &rect)) {
			    rfm_thread_expose_rect (view_p, &rect);
			}
		    }
		}

		// Should the referred item have a thumbnail displayed?
		if(view_p->en && (view_p->flags.preferences & __SHOW_IMAGES) &&
			population_p->flags & POPULATION_THUMBNAILED) {
		    // The population_p has a thumbnail and should be redone.
		    population_p->flags &= (POPULATION_THUMBNAILED ^ 0xffffffff);
		    NOOP ("THREAD: thread(rfm_preview_thread_f) at primary-xfdir.i\n");
		    // Now assign memory block which is to be used by the thumbnail
		    // preview thread.
		    fireup_thread_manager_t *fireup_thread_manager_p = 
			(fireup_thread_manager_t *)malloc(sizeof(fireup_thread_manager_t));
		    memset(fireup_thread_manager_p, 0, sizeof(fireup_thread_manager_t));
		    fireup_thread_manager_p->population_p = population_p;
		    fireup_thread_manager_p->flags = population_p->flags;
		    fireup_thread_manager_p->path = g_strdup(population_p->en->path);
		    fireup_thread_manager_p->preview_manager_v.view_p = view_p;
		    fireup_thread_manager_p->preview_manager_v.population_serial = 
			rodent_get_population_serial(view_p);
		    fireup_thread_manager_p->preview_manager_v.diagnostics = 
			view_p->widgets.diagnostics;
		    fireup_thread_manager_p->preview_manager_v.diagnostics_window = 
			view_p->widgets.diagnostics_window;

		    // Fire up preview thread and forget (this will expose).
		    NOOP(stderr, "xfdir monitor: stat change thumbnailing: %s\n", population_p->en->path);
		    THREAD_CREATE (rfm_preview_thread_f, (gpointer) fireup_thread_manager_p, "rfm_preview_thread_f");
		}
		// Since we have found the population_p item, we may
		// exit the search loop.
		break; 
           } // This is the end of the item in population search.
           // Release the population read lock.
           rfm_population_read_unlock (view_p);
	   // Directory is updated
        } // This is the end of the stat comparison test.
	if (en) rfm_destroy_entry(en);
    } // This is the end of the stat comparison loop on all items of xfdir structure
    if (something_changed && 
	    !(xfdir_p->sort_column == NAME_SORT ||xfdir_p->sort_column == TYPE_SORT)) 
    {
	// We need to resort the icons, 
	// so we indicate that a transfer will be necessary.
	return TRUE;
    }
	
    return FALSE;
}

static gint
pathv_compare_up (const gchar * a, const gchar * b) {
    if(!a && !b) return 0;
    if(!a) return -1;
    if(!b) return 1;
    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
    return strcmp (a, b);
}

static gint
pathv_compare_down (const gchar * a, const gchar * b) {
    if(!a && !b) return 0;
    if(!a) return 1;
    if(!b) return -1;
    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
    return strcmp (b, a);
}

static
  gint
entry_compare_up (int caso, record_entry_t * en_a, record_entry_t * en_b) {
    if(!en_a && !en_b) return 0;
    if(!en_a) return -1;
    if(!en_b) return 1;

    if(caso == TYPE_SORT || caso == NAME_SORT) {
	if(!en_a->path && !en_b->path) return 0;
	if(!en_a->path) return -1;
	if(!en_b->path) return 1;
    }

    /* subsorting... */
    if(caso == TYPE_SORT && IS_FILE (en_a->type) && IS_FILE (en_b->type)) {
        gchar *a, *b;
        a = strrchr (en_a->path, '.');
        b = strrchr (en_b->path, '.');
        if(a || b) {
            if(!a) return -1;
            if(!b) return 1;
	    gint result = strcmp (a, b);
            if(result) return result;
        }
    }

    switch (caso) {
	case TYPE_SORT:
	case NAME_SORT: {
	    gchar *a = en_a->path;
	    gchar *b = en_b->path;
	    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
	    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
	    return strcmp (a, b);
	}
	default:
	    if(!en_a->st && !en_b->st) return 0;
	    if(!en_a->st) return -1;
	    if(!en_b->st) return 1;
	    switch (caso) {
		case SIZE_SORT:
		    return en_a->st->st_size - en_b->st->st_size;
		case DATE_SORT:
		    return en_a->st->st_mtime - en_b->st->st_mtime;
		case OWNER_SORT:
		    return en_a->st->st_uid - en_b->st->st_uid;
		case GROUP_SORT:
		    return en_a->st->st_gid - en_b->st->st_gid;
		case MODE_SORT:
		    return en_a->st->st_mode - en_b->st->st_mode;
	    }
    }
    return 0;
}

static gint
entry_compare_down (gint caso, record_entry_t * en_a, record_entry_t * en_b) {
    if(!en_a && !en_b) return 0;
    if(!en_a) return 1;
    if(!en_b) return -1;

    if(caso == TYPE_SORT || caso == NAME_SORT) {
	if(!en_a->path && !en_b->path) return 0;
	if(!en_a->path) return 1;
	if(!en_b->path) return -1;
    }

    /* Type subsorting... */
    if(caso == TYPE_SORT && IS_FILE (en_a->type) && IS_FILE (en_b->type)) {
        gchar *a, *b;
        a = strrchr (en_a->path, '.');
        b = strrchr (en_b->path, '.');
        if(a || b) {
            if(!a) return 1;
            if(!b) return -1;
	    gint result = strcmp (b, a);
            if(result) return result;
        }
    }

    switch (caso) {
	case TYPE_SORT:
	case NAME_SORT:{
	    gchar *a = en_a->path;
	    gchar *b = en_b->path;
	    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
	    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
	    return strcmp (b, a);
	}
	default:
	    if(!en_a->st && !en_b->st) return 0;
	    if(!en_a->st) return 1;
	    if(!en_b->st) return -1;
	    switch (caso) {
		case SIZE_SORT:
		    return en_b->st->st_size - en_a->st->st_size;
		case DATE_SORT:
		    return en_b->st->st_mtime - en_a->st->st_mtime;
		case OWNER_SORT:
		    return en_b->st->st_uid - en_a->st->st_uid;
		case GROUP_SORT:
		    return en_b->st->st_gid - en_a->st->st_gid;
		case MODE_SORT:
		    return en_b->st->st_mode - en_a->st->st_mode;
	    }
    }
    return 0;
}


static gint compare_name_up (const void *a, const void *b);
static gint compare_name_down (const void *a, const void *b);

static gint
null_test (const dir_t *a, const dir_t *b) {
    // Null entry is always on top
    if(!a && !b) return 0;
    if(!a) return -1;
    if(!b) return 1;
    return 2;
}

static gint
dummy_test (const dir_t *d1, const dir_t *d2) {
    if(!d1->en || IS_DUMMY_TYPE (d1->en->type))	return -1;
    if(!d2->en || IS_DUMMY_TYPE (d2->en->type))	return 1;
    return 2;
}
static gint
directory_test (const dir_t *d1, const dir_t *d2) {
    gboolean d1_is_dir = IS_SDIR (d1->en->type);
    gboolean d2_is_dir = IS_SDIR (d2->en->type);
    if(d1_is_dir && !d2_is_dir)  return -1;
    if(!d1_is_dir && d2_is_dir)  return 1;
    return 2;
}

static gint
standard_test (const dir_t *d1, const dir_t *d2){
    // Null entries on top
    gint result = null_test(d1, d2);
    if (result < 2) return result;
    // Dummies go next
    result = dummy_test(d1, d2);
    if (result < 2) return result;
    // Directories follow dummy types.
    result = directory_test(d1, d2);
    if (result < 2) return result;
    return 2;
}


// Date comparison functions.//////////////////////////////////////////
static gint
compare_date_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (DATE_SORT, d1->en, d2->en);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_date_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (DATE_SORT, d1->en, d2->en);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

// SIZE_SORT comparison functions.//////////////////////////////////////////
static gint
compare_size_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (SIZE_SORT, d1->en, d2->en);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_size_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (SIZE_SORT, d1->en, d2->en);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}


// OWNER_SORT comparison functions.//////////////////////////////////////////
static gint
compare_owner_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (OWNER_SORT, d1->en, d2->en);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_owner_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (OWNER_SORT, d1->en, d2->en);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

// GROUP_SORT comparison functions.//////////////////////////////////////////
static gint
compare_group_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (GROUP_SORT, d1->en, d2->en);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_group_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (GROUP_SORT, d1->en, d2->en);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

// MODE_SORT comparison functions.//////////////////////////////////////////
static gint
compare_mode_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (MODE_SORT, d1->en, d2->en);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_mode_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (MODE_SORT, d1->en, d2->en);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}



// Name comparison functions.//////////////////////////////////////////
static gint
compare_name_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method
    result = pathv_compare_up (d1->pathv, d2->pathv);
    return result;
    // If we try to break ties with date comparison,
    // we may run into a circular dependency.
    // return (compare_date_up (d1, d2));
 }

static gint
compare_name_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method 
    result = pathv_compare_down (d1->pathv, d2->pathv);
    return result;
    // If we try to break ties with date comparison,
    // we may run into a circular dependency.
    // return (compare_date_down (d1, d2));
}

// TYPE_SORT comparison functions.//////////////////////////////////////////
static gint
compare_type_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    result = entry_compare_up (TYPE_SORT, d1->en, d2->en);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_type_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    result = entry_compare_down (TYPE_SORT, d1->en, d2->en);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

///////////////////////////////////////////////////////////////////////////////////

/**
 * xfdir_sort:
 * @xfdir_p: pointer to an xfdir_t structure
 *
 * Sorts an xfdir_t structure according to the settings in xfdir_p->ascending 
 * and xfdir_p->sort_column. Ascending is TRUE or FALSE and sort_column may be: 
 * 
 * TYPE_SORT, NAME_SORT, SIZE_SORT,
 * DATE_SORT, OWNER_SORT, GROUP_SORT, MODE_SORT.
 * <para>
 * qsort library routine is used as a sorting method.
 * </para>
 * 
 **/
static void
xfdir_sort (xfdir_t * xfdir_p) {
    NOOP ("reload_iconview: do quicksort here: sortcolumn is %d, ascending=%d\n",
	    xfdir_p->sort_column, xfdir_p->ascending);
    NOOP ("xfdir_p=0x%x, pathc is %d\n", 
	    GPOINTER_TO_INT(xfdir_p), xfdir_p->pathc);

//    set_sort_column (xfdir_p->sort_column);

    gint (*sort_f)(const void *, const void *) = NULL;
    if ((xfdir_p->ascending)) {
	switch (xfdir_p->sort_column) {
	    case NAME_SORT:
		sort_f = compare_name_up;
		break;
	    case SIZE_SORT:
		sort_f = compare_size_up;
		break;
	    case DATE_SORT:
		sort_f = compare_date_up;
		break;
	    case OWNER_SORT:
		sort_f = compare_owner_up;
		break;
	    case GROUP_SORT:
		sort_f = compare_group_up;
		break;
	    case MODE_SORT:
		sort_f = compare_mode_up;
		break;
	    default: // TYPE_SORT
		sort_f = compare_type_up;
		break;
	}
    } else {
	switch (xfdir_p->sort_column) {
	    case NAME_SORT:
		sort_f = compare_name_down;
		break;
	    case SIZE_SORT:
		sort_f = compare_size_down;
		break;
	    case DATE_SORT:
		sort_f = compare_date_down;
		break;
	    case OWNER_SORT:
		sort_f = compare_owner_down;
		break;
	    case GROUP_SORT:
		sort_f = compare_group_down;
		break;
	    case MODE_SORT:
		sort_f = compare_mode_down;
		break;
	    default: // TYPE_SORT
		sort_f = compare_type_down;  
		break;
	}
    }
    
    qsort ((void *)xfdir_p->gl, xfdir_p->pathc, sizeof (dir_t),
#ifdef __COMPAR_FN_T
               (__compar_fn_t)
#endif
               sort_f);

}


// function to do a background reload and set things up for the update
static xfdir_t *
get_new_xfdir (xfdir_t *xfdir_p){
    // allocate memory for reloaded xfdir structure
    xfdir_t *new_xfdir_p = 
	new_xfdir_p = (xfdir_t *) malloc (sizeof (xfdir_t));

    NOOP ("monitor new_xfdir_p=0x%lx\n", (long unsigned)new_xfdir_p);

    // initialize new xfdir structure with current xfdir structure information
    memcpy (new_xfdir_p, xfdir_p, sizeof (xfdir_t));

    // new xfdir structure specific initializations:
    new_xfdir_p->pathc = 0;
    new_xfdir_p->gl = NULL;
    new_xfdir_p->en = rfm_copy_entry(xfdir_p->en);
    // reload directory (or module) into new xfdir structure
    // Since this is background thread, fullstat is best.
    if (update_xfdir (new_xfdir_p, TRUE, NULL) < 0){
	// Error in read step. count is <= 0, but should at least be 1, 
	// where that one would be the "../" record...
	g_free(new_xfdir_p);
	return NULL;
    }

    // sort directory (or module) using current user settings
    xfdir_sort (new_xfdir_p);

    return new_xfdir_p;
}


static gboolean
get_current_stat(record_entry_t *en, struct stat *st){
    memset(st, 0, sizeof(struct stat));
    const gchar *module = en->module;
    if(module) {
	if (rfm_rational(PLUGIN_DIR, en->module, NULL, NULL, "module_stat")
		&&
	    !rfm_rational(PLUGIN_DIR, en->module, NULL, st, "module_stat"))
	{
	    DBG ("get_current_stat(): module_stat in correct for %s\n", 
		module);
	    return FALSE;
	}
    } else {
	if (stat(en->path, st) < 0) return FALSE;
	// return thread_stat(en->path, st);
    }
    return TRUE;
}

// thread_monitor_f is in charge of xfdir_p cleanup!
// AND view_p cleanup!
static gpointer
thread_monitor_f (gpointer data) {
    //return NULL;
    // memory structure should be received from parent thread
    xfdir_t *xfdir_p = (xfdir_t *) data;
    if(xfdir_p == NULL || !xfdir_p->en || !xfdir_p->en->path) {
	g_error("thread_monitor_f(): xfdir_p == NULL || !xfdir_p->path");
    }
    view_t *view_p = xfdir_p->view_p;


    // We ge a copy of the entry, since view entry will become
    // invalid after a reload.
    //xfdir_p->en = rfm_copy_entry(view_p->en);
    TRACE("xfdir monitor: starting thread 0x%x (module=%s)\n", 
	    GPOINTER_TO_INT(g_thread_self()),
	    xfdir_p->en->module);
    increment_view_ref(view_p);
    TRACE("monitor increments view ref: 0x%x\n", GPOINTER_TO_INT(view_p));
	
    g_static_rw_lock_reader_lock(&(rfm_global_p->icon_theme_lock));
 
    // count items present in directory (or module)
    gint current_count = xfdir_p->pathc;
    gboolean refresh_action = FALSE;


    // check whether stat condition has changed on directory (or module)
    struct stat current_st;
    if (!get_current_stat(xfdir_p->en, &current_st)) {
	xfdir_exit_monitor(view_p);
    }

    // Setup to enter inner loop is complete. For this we have a count of 
    // items which should be present (inode count) and a stat structure 
    // with current information.
    NOOP ("monitor %d setup for %s\n", xfdir_p->monitor_id, xfdir_p->en->path);

    do { // inner loop (2)
	// decrement monitor count to allow barrier
	//  (value has been incremented on entering function
	//   and will be incremented on repeating this loop) 
	g_static_rw_lock_reader_unlock(&(rfm_global_p->icon_theme_lock));
	// if lock is requested from another thread, this will block
	// otherwise barrier will be disabled.
	// This pause gives the icontheme thread a break to do her thing.
	g_static_rw_lock_reader_lock(&(rfm_global_p->icon_theme_lock));
	if (xfdir_p->view_p->flags.status == STATUS_EXIT) break;


	NOOP(stderr, "monitor 0x%x past barrier in inner loop... \n",
		GPOINTER_TO_INT(g_thread_self()));
	// check if the inner event loop must be broken
	// this happens if monitor thread will exit or a reload condition
	// has been detected
	if(!xfdir_monitor_continue (xfdir_p)) {
	    NOOP(stdout, "xfdir monitor: !xfdir_monitor_continue\n");
	    //rfm_threadwait ();
	    break;
	}
	NOOP(stdout, "xfdir monitor: xfdir_monitor_continue... \n");

	// Check if the event loop must be placed in suspend mode.
	// Skip mode will bypass control thread wait.
	gboolean skip = xfdir_monitor_skip (xfdir_p);
	if(skip) {
	    NOOP(stdout, "monitor(0x%x): skip = %d\n", GPOINTER_TO_INT(g_thread_self()), skip);
	    rfm_threadwait ();  
	    continue;
	} 

	gboolean disabled = xfdir_monitor_disabled (xfdir_p);;
	if (disabled){
	    NOOP(stdout, "monitor disabled=%d \n", disabled);
	    goto breath;
	}
	// check for an immediate reload condition which requires
	// immediate reload action (gboolean refresh_action)
	if(refresh_action || 
		reload_condition (xfdir_p, &current_count, &current_st))
	{
	    NOOP(stdout, "xfdir monitor: current count is now %d\n",
		    current_count);

	    // Get stat of directory corresponding to the
	    // upcoming reload.
	    if (!get_current_stat(xfdir_p->en, &current_st)){
		xfdir_exit_monitor(view_p);
		continue;
	    }
	    // obtain new xfdir structure
	    g_mutex_lock (view_p->mutexes.monitor_loop);
	    xfdir_t *reloaded_xfdir_p = get_new_xfdir(xfdir_p);
	    g_mutex_unlock (view_p->mutexes.monitor_loop);
	    // Check if reload action was successful (non NULL return value)
	    // And check whether there was an actual reload (new xfdir structure)
	    if (reloaded_xfdir_p && reloaded_xfdir_p != xfdir_p) {

		// safeguard check: if updated xfdir structure is no longer
		// required, discard updated directory 
		if(exit_monitor (xfdir_p)) {
		    NOOP ("monitor now exiting and dumping new_xfdir_p!\n");
		    // rodent no longer needs this.
		    xfdir_free_data (reloaded_xfdir_p);
		    g_free (reloaded_xfdir_p);
		    // rfm_threadwait ();
		    break;
		}
    
		TRACE("monitor now setting up actual refresh!\n");
		xfdir_t *obsolete_xfdir_p = xfdir_p;
		xfdir_p = reloaded_xfdir_p;
		xfdir_free_data (obsolete_xfdir_p);
		g_free (obsolete_xfdir_p);

		current_count = xfdir_p->pathc;
		// RELOAD HERE
		rodent_thread_reload_view (view_p, xfdir_p);
		// No breath...
		rfm_threadwait();
		continue;
	    } else {
		NOOP(stderr, "1. XXXX  doing skipwait\n");
		NOOP(stdout, "!!!!!! xfdir monitor: Ignoring failed background reload 0x%x\n",
			GPOINTER_TO_INT(reloaded_xfdir_p));
		continue;
	    }
	}
	else {
	    // reload condition has not occurred, but stat information
	    // or thumbnail image may have changed for an existing item
	    // without affecting directory stat information. In this case
	    // we update stat information for individual xfdir structure items. 
	    NOOP (stdout, "monitor now checking update_stat_info...\n");
	    if (update_stat_info (xfdir_p)){
		NOOP (stderr, "refresh_pending on update_stat_info()...\n");
		// A TRUE return value indicates that the directory
		// count does not correspond to the actual files 
		// which currently exist and the file count is wrong.
		// In this case we queue a refresh.
		refresh_action = TRUE;
		continue; // this implies no wait on loop cycle.
	    } 
	    if ( xfdir_p->en->module &&
		    rfm_natural(PLUGIN_DIR, xfdir_p->en->module, view_p, "monitor_skipwait"))
	    {
		NOOP(stderr, "XXXX  doing skipwait\n");
		rfm_threadwait ();
		continue; // this implies no wait on loop cycle.
	    } else {
		NOOP(stderr, "XXXX  monitor says skipwait is FALSE\n");
	    }

	}
breath:
	
	// Controlled breath condition
	// Skip mode will not enter here.

	g_mutex_lock(view_p->mutexes.monitor_control);
	//if (view_p->flags.status == STATUS_EXIT) break;
	if (view_p->flags.monitor_go == FALSE) {
#define MONITOR_WAIT_PERIOD 3
#if GLIB_MAJOR_VERSION==2 && GTK_MINOR_VERSION<32
	    GTimeVal tv;
	    g_get_current_time (&tv);
	    tv.tv_sec += MONITOR_WAIT_PERIOD;
	    g_cond_timed_wait(view_p->mutexes.monitor_signal, view_p->mutexes.monitor_control, &tv);
#else
	    gint64 end_time;
	    end_time = g_get_monotonic_time () + MONITOR_WAIT_PERIOD * G_TIME_SPAN_SECOND;
	    g_cond_wait_until (view_p->mutexes.monitor_signal, view_p->mutexes.monitor_control, end_time);
#endif
	} else {
	    view_p->flags.monitor_go = FALSE;
	}
	    
	g_mutex_unlock(view_p->mutexes.monitor_control);


    } while (2);   // end inner loop 2

    DBG("xfdir monitor: monitor id=%d loop broken for %s\n", xfdir_p->monitor_id, xfdir_p->en->path);
    // inner loop has exited.
    //  * monitor thread has met an exit condition and is no longer necessary
    //
    // Here we consider case 1, monitor thread had met an exit condition:
    TRACE ("monitor: all is cool to release xfdir memory\n");
    NOOP(stdout, "monitor: id=%d ending now\n", xfdir_p->monitor_id);
    TRACE ("monitor: will free xfdir_p=0x%lx\n", (long unsigned)xfdir_p);

    // monitor thread ends here: 
    TRACE("monitor decrements view ref: 0x%x\n", GPOINTER_TO_INT(view_p));
    decrement_view_ref(view_p);
    // current xfdir structure (no longer needed) cleanup
    xfdir_free_data (xfdir_p);
    //rfm_destroy_entry(xfdir_p->en);
    g_free (xfdir_p);
    xfdir_p = NULL;


    g_static_rw_lock_reader_unlock(&(rfm_global_p->icon_theme_lock));

    TRACE("xfdir monitor: ending thread 0x%x\n", 
	    GPOINTER_TO_INT(g_thread_self()));


    // monitor event loop ends here (end outer loop)
    return NULL;
}

static void
set_local_up_item(xfdir_t *xfdir_p, gint count){
    gchar *fullpath = g_path_get_dirname (xfdir_p->en->path);
    gchar *basepath = g_path_get_basename (fullpath);
    if(strcmp (xfdir_p->en->path, "/") == 0) {
	xfdir_p->gl[count].en = NULL;
	xfdir_p->gl[count].pathv = g_strdup (g_get_host_name ());
	g_free(fullpath);
	g_free(basepath);
    } else {
	xfdir_p->gl[count].en = rfm_stat_entry (fullpath, 0);
	xfdir_p->gl[count].pathv = basepath;
	xfdir_p->gl[count].en->path = fullpath;
	SET_DUMMY_TYPE (xfdir_p->gl[count].en->type);
	SET_UP_TYPE (xfdir_p->gl[count].en->type);
	NOOP(stdout, "UP item type= 0x%x\n", xfdir_p->gl[count].en->type);
    }
}


static void
set_directory_item(xfdir_t *xfdir_p, gint count, xd_t *xd_p, gboolean fullstat){
    xfdir_p->gl[count].pathv = xd_p->d_name;
    gchar *fullpath = g_build_filename (xfdir_p->en->path, xd_p->d_name, NULL);

    gint type=0;
#ifdef HAVE_STRUCT_DIRENT_D_TYPE
    switch (xd_p->d_type){
      case DT_BLK:
	//This is a block device.
	SET_SBLK(type);
	break;
      case DT_CHR:
	//This is a character device.
	SET_SCHR(type);
	break;
      case DT_DIR:
	//This is a directory.
	SET_SDIR(type);
	break;
      case DT_FIFO:
	//This is a named pipe (FIFO).
	SET_SFIFO(type);
	break;
      case DT_LNK:
	//This is a symbolic link.
	SET_SLNK(type);
	break;
      case DT_REG:
	//This is a regular file.
	SET_SREG(type);
	break;
      case DT_SOCK:
	//This is a Unix domain socket.
	SET_SSOCK(type);
	break;
      case DT_UNKNOWN:
	//The file type is unknown.
	break;			
    }
    if (IS_LOCAL_TYPE(xfdir_p->en->type)){ 
	SET_LOCAL_TYPE(type);
    }
#endif


    if (fullstat || count < FULL_LOAD_COUNT) 
    {
	// Create en and en->st, fill in en->st.
	xfdir_p->gl[count].en = rfm_stat_entry (fullpath, type);
    } 
    else
    {
	// Create en and en->st, leave en->st memset to 0.
	xfdir_p->gl[count].en = rfm_mk_entry_path (fullpath, type);
    }
    xfdir_p->gl[count].en->path = fullpath;


}
// update xfdir structure for directory (or module)
static int
update_xfdir (xfdir_t * xfdir_p, gboolean fullstat, gint *heartbeat) {
    record_entry_t *en = xfdir_p->en;
    const gchar *module = en->module;
    
    if(module) {
        NOOP (stderr, "obtaining xfdir_p from module %s\n", module);
        if (rfm_rational(PLUGIN_DIR, module, NULL, NULL, "module_stat")
		&&
	    !rfm_rational(PLUGIN_DIR, module, NULL, en->st, "module_stat"))
	{

            DBG ("update_xfdir(): module_stat not correctly defined for %s\n", 
		    module);
        }
        xfdir_p->data = rfm_natural(PLUGIN_DIR, module, xfdir_p, "module_data");
	gint retval = 
	    GPOINTER_TO_INT(rfm_natural(PLUGIN_DIR, 
			xfdir_p->en->module, xfdir_p, 
			"module_xfdir_get"));
	NOOP(stderr, "update_xfdir got 0x%x\n", retval);
	if (retval) return 1;
	else return 0;
    } 

    xfdir_p->tama = 0;

    // Potential block: stat(). 
    // This would block the  thread.
    if (stat(en->path, en->st) < 0) return 0;
    //if (!thread_stat(en->path, en->st)) return 0;
    
    // This just does a read, not stat. May be retrieved from kernel cache 
    // and is extremely fast.
    GSList *directory_list = read_files_local (xfdir_p, heartbeat);
    if(directory_list == NULL) return 0;
    
    xfdir_p->pathc = g_slist_length(directory_list);
    xfdir_p->gl = (dir_t *) malloc (xfdir_p->pathc * sizeof (dir_t));
    if(xfdir_p->gl == NULL) g_error("malloc: %s\n", strerror(errno));
    memset (xfdir_p->gl, 0, xfdir_p->pathc * sizeof (dir_t));
    GSList *list = directory_list;
    gint count = 0;
    // This is not so fast. We may be doing a stat on all or first items.
    for(;list && list->data; list = list->next, count++){
	struct xd_t *xd_p = list->data;
	if(strcmp (xd_p->d_name, "..") == 0){
	    set_local_up_item(xfdir_p, count);
	    // d_name is not used
	    g_free(xd_p->d_name);
	} else {
	    // d_name is passed on to xfdir_p
	    if (heartbeat)
		NOOP("heartbeat records stat: %s...%d\n", xd_p->d_name, *heartbeat);
	    set_directory_item(xfdir_p, count, xd_p, fullstat);
	    // en and en->st are not NULL now (allocated in set_directory_item()).
	    xfdir_p->tama += xfdir_p->gl[count].en->st->st_size;
	}
	if (heartbeat) (*heartbeat)++;
	g_free(xd_p);
    }
    g_slist_free(directory_list);
    
    return 1;
}

