/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent_libs.h"
#include "rodent_touch.i"
#include "rodent_popup.i"

/*****************************************************************************/
GtkWidget *
rodent_get_widget_by_name (GtkWidget * parent, const gchar * name) {
    return rfm_get_widget_by_name (parent, name);

}

void
rodent_set_widget_by_name (GtkWidget * parent, const gchar * name, GtkWidget * widget) {
    rfm_set_widget_by_name (parent, name, widget);
    return ;
}

void
rodent_push_view_go_history (view_t * view_p) {
    record_entry_t *history_en;
    /*if (!view_p->en || !view_p->en->path) return; */
    if(view_p->go_list) {
        GList *last = g_list_last (view_p->go_list);
        record_entry_t *last_en;
        last_en = last->data;
        if(!last_en && !view_p->en)
            return;             /* NULL case */
        if(last_en && view_p->en) {
            if(!last_en->path && !view_p->en->path)
                return;
            if(last_en->path && view_p->en->path) {
                if(strcmp (last_en->path, view_p->en->path) == 0)
                    return;
            }
        }
    }
    history_en = rfm_copy_entry (view_p->en);
    view_p->go_list = g_list_append (view_p->go_list, history_en);
    return;
}

/******************** common menu configuration ******************************/
gint
rodent_refresh (widgets_t * widgets_p, record_entry_t * en) {
    TRACE ("rodent_refresh(%s)...\n", (en) ? en->path : "en==NULL");
    if (en && en->st) NOOP ("en->st->st_uid  %d\n", en->st->st_uid );
    g_return_val_if_fail (widgets_p != NULL, 0);
    view_t *view_p = widgets_p->view_p;
    //view_p->redraw_pixmap = TRUE;
    gint retval;
    if(en) {
	if (en->path && g_path_is_absolute(en->path)){
	    gboolean exists;
	    if (IS_LOCAL_TYPE(en->type)) exists = g_file_test(en->path, G_FILE_TEST_IS_DIR);
	    else exists = rfm_g_file_test_with_wait(en->path, G_FILE_TEST_IS_DIR);
	    if (!exists) {
		rfm_show_text(widgets_p);
		rfm_diagnostics(widgets_p, "xffm/stock_dialog-error", NULL);
		rfm_diagnostics(widgets_p, "xffm_tag/stderr", en->path, ": ", 
			strerror(ENOENT), "\n",NULL);
		return 0;
	    }
	}
        UNSET_UP_TYPE (en->type);
        if (en->st && view_p->en && view_p->en->st) NOOP("source inode=%d target inode=%d\n",
	    (gint)view_p->en->st->st_ino,
	    (gint)en->st->st_ino);
    }
    // The purpose of the following lines is to check whether the target 
    // to refresh is the same as the current view, but path differs.
    // XXX Only for local files (modules may trip here until verified).
    if (en && IS_LOCAL_TYPE(en->type) && !en->module){
	gboolean valid_entry = (en->st != NULL);
	gboolean valid_view = view_p->en && view_p->en->st;
	if (valid_view && valid_entry) {
	    gboolean same_inode = view_p->en->st->st_ino == en->st->st_ino &&
			       view_p->en->st->st_dev == en->st->st_dev;
	    gboolean redundant_condition= strcmp(view_p->en->path, en->path)!=0;
	    if (same_inode && redundant_condition){
		// this condition also is true if a reload is attempted while the
		// original load has not completed.
		//rfm_diagnostics(widgets_p, "xffm/stock_dialog-warning", NULL);
		//rfm_diagnostics(widgets_p, "xffm_tag/stderr", 
		//	strerror(EXDEV),"\n",NULL);
		g_warning("rodent_refresh cancelled upon redundant_condition\n");
		return 0;
	    }
	}
    }
    NOOP("rodent_refresh calling rodent_full_reload_view()\n");
    retval = rodent_full_reload_view ((gpointer) view_p, en);

    return retval;
}

/**************************************************************/

gpointer
rodent_create_popup_bythread(gpointer data){
    TRACE("rodent_create_popup_bythread() ...\n");
    view_t * view_p=data;
    widgets_t *widgets_p=&(view_p->widgets);
    view_p->widgets.popup = thread_mk_popup_menu (widgets_p, NULL, NULL);
    //view_p->widgets.popup = mk_popup_menu (NULL, NULL, view_p);
    thread_add_autotype_C_widgets (&(view_p->widgets), 
                                    view_p->widgets.popup, 
				    (gpointer) execute_autotype,
                                    (gpointer) rodent_mount, 
				    (gpointer) rodent_unmount);
//GDK_THREADS_LEAVE();
    TRACE("rodent_create_popup_bythread() done!\n");
	return NULL;
}

void
rodent_do_popup (view_t * view_p, const population_t * population_p, GdkEventButton * event) {
    if (g_thread_self() != rfm_global_p->self) {
	g_warning("rodent_do_popup(): this function may only be called from main thread.");
	return;
    }
    if(!view_p) return;
    widgets_t *widgets_p=&(view_p->widgets);

    if (!view_p->widgets.popup) {
#ifdef DEBUG
	rfm_show_text(widgets_p);
	//rfm_diagnostics(widgets_p, "xffm_tag/stderr", _("Connection timed out. Please try again."), "\n", NULL);
	rfm_diagnostics(widgets_p, "xffm_tag/stderr", "DBG: oops! popup menu not ready yet!\n", NULL);
#endif
	TRACE("oops! popup menu not ready yet!\n");
	return;
    }
	TRACE( "popup menu ready!\n");
    if (GTK_IS_WIDGET(view_p->widgets.popup)) {
#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
	// This is deprecated as of 2.24, which forces us
	// to get children and hide each one of them... bleah!
	gtk_widget_hide_all (view_p->widgets.popup);
#else
	GList *children = gtk_container_get_children(GTK_CONTAINER(view_p->widgets.popup));
	GList *tmp=children;
	for (; tmp && tmp->data; tmp = tmp->next){
	    GtkWidget *w=tmp->data;
	    if (GTK_WIDGET(w)) gtk_widget_hide (w);
	}
	g_list_free(children);
#endif
    }

    NOOP ("DO_POPUPx\n");

    if(population_p==NULL  && view_p->module && 
	    rfm_rational (PLUGIN_DIR, view_p->module,  widgets_p, NULL, 
		"private_popup"))
    {
        NOOP ("private module POPUPx menu here (no population).\n");
        return;
    }

    if( population_p && population_p->en && 
	    ((IS_DUMMY_TYPE (population_p->en->type)
	      && 
	      !g_path_is_absolute(population_p->en->path)
	      )
	     ||
	    view_p->en==NULL)) 
    {
        refresh_item (view_p, population_p);
        if (!IS_LOCAL_TYPE(population_p->en->type) || 
		rfm_g_file_test_with_wait(population_p->en->path, G_FILE_TEST_EXISTS)) {
            gui_autostuff (widgets_p, widgets_p->popup,
                   population_p->en, (gpointer) execute_autotype, 
                   (gpointer) rodent_mount, (gpointer) rodent_unmount);
        }
        gtk_menu_popup (GTK_MENU (widgets_p->navigation_popup), NULL, NULL, NULL, NULL, 3, event->time);
        return;
    }
    const gchar *module = POPULATION_MODULE(population_p);
    if(population_p && module && population_p->en )
    {
	//if (rfm_rational (PLUGIN_DIR, POPULATION_MODULE(population_p), 
	if (rfm_rational (PLUGIN_DIR, module, 
		widgets_p, (void *)(population_p->en), "private_popup")) {
	    NOOP ("private module POPUPx menu here.\n");
	    return;
	}
    }
//jumpit:;
    {
        ///XXX hmmm works when menu is created, but not here.....
        GtkWidget *w,
         *inner_label = rodent_get_widget_by_name (widgets_p->paper, "sort1");
        GdkPixbuf *pb;
        if(view_p->flags.preferences & SORT_ASCENDING)
            pb = rfm_get_pixbuf ("xffm/stock_sort-ascending", SIZE_BUTTON);
        else
            pb = rfm_get_pixbuf ("xffm/stock_sort-descending", SIZE_BUTTON);
        w = gtk_image_new_from_pixbuf (pb);
        gtk_widget_show (w);
        NOOP ("now setting image for sort order menu item...\n");
        gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (inner_label), w);
    }

    if(!population_p ||
	    !population_p->en ||
	    !population_p->en->path) 
    {
        refresh_item (view_p, population_p);
        set_menu_toggles (view_p);
        if(view_p->module) {
            rfm_rational (PLUGIN_DIR, view_p->module, widgets_p, NULL,  
		    "hide_local_menu_items");
            rfm_rational (PLUGIN_DIR, view_p->module, widgets_p, NULL,    
		    "show_local_menu_items");
        }
	HIDE_IT (view_p->widgets.paper, "addbookmark_menuitem");
	HIDE_IT (view_p->widgets.paper, "removebookmark_menuitem");
	HIDE_IT (view_p->widgets.paper, "subbookmark_item_yes");
	HIDE_IT (view_p->widgets.paper, "subbookmark_item_no");
	if (view_p->en && IS_SDIR(view_p->en->type)){
	  gboolean on=rodent_path_is_bookmarkable(view_p->en->path);
	  if (on){
	    setup_bookmark_menuitem(widgets_p, NULL, 
		    "subbookmark_item_yes", on);
	  } else {
	    setup_bookmark_menuitem(widgets_p, NULL, 
		    "subbookmark_item_no", on);
	  }
	}

        gtk_menu_popup (GTK_MENU (view_p->widgets.popup), NULL, NULL, NULL, NULL, 3, event->time);
        TRACE("gtk_menu_popup now...\n");
        return;
    }

    NOOP(stderr, "POPUPx gui_autostuff here...\n");

    // re-arrange autostuff here:
    gui_autostuff (widgets_p, widgets_p->popup,
                   population_p->en, (gpointer) execute_autotype, 
                   (gpointer) rodent_mount, (gpointer) rodent_unmount);

    NOOP(stderr, "POPUPx  mount_item here...\n");
    mount_item (view_p, population_p);

    NOOP(stderr, "POPUPx  remove_item here...\n");
    remove_item (view_p, population_p);
    NOOP(stderr, "POPUPx open_item  here...\n");
    open_item (view_p, population_p);
    NOOP(stderr, "POPUPx properties_item  here...\n");
    properties_item (view_p, population_p);
    NOOP(stderr, "set_menu_toggles here...\n");
    set_menu_toggles (view_p);
    if(view_p->module) {
        rfm_rational (PLUGIN_DIR, view_p->module, widgets_p, population_p->en, "hide_local_menu_items");
        rfm_rational (PLUGIN_DIR, view_p->module, widgets_p, population_p->en, "show_local_menu_items");
    }

    NOOP(stderr, "RODENT_MK_MODULE_POPUP_MENU here...\n");
    rodent_mk_module_popup_menu (view_p->module, view_p, view_p->en);
    NOOP(stderr, "gtk_menu_popup now...\n");

        // Bookmarks stuff...

    if (population_p->en && population_p->en->path &&
	    IS_SDIR(population_p->en->type)){
      gboolean on=rodent_path_is_bookmarkable(population_p->en->path);
      if (population_p->en && on){
	setup_bookmark_menuitem(widgets_p, population_p, 
		"addbookmark_menuitem", on);
      } else {
	setup_bookmark_menuitem(widgets_p, population_p, 
		"removebookmark_menuitem", on);
      }
    }

    gtk_menu_popup (GTK_MENU (widgets_p->popup), NULL, NULL, NULL, NULL, 3, event->time);
}

void
rodent_open_with (widgets_t * widgets_p, record_entry_t * en) {
    /* open with */
    gchar *command=NULL;
    gchar *command_fmt=NULL;
    TRACE ("rodent_open_with()... \n");

    if(!en || !en->path) {
        DBG ("OPEN: rodent_open_with !en || !en->path\n");
        return;
    }

    gchar *wd = g_path_get_dirname (en->path);
    if (!rfm_g_file_test_with_wait (wd, G_FILE_TEST_EXISTS)){
	 rodent_time_out(widgets_p, wd);
	 g_free(wd);
	 wd = g_strdup(g_get_home_dir());
     }
	
    g_free (widgets_p->workdir);
    widgets_p->workdir = wd;

    // Here we take special consideration for shell scripts.
    // Shell scripts will be editable files, therefore will
    // have an associated mime_command to open the editor.
    // tests 

    if (!en->mimetype) en->mimetype=MIME_type (en->path, en->st);
    if(!en->mimemagic){
	if (IS_LOCAL_TYPE(en->type)) en->mimemagic = MIME_magic (en->path);
	else en->mimemagic = g_strdup(_("unknown"));
    }

    if(!en->filetype) {
	if (IS_LOCAL_TYPE(en->type)) en->filetype = MIME_file (en->path);
	else en->filetype = g_strdup(_("unknown"));
    }

    command_fmt = MIME_command (en->mimetype);
    NOOP ("OPEN: command_fmt(%s) = %s\n", en->mimetype, command_fmt);
    if(!command_fmt) {
        command_fmt = MIME_command (en->mimemagic);
    }

    gboolean is_script= ((en->mimetype && strstr (en->mimetype, "/x-sh")) ||
			 (en->mimemagic && strstr (en->mimemagic, "/x-sh")) ||
		 (en->mimetype && strstr (en->mimetype, "/x-shellscript")) ||   
		 (en->mimemagic && strstr (en->mimemagic, "/x-shellscript")) ||
			 (en->mimetype && strstr (en->mimetype, "/x-csh")) ||   
			 (en->mimemagic && strstr (en->mimemagic, "/x-csh")) ||
			 (en->mimetype && strstr (en->mimetype, "/x-perl")) ||   
			 (en->mimemagic && strstr (en->mimemagic, "/x-perl")) 			 );
    if (is_script && !IS_EXE_TYPE(en->type)){
	g_free(command_fmt);
	command_fmt = NULL;
    }

    // for default editor...
    gchar *text_editor = NULL;
    if(!command_fmt) {
	text_editor = get_text_editor(en);
	NOOP ("OPEN: text_editor = %s\n", text_editor);
	if(text_editor) {
	    /* OK to apply an editor */
	    command_fmt = g_strconcat(text_editor, " ", NULL);
	}
    }

    //command_fmt=get_command_fmt(en);
    if (is_script) {
	rfm_show_text(widgets_p);
	if (!IS_EXE_TYPE(en->type)){
	    rfm_diagnostics(widgets_p, "xffm/stock_dialog-warning", 
		    en->path, "\n", NULL);
	    rfm_diagnostics(widgets_p, "xffm_tag/stderr",
		    _("The program exists, but is not executable.\nPlease check your installation and/or install the binary properly."), 
		    "\n", NULL);
	    rfm_diagnostics(widgets_p, "xffm/stock_dialog-info", NULL);
	    gchar *text=g_strdup_printf (_("Open with %s"), _("Text Editor"));
	    gchar *base=g_path_get_basename(en->path);
	    rfm_diagnostics(widgets_p, "xffm_tag/green", text, ": ", base, "\n", NULL);
	    g_free(base);
	    g_free(text);
	}
    }

    DBG ("rodent_open_with(): magic=%s, mime=%s, command_fmt=%s, editor=%s\n",
	    en->mimemagic, en->mimetype, command_fmt, text_editor);
    g_free(text_editor);

    if(command_fmt) {
        command = MIME_mk_command_line (command_fmt, en->path);
        TRACE( "OPEN: command = %s\n", command);

        RFM_THREAD_RUN2ARGV (widgets_p, command, FALSE);
        g_free (command);
        g_free (command_fmt);
    } else {
        on_open_with_activate (NULL, (gpointer) widgets_p);
    }
    return;
}


void
rodent_remove_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 

    if(!rfm_population_try_read_lock (view_p)) {
        return;
    }
    subthread_t *subthread_p=create_subthread_p(widgets_p);
    // set up values to determine if widgets_p remains valid
    // this is done by using the same memory locations as
    // the widgets_p, since these memory locations are not
    // destroyed (insignificant leak of 64/128 bits for each
    // view_p, much much less than a single pixbuf which
    // remains in cache. These are the default values set
    // up in create_subthread_p()
    //subthread_p->widgets.diagnostics = widgets_p->diagnostics;
    //subthread_p->widgets.diagnostics_window = widgets_p->diagnostics_window;

    if (view_p && g_slist_length(view_p->selection_list) >= 1){
	GSList *tmp=view_p->selection_list;
	for (;tmp && tmp->data; tmp=tmp->next) {
	    record_entry_t *en=tmp->data;
	    subthread_p->selection_list=
		g_slist_prepend(subthread_p->selection_list, g_strdup (en->path));
	    NOOP("creating remove thread for %s -- %s\n",en->path,
		    (gchar *)subthread_p->selection_list->data);
	}
	// unselect all items now so they won't remain in selection
	// list when the thread is done.
	rodent_unselect_all_pixbuf (view_p);
	THREAD_CREATE(rodent_remove, subthread_p, "rodent_remove");
    }

    rfm_population_read_unlock (view_p);
    
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

/****************/

extern char **environ;

/***************/

gboolean
rodent_common_mask_key (GdkEventKey * event, void *data) {
    if(!(event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK))) {
        NOOP ("no mask\n");
        return FALSE;
    }
    if(event->state & GDK_SHIFT_MASK)
        return gui_keybind_exec (event->keyval, GDK_SHIFT_MASK, data);
    else if(event->state & GDK_CONTROL_MASK)
        return gui_keybind_exec (event->keyval, GDK_CONTROL_MASK, data);
    else if(event->state & GDK_MOD1_MASK)
        return gui_keybind_exec (event->keyval, GDK_MOD1_MASK, data);
    return FALSE;
}


// This function should be entered with the GDK mutex if 
// called from the main thread. Otherwise, the GDK mutex
// will be enabled when needed.
GtkWidget *
rodent_mk_menu (widgets_t * widgets_p,
                const gchar * label, 
		const gchar * name, 
		GtkWidget * parent, 
		gpointer callback, 
		const gchar * iconfile) {

    GdkPixbuf *pixbuf=NULL;
    if (iconfile) {
	pixbuf=rfm_get_pixbuf(iconfile, TINY_ICON_SIZE);
    }
    if (rfm_global_p->self != g_thread_self()){
	GDK_THREADS_ENTER();
    }
    DBG ("rodent_mk_menu(): RSS request starts\n");
	
    GtkWidget *menu_menu = gtk_menu_new ();
    if(name) {
        gchar *g = g_strconcat (name, "_menu", NULL);
        RODENT_HOOKUP_OBJECT (widgets_p->paper, menu_menu, g);
    }
    if (parent) {
	GtkWidget *w = gtk_image_menu_item_new_with_mnemonic (label);
	gtk_image_menu_item_set_always_show_image ((GtkImageMenuItem *) w, TRUE);
	if (pixbuf) {
	    GtkWidget *image=gtk_image_new_from_pixbuf(pixbuf);
	    gtk_image_menu_item_set_image((GtkImageMenuItem *)w, image);
	}
	if(name){
	    RODENT_HOOKUP_OBJECT (widgets_p->paper, w, name);
	}
	gtk_widget_show_all (w);
	// add label menuitem to parent
	gtk_container_add (GTK_CONTAINER (parent), w);
	// attach submenu to label menuitem
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (w), menu_menu);
	if(callback) {
	    g_signal_connect ((gpointer) w, "activate", G_CALLBACK (callback), NULL);
	}
    }

    GtkWidget *v;
    v = gtk_menu_item_new_with_mnemonic (label);
    gtk_widget_show (v);
    gtk_container_add (GTK_CONTAINER (menu_menu), v);
    gtk_widget_set_sensitive (v, FALSE);
    if (rfm_global_p->self != g_thread_self()){
	GDK_THREADS_LEAVE();
    }

// thread_add_menu_separator always protects witk GDK mutex.
    thread_add_menu_separator (menu_menu);

    return menu_menu;
}

GtkWidget *
rodent_mk_pixmap_menu (const gchar * icon_id, GtkWidget * parent, int caso) {

    GtkWidget *w;
    GdkPixbuf *pb = NULL;

    NOOP (" gui_mk_pixmap_menu, icon_id: %s\n", icon_id);
    switch (caso) {
    case MENU_PIXMAP:
        pb = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
        break;
    case BUTTON_PIXMAP:
        pb = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
        break;
    case BIGGER_MENU_PIXMAP:
    case BIGGER_BUTTON_PIXMAP:
        pb = rfm_get_pixbuf (icon_id, SIZE_DIALOG);
        break;
    case OTHER_PIXMAP:
        pb = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
        break;
    }
    if(!pb) {
        pb = rfm_get_pixbuf ("xffm/emote_cool", 
		(caso==BIGGER_BUTTON_PIXMAP || caso==BIGGER_MENU_PIXMAP)?
		SIZE_DIALOG:
		SIZE_BUTTON);
        NOOP ("Unable to get pixbuf for %s (unable to get any pixbuf at all!)\n", icon_id);
        return NULL;
    }
    NOOP (" gui_mk_pixmap_menu ok, icon_id: %s\n", icon_id);

    w = gtk_image_new_from_pixbuf (pb);
    gtk_widget_show (w);

    switch (caso) {
    case BIGGER_MENU_PIXMAP:
    case MENU_PIXMAP:
        NOOP ("gtk_image_menu_item_set_image now...\n");
        gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (parent), w);
        break;
    case BIGGER_BUTTON_PIXMAP:
    case BUTTON_PIXMAP:
        gtk_container_add (GTK_CONTAINER (parent), w);
        break;
    case OTHER_PIXMAP:
        gtk_box_pack_start (GTK_BOX (parent), w, FALSE, TRUE, 0);
        gtk_misc_set_padding (GTK_MISC (w), 4, 0);
        break;
    }
    return w;
}

const gchar *
rodent_get_response (widgets_t * widgets_p, const gchar * title_txt, const gchar * label_txt, const gchar * default_txt) {
    static gchar *response_txt = NULL;
    int response = GTK_RESPONSE_NONE;
    GtkWidget *hbox, *label, *entry, *button, *dialog;
    if(!default_txt)
        default_txt = "";

    dialog = gtk_dialog_new ();

    g_free (response_txt);
    response_txt = NULL;
    if(widgets_p) {
	view_t *view_p=widgets_p->view_p;
        if(view_p && view_p->flags.type == DESKVIEW_TYPE) {
	    gtk_window_set_keep_above (GTK_WINDOW(dialog), TRUE);
	    gtk_window_stick (GTK_WINDOW(dialog));
	} else {   
            gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
            gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (widgets_p->window));
        } 
    } else {
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    }
    gtk_window_set_resizable (GTK_WINDOW (dialog), TRUE);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);

    if(label_txt)
        label = gtk_label_new (label_txt);
    else
        label = gtk_label_new (_("Preparing"));

    hbox = rfm_hbox_new (TRUE, 6);
    gtk_box_pack_start (GTK_BOX (gtk_dialog_get_content_area(GTK_DIALOG (dialog))), hbox, FALSE, FALSE, 0);

    entry = gtk_entry_new ();

    gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
    gtk_widget_show_all (hbox);


    gtk_entry_set_text ((GtkEntry *) entry, default_txt);

    g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (activate_entry), dialog);

    button = rfm_dialog_button ("xffm/stock_cancel", _("Cancel"));
    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_CANCEL);
    button = rfm_dialog_button ("xffm/stock_ok", _("Ok"));

    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_YES);
    gtk_widget_realize (dialog);

    if(title_txt)
        gtk_window_set_title (GTK_WINDOW (dialog), title_txt);
    else
        gdk_window_set_decorations (gtk_widget_get_window(dialog), GDK_DECOR_BORDER);

    /* show dialog and return */
    gtk_widget_show_all (dialog);
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    if(response == GTK_RESPONSE_YES) {
        const gchar *et = gtk_entry_get_text (GTK_ENTRY(entry));
        if(et && strlen (et)) {
            response_txt = g_strdup (et);
        }
    }
    gtk_widget_hide (dialog);
    gtk_widget_destroy (dialog);
    if(response_txt != NULL){
	g_strstrip (response_txt);
    }

    /* use slocate_string to select... */
    return (const gchar *)(response_txt);
}



gchar *
rodent_get_radio_response (widgets_t * widgets_p, const gchar * title_txt, const gchar * label_txt, gchar **radio_options) {
    gint response = GTK_RESPONSE_NONE;
    GtkWidget *hbox, *label,  *button, *dialog;

    
    gint radiobuttons = 0;
    gchar **p;
    for (p = radio_options;p && *p; p++, radiobuttons++);

    GtkWidget *radio[radiobuttons];

    dialog = gtk_dialog_new ();

    if(widgets_p) {
	view_t *view_p=widgets_p->view_p;
        if(view_p && view_p->flags.type == DESKVIEW_TYPE) {
	    gtk_window_set_keep_above (GTK_WINDOW(dialog), TRUE);
	    gtk_window_stick (GTK_WINDOW(dialog));
	} else {   
            gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
            gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (widgets_p->window));
        } 
    } else {
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    }
    gtk_window_set_resizable (GTK_WINDOW (dialog), TRUE);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);

    if(label_txt) label = gtk_label_new (label_txt);
    else label = gtk_label_new ("rodent_get_radio_response");

    hbox = rfm_hbox_new (TRUE, 6);
    gtk_box_pack_start (GTK_BOX (gtk_dialog_get_content_area(GTK_DIALOG (dialog))), hbox, FALSE, FALSE, 0);

    
    GtkWidget *vbox = rfm_vbox_new (TRUE, 6);
    GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_box_pack_start (GTK_BOX (hbox), scrolled_window, TRUE, TRUE, 0);

    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), vbox);

    gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
    gtk_widget_set_size_request(scrolled_window, -1, 200);

    GSList *group=NULL;
    gint i=0;
    for (p = radio_options;p && *p; p++, i++){
	if (i==0) {
	    radio[i] = gtk_radio_button_new_with_label(group, *p);
	} else {
	    radio[i] = gtk_radio_button_new_with_label(group, *p);
	}
	group = gtk_radio_button_get_group (GTK_RADIO_BUTTON(radio[i]));
	g_object_set_data(G_OBJECT(radio[i]), "type", *p);
	gtk_box_pack_start (GTK_BOX (vbox), radio[i], TRUE, TRUE, 0);
    }



    gtk_widget_show_all (hbox);


    button = rfm_dialog_button ("xffm/stock_cancel", _("Cancel"));
    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_CANCEL);
    button = rfm_dialog_button ("xffm/stock_ok", _("Ok"));

    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_YES);
    gtk_widget_realize (dialog);

    if(title_txt)
        gtk_window_set_title (GTK_WINDOW (dialog), title_txt);
    else
        gdk_window_set_decorations (gtk_widget_get_window(dialog), GDK_DECOR_BORDER);

    /* show dialog and return */
    gtk_widget_show_all (dialog);
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    const gchar *type = NULL;
    if(response == GTK_RESPONSE_YES) {
	for (i=0; i<radiobuttons; i++){
	    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radio[i]))){
		type = g_object_get_data(G_OBJECT(radio[i]), "type");
		break;
	    }
	}
    }
    gtk_widget_hide (dialog);
    gtk_widget_destroy (dialog);

    if (!type) return NULL;
    /* use slocate_string to select... */
    return g_strdup(type);
}


const gchar *
rodent_get_response_history (widgets_t * widgets_p,
                             const gchar * title_txt,
                             const gchar * label_txt,
                             const gchar * extra_txt,
                             gchar * history_file,
                             const gchar * path,
                             const gchar * entry_text,
                             gchar * flagfile, 
			     const gchar * check_label, 
			     int filechooser_action, 
			     const gchar * folder,
			     gint completion_type) {
    static extra_key_t extra_key;
    static gchar *response_txt = NULL;
    int response = GTK_RESPONSE_NONE;
    GtkWidget *hbox,
     *label,
     *button,
     *dialog;
    GtkWidget *combo = NULL;
    void *combo_info = NULL;
    filechooser_t filechooser_v;

    if (chdir(folder) < 0){
	g_warning("cannot chdir(%s)", folder);
    }
    dialog = gtk_dialog_new ();

    g_free (response_txt);
    response_txt = NULL;
    if(widgets_p) {
	view_t *view_p=widgets_p->view_p;
        if(view_p && view_p->flags.type == DESKVIEW_TYPE) {
	    gtk_window_set_keep_above (GTK_WINDOW(dialog), TRUE);
	    gtk_window_stick (GTK_WINDOW(dialog));
	} else {   
            gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
            gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (widgets_p->window));
        } 
    } else {
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    }


    gtk_window_set_resizable (GTK_WINDOW (dialog), TRUE);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);


#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
    // this is deprecated...
    combo = gtk_combo_box_entry_new ();
#else
    combo =  gtk_combo_box_new_with_entry ();
#endif
    gtk_widget_set_size_request (GTK_WIDGET (combo), 350, -1);
    if (extra_txt) {
	gchar *markup;
	label=gtk_label_new ("");
	markup = g_markup_printf_escaped ("<span style=\"italic\">%s</span>\n", extra_txt);
	gtk_label_set_markup (GTK_LABEL (label), markup);
	g_free (markup);	
	/*gchar *marked_up=g_strdup_printf("<b>%s</b>\n", extra_txt);
        label = gtk_label_new (marked_up);
	g_free(marked_up);*/
        hbox = rfm_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (gtk_dialog_get_content_area(GTK_DIALOG (dialog))), hbox, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show_all(hbox);
    }

    if(label_txt)
        label = gtk_label_new (label_txt);
    else
        label = gtk_label_new (_("Preparing"));
    hbox = rfm_hbox_new (FALSE, 6);
    gtk_box_pack_start (GTK_BOX (gtk_dialog_get_content_area(GTK_DIALOG (dialog))), hbox, FALSE, FALSE, 0);
    GtkWidget *vbox = rfm_vbox_new (FALSE, 6);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
    vbox = rfm_vbox_new (FALSE, 6);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), (GtkWidget *) combo, FALSE, FALSE, 0);


    gboolean combobox_active =GPOINTER_TO_INT(rfm_void(MODULE_DIR, "combobox", "module_active"));
    if(combo_info == NULL) {
	    NOOP ("signal widgets_p is 0x%lx\n", (unsigned long)widgets_p);
	    combo_info = COMBOBOX_init_combo (combo, completion_type);
    } else {
	COMBOBOX_clear_history (combo_info);
    }
    COMBOBOX_set_quick_activate(combo_info, GINT_TO_POINTER(TRUE));
    COMBOBOX_set_activate_function(combo_info, activate_entry);
    COMBOBOX_set_cancel_function(combo_info, cancel_entry);
    COMBOBOX_set_activate_user_data(combo_info, dialog);
    COMBOBOX_set_cancel_user_data(combo_info, dialog);
    COMBOBOX_set_extra_key_completion_function(combo_info, extra_key_completionR);
    COMBOBOX_set_extra_key_completion_data(combo_info, &extra_key);
    /*combo_info->activate_func = activate_entry;
    combo_info->activate_user_data = dialog;
    combo_info->cancel_func = cancel_entry;
    combo_info->cancel_user_data = dialog;
    xfc_fun->extra_key_completion = extra_key_completionR;
    xfc_fun->extra_key_data = &extra_key;*/

    COMBOBOX_read_history (combo_info, history_file);
    COMBOBOX_set_combo (combo_info);


    if(filechooser_action == GTK_FILE_CHOOSER_ACTION_OPEN || filechooser_action == GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER) {
        filechooser_v.combo_info = combo_info;
        filechooser_v.parent = dialog;
        filechooser_v.folder = folder;
        filechooser_v.title = title_txt;
	if (combobox_active) {
	    filechooser_v.entry = GTK_ENTRY(COMBOBOX_get_entry_widget (combo_info));
	} else {
	    filechooser_v.entry = GTK_ENTRY(gtk_bin_get_child (GTK_BIN(combo)));

	}
        filechooser_v.activate_func = activate_entry;
        filechooser_v.activate_user_data = dialog;

        filechooser_v.filechooser_action = filechooser_action;
        preload (filechooser_v.folder);

        button = gtk_button_new ();
	GdkPixbuf *pixbuf=rfm_get_pixbuf("xffm/stock_directory", SIZE_BUTTON);
        GtkWidget *image;
	if (pixbuf) {
	    image = gtk_image_new_from_pixbuf (pixbuf);
	} else {
	    image = gtk_image_new_from_stock (GTK_STOCK_DIRECTORY, GTK_ICON_SIZE_BUTTON);
	}
        gtk_button_set_image ((GtkButton *) button, image);
        vbox = rfm_vbox_new (FALSE, 6);
        gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
        g_signal_connect (button, "clicked", G_CALLBACK (filechooser), (gpointer) (&filechooser_v));
        gtk_widget_show (button);
    }
    gtk_widget_show_all (hbox);

    if(path) {
        gchar *type = MIME_type ((void *)path, NULL);
        gchar *p = NULL;
	//   This may be borked if path is a non local file (i.e., obexfs)
	//   so we verify on the parent folder.
        if(!type){
	    if (widgets_p && widgets_p->view_p && widgets_p->view_p->en &&
		    IS_LOCAL_TYPE(widgets_p->view_p->en->type)){
		type = MIME_magic (path);
	    }
	} 
	if (!type) type = g_strdup(_("unknown"));
        p = MIME_command (type);
        g_free (type);
        if(p){
	    NOOP("COMBO: setting entry to %s\n", p);
	    if (combobox_active) {
		COMBOBOX_set_entry (combo_info, p);
	    } else {
		gtk_entry_set_text (filechooser_v.entry,p);
	    }
	}
    }

    if (entry_text) {
	NOOP ("COMBO: setting combo to %s\n", entry_text);
	if (combobox_active) {
	    COMBOBOX_set_entry (combo_info, entry_text);
 	} else {
	    gtk_entry_set_text (filechooser_v.entry,entry_text);
	}
   }
    if (!entry_text && ! path) {
	COMBOBOX_set_default (combo_info);
	NOOP ("COMBO: setting combo to default (!entry_text && ! path)\n");
    }

    if(check_label && flagfile) {
        extra_key.check1 = (GtkWidget *) gtk_check_button_new_with_mnemonic (check_label);
        g_signal_connect (extra_key.check1, "toggled", G_CALLBACK (toggle_flag), (gpointer) & extra_key);
        //extra_key.entry = GTK_ENTRY(COMBOBOX_get_entry_widget(combo_info));
        extra_key.entry = filechooser_v.entry;
        gtk_box_pack_start (GTK_BOX (gtk_dialog_get_action_area (GTK_DIALOG (dialog))), GTK_WIDGET (extra_key.check1), FALSE, FALSE, 0);
        gtk_widget_show (extra_key.check1);
    }

    button = rfm_dialog_button ("xffm/stock_cancel", _("Cancel"));
    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_CANCEL);
    button = rfm_dialog_button ("xffm/stock_ok", _("Ok"));


    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_YES);
    gtk_widget_realize (dialog);
    if(flagfile) {
        extra_key.flagfile = flagfile;
        extra_key_completionR (&extra_key);
    } else
        extra_key.flagfile = NULL;

    if(title_txt)
        gtk_window_set_title (GTK_WINDOW (dialog), title_txt);
    else
        gdk_window_set_decorations (gtk_widget_get_window(dialog), GDK_DECOR_BORDER);

    /* show dialog and return */
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    if(response == GTK_RESPONSE_YES) {
        if(response_txt)
            g_strstrip (response_txt);
        const gchar *et = (combobox_active)?
	    COMBOBOX_get_entry (combo_info):
	    gtk_entry_get_text (filechooser_v.entry);
        if(et && strlen (et)) {
            response_txt = (gchar *) malloc ((strlen (et) + 3) * sizeof (gchar));
	    if (!response_txt) g_error("malloc: %s", strerror(errno));
            memset (response_txt, 0, (strlen (et) + 3) * sizeof (gchar));
            strcpy (response_txt, et);
            COMBOBOX_save_to_history ((char *)history_file, (char *)response_txt);
            if(flagfile)
                save_flags (&extra_key);
        }
        if(flagfile && GTK_IS_TOGGLE_BUTTON(extra_key.check1)) {
            gboolean active = gtk_toggle_button_get_active ((GtkToggleButton *)
                                                            extra_key.check1);
            //TRACE("active=%d\n",active);
            if(active)
                response_txt[strlen (response_txt) + 1] = 1;
        }
    }
    gtk_widget_hide (dialog);
    // cleanup combobox module objects:
    COMBOBOX_destroy_combo(combo_info);
    
    gtk_widget_destroy (dialog);
    if (chdir(g_get_home_dir()) < 0){
	g_warning("cannot chdir(g_get_home_dir())");
    }

    return (const gchar *)(response_txt);
}
/*********   find callbacks   *********/

void
rodent_glob_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    TRACE ("rodent_glob_activate(): view_p=0x%lx\n", (unsigned long)view_p);
    record_entry_t *en = NULL;
    if(rfm_population_try_read_lock (view_p)) {
        if(view_p->mouse_event.selected_p && view_p->mouse_event.selected_p->en 
		&& IS_SDIR(view_p->mouse_event.selected_p->en->type)
           && g_slist_length (view_p->selection_list) == 1) {
            en = view_p->mouse_event.selected_p->en;
        } else if(view_p->en) {
            en = view_p->en;
        }

        if(!rfm_rational (MODULE_DIR, "fgr", en, widgets_p, "do_find")) {
            DBG ("rodent_glob_activate(): unable to open fgr dialog\n");
        } else {
            NOOP ("FGR: fgr dialog OK\n");
        }
        rfm_population_read_unlock (view_p);
    }
    return;
}

static void *
cleanup_thumbnail_f(void *data){
    gchar *path = data;
    // This should always resolve to a local absolute path...
    if (!g_file_test(path, G_FILE_TEST_IS_DIR)) return NULL;
    // Comments are for reading and understanding code:
    // clear any thumbnail previews:
    gchar *test_image = g_build_filename (path, "test.jpg", NULL);
    gchar *filename = rfm_get_thumbnail_path (test_image, 48);
    gchar *dirname = g_path_get_dirname (filename);
    DIR *directory;
    struct dirent *d;
    directory = opendir (dirname);
    if(!directory) return NULL;
    
    while((d = readdir (directory)) != NULL) {
        if(strcmp (d->d_name, ".") == 0) continue;
        if(strcmp (d->d_name, "..") == 0) continue;
	gchar *fullpath = g_build_filename (dirname, d->d_name, NULL);
	if (unlink(fullpath) < 0);
	g_free (fullpath);
    }

    g_free (test_image);
    g_free (filename);
    g_free (dirname);
    g_free (path);
    return NULL;
}

void
rodent_refresh_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rodent_entry_available(widgets_p, view_p->en)) return; 
    if (!view_p->en) return;
    
    if(view_p->en->path) {
	gchar *path = g_strdup( view_p->en->path);
	cleanup_thumbnail_f(path);
    }
    if (view_p->en->module){
	if (rfm_natural(PLUGIN_DIR, view_p->en->module, view_p, "reload")) return;
    }
    record_entry_t *en = rfm_copy_entry (view_p->en);
    if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
    return;
}

void
rodent_back_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;

    GList *last;
    record_entry_t *en;
    TRACE ("rodent_back_activate()\n");
last_one:
    last = g_list_last (view_p->go_list);
    if(!last) return;
    en = last->data;

    if (!rodent_entry_available(widgets_p, en)) return; 
    if(en && !en->module && !rfm_g_file_test_with_wait(en->path, G_FILE_TEST_EXISTS)){
	rodent_pop_view_go_history (view_p);
	goto last_one;
    }

    if(en) view_p->module = en->module;
    else view_p->module = NULL;

    rodent_pop_view_go_history (view_p);

    rodent_full_reload_view ((gpointer) view_p, en);
    
}

void
rodent_jump_to_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;

    gchar *f = g_build_filename (GOTO_DBH_FILE, NULL);
    const gchar *response = rodent_get_response_history (NULL, 
	    _("Go To"), _("Path"), 
	    NULL, // extra_text
	    f,
	    NULL, NULL, NULL, NULL,
            GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
            g_get_home_dir (),
	    MATCH_FILE);
    g_free (f);

    NOOP("dialog returned with %s\n", response);
    gchar *choice = (response)? g_strdup(response) : NULL;
    if (!choice) return;

    g_strstrip(choice);
    if (strlen(choice)==0) {
	g_free(choice);
	return;
    }
    if (g_path_is_absolute(choice)){
	if (!rfm_g_file_test_with_wait (choice, G_FILE_TEST_EXISTS)){
	   rodent_time_out(widgets_p, choice);
	   g_free(choice);
	   return;
	}
    } else {
	// make path absolute
	const gchar *dirname;
	if (view_p->en && view_p->en->path) dirname = view_p->en->path;
	else dirname = g_get_home_dir();
	gchar *fullpath = g_build_filename(dirname, choice, NULL);
	if (!rfm_g_file_test_with_wait (fullpath, G_FILE_TEST_EXISTS)){
	   rodent_time_out(widgets_p, fullpath);
	   g_free(fullpath);
	   g_free(choice);
	   return;
	}
	g_free(fullpath);
    }


    record_entry_t *new_en;
    if(!rfm_g_file_test_with_wait (choice, G_FILE_TEST_IS_DIR)) {
	gchar *text=g_strdup_printf(_("%s does not exist."), choice);
	rfm_confirm (NULL, GTK_MESSAGE_ERROR, text, NULL, NULL);
	g_free(text);
        return;
    }
    else if(chdir (choice) < 0) {
        rfm_show_text(widgets_p);
        rfm_diagnostics (widgets_p, "xffm/stock_dialog-error", NULL);
        rfm_diagnostics (widgets_p, "xffm_tag/stderr", choice, ": ", strerror (errno), "\n", NULL);
	g_free(choice);
	return;
    } 
    gchar *g = g_get_current_dir ();
    SETWD();

    if(view_p->child_constructor) {
        (*view_p->child_constructor) (widgets_p, (char *)g);
    } else {
        rodent_push_view_go_history (view_p);
        new_en = rfm_stat_entry (g, 0);
        rfm_save_to_go_history ((gchar *) g);
	if (!rodent_refresh (widgets_p, new_en)){ rfm_destroy_entry(new_en); }
    }
    g_free(choice);
    g_free (g);
}

void
rodent_home_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    if (!rfm_g_file_test_with_wait (g_get_home_dir(), G_FILE_TEST_IS_DIR)){
	 rodent_time_out(widgets_p, g_get_home_dir());
	 return; 
     }
    if(!view_p) {
        g_warning ("!view_p");
        return;
    }
    if(view_p->child_constructor) {
        (*view_p->child_constructor) (widgets_p, (char *)g_get_home_dir ());
    } else {
        record_entry_t *en = rfm_stat_entry ((gchar *) g_get_home_dir (), 0);
        rodent_push_view_go_history (view_p);
	if (!rodent_refresh (widgets_p, en)){ rfm_destroy_entry(en); }
    }
}

void
rodent_host_activate (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    view_t *view_p = widgets_p->view_p;
    record_entry_t *en = NULL;
    if(!view_p)
        return;
    if(view_p->child_constructor) {
        (*view_p->child_constructor) (widgets_p, NULL);
    } else {
        rodent_push_view_go_history (view_p);
        rodent_refresh (widgets_p, en);
    }
}

void
rodent_add_navigation_menu_items (widgets_t * widgets_p, GtkWidget *target_menu){
     thread_navigation_menu_items (widgets_p, NULL, target_menu);
    return;
}


void
rodent_add_general_menu_items (widgets_t * widgets_p, GtkWidget *target_menu){
    thread_general_menu_items (widgets_p, NULL, target_menu);
    return;
}

void
rodent_add_file_menu_items (widgets_t * widgets_p, GtkWidget *target_menu){
    thread_file_menu_items (widgets_p, NULL, target_menu);
    return;
}

void
rodent_add_exec_menu_items (widgets_t * widgets_p, GtkWidget *target_menu){
     thread_exec_menu_items (widgets_p, NULL, target_menu);
    return;
}

void
rodent_add_fileexec_menu_items (widgets_t * widgets_p, GtkWidget *target_menu){
    thread_fileexec_menu_items (widgets_p, NULL, target_menu);
    return;
}

void
rodent_add_size_menu_items (widgets_t * widgets_p, GtkWidget *target_menu){
     thread_size_menu_items (widgets_p, NULL, target_menu);
    return;
}
